#!/bin/sh
#
# Copyright 2005-2007 VMware, Inc.  All rights reserved.
#
# This script manages the services needed to use networking with VMware software
#

# BEGINNING_OF_UTIL_DOT_SH
#!/bin/sh

# A few utility functions used by our shell scripts. These are patched in 
# during make.

vmware_failed() {
  if [ "`type -t 'echo_failure' 2>/dev/null`" = 'function' ]; then
    echo_failure
  else
    echo -n "$rc_failed"
  fi
}

vmware_success() {
  if [ "`type -t 'echo_success' 2>/dev/null`" = 'function' ]; then
    echo_success
  else
    echo -n "$rc_done"
  fi
}

# Execute a macro
vmware_exec() {
  local msg="$1"  # IN
  local func="$2" # IN
  shift 2

  echo -n '   '"$msg"

  # On Caldera 2.2, SIGHUP is sent to all our children when this script exits
  # I wanted to use shopt -u huponexit instead but their bash version
  # 1.14.7(1) is too old
  #
  # Ksh does not recognize the SIG prefix in front of a signal name
  if [ "$VMWARE_DEBUG" = 'yes' ]; then
    (trap '' HUP; "$func" "$@")
  else
    (trap '' HUP; "$func" "$@") >/dev/null 2>&1
  fi
  if [ "$?" -gt 0 ]; then
    vmware_failed
    echo
    return 1
  fi

  vmware_success
  echo
  return 0
}

# Execute a macro in the background
vmware_bg_exec() {
  local msg="$1"  # IN
  local func="$2" # IN
  shift 2

  if [ "$VMWARE_DEBUG" = 'yes' ]; then
    # Force synchronism when debugging
    vmware_exec "$msg" "$func" "$@"
  else
    echo -n '   '"$msg"' (background)'

    # On Caldera 2.2, SIGHUP is sent to all our children when this script exits
    # I wanted to use shopt -u huponexit instead but their bash version
    # 1.14.7(1) is too old
    #
    # Ksh does not recognize the SIG prefix in front of a signal name
    (trap '' HUP; "$func" "$@") 2>&1 | logger -t 'VMware[init]' -p daemon.err &

    vmware_success
    echo
    return 0
  fi
}

# This is a function in case a future product name contains language-specific
# escape characters.
vmware_product_name() {
  echo 'VMware Workstation'
  exit 0
}

# This is a function in case a future product contains language-specific
# escape characters.
vmware_product() {
  echo 'ws'
  exit 0
}

# Used to check whether the rpm built matches the platform to run on.
# I.E.  an x86_64 product shouldn't run on an x86/i386 platform.
vmware_product_architecture() {
  echo 'i386'
  exit 0
}

# Check to see if the platform is x86_64 or not.  Platforms that are
# not x86_64 may be of several types: i386, i686, etc but there is
# only one x86_64.
vmware_product_platform_match() {
  PLATFORM_ARCH=`uname -m`
  PRODUCT_ARCH=`vmware_product_architecture`

  if [ "$PLATFORM_ARCH" == 'x86_64' ] && [ "$PRODUCT_ARCH" != 'x86_64' ] ; then
    echo '.'
  fi
  if [ "$PLATFORM_ARCH" != 'x86_64' ] && [ "$PRODUCT_ARCH" == 'x86_64' ] ; then
    echo '.'
  fi
  echo ''
}

# They are a lot of small utility programs to create temporary files in a
# secure way, but none of them is standard. So I wrote this --hpreg
make_tmp_dir() {
  local dirname="$1" # OUT
  local prefix="$2"  # IN
  local tmp
  local serial
  local loop

  tmp="${TMPDIR:-/tmp}"

  # Don't overwrite existing user data
  # -> Create a directory with a name that didn't exist before
  #
  # This may never succeed (if we are racing with a malicious process), but at
  # least it is secure
  serial=0
  loop='yes'
  while [ "$loop" = 'yes' ]; do
    # Check the validity of the temporary directory. We do this in the loop
    # because it can change over time
    if [ ! -d "$tmp" ]; then
      echo 'Error: "'"$tmp"'" is not a directory.'
      echo
      exit 1
    fi
    if [ ! -w "$tmp" -o ! -x "$tmp" ]; then
      echo 'Error: "'"$tmp"'" should be writable and executable.'
      echo
      exit 1
    fi

    # Be secure
    # -> Don't give write access to other users (so that they can not use this
    # directory to launch a symlink attack)
    if mkdir -m 0755 "$tmp"'/'"$prefix$serial" >/dev/null 2>&1; then
      loop='no'
    else
      serial=`expr $serial + 1`
      serial_mod=`expr $serial % 200`
      if [ "$serial_mod" = '0' ]; then
        echo 'Warning: The "'"$tmp"'" directory may be under attack.'
        echo
      fi
    fi
  done

  eval "$dirname"'="$tmp"'"'"'/'"'"'"$prefix$serial"'
}

# Checks if the given pid represents a live process.
# Returns 0 if the pid is a live process, 1 otherwise
vmware_is_process_alive() {
  local pid="$1" # IN

  ps -p $pid | grep $pid > /dev/null 2>&1
}

# Check if the process associated to a pidfile is running.
# Return 0 if the pidfile exists and the process is running, 1 otherwise
vmware_check_pidfile() {
  local pidfile="$1" # IN
  local pid

  pid=`cat "$pidfile" 2>/dev/null`
  if [ "$pid" = '' ]; then
    # The file probably does not exist or is empty. Failure
    return 1
  fi
  # Keep only the first number we find, because some Samba pid files are really
  # trashy: they end with NUL characters
  # There is no double quote around $pid on purpose
  set -- $pid
  pid="$1"

  vmware_is_process_alive $pid
}

# Note:
#  . Each daemon must be started from its own directory to avoid busy devices
#  . Each PID file doesn't need to be added to the installer database, because
#    it is going to be automatically removed when it becomes stale (after a
#    reboot). It must go directly under /var/run, or some distributions
#    (RedHat 6.0) won't clean it
#

# Terminate a process synchronously
vmware_synchrone_kill() {
   local pid="$1"    # IN
   local signal="$2" # IN
   local second

   kill -"$signal" "$pid"

   # Wait a bit to see if the dirty job has really been done
   for second in 0 1 2 3 4 5 6 7 8 9 10; do
      vmware_is_process_alive "$pid"
      if [ "$?" -ne 0 ]; then
         # Success
         return 0
      fi

      sleep 1
   done

   # Timeout
   return 1
}

# Kill the process associated to a pidfile
vmware_stop_pidfile() {
   local pidfile="$1" # IN
   local pid

   pid=`cat "$pidfile" 2>/dev/null`
   if [ "$pid" = '' ]; then
      # The file probably does not exist or is empty. Success
      return 0
   fi
   # Keep only the first number we find, because some Samba pid files are really
   # trashy: they end with NUL characters
   # There is no double quote around $pid on purpose
   set -- $pid
   pid="$1"

   # First try a nice SIGTERM
   if vmware_synchrone_kill "$pid" 15; then
      return 0
   fi

   # Then send a strong SIGKILL
   if vmware_synchrone_kill "$pid" 9; then
      return 0
   fi

   return 1
}

# Determine if SELinux is enabled
isSELinuxEnabled() {
   if [ "`cat /selinux/enforce 2> /dev/null`" = "1" ]; then
      echo "yes"
   else
      echo "no"
   fi
}

# Runs a command and retries under the provided SELinux context if it fails
vmware_exec_selinux() {
   local command="$1"
   # XXX We should probably ask the user at install time what context to use
   # when we retry commands.  unconfined_t is the correct choice for Red Hat.
   local context="unconfined_t"
   local retval

   $command
   retval=$?
   if [ $retval -ne 0 -a "`isSELinuxEnabled`" = 'yes' ]; then
      runcon -t $context -- $command
      retval=$?
   fi

   return $retval
}

# Start the blocking file system.  This consists of loading the module and
# mounting the file system.
vmware_start_vmblock() {
   mkdir -p /tmp/VMwareDnD && chmod 1777 /tmp/VMwareDnD
   vmware_exec 'Loading module' vmware_load_module $vmblock
   exitcode=`expr $exitcode + $?`
   vmware_exec_selinux "mount -t vmblock none /proc/fs/vmblock/mountPoint"
}

# Stop the blocking file system
vmware_stop_vmblock() {
   # It's okay for us to not factor the umount command's error code into
   # exitcode since the module unload will fail if the file system remains
   # mounted.
   vmware_exec_selinux "umount /proc/fs/vmblock/mountPoint"

   vmware_unload_module $vmblock
}

is_ESX_running() {
  if [ ! -f "$vmdb_answer_SBINDIR"/vmware-checkvm ] ; then
    echo no
    return
  fi
  if "$vmdb_answer_SBINDIR"/vmware-checkvm -p | grep -q ESX; then
    echo yes
  else
    echo no
  fi
}

#
# Start vmblock only if ESX is not running and the config script
# built/loaded it (kernel is >= 2.4.0 and  product is tools-for-linux).
# 
is_vmblock_needed() {
  if [ "`is_ESX_running`" = 'yes' ]; then
    echo no
  else
    if [ "$vmdb_answer_VMBLOCK_CONFED" = 'yes' ]; then
      echo yes
    else
      echo no
    fi
  fi
}

wrap () {
  AMSG="$1"
  while [ `echo $AMSG | wc -c` -gt 75 ] ; do
    AMSG1=`echo $AMSG | sed 's/\(.\{1,75\} \).*/\1/'`
    AMSG=`echo $AMSG | sed -e 's/.\{1,75\} //'`
    echo "  $AMSG1"
  done
  echo "  $AMSG"
}

# END_OF_UTIL_DOT_SH

vmware_etc_dir=/etc/vmware

# Since this script is installed, our main database should be installed too and
# should contain the basic information
vmware_db="$vmware_etc_dir"/locations
if [ ! -r "$vmware_db" ]; then
   echo 'Warning: Unable to find '"`vmware_product_name`""'"'s main database '"$vmware_db"'.'
   echo

   exit 1
fi

# BEGINNING_OF_DB_DOT_SH
#!/bin/sh

#
# Manage an installer database
#

# Add an answer to a database in memory
db_answer_add() {
  local dbvar="$1" # IN/OUT
  local id="$2"    # IN
  local value="$3" # IN
  local answers
  local i

  eval "$dbvar"'_answer_'"$id"'="$value"'

  eval 'answers="$'"$dbvar"'_answers"'
  # There is no double quote around $answers on purpose
  for i in $answers; do
    if [ "$i" = "$id" ]; then
      return
    fi
  done
  answers="$answers"' '"$id"
  eval "$dbvar"'_answers="$answers"'
}

# Remove an answer from a database in memory
db_answer_remove() {
  local dbvar="$1" # IN/OUT
  local id="$2"    # IN
  local new_answers
  local answers
  local i

  eval 'unset '"$dbvar"'_answer_'"$id"

  new_answers=''
  eval 'answers="$'"$dbvar"'_answers"'
  # There is no double quote around $answers on purpose
  for i in $answers; do
    if [ "$i" != "$id" ]; then
      new_answers="$new_answers"' '"$i"
    fi
  done
  eval "$dbvar"'_answers="$new_answers"'
}

# Load all answers from a database on stdin to memory (<dbvar>_answer_*
# variables)
db_load_from_stdin() {
  local dbvar="$1" # OUT

  eval "$dbvar"'_answers=""'

  # read doesn't support -r on FreeBSD 3.x. For this reason, the following line
  # is patched to remove the -r in case of FreeBSD tools build. So don't make
  # changes to it. -- Jeremy Bar
  while read -r action p1 p2; do
    if [ "$action" = 'answer' ]; then
      db_answer_add "$dbvar" "$p1" "$p2"
    elif [ "$action" = 'remove_answer' ]; then
      db_answer_remove "$dbvar" "$p1"
    fi
  done
}

# Load all answers from a database on disk to memory (<dbvar>_answer_*
# variables)
db_load() {
  local dbvar="$1"  # OUT
  local dbfile="$2" # IN

  db_load_from_stdin "$dbvar" < "$dbfile"
}

# Iterate through all answers in a database in memory, calling <func> with
# id/value pairs and the remaining arguments to this function
db_iterate() {
  local dbvar="$1" # IN
  local func="$2"  # IN
  shift 2
  local answers
  local i
  local value

  eval 'answers="$'"$dbvar"'_answers"'
  # There is no double quote around $answers on purpose
  for i in $answers; do
    eval 'value="$'"$dbvar"'_answer_'"$i"'"'
    "$func" "$i" "$value" "$@"
  done
}

# If it exists in memory, remove an answer from a database (disk and memory)
db_remove_answer() {
  local dbvar="$1"  # IN/OUT
  local dbfile="$2" # IN
  local id="$3"     # IN
  local answers
  local i

  eval 'answers="$'"$dbvar"'_answers"'
  # There is no double quote around $answers on purpose
  for i in $answers; do
    if [ "$i" = "$id" ]; then
      echo 'remove_answer '"$id" >> "$dbfile"
      db_answer_remove "$dbvar" "$id"
      return
    fi
  done
}

# Add an answer to a database (disk and memory)
db_add_answer() {
  local dbvar="$1"  # IN/OUT
  local dbfile="$2" # IN
  local id="$3"     # IN
  local value="$4"  # IN

  db_remove_answer "$dbvar" "$dbfile" "$id"
  echo 'answer '"$id"' '"$value" >> "$dbfile"
  db_answer_add "$dbvar" "$id" "$value"
}

# Add a file to a database on disk
# 'file' is the file to put in the database (it may not exist on the disk)
# 'tsfile' is the file to get the timestamp from, '' if no timestamp
db_add_file() {
  local dbfile="$1" # IN
  local file="$2"   # IN
  local tsfile="$3" # IN
  local date

  if [ "$tsfile" = '' ]; then
    echo 'file '"$file" >> "$dbfile"
  else
    date=`date -r "$tsfile" '+%s' 2> /dev/null`
    if [ "$date" != '' ]; then
      date=' '"$date"
    fi
    echo 'file '"$file$date" >> "$dbfile"
  fi
}

# Add a directory to a database on disk
db_add_dir() {
  local dbfile="$1" # IN
  local dir="$2"    # IN

  echo 'directory '"$dir" >> "$dbfile"
}
# END_OF_DB_DOT_SH

db_load 'vmdb' "$vmware_db"

# This defines echo_success() and echo_failure() on RedHat
if [ -r "$vmdb_answer_INITSCRIPTSDIR"'/functions' ]; then
   . "$vmdb_answer_INITSCRIPTSDIR"'/functions'
fi

# This defines $rc_done and $rc_failed on S.u.S.E.
if [ -f /etc/rc.config ]; then
   # Don't include the entire file: there could be conflicts
   rc_done=`(. /etc/rc.config; echo "$rc_done")`
   rc_failed=`(. /etc/rc.config; echo "$rc_failed")`
else
   # Make sure the ESC byte is literal: Ash does not support echo -e
   rc_done='[71G done'
   rc_failed='[71Gfailed'
fi

bridge=vmnet-bridge
dhcpd=vmnet-dhcpd
netifup=vmnet-netifup
natd=vmnet-natd
netdetect=vmnet-detect
ping=vmware-ping
vnet=vmnet

allservices=bridge,hostonly,nat,dhcp,vmnet-detect

#
# Utilities
#

# BEGINNING_OF_IPV4_DOT_SH
#!/bin/sh

#
# IPv4 address functions
#
# Thanks to Owen DeLong <owen@delong.com> for pointing me at bash's arithmetic
# expansion ability, which is a lot faster than using 'expr' --hpreg
#

# Compute the subnet address associated to a couple IP/netmask
ipv4_subnet() {
  local ip="$1"
  local netmask="$2"

  # Split quad-dotted addresses into bytes
  # There is no double quote around the back-quoted expression on purpose
  # There is no double quote around $ip and $netmask on purpose
  set -- `IFS='.'; echo $ip $netmask`

  echo $(($1 & $5)).$(($2 & $6)).$(($3 & $7)).$(($4 & $8))
}

# Compute the broadcast address associated to a couple IP/netmask
ipv4_broadcast() {
  local ip="$1"
  local netmask="$2"

  # Split quad-dotted addresses into bytes
  # There is no double quote around the back-quoted expression on purpose
  # There is no double quote around $ip and $netmask on purpose
  set -- `IFS='.'; echo $ip $netmask`

  echo $(($1 | (255 - $5))).$(($2 | (255 - $6))).$(($3 | (255 - $7))).$(($4 | (255 - $8)))
}
# END_OF_IPV4_DOT_SH

# Is a given module loaded?
isLoaded() {
   local module="$1"

   /sbin/lsmod | awk 'BEGIN {n = "no";} {if ($1 == "'"$module"'") n = "yes";} END {print n;}'
}

# Check if there is an IP route for a given subnet via a given interface
# Return true if there is _NO_ such route
noRoutePresent() {
   local subnet="$1" # IN
   local intf="$2"   # IN

   # Beware, there may be several identical routes
   [ "`/sbin/route -n | grep '^'"$subnet"'.*'"$intf"'$'`" = '' ]
}

#
# Check that the IP address we are going to assign to the host machine on
# a private IP network does not already exist
#
# NB: If you don't want to do this test, just substitute
#     false for it.
#
lookForHostOnlyNetwork() {
   local ip="$1"

   "$vmdb_answer_BINDIR"/"$ping" -q "$ip"
}

# Create /dev/vmnetXX device
vmware_create_vmnet() {
   local vHubNr="$1" # IN
   local vDevice="/dev/vmnet$vHubNr"

   if [ ! -e "$vDevice" ]; then
      mknod -m 600 "$vDevice" c 119 "$vHubNr"
   fi
}

# Create a virtual host ethernet interface and connect it to a virtual
# ethernet hub
vmware_start_netifup() {
   local vHostIf="$1" # IN
   local vHubNr="$2"  # IN

   cd "$vmdb_answer_BINDIR" && "$vmdb_answer_BINDIR"/"$netifup" \
      -d /var/run/"$netifup"-"$vHostIf".pid /dev/vmnet"$vHubNr" "$vHostIf"
}

# Disconnect a virtual host ethernet interface from a virtual ethernet hub
# and destroy the virtual host ethernet interface
vmware_stop_netifup() {
   local vHostIf="$1" # IN

   if vmware_stop_pidfile /var/run/"$netifup"-"$vHostIf".pid; then
      rm -f /var/run/"$netifup"-"$vHostIf".pid
   fi
}

# Connect a physical host ethernet interface to a virtual ethernet hub
vmware_start_bridge() {
   local vHubNr="$1"  # IN
   local pHostIf="$2" # IN

   cd "$vmdb_answer_BINDIR" && "$vmdb_answer_BINDIR"/"$bridge" \
      -d /var/run/"$bridge"-"$vHubNr".pid /dev/vmnet"$vHubNr" "$pHostIf"
}

# Disconnect a physical host ethernet interface from a virtual ethernet hub
vmware_stop_bridge() {
   local vHubNr="$1"  # IN

   if vmware_stop_pidfile /var/run/"$bridge"-"$vHubNr".pid; then
      rm -f /var/run/"$bridge"-"$vHubNr".pid
   fi
}

# Start a DHCP server on a private IP network
vmware_start_dhcpd() {
   local vHostIf="$1" # IN

   # The daemon already logs its output in the system log, so we can safely
   # trash it
   cd "$vmdb_answer_BINDIR" && "$vmdb_answer_BINDIR"/"$dhcpd" \
      -cf "$vmware_etc_dir"/"$vHostIf"/dhcpd/dhcpd.conf \
      -lf "$vmware_etc_dir"/"$vHostIf"/dhcpd/dhcpd.leases \
      -pf /var/run/"$dhcpd"-"$vHostIf".pid "$vHostIf"
}

# Stop a DHCP server on a private IP network
vmware_stop_dhcpd() {
   local vHostIf="$1" # IN

   if vmware_stop_pidfile /var/run/"$dhcpd"-"$vHostIf".pid; then
      rm -f /var/run/"$dhcpd"-"$vHostIf".pid
   fi
}

# Start the host-only network user service
vmware_start_hostonly() {
   local vHubNr="$1"    # IN
   local vHostIf="$2"   # IN
   local ifIp="$3"      # IN
   local ifMask="$4"    # IN
   local ifNet

   #
   # Do a cursory check to see if the host-only network
   # configuration is still ok.  We do this so that mobile
   # hosts don't get setup at install time and then moved to
   # a new locale where the host-only network config is no
   # longer valid.
   #
   # NB: This really needs to be done at power-on time when
   #     VM is configured to use host-only networking so that
   #     we aren't fooled by dynamic changes in the network.
   #
   # XXX ping takes 10 seconds to timeout if nobody answers
   #     that slows boot too much so we do this bit in the
   #     background.
   #
   if lookForHostOnlyNetwork "$ifIp"; then
    echo 'Host-only networking disabled because '"$ifIp"
    echo 'appears to be a real, physical, existing address.'
    echo 'Please run "'"$vmdb_answer_BINDIR"'/vmware-config.pl" to'
    echo 'modify your host-only network configuration.'
    exit 1
   fi

   vmware_start_netifup "$vHostIf" "$vHubNr" || exit 1

   # Configure the virtual host ethernet interface and define the private IP
   # network
   #
   # . We provide the broadcast address explicitly because versions of ifconfig
   #   prior to 1.39 (1999-03-18) seem to miscompute it
   # . 2.0.x kernels don't install a route when the interface is marked up, but
   #   2.2.x kernel do. Since we want to see any errors from route we don't
   #   just discard messages from route, but instead check if the route got
   #   installed before manually adding one.
   ifNet=`ipv4_subnet "$ifIp" "$ifMask"`
   if ifconfig "$vHostIf" inet "$ifIp" netmask "$ifMask" \
         broadcast "`ipv4_broadcast "$ifIp" "$ifMask"`" up \
         && noRoutePresent "$ifNet" "$vHostIf"; then
      route add -net "$ifNet" netmask "$ifMask" "$vHostIf"
   fi

   exit 0
}

# Stop the host-only network user service
vmware_stop_hostonly() {
   local vHostIf="$1"   # IN
   local ifIp="$2"      # IN
   local ifMask="$3"    # IN
   local ifNet

   # Terminate the private network
   ifNet=`ipv4_subnet "$ifIp" "$ifMask"`
   noRoutePresent "$ifNet" "$vHostIf" || \
      route del -net "$ifNet" netmask "$ifMask" || exit 1
   # To test if the interface exists, we can not just look at the exitcode
   # because old versions of ifconfig don't exit with 1 when invoked with a
   # non-existing interface
   if [ "`ifconfig "$vHostIf" 2>/dev/null`" != '' ]; then
      ifconfig "$vHostIf" down || exit 1
   fi

   vmware_stop_netifup "$vHostIf" || exit 1

   exit 0
}

# Start the NAT network user service
vmware_start_nat() {
   local vHubNr="$1"    # IN

   cd "$vmdb_answer_BINDIR" && "$vmdb_answer_BINDIR"/"$natd" \
      -d /var/run/"$natd"-"$vHubNr".pid \
      -m /var/run/"$natd"-"$vHubNr".mac \
      -c "$vmware_etc_dir"/vmnet"$vHubNr"/nat/nat.conf
}

# Stop the NAT network user service
vmware_stop_nat() {
   local vHubNr="$1"   # IN

   if vmware_stop_pidfile /var/run/"$natd"-"$vHubNr".pid; then
      rm -f /var/run/"$natd"-"$vHubNr".pid
      rm -f /var/run/"$natd"-"$vHubNr".mac
   fi
}

# Start the vmnet-detect daemon
vmware_start_netdetect() {
   cd "$vmdb_answer_BINDIR" && "$vmdb_answer_BINDIR"/"$netdetect" \
      -d /var/run/"$netdetect".pid
}

# Stop the vmnet-detect daemon gracefully
# This sends a SIGUSR1 to the netdetect daemon, unlike vmware_stop_pidfile()
vmware_stop_netdetect() {
   local pid

   pid=`cat /var/run/"$netdetect".pid 2>/dev/null`
   if [ "$pid" = '' ]; then
      # The file probably does not exist or is empty. Success
      return 0
   fi
   # Keep only the first number we find
   set -- $pid
   pid="$1"

   # First try a graceful shutdown using SIGUSR1
   if ! vmware_synchrone_kill "$pid" "USR1"; then
      # Try the standard way of killing VMware daemons
      vmware_stop_pidfile /var/run/"$netdetect".pid
   fi
   rm -f /var/run/"$netdetect".pid
}

# Determine if needle is present in comma-delimited haystack
vmware_find_comma() {
   local haystack="$1" # IN
   local needle="$2"   # IN

   set -- `IFS=','; echo $haystack`
   while [ "$#" -gt 0 ]; do
      if [ "$1" = "$needle" ]; then
         return 0
      fi
      shift
   done
   return 1
}

# Start services on specified vnet
vmware_start_services() {
   local vHubNr="$1"  # IN
   local services="$2" # IN

   local exitcode='0'
   local interface
   local hostaddr
   local netmask
   local dhcp
   local nat

   eval 'interface="$vmdb_answer_VNET_'"$vHubNr"'_INTERFACE"'
   eval 'hostaddr="$vmdb_answer_VNET_'"$vHubNr"'_HOSTONLY_HOSTADDR"'
   eval 'netmask="$vmdb_answer_VNET_'"$vHubNr"'_HOSTONLY_NETMASK"'
   if [ -n "$interface" ]; then
      vmware_create_vmnet "$vHubNr"

      if vmware_find_comma "$services" 'bridge'; then
	 vmware_exec 'Bridged networking on /dev/vmnet'"$vHubNr" \
	    vmware_start_bridge "$vHubNr" "$interface"
	 exitcode=$(($exitcode + $?))
      fi
   elif [ -n "$hostaddr" -a -n "$netmask" ]; then
      vmware_create_vmnet "$vHubNr"

      if vmware_find_comma "$services" 'hostonly'; then
	 vmware_bg_exec 'Host-only networking on /dev/vmnet'"$vHubNr" \
	    vmware_start_hostonly "$vHubNr" 'vmnet'"$vHubNr" \
	    "$hostaddr" "$netmask"
	 exitcode=$(($exitcode + $?))
      fi
      
      eval 'dhcp="$vmdb_answer_VNET_'"$vHubNr"'_DHCP"'
      if [ "$dhcp" = 'yes' ] && vmware_find_comma "$services" 'dhcp'; then
	 vmware_exec 'DHCP server on /dev/vmnet'"$vHubNr" \
	    vmware_start_dhcpd 'vmnet'"$vHubNr"
	 exitcode=$(($exitcode + $?))
      fi
      
      eval 'nat="$vmdb_answer_VNET_'"$vHubNr"'_NAT"'
      if [ "$nat" = 'yes' ] && vmware_find_comma "$services" 'nat'; then
	 vmware_exec 'NAT service on /dev/vmnet'"$vHubNr" \
	    vmware_start_nat "$vHubNr"
	 exitcode=$(($exitcode + $?))
      fi
   fi
   if [ $vHubNr -eq 0 ] && vmware_find_comma "$services" 'vmnet-detect'; then
      vmware_exec 'Host network detection' vmware_start_netdetect
   fi
   
   return "$exitcode"
}

# Stop services on specified vnet
vmware_stop_services() {
   local vHubNr="$1"  # IN
   local services="$2" # IN

   local exitcode='0'
   local interface
   local hostaddr
   local netmask
   local dhcp
   local nat

   eval 'interface="$vmdb_answer_VNET_'"$vHubNr"'_INTERFACE"'
   eval 'hostaddr="$vmdb_answer_VNET_'"$vHubNr"'_HOSTONLY_HOSTADDR"'
   eval 'netmask="$vmdb_answer_VNET_'"$vHubNr"'_HOSTONLY_NETMASK"'
   if [ -n "$interface" ] && vmware_find_comma "$services" 'bridge'; then
      vmware_exec 'Bridged networking on /dev/vmnet'"$vHubNr" \
	 vmware_stop_bridge "$vHubNr"
      exitcode=$(($exitcode + $?))
   elif [ -n "$hostaddr" -a -n "$netmask" ]; then
      eval 'dhcp="$vmdb_answer_VNET_'"$vHubNr"'_DHCP"'
      if [ "$dhcp" = 'yes' ] && vmware_find_comma "$services" 'dhcp'; then
	 vmware_exec 'DHCP server on /dev/vmnet'"$vHubNr" \
	    vmware_stop_dhcpd 'vmnet'"$vHubNr"
	 exitcode=$(($exitcode + $?))
      fi
      
      eval 'nat="$vmdb_answer_VNET_'"$vHubNr"'_NAT"'
      if [ "$nat" = 'yes' ] && vmware_find_comma "$services" 'nat'; then
	 vmware_exec 'NAT service on /dev/vmnet'"$vHubNr" \
	    vmware_stop_nat "$vHubNr"
	 exitcode=$(($exitcode + $?))
      fi

      if vmware_find_comma "$services" 'hostonly'; then
	 vmware_exec 'Host-only networking on /dev/vmnet'"$vHubNr" \
	    vmware_stop_hostonly 'vmnet'"$vHubNr" "$hostaddr" "$netmask"
	 exitcode=$(($exitcode + $?))
      fi
   fi

   if [ $vHubNr -eq 0 ] && vmware_find_comma "$services" 'vmnet-detect'; then
      vmware_exec 'Host network detection' vmware_stop_netdetect
   fi
   
   return "$exitcode"
}

# Print status of services on specified vnet
vmware_status_services() {
   local vHubNr="$1"  # IN
   local services="$2" # IN

   local exitcode='0'
   local interface
   local hostaddr
   local netmask
   local dhcp
   local nat

   eval 'interface="$vmdb_answer_VNET_'"$vHubNr"'_INTERFACE"'
   eval 'hostaddr="$vmdb_answer_VNET_'"$vHubNr"'_HOSTONLY_HOSTADDR"'
   eval 'netmask="$vmdb_answer_VNET_'"$vHubNr"'_HOSTONLY_NETMASK"'
   if [ -n "$interface" ] && vmware_find_comma "$services" 'bridge'; then
      echo -n 'Bridged networking on /dev/vmnet'"$vHubNr"
      if vmware_check_pidfile '/var/run/'"$bridge"'-'"$vHubNr"'.pid'; then
	 echo ' is running'
      else 
	 echo ' is not running'
	 exitcode=$(($exitcode + 1))
      fi
   elif [ -n "$hostaddr" -a -n "$netmask" ]; then
      if vmware_find_comma "$services" 'hostonly'; then
	 echo -n 'Host-only networking on /dev/vmnet'"$vHubNr"
	 if vmware_check_pidfile '/var/run/'"$netifup"'-vmnet'"$vHubNr".'pid'; then
	    echo ' is running'
	 else 
	    echo ' is not running'
	    exitcode=$(($exitcode + 1))
	 fi
      fi
      
      eval 'dhcp="$vmdb_answer_VNET_'"$vHubNr"'_DHCP"'
      if [ "$dhcp" = 'yes' ] && vmware_find_comma "$services" 'dhcp'; then
	 echo -n 'DHCP server on /dev/vmnet'"$vHubNr"
	 if vmware_check_pidfile '/var/run/'"$dhcpd"'-vmnet'"$vHubNr"'.pid'; then
	    echo ' is running'
	 else 
	    echo ' is not running'
	    exitcode=$(($exitcode + 1))
	 fi
      fi
      
      eval 'nat="$vmdb_answer_VNET_'"$vHubNr"'_NAT"'
      if [ "$nat" = 'yes' ] && vmware_find_comma "$services" 'nat'; then
	 echo -n 'NAT networking on /dev/vmnet'"$vHubNr"
	 if vmware_check_pidfile '/var/run/'"$natd"'-'"$vHubNr"'.pid'; then
	    echo ' is running'
	 else 
	    echo ' is not running'
	    exitcode=$(($exitcode + 1))
	 fi
      fi
   fi
   if [ $vHubNr -eq 0 ] && vmware_find_comma "$services" 'vmnet-detect'; then
      echo -n 'Host network detection'
      if vmware_check_pidfile '/var/run/'"$netdetect"'.pid'; then
         echo ' is running'
      else
         echo ' is not running'
      fi
   fi

   return "$exitcode"
}

# Check arguments
if [ "$#" -gt 1 ]; then
   if [ "$2" != 'all' ] && ! [ "$2" -lt 256 ]; then
      echo 'Invalid hub number specified: "'"$2"'".'
      echo

      exit 1
   fi

   vHubNr=$2
else
   vHubNr='all'
fi

if [ "$#" -gt 2 ]; then
   services="$3"
   args="$@"
   set -- `IFS=','; echo $services`
   while [ "$#" -gt 0 ]; do
      if ! vmware_find_comma "$allservices" "$1"; then
	 echo 'Invalid service specified: "'"$1"'".'
	 echo
	 
	 exit 1
      fi
      shift
   done
   set -- $args
else
   services=$allservices
fi


# See how we were called.
case "$1" in
   start)
      if [ -e "$vmware_etc_dir"/not_configured ]; then
         echo "`vmware_product_name`"' is installed, but it has not been (correctly) configured'
         echo 'for the running kernel. To (re-)configure it, invoke the'
         echo 'following command: '"$vmdb_answer_BINDIR"'/vmware-config.pl.'
         echo

         exit 1
      fi

      exitcode='0'
      if [ "$vmdb_answer_NETWORKING" = 'yes' ]; then
         if [ "`isLoaded "$vnet"`" = 'no' ]; then
	    echo 'Module '"$vnet"' is not loaded.  Please verify that it is loaded before'
	    echo 'running this script.'
	    echo

            exit 1
         fi

	 if [ "$vHubNr" != 'all' ]; then
	    vmware_start_services "$vHubNr" "$services"
	    exitcode=$(($exitcode + $?))
	 else
	    vHubNr=0
	    while [ $vHubNr -lt 256 ]; do
	       vmware_start_services "$vHubNr" "$services"
	       exitcode=$(($exitcode + $?))
	       
	       vHubNr=$(($vHubNr + 1))
	    done
	 fi
      fi

      if [ "$exitcode" -gt 0 ]; then
         exit 1
      fi
   ;;

   stop)
      exitcode='0'
      if [ "$vmdb_answer_NETWORKING" = 'yes' ]; then
	 if [ "$vHubNr" != 'all' ]; then
	    vmware_stop_services "$vHubNr" "$services"
	    exitcode=$(($exitcode + $?))
	 else
	    vHubNr=0
	    while [ $vHubNr -lt 256 ]; do
	       vmware_stop_services "$vHubNr" "$services"
	       exitcode=$(($exitcode + $?))
	       
	       vHubNr=$(($vHubNr + 1))
	    done
	 fi
      fi

      if [ "$exitcode" -gt 0 ]; then
         exit 1
      fi
   ;;

   status)
      exitcode='0'
      if [ "$vmdb_answer_NETWORKING" = 'yes' ]; then
	 if [ "$vHubNr" != 'all' ]; then
	    vmware_status_services "$vHubNr" "$services"
	    exitcode=$(($exitcode + $?))
	 else
	    vHubNr=0
	    while [ $vHubNr -lt 256 ]; do
	       vmware_status_services "$vHubNr" "$services"
	       exitcode=$(($exitcode + $?))
	       
	       vHubNr=$(($vHubNr + 1))
	    done
	 fi
      fi
 
      if [ "$exitcode" -gt 0 ]; then
         exit 1
      fi
   ;;
   restart)
      shift
      "$0" stop "$@" && "$0" start "$@"
   ;;

   *)
      echo "Usage: `basename "$0"` {start|stop|status|restart}" \
	 "[{<vHubNr>|all} [<comma-separated subset of " \
         "{bridge,hostonly,dhcp,nat,vmnet-detect}>]]"
      exit 1
esac

exit 0
