/* **********************************************************
 * Copyright  2006-2007 VMware, Inc.  All rights reserved.
 * **********************************************************/

#ifndef _VMCI_CALL_DEFS_H_
#define _VMCI_CALL_DEFS_H_


#include "vm_basic_types.h"

/*
 * All structs here are an integral size of their largest member, ie. a struct 
 * with at least one 8-byte member will have a size that is an integral of 8.
 * A struct which has a largest member of size 4 will have a size that is an
 * integral of 4. This is because Windows CL enforces this rule. 32 bit gcc 
 * doesn't e.g. 32 bit gcc can misalign an 8 byte member if it is preceeded by
 * a 4 byte member. 
 */


/* Generic guestcall/hypercall header */
typedef struct VMCICallHeader {
   VMCI_Vector vector;// FIXME: VMCI_Vector is enum.  Enum's size is implementation dependent.
   uint32    size;  // Total size including this header as well as arguments.
} VMCICallHeader;


/* Useful call struct to access args. */
typedef struct VMCICall {
   VMCICallHeader h;
   uint8     args[1];
} VMCICall;


/* Size of guest/hypercalls excluding the arguments. */
#define VMCI_CALL_HEADERSIZE sizeof(VMCICallHeader)


/*
 * Struct used for making VMCI_VECTORS_* hypercalls. Struct size
 * is 16 bytes. All fields in struct are aligned to their natural
 * alignment.
 */
typedef struct VMCIVectorsHdr {
   VMCICallHeader hdr;
   uint32 numVectors;
   uint32 _padding;
} VMCIVectorsHdr;


/*
 * Convenience struct for negotiating vectors. Must match layout of
 * VMCIVectorsHdr minus the VMCICallHeader.
 */
typedef struct VMCIVectorsMsg {
   uint32 numVectors;
   uint32 _padding;
   VMCI_Vector vectors[1];
} VMCIVectorsMsg;


/* Maximum size needed for VMCI_VECTORS_* hypercalls. */
#define VMCI_VECTORS_CALL_MAX_SIZE sizeof(VMCIVectorsHdr) \
        + VMCI_VECTOR_MAX * sizeof(VMCI_Vector)


/* 
 * Struct used for making VMCI_DATAGRAM_CREATE_HND hypercalls. Struct size is 
 * 16 bytes. All fields in struct are aligned to their natural alignment.
 */
typedef struct VMCIDatagramCreateHdr {
   VMCICallHeader hdr;
   VMCIId resourceID;
   uint32 flags;
} VMCIDatagramCreateHdr;


/* 
 * Struct used for making VMCI_DATAGRAM_DESTROY_HND hypercalls. Struct size is 
 * 16 bytes. All fields in struct are aligned to their natural alignment.
 */
typedef struct VMCIDatagramDestroyHdr {
   VMCICallHeader hdr;
   VMCIHandle handle;
} VMCIDatagramDestroyHdr;


/* 
 * Struct used for making VMCI_DATAGRAM_SEND hypercalls. Struct size is 32 bytes.
 * All fields in struct are aligned to their natural alignment.
 */
typedef struct VMCIDatagramSendHdr {
   VMCICallHeader hdr;
   VMCIHandle dstHandle;
   VMCIHandle srcHandle;
   uint32   payloadSize;
   uint32   _padding;
   /* Payload placed after struct. */
} VMCIDatagramSendHdr;


/* 
 * Struct used for making VMCI_DATAGRAM_RECEIVE calls. Same as struct used for
 * datagram send calls.
 */
typedef VMCIDatagramSendHdr VMCIDatagramReceiveHdr;


/*
 * Convenient struct for working with datagrams. Must match layout of 
 * VMCIDatagramSendHdr minus the VMCICallHeader.
 */
typedef struct VMCIDatagramMsg {
   VMCIHandle dstHandle;
   VMCIHandle srcHandle;
   uint32 payloadSize;
   uint32   _padding;
   char payload[1];
} VMCIDatagramMsg;


/* 
 * Struct used for making VMCI_SHAREDMEM_CREATE hypercalls. Struct size is 24 bytes.
 * All fields in struct are aligned to their natural alignment.
 */
typedef struct VMCISharedMemCreateHdr {
   VMCICallHeader hdr;
   VMCIHandle handle;
   uint32   memSize;
   uint32   _padding;
   /* PPNs placed after struct. */
} VMCISharedMemCreateHdr;


/* 
 * Struct used for making VMCI_SHAREDMEM_ATTACH hypercalls. Same as struct used for
 * create calls.
 */
typedef VMCISharedMemCreateHdr VMCISharedMemAttachHdr;


/* 
 * Struct used for making VMCI_SHAREDMEM_DETACH hypercalls. Struct size is 16 bytes.
 * All fields in struct are aligned to their natural alignment.
 */
typedef struct VMCISharedMemDetachHdr {
   VMCICallHeader hdr;
   VMCIHandle handle;
} VMCISharedMemDetachHdr;


/* 
 * Struct used for making VMCI_SHAREDMEM_QUERY hypercalls. Same as struct used for
 * detach calls.
 */
typedef VMCISharedMemDetachHdr VMCISharedMemQueryHdr;


/* 
 * Struct used for making VMCI_RESOURCE_ADD_CLIENT_PRIVS hypercalls. Struct size 
 * is 32 bytes. All fields in struct are aligned to their natural alignment.
 */
typedef struct VMCIResourceAddClientPrivsHdr {
   VMCICallHeader hdr;
   VMCIHandle resourceHandle;
   VMCIHandle clientHandle;
   uint32 numAllowPrivs;
   uint32 numDenyPrivs;
   /*
    * AllowPrivs are placed first, followed by deny privs. Only one of which
    * can be empty.
    */
} VMCIResourceAddClientPrivsHdr;


/* 
 * Struct used for making VMCI_RESOURCE_REMOVE_CLIENT_PRIVS hypercalls. Struct 
 * size is 32 bytes. All fields in struct are aligned to their natural alignment.
 */
typedef struct VMCIResourceRemoveClientPrivsHdr {
   VMCICallHeader hdr;
   VMCIHandle resourceHandle;
   VMCIHandle clientHandle;
   uint32 numPrivs;
   uint32 _padding;
   /* Privs placed after struct. */
} VMCIResourceRemoveClientPrivsHdr;


/* 
 * Struct used for making VMCI_RESOURCE_REMOVE_ALL_CLIENT_PRIVS hypercalls. 
 * Struct size is 16 bytes. All fields in struct are aligned to their natural 
 * alignment.
 */
typedef struct VMCIResourceRemoveAllClientPrivsHdr {
   VMCICallHeader hdr;
   VMCIHandle resourceHandle;
   VMCIHandle clientHandle;
} VMCIResourceRemoveAllClientPrivsHdr;


/*
 * Struct used for making VMCI_TEST_CALL driver/hypercalls. Struct size is
 * 16 bytes. All fields in struct are aligned to their natural alignment.
 */
typedef struct VMCITestCallHdr {
   VMCICallHeader hdr;
   int command;
   uint32 _padding;
} VMCITestCallHdr;


/*
 * Struct used for making VMCI_READ_CALL driver calls / hypercalls. Struct
 * size is 16 bytes. All fields in struct are aligned to their natural
 * alignment.
 */
typedef struct VMCIReadCallInfo {
   uint64 bufferUVA; /* VA of buffer in userland. */
   uint64 size; /* Size of the buffer. */
} VMCIReadCallInfo;

#endif
