/* **********************************************************
 * Copyright  2005-2007 VMware, Inc.  All rights reserved.
 * **********************************************************/

#ifndef _VMCI_DEF_H_
#define _VMCI_DEF_H_


#include "vm_basic_types.h"

/* Register offsets. */
#define VMCI_STATUS_ADDR      0x00
#define VMCI_CONTROL_ADDR     0x04
#define VMCI_ICR_ADDR	      0x08
#define VMCI_IMR_ADDR         0x0c
#define VMCI_HYPERCALL_ADDR   0x10
#define VMCI_GUESTCALL_ADDR   0x14
#define VMCI_CAPS_ADDR        0x18

/* Max number of devices. */
#define VMCI_MAX_DEVICES 1

/* Status register bits. */
#define VMCI_STATUS_INT_ON     0x1

/* Control register bits. */
#define VMCI_CONTROL_RESET        0x1
#define VMCI_CONTROL_INT_ENABLE   0x2
#define VMCI_CONTROL_INT_DISABLE  0x4

/* Capabilities register bits. */
#define VMCI_CAPS_HYPERCALL    0x1
#define VMCI_CAPS_GUESTCALL    0x2

/* Interrupt Cause register bits. */
#define VMCI_ICR_GUESTCALL     0x1

/* Interrupt Mask register bits. */
#define VMCI_IMR_GUESTCALL     0x1

#define VMCI_IOSPACE_SIZE      0x20

/* 
 * Standard maximum size of a VMCI hypercall or guestcall.
 * Calls are allowed to be larger than this, but for most calls,
 * a buffer of this size should suffice.
 */
#define VMCI_CALL_STANDARD_SIZE 4096

#define VMCI_MAX_NUM_CALLS  100

/* VMCI reserved API vectors. */
typedef enum  {
   VMCI_VECTORS_INFORM                    = 0,
   VMCI_VECTORS_QUERY                     = 1,
   VMCI_GET_CONTEXT_ID,
   VMCI_RESOURCE_ADD_CLIENT_PRIVS,
   VMCI_RESOURCE_REMOVE_CLIENT_PRIVS,
   VMCI_RESOURCE_REMOVE_ALL_CLIENT_PRIVS,
   VMCI_SHAREDMEM_CREATE,
   VMCI_SHAREDMEM_ATTACH,
   VMCI_SHAREDMEM_DETACH,
   VMCI_SHAREDMEM_QUERY,
   VMCI_DATAGRAM_CREATE_HND,
   VMCI_DATAGRAM_DESTROY_HND,
   VMCI_DATAGRAM_SEND,
   VMCI_DATAGRAM_RECEIVE,
   VMCI_TEST_CALL,
   VMCI_VECTOR_MAX
} VMCI_Vector;

/* VMCI Ids. */
typedef uint32 VMCIId;

#define VMCI_INVALID_ID      -1
#define VMCI_HOST_CONTEXT_ID  0

/* 
 * The VMCI_CONTEXT_RESOURCE_ID is used together with VMCI_MAKE_HANDLE to make 
 * handles that refer to a specific context.
 */
#define VMCI_CONTEXT_RESOURCE_ID 0

/* VMCI error codes. */
#define VMCI_SUCCESS_LAST_DETACH          3
#define VMCI_SUCCESS_ACCESS_GRANTED       2
#define VMCI_SUCCESS_ENTRY_DEAD           1
#define VMCI_SUCCESS                      0
#define VMCI_ERROR_INVALID_VECTOR        -1
#define VMCI_ERROR_INVALID_ARGS          -2
#define VMCI_ERROR_NO_MEM                -3
#define VMCI_ERROR_HYPERCALL_FAILED      -4
#define VMCI_ERROR_MORE_DATA             -5
#define VMCI_ERROR_NO_MORE_CALLS         -6
#define VMCI_ERROR_NO_ACCESS             -7
#define VMCI_ERROR_NO_HANDLE             -8
#define VMCI_ERROR_DUPLICATE_ENTRY       -9
#define VMCI_ERROR_DST_UNREACHABLE       -10
#define VMCI_ERROR_PAYLOAD_TOO_LARGE     -11
#define VMCI_ERROR_INVALID_PRIV          -12
#define VMCI_ERROR_GENERIC               -13
#define VMCI_ERROR_PAGE_ALREADY_SHARED   -14
#define VMCI_ERROR_CANNOT_SHARE_PAGE     -15
#define VMCI_ERROR_CANNOT_UNSHARE_PAGE   -16
#define VMCI_ERROR_NO_PROCESS            -17
#define VMCI_ERROR_NO_GUESTCALL          -18
#define VMCI_ERROR_NO_RESOURCES          -19
#define VMCI_ERROR_UNAVAILABLE           -20
#define VMCI_ERROR_NOT_FOUND             -21
#define VMCI_ERROR_ALREADY_EXISTS        -22
#define VMCI_ERROR_NOT_PAGE_ALIGNED      -23
#define VMCI_ERROR_INVALID_SIZE          -24
#define VMCI_ERROR_REGION_ALREADY_SHARED -25
#define VMCI_ERROR_TIMEOUT               -26
#define VMCI_ERROR_HYPERCALL_INCOMPLETE  -27
 
/* Internal error codes. */
#define VMCI_SHAREDMEM_ERROR_BAD_CONTEXT -1000

/* Handle definitions. */
typedef uint64 VMCIHandle;

#define VMCI_INVALID_HANDLE CONST64U(-1)

#define VMCI_MAKE_HANDLE(_cid, _rid) (((uint64)(_cid) << 32) | (_rid))
#define VMCI_HANDLE_TO_CONTEXT_ID(_handle) ((VMCIId)((_handle) >> 32))
#define VMCI_HANDLE_TO_RESOURCE_ID(_handle) ((VMCIId)((_handle) & 0xffffffff))

#define VMCI_PATH_MAX 256

struct VMCIDatagramMsg;

typedef void
(*VMCIDatagramCB)(void *notifyData,             // IN: client data for handler
                  uint32 msgSize,               // IN: size of msg
                  struct VMCIDatagramMsg *msg); // IN: 

/* VMCI privileges. */
typedef enum {
   VMCI_PRIV_CH_PRIV,
   VMCI_PRIV_DESTROY_RESOURCE,
   VMCI_PRIV_ASSIGN_CLIENT,
   VMCI_PRIV_DG_CREATE,
   VMCI_PRIV_DG_SEND,
   VMCI_PRIV_SM_CREATE,
   VMCI_PRIV_SM_ATTACH,
   VMCI_NUM_PRIVILEGES,
} VMCIResourcePrivilegeType;


/* VMCI Discovery Service. */

/* Well-known handle to the discovery service. */
#define VMCI_DS_RESOURCE_ID 1 /* Reserved resource ID for discovery service. */
#define VMCI_DS_HANDLE VMCI_MAKE_HANDLE(VMCI_HOST_CONTEXT_ID, VMCI_DS_RESOURCE_ID)
#define VMCI_DS_CONTEXT VMCI_MAKE_HANDLE(VMCI_HOST_CONTEXT_ID, VMCI_CONTEXT_RESOURCE_ID)

/* Maximum length of a DS message. */
#define VMCI_DS_MAX_MSG_SIZE        300

/* Command actions. */
#define VMCI_DS_ACTION_LOOKUP         0
#define VMCI_DS_ACTION_REGISTER       1
#define VMCI_DS_ACTION_UNREGISTER     2

/* Defines wire-protocol format for a request send to the DS from a context. */
typedef struct VMCIDsRequestHeader {
   int32   action;
   int32   msgid;
   int64   handle;
   int32   nameLen;
   int8    name[1];   
} VMCIDsRequestHeader;


/* Defines the wire-protocol format for a request send from the DS to a context. */
typedef struct VMCIDsReplyHeader {
   int32 msgid;
   int32 code;
   int64 handle;
   int32 msgLen;
   int8  msg[1];
} VMCIDsReplyHeader;

/* VMCI_TEST_CALL commands */
#define VMCI_TEST_INIT      0
#define VMCI_TEST_CLEANUP   1
#define VMCI_TEST_GUESTCALL 2

#define VMCI_PUBLIC_GROUP_NAME "vmci public group"
/* 0 through VMCI_RESERVED_RESOURCE_ID_MAX are reserved. */
#define VMCI_RESERVED_RESOURCE_ID_MAX 1023

#endif
