/* **********************************************************
 * Copyright  2006-2007 VMware, Inc.  All rights reserved.
 * **********************************************************/

/*
 * vmcilib.h --
 *
 *	Virtual Machine Communication Interface (VMCI) library interface.
 */

#ifndef _VMCI_LIB_H_
#define _VMCI_LIB_H_

/* VMCI library version number. */
#define VMCI_LIB_VERSION    1

#ifdef _WIN32
#include <windows.h>
#endif // _WIN32

#include "vm_basic_types.h"
#include "vmci_defs.h"
#include "vmci_call_defs.h"

#ifdef _WIN32
typedef HANDLE VMCIFileHandle;
#  define VMCI_INVALID_FILE_HANDLE INVALID_HANDLE_VALUE
#  define VMCI_INVALID_WAIT_HANDLE NULL
#else
typedef int VMCIFileHandle;
#  define VMCI_INVALID_FILE_HANDLE -1
#  define VMCI_INVALID_WAIT_HANDLE -1
#endif

/* Can poll on these many handles at once. */
#ifdef _WIN32
#  define VMCIPOLL_MAX_HANDLES MAXIMUM_WAIT_OBJECTS
#elif defined(__linux__) || defined(__APPLE__)
#  define VMCIPOLL_MAX_HANDLES 1024
#else
#  error "Unsupported platform"
#endif

/* VMCI datagram handle flag bits */
#define VMCIDATAGRAM_NONBLOCK 0x00000001
#define VMCIDATAGRAM_FLAG_BITS (VMCIDATAGRAM_NONBLOCK) /* One bit flag for now. */ 

/* VMCI poll event filters. */
#define VMCIPOLL_READ  0x0001

typedef struct VMCIPollHandle {
   VMCIHandle       handle;  /* Datagram handle. */
   unsigned short events;  /* Events of interest. */
   unsigned short revents; /* Ready events. */
} VMCIPollHandle;

int VMCI_InitApp(const char *devName);
void VMCI_CleanupApp(void);

int VMCI_Poll(VMCIPollHandle *pollHandles,
              unsigned int nHandles,
              int timeout);
VMCIId VMCI_GetContextID(void);

/* Testing. */
int VMCI_TestDriver(int command);

/* Resource access control API. */
int VMCIResource_AddClientPrivileges(VMCIHandle resourceHandle,
                                     VMCIHandle clientHandle,
                                     int numAllowPrivs,
                                     VMCIResourcePrivilegeType *allowPrivs,
                                     int numDenyPrivs,
                                     VMCIResourcePrivilegeType *denyPrivs);
int VMCIResource_RemoveClientPrivileges(VMCIHandle resourceHandle,
                                        VMCIHandle clientHandle,
                                        int numPrivs,
                                        VMCIResourcePrivilegeType *privs);
int VMCIResource_RemoveAllClientPrivileges(VMCIHandle resourceHandle,
                                           VMCIHandle clientHandle);


/* Datagram API. */
VMCIHandle VMCIDatagram_CreateHnd(uint32 flags);
void VMCIDatagram_DestroyHnd(VMCIHandle handle);
int VMCIDatagram_SendTo(VMCIHandle dstHandle, VMCIHandle srcHandle,
                        uint32 payloadSize, void *payload);
int VMCIDatagram_RecvFrom(VMCIHandle handle, VMCIHandle *srcHandle,
                          void *buf, size_t count);
int VMCIDatagram_GetFlags(VMCIHandle handle, int *outFlags);
int VMCIDatagram_SetFlags(VMCIHandle handle, const int flags);
VMCIFileHandle VMCIDatagram_GetWaitableHandle(VMCIHandle datagramHandle);
VMCIHandle VMCIDatagram_FindHandle(VMCIFileHandle waitableHandle);


/* Shared memory API. */
VMCIHandle VMCISharedMem_Create(uint32 size, void **addr);
int VMCISharedMem_Attach(VMCIHandle handle, void **addr, uint32 *size);
int VMCISharedMem_Detach(VMCIHandle handle);
int VMCISharedMem_Query(VMCIHandle handle, uint32 *size);

/* Discovery Service Library Interface */
int  VMCIDs_Register(const char *name, VMCIHandle handle);
int  VMCIDs_Unregister(const char *name);
int  VMCIDs_Lookup(const char *name, VMCIHandle *handle);

#endif // _VMCI_LIB_H_

