/* **********************************************************
 * Copyright 2006 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/

/*
 * vmciResource.c --
 *
 *     Implementation of the VMCI Resource Access Control API.
 */

#ifdef linux
#   include "driver-config.h"
#   include <linux/string.h> /* memset() in the kernel */
#elif defined(WINNT_DDK)
#   include <string.h>
#elif !defined(__APPLE__)
#   error "Unknown platform"
#endif

#include "vmci_defs.h"
#include "vmciContext.h"
#include "vmci_infrastructure.h"
#include "vmci_handle_array.h"
#include "vmciResource.h"
#include "vmciGroup.h"
#include "vmciDriver.h"
#include "vm_atomic.h"
#include "vmciCommonInt.h"

#define LGPFX "VMCIResource: "

/* 0 through VMCI_RESERVED_RESOURCE_ID_MAX are reserved. */
static Atomic_uint32 resourceID = { VMCI_RESERVED_RESOURCE_ID_MAX + 1 };

static int ResourceAddClient(VMCIResource *resource, 
                             VMCIHandle clientHandle,
                             int numAllowPrivs,
                             VMCIResourcePrivilegeType *allowPrivs,
                             int numDenyPrivs,
                             VMCIResourcePrivilegeType *denyPrivs);
static void ResourceRemoveClient(VMCIResource *resource,
                                 VMCIResourceClient *client);
static int ResourceAddClientPrivileges(VMCIId contextID, 
                                       void* clientData,
                                       VMCICall *call);
static int ResourceRemoveClientPrivileges(VMCIId contextID, 
                                          void* clientData,
                                          VMCICall *call);
static int ResourceRemoveAllClientPrivileges(VMCIId contextID, 
                                             void* clientData,
                                             VMCICall *call);
static int64 ResourceDriverHandler(void *vmciObj,
                                   VMCIObjType vmciObjType,
                                   VMCICall *call);
static Bool ResourceCanSeeClient(VMCIHandle callerHnd, VMCIHandle clientHnd);

static VMCIHashTable *resourceTable = NULL;

/* Helper functions. */

/*
 *-----------------------------------------------------------------------------------
 *
 * ResourceValidatePrivileges --
 *
 *      Checks given privileges are valid for the given resource.
 *
 * Results:
 *      Returns VMCI_SUCCESS on success, appropriate error code otherwise.
 *
 * Side effects:
 *      None.
 *
 *-----------------------------------------------------------------------------------
 */

static INLINE int
ResourceValidatePrivileges(VMCIResource *resource, 
                           int numPrivs,
                           VMCIResourcePrivilegeType *privs)
{
   int i;

   for (i = 0; i < numPrivs; i++) {
      if (resource->validPrivs[privs[i]] != VMCI_PRIV_VALID) {
	 return VMCI_ERROR_INVALID_PRIV;
      }
   }
   return VMCI_SUCCESS;
}


/*
 *-----------------------------------------------------------------------------------
 *
 * ResourceGetClient --
 *
 *      Traverses resource's client list and returns client struct if found.
 *      Assumes resource->clientsLock is held.
 *
 * Results:
 *      Returns VMCI_SUCCESS on success, appropriate error code otherwise.
 *
 * Side effects:
 *      None.
 *
 *-----------------------------------------------------------------------------------
 */

static INLINE VMCIResourceClient *
ResourceGetClient(VMCIResource *resource, 
                  VMCIHandle clientHandle)
{
   VMCIResourceClient *client = resource->clients;
   while (client && client->handle != clientHandle) {
      client = client->next;
   }
   if (client != NULL) {
      client->refCount++;
   }

   return client;
}


/*
 *-----------------------------------------------------------------------------------
 *
 * ResourceReleaseClient --
 *
 *      Releases a client and checks if it should be freed. 
 *      Assumes resource->clientsLock is held.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *-----------------------------------------------------------------------------------
 */

static INLINE void
ResourceReleaseClient(VMCIResource *resource,
                      VMCIResourceClient *client)
{
   ASSERT(client && client->refCount > 0);

   client->refCount--;
   if (client->refCount == 0) {
      VMCI_FreeKernelMem(client);
   }
}


/*
 *-----------------------------------------------------------------------------------
 *
 * ResourceAddClient --
 *
 *      Creates a new client for resource, set given privileges at the same time.
 *      Assumes resource->clientsLock is held.
 *
 * Results:
 *      Returns VMCI_SUCCESS on success, appropriate error code otherwise.
 *
 * Side effects:
 *      None.
 *
 *-----------------------------------------------------------------------------------
 */

static int
ResourceAddClient(VMCIResource *resource, 
                  VMCIHandle clientHandle,
                  int numAllowPrivs,
                  VMCIResourcePrivilegeType *allowPrivs,
                  int numDenyPrivs,
                  VMCIResourcePrivilegeType *denyPrivs)
{
   int i;
   VMCIResourceClient *client;
   
   if (clientHandle == VMCI_INVALID_HANDLE) {
      return VMCI_ERROR_INVALID_ARGS;
   }

   client = VMCI_AllocKernelMem(sizeof *client, 
                                VMCI_MEMORY_NONPAGED | VMCI_MEMORY_ATOMIC);
   if (client == NULL) {
      Log(LGPFX"Failed to create new client for resource %p.\n",
	 resource);
      return VMCI_ERROR_NO_MEM;
   }
   client->handle = clientHandle;
   client->refCount = 1;

   /* Initialize all privs to VMCI_PRIV_NOT_SET. */
   for (i = 0; i < VMCI_NUM_PRIVILEGES; i++) {
      client->privilege[i] = VMCI_PRIV_NOT_SET;
   }

   /* Set allow privileges. */
   for (i = 0; i < numAllowPrivs; i++) {
      client->privilege[allowPrivs[i]] = VMCI_PRIV_ALLOW;
   }

   /* Set deny privileges, any overlap results in privilege being denied. */
   for (i = 0; i < numDenyPrivs; i++) {
      client->privilege[denyPrivs[i]] = VMCI_PRIV_DENY;
   }

#ifdef VMX86_DEBUG
   {
      VMCIResourceClient *cur = resource->clients;
      while (cur && cur->handle != clientHandle) {
	 cur = cur->next;
      }
      ASSERT(cur == NULL);
   }
#endif // VMX86_DEBUG
   client->next = resource->clients;
   resource->clients = client;

   return VMCI_SUCCESS;
}


/*
 *-----------------------------------------------------------------------------------
 *
 * ResourceRemoveClient --
 *
 *      Removes a client from the resource's client list and decrements the reference
 *      count. Assumes resource->clientsLock is held.
 *
 * Results:
 *      Returns VMCI_SUCCESS on success, appropriate error code otherwise.
 *
 * Side effects:
 *      None.
 *
 *-----------------------------------------------------------------------------------
 */

static void
ResourceRemoveClient(VMCIResource *resource,
                     VMCIResourceClient *client)
{
   VMCIResourceClient *prev, *cur;

   ASSERT(resource && client && client->refCount > 0);
   prev = NULL;
   cur = resource->clients;
   while (cur && cur->handle != client->handle) {
      prev = cur;
      cur = cur->next;
   }
   ASSERT(cur && cur == client);
   if (prev != NULL) {
      prev->next = cur->next;
   } else {
      resource->clients = cur->next;
   }

   ResourceReleaseClient(resource, client);
}


/*
 *------------------------------------------------------------------------------
 *
 *  ResourceCanSeeClient --
 *
 *     Helper function to check if a caller context can see a client context.
 *
 *  Result:
 *     VMCI_SUCCESS, if succesful, error code if not.
 *
 *  Side effects:
 *     None.
 *     
 *------------------------------------------------------------------------------
 */

static Bool
ResourceCanSeeClient(VMCIHandle callerHnd,  // IN:
                     VMCIHandle clientHnd)  // IN:
{
   Bool canSeeClient;
   
   /*
    * In order to add or remove a client, the caller context must be able to see
    * the client.
    * Visibility is defined by at least one of the following properties being 
    * true:
    *   1. Caller == client,
    *   2. Caller context has VMCI_PRIV_ASSIGN_CLIENT privilege to the client, or
    *   3. Caller context and client context are both members of a group, to 
    *      which the caller context has VMCI_PRIV_ASSIGN_CLIENT privilege.
    */

   canSeeClient = callerHnd == clientHnd || 
                  VMCIResource_CheckClientPrivilege(clientHnd, callerHnd, 
                                                    VMCI_PRIV_ASSIGN_CLIENT) == 
                  VMCI_SUCCESS_ACCESS_GRANTED;

   /* 
    * If the caller context is not the same as the client or we don't have the 
    * VMCI_PRIV_ASSIGN_CLIENT privilege, check if the client and the caller 
    * are both members of the same group where the caller context has the
    * VMCI_PRIV_ASSIGN_CLIENT privilege. This is only done if the client is
    * a context, since groups don't nest.
    */

   if (VMCI_HANDLE_TO_RESOURCE_ID(clientHnd) == VMCI_CONTEXT_RESOURCE_ID) {
      int i = 0;
      VMCIContext *callerContext = 
         VMCIContext_Get(VMCI_HANDLE_TO_CONTEXT_ID(callerHnd));
      if (callerContext == NULL) {
	 Log(LGPFX"Could get context for handle 0x%"FMT64"x.\n", callerHnd);
	 return FALSE;
      }

      while (!canSeeClient && 
	     i < VMCIHandleArray_GetSize(callerContext->groupArray)) {
	 VMCIHandle groupHandle =
	    VMCIHandleArray_GetEntry(callerContext->groupArray, i);
	 canSeeClient = VMCIGroup_IsMember(groupHandle, clientHnd) &&
	                VMCIResource_CheckClientPrivilege(groupHandle, callerHnd,
                                                     VMCI_PRIV_ASSIGN_CLIENT) ==
                   VMCI_SUCCESS_ACCESS_GRANTED;
	 i++;
      }
   }

   return canSeeClient;
}


/* Handlers for Guest Resource Access Control API calls. */

/*
 *------------------------------------------------------------------------------
 *
 *  ResourceAddClientPrivileges --
 *
 *     Hypercall handler to add a client to or update an existing clients
 *     privileges to a resource.
 *
 *  Result:
 *     VMCI_SUCCESS, if succesful, error code if not.
 *
 *  Side effects:
 *     None.
 *     
 *------------------------------------------------------------------------------
 */

static int
ResourceAddClientPrivileges(VMCIId contextID,  // IN:
                            void* clientData,       // IN: 
                            VMCICall *call)    // IN
{
   int result;
   VMCIResourcePrivilegeType *allowPrivs = NULL, *denyPrivs = NULL;
   VMCIResourceAddClientPrivsHdr *addCall = (VMCIResourceAddClientPrivsHdr *)call;
   VMCIHandle callerContextHnd = 
      VMCI_MAKE_HANDLE(contextID, VMCI_CONTEXT_RESOURCE_ID);
   
   if (addCall == NULL || 
       addCall->numAllowPrivs + addCall->numDenyPrivs < 1 ||
       (sizeof *addCall + (addCall->numAllowPrivs + addCall->numDenyPrivs) * 
	sizeof(VMCIResourcePrivilegeType)) != addCall->hdr.size) {
      Log(LGPFX "AddClientPrivsHandler: Invalid args.\n");
      return VMCI_ERROR_INVALID_ARGS;
   }
   
   if (addCall->numAllowPrivs > 0) {
      allowPrivs = 
	 (VMCIResourcePrivilegeType *)((char *)addCall + sizeof *addCall);
   }
   if (addCall->numDenyPrivs > 0) {
      int offset = sizeof *addCall + 
	 addCall->numAllowPrivs * sizeof(VMCIResourcePrivilegeType);
      denyPrivs = (VMCIResourcePrivilegeType *)((char *)addCall + offset);
   }

   /* 
    * Check if callerContext has the privilege to add clients/change privileges
    * to the resource.
    */
   result = VMCIResource_CheckClientPrivilege(addCall->resourceHandle,
                                              callerContextHnd,
                                              VMCI_PRIV_CH_PRIV);
   if (result != VMCI_SUCCESS_ACCESS_GRANTED) {
      return VMCI_ERROR_NO_ACCESS;
   }

   if (!ResourceCanSeeClient(callerContextHnd, addCall->clientHandle)) {
      return VMCI_ERROR_NO_ACCESS;
   }

   return VMCIResource_AddClientPrivileges(addCall->resourceHandle,
                                           addCall->clientHandle,
                                           addCall->numAllowPrivs, allowPrivs,
                                           addCall->numDenyPrivs, denyPrivs);
}


/*
 *------------------------------------------------------------------------------
 *
 *  ResourceRemoveClientPrivileges --
 *
 *     Hypercall handler to remove privileges from a client. If the last 
 *     privilege is removed the client will be removed as well.
 *
 *  Result:
 *     VMCI_SUCCESS, if successful, error code if not.
 *
 *  Side effects:
 *     None.
 *     
 *------------------------------------------------------------------------------
 */

static int
ResourceRemoveClientPrivileges(VMCIId contextID,  // IN:
                               void* clientData,       // IN: 
                               VMCICall *call)    // IN:
{
   int result;
   VMCIResourcePrivilegeType *privs;
   VMCIResourceRemoveClientPrivsHdr *removeCall =
      (VMCIResourceRemoveClientPrivsHdr *)call;
   VMCIHandle callerContextHnd = 
      VMCI_MAKE_HANDLE(contextID, VMCI_CONTEXT_RESOURCE_ID);

   if (removeCall == NULL || removeCall->numPrivs < 1 || 
       (sizeof *removeCall + removeCall->numPrivs * 
	sizeof(VMCIResourcePrivilegeType)) != removeCall->hdr.size) {
      return VMCI_ERROR_INVALID_ARGS;
   }
   
   result =
      VMCIResource_CheckClientPrivilege(removeCall->resourceHandle, 
                                        callerContextHnd, VMCI_PRIV_CH_PRIV);
   if (result != VMCI_SUCCESS_ACCESS_GRANTED) {
      return VMCI_ERROR_NO_ACCESS;
   }

   if (!ResourceCanSeeClient(callerContextHnd, removeCall->clientHandle)) {
      return VMCI_ERROR_NO_ACCESS;
   }

   privs = (VMCIResourcePrivilegeType *)((char *)removeCall + sizeof *removeCall);
   return VMCIResource_RemoveClientPrivileges(removeCall->resourceHandle, 
                                              removeCall->clientHandle,
                                              removeCall->numPrivs, privs);
}


/*
 *------------------------------------------------------------------------------
 *
 *  ResourceRemoveAllClientPrivileges --
 *
 *     Hypercall handler to remove all privileges from a client.
 *
 *  Result:
 *     VMCI_SUCCESS if successful, error code if not.
 *
 *  Side effects:
 *     None.
 *     
 *     
 *------------------------------------------------------------------------------
 */

static int
ResourceRemoveAllClientPrivileges(VMCIId contextID,  // IN:
                                  void* clientData,       // IN: 
                                  VMCICall *call)    // IN:
{
   int result;
   VMCIHandle callerContextHnd = VMCI_MAKE_HANDLE(contextID,
					      VMCI_CONTEXT_RESOURCE_ID);
   VMCIResourceRemoveAllClientPrivsHdr *removeAllCall = 
      (VMCIResourceRemoveAllClientPrivsHdr *)call;

   if (removeAllCall == NULL || 
       removeAllCall->hdr.size != sizeof *removeAllCall) {
      return VMCI_ERROR_INVALID_ARGS;
   }

   result =
      VMCIResource_CheckClientPrivilege(removeAllCall->resourceHandle, 
                                        callerContextHnd, VMCI_PRIV_CH_PRIV);
   if (result != VMCI_SUCCESS_ACCESS_GRANTED) {
      return VMCI_ERROR_NO_ACCESS;
   }

   if (!ResourceCanSeeClient(callerContextHnd, removeAllCall->clientHandle)) {
      return VMCI_ERROR_NO_ACCESS;
   }

   return VMCIResource_RemoveAllClientPrivileges(removeAllCall->resourceHandle, 
					       removeAllCall->clientHandle);
}


/* Handlers for Driver Resource Access Control API calls. */

/*
 *------------------------------------------------------------------------------
 *
 *  ResourceDriverHandler --
 *
 *     Hypercall handler to add a client to or update an existing clients
 *     privileges to a resource.
 *
 *  Result:
 *     VMCI_SUCCESS, if succesful, error code if not.
 *
 *  Side effects:
 *     None.
 *     
 *------------------------------------------------------------------------------
 */

static int64
ResourceDriverHandler(void *vmciObj,           // IN:
                      VMCIObjType vmciObjType, // IN:
                      VMCICall *call)          // IN:
{
   /* 
    * XXX Currently we call the guest call handlers to parse the parameters. 
    * This means that the process calling is under the same restrictions as 
    * guest which might not be what we want. We must determine what restrictions
    * we want to enforce regarding host processes changing privileges. 
    */

   return VMCI_Dispatch(VMCI_HOST_CONTEXT_ID, call);
}


/* Public Resource Access Control API. */

/*
 *-----------------------------------------------------------------------------------
 *
 * VMCIResource_Init --
 *
 *      Initializes the VMCI Resource Access Control API. Creates a hashtable
 *      to hold all resources, and registers vectors and callbacks for hypercalls.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *-----------------------------------------------------------------------------------
 */

int
VMCIResource_Init(void)
{
   resourceTable = VMCIHashTable_Create(128);
   if (resourceTable == NULL) {
      Log(LGPFX"Failed creating a resource hash table for VMCI.\n");
      return VMCI_ERROR_NO_MEM;
   }

   /* Register function for resource access vectors. */
   VMCI_RegisterHandler(VMCI_RESOURCE_ADD_CLIENT_PRIVS,
                        ResourceAddClientPrivileges, 
                        VMCI_IMMEDIATE_DISPATCH, NULL);
   VMCI_RegisterHandler(VMCI_RESOURCE_REMOVE_CLIENT_PRIVS,
                        ResourceRemoveClientPrivileges, 
                        VMCI_IMMEDIATE_DISPATCH, NULL);
   VMCI_RegisterHandler(VMCI_RESOURCE_REMOVE_ALL_CLIENT_PRIVS,
                        ResourceRemoveAllClientPrivileges, 
                        VMCI_IMMEDIATE_DISPATCH, NULL);

   /* Register driver call handlers. */
   VMCI_RegisterDriverHandler(VMCI_RESOURCE_ADD_CLIENT_PRIVS,
                              ResourceDriverHandler);
   VMCI_RegisterDriverHandler(VMCI_RESOURCE_REMOVE_CLIENT_PRIVS,
                              ResourceDriverHandler);
   VMCI_RegisterDriverHandler(VMCI_RESOURCE_REMOVE_ALL_CLIENT_PRIVS,
                              ResourceDriverHandler);

   return VMCI_SUCCESS;
}


/*
 *-----------------------------------------------------------------------------------
 *
 * VMCIResource_Exit --
 *
 *      Cleans up resources.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *-----------------------------------------------------------------------------------
 */

void
VMCIResource_Exit(void)
{
   /* Cleanup resources.*/
  if (resourceTable) {
    VMCIHashTable_Destroy(resourceTable);
  }
}


/*
 *-------------------------------------------------------------------------
 *
 *  VMCIResource_GetID --
 *
 *     Return resource ID. The VMCI_CONTEXT_RESOURCE_ID is reserved so we we
 *     start from its value + 1. XXX should keep account to know when id is 
 *     free to use again.
 *
 *  Result:
 *     VMCI resource id.
 *
 *  Side effects:
 *     None.
 *     
 *     
 *-------------------------------------------------------------------------
 */

VMCIId
VMCIResource_GetID(void)
{
   VMCIId cid = Atomic_FetchAndInc(&resourceID);
   if (0 == cid) {
      /* Counter overflow -- FIXME */
      Warning("VMCIResource_GetID() counter overflow.\n");
      PANIC();
   }
   return cid;
}


/*
 *-----------------------------------------------------------------------------------
 *
 * VMCIResource_Add --
 *
 * Results:
 *      VMCI_SUCCESS if successful, error code if not.
 *
 * Side effects:
 *      None.
 *
 *-----------------------------------------------------------------------------------
 */

int
VMCIResource_Add(VMCIResource *resource,              // IN
                 VMCIResourceType resourceType,       // IN
                 VMCIHandle resourceHandle,             // IN
                 VMCIHandle ownerHandle,                // IN
                 int numValidPrivs,                   // IN
                 VMCIResourcePrivilegeType *validPrivs, // IN
                 VMCIResourceFreeCB containerFreeCB,  // IN
                 void *containerObject)               // IN
{
   int result, i;
   VMCIResourcePrivilegeType ownerPrivs[2] = {VMCI_PRIV_CH_PRIV, 
					    VMCI_PRIV_DESTROY_RESOURCE};
   ASSERT(resource);

   if (resourceHandle == VMCI_INVALID_HANDLE || 
       ownerHandle == VMCI_INVALID_HANDLE ||
       numValidPrivs < 1) {
      Log(LGPFX"Invalid arguments resource 0x%"FMT64"x, owner 0x%"FMT64"x, "
	  "num valid privs %d.\n", resourceHandle, ownerHandle, numValidPrivs);
      return VMCI_ERROR_INVALID_ARGS;
   }

   VMCIHashTable_InitEntry(&resource->hashEntry, resourceHandle);
   resource->type = resourceType;
   resource->containerFreeCB = containerFreeCB;
   resource->containerObject = containerObject;
   resource->handle = resourceHandle;
   resource->registrationCount = 0;
   
   for (i = 0; i < VMCI_NUM_PRIVILEGES; i++) {
      resource->validPrivs[i] = VMCI_PRIV_NOT_SET;
   }

   /* Owner privs are always valid. */
   resource->validPrivs[VMCI_PRIV_CH_PRIV] = VMCI_PRIV_VALID;
   resource->validPrivs[VMCI_PRIV_DESTROY_RESOURCE] = VMCI_PRIV_VALID;

   /* Specify what privs aside from owner privs can be set. */
   for (i = 0; i < numValidPrivs; i++) {
      resource->validPrivs[validPrivs[i]] = VMCI_PRIV_VALID;
   }

   VMCI_InitLock(&resource->clientsLock);   
   resource->clients = NULL;

   /* Add owner as client with the ownerPrivs privileges. */
   result = ResourceAddClient(resource, ownerHandle, 2, ownerPrivs, 0, NULL);
   if (result != VMCI_SUCCESS) {
      Log(LGPFX"Failed to create owner client.\n");
      return result;
   }

   /* Add resource to hashtable. */
   result = VMCIHashTable_AddEntry(resourceTable, &resource->hashEntry);
   if (result != VMCI_SUCCESS) {
      Log(LGPFX"Failed to add entry to hash table.\n");
      return result;
   }

   return result;
}


/*
 *-----------------------------------------------------------------------------------
 *
 * VMCIResource_Remove --
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *-----------------------------------------------------------------------------------
 */

void
VMCIResource_Remove(VMCIHandle resourceHandle, VMCIResourceType resourceType)
{
   VMCILockFlags flags;
   VMCIResource *resource = VMCIResource_Get(resourceHandle, resourceType);

   if (resource == NULL) {
      return;
   }
   
   /*
    * Remove all clients from resource, this will cause others to fail accessing
    * the resource.
    */
   VMCI_GrabLock(&resource->clientsLock, &flags);
   while (resource->clients) {
      ResourceRemoveClient(resource, resource->clients);
   }
   VMCI_ReleaseLock(&resource->clientsLock, flags);

   /* Remove resource from hashtable. */
   VMCIHashTable_RemoveEntry(resourceTable, &resource->hashEntry);
   
   VMCIResource_Release(resource);
   /* resource could be freed by now. */
}


/*
 *-----------------------------------------------------------------------------------
 *
 * VMCIResource_Get --
 *
 * Results:
 *      Resource is successful. Otherwise NULL.
 *
 * Side effects:
 *      None.
 *
 *-----------------------------------------------------------------------------------
 */

VMCIResource *
VMCIResource_Get(VMCIHandle resourceHandle, VMCIResourceType resourceType)
{
   VMCIResource *resource;
   VMCIHashEntry *entry = VMCIHashTable_GetEntry(resourceTable, resourceHandle);
   if (entry == NULL) {
      return NULL;
   }
   resource = RESOURCE_CONTAINER(entry, VMCIResource, hashEntry);
   if ((resourceType == VMCI_RESOURCE_TYPE_ANY) || (resource->type == resourceType)) {
      return resource;
   }
   return NULL;
}


/*
 *-----------------------------------------------------------------------------------
 *
 * VMCIResource_Release --
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      resource's containerFreeCB will get called if last reference.
 *
 *-----------------------------------------------------------------------------------
 */

int
VMCIResource_Release(VMCIResource *resource)
{
   VMCILockFlags flags;
   int result;

   ASSERT(resource);

   result = VMCIHashTable_ReleaseEntry(resourceTable, &resource->hashEntry);
   if (result == VMCI_SUCCESS_ENTRY_DEAD) {
      /*
       * Nobody but us are referencing this resource. Remove any remaining
       * clients, and invoke the resource's free-callback.
       */
      VMCI_GrabLock(&resource->clientsLock, &flags);
      while (resource->clients) {
         ResourceRemoveClient(resource, resource->clients);
      }
      VMCI_ReleaseLock(&resource->clientsLock, flags);

      if (resource->containerFreeCB) {
	 resource->containerFreeCB(resource->containerObject);
	 /* Resource has been freed don't dereference it. */
      }
   }

   /* 
    * We propagate the information back to caller in case it wants to know
    * whether entry was freed.
    */      
   return result;
}


/*
 *-----------------------------------------------------------------------------------
 *
 * VMCIResource_AddClientPrivileges --
 *
 * Results:
 *      VMCI_SUCCESS if successful, error code if not.
 *
 * Side effects:
 *      None.
 *
 *-----------------------------------------------------------------------------------
 */

int
VMCIResource_AddClientPrivileges(VMCIHandle resourceHandle,
                                 VMCIHandle clientHandle,
                                 int numAllowPrivs,
                                 VMCIResourcePrivilegeType *allowPrivs,
                                 int numDenyPrivs,
                                 VMCIResourcePrivilegeType *denyPrivs)
{
   int i;
   VMCIResource *resource;
   VMCIResourceClient *client;
   int result = VMCI_SUCCESS;
   VMCILockFlags flags;

   if (resourceHandle == VMCI_INVALID_HANDLE || 
       clientHandle == VMCI_INVALID_HANDLE ||
       numAllowPrivs + numDenyPrivs < 1) {
      Log(LGPFX"AddClientPrivs: Invalid args.\n");
      return VMCI_ERROR_INVALID_ARGS;
   }

#ifdef VMX86_DEBUG
   VMCI_LOG((LGPFX"AddClientPrivs: Adding allow privs:\n"));
   for (i = 0; i < numAllowPrivs; i++) {
      VMCI_LOG((LGPFX"AddClientPrivs: %d. 0x%x.\n", i, allowPrivs[i]));
   }
   VMCI_LOG((LGPFX"AddClientPrivs: Adding deny privs:\n"));
   for (i = 0; i < numDenyPrivs; i++) {
      VMCI_LOG((LGPFX"AddClientPrivs: %d. 0x%x.\n", i, denyPrivs[i]));
   }
   VMCI_LOG((LGPFX"AddClientPrivs: to client 0x%"FMT64"x for resource "
             "0x%"FMT64"x.\n", clientHandle, resourceHandle));
#endif // VMX86_DEBUG

   resource = VMCIResource_Get(resourceHandle, VMCI_RESOURCE_TYPE_ANY);
   if (resource == NULL) {
      Log(LGPFX"AddClientPrivs: No resource.\n");
      return VMCI_ERROR_INVALID_ARGS;
   }

   /* Validate privileges up front. */
   result = ResourceValidatePrivileges(resource, numAllowPrivs, allowPrivs);
   if (result != VMCI_SUCCESS) {
      Log(LGPFX"AddClientPrivs: Invalid allow privs.\n");
      goto done;
   }
   result = ResourceValidatePrivileges(resource, numDenyPrivs, denyPrivs);
   if (result != VMCI_SUCCESS) {
      Log(LGPFX"AddClientPrivs: Invalid deny privs.\n");
      goto done;
   }

   /* If client doesn't exists, create it. */
   VMCI_GrabLock(&resource->clientsLock, &flags);
   client = ResourceGetClient(resource, clientHandle);
   if (client == NULL) {
      result = ResourceAddClient(resource, clientHandle, numAllowPrivs, 
                                 allowPrivs, numDenyPrivs, denyPrivs);
      VMCI_ReleaseLock(&resource->clientsLock, flags);
      goto done;
   }

   /* 
    * If same privilege is present in both the allow and deny array. The deny
    * privilege takes precedence.
    */
   for (i = 0; i < numAllowPrivs; i++) {
      client->privilege[allowPrivs[i]] = VMCI_PRIV_ALLOW;
   }
   for (i = 0; i < numDenyPrivs; i++) {
      client->privilege[denyPrivs[i]] = VMCI_PRIV_DENY;
   }

   ResourceReleaseClient(resource, client);
   VMCI_ReleaseLock(&resource->clientsLock, flags);
  done:
   if (resource) {
      VMCIResource_Release(resource);
   }

   return result;
}


/*
 *-----------------------------------------------------------------------------------
 *
 * VMCIResource_RemoveClientPrivileges --
 *
 * Results:
 *      VMCI_SUCCESS if successful, error code if not.
 *
 * Side effects:
 *      None.
 *
 *-----------------------------------------------------------------------------------
 */

int
VMCIResource_RemoveClientPrivileges(VMCIHandle resourceHandle,
                                    VMCIHandle clientHandle,
                                    int numPrivs, 
                                    VMCIResourcePrivilegeType *privs)
{
   int i;
   VMCIResource *resource;
   VMCIResourceClient *client;
   Bool noPrivs;
   int result = VMCI_SUCCESS;
   VMCILockFlags flags;

   if (resourceHandle == VMCI_INVALID_HANDLE || 
       clientHandle == VMCI_INVALID_HANDLE ||
       numPrivs < 1) {
      Log(LGPFX"RemoveClientPrivs: Invalid args.\n");
      return VMCI_ERROR_INVALID_ARGS;
   }

#ifdef VMX86_DEBUG
   VMCI_LOG((LGPFX"RemoveClientPrivs: Removing privs:\n"));
   for (i = 0; i < numPrivs; i++) {
      VMCI_LOG((LGPFX"RemoveClientPrivs: %d. 0x%x.\n", i, privs[i]));
   }
   VMCI_LOG((LGPFX"RemoveClientPrivs: from client 0x%"FMT64"x for resource "
             "0x%"FMT64"x.\n", clientHandle, resourceHandle));
#endif // VMX86_DEBUG

   resource = VMCIResource_Get(resourceHandle, VMCI_RESOURCE_TYPE_ANY);
   if (resource == NULL) {
      Log(LGPFX"RemoveClientPrivs: Failed getting resource.\n");
      result = VMCI_ERROR_INVALID_ARGS;
      goto done;
   }

   /* Validate privileges up front to avoid partial changes of privileges. */
   result = ResourceValidatePrivileges(resource, numPrivs, privs);
   if (result != VMCI_SUCCESS) {
      Log(LGPFX"RemoveClientPrivs: Invalid privs.\n");
      goto done;
   }

   VMCI_GrabLock(&resource->clientsLock, &flags);
   client = ResourceGetClient(resource, clientHandle);
   if (client == NULL) {
      VMCI_ReleaseLock(&resource->clientsLock, flags);
      Log(LGPFX"RemoveClientPrivs: No client.\n");
      result = VMCI_ERROR_INVALID_ARGS;
      goto done;
   }

   for (i = 0; i < numPrivs; i++) {
      /* Remove client privilege. */
      client->privilege[privs[i]] = VMCI_PRIV_NOT_SET;
   }

   /* Validate if client has no more privileges set and remove if so. */
   noPrivs = TRUE;
   for (i = 0; i < VMCI_NUM_PRIVILEGES; i++) {
      if (client->privilege[i] != VMCI_PRIV_NOT_SET) {
	 noPrivs = FALSE;
	 break;
      }
   }
   if (noPrivs) {
      /* 
       * This client no longer has any privileges set for resource. We remove it 
       * which also decrements the reference count.
       */
      VMCI_LOG((LGPFX"RemoveClientPrivs: Removing client 0x%"FMT64"x.\n",
                clientHandle));
      ResourceRemoveClient(resource, client);
   }
   ResourceReleaseClient(resource, client);
   VMCI_ReleaseLock(&resource->clientsLock, flags);

  done:
   if (resource) {
      VMCIResource_Release(resource);
   }

   return result;
}


/*
 *-----------------------------------------------------------------------------------
 *
 * VMCIResource_RemoveAllClientPrivileges --
 *
 * Results:
 *      VMCI_SUCCESS if successful, error code if not.
 *
 * Side effects:
 *      None.
 *
 *-----------------------------------------------------------------------------------
 */

int
VMCIResource_RemoveAllClientPrivileges(VMCIHandle resourceHandle,
                                       VMCIHandle clientHandle)
{
   VMCIResource *resource;
   VMCIResourceClient *client;
   int result = VMCI_SUCCESS;
   VMCILockFlags flags;

   if (resourceHandle == VMCI_INVALID_HANDLE || 
       clientHandle == VMCI_INVALID_HANDLE) {
      Log(LGPFX"RemoveAllClientPrivs: Invalid args.\n");
      return VMCI_ERROR_INVALID_ARGS;
   }

   resource = VMCIResource_Get(resourceHandle, VMCI_RESOURCE_TYPE_ANY);
   if (resource == NULL) {
      result = VMCI_ERROR_INVALID_ARGS;
      goto done;
   }

   VMCI_GrabLock(&resource->clientsLock, &flags);
   client = ResourceGetClient(resource, clientHandle);
   if (client == NULL) {
      VMCI_ReleaseLock(&resource->clientsLock, flags);
      result = VMCI_ERROR_INVALID_ARGS;
      goto done;
   }

   ResourceRemoveClient(resource, client);

   ResourceReleaseClient(resource, client);
   VMCI_ReleaseLock(&resource->clientsLock, flags);
  done:
   if (resource) {
      VMCIResource_Release(resource);
   }

   return result;
}


/*
 *-----------------------------------------------------------------------------------
 *
 * VMCIResource_CheckClientPrivilege --
 *
 * Results:
 *      VMCI_SUCCESS_ACCESS_GRANTED if privilege is allowed, VMCI_ERROR_NO_ACCESS if 
 *      privilege denied, error code otherwise.
 *
 * Side effects:
 *      None.
 *
 *-----------------------------------------------------------------------------------
 */

int
VMCIResource_CheckClientPrivilege(VMCIHandle resourceHandle,       // IN:
                                  VMCIHandle clientHandle,         // IN:
                                  VMCIResourcePrivilegeType priv)  // IN:
{
   VMCILockFlags flags;
   VMCIResource *resource;
   VMCIResourceClient *client;
   int result = VMCI_ERROR_INVALID_PRIV;

   if (resourceHandle == VMCI_INVALID_HANDLE || 
       clientHandle == VMCI_INVALID_HANDLE || 
       priv >= VMCI_NUM_PRIVILEGES) {
      Log(LGPFX"CheckClientPriv: Invalid args.\n");
      return VMCI_ERROR_INVALID_ARGS;
   }

   resource = VMCIResource_Get(resourceHandle, VMCI_RESOURCE_TYPE_ANY);
   if (resource == NULL) {
      return VMCI_ERROR_INVALID_ARGS;
   }

   /*
    * Clients can be either groups or contexts. No other clients are supported
    * at this point. ResourceAddClientPrivileges enforces this via 
    * ResourceCanSeeClient by looking for the VMCI_PRIV_ASSIGN_CLIENT privilege
    * or by verifying that the client is the caller context.
    * For both group and context clients we check the client directly to see if
    * it has the privilege set. If the privilege is not set we return error 
    * for groups but for contexts we continue by checking if the context is
    * a member of a group that has the requested privilege to the given resource
    * and grant access if so.
    */

   VMCI_GrabLock(&resource->clientsLock, &flags);
   client = ResourceGetClient(resource, clientHandle);

   if (VMCI_HANDLE_TO_RESOURCE_ID(clientHandle) == VMCI_CONTEXT_RESOURCE_ID && 
       (client == NULL || client->privilege[priv] == VMCI_PRIV_NOT_SET)) {

      /* 
       * At this point we know the client is a context. Check if it is a member
       * of a group that is a client to the resource and has the privilege set.
       */

      int i;
      VMCIId contextID = VMCI_HANDLE_TO_CONTEXT_ID(clientHandle);
      VMCIContext *context;

      if (client) {
	 ResourceReleaseClient(resource, client);
	 client = NULL;
      }

      context = VMCIContext_Get(contextID);
      if (context == NULL) {
	 result = VMCI_ERROR_INVALID_ARGS;
	 goto done;
      }
    
      for (i = 0; i < VMCIHandleArray_GetSize(context->groupArray); i++) {
	 VMCIHandle groupHandle = VMCIHandleArray_GetEntry(context->groupArray, i);
	 client = ResourceGetClient(resource, groupHandle);
	 if (client != NULL) {
	    /* 
	     * Check if client has privilege, if so stop, otherwise continue.
	     * Semantic currently is first group with privilege set has
	     * precedence. This could be enhance with a group priority where
	     * higher priority was checked first.
	     */
	    if (client->privilege[priv] != VMCI_PRIV_NOT_SET) {
	       VMCI_LOG((LGPFX"Client 0x%"FMT64"x is a member of group "
                         "0x%"FMT64"x which has priv 0x%x set to %d for "
                         "resource 0x%"FMT64"x.\n", 
                         clientHandle, groupHandle, priv, 
                         client->privilege[priv], resourceHandle));
	       break;	       
	    }
	    ResourceReleaseClient(resource, client);
	    client = NULL;
	 }
      }
   }

   if (client) {
      if (client->privilege[priv] == VMCI_PRIV_ALLOW) {
	 result = VMCI_SUCCESS_ACCESS_GRANTED;
      }
      if (client->privilege[priv] == VMCI_PRIV_DENY) {
	 result = VMCI_ERROR_NO_ACCESS;
      }
      ResourceReleaseClient(resource, client);
   }
   VMCI_LOG((LGPFX"Checking if client 0x%"FMT64"x has priv 0x%x for resource "
             "0x%"FMT64"x, result %d.\n", clientHandle, priv, resourceHandle,
             result));

  done:
   VMCI_ReleaseLock(&resource->clientsLock, flags);
   VMCIResource_Release(resource);
   return result;
}


/*
 *-----------------------------------------------------------------------------------
 *
 * VMCIResource_IncDsRegCount --
 *       Increments the registrationCount associated with a resource.
 *
 * Results:
 *       VMCI_SUCCESS on success, error code otherwise.      
 *
 * Side effects:
 *       None.
 *
 *-----------------------------------------------------------------------------------
 */

int
VMCIResource_IncDsRegCount(VMCIResource *resource) // IN:
{
   VMCILockFlags flags;

   if (!resource) {
      return VMCI_ERROR_INVALID_ARGS;
   }

   VMCI_GrabLock(&resource->clientsLock, &flags);
   resource->registrationCount++;
   VMCI_ReleaseLock(&resource->clientsLock, flags);

   return VMCI_SUCCESS;
}


/*
 *-----------------------------------------------------------------------------------
 *
 * VMCIResource_DecDsRegCount --
 *       Decrements the registrationCount associated with a resource.
 *
 * Results:
 *       VMCI_SUCCESS on success, error code otherwise.       
 *
 * Side effects:
 *       None.
 *
 *-----------------------------------------------------------------------------------
 */

int
VMCIResource_DecDsRegCount(VMCIResource *resource) // IN:
{
   VMCILockFlags flags;

   if (!resource) {
      return VMCI_ERROR_INVALID_ARGS;
   }

   VMCI_GrabLock(&resource->clientsLock, &flags);
   ASSERT(resource->registrationCount > 0);
   resource->registrationCount--;
   VMCI_ReleaseLock(&resource->clientsLock, flags);

   return VMCI_SUCCESS;
}


/*
 *-----------------------------------------------------------------------------
 *
 * VMCIResource_CheckGuestCalls --
 *
 *      Make sure that context supports the guestcalls we need.
 *
 * Results:
 *      TRUE if context is supported, FALSE otherwise.
 *
 * Side effects:
 *      None.
 *
 *-----------------------------------------------------------------------------
 */

Bool
VMCIResource_CheckGuestCalls(VMCIContext *context)
{
   /* This part of the API does not need any guestcalls. */
   return TRUE;
}

