/* **********************************************************
 * Copyright 1998 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/

/*
 *  vmx86.h - Platform independent data and interface for supporting 
 *            the vmx86 device driver. 
 */

#ifndef VMX86_H
#define VMX86_H

#define INCLUDE_ALLOW_VMMON
#define INCLUDE_ALLOW_VMCORE
#include "includeCheck.h"

#include "x86.h"
#include "modulecall.h"
#include "vcpuid.h"
#include "initblock.h"
#include "iocontrols.h"
#include "numa_defs.h"

#define INVALID_HOST_CPU ((uint32)(-1))

/*
 * VMDriver - the main data structure for the driver side of a 
 *            virtual machine.
 */ 

typedef struct VMDriver {
   int                 id;           /* unique VM ID */
   struct VMDriver    *nextDriver;   /* Next on list of all VMDrivers */

   Vcpuid              numVCPUs;     /* Number of vcpus in VM. */
   struct VMHost      *vmhost;       /* Host-specific fields. */

   /* Pointers to the crossover pages shared with the monitor. */
   struct VMCrossPage *crosspage[MAX_INITBLOCK_CPUS];
   volatile uint32    currentHostCpu[MAX_INITBLOCK_CPUS];
   volatile uint32   (*hostAPIC)[4]; /* kseg pointer to host APIC */

   void               *memtracker;   /* Memory tracker pointer */
   struct PhysTracker *physTracker;
   int                 releaseCount;
   Bool                checkFuncFailed;
   struct PerfCounter *perfCounter;
   VMMemMgmtInfo       memInfo;
   unsigned            fastClockRate;/* Modified while holding fastClock lock only */
   PShare_HintUpdate   hintUpdate[PSHARE_HINT_UPDATES_MAX];
   int                 hintIndex; 
   int                 cowID;
   struct {
      Bool             performingSwitch; /* Used by old VM */
      int              oldVmId;          /* Used by new VM */
   } fastMonSwitch;
} VMDriver;

typedef struct VmTimeStart {
   uint64 count;
   uint64 time;
} VmTimeStart;

#define MAX_LOCKED_PAGES (-1)

extern VMDriver *Vmx86_CreateVM(void);
extern int Vmx86_ReleaseVM(VMDriver *vm);
extern int Vmx86_InitVM(VMDriver *vm, InitBlock *initParams);
extern Bool Vmx86_InitNUMAInfo(NUMAInfoArgs *initParams);
extern void Vmx86_DestroyNUMAInfo(void);
extern Bool Vmx86_GetNUMAMemStats(VMDriver *curVM,
				 VMNUMAMemStatsArgs *outArgs);
extern NUMA_Node Vmx86_MPNToNodeNum(MPN mpn);
extern int Vmx86_LateInitVM(VMDriver *vm);
extern int Vmx86_RunVM(VMDriver *vm, Vcpuid vcpuid);
extern void Vmx86_SetStartTime(VmTimeStart *st);
extern uint32 Vmx86_GetkHzEstimate(VmTimeStart *st);
extern int Vmx86_SetHostClockRate(VMDriver *vm, int rate);
extern MPN Vmx86_LockPage(VMDriver *vm, void *addr, Bool allowMultipleMPNsPerVA);
extern int Vmx86_UnlockPage(VMDriver *vm, void *addr);
extern int Vmx86_UnlockPageByMPN(VMDriver *vm, MPN mpn, void *addr);
extern MPN Vmx86_GetRecycledPage(VMDriver *vm);
extern int Vmx86_ReleaseAnonPage(VMDriver *vm, MPN mpn);
extern int Vmx86_AllocLockedPages(VMDriver* vm, MPN32* mpns, 
				  unsigned numPages, Bool kernelMPNBuffer);
extern int Vmx86_FreeLockedPages(VMDriver *vm, const MPN32* mpns, 
				 unsigned numPages, Bool kernelMPNBuffer);
extern int Vmx86_GetLockedPageList(VMDriver* vm, MPN32* mpns, 
				   unsigned int numPages);
extern Bool Vmx86_IsAnonPage(VMDriver *vm, const MPN32 mpn);

extern int32 Vmx86_GetNumVMs(void);
extern int32 Vmx86_GetTotalMemUsage(void);
extern Bool Vmx86_SetConfiguredLockedPagesLimit(unsigned limit);
extern void Vmx86_SetDynamicLockedPagesLimit(unsigned limit);
extern Bool Vmx86_GetMemInfo(VMDriver *curVM,
                             int32 curVMOnly,
                             VMMemInfoArgs *args,
                             int outArgsLength);
extern Bool Vmx86_GetMemInfoCopy(VMDriver *curVM,
				 VMMemInfoArgs *args);
extern void Vmx86_Admit(VMDriver *curVM, VMMemInfoArgs *args);
extern Bool Vmx86_Readmit(VMDriver *curVM, int32 pageDelta);
extern void Vmx86_SetMemUsage(VMDriver *curVM, unsigned usage);
extern void Vmx86_Add2MonPageTable(VMDriver *vm, VPN vpn, MPN mpn,
				   Bool readOnly);
extern Bool Vmx86_PAEEnabled(void);
extern Bool Vmx86_SVMEnabledCPU(void);
extern Bool Vmx86_VTEnabledCPU(void);
extern Bool Vmx86_VTSupportedCPU(Bool use64Bit);
extern Bool Vmx86_BrokenCPUHelper(void);
extern void Vmx86_CompleteUserCall(VMDriver *vm, Vcpuid vcpuid);
extern void Vmx86_MonitorPollIPI(void);
extern void Vmx86_COWSharePages(VMDriver *vm, COWShareInfo *info);
extern void Vmx86_COWGetHintUpdates(VMDriver *vm, COWHintInfo *info);
extern void Vmx86_COWCopyPage(VMDriver *vm, MPN sharedMPN);
extern void Vmx86_COWRemoveHint(VMDriver *vm, MPN mpn);
extern void Vmx86_InitCOWList(void);
extern void Vmx86_COWCleanup(void);
extern void Vmx86_COWCheckPages(VMDriver *vm, COWCheckInfo *info);
extern MPN  Vmx86_COWGetZeroMPN(void);
extern Bool Vmx86_FastMonSwitchTransfer(VMDriver *vm, int id);
extern Bool Vmx86_FastMonSwitchEnd(VMDriver *vm);

#ifdef __APPLE__
extern Bool Vmx86_InCompatMode(void);
extern Bool Vmx86_InLongMode(void);
#else
static INLINE Bool
Vmx86_InCompatMode(void)
{
   return FALSE;
}

static INLINE Bool
Vmx86_InLongMode(void)
{
   return vm_x86_64;
}
#endif

#ifdef linux
#include <linux/errno.h>
#define USERCALL_RESTART (-ERESTARTNOINTR)
#else
#include <usercall.h>
#define USERCALL_RESTART (USERCALL_NOP)
#endif

#endif 
