/* **********************************************************
 * Copyright 2006 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/

/*
 * vmci_handle_array.h --
 *
 *	Simple dynamic array.
 */

#ifndef _VMCI_HANDLE_ARRAY_H_
#define _VMCI_HANDLE_ARRAY_H_

#define INCLUDE_ALLOW_VMMON
#define INCLUDE_ALLOW_VMCORE
#define INCLUDE_ALLOW_MODULE
#include "includeCheck.h"

#ifdef _WIN32
#include <ntddk.h>
#define MemAlloc(size) ExAllocatePoolWithTag(NonPagedPool, size, 'MMTC')
#define MemFree(ptr) ExFreePool(ptr);
#elif linux
#include "compat_slab.h"
#define MemAlloc(size) kmalloc(size, GFP_KERNEL)
#define MemFree(ptr) kfree(ptr)
#elif defined(__APPLE__)
#  include "hostif.h"
#  define MemAlloc(size) HostIF_AllocKernelMem(size, FALSE)
#  define MemFree(ptr) HostIF_FreeKernelMem(ptr)
#else 
#error "HandleArray only implemented on Linux, Windows and Mac OS."
#endif // _WIN32

#include "vmware.h"
#include "vmci_defs.h"

#define VMCI_HANDLE_ARRAY_DEFAULT_SIZE 4

typedef struct VMCIHandleArray {
   uint32          capacity;
   uint32          size;
   VMCIHandle      entries[1];
} VMCIHandleArray;


/*
 *-----------------------------------------------------------------------------------
 *
 * VMCIHandleArray_Create --
 *
 * Results:
 *      Array if successful, NULL if not.
 *
 * Side effects:
 *      None.
 *
 *-----------------------------------------------------------------------------------
 */

static INLINE VMCIHandleArray *
VMCIHandleArray_Create(uint32 capacity) 
{
   VMCIHandleArray *array;
   
   if (capacity == 0) {
      capacity = VMCI_HANDLE_ARRAY_DEFAULT_SIZE;
   }

   array = (VMCIHandleArray *)MemAlloc(2 * sizeof(uint32) + 
				     capacity * sizeof(VMCIHandle));
   if (array == NULL) {
      return NULL;
   }
   array->capacity = capacity;
   array->size = 0;
   
   return array;
}


/*
 *-----------------------------------------------------------------------------------
 *
 * VMCIHandleArray_Destroy --
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *-----------------------------------------------------------------------------------
 */

static INLINE void
VMCIHandleArray_Destroy(VMCIHandleArray *array) 
{
   MemFree(array);
}


/*
 *-----------------------------------------------------------------------------------
 *
 * VMCIHandleArray_AppendEntry --
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      Array may be reallocated.
 *
 *-----------------------------------------------------------------------------------
 */

static INLINE void
VMCIHandleArray_AppendEntry(VMCIHandleArray **arrayPtr,
                            VMCIHandle handle)
{
   VMCIHandleArray *array;

   ASSERT(arrayPtr && *arrayPtr);
   array = *arrayPtr;

   if (UNLIKELY(array->size >= array->capacity)) {
      /* reallocate. */
      uint32 arraySize = 2 * sizeof(uint32) + array->capacity * sizeof(VMCIHandle);
      VMCIHandleArray *newArray = 
	 MemAlloc(arraySize + array->capacity * sizeof(VMCIHandle));
      if (newArray == NULL) {
	 return;
      }
      memcpy(newArray, array, arraySize);
      newArray->capacity *= 2;
      MemFree(array);
      *arrayPtr = newArray;
      array = newArray;
   }
   array->entries[array->size] = handle;
   array->size++;
}


/*
 *-----------------------------------------------------------------------------------
 *
 * VMCIHandleArray_RemoveEntry --
 *
 * Results:
 *      Handle that was removed, VMCI_INVALID_HANDLE if entry not found.
 *
 * Side effects:
 *      None.
 *
 *-----------------------------------------------------------------------------------
 */

static INLINE VMCIHandle
VMCIHandleArray_RemoveEntry(VMCIHandleArray *array,
                            VMCIHandle entryHandle)
{
   int i;
   VMCIHandle handle = VMCI_INVALID_HANDLE;

   ASSERT(array);
   for (i = 0; i < array->size; i++) {
      if (array->entries[i] == entryHandle) {
	 handle = array->entries[i];
	 array->entries[i] = array->entries[array->size-1];
	 array->entries[array->size-1] = VMCI_INVALID_HANDLE;
	 array->size--;
	 break;
      }
   }

   return handle;
}


/*
 *-----------------------------------------------------------------------------------
 *
 * VMCIHandleArray_RemoveTail --
 *
 * Results:
 *      Handle that was removed, VMCI_INVALID_HANDLE if array was empty.
 *
 * Side effects:
 *      None.
 *
 *-----------------------------------------------------------------------------------
 */

static INLINE VMCIHandle
VMCIHandleArray_RemoveTail(VMCIHandleArray *array)
{
   VMCIHandle handle;
   
   if (array->size == 0) {
      return VMCI_INVALID_HANDLE;
   }
   handle = array->entries[array->size-1];
   array->entries[array->size-1] = VMCI_INVALID_HANDLE;
   array->size--;
   
   return handle;
}


/*
 *-----------------------------------------------------------------------------------
 *
 * VMCIHandleArray_GetEntry --
 *
 * Results:
 *      Handle at given index, VMCI_INVALID_HANDLE if invalid index.
 *
 * Side effects:
 *      None.
 *
 *-----------------------------------------------------------------------------------
 */

static INLINE VMCIHandle
VMCIHandleArray_GetEntry(const VMCIHandleArray *array,
                         uint32 index)
{
   ASSERT(array);
   if (UNLIKELY(index >= array->size)) {
      return VMCI_INVALID_HANDLE;
   }
 
  return array->entries[index];
}


/*
 *-----------------------------------------------------------------------------------
 *
 * VMCIHandleArray_GetSize --
 *
 * Results:
 *      Number of entries in array.
 *
 * Side effects:
 *      None.
 *
 *-----------------------------------------------------------------------------------
 */

static INLINE uint32
VMCIHandleArray_GetSize(const VMCIHandleArray *array)
{
   ASSERT(array);
   return array->size;
}


/*
 *-----------------------------------------------------------------------------------
 *
 * VMCIHandleArray_HasEntry --
 *
 * Results:
 *      TRUE is entry exists in array, FALSE if not.
 *
 * Side effects:
 *      None.
 *
 *-----------------------------------------------------------------------------------
 */

static INLINE Bool
VMCIHandleArray_HasEntry(const VMCIHandleArray *array,
                         VMCIHandle entryHandle)
{
   int i;

   ASSERT(array);
   for (i = 0; i < array->size; i++) {
      if (array->entries[i] == entryHandle) {
	 return TRUE;
      }
   }

   return FALSE;
}

#endif // _VMCI_HANDLE_ARRAY_H_
