/* **********************************************************
 * Copyright 2006 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/

/*
 * vmci_kernel_defs.h -- 
 * 
 *      This file implements defines and helper functions for VMCI
 *      host _and_ guest kernel code. It must work for both windows and 
 *      linux kernel, ie. using defines where necessary.
 */ 
 
#ifndef _VMCI_KERNEL_DEFS_H_
#define _VMCI_KERNEL_DEFS_H_

#if !defined(linux) && !defined(_WIN32) && !defined(__APPLE__)
#error "Platform not supported."
#endif

#if defined(_WIN32)
#include <ntddk.h>
#endif 

#include "vm_basic_types.h"
#ifdef linux
#include "compat_wait.h"
#include "compat_interrupt.h"
#include "compat_spinlock.h"
#include "compat_slab.h"
#endif // linux

#ifdef __APPLE__
#  include <IOKit/IOLib.h>
#  include "hostif.h"
#  include "driver.h" /* bora/modules/vmci/macos/driver.h */
#endif


/* Flags for specifying memory type. */
#define VMCI_MEMORY_ATOMIC   0x1
#define VMCI_MEMORY_NONPAGED 0x2

/* Shared lock for use on both windows and linux. */

#ifdef linux
typedef spinlock_t VMCILock;
#elif defined(__APPLE__)
typedef IOLock *VMCILock;
#else
typedef KSPIN_LOCK VMCILock;
#endif // linux

#if defined(linux) || defined(__APPLE__) 
typedef unsigned long VMCILockFlags;
#else 
typedef KIRQL VMCILockFlags;
#endif 

/*
 * Host specific struct for use on both windows and linux, used for
 * signalling.
 */

typedef struct VMCIHost {
#ifdef linux
   wait_queue_head_t  waitQueue;
#elif defined(__APPLE__)
   Socket *socket; /* vmci Socket object on Mac OS. */
#else
   KEVENT *callEvent; /* Ptr to userlevel event used when signalling 
                       * new pending guestcalls in kernel. */
#endif // linux
} VMCIHost;


/*
 *-----------------------------------------------------------------------------
 *
 * VMCI_InitLock
 *
 *      Initializes the lock. Must be called before use. XXX Move locking code
 *      into hostif if VMCI stays in vmmon.
 *
 * Results:
 *      None
 *
 * Side effects:
 *      Thread can block.
 *
 *-----------------------------------------------------------------------------
 */

static INLINE void
VMCI_InitLock(VMCILock *lock)
{
#ifdef linux
   spin_lock_init(lock);
#elif defined(__APPLE__)
   *lock = IOLockAlloc();
#else
   KeInitializeSpinLock(lock);
#endif // linux
}


/*
 *-----------------------------------------------------------------------------
 *
 * VMCI_GrabLock
 *
 *      Grabs the given lock. XXX Fill in specific lock requirements. XXX Move
 *      locking code into hostif if VMCI stays in vmmon.
 *
 * Results:
 *      None
 *
 * Side effects:
 *      Thread can block.
 *
 *-----------------------------------------------------------------------------
 */

static INLINE void
VMCI_GrabLock(VMCILock *lock,       // IN
              VMCILockFlags *flags) // OUT: used to restore irql on windows
{
#ifdef linux
   spin_lock(lock);
#elif defined(__APPLE__)
   IOLockLock(*lock);
#else
   KeAcquireSpinLock(lock, flags);
#endif // linux
}


/*
 *-----------------------------------------------------------------------------
 *
 * VMCI_ReleaseLock
 *
 *      Releases the given lock. XXX Move locking code into hostif if VMCI
 *      stays in vmmon.
 *
 * Results:
 *      None
 *
 * Side effects:
 *      A thread blocked on this lock may wake up.
 *
 *-----------------------------------------------------------------------------
 */

static INLINE void
VMCI_ReleaseLock(VMCILock *lock,      // IN
                 VMCILockFlags flags) // IN
{
#ifdef linux
   spin_unlock(lock);
#elif defined(__APPLE__)
   IOLockUnlock(*lock);
#else
   KeReleaseSpinLock(lock, flags);
#endif // linux
}


/*
 *-----------------------------------------------------------------------------
 *
 * VMCI_GrabLock_BH
 *
 *      Grabs the given lock and for linux kernels disables bottom half execution.
 * .    This should be used with locks accessed both from bottom half/tasklet
 *      contexts, ie. guestcall handlers, and from process contexts to avoid
 *      deadlocks where the process has the lock and gets descheduled due to a
 *      bh/tasklet coming in.
 *      For Windows acquiring a spinlock always raises the caller to 
 *      DISPATCH_LEVEL which means DPCs are blocked from running.
 *
 * Results:
 *      None
 *
 * Side effects:
 *      None.
 *
 *-----------------------------------------------------------------------------
 */

static INLINE void
VMCI_GrabLock_BH(VMCILock *lock, // IN
                 VMCILockFlags *flags)  // OUT: used to restore
{
#ifdef linux
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2, 3, 4)
   spin_lock_bh(lock);
#else

   /* 
    * Before 2.3.4 linux kernels spin_unlock_bh didn't exist so we are using 
    * spin_lock_irqsave/restore instead. I wanted to define spin_[un]lock_bh
    * functions in compat_spinlock.h as local_bh_disable;spin_lock(lock) and
    * so on, but local_bh_disable/enable does not exist on 2.2.26.
    */
   spin_lock_irqsave(lock, *flags);
#endif // LINUX_VERSION_CODE
#elif defined(__APPLE__)
   /* Don't need on Mac OS host. Only needed on guests, no Mac OS guest yet. */
#else
   KeAcquireSpinLock(lock, flags);
#endif // linux
}


/*
 *-----------------------------------------------------------------------------
 *
 * VMCI_ReleaseLock_BH
 *
 *      Releases the given lock and for linux kernels reenables bottom half 
 *      execution.
 * .    This should be used with locks accessed both from bottom half/tasklet
 *      contexts, ie. guestcall handlers, and from process contexts to avoid
 *      deadlocks where the process has the lock and get descheduled due to a
 *      bh/tasklet coming in.
 *      For Windows acquiring a spinlock always raises the caller to 
 *      DISPATCH_LEVEL which means DPCs are blocked from running.
 *
 * Results:
 *      None
 *
 * Side effects:
 *      None.
 *
 *-----------------------------------------------------------------------------
 */

static INLINE void
VMCI_ReleaseLock_BH(VMCILock *lock, // IN
                    VMCILockFlags flags)   // IN
{
#ifdef linux
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2, 3, 4)
   spin_unlock_bh(lock);
#else

   /* 
    * Before 2.3.4 linux kernels spin_unlock_bh didn't exist so we are using 
    * spin_lock_irqsave/restore instead. I wanted to define spin_[un]lock_bh
    * functions in compat_spinlock.h as local_bh_disable;spin_lock(lock) and
     * so on, but local_bh_disable/enable does not exist on 2.2.26.
     */
   spin_unlock_irqrestore(lock, flags);
#endif // LINUX_VERSION_CODE
#elif defined(__APPLE__)
   /* Don't need on Mac OS host. Only needed on guests, no Mac OS guest yet. */
#else
   KeReleaseSpinLock(lock, flags);
#endif // linux
}


/*
 *----------------------------------------------------------------------
 *
 * VMCIHost_InitContext --
 *
 *      Host-specific initialization of VMCI context state.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

static INLINE void
VMCIHost_InitContext(VMCIHost *hostContext, // IN
                     uintptr_t eventHnd)    // IN
{
#ifdef linux
   init_waitqueue_head(&hostContext->waitQueue);
#elif defined(__APPLE__)
   hostContext->socket = (Socket *) eventHnd;
#else // _WIN32
   if (eventHnd != -1) {
      NTSTATUS s;
      HANDLE hnd = (HANDLE)(uintptr_t)eventHnd;
      
      ASSERT(hostContext);
      s = ObReferenceObjectByHandle(hnd, SYNCHRONIZE, NULL, KernelMode,
				    &hostContext->callEvent, NULL);
   } else {
      hostContext->callEvent = NULL;
   }
#endif // __linux
}


/*
 *----------------------------------------------------------------------
 *
 * VMCIHost_SignalCall --
 *
 *      Signal to userlevel that a VMCI call is waiting.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

static INLINE void
VMCIHost_SignalCall(VMCIHost *hostContext)     // IN
{
#ifdef linux
   wake_up(&hostContext->waitQueue);
#elif defined(__APPLE__)
   MacOSDriver_SignalCall(hostContext->socket);
#else // _WIN32
   if (hostContext->callEvent) {
      KeSetEvent(hostContext->callEvent, (KPRIORITY)0, FALSE);
   }
#endif // linux
}


/*
 *----------------------------------------------------------------------
 *
 * VMCIHost_ClearCall --
 *
 *      Clear the pending call signal.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

static INLINE void
VMCIHost_ClearCall(VMCIHost *hostContext)     // IN
{
#ifdef _WIN32
   if (hostContext->callEvent) {
      KeClearEvent(hostContext->callEvent);
   }
#elif defined(__APPLE__)
   MacOSDriver_ClearCall(hostContext->socket);
#endif // _WIN32
}

/*
 *----------------------------------------------------------------------
 *
 * VMCI_AllocKernelMem
 *
 *      Allocate some kernel memory for the VMCI driver. 
 *
 * Results:
 *      The address allocated or NULL on error. 
 *      
 *
 * Side effects:
 *      memory is malloced
 *----------------------------------------------------------------------
 */

static INLINE void *
VMCI_AllocKernelMem(size_t size, int flags)
{
   void *ptr;
#ifdef _WIN32
   if ((flags & VMCI_MEMORY_NONPAGED) != 0) { 
      ptr = ExAllocatePoolWithTag(NonPagedPool, size, 'MMTC');
   } else {
      ptr = ExAllocatePoolWithTag(PagedPool, size, 'MMTC');
   }
#elif defined(__APPLE__)
   if ((flags & VMCI_MEMORY_NONPAGED) != 0) {
      ptr = HostIF_AllocKernelMem(size, TRUE);
   } else {
      ptr = HostIF_AllocKernelMem(size, FALSE);
   }
#else // linux
   if ((flags & VMCI_MEMORY_ATOMIC) != 0) {
      ptr = kmalloc(size, GFP_ATOMIC);
   } else {
      ptr = kmalloc(size, GFP_KERNEL);
   }
#endif // _WIN32
   return ptr;
}


/*
 *----------------------------------------------------------------------
 *
 * VMCI_FreeKernelMem
 *
 *      Free kernel memory allocated for the VMCI driver. 
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      memory is freed.
 *----------------------------------------------------------------------
 */

static INLINE void
VMCI_FreeKernelMem(void *ptr)
{
#ifdef _WIN32
   ExFreePool(ptr);
#elif defined(__APPLE__)
   HostIF_FreeKernelMem(ptr);
#else // linux
   kfree(ptr);
#endif // _WIN32
}


#endif // _VMCI_KERNEL_DEFS_

