/*
 * Copyright 2006.  VMware, Inc.  The use of this code is subject to 
 * your agreement to the terms of an applicable VMware license.
 */


#include "EXTERN.h"
#include "perl.h"
#include "XSUB.h"

#include "ppport.h"

#include "vix.h"

static void TheCallback(VixHandle handle,
                        VixEventType eventType,
                        VixHandle moreEventInfo,
                        void *clientData);

/* Default "NULL" parameter, correct for Perl */
#define NULL_SV   &PL_sv_undef

/*-------------------------------------------------------------------
 *
 * TheCallback
 *
 *    All Vix callbacks will go through this function.
 *
 *-------------------------------------------------------------------
 */

void 
TheCallback(VixHandle handle,
            VixEventType eventType,
            VixHandle moreEventInfo,
            void *clientData)
{
   /* No uses.  Here as a useful stub only */
}

/*------------------------------------------------------------------
 *
 * Here is an implementation of FindItems, the enumerator for 
 * VMs on the connected machine
 *
 *------------------------------------------------------------------
 */

/*
 * XXX malloc() has issues in Win32 perl (at least ActivePerl), so
 * sleaze it and use an array for the results
 *
 * this will fail if more there are more  MAX_FIND_RESULTS results
 */

#ifdef WIN32
#define PERL_MALLOC_BUG_HACK
#define MAX_FIND_RESULTS 100
#endif

#ifdef PERL_MALLOC_BUG_HACK

typedef struct vmArray {
   int num;
   char loc[MAX_FIND_RESULTS][512];
} vmArray;

#else

typedef struct vmList {
   char *loc;
   struct vmList *next;
}  vmList;

#endif


static void
findvms_cb(VixHandle jobHandle, VixEventType evType, VixHandle propHandle,
           void *clientdata)
{
#ifdef PERL_MALLOC_BUG_HACK
   vmArray *results = (vmArray *) clientdata;
#else
   vmList **lst = (vmList **) clientdata;
   vmList *ne;
#endif
   char *loc;
   VixError err;

   if (VIX_EVENTTYPE_FIND_ITEM != evType) {
      return;
   }

   err = Vix_GetProperties(propHandle, VIX_PROPERTY_FOUND_ITEM_LOCATION, &loc, 
                           VIX_PROPERTY_NONE);

   if (VIX_SUCCEEDED(err)) {
#ifdef PERL_MALLOC_BUG_HACK
      strcpy(results->loc[results->num++], loc);
      Vix_FreeBuffer(loc);
#else
      ne = (vmList *)malloc(sizeof(vmList));
      if (ne) {
         ne->loc = strdup(loc);
         ne->next = *lst;
         *lst = ne;
         Vix_FreeBuffer(loc);
      }
#endif
   }
}

static VixError
#ifdef PERL_MALLOC_BUG_HACK
doFindVMs(VixHandle hostHandle, int timeout, vmArray *vmlist)
#else
doFindVMs(VixHandle hostHandle, int timeout, vmList **vmlist)
#endif
{
   VixError err = VIX_OK;
   VixHandle jobHandle;

   jobHandle = VixHost_FindItems(hostHandle, VIX_FIND_RUNNING_VMS,
                                 VIX_INVALID_HANDLE, timeout,
                                 findvms_cb, vmlist);
   err = VixJob_Wait(jobHandle, VIX_PROPERTY_NONE);
   Vix_ReleaseHandle(jobHandle);

   return err;
}

/*-----------------------------------------------------------------
 *
 * Common property-collection code.  Handles mappings between Vix
 * types and Perl types.
 *
 *-----------------------------------------------------------------
 */

/* Presently, Vix_GetProperties and VixJob_Wait fit this prototype */
typedef VixError (*propertiesFunction)(VixHandle handle, VixPropertyID propID, ...);
/* Presently, Vix_GetNthProperties fit this prototype */
typedef VixError (*propertiesNthFunction)(VixHandle handle, int index, VixPropertyID propID, ...);

static VixError
doProperty(VixHandle handle, SV* propertySV, SV* resultSV, propertiesFunction func)
{
   char *sarg = NULL;
   int   iarg = 0;
   int property;
   VixError err;
   VixPropertyType propertyType = -1;

   property = SvIV(propertySV);

   if (VIX_PROPERTY_NONE == property) {
      return VIX_OK;
   }
   err = Vix_GetPropertyType(handle, property, &propertyType);
   if (VIX_FAILED(err)) {
      warn("Failed to get property type for %d on handle %d.\n", property, handle);
      return err;
   }

   switch (propertyType) {
   case VIX_PROPERTYTYPE_INTEGER:
   case VIX_PROPERTYTYPE_INT64:
   case VIX_PROPERTYTYPE_HANDLE:
   case VIX_PROPERTYTYPE_BOOL:
       err = func(handle, property, &iarg, VIX_PROPERTY_NONE);
       if (VIX_SUCCEEDED(err)) {
          sv_setiv(resultSV, iarg);
       }
       break;
   case VIX_PROPERTYTYPE_STRING:
       err = func(handle, property, &sarg, VIX_PROPERTY_NONE);
       if (VIX_SUCCEEDED(err)) {
          sv_setpv(resultSV, sarg);
          Vix_FreeBuffer(sarg);
       }
       break;
   default:
       warn("Unrecognized property type\n");
       err = VIX_E_UNRECOGNIZED_PROPERTY;
   }
   return err;
}

static VixError
doNthProperty(VixHandle handle, int index, SV* propertySV, SV* resultSV, propertiesNthFunction func)
{
   char *sarg = NULL;
   int   iarg = 0;
   int property;
   VixError err;
   VixPropertyType propertyType;

   property = SvIV(propertySV);

   if (VIX_PROPERTY_NONE == property) {
      return VIX_OK;
   }
   err = Vix_GetPropertyType(handle, property, &propertyType);
   if (VIX_FAILED(err)) {
      warn("Failed to get property type for %d on handle %d.\n", property, handle);
      return err;
   }

   switch (propertyType) {
   case VIX_PROPERTYTYPE_INTEGER:
   case VIX_PROPERTYTYPE_INT64:
   case VIX_PROPERTYTYPE_HANDLE:
   case VIX_PROPERTYTYPE_BOOL:
       err = func(handle, index, property, &iarg, VIX_PROPERTY_NONE);
       if (VIX_SUCCEEDED(err)) {
          sv_setiv(resultSV, iarg);
       }
       break;
   case VIX_PROPERTYTYPE_STRING:
       err = func(handle, index, property, &sarg, VIX_PROPERTY_NONE);
       if (VIX_SUCCEEDED(err)) {
          sv_setpv(resultSV, sarg);
          Vix_FreeBuffer(sarg);
       }
       break;
   default:
       warn("Unrecognized property type\n");
       err = VIX_E_UNRECOGNIZED_PROPERTY;
   }
   return err;
}


static VixError
doBlobProperty(VixHandle handle, SV* propertySV, SV* sizeSV, SV* dataSV, propertiesFunction func)
{
   char *sarg;
   int   iarg;
   int property;
   VixError err;
   VixPropertyType propertyType = -1;

   property = SvIV(propertySV);
   if (VIX_PROPERTY_NONE == property) {
      return VIX_OK;
   }

   err = Vix_GetPropertyType(handle, property, &propertyType);
   if (VIX_FAILED(err)) {
      warn("Failed to get property type for %d on handle %d.\n", property, handle);
      return err;
   }

   switch (propertyType) {
   case VIX_PROPERTYTYPE_BLOB:
       err = func(handle, property, &iarg, &sarg, VIX_PROPERTY_NONE);
       if (VIX_SUCCEEDED(err)) {
          sv_setiv(sizeSV, iarg);
          sv_setpvn(dataSV, sarg, iarg);
          Vix_FreeBuffer(sarg);
       }
       break;
   default:
       warn("Unrecognized property type\n");
       err = VIX_E_UNRECOGNIZED_PROPERTY;
   }
   return err;
}

/*******************************************************************/
/*******************************************************************/
/*******************************************************************/

/* MODULE vs. PACKAGE
 * 
 * This is a bootstrap hack.
 *
 * All functions in this file are in the VMware::Vix namespace.
 * On Windows, Perl creates a Vix.DLL to contain the functions
 * within said namespace.  However, Vix.DLL is already used as the
 * C implementations of all Vix functions.  So, this file (and
 * Makefile.PL and VixBinding.pm) contains some trickery to actually
 * build everything as the VMware::VixBinding module, even though
 * all the functions in this module are exported into the VMware::Vix
 * namespace.
 */


/*******************************************************************/

MODULE = VMware::VixBinding PACKAGE = VMware::Vix::API::API PREFIX = Vix_
PROTOTYPES: ENABLE

SV *
Vix_GetErrorText(err,language)
        VixError                       err
        const char *                    language
   PREINIT:
        char *txt;
   CODE:
        txt = (char *) Vix_GetErrorText(err, language);
        RETVAL = newSVpv(txt, 0);
   OUTPUT:
        RETVAL

void
Vix_ReleaseHandle(handle)
        VixHandle                      handle
   CODE:
        Vix_ReleaseHandle(handle);

VixHandleType
Vix_GetHandleType(handle)
        VixHandle                      handle

VixError
Vix_GetProperties(handle, ...)
        VixHandle                      handle
   PREINIT:
       int property;
       int i;
       int err;
       int propertyType;
   CODE:
        for (i = 1; i < items; i+= 2) {
           property = SvIV(ST(i));
           if (VIX_PROPERTY_NONE == property) {
               break;
           }
           err = Vix_GetPropertyType(handle, property, &propertyType);
           if (VIX_FAILED(err)) {
              warn("Failed to get property type for %d on handle %d.\n",
                  property, handle);
              goto abort;
           }
           if (VIX_PROPERTYTYPE_BLOB == propertyType) {
              // blobs return 2 values, size and data
              err = doBlobProperty(handle, ST(i), ST(i + 1), ST(i + 2),
                             Vix_GetProperties);
              i++;
           } else {
              err = doProperty(handle, ST(i), ST(i + 1), Vix_GetProperties);
           }
           if (VIX_FAILED(err)) {
              goto abort;
           }
        }
        // should always be NULL
        property = SvIV(ST(items-1)); /* last property */
        if (property != VIX_PROPERTY_NONE) {
            warn("Final argument to GetProperties() should be VIX_PROPERTY_NONE\n");
        }
        err = Vix_GetProperties(handle, property);
   abort:
       RETVAL = err;
   OUTPUT:
        RETVAL

VixError
Vix_GetPropertyType(handle,propertyID,propertyType)
      VixHandleType     handle
      VixPropertyID     propertyID
      VixPropertyType   &propertyType
   CODE:
      RETVAL = Vix_GetPropertyType(handle, propertyID, &propertyType);
   OUTPUT:
      propertyType
      RETVAL


void
Vix_PumpEvents(hostHandle,options)
        VixHandle                      hostHandle
        int                             options
   CODE:
      Vix_PumpEvents(hostHandle, (VixPumpEventsOptions)options);


####################################################
# Note: Vix_FreeBuffer is not implemented.        #
# The binding frees all buffers that are returned  #
# after wrapping them in a Perl SV, because of the #
# difference in memory management.                 #
####################################################


MODULE = VMware::VixBinding PACKAGE = VMware::Vix::API::Host PREFIX = VixHost_
PROTOTYPES: ENABLE

VixHandle
VixHost_Connect(apiVersion,hostType,hostName,hostPort,username,password,options,propertyListHandle,callbackProc=NULL_SV,clientData=NULL_SV)
        int                             apiVersion
        VixServiceProvider                    hostType
        const char *                    hostName
        int                             hostPort
        const char *                    username
        const char *                    password
        int                             options
        VixHandle                      propertyListHandle
        SV *                            callbackProc
        SV *                            clientData
   CODE:
        RETVAL = VixHost_Connect(apiVersion, 
                              hostType, hostName, hostPort,
                              username, password,
                              options, 
                              propertyListHandle,
                              NULL, NULL);
   OUTPUT:
        RETVAL

void
VixHost_Disconnect(hostHandle)
        VixHandle                       hostHandle
   CODE:
        VixHost_Disconnect(hostHandle);


#
# This breaks the raw Perl<->C model used elsewhere, because of the pain
# of using Perl callbacks.  So rather than try to return a C type,
# this code stuffs a Perl array on the stack, with the first element
# being the error code.
#
void
VixHost_FindItems(hostHandle,searchType,searchCriteria,timeout,callbackProc=NULL_SV,clientData=NULL_SV)
        VixHandle                      hostHandle
        VixFindItemType                searchType
        VixHandle                      searchCriteria
        int                            timeout
        SV *                           callbackProc
        SV *                           clientData
   PREINIT:
        VixError err;
#ifdef PERL_MALLOC_BUG_HACK
        vmArray results;
#else
        vmList *vmlist = NULL;
        vmList *l;
        vmList *t;
#endif
        int i;
   PPCODE:
#ifdef PERL_MALLOC_BUG_HACK
        results.num = 0;
        err = doFindVMs(hostHandle, timeout, &results);
        if (VIX_SUCCEEDED(err)) {
           XPUSHs(sv_2mortal(newSVnv(err)));
           for (i = 0; i < results.num; i++) {
              XPUSHs(sv_2mortal(newSVpv(results.loc[i], 0)));
           }
        }
#else
        err = doFindVMs(hostHandle, timeout, &vmlist);
        if (VIX_SUCCEEDED(err)) {
           l = vmlist;
           i = 0;
           XPUSHs(sv_2mortal(newSVnv(err)));
           i++;
           while (l) {
              XPUSHs(sv_2mortal(newSVpv(l->loc, 0)));
              t = l;
              l = l->next;
              free(t->loc);
              free(t);
              i++;
           }
        }
#endif

VixHandle 
VixHost_RegisterVM(hostHandle,vmxFilePathNameArg,callbackProc=NULL_SV,clientData=NULL_SV)
        VixHandle                      hostHandle
        const char *                   vmxFilePathNameArg
        SV *                           callbackProc
        SV *                           clientData
   CODE:
        RETVAL = VixHost_RegisterVM(hostHandle, vmxFilePathNameArg, NULL, NULL);
   OUTPUT:
        RETVAL

VixHandle 
VixHost_UnregisterVM(hostHandle,vmxFilePathNameArg,callbackProc=NULL_SV,clientData=NULL_SV)
        VixHandle                      hostHandle
        const char *                   vmxFilePathNameArg
        SV *                           callbackProc
        SV *                           clientData
   CODE:
        RETVAL = VixHost_UnregisterVM(hostHandle, vmxFilePathNameArg, NULL, NULL);
   OUTPUT:
        RETVAL


#####################################################################
# VMware::Vix::API::VM
# 
# Per-VM functions
# - General functions (Open, Delete)
# - Power Ops (PowerOn, PowerOff, Reset, Suspend)
# - Snapshots
#####################################################################
MODULE = VMware::VixBinding PACKAGE = VMware::Vix::API::VM PREFIX = VixVM_
PROTOTYPES: ENABLE

##########################################################
# General Functions
###################
VixHandle 
VixVM_Open(hostHandle,vmxFilePathNameArg,callbackProc=NULL_SV,clientData=NULL_SV)
        VixHandle                      hostHandle
        const char *                    vmxFilePathNameArg
        SV *                           callbackProc
        SV *                           clientData
   CODE:
/* XXX temp hack XXX
 * don't mount vmdb so it mostly works with current workstation GUI
 */
#ifndef notyet
        RETVAL = VixVM_Open(hostHandle, vmxFilePathNameArg, NULL, NULL);
#else


         RETVAL = VixVM_OpenEx(hostHandle, vmxFilePathNameArg,
                    0x0001, // VIX_VMOPEN_DONT_MOUNT_IN_VMDB
                    NULL, 0, NULL, NULL, NULL);
#endif

   OUTPUT:
        RETVAL

VixHandle
VixVM_Delete(vmHandle,deleteOptions,callbackProc=NULL_SV,clientData=NULL_SV)
        VixHandle                      vmHandle
        int                             deleteOptions
        SV *                            callbackProc
        SV *                            clientData
   CODE:
        RETVAL = VixVM_Delete(vmHandle, deleteOptions,
                              NULL, NULL);
   OUTPUT:
        RETVAL

##########################################################
# Power Ops
###########

VixHandle
VixVM_PowerOn(vmHandle,powerOnOptions,propertyListHandle,callbackProc=NULL_SV,clientData=NULL_SV)
        VixHandle                      vmHandle
        int                            powerOnOptions
        VixHandle                      propertyListHandle
        SV *                           callbackProc
        SV *                           clientData
   CODE:
        RETVAL = VixVM_PowerOn(vmHandle, 
                                powerOnOptions, 
                                propertyListHandle, 
                                NULL, NULL);
     OUTPUT:
        RETVAL

VixHandle
VixVM_PowerOff(vmHandle,powerOffOptions,callbackProc=NULL_SV,clientData=NULL_SV)
        VixHandle                      vmHandle
        VixVMPowerOpOptions            powerOffOptions
        SV *                           callbackProc
        SV *                           clientData
   CODE:
        RETVAL = VixVM_PowerOff(vmHandle,
                                powerOffOptions,
                                NULL, NULL);
    OUTPUT:
        RETVAL

VixHandle
VixVM_Reset(vmHandle,powerOnOptions,callbackProc=NULL_SV,clientData=NULL_SV)
        VixHandle                      vmHandle
        VixVMPowerOpOptions            powerOnOptions
        SV *                            callbackProc
        SV *                            clientData
   CODE:
        RETVAL = VixVM_Reset(vmHandle, 
                              powerOnOptions,
                              NULL, NULL);
   OUTPUT:
        RETVAL

VixHandle
VixVM_Suspend(vmHandle,powerOffOptions,callbackProc=NULL_SV,clientData=NULL_SV)
        VixHandle                       vmHandle
        VixVMPowerOpOptions             powerOffOptions
        SV *                            callbackProc
        SV *                            clientData
   CODE:
        RETVAL = VixVM_Suspend(vmHandle,
                                powerOffOptions,
                                NULL, NULL);
   OUTPUT:
        RETVAL

##########################################################
# VM Guest Functions
####################

VixHandle
VixVM_LoginInGuest(vmHandle,userName,password,options,callbackProc=NULL_SV,clientData=NULL_SV)
        VixHandle                      vmHandle
        char *                         userName
        char *                         password
        int                            options
        SV *                           callbackProc
        SV *                           clientData
   CODE:
        RETVAL = VixVM_LoginInGuest(vmHandle, userName,
                                         password, options,
                                         NULL, NULL);
   OUTPUT:
        RETVAL

VixHandle
VixVM_LogoutFromGuest(vmHandle,callbackProc=NULL_SV,clientData=NULL_SV)
        VixHandle                      vmHandle
        SV *                           callbackProc
        SV *                           clientData
   CODE:
        RETVAL = VixVM_LogoutFromGuest(vmHandle, NULL, NULL);
   OUTPUT:
        RETVAL

VixHandle
VixVM_RunProgramInGuest(vmHandle,guestProgramName,commandLineArgs,options,propertyListHandle,callbackProc=NULL_SV,clientData=NULL_SV)
        VixHandle                      vmHandle
        char *                         guestProgramName
        char *                         commandLineArgs
        VixRunProgramOptions           options
        VixHandle                      propertyListHandle
        SV *                           callbackProc
        SV *                           clientData
   CODE:
        RETVAL = VixVM_RunProgramInGuest(vmHandle, guestProgramName,
                                         commandLineArgs, options,
                                         propertyListHandle,
                                         NULL, NULL);
   OUTPUT:
        RETVAL

VixHandle
VixVM_ListProcessesInGuest(vmHandle,options,callbackProc=NULL_SV,clientData=NULL_SV)
        VixHandle                      vmHandle
        int                            options
        SV *                           callbackProc
        SV *                           clientData
   CODE:
        RETVAL = VixVM_ListProcessesInGuest(vmHandle, 
                                         options, NULL, NULL);
   OUTPUT:
        RETVAL

VixHandle
VixVM_KillProcessInGuest(vmHandle,pid,options,callbackProc=NULL_SV,clientData=NULL_SV)
        VixHandle                      vmHandle
        int                            pid
        int                            options
        SV *                           callbackProc
        SV *                           clientData
   CODE:
        RETVAL = VixVM_KillProcessInGuest(vmHandle, pid,
                                         options, NULL, NULL);
   OUTPUT:
        RETVAL

VixHandle
VixVM_RunScriptInGuest(vmHandle,interpreter,scriptText,options,propertyListHandle,callbackProc=NULL_SV,clientData=NULL_SV)
        VixHandle                      vmHandle
        char                          *interpreter
        char                          *scriptText
        VixRunProgramOptions           options
        VixHandle                      propertyListHandle
        SV *                           callbackProc
        SV *                           clientData
   CODE:
        RETVAL = VixVM_RunScriptInGuest(vmHandle,
                                        interpreter,
                                        scriptText, options,
                                        propertyListHandle,
                                        NULL, NULL);
   OUTPUT:
        RETVAL

VixHandle
VixVM_OpenUrlInGuest(vmHandle,url,windowState,propertyListHandle,callbackProc=NULL_SV,clientData=NULL_SV)
        VixHandle                      vmHandle
        char                          *url
        int                            windowState
        VixHandle                      propertyListHandle
        SV *                           callbackProc
        SV *                           clientData
   CODE:
        RETVAL = VixVM_OpenUrlInGuest(vmHandle,
                                        url,
                                        windowState,
                                        propertyListHandle,
                                        NULL, NULL);
   OUTPUT:
        RETVAL

VixHandle
VixVM_CopyFileFromHostToGuest(vmHandle,hostPathName,guestPathName,options,propertyListHandle,callbackProc=NULL_SV,clientData=NULL_SV)
        VixHandle                      vmHandle
        char *                         hostPathName
        char *                         guestPathName
        int                            options
        VixHandle                      propertyListHandle
        SV *                           callbackProc
        SV *                           clientData
   CODE:
        RETVAL = VixVM_CopyFileFromHostToGuest(vmHandle, hostPathName,
                                         guestPathName, options,
                                         propertyListHandle,
                                         NULL, NULL);
   OUTPUT:
        RETVAL

VixHandle
VixVM_CopyFileFromGuestToHost(vmHandle,guestPathName,hostPathName,options,propertyListHandle,callbackProc=NULL_SV,clientData=NULL_SV)
        VixHandle                      vmHandle
        char *                         guestPathName
        char *                         hostPathName
        int                            options
        VixHandle                      propertyListHandle
        SV *                           callbackProc
        SV *                           clientData
   CODE:
        RETVAL = VixVM_CopyFileFromGuestToHost(vmHandle, guestPathName,
                                         hostPathName, options,
                                         propertyListHandle,
                                         NULL, NULL);
   OUTPUT:
        RETVAL

VixHandle
VixVM_DeleteFileInGuest(vmHandle,guestPathName,callbackProc=NULL_SV,clientData=NULL_SV)
        VixHandle                      vmHandle
        char *                         guestPathName
        SV *                           callbackProc
        SV *                           clientData
   CODE:
        RETVAL = VixVM_DeleteFileInGuest(vmHandle, guestPathName,
                                         NULL, NULL);
   OUTPUT:
        RETVAL

VixHandle
VixVM_FileExistsInGuest(vmHandle,guestPathName,callbackProc=NULL_SV,clientData=NULL_SV)
        VixHandle                      vmHandle
        char *                         guestPathName
        SV *                           callbackProc
        SV *                           clientData
   CODE:
        RETVAL = VixVM_FileExistsInGuest(vmHandle, guestPathName,
                                         NULL, NULL);
   OUTPUT:
        RETVAL

VixHandle
VixVM_RenameFileInGuest(vmHandle,oldName,newName,options,propertyListHandle,callbackProc=NULL_SV,clientData=NULL_SV)
        VixHandle                      vmHandle
        char *                         oldName
        char *                         newName
        int                            options
        VixHandle                      propertyListHandle
        SV *                           callbackProc
        SV *                           clientData
   CODE:
        RETVAL = VixVM_RenameFileInGuest(vmHandle, oldName,
                                         newName, options,
                                         propertyListHandle,
                                         NULL, NULL);
   OUTPUT:
        RETVAL

VixHandle
VixVM_CreateTempFileInGuest(vmHandle,options,propertyListHandle,callbackProc=NULL_SV,clientData=NULL_SV)
        VixHandle                      vmHandle
        int                            options
        VixHandle                      propertyListHandle
        SV *                           callbackProc
        SV *                           clientData
   CODE:
        RETVAL = VixVM_CreateTempFileInGuest(vmHandle, options,
                                         propertyListHandle,
                                         NULL, NULL);
   OUTPUT:
        RETVAL

VixHandle
VixVM_ListDirectoryInGuest(vmHandle,pathName,options,callbackProc=NULL_SV,clientData=NULL_SV)
        VixHandle                      vmHandle
        char *                         pathName
        int                            options
        SV *                           callbackProc
        SV *                           clientData
   CODE:
        RETVAL = VixVM_ListDirectoryInGuest(vmHandle, pathName,
                                         options,
                                         NULL, NULL);
   OUTPUT:
        RETVAL

VixHandle
VixVM_CreateDirectoryInGuest(vmHandle,pathName,propertyListHandle,callbackProc=NULL_SV,clientData=NULL_SV)
        VixHandle                      vmHandle
        char *                         pathName
        VixHandle                      propertyListHandle
        SV *                           callbackProc
        SV *                           clientData
   CODE:
        RETVAL = VixVM_CreateDirectoryInGuest(vmHandle, pathName,
                                         propertyListHandle,
                                         NULL, NULL);
   OUTPUT:
        RETVAL

VixHandle
VixVM_DeleteDirectoryInGuest(vmHandle,pathName,options,callbackProc=NULL_SV,clientData=NULL_SV)
        VixHandle                      vmHandle
        char *                         pathName
        int                            options
        SV *                           callbackProc
        SV *                           clientData
   CODE:
        RETVAL = VixVM_DeleteDirectoryInGuest(vmHandle, pathName,
                                         options,
                                         NULL, NULL);
   OUTPUT:
        RETVAL


VixHandle
VixVM_DirectoryExistsInGuest(vmHandle,pathName,callbackProc=NULL_SV,clientData=NULL_SV)
        VixHandle                      vmHandle
        char *                         pathName
        SV *                           callbackProc
        SV *                           clientData
   CODE:
        RETVAL = VixVM_DirectoryExistsInGuest(vmHandle, pathName,
                                         NULL, NULL);
   OUTPUT:
        RETVAL

##########################################################
# VM Snapshot Functions
####################

VixError
VixVM_GetNumRootSnapshots(vmHandle,result)
        VixHandle                      vmHandle
        int                            &result
   CODE:
        RETVAL = VixVM_GetNumRootSnapshots(vmHandle, &result);
   OUTPUT:
        RETVAL
        result

VixError
VixVM_GetRootSnapshot(vmHandle,index,snapshotHandle)
        VixHandle                      vmHandle
        int                             index
        VixHandle                     &snapshotHandle
   CODE:
        RETVAL = VixVM_GetRootSnapshot(vmHandle, index,
                                           &snapshotHandle);
   OUTPUT:
        RETVAL
        snapshotHandle

VixHandle
VixVM_RemoveSnapshot(vmHandle,snapshotHandle,options,callbackProc=NULL_SV,clientData=NULL_SV)
        VixHandle                      vmHandle
        VixHandle                      snapshotHandle
        int                             options
        SV *                            callbackProc
        SV *                            clientData
   CODE:
        RETVAL = VixVM_RemoveSnapshot(vmHandle, snapshotHandle,
                                       options,
                                       NULL, NULL);
   OUTPUT:
        RETVAL
        
VixHandle
VixVM_RevertToSnapshot(vmHandle,snapshotHandle,options,propertyListHandle,callbackProc=NULL_SV,clientData=NULL_SV)
        VixHandle                      vmHandle
        VixHandle                      snapshotHandle
        int                             options
        VixHandle                      propertyListHandle
        SV *                            callbackProc
        SV *                            clientData
   CODE:
        RETVAL = VixVM_RevertToSnapshot(vmHandle, snapshotHandle,
                                         options, propertyListHandle,
                                         NULL, NULL);
   OUTPUT:
        RETVAL

VixHandle
VixVM_CreateSnapshot(vmHandle,name,description,options,propertyListHandle,callbackProc=NULL_SV,clientData=NULL_SV)
        VixHandle                      vmHandle
        const char *                    name
        const char *                    description
        int                             options
        VixHandle                      propertyListHandle
        SV *                            callbackProc
        SV *                            clientData
   CODE:
        RETVAL = VixVM_CreateSnapshot(vmHandle, name, description,
                                       options, propertyListHandle,
                                       NULL, NULL);
   OUTPUT:
        RETVAL

VixError
VixVM_GetCurrentSnapshot(vmHandle,result)
        VixHandle                      vmHandle
        VixHandle                     &result
   CODE:
        RETVAL = VixVM_GetCurrentSnapshot(vmHandle, &result);
   OUTPUT:
        RETVAL
        result

VixError
VixVM_GetNamedSnapshot(vmHandle,name,result)
        VixHandle                      vmHandle
        const char *                   name
        VixHandle                     &result
   CODE:
        RETVAL = VixVM_GetNamedSnapshot(vmHandle, name, &result);
   OUTPUT:
        RETVAL
        result

##########################################################
# VM Shared Folders Functions
###################

VixHandle
VixVM_GetNumSharedFolders(vmHandle,callbackProc=NULL_SV,clientData=NULL_SV)
        VixHandle                      vmHandle
        SV *                           callbackProc
        SV *                           clientData
   CODE:
        RETVAL = VixVM_GetNumSharedFolders(vmHandle,
                                           NULL, NULL);
   OUTPUT:
        RETVAL

VixHandle
VixVM_GetSharedFolderState(vmHandle,index,callbackProc=NULL_SV,clientData=NULL_SV)
        VixHandle                      vmHandle
        int                            index
        SV *                           callbackProc
        SV *                           clientData
   CODE:
        RETVAL = VixVM_GetSharedFolderState(vmHandle, index,
                                            NULL, NULL);
   OUTPUT:
        RETVAL

VixHandle
VixVM_SetSharedFolderState(vmHandle,shareName,hostPathName,flags,callbackProc=NULL_SV,clientData=NULL_SV)
        VixHandle                      vmHandle
        char                          *shareName
        char                          *hostPathName
        VixMsgSharedFolderOptions      flags
        SV *                           callbackProc
        SV *                           clientData
   CODE:
        RETVAL = VixVM_SetSharedFolderState(vmHandle, shareName,
                                            hostPathName, flags,
                                            NULL, NULL);
   OUTPUT:
        RETVAL

VixHandle
VixVM_AddSharedFolder(vmHandle,shareName,hostPathName,flags,callbackProc=NULL_SV,clientData=NULL_SV)
        VixHandle                      vmHandle
        char                          *shareName
        char                          *hostPathName
        VixMsgSharedFolderOptions      flags
        SV *                           callbackProc
        SV *                           clientData
   CODE:
        RETVAL = VixVM_AddSharedFolder(vmHandle, shareName, hostPathName,
                                       flags, NULL, NULL);
   OUTPUT:
        RETVAL

VixHandle
VixVM_RemoveSharedFolder(vmHandle,shareName,flags,callbackProc=NULL_SV,clientData=NULL_SV)
        VixHandle                      vmHandle
        char                          *shareName
        VixMsgSharedFolderOptions      flags
        SV *                           callbackProc
        SV *                           clientData
   CODE:
        RETVAL = VixVM_RemoveSharedFolder(vmHandle, shareName,
                                       flags, NULL, NULL);
   OUTPUT:
        RETVAL

VixHandle
VixVM_EnableSharedFolders(vmHandle,enable,options,callbackProc=NULL_SV,clientData=NULL_SV)
        VixHandle                      vmHandle
        Bool                           enable
        int                            options
        SV *                           callbackProc
        SV *                           clientData
   CODE:
        RETVAL = VixVM_EnableSharedFolders(vmHandle, enable,
                                           options, NULL, NULL);
   OUTPUT:
        RETVAL


##########################################################
# General VM Functions
###################

VixHandle
VixVM_WaitForToolsInGuest(vmHandle,timeoutInSeconds,callbackProc=NULL_SV,clientData=NULL_SV)
        VixHandle                      vmHandle
        int                             timeoutInSeconds
        SV *                            callbackProc
        SV *                            clientData
   CODE:
        RETVAL = VixVM_WaitForToolsInGuest(vmHandle,
                                timeoutInSeconds,
                                NULL, NULL);
   OUTPUT:
        RETVAL

VixHandle 
VixVM_UpgradeVirtualHardware(vmHandle,options,callbackProc=NULL_SV,clientData=NULL_SV)
        VixHandle                      vmHandle
        int                             options
        SV *                            callbackProc
        SV *                            clientData
   CODE:
        RETVAL = VixVM_UpgradeVirtualHardware(vmHandle, options,
                                      NULL, NULL);
   OUTPUT:
        RETVAL

VixHandle 
VixVM_InstallTools(vmHandle,options,commandLineArgs,callbackProc=NULL_SV,clientData=NULL_SV)
        VixHandle                      vmHandle
        int                             options
        char *                          commandLineArgs
        SV *                            callbackProc
        SV *                            clientData
   CODE:
        RETVAL = VixVM_InstallTools(vmHandle, options, commandLineArgs,
                                     NULL, NULL);
   OUTPUT:
        RETVAL

##########################################################
# Read/Write variable VM Functions
###################

VixHandle
VixVM_ReadVariable(vmHandle,variableType,name,options,callback=NULL_SV,clientData=NULL_SV)
      VixHandle                     vmHandle
      int                           variableType
      const char *                  name
      int                           options
      SV *                          callback
      SV *                          clientData
   CODE:
      RETVAL = VixVM_ReadVariable(vmHandle, variableType,
                                  name, options, NULL, NULL);
   OUTPUT:
      RETVAL

VixHandle
VixVM_WriteVariable(vmHandle,variableType,valueName,value,options,callback=NULL_SV,clientData=NULL_SV)
      VixHandle                     vmHandle
      int                           variableType
      const char *                  valueName
      const char *                  value
      int                           options
      SV *                          callback
      SV *                          clientData
   CODE:
      RETVAL = VixVM_WriteVariable(vmHandle, variableType,
                                  valueName, value, options, NULL, NULL);
   OUTPUT:
      RETVAL


##########################################################
# Snapshot Handle Functions
####################

MODULE = VMware::VixBinding PACKAGE = VMware::Vix::API::Snapshot PREFIX = VixSnapshot_
PROTOTYPES: ENABLE

VixError
VixSnapshot_GetNumChildren(snapshotHandle, numChildren)
      VixHandle                        snapshotHandle
      int                              &numChildren
   CODE:
        RETVAL = VixSnapshot_GetNumChildren(snapshotHandle, &numChildren);
   OUTPUT:
        RETVAL
        numChildren

VixError
VixSnapshot_GetChild(snapshotHandle, index, childSnapshotHandle)
      VixHandle                        snapshotHandle
      int                              index
      VixHandle                        &childSnapshotHandle
   CODE:
        RETVAL = VixSnapshot_GetChild(snapshotHandle, index, &childSnapshotHandle);
   OUTPUT:
        RETVAL
        childSnapshotHandle

VixError
VixSnapshot_GetParent(snapshotHandle, parentSnapshotHandle)
      VixHandle                        snapshotHandle
      VixHandle                        &parentSnapshotHandle
   CODE:
        RETVAL = VixSnapshot_GetParent(snapshotHandle, &parentSnapshotHandle);
   OUTPUT:
        RETVAL
        parentSnapshotHandle


MODULE = VMware::VixBinding PACKAGE = VMware::Vix::API::Job PREFIX = VixJob_
PROTOTYPES: ENABLE

VixError
VixJob_Wait(jobHandle, ...)
        VixHandle                       jobHandle
   PREINIT:
       int property;
       int i;
       int err;
       int propertyType;
   CODE:
        for (i = 1; i < items; i+= 2) {
           property = SvIV(ST(i));
           if (VIX_PROPERTY_NONE == property) {
               break;
           }
           err = Vix_GetPropertyType(jobHandle, property, &propertyType);
           if (VIX_FAILED(err)) {
              warn("Failed to get property type for %d on handle %d.\n",
                  property, jobHandle);
              goto abort;
           }
           if (VIX_PROPERTYTYPE_BLOB == propertyType) {
              // blobs return 2 values, size and data
              err = doBlobProperty(jobHandle, ST(i), ST(i + 1), ST(i + 2),
                             VixJob_Wait);
              i++;
           } else {
              err = doProperty(jobHandle, ST(i), ST(i + 1), VixJob_Wait);
           }
           if (VIX_FAILED(err)) {
              goto abort;
           }
        }
        // should always be NULL
        property = SvIV(ST(items-1)); /* last property */
        if (property != VIX_PROPERTY_NONE) {
            warn("Final argument to Wait() should be VIX_PROPERTY_NONE\n");
        }
        err = VixJob_Wait(jobHandle, property);
   abort:
       RETVAL = err;
   OUTPUT:
        RETVAL

VixError 
VixJob_CheckCompletion(jobHandle,complete)
        VixHandle                      jobHandle
        Bool                           &complete
   OUTPUT:
        complete
        RETVAL

VixError
VixJob_GetError(jobHandle)
        VixHandle                      jobHandle

int
VixJob_GetNumProperties(jobHandle, resultPropertyID)
         VixHandle                     jobHandle
         int                           resultPropertyID

VixError
VixJob_GetNthProperties(handle, index, propertyID, ...)
        VixHandle                      handle
        int                            index
        int                            propertyID
   PREINIT:
       int property;
       int i;
       VixError err;
   CODE:
        if ((items < 3) || ((items - 1) % 2) != 0) {
           warn("VixJob_GetNthProperties must have an even number of arguments.\n");
        }
        for (i = 3; i < items; i+= 2) {
           err = doNthProperty(handle, index, ST(i - 1), ST(i), VixJob_GetNthProperties);
           if (VIX_FAILED(err)) {
              goto abort;
           }
        }
        // should always be NULL
        property = SvIV(ST(items-1)); /* last property */
        if (property != VIX_PROPERTY_NONE) {
            warn("Final argument to GetProperties() should be VIX_PROPERTY_NONE\n");
        }
        err = VixJob_GetNthProperties(handle, index, property);
   abort:
        RETVAL = err;
   OUTPUT:
        RETVAL

