/* $Id: tstBase64.cpp $ */
/** @file
 * IPRT Testcase - Base64.
 */

/*
 * Copyright (C) 2009 Sun Microsystems, Inc.
 *
 * This file is part of VirtualBox Open Source Edition (OSE), as
 * available from http://www.virtualbox.org. This file is free software;
 * you can redistribute it and/or modify it under the terms of the GNU
 * General Public License (GPL) as published by the Free Software
 * Foundation, in version 2 as it comes in the "COPYING" file of the
 * VirtualBox OSE distribution. VirtualBox OSE is distributed in the
 * hope that it will be useful, but WITHOUT ANY WARRANTY of any kind.
 *
 * The contents of this file may alternatively be used under the terms
 * of the Common Development and Distribution License Version 1.0
 * (CDDL) only, as it comes in the "COPYING.CDDL" file of the
 * VirtualBox OSE distribution, in which case the provisions of the
 * CDDL are applicable instead of those of the GPL.
 *
 * You may elect to license modified versions of this file under the
 * terms and conditions of either the GPL or the CDDL or both.
 *
 * Please contact Sun Microsystems, Inc., 4150 Network Circle, Santa
 * Clara, CA 95054 USA or visit http://www.sun.com if you need
 * additional information or have any questions.
 */

/*******************************************************************************
*   Header Files                                                               *
*******************************************************************************/
#include <iprt/base64.h>
#include <iprt/err.h>
#include <iprt/initterm.h>
#include <iprt/stream.h>
#include <iprt/string.h>
#include <iprt/stdarg.h>


/*******************************************************************************
*   Defined Constants And Macros                                               *
*******************************************************************************/
#if defined(RT_OS_OS2) || defined(RT_OS_WINDOWS)
# define MY_NL "\r\n"
#else
# define MY_NL "\n"
#endif


/*******************************************************************************
*   Global Variables                                                           *
*******************************************************************************/
static int g_cErrors = 0;


static void tstBase64(const void *pvData, size_t cbData, const char *pszEnc, size_t cchEnc,
                      int fTextData, int fNormalEnc, const char *pszTestFmt, ...)
{
    char    szOut[0x10000];
    size_t  cchOut = 0;

    /* Format the test name first. */
    va_list va;
    va_start(va, pszTestFmt);
    char szTest[32];
    RTStrPrintfV(szTest, sizeof(szTest), pszTestFmt, va);
    va_end(va);

    /*
     * Test decoding.
     */
    int rc = RTBase64Decode(pszEnc, szOut, cbData, &cchOut, NULL);
    if (RT_FAILURE(rc))
    {
        RTPrintf("tstBase64: FAILURE - %s: RTBase64Decode -> %Rrc\n", szTest, rc);
        g_cErrors++;
    }
    else if (cchOut != cbData)
    {
        RTPrintf("tstBase64: FAILURE - %s: RTBase64Decode returned %zu bytes, expected %zu.\n",
                 szTest, cchOut, cbData);
        g_cErrors++;
    }
    else if (memcmp(szOut, pvData, cchOut))
    {
        if (fTextData)
            RTPrintf("tstBase64: FAILURE - %s: RTBase64Decode returned:\n%.*s\nexpected:\n%s\n",
                     szTest, (int)cchOut, szOut, pvData);
        else
            RTPrintf("tstBase64: FAILURE - %s: RTBase64Decode return mismatching output\n", szTest);
        g_cErrors++;
    }

    cchOut = RTBase64DecodedSize(pszEnc, NULL);
    if (cchOut != cbData)
    {
        RTPrintf("tstBase64: FAILURE - %s: RTBase64DecodedSize returned %zu bytes, expected %zu.\n",
                 szTest, cchOut, cbData);
        g_cErrors++;
    }

    /*
     * Test encoding.
     */
    rc = RTBase64Encode(pvData, cbData, szOut, cchEnc + 1, &cchOut);
    if (RT_FAILURE(rc))
    {
        RTPrintf("tstBase64: FAILURE - %s: RTBase64Encode -> %Rrc\n", szTest, rc);
        g_cErrors++;
    }
    else if (fNormalEnc && cchOut != cchEnc)
    {
        RTPrintf("tstBase64: FAILURE - %s: RTBase64Encode returned %zu bytes, expected %zu.\n",
                 szTest, cchOut, cchEnc);
        g_cErrors++;
    }
    else if (fNormalEnc && memcmp(szOut, pszEnc, cchOut + 1))
    {
        RTPrintf("tstBase64: FAILURE - %s: RTBase64Encode returned:\n%*s\nexpected:\n%s\n",
                 szTest, szOut, pszEnc);
        g_cErrors++;
    }

    size_t cchOut2 = RTBase64EncodedLength(cbData);
    if (cchOut != cchOut2)
    {
        RTPrintf("tstBase64: FAILURE - %s: RTBase64EncodedLength returned %zu bytes, expected %zu.\n",
                 szTest, cchOut2, cchOut);
        g_cErrors++;
    }

    /** @todo negative testing. */
}


int main()
{
    RTR3Init();


    /*
     * Series of simple tests.
     */
    RTPrintf("tstBase64: TESTING - Test 1\n");
    static const struct
    {
        const char *pszText;
        size_t      cchText;
        const char *pszEnc;
        size_t      cchEnc;
    } g_aTests[] =
    {
#define TEST_ENTRY(szText, szEnc) { szText, sizeof(szText) - 1, szEnc, sizeof(szEnc) - 1 }
        TEST_ENTRY("Hey", "SGV5"),
        TEST_ENTRY("Base64", "QmFzZTY0"),
        TEST_ENTRY("Call me Ishmael.", "Q2FsbCBtZSBJc2htYWVsLg=="),
        TEST_ENTRY(
        "Man is distinguished, not only by his reason, but by this singular passion "
        "from other animals, which is a lust of the mind, that by a perseverance of "
        "delight in the continued and indefatigable generation of knowledge, exceeds "
        "the short vehemence of any carnal pleasure." /* Thomas Hobbes's Leviathan */,
        "TWFuIGlzIGRpc3Rpbmd1aXNoZWQsIG5vdCBvbmx5IGJ5IGhpcyByZWFzb24sIGJ1" MY_NL
        "dCBieSB0aGlzIHNpbmd1bGFyIHBhc3Npb24gZnJvbSBvdGhlciBhbmltYWxzLCB3" MY_NL
        "aGljaCBpcyBhIGx1c3Qgb2YgdGhlIG1pbmQsIHRoYXQgYnkgYSBwZXJzZXZlcmFu" MY_NL
        "Y2Ugb2YgZGVsaWdodCBpbiB0aGUgY29udGludWVkIGFuZCBpbmRlZmF0aWdhYmxl" MY_NL
        "IGdlbmVyYXRpb24gb2Yga25vd2xlZGdlLCBleGNlZWRzIHRoZSBzaG9ydCB2ZWhl" MY_NL
        "bWVuY2Ugb2YgYW55IGNhcm5hbCBwbGVhc3VyZS4="
        )
#undef TEST_ENTRY
    };

    for (unsigned i = 0; i < RT_ELEMENTS(g_aTests); i++)
        tstBase64(g_aTests[i].pszText, g_aTests[i].cchText,
                  g_aTests[i].pszEnc, g_aTests[i].cchEnc,
                  1 /* fTextData */, 1 /* fNormalEnc */, "Test 1-#u", i);

    /*
     * Try with some more junk in the encoding and different line length.
     */
    RTPrintf("tstBase64: TESTING - Test 2\n");
    static const char s_szText2[] =
        "Man is distinguished, not only by his reason, but by this singular passion "
        "from other animals, which is a lust of the mind, that by a perseverance of "
        "delight in the continued and indefatigable generation of knowledge, exceeds "
        "the short vehemence of any carnal pleasure."; /* Thomas Hobbes's Leviathan */

    static const char s_szEnc2[] =
        "  TWFuIGlzIGRpc3Rpbmd1aXNoZWQsIG5vdCBvbmx5IGJ5IGhpcyByZWFzb24sIGJ1dCBieSB0aGlz\r\n"
        "  IHNpbmd1bGFyIHBhc3Npb24gZnJvbSBvdGhlciBhbmltYWxzLCB3aGljaCBpcyBhIGx1c3Qgb2Yg\n\r\t\t\t\v"
          "dGhlIG1pbmQsIHRoYXQgYnkgYSBwZXJzZXZlcmFuY2Ugb2YgZGVsaWdodCBpbiB0aGUgY29udGlu\n"
        "\tdWVkIGFuZCBpbmRlZmF0aWdhYmxlIGdlbmVyYXRpb24gb2Yga25vd2xlZGdlLCBleGNlZWRzIHRo\n\r"
        "  ZSBzaG9ydCB2ZWhlbWVuY2Ugb2YgYW55IGNhcm5hbCBwbGVhc3VyZS4=\n \n   \r   \n \t";

    tstBase64(s_szText2, sizeof(s_szText2) - 1,
              s_szEnc2,  sizeof(s_szEnc2) - 1,
              1 /* fTextData */, 0 /* fNormalEnc */, "Test 2");

    /*
     * Finally, a more extensive test.
     */
    RTPrintf("tstBase64: TESTING - Test 3\n");
    static uint8_t s_abData3[12*256];
    for (unsigned i = 0; i < 256; i++)
    {
        unsigned j = i*12;
        s_abData3[j + 0] = i;      /* */
        s_abData3[j + 1] = 0xff;
        s_abData3[j + 2] = i;
        s_abData3[j + 3] = 0xff;   /* */
        s_abData3[j + 4] = i;
        s_abData3[j + 5] = 0xff;
        s_abData3[j + 6] = i;      /* */
        s_abData3[j + 7] = 0x00;
        s_abData3[j + 8] = i;
        s_abData3[j + 9] = 0x00;   /* */
        s_abData3[j + 10]= i;
        s_abData3[j + 11]= 0x00;
    }

    static const char s_szEnc3[] =
        "AP8A/wD/AAAAAAAAAf8B/wH/AQABAAEAAv8C/wL/AgACAAIAA/8D/wP/AwADAAMA" MY_NL
        "BP8E/wT/BAAEAAQABf8F/wX/BQAFAAUABv8G/wb/BgAGAAYAB/8H/wf/BwAHAAcA" MY_NL
        "CP8I/wj/CAAIAAgACf8J/wn/CQAJAAkACv8K/wr/CgAKAAoAC/8L/wv/CwALAAsA" MY_NL
        "DP8M/wz/DAAMAAwADf8N/w3/DQANAA0ADv8O/w7/DgAOAA4AD/8P/w//DwAPAA8A" MY_NL
        "EP8Q/xD/EAAQABAAEf8R/xH/EQARABEAEv8S/xL/EgASABIAE/8T/xP/EwATABMA" MY_NL
        "FP8U/xT/FAAUABQAFf8V/xX/FQAVABUAFv8W/xb/FgAWABYAF/8X/xf/FwAXABcA" MY_NL
        "GP8Y/xj/GAAYABgAGf8Z/xn/GQAZABkAGv8a/xr/GgAaABoAG/8b/xv/GwAbABsA" MY_NL
        "HP8c/xz/HAAcABwAHf8d/x3/HQAdAB0AHv8e/x7/HgAeAB4AH/8f/x//HwAfAB8A" MY_NL
        "IP8g/yD/IAAgACAAIf8h/yH/IQAhACEAIv8i/yL/IgAiACIAI/8j/yP/IwAjACMA" MY_NL
        "JP8k/yT/JAAkACQAJf8l/yX/JQAlACUAJv8m/yb/JgAmACYAJ/8n/yf/JwAnACcA" MY_NL
        "KP8o/yj/KAAoACgAKf8p/yn/KQApACkAKv8q/yr/KgAqACoAK/8r/yv/KwArACsA" MY_NL
        "LP8s/yz/LAAsACwALf8t/y3/LQAtAC0ALv8u/y7/LgAuAC4AL/8v/y//LwAvAC8A" MY_NL
        "MP8w/zD/MAAwADAAMf8x/zH/MQAxADEAMv8y/zL/MgAyADIAM/8z/zP/MwAzADMA" MY_NL
        "NP80/zT/NAA0ADQANf81/zX/NQA1ADUANv82/zb/NgA2ADYAN/83/zf/NwA3ADcA" MY_NL
        "OP84/zj/OAA4ADgAOf85/zn/OQA5ADkAOv86/zr/OgA6ADoAO/87/zv/OwA7ADsA" MY_NL
        "PP88/zz/PAA8ADwAPf89/z3/PQA9AD0APv8+/z7/PgA+AD4AP/8//z//PwA/AD8A" MY_NL
        "QP9A/0D/QABAAEAAQf9B/0H/QQBBAEEAQv9C/0L/QgBCAEIAQ/9D/0P/QwBDAEMA" MY_NL
        "RP9E/0T/RABEAEQARf9F/0X/RQBFAEUARv9G/0b/RgBGAEYAR/9H/0f/RwBHAEcA" MY_NL
        "SP9I/0j/SABIAEgASf9J/0n/SQBJAEkASv9K/0r/SgBKAEoAS/9L/0v/SwBLAEsA" MY_NL
        "TP9M/0z/TABMAEwATf9N/03/TQBNAE0ATv9O/07/TgBOAE4AT/9P/0//TwBPAE8A" MY_NL
        "UP9Q/1D/UABQAFAAUf9R/1H/UQBRAFEAUv9S/1L/UgBSAFIAU/9T/1P/UwBTAFMA" MY_NL
        "VP9U/1T/VABUAFQAVf9V/1X/VQBVAFUAVv9W/1b/VgBWAFYAV/9X/1f/VwBXAFcA" MY_NL
        "WP9Y/1j/WABYAFgAWf9Z/1n/WQBZAFkAWv9a/1r/WgBaAFoAW/9b/1v/WwBbAFsA" MY_NL
        "XP9c/1z/XABcAFwAXf9d/13/XQBdAF0AXv9e/17/XgBeAF4AX/9f/1//XwBfAF8A" MY_NL
        "YP9g/2D/YABgAGAAYf9h/2H/YQBhAGEAYv9i/2L/YgBiAGIAY/9j/2P/YwBjAGMA" MY_NL
        "ZP9k/2T/ZABkAGQAZf9l/2X/ZQBlAGUAZv9m/2b/ZgBmAGYAZ/9n/2f/ZwBnAGcA" MY_NL
        "aP9o/2j/aABoAGgAaf9p/2n/aQBpAGkAav9q/2r/agBqAGoAa/9r/2v/awBrAGsA" MY_NL
        "bP9s/2z/bABsAGwAbf9t/23/bQBtAG0Abv9u/27/bgBuAG4Ab/9v/2//bwBvAG8A" MY_NL
        "cP9w/3D/cABwAHAAcf9x/3H/cQBxAHEAcv9y/3L/cgByAHIAc/9z/3P/cwBzAHMA" MY_NL
        "dP90/3T/dAB0AHQAdf91/3X/dQB1AHUAdv92/3b/dgB2AHYAd/93/3f/dwB3AHcA" MY_NL
        "eP94/3j/eAB4AHgAef95/3n/eQB5AHkAev96/3r/egB6AHoAe/97/3v/ewB7AHsA" MY_NL
        "fP98/3z/fAB8AHwAff99/33/fQB9AH0Afv9+/37/fgB+AH4Af/9//3//fwB/AH8A" MY_NL
        "gP+A/4D/gACAAIAAgf+B/4H/gQCBAIEAgv+C/4L/ggCCAIIAg/+D/4P/gwCDAIMA" MY_NL
        "hP+E/4T/hACEAIQAhf+F/4X/hQCFAIUAhv+G/4b/hgCGAIYAh/+H/4f/hwCHAIcA" MY_NL
        "iP+I/4j/iACIAIgAif+J/4n/iQCJAIkAiv+K/4r/igCKAIoAi/+L/4v/iwCLAIsA" MY_NL
        "jP+M/4z/jACMAIwAjf+N/43/jQCNAI0Ajv+O/47/jgCOAI4Aj/+P/4//jwCPAI8A" MY_NL
        "kP+Q/5D/kACQAJAAkf+R/5H/kQCRAJEAkv+S/5L/kgCSAJIAk/+T/5P/kwCTAJMA" MY_NL
        "lP+U/5T/lACUAJQAlf+V/5X/lQCVAJUAlv+W/5b/lgCWAJYAl/+X/5f/lwCXAJcA" MY_NL
        "mP+Y/5j/mACYAJgAmf+Z/5n/mQCZAJkAmv+a/5r/mgCaAJoAm/+b/5v/mwCbAJsA" MY_NL
        "nP+c/5z/nACcAJwAnf+d/53/nQCdAJ0Anv+e/57/ngCeAJ4An/+f/5//nwCfAJ8A" MY_NL
        "oP+g/6D/oACgAKAAof+h/6H/oQChAKEAov+i/6L/ogCiAKIAo/+j/6P/owCjAKMA" MY_NL
        "pP+k/6T/pACkAKQApf+l/6X/pQClAKUApv+m/6b/pgCmAKYAp/+n/6f/pwCnAKcA" MY_NL
        "qP+o/6j/qACoAKgAqf+p/6n/qQCpAKkAqv+q/6r/qgCqAKoAq/+r/6v/qwCrAKsA" MY_NL
        "rP+s/6z/rACsAKwArf+t/63/rQCtAK0Arv+u/67/rgCuAK4Ar/+v/6//rwCvAK8A" MY_NL
        "sP+w/7D/sACwALAAsf+x/7H/sQCxALEAsv+y/7L/sgCyALIAs/+z/7P/swCzALMA" MY_NL
        "tP+0/7T/tAC0ALQAtf+1/7X/tQC1ALUAtv+2/7b/tgC2ALYAt/+3/7f/twC3ALcA" MY_NL
        "uP+4/7j/uAC4ALgAuf+5/7n/uQC5ALkAuv+6/7r/ugC6ALoAu/+7/7v/uwC7ALsA" MY_NL
        "vP+8/7z/vAC8ALwAvf+9/73/vQC9AL0Avv++/77/vgC+AL4Av/+//7//vwC/AL8A" MY_NL
        "wP/A/8D/wADAAMAAwf/B/8H/wQDBAMEAwv/C/8L/wgDCAMIAw//D/8P/wwDDAMMA" MY_NL
        "xP/E/8T/xADEAMQAxf/F/8X/xQDFAMUAxv/G/8b/xgDGAMYAx//H/8f/xwDHAMcA" MY_NL
        "yP/I/8j/yADIAMgAyf/J/8n/yQDJAMkAyv/K/8r/ygDKAMoAy//L/8v/ywDLAMsA" MY_NL
        "zP/M/8z/zADMAMwAzf/N/83/zQDNAM0Azv/O/87/zgDOAM4Az//P/8//zwDPAM8A" MY_NL
        "0P/Q/9D/0ADQANAA0f/R/9H/0QDRANEA0v/S/9L/0gDSANIA0//T/9P/0wDTANMA" MY_NL
        "1P/U/9T/1ADUANQA1f/V/9X/1QDVANUA1v/W/9b/1gDWANYA1//X/9f/1wDXANcA" MY_NL
        "2P/Y/9j/2ADYANgA2f/Z/9n/2QDZANkA2v/a/9r/2gDaANoA2//b/9v/2wDbANsA" MY_NL
        "3P/c/9z/3ADcANwA3f/d/93/3QDdAN0A3v/e/97/3gDeAN4A3//f/9//3wDfAN8A" MY_NL
        "4P/g/+D/4ADgAOAA4f/h/+H/4QDhAOEA4v/i/+L/4gDiAOIA4//j/+P/4wDjAOMA" MY_NL
        "5P/k/+T/5ADkAOQA5f/l/+X/5QDlAOUA5v/m/+b/5gDmAOYA5//n/+f/5wDnAOcA" MY_NL
        "6P/o/+j/6ADoAOgA6f/p/+n/6QDpAOkA6v/q/+r/6gDqAOoA6//r/+v/6wDrAOsA" MY_NL
        "7P/s/+z/7ADsAOwA7f/t/+3/7QDtAO0A7v/u/+7/7gDuAO4A7//v/+//7wDvAO8A" MY_NL
        "8P/w//D/8ADwAPAA8f/x//H/8QDxAPEA8v/y//L/8gDyAPIA8//z//P/8wDzAPMA" MY_NL
        "9P/0//T/9AD0APQA9f/1//X/9QD1APUA9v/2//b/9gD2APYA9//3//f/9wD3APcA" MY_NL
        "+P/4//j/+AD4APgA+f/5//n/+QD5APkA+v/6//r/+gD6APoA+//7//v/+wD7APsA" MY_NL
        "/P/8//z//AD8APwA/f/9//3//QD9AP0A/v/+//7//gD+AP4A/////////wD/AP8A";

    tstBase64(s_abData3, sizeof(s_abData3),
              s_szEnc3,  sizeof(s_szEnc3) - 1,
              0 /* fTextData */, 0 /* fNormalEnc */, "Test 3");

    /*
     * Summary.
     */
    if (!g_cErrors)
        RTPrintf("tstBase64: SUCCESS\n");
    else
        RTPrintf("tstBase64: FAILURE - %d errors\n", g_cErrors);
    return !!g_cErrors;
}

