/*
 * VARIANT test program
 *
 * Copyright 1998 Jean-Claude Cote
 *
 * The purpose of this program is validate the implementation
 * of the APIs related to VARIANTs. The validation is done
 * by comparing the results given by the Windows implementation
 * versus the Wine implementation.
 *
 * This program tests the creation/coercion/destruction of VARIANTs.
 *
 * The program does not currently test any API that takes
 * arguments of type: IDispatch, IUnknown, DECIMAL, CURRENCY.
 *
 * Since the purpose of this program is to compare the results
 * from Windows and Wine it is written so that with a simple
 * define it can be compiled either in Windows or Linux.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * NOTES
 *	 - The Variant structure in Windows has a non-named union. This means
 *	 the member of the union are accessible simply by doing
 *	 pVariant->pfltVal. With gcc it is not possible to have non-named
 *	 union so it has been named 'u'.  So it's members are accessible
 *	 using this name like so pVariant->u.pfltVal.  So if this program is
 *	 compiled in Windows the references to 'u' will need to be take out
 *       of this file.
 *
 *	 - Also the printf is a little different so the format specifiers may
 *	 need to be tweaked if this file is compile in Windows.
 *	 Printf is also different in that it continues printing numbers
 *	 even after there is no more significative digits left to print.  These
 *	 number are garbage and in windows they are set to zero but not
 *	 on Linux.
 *
 *	 - The VarDateFromStr is not implemented yet.
 *
 *	 - The date and floating point format may not be the exact same
 *	 format has the one inwindows depending on what the Internatinal
 *       setting are in windows.
 *
 */


#include <winsock.h>

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <float.h>
#include <time.h>

#include "wine/test.h"
#include "winbase.h"
#include "winuser.h"
#include "wingdi.h"
#include "winnls.h"
#include "winerror.h"
#include "winnt.h"

#include "wtypes.h"
#include "oleauto.h"


static HRESULT (WINAPI *pVarBstrFromI1)(CHAR,LCID,ULONG,BSTR*)=NULL;

static HRESULT (WINAPI *pVarI1FromBool)(VARIANT_BOOL,CHAR*)=NULL;
static HRESULT (WINAPI *pVarI1FromDate)(DATE,CHAR*)=NULL;
static HRESULT (WINAPI *pVarI1FromI4)(LONG,CHAR*)=NULL;
static HRESULT (WINAPI *pVarI1FromR8)(double,CHAR*)=NULL;
static HRESULT (WINAPI *pVarI1FromStr)(OLECHAR*,LCID,ULONG,CHAR*);
static HRESULT (WINAPI *pVarI1FromUI1)(BYTE,CHAR*)=NULL;

static HRESULT (WINAPI *pVarI2FromUI2)(USHORT,short*)=NULL;

static HRESULT (WINAPI *pVarUI2FromBool)(VARIANT_BOOL,USHORT*)=NULL;
static HRESULT (WINAPI *pVarUI2FromDate)(DATE,USHORT*)=NULL;
static HRESULT (WINAPI *pVarUI2FromI2)(short,USHORT*)=NULL;
static HRESULT (WINAPI *pVarUI2FromI4)(LONG,USHORT*);
static HRESULT (WINAPI *pVarUI2FromR8)(double,USHORT*)=NULL;
static HRESULT (WINAPI *pVarUI2FromStr)(OLECHAR*,LCID,ULONG,USHORT*)=NULL;

static HRESULT (WINAPI *pVarUI4FromBool)(VARIANT_BOOL,ULONG*)=NULL;
static HRESULT (WINAPI *pVarUI4FromDate)(DATE,ULONG*)=NULL;
static HRESULT (WINAPI *pVarUI4FromI2)(short,ULONG*)=NULL;
static HRESULT (WINAPI *pVarUI4FromR8)(double,ULONG*)=NULL;
static HRESULT (WINAPI *pVarUI4FromStr)(OLECHAR*,LCID,ULONG,ULONG*)=NULL;

/* When comparing floating point values we cannot expect an exact match
 * because the rounding errors depend on the exact algorithm.
 */
#define EQ_DOUBLE(a,b)     (fabs((a)-(b))<1e-14)

#define MAX_BUFFER  1024

static char* WtoA( OLECHAR* p )
{
    static char buffer[MAX_BUFFER];
    DWORD len = WideCharToMultiByte( CP_ACP, 0, p, -1, buffer+1, sizeof(buffer)-3, NULL, NULL );
    buffer[0] = '\"';
    buffer[len] = '\"';
    buffer[len+1] = 0;
    return buffer;
}

static OLECHAR* AtoW( char* p )
{
    OLECHAR *buffer;
    DWORD len = MultiByteToWideChar( CP_ACP, 0, p, -1, NULL, 0 );
    buffer = malloc( len * sizeof(OLECHAR) );
    MultiByteToWideChar( CP_ACP, 0, p, -1, buffer, len );
    return buffer;
}

static const struct _vartypes {
    int ind;
    HRESULT vcind1,vcind2,vcex1,vcex2;
    int todoind1,todoind2,todowcex1,todowcex2;
} vartypes[] = {
    {0, 0,          0x80070057, 0,          0x80020008,0,1 },
    {1, 0,          0x80070057, 0,          0x80020008,0,1 },
    {2, 0,          0,          0,          0x80020005 },
    {3, 0,          0,          0,          0x80020005 },
    {4, 0,          0,          0,          0x80020005 },
    {5, 0,          0,          0,          0x80020005 },
    {6, 0,          0,          0,          0x80020005 },
    {7, 0,          0,          0,          0x80020005 },
    {77,0x80020008, 0x80070057, 0,          0x80020005,0,1 },
    {78,0x80020008, 0x80070057, 0x80020005, 0x80020005,0,1 },
    {79,0x80020008, 0x80070057, 0x80020005, 0x80020005,0,1 },
    {80,0x80020008, 0x80070057, 0,          0x80020005,0,1 },
    {81,0x80020008, 0x80070057, 0x80020005, 0x80020005,0,1 },
    {82,0x80020008, 0x80070057, 0x80020005, 0x80020005,0,1 },
    {83,0x80020008, 0x80070057, 0,          0x80020005,0,1,1 },
    {84,0x80020008, 0x80070057, 0x80020008, 0x80020008,0,1,1,1 },
    {85,0x80020008, 0x80070057, 0,          0x80020005,0,1 },
    {86,0x80020008, 0x80070057, 0,          0x80020005,0,1 },
    {87,0x80020008, 0x80070057, 0,          0x80020005,0,1 },
    {88,0x80020008, 0x80070057, 0,          0x80020005,0,1 },
    {89,0x80020008, 0x80070057, 0,          0x80020005,0,1,1 },
    {90,0x80020008, 0x80070057, 0,          0x80020005,0,1,1 },
    {91,0x80020008, 0x80070057, 0,          0x80020005,0,1 },
    {92,0x80020008, 0x80070057, 0,          0x80020005,0,1 },
    {93,0x80020008, 0x80070057, 0x80020008, 0x80020008,0,1,1,1 },
    {94,0x80020008, 0x80070057, 0x80020008, 0x80020008,0,1,1,1 },
    {95,0x80020008, 0x80070057, 0x80020008, 0x80020008,0,1,1,1 },
    {96,0x80020008, 0x80070057, 0x80020008, 0x80020008,0,1,1,1 },
    {97,0x80020008, 0x80070057, 0x80020008, 0x80020008,0,1,1,1 },
    {98,0x80020008, 0x80070057, 0x80020008, 0x80020008,0,1,1,1 },
    {99,0x80020008, 0x80070057, 0x80020008, 0x80020008,0,1,1,1 },
};

static const char *strfromr8[] = {
"1",
"-1",
"21",
"-21",
"321",
"-321",
"4321",
"-4321",
"54321",
"-54321",
"654321",
"-654321",
"7654321",
"-7654321",
"87654321",
"-87654321",
"987654321",
"-987654321",
"1987654321",
"-1987654321",
"21987654321",
"-21987654321",
"321987654321",
"-321987654321",
"4321987654321",
"-4321987654321",
"54321987654321",
"-54321987654321",
"654321987654321",
"-654321987654321",
"7.65432198765432E+15",
"-7.65432198765432E+15",
"8.76543219876543E+16",
"-8.76543219876543E+16",
"9.87654321987654E+17",
"-9.87654321987654E+17",
"1.98765432198765E+18",
"-1.98765432198765E+18",
"2.19876543219877E+19",
"-2.19876543219877E+19",
"1",
"0",
"-1",
"1.2",
"0.2",
"-1.2",
"1.23",
"0.23",
"-1.23",
"1.234",
"0.234",
"-1.234",
"1.2345",
"0.2345",
"-1.2345",
"1.23456",
"0.23456",
"-1.23456",
"1.234567",
"0.234567",
"-1.234567",
"1.2345678",
"0.2345678",
"-1.2345678",
"1.23456789",
"0.23456789",
"-1.23456789",
"1.234567891",
"0.234567891",
"-1.234567891",
"1.2345678912",
"0.2345678912",
"-1.2345678912",
"1.23456789123",
"0.23456789123",
"-1.23456789123",
"1.234567891234",
"0.234567891234",
"-1.234567891234",
"1.2345678912345",
"0.2345678912345",
"-1.2345678912345",
"1.23456789123456",
"0.23456789123456",
"-1.23456789123456",
"1.23456789123457",
"0.234567891234567",
"-1.23456789123457",
"1.23456789123457",
"0.234567891234568",
"-1.23456789123457",
"1.23456789123457",
"0.234567891234568",
"-1.23456789123457",
"1.23456789123457",
"0.234567891234568",
"-1.23456789123457",
"1.23456789123457",
"0.234567891234568",
"-1.23456789123457",
"2",
"-2",
"22.2",
"-22.2",
"322.23",
"-322.23",
"4322.234",
"-4322.234",
"54322.2345",
"-54322.2345",
"654322.23456",
"-654322.23456",
"7654322.234567",
"-7654322.234567",
"87654322.2345678",
"-87654322.2345678",
"987654322.234568",
"-987654322.234568",
"1987654322.23457",
"-1987654322.23457",
"21987654322.2346",
"-21987654322.2346",
"321987654322.235",
"-321987654322.235",
"4321987654322.23",
"-4321987654322.23",
"54321987654322.2",
"-54321987654322.2",
"654321987654322",
"-654321987654322",
"7.65432198765432E+15",
"-7.65432198765432E+15",
"8.76543219876543E+16",
"-8.76543219876543E+16",
"9.87654321987654E+17",
"-9.87654321987654E+17",
"1.98765432198765E+18",
"-1.98765432198765E+18",
"2.19876543219877E+19",
"-2.19876543219877E+19",
/* r4 tests */
"1",
"-1",
"21",
"-21",
"321",
"-321",
"4321",
"-4321",
"54321",
"-54321",
"654321",
"-654321",
"7654321",
"-7654321",
"8.765432E+07",
"-8.765432E+07",
"9.876543E+08",
"-9.876543E+08",
"1.987654E+09",
"-1.987654E+09",
"1",
"0",
"-1",
"1.2",
"0.2",
"-1.2",
"1.23",
"0.23",
"-1.23",
"1.234",
"0.234",
"-1.234",
"1.2345",
"0.2345",
"-1.2345",
"1.23456",
"0.23456",
"-1.23456",
"1.234567",
"0.234567",
"-1.234567",
"1.234568",
"0.2345678",
"-1.234568",
"1.234568",
"0.2345679",
"-1.234568",
"1.234568",
"0.2345679",
"-1.234568",
"2",
"-2",
"22.2",
"-22.2",
"322.23",
"-322.23",
"4322.234",
"-4322.234",
"54322.23",
"-54322.23",
"654322.3",
"-654322.3",
"7654322",
"-7654322",
"8.765432E+07",
"-8.765432E+07",
"9.876543E+08",
"-9.876543E+08",
"1.987654E+09",
"-1.987654E+09",
};

/* These are the strings we use for the XxxFromStr tests.
 * The arrays that follow define the expected results for each type.
 */
static char* _pTestStrA[] = {
    "-2",
    "-1",
    "-0.51",
    "-0.5",
    "-0.49",
    "-0.0",
    "0.0",
    "0.49",
    "0.5",
    "0.51",
    "1",
    "127",
    "128",
    "129",
    "255",
    "256",
    "257",
    "32767",
    "32768",
    "-32768",
    "-32769",
    "16777216",
    "16777217",
    "-16777216",
    "16777217",
    "2147483647",
    "2147483648",
    "-2147483647",
    "-2147483648",

    "",
    " ",
    "1F",
    "1G",
    " 1 ",
    " 1 2 ",
    "1,2,3",
    "1 2 3",
    "1,2, 3",
    "1;2;3",
    "1.2.3",

    "0.",
    ".0",
    "0.1E12",
    "2.4,E1",
    "	+3.2,E1",
    "4E2.5",
    "	2E+2",
    "1 E+2",
    ".",
    ".E2",
    "1000000000000000000000000000000000000000000000000000000000000000",
    "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000001",
    "00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000001",
    "100000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000001",
    "0000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000001",
    "65535",
    "65535.5",
    "65536",
    "4294967295",
    "4294967296",

    "1 January 99",
    "December 31, 2078",
    "January 1, 1900",
    "January 2 1900",
    "11.11.1999",
    "11/11/1999",
    " 11 / 11 / 1999",
    "11/11/1999:11:11:1134",
    "11/11/1999 11:11:11:1",
    "\t1999/\t11/21 11 :11:11am",

    "11/11/1999 11:11:11Am",
    "11/11/1999 11:11:11PM",
    "11/11/199911:11:11PM",
    "11/11/1999 0:0:11am",
    "11/11/1999 11,11:11am",
    "11/11/1999 11:11:11am",
    "11/11/1999 11/11:11am",
    "11/11/1999 11:11AM",
    "11/11/1999 1AM",
    "11/11/1999 0AM",

    "11/11/1999 11:11:11",
    "11/13/1999 0AM",
    "13/13/1999 0AM",
    "13/11/1999 0AM",
    "11/33/1999 0AM",
    "11/11/1999 AM",
    "1/1/0 0AM",
    "1/1/-1 0AM",
    "1999 January 3 9AM",
    "1 January 1999 11AM",

    "4AM 11/11/1999",
    "4:22 11/11/1999 AM",
    " 1 1 /11/1999",
    "11-11/1999 11:11:11.12AM",
    "1999 January 3, 9AM",
    "December, 31, 2078",
    "December, 31, 2078,",
    "December, 31 2078",
    "11/99",
    "11-1999",

    "true",
    "True",
    "TRue",
    "TRUE",
    " TRUE",
    "FALSE ",
    "False",
    "JustSomeText",
    "Just Some Text",

    "1.5",
    "2.5",
    "3.5",
    "4.5",
};
#define NB_OLE_STRINGS (sizeof(_pTestStrA)/sizeof(*_pTestStrA))

static const struct _strret_date {
    HRESULT error;
    DATE retval;
    BOOL todo_rc;
    BOOL todo_val;
} strrets_DATE[NB_OLE_STRINGS] = {
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0,           0.0, 1 },
    { 0,           0.03402777777777, 1 },
    { 0,           0.00347222222222, 1 },
    { 0,           0.03541666666666, 1 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005,  0.0, 1 },
    { 0x80020005,  0.0, 1 },
    { 0x80020005 },
    { 0x80020005,  0.0, 1 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0,           37623.0, 1 },
    { 0,           37623.0 },
    { 0,           37623.0 },
    { 0,           37623.0 },
    { 0x80020005,  0.0, 1 },
    { 0,           0.04309027777777, 0, 1 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005,  0.0, 1 },
    { 0x80020005,  0.0, 1 },
    { 0x80020005 },
    { 0x80020005,  0.0, 1 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0,           36161.0, 1 },
    { 0,           65380.0 },
    { 0,           2.0},
    { 0,           3.0 },
    { 0x80020005,  0.0, 1 },
    { 0,           36475.0 },
    { 0,           36475.0 },
    { 0x80020005 },
    { 0x80020005 },
    { 0,           36485.466099537036, 1 },
    { 0,           36475.466099537036 },
    { 0,           36475.966099537036 },
    { 0x80020005,  0.0, 1 },
    { 0,           36475.000127314815 },
    { 0x80020005 },
    { 0,           36475.466099537036 },
    { 0x80020005 },
    { 0,           36475.465972222222 },
    { 0,           36475.041666666664, 1 },
    { 0,           36475.0, 1 },
    { 0,           36475.466099537036 },
    { 0,           36477.0, 1 },
    { 0x80020005 },
    { 0,           36477.0, 1 },
    { 0x80020005 },
    { 0x80020005,  0.0, 1 },
    { 0,           36526.0, 1 },
    { 0x80020005 },
    { 0,           36163.375000000000, 1 },
    { 0,           36161.458333333336, 1 },
    { 0,           36475.166666666664, 1 },
    { 0x80020005,  0.0, 1 },
    { 0x80020005 },
    { 0x80020005,  0.0, 1 },
    { 0x80020005 },
    { 0,           65380.0 },
    { 0x80020005,  0.0, 1 },
    { 0,           65380.0 },
    { 0,           36465.0, 1 },
    { 0,           36465.0, 1 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0,           0.045138888888889, 1 },
    { 0,           0.086805555555556, 1 },
    { 0,           0.128472222222222, 1 },
    { 0,           0.170138888888889, 1 },
};
static const struct _strret_b {
    HRESULT error;
    BOOL retval;
} strrets_B[NB_OLE_STRINGS] = {
    { 0,           VARIANT_TRUE },
    { 0,           VARIANT_TRUE },
    { 0,           VARIANT_TRUE },
    { 0,           VARIANT_TRUE },
    { 0,           VARIANT_TRUE },
    { 0,           VARIANT_FALSE },
    { 0,           VARIANT_FALSE },
    { 0,           VARIANT_TRUE },
    { 0,           VARIANT_TRUE },
    { 0,           VARIANT_TRUE },
    { 0,           VARIANT_TRUE },
    { 0,           VARIANT_TRUE },
    { 0,           VARIANT_TRUE },
    { 0,           VARIANT_TRUE },
    { 0,           VARIANT_TRUE },
    { 0,           VARIANT_TRUE },
    { 0,           VARIANT_TRUE },
    { 0,           VARIANT_TRUE },
    { 0,           VARIANT_TRUE },
    { 0,           VARIANT_TRUE },
    { 0,           VARIANT_TRUE },
    { 0,           VARIANT_TRUE },
    { 0,           VARIANT_TRUE },
    { 0,           VARIANT_TRUE },
    { 0,           VARIANT_TRUE },
    { 0,           VARIANT_TRUE },
    { 0,           VARIANT_TRUE },
    { 0,           VARIANT_TRUE },
    { 0,           VARIANT_TRUE },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0,           VARIANT_TRUE },
    { 0x80020005 },
    { 0,           VARIANT_TRUE },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0,           VARIANT_FALSE },
    { 0,           VARIANT_FALSE },
    { 0,           VARIANT_TRUE },
    { 0,           VARIANT_TRUE },
    { 0,           VARIANT_TRUE },
    { 0x80020005 },
    { 0,           VARIANT_TRUE },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0,           VARIANT_TRUE },
    { 0,           VARIANT_TRUE },
    { 0,           VARIANT_TRUE },
    { 0,           VARIANT_TRUE },
    { 0,           VARIANT_TRUE },
    { 0,           VARIANT_TRUE },
    { 0,           VARIANT_TRUE },
    { 0,           VARIANT_TRUE },
    { 0,           VARIANT_TRUE },
    { 0,           VARIANT_TRUE },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0,           VARIANT_TRUE },
    { 0,           VARIANT_TRUE },
    { 0,           VARIANT_TRUE },
    { 0,           VARIANT_TRUE },
    { 0x80020005 },
    { 0x80020005 },
    { 0,           VARIANT_FALSE },
    { 0x80020005 },
    { 0x80020005 },
    { 0,           VARIANT_TRUE },
    { 0,           VARIANT_TRUE },
    { 0,           VARIANT_TRUE },
    { 0,           VARIANT_TRUE },
};
static const struct _strret_r8 {
    HRESULT error;
    DOUBLE retval;
} strrets_R8[NB_OLE_STRINGS] = {
    { 0,           -2.000000 },
    { 0,           -1.000000 },
    { 0,           -0.510000 },
    { 0,           -0.500000 },
    { 0,           -0.490000 },
    { 0,           0.000000 },
    { 0,           0.000000 },
    { 0,           0.490000 },
    { 0,           0.500000 },
    { 0,           0.510000 },
    { 0,           1.000000 },
    { 0,           127.000000 },
    { 0,           128.000000 },
    { 0,           129.000000 },
    { 0,           255.000000 },
    { 0,           256.000000 },
    { 0,           257.000000 },
    { 0,           32767.000000 },
    { 0,           32768.000000 },
    { 0,           -32768.000000 },
    { 0,           -32769.000000 },
    { 0,           16777216.000000 },
    { 0,           16777217.000000 },
    { 0,           -16777216.000000 },
    { 0,           16777217.000000 },
    { 0,           2147483647.000000 },
    { 0,           2147483648.000000 },
    { 0,           -2147483647.000000 },
    { 0,           -2147483648.000000 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0,           1.000000 },
    { 0x80020005 },
    { 0,           123.000000 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0,           0.000000 },
    { 0,           0.000000 },
    { 0,           100000000000.000000 },
    { 0,           24.000000 },
    { 0,           32.000000 },
    { 0x80020005 },
    { 0,           200.000000 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0,           1e63 },
    { 0,           1.000000 },
    { 0,           1.000000 },
    { 0,           99999999999999997e183 },
    { 0,           1.000000 },
    { 0,           65535.000000 },
    { 0,           65535.500000 },
    { 0,           65536.000000 },
    { 0,           4294967295.000000 },
    { 0,           4294967296.000000 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0,           1.500000 },
    { 0,           2.500000 },
    { 0,           3.500000 },
    { 0,           4.500000 },
};
static const struct _strret_r4 {
    HRESULT error;
    FLOAT retval;
} strrets_R4[NB_OLE_STRINGS] = {
    { 0,           -2.000000F },
    { 0,           -1.000000F },
    { 0,           -0.510000F },
    { 0,           -0.500000F },
    { 0,           -0.490000F },
    { 0,           0.000000F },
    { 0,           0.000000F },
    { 0,           0.490000F },
    { 0,           0.500000F },
    { 0,           0.510000F },
    { 0,           1.000000F },
    { 0,           127.000000F },
    { 0,           128.000000F },
    { 0,           129.000000F },
    { 0,           255.000000F },
    { 0,           256.000000F },
    { 0,           257.000000F },
    { 0,           32767.000000F },
    { 0,           32768.000000F },
    { 0,           -32768.000000F },
    { 0,           -32769.000000F },
    { 0,           16777216.000000F },
    { 0,           16777216.000000F },
    { 0,           -16777216.000000F },
    { 0,           16777216.000000F },
    { 0,           2147483648.000000F },
    { 0,           2147483648.000000F },
    { 0,           -2147483648.000000F },
    { 0,           -2147483648.000000F },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0,           1.000000F },
    { 0x80020005 },
    { 0,           123.000000F },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0,           0.000000F },
    { 0,           0.000000F },
    { 0,           99999997952.000000F },
    { 0,           24.000000F },
    { 0,           32.000000F },
    { 0x80020005 },
    { 0,           200.000000F },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x8002000A },
    { 0,           1.000000F },
    { 0,           1.000000F },
    { 0x8002000A },
    { 0,           1.000000F },
    { 0,           65535.000000F },
    { 0,           65535.500000F },
    { 0,           65536.000000F },
    { 0,           4294967296.000000F },
    { 0,           4294967296.000000F },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0,           1.500000F },
    { 0,           2.500000F },
    { 0,           3.500000F },
    { 0,           4.500000F },
};
static const struct _strret_i4 {
    HRESULT error;
    LONG retval;
} strrets_I4[NB_OLE_STRINGS] = {
    { 0,           -2L },
    { 0,           -1L },
    { 0,           -1L },
    { 0,           0L },
    { 0,           0L },
    { 0,           0L },
    { 0,           0L },
    { 0,           0L },
    { 0,           0L },
    { 0,           1L },
    { 0,           1L },
    { 0,           127L },
    { 0,           128L },
    { 0,           129L },
    { 0,           255L },
    { 0,           256L },
    { 0,           257L },
    { 0,           32767L },
    { 0,           32768L },
    { 0,           -32768L },
    { 0,           -32769L },
    { 0,           16777216L },
    { 0,           16777217L },
    { 0,           -16777216L },
    { 0,           16777217L },
    { 0,           2147483647L },
    { 0x8002000A },
    { 0,           -2147483647L },
    { 0,           0x80000000L },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0,           1L },
    { 0x80020005 },
    { 0,           123L },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0,           0L },
    { 0,           0L },
    { 0x8002000A },
    { 0,           24L },
    { 0,           32L },
    { 0x80020005 },
    { 0,           200L },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x8002000A },
    { 0,           1L },
    { 0,           1L },
    { 0x8002000A },
    { 0,           1L },
    { 0,           65535L },
    { 0,           65536L },
    { 0,           65536L },
    { 0x8002000A },
    { 0x8002000A },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0,           2L },
    { 0,           2L },
    { 0,           4L },
    { 0,           4L },
};
static const struct _strret_i2 {
    HRESULT error;
    SHORT retval;
} strrets_I2[NB_OLE_STRINGS] = {
    { 0,           -2 },
    { 0,           -1 },
    { 0,           -1 },
    { 0,           0 },
    { 0,           0 },
    { 0,           0 },
    { 0,           0 },
    { 0,           0 },
    { 0,           0 },
    { 0,           1 },
    { 0,           1 },
    { 0,           127 },
    { 0,           128 },
    { 0,           129 },
    { 0,           255 },
    { 0,           256 },
    { 0,           257 },
    { 0,           32767 },
    { 0x8002000A },
    { 0,           -32768 },
    { 0x8002000A },
    { 0x8002000A },
    { 0x8002000A },
    { 0x8002000A },
    { 0x8002000A },
    { 0x8002000A },
    { 0x8002000A },
    { 0x8002000A },
    { 0x8002000A },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0,           1 },
    { 0x80020005 },
    { 0,           123 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0,           0 },
    { 0,           0 },
    { 0x8002000A },
    { 0,           24 },
    { 0,           32 },
    { 0x80020005 },
    { 0,           200 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x8002000A },
    { 0,           1 },
    { 0,           1 },
    { 0x8002000A },
    { 0,           1 },
    { 0x8002000A },
    { 0x8002000A },
    { 0x8002000A },
    { 0x8002000A },
    { 0x8002000A },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0,           2 },
    { 0,           2 },
    { 0,           4 },
    { 0,           4 },
};
static const struct _strret_i1 {
    HRESULT error;
    CHAR retval;
} strrets_I1[NB_OLE_STRINGS] = {
    { 0,           -2 },
    { 0,           -1 },
    { 0,           -1 },
    { 0,           0 },
    { 0,           0 },
    { 0,           0 },
    { 0,           0 },
    { 0,           0 },
    { 0,           0 },
    { 0,           1 },
    { 0,           1 },
    { 0,           127 },
    { 0x8002000A },
    { 0x8002000A },
    { 0x8002000A },
    { 0x8002000A },
    { 0x8002000A },
    { 0x8002000A },
    { 0x8002000A },
    { 0x8002000A },
    { 0x8002000A },
    { 0x8002000A },
    { 0x8002000A },
    { 0x8002000A },
    { 0x8002000A },
    { 0x8002000A },
    { 0x8002000A },
    { 0x8002000A },
    { 0x8002000A },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0,           1 },
    { 0x80020005 },
    { 0,           123 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0,           0 },
    { 0,           0 },
    { 0x8002000A },
    { 0,           24 },
    { 0,           32 },
    { 0x80020005 },
    { 0x8002000A },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x8002000A },
    { 0,           1 },
    { 0,           1 },
    { 0x8002000A },
    { 0,           1 },
    { 0x8002000A },
    { 0x8002000A },
    { 0x8002000A },
    { 0x8002000A },
    { 0x8002000A },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0,           2 },
    { 0,           2 },
    { 0,           4 },
    { 0,           4 },
};
static const struct _strret_u1 {
    HRESULT error;
    BYTE retval;
} strrets_U1[NB_OLE_STRINGS] = {
    { 0x8002000A },
    { 0x8002000A },
    { 0x8002000A },
    { 0,           0 },
    { 0,           0 },
    { 0,           0 },
    { 0,           0 },
    { 0,           0 },
    { 0,           0 },
    { 0,           1 },
    { 0,           1 },
    { 0,           0x7F },
    { 0,           0x80 },
    { 0,           0x81 },
    { 0,           0xFF },
    { 0x8002000A },
    { 0x8002000A },
    { 0x8002000A },
    { 0x8002000A },
    { 0x8002000A },
    { 0x8002000A },
    { 0x8002000A },
    { 0x8002000A },
    { 0x8002000A },
    { 0x8002000A },
    { 0x8002000A },
    { 0x8002000A },
    { 0x8002000A },
    { 0x8002000A },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0,           1 },
    { 0x80020005 },
    { 0,           0x7B },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0,           0 },
    { 0,           0 },
    { 0x8002000A },
    { 0,           0x18 },
    { 0,           0x20 },
    { 0x80020005 },
    { 0,           0xC8 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x8002000A },
    { 0,           1 },
    { 0,           1 },
    { 0x8002000A },
    { 0,           1 },
    { 0x8002000A },
    { 0x8002000A },
    { 0x8002000A },
    { 0x8002000A },
    { 0x8002000A },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0,           2 },
    { 0,           2 },
    { 0,           4 },
    { 0,           4 },
};

static const struct _strret_U2 {
    HRESULT error;
    WORD retval;
} strrets_U2[NB_OLE_STRINGS] = {
    { 0x8002000A },
    { 0x8002000A },
    { 0x8002000A },
    { 0,           0 },
    { 0,           0 },
    { 0,           0 },
    { 0,           0 },
    { 0,           0 },
    { 0,           0 },
    { 0,           1 },
    { 0,           1 },
    { 0,           127 },
    { 0,           128 },
    { 0,           129 },
    { 0,           255 },
    { 0,           256 },
    { 0,           257 },
    { 0,           32767 },
    { 0,           32768 },
    { 0x8002000A },
    { 0x8002000A },
    { 0x8002000A },
    { 0x8002000A },
    { 0x8002000A },
    { 0x8002000A },
    { 0x8002000A },
    { 0x8002000A },
    { 0x8002000A },
    { 0x8002000A },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0,           1 },
    { 0x80020005 },
    { 0,           123 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0,           0 },
    { 0,           0 },
    { 0x8002000A },
    { 0,           24 },
    { 0,           32 },
    { 0x80020005 },
    { 0,           200 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x8002000A },
    { 0,           1 },
    { 0,           1 },
    { 0x8002000A },
    { 0,           1 },
    { 0,           65535 },
    { 0x8002000A },
    { 0x8002000A },
    { 0x8002000A },
    { 0x8002000A },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0,           2 },
    { 0,           2 },
    { 0,           4 },
    { 0,           4 },
};

static const struct _strret_U4 {
    HRESULT error;
    DWORD retval;
} strrets_U4[NB_OLE_STRINGS] = {
    { 0x8002000A },
    { 0x8002000A },
    { 0x8002000A },
    { 0,           0 },
    { 0,           0 },
    { 0,           0 },
    { 0,           0 },
    { 0,           0 },
    { 0,           0 },
    { 0,           1 },
    { 0,           1 },
    { 0,           127 },
    { 0,           128 },
    { 0,           129 },
    { 0,           255 },
    { 0,           256 },
    { 0,           257 },
    { 0,           32767 },
    { 0,           32768 },
    { 0x8002000A },
    { 0x8002000A },
    { 0,           16777216 },
    { 0,           16777217 },
    { 0x8002000A },
    { 0,           16777217 },
    { 0,           2147483647 },
    { 0,           2147483648U },
    { 0x8002000A },
    { 0x8002000A },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0,           1 },
    { 0x80020005 },
    { 0,           123 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0,           0 },
    { 0,           0 },
    { 0x8002000A },
    { 0,           24 },
    { 0,           32 },
    { 0x80020005 },
    { 0,           200 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x8002000A },
    { 0,           1 },
    { 0,           1 },
    { 0x8002000A },
    { 0,           1 },
    { 0,           65535 },
    { 0,           65536 },
    { 0,           65536 },
    { 0,           4294967295U },
    { 0x8002000A },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0x80020005 },
    { 0,           2 },
    { 0,           2 },
    { 0,           4 },
    { 0,           4 },
};

START_TEST(vartest)
{
	HMODULE hdll;
	VARIANTARG va;
	VARIANTARG vb;
	VARIANTARG vc;
	VARIANTARG vd;
	VARIANTARG ve;

	HRESULT rc;
	LCID lcid;
	int theInt = 0;
	int* pInt = &theInt;
	VARIANT_BOOL b = 0;
	VARIANT_BOOL* pBool = &b;
	unsigned short uShort = 0;
	unsigned short* pUShort = &uShort;
	unsigned long uLong = 0;
	unsigned long* pULong = &uLong;
	CHAR theChar;
	CHAR* pChar = &theChar;
	BYTE byte;
	BYTE* pByte = &byte;
	short s = 0;
	short* pShort = &s;
	long Long = 0;
	long* pLong = &Long;
	float f = 0;
	float* pFloat = &f;
	double d = 0;
	double* pDouble = &d;

	BSTR bstr = NULL;
	int off, i = 0;
	OLECHAR* pOleChar[NB_OLE_STRINGS];

	for (i=0; i<NB_OLE_STRINGS;i++) {
		pOleChar[i]=AtoW(_pTestStrA[i]);
	}
	lcid = MAKELCID(MAKELANGID(LANG_ENGLISH, SUBLANG_ENGLISH_US), SORT_DEFAULT );

	/* Start testing the Low-Level API ( the coercions )
	 */
    hdll=LoadLibraryA("netapi32.dll");
    pVarI1FromBool=(void*)GetProcAddress(hdll,"VarI1FromBool");
    pVarI1FromDate=(void*)GetProcAddress(hdll,"VarI1FromDate");
    pVarI1FromI4=(void*)GetProcAddress(hdll,"VarI1FromI4");
    pVarI1FromR8=(void*)GetProcAddress(hdll,"VarI1FromR8");
    pVarI1FromStr=(void*)GetProcAddress(hdll,"VarI1FromStr");
    pVarI1FromUI1=(void*)GetProcAddress(hdll,"VarI1FromUI1");

    pVarI2FromUI2=(void*)GetProcAddress(hdll,"VarI2FromUI2");

    pVarUI2FromBool=(void*)GetProcAddress(hdll,"VarUI2FromBool");
    pVarUI2FromDate=(void*)GetProcAddress(hdll,"VarUI2FromDate");
    pVarUI2FromI2=(void*)GetProcAddress(hdll,"VarUI2FromI2");
    pVarUI2FromI4=(void*)GetProcAddress(hdll,"VarUI2FromI4");
    pVarUI2FromR8=(void*)GetProcAddress(hdll,"VarUI2FromR8");
    pVarUI2FromStr=(void*)GetProcAddress(hdll,"VarUI2FromStr");

    pVarUI4FromBool=(void*)GetProcAddress(hdll,"VarUI4FromBool");
    pVarUI4FromDate=(void*)GetProcAddress(hdll,"VarUI4FromDate");
    pVarUI4FromI2=(void*)GetProcAddress(hdll,"VarUI4FromI2");
    pVarUI4FromR8=(void*)GetProcAddress(hdll,"VarUI4FromR8");
    pVarUI4FromStr=(void*)GetProcAddress(hdll,"VarUI4FromStr");

	/* unsigned char from...
	 */
	trace( "======== Testing VarUI1FromXXX ========\n");

#define XOK "should return S_OK"
#define XOV "should return DISP_E_OVERFLOW"
	/* Crashes on Win95: VarUI1FromI2( 0, NULL ) */

	ok(VarUI1FromStr(NULL, lcid, 0, pByte) == DISP_E_TYPEMISMATCH,"should return DISP_E_TYPEMISMATCH");
	ok(S_OK == VarUI1FromI2( 0, pByte ), XOK);
	ok(*pByte == 0,"should give 0 byte value");

	ok(S_OK == VarUI1FromI2( 69, pByte ), XOK);
	ok(*pByte == 69,"should give 69 byte value");

	ok(S_OK == VarUI1FromI2( 70, pByte ), XOK);
	ok(*pByte == 70,"should give 70 byte value");

	ok(S_OK == VarUI1FromI2( 128, pByte ), XOK);
	ok(*pByte == 128,"should give 128 byte value");

	ok(S_OK == VarUI1FromI2( 255, pByte ), XOK);
	ok(*pByte == 255,"should give 255 byte value");

	ok(DISP_E_OVERFLOW == VarUI1FromI2( 256, pByte ), XOV);
	ok(DISP_E_OVERFLOW == VarUI1FromI2( 257, pByte ), XOV);

	ok(S_OK == VarUI1FromR8( 0.0, pByte ), XOK);
	ok(*pByte == 0,"0.0 float should be converted to 0");

	ok(S_OK == VarUI1FromR8( 69.33, pByte ), XOK);
	ok(*pByte == 0x45, "expected 69 (hex 0x45) as byte value");

	ok(S_OK == VarUI1FromR8( 69.66, pByte ), XOK);
	ok(*pByte == 0x46, "expected 70 (hex 0x46) as byte value");

	ok(DISP_E_OVERFLOW == VarUI1FromR8( -69.33, pByte ), XOV);
	ok(DISP_E_OVERFLOW == VarUI1FromR8( -69.66, pByte ), XOV);

	ok(S_OK == VarUI1FromR8( -0.5, pByte ), XOK);
	ok(*pByte == 0,"-0.5 should give return 0");

	ok(DISP_E_OVERFLOW == VarUI1FromR8( -0.51, pByte ), XOV);

	ok(S_OK == VarUI1FromR8( -0.49, pByte ), XOK);
	ok(*pByte == 0,"-0.49 should give return 0");

	ok(S_OK == VarUI1FromR8( 0.5, pByte ), XOK);
	ok(*pByte == 0,"0.5 should give return 0");

	ok(S_OK == VarUI1FromR8( 0.51, pByte ), XOK);
	ok(*pByte == 1,"0.51 should give return 1");

	ok(S_OK == VarUI1FromR8( 0.49, pByte ), XOK);
	ok(*pByte == 0,"0.49 should give return 0");

	ok(S_OK == VarUI1FromDate( 0.0, pByte ), XOK);
	ok(*pByte == 0,"0.0 date should give return 0");

	ok(S_OK == VarUI1FromDate( 69.33, pByte ), XOK);
	ok(*pByte == 0x45,"69.33 date should give return 0x45");

	ok(S_OK == VarUI1FromDate( 69.66, pByte ), XOK);
	ok(*pByte == 0x46,"69.66 date should give return 0x46");

	ok(DISP_E_OVERFLOW == VarUI1FromDate( -69.33, pByte ), XOV);

	ok(DISP_E_OVERFLOW == VarUI1FromDate( -69.66, pByte ), XOV);

	ok(S_OK == VarUI1FromBool( VARIANT_TRUE, pByte ), XOK);
	ok(*pByte == 0xff, "true should be converted to 0xff");

	ok(S_OK == VarUI1FromBool( VARIANT_FALSE, pByte ), XOK);
	ok(*pByte == 0, "false should be converted to 0");

	for (i = 0; i < NB_OLE_STRINGS; i++)
	{
            *pByte= 42;
            rc=VarUI1FromStr( pOleChar[i], lcid, 0, pByte );
            ok(rc == strrets_U1[i].error,
               "VarUI1FromStr([%d]=\"%s\") rc=%lx instead of %lx",
               i,_pTestStrA[i],rc,strrets_U1[i].error);
            if (rc == 0 && strrets_U1[i].error == 0) {
                ok(*pByte == strrets_U1[i].retval,
                   "VarUI1FromStr([%d]=\"%s\") got %02x instead of %02x",
                   i,_pTestStrA[i],*pByte,strrets_U1[i].retval);
            }
	}

	/* unsigned short from ... */
	trace( "======== Testing VarUI2FromXXX ========\n");

    if (pVarUI2FromI2) {
        ok(DISP_E_OVERFLOW == pVarUI2FromI2( -1, pUShort ), XOV);
        ok(S_OK == pVarUI2FromI2( 0, NULL ), XOK);

        ok(S_OK == pVarUI2FromI2( 0, pUShort ), XOK);
        ok(*pUShort == 0,"0 should be 0");
        ok(S_OK == pVarUI2FromI2( 69, pUShort ), XOK);
        ok(*pUShort == 69,"69 should be 69");
        ok(S_OK == pVarUI2FromI2( 70, pUShort ), XOK);
        ok(*pUShort == 70,"70 should be 70");

        ok(S_OK == pVarUI2FromI2( 128, pUShort ), XOK);
        ok(*pUShort == 128,"128 should be 128");
    }

    if (pVarUI2FromI4) {
        ok(S_OK == pVarUI2FromI4( 65535, pUShort ), XOK);
        ok(*pUShort == 65535,"65535 should be 65535");
        ok(DISP_E_OVERFLOW == pVarUI2FromI4( 65536, pUShort ), XOV);
        ok(DISP_E_OVERFLOW == pVarUI2FromI4( 65537, pUShort ), XOV);
    }

    if (pVarUI2FromR8) {
        ok(S_OK == pVarUI2FromR8( 0.0, pUShort ), XOK);
        ok(*pUShort == 0,"0.0 should be 0");
        ok(S_OK == pVarUI2FromR8( 69.33, pUShort ), XOK);
        ok(*pUShort == 69,"69.33 should be 69");
        ok(S_OK == pVarUI2FromR8( 69.66, pUShort ), XOK);
        ok(*pUShort == 70,"69.66 should be 70");

        ok(DISP_E_OVERFLOW == pVarUI2FromR8( -69.33, pUShort ), XOV);
        ok(DISP_E_OVERFLOW == pVarUI2FromR8( -69.66, pUShort ), XOV);

        ok(S_OK == pVarUI2FromR8( -0.5, pUShort ), XOK);
        ok(*pUShort == 0, "-0.5 -> 0");
        ok(DISP_E_OVERFLOW == pVarUI2FromR8( -0.51, pUShort ), XOV);
        ok(S_OK == pVarUI2FromR8( -0.49, pUShort ), XOK);
        ok(*pUShort == 0, "-0.49 -> 0");

        ok(S_OK == pVarUI2FromR8( 0.5, pUShort ), XOK);
        ok(*pUShort == 0,"0.5 should be 0");
        ok(S_OK == pVarUI2FromR8( 0.51, pUShort ), XOK);
        ok(*pUShort == 1,"0.51 should be 1");
        ok(S_OK == pVarUI2FromR8( 0.49, pUShort ), XOK);
        ok(*pUShort == 0,"0.49 should be 0");
    }

    if (pVarUI2FromDate) {
        ok(S_OK == pVarUI2FromDate( 0.0, pUShort ), XOK);
        ok(*pUShort == 0,"0.0 should be 0");
        ok(S_OK == pVarUI2FromDate( 69.33, pUShort ), XOK);
        ok(*pUShort == 69,"69.33 should be 69");
        ok(S_OK == pVarUI2FromDate( 69.66, pUShort ), XOK);
        ok(*pUShort == 70,"69.66 should be 70");
        ok(DISP_E_OVERFLOW == pVarUI2FromDate( -69.33, pUShort ), XOV);
        ok(DISP_E_OVERFLOW == pVarUI2FromDate( -69.66, pUShort ), XOV);
    }

    if (pVarUI2FromBool) {
        ok(S_OK == pVarUI2FromBool( VARIANT_TRUE, pUShort ), XOK);
        ok(*pUShort == 65535,"TRUE should be 65535");
        ok(S_OK == pVarUI2FromBool( VARIANT_FALSE, pUShort ), XOK);
        ok(*pUShort == 0,"FALSE should be 0");
    }

    if (pVarUI2FromStr) {
        ok(DISP_E_TYPEMISMATCH == pVarUI2FromStr( NULL, lcid, 0, pUShort ), "should return DISP_E_TYPEMISMATCH");

        for (i = 0; i < NB_OLE_STRINGS; i++)
        {
            *pUShort=42;
            rc=pVarUI2FromStr( pOleChar[i], lcid, 0, pUShort );
            ok(rc == strrets_U2[i].error,
               "VarUI2FromStr([%d]=\"%s\") rc=%lx instead of %lx",
               i,_pTestStrA[i],rc,strrets_U2[i].error);
            if (rc == 0 && strrets_U2[i].error == 0) {
                ok(*pUShort == strrets_U2[i].retval,
                   "VarUI2FromStr([%d]=\"%s\") got %u instead of %u",
                   i,_pTestStrA[i],*pUShort,strrets_U2[i].retval);
            }
        }
    }

	/* unsigned long from ...
	 */
	trace( "======== Testing VarUI4FromXXX ========\n");

    if (pVarUI4FromI2) {
        ok(S_OK == pVarUI4FromI2( 0, NULL ), XOK);

        ok(S_OK == pVarUI4FromI2( 0, pULong ), XOK);
        ok(*pULong == 0,"0 should be 0");
        ok(S_OK == pVarUI4FromI2( 69, pULong ), XOK);
        ok(*pULong == 69,"69 should be 69");

        ok(S_OK == pVarUI4FromI2( 70, pULong ), XOK);
        ok(*pULong == 70,"70 should be 70");

        ok(S_OK == pVarUI4FromI2( 128, pULong ), XOK);
        ok(*pULong == 128,"128 should be 128");
        ok(S_OK == pVarUI4FromI2( 255, pULong ), XOK);
        ok(*pULong == 255,"255 should be 255");
    }

    if (pVarUI4FromR8) {
        ok(S_OK == pVarUI4FromR8( 4294967295.0, pULong ), XOK);
        ok(*pULong == 4294967295U,"4294967295.0 should be 4294967295");
        ok(DISP_E_OVERFLOW == pVarUI4FromR8( 4294967296.0, pULong ), XOV);

        ok(S_OK == pVarUI4FromR8( 0.0, pULong ), XOK);
        ok(*pULong == 0,"0 should be 0");
        ok(S_OK == pVarUI4FromR8( 69.33, pULong ), XOK);
        ok(*pULong == 69,"69.33 should be 69");
        ok(S_OK == pVarUI4FromR8( 69.66, pULong ), XOK);
        ok(*pULong == 70,"69.66 should be 70");
        ok(DISP_E_OVERFLOW == pVarUI4FromR8( -69.33, pULong ), XOV);
        ok(DISP_E_OVERFLOW == pVarUI4FromR8( -69.66, pULong ), XOV);

        ok(S_OK == pVarUI4FromR8( -0.5, pULong ), XOK);
        ok(*pULong == 0,"-0.5 should be 0");

        ok(DISP_E_OVERFLOW == pVarUI4FromR8( -0.51, pULong ), XOV);

        ok(S_OK == pVarUI4FromR8( -0.49, pULong ), XOK);
        ok(*pULong == 0,"-0.49 should be 0");

        ok(S_OK == pVarUI4FromR8( 0.5, pULong ), XOK);
        ok(*pULong == 0,"0.5 should be 0");
        ok(S_OK == pVarUI4FromR8( 0.51, pULong ), XOK);
        ok(*pULong == 1,"0.51 should be 1");
        ok(S_OK == pVarUI4FromR8( 0.49, pULong ), XOK);
        ok(*pULong == 0,"0.49 should be 0");
    }

    if (pVarUI4FromDate) {
        ok(S_OK == pVarUI4FromDate( 0.0, pULong ), XOK);
        ok(*pULong == 0,"0.0 should be 0");
        ok(S_OK == pVarUI4FromDate( 69.33, pULong ), XOK);
        ok(*pULong == 69,"69.33 should be 69");
        ok(S_OK == pVarUI4FromDate( 69.66, pULong ), XOK);
        ok(*pULong == 70,"69.66 should be 70");
        ok(DISP_E_OVERFLOW == pVarUI4FromDate( -69.33, pULong ), XOV);
        ok(DISP_E_OVERFLOW == pVarUI4FromDate( -69.66, pULong ), XOV);
    }

    if (pVarUI4FromBool) {
        ok(S_OK == pVarUI4FromBool( VARIANT_TRUE, pULong ), XOK);
        ok(*pULong == 4294967295U, "TRUE should be 4294967295");
        ok(S_OK == pVarUI4FromBool( VARIANT_FALSE, pULong ), XOK);
        ok(*pULong == 0, "FALSE should be 0");
    }

    if (pVarUI4FromStr) {
        ok(DISP_E_TYPEMISMATCH == pVarUI4FromStr( NULL, lcid, 0, pULong ), "should erturn DISP_E_TYPEMISMATCH");
        for (i = 0; i < NB_OLE_STRINGS; i++)
        {
            *pULong=42;
            rc=pVarUI4FromStr( pOleChar[i], lcid, 0, pULong );
            ok(rc == strrets_U4[i].error,
               "VarUI4FromStr([%d]=\"%s\") rc=%lx instead of %lx",
               i,_pTestStrA[i],rc,strrets_U4[i].error);
            if (rc == 0 && strrets_U4[i].error == 0) {
                ok(*pULong == strrets_U4[i].retval,
                   "VarUI4FromStr([%d]=\"%s\") got %lu instead of %lu",
                   i,_pTestStrA[i],*pULong,strrets_U4[i].retval);
            }
        }
    }

	/* CHAR from ...
	 */
	trace( "======== Testing VarI1FromXXX ========\n");

    if (pVarI1FromBool) {
        ok(S_OK == pVarI1FromBool( VARIANT_TRUE, pByte ), XOK);
        ok(*pByte == 0xff,"true should be 0xff");

        ok(S_OK == pVarI1FromBool( VARIANT_TRUE, pChar ), XOK);
        ok(*pChar == -1, "TRUE should be -1");

        ok(S_OK == pVarI1FromBool( VARIANT_FALSE, pChar ), XOK);
        ok(*pChar == 0, "FALSE should be 0");
    }

    if (pVarI1FromUI1) {
        ok(DISP_E_OVERFLOW == pVarI1FromUI1( (unsigned char)32767, pChar ), XOV);
        ok(*pChar == 0, "should still be 0");
        ok(DISP_E_OVERFLOW == pVarI1FromUI1( (unsigned char)65535, pChar ), XOV);
        ok(*pChar == 0, "should still be 0");
    }

    if (pVarI1FromI4) {
        ok(DISP_E_OVERFLOW == pVarI1FromI4( 32767, pChar ), XOV);
        ok(*pChar == 0, "should still be 0");
        ok(DISP_E_OVERFLOW == pVarI1FromI4( 32768, pChar ), XOV);
        ok(*pChar == 0, "should still be 0");
        ok(DISP_E_OVERFLOW == pVarI1FromI4( -32768, pChar ), XOV);
        ok(*pChar == 0, "should still be 0");
        ok(DISP_E_OVERFLOW == pVarI1FromI4( -32769, pChar ), XOV);
        ok(*pChar == 0, "should still be 0");
    }

    if (pVarI1FromR8) {
        ok(S_OK == pVarI1FromR8( 69.33, pChar ), XOK);
        ok(*pChar == 69, "69.33 should be 69");
        ok(S_OK == pVarI1FromR8( 69.66, pChar ), XOK);
        ok(*pChar == 70, "69.66 should be 70");

        ok(S_OK == pVarI1FromR8( -69.33, pChar ), XOK);
        ok(*pChar == -69, "-69.33 should be -69");
        ok(S_OK == pVarI1FromR8( -69.66, pChar ), XOK);
        ok(*pChar == -70, "-69.66 should be -70");
    }

    if (pVarI1FromDate) {
        ok(S_OK == pVarI1FromDate( -69.66, pChar ), XOK);
        ok(*pChar == -70, "-69.66 should be -70");
    }

    if (pVarI1FromStr) {
        for (i = 0; i < NB_OLE_STRINGS; i++)
        {
            *pChar=42;
            rc=pVarI1FromStr( pOleChar[i], lcid, 0, pChar );
            ok(rc == strrets_I1[i].error,
               "VarI1FromStr([%d]=\"%s\") rc=%lx instead of %lx",
               i,_pTestStrA[i],rc,strrets_I1[i].error);
            if (rc == 0 && strrets_I1[i].error == 0) {
                ok(*pChar == strrets_I1[i].retval,
                   "VarI1FromStr([%d]=\"%s\") got %d instead of %d",
                   i,_pTestStrA[i],*pChar,strrets_I1[i].retval);
            }
        }
    }

	/* short from ...
	 */
	trace( "======== Testing VarI2FromXXX ========\n");

    if (pVarI2FromUI2) {
        ok(S_OK == pVarI2FromUI2( 32767, pShort ), XOK);
        ok(*pShort == 32767, "should be 32767");
        ok(DISP_E_OVERFLOW == pVarI2FromUI2( 65535, pShort ), XOV);
        ok(*pShort == 32767, "pShort should be unchanged");
    }

	ok(S_OK == VarI2FromI4( 32767, pShort ), XOK);
	ok(*pShort == 32767, "should be 32767");
	ok(DISP_E_OVERFLOW == VarI2FromI4( 32768, pShort ), XOV);
	ok(*pShort == 32767, "should still be 32767");
	ok(S_OK == VarI2FromI4( -32768, pShort ), XOK);
	ok(*pShort == -32768, "should be -32768");
	ok(DISP_E_OVERFLOW == VarI2FromI4( -32769, pShort ), XOV);
	ok(*pShort == -32768, "should still be -32768");

	ok(S_OK == VarI2FromR8( 69.33, pShort ), XOK);
	ok(*pShort == 69, "should be 69");
	ok(S_OK == VarI2FromR8( 69.66, pShort ), XOK);
	ok(*pShort == 70, "should be 70");
	ok(S_OK == VarI2FromR8( -69.33, pShort ), XOK);
	ok(*pShort == -69, "should be -69");
	ok(S_OK == VarI2FromR8( -69.66, pShort ), XOK);
	ok(*pShort == -70, "should be -70");
	ok(S_OK == VarI2FromDate( -69.66, pShort ), XOK);
	ok(*pShort == -70, "should be -70");

	for (i = 0; i < NB_OLE_STRINGS; i++)
	{
            *pShort=42;
            rc=VarI2FromStr( pOleChar[i], lcid, 0, pShort );
            ok(rc == strrets_I2[i].error,
               "VarI2FromStr([%d]=\"%s\") rc=%lx instead of %lx",
               i,_pTestStrA[i],rc,strrets_I2[i].error);
            if (rc == 0 && strrets_I2[i].error == 0) {
                ok(*pShort == strrets_I2[i].retval,
                   "VarI2FromStr([%d]=\"%s\") got %d instead of %d",
                   i,_pTestStrA[i],*pShort,strrets_I2[i].retval);
            }
	}

	/* long from ...
	 */
	trace( "======== Testing VarI4FromXXX ========\n");

	ok(S_OK == VarI4FromI2( 3, (long*)pInt ), XOK);
	ok(*pInt == 3,"should be 3");

	ok(S_OK == VarI4FromR8( 69.33, pLong ), XOK);
	ok(*pLong == 69,"should be 69");
	ok(S_OK == VarI4FromR8( 69.66, pLong ), XOK);
	ok(*pLong == 70,"should be 70");
	ok(S_OK == VarI4FromR8( -69.33, pLong ), XOK);
	ok(*pLong == -69,"should be -69");
	ok(S_OK == VarI4FromR8( -69.66, pLong ), XOK);
	ok(*pLong == -70,"should be -70");

	ok(S_OK == VarI4FromR8( 2147483647.0, pLong ), XOK);
	ok(*pLong == 2147483647,"should be 2147483647");
	ok(DISP_E_OVERFLOW == VarI4FromR8( 2147483648.0, pLong ), XOV);
	ok(*pLong == 2147483647,"should still be 2147483647");

	ok(S_OK == VarI4FromR8( -2147483647.0, pLong ), XOK);
	ok(*pLong == -2147483647,"should be -2147483647");
	ok(S_OK == VarI4FromR8( -2147483648.0, pLong ), XOK);
	ok(*pLong == 0x80000000L,"should be -2147483648");
	ok(DISP_E_OVERFLOW == VarI4FromR8( -2147483649.0, pLong ), XOV);
	ok(*pLong == 0x80000000L,"should still be -2147483648");
	ok(DISP_E_OVERFLOW == VarI4FromDate( -2147483649.0, pLong ), XOV);
	ok(*pLong == 0x80000000L,"should still be -2147483648");

	for (i = 0; i < NB_OLE_STRINGS; i++)
	{
            *pLong=42;
            rc=VarI4FromStr( pOleChar[i], lcid, 0, pLong );
            ok(rc == strrets_I4[i].error,
               "VarI4FromStr([%d]=\"%s\") rc=%lx instead of %lx",
               i,_pTestStrA[i],rc,strrets_I4[i].error);
            if (rc == 0 && strrets_I4[i].error == 0) {
                ok(*pLong == strrets_I4[i].retval,
                   "VarI4FromStr([%d]=\"%s\") got %ld instead of %ld",
                   i,_pTestStrA[i],*pLong,strrets_I4[i].retval);
            }
	}

	/* float from ...
	 */
	trace( "======== Testing VarR4FromXXX ========\n");

	ok(S_OK == VarR4FromI4( 16777216, pFloat ), XOK);
	ok(16777216.0 == *pFloat,"should be 16777216.0");

	ok(S_OK == VarR4FromI4( 16777217, pFloat ), XOK);
	ok(16777216.0 == *pFloat,"should be 16777216.0");
	ok(S_OK == VarR4FromI4( -16777216, pFloat ), XOK);
	ok(-16777216.0 == *pFloat,"should be -16777216.0");
	ok(S_OK == VarR4FromI4( -16777217, pFloat ), XOK);
	ok(-16777216.0 == *pFloat,"should be -16777216.0");

	ok(S_OK == VarR4FromR8( 16777216.0, pFloat ), XOK);
	ok(16777216.0 == *pFloat,"should be 16777216.0");
	ok(S_OK == VarR4FromR8( 16777217.0, pFloat ), XOK);
	ok(16777216.0 == *pFloat,"should be 16777216.0");
	ok(S_OK == VarR4FromR8( -16777216.0, pFloat ), XOK);
	ok(-16777216.0 == *pFloat,"should be -16777216.0");
	ok(S_OK == VarR4FromR8( -16777217.0, pFloat ), XOK);
	ok(-16777216.0 == *pFloat,"should be -16777216.0");

	ok(S_OK == VarR4FromR8( 16777218e31, pFloat ), XOK);
	ok(*pFloat == 167772177736353110000000000000000000000.000000,
		"should be 167772177736353110000000000000000000000.000000");
	ok(DISP_E_OVERFLOW == VarR4FromR8( 16777218e32, pFloat ), XOV);
	ok(*pFloat == 167772177736353110000000000000000000000.000000,
		"should still be 167772177736353110000000000000000000000.000000");
	ok(S_OK == VarR4FromDate( 16777218e31, pFloat ), XOK);
	ok(*pFloat == 167772177736353110000000000000000000000.000000,
		"should be 167772177736353110000000000000000000000.000000");

	for (i = 0; i < NB_OLE_STRINGS; i++)
	{
            *pFloat=42.0;
            rc=VarR4FromStr( pOleChar[i], lcid, 0, pFloat );
            ok(rc == strrets_R4[i].error,
                   "VarR4FromStr([%d]=\"%s\") rc=%lx instead of %lx",
                   i,_pTestStrA[i],rc,strrets_R4[i].error);
            if (rc == 0 && strrets_R4[i].error == 0) {
                ok(*pFloat == strrets_R4[i].retval,
                   "VarR4FromStr([%d]=\"%s\")  got %f instead of %f",
                   i,_pTestStrA[i],*pFloat,strrets_R4[i].retval);
            }
	}

	/* double from ...
	 */
	trace( "======== Testing VarR8FromXXX ========\n");

	ok(S_OK == VarR8FromDate( 900719925474099.0, pDouble ), XOK);
	ok(*pDouble == 900719925474099.000000,"should be 900719925474099.000000\n");
	for (i = 0; i < NB_OLE_STRINGS; i++)
	{
            *pDouble=42.0;
            rc=VarR8FromStr( pOleChar[i], lcid, 0, pDouble );
            ok(rc == strrets_R8[i].error,
               "VarR8FromStr([%d]=\"%s\") rc=%lx instead of %lx",
               i,_pTestStrA[i],rc,strrets_R8[i].error);
            if (rc == 0 && strrets_R8[i].error == 0) {
                ok(*pDouble == strrets_R8[i].retval,
                   "VarR8FromStr([%d]=\"%s\") got %.15f instead of %.15f",
                   i,_pTestStrA[i],*pDouble,strrets_R8[i].retval);
            }
	}

	/* date from ...
	 */
	trace( "======== Testing VarDateFromXXX ========\n");

	ok(S_OK == VarDateFromI4( 2958465, pDouble ), XOK);
	ok(*pDouble == 2958465.000000,"should be 2958465.000000");
    rc=VarDateFromI4( 2958466, pDouble );
    ok(rc==DISP_E_OVERFLOW || rc==DISP_E_TYPEMISMATCH /* Win95 */,
       "got %lx",rc);
	ok(*pDouble == 2958465.000000,"should still be 2958465.000000");
	ok(S_OK == VarDateFromI4( -657434, pDouble ), XOK);
	ok(*pDouble == -657434.000000,"should be -657434.000000");
    rc=VarDateFromI4( -657435, pDouble );
    ok(rc==DISP_E_OVERFLOW || rc==DISP_E_TYPEMISMATCH /* Win95 */,
       "got %lx",rc);
	ok(*pDouble == -657434.000000,"should still be -657434.000000");

	ok(S_OK == VarDateFromR8( 2958465.9999, pDouble ), XOK);
	ok(*pDouble == 2958465.999900, "should be 2958465.999900");
    rc=VarDateFromR8( 2958466, pDouble );
    ok(rc==DISP_E_OVERFLOW || rc==DISP_E_TYPEMISMATCH /* Win95 */,
       "got %lx",rc);
	ok(*pDouble == 2958465.999900, "should still be 2958465.999900");
	ok(S_OK == VarDateFromR8( -657434.9999, pDouble ), XOK);
	ok(*pDouble == -657434.999900,"should be -657434.999900");
    rc=VarDateFromR8( -657435, pDouble );
    ok(rc==DISP_E_OVERFLOW || rc==DISP_E_TYPEMISMATCH /* Win95 */,
       "got %lx",rc);
	ok(*pDouble == -657434.999900,"should still be -657434.999900");

	ok(S_OK == VarDateFromR8( 0.0, pDouble ), XOK);
	ok(*pDouble == 0.0,"0.0 should be 0.0");
	ok(S_OK == VarDateFromR8( 1.0, pDouble ), XOK);
	ok(*pDouble == 1.0,"1.0 should be 1.0");
	ok(S_OK == VarDateFromR8( 2.25, pDouble ), XOK);
	ok(*pDouble == 2.25,"2.25 should be 2.25");
	ok(S_OK == VarDateFromR8( -2.0, pDouble ), XOK);
	ok(*pDouble == -2.0,"-2.0 should be -2.0");

	/* Need some parsing function in Linux to emulate this...
	 * Still in progess.
	 */
	for (i = 0; i < NB_OLE_STRINGS; i++)
	{
            *pDouble=42.0;
            rc=VarDateFromStr( pOleChar[i], lcid, 0, pDouble );
            if (strrets_DATE[i].todo_rc) {
                todo_wine {
                    ok(rc == strrets_DATE[i].error,
                       "VarDateFromStr([%d]=\"%s\") rc= %lx instead of %lx",
                       i,_pTestStrA[i],rc,strrets_DATE[i].error);
                }
            } else {
                ok(rc == strrets_DATE[i].error,
                   "VarDateFromStr([%d]=\"%s\") rc= %lx instead of %lx",
                   i,_pTestStrA[i],rc,strrets_DATE[i].error);
            }
            if (rc == 0 && strrets_DATE[i].error == 0) {
	            if (strrets_DATE[i].todo_rc || strrets_DATE[i].todo_val) {
		            todo_wine {
                        ok(EQ_DOUBLE(*pDouble,strrets_DATE[i].retval),
                           "VarDateFromStr([%d]=\"%s\") got %.15f instead of %.15f",
                           i,_pTestStrA[i],*pDouble,strrets_DATE[i].retval);
                    }
	            } else {
                    ok(EQ_DOUBLE(*pDouble,strrets_DATE[i].retval),
                       "VarDateFromStr([%d]=\"%s\") got %.15f instead of %.15f",
                       i,_pTestStrA[i],*pDouble,strrets_DATE[i].retval);
                }
            }
	}
	/* bool from ...
	 */
	trace( "======== Testing VarBoolFromXXX ========\n");

	ok(S_OK == VarBoolFromI4( 0, pBool ), XOK);
	ok(VARIANT_FALSE == *pBool, "expected FALSE");
	ok(S_OK == VarBoolFromI4( 1, pBool ), XOK);
	ok(VARIANT_TRUE == *pBool, "expected TRUE");
	ok(S_OK == VarBoolFromI4( -1, pBool ), XOK);
	ok(VARIANT_TRUE == *pBool, "expected TRUE");
	ok(S_OK == VarBoolFromI4( 2, pBool ), XOK);
	ok(VARIANT_TRUE == *pBool, "expected TRUE");

	ok(S_OK == VarBoolFromUI1( ' ', pBool ), XOK);
	ok(VARIANT_TRUE == *pBool, "expected TRUE");
	ok(S_OK == VarBoolFromUI1( '\0', pBool ), XOK);
	ok(VARIANT_FALSE == *pBool, "expected FALSE");
	ok(S_OK == VarBoolFromUI1( 0x0000, pBool ), XOK);
	ok(VARIANT_FALSE == *pBool, "expected FALSE");
	ok(S_OK == VarBoolFromUI1( (unsigned char)0xFFF, pBool ), XOK);
	ok(VARIANT_TRUE == *pBool, "expected TRUE");
	ok(S_OK == VarBoolFromUI1( (unsigned char)0xFFFF, pBool ), XOK);
	ok(VARIANT_TRUE == *pBool, "expected TRUE");

	ok(S_OK == VarBoolFromR8( 0.0, pBool ), XOK);
	ok(VARIANT_FALSE == *pBool, "expected FALSE");
	ok(S_OK == VarBoolFromR8( 1.1, pBool ), XOK);
	ok(VARIANT_TRUE == *pBool, "expected TRUE");
	ok(S_OK == VarBoolFromR8( 0.5, pBool ), XOK);
	ok(VARIANT_TRUE == *pBool, "expected TRUE");
	ok(S_OK == VarBoolFromR8( 0.49, pBool ), XOK);
	ok(VARIANT_TRUE == *pBool, "expected TRUE");
	ok(S_OK == VarBoolFromR8( 0.51, pBool ), XOK);
	ok(VARIANT_TRUE == *pBool, "expected TRUE");
	ok(S_OK == VarBoolFromR8( -0.5, pBool ), XOK);
	ok(VARIANT_TRUE == *pBool, "expected TRUE");
	ok(S_OK == VarBoolFromR8( -0.49, pBool ), XOK);
	ok(VARIANT_TRUE == *pBool, "expected TRUE");
	ok(S_OK == VarBoolFromR8( -0.51, pBool ), XOK);
	ok(VARIANT_TRUE == *pBool, "expected TRUE");


	for (i = 0; i < NB_OLE_STRINGS; i++)
	{
            *pBool=42;
            rc=VarBoolFromStr( pOleChar[i], lcid, 0, pBool );
            ok(rc == strrets_B[i].error,
               "VarBoolFromStr([%d]=\"%s\") rc=%lx instead of %lx",
               i,_pTestStrA[i],rc,strrets_B[i].error);
            if (rc == 0 && strrets_B[i].error == 0) {
                ok(*pBool == strrets_B[i].retval,
                   "VarBoolFromStr([%d]=\"%s\") got %x instead of %x",
                   i,_pTestStrA[i],*pBool,strrets_B[i].retval);
            }
	}

	/* BSTR from ...
	 */
	trace( "======== Testing VarBSTRFromXXX ========\n");

	/* integers...
	 */
    if (pVarBstrFromI1) {
        ok(S_OK == pVarBstrFromI1( -100, 0, 0, &bstr ), XOK);
        ok(!strcmp(WtoA(bstr),"\"-100\""),"should be string -100");
    }

	ok(S_OK == VarBstrFromUI1( 0x5A, 0, 0, &bstr ), XOK);
	ok(!strcmp(WtoA(bstr),"\"90\""),"should be string 90");
	ok(S_OK == VarBstrFromI4( 2958465, 0, 0, &bstr ), XOK);
	ok(!strcmp(WtoA(bstr),"\"2958465\""),"should be string 2958465");

	/* reals...
	 */
	off = 0;
	d=0;
	for( i=0; i<20; i++ )
	{
	        char xval[80];
		/* add an integer to the real number
		 */
		d += ((i%9)+1) * pow( 10, i );

		ok(S_OK == VarBstrFromR8( d, lcid, 0, &bstr ), XOK);
		sprintf(xval,"\"%s\"",strfromr8[off]);
		ok(!strcmp(xval,WtoA(bstr)),
                   "d is %.15f, should be cvt. to %s, but return val is %s",
                   d,strfromr8[off],WtoA(bstr));
                off++;

		ok(S_OK == VarBstrFromR8( -d, lcid, 0, &bstr ), XOK);
		sprintf(xval,"\"%s\"",strfromr8[off]);
		ok(!strcmp(xval,WtoA(bstr)),
                   "d is %.15f, should be cvt. to %s, but return val is %s",
                   -d,strfromr8[off],WtoA(bstr));
                off++;
	}
	d=0;
	for( i=0; i<20; i++ )
	{
	        char xval[80];
		/* add a decimal to the real number
		 */
		d += ((i%9)+1) * pow( 10, (i*-1) );
		ok(S_OK == VarBstrFromR8( d, lcid, 0, &bstr ), XOK);
		sprintf(xval,"\"%s\"",strfromr8[off]);
		ok(!strcmp(xval,WtoA(bstr)),
                   "d is %.15f, should be cvt. to %s, but return val is %s",
                   d,strfromr8[off],WtoA(bstr));
                off++;
		ok(S_OK == VarBstrFromR8( d-1, lcid, 0, &bstr ), XOK);
		sprintf(xval,"\"%s\"",strfromr8[off]);
		ok(!strcmp(xval,WtoA(bstr)),
                   "d is %.15f, should be cvt. to %s, but return val is %s",
                   d-1,strfromr8[off],WtoA(bstr));
                off++;
		ok(S_OK == VarBstrFromR8( -d, lcid, 0, &bstr ), XOK);
		sprintf(xval,"\"%s\"",strfromr8[off]);
		ok(!strcmp(xval,WtoA(bstr)),
                   "d is %.15f, should be cvt. to %s, but return val is %s",
                   -d,strfromr8[off],WtoA(bstr));
                off++;
	}

	d=0;
	for( i=0; i<20; i++ )
	{
	        char xval[80];
		/* add an integer to the real number
		 */
		d += ((i%9)+1) * pow( 10, i );
		/* add a decimal to the real number
		 */
		d += ((i%9)+1) * pow( 10, (i*-1) );
		ok(S_OK == VarBstrFromR8( d, lcid, 0, &bstr ), XOK);
		sprintf(xval,"\"%s\"",strfromr8[off]);
		ok(!strcmp(xval,WtoA(bstr)),
                   "d is %.15f, should be cvt. to %s, but return val is %s",
                   d,strfromr8[off],WtoA(bstr));
                        off++;
		ok(S_OK == VarBstrFromR8( -d, lcid, 0, &bstr ), XOK);
		sprintf(xval,"\"%s\"",strfromr8[off]);
		ok(!strcmp(xval,WtoA(bstr)),
                   "d is %.15f, should be cvt. to %s, but return val is %s",
                   -d,strfromr8[off],WtoA(bstr));
                        off++;
	}

	d=0;
	for( i=0; i<10; i++ )
	{
	        char xval[80];
		/* add an integer to the real number
		 */
		d += ((i%9)+1) * pow( 10, i );
		ok(S_OK == VarBstrFromR4( (float)d, lcid, 0, &bstr ), XOK);
		sprintf(xval,"\"%s\"",strfromr8[off]);
		ok(!strcmp(xval,WtoA(bstr)),
                   "%d: d is %.8f, should be cvt. to %s, but return val is %s",
                   i,d,strfromr8[off],WtoA(bstr));
                off++;
		ok(S_OK == VarBstrFromR4( (float)-d, lcid, 0, &bstr ), XOK);
		sprintf(xval,"\"%s\"",strfromr8[off]);
		ok(!strcmp(xval,WtoA(bstr)),
                   "%d: d is %.8f, should be cvt. to %s, but return val is %s",
                   i,-d,strfromr8[off],WtoA(bstr));
                off++;
	}
	d=0;
	for( i=0; i<10; i++ )
	{
	        char xval[80];
		/* add a decimal to the real number
		 */
		d += ((i%9)+1) * pow( 10, (i*-1) );
		ok(S_OK == VarBstrFromR4( (float)d, lcid, 0, &bstr ), XOK);
		sprintf(xval,"\"%s\"",strfromr8[off]);
		ok(!strcmp(xval,WtoA(bstr)),
                   "%d: d is %.8f, should be cvt. to %s, but return val is %s",
                   i,d,strfromr8[off],WtoA(bstr));
                off++;
		ok(S_OK == VarBstrFromR4( (float)(d-1), lcid, 0, &bstr ), XOK);
		sprintf(xval,"\"%s\"",strfromr8[off]);
		ok(!strcmp(xval,WtoA(bstr)),
                   "%d: d is %.8f, should be cvt. to %s, but return val is %s",
                   i,d-1,strfromr8[off],WtoA(bstr));
                off++;
		ok(S_OK == VarBstrFromR4( (float)-d, lcid, 0, &bstr ), XOK);
		sprintf(xval,"\"%s\"",strfromr8[off]);
                ok(!strcmp(xval,WtoA(bstr)),
                   "%d: d is %.8f, should be cvt. to %s, but return val is %s",
                   i,-d,strfromr8[off],WtoA(bstr));
                off++;
	}

	d=0;
	for( i=0; i<10; i++ )
	{
        static int istodo[10]={0,0,0,0,0,1,0,0,0,0};
	    char xval[80];
		/* add an integer to the real number
		 */
		d += ((i%9)+1) * pow( 10, i );
		/* add a decimal to the real number
		 */
		d += ((i%9)+1) * pow( 10, (i*-1) );
		ok(S_OK == VarBstrFromR4( (float)d, lcid, 0, &bstr ), XOK);
		sprintf(xval,"\"%s\"",strfromr8[off]);
        if (istodo[i]) {
            todo_wine {
                ok(!strcmp(xval,WtoA(bstr)),
                       "%d: d is %.8f, should be cvt. to %s, but return val is %s",
                       i,d,strfromr8[off],WtoA(bstr));
            }
        } else {
            ok(!strcmp(xval,WtoA(bstr)),
                   "%d: d is %.8f, should be cvt. to %s, but return val is %s",
                   i,d,strfromr8[off],WtoA(bstr));
        }
        off++;
		ok(S_OK == VarBstrFromR4( (float)-d, lcid, 0, &bstr ), XOK);
		sprintf(xval,"\"%s\"",strfromr8[off]);
        if (istodo[i]) {
            todo_wine {
		        ok(!strcmp(xval,WtoA(bstr)),
                           "%d: d is %.8f, should be cvt. to %s, but return val is %s",
                           i,-d,strfromr8[off],WtoA(bstr));
            }
        } else {
		    ok(!strcmp(xval,WtoA(bstr)),
                       "%d: d is %.8f, should be cvt. to %s, but return val is %s",
                       i,-d,strfromr8[off],WtoA(bstr));
        }
        off++;
	}

	ok(S_OK == VarBstrFromBool( 0x00, lcid, 0, &bstr ), XOK);
	ok(!strcmp(WtoA(bstr),"\"False\""),"should be 'False'");
	ok(S_OK == VarBstrFromBool( 0xFF, lcid, 0, &bstr ), XOK);
	ok(!strcmp(WtoA(bstr),"\"True\""),"should be 'True'");

	ok(S_OK == VarBstrFromDate( 0.0, lcid, 0, &bstr ), XOK);
	todo_wine {
	    ok(!strcmp(WtoA(bstr),"\"12:00:00 AM\""),
               "should be '12:00:00 AM', but is %s\n",WtoA(bstr));
	}

	ok(S_OK == VarBstrFromDate( 3.34, lcid, 0, &bstr ), XOK);
	todo_wine {
	    ok(strcmp(WtoA(bstr),"\"1/2/1900 8:09:36 AM\"")==0 ||
           strcmp(WtoA(bstr),"\"1/2/00 8:09:36 AM\"")==0 /* Win95 */,
               "should be '1/2/1900 8:09:36 AM', but is %s\n",WtoA(bstr));
	}

	ok(S_OK == VarBstrFromDate( 3339.34, lcid, 0, &bstr ), XOK);
	todo_wine {
	    ok(strcmp(WtoA(bstr),"\"2/20/1909 8:09:36 AM\"")==0 ||
           strcmp(WtoA(bstr),"\"2/20/09 8:09:36 AM\"")==0 /* Win95 */,
               "should be '2/20/1909 8:09:36 AM', but is %s\n",WtoA(bstr));
	}

	ok(S_OK == VarBstrFromDate( 365.00, lcid, 0, &bstr ), XOK);
	todo_wine {
	    ok(strcmp(WtoA(bstr),"\"12/30/1900\"")==0 ||
           strcmp(WtoA(bstr),"\"12/30/00\"")==0 /* Win95 */,
               "should be '12/30/1900', but is %s\n",WtoA(bstr));
	}

	ok(S_OK == VarBstrFromDate( 365.25, lcid, 0, &bstr ), XOK);
	todo_wine {
	    ok(strcmp(WtoA(bstr),"\"12/30/1900 6:00:00 AM\"")==0 ||
           strcmp(WtoA(bstr),"\"12/30/00 6:00:00 AM\"")==0 /* Win95 */,
               "should be '12/30/1900 6:00:00 AM', but is %s\n",WtoA(bstr));
	}

	ok(S_OK == VarBstrFromDate( 1461.0, lcid, 0, &bstr ), XOK);
	todo_wine {
	    ok(strcmp(WtoA(bstr),"\"12/31/1903\"")==0 ||
           strcmp(WtoA(bstr),"\"12/31/03\"")==0 /* Win95 */,
               "should be '12/31/1903', but is %s\n",WtoA(bstr));
	}

	ok(S_OK == VarBstrFromDate( 1461.5, lcid, 0, &bstr ), XOK);
	todo_wine {
	    ok(strcmp(WtoA(bstr),"\"12/31/1903 12:00:00 PM\"")==0 ||
           strcmp(WtoA(bstr),"\"12/31/03 12:00:00 PM\"")==0 /* Win95 */,
               "should be '12/31/1903 12:00:00 PM', but is %s\n",WtoA(bstr));
	}

	/* Test variant API...
	 */
	trace( "======== Testing Hi-Level Variant API ========\n");

	bstr = SysAllocString( pOleChar[4] );

	VariantClear( &va );

	VariantInit( &va );
	VariantInit( &vb );
	VariantInit( &vc );
	VariantInit( &vd );
	VariantInit( &ve );

	V_VT(&va) = VT_BSTR;
	V_UNION(&va,bstrVal) = bstr;
	ok(S_OK == VariantClear( &va ), XOK);
	SysFreeString( bstr );
	SysFreeString( bstr );

	ok(S_OK == VariantCopy( &vb, &va ), XOK);
	ok(S_OK == VariantClear( &vb ), XOK);
	ok(S_OK == VariantClear( &va ), XOK);

	V_VT(&va) = VT_R8;
	d = 4.123;
	V_UNION(&va,dblVal) = d;
	ok(S_OK == VariantCopy( &va, &va ), XOK);
	ok(V_R8(&va) == 4.123,"should be 4.123");

	V_VT(&va) = VT_R8 | VT_BYREF;
	d = 31.123;
	V_UNION(&va,pdblVal) = &d;
	ok(S_OK == VariantCopyInd( &va, &va ), XOK);
	ok(V_R8(&va) == 31.123,"should be 31.123");

	V_VT(&va) = VT_R8;
	d = 1.123;
	V_UNION(&va,dblVal) = d;
	ok(S_OK == VariantCopy( &vb, &va ), XOK);
	ok(V_R8(&vb) == 1.123,"should be 1.123");

	V_VT(&va) = VT_R8 | VT_BYREF;
	d = 123.123;
	V_UNION(&va,pdblVal) = &d;
	ok(S_OK == VariantCopy( &vb, &va ), XOK);
	ok(*(V_R8REF(&vb)) == 123.123,"should be 123.123");

	V_VT(&va) = VT_R8 | VT_BYREF;
	d = 111.2;
	V_UNION(&va,pdblVal) = &d;
	ok(S_OK == VariantCopyInd( &vb, &va ), XOK);
	ok(V_R8(&vb) == 111.2,"should be 111.2");

	V_VT(&va) = VT_R8 | VT_BYREF;
	d = 1211.123453;
	V_UNION(&va,pdblVal) = &d;
	ok(S_OK == VariantChangeTypeEx( &va, &va, lcid, 0, VT_I2 ), XOK);
	ok(V_VT(&va) == VT_I2,"should be type VT_I2");

	V_VT(&va) = VT_I4;
	V_UNION(&va,intVal) = 4;
	ok(S_OK == VariantChangeTypeEx(&vb, &va, lcid, 0, VT_BSTR ), XOK);
	ok(!strcmp(WtoA(V_BSTR(&vb)),"\"4\""),"should be 4");

	V_VT(&va) = VT_DATE;
	V_UNION(&va,date) = 34465.332431;
	ok(S_OK == VariantChangeTypeEx(&vb, &va, lcid, 0, VT_BSTR ), XOK);
	todo_wine {
	    ok(strcmp(WtoA(V_BSTR(&vb)),"\"5/11/1994 7:58:42 AM\"")==0 ||
           strcmp(WtoA(V_BSTR(&vb)),"\"5/11/94 7:58:42 AM\"")==0 /* Win95 */,
           "should be 5/11/94 7:58:42 AM got %s",WtoA(V_BSTR(&vb)));
	}

	bstr = pOleChar[4];
	V_VT(&va) = VT_BSTR;
	V_UNION(&va,bstrVal) = bstr;
	ok(S_OK == VariantChangeTypeEx(&vb, &va, lcid, 0, VT_R8 ), XOK);
	ok(V_R8(&vb) == -0.490000,"should be -0.49");

	V_VT(&vc) = VT_BSTR | VT_BYREF;
	V_UNION(&vc,pbstrVal) = &bstr;
	V_VT(&vb) = VT_VARIANT | VT_BYREF;
	V_UNION(&vb,pvarVal) = &vc;
	V_VT(&va) = VT_VARIANT | VT_BYREF;
	V_UNION(&va,pvarVal) = &vb;
	ok(E_INVALIDARG == VariantCopyInd( &vd, &va ), "expect E_INVALIDARG");

	/* test what happens when bad vartypes are passed in */
	trace( "======== Testing different VARTYPES ========\n" );

	for( i=0; i<sizeof(vartypes)/sizeof(vartypes[0]); i++ )
	{
		    /* Trying to use variants that are set to be BSTR but
		     * do not contain a valid pointer makes the program crash
		     * in Windows so we will skip those. We do not need them
		     * anyway to illustrate the behavior.
		     */
		    V_VT(&va) = vartypes[i].ind;
		    d = 4.123;
		    V_UNION(&va,dblVal) = d;
		    rc = VariantCopyInd( &vb, &va );
            if (vartypes[i].todoind1) {
                todo_wine {
		            ok(vartypes[i].vcind1 == rc,
                               "%d: vt %d, return value %lx, expected was %lx",
                               i,vartypes[i].ind,rc,vartypes[i].vcind1);
                }
            } else {
		        ok(vartypes[i].vcind1 == rc,
                           "%d: vt %d, return value %lx, expected was %lx",
                           i,vartypes[i].ind,rc,vartypes[i].vcind1);
            }
		    V_VT(&va) = vartypes[i].ind | VT_BYREF;
		    d = 4.123;
		    V_UNION(&va,pdblVal) = &d;
		    rc = VariantCopyInd( &vb, &va );
            if (vartypes[i].todoind2) {
                todo_wine {
		            ok(vartypes[i].vcind2 == rc,
                               "%d: vt %d, return value %lx, expected was %lx",
                               i,vartypes[i].ind,rc,vartypes[i].vcind2);
                }
            } else {
		        ok(vartypes[i].vcind2 == rc,
                           "%d: vt %d, return value %lx, expected was %lx",
                           i,vartypes[i].ind,rc,vartypes[i].vcind2);
            }
		    V_VT(&va) = VT_R8;
		    d = 4.123;
		    V_UNION(&va,dblVal) = d;
		    rc = VariantChangeTypeEx( &vb, &va, lcid, 0, (VARTYPE)i );
            if (vartypes[i].todowcex1) {
                todo_wine {
		            ok(vartypes[i].vcex1 == rc || rc == DISP_E_BADVARTYPE,
                               "%d: vt %d, return value %lx, expected was %lx",
                               i,vartypes[i].ind,rc,vartypes[i].vcex1);
                }
            } else {
		        ok(vartypes[i].vcex1 == rc || rc == DISP_E_BADVARTYPE,
                           "%d: vt %d, return value %lx, expected was %lx",
                           i,vartypes[i].ind,rc,vartypes[i].vcex1);
            }
		    V_VT(&va) = VT_R8;
		    d = 4.123;
		    V_UNION(&va,dblVal) = d;
		    rc = VariantChangeTypeEx( &vb, &va, lcid, 0, (VARTYPE)(i | VT_BYREF) );
            if (vartypes[i].todowcex2) {
                todo_wine {
		            ok(vartypes[i].vcex2 == rc || rc == DISP_E_BADVARTYPE,
                               "%d: vt %d, return value %lx, expected was %lx",
                               i,vartypes[i].ind,rc,vartypes[i].vcex2);
                }
            } else {
		        ok(vartypes[i].vcex2 == rc || rc == DISP_E_BADVARTYPE,
                           "%d: vt %d, return value %lx, expected was %lx",
                           i,vartypes[i].ind,rc,vartypes[i].vcex2);
            }

		V_VT(&va) = 99;
		d = 4.123;
		V_UNION(&va,dblVal) = d;
		ok(DISP_E_BADVARTYPE == VariantClear( &va ), "should give DISP_E_BADVARTYPE");
	}
	VariantClear( &va );
	VariantClear( &vb );
	VariantClear( &vc );
	VariantClear( &vd );
	VariantClear( &ve );
	/* There is alot of memory leaks but this is simply a test program.
	 */
}
