/* Unit test suite for edit control.
 *
 * Copyright 2004 Vitaliy Margolen
 * Copyright 2005 C. Scott Ananian
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <assert.h>
#include <windows.h>
#include <windowsx.h>
#include <commctrl.h>

#include "wine/test.h"

#ifndef ES_COMBO
#define ES_COMBO 0x200
#endif

#define ID_EDITTEST2 99
#define MAXLEN 200

static char szEditTest2Name[] = "Edit Test 2 window class";
static HINSTANCE hinst;
static HWND hwndET2;

HWND create_editcontrol (DWORD style, DWORD exstyle)
{
    HWND handle;

    handle = CreateWindowEx(exstyle,
			  "EDIT",
			  NULL,
			  ES_AUTOHSCROLL | ES_AUTOVSCROLL | style,
			  10, 10, 300, 300,
			  NULL, NULL, NULL, NULL);
    assert (handle);
    if (winetest_interactive)
	ShowWindow (handle, SW_SHOW);
    return handle;
}

static LONG get_edit_style (HWND hwnd)
{
    return GetWindowLongA( hwnd, GWL_STYLE ) & (
	ES_LEFT |
/* FIXME: not implemented
	ES_CENTER |
	ES_RIGHT |
	ES_OEMCONVERT |
*/
	ES_MULTILINE |
	ES_UPPERCASE |
	ES_LOWERCASE |
	ES_PASSWORD |
	ES_AUTOVSCROLL |
	ES_AUTOHSCROLL |
	ES_NOHIDESEL |
	ES_COMBO |
	ES_READONLY |
	ES_WANTRETURN |
	ES_NUMBER
	);
}

static void set_client_height(HWND Wnd, unsigned Height)
{
    RECT ClientRect, WindowRect;

    GetWindowRect(Wnd, &WindowRect);
    GetClientRect(Wnd, &ClientRect);
    SetWindowPos(Wnd, NULL, WindowRect.left, WindowRect.top,
                 WindowRect.right - WindowRect.left,
                 Height + (WindowRect.bottom - WindowRect.top) - (ClientRect.bottom - ClientRect.top),
                 SWP_NOMOVE | SWP_NOZORDER);
}

static void test_edit_control_1(void)
{
    HWND hwEdit;
    MSG msMessage;
    int i;
    LONG r;
    HFONT Font, OldFont;
    HDC Dc;
    TEXTMETRIC Metrics;
    RECT FormatRect;

    msMessage.message = WM_KEYDOWN;

    trace("EDIT: Single line\n");
    hwEdit = create_editcontrol(0, 0);
    r = get_edit_style(hwEdit);
    ok(r == (ES_AUTOVSCROLL | ES_AUTOHSCROLL), "Wrong style expected 0xc0 got: 0x%lx\n", r); 
    for (i=0;i<65535;i++)
    {
	msMessage.wParam = i;
	r = SendMessage(hwEdit, WM_GETDLGCODE, 0, (LPARAM) &msMessage);
	ok(r == (DLGC_WANTCHARS | DLGC_HASSETSEL | DLGC_WANTARROWS),
	    "Expected DLGC_WANTCHARS | DLGC_HASSETSEL | DLGC_WANTARROWS got %lx\n", r);
    }
    DestroyWindow (hwEdit);

    trace("EDIT: Single line want returns\n");
    hwEdit = create_editcontrol(ES_WANTRETURN, 0);
    r = get_edit_style(hwEdit);
    ok(r == (ES_AUTOVSCROLL | ES_AUTOHSCROLL | ES_WANTRETURN), "Wrong style expected 0x10c0 got: 0x%lx\n", r); 
    for (i=0;i<65535;i++)
    {
	msMessage.wParam = i;
	r = SendMessage(hwEdit, WM_GETDLGCODE, 0, (LPARAM) &msMessage);
	ok(r == (DLGC_WANTCHARS | DLGC_HASSETSEL | DLGC_WANTARROWS),
	    "Expected DLGC_WANTCHARS | DLGC_HASSETSEL | DLGC_WANTARROWS got %lx\n", r);
    }
    DestroyWindow (hwEdit);

    trace("EDIT: Multiline line\n");
    hwEdit = create_editcontrol(ES_MULTILINE | WS_VSCROLL | ES_AUTOVSCROLL, 0);
    r = get_edit_style(hwEdit);
    ok(r == (ES_AUTOHSCROLL | ES_AUTOVSCROLL | ES_MULTILINE), "Wrong style expected 0xc4 got: 0x%lx\n", r); 
    for (i=0;i<65535;i++)
    {
	msMessage.wParam = i;
	r = SendMessage(hwEdit, WM_GETDLGCODE, 0, (LPARAM) &msMessage);
	ok(r == (DLGC_WANTCHARS | DLGC_HASSETSEL | DLGC_WANTALLKEYS | DLGC_WANTARROWS),
	    "Expected DLGC_WANTCHARS | DLGC_HASSETSEL | DLGC_WANTALLKEYS | DLGC_WANTARROWS got %lx\n", r);
    }
    DestroyWindow (hwEdit);

    trace("EDIT: Multi line want returns\n");
    hwEdit = create_editcontrol(ES_MULTILINE | WS_VSCROLL | ES_AUTOVSCROLL | ES_WANTRETURN, 0);
    r = get_edit_style(hwEdit);
    ok(r == (ES_WANTRETURN | ES_AUTOHSCROLL | ES_AUTOVSCROLL | ES_MULTILINE), "Wrong style expected 0x10c4 got: 0x%lx\n", r); 
    for (i=0;i<65535;i++)
    {
	msMessage.wParam = i;
	r = SendMessage(hwEdit, WM_GETDLGCODE, 0, (LPARAM) &msMessage);
	ok(r == (DLGC_WANTCHARS | DLGC_HASSETSEL | DLGC_WANTALLKEYS | DLGC_WANTARROWS),
	    "Expected DLGC_WANTCHARS | DLGC_HASSETSEL | DLGC_WANTALLKEYS | DLGC_WANTARROWS got %lx\n", r);
    }
    DestroyWindow (hwEdit);

    /* Get a stock font for which we can determine the metrics */
    Font = GetStockObject(SYSTEM_FONT);
    assert(NULL != Font);
    Dc = GetDC(NULL);
    assert(NULL != Dc);
    OldFont = SelectObject(Dc, Font);
    assert(NULL != OldFont);
    if (! GetTextMetrics(Dc, &Metrics))
    {
	assert(FALSE);
    }
    SelectObject(Dc, OldFont);
    ReleaseDC(NULL, Dc);

    trace("EDIT: vertical text position\n");
    hwEdit = create_editcontrol(WS_POPUP, 0);
    SendMessage(hwEdit, WM_SETFONT, (WPARAM) Font, (LPARAM) FALSE);
    set_client_height(hwEdit, Metrics.tmHeight - 1);
    SendMessage(hwEdit, EM_GETRECT, 0, (LPARAM) &FormatRect);
    ok(0 == FormatRect.top, "wrong vertical position expected 0 got %ld\n", FormatRect.top);
    ok(Metrics.tmHeight - 1 == FormatRect.bottom - FormatRect.top, "wrong height expected %ld got %ld\n", Metrics.tmHeight - 1, FormatRect.bottom - FormatRect.top);
    set_client_height(hwEdit, Metrics.tmHeight);
    SendMessage(hwEdit, EM_GETRECT, 0, (LPARAM) &FormatRect);
    ok(0 == FormatRect.top, "wrong vertical position expected 0 got %ld\n", FormatRect.top);
    ok(Metrics.tmHeight == FormatRect.bottom - FormatRect.top, "wrong height expected %ld got %ld\n", Metrics.tmHeight, FormatRect.bottom - FormatRect.top);
    set_client_height(hwEdit, Metrics.tmHeight + 1);
    SendMessage(hwEdit, EM_GETRECT, 0, (LPARAM) &FormatRect);
    ok(0 == FormatRect.top, "wrong vertical position expected 0 got %ld\n", FormatRect.top);
    ok(Metrics.tmHeight == FormatRect.bottom - FormatRect.top, "wrong height expected %ld got %ld\n", Metrics.tmHeight, FormatRect.bottom - FormatRect.top);
    set_client_height(hwEdit, Metrics.tmHeight + 2);
    SendMessage(hwEdit, EM_GETRECT, 0, (LPARAM) &FormatRect);
    ok(0 == FormatRect.top, "wrong vertical position expected 0 got %ld\n", FormatRect.top);
    ok(Metrics.tmHeight == FormatRect.bottom - FormatRect.top, "wrong height expected %ld got %ld\n", Metrics.tmHeight, FormatRect.bottom - FormatRect.top);
    set_client_height(hwEdit, Metrics.tmHeight + 10);
    SendMessage(hwEdit, EM_GETRECT, 0, (LPARAM) &FormatRect);
    ok(0 == FormatRect.top, "wrong vertical position expected 0 got %ld\n", FormatRect.top);
    ok(Metrics.tmHeight == FormatRect.bottom - FormatRect.top, "wrong height expected %ld got %ld\n", Metrics.tmHeight, FormatRect.bottom - FormatRect.top);
    DestroyWindow(hwEdit);

    hwEdit = create_editcontrol(WS_POPUP | WS_BORDER, 0);
    SendMessage(hwEdit, WM_SETFONT, (WPARAM) Font, (LPARAM) FALSE);
    set_client_height(hwEdit, Metrics.tmHeight - 1);
    SendMessage(hwEdit, EM_GETRECT, 0, (LPARAM) &FormatRect);
    ok(0 == FormatRect.top, "wrong vertical position expected 0 got %ld\n", FormatRect.top);
    ok(Metrics.tmHeight - 1 == FormatRect.bottom - FormatRect.top, "wrong height expected %ld got %ld\n", Metrics.tmHeight - 1, FormatRect.bottom - FormatRect.top);
    set_client_height(hwEdit, Metrics.tmHeight);
    SendMessage(hwEdit, EM_GETRECT, 0, (LPARAM) &FormatRect);
    ok(0 == FormatRect.top, "wrong vertical position expected 0 got %ld\n", FormatRect.top);
    ok(Metrics.tmHeight == FormatRect.bottom - FormatRect.top, "wrong height expected %ld got %ld\n", Metrics.tmHeight, FormatRect.bottom - FormatRect.top);
    set_client_height(hwEdit, Metrics.tmHeight + 2);
    SendMessage(hwEdit, EM_GETRECT, 0, (LPARAM) &FormatRect);
    ok(0 == FormatRect.top, "wrong vertical position expected 0 got %ld\n", FormatRect.top);
    ok(Metrics.tmHeight == FormatRect.bottom - FormatRect.top, "wrong height expected %ld got %ld\n", Metrics.tmHeight, FormatRect.bottom - FormatRect.top);
    set_client_height(hwEdit, Metrics.tmHeight + 3);
    SendMessage(hwEdit, EM_GETRECT, 0, (LPARAM) &FormatRect);
    ok(0 == FormatRect.top, "wrong vertical position expected 0 got %ld\n", FormatRect.top);
    ok(Metrics.tmHeight == FormatRect.bottom - FormatRect.top, "wrong height expected %ld got %ld\n", Metrics.tmHeight, FormatRect.bottom - FormatRect.top);
    set_client_height(hwEdit, Metrics.tmHeight + 4);
    SendMessage(hwEdit, EM_GETRECT, 0, (LPARAM) &FormatRect);
    ok(2 == FormatRect.top, "wrong vertical position expected 2 got %ld\n", FormatRect.top);
    ok(Metrics.tmHeight == FormatRect.bottom - FormatRect.top, "wrong height expected %ld got %ld\n", Metrics.tmHeight, FormatRect.bottom - FormatRect.top);
    set_client_height(hwEdit, Metrics.tmHeight + 10);
    SendMessage(hwEdit, EM_GETRECT, 0, (LPARAM) &FormatRect);
    ok(2 == FormatRect.top, "wrong vertical position expected 0 got %ld\n", FormatRect.top);
    ok(Metrics.tmHeight == FormatRect.bottom - FormatRect.top, "wrong height expected %ld got %ld\n", Metrics.tmHeight, FormatRect.bottom - FormatRect.top);
    DestroyWindow(hwEdit);

    hwEdit = create_editcontrol(WS_POPUP, WS_EX_CLIENTEDGE);
    SendMessage(hwEdit, WM_SETFONT, (WPARAM) Font, (LPARAM) FALSE);
    set_client_height(hwEdit, Metrics.tmHeight - 1);
    SendMessage(hwEdit, EM_GETRECT, 0, (LPARAM) &FormatRect);
    ok(0 == FormatRect.top, "wrong vertical position expected 0 got %ld\n", FormatRect.top);
    ok(Metrics.tmHeight - 1 == FormatRect.bottom - FormatRect.top, "wrong height expected %ld got %ld\n", Metrics.tmHeight - 1, FormatRect.bottom - FormatRect.top);
    set_client_height(hwEdit, Metrics.tmHeight);
    SendMessage(hwEdit, EM_GETRECT, 0, (LPARAM) &FormatRect);
    ok(0 == FormatRect.top, "wrong vertical position expected 0 got %ld\n", FormatRect.top);
    ok(Metrics.tmHeight == FormatRect.bottom - FormatRect.top, "wrong height expected %ld got %ld\n", Metrics.tmHeight, FormatRect.bottom - FormatRect.top);
    set_client_height(hwEdit, Metrics.tmHeight + 1);
    SendMessage(hwEdit, EM_GETRECT, 0, (LPARAM) &FormatRect);
    ok(0 == FormatRect.top, "wrong vertical position expected 0 got %ld\n", FormatRect.top);
    ok(Metrics.tmHeight == FormatRect.bottom - FormatRect.top, "wrong height expected %ld got %ld\n", Metrics.tmHeight, FormatRect.bottom - FormatRect.top);
    set_client_height(hwEdit, Metrics.tmHeight + 2);
    SendMessage(hwEdit, EM_GETRECT, 0, (LPARAM) &FormatRect);
    ok(1 == FormatRect.top, "wrong vertical position expected 1 got %ld\n", FormatRect.top);
    ok(Metrics.tmHeight == FormatRect.bottom - FormatRect.top, "wrong height expected %ld got %ld\n", Metrics.tmHeight, FormatRect.bottom - FormatRect.top);
    set_client_height(hwEdit, Metrics.tmHeight + 4);
    SendMessage(hwEdit, EM_GETRECT, 0, (LPARAM) &FormatRect);
    ok(1 == FormatRect.top, "wrong vertical position expected 1 got %ld\n", FormatRect.top);
    ok(Metrics.tmHeight == FormatRect.bottom - FormatRect.top, "wrong height expected %ld got %ld\n", Metrics.tmHeight, FormatRect.bottom - FormatRect.top);
    set_client_height(hwEdit, Metrics.tmHeight + 10);
    SendMessage(hwEdit, EM_GETRECT, 0, (LPARAM) &FormatRect);
    ok(1 == FormatRect.top, "wrong vertical position expected 1 got %ld\n", FormatRect.top);
    ok(Metrics.tmHeight == FormatRect.bottom - FormatRect.top, "wrong height expected %ld got %ld\n", Metrics.tmHeight, FormatRect.bottom - FormatRect.top);
    DestroyWindow(hwEdit);

    hwEdit = create_editcontrol(WS_POPUP | WS_BORDER, WS_EX_CLIENTEDGE);
    SendMessage(hwEdit, WM_SETFONT, (WPARAM) Font, (LPARAM) FALSE);
    set_client_height(hwEdit, Metrics.tmHeight - 1);
    SendMessage(hwEdit, EM_GETRECT, 0, (LPARAM) &FormatRect);
    ok(0 == FormatRect.top, "wrong vertical position expected 0 got %ld\n", FormatRect.top);
    ok(Metrics.tmHeight - 1 == FormatRect.bottom - FormatRect.top, "wrong height expected %ld got %ld\n", Metrics.tmHeight - 1, FormatRect.bottom - FormatRect.top);
    set_client_height(hwEdit, Metrics.tmHeight);
    SendMessage(hwEdit, EM_GETRECT, 0, (LPARAM) &FormatRect);
    ok(0 == FormatRect.top, "wrong vertical position expected 0 got %ld\n", FormatRect.top);
    ok(Metrics.tmHeight == FormatRect.bottom - FormatRect.top, "wrong height expected %ld got %ld\n", Metrics.tmHeight, FormatRect.bottom - FormatRect.top);
    set_client_height(hwEdit, Metrics.tmHeight + 1);
    SendMessage(hwEdit, EM_GETRECT, 0, (LPARAM) &FormatRect);
    ok(0 == FormatRect.top, "wrong vertical position expected 0 got %ld\n", FormatRect.top);
    ok(Metrics.tmHeight == FormatRect.bottom - FormatRect.top, "wrong height expected %ld got %ld\n", Metrics.tmHeight, FormatRect.bottom - FormatRect.top);
    set_client_height(hwEdit, Metrics.tmHeight + 2);
    SendMessage(hwEdit, EM_GETRECT, 0, (LPARAM) &FormatRect);
    ok(1 == FormatRect.top, "wrong vertical position expected 1 got %ld\n", FormatRect.top);
    ok(Metrics.tmHeight == FormatRect.bottom - FormatRect.top, "wrong height expected %ld got %ld\n", Metrics.tmHeight, FormatRect.bottom - FormatRect.top);
    set_client_height(hwEdit, Metrics.tmHeight + 4);
    SendMessage(hwEdit, EM_GETRECT, 0, (LPARAM) &FormatRect);
    ok(1 == FormatRect.top, "wrong vertical position expected 1 got %ld\n", FormatRect.top);
    ok(Metrics.tmHeight == FormatRect.bottom - FormatRect.top, "wrong height expected %ld got %ld\n", Metrics.tmHeight, FormatRect.bottom - FormatRect.top);
    set_client_height(hwEdit, Metrics.tmHeight + 10);
    SendMessage(hwEdit, EM_GETRECT, 0, (LPARAM) &FormatRect);
    ok(1 == FormatRect.top, "wrong vertical position expected 1 got %ld\n", FormatRect.top);
    ok(Metrics.tmHeight == FormatRect.bottom - FormatRect.top, "wrong height expected %ld got %ld\n", Metrics.tmHeight, FormatRect.bottom - FormatRect.top);
    DestroyWindow(hwEdit);
}

/* WM_SETTEXT is implemented by selecting all text, and then replacing the
 * selection.  This test checks that the first 'select all' doesn't generate
 * an UPDATE message which can escape and (via a handler) change the
 * selection, which would cause WM_SETTEXT to break.  This old bug
 * was fixed 18-Mar-2005; we check here to ensure it doesn't regress.
 */
static void test_edit_control_2(void)
{
    HWND hwndMain;
    char szLocalString[MAXLEN];

    /* Create main and edit windows. */
    hwndMain = CreateWindow(szEditTest2Name, "ET2", WS_OVERLAPPEDWINDOW,
                            0, 0, 200, 200, NULL, NULL, hinst, NULL);
    assert(hwndMain);
    if (winetest_interactive)
        ShowWindow (hwndMain, SW_SHOW);

    hwndET2 = CreateWindow("EDIT", NULL,
                           WS_CHILD|WS_BORDER|ES_LEFT|ES_AUTOHSCROLL,
                           0, 0, 150, 50, /* important this not be 0 size. */
                           hwndMain, (HMENU) ID_EDITTEST2, hinst, NULL);
    assert(hwndET2);
    if (winetest_interactive)
        ShowWindow (hwndET2, SW_SHOW);

    trace("EDIT: SETTEXT atomicity\n");
    /* Send messages to "type" in the word 'foo'. */
    SendMessage(hwndET2, WM_CHAR, 'f', 1);
    SendMessage(hwndET2, WM_CHAR, 'o', 1);
    SendMessage(hwndET2, WM_CHAR, 'o', 1);
    /* 'foo' should have been changed to 'bar' by the UPDATE handler. */
    GetWindowText(hwndET2, szLocalString, MAXLEN);
    ok(lstrcmp(szLocalString, "bar")==0,
       "Wrong contents of edit: %s\n", szLocalString);

    /* OK, done! */
    DestroyWindow (hwndET2);
    DestroyWindow (hwndMain);
}

static void ET2_check_change() {
   char szLocalString[MAXLEN];
   /* This EN_UPDATE handler changes any 'foo' to 'bar'. */
   GetWindowText(hwndET2, szLocalString, MAXLEN);
   if (lstrcmp(szLocalString, "foo")==0) {
       lstrcpy(szLocalString, "bar");
       SendMessage(hwndET2, WM_SETTEXT, 0, (LPARAM) szLocalString);
   }
   /* always leave the cursor at the end. */
   SendMessage(hwndET2, EM_SETSEL, MAXLEN - 1, MAXLEN - 1);
}
static void ET2_OnCommand(HWND hwnd, int id, HWND hwndCtl, UINT codeNotify)
{
    if (id==ID_EDITTEST2 && codeNotify == EN_UPDATE)
        ET2_check_change();
}
static LRESULT CALLBACK ET2_WndProc(HWND hwnd, UINT iMsg, WPARAM wParam, LPARAM lParam)
{
    switch (iMsg) {
        HANDLE_MSG(hwnd, WM_COMMAND, ET2_OnCommand);
    }
    return DefWindowProc(hwnd, iMsg, wParam, lParam);
}

static BOOL RegisterWindowClasses (void)
{
    WNDCLASSA cls;
    cls.style = 0;
    cls.lpfnWndProc = ET2_WndProc;
    cls.cbClsExtra = 0;
    cls.cbWndExtra = 0;
    cls.hInstance = hinst;
    cls.hIcon = NULL;
    cls.hCursor = LoadCursorA (NULL, IDC_ARROW);
    cls.hbrBackground = (HBRUSH)(COLOR_WINDOW + 1);
    cls.lpszMenuName = NULL;
    cls.lpszClassName = szEditTest2Name;
    if (!RegisterClassA (&cls)) return FALSE;

    return TRUE;
}

START_TEST(edit)
{
    hinst = GetModuleHandleA (NULL);
    if (!RegisterWindowClasses())
        assert(0);

    test_edit_control_1();
    test_edit_control_2();
}
