/*
 * BlueALSA - io.c
 * Copyright (c) 2016 Arkadiusz Bokowy
 *
 * This file is a part of bluez-alsa.
 *
 * This project is licensed under the terms of the MIT license.
 *
 */

#include "io.h"

#include <errno.h>
#include <fcntl.h>
#include <poll.h>
#include <signal.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <unistd.h>
#include <sys/socket.h>
#include <sys/types.h>

#include <sbc/sbc.h>

#include "a2dp-codecs.h"
#include "a2dp-rtp.h"
#include "log.h"
#include "transport.h"
#include "utils.h"


/**
 * Wrapper for release callback, which can be used by pthread cleanup. */
static void io_thread_release_bt(void *arg) {
	struct ba_transport *t = (struct ba_transport *)arg;

	/* During the normal operation mode, the release callback should not
	 * be NULL. Hence, we will relay on this callback - file descriptors
	 * are closed in it. */
	if (t->release != NULL)
		t->release(t);

	/* XXX: If the order of the cleanup push is right, this function will
	 *      indicate the end of the IO thread. */
	debug("Exiting IO thread");
}

/**
 * Pause IO thread until the resume signal is received. */
static void io_thread_pause(struct ba_transport *t) {

	struct timespec ts0, ts;
	clock_gettime(CLOCK_MONOTONIC, &ts0);

	debug("Pausing IO thread: %s", t->name);
	pthread_mutex_lock(&t->resume_mutex);
	pthread_cond_wait(&t->resume, &t->resume_mutex);
	pthread_mutex_unlock(&t->resume_mutex);

	clock_gettime(CLOCK_MONOTONIC, &ts);
	t->ts_paused.tv_sec += ts.tv_sec - ts0.tv_sec;
	t->ts_paused.tv_nsec += ts.tv_nsec - ts0.tv_nsec;

	debug("Resuming IO thread: %s", t->name);
}

void *io_thread_a2dp_sbc_forward(void *arg) {
	struct ba_transport *t = (struct ba_transport *)arg;

	/* Cancellation should be possible only in the carefully selected place
	 * in order to prevent memory leaks and resources not being released. */
	pthread_setcancelstate(PTHREAD_CANCEL_DISABLE, NULL);

	if (t->bt_fd == -1) {
		error("Invalid BT socket: %d", t->bt_fd);
		return NULL;
	}

	/* Check for invalid (e.g. not set) reading MTU. If buffer allocation does
	 * not return NULL (allocating zero bytes might return NULL), we will read
	 * zero bytes from the BT socket, which will be wrongly identified as a
	 * "connection closed" action. */
	if (t->mtu_read <= 0) {
		error("Invalid reading MTU: %zu", t->mtu_read);
		return NULL;
	}

	sbc_t sbc;

	if ((errno = -sbc_init_a2dp(&sbc, 0, t->config, t->config_size)) != 0) {
		error("Couldn't initialize %s codec: %s", "SBC", strerror(errno));
		return NULL;
	}

	const size_t sbc_codesize = sbc_get_codesize(&sbc);
	const size_t sbc_frame_len = sbc_get_frame_length(&sbc);

	const size_t rbuffer_size = t->mtu_read;
	const size_t wbuffer_size = sbc_codesize * (rbuffer_size / sbc_frame_len + 1);
	uint8_t *rbuffer = malloc(rbuffer_size);
	uint8_t *wbuffer = malloc(wbuffer_size);

	pthread_cleanup_push(io_thread_release_bt, t);
	pthread_cleanup_push(sbc_finish, &sbc);
	pthread_cleanup_push(free, wbuffer);
	pthread_cleanup_push(free, rbuffer);

	if (rbuffer == NULL || wbuffer == NULL) {
		error("Couldn't create data buffers: %s", strerror(errno));
		goto fail;
	}

	struct sigaction sigact = { .sa_handler = SIG_IGN };
	sigaction(SIGPIPE, &sigact, NULL);

	struct pollfd pfds[1] = {{ t->bt_fd, POLLIN, 0 }};

	/* TODO: support for "out of the hat" reading MTU */

	debug("Starting forward IO loop");
	while (TRANSPORT_RUN_IO_THREAD(t)) {

		ssize_t len;

		pthread_setcancelstate(PTHREAD_CANCEL_ENABLE, NULL);

		if (t->state == TRANSPORT_PAUSED)
			io_thread_pause(t);

		if (poll(pfds, 1, -1) == -1) {
			error("Transport poll error: %s", strerror(errno));
			break;
		}

		pthread_setcancelstate(PTHREAD_CANCEL_DISABLE, NULL);

		if ((len = read(pfds[0].fd, rbuffer, rbuffer_size)) == -1) {
			debug("BT read error: %s", strerror(errno));
			continue;
		}

		/* It seems that this block of code is not executed... */
		if (len == 0) {
			debug("BT socket has been closed: %d", pfds[0].fd);
			/* Prevent sending the release request to the BlueZ. If the socket has
			 * been closed, it means that BlueZ has already closed the connection. */
			close(pfds[0].fd);
			t->bt_fd = -1;
			break;
		}

		/* transport PCM FIFO has not been requested */
		if (t->pcm_fifo == NULL)
			continue;

		if (t->pcm_fd == -1) {

			if ((t->pcm_fd = open(t->pcm_fifo, O_WRONLY | O_NONBLOCK)) == -1) {
				if (errno != ENXIO)
					error("Couldn't open FIFO: %s", strerror(errno));
				/* FIFO endpoint is not connected yet */
				continue;
			}

			/* Restore the blocking mode of our FIFO. Non-blocking mode was required
			 * only for the opening process - we do not want to block if the reading
			 * endpoint is not connected yet. Blocking upon data write will prevent
			 * frame dropping. */
			fcntl(t->pcm_fd, F_SETFL, fcntl(t->pcm_fd, F_GETFL) & ~O_NONBLOCK);
		}

		const rtp_header_t *rtp_header = (rtp_header_t *)rbuffer;
		const rtp_payload_sbc_t *rtp_payload = (rtp_payload_sbc_t *)&rtp_header->csrc[rtp_header->cc];

		if (rtp_header->paytype != 96) {
			warn("Unsupported RTP payload type: %u", rtp_header->paytype);
			continue;
		}

		const uint8_t *input = (uint8_t *)(rtp_payload + 1);
		uint8_t *output = wbuffer;
		size_t input_len = len - (input - rbuffer);
		size_t output_len = wbuffer_size;
		size_t frames = rtp_payload->frame_count;

		/* decode retrieved SBC frames */
		while (frames && input_len >= sbc_frame_len) {

			ssize_t len;
			size_t decoded;

			if ((len = sbc_decode(&sbc, input, input_len, output, output_len, &decoded)) < 0) {
				error("SBC decoding error: %s", strerror(-len));
				break;
			}

			input += len;
			input_len -= len;
			output += decoded;
			output_len -= decoded;
			frames--;

		}

		if (write(t->pcm_fd, wbuffer, wbuffer_size - output_len) == -1) {

			if (errno == EPIPE) {
				debug("FIFO endpoint has been closed: %d", t->pcm_fd);
				transport_release_pcm(t);
				continue;
			}

			error("FIFO write error: %s", strerror(errno));
		}

	}

fail:
	warn("IO thread failure");
	pthread_cleanup_pop(1);
	pthread_cleanup_pop(1);
	pthread_cleanup_pop(1);
	pthread_cleanup_pop(1);
	return NULL;
}

void *io_thread_a2dp_sbc_backward(void *arg) {
	struct ba_transport *t = (struct ba_transport *)arg;

	pthread_setcancelstate(PTHREAD_CANCEL_DISABLE, NULL);

	sbc_t sbc;

	if ((errno = -sbc_init_a2dp(&sbc, 0, t->config, t->config_size)) != 0) {
		error("Couldn't initialize %s codec: %s", "SBC", strerror(errno));
		return NULL;
	}

	const size_t sbc_codesize = sbc_get_codesize(&sbc);
	const size_t sbc_frame_len = sbc_get_frame_length(&sbc);
	const unsigned sbc_frame_duration = sbc_get_frame_duration(&sbc);

	/* Writing MTU should be big enough to contain RTP header, SBC payload
	 * header and at least one SBC frame. In general, there is no constraint
	 * for the MTU value, but the speed might suffer significantly. */
	size_t mtu_write = t->mtu_write;
	if (mtu_write < sizeof(rtp_header_t) + sizeof(rtp_payload_sbc_t) + sbc_frame_len) {
		mtu_write = sizeof(rtp_header_t) + sizeof(rtp_payload_sbc_t) + sbc_frame_len;
		warn("Writing MTU too small for one single SBC frame: %zu < %zu", t->mtu_write, mtu_write);
	}

	rtp_header_t *rtp_header;
	rtp_payload_sbc_t *rtp_payload;

	/* The internal timestamp has to be stored in a variable, which can handle
	 * reasonable big time (more then one hour) with a microsecond precision.
	 * Unsigned 32 bit integer is not enough for this purpose. It is possible
	 * to store up to 0xFFFFFFFF microseconds which is ~71.6 minutes. */
	struct timespec ts_timestamp = { 0 };
	uint16_t seq_number = 0;
	uint32_t timestamp = 0;

	const size_t rbuffer_size = sbc_codesize * (mtu_write / sbc_frame_len);
	const size_t wbuffer_size = mtu_write;
	uint8_t *rbuffer = malloc(rbuffer_size);
	uint8_t *wbuffer = malloc(wbuffer_size);

	pthread_cleanup_push(io_thread_release_bt, t);
	pthread_cleanup_push(sbc_finish, &sbc);
	pthread_cleanup_push(free, wbuffer);
	pthread_cleanup_push(free, rbuffer);

	if (rbuffer == NULL || wbuffer == NULL) {
		error("Couldn't create data buffers: %s", strerror(errno));
		goto fail;
	}

	/* XXX: This check allows testing. During normal operation PCM FIFO
	 *      should not be opened by some external entity. */
	if (t->pcm_fd == -1) {
		debug("Opening FIFO for reading: %s", t->pcm_fifo);
		/* this call will block until writing end is opened */
		if ((t->pcm_fd = open(t->pcm_fifo, O_RDONLY)) == -1) {
			error("Couldn't open FIFO: %s", strerror(errno));
			goto fail;
		}
	}

	struct sigaction sigact = { .sa_handler = SIG_IGN };
	sigaction(SIGPIPE, &sigact, NULL);

	/* initialize RTP headers (the constant part) */
	rtp_header = (rtp_header_t *)wbuffer;
	memset(rtp_header, 0, sizeof(*rtp_header));
	rtp_payload = (rtp_payload_sbc_t *)&rtp_header->csrc[rtp_header->cc];
	memset(rtp_payload, 0, sizeof(*rtp_payload));
	rtp_header->version = 2;
	rtp_header->paytype = 96;

	/* reading head position and available read length */
	uint8_t *rhead = rbuffer;
	size_t rlen = rbuffer_size;

	int initialized = 0;
	struct timespec ts0;

	debug("Starting backward IO loop");
	while (TRANSPORT_RUN_IO_THREAD(t)) {

		ssize_t len;

		pthread_setcancelstate(PTHREAD_CANCEL_ENABLE, NULL);

		if (t->state == TRANSPORT_PAUSED)
			io_thread_pause(t);

		/* This call will block until data arrives. If the passed file descriptor
		 * is invalid (e.g. -1) is means, that other thread (the controller) has
		 * closed the connection. If the connection was closed during the blocking
		 * part, we will still read correct data, because Linux kernel does not
		 * decrement file descriptor reference counter until the read returns. */
		if ((len = read(t->pcm_fd, rhead, rlen)) == -1) {
			if (errno == EINTR)
				continue;
		}

		pthread_setcancelstate(PTHREAD_CANCEL_DISABLE, NULL);

		if (len <= 0) {
			if (len == -1 && errno != EBADF)
				error("FIFO read error: %s", strerror(errno));
			else if (len == 0)
				debug("FIFO endpoint has been closed: %d", t->pcm_fd);
			transport_release_pcm(t);
			goto fail;
		}

		/* When the thread is created, there might be no data in the FIFO. In fact
		 * there might be no data for a long time - until client starts playback.
		 * In order to correctly calculate time drift, the zero time point has to
		 * be obtained after the stream has started. */
		if (!initialized) {
			initialized = 1;
			/* Get initial time point. This time point is used to calculate time
			 * drift during data transmission. The transfer should be kept at a
			 * constant pace (audio bit rate). */
			clock_gettime(CLOCK_MONOTONIC, &ts0);
		}

		const uint8_t *input = rbuffer;
		size_t input_len = (rhead - rbuffer) + len;

		/* Get a snapshot of audio properties. Please note, that mutex is not
		 * required here, because we are not modifying these variables. */
		const uint8_t volume = t->volume;
		const uint8_t muted = t->muted;

		if (muted || volume == 0)
			snd_pcm_mute_s16le(rbuffer, input_len);
		else if (volume != 100)
			snd_pcm_scale_s16le(rbuffer, input_len, volume);

		/* encode and transfer obtained data */
		while (input_len >= sbc_codesize) {

			uint8_t *output = (uint8_t *)(rtp_payload + 1);
			size_t output_len = wbuffer_size - (output - wbuffer);
			size_t frames = 0;

			/* Generate as many SBC frames as possible to fill the output buffer
			 * without overflowing it. The size of the output buffer is based on
			 * the socket MTU, so such a transfer should be most efficient. */
			while (input_len >= sbc_codesize && output_len >= sbc_frame_len) {

				ssize_t len;
				ssize_t encoded;

				if ((len = sbc_encode(&sbc, input, input_len, output, output_len, &encoded)) < 0) {
					error("SBC encoding error: %s", strerror(-len));
					break;
				}

				input += len;
				input_len -= len;
				output += encoded;
				output_len -= encoded;
				frames++;

			}

			rtp_header->seq_number = htons(++seq_number);
			rtp_header->timestamp = htonl(timestamp);
			rtp_payload->frame_count = frames;

			struct timespec ts;
			clock_gettime(CLOCK_MONOTONIC, &ts);

			/* keep transfer at constant rate, always 10 ms ahead */
			int rt_delta = (ts_timestamp.tv_sec + t->ts_paused.tv_sec - (ts.tv_sec - ts0.tv_sec)) * 1e6 +
				(ts_timestamp.tv_nsec + t->ts_paused.tv_nsec - (ts.tv_nsec - ts0.tv_nsec)) / 1e3 - 10e3;
			if (rt_delta > 0)
				usleep(rt_delta);

			if (write(t->bt_fd, wbuffer, output - wbuffer) == -1) {
				if (errno == ECONNRESET || errno == ENOTCONN) {
					/* exit the thread upon BT socket disconnection */
					debug("BT socket disconnected: %s", strerror(errno));
					goto fail;
				}
				error("BT socket write error: %s", strerror(errno));
			}

			/* get timestamp for the next frame */
			const unsigned payload_duration = sbc_frame_duration * frames;
			ts_timestamp.tv_nsec += payload_duration * 1000;
			ts_timestamp.tv_sec += ts_timestamp.tv_nsec / (long)1e9;
			ts_timestamp.tv_nsec = ts_timestamp.tv_nsec % (long)1e9;
			timestamp += payload_duration;

		}

		/* If the input buffer was not consumed (due to codesize limit), we
		 * have to append new data to the existing one. Since we do not use
		 * ring buffer, we will simply move unprocessed data to the front
		 * of our linear buffer. */
		if (input_len > 0 && rbuffer != input)
			memmove(rbuffer, input, input_len);

		/* reposition our reading head */
		rhead = rbuffer + input_len;
		rlen = rbuffer_size - input_len;

	}

fail:
	pthread_cleanup_pop(1);
	pthread_cleanup_pop(1);
	pthread_cleanup_pop(1);
	pthread_cleanup_pop(1);
	return NULL;
}
