//  BMPx - The Dumb Music Player
//  Copyright (C) 2005-2007 BMPx development team.
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License Version 2
//  as published by the Free Software Foundation.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
//  --
//
//  The BMPx project hereby grants permission for non-GPL compatible GStreamer
//  plugins to be used and distributed together with GStreamer and BMPx. This
//  permission is above and beyond the permissions granted by the GPL license
//  BMPx is covered by.

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif //HAVE_CONFIG_H

#include <glibmm.h>
#include <glibmm/i18n.h>
#include <gtkmm.h>
#include <libglademm.h>
#include <glade/glade.h>
#include <glib/gstdio.h>

#include <cstring>
#include <iostream>
#include <sstream>
#include <string>

#include <boost/format.hpp>
#include <boost/optional.hpp>
#include <boost/algorithm/string.hpp>

#include "dialog-track-details.hh"

#include "database.hh"
#include "lastfm.hh"
#include "main.hh"
#include "network.hh"
#include "paths.hh"
#include "playbacksource.hh"
#include "stock.hh"
#include "ui-tools.hh"
#include "util.hh"
#include "uri++.hh"

#include "amazon.hh"
#include "x_library.hh"
#include "x_mcsbind.hh"

#include "musicbrainz/mb-utils.hh"

using namespace Glib;
using namespace Gtk;
using namespace Bmp;
using namespace DB;
using namespace std; 

namespace Bmp
{
  typedef RefPtr<TextTag> RefTextTag;
  typedef std::vector<RefTextTag> TagV;

  class LinkTag
    : public TextTag
  {
    private:

      LinkTag (ustring const& url)
      : ObjectBase (typeid(LinkTag))
      , TextTag ()
      , m_url   (url)
      {
        property_foreground() = "#0000ff"; 
      }

    public:

      static RefPtr<LinkTag> create (ustring const& url)
      {
        return RefPtr<LinkTag> (new LinkTag (url)); 
      }

      virtual ~LinkTag () {}

    public:

      typedef sigc::signal<void, ustring const&> SignalUrlActivated;
    
    private:

      SignalUrlActivated signal_url_activated_;

    public:

      SignalUrlActivated&
      signal_url_activated ()
      {
        return signal_url_activated_;
      }

    protected:

      virtual bool
      on_event (RefPtr<Object> const& event_object, GdkEvent* event, TextIter const& iter) 
      {
        if (event->type == GDK_BUTTON_PRESS)
        {
              GdkEventButton * ev = (GdkEventButton*)(event);
              if (ev->button == 1)
              {
                    return true;
              }
        }
        if (event->type == GDK_BUTTON_RELEASE)
        {
              GdkEventButton * ev = (GdkEventButton*)(event);
              if (ev->button == 1)
              { 
                    signal_url_activated_.emit (m_url);
                    return true;
              }
        }
        return false;
      }

    private:
    
      ustring m_url;
  };

  class LastFmWikiView
    : public TextView
  {
      RefTextTag  m_current_tag;
      bool        m_hand;

    public:

      LastFmWikiView (BaseObjectType*                  obj,
                     RefPtr<Gnome::Glade::Xml> const& xml)
      : TextView  (obj) 
      , m_hand    (0) 
      { 
        gtk_widget_add_events (GTK_WIDGET (gobj()), 
                               GDK_POINTER_MOTION_MASK | GDK_POINTER_MOTION_HINT_MASK | GDK_LEAVE_NOTIFY_MASK);
      }

      virtual ~LastFmWikiView ()
      {}

    protected:

      virtual bool on_motion_notify_event (GdkEventMotion * event)
      {
        int x, y;
        int x_orig, y_orig;
        GdkModifierType state;

        if (event->is_hint)
        {
          gdk_window_get_pointer (event->window, &x_orig, &y_orig, &state);
        }
        else
        {
          x_orig = int (event->x);
          y_orig = int (event->y);
          state = GdkModifierType (event->state);
        }

        window_to_buffer_coords (TEXT_WINDOW_WIDGET, x_orig, y_orig, x, y);

        RefPtr<TextBuffer> buf = get_buffer();
        TextBuffer::iterator iter;
        get_iter_at_location (iter, x, y);      
      
        TagV tags = iter.get_tags();
    
        for (TagV::const_iterator i = tags.begin(); i != tags.end(); ++i) 
        {
          RefPtr<LinkTag> tag = RefPtr<LinkTag>::cast_dynamic (*i);
          if (tag)
          {
            GdkWindow * window = GDK_WINDOW (gtk_text_view_get_window (GTK_TEXT_VIEW (gobj()), GTK_TEXT_WINDOW_TEXT));
            GdkCursor * cursor = gdk_cursor_new_from_name (gdk_display_get_default (), "hand2");
            if (!cursor)
            {
              cursor = gdk_cursor_new_for_display (gdk_display_get_default (), GDK_HAND2);
            }
            gdk_window_set_cursor (window, cursor);
            m_hand = 1;
            return false;
          }
        }
 
        if (m_hand) 
        {
          GdkWindow * window = GDK_WINDOW (gtk_text_view_get_window (GTK_TEXT_VIEW (gobj()), GTK_TEXT_WINDOW_TEXT));
          GdkCursor * cursor = gdk_cursor_new_from_name (gdk_display_get_default (), "xterm");
          if (!cursor)
          {
            cursor = gdk_cursor_new_for_display (gdk_display_get_default (), GDK_XTERM);
          }
          gdk_window_set_cursor (window, cursor);
          m_hand = 0;
       }
    
        return false;
      }
  };

  class TrackInfoView
    : public TreeView
  {
      class Columns
        : public TreeModel::ColumnRecord
      {
        public:

          TreeModelColumn <ustring>  value; 
          TreeModelColumn <ustring>  title; 
          TreeModelColumn <string>   attribute; 

          Columns ()
          {
            add (value);
            add (title);
            add (attribute); 
          }
      };

      Columns             m_columns;
      RefPtr <ListStore>  m_store;
      Label               m_label1, m_label2;
      Bmp::DB::Row        m_row;

    public:

      TrackInfoView (BaseObjectType                 * obj,
                     RefPtr<Gnome::Glade::Xml> const& xml)
      : TreeView (obj)
      {
        {
          CellRendererText * cell = manage (new CellRendererText());
          TreeViewColumn * column = manage (new TreeViewColumn ("", *cell));
          column->add_attribute (*cell, "text", m_columns.title);
          column->set_resizable (false);
          column->set_expand (false);
          append_column (*column);
        }

        {
          CellRendererText * cell = manage (new CellRendererText());
          TreeViewColumn * column = manage (new TreeViewColumn ("", *cell));
          cell->property_editable() = true; // This way people can copy out text, but we don't actually edit it here
          column->set_cell_data_func (*cell, sigc::mem_fun (this, &Bmp::TrackInfoView::cell_data_func));
          column->set_resizable (false);
          column->set_expand (false);
          append_column (*column);
        }

        m_label1.property_xalign() = 0.0;
        m_label2.property_xalign() = 0.0;
          
        m_label1.property_label() = (boost::format ("<small><b>%s</b></small>") % (_("Attribute"))).str();
        m_label2.property_label() = (boost::format ("<small><b>%s</b></small>") % (_("Value"))).str();

        m_label1.property_use_markup() = true;
        m_label2.property_use_markup() = true; 

        m_label1.show_all ();
        m_label2.show_all ();

        get_column(0)->set_widget (m_label1);
        get_column(1)->set_widget (m_label2);

        m_store = ListStore::create (m_columns);  
        m_store->set_default_sort_func (sigc::mem_fun (this, &Bmp::TrackInfoView::default_sort_func));
        m_store->set_sort_column (TreeSortable::DEFAULT_SORT_COLUMN_ID, SORT_ASCENDING);

        set_grid_lines (TREE_VIEW_GRID_LINES_BOTH);
        set_model (m_store);
      }

      ~TrackInfoView()
      {}

      int
      default_sort_func (TreeModel::iterator const& iter_a,
                         TreeModel::iterator const& iter_b)
      {
        ustring str_a = (*iter_a)[m_columns.title];
        ustring str_b = (*iter_b)[m_columns.title];
        return str_a.lowercase().compare (str_b.lowercase());
      }

      void
      cell_data_func (CellRenderer * basecell, TreeModel::iterator const& iter)
      {
        CellRendererText * cell = dynamic_cast <CellRendererText*> (basecell);

        Row::const_iterator i = m_row.find ((*iter)[m_columns.attribute]);
        if (i == m_row.end())
        {
          cell->property_text() = "";
          return;
        }
      
        switch (i->second.which()) 
        {
          case VALUE_TYPE_INT:
            cell->property_text() = ((boost::format ("%1%") % boost::get <guint64> (i->second)).str());
            break;

          case VALUE_TYPE_REAL:
            cell->property_text() = ((boost::format ("%1%") % boost::get <double> (i->second)).str());
            break;

          case VALUE_TYPE_STRING:
            cell->property_text() = boost::get <string> (i->second);
            break;
        }
      }

#define ADD_ATTR(member,attr) \
        if (metadata. member ) \
        { \
          iter = m_store->append(); \
          (*iter)[m_columns.attribute] = get_attribute_info (AttributeId ( attr )).id; \
          (*iter)[m_columns.title] = get_attribute_info (AttributeId ( attr )).title; \
          m_row.insert (make_pair (get_attribute_info (AttributeId ( attr )).id, ::Bmp::DB::Variant (metadata. member .get()))); \
        } 

#define ADD_ATTR_NAME(member,attr,name) \
        if (metadata. member) \
        { \
          iter = m_store->append(); \
          (*iter)[m_columns.attribute] = get_attribute_info (AttributeId (attr)).id; \
          (*iter)[m_columns.title] = name; \
          m_row.insert (make_pair (get_attribute_info (AttributeId (attr)).id, ::Bmp::DB::Variant (metadata. member .get()))); \
        }

#define ADD_ATTR_CUSTOM(name,id) \
        if(metadata.m_row.count(id)) \
        { \
          iter = m_store->append(); \
          (*iter)[m_columns.attribute] = std::string(id); \
          (*iter)[m_columns.title] = name; \
          m_row.insert (make_pair (id, ::Bmp::DB::Variant (metadata.m_row.find(id)->second))); \
        }

      void
      set_track (std::string const& location)
      {
        m_row = DB::Row();
        Library::Obj()->get_metadata (location, m_row);  

        TreeIter iter;
        for (guint n = 0; n <= ATTRIBUTE_TYPE; ++n)
        {
          iter = m_store->append();
          (*iter)[m_columns.attribute] = get_attribute_info (AttributeId (n)).id;
          (*iter)[m_columns.title] = get_attribute_info (AttributeId (n)).title;
        }
      }

      void
      set_track (TrackMetadata const& metadata)
      {
        m_row = DB::Row();
        TreeIter iter;

        ADD_ATTR_NAME(album, ATTRIBUTE_ALBUM, _("Album/Source"));
        ADD_ATTR(artist, ATTRIBUTE_ARTIST);
        ADD_ATTR(title, ATTRIBUTE_TITLE);
        ADD_ATTR(genre, ATTRIBUTE_GENRE);
        ADD_ATTR(location, ATTRIBUTE_LOCATION) 
        ADD_ATTR_CUSTOM("Audio Codec", "audio-codec"); 
        ADD_ATTR_CUSTOM("Video Codec", "video-codec"); 
      }

      void
      clear ()
      {
        m_store->clear ();
      }
  };
}

namespace Bmp
{
  TrackDetails::TrackDetails (BaseObjectType                 * obj,
                              RefPtr<Gnome::Glade::Xml> const& xml)
  : Window            (obj)
  , m_ref_xml         (xml)
  , m_lyrics_request  (LyricWiki::TextRequestRefP (0))
  , m_artist_request  (LastFM::XMLRPC::ArtistMetadataRequestRefPtr (0))
  {
    Util::window_set_icon_list (*this, "player");

    m_ref_xml->get_widget_derived ("textview-artist", m_widget_artist);

    m_ref_xml->get_widget ("image-cover",     m_widget_cover);
    m_ref_xml->get_widget ("textview-lyrics", m_widget_lyrics);
    m_ref_xml->get_widget ("notebook-artist", m_notebook_artist);
    m_ref_xml->get_widget ("notebook-lyrics", m_notebook_lyrics);

    m_ref_xml->get_widget ("lyrics-artist", m_widget_lyrics_artist);
    m_ref_xml->get_widget ("lyrics-title", m_widget_lyrics_title);

    dynamic_cast<Image*>(m_ref_xml->get_widget ("throbber-artist"))->set (build_filename (BMP_IMAGE_DIR, BMP_THROBBER));
    dynamic_cast<Image*>(m_ref_xml->get_widget ("throbber-lyrics"))->set (build_filename (BMP_IMAGE_DIR, BMP_THROBBER));

    m_ref_xml->get_widget_derived ("treeview-trackinfo", m_widget_trackinfo);
    m_widget_trackinfo->show_all ();

    dynamic_cast<Button*>(m_ref_xml->get_widget ("button-refresh-lyrics"))->signal_clicked().connect
      (sigc::bind (sigc::mem_fun (*this, &Bmp::TrackDetails::get_lyrics), true));
    dynamic_cast<Button*>(m_ref_xml->get_widget ("button-refresh-artist"))->signal_clicked().connect
      (sigc::mem_fun (*this, &Bmp::TrackDetails::get_artist));
    dynamic_cast<Button*>(m_ref_xml->get_widget ("button-split-title"))->signal_clicked().connect
      (sigc::mem_fun (*this, &Bmp::TrackDetails::split_title));

    m_cover = Gdk::Pixbuf::create_from_file (build_filename(BMP_IMAGE_DIR, BMP_COVER_IMAGE_DEFAULT))->scale_simple (256, 256, Gdk::INTERP_BILINEAR);

    m_size_group = Gtk::SizeGroup::create (Gtk::SIZE_GROUP_HORIZONTAL);
    m_size_group->add_widget( *m_ref_xml->get_widget( "alignment6" ) );
    m_size_group->add_widget( *m_ref_xml->get_widget( "alignment7" ) );
    m_size_group->add_widget( *m_ref_xml->get_widget( "alignment8" ) );
    m_size_group->add_widget( *m_ref_xml->get_widget( "alignment9" ) );
  }


  TrackDetails::~TrackDetails ()
  {
    m_lyrics_request.clear ();
    m_artist_request.clear ();
  }

  TrackDetails*
  TrackDetails::create ()
  {
    const string path (BMP_GLADE_DIR G_DIR_SEPARATOR_S "dialog-track-details.glade");
    RefPtr<Gnome::Glade::Xml> glade_xml = Gnome::Glade::Xml::create (path);
    TrackDetails* dialog = 0;
    glade_xml->get_widget_derived ("dialog-track-details", dialog);
    glade_xml_signal_autoconnect (glade_xml->gobj());
    return dialog;
  }

  void
  TrackDetails::split_title ()
  {
    using boost::algorithm::split;
    using boost::algorithm::find_nth;
    using boost::algorithm::trim;
    using boost::iterator_range;

    std::string value = m_widget_lyrics_title->get_text ();
    iterator_range<std::string::iterator> match = find_nth (value, "-", 0);
    if (match)
    {
      std::string artist = std::string(value.begin(), match.begin());
      std::string title  = std::string(match.end(), value.end());

      trim (artist);
      trim (title);

      m_widget_lyrics_artist->set_text (artist);
      m_widget_lyrics_title->set_text (title);

      get_lyrics (false);
    }
  }

  void
  TrackDetails::open_uri (ustring const& url)
  {
    system (((boost::format ("xdg-open %s &") % url.c_str()).str()).c_str());
  }

  void
  TrackDetails::got_artist (std::string const& metadata, guint code)
  {
    if( code == 200 )
    {
      std::string text = Util::sanitize_lastfm (metadata);
      if( !text.empty() )
      {
          m_widget_artist->get_buffer()->set_text (text);
          RefPtr<LinkTag> tag = LinkTag::create ((boost::format ("http://last.fm/music/%s/+wiki") % URI::escape_string ((*MData.Iter).artist.get())).str());
          tag->signal_url_activated().connect (sigc::mem_fun (*this, &TrackDetails::open_uri));
          m_widget_artist->get_buffer()->get_tag_table()->add (tag);
          TextBuffer::iterator iter = m_widget_artist->get_buffer()->end();
          m_widget_artist->get_buffer()->insert_with_tag (iter, _("[Read More]"), tag);
      }
    }

    m_ref_xml->get_widget ("button-refresh-artist")->set_sensitive (1);
    m_notebook_artist->set_current_page (0);
  }

  void
  TrackDetails::get_artist ()
  {
    m_artist_request.clear ();

    if ((*MData.Iter).artist)
    {
      m_ref_xml->get_widget ("button-refresh-artist")->set_sensitive (0);
      m_notebook_artist->set_current_page (1);
      m_artist_request = LastFM::XMLRPC::ArtistMetadataRequest::create ((*MData.Iter).artist.get());
      m_artist_request->reply().connect (sigc::mem_fun (*this, &TrackDetails::got_artist));
      m_artist_request->run ();
      return;
    }
  }

  void
  TrackDetails::got_lyrics (std::string const& lyrics, bool have_em)
  {
    if (have_em)
    {
      m_widget_lyrics->get_buffer()->set_text (lyrics);
    }

    m_notebook_lyrics->set_current_page (0);
    m_ref_xml->get_widget ("button-refresh-lyrics")->set_sensitive (1);
  }

  void
  TrackDetails::get_lyrics (bool forced)
  {
    m_lyrics_request.clear ();
    m_notebook_lyrics->set_current_page (1);
    m_ref_xml->get_widget ("button-refresh-lyrics")->set_sensitive (0);
    m_lyrics_request = LyricWiki::TextRequest::create (m_widget_lyrics_artist->get_text(), m_widget_lyrics_title->get_text(), forced);
    m_lyrics_request->lyrics().connect (sigc::mem_fun (*this, &TrackDetails::got_lyrics));
    m_lyrics_request->run ();
  }

  void
  TrackDetails::display (TrackMetadata const& metadata)
  {
    Glib::Mutex::Lock L (MData.Lock);

    MData.V.push_back (metadata);
    MData.Iter = MData.V.begin();

    static boost::format title_f1 ("%s - Track Details (BMP)");
    static boost::format title_f2 ("%s / %s - Track Details (BMP)");

    if((*MData.Iter).artist && (*MData.Iter).title)
    {
      set_title ((title_f2 % (*MData.Iter).artist.get() % (*MData.Iter).title.get()).str());
    }
    else
    if((*MData.Iter).title)
    { 
      set_title ((title_f1 % (*MData.Iter).title.get()).str());
    }
    else
    {
      set_title (_("(Unknown Title) - Track Details (BMP)"));
    }

    if ((*MData.Iter).new_item)
      m_widget_trackinfo->set_track ((*MData.Iter).location.get());
    else
      m_widget_trackinfo->set_track ((*MData.Iter));

    if ((*MData.Iter).artist)
      m_widget_lyrics_artist->set_text( (*MData.Iter).artist.get() );

    if ((*MData.Iter).title)
      m_widget_lyrics_title->set_text( (*MData.Iter).title.get() );

    if ((*MData.Iter).image)
    {
      m_widget_cover->set ((*MData.Iter).image->scale_simple (256, 256, Gdk::INTERP_BILINEAR));
    }
    else if ((*MData.Iter).asin && Network::check_connected())
    {
      try{
        RefPixbuf cover; 
        Amazon::Covers::Obj()->fetch ((*MData.Iter).asin.get(), cover, true);
        m_widget_cover->set (cover->scale_simple (256, 256, Gdk::INTERP_BILINEAR));
      }
      catch (...)
      {
        m_widget_cover->set (m_cover);
      }
    }
    else
    {
        m_widget_cover->set (m_cover);
    }

    get_lyrics (0);
    get_artist ();
   
    show_all (); 
  }
} // namespace Bmp
