// Copyright (c) 2011 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "base/command_line.h"
#include "base/file_util.h"
#include "base/path_service.h"
#include "chrome/browser/ui/browser.h"
#include "chrome/common/chrome_paths.h"
#include "chrome/test/base/in_process_browser_test.h"
#include "chrome/test/base/ui_test_utils.h"
#include "content/public/common/content_switches.h"
#include "net/base/net_util.h"

namespace {

class WebGLConformanceTests : public InProcessBrowserTest {
 public:
  WebGLConformanceTests() {}

  virtual void SetUpCommandLine(CommandLine* command_line) {
    // This enables DOM automation for tab contents.
    EnableDOMAutomation();

    // Allow privileged WebGL extensions.
    command_line->AppendSwitch(switches::kEnablePrivilegedWebGLExtensions);
  }

  virtual void SetUpInProcessBrowserTestFixture() {
    FilePath webgl_conformance_path;
    PathService::Get(base::DIR_SOURCE_ROOT, &webgl_conformance_path);
    webgl_conformance_path = webgl_conformance_path.Append(
        FILE_PATH_LITERAL("third_party"));
    webgl_conformance_path = webgl_conformance_path.Append(
        FILE_PATH_LITERAL("webgl_conformance"));
    ASSERT_TRUE(file_util::DirectoryExists(webgl_conformance_path))
        << "Missing conformance tests: " << webgl_conformance_path.value();

    PathService::Get(chrome::DIR_TEST_DATA, &test_path_);
    test_path_ = test_path_.Append(FILE_PATH_LITERAL("gpu"));
    test_path_ = test_path_.Append(FILE_PATH_LITERAL("webgl_conformance.html"));
  }

  void RunTest(const std::string& url) {
    ui_test_utils::DOMMessageQueue message_queue;
    ui_test_utils::NavigateToURL(browser(), net::FilePathToFileURL(test_path_));
    ui_test_utils::NavigateToURL(
        browser(), GURL("javascript:start('" + url + "');"));

    std::string message;
    // Wait for message indicating the test has finished running.
    ASSERT_TRUE(message_queue.WaitForMessage(&message));
    EXPECT_STREQ("\"SUCCESS\"", message.c_str()) << message;
  }

 private:
  FilePath test_path_;
};

#define CONFORMANCE_TEST(name, url) \
IN_PROC_BROWSER_TEST_F(WebGLConformanceTests, name) { \
  RunTest(url); \
}

// The test declarations are located in webgl_conformance_test_list_autogen.h,
// because the list is automatically generated by a script.
// See: generate_webgl_conformance_test_list.py
#include "webgl_conformance_test_list_autogen.h"

}  // namespace
