// Copyright (c) 2011 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef NET_BASE_SSL_FALSE_START_BLACKLIST_H_
#define NET_BASE_SSL_FALSE_START_BLACKLIST_H_

#include <string>

#include "base/logging.h"
#include "net/base/net_export.h"

namespace net {

// SSLFalseStartBlacklist is a set of domains which we believe to be intolerant
// to TLS False Start. Because this set is several hundred long, it's
// precompiled by the code in ssl_false_start_blacklist_process.cc into a hash
// table for fast lookups.
class SSLFalseStartBlacklist {
 public:
  // Returns true if |host| (a DNS name in dotted form, e.g. "www.example.com")
  // is in the blacklist.
  NET_EXPORT_PRIVATE static bool IsMember(const std::string& host);

  // Returns the modified djb2 hash of |host|.
  // NOTE: This is inline because the code which generates the hash table needs
  // to use it. However, the generating code cannot link against
  // ssl_false_start_blacklist.cc because that needs the tables which it
  // generates.
  static uint32 Hash(const std::string& host) {
    uint32 hash = 5381;
    for (const uint8* in = reinterpret_cast<const uint8*>(host.c_str());
         *in != 0; ++in)
      hash = ((hash << 5) + hash) ^ *in;
    return hash;
  }

  // Returns the last two dot-separated components of |host|, ignoring any
  // trailing dots.  For example, returns "c.d" for "a.b.c.d.".  Returns an
  // empty string if |host| does not have two dot-separated components.
  // NOTE: Inline for the same reason as Hash().
  static std::string LastTwoComponents(const std::string& host) {
    size_t last_nondot = host.find_last_not_of('.');
    if (last_nondot == std::string::npos)
      return std::string();
    size_t last_dot = host.find_last_of('.', last_nondot);
    if ((last_dot == 0) || (last_dot == std::string::npos))
      return std::string();
    // NOTE: This next line works correctly even when the call returns npos.
    size_t components_begin = host.find_last_of('.', last_dot - 1) + 1;
    return host.substr(components_begin, last_nondot - components_begin + 1);
  }

  // This is the number of buckets in the blacklist hash table. (Must be a
  // power of two).
  static const size_t kBuckets = 128;

 private:
  // The following two members are defined in
  // ssl_false_start_blacklist_data.cc, which is generated by
  // ssl_false_start_blacklist_process.cc

  // kHashTable contains an offset into |kHashData| for each bucket. The
  // additional element at the end contains the length of |kHashData|.
  static const uint32 kHashTable[kBuckets + 1];
  // kHashData contains the contents of the hash table. |kHashTable| indexes
  // into this array. Each bucket consists of zero or more, 8-bit length
  // prefixed strings. Each string is a DNS name in dotted form. For a given
  // string x, x and *.x are considered to be in the blacklist. In order to
  // assign a string to a hash bucket, the last two labels (not including the
  // root label) are hashed. Thus, the bucket for "www.example.com" is
  // Hash("example.com"). No names that are less than two labels long are
  // included in the blacklist.
  static const char kHashData[];
};

}  // namespace net

#endif  // NET_BASE_SSL_FALSE_START_BLACKLIST_H_
