/*
 * Copyright (c) 2010, Google, Inc.
 *
 * This file is part of FFmpeg.
 *
 * FFmpeg is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * FFmpeg is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with FFmpeg; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */

/*
 * @file
 * WebP muxer
 * http://code.google.com/p/webp
 *
 * supports metadata: artist, copyright, title, comment
 * Encoding/decoding examples:
 *  ffmpeg -i my_image
 *     -metadata artist=me -metadata title="sunset"
 *     -metadata copyright=2010 -metadata comment="nice pic!"
 *     -profile 1 -qmin 5 my_image.webp
 *  ffmpeg -i my_image.webp -y my_image.png
 */

#include "webp.h"

typedef struct {
  uint64_t riff_start;
} WEBPContext;

static int write_header(AVFormatContext *s)
{
    ByteIOContext *pb = s->pb;
    WEBPContext *webp = s->priv_data;
    AVStream *stream;
    AVCodecContext *codec;

    if (s->nb_streams != 1) {
        av_log(s, AV_LOG_ERROR, "muxer only support 1 video stream.");
        return AVERROR(EINVAL);
    }
    stream = s->streams[0];
    codec = stream->codec;
    if (codec->codec_type != AVMEDIA_TYPE_VIDEO)
        return AVERROR(EINVAL);
    if (codec->codec_id != CODEC_ID_VP8)
       return AVERROR(EINVAL);

    webp->riff_start = ff_start_tag(pb, "RIFF");
    put_tag(pb, "WEBP");
    return 0;
}

static int write_packet(AVFormatContext *s, AVPacket *pkt)
{
    AVStream *stream = s->streams[pkt->stream_index];
    AVCodecContext *codec = stream->codec;
    ByteIOContext *pb = s->pb;
    uint64_t vp8_start;

    if (codec->codec_id != CODEC_ID_VP8) {
        av_log(s, AV_LOG_ERROR, "muxer only supports VP8 codec.");
        return AVERROR(EINVAL);
    }
    vp8_start = ff_start_tag(pb, "VP8 ");
    put_buffer(pb, pkt->data, pkt->size);
    if (pkt->size & 1) put_byte(pb, 0);   // pad
    ff_end_tag(pb, vp8_start);
    put_flush_packet(pb);
    return 0;
}

static void put_metadata(AVFormatContext *s,
                         const AVMetadataConv* metadata)
{
    ByteIOContext *pb = s->pb;
    AVMetadataTag *t;
    uint64_t pos;
    int len;
    t = av_metadata_get(s->metadata, metadata->generic, NULL, 0);
    if (!t) return;
    pos = ff_start_tag(pb, metadata->native);
    len = strlen(t->value) + 1;
    put_buffer(pb, t->value, len);
    if (len & 1) put_byte(pb, 0);   // pad
    ff_end_tag(pb, pos);
}

static int write_trailer(AVFormatContext *s)
{
    ByteIOContext *pb = s->pb;
    WEBPContext *webp = s->priv_data;
    int i;
    for (i = 0; ff_webp_metadata_conv[i].native; ++i) {
        put_metadata(s, &ff_webp_metadata_conv[i]);
    }
    ff_end_tag(pb, webp->riff_start);
    put_flush_packet(pb);
    return 0;
}

AVOutputFormat webp_muxer = {
    .name           = "webp",
    .long_name      = NULL_IF_CONFIG_SMALL("WebP"),
    .mime_type      = "image/webp",
    .extensions     = "webp",
    .priv_data_size = sizeof(WEBPContext),
    .audio_codec    = CODEC_ID_NONE,
    .video_codec    = CODEC_ID_VP8,
    .write_header   = write_header,
    .write_packet   = write_packet,
    .write_trailer  = write_trailer,
};
