// Copyright 2016 The go-ethereum Authors
// This file is part of the go-ethereum library.
//
// The go-ethereum library is free software: you can redistribute it and/or modify
// it under the terms of the GNU Lesser General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// The go-ethereum library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public License
// along with the go-ethereum library. If not, see <http://www.gnu.org/licenses/>.

package light

import (
	"math"
	"math/big"
	"testing"
	"time"

	"github.com/ethereum/go-ethereum/common"
	"github.com/ethereum/go-ethereum/core"
	"github.com/ethereum/go-ethereum/core/types"
	"github.com/ethereum/go-ethereum/ethdb"
	"github.com/ethereum/go-ethereum/event"
	"github.com/ethereum/go-ethereum/params"
	"golang.org/x/net/context"
)

type testTxRelay struct {
	send, nhMined, nhRollback, discard int
}

func (self *testTxRelay) Send(txs types.Transactions) {
	self.send = len(txs)
}

func (self *testTxRelay) NewHead(head common.Hash, mined []common.Hash, rollback []common.Hash) {
	self.nhMined = len(mined)
	self.nhRollback = len(rollback)
}

func (self *testTxRelay) Discard(hashes []common.Hash) {
	self.discard = len(hashes)
}

const poolTestTxs = 1000
const poolTestBlocks = 100

// test tx 0..n-1
var testTx [poolTestTxs]*types.Transaction

// txs sent before block i
func sentTx(i int) int {
	return int(math.Pow(float64(i)/float64(poolTestBlocks), 0.9) * poolTestTxs)
}

// txs included in block i or before that (minedTx(i) <= sentTx(i))
func minedTx(i int) int {
	return int(math.Pow(float64(i)/float64(poolTestBlocks), 1.1) * poolTestTxs)
}

func txPoolTestChainGen(i int, block *core.BlockGen) {
	s := minedTx(i)
	e := minedTx(i + 1)
	for i := s; i < e; i++ {
		block.AddTx(testTx[i])
	}
}

func TestTxPool(t *testing.T) {
	for i, _ := range testTx {
		testTx[i], _ = types.NewTransaction(uint64(i), acc1Addr, big.NewInt(10000), params.TxGas, nil, nil).SignECDSA(types.HomesteadSigner{}, testBankKey)
	}

	var (
		evmux   = new(event.TypeMux)
		pow     = new(core.FakePow)
		sdb, _  = ethdb.NewMemDatabase()
		ldb, _  = ethdb.NewMemDatabase()
		genesis = core.WriteGenesisBlockForTesting(sdb, core.GenesisAccount{Address: testBankAddress, Balance: testBankFunds})
	)
	core.WriteGenesisBlockForTesting(ldb, core.GenesisAccount{Address: testBankAddress, Balance: testBankFunds})
	// Assemble the test environment
	blockchain, _ := core.NewBlockChain(sdb, testChainConfig(), pow, evmux)
	chainConfig := &params.ChainConfig{HomesteadBlock: new(big.Int)}
	gchain, _ := core.GenerateChain(chainConfig, genesis, sdb, poolTestBlocks, txPoolTestChainGen)
	if _, err := blockchain.InsertChain(gchain); err != nil {
		panic(err)
	}

	odr := &testOdr{sdb: sdb, ldb: ldb}
	relay := &testTxRelay{}
	lightchain, _ := NewLightChain(odr, testChainConfig(), pow, evmux)
	lightchain.SetValidator(bproc{})
	txPermanent = 50
	pool := NewTxPool(testChainConfig(), evmux, lightchain, relay)

	for ii, block := range gchain {
		i := ii + 1
		ctx, _ := context.WithTimeout(context.Background(), 200*time.Millisecond)
		s := sentTx(i - 1)
		e := sentTx(i)
		for i := s; i < e; i++ {
			relay.send = 0
			pool.Add(ctx, testTx[i])
			got := relay.send
			exp := 1
			if got != exp {
				t.Errorf("relay.Send expected len = %d, got %d", exp, got)
			}
		}

		relay.nhMined = 0
		relay.nhRollback = 0
		relay.discard = 0
		if _, err := lightchain.InsertHeaderChain([]*types.Header{block.Header()}, 1); err != nil {
			panic(err)
		}
		time.Sleep(time.Millisecond * 30)

		got := relay.nhMined
		exp := minedTx(i) - minedTx(i-1)
		if got != exp {
			t.Errorf("relay.NewHead expected len(mined) = %d, got %d", exp, got)
		}

		got = relay.discard
		exp = 0
		if i > int(txPermanent)+1 {
			exp = minedTx(i-int(txPermanent)-1) - minedTx(i-int(txPermanent)-2)
		}
		if got != exp {
			t.Errorf("relay.Discard expected len = %d, got %d", exp, got)
		}
	}
}
