{-
    Hima - Himerge Authentication module for Privileged Processes.  
    Copyright (C) 2007, 2008 Luis Francisco Araujo <araujo@gentoo.org>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

    ---------------------------------------------------------------------------
      Interface:
      hima <communication_file> <command>
-}

module Main
    where

import System.IO
import System.Environment
import System.Posix.User
import System.Posix.Types
import System.Posix.Files
import System.Process
import System.Posix.Process
import System.Exit
import System.Directory
import Control.Concurrent
import qualified Control.Exception as E

data HCF = HCF { pid :: ProcessID }

groupname :: String
groupname = "himerge"

run :: [String] -> IO ()
run str = 
    do
      let (file, command) = (head str, tail str)
      e <- getEffectiveUserID
      setUserID e
      g <- getEffectiveGroupID
      setGroupID g
      p <- getProcessID
      createHCFile file (HCF { pid = p })
      ph <- runCommand $ unwords command
      ph `onExit` ferror

simpleRun :: String -> IO ()
simpleRun cmd =
    do
      e <- getEffectiveUserID
      setUserID e
      g <- getEffectiveGroupID
      setGroupID g
      ph <- runCommand cmd
      ph `onExit` ferror

onExit :: ProcessHandle -> (ExitCode -> IO ()) -> IO ()
onExit ph func = 
    do
    threadDelay 3000
    exitcode <- getProcessExitCode ph
		     `E.catch` (\ _ -> getProcessExitCode ph)
    case exitcode of
        Nothing -> onExit ph func
	Just status -> func status

ferror :: ExitCode -> IO ()
ferror ExitSuccess = return ()
ferror (ExitFailure e) = error ("hima error: " ++ (show e))

createHCFile :: FilePath -> HCF -> IO ()
createHCFile path hcf = writeFile path (show $ pid hcf)

runAuthCommands :: [String] -> IO ()
runAuthCommands commands =
    case commands of
      a@(_:"/usr/bin/emerge":_) -> run a
      b@(_:"/usr/bin/update-eix":_) -> run b
      ("copy":old:new:_) -> copy old new
      ("kill":p:_) -> simpleRun ("kill -9 " ++ p)
      ("t":_) -> simpleRun "echo \"m00\""
      _ -> error "Not a valid auth option."

copy :: FilePath -> FilePath -> IO ()
copy old new = 
    do
      fstat <- getFileStatus new
      readFile old >>= writeFile new
      removeFile old
      setFileMode new (fileMode fstat)
      setOwnerAndGroup new (fileOwner fstat) (fileGroup fstat)

checkUserInGroup :: String -> [String] -> Bool
checkUserInGroup = any . (==)

main :: IO ()
main =
    do
      uid <- getRealUserID
      entryuid <- getUserEntryForID uid
      let loginame = userName entryuid
      gentry <- getGroupEntryForName groupname
      if (checkUserInGroup loginame (groupMembers gentry))
         then getArgs >>= runAuthCommands
         else error (loginame ++ " doesn't belong to group " ++ groupname)