{-
    Portage's emerge command entry functions.
    Copyright (C) 2007, 2008 Luis Francisco Araujo <araujo@gentoo.org>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
-}

module Emerge
    where

import Util
import SCKeys
import Graphics.UI.Gtk
import Data.Char (isAlpha)
import System.Directory
import System.IO
import System.Process
import Control.Concurrent
import System.Exit
import qualified Control.Exception as E


data EmergeParse = EmergeSuccessDep [String] | EmergeBlockError String 
		   deriving Show

blockerrormsg :: String
blockerrormsg = "\nFor more information about Blocked Packages, please refer to the following\n\
		 \section of the Gentoo Linux x86 Handbook (architecture is irrelevant):\n\
		 \\n\      
                 \http://www.gentoo.org/doc/en/handbook/handbook-x86.xml?full=1#blocked"

emergeDeps :: Handle -> IO [(String, String)]
emergeDeps out =
    do
    output <- hGetContents out
    E.evaluate output    
    let pkgs = lines $ dropWhile (/= '[') output
    case checkDepOutput pkgs [] of
      EmergeSuccessDep parsedpkgs -> return $ map f $ takeWhile (not . null) $ parsedpkgs
      EmergeBlockError xs -> popErrorWindow xs >> return []
    where
	f pkg = let (s:rs) = words $ dropWhile (/= ' ') pkg 
                   in
		     let p = dropWhile (not . isAlpha) $ unwords rs in (filter isAlpha s, p)

checkDepOutput :: [String] -> [String] -> EmergeParse
checkDepOutput [] pkgs = EmergeSuccessDep pkgs
checkDepOutput (('[':'b':'l':'o':'c':'k':'s':xs):_) _ =
    EmergeBlockError ((drop 2 $ dropWhile (/= ']') xs) ++ blockerrormsg)
checkDepOutput (('*':'*':'*':' ':'P':_):xs) pkgs = checkDepOutput xs pkgs
checkDepOutput ((' ':_):xs) pkgs = checkDepOutput xs pkgs
checkDepOutput (x:xs) pkgs = checkDepOutput xs (pkgs ++ [x])

emergeDepsClean :: Handle -> IO [(String, String)]
emergeDepsClean out =
    do
      output <- hGetContents out
      E.evaluate output
      let pkglist = lines output
      return $ findDeps (map strip pkglist)
    where
      findDeps [] = []
      findDeps (x:ys@(y:xs))
          | any (== '/') x = case y of
                               ('s':'e':'l':'e':'c':'t':'e':'d':':':' ':version) -> ("DC", (x ++ "-" ++ version)) : findDeps xs
                               _ -> findDeps ys
          | otherwise = findDeps ys
      findDeps (_:zs) = findDeps zs

equeryOutput :: Handle -> IO [(String, String)]
equeryOutput out = 
    do
    output <- hGetContents out
    E.evaluate output
    let xs = filter (not . null) $ map (takeWhile (/= ' ')) $ lines output
    return $ map ((,) "Y") xs

emergeError :: String -> Handle -> IO ()
emergeError = flip ((>>=) . hGetContents) . (popErrorWindow .) . (++)

sync :: Notebook -> TextView -> Statusbar -> ProgressBar -> IO ()
sync = commandView (emergecmd ++ "--sync", "Updating portage tree",
	            "Portage tree successfully updated.", "Error updating portage tree.") EmergeSync

emergeInfo, emergeMetadata :: Notebook -> TextView -> Statusbar -> ProgressBar -> IO ()
emergeInfo = commandView (emergecmd ++ "--info", "Showing portage information",
			  "Portage information.", "Error showing portage information.") Emerge
emergeMetadata = commandView (emergecmd ++ "--metadata", "Regenerating metadata",
			      "Portage metadata regenerated.", "Error regenerating metadata.") EmergeSync

emerge :: [String] -> Notebook -> TextView -> Statusbar -> ProgressBar 
       -> String -> (TreeView, ListStore, [(String, TreeIter)]) -> IO ()
emerge _ _ _ statbar _ [] (_, _, []) = 
    updateStatBar statbar "No package on queue." >> return ()
emerge _ _ _ statbar _ pkgmsg (_, _, []) =
    updateStatBar statbar pkgmsg
emerge msgs panel logview statbar progbar _ (tv, st, ((package, pkgiter):ps)) =
    do
    -- Create the notebook page.
    (pkgscroll, pkgview) <- makeView False
    -- Build the page panel with the proper tab buttons.
    (tooltips, (Just stopbutton), (Just closebutton))
	<- buildPanelTab pkgscroll panel package StopCloseButton
    widgetShowAll pkgscroll
    -- Jump to the page processing the operation.
    notebookSetCurrentPage panel (-1)
    -- Start emerge operation.
    -- Set package status on queue.
    progressBarSetFraction progbar 0.0
    -- Log information.
    let loginfo = ((msgs !! 1) ++ " " ++ package)
    -- Write the log.
    writeLog logview loginfo
    -- Keep updating the status bar with the log information.
    statid <- forkIO $ refreshStatBar statbar (loginfo ++ " ...")
    listStoreSetValue st pkgiter 0 (GVstring $ Just (setQueueColor "green" (msgs !! 1)))
    listStoreSetValue st pkgiter 4 (GVstring $ Just "purple")
    scrollViewToCell st pkgiter tv
    -- Run 'emerge' through the authentication module.
    -- Get the process handles out of it.
    (hcfile, hchandle) <- writeTempFile "hcfile"
    (_,out,err,ph) <- runInteractiveCommand (hima++" "++hcfile++" "++(msgs !! 0)++" ="++package)
    barthread <- forkIO $ updateBar progbar
    -- show stdout buffer.
    ebuf <- textViewGetBuffer pkgview
    forkIO $ showEmergeBuffer Emerge out pkgview ebuf
    -- show stderr buffer.
    forkIO $ showEmergeBuffer Emerge err pkgview ebuf
    -- Connect signal to the stop-process and close-tab buttons.
    stopbutton `onClicked` 
	       do
		 exitcode <- getProcessExitCode ph
		 case exitcode of
		   Nothing -> do
                                popSelectWindow ("Stop " ++ (msgs !! 1) ++ " " ++ package)
                                                (killHandleThroughAuth hchandle ph)
                   Just _ -> popInfoWindow "This process is already stopped."
    let closefunc = (do
	              exitcode <- getProcessExitCode ph
		      case exitcode of
	                Nothing -> popWarningWindow "The process is still running.\n\
						    \Stop this process first and then close the tab."
	                Just _ -> do
		          (Just text) <- notebookGetMenuLabelText panel pkgscroll
		          popSelectWindow ("Do you really want to close [" ++ text ++ "]?")
                 		              (tooltipsEnable tooltips >> closeTab panel pkgscroll))
    closebutton `onClicked` closefunc
    pkgview `onKeyPress` closeProcTab closefunc
    -- Avoid individual tooltips to be GC'ed.
    -- Test for process termination.
    forkIO $ onExit ph ((.) ((>>) (killThread statid)) (handleExitCode barthread (hcfile, hchandle)))
    return ()
	where
	handleExitCode thread (hcfile, hchandle) exitc =
            do
	      killThread thread
              hSeek hchandle AbsoluteSeek 0
              iseof <- hIsEOF hchandle
              exitcode <- 
                  if not iseof
                  then do
                    stat <- hGetLine hchandle
                    if stat == "killed" then return (ExitFailure 115) else return exitc
                  else return exitc
              removeFile hcfile
	      case exitcode of
	        ExitSuccess -> do
			      progressBarSetFraction progbar 1.0 
			      listStoreSetValue st pkgiter 0 (GVstring $ Just (setQueueColor "purple" (msgs !! 2)))
			      listStoreSetValue st pkgiter 4 (GVstring $ Just "lightgreen")
			      scrollViewToCell st pkgiter tv
			      let rlog = (package ++ " => " ++ (msgs !! 2) ++ " successfully.") 
			      writeLog logview rlog
			      emerge msgs panel logview statbar progbar rlog (tv, st, ps)
	        ExitFailure 115 -> do
			      listStoreSetValue st pkgiter 0 (GVstring $ Just (setQueueColor "blue" "Stopped"))
			      listStoreSetValue st pkgiter 4 (GVstring $ Just "red")
			      scrollViewToCell st pkgiter tv
			      let rlog =  ((msgs !! 1) ++ " " ++ package ++ " stopped.")
			      writeLog logview rlog
			      updateStatBar statbar rlog
			      progressBarSetFraction progbar 0.0
	        ExitFailure _ -> do 
			      listStoreSetValue st pkgiter 0 (GVstring $ Just (setQueueColor "black" (msgs !! 3)))
			      listStoreSetValue st pkgiter 4 (GVstring $ Just "red")
			      scrollViewToCell st pkgiter tv
			      let rlog = ((msgs !! 3) ++ " " ++ package ++ ".")
			      popErrorWindow rlog
			      writeLog logview rlog
			      updateStatBar statbar rlog
			      progressBarSetFraction progbar 0.0

packagesToEmerge :: ListStore -> IO (ListStore, [(String, TreeIter)])
packagesToEmerge st = collectIters st 0 >>= mapM f >>= return . (,) st
	where f i = do
		    (GVstring (Just pkgatom)) <- treeModelGetValue st i 3
		    return $ (takeWhile (/= ' ') pkgatom, i)

{- | Emerge functions. -}
emergePackages, unmergePackages, fetchPackages
    :: Notebook -> TextView -> Statusbar -> ProgressBar -> IO ()
binaryPackages, useBinaryPackages
    :: Notebook -> TextView -> Statusbar -> ProgressBar -> IO ()

emergeOperation :: Notebook -> TextView -> Statusbar 
		-> ProgressBar -> [String] -> IO ()
emergeOperation panel logview statbar progbar info =
    isCurrentPagePackagePanel panel
       (\ pkgtv -> getStoreFromView pkgtv >>= packagesToEmerge >>=
	\ (st, xs) -> emerge info panel logview statbar progbar [] (pkgtv, st, xs))

emergePackages panel logview statbar progbar =
    emergeOperation panel logview statbar progbar
       [emergecmd, "Installing", "Installed", "Error installing"]

unmergePackages panel logview statbar progbar =
    emergeOperation panel logview statbar progbar
       [emergecmd ++ " --unmerge" , "Uninstalling", "Uninstalled", "Error uninstalling"]
	
fetchPackages panel logview statbar progbar =
    emergeOperation panel logview statbar progbar
       [emergecmd ++ " --nodeps --fetchonly", "Fetching", "Fetched", "Error fetching"]

binaryPackages panel logview statbar progbar =
    emergeOperation panel logview statbar progbar
       [emergecmd ++ " --buildpkg", "Installing (building binary package too)"
       , "Installed", "Error installing"]

useBinaryPackages panel logview statbar progbar =
    emergeOperation panel logview statbar progbar
       [emergecmd ++ " --usepkg", "Installing (using binary package if available)"
       , "Installed", "Error installing"]