{-
    Himerge initialization routines.
    Copyright (C) 2007, 2008 Luis Francisco Araujo <araujo@gentoo.org>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
-}

module Init
    where

import Util
import Components
import System.Console.GetOpt
import System.FilePath.Posix
import System.Environment
import System.Directory
import System.Exit


data Flag = Version | Help | None | WebBrowser String | PortageRepo String
            deriving Show

header :: String
header = comment ++ "\n" ++ url ++ "\n" ++ copyright ++
         "\n\nUsage: himerge [options]\n"

himergeDir :: IO FilePath
himergeDir = getAppUserDataDirectory "himerge" >>= 
             return . setPathTrailingSeparator

files :: [String]
files = [makeconfpath]

binaries :: [String]
binaries = [eixpath, equerypath, updateeix, emergepath]

initialize :: IO [Flag]
{- | Main initialization function entry point. -}
initialize =
    do
      args <- getArgs
      (opts, _) <- parseArgs args
      -- Hey, let's check command line help options first!
      helperOpts opts
      checkForFiles
      checkForBinaries
      initDirectory
      return opts

helperOpts :: [Flag] -> IO ()
helperOpts flags =
    case findHelperOpt flags of
      Help -> putStrLn (usageInfo header options) >> exitWith ExitSuccess
      Version -> putStrLn (hname ++ " version " ++ hversion) >> exitWith ExitSuccess
      _ -> return ()

initDirectory :: IO ()
initDirectory =
    do
      cfgdir <- himergeDir
      createDirectoryIfMissing True cfgdir
      mapM_ backUpFiles [makeconfpath, packagekeywords, packagemask, packageuse]

options :: [OptDescr Flag]
options =
    [ Option ['v'] ["version"]   (NoArg Version)               "show version number",
      Option ['h'] ["help"]      (NoArg Help)                  "show help menu",
      Option ['w'] ["webrowser"] (ReqArg WebBrowser "browser") "choose web browser",
      Option ['r'] ["repo"]      (ReqArg PortageRepo "path")   "portage repository path" ]

parseArgs :: [String] -> IO ([Flag], [String])
parseArgs args =
    case getOpt Permute options args of
      (o, n, []) -> return (o, n)
      (_, _, errs) -> ioError (userError (concat errs ++ (usageInfo header options)))

findOpt :: [Flag] -> String -> String
findOpt [] _ = []
findOpt (PortageRepo r:flags) opt
    | opt == "portagerepo" = r
    | otherwise = findOpt flags opt
findOpt (WebBrowser w:flags) opt
    | opt == "webrowser" = w
    | otherwise = findOpt flags opt
findOpt flags opt = findOpt flags opt

findHelperOpt :: [Flag] -> Flag
findHelperOpt [] = None
findHelperOpt (Version:_) = Version
findHelperOpt (Help:_) = Help
findHelperOpt (_:flags) = findHelperOpt flags

getDirname :: FilePath -> FilePath
getDirname = takeFileName .  dropTrailingPathSeparator

checkForFiles :: IO ()
checkForFiles =
    mapM_ (\ f -> do
             isfile <- doesFileExist f 
             if isfile
                then return ()
                else do
                  isdir <- doesDirectoryExist f
                  case isdir of 
                    {
                      False -> popErrorWindow ("'" ++ f ++ "' does not exist!\n\
                                               \ You need '" ++ f ++ "'to run Himerge.") >>
                               exitFailure
                    ;
                      True -> return ()
                    }) files

checkForBinaries :: IO ()
checkForBinaries =
    mapM_ (\ bin -> do
             -- Commands contain a white space at the end. Remove it.
             b <- doesFileExist $ init bin
             if b
                then
                    return ()
                else
                    popErrorWindow ("Program '" ++ bin ++ "' does not exist.\n\
                                    \ You need to install this program to run Himerge.") >>
                    exitFailure) binaries

backUpFiles :: FilePath -> IO ()
backUpFiles filepath =
    do
      isfile <- doesFileExist filepath
      case isfile of
        True -> himergeDir >>= 
                \ p -> copyFile filepath (joinPath [p , (getDirname filepath)])
        False ->
            do
              isdir <- doesDirectoryExist filepath
              case isdir of
                True -> himergeDir >>= 
                        \ p -> return (joinPath [p , (getDirname filepath)]) >>=
                        makeCFGDirectory filepath
                False -> return ()

makeCFGDirectory :: FilePath -> FilePath -> IO ()
makeCFGDirectory destdir newdir =
    do
      createDirectoryIfMissing True newdir
      contents <- getDirectoryContents destdir
      mapM_ (\ f -> do
               let dir = joinPath [destdir, f]
               isdir <- doesDirectoryExist dir
               case isdir of
                 True -> makeCFGDirectory dir (joinPath [newdir, f])
                 False -> copyFile dir (joinPath [newdir, f])) (removeOddDirs contents)
