{-
    Functions for processes operations related to /proc.
    Copyright (C) 2007, 2008 Luis Francisco Araujo <araujo@gentoo.org>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
-}

module Proc (findAllChildrenFromPID)
    where

import System.Directory
import Data.Char
import System.FilePath.Posix


data PID = PID String String

proc :: FilePath
proc = "/proc"

getAllProcessesDir :: IO [String]
getAllProcessesDir =
    getDirectoryContents proc >>= return . filter (all isNumber) >>=
    return . map (joinPath . ((proc :) . return))

readPID :: FilePath -> IO PID
readPID procpath =
    do
      let file = joinPath [procpath, "status"]
      b <- doesFileExist file
      case b of
        True -> do
               str <- readFile file
               let pids = (f . map words . lines) str
               return (PID (head pids) (last pids))
        False -> return $ PID [] []
    where
      f [] = []
      f (("Pid:":pid):xs) = (head pid) : f xs
      f (("PPid:":pid):xs) = (head pid) : f xs
      f (_:xs) = f xs

findPID :: String -> [PID] -> String
findPID _ [] = []
findPID pid ((PID cpid parentid):xs)
    | parentid == pid = cpid
    | otherwise = findPID pid xs

childPID :: String -> IO String
{- Get the child PID of the process 'pid'. -}
childPID pid = getAllProcessesDir >>= mapM readPID >>=
               (return . findPID pid)

findAllChildrenFromPID :: String -> IO [String]
findAllChildrenFromPID pid = f pid []
    where
      f pid' children =
          do
            child <- childPID pid'
            case child of
              [] -> return $ reverse children
              _ -> f child (child : children)