/*
* IceBreaker
* Copyright (c) 2001-2002 Matthew Miller <mattdm@mattdm.org> and
*   Enrico Tassi <gareuselesinge@infinito.it>
*
* <http://www.mattdm.org/icebreaker/>
*
* This program is free software; you can redistribute it and/or modify it
* under the terms of the GNU General Public License as published by the Free
* Software Foundation; either version 2 of the License, or (at your option)
* any later version.
*
* This program is distributed in the hope that it will be useful, but
* WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
* or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
* for more details.
*
* You should have received a copy of the GNU General Public License along
* with this program; if not, write to the Free Software Foundation, Inc., 59
* Temple Place, Suite 330, Boston, MA 02111-1307 USA
*
*/

#include <SDL.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include "icebreaker.h"
#include "globals.h"
#include "icebreaker.h"
#include "options.h"
#include "themes.h"

GameOptionsType options;
GameOptionsType commandline;

GameFlagsType gameflags;

static int findflag(int argc,char** argv,char* option);
static void printversion(void);
static void printhelp(void);
static void printthemelist(void);

#define FLAGVERSION        "-v"
#define FLAGVERSIONLONG    "--version"
#define FLAGHELP           "-h"
#define FLAGHELPLONG       "--help"
#define FLAGSOUND          "-s"
#define FLAGSOUNDLONG      "--sound"
#define FLAGNOSOUND        "-n"
#define FLAGNOSOUNDLONG    "--nosound"
#define FLAGFULLSCREEN     "-f"
#define FLAGFULLSCREENLONG "--fullscreen"
#define FLAGWINDOWED       "-w"
#define FLAGWINDOWEDLONG   "--windowed"
#define FLAGTHEME          "-t"
#define FLAGTHEMELONG      "--theme"
#define FLAGLISTTHEMES     "-l"
#define FLAGLISTTHEMESLONG "--listthemes"
#define FLAGBENCHMARK      "-b"
#define FLAGBENCHMARKLONG  "--benchmark"

void setdefaultoptions(void)
{
	options.sound=SOUNDON;
	options.autopause=AUTOPAUSEOFF;	
	options.difficulty=NORMAL;
	options.fullscreen=FULLSCREENOFF;
	snprintf(options.theme,MAXTHEMENAMELENGTH+1,"linux");
}

int readoptions(void)
{
	FILE * optionfile;
	char linebuf[50];
	char filename[255]; // fix -- use defined OS constant
	char optbuf[21];
	char valbuf[10+MAXTHEMENAMELENGTH];
	char scanformat[20];
	int i;

	setdefaultoptions();
	
	snprintf(filename,255,"%s/%s",homedir,OPTIONFILE);
	
	optionfile=fopen(filename,"r");
	if (optionfile==NULL)
	{
		fprintf(stderr, OPTIONFILE " doesn't exist.\nWelcome to IceBreaker.\n");
		return true;
	}
	
	while(fgets(linebuf,50,optionfile))
	{
		for (i=0;i<50;i++)
		{
			if (linebuf[i]=='\0') break;
			linebuf[i]=tolower(linebuf[i]);
		}
		
		sprintf(scanformat,"%%20s %%%ds",10+MAXTHEMENAMELENGTH);	
		if (sscanf(linebuf,"%20s %10s",optbuf,valbuf)==2)
		{
			if (!strcmp(optbuf,"sound"))
			{
				if (!strcmp(valbuf,"on"))
					options.sound=SOUNDON;
				else if (!strcmp(valbuf,"off"))
					options.sound=SOUNDOFF;
			}
			else if (!strcmp(optbuf,"autopause"))
			{
				if (!strcmp(valbuf,"on"))
					options.autopause=AUTOPAUSEON;
				else if (!strcmp(valbuf,"off"))
					options.autopause=AUTOPAUSEOFF;
			}
			else if (!strcmp(optbuf,"fullscreen"))
			{
				if (!strcmp(valbuf,"off"))
					options.fullscreen=FULLSCREENOFF;
				else if (!strcmp(valbuf,"on"))
					options.fullscreen=FULLSCREENON;
				else if (!strcmp(valbuf,"always"))
					options.fullscreen=FULLSCREENALWAYS;
			}
			else if (!strcmp(optbuf,"difficulty"))
			{
				if (!strcmp(valbuf,"normal"))
					options.difficulty=NORMAL;
				else if (!strcmp(valbuf,"hard"))
					options.difficulty=HARD;
				else if (!strcmp(valbuf,"easy"))
					options.difficulty=EASY;
			}
			else if (!strcmp(optbuf,"theme"))
			{
				snprintf(options.theme,MAXTHEMENAMELENGTH+1,"%s",valbuf);
			}
			// FIX: add username
		}
	}
	
	fclose(optionfile);

	return false;
}

int writeoptions(void)
{
	FILE * optionfile;
	char filename[255];
	snprintf(filename,255,"%s/%s",homedir,OPTIONFILE);
	
	optionfile=fopen(filename,"w");
	if (optionfile==NULL)
	{
		fprintf(stderr, "Can't write to " OPTIONFILE ".\n");
		return true;
	}
	
	fprintf(optionfile,"# Icebreaker config file 1.0\n#\n");
	fprintf(optionfile,"# Separate keywords from values by whitespace. Not case sensitive.\n#\n");
	fprintf(optionfile,"# %s/" OPTIONFILE " will be overwritten automatically.\n#\n",homedir);
	
	fprintf(optionfile,"\n# Change this if the crashing noise annoys your neighbors.\n");
	if (options.sound==SOUNDON)
		fprintf(optionfile,"Sound On\n");
	else if (options.sound==SOUNDOFF)
		fprintf(optionfile,"Sound Off\n");
		
	fprintf(optionfile,"\n# AutoPause makes the game pause when the window is out of focus.\n");
	if (options.autopause==AUTOPAUSEON)
		fprintf(optionfile,"AutoPause On\n");
	else if (options.autopause==AUTOPAUSEOFF)
		fprintf(optionfile,"AutoPause Off\n");

	fprintf(optionfile,"\n# Set FullScreen to Always if you want it that way every time.\n");
	fprintf(optionfile,"# On will use full screen mode once, but then change back to Off.\n");
	if (options.fullscreen==FULLSCREENOFF || options.fullscreen==FULLSCREENON)
		fprintf(optionfile,"FullScreen Off\n");
	else if (options.fullscreen==FULLSCREENALWAYS)
		fprintf(optionfile,"FullScreen Always\n");

	fprintf(optionfile,"\n# Normal is the best way to play. Easy is okay to get started,\n");
	fprintf(optionfile,"# but you won't get very high scores. Hard is for those who really\n");
	fprintf(optionfile,"# want a challenge, but scores only slightly higher than normal.\n");
	

	if (options.difficulty==NORMAL)
		fprintf(optionfile,"Difficulty Normal\n");
	else if (options.difficulty==HARD)
		fprintf(optionfile,"Difficulty Hard\n");
	else if (options.difficulty==EASY)
		fprintf(optionfile,"Difficulty Easy\n");


	fprintf(optionfile,"\n# Themes provide an easy way to select (or tweak!) the appearance\n");
	fprintf(optionfile,"# and sound of the game. Select from " THEMEFILEEXTENSION " files in the game data\n");
	fprintf(optionfile,"# directory. Check the docs for info on creating your own themes, too.\n");	

	fprintf(optionfile,"Theme %s\n",options.theme);

	fclose(optionfile);
	
	return false;
}

int parsecommandline(int argc, char** argv)
{
	int i;

	commandline.sound=SOUNDON;
	commandline.autopause=AUTOPAUSEOFF;	
	commandline.difficulty=NORMAL;
	commandline.fullscreen=FULLSCREENUNKNOWN;
	*(commandline.theme)='\0'; // makes this ""

	if ( findflag(argc,argv,FLAGVERSION) || findflag(argc,argv,FLAGVERSIONLONG) ) 
	{
		printversion();
		return 1;
	}

	if ( findflag(argc,argv,FLAGHELP) || findflag(argc,argv,FLAGHELPLONG) ) 
	{
		printhelp();
		return 2;
	}

	if ( findflag(argc,argv,FLAGLISTTHEMES) || findflag(argc,argv,FLAGLISTTHEMESLONG) ) 
	{
		printthemelist();
		return 3;
	}

	if ( findflag(argc,argv,FLAGNOSOUND) || findflag(argc,argv,FLAGNOSOUNDLONG) ) 
		commandline.sound=SOUNDOFF;

	if ( findflag(argc,argv,FLAGBENCHMARK) || findflag(argc,argv,FLAGBENCHMARKLONG) ) 
		gameflags.benchmarkmode=true;
	else
		gameflags.benchmarkmode =false;


	if ( findflag(argc,argv,FLAGSOUND) || findflag(argc,argv,FLAGSOUNDLONG) )
	{
		if (commandline.sound==SOUNDOFF)
		{	
			fprintf(stderr,"You asked for sound to be both on and off. Sorry Schrodinger, we can't do that.\n"
			               "We'll assume \'off\' is what you really meant.\n");
		}
		else
		{
			commandline.sound=SOUNDON;  // redundant. but more clear. :)
		}
	}


	if ( findflag(argc,argv,FLAGFULLSCREEN) || findflag(argc,argv,FLAGFULLSCREENLONG) ) 
		commandline.fullscreen=FULLSCREENON;

	if ( findflag(argc,argv,FLAGWINDOWED) || findflag(argc,argv,FLAGWINDOWEDLONG) )
	{
		if (commandline.fullscreen==FULLSCREENON)
		{	
			fprintf(stderr,"You asked for both fullscreen and windowed mode. You're mad! Mad, I tell you.\n"
			               "We'll assume \'windowed\' is what you really meant.\n");
		}
		commandline.fullscreen=FULLSCREENOFF;
	}

	// find theme name -- can't use findflag because this option
	// takes a parameter and that function isn't that smart.
	for(i=1; i<argc; i++)
	{
		if (strncmp(argv[i],FLAGTHEME,strlen(FLAGTHEME)) == 0)
		{
			if (strlen(argv[i])==strlen(FLAGTHEME))
			{
				fprintf(stderr,"Warning: " FLAGTHEME " option given but no theme specified. Perhaps this is because you\n"
				               " wrote \"" FLAGTHEME " themename\" or \"" FLAGTHEME "=themename\". Sorry, I'm too dumb to understand\n"
		               	               " that -- you'll have to use the form " FLAGTHEME "themename or " FLAGTHEMELONG "=themename. As it\n"
	               		               " is, I'm just going to ignore you.\n");
			}
			else if (strlen(argv[i]) > MAXTHEMENAMELENGTH+strlen(FLAGTHEME))
			{
				fprintf(stderr,"Warning: the theme name you've given is too long. The maximum is %d characters.\n"
	               		               " Not because I don't understand long filenames, by the way -- it's so the\n"
				               " options menu looks nice. I'm very fussy about my appearance.\n",MAXTHEMENAMELENGTH);
			}
			else
			{
				// fix -- we should probably search for malicious characters here.
				snprintf(commandline.theme,MAXTHEMENAMELENGTH+1,"%s",argv[i]+strlen(FLAGTHEME));
			}	
		}
		else if (strncmp(argv[i],FLAGTHEMELONG "=" ,strlen(FLAGTHEMELONG "=")) == 0)
		{
			if (strlen(argv[i])==strlen(FLAGTHEMELONG "="))
			{
				fprintf(stderr,"Warning: " FLAGTHEMELONG "= what exactly? The theme name has to go right after the equals\n"
				               " sign or else I will ignore you. Which is what I'm doing now. La la la la la la\n"
		               	               " la la la I can't hear you la la la....\n");
			}
			else if (strlen(argv[i]) > MAXTHEMENAMELENGTH+strlen(FLAGTHEMELONG "="))
			{
				fprintf(stderr,"Warning: the theme name you've given is too long. The maximum is %d characters.\n"
	               		               " Not because I don't understand long filenames, by the way -- it's so the\n"
				               " options menu looks nice. I'm very fussy about my appearance.\n",MAXTHEMENAMELENGTH);
			}
			else
			{
				// fix -- we should probably search for malicious characters here.
				snprintf(commandline.theme,MAXTHEMENAMELENGTH+1,"%s",argv[i]+strlen(FLAGTHEMELONG "="));
			}	
		}
		else if (strncmp(argv[i],FLAGTHEMELONG,strlen(FLAGTHEMELONG)) == 0)
		{
			fprintf(stderr,"Warning: the " FLAGTHEMELONG " parameter needs to be followed directly by an \'=\' and\n"
			               " then the name of the theme you want. Like, \"" FLAGTHEMELONG "=linux\". Otherwise, I\n"
	               	               " will pretend to not understand you.\n");
		}
	

	}

	return 0;
}

int findflag(int argc,char** argv, char* option)
{
	int i;
	for (i=1; i<argc; i++)
	{	
		//printf("[%d] = %s\n",i,argv[i]);
		if (strcmp(argv[i],option) == 0)
			return true;
	}
	return false;
}

void printversion()
{
	printf("\nIceBreaker v%d.%d.%d  Copyright 2000-2002 Matthew Miller, et al.\n\n",VERMAJOR,VERMINOR,VERSUB);

	printf("Written by Matthew Miller with additional code and help from Enrico Tassi\n"
	       "and others. Like to contribute to the project? Good code, graphics, sounds,\n"
       	       "and even just advice are all appreciated and often even accepted.\n\n"
	       "Visit the IceBreaker web site at <http://www.mattdm.org/icebreaker/> for more\n"
               "information or to download the latest version.\n\n"
	       "This program is free software; you can redistribute it and/or modify it\n"
               "under the terms of the GNU General Public License as published by the Free\n"
	       "Software Foundation; either version 2 of the License, or (at your option)\n"
               "any later version. IceBreaker uses the SDL library, which is distributed\n"
	       "under the GNU LGPL; see <http://www.libsdl.org/> for details.\n\n");
}
      	        
void printhelp()
{
	printf("Usage: icebreaker [OPTION...]\n");
	printf("%20s, %-15s %-35s\n",FLAGNOSOUND,FLAGNOSOUNDLONG,"disable game sounds");
	printf("%20s, %-15s %-35s\n",FLAGFULLSCREEN,FLAGFULLSCREENLONG,"start fullscreen if possible");
	printf("%20s, %-15s %-35s\n",FLAGWINDOWED,FLAGWINDOWEDLONG,"start windowed (resets \"Always\" option)");
	printf("%20s, %-15s %-35s\n",FLAGVERSION,FLAGVERSIONLONG,"display version and copyright info");
	printf("%20s, %-15s %-35s\n",FLAGHELP,FLAGHELPLONG,"display this help screen");
	printf("%20s, %-15s %-35s\n",FLAGLISTTHEMES,FLAGLISTTHEMESLONG,"list available themes");
	printf("%18s%s,\n","",FLAGTHEME "themename");
	printf("%18s %-18s %-35s\n","",FLAGTHEMELONG "=themename","select theme");

	printf("\nFor game play help, see the included documentation, the in-game help, or the\n"
	       "web site at <http://www.mattdm.org/icebreaker/>\n\n");
}

void printthemelist()
{
	char** themelist;
	int themecount;
	int t;

	printf("I've found the following themes:\n\n");

	themecount=getthemenames(&themelist);
	if (themecount==0)
	{
		printf("  (none)\n");
		printf("\nThis probably means IceBreaker was installed incorrectly.\n");
	}
	else
	{
		for (t=0;t<themecount;t++)
			printf("  %s\n",themelist[t]);

		printf("\nAdditionally, you can use \'random\', which does about what you'd expect.\n");
        }
        freethemenames(&themelist,themecount);
}
