# $Header: /var/cvsroot/gentoo-src/java-config/python/java_config/jc_envgen.py,v 1.4 2004/07/03 06:33:44 aether Exp $

import os
import errno

from jc_util import collect_vms
from jc_exceptions import OptionRuntimeError

__version__ = "$Revision: 1.4 $" [11:-2]


def find_vm (name):
	'''
	Find a VM, match prefixes, raise an error if the VM couldn't be
	found or there was more than one match.
	'''

	vms = collect_vms()
	matches = [
		S for S in vms.keys()
		if S.lstrip('20').startswith(name) 
	]

	if not matches:
		raise OptionRuntimeError('VM not found, use --list-available-vms')

	if len(matches) > 1:

		print 'Matches:'

		for n in matches:
			print '\t%s' % (n[2:],)

		raise OptionRuntimeError('More than one VM matched the request')

	return vms[matches[0]]

		
class GenerateEnvEntry:

	def __init__ (self, vmName):

		self.vm = find_vm(vmName)
		self.vmName = vmName
		self.makers = []

	def installMaker (self, envMkr):

		self.makers.append(envMkr)

	def makeEnvEntry (self):

		self._doHeaders()

		for (command, value,) in self.vm.iteritems():

			handler = 'handle_' + command	
			self._mux(handler, command, value)

		self._cleanUp()

	def _doHeaders (self):

		for (maker) in self.makers:
			O = maker.getHeader(self.vmName)
			if O is not None:
				maker.doOutput(O)

	def _mux (self, handler, cmd, cmdval):
			
		for (maker) in self.makers:
			if hasattr(maker, handler):
				hdl = getattr(maker, handler)
			else:
				# if this fails then something is wrong
				hdl = maker.handle_special
				
			O = hdl(cmd, cmdval)

			#if O == 'export ':
			#	print hdl

			if O is not None:
				maker.doOutput(O)
		
	def _cleanUp (self):

		for maker in self.makers:
			maker.cleanUp()

class EnvMaker:
	
	def __init__ (self, system=False):

		self.system = system
		self.outputHandlers = []

	def installOutputHandler (self, handler):
		
		self.outputHandlers.append(handler)

	def doOutput (self, L):

		for handler in self.outputHandlers:
			handler.line(L)

	def cleanUp (self):

		for output in self.outputHandlers:
			output.cleanUp()

	def getHeader (self, vm_name):

		if self.system:
			cmd = 'set-system-vm' 
		else:
			cmd = 'set-user-vm'

		auto = '# Autogenerated by java-config\n# Command: --%s=%s' % (cmd, vm_name,)
 	
		return auto		

class BashEnvEntry (EnvMaker):

	def handle_ADDPATH (self, cmd, value):

		conf_str = None

		if self.system:
			conf_str = 'PATH=%s\nROOTPATH=%s' % (value, value,)
		else:
			conf_str = 'PATH=%s:${PATH}' % (value.strip('"'),)

		assert conf_str is not None

		return conf_str
		
	def handle_MANPATH (self, cmd, value):

		return 'MANPATH=${MANPATH}:%s' % (value.strip('"'),)

	def handle_ADDLDPATH (self, cmd, value):

		conf_str = None

		if self.system: 
			conf_str = 'LDPATH=%s' % (value,)

		return conf_str

	def handle_CLASSPATH (self, cmd, value):

		return 'CLASSPATH=%s' % (value,)	

	def handle_VERSION (self, cmd, value):

		return '# VERSION=%s' % (value,)

	def handle_ENV_VARS (self, cmd, val):

		return None

	def handle_special (self, cmd, value):

		ret = None 

		if cmd == 'ROOTPATH' and not self.system:
			ret = None
		else:
			ret = '%s=%s' % (cmd, value,)

		return ret

class TcshEnvEntry (EnvMaker):

	def handle_ADDPATH (self, cmd, value):

		if self.system:
			return ''
		else:
			return 'setenv PATH %s:${PATH}' % value.strip('"')

	def handle_MANPATH (self, cmd, value):

		return 'setenv MANPATH ${MANPATH}:%s' % (value.strip('"'),)

	def handle_ADDLDPATH (self, cmd, value):

		return None

	def handle_CLASSPATH (self, cmd, value):

		return 'setenv CLASSPATH %s' % (value,)	

	def handle_VERSION (self, cmd, value):

		return '# %s=%s' % (cmd, value)

	def handle_ENV_VARS (self, c, v):

		return None

	def handle_special (self, cmd, value):

		return 'setenv %s %s' % (cmd, value)


class StdOutBashEnvEntry (BashEnvEntry):

	def getHeader (self, vm_name):
		return None

	def _gh (self, name, c, v):

		O = getattr(BashEnvEntry, 'handle_' + name)(self, c, v)
		if O is not None:
			if O.startswith('#'):
				return None
			else:
			 	return 'export ' + O
		else:
			return None

	def handle_ADDPATH (self, c, v): return self._gh('ADDPATH', c, v)
	def handle_MANPATH (self, c, v): return self._gh('MANPATH', c, v)
	def handle_ADDLDPATH (self, c, v): return self._gh('ADDLDPATH', c, v)
	def handle_CLASSPATH (self, c, v): return self._gh('CLASSPATH', c, v)
	def handle_VERSION (self, c, v): return self._gh('VERSION', c, v)
	def handle_special (self, c, v): return self._gh('special', c, v)


class StdOutTcshEnvEntry (StdOutBashEnvEntry, TcshEnvEntry):

	def _gh (self, name, c, v):

		O = getattr(TcshEnvEntry, 'handle_' + name)(self, c, v)
		if O is not None:
			if O.startswith('#'):
				return None
			else:
				return O 
		else:
			return None


class EnvOutput:

	def header (self, headStr):

		return None


class EnvFileOutput (EnvOutput):

	def __init__ (self, config_file='20java', config_path='/etc/env.d'):

		try:
			self.configFile = open(os.path.join(config_path, config_file), 'w')
		except (IOError, OSError), e:
			raise OptionRuntimeError(
				'Error opening JVM configuration file %s' % (str(e),)
			)

	def header (self, headStr):
		
		self.configFile.write(headStr + '\n')

	def line (self, line):
		self.configFile.write(line + '\n')

	def cleanUp (self):
		self.configFile.close()


class EnvStdoutOutput (EnvOutput):

	def line (self, line):
		print line

	def cleanUp (self):
		pass

# vim: noet:ts=4:
