#!/usr/sbin/env python2
# -*- coding: UTF-8 -*-

# Author: Jason Mobarak
# $Header: /var/cvsroot/gentoo-src/java-config/python/java_config/jc_options.py,v 1.46 2006/09/06 23:35:52 nichoj Exp $

__version__ = "$Revision: 1.46 $" [11:-2]

import os
import re
from glob import glob

import jc_iface

from jc_exceptions import OptionRuntimeError
from jc_exceptions import OptionNamesAlreadyDefinedException

from jc_util import get_line_from_env_file
from jc_util import find_exec
from jc_util import may_want_to_source
from jc_util import get_java_home
from jc_util import get_jvm_env_var
from jc_util import norm_cp
from jc_util import collect_packages
from jc_util import collect_library
from jc_util import collect_vms
from jc_util import die_no_jh
from jc_util import warn

from jc_envgen import GenerateEnvEntry
from jc_envgen import EnvFileOutput
from jc_envgen import EnvStdoutOutput
from jc_envgen import BashEnvEntry
from jc_envgen import TcshEnvEntry
from jc_envgen import StdOutBashEnvEntry
from jc_envgen import StdOutTcshEnvEntry

from jc_output import *

env = os.environ
JAVA_HOME = get_java_home()

# I'm assuming it's safe to assume that keys like 'HOME' are 
# always defined in os.environ
	
def collect(): 

	""" 
	The bread and butter of how java-config works -- searches the __dict__
	of the jc_options(this) module and looks for classes that subclass the
	jc_iface.OptInterface class. Hopefully, these class will correctly
	subclass OptInterface and define everything that's required. The classes
	that are found are collected into a working program and iterations of
	the main program dispatch command line options to the class that says it
	implements that option. See jc_iface.OptInterface for more detailed info
	on these classes. 
	"""

	options = {}
	option_to_class = {}
	import jc_options
	jco_dict = jc_options.__dict__

	for name in jco_dict.iterkeys():
		obj = jco_dict[name] 
		try: 
			if issubclass(obj, jc_iface.OptInterface): 
				short_name, long_name, usage = \
					obj.names[0], obj.names[1], obj.usage 

				if not options.has_key(short_name) and \
					not option_to_class.has_key(long_name):

					option_to_class[long_name] = obj
					options[short_name] = obj
				else:
					raise \
					OptionNamesAlreadyDefinedException, \
					'Class %s, defines already defined ' \
					'option names.' % name 

		except TypeError: pass # skip non-classes

	return options, option_to_class


def report_env_update():

	print "Env files in %s/.gentoo updated. Source these in your shell's profile." % (env['HOME'])
			

class Help (jc_iface.OptInterface):
	
	usage = (
	  'List all command line options with --help , for '
	  'command specific long help messages use --help=command (ex. --help=javac)',
	)

	usage += (usage[0] + ' or all long help messages with --help=*',)
	
	names = ('h:', 'help=')

	# Order that options are printed
	order = [ 
	  'J', 'c', 'j', 'O', 'o','f', 'v', 'p:', 'g:', 'e:', 'l', 'L', 'P:', 
	  'S:', 's:', 'A:', 'a:', 'B:', 'b:', 'X', 'x', 'n','i:',
	]
	
	def _justify(self, S):
		'Appends some spaces to make the printing of help prettier'

		check = Help._justify.LINE_LEN - 5 
		space = ' ' * Help._justify.SPACE

		L = S.split()
		new_S, fin_S = '', ''
		for word in L:
			if len(new_S+word) <= check:
				new_S += '%s ' % word
			else:
				fin_S += '%s%s\n' % (space, new_S)
				new_S = word + ' '
		fin_S += '%s%s' % (space, new_S)
		return fin_S

	# Function constants for customizing how stuff is spaced
	_justify.LINE_LEN = 80
	_justify.SPACE = 4

	head = lambda s: \
		bold('Usage:')+' '+turquoise('java-config')+' [ '+ \
		green('OPTIONS')+' ]\n' + bold('Options:')


	def _print_help(self, klass):

		short_name, long_name = self._sub_args(klass)
		long_help = self._justify(klass.usage[1])
		return ' << %s >>\n%s' % \
			(bold('-%s --%s' % (short_name, long_name)), long_help)


	def _sub_args (self, klass):
		'''
		Change the : and = at the of the "names" to arg1[,arg2] or just
		arg if the function defines "single_arg" as true
		'''

		if klass.single_arg:
			short_name = klass.names[0].replace(':', ' argument')
			long_name = klass.names[1].replace('=', '=argument')
		else:	
			short_name = klass.names[0].replace(':', ' arg1[,arg2,...]')
			long_name = klass.names[1].replace('=', '=arg1[,arg2,...]')

		return short_name, long_name


	def __call__(self, args):

		output_str = ''
		short_list, long_list = collect()
		klass = None
		if args == '*': args = ','.join(Help.order)
		if args == '':
			output_str += self.head() + '\n'
			for option in ['h:'] + Help.order:
				klass = short_list[option]
				short_name, long_name = self._sub_args(klass)
				output_str += ' << %s >> \n%s\n' % \
					(bold('-%s --%s' % (short_name, long_name)), \
						self._justify(klass.usage[0]))
		else:
			args = map(lambda S: S.lstrip('-'), args.split(','))
			if args == ['']: args = Help.order
			output_str += self.head() + '\n'
			for arg in args:
				if short_list.has_key(arg):
					klass = short_list[arg]
				elif short_list.has_key(arg + ':'):
					klass = short_list[arg + ':']
				elif long_list.has_key(arg):
					klass = long_list[arg]
				elif long_list.has_key(arg + '='):
					klass = long_list[arg + '=']
				else:
					raise OptionRuntimeError('command %s not found' % arg)	
				if klass: 
					output_str += self._print_help(klass) + '\n'

		print output_str

class Java (jc_iface.OptInterface):

	usage = (
	  'Print the location of the java executable',

	  'Prints the location of the java virtual machine in the bin '
	  'directory of the java enviroment as specified by the JAVA_HOME '
	  'enviroment variable.'
	)

	names= ('J', 'java')

	def __call__(self, args):
		print find_exec('java', JAVA_HOME)

class Javac (jc_iface.OptInterface): 

	usage = (
	  'Print the location of the java byte compiler', 

	  'Prints the location of the java byte compiler (javac) in bin '
	  'directory of the java enviroment as specified by the JAVA_HOME '
	  'enviroment variable.'
	)

	names = ('c', 'javac')

	def __call__(self, args):
		print find_exec('javac', JAVA_HOME)


class Jar (jc_iface.OptInterface):
	
	usage = (
	  'Print the location of the jar executable', 

	  'Prints the location of the jar program in bin directory of the '
	  'java enviroment as specified by the JAVA_HOME enviroment '
	  'variable.'
	)

	names = ('j', 'jar') 

	def __call__(self, args):
		print find_exec('jar', JAVA_HOME)
   

class JDKHome (jc_iface.OptInterface):

	usage = (
	  'Print the location of the current jdk home', 

	  'Prints the location of the current java developement kit home, '
	  'as specified by the JDK_HOME enviroment variable.'
	)

	names = ('O', 'jdk-home')

	def __call__(self, args):

        # So, we'll shortcut this a little...
        # With new Java system, we didn't want JDK_HOME in the environment.
        # This is ok though, since we only allow JDKs to be visible to
        # java-config-1, so we can just return the value of JAVA_HOME :)
		#jdkh = get_jvm_env_var('JDK_HOME')
		jdkh = get_jvm_env_var('JAVA_HOME')

		if jdkh is None: 
			print
		else:
			print jdkh


class JREHome (jc_iface.OptInterface):

	usage = (
	  'Print the location of the current jre home',

	  'Prints the location of the current java runtime enviroment '
	  'home, as specified by the JRE_HOME enviroment variable.'
	)

	names = ('o', 'jre-home')

	def __call__(self, args):

        # So, we'll shortcut this a little...
        # With new Java system, we didn't want JRE_HOME in the environment.
        # This is ok though, since we only allow JDKs to be visible to
        # java-config-1, so we can just return the value of JAVA_HOME :)
		#jreh = get_jvm_env_var('JRE_HOME')
		jreh = get_jvm_env_var('JAVA_HOME')

		if jreh is None: 
			print
		else:
			print jreh
                        
class ShowActiveVM(jc_iface.OptInterface):
	usage = (
	  'Show the active virtual machine',
	  'Show the active virtual machine'
	)
        
	names = ('f', 'show-active-vm')
                
	def __init__(self):
		self.vmlist = collect_vms()

	def __call__(self,args):
		for vm in self.vmlist.iterkeys():
			if (self.vmlist[vm]['JAVA_HOME'] == JAVA_HOME):
				print vm[2:]
                        
class JavaVersion (jc_iface.OptInterface):

	usage = (
	  'Print the version information for the in-use jvm',

	  'Outputs all version information given by the in-use java '
	  'virtual machine'
	)

	names = ('v', 'java-version')

	def __call__(self, args):

		from commands import getoutput
		print getoutput('%s -version' % find_exec('java', JAVA_HOME))


class Classpath (jc_iface.OptInterface):

	usage = (
	  'Print entries in the enviroment or package classpath.',

	  'Print the enviroment class with just --classpath. Print the '
	  'classpath for a package with --classpath=pkg1,pkg2 .'
	)

	names = ('p:', 'classpath=')

	def __call__(self, args):
		cp = []
		if args != '':
			for arg in args.split(','): 
				#print bold('Classpath entries in package %s:' % arg)
				try:
					cp.append(collect_packages(arg))
				except OptionRuntimeError, err_msg:
					warn(str(err_msg))
			print ':'.join(cp)
		else:			
			#print bold('Entries in the classpath: \n')
			try:
				print env['CLASSPATH']
			except KeyError:
				# there is no CLASSPATH env set
				print

class Librarypath (jc_iface.OptInterface):

	usage = (
	  'Print entries in the package library.',

	  'Print the library path for a package '
	  'with --library=pkg1,pkg2 .'
	)

	names = ('i:', 'library=')

	def __call__(self, args):
		cp = []
		for arg in args.split(','): 
			try:
				cp.append(collect_library(arg))
			except OptionRuntimeError, err_msg:
				warn(str(err_msg))
		print ':'.join(cp)


class Exec (jc_iface.OptInterface):

	usage = (
	  'Execute something that should be somewhere in the JAVA_HOME',

	  'Searchs the bin directories of the Java Virtual Machine '
	  'specified by the JAVA_HOME enviroment variable and try to '
	  'execute it.'
	)

	names = ('e:', 'exec=')

	def __call__(self, args):

		#print args
		os.system(find_exec(args, JAVA_HOME))

class ListAvailablePackages (jc_iface.OptInterface):

	usage = (
	  'List all available packages', 

	  'List all java packages, package names can be used when using '
	  'the --set-{system, user}-classpath commands'
	)

	names = ('l', 'list-available-packages')

	def __init__ (self):

		self.glob_pkg = os.path.join(
		  '/', 'usr', 'share', '*', 'package.env'
		)

		self.glob_cp = os.path.join(
		  '/', 'usr', 'share', '*', 'classpath.env'
		)

		self.glob_pkg_dir = os.path.join(
		  '/', 'usr', 'share', 'java', 'packages', '*'
		)

	def __call__(self, args):

		def scour (pack):

			desc = get_line_from_env_file('DESC', pack)
			if desc == None:
				desc = '*Error getting description*'

			return desc
			
		for pack in glob(self.glob_pkg):

			desc = scour(pack)

			pkg_name = os.path.basename(os.path.dirname(pack))
			self.output('[%s] %s (%s)' % (pkg_name, desc, pack))

		for pack in glob(self.glob_cp):

			pkg_name = os.path.basename(os.path.dirname(pack))
			self.output('[%s] No description (%s)' % (pkg_name, pack))

		for pack in glob(self.glob_pkg_dir):

			desc = scour(pack)

			pkg_name = pack.split(os.sep)[-1]
			self.output('[%s] %s (%s)' % (pkg_name, desc, pack))

	def output (self, L):

		print L

		
class ListAvailableVMs (jc_iface.OptInterface):

	usage = (
	  'List all available virtual machines',

	  'Lists all virtuals machines in the /etc/env.d/java '
		'directory, displays wether a VM is active'
	)

	names = ('L', 'list-available-vms')

	def __init__(self):

		self.vm_list = collect_vms()

	def _active_vm (self, java_home):
		# Should fix bug #32611, no need to die if JVM isn't set when
		# listing JVMs
		#die_no_jh()

		if java_home == JAVA_HOME: return True
		else: return False
	
	def _disp_vm (self, file_name, cfg): 

		short_name = file_name[2:]
		vmstr = '[%s] %s (/etc/env.d/java/%s)' \
			% (short_name, cfg['VERSION'], file_name)  
		if self._active_vm(cfg['JAVA_HOME']): 
			return green(vmstr + ' *')
		else: return vmstr  

	def __call__(self, args):

		for vm in self.vm_list.iterkeys():
			print self._disp_vm(vm, self.vm_list[vm])  

source_java_classpath = \
'''
if [ -f $HOME/.gentoo/java ] ; then
	. $HOME/.gentoo/java
fi
'''

source_java_classpath_csh = \
'''
if ( -f $HOME/.gentoo/java.csh ) then
	source $HOME/.gentoo/java.csh
endif
'''

executables_a_system_vm_should_have = [
	'javac',
	'javadoc',
	'jar',
]

def vm_ok_for_system (vm_name, env_gen_obj):
	'''
	Check that a VM has all the binaries it needs to be a
	proper system VM
	'''

	suk = False
	java_home = env_gen_obj.vm['JAVA_HOME']

	for exe in executables_a_system_vm_should_have:
		try:
			find_exec(exe, java_home=java_home)
		except OptionRuntimeError, e:
			suk = True
			warn(str(e))	

	try:
		find_exec('rmic', java_home=java_home)
	except OptionRuntimeError, e:
		warn(str(e))

	if suk:
		warn(
			'THIS SYSTEM VM IS NOT SUFFICIENT, '
			'REQUIRED BINARIES WERE NOT FOUND'	
		)

def set_vm(vm_name, system=False):

	systemVM = system
	genv = GenerateEnvEntry(vm_name)

	if systemVM:
		vm_ok_for_system(vm_name, genv)
		sh_out = EnvFileOutput()
		sh_mkr = BashEnvEntry(system=True)
	else:
		sys.stderr.write(red('Setting User VM with java-config-1 is no longer supported. Use java-config-2 instead\n'))
		sys.exit(1)
#		sh_out = EnvFileOutput(
#		    config_file='java', 
#		    config_path=os.path.join(env['HOME'], '.gentoo')
#		)
#		sh_mkr = BashEnvEntry(system=False)
#
#		csh_out = EnvFileOutput(
#		    config_file='java.csh', 
#		    config_path=os.path.join(env['HOME'], '.gentoo')
#		)
#
#		csh_mkr = TcshEnvEntry(system=False)
#		csh_mkr.installOutputHandler(csh_out)
#		genv.installMaker(csh_mkr)

	sh_mkr.installOutputHandler(sh_out)
	genv.installMaker(sh_mkr)

	genv.makeEnvEntry()

	if systemVM: 
		print 'System Virtual Machine updated'
		#may_want_to_source()
	else: 
		report_env_update()
				

class PrintVMEnvVar (jc_iface.OptInterface):

	usage = (
	'Print an environment variable from the currently selected VM',

	'Outputs a specific environment variable for the specified VM'
	)

	names = ('g:', 'get-env=')
	
	single_arg = True

	def __call__(self, args):
		
		env = get_jvm_env_var(args)
		if env is None: 
			print	
		else:
			print env


class PrintVM (jc_iface.OptInterface):


	usage = (
      'Print the environment information for a specified VM',

	  'Outputs all the environment information for a specified VM so that '
	  'the output can be evaluated or viewed, takes two arguments, first argument '
	  'is a JVM name (see --list-available-vms), second argument is a shell type '
	  'currently supported is "sh" or "csh", if only one argument is specified "sh" '
	  'is assumed'
	)

	names = ('P:', 'print=')

	#single_arg = True

	def __call__(self, args): 
		
		args = args.split(',')

		if len(args) == 1:
			shellType = 'sh'
		else:
			shellType = args[1]
		
		try:
			genv = GenerateEnvEntry(args[0])
		except OptionRuntimeError:
			# die silently
			print ''
			return

		if shellType == 'sh':
			_ = StdOutBashEnvEntry(False)
			_.installOutputHandler(EnvStdoutOutput())
			genv.installMaker(_)
		elif shellType == 'csh':
			_ = StdOutTcshEnvEntry(False)
			_.installOutputHandler(EnvStdoutOutput())
			genv.installMaker(_)
		else:
			raise OptionRuntimeError('Unsupported shell type specified')

		genv.makeEnvEntry()


class SetSystemVM (jc_iface.OptInterface):


	usage = (
	  'Set the VM for the entire system',

	  'Set the virtual machine that the entire system will use by '
	  'default'
	)

	names = ('S:', 'set-system-vm=')

	require_root = True
	single_arg = True

	def __call__(self, args): 
		set_vm(args, system=True)


class SetUserVM (jc_iface.OptInterface):


	usage = (
	  'Set the VM for a user', 

	  'Set the virtual machine that the current user will use by '
	  'default'
	)

	names = ('s:', 'set-user-vm=')
	single_arg = True

	def __call__(self, args):
		sys.stderr.write(red('Setting the user vm with java-config-1 is longer supported by java-config-1. Use java-config-2 instead\n'))
		sys.exit(1)

#		jc_dir = env['HOME'] + '/.gentoo'
#
#		if os.getuid() == 0: 
#			raise OptionRuntimeError, 'root should always use the system vm'
#
#		can_write = os.access(os.path.split(jc_dir)[0], os.R_OK|os.W_OK|os.X_OK)
#		if not can_write:
#			raise OptionRuntimeError('Cannot write to ${HOME}')
#
#		if not os.path.isdir(jc_dir):
#			os.mkdir(jc_dir)
#		elif not os.path.exists(jc_dir):
#			raise OptionRuntimeError('${HOME}/.gentoo exists but is not a directory')
#
#		set_vm(args)
#
#		if not os.path.isfile(jc_dir + '/java-env'): 
#			file(os.path.join(jc_dir, 'java-env'), 'w').write(
#			  source_java_classpath
#			)
#
#		if not os.path.isfile(jc_dir + '/java-env.csh'): 
#			file(os.path.join(jc_dir, 'java-env.csh'), 'w').write(
#			  source_java_classpath_csh
#			)


class SetSystemClasspath (jc_iface.OptInterface):

	# /etc/env.d/21java-classpath 

	usage = (
	  'Set the system classpath', 

	  'Set the system classpath based on the package names specified, '
	  'available packages can be found with the '
	  '--list-available-packages option.'
	)

	names = ('A:', 'set-system-classpath=')
	require_root = True

	def __call__(self, args):

		#if os.getuid() != 0: raise OptionRuntimeError, 'root access required'
		if args == '':
			raise OptionRuntimeError, \
			'if you want to clean the classpath use --clean-system-classpath'

		else: 
			try:
				fn = os.path.join('/', 'etc', 'env.d', '21java-classpath')
				fp = file(fn, 'w')
			except (OSError, IOError), e:
				raise OptionRuntimeError(
				  'Error opening configuration file, ' + str(e)
				)

			try:
				fp.write('CLASSPATH=%s\n' % (norm_cp(collect_packages(args)),))
			except (OSError, IOError), e:
				raise OptionRuntimeError(
				  'Error writing configuration file, ' + str(e)
				)
				
			print 'System classpath set'
			may_want_to_source()


class AddSystemClasspath (jc_iface.OptInterface):

	# /etc/env.d/21java-classpath 

	usage = (
	  'Append to the system classpath', 

	  'Append the package names specified to the system '
	  'classpath. Packages can be found with the '
	  '--list-available-packages option.'
	)

	names = ('B:', 'add-system-classpath=')
	require_root = True

	conf_file = os.path.join('/', 'etc', 'env.d', '21java-classpath')

	def __call__(self, args):
		sys.stderr.write(red('Classpath operations are no longer supported by java-config-1. Use java-config-2 instead\n'))
		sys.exit(1)

#		if args == '':
#			raise OptionRuntimeError(
#				'if you want to clean the classpath use --clean-system-classpath')
#
#		else:
#			old_classpath = None 
#
#			if os.path.exists(self.conf_file):
#				try:
#					old_classpath = file(self.conf_file, 'r').read().strip()
#
#				except (OSError, IOError), e:
#					warn ("Error when reading classpath file " + str(e))
#					return
#
#				try:
#					old_classpath = old_classpath.split('=', 1)[1]
#					#old_classpath = re.sub('CLASSPATH=', '', old_classpath)
#		
#				except IndexError:
#					warn ("Malformed CLASSPATH in /etc/env.d/21java-classpath")
#					return
#			else:
#				old_classpath = ""
#
#
#			try:
#				
#				file(self.conf_file, 'w').write(
#			  		'CLASSPATH=%s\n' % (
#			    		norm_cp(old_classpath + ':' + collect_packages(args)),))
#
#			except (OSError, IOError), e:
#				warn ("Error writing classpath file " + str(e))
#				return
#
#
#			print 'System classpath set'
#			may_want_to_source() 

class SetUserClasspath (jc_iface.OptInterface):

	usage = (
	  'Set the user classpath', 

	  'Set the user classpath based on the package names '
	  'specified, available packages can be found with the '
	  '--list-available-packages option.'
	)

	names = ('a:', 'set-user-classpath=')

	def __call__(self, args):
		sys.stderr.write(red('Classpath operations are no longer supported by java-config-1. Use java-config-2 instead\n'))
		sys.exit(1)

#		jc_dir = os.path.join(env['HOME'], '.gentoo')
#
#		if not os.path.isdir(jc_dir):
#			os.mkdir(jc_dir)
#
#		file(os.path.join(jc_dir, 'java-env-classpath'), 'w').write(
#		  'CLASSPATH=${CLASSPATH}:' + collect_packages(args)
#		)
#
#		if not os.path.isfile(os.path.join(jc_dir, 'java-env')): 
#
#			file(os.path.join(jc_dir, 'java-env'), 'w').write(
#				source_java_classpath
#			)
#		
#			report_env_update()


class AppendUserClasspath (jc_iface.OptInterface):

	usage = (
	  'Append to the user classpath', 

	  'Append the package names specified to the user ' 
	  'classpath, available packages can be found with '
	  'the --list-available-packages option.'
	)

	names = ('b:', 'add-user-classpath=')

	def __call__(self, args):
		sys.stderr.write(red('Classpath operations are no longer supported by java-config-1. Use java-config-2 instead\n'))
		sys.exit(1)

#		jc_dir = os.path.join(env['HOME'], '.gentoo')
#		old_classpath = '' 
#
#		if not os.path.isdir(jc_dir):
#			os.mkdir(jc_dir)
#
#		elif os.path.isfile(os.path.join(jc_dir, 'java-env-classpath')):
#			old_classpath = file(os.path.join(jc_dir, 'java-env-classpath'), 'r').read()
#			old_classpath = re.sub('CLASSPATH=\${CLASSPATH}:', '', old_classpath)
#
#		file(os.path.join(jc_dir, 'java-env-classpath'), 'w').write(
#		  'CLASSPATH=${CLASSPATH}:' + norm_cp(old_classpath + ':' + collect_packages(args))
#		)
#
#		if not os.path.isfile(os.path.join(jc_dir, 'java-env')): 
#			file(os.path.join(jc_dir, 'java-env'), 'w').write(source_java_classpath)
#
#		report_env_update()
		
class CleanSystemClasspath (jc_iface.OptInterface):

	usage = (
	  'Clean the classpath of all packages',

	  'Whipes the classpath of all packages, this is done by '
	  'removing the /etc/env.d/21java-classpath file.'
	) 

	names = ('X', 'clean-system-classpath')

	require_root = True
	env_file = '/etc/env.d/21java-classpath'

	def __call__(self, args):
		sys.stderr.write(red('Classpath operations are no longer supported by java-config-1. Use java-config-2 instead\n'))
		sys.exit(1)

		#if os.getuid() != 0: raise OptionRuntimeError, 'root access required'
#		if os.path.isfile(self.env_file):
#			print 'Cleaning system classpath...'
#			os.remove(self.env_file)
#			#if os.system('/usr/sbin/env-update') != 0:
#			#	raise OptionRuntimeError, 'env-update command not found!'
#			print 'System classpath cleaned'
#			may_want_to_source()

class CleanUserClasspath (jc_iface.OptInterface):

	usage = (
	  'Clean the user classpath of all packages',

	  'Whipes the user classpath of all packages, '
	  'this is done by removing the '
	  '~/.gentoo/java-env-classpath file.'
	)

	names = ('x', 'clean-user-classpath')
	require_root = False

	def __init__ (self):

		self.env_file = os.path.join(env['HOME'], '.gentoo', 'java-env-classpath')

	def __call__(self, args):
		sys.stderr.write(red('Classpath operations are no longer supported by java-config-1. Use java-config-2 instead\n'))
		sys.exit(1)

#		if os.path.isfile(self.env_file):
#			print 'Cleaning user classpath...'
#			os.remove(self.env_file)
#			print 'User classpath cleaned' 
#			report_env_update()
#		else:
#			print 'No user classpath file found'


		
class NoColors (jc_iface.OptInterface):

	usage = (
	  'Turn off all colors.', 

	  'Turn off all instances of color or bold escape '
	  'codes that java-config might use.'
	)

	names = ('n', 'nocolor')
	require_root = False

	def __call__ (self, args):
		nocolor()

if __name__ == '__main__':

	import __main__ 
	name, thing = None, None
	for (name, thing) in __main__.__dict__.iteritems():
		try:
			if issubclass(thing, __main__.jc_iface.OptInterface):
				print blue('jc_iface.OptInterface: ') + bold(name)
			else:
				print 'Not a subclass: ' + name 
		except TypeError:
			print 'This was not a class: ' + name

# vim: noet:ts=4:
