package org.apache.jetspeed.portlet;

import javax.servlet.*;
import javax.servlet.http.*;

import java.io.IOException;
import java.io.Serializable;
import java.io.InvalidObjectException;
import java.io.ObjectStreamException;
import java.io.PrintWriter;

import java.util.*;
import java.lang.Boolean;

import org.apache.jetspeed.portlet.event.*;
import org.apache.jetspeed.portlet.spi.SPIPortletInterceptor;
import org.apache.jetspeed.portlet.spi.Constants;

/**
 * The abstract <CODE>Portlet</CODE> is used by the portlet container to
 * invoke the portlet. Every portlet has to implement this abstract class,
 * either by deriving directly from it, or by using one of the abstract
 * portlet implementations.
 * <P>
 * A portlet is a small Java program that runs within a portlet container.
 * Portlets receive and respond to requests from the portlet container.
 * There is ever only one portlet object instance per portlet configuration
 * in the web deployment descriptor. There may be many <CODE>PortletSettings</CODE> objects
 * parametersing the same portlet object according to the Flyweight pattern,
 * provided on a per-request basis. A concrete parameterization of a portlet
 * object is referred to as a <I>concrete portlet</I>. The settings of <I>concrete portlets</I>
 * may change at any time caused by administrators modifying portlet settings,
 * e.g. using the config mode of a portlet.
 * <P>
 * Additionally, user can have personal views of <I>concrete portlets</I>. Therefore,
 * the transient portlet session and persistent concrete portlet data carries
 * vital information for the portlet to create a personalized user experience.
 * A <I>concrete portlet</I> in conjunction with portlet data creates a <I>concrete portlet
 * instance</I>. This is similar to why a servlet may not store things depending
 * on requests or sessions in instance variables. As a consequence, the portlet
 * should not attempt to store any data that depends on portlet settings, portlet
 * data or the portlet session or any other user-related information as instance
 * or class variables. The general programming rules for servlets also apply to
 * portlets - instance variables should only used when the intent is to share
 * them between all the parallel threads that concurrently execute a portlet, and
 * portlet code should avoid synchronization unless absolutely required.
 *
 * <P>
 * As part of running within the portlet container each portlet has a
 * life-cycle. The corresponding methods are called in the following
 * sequence:
 * <OL>
 * <LI>The portlet is constructed, then initialized with the
 *     <CODE>init()</CODE> method.
 * <LI>A <I>concrete portlet</I> s initialized with the
 *     <CODE>initConcrete()</CODE> method for each PortletSettings.
 * <LI>Any calls from the portlet container to the <CODE>service()</CODE>
 *     method are handled.
 * <LI>The <I>concrete portlet</I> is taken out of service with the
 *     <CODE>destroyConcrete()</CODE> method.
 * <LI>The portlet is taken out of service, then destroyed with the
 *     <CODE>destroy()</CODE> method, then garbage collected and finalized.
 * </OL>
 *
 * <P>
 * The <I>concrete portlet instance</I> is created and destroyed with the
 * <CODE>login()</CODE> and <CODE>logout()</CODE> methods,
 * respectively. If a portlet provides personalized views these methods
 * should be implemented.
 *
 * <P>
 * The portlet container loads and instantiates the portlet class. This
 * can happen during startup of the portal server or later, but no later
 * then when the first request to the portlet has to be serviced.
 * Also, if a portlet is taken out of service temporarily, for example
 * while administrating it, the portlet container may finish the life-cycle
 * before taking the portlet out of service. When the administration
 * is done, the portlet will be newly initialized.
 *
 * @author <A HREF="mailto:shesmer@apache.org">Stephan Hesmer</A>
 */
public abstract class Portlet
extends com.ibm.wps.portletcontainer.cache.CacheablePortlet
implements PortletSessionListener // temp
{
    /**
     ** The <CODE>Mode</CODE> class is a finite enumeration of
     ** the possible modes that a portlet can assume.
     **/

    public static class Mode implements Serializable
    {
        /**
         ** The standard "one-of many" portlet view on a page.
         **/

        public final static Mode VIEW = new Mode ("View", 0);

        /**
         ** This mode allows the portlet to capture user-specific
         ** parameterization, which leads to a personalized view of the
         ** portlet.
         **/

        public final static Mode EDIT = new Mode ("Edit", 1);

        /**
         ** A portlet should provide useful online help in this mode.
         ** This may be a short description or a multi-page instruction on
         ** how to use the portlet.
         **/

        public final static Mode HELP = new Mode ("Help", 2);

        /**
         ** Allows the portlet to bring its own configuration screen if
         ** required. Only a user with administrator privileges should
         ** be able to call a portlet in this mode.
         **/

        public final static Mode CONFIGURE = new Mode ("Configure", 3);

        private final static Mode[] MODES = { VIEW, EDIT, HELP, CONFIGURE };

        private String identifier;

        private int value;

        private Mode (String identifier, int value)
        {
            this.identifier = identifier;
            this.value = value;
        }

        public int getId()
        {
            return value;
        }

        public String toString ()
        {
            return (identifier);
        }

        public Object readResolve() throws ObjectStreamException {
            try {
                return MODES[value];
            }
            catch (IndexOutOfBoundsException e) {
                throw new InvalidObjectException("Unknown Portlet Mode");
            }
        }
    }

    /**
     ** The <CODE>ModeModifier</CODE> class is a finite enumeration of
     ** the possible modifications a portlet can apply on modes.
     **/

    public static class ModeModifier implements java.io.Serializable
    {
        /**
         ** The mode is changed as requested.
         **/

        public final static ModeModifier REQUESTED = new ModeModifier ("Requested", 0);

        /**
         ** The mode is not changed and stays in the current one.
         **/

        public final static ModeModifier CURRENT = new ModeModifier ("Current", 1);

        /**
         ** The mode is changed to the previous one.
         **/

        public final static ModeModifier PREVIOUS = new ModeModifier ("Previous", 2);

        private final static ModeModifier[] MODE_MODIFIERS = { REQUESTED, CURRENT, PREVIOUS };

        private String identifier;

        private int value;

        private ModeModifier (String identifier, int value)
        {
            this.identifier = identifier;
            this.value = value;
        }

        public int getId()
        {
            return value;
        }

        public String toString ()
        {
            return (identifier);
        }

        public Object readResolve() throws ObjectStreamException {
            try {
                return MODE_MODIFIERS[value];
            }
            catch (IndexOutOfBoundsException e) {
                throw new InvalidObjectException("Unknown Portlet ModeModifier");
            }
        }
    }

    /**
     * Called by the portlet container to indicate to this portlet
     * that it is put into service.
     *
     * <P>
     * The portlet container calls the <CODE>init()</CODE> method
     * for the whole life-cycle of the portlet. The
     * <CODE>init()</CODE> method must complete successfully before
     * concrete portlets are created through the
     * <CODE>initConcrete()</CODE> method.
     *
     * <P>
     * The portlet container cannot place the portlet into service
     * if the <CODE>init()</CODE> method
     *
     * <OL>
     * <LI>throws <CODE>UnavailableException</CODE>
     * <LI>does not return within a time period defined by the portlet
     *     container.
     * </OL>
     *
     * @param config
     *               the portlet configuration
     * @exception UnavailableException
     *                   if an exception has occurrred that interferes
     *                   with the portlet's normal initialization
     */
    abstract public void init (PortletConfig config) throws UnavailableException;

    /**
     ** Called by the portlet container to indicate to this portlet
     ** that it is taken out of service. This method is only called
     ** once all threads within the portlet's <CODE>service()</CODE>
     ** method have exited or after a timeout period has passed. After
     ** the portlet container calls this method, it will not call the
     ** <CODE>service()</CODE> method again on this portlet.
     **
     ** <P>
     ** This method gives the portlet an opportunity to clean up any
     ** resources that are being held (for example, memory, file
     ** handles, threads).
     **/

    abstract public void destroy (PortletConfig config);

    /**
     * Called by the portlet container to indicate that the concrete portlet
     * is put into service.
     *
     * <P>
     * The portlet container calls the <CODE>initConcrete()</CODE> method
     * for the whole life-cycle of the portlet. The
     * <CODE>initConcrete()</CODE> method must complete successfully before
     * concrete portlet instances can be created through the <CODE>login()</CODE> method.
     *
     * <P>
     * The portlet container cannot place the portlet into service
     * if the <CODE>initConcrete()</CODE> method
     *
     * <OL>
     * <LI>throws <CODE>UnavailableException</CODE>
     * <LI>does not return within a time period defined by the portlet
     *     container.
     * </OL>
     *
     * @param settings
     *                 the portlet settings
     * @exception UnavailableException
     *                   if an exception has occurrred that interferes
     *                   with the portlet's normal initialization
     */
    abstract public void initConcrete (PortletSettings settings) throws UnavailableException;

    /**
     * Called by the portlet container to indicate that the concrete portlet
     * is taken out of service. This method is only called
     * once all threads within the portlet's <CODE>service()</CODE>
     * method have exited or after a timeout period has passed. After
     * the portlet container calls this method, it will not call the
     * <CODE>service()</CODE> method again on this portlet.
     *
     * <P>
     * This method gives the portlet an opportunity to clean up any
     * resources that are being held (for example, memory, file
     * handles, threads).
     *
     * @param settings
     *                 the portlet settings
     */
    abstract public void destroyConcrete (PortletSettings settings);

    /**
     ** Called by the portlet container to ask this portlet to generate
     ** its markup using the given request/response pair. Depending
     ** on the mode of the portlet and the requesting client device,
     ** the markup will be different. Also, the portlet can take language
     ** preferences and/or personalized settings into account.
     **
     ** @param   request
     **          the portlet request
     ** @param   response
     **          the portlet response
     **
     ** @exception   PortletException
     **              if the portlet has trouble fulfilling the
     **              rendering request
     ** @exception   IOException
     **              if the streaming causes an I/O problem
     **/

    abstract public void service (PortletRequest request,
                                  PortletResponse response) throws PortletException,
                                                                   IOException;

    /**
     * Returns the time the response of the <code>Portlet</code>
     * object was last modified,
     * in milliseconds since midnight January 1, 1970 GMT.
     * If the time is unknown, this method returns a negative
     * number (the default).
     *
     * <p>Portlets that can quickly determine
     * their last modification time should override this method.
     * This makes browser and proxy caches work more effectively,
     * reducing the load on server and network resources.
     *
     * @param request the <code>PortletRequest</code>
     *                object that is sent to the portlet
     * @return a <code>long</code> integer specifying
     *         the time the response of the <code>Portlet</code>
     *         object was last modified, in milliseconds
     *         since midnight, January 1, 1970 GMT, or
     *         -1 if the time is not known
     */

    abstract public long getLastModified(PortletRequest request);

    /**
     * Returns the <CODE>PortletConfig</CODE> object of the portlet
     *
     * @return the PortletConfig object
     */
    abstract public PortletConfig getPortletConfig ();

    /**
     * Returns the <CODE>PortletSettings</CODE> object of the concrete portlet.
     *
     * @return the PortletSettings object, or NULL if no PortletSettings object is available.
     */
    protected PortletSettings getPortletSettings()
    {
        return portletSettingsThreadContainer.getSettings();
    }

    abstract public void login (PortletRequest request) throws PortletException;

    abstract public void logout (PortletSession session) throws PortletException;

// servlet specific functions

    public final void init(ServletConfig config)
        throws ServletException
    {
        super.init(config);
        portletInitialized = false;
    }

    public final void init()
        throws ServletException
    {
    }

    final public ServletConfig getServletConfig ()
    {
        return super.getServletConfig();
    }

    final public String getInitParameter(String name) {
       return getServletConfig().getInitParameter(name);
    }

    final public Enumeration getInitParameterNames() {
        return getServletConfig().getInitParameterNames();
    }

    public ServletContext getServletContext() {
// TBD: wrapping of servlet context is needed
        return getServletConfig().getServletContext();
    }

    final protected long getLastModified(HttpServletRequest req) {
        // tbd
	return -1;
    }

    final public String getServletInfo()
    {
        return "";
    }

    final public void service(ServletRequest request, ServletResponse response)
	throws ServletException, IOException
    {
        try {
            super.service(request,response);
        }
        catch (Throwable t) {
            throw new PortletException(t);
        }
    }

    protected void doGet(HttpServletRequest req, HttpServletResponse resp)
	throws ServletException, IOException
    {
        dispatch(req,resp);
    }

    protected void doPost(HttpServletRequest req, HttpServletResponse resp)
	throws ServletException, IOException
    {
        dispatch(req,resp);
    }

    protected void doPut(HttpServletRequest req, HttpServletResponse resp)
	throws ServletException, IOException
    {
        dispatch(req,resp);
    }

    protected void doDelete(HttpServletRequest req,
			    HttpServletResponse resp)
	throws ServletException, IOException
    {
        super.doDelete(req,resp);
    }

    protected void doOptions(HttpServletRequest req, HttpServletResponse resp)
	throws ServletException, IOException
    {
        super.doOptions(req,resp);
    }

    protected void doTrace(HttpServletRequest req, HttpServletResponse resp)
	throws ServletException, IOException
    {
        super.doTrace(req,resp);
    }

    private void dispatch(HttpServletRequest request, HttpServletResponse response)
	throws ServletException, IOException
    {
        Integer methodIdObject = (Integer)request.getAttribute(Constants.METHOD_ID);
        int methodId = methodIdObject.intValue();

        if (methodId==Constants.METHOD_PORTLET_LOGOUT) {
            // special handling for logout
            org.apache.turbine.util.Log.debug("Portlet.service METHOD_PORTLET_LOGOUT ...");

            if (this instanceof PortletSessionListener)
            {
                String cpiid = (String)request.getAttribute(org.apache.jetspeed.portletcontainer.Constants.PORTLET_CPIID);
                PortletSession portletSession = (PortletSession)
                    request.getAttribute(Constants.PARAM_PORTLETSESSION);
                PortletSettings portletSettings = (PortletSettings)
                    request.getAttribute(Constants.PARAM_PORTLETSETTINGS);
                portletSettingsThreadContainer.setSettings(portletSettings);

                if (getPortletLinkedWithSession(cpiid, portletSession)!=null)
                {
                    org.apache.turbine.util.Log.debug("Portlet.service METHOD_PORTLET_LOGOUT called");
                    unlinkPortletWithSession(cpiid, portletSession);
                    ((PortletSessionListener)this).logout(portletSession);
                }
            }
            return;
        }

        if (!portletInitialized)
        {
            synchronized (this) {
                if (!portletInitialized)
                {
                    org.apache.turbine.util.Log.debug("Portlet.service METHOD_PORTLET_INIT");

                    interceptor = (SPIPortletInterceptor)request.getAttribute(Constants.PARAM_INTERCEPTOR);
                    if (interceptor==null) {
                        throw new UnavailableException("SPIPortletInterceptor was not provided!");
                    }

                    PortletConfig portletConfig = (PortletConfig)
                        request.getAttribute(Constants.PARAM_PORTLETCONFIG);
                    if (portletConfig==null) // error
                        throw new UnavailableException("No PortletConfig provided.");

                    interceptor.init(this,
                                     portletConfig);

                    init(portletConfig);

                    this.portletConfig = portletConfig;
                    portletInitialized = true;
                }
            }
        }
        if (interceptor==null) {
            throw new UnavailableException(this, "SPIPortletInterceptor was not provided!");
        }

        interceptor.preService(request,response);

        try
        {
            // events, not request-based
            if ((methodId==Constants.METHOD_PORTLET_SETTINGS_EVENT) ||
                (methodId==Constants.METHOD_PORTLETAPPLICATION_SETTINGS_EVENT))
            {
// SHESMER-TODO
//                PortletSettings portletSettings = portletRequest.getPortletSettings();
//                portletSettingsThreadContainer.setSettings(portletSettings);

                interceptor.handleEvents(this,
                                         request);
            }
            else
            {
                // events, request based

                // do some general initializing
                PortletRequest portletRequest = (PortletRequest)
                    request.getAttribute(Constants.PARAM_PORTLETREQUEST);
                PortletResponse portletResponse = (PortletResponse)
                    request.getAttribute(Constants.PARAM_PORTLETRESPONSE);

                PortletSettings portletSettings = portletRequest.getPortletSettings();
                portletSettingsThreadContainer.setSettings(portletSettings);

                if (!concretePortlets.contains(portletSettings)) // concrete portlet not yet initialized
                {
                    org.apache.turbine.util.Log.debug("Portlet.service METHOD_PORTLET_INIT_CONCRETE");

                    initConcrete(portletSettings);

                    concretePortlets.add(portletSettings);
                }

                // start dispatching
                switch (methodId) {
                    case Constants.METHOD_PORTLET_LOGIN:
                        org.apache.turbine.util.Log.debug("Portlet.service METHOD_PORTLET_LOGIN ...");

                        if (this instanceof PortletSessionListener)
                        {
                            PortletSession portletSession = portletRequest.getPortletSession();
                            if (getPortletLinkedWithSession(portletRequest,
                                                            portletSession)==null)
                            {
                                org.apache.turbine.util.Log.debug("Portlet.service METHOD_PORTLET_LOGIN called");

                                linkPortletWithSession(portletRequest,
                                                       portletSession,
                                                       this);
                                ((PortletSessionListener)this).login(portletRequest);
                            }
                        }
                        break;
                    case Constants.METHOD_PORTLET_BEGINPAGE:
                        org.apache.turbine.util.Log.debug("Portlet.service METHOD_PORTLET_BEGINPAGE");

                        if (this instanceof PortletPageListener)
                        {
                            ((PortletPageListener)this).beginPage(portletRequest, portletResponse);
                        }
                        break;
                    case Constants.METHOD_PORTLET_SERVICE:
                        org.apache.turbine.util.Log.debug("Portlet.service METHOD_PORTLET_SERVICE");

                        service(portletRequest, portletResponse);

                        break;
                    case Constants.METHOD_PORTLET_ENDPAGE:
                        org.apache.turbine.util.Log.debug("Portlet.service METHOD_PORTLET_ENDPAGE");

                        if (this instanceof PortletPageListener)
                        {
                            ((PortletPageListener)this).endPage(portletRequest, portletResponse);
                        }
                        break;
                    case Constants.METHOD_PORTLET_DESTROY_CONCRETE:
                        org.apache.turbine.util.Log.debug("Portlet.service METHOD_PORTLET_DESTROY_CONCRETE");

                        if (portletSettings==null) // error
                            throw new UnavailableException("No PortletSettings provided.");
                        if (!concretePortlets.contains(portletSettings))
                            throw new UnavailableException("Concrete Portlet was not initialized.");

                        destroyConcrete(portletSettings);

                        concretePortlets.remove(portletSettings);

                        break;
                    case Constants.METHOD_INCLUDE_PORTLET_TITLE:
                        org.apache.turbine.util.Log.debug("Portlet.service METHOD_INCLUDE_PORTLET_TITLE");

                        if (this instanceof PortletTitleListener)
                        {
                            ((PortletTitleListener)this).doTitle(portletRequest, portletResponse);
                        }
                        else { // static title
                            String title = (String)request.getAttribute(Constants.PARAM_STATICTITLE);
                            PrintWriter writer = portletResponse.getWriter();
                            writer.print(title);
                        }
                        break;
                    case Constants.METHOD_PERFORM_ACTION:
                    case Constants.METHOD_PERFORM_MESSAGE:
                    case Constants.METHOD_PERFORM_WINDOW:
                        interceptor.handleEvents(this,
                                                 request);
                        break;
                }
            }
        }
        finally
        {
            interceptor.postService(request,response);
        }

    }

    final public void destroy()
    {
        if (portletInitialized)
        {
            org.apache.turbine.util.Log.debug("Portlet.destroy METHOD_PORTLET_DESTROY");

            interceptor.destroy();

            Enumeration allConcretePortlets = concretePortlets.elements();
            while (allConcretePortlets.hasMoreElements())
            {
                org.apache.turbine.util.Log.debug("  Portlet.destroy METHOD_PORTLET_DESTROY_CONCRETE");
                PortletSettings portletSettings = (PortletSettings)allConcretePortlets.nextElement();
                destroyConcrete(portletSettings);
            }
            concretePortlets.clear();

            destroy(portletConfig);

            portletConfig = null;
            portletInitialized = false;
        }
        super.destroy();
    }

    // internal classes and variables

    private static class PortletSettingsThreadContainer extends ThreadLocal
    {
        public PortletSettings getSettings()
        {
            return (PortletSettings)super.get();
        }

        public void setSettings(PortletSettings settings)
        {
            super.set(settings);
        }
    }

    private PortletSettingsThreadContainer portletSettingsThreadContainer =
        new PortletSettingsThreadContainer();

    private boolean portletInitialized = false;
    private PortletConfig portletConfig = null;
    private SPIPortletInterceptor interceptor = null;
    private Vector concretePortlets = new Vector();

    // managment functionality for session handling

    private static Hashtable sessionPortlets = new Hashtable();

    public static class PortletSessionStub
    {
        private Portlet portlet;
        private String cpiid;
        private String cpid;

        PortletSessionStub(Portlet portlet, String cpiid, String cpid)
        {
            this.portlet = portlet;
            this.cpiid = cpiid;
            this.cpid = cpid;
        }
        public Portlet getPortlet()
        {
            return portlet;
        }
        public String getCPiid()
        {
            return cpiid;
        }
        public String getCPid()
        {
            return cpid;
        }
    }

    public static Enumeration getAllPortletsLinkedWithSession(PortletSession portletSession)
    {
        Vector portlets = new Vector();
        org.apache.jetspeed.portletcontainer.PortletSessionImpl sessionImpl =
            (org.apache.jetspeed.portletcontainer.PortletSessionImpl)portletSession;

        String _key = sessionImpl.getServletSession().getId()+
                      sessionImpl.getSessionData().getCAiid();

        Enumeration enum = sessionPortlets.keys();
        while (enum.hasMoreElements())
        {
            String key = (String)enum.nextElement();
            if (key.startsWith(_key))
            {
                portlets.add(sessionPortlets.get(key));
            }
        }
        return portlets.elements();
    }

    public static Portlet getPortletLinkedWithSession(PortletRequest portletRequest,
                                                      PortletSession portletSession)
    {
        org.apache.jetspeed.portletcontainer.PortletRequestImpl requestImpl =
            (org.apache.jetspeed.portletcontainer.PortletRequestImpl)portletRequest;
        return getPortletLinkedWithSession(requestImpl.getPortletInstanceEntry().getPiid().toString(),
                                           portletSession);
    }

    public static Portlet getPortletLinkedWithSession(String cpiid,
                                                      PortletSession portletSession)
    {
        org.apache.jetspeed.portletcontainer.PortletSessionImpl sessionImpl =
            (org.apache.jetspeed.portletcontainer.PortletSessionImpl)portletSession;
        // we need to apply the session id and PIID because there can be more than one portlet instance per session
        String key = sessionImpl.getServletSession().getId()+
                     sessionImpl.getSessionData().getCAiid()+
                     cpiid;
        PortletSessionStub stub = ((PortletSessionStub)sessionPortlets.get(key));
        if (stub==null) return null;
        return stub.getPortlet();
    }

    private static void linkPortletWithSession(PortletRequest portletRequest,
                                               PortletSession portletSession,
                                               Portlet portlet)
    {
        org.apache.jetspeed.portletcontainer.PortletRequestImpl requestImpl =
            (org.apache.jetspeed.portletcontainer.PortletRequestImpl)portletRequest;
        org.apache.jetspeed.portletcontainer.PortletSessionImpl sessionImpl =
            (org.apache.jetspeed.portletcontainer.PortletSessionImpl)portletSession;
        // we need to apply the session id and PIID because there can be more than one portlet instance per session
        String key = sessionImpl.getServletSession().getId()+
                     sessionImpl.getSessionData().getCAiid()+
                     requestImpl.getPortletInstanceEntry().getPiid().toString();
        PortletSessionStub stub = new PortletSessionStub(portlet,
                                                         requestImpl.getPortletInstanceEntry().getPiid().toString(),
                                                         requestImpl.getPortletInstanceEntry().getConcretePortlet().getPid().toString());
        sessionPortlets.put(key, stub);
    }

    private static void unlinkPortletWithSession(String cpiid,
                                                 PortletSession portletSession)
    {
        org.apache.jetspeed.portletcontainer.PortletSessionImpl sessionImpl =
            (org.apache.jetspeed.portletcontainer.PortletSessionImpl)portletSession;
        String key = sessionImpl.getServletSession().getId()+
                     sessionImpl.getSessionData().getCAiid()+
                     cpiid;
        sessionPortlets.remove(key);
    }

}
