package org.apache.jetspeed.portlet;

import java.io.IOException;
import java.util.Hashtable;

import javax.servlet.*;
import javax.servlet.http.*;

import com.ibm.wps.portletcontainer.invoker.PortletInvokerConstants;

/**
 * The <CODE>PortletAdapter</CODE> provides a default implementation
 * for the <CODE>Portlet</CODE> interface. It is recommended not to
 * extend the portlet interface directly. Rather, a portlet should derive
 * from this or any other derived class, because changes in the
 * <CODE>Portlet</CODE> interface are then mostly likely to be catched
 * by the default implementation, rather than breaking your portlet
 * implementation.
 * <P>
 * The virtual instance is created and destroyed with the
 * <CODE>login()</CODE> and <CODE>logout()</CODE> methods,
 * respectively. If a portlet provides personalized views these methods
 * should be implemented.
 * 
 * @author <A HREF="mailto:shesmer@apache.org">Stephan Hesmer</A>
 */
public abstract class PortletAdapter extends Portlet
implements PortletSessionListener
{

    public void init (PortletConfig config) throws UnavailableException
    {
        portletConfig = config;
    }

    public void destroy (PortletConfig config)
    {
        // do nothing
    }

    public void initConcrete (PortletSettings settings) throws UnavailableException
    {
        // do nothing
    }

    public void destroyConcrete (PortletSettings settings)
    {
        // do nothing
    }

    public void login (PortletRequest request) throws PortletException
    {
        // do nothing
    }

    public void logout (PortletSession session) throws PortletException
    {
        // do nothing
    }

    /**
     ** The default implementation of this method defines that
     ** the portlet never expires.
     **
     ** @param request	the <code>PortletRequest</code> 
     **			object that is sent to the portlet
     **
     ** @return		a <code>long</code> integer specifying
     **			the time the <code>PortletRequest</code>
     **			object was last modified, in milliseconds
     **			since midnight, January 1, 1970 GMT, or
     **			-1 if the time is not known
     **/
    public long getLastModified(PortletRequest request)
    {
        return -1;
    }

    /**
     ** The default implementation of this method routes the request
     ** to a set of helper methods that are already split out by
     ** portlet mode. Therefore if this portlet only supports the
     ** VIEW mode, only the <CODE>doView</CODE> method needs to be
     ** implemented.
     **
     ** @param   request
     **          the portlet request
     ** @param   response
     **          the portlet response
     **
     ** @see   #doView
     ** @see   #doEdit
     ** @see   #doHelp
     ** @see   #doConfigure
     **/

    public void service (PortletRequest request,
                         PortletResponse response) throws PortletException,
                                                          IOException
    {
        if (request.getMode () == Portlet.Mode.VIEW)
            doView (request, response);
        else if (request.getMode () == Portlet.Mode.EDIT)
            doEdit (request, response);
        else if (request.getMode () == Portlet.Mode.HELP)
            doHelp (request, response);
        else if (request.getMode () == Portlet.Mode.CONFIGURE)
            doConfigure (request, response);
    }

    /**
     ** Helper method to serve up the VIEW mode.
     **
     ** @param    request
     **           the portlet request
     ** @param    response
     **           the portlet response
     **/

    public void doView (PortletRequest request,
                        PortletResponse response) throws PortletException,
                                                         IOException
    {
        // do nothing
    }

    /**
     ** Helper method to serve up the EDIT mode.
     **
     ** @param    request
     **           the portlet request
     ** @param    response
     **           the portlet response
     **/

    public void doEdit (PortletRequest request,
                        PortletResponse response) throws PortletException,
                                                         IOException
    {
        // do nothing
    }

    /**
     ** Helper method to serve up the HELP mode.
     **
     ** @param    request
     **           the portlet request
     ** @param    response
     **           the portlet response
     **/

    public void doHelp (PortletRequest request,
                        PortletResponse response) throws PortletException,
                                                         IOException
    {
        // do nothing
    }

    /**
     ** Helper method to serve up the CONFIGURE mode.
     **
     ** @param    request
     **           the portlet request
     ** @param    response
     **           the portlet response
     **/

    public void doConfigure (PortletRequest request,
                             PortletResponse response) throws PortletException,
                                                              IOException
    {
        // do nothing
    }

    /**
     ** Returns the portlet configuration.
     **
     ** @return   the portlet configuration
     **/

    public PortletConfig getPortletConfig ()
    {
        return (portletConfig);
    }

    /**
     ** Returns the portlet log.
     **
     ** @return   the portlet log
     **/

    public PortletLog getPortletLog ()
    {
        return (getPortletConfig ().getContext ().getLog ());
    }

// Transient concrete portlet variable methods

    /**
     * Sets a transient variable of the concrete portlet.
     * 
     * @param name   the name of the transient variable
     * @param value  the value of the transient variable
     * @exception AccessDeniedException
     *                   if the method is called outside of a concrete portlet
     */
    public void setVariable (String name, Object value) throws AccessDeniedException
    {
        if (getPortletSettings()==null)
            throw new AccessDeniedException();
        String key = getPortletSettings().hashCode()+name;
        variables.put(key, value);
    }

    /**
     * Returns a transient variable of the concrete portlet.
     * 
     * @param name the name of the transient variable
     * @param value the value of the transient variable
     * @exception AccessDeniedException
     *                   if the method is called outside of a concrete portlet
     */
    public Object getVariable (String name) throws AccessDeniedException
    {
        if (getPortletSettings()==null)
            throw new AccessDeniedException();
        String key = getPortletSettings().hashCode()+name;
        return variables.get(key);
    }

    /**
     * Removes a transient variable of the concrete portlet.
     * 
     * @param name the name of the transient variable
     * @param value the value of the transient variable
     * @exception AccessDeniedException
     *                   if the method is called outside of a concrete portlet
     */
    public void removeVariable (String name) throws AccessDeniedException
    {
        if (getPortletSettings()==null)
            throw new AccessDeniedException();
        String key = getPortletSettings().hashCode()+name;
        variables.remove(key);
    }

    // --- PRIVATE MEMBERS --- //

    private PortletConfig portletConfig = null;

    private Hashtable variables = new Hashtable();

}
