package org.apache.jetspeed.portlet;

import org.apache.jetspeed.portlet.event.MessageEvent;
import org.apache.jetspeed.portlet.event.MessageListener;

import org.apache.jetspeed.portlet.service.PortletService;
import org.apache.jetspeed.portlet.service.PortletServiceUnavailableException;
import org.apache.jetspeed.portlet.service.PortletServiceNotFoundException;

import java.io.InputStream;
import java.io.IOException;

import java.util.Locale;
import java.util.Enumeration;

/**
 ** The <CODE>PortletContext</CODE> interface defines a portlet's view
 ** of the portlet container within which each portlet is running.
 ** The <CODE>PortletContext</CODE> also allows a portlet to access
 ** resources available to it. Using the context, a portlet can access
 ** the portlet log, and obtain URL references to resources.
 **
 ** @author <A HREF="mailto:shesmer@apache.org">Stephan Hesmer</A>
 **/

public interface PortletContext extends javax.servlet.ServletContext
{
    /**
     * Returns the value of the context-wide initialization parameter with
     * the given name, or <CODE>null</CODE> if the parameter does not exist.
     * 
     * <P>
     * This method can make available configuration information useful to
     * all portlets. The configuration is read from the portlet deployment
     * descriptor.
     * 
     * @param name
     *               the parameter name
     * @return the parameter value, <BR>
     *         or <CODE>null</CODE> if it does not exist
     */
    public String getInitParameter (String name);

    /**
     * Returns the names of this contexts initialization parameters as
     * an enumeration, or an empty enumeration if this context has no
     * initialization parameters.
     * <P>
     * The configuration is read from the portlet deployment descriptor.
     * 
     * @return an enumeration of parameter names
     */
    public Enumeration getInitParameterNames ();

    /**
     ** Associates an attribute with the given name and value with this
     ** context. If a portlet needs to communicate information to
     ** embedded servlets or JSP, this methods can used carry the
     ** information along.
     **
     ** @param   name
     **          the attribute name
     ** @param   value
     **          the attribute value
     **/

    public void setAttribute (String name, Object value);

    /**
     ** Returns the attribute value with the given name, or <CODE>null</CODE>
     ** if no such attribute exists.
     **
     ** <P>
     ** The context attributes can be used to share information between
     ** the portlets of one portlet application.
     **
     ** @param   name
     **          the attribute name
     **
     ** @return   the attribute value
     **/

    public Object getAttribute (String name);

    /**
     ** Returns an enumeration of the attribute names that this portlet
     ** context is aware of.
     **
     ** @return    an enumeration of attribute names
     **/

    public Enumeration getAttributeNames ();

    /**
     ** Removes the attribute with the given name.
     **
     ** @param   name
     **          the name of attribute to be removed
     **/

    public void removeAttribute (String name);

    /**
     ** Allows the portlet to delegate the rendering to another resource
     ** as specified by the given path. The path has to be relative
     ** and will be resolved by this method, so that the portlet's
     ** resources are accessed.
     **
     ** <P>
     ** To access protected resources the path has to be prefixed with
     ** /PORTLET-INF/ (e.g. /PORTLET-INF/myportlet/myportlet.jsp).
     ** Otherwise, the direct path is used. (e.g. /myportlet/myportlet.jsp).
     **
     ** <P>
     ** This method is enabled for multi-language and multi-device support.
     ** For example, a jsp file "/myportlet/mytemplate.jsp" will be searched for
     ** in the following order, when accessing via HTML-Browser:
     **
     ** <OL>
     ** <LI> /html/en/US/IE/mytemplate.jsp
     ** <LI> /html/en/US/mytemplate.jsp
     ** <LI> /html/en/mytemplate.jsp
     ** <LI> /html/mytemplate.jsp
     ** <LI> /en/US/IE/mytemplate.jsp
     ** <LI> /en/US/mytemplate.jsp
     ** <LI> /en/mytemplate.jsp
     ** <LI> /mytemplate.jsp
     ** </OL>
     **
     ** @param   path
     **          the path of the delegate resource
     ** @param   request
     **          the portlet request
     ** @param   response
     **          the portlet response
     **
     ** @exception   PortletException
     **              if the delegated resource has trouble fulfilling the
     **              rendering request
     ** @exception   IOException
     **              if the streaming causes an I/O problem
     **/

    public void include (String path,
                         PortletRequest request,
                         PortletResponse response) throws PortletException,
                                                          IOException;

     /**
      ** Returns the resource located at the given path as an
      ** <CODE>InputStream</CODE> object.
      **
      ** <P>
      ** The data in the <CODE>InputStream</CODE> can be of any type
      ** or length. The method returns <CODE>null</CODE> if no resource
      ** exists at the given path.
      **
      ** <P>
      ** To access protected resources the path has to be prefixed with
      ** /PORTLET_INF/ (e.g. /PORTLET-INF/myportlet/myportlet.jsp).
      ** Otherwise, the direct path is used. (e.g. /myportlet/myportlet.jsp).
      **
      ** @param   path
      **          the path to the resource
      **
      ** @return   the input stream
      **/

     public InputStream getResourceAsStream (String path);

     /**
      ** Returns the resource located at the given path as an
      ** <CODE>InputStream</CODE> object.
      **
      ** <P>
      ** The data in the <CODE>InputStream</CODE> can be of any type
      ** or length. The method returns <CODE>null</CODE> if no resource
      ** exists at the given path.
      **
      ** <P>
      ** To access protected resources the path has to be prefixed with
      ** /PORTLET_INF/ (e.g. /PORTLET-INF/myportlet/myportlet.jsp).
      ** Otherwise, the direct path is used. (e.g. /myportlet/myportlet.jsp).
      ** <P>
      ** This method is enabled for multi-language and multi-device support.
      ** For example, a jsp file "/myportlet/mytemplate.jsp" will be searched for
      ** in the following order, when accessing via HTML-Browser:
      **
      ** <OL>
      ** <LI> /html/en/US/IE/mytemplate.jsp
      ** <LI> /html/en/US/mytemplate.jsp
      ** <LI> /html/en/mytemplate.jsp
      ** <LI> /html/mytemplate.jsp
      ** <LI> /en/US/IE/mytemplate.jsp
      ** <LI> /en/US/mytemplate.jsp
      ** <LI> /en/mytemplate.jsp
      ** <LI> /mytemplate.jsp
      ** </OL>
      **
      ** @param   path
      **          the path to the resource
      ** @param   client
      **          the client
      ** @param   locale
      **          the locale
      **
      ** @return   the input stream
      **/

     public InputStream getResourceAsStream (String path, Client client, Locale locale);

     /**
      ** Returns the localized text resource with the given key and
      ** using the given locale.
      **
      ** <P>
      ** To use this feature, the portlet application's
      ** <CODE>CLASSPATH</CODE> has to contain a resource bundle with
      ** the same name (including the package) as the portlet.
      **
      ** @param   bundle
      **          the name of the resource bundle
      ** @param   key
      **          the text key
      ** @param   locale
      **          the locale to observe
      **
      ** @return   the localized text resource
      **/

     public String getText (String bundle, String key, Locale locale);

     /**
      * Sends the given message to all portlets in the same portlet
      * application on the same page that have the given name.
      * If the portlet name is <CODE>null</CODE> the message is
      * broadcast to all portlets in the same portlet application.
      * If more than one instance of the portlet with the given
      * name exists on the current page, the message is
      * sent to every single instance of that portlet. If the
      * source portlet has the same name as the target portlet(s),
      * the message will not be sent to avoid possible cyclic calls.
      * 
      * <P>
      * The portlet(s) with the given name will only receive the
      * message event if it has/they have implemented the appropriate
      * listener.
      * 
      * <P>
      * This function may only be used during event processing,
      * in any other case the method throws an AccessDeniedException.
      * 
      * @param portletName
      *                the name of the portlet(s) that this
      * @param message
      *                the message to be sent
      * @param request
      *                the portlet request
      * @exception AccessDeniedException
      *                   if the portlet tries to access this function
      *                   outside of the event processing
      * @deprecated since Version 1.1
      */
     public void send (String portletName,
                       PortletMessage message,
                       PortletRequest request) throws AccessDeniedException;

     /**
      * Sends the given message to all portlets in the same portlet
      * application on the same page that have the given name.
      * If the portlet name is <CODE>null</CODE> the message is
      * broadcast to all portlets in the same portlet application.
      * If more than one instance of the portlet with the given
      * name exists on the current page, the message is
      * sent to every single instance of that portlet. If the
      * source portlet has the same name as the target portlet(s),
      * the message will not be sent to avoid possible cyclic calls.
      * 
      * <P>
      * The portlet(s) with the given name will only receive the
      * message event if it has/they have implemented the appropriate
      * listener.
      * 
      * <P>
      * This function may only be used during event processing,
      * in any other case the method throws an AccessDeniedException.
      * 
      * @param portletName
      *                the name of the portlet(s) that this
      * @param message
      *                the message to be sent
      * @exception AccessDeniedException
      *                   if the portlet tries to access this function
      *                   outside of the event processing
      */
     public void send (String portletName,
                       PortletMessage message) throws AccessDeniedException;

     /**
      * Sends the given message to all portlets on the same page
      * that have the given name regardless of the portlet application.
      * If the portlet name is <CODE>null</CODE> the message is
      * broadcast to all portlets on the same page across all portlet application.
      * If more than one instance of the portlet with the given
      * name exists on the current page, the message is
      * sent to every single instance of that portlet. If the
      * source portlet has the same name as the target portlet(s),
      * the message will not be sent to avoid possible cyclic calls.
      * 
      * <P>
      * The portlet(s) with the given name will only receive the
      * message event if it has/they have implemented the appropriate
      * listener.
      * 
      * <P>
      * This function may only be used during event processing,
      * in any other case the method throws an AccessDeniedException.
      * 
      * @param portletName
      *                the name of the portlet(s) that this
      * @param message
      *                the message to be sent
      * @exception AccessDeniedException
      *                   if the portlet tries to access this function
      *                   outside of the event processing
      */
     public void send (String portletName,
                       DefaultPortletMessage message) throws AccessDeniedException;

     /**
      ** This function looks up a portlet service with the
      ** given classname.
      **
      ** <P>
      ** Using this method a portlet is able to get additional functionality
      ** like a service to get external content over a firewall or to include
      ** a servlet.
      **
      ** @param   service
      **          the classname of the service to load
      **
      ** @return   the portlet service
      **
      ** @exception   PortletServiceUnavailableException
      **              if an exception has occurrred that interferes
      **              with the portlet service's normal initialization
      ** @exception   PortletServiceNotFoundException
      **              if the PortletService is not found
      **/

     public PortletService getService(Class service) throws PortletServiceUnavailableException,
                                                            PortletServiceNotFoundException;

     /**
      ** Returns the major version of the Portlet API that this portlet
      ** container supports.
      **
      ** @return   the major version
      **
      ** @see   #getMinorVersion()
      **/

     public int getMajorVersion ();

     /**
      ** Returns the minor version of the Portlet API that this portlet
      ** container supports.
      **
      ** @return   the minor version
      **
      ** @see   #getMajorVersion()
      **/

     public int getMinorVersion ();

     /**
      ** Returns the name and version of the portlet container which the
      ** portlet is running in.
      **
      ** <P>
      ** The form of the returned string is <IT>servername/versionnumber</IT>.
      ** For the Jetspeed Framework this method may return the string
      ** <CODE>Apache Jetspeed/1.3a1</CODE>.
      **
      ** @return   the string containing at least name and version number
      **/

      public String getContainerInfo ();

      /**
       ** Returns the portlet log which allows the portlet to write
       ** debug, informational, warning, or error messages to a log.
       **
       ** @return   the portlet log
       **/

      public PortletLog getLog ();
}
