/***************************************************************************
begin                : Mon Jan 28 2002
copyright            : (C) 2002 by Christian Hubinger
email                : chubinger@gmail.com
***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef IPTCHAIN_H
#define IPTCHAIN_H

#include "netfilterobject.h"

#include "qptrlist.h"
#include "qstringlist.h"
#include "qdom.h"

#include <kdemacros.h>

class KMFError;
class IPTRule;
class IPTable;

/**
  *@author Christian Hubinger
  */


class KDE_EXPORT IPTChain : public NetfilterObject {
public:

	/** The Constructor that will set the "Name" and the "Table" of the chain. If buildin is set true the chain will be handeled as bild-in chain. Normaly this is only done at the initiallistaion of the whole KMFIPTDoc.
	  */
	IPTChain( IPTable* table, const QString& name , bool buildin = false );
	~IPTChain();

	/** Sets the table to which this chain will be added */
	void setTable( IPTable* table );

	/** Return the IPTAble to which this chain belongs */
	IPTable* table() const;

	/** Sets the Default Policy of the Chain if it is a
		build in chain like ( $IPT -P INPUT -j DROP )
		For user defined chain this function generates a
		rule like ( $IPT -A CHAINNAME -j DROP ) which will
		be added at the end of the chain to avoid that the
		packetes jump back to the chain that feeded this chain. */
	void setDefaultTarget( const QString& );

	/** Enabbles/Disables the build in logging for this Chain.
		This will generate a Rule like: $IPT -A CHAINNAME --match
		limit --limit <limit> --limit-burst <burst> -j LOG --log-prefix
		<prefix> This rule will be placed right before the
		Default Policy to log just the Packets that don't match any
		of the rules defined by this chain. limit, burst, prefix may be
		empty strings to disable those features.
		If enable is "false" the logging will be deactivated. */
	void setDropLogging( bool enable,
	                     QString& limit ,
	                     QString& burst ,
	                     QString& prefix );

	/** Defines weather a Chain is build-in or is user-defined */
	void setBuildIn( bool ); 
	
	/** will set a switch that lets the chain be shown by the GUI */
	void setUsed( bool ); 

	/** Sets if a chain has a "Default Policy" set. */
	void hasCustomDefaultTarget( bool has_target );

	/** Moves the rule up (to beginning of the chain) if how_much < 0 else
	  moves the rule down (to end of the chain) */
	bool moveRule( IPTRule* rule, int how_much );

	/** Simply adds a Rule to the Chains Ruleset. */
	IPTRule* addRule( const QString& name, KMFError*, int index = -1 );

	/** The the index of the rule in the chain **/
	int indexOf( const IPTRule& );
	
	/** Deletes a given Rule in the Ruleset. */
	KMFError* delRule( IPTRule* );

	/** Returns a pointer to a rule for the given name if it exists in
		this chain. If no rule is found 0 is returned */
	IPTRule* ruleForName( const QString& name );

	/** Returns a pointer to a rule for the given object id if it exists in
		this chain. If no rule is found 0 is returned */
	IPTRule* ruleForID( int objectID );


	/** Returns "true" if the logging is enabled, otherwhise "false". */
	bool logging() const {
		return enable_log;
	};

	/** Returns "true" if the Chain is a build-in chain and "fasle" if it is
	  an user defined chain */
	bool isBuildIn() const {
		return is_build_in_chain;
	};

	/** Returns "true" if the Chain has a "Default Policy" set or not. */
	bool hasDefaultTarget() const {
		return has_default_target;
	};

	/** Returns "true" if the Chain should be visible inn the GUI. */
	bool isUsed() const {
		return is_used;
	};
	
	/** Reset the chain to its initial state */
	void reset();

	/** Returns the limit of matches for the logging - rule. */
	const QString& logLimit() const {
		return m_log_limit;
	};

	/** Returns the log prefix */
	const QString& logPrefix() const {
		return m_log_prefix;
	};

	/** Returns the log limit burst number. */
	const QString& logBurst() const {
		return m_log_burst;
	};

	/** Returns the returns the Chaindefinition ($IPT -N chainname)
		if this is a user defined chain. if it's a build in chain
		like INPUT it returnes an empty string. */
	const QString& chainDefinitions() const {
		return m_cmd_chain_definition ;
	} ;

	/** Returns the Chain Policy ( "DROP" "ACCEPT" "LOG" etc ) */
	const QString& defaultTarget() const {
		return m_default_target;
	} ;

	/** Returns rules that forward packets to other chains.
	  Those Rules have a chain as their target ( -j <chainname>) */
	QPtrList<IPTRule>& chainFwds();

	/** Returns rules that forward packets to this chain.
	  Those Rules have this chain as their target ( -j <chainname>) */
	QPtrList<IPTRule>& chainFeeds();

	/** Returns the whole ruleset defined by the Chain as
	    an QPtrList<IPTRule>. */
	QPtrList<IPTRule>& chainRuleset() const;

	/** Returns a list of strings like: "$IPT -A CHAINNAME OPTIONS
		[ all kind of allowed optiions p.e --match state --state NEW,
		RELATED --source 192.168.0.2 ... ] -j TARGET [target options
		p.e --log-prefix 'KMF:' ] */
	QPtrList<QStringList>& createIPTablesChainRules();

	/** Returns a String like: "$IPT -P INPUT DROP" for build-in
		and "$IPT -A CHAINNAME -j DROP" for user defined chans. */
	QString createIPTablesChainDefaultTarget();

	/** Returns a String like: "$IPT -N CHAINNAME" that is written to the script. */
	QString createIPTablesChainDefinition();  

	/** Return DomDocument of this Chain */
	const QDomDocument& getDOMTree();

	/** Lad the XML sniplet representing the chain */
	void loadXML( const QDomDocument& );


private:     // functions

private: //data	
	// initialisized by the constructor
	IPTable* m_table;
	KMFError* m_err;

	bool is_used;
	bool enable_log;
	bool is_build_in_chain;
	bool has_default_target;

	QString m_log_limit;
	QString m_log_prefix;
	QString m_log_burst;

	QString m_default_target;
	QString m_cmd_chain_definition;
	QString m_cmd_default_target;

	QPtrList<IPTRule> m_ruleset;
};

#endif
