/***************************************************************************
*   Copyright (C) 2010 by Thomas Fischer                             *
*   fischer@unix-ag.uni-kl.de                                             *
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as published by  *
*   the Free Software Foundation; either version 3 of the License, or     *
*   (at your option) any later version.                                   *
*                                                                         *
*   This program is distributed in the hope that it will be useful,       *
*   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
*   GNU General Public License for more details.                          *
*                                                                         *
*   You should have received a copy of the GNU General Public License     *
*   along with this program; if not, write to the                         *
*   Free Software Foundation, Inc.,                                       *
*   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
***************************************************************************/

#include <QMetaClassInfo>
#include <QLayout>
#include <QLabel>
#include <QIODevice>
#include <QTreeWidget>
#include <QMenu>
#include <QSignalMapper>
#include <QApplication>
#include <QCheckBox>

#include <KListWidget>
#include <KMessageBox>
#include <KTemporaryFile>
#include <KFileDialog>
#include <KPushButton>
#include <KMimeType>
#include <KMimeTypeTrader>
#include <KIcon>
#include <KLocale>
#include <KRun>
#include <KDebug>
#include <KStandardDirs>

#include <qtbrowserplugin.h>
#include "kpartsplugin.h"


/// To order the mimetype list
static bool mimeTypeLessThan(const KMimeType::Ptr& m1, const KMimeType::Ptr& m2)
{
    return m1->name() < m2->name();
}

static QStringList allMimeTypes, enabledMimeTypes;

static const QString configFilename(QLatin1String("kpartsplugin-mimetypes.rc"));
static const QString configSectionBlacklisted(QLatin1String("Blacklisted"));
static const QString configSectionPreferredService(QLatin1String("PreferredService"));

/// built-in list of mime types that should never be loaded with this plugin
/// comparison is done with "startsWith", so "inode/" covers e.g. "inode/directory"
static const QStringList builtinBlacklisted = QStringList() << QLatin1String("all/") << QLatin1String("x-") << QLatin1String("inode/") << QLatin1String("application/x-shockwave") << QLatin1String("application/force-download") << QLatin1String("interface/") << QLatin1String("message/") << QLatin1String("multipart/");

static void initAllMimeTypes()
{
    if (!allMimeTypes.isEmpty()) return;

    /// load configuration to check which mime types are black-listed by user
    KSharedPtr<KSharedConfig> userConfig = KSharedConfig::openConfig(KStandardDirs::locateLocal("config", configFilename), KConfig::SimpleConfig);
    KConfigGroup config(userConfig, configSectionBlacklisted);

    /// fetch complete list of mime types from KDE subsystem and sort them
    KMimeType::List mimetypes = KMimeType::allMimeTypes();
    qSort(mimetypes.begin(), mimetypes.end(), mimeTypeLessThan);

    /// go through each mime type
    for (KMimeType::List::ConstIterator it(mimetypes.constBegin()); it != mimetypes.constEnd(); ++it) {
        const QString mimetype = (*it)->name();

        /// ignore special mime types
        bool isBuiltinBlacklisted = false;
        for (QStringList::ConstIterator bbit = builtinBlacklisted.constBegin(); !isBuiltinBlacklisted && bbit != builtinBlacklisted.constEnd(); ++bbit)
            isBuiltinBlacklisted |= mimetype.startsWith(*bbit);
        if (isBuiltinBlacklisted) {
            kDebug() << "Skipping blacklisted (built-in) mime type " << mimetype;
            continue;
        }

        /// fetch additional info (extension and description)
        const QString extension = (*it)->mainExtension().mid(1);
        const QString description = (*it)->comment();

        /// search for read-only parts that can display this mimetype
        KService::List list = KMimeTypeTrader::self()->query(mimetype, QString::fromLatin1("KParts/ReadOnlyPart"));
        if (!list.isEmpty()) {
            /// add three-column info to list of all mime types
            allMimeTypes.append(QString("%1:%2:%3").arg(mimetype).arg(extension).arg(description));
            if (!config.readEntry(mimetype, false)) {
                /// add three-column info to list of enabled mime types only if not blacklisted
                enabledMimeTypes.append(QString("%1:%2:%3").arg(mimetype).arg(extension).arg(description));
            } else
                kDebug() << "Skipping blacklisted (user config) mime type " << mimetype;
        }
    }
}

class KPPServiceListDialog : public KDialog
{
public:
    KListWidget *listWidget;
    QCheckBox *checkBoxRemember;

    /**
     * Create a dialog which presents the user a list of choices. One choice is pre-selected.
     * A checkbox allows the user to store his/her preference.
     */
    KPPServiceListDialog(const QStringList &list, const QString &selected, const QString &caption, const QString &text, QWidget *parent = NULL, Qt::WFlags flags = NULL)
            : KDialog(parent, flags) {
        /// set various dialog configurations
        setWindowModality(Qt::NonModal);
        setCaption(caption);
        setButtons(KDialog::Ok);

        /// central widget for dialog
        QWidget *container = new QWidget(this);
        setMainWidget(container);
        QGridLayout *layout = new QGridLayout(container);

        /// label holding the icon in the dialog's upper left corner
        QLabel *label = new QLabel(container);
        label->setPixmap(KIconLoader::global()->loadIcon("preferences-desktop-filetype-association", KIconLoader::NoGroup, KIconLoader::SizeLarge));
        layout->addWidget(label, 0, 0, 3, 1, Qt::AlignTop);

        /// label showing the text message
        label = new QLabel(text, container);
        label->setTextFormat(Qt::RichText);
        layout->addWidget(label, 0, 1, 1, 1, Qt::AlignTop);
        label->setWordWrap(true);

        /// list with all choices for the user
        listWidget = new KListWidget(container);
        label->setBuddy(listWidget);
        for (QStringList::ConstIterator it = list.constBegin(); it != list.constEnd(); ++it) {
            QListWidgetItem *item = new QListWidgetItem(*it, listWidget);
            item->setSelected(*it == selected);
            listWidget->addItem(item);
        }
        listWidget->setSelectionMode(QAbstractItemView::SingleSelection);
        layout->addWidget(listWidget, 1, 1, 1, 1);

        /// checkbox to set remembering the default choice
        checkBoxRemember = new QCheckBox(i18n("Remember selection as default"), container);
        checkBoxRemember->setTristate(false);
        layout->addWidget(checkBoxRemember, 2, 1, 1, 1);

        /// double-clicking on list item is like pressing "Ok"
        connect(listWidget, SIGNAL(doubleClicked(QListWidgetItem *)), this, SLOT(accept()));
    }

    /**
     * Static function simplifying using KPPServiceListDialog. Variable rememberChoice is a reference
     * returning the checkbox's check state.
     */
    static QString selectStringFromList(bool &rememberChoice, const QStringList &list, const QString &selected, const QString &caption, const QString &text, QWidget *parent = NULL, Qt::WFlags flags = NULL) {
        Q_ASSERT(list.size() > 0);

        KPPServiceListDialog dlg(list, selected, caption, text, parent, flags);
        dlg.activateWindow();
        dlg.raise();
        dlg.exec();
        rememberChoice = dlg.checkBoxRemember->checkState() == Qt::Checked;
        /// fetch first (and only) selected item in list
        QList<QListWidgetItem*>::ConstIterator it(dlg.listWidget->selectedItems().constBegin());
        return (*it)->text();
    }

};

KPartsPlugin::KPartsPlugin(QWidget *parent)
        : QWidget(parent), QtNPBindable(), m_part(NULL), m_listMimeTypes(NULL), m_calledOnce(false)
{
    /// set cursor to hourglass or similar
    setCursor(Qt::WaitCursor);
    /// initialize the plugin's visual layout
    setLayout(m_gridLayout = new QGridLayout());
    /// change focus policy to receive keyboard input
    setFocusPolicy(Qt::StrongFocus);
    /// force activation of this plugin
    QApplication::setActiveWindow(this);

    /// configure temporary file
    m_tempFile.setPrefix("KPartsPlugin");
    m_tempFile.setAutoRemove(true);

    /// fetch all mime types
    initAllMimeTypes();

    /// create user interface
    createGUI();
}

KService::Ptr KPartsPlugin::selectService(const QString & format)
{
    QMap<QString, KService::Ptr> nameToKService;
    QStringList serviceList;
    /// determine preferred service as configured in the KDE settings
    KService::Ptr preferredService = KMimeTypeTrader::self()->preferredService(format, QString::fromLatin1("KParts/ReadOnlyPart"));
    /// stick with preferred service unless there are several to choose from
    KService::Ptr service = preferredService;

    /// fetch list of available services
    KService::List list = KMimeTypeTrader::self()->query(format, QString::fromLatin1("KParts/ReadOnlyPart"));
    for (KService::List::Iterator it = list.begin(); it != list.end(); ++it)
        /// exclude Netscape plugins (remember, this is a *KDE* plugin)
        if (!(*it)->name().contains(QLatin1String("Netscape"), Qt::CaseInsensitive)) {
            /// store both service's name and mapping from name to actual service
            serviceList << (*it)->name();
            nameToKService.insert((*it)->name(), *it);
        }

    /// fall back to plain text handler for unknown text file types
    if (serviceList.isEmpty() && format.startsWith(QString::fromLatin1("text/"), Qt::CaseInsensitive)) {
        KService::List list = KMimeTypeTrader::self()->query(QString::fromLatin1("text/plain"), QString::fromLatin1("KParts/ReadOnlyPart"));
        for (KService::List::Iterator it = list.begin(); it != list.end(); ++it)
            /// exclude Netscape plugins (remember, this is a *KDE* plugin)
            if (!(*it)->name().contains(QLatin1String("Netscape"), Qt::CaseInsensitive)) {
                /// store both service's name and mapping from name to actual service
                serviceList << (*it)->name();
                nameToKService.insert((*it)->name(), *it);
            }
    }

    /// if more than one service has been found, use stored preferred service or user choice to select
    if (serviceList.count() > 1) {
        /// load configuration data for preferred services (independent from KDE's preferred service!)
        KSharedPtr<KSharedConfig> userConfig = KSharedConfig::openConfig(KStandardDirs::locateLocal("config", configFilename), KConfig::SimpleConfig);
        KConfigGroup config(userConfig, configSectionPreferredService);

        QString name = config.readEntry(format, "");
        /// if no preference for this mime type has been saved, show selection dialog
        if (name.isEmpty()) {
            bool rememberChoice = false;
            setEnabled(false);
            name = KPPServiceListDialog::selectStringFromList(rememberChoice, serviceList, preferredService->name(), i18n("KPart Selection"), i18n("Select the KPart to be used for the mime type<br/><b>%1</b>.<br/>The default part<br/><b>%2</b><br/>has been selected.", format, preferredService->name()));
            /// if user has checked to store preference ...
            if (rememberChoice) {
                /// write updated configuration
                config.writeEntry(format, name);
                userConfig->sync();
            }
            setEnabled(true);
        }
        /// determine which service to use eventually
        if (nameToKService.contains(name))
            service = nameToKService[name];
    }

    return service;
}

bool KPartsPlugin::readData(QIODevice * source, const QString & format)
{
    /// fetch information on mime type
    KMimeType::Ptr mimeTypeKDE = KMimeType::mimeType(format);
    m_mimeTypeLabel->setText(i18n("<b>%1</b> (%2)", format, mimeTypeKDE->comment()));

    /// prevent running this function twice
    if (m_calledOnce) {
        kWarning() << " readData was called multiple times!";
        return false;
    }
    m_calledOnce = true;

    /// enable buttons for opening in external program and saving as file
    m_openButton->setEnabled(true);
    m_saveButton->setEnabled(true);
    /// allow UI to update
    QCoreApplication::instance()->processEvents();

    /// copy io data into temporary file using a suffix matching to mime type
    m_label->setText(i18n("Copying data ..."));
    m_tempFile.setSuffix(mimeTypeKDE->mainExtension());
    copyIODevice(source, &m_tempFile);
    /// allow UI to update
    QCoreApplication::instance()->processEvents();

    /// locate and load KPart plugin for this mime type
    m_label->setText(i18n("Locating KPart ..."));
    KService::Ptr service = selectService(format);
    /// allow UI to update
    QCoreApplication::instance()->processEvents();

    if (service->isValid()) {
        m_label->setText(i18n("Initializing ..."));
        m_part = service->createInstance<KParts::ReadOnlyPart>((QWidget*)parent(), (QObject*)parent());

        /// initialize plugin, set layout
        QWidget *partWidget = m_part->widget();
        m_gridLayout->addWidget(partWidget, 1, 0, 2, 4);
        /// open temporary file in KPart plugin
        m_part->openUrl(KUrl(m_tempFile.fileName()));

        /// clean up memory and restore curser
        delete m_label;
        delete m_listMimeTypes;
        m_listMimeTypes = NULL;
        setCursor(Qt::ArrowCursor);
    } else {
        /// show error message and exit gracefully
        m_label->setText(i18n("Failed to load KPart for mime type \"%1\".", format));
        setCursor(Qt::ArrowCursor);
        return false;
    }

    return true;
}

bool KPartsPlugin::copyIODevice(QIODevice * source, QIODevice * target)
{
    /// initialize buffer for copy operation
    const int bufferSize = 32768;
    char buffer[bufferSize];

    /// open input and output devices
    source->open(QIODevice::ReadOnly);
    target->open(QIODevice::WriteOnly);

    /// as long as there are bytes to load ...
    qint64 ba = source->bytesAvailable();
    while (ba > 0) {
        /// read from input, write to output
        qint64 br = source->read(buffer, bufferSize);
        qint64 bw = target->write(buffer, br);
        if (br != bw) {
            /// io operations messed up
            source->close();
            target->close();
            return false;
        }
        ba = source->bytesAvailable();
    }
    source->close();
    target->close();

    return true;
}

bool KPartsPlugin::createGUI()
{
    /// set streching for columns and rows
    m_gridLayout->setColumnStretch(0, 1);
    m_gridLayout->setColumnStretch(1, 0);
    m_gridLayout->setColumnStretch(2, 0);
    m_gridLayout->setColumnStretch(3, 0);
    m_gridLayout->setRowStretch(0, 0);
    m_gridLayout->setRowStretch(1, 0);
    m_gridLayout->setRowStretch(2, 1);

    /// configure central message label
    m_label = new QLabel(i18n("Loading ..."));
    m_label->setAlignment(Qt::AlignCenter);

    /// add label about which mime type has been loaded
    m_mimeTypeLabel = new QLabel();
    m_mimeTypeLabel->setAlignment(Qt::AlignVCenter | Qt::AlignRight);
    m_mimeTypeLabel->setWordWrap(true);

    /// add push button with context menu to enable/disable mime types
    m_supportedMimeTypesButton = new KPushButton(KIcon("preferences-desktop-filetype-association"), i18n("Supported Mime Types"));
    m_supportedMimeTypesMenu = new QMenu(m_supportedMimeTypesButton);
    m_supportedMimeTypesButton->setMenu(m_supportedMimeTypesMenu);
    QSignalMapper *supportedMimeTypesSignalMapper = new QSignalMapper(this);
    connect(supportedMimeTypesSignalMapper, SIGNAL(mapped(const QString &)), this, SLOT(slotSwitchMimeType(const QString &)));

    /// add push button to open current file in external program
    m_openButton = new KPushButton(KIcon("document-open"), i18n("Open in External Program ..."));
    connect(m_openButton, SIGNAL(clicked()), this, SLOT(slotOpenTempFile()));
    m_openButton->setEnabled(false);

    /// add push button to save current file in an user-specified file
    m_saveButton = new KPushButton(KIcon("document-save-as"), i18n("Save Copy ..."));
    connect(m_saveButton, SIGNAL(clicked()), this, SLOT(slotSaveTempFile()));
    m_saveButton->setEnabled(false);

    /// setup large list with mime types shown while data for file is transferred
    m_listMimeTypes = new QTreeWidget();
    QFontMetrics fm(m_listMimeTypes->font());
    m_listMimeTypes->setSelectionMode(QAbstractItemView::NoSelection);
    m_listMimeTypes->setHeaderLabels(QStringList() << i18n("Mime type") << QLatin1String("Extension") << QLatin1String("Description"));
    m_listMimeTypes->setColumnWidth(0, fm.width("AwIkKJ0235W") * 6);
    m_listMimeTypes->setColumnWidth(1, fm.width("AwIkKJ0235W"));
    m_listMimeTypes->setColumnWidth(2, fm.width("AwIkKJ0235W") * 12);

    /// fill list of mime types and initialize menu structure for button to enable/disable mimetypes
    QMap<QString, QMenu*> subMenuMap;
    for (QStringList::ConstIterator it = allMimeTypes.constBegin(); it != allMimeTypes.constEnd(); ++it) {
        /// setup list item
        KIcon icon = enabledMimeTypes.contains(*it) ? KIcon("button_more") : KIcon("button_fewer");
        QTreeWidgetItem *item = new QTreeWidgetItem(m_listMimeTypes);
        item->setIcon(0, icon);
        QStringList list = (*it).split(":");
        item->setText(0, list[0]);
        item->setText(1, list[1]);
        item->setText(2, list[2]);

        /// initialize menu
        QStringList mimeCat = list[0].split("/");
        QMenu *subMenu = NULL;
        if (subMenuMap.contains(mimeCat[0]))
            subMenu = subMenuMap.value(mimeCat[0]);
        else {
            subMenu = m_supportedMimeTypesMenu->addMenu(mimeCat[0]);
            subMenuMap.insert(mimeCat[0], subMenu);
        }
        QAction *action = subMenu->addAction(QString("%1 (%2)").arg(mimeCat[1]).arg(list[2]), supportedMimeTypesSignalMapper, SLOT(map()));
        action->setCheckable(true);
        action->setChecked(enabledMimeTypes.contains(*it));
        supportedMimeTypesSignalMapper->setMapping(action, list[0]);
    }

    /// add widgets to layout
    m_gridLayout->addWidget(m_mimeTypeLabel, 0, 0, 1, 1);
    m_gridLayout->addWidget(m_supportedMimeTypesButton, 0, 1, 1, 1);
    m_gridLayout->addWidget(m_openButton, 0, 2, 1, 1);
    m_gridLayout->addWidget(m_saveButton, 0, 3, 1, 1);
    m_gridLayout->addWidget(m_label, 1, 0, 1, 4);
    m_gridLayout->addWidget(m_listMimeTypes, 2, 0, 1, 4);

    return true;
}

void KPartsPlugin::slotOpenTempFile()
{
    /// use KDE subsystem to open temporary file with proper application
    KRun::runUrl(KUrl(m_tempFile.fileName()), mimeType(), this);
}

void KPartsPlugin::slotSaveTempFile()
{
    /// determine file name to save data to
    QString fileName = KFileDialog::getSaveFileName(KUrl(), mimeType());
    if (fileName.isEmpty() || fileName.isNull()) return;

    /// save data by copying temporary file to new file
    QFile file(fileName);
    copyIODevice(&m_tempFile, &file);
}

void KPartsPlugin::slotSwitchMimeType(const QString & text)
{
    /// load configuration data
    KSharedPtr<KSharedConfig> userConfig = KSharedConfig::openConfig(KStandardDirs::locateLocal("config", configFilename), KConfig::SimpleConfig);
    KConfigGroup config(userConfig, configSectionBlacklisted);
    /// toggle blacklist status from how it is set in current configuration
    bool isBlacklisted = !config.readEntry(text, false);
    config.writeEntry(text, isBlacklisted);
    /// write updated configuration
    userConfig->sync();

    KMessageBox::information((QWidget*)parent(), (isBlacklisted ? i18n("The mime type \"%1\" has been disabled for KPartsPlugin.", text) : i18n("The mime type \"%1\" has been enabled for KPartsPlugin.", text)) + i18n("\n\nThe browser has to be restarted to make the change effective."), QLatin1String("KPartsPlugin"));
}

void KPartsPlugin::enterEvent(QEvent *event)
{
    /** code from Jeremy Sanders */
    // this is required because firefox stops sending keyboard
    // events to the plugin after opening windows (e.g. download dialog)
    // setting the active window brings the events back
    if (QApplication::activeWindow() == NULL) {
        QApplication::setActiveWindow(this);
    }

    QWidget::enterEvent(event);
}


#include "kpartsplugin.moc"


class QtNPClassList : public QtNPFactory
{
private:
    QString m_name, m_description;

public:
    QtNPClassList()
            : m_name("KParts Plugin"), m_description("File viewer using KDE's KParts technology (2010-07-23)") {
        initAllMimeTypes();
    }

    ~QtNPClassList() {
        // nothing
    }

    QObject *createObject(const QString &) {
        return new KPartsPlugin();
    }

    QStringList mimeTypes() const {
        return enabledMimeTypes;
    }

    QString pluginName() const {
        return m_name;
    }

    QString pluginDescription() const {
        return m_description;
    }

};

QtNPFactory *qtns_instantiate()
{
    return new QtNPClassList;
}
