/* Copyright 2004 Red Hat, Inc.
 *
 * This software may be freely redistributed under the terms of the GNU
 * public license.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 * 
 */

#include <ctype.h>
#include <dirent.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/ioctl.h>
#include <linux/fd.h>

#include "vio.h"
#include "modules.h"

static void vioFreeDevice(struct vioDevice *dev)
{
	freeDevice((struct device *) dev);
}

static void vioWriteDevice(FILE *file, struct vioDevice *dev)
{
	writeDevice(file, (struct device *)dev);
}

static int vioCompareDevice(struct vioDevice *dev1, struct vioDevice *dev2)
{
	return compareDevice( (struct device *)dev1, (struct device *)dev2);
}

struct vioDevice *vioNewDevice(struct vioDevice *old)
{
	struct vioDevice *ret;

	ret = malloc(sizeof(struct vioDevice));
	memset(ret, '\0', sizeof(struct vioDevice));
	ret = (struct vioDevice *) newDevice((struct device *) old, (struct device *) ret);
	ret->bus = BUS_VIO;
	ret->newDevice = vioNewDevice;
	ret->freeDevice = vioFreeDevice;
	ret->writeDevice = vioWriteDevice;
	ret->compareDevice = vioCompareDevice;
	return ret;
}

struct device *vioProbe(enum deviceClass probeClass, int probeFlags,
			struct device *devlist)
{
    struct vioDevice *viodev;
    int fd;

#ifdef __powerpc__
    if (probeClass & CLASS_HD) {
	char * buf = NULL, * start, * end, * chptr, * next, * model, * ptr;
	int ctlNum = 0;

	if (access("/proc/iSeries/viodasd", R_OK))
	    goto dasddone;

	/* Read from /proc/iSeries/viodasd */
	fd = open("/proc/iSeries/viodasd", O_RDONLY);
	if (fd < 0) {
	    /* fprintf(stderr, "failed to open /proc/iSeries/viodasd!\n");*/
	    goto dasddone;
	}

	start = buf = bufFromFd(fd);
	close(fd);

	end = start + strlen(start);
	while (*start && start < end) {
	    /* parse till end of line and store the start of next line. */
	    chptr = start;
	    while (*chptr != '\n') chptr++;
	    *chptr = '\0';
	    next = chptr + 1;

	    /* get rid of anything which is not alpha */
	    while (!(isalpha(*start))) start++;

	    model = NULL;
	    if (!strncmp("DISK ", start, 5))
		model = "IBM Virtual DASD";

	    if (model) {
		chptr = start += 5;
		ptr = strchr(chptr, ' ');
		*ptr = '\0';
		ctlNum = atoi(chptr);

		viodev = vioNewDevice(NULL);
		viodev->device = malloc(20);
		if (ctlNum < 26) {
		    snprintf(viodev->device, 19, "iseries/vd%c", 'a' + ctlNum);
		} else {
		    snprintf(viodev->device, 19, "iseries/vda%c", 'a' + ctlNum - 26);
		}
		viodev->desc = malloc(64);
		viodev->desc = strdup(model);
		viodev->type = CLASS_HD;
		viodev->driver = strdup("viodasd");
		if (devlist) 
		    viodev->next = devlist;
		devlist = (struct device *) viodev;
		//    printf("model is %s, ctl is %s\n", model, ctl);
	    }

	    start = next;
	    end = start + strlen(start);
	} /* end of while */

dasddone:
	if (buf)
	    free (buf);
    } 

    if (probeClass & CLASS_NETWORK) {
	DIR * dir;
	struct dirent * ent;
	char * buf = NULL, * start, * end, * chptr, * next, * model, * ptr;
	char * path;

	if (access("/proc/iSeries/veth", R_OK))
	    goto vethdone;

	/* Read from /proc/iSeries/veth */
	dir = opendir("/proc/iSeries/veth");
	while ((ent = readdir(dir))) {
	    if (strncmp("veth", ent->d_name, 4))
		continue;
	    path = malloc(30);
	    snprintf(path, 29, "/proc/iSeries/veth/%s", ent->d_name);
            if ((fd = open(path, O_RDONLY)) >= 0) {
		start = buf = bufFromFd(fd);
		close(fd);
		if (!start)
		    continue;

		end = start + strlen(start);
		while (start && *start && start < end) {
		    /* parse till end of line and store the start of next line. */
		    chptr = start;
		    while (*chptr != '\n') chptr++;
		    *chptr = '\0';
		    next = chptr + 1;
		    
		    /* get rid of anything which is not alpha */
		    while (!(isalpha(*start))) start++;

		    if (!strncmp("Net device name:", start, 16)) {
			start += 16;
			while ((isspace(*start))) start++;

			viodev = vioNewDevice(NULL);
			viodev->device = strdup(start);
			viodev->desc = strdup("IBM Virtual Ethernet");
			viodev->type = CLASS_NETWORK;
			viodev->driver = strdup("veth");
			if (devlist)
			    viodev->next = devlist;
			devlist = (struct device *) viodev;
			break;
		    }

		    start = next;
		    end = start + strlen(start);
		}
		free(buf);
	    }
	}

vethdone:
    } 

    if (probeClass & CLASS_CDROM) {
	char *buf, *ptr, *b;
	int cdnum;
	int modloaded = 0;

	if (!loadModule("viocd"))
	    modloaded = 1;
			
	if (access("/proc/iSeries/viocd", R_OK))
	    goto viocddone;
	fd = open("/proc/iSeries/viocd", O_RDONLY);
	if (fd < 0)
	    goto viocddone;
	b = buf = bufFromFd(fd);
	while (*buf) {
	    ptr = buf;
	    while (*ptr && *ptr != '\n')
		ptr++;
	    if (*ptr) {
		*ptr = '\0';
		ptr++;
	    }
	    if (strncmp("viocd", buf, 5)) {
		buf = ptr;
		continue;
	    }
	    buf +=13;
	    cdnum = atoi(buf);
	    buf = strstr(buf,"type");
	    viodev = vioNewDevice(NULL);
	    viodev->device = malloc(20);
	    snprintf(viodev->device,19,"iseries/vcd%c", cdnum + 'a');
	    viodev->desc = malloc(64);
	    if (buf)
		snprintf(viodev->desc,63,"IBM Virtual CD-ROM %s",buf);
	    else
		snprintf(viodev->desc,63,"IBM Virtual CD-ROM");
	    viodev->type = CLASS_CDROM;
	    viodev->driver = strdup("ignore");
	    if (devlist)
		viodev->next = devlist;
	    devlist = (struct device *) viodev;
	    buf = ptr;
	    continue;
	}
	free(b);
viocddone:
	if (modloaded)
	    removeModule("viocd");
    }
#endif
    return devlist;
}
