/*
 * Copyright (c) 2001-2002 The Trustees of Indiana University.  
 *                         All rights reserved.
 * Copyright (c) 1998-2001 University of Notre Dame. 
 *                         All rights reserved.
 * Copyright (c) 1994-1998 The Ohio State University.  
 *                         All rights reserved.
 * 
 * This file is part of the LAM/MPI software package.  For license
 * information, see the LICENSE file in the top level directory of the
 * LAM/MPI source distribution.
 * 
 *
 *	$Id: lamthreads.c,v 6.3.2.1 2002/10/09 19:49:11 brbarret Exp $
 *
 *	Function:	- LAM interface for threads
 */

#include <lam_config.h>

#include <stdio.h>
#include <mpi.h>
#include <lamthreads.h>


/*
 * Exported varaibles
 */
int lam_thread_level = MPI_THREAD_SINGLE;
lam_thread_t lam_main_thread;
lam_mutex_t lam_mpi_mutex;
lam_mutex_t lam_updown_mutex;


/*
 *	lam_thread_init
 *
 *	Function:	- initialize the LAM thread interface
 */
void
lam_thread_init(int requested, int *pprovided)
{
#if LAM_HAVE_THREADS
/*
 * Set the provided level, and the global supported variable
 */
  if (requested < MPI_THREAD_MULTIPLE)
    lam_thread_level = *pprovided = requested;
  else
    lam_thread_level = *pprovided = MPI_THREAD_SERIALIZED;

/*
 * Initialize the global mutexes
 */
  lam_mutex_init(&lam_mpi_mutex);
  lam_mutex_init(&lam_updown_mutex);

#else /* LAM_HAVE_THREADS */
/*
 * If we don't have threads, set the return code to be MPI_THREAD_SINGLE
 */
  *pprovided = MPI_THREAD_SINGLE;

/*
 * Set the main thread variable -- even if we don't have threads, so
 * that MPI_IS_THREAD_MAIN will return something predicable, at least.
 */
  lam_main_thread = lam_thread_self();
#endif
}


/*
 *	lam_thread_self
 *
 *	Function:	- get thread specific identifier
 *      Returns:        - thread specific identifier
 */
lam_thread_t
lam_thread_self(void)
{
#if LAM_HAVE_SOL_THREADS
  return thr_self();
#elif LAM_HAVE_PTHREADS
  return pthread_self();
#else
  return 0;
#endif
}


/*
 *	lam_thread_compare
 *
 *	Function:	- compare two thread IDs
 *	Returns:	- 1 if they are the same, 0 otherwise
 *			- 1 if threading is not enabled (on assumption 
 *			  that there is only one thread)
 */
int
lam_thread_compare(lam_thread_t a, lam_thread_t b)
{
#if LAM_HAVE_SOL_THREADS
  return (a == b);
#elif LAM_HAVE_PTHREADS
  return !pthread_equal(a, b);
#else
  return 1;
#endif
}


/*
 *	lam_thread_finalize
 *
 *	Function:	- finalize the LAM thread interface
 */
void
lam_thread_finalize(void)
{
#if LAM_HAVE_THREADS
  if (lam_thread_level > MPI_THREAD_SINGLE) {
    lam_mutex_finalize(&lam_updown_mutex);
    lam_mutex_finalize(&lam_mpi_mutex);
  }
#endif
}
