/*
 * LavaLauncher - A simple launcher panel for Wayland
 *
 * Copyright (C) 2020 Leon Henrik Plickat
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

#define _POSIX_C_SOURCE 200809L

#include<stdio.h>
#include<stdlib.h>
#include<stdbool.h>
#include<unistd.h>
#include<string.h>

#include<cairo/cairo.h>
#include<wayland-server.h>
#include<wayland-client.h>
#include<wayland-client-protocol.h>

#include"wlr-layer-shell-unstable-v1-protocol.h"
#include"xdg-output-unstable-v1-protocol.h"
#include"xdg-shell-protocol.h"

#include"lavalauncher.h"
#include"log.h"
#include"output.h"
#include"draw-generics.h"
#include"bar/bar-pattern.h"
#include"bar/bar.h"
#include"bar/indicator.h"
#include"bar/layersurface.h"
#include"bar/render.h"
#include"item/item.h"
#include"types/string-container.h"

/* Positions and dimensions for MODE_DEFAULT. */
static void mode_default_dimensions (struct Lava_bar *bar)
{
	struct Lava_bar_pattern *pattern = bar->pattern;
	struct Lava_output      *output  = bar->output;

	/* Position of item area. */
	if ( pattern->orientation == ORIENTATION_HORIZONTAL ) switch (pattern->alignment)
	{
		case ALIGNMENT_START:
			bar->item_area_x = pattern->border_left + (uint32_t)pattern->margin_left;
			bar->item_area_y = pattern->border_top;
			break;

		case ALIGNMENT_CENTER:
			bar->item_area_x = (output->w / 2) - (bar->item_area_width / 2)
				+ (uint32_t)(pattern->margin_left - pattern->margin_right);
			bar->item_area_y = pattern->border_top;
			break;

		case ALIGNMENT_END:
			bar->item_area_x = output->w - bar->item_area_width
				- pattern->border_right - (uint32_t)pattern->margin_right;
			bar->item_area_y = pattern->border_top;
			break;
	}
	else switch (pattern->alignment)
	{
		case ALIGNMENT_START:
			bar->item_area_x = pattern->border_left;
			bar->item_area_y = pattern->border_top + (uint32_t)pattern->margin_top;
			break;

		case ALIGNMENT_CENTER:
			bar->item_area_x = pattern->border_left;
			bar->item_area_y = (output->h / 2) - (bar->item_area_height / 2)
				+ (uint32_t)(pattern->margin_top - pattern->margin_bottom);
			break;

		case ALIGNMENT_END:
			bar->item_area_x = pattern->border_left;
			bar->item_area_y = output->h - bar->item_area_height
				- pattern->border_bottom - (uint32_t)pattern->margin_bottom;
			break;
	}

	/* Position of bar. */
	bar->bar_x = bar->item_area_x - pattern->border_left;
	bar->bar_y = bar->item_area_y - pattern->border_top;

	/* Size of bar. */
	bar->bar_width  = bar->item_area_width  + pattern->border_left + pattern->border_right;
	bar->bar_height = bar->item_area_height + pattern->border_top  + pattern->border_bottom;

	/* Size of buffer / surface. */
	if ( pattern->orientation == ORIENTATION_HORIZONTAL )
	{
		bar->buffer_width  = bar->output->w;
		bar->buffer_height = pattern->size
			+ pattern->border_top
			+ pattern->border_bottom;
	}
	else
	{
		bar->buffer_width  = pattern->size
			+ pattern->border_left
			+ pattern->border_right;
		bar->buffer_height = bar->output->h;
	}
}

/* Positions and dimensions for MODE_FULL. */
static void mode_full_dimensions (struct Lava_bar *bar)
{
	struct Lava_bar_pattern *pattern = bar->pattern;
	struct Lava_output      *output  = bar->output;

	/* Position of item area. */
	if ( pattern->orientation == ORIENTATION_HORIZONTAL ) switch (pattern->alignment)
	{
		case ALIGNMENT_START:
			bar->item_area_x = pattern->border_left + (uint32_t)pattern->margin_left;
			bar->item_area_y = pattern->border_top;
			break;

		case ALIGNMENT_CENTER:
			bar->item_area_x = (output->w / 2) - (bar->item_area_width / 2)
				+ (uint32_t)(pattern->margin_left - pattern->margin_right);
			bar->item_area_y = pattern->border_top;
			break;

		case ALIGNMENT_END:
			bar->item_area_x = output->w - bar->item_area_width
				- pattern->border_right - (uint32_t)pattern->margin_right;
			bar->item_area_y = pattern->border_top;
			break;
	}
	else switch (pattern->alignment)
	{
		case ALIGNMENT_START:
			bar->item_area_x = pattern->border_left;
			bar->item_area_y = pattern->border_top + (uint32_t)pattern->margin_top;
			break;

		case ALIGNMENT_CENTER:
			bar->item_area_x = pattern->border_left;
			bar->item_area_y = (output->h / 2) - (bar->item_area_height / 2)
				+ (uint32_t)(pattern->margin_top - pattern->margin_bottom);
			break;

		case ALIGNMENT_END:
			bar->item_area_x = pattern->border_left;
			bar->item_area_y = output->h - bar->item_area_height
				- pattern->border_bottom - (uint32_t)pattern->margin_bottom;
			break;
	}

	/* Position and size of bar and size of buffer / surface. */
	if ( pattern->orientation == ORIENTATION_HORIZONTAL )
	{
		bar->bar_x = (uint32_t)pattern->margin_left;
		bar->bar_y = 0;

		bar->bar_width  = output->w - (uint32_t)(pattern->margin_left + pattern->margin_right);
		bar->bar_height = bar->item_area_height + pattern->border_top + pattern->border_bottom;

		bar->buffer_width  = bar->output->w;
		bar->buffer_height = pattern->size
			+ pattern->border_top
			+ pattern->border_bottom;
	}
	else
	{
		bar->bar_x = 0;
		bar->bar_y = (uint32_t)pattern->margin_top;

		bar->bar_width  = bar->item_area_width + pattern->border_left + pattern->border_right;
		bar->bar_height = output->h - (uint32_t)(pattern->margin_top + pattern->margin_bottom);

		bar->buffer_width  = pattern->size
			+ pattern->border_left
			+ pattern->border_right;
		bar->buffer_height = bar->output->h;
	}
}

/* Position and size for MODE_SIMPLE. */
static void mode_simple_dimensions (struct Lava_bar *bar)
{
	struct Lava_bar_pattern *pattern = bar->pattern;

	/* Position of item area. */
	bar->item_area_x = pattern->border_left;
	bar->item_area_y = pattern->border_top;

	/* Position of bar. */
	bar->bar_x = 0;
	bar->bar_y = 0;

	/* Size of bar. */
	bar->bar_width  = bar->item_area_width  + pattern->border_left + pattern->border_right;
	bar->bar_height = bar->item_area_height + pattern->border_top  + pattern->border_bottom;

	/* Size of buffer / surface. */
	bar->buffer_width  = bar->bar_width;
	bar->buffer_height = bar->bar_height;
}

static void update_dimensions (struct Lava_bar *bar)
{
	struct Lava_bar_pattern *pattern = bar->pattern;
	struct Lava_output      *output  = bar->output;

	if ( output->w == 0 || output->h == 0 )
		return;

	/* Size of item area. */
	if ( pattern->orientation == ORIENTATION_HORIZONTAL )
	{
		bar->item_area_width  = get_item_length_sum(pattern);
		bar->item_area_height = pattern->size;
	}
	else
	{
		bar->item_area_width  = pattern->size;
		bar->item_area_height = get_item_length_sum(pattern);
	}

	/* Other dimensions. */
	switch (bar->pattern->mode)
	{
		case MODE_DEFAULT: mode_default_dimensions(bar); break;
		case MODE_FULL:    mode_full_dimensions(bar);    break;
		case MODE_SIMPLE:  mode_simple_dimensions(bar);  break;
	}
}

bool create_bar (struct Lava_bar_pattern *pattern, struct Lava_output *output)
{
	struct Lava_data *data = pattern->data;
	log_message(data, 1, "[bar] Creating bar: global_name=%d\n", output->global_name);

	struct Lava_bar *bar = calloc(1, sizeof(struct Lava_bar));
	if ( bar == NULL )
	{
		log_message(NULL, 0, "ERROR: Could not allocate.\n");
		return false;
	}

	wl_list_insert(&output->bars, &bar->link);
	bar->data          = data;
	bar->pattern       = pattern;
	bar->output        = output;
	bar->bar_surface   = NULL;
	bar->icon_surface  = NULL;
	bar->layer_surface = NULL;
	bar->subsurface    = NULL;
	bar->configured    = false;

	wl_list_init(&bar->indicators);

	/* Main surface for the bar. */
	if ( NULL == (bar->bar_surface = wl_compositor_create_surface(data->compositor)) )
	{
		log_message(NULL, 0, "ERROR: Compositor did not create wl_surface.\n");
		return false;
	}
	if ( NULL == (bar->layer_surface = zwlr_layer_shell_v1_get_layer_surface(
					data->layer_shell, bar->bar_surface,
					output->wl_output, pattern->layer,
					string_container_get_string_or_else(
						pattern->namespace, "LavaLauncher"))) )
	{
		log_message(NULL, 0, "ERROR: Compositor did not create layer_surface.\n");
		return false;
	}

	/* Subsurface for the icons. */
	if ( NULL == (bar->icon_surface = wl_compositor_create_surface(data->compositor)) )
	{
		log_message(NULL, 0, "ERROR: Compositor did not create wl_surface.\n");
		return false;
	}
	if ( NULL == (bar->subsurface = wl_subcompositor_get_subsurface(
					data->subcompositor, bar->icon_surface,
					bar->bar_surface)) )
	{
		log_message(NULL, 0, "ERROR: Compositor did not create wl_subsurface.\n");
		return false;
	}

	update_dimensions(bar);
	configure_layer_surface(bar);
	configure_subsurface(bar);
	zwlr_layer_surface_v1_add_listener(bar->layer_surface,
			&layer_surface_listener, bar);
	wl_surface_commit(bar->icon_surface);
	wl_surface_commit(bar->bar_surface);

	return true;
}

void destroy_bar (struct Lava_bar *bar)
{
	if ( bar == NULL )
		return;

	struct Lava_item_indicator *indicator, *temp;
	wl_list_for_each_safe(indicator, temp, &bar->indicators, link)
		destroy_indicator(indicator);

	if ( bar->layer_surface != NULL )
		zwlr_layer_surface_v1_destroy(bar->layer_surface);
	if ( bar->subsurface != NULL )
		wl_subsurface_destroy(bar->subsurface);
	if ( bar->bar_surface != NULL )
		wl_surface_destroy(bar->bar_surface);
	if ( bar->icon_surface != NULL )
		wl_surface_destroy(bar->icon_surface);

	finish_buffer(&bar->bar_buffers[0]);
	finish_buffer(&bar->bar_buffers[1]);
	finish_buffer(&bar->icon_buffers[0]);
	finish_buffer(&bar->icon_buffers[1]);

	wl_list_remove(&bar->link);
	free(bar);
}

void destroy_all_bars (struct Lava_output *output)
{
	log_message(output->data, 1, "[bar] Destroying bars: global-name=%d\n", output->global_name);
	struct Lava_bar *b1, *b2;
	wl_list_for_each_safe(b1, b2, &output->bars, link)
		destroy_bar(b1);
}

void update_bar (struct Lava_bar *bar)
{
	/* It is possible that this function is called by output events before
	 * the bar has been created. This function will return and abort unless
	 * it is called either when a surface configure event has been received
	 * at least once, it is called by a surface configure event or
	 * it is called during the creation of the surface.
	 */
	if ( bar == NULL || ! bar->configured )
		return;

	update_dimensions(bar);

	configure_subsurface(bar);
	render_icon_frame(bar);
	wl_surface_commit(bar->icon_surface);

	configure_layer_surface(bar);
	render_bar_frame(bar);
	wl_surface_commit(bar->bar_surface);
}

struct Lava_bar *bar_from_surface (struct Lava_data *data, struct wl_surface *surface)
{
	if ( data == NULL || surface == NULL )
		return NULL;
	struct Lava_output *op1, *op2;
	struct Lava_bar *b1, *b2;
	wl_list_for_each_safe(op1, op2, &data->outputs, link)
	{
		wl_list_for_each_safe(b1, b2, &op1->bars, link)
		{
			if ( b1->bar_surface == surface )
				return b1;
		}
	}
	return NULL;
}

