#!/usr/bin/python
# -*- coding: utf-8 -*-
#################################################################################
# LAYMAN ACTIONS
#################################################################################
# File:       action.py
#
#             Handles layman actions.
#
# Copyright:
#             (c) 2005 - 2006 Gunnar Wrobel
#             Distributed under the terms of the GNU General Public License v2
#
# Author(s):
#             Gunnar Wrobel <wrobel@gentoo.org>
#

__version__ = "$Id: action.py 218 2006-07-25 12:31:27Z wrobel $"

#===============================================================================
#
# Dependencies
#
#-------------------------------------------------------------------------------

from   layman.overlay           import Overlays
from   layman.db                import DB, RemoteDB

from   layman.debug             import OUT

#===============================================================================
#
# Class Fetch
#
#-------------------------------------------------------------------------------

class Fetch:
    ''' Fetches the overlay listing.'''

    def __init__(self, config):
        self.db = RemoteDB(config)

    def run(self):
        try:
            self.db.cache()
        except Exception, e:
            OUT.die('Failed to fetch overlay list!\nError was: '
                    + str(e))

#===============================================================================
#
# Class Sync
#
#-------------------------------------------------------------------------------

class Sync:
    ''' Syncs the selected overlays.'''

    def __init__(self, config):

        self.db = DB(config)

        self.rdb = RemoteDB(config)

        self.selection = config['sync']

        if config['sync_all'] or 'ALL' in self.selection:
            self.selection = self.db.overlays.keys()

    def run(self):

        OUT.debug('Updating selected overlays', 6)

        for i in self.selection:
            ordb = self.rdb.select(i)
            odb = self.db.select(i)
            if ordb and odb and ordb.src != odb.src:
                OUT.warn('The source of the overlay "' + i + '" seems to have c'
                         'hanged. You currently sync from "' + odb.src + '" whi'
                         'le the global layman list reports "' + ordb.src + '" '
                         'as correct location. Please consider removing and rea'
                         'dding the overlay!')

            try:
                self.db.sync(i)
                OUT.info('Successfully synchronized overlay "' + i + '".', 1)
            except Exception, e:
                OUT.warn('Failed to sync overlay "' + i + '".\nError was: '
                         + str(e))

#===============================================================================
#
# Class Add
#
#-------------------------------------------------------------------------------

class Add:
    ''' Adds the selected overlays.'''

    def __init__(self, config):

        self.config = config

        self.db = DB(config)

        self.rdb = RemoteDB(config)

        self.selection = config['add']

        if 'ALL' in self.selection:
            self.selection = self.rdb.overlays.keys()

    def run(self):

        OUT.debug('Adding selected overlays', 6)

        for i in self.selection:
            overlay = self.rdb.select(i)

            OUT.debug('Selected overlay', 7)

            if overlay:
                try:
                    self.db.add(overlay)
                    OUT.info('Successfully added overlay "' + i + '".', 1)
                except Exception, error:
                    OUT.warn('Failed to add overlay "' + i + '".\nError was: '
                             + str(error))
            else:
                OUT.warn('Overlay "' + i + '" does not exist!')

#===============================================================================
#
# Class Delete
#
#-------------------------------------------------------------------------------

class Delete:
    ''' Deletes the selected overlays.'''

    def __init__(self, config):

        self.db = DB(config)

        self.selection = config['delete']

        if 'ALL' in self.selection:
            self.selection = self.db.overlays.keys()

    def run(self):

        OUT.debug('Deleting selected overlays', 6)

        for i in self.selection:
            overlay = self.db.select(i)

            OUT.debug('Selected overlay', 7)

            if overlay:
                try:
                    self.db.delete(overlay)
                    OUT.info('Successfully deleted overlay "' + i + '".', 1)
                except Exception, e:
                    OUT.warn('Failed to delete overlay "' + i + '".\nError was: '
                             + str(e))
            else:
                OUT.warn('Overlay "' + i + '" does not exist!')
                
#===============================================================================
#
# Class List
#
#-------------------------------------------------------------------------------

class List:
    ''' Lists the available overlays.'''

    def __init__(self, config):

        OUT.debug('Creating RemoteDB handler', 6)

        self.rdb    = RemoteDB(config)
        self.config = config

    def run(self):
        for i in self.rdb.list(self.config['verbose']):
            if i[1]:
                OUT.info(i[0])
            else:
                OUT.error(i[0])

#===============================================================================
#
# Class ListLocal
#
#-------------------------------------------------------------------------------

class ListLocal:
    ''' Lists the local overlays.'''

    def __init__(self, config):
        self.db = DB(config)
        self.config = config

    def run(self):
        for i in self.db.list(self.config['verbose']):
            if i[1]:
                OUT.info(i[0])
            else:
                OUT.error(i[0])

#===============================================================================
#
# Class Actions
#
#-------------------------------------------------------------------------------

class Actions:

    # Given in order of precedence
    actions = [('fetch',      Fetch),
               ('add',        Add),
               ('sync',       Sync),
               ('sync_all',   Sync),
               ('delete',     Delete),
               ('list',       List),
               ('list_local', ListLocal),]

    def __init__(self, config):
        
        # Make fetching the overlay list a default action
        if not 'nofetch' in config.keys():
            # Actions that implicitely call the fetch operation before
            fetch_actions = ['fetch', 'sync', 'sync_all', 'list']
            for i in fetch_actions:
                if i in config.keys():
                    # Implicitely call fetch, break loop
                    Fetch(config).run()
                    break

        for i in self.actions:

            OUT.debug('Checking for action', 7)

            if i[0] in config.keys():
                i[1](config).run()
