#!/usr/bin/python
# -*- coding: utf-8 -*-
#################################################################################
# LAYMAN OVERLAY DB
#################################################################################
# File:       db.py
#
#             Access to the db of overlays
#
# Copyright:
#             (c) 2005 - 2006 Gunnar Wrobel
#             Distributed under the terms of the GNU General Public License v2
#
# Author(s):
#             Gunnar Wrobel <wrobel@gentoo.org>
#

__version__ = "$Id: db.py 218 2006-07-25 12:31:27Z wrobel $"

#===============================================================================
#
# Dependencies
#
#-------------------------------------------------------------------------------

import os, os.path, urllib, re, md5

from   layman.utils             import path
from   layman.overlay           import Overlays

from   layman.debug             import OUT

#===============================================================================
#
# Class DB
#
#-------------------------------------------------------------------------------

class DB(Overlays):
    ''' Handle the list of local overlays.'''

    def __init__(self, config):

        self.config = config

        self.path = config['local_list']

        Overlays.__init__(self, [config['local_list'], ])

        OUT.debug('DB handler initiated', 6)

    def add(self, overlay):
        '''
        Add an overlay to the local list of overlays.

        >>> write2 = os.tmpnam()
        >>> here = os.path.dirname(os.path.realpath(__file__))
        >>> m = MakeConf(here + '/tests/testfiles/make.conf')
        >>> m.path = write2
        >>> m.write()

        >>> write = os.tmpnam()
        >>> write3 = os.tmpnam()
        >>> here = os.path.dirname(os.path.realpath(__file__))
        >>> config = {'local_list' : 
        ...           here + '/tests/testfiles/global-overlays.xml',
        ...           'make_conf' : write2,
        ...           'storage'   : write3}
        >>> a = DB(config)
        >>> config['local_list'] = write
        >>> b = DB(config)
        >>> OUT.color_off()
        >>> b.add(a.select('wrobel-stable')) #doctest: +ELLIPSIS
        * Running command "/usr/bin/rsync -rlptDvz --progress --delete --delete-after --timeout=180 --exclude="distfiles/*" --exclude="local/*" --exclude="packages/*" rsync://gunnarwrobel.de/wrobel-stable/* /tmp/file.../wrobel-stable"...

        >>> c = Overlays([write, ])
        >>> c.overlays.keys()
        [u'wrobel-stable']
        
        >>> m = MakeConf(write2)
        >>> m.overlays #doctest: +ELLIPSIS
        ['/usr/portage/local/ebuilds/broken', '/usr/portage/local/ebuilds/testing', '/usr/portage/local/ebuilds/stable', '/usr/portage/local/kolab2', '/usr/portage/local/gentoo-webapps-overlay/experimental', '/usr/portage/local/gentoo-webapps-overlay/production-ready', '/tmp/file.../wrobel-stable']


        >>> os.unlink(write)
        >>> os.unlink(write2)
        >>> import shutil
        >>> shutil.rmtree(write3)
        '''

        if overlay.name not in self.overlays.keys():
            result = overlay.add(self.config['storage'])
            if result == 0:
                self.overlays[overlay.name] = overlay
                self.write(self.path)
                make_conf = MakeConf(self.config['make_conf'])
                make_conf.add(path([self.config['storage'], overlay.name]))
            else:
                overlay.delete(self.config['storage'])
                raise Exception('Adding the overlay failed!')
        else:
            raise Exception('Overlay "' + overlay.name + '" already in the loca'
                            'l list!')

    def delete(self, overlay):
        '''
        Add an overlay to the local list of overlays.

        >>> write2 = os.tmpnam()
        >>> here = os.path.dirname(os.path.realpath(__file__))
        >>> m = MakeConf(here + '/tests/testfiles/make.conf')
        >>> m.path = write2
        >>> m.write()

        >>> write = os.tmpnam()
        >>> write3 = os.tmpnam()
        >>> here = os.path.dirname(os.path.realpath(__file__))
        >>> config = {'local_list' : 
        ...           here + '/tests/testfiles/global-overlays.xml',
        ...           'make_conf' : write2,
        ...           'storage'   : write3}
        >>> a = DB(config)
        >>> config['local_list'] = write
        >>> b = DB(config)
        >>> OUT.color_off()
        >>> b.add(a.select('wrobel-stable')) #doctest: +ELLIPSIS
        * Running command "/usr/bin/rsync -rlptDvz --progress --delete --delete-after --timeout=180 --exclude="distfiles/*" --exclude="local/*" --exclude="packages/*" rsync://gunnarwrobel.de/wrobel-stable/* /tmp/file.../wrobel-stable"...
        >>> b.add(a.select('webapps-experimental')) #doctest: +ELLIPSIS
        * Running command "/usr/bin/svn co http://svn.gnqs.org/svn/gentoo-webapps-overlay/experimental/ /tmp/file.../webapps-experimental"...
        >>> c = Overlays([write, ])
        >>> c.overlays.keys()
        [u'webapps-experimental', u'wrobel-stable']
        >>> b.delete(b.select('webapps-experimental'))
        >>> c = Overlays([write, ])
        >>> c.overlays.keys()
        [u'wrobel-stable']
        
        >>> m = MakeConf(write2)
        >>> m.overlays #doctest: +ELLIPSIS
        ['/usr/portage/local/ebuilds/broken', '/usr/portage/local/ebuilds/testing', '/usr/portage/local/ebuilds/stable', '/usr/portage/local/kolab2', '/usr/portage/local/gentoo-webapps-overlay/experimental', '/usr/portage/local/gentoo-webapps-overlay/production-ready', '/tmp/file.../wrobel-stable']

        >>> os.unlink(write)
        >>> os.unlink(write2)
        >>> import shutil
        >>> shutil.rmtree(write3)
        '''
        
        if overlay.name in self.overlays.keys():
            overlay.delete(self.config['storage'])
            del self.overlays[overlay.name]
            self.write(self.path)
            make_conf = MakeConf(self.config['make_conf'])
            make_conf.delete(path([self.config['storage'], overlay.name]))
        else:
            raise Exception('No local overlay named "' + overlay.name + '"!')

    def sync(self, overlay_name):

        overlay = self.select(overlay_name)

        if overlay:
            overlay.sync(self.config['storage'])
        else:
            raise Exception('No such overlay ("' + overlay_name + '")!')

#===============================================================================
#
# Class RemoteDB
#
#-------------------------------------------------------------------------------

class RemoteDB(Overlays):
    '''Handles fetching the remote overlay list.'''

    def __init__(self, config):

        self.config = config

        self.proxies = {}

        if config['proxy']:
            self.proxies['http']  = config['proxy']

        self.urls  = [i.strip() for i in config['overlays'].split('\n') if i]

        paths = [self.path(i) for i in self.urls]

        Overlays.__init__(self, paths)

    def cache(self):
        ''' 
        Copy the remote overlay list to the local cache.

        >>> cache = os.tmpnam()
        >>> config = {'overlays' : 
        ...           'http://dev.gentoo.org/~wrobel/layman/global-overlays-test.xml',
        ...           'cache' : cache}
        >>> a = RemoteDB(config)
        >>> a.cache()
        >>> b = open(a.path(config['overlays']))
        >>> b.readlines()[23]
        '      A collection of ebuilds from Gunnar Wrobel [wrobel@gentoo.org].\\n'
        
        >>> b.close()
        >>> os.unlink(a.path(config['overlays']))

        >>> a.overlays.keys()
        [u'webapps-experimental', u'wrobel-stable']
        '''
        for url in self.urls:

            path = self.path(url)

            try:

                # Fetch the remote list
                olist = urllib.urlopen(url, proxies = self.proxies).read()

                # Create our storage directory if it is missing
                if not os.path.exists(os.path.dirname(path)):
                    try:
                        os.makedirs(os.path.dirname(path))
                    except OSError, e:
                        raise OSError('Failed to create layman storage direct'
                                      + 'ory ' + os.path.dirname(path) + '\nE'
                                      + 'rror was:' + str(e))
                
                # Before we overwrite the old cache, check that the downloaded
                # file is intact and can be parsed
                try:
                    self.read(olist)
                except Exception, e:
                    raise IOError('Failed to parse the overlays list fetched fro' 
                                  'm ' + url + '\nThis means that the downloaded'
                                  ' file is somehow corrupt or there was a probl'
                                  'em with the webserver. Check the content of t'
                                  'he file. Error was:\n' + str(e))
                
                # Ok, now we can overwrite the old cache
                try:
                    out_file = open(path, 'w')
                    out_file.write(olist)
                    out_file.close()

                except Exception, e:
                    raise IOError('Failed to temporarily cache overlays list in ' 
                                  + path + '\nError was:\n' + str(e))


            except IOError, e:
                OUT.warn('Failed to update the overlay list from: ' 
                         + url + '\nError was:\n' + str(e))

            try:
                # Finally parse the contents of the cache
                self.read_file(path)
            except IOError, e:
                OUT.warn('Failed to read a cached version of the overlay list fr' 
                         'om ' + url + '. You probably did not download the file'
                         ' before. The corresponding entry in your layman.cfg fi'
                         'le will be disregarded.\nError was:\n' + str(e))

    def path(self, url):

        base = self.config['cache']

        OUT.debug('Generating cache path.', 6)

        return base + '_' + md5.md5(url).hexdigest() + '.xml'

#===============================================================================
#
# Helper class MakeConf
#
#-------------------------------------------------------------------------------

class MakeConf:
    '''
    Handles modifications to /etc/make.conf

    Check that an add/remove cycle does not modify the make.conf:

    >>> import md5
    >>> write = os.tmpnam()
    >>> here = os.path.dirname(os.path.realpath(__file__))
    >>> a = MakeConf(here + '/tests/testfiles/make.conf')
    >>> o_md5 = str(md5.md5(open(here + '/tests/testfiles/make.conf').read()).hexdigest())
    >>> a.path = write
    >>> a.add('test   ')
    >>> a.delete('/usr/portage/local/gentoo-webapps-overlay/production-ready')
    >>> a.add('/usr/portage/local/gentoo-webapps-overlay/production-ready')
    >>> a.delete('test   ')
    >>> n_md5 = str(md5.md5(open(write).read()).hexdigest())
    >>> o_md5 == n_md5
    True
    >>> #os.unlink(write)
    '''

    my_re = re.compile('PORTDIR_OVERLAY\s*=\s*"([^"]*)"')

    def __init__(self, path):
        self.path = path

        self.read()

    def add(self, overlay_path):
        '''
        Add an overlay to make.conf.

        >>> write = os.tmpnam()
        >>> here = os.path.dirname(os.path.realpath(__file__))
        >>> a = MakeConf(here + '/tests/testfiles/make.conf')
        >>> a.path = write
        >>> a.add('test   ')
        >>> b = MakeConf(write)
        >>> b.overlays
        ['/usr/portage/local/ebuilds/broken', '/usr/portage/local/ebuilds/testing', '/usr/portage/local/ebuilds/stable', '/usr/portage/local/kolab2', '/usr/portage/local/gentoo-webapps-overlay/experimental', '/usr/portage/local/gentoo-webapps-overlay/production-ready', 'test']
        
        >>> os.unlink(write)
        '''
        self.overlays.append(path(overlay_path.strip()))
        self.write()

    def delete(self, overlay_path):
        '''
        Delete an overlay from make.conf.

        >>> write = os.tmpnam()
        >>> here = os.path.dirname(os.path.realpath(__file__))
        >>> a = MakeConf(here + '/tests/testfiles/make.conf')
        >>> a.path = write
        >>> a.delete('/usr/portage/local/ebuilds/broken')
        >>> b = MakeConf(write)
        >>> b.overlays
        ['/usr/portage/local/ebuilds/testing', '/usr/portage/local/ebuilds/stable', '/usr/portage/local/kolab2', '/usr/portage/local/gentoo-webapps-overlay/experimental', '/usr/portage/local/gentoo-webapps-overlay/production-ready']
        
        >>> os.unlink(write)
        '''
        self.overlays = [path(i) 
                         for i in self.overlays
                         if path(i) != path(overlay_path.strip())]
        self.write()

    def read(self):
        '''
        Read the list of registered overlays from /etc/make.conf.

        >>> here = os.path.dirname(os.path.realpath(__file__))
        >>> a = MakeConf(here + '/tests/testfiles/make.conf')
        >>> a.overlays
        ['/usr/portage/local/ebuilds/broken', '/usr/portage/local/ebuilds/testing', '/usr/portage/local/ebuilds/stable', '/usr/portage/local/kolab2', '/usr/portage/local/gentoo-webapps-overlay/experimental', '/usr/portage/local/gentoo-webapps-overlay/production-ready']
        '''
        if os.path.isfile(self.path):
            self.content()

            overlays = self.my_re.search(self.data)
        
            if not overlays:
                raise Exception('Did not find a PORTDIR_OVERLAY entry in file ' +
                                self.path +'! Did you specify the correct file?')

            self.overlays = [i.strip() 
                             for i in overlays.group(1).split('\n') 
                             if i.strip()]
        else:
            self.overlays = []
            self.data     = 'PORTDIR_OVERLAY="\n"\n'

        self.overlays = [i for i in self.overlays 
                         if (i != '$PORTDIR_OVERLAY' 
                             and i != '${PORTDIR_OVERLAY}')]

    def write(self):
        '''
        Write the list of registered overlays to /etc/make.conf.

        >>> write = os.tmpnam()
        >>> here = os.path.dirname(os.path.realpath(__file__))
        >>> a = MakeConf(here + '/tests/testfiles/make.conf')
        >>> a.path = write
        >>> a.write()
        >>> b = MakeConf(write)
        >>> b.overlays
        ['/usr/portage/local/ebuilds/broken', '/usr/portage/local/ebuilds/testing', '/usr/portage/local/ebuilds/stable', '/usr/portage/local/kolab2', '/usr/portage/local/gentoo-webapps-overlay/experimental', '/usr/portage/local/gentoo-webapps-overlay/production-ready']
        
        >>> os.unlink(write)
        '''
        overlays = 'PORTDIR_OVERLAY="$PORTDIR_OVERLAY\n'
        overlays += '\n'.join(self.overlays)
        overlays += '"'

        content = self.my_re.sub(overlays, self.data)
        
        if not self.my_re.search(content):
            raise Exception('Ups, failed to set a proper PORTDIR_OVERLAY entry '
                            'in file ' + self.path +'! Did not overwrite the fi'
                            'le.')

        try:
            make_conf = open(self.path, 'w')
            
            self.content = make_conf.write(content)

            make_conf.close()

        except Exception, e:
            raise Exception('Failed to read "' + self.path + '".\nError was:\n'  
                            + str(e))

    def content(self):
        '''
        Returns the content of the /etc/make.conf file.
        '''
        try:
            make_conf = open(self.path)
            
            self.data = make_conf.read()

            make_conf.close()

        except Exception, e:
            raise Exception('Failed to read "' + self.path + '".\nError was:\n'
                            + str(e))

if __name__ == '__main__':
    import doctest, sys

    # Ignore warnings here. We are just testing
    from warnings     import filterwarnings, resetwarnings
    filterwarnings('ignore')

    doctest.testmod(sys.modules[__name__])

    resetwarnings()
