#!/usr/bin/python
# -*- coding: utf-8 -*-
#################################################################################
# LAYMAN OVERLAY HANDLER
#################################################################################
# File:       overlay.py
#
#             Access to an xml list of overlays
#
# Copyright:
#             (c) 2005 - 2006 Gunnar Wrobel
#             Distributed under the terms of the GNU General Public License v2
#
# Author(s):
#             Gunnar Wrobel <wrobel@gentoo.org>
#

__version__ = "$Id: overlay.py 206 2006-07-18 13:19:56Z wrobel $"

#===============================================================================
#
# Dependencies
#
#-------------------------------------------------------------------------------

import os, os.path, xml.dom.minidom

from   layman.overlays.bzr      import BzrOverlay
from   layman.overlays.darcs    import DarcsOverlay
from   layman.overlays.git      import GitOverlay
from   layman.overlays.svn      import SvnOverlay
from   layman.overlays.rsync    import RsyncOverlay
from   layman.overlays.tar      import TarOverlay

from   layman.debug             import OUT

#===============================================================================
#
# Constants
#
#-------------------------------------------------------------------------------

OVERLAY_TYPES = {'git'  : GitOverlay,
                 'svn'  : SvnOverlay,
                 'rsync': RsyncOverlay,
                 'tar'  : TarOverlay,
                 'bzr'  : BzrOverlay,
                 'darcs': DarcsOverlay}

#===============================================================================
#
# Class Overlays
#
#-------------------------------------------------------------------------------

class Overlays:
    ''' Handle a list of overlays.'''

    def __init__(self, paths):

        self.paths = paths

        self.overlays = {}

        for path in self.paths:
            if os.path.exists(path):
                self.read_file(path)

    def read_file(self, path):

        try:

            document = open(path).read()

        except Exception, e:
            raise IOError('Failed to read the overlay list at ("' 
                          + path + '")!\nError was:\n' + str(e))
        

        self.read(document)

    def read(self, document):
        '''
        Read an xml list of overlays.

        >>> here = os.path.dirname(os.path.realpath(__file__))

        >>> a = Overlays([here + '/tests/testfiles/global-overlays.xml', ])
        >>> a.overlays.keys()
        [u'webapps-experimental', u'wrobel-stable']

        >>> a.overlays['wrobel-stable'].data['&src']
        u'rsync://gunnarwrobel.de/wrobel-stable'
        '''
        try:

            document = xml.dom.minidom.parseString(document)

        except Exception, e:
            raise Exception('Failed to parse the overlay list!\nError was:\n' 
                            + str(e))

        overlays = document.getElementsByTagName('overlay')
        
        for overlay in overlays:

            for index in range(0, overlay.attributes.length):
                attr = overlay.attributes.item(index)
                if attr.name == 'type':
                    if attr.nodeValue in OVERLAY_TYPES.keys():
                        ol = OVERLAY_TYPES[attr.nodeValue](overlay)
                    else:
                        raise Exception('Unknown overlay type "' + 
                                        attr.nodeValue + '"!')
                
            self.overlays[ol.name] = ol

    def write(self, path):
        '''
        Write the list of overlays to a file.

        >>> write = os.tmpnam()
        >>> here = os.path.dirname(os.path.realpath(__file__))

        >>> a = Overlays([here + '/tests/testfiles/global-overlays.xml', ])
        >>> b = Overlays([write,])
        >>> b.overlays['wrobel-stable'] = a.overlays['wrobel-stable']
        >>> b.write(write)
        >>> c = Overlays([write,])
        >>> c.overlays.keys()
        [u'wrobel-stable']
        
        >>> os.unlink(write)
        '''

        imp = xml.dom.minidom.getDOMImplementation()
        
        doc = imp.createDocument('layman', 'overlays', None)

        root =  doc.childNodes[0]

        for name, overlay in self.overlays.items():

            root.appendChild(overlay.to_minidom(doc))

        try:

            out_file = open(path, 'w')

            doc.writexml(out_file, '', '  ', '\n')

        except Exception, e:
            raise Exception('Failed to write to local overlays file: ' 
                            + path + '\nError was:\n' + str(e))

    def select(self, overlay):
        '''
        Select an overlay from the list.

        >>> here = os.path.dirname(os.path.realpath(__file__))
        >>> a = Overlays([here + '/tests/testfiles/global-overlays.xml', ])
        >>> a.select('wrobel-stable').data['&src']
        u'rsync://gunnarwrobel.de/wrobel-stable'
        '''

        if overlay in self.overlays.keys():
            return self.overlays[overlay]

    def list(self, verbose = False):
        '''
        List all overlays.

        >>> here = os.path.dirname(os.path.realpath(__file__))
        >>> a = Overlays([here + '/tests/testfiles/global-overlays.xml', ])
        >>> for i in a.list(True):
        ...     print i[0]
        webapps-experimental
        ~~~~~~~~~~~~~~~~~~~~
        <BLANKLINE>
        Type    : Subversion
        Source  : http://svn.gnqs.org/svn/gentoo-webapps-overlay/experimental
        <BLANKLINE>
        Description:
        <BLANKLINE>
          This is the home of Gentoo's wider collection of ebuilds for
          web-based applications. This is where we collect all the ebuilds
          submitted to Bugzilla by our users, and make them available in
          an easy-to-use overlay for wider testing.
        <BLANKLINE>
        wrobel-stable
        ~~~~~~~~~~~~~
        <BLANKLINE>
        Type    : Rsync
        Source  : rsync://gunnarwrobel.de/wrobel-stable
        <BLANKLINE>
        Description:
        <BLANKLINE>
          A collection of ebuilds from Gunnar Wrobel [wrobel@gentoo.org].
        <BLANKLINE>

        >>> for i in a.list(False):
        ...     print i[0]
        webapps-experimental      [Subversion] (source: http://svn.gnqs.org/svn/ge...)
        wrobel-stable             [Rsync     ] (source: rsync://gunnarwrobel.de/wr...)
        '''
        result = []

        for name, overlay in self.overlays.items():

            if verbose:
                result.append((str(overlay), overlay.is_supported()))
            else:
                result.append((overlay.short_list(), overlay.is_supported()))

        result = sorted(result)

        return result

if __name__ == '__main__':
    import doctest, sys

    # Ignore warnings here. We are just testing
    from warnings     import filterwarnings, resetwarnings
    filterwarnings('ignore')

    doctest.testmod(sys.modules[__name__])

    resetwarnings()
