#!/usr/bin/python
# -*- coding: utf-8 -*-
#################################################################################
# LAYMAN OVERLAY BASE CLASS
#################################################################################
# File:       overlay.py
#
#             Base class for the different overlay types.
#
# Copyright:
#             (c) 2005 - 2006 Gunnar Wrobel
#             Distributed under the terms of the GNU General Public License v2
#
# Author(s):
#             Gunnar Wrobel <wrobel@gentoo.org>
#

__version__ = "$Id: overlay.py 200 2006-07-17 10:12:41Z wrobel $"

#===============================================================================
#
# Dependencies
#
#-------------------------------------------------------------------------------

import re, os, os.path, shutil

from   layman.utils             import node_to_dict, dict_to_node, path

from   layman.debug             import OUT

#===============================================================================
#
# Class Overlay
#
#-------------------------------------------------------------------------------

class Overlay:
    ''' Derive the real implementations from this.'''

    type = 'None'
    
    def __init__(self, xml):

        self.data = node_to_dict(xml)

        if '&name' in self.data.keys():
            self.name = self.data['&name']
        else:
            raise Exception('Overlay is missing a "name" attribute!')
        
        if '&src' in self.data.keys():
            self.src = self.data['&src']
        else:
            raise Exception('Overlay is missing a "src" attribute!')
        
    def to_minidom(self, document):
    
        return dict_to_node(self.data, document, 'overlay')

    def add(self, base):

        dir = path([base, self.name])

        if os.path.exists(dir):
            raise Exception('Directory ' + dir + ' already exists. Will not ove'
                            'rwrite its contents!')

        os.makedirs(dir)

    def sync(self, base):
        pass

    def delete(self, base):
        dir = path([base, self.name])

        if not os.path.exists(dir):
            raise Exception('Directory ' + dir + ' does not exist. Cannot remov'
                            'e the overlay!')

        shutil.rmtree(dir)

    def cmd(self, command):

        OUT.info('Running command "' + command + '"...', 1)

        return os.system(command)

    def __str__(self):
        
        result = ''

        result += self.name + '\n' + (len(self.name) * '~') + '\n'

        result += '\nType    : ' + self.type

        if '&priority' in self.data.keys():
            result += '\nPriority: ' + self.data['&priority']

        result += '\nSource  : ' + self.src + '\n'

        if '<description>1' in self.data.keys():
            description = self.data['<description>1']['@'].strip()
            description = re.compile(' +').sub(' ', description)
            description = re.compile('\n ').sub('\n', description)
            result += '\nDescription:\n'
            result += '\n  '.join(('\n' + description).split('\n'))
            result += '\n'

        if '<link>1' in self.data.keys():
            link = self.data['<link>1']['@'].strip()
            link = re.compile(' +').sub(' ', link)
            link = re.compile('\n ').sub('\n', link)
            result += '\nLink:\n'
            result += '\n  '.join(('\n' + link).split('\n'))
            result += '\n'

        return result

    def short_list(self):

        def pad(string, length):
            if len(string) <= length:
                return string + ' ' * (length - len(string))
            else:
                return string[:length - 3] + '...'
            
        name   = pad(self.name, 25)
        type   = ' [' + pad(self.type, 10) + ']'
        source = ' (source: ' + pad(self.src, 29) + ')'
        
        return name + type + source

    def supported(self, binaries = []):

        if binaries:
            for path, type, package in binaries:
                if not os.path.exists(path):
                    raise Exception('Binary ' + path + ' seems to be missing! '
                                    'Overlay type "' + type + '" not supported. '
                                    'Did you emerge ' + package + '?')
                
        return True

    def is_supported(self):

        try:
            self.supported()
            return True
        except Exception, error:
            return False
