/* -*- Mode: C++; tab-width: 2; indent-tabs-mode: nil; c-basic-offset: 2 -*- */
/* libe-book
 * Version: MPL 2.0 / LGPLv2.1+
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/.
 *
 * Alternatively, the contents of this file may be used under the terms
 * of the GNU Lesser General Public License Version 2.1 or later
 * (LGPLv2.1+), in which case the provisions of the LGPLv2.1+ are
 * applicable instead of those above.
 *
 * For further information visit http://libebook.sourceforge.net
 */

#include <boost/scoped_ptr.hpp>

#include <libwpd-stream/libwpd-stream.h>

#include <libe-book/FB2Document.h>

#include "FB2Parser.h"
#include "FB2ParserContext.h"
#include "FB2Token.h"
#include "WPXZipStream.h"

using boost::scoped_ptr;

namespace libebook
{

namespace
{

class TesterContext : public FB2XMLParserContext
{
  // disable copying
  TesterContext(const TesterContext &other);
  TesterContext &operator=(const TesterContext &other);

public:
  TesterContext(WPXInputStream *input);

private:
  virtual FB2XMLParserContext *leaveContext() const;

  virtual FB2XMLParserContext *element(const EBOOKToken &name, const EBOOKToken &ns);
  virtual void startOfElement();
  virtual void endOfElement();
  virtual void attribute(const EBOOKToken &name, const EBOOKToken *ns, const char *value);
  virtual void endOfAttributes();
  virtual void text(const char *text);

private:
  WPXInputStream *const m_input;
};

TesterContext::TesterContext(WPXInputStream *const input)
  : FB2XMLParserContext()
  , m_input(input)
{
}

FB2XMLParserContext *TesterContext::leaveContext() const
{
  return 0;
}

FB2XMLParserContext *TesterContext::element(const EBOOKToken &name, const EBOOKToken &ns)
{
  if ((FB2Token::NS_FICTIONBOOK == getFB2TokenID(ns)) && (FB2Token::FictionBook == getFB2TokenID(name)))
    m_input->seek(0, WPX_SEEK_END);

  return 0;
}

void TesterContext::startOfElement()
{
}

void TesterContext::endOfElement()
{
}

void TesterContext::attribute(const EBOOKToken &, const EBOOKToken *, const char *)
{
}

void TesterContext::endOfAttributes()
{
}

void TesterContext::text(const char *)
{
}

bool isXml(WPXInputStream *const input)
{
  const unsigned char xml[] = "<?xml ";
  unsigned long readBytes = 0;
  const unsigned char *s = input->read(sizeof(xml), readBytes);

  return (sizeof(xml) == readBytes) && std::equal(xml, xml + sizeof(xml) - 1, s);
}

}

bool FB2Document::isSupported(WPXInputStream *input) try
{
  scoped_ptr<WPXInputStream> zipStream;

  input->seek(0, WPX_SEEK_SET);
  if (WPXZipStream::isZipFile(input))
  {
    zipStream.reset(WPXZipStream::getSingleSubstream(input));
    input = zipStream.get();
    if (!input)
      return false;
  }

  input->seek(0, WPX_SEEK_SET);
  if (!isXml(input))
    return false;

  input->seek(0, WPX_SEEK_SET);
  FB2Parser tester(input);
  TesterContext context(input);
  return tester.parse(&context);
}
catch (...)
{
  return false;
}

bool FB2Document::parse(WPXInputStream *input, WPXDocumentInterface *document) try
{
  if (!FB2Document::isSupported(input))
    return false;

  input->seek(0, WPX_SEEK_SET);

  scoped_ptr<WPXInputStream> zipStream;

  if (WPXZipStream::isZipFile(input))
  {
    zipStream.reset(WPXZipStream::getSingleSubstream(input));
    input = zipStream.get();
    if (!input)
      return false;
  }

  input->seek(0, WPX_SEEK_SET);
  FB2Parser parser(input);
  return parser.parse(document);
}
catch (...)
{
  return false;
}

}

/* vim:set shiftwidth=2 softtabstop=2 expandtab: */
