#include "libfilezilla/tls_layer.hpp"
#include "tls_layer_impl.hpp"
#include "libfilezilla/tls_info.hpp"
#include "tls_system_trust_store_impl.hpp"

#include "libfilezilla/file.hpp"
#include "libfilezilla/iputils.hpp"
#include "libfilezilla/translate.hpp"
#include "libfilezilla/util.hpp"

#include <gnutls/x509.h>

#include <algorithm>
#include <set>

#include <string.h>

#if DEBUG_SOCKETEVENTS
#include <assert.h>

namespace fz {
bool FZ_PRIVATE_SYMBOL has_pending_event(event_handler * handler, socket_event_source const* const source, socket_event_flag event);
}
#endif

static_assert(GNUTLS_VERSION_NUMBER != 0x030604, "Using TLS 1.3 with this version of GnuTLS does not work, update your version of GnuTLS");

namespace fz {

namespace {

#if FZ_USE_GNUTLS_SYSTEM_CIPHERS
char const ciphers[] = "@SYSTEM:-ARCFOUR-128:-3DES-CBC:-MD5:-SIGN-RSA-MD5:-VERS-SSL3.0";
#else
	#if GNUTLS_VERSION_NUMBER >= 0x030600
		char const ciphers[] = "SECURE256:+SECURE128:-ARCFOUR-128:-3DES-CBC:-MD5:+SIGN-ALL:-SIGN-RSA-MD5:+CTYPE-X509:-VERS-SSL3.0";
	#else
		char const ciphers[] = "SECURE256:+SECURE128:-ARCFOUR-128:-3DES-CBC:-MD5:+SIGN-ALL:-SIGN-RSA-MD5:+CTYPE-X509:-CTYPE-OPENPGP:-VERS-SSL3.0";
	#endif
#endif

#define TLSDEBUG 0
#if TLSDEBUG
// This is quite ugly
logger_interface* pLogging;
extern "C" void log_func(int level, char const* msg)
{
	if (!msg || !pLogging) {
		return;
	}
	std::wstring s = to_wstring(msg);
	trim(s);
	pLogging->log(logmsg::debug_debug, L"tls: %d %s", level, s);
}
#endif

void remove_verification_events(event_handler* handler, tls_layer const* const source)
{
	if (!handler) {
		return;
	}

	auto event_filter = [&](event_loop::Events::value_type const& ev) -> bool {
		if (ev.first != handler) {
			return false;
		}
		else if (ev.second->derived_type() == certificate_verification_event::type()) {
			return std::get<0>(static_cast<certificate_verification_event const&>(*ev.second).v_) == source;
		}
		return false;
	};

	handler->event_loop_.filter_events(event_filter);
}

extern "C" ssize_t c_push_function(gnutls_transport_ptr_t ptr, const void* data, size_t len)
{
	return ((tls_layer_impl*)ptr)->push_function(data, len);
}

extern "C" ssize_t c_pull_function(gnutls_transport_ptr_t ptr, void* data, size_t len)
{
	return ((tls_layer_impl*)ptr)->pull_function(data, len);
}
}

class tls_layerCallbacks
{
public:
	static int handshake_hook_func(gnutls_session_t session, unsigned int htype, unsigned int post, unsigned int incoming)
	{
		if (!session) {
			return 0;
		}
		auto* tls = reinterpret_cast<tls_layer_impl*>(gnutls_session_get_ptr(session));
		if (!tls) {
			return 0;
		}

		char const* prefix;
		if (incoming) {
			if (post) {
				prefix = "Processed";
			}
			else {
				prefix = "Received";
			}
		}
		else {
			if (post) {
				prefix = "Sent";
			}
			else {
				prefix = "About to send";
			}
		}

		char const* name = gnutls_handshake_description_get_name(static_cast<gnutls_handshake_description_t>(htype));

		tls->logger_.log(logmsg::debug_debug, L"TLS handshake: %s %s", prefix, name);

		return 0;
	}
};

namespace {
extern "C" int handshake_hook_func(gnutls_session_t session, unsigned int htype, unsigned int post, unsigned int incoming, gnutls_datum_t const*)
{
	return tls_layerCallbacks::handshake_hook_func(session, htype, post, incoming);
}

struct cert_list_holder final
{
	cert_list_holder() = default;
	~cert_list_holder() {
		for (unsigned int i = 0; i < certs_size; ++i) {
			gnutls_x509_crt_deinit(certs[i]);
		}
		gnutls_free(certs);
	}

	cert_list_holder(cert_list_holder const&) = delete;
	cert_list_holder& operator=(cert_list_holder const&) = delete;

	gnutls_x509_crt_t * certs{};
	unsigned int certs_size{};
};

std::string to_string(gnutls_datum_t const& d)
{
	if (d.data && d.size) {
		return std::string(d.data, d.data + d.size);
	}
	return {};
}

struct datum_holder final : gnutls_datum_t
{
	datum_holder() {
		data = nullptr;
		size = 0;
	}

	~datum_holder() {
		gnutls_free(data);
	}

	void clear()
	{
		gnutls_free(data);
		data = nullptr;
		size = 0;
	}

	datum_holder(datum_holder const&) = delete;
	datum_holder& operator=(datum_holder const&) = delete;

	std::string to_string() const {
		return data ? std::string(data, data + size) : std::string();
	}

	std::string_view to_string_view() const {
		return data ? std::string_view(reinterpret_cast<char *>(data), size) : std::string_view();
	}
};

void clone_cert(gnutls_x509_crt_t in, gnutls_x509_crt_t &out)
{
	gnutls_x509_crt_deinit(out);
	out = nullptr;

	if (in) {
		datum_holder der;
		if (gnutls_x509_crt_export2(in, GNUTLS_X509_FMT_DER, &der) == GNUTLS_E_SUCCESS) {
			gnutls_x509_crt_init(&out);
			if (gnutls_x509_crt_import(out, &der, GNUTLS_X509_FMT_DER) != GNUTLS_E_SUCCESS) {
				gnutls_x509_crt_deinit(out);
				out = nullptr;
			}
		}
	}
}
}

tls_layer_impl::tls_layer_impl(tls_layer& layer, tls_system_trust_store* systemTrustStore, logger_interface & logger)
	: tls_layer_(layer)
	, logger_(logger)
	, system_trust_store_(systemTrustStore)
{
}

tls_layer_impl::~tls_layer_impl()
{
	deinit();
}

bool tls_layer_impl::init()
{
	// This function initializes GnuTLS
	if (!initialized_) {
		initialized_ = true;
		int res = gnutls_global_init();
		if (res) {
			log_error(res, L"gnutls_global_init");
			deinit();
			return false;
		}

#if TLSDEBUG
		if (!pLogging) {
			pLogging = &logger_;
			gnutls_global_set_log_function(log_func);
			gnutls_global_set_log_level(99);
		}
#endif
	}

	if (!cert_credentials_) {
		int res = gnutls_certificate_allocate_credentials(&cert_credentials_);
		if (res < 0) {
			log_error(res, L"gnutls_certificate_allocate_credentials");
			deinit();
			return false;
		}
	}

	return true;
}

bool tls_layer_impl::set_certificate_file(native_string const& keyfile, native_string const& certsfile, native_string const& password, bool pem)
{
	// Load the files ourselves instead of calling gnutls_certificate_set_x509_key_file2
	// as it takes narrow strings on MSW, thus being unable to open all files.

	file kf(keyfile, file::reading, file::existing);
	if (!kf.opened()) {
		logger_.log(logmsg::error, fztranslate("Could not open key file."));
		return false;
	}
	int64_t const ks = kf.size();
	if (ks < 0 || ks > 1024 * 1024) {
		logger_.log(logmsg::error, fztranslate("Key file too big."));
		return false;
	}
	std::string k;
	k.resize(ks);
	auto read = kf.read(k.data(), ks);
	if (read != ks) {
		logger_.log(logmsg::error, fztranslate("Could not read key file."));
		return false;
	}


	file cf(certsfile, file::reading, file::existing);
	if (!cf.opened()) {
		logger_.log(logmsg::error, fztranslate("Could not open certificate file."));
		return false;
	}
	int64_t const cs = cf.size();
	if (cs < 0 || cs > 1024 * 1024) {
		logger_.log(logmsg::error, fztranslate("Certificate file too big."));
		return false;
	}
	std::string c;
	c.resize(cs);
	read = cf.read(c.data(), cs);
	if (read != cs) {
		logger_.log(logmsg::error, fztranslate("Could not read certificate file."));
		return false;
	}

	return set_certificate(k, c, password, pem);
}

bool tls_layer_impl::set_certificate(std::string const& key, std::string const& certs, native_string const& password, bool pem)
{
	if (!init()) {
		return false;
	}

	if (!cert_credentials_) {
		return false;
	}

	gnutls_datum_t c;
	c.data = const_cast<unsigned char*>(reinterpret_cast<unsigned char const*>(certs.data()));
	c.size = certs.size();

	gnutls_datum_t k;
	k.data = const_cast<unsigned char*>(reinterpret_cast<unsigned char const*>(key.data()));
	k.size = key.size();

	int res = gnutls_certificate_set_x509_key_mem2(cert_credentials_, &c,
		&k, pem ? GNUTLS_X509_FMT_PEM : GNUTLS_X509_FMT_DER, password.empty() ? nullptr : to_utf8(password).c_str(), 0);
	if (res < 0) {
		log_error(res, L"gnutls_certificate_set_x509_key_mem2");
		deinit();
		return false;
	}

	return true;
}

bool tls_layer_impl::init_session(bool client)
{
	if (!cert_credentials_) {
		deinit();
		return false;
	}

	int res = gnutls_init(&session_, client ? GNUTLS_CLIENT : GNUTLS_SERVER);
	if (res) {
		log_error(res, L"gnutls_init");
		deinit();
		return false;
	}

	if (!client) {
		if (ticket_key_.empty()) {
			datum_holder h;
			res = gnutls_session_ticket_key_generate(&h);
			if (res) {
				log_error(res, L"gnutls_session_ticket_key_generate");
				deinit();
				return false;
			}
			ticket_key_.assign(h.data, h.data + h.size);
		}

		gnutls_datum_t k;
		k.data = ticket_key_.data();
		k.size = ticket_key_.size();
		res = gnutls_session_ticket_enable_server(session_, &k);
		if (res) {
			log_error(res, L"gnutls_session_ticket_enable_server");
			deinit();
			return false;
		}
	}

	// For use in callbacks
	gnutls_session_set_ptr(session_, this);

	// Even though the name gnutls_db_set_cache_expiration
	// implies expiration of some cache, it also governs
	// the actual session lifetime, independend whether the
	// session is cached or not.
	gnutls_db_set_cache_expiration(session_, 100000000);

	res = gnutls_priority_set_direct(session_, ciphers, nullptr);
	if (res) {
		log_error(res, L"gnutls_priority_set_direct");
		deinit();
		return false;
	}

	gnutls_dh_set_prime_bits(session_, 1024);

	gnutls_credentials_set(session_, GNUTLS_CRD_CERTIFICATE, cert_credentials_);

	// Setup transport functions
	gnutls_transport_set_push_function(session_, c_push_function);
	gnutls_transport_set_pull_function(session_, c_pull_function);
	gnutls_transport_set_ptr(session_, (gnutls_transport_ptr_t)this);

	if (!do_set_alpn()) {
		deinit();
		return false;
	}

	return true;
}

void tls_layer_impl::deinit()
{
	deinit_session();

	if (cert_credentials_) {
		gnutls_certificate_free_credentials(cert_credentials_);
		cert_credentials_ = nullptr;
	}

	if (initialized_) {
		initialized_ = false;
		gnutls_global_deinit();
	}

	ticket_key_.clear();

	state_ = socket_state::failed;

#if TLSDEBUG
	if (pLogging == &logger_) {
		pLogging = nullptr;
	}
#endif

	remove_verification_events(verification_handler_, &tls_layer_);
	verification_handler_ = nullptr;
}


void tls_layer_impl::deinit_session()
{
	if (session_) {
		gnutls_deinit(session_);
		session_ = nullptr;
	}
}


void tls_layer_impl::log_error(int code, std::wstring const& function, logmsg::type logLevel)
{
	if (logLevel < logmsg::debug_warning && state_ >= socket_state::shut_down && shutdown_silence_read_errors_) {
		logLevel = logmsg::debug_warning;
	}

	if (code == GNUTLS_E_WARNING_ALERT_RECEIVED || code == GNUTLS_E_FATAL_ALERT_RECEIVED) {
		log_alert(logLevel);
	}
	else if (code == GNUTLS_E_PULL_ERROR) {
		if (function.empty()) {
			logger_.log(logmsg::debug_warning, L"GnuTLS could not read from socket: %s", socket_error_description(socket_error_));
		}
		else {
			logger_.log(logmsg::debug_warning, L"GnuTLS could not read from socket in %s: %s", function, socket_error_description(socket_error_));
		}
	}
	else if (code == GNUTLS_E_PUSH_ERROR) {
		if (function.empty()) {
			logger_.log(logmsg::debug_warning, L"GnuTLS could not write to socket: %s", socket_error_description(socket_error_));
		}
		else {
			logger_.log(logmsg::debug_warning, L"GnuTLS could not write to socket in %s: %s", function, socket_error_description(socket_error_));
		}
	}
	else {
		char const* error = gnutls_strerror(code);
		if (error) {
			if (function.empty()) {
				logger_.log(logLevel, fztranslate("GnuTLS error %d: %s"), code, error);
			}
			else {
				logger_.log(logLevel, fztranslate("GnuTLS error %d in %s: %s"), code, function, error);
			}
		}
		else {
			if (function.empty()) {
				logger_.log(logLevel, fztranslate("GnuTLS error %d"), code);
			}
			else {
				logger_.log(logLevel, fztranslate("GnuTLS error %d in %s"), code, function);
			}
		}
	}
}

void tls_layer_impl::log_alert(logmsg::type logLevel)
{
	gnutls_alert_description_t last_alert = gnutls_alert_get(session_);
	char const* alert = gnutls_alert_get_name(last_alert);
	if (alert) {
		logger_.log(logLevel,
					server_ ? fztranslate("Received TLS alert from the client: %s (%d)") : fztranslate("Received TLS alert from the server: %s (%d)"),
					alert, last_alert);
	}
	else {
		logger_.log(logLevel,
					server_ ? fztranslate("Received unknown TLS alert %d from the client") : fztranslate("Received unknown TLS alert %d from the server"),
					last_alert);
	}
}

ssize_t tls_layer_impl::push_function(void const* data, size_t len)
{
#if TLSDEBUG
	logger_.log(logmsg::debug_debug, L"tls_layer_impl::push_function(%d)", len);
#endif
	if (!can_write_to_socket_) {
		gnutls_transport_set_errno(session_, EAGAIN);
		return -1;
	}

	int error;
	int written = tls_layer_.next_layer_.write(data, static_cast<unsigned int>(len), error);

	if (written < 0) {
		can_write_to_socket_ = false;
		if (error != EAGAIN) {
			socket_error_ = error;
		}
		gnutls_transport_set_errno(session_, error);
#if TLSDEBUG
		logger_.log(logmsg::debug_debug, L"  returning -1 due to %d", error);
#endif
		return -1;
	}

#if TLSDEBUG
	logger_.log(logmsg::debug_debug, L"  returning %d", written);
#endif

	return written;
}

ssize_t tls_layer_impl::pull_function(void* data, size_t len)
{
#if TLSDEBUG
	logger_.log(logmsg::debug_debug, L"tls_layer_impl::pull_function(%d)",  (int)len);
#endif

	if (!can_read_from_socket_) {
		gnutls_transport_set_errno(session_, EAGAIN);
		return -1;
	}

	int error;
	int read = tls_layer_.next_layer_.read(data, static_cast<unsigned int>(len), error);
	if (read < 0) {
		if (error != EAGAIN) {
			socket_error_ = error;
		}
		else {
			can_read_from_socket_ = false;
		}
		gnutls_transport_set_errno(session_, error);
#if TLSDEBUG
		logger_.log(logmsg::debug_debug, L"  returning -1 due to %d", error);
#endif
		return -1;
	}

	if (!read) {
		socket_eof_ = true;
	}

#if TLSDEBUG
	logger_.log(logmsg::debug_debug, L"  returning %d", read);
#endif

	return read;
}

void tls_layer_impl::operator()(event_base const& ev)
{
	dispatch<socket_event, hostaddress_event>(ev, this
		, &tls_layer_impl::on_socket_event
		, &tls_layer_impl::forward_hostaddress_event);
}

void tls_layer_impl::forward_hostaddress_event(socket_event_source* source, std::string const& address)
{
	tls_layer_.forward_hostaddress_event(source, address);
}

void tls_layer_impl::on_socket_event(socket_event_source* s, socket_event_flag t, int error)
{
	if (!session_) {
		return;
	}

	if (t == socket_event_flag::connection_next) {
		tls_layer_.forward_socket_event(s, t, error);
		return;
	}

	if (error) {
		socket_error_ = error;
		deinit();
		tls_layer_.forward_socket_event(s, t, error);
		return;
	}

	switch (t)
	{
	case socket_event_flag::read:
		on_read();
		break;
	case socket_event_flag::write:
		on_send();
		break;
	case socket_event_flag::connection:
		if (hostname_.empty()) {
			set_hostname(tls_layer_.next_layer_.peer_host());
		}
		on_send();
		break;
	default:
		break;
	}
}

void tls_layer_impl::on_read()
{
	logger_.log(logmsg::debug_debug, L"tls_layer_impl::on_read()");

	can_read_from_socket_ = true;

	if (!session_) {
		return;
	}

	if (state_ == socket_state::connecting) {
		continue_handshake();
	}
	else if (state_ == socket_state::connected || state_ == socket_state::shutting_down || state_ == socket_state::shut_down) {
		if (tls_layer_.event_handler_) {
#if DEBUG_SOCKETEVENTS
			assert(!debug_can_read_);
			debug_can_read_ = true;
#endif
			tls_layer_.event_handler_->send_event<socket_event>(&tls_layer_, socket_event_flag::read, 0);
		}
	}
}

void tls_layer_impl::on_send()
{
	logger_.log(logmsg::debug_debug, L"tls_layer_impl::on_send()");

	can_write_to_socket_ = true;

	if (!session_) {
		return;
	}

	if (state_ == socket_state::connecting) {
		continue_handshake();
	}
	else if (state_ == socket_state::shutting_down) {
		int res = continue_write();
		if (res) {
			return;
		}

		res = continue_shutdown();
		if (res != EAGAIN) {
			if (tls_layer_.event_handler_) {
				tls_layer_.event_handler_->send_event<socket_event>(&tls_layer_, socket_event_flag::write, res);
			}
		}
	}
	else if (state_ == socket_state::connected) {
		continue_write();
	}
}

int tls_layer_impl::continue_write()
{
	if (send_buffer_.empty()) {
		return 0;
	}

	do {
		ssize_t res = GNUTLS_E_AGAIN;
		while ((res == GNUTLS_E_INTERRUPTED || res == GNUTLS_E_AGAIN) && can_write_to_socket_) {
			res = gnutls_record_send(session_, send_buffer_.get(), send_buffer_.size());
		}

		if (res == GNUTLS_E_INTERRUPTED || res == GNUTLS_E_AGAIN) {
			return EAGAIN;
		}

		if (res < 0) {
			failure(static_cast<int>(res), true);
			return ECONNABORTED;
		}

		if (static_cast<size_t>(res) > send_buffer_.size()) {
			logger_.log(logmsg::error, L"gnutls_record_send has processed more data than it has buffered");
			failure(0, true);
			return ECONNABORTED;
		}

		send_buffer_.consume(static_cast<size_t>(res));
	}
	while (!send_buffer_.empty());

	if (write_blocked_by_send_buffer_) {
		write_blocked_by_send_buffer_ = false;

		if (state_ == socket_state::connected) {
			if (tls_layer_.event_handler_) {
#if DEBUG_SOCKETEVENTS
				assert(!debug_can_write_);
				debug_can_write_ = true;
#endif
				tls_layer_.event_handler_->send_event<socket_event>(&tls_layer_, socket_event_flag::write, 0);
			}
		}
	}

	return 0;
}

bool tls_layer_impl::resumed_session() const
{
	return gnutls_session_is_resumed(session_) != 0;
}

bool tls_layer_impl::client_handshake(std::vector<uint8_t> const& session_to_resume, native_string const& session_hostname, std::vector<uint8_t> const& required_certificate, event_handler *const verification_handler)
{
	logger_.log(logmsg::debug_verbose, L"tls_layer_impl::client_handshake()");

	if (state_ != socket_state::none) {
		logger_.log(logmsg::debug_warning, L"Called tls_layer_impl::client_handshake on a socket that isn't idle");
		return false;
	}

	if (!init() || !init_session(true)) {
		return false;
	}

	state_ = socket_state::connecting;

	if (!required_certificate.empty()) {
		std::string_view v(reinterpret_cast<char const*>(required_certificate.data()), required_certificate.size());
		size_t i = v.find_first_not_of("-");
		size_t p = v.find("BEGIN ");
		if (i != std::string_view::npos && i >= 4 && i == p) {
			// It's PEM
			gnutls_datum_t in;
			in.data = const_cast<unsigned char*>(reinterpret_cast<unsigned char const*>(required_certificate.data()));
			in.size = required_certificate.size();

			datum_holder der;
			gnutls_pem_base64_decode2(nullptr, &in, &der);

			required_certificate_.assign(der.data, der.data + der.size);
		}
		else {
			// Must be DER
			required_certificate_ = required_certificate;
		}
	}

	verification_handler_ = verification_handler;

	if (!session_to_resume.empty()) {
		int res = gnutls_session_set_data(session_, session_to_resume.data(), session_to_resume.size());
		if (res) {
			logger_.log(logmsg::debug_info, L"gnutls_session_set_data failed: %d. Going to reinitialize session.", res);
			deinit_session();
			if (!init_session(true)) {
				return false;
			}
		}
		else {
			logger_.log(logmsg::debug_info, L"Trying to resume existing TLS session.");
		}
	}

	if (logger_.should_log(logmsg::debug_debug)) {
		gnutls_handshake_set_hook_function(session_, GNUTLS_HANDSHAKE_ANY, GNUTLS_HOOK_BOTH, &handshake_hook_func);
	}

	if (!session_hostname.empty()) {
		set_hostname(session_hostname);
	}
	else if (!hostname_.empty()) {
		set_hostname(hostname_);
	}

	if (tls_layer_.next_layer_.get_state() == socket_state::none || tls_layer_.next_layer_.get_state() == socket_state::connecting) {
		// Wait until the socket gets connected
		return true;
	}
	else if (tls_layer_.next_layer_.get_state() != socket_state::connected) {
		// We're too late
		return false;
	}

	if (hostname_.empty()) {
		set_hostname(tls_layer_.next_layer_.peer_host());
	}
	return continue_handshake() == EAGAIN;
}

bool tls_layer_impl::server_handshake(std::vector<uint8_t> const& session_to_resume, std::string_view const& preamble)
{
	logger_.log(logmsg::debug_verbose, L"tls_layer_impl::server_handshake()");

	if (state_ != socket_state::none) {
		logger_.log(logmsg::debug_warning, L"Called tls_layer_impl::server_handshake on a socket that isn't idle");
		return false;
	}

	server_ = true;

	ticket_key_ = session_to_resume;

	if (!init() || !init_session(false)) {
		return false;
	}

	state_ = socket_state::connecting;

	if (logger_.should_log(logmsg::debug_debug)) {
		gnutls_handshake_set_hook_function(session_, GNUTLS_HANDSHAKE_ANY, GNUTLS_HOOK_BOTH, &handshake_hook_func);
	}

	if (tls_layer_.next_layer_.get_state() == socket_state::none || tls_layer_.next_layer_.get_state() == socket_state::connecting) {
		// Wait until the socket gets connected
		return true;
	}
	else if (tls_layer_.next_layer_.get_state() != socket_state::connected) {
		// We're too late
		return false;
	}

	preamble_.append(preamble);

	return continue_handshake() == EAGAIN;
}

int tls_layer_impl::continue_handshake()
{
	logger_.log(logmsg::debug_verbose, L"tls_layer_impl::continue_handshake()");
	if (!session_ || state_ != socket_state::connecting) {
		return ENOTCONN;
	}

	while (!preamble_.empty()) {
		if (!can_write_to_socket_) {
			return EAGAIN;
		}

		int error{};
		int written = tls_layer_.next_layer_.write(preamble_.get(), static_cast<int>(preamble_.size()), error);
		if (written < 0) {
			can_write_to_socket_ = false;
			if (error != EAGAIN) {
				socket_error_ = error;
				failure(0, true);
			}
			return error;
		}
		preamble_.consume(static_cast<size_t>(written));
	}

	int res = gnutls_handshake(session_);
	while (res == GNUTLS_E_AGAIN || res == GNUTLS_E_INTERRUPTED) {
		if (!(gnutls_record_get_direction(session_) ? can_write_to_socket_ : can_read_from_socket_)) {
			break;
		}
		res = gnutls_handshake(session_);
	}
	if (!res) {
		logger_.log(logmsg::debug_info, L"TLS Handshake successful");
		handshake_successful_ = true;

		if (resumed_session()) {
			logger_.log(logmsg::debug_info, L"TLS Session resumed");
		}

		std::string const protocol = get_protocol();
		std::string const keyExchange = get_key_exchange();
		std::string const cipherName = get_cipher();
		std::string const macName = get_mac();

		logger_.log(logmsg::debug_info, L"Protocol: %s, Key exchange: %s, Cipher: %s, MAC: %s", protocol, keyExchange, cipherName, macName);

		if (is_client()) {
			return verify_certificate();
		}
		else {
			state_ = socket_state::connected;

			if (tls_layer_.event_handler_) {
				tls_layer_.event_handler_->send_event<socket_event>(&tls_layer_, socket_event_flag::connection, 0);
				if (can_read_from_socket_) {
#if DEBUG_SOCKETEVENTS
					assert(!debug_can_read_);
					debug_can_read_ = true;
#endif
					tls_layer_.event_handler_->send_event<socket_event>(&tls_layer_, socket_event_flag::read, 0);
				}
#if DEBUG_SOCKETEVENTS
				assert(!debug_can_write_);
				debug_can_write_ = true;
#endif
			}
		}

		return 0;
	}
	else if (res == GNUTLS_E_AGAIN || res == GNUTLS_E_INTERRUPTED) {
		if (!socket_error_) {
			return EAGAIN;
		}

		// GnuTLS has a writev() emulation that ignores trailing errors if
		// at least some data got sent
		res = GNUTLS_E_PUSH_ERROR;
	}

	failure(res, true);

	return socket_error_ ? socket_error_ : ECONNABORTED;
}

int tls_layer_impl::read(void *buffer, unsigned int len, int& error)
{
	if (state_ == socket_state::connecting) {
		error = EAGAIN;
		return -1;
	}
	else if (state_ != socket_state::connected && state_ != socket_state::shutting_down && state_ != socket_state::shut_down) {
		error = ENOTCONN;
		return -1;
	}

#if DEBUG_SOCKETEVENTS
	assert(debug_can_read_);
	assert(!has_pending_event(tls_layer_.event_handler_, &tls_layer_, socket_event_flag::read));
#endif

	int res = do_call_gnutls_record_recv(buffer, len);
	if (res >= 0) {
		error = 0;
		return res;
	}
	else if (res == GNUTLS_E_INTERRUPTED || res == GNUTLS_E_AGAIN) {
#if DEBUG_SOCKETEVENTS
		debug_can_read_ = false;
#endif
		error = EAGAIN;
	}
	else {
		failure(res, false, L"gnutls_record_recv");
		error = socket_error_ ? socket_error_ : ECONNABORTED;
	}

	return -1;
}

int tls_layer_impl::write(void const* buffer, unsigned int len, int& error)
{
	if (state_ == socket_state::connecting) {
		error = EAGAIN;
		return -1;
	}
	else if (state_ == socket_state::shutting_down || state_ == socket_state::shut_down) {
		error = ESHUTDOWN;
		return -1;
	}
	else if (state_ != socket_state::connected) {
		error = ENOTCONN;
		return -1;
	}

#if DEBUG_SOCKETEVENTS
	assert(debug_can_write_);
	assert(!has_pending_event(tls_layer_.event_handler_, &tls_layer_, socket_event_flag::write));
#endif

	if (!send_buffer_.empty()) {
		write_blocked_by_send_buffer_ = true;
#if DEBUG_SOCKETEVENTS
		debug_can_write_ = false;
#endif
		error = EAGAIN;
		return -1;
	}

	ssize_t res = gnutls_record_send(session_, buffer, len);

	while ((res == GNUTLS_E_INTERRUPTED || res == GNUTLS_E_AGAIN) && can_write_to_socket_) {
		res = gnutls_record_send(session_, nullptr, 0);
	}

	if (res >= 0) {
		error = 0;
		return static_cast<int>(res);
	}

	if (res == GNUTLS_E_INTERRUPTED || res == GNUTLS_E_AGAIN) {
		if (!socket_error_) {
			// Unfortunately we can't return EAGAIN here as GnuTLS has already consumed some data.
			// With our semantics, EAGAIN means nothing has been handed off yet.
			// Thus remember up to gnutls_record_get_max_size bytes from the input.
			unsigned int max = static_cast<unsigned int>(gnutls_record_get_max_size(session_));
			if (len > max) {
				len = max;
			}
			send_buffer_.append(reinterpret_cast<unsigned char const*>(buffer), len);
			return static_cast<int>(len);
		}

		// GnuTLS has a writev() emulation that ignores trailing errors if
		// at least some data got sent
		res = GNUTLS_E_PUSH_ERROR;
	}

	failure(static_cast<int>(res), false, L"gnutls_record_send");
	error = socket_error_ ? socket_error_ : ECONNABORTED;
	return -1;
}

void tls_layer_impl::failure(int code, bool send_close, std::wstring const& function)
{
	logger_.log(logmsg::debug_debug, L"tls_layer_impl::failure(%d)", code);
	if (code) {
		log_error(code, function);
		if (socket_eof_) {
			if (code == GNUTLS_E_UNEXPECTED_PACKET_LENGTH
#ifdef GNUTLS_E_PREMATURE_TERMINATION
				|| code == GNUTLS_E_PREMATURE_TERMINATION
#endif
				)
			{
				if (state_ != socket_state::shut_down || !shutdown_silence_read_errors_) {
					logger_.log(logmsg::status, server_ ? fztranslate("Client did not properly shut down TLS connection") : fztranslate("Server did not properly shut down TLS connection"));
				}
			}
		}
	}

	auto const oldState = state_;

	deinit();

	if (send_close && tls_layer_.event_handler_) {
		int error = socket_error_;
		if (!error) {
			error = ECONNABORTED;
		}
		if (oldState == socket_state::connecting) {
			tls_layer_.event_handler_->send_event<socket_event>(&tls_layer_, socket_event_flag::connection, error);
		}
		else {
			tls_layer_.event_handler_->send_event<socket_event>(&tls_layer_, socket_event_flag::read, error);
		}
	}
}

int tls_layer_impl::shutdown()
{
	logger_.log(logmsg::debug_verbose, L"tls_layer_impl::shutdown()");

	if (state_ == socket_state::shut_down) {
		return 0;
	}
	else if (state_ == socket_state::shutting_down) {
		return EAGAIN;
	}
	else if (state_ != socket_state::connected) {
		return ENOTCONN;
	}

	state_ = socket_state::shutting_down;

	if (!send_buffer_.empty()) {
		logger_.log(logmsg::debug_verbose, L"Postponing shutdown, send_buffer_ not empty");
		return EAGAIN;
	}

	return continue_shutdown();
}

int tls_layer_impl::continue_shutdown()
{
	logger_.log(logmsg::debug_verbose, L"tls_layer_impl::continue_shutdown()");

	if (!sent_closure_alert_) {
		int res = gnutls_bye(session_, GNUTLS_SHUT_WR);
		while ((res == GNUTLS_E_INTERRUPTED || res == GNUTLS_E_AGAIN) && can_write_to_socket_) {
			res = gnutls_bye(session_, GNUTLS_SHUT_WR);
		}
		if (res == GNUTLS_E_INTERRUPTED || res == GNUTLS_E_AGAIN) {
			if (!socket_error_) {
				return EAGAIN;
			}

			// GnuTLS has a writev() emulation that ignores trailing errors if
			// at least some data got sent
			res = GNUTLS_E_PUSH_ERROR;
		}
		if (res) {
			failure(res, false, L"gnutls_bye");
			return socket_error_ ? socket_error_ : ECONNABORTED;
		}
		sent_closure_alert_ = true;
	}

	int res = tls_layer_.next_layer_.shutdown();
	if (res == EAGAIN) {
		return EAGAIN;
	}

	if (!res) {
		state_ = socket_state::shut_down;
	}
	else {
		socket_error_ = res;
		failure(0, false);
	}
	return res;
}

void tls_layer_impl::set_verification_result(bool trusted)
{
	if (state_ != socket_state::connecting && !handshake_successful_) {
		logger_.log(logmsg::debug_warning, L"TrustCurrentCert called at wrong time.");
		return;
	}

	remove_verification_events(verification_handler_, &tls_layer_);
	verification_handler_ = nullptr;

	if (trusted) {
		state_ = socket_state::connected;

		if (tls_layer_.event_handler_) {
			tls_layer_.event_handler_->send_event<socket_event>(&tls_layer_, socket_event_flag::connection, 0);
			if (can_read_from_socket_) {
#if DEBUG_SOCKETEVENTS
				assert(!debug_can_read_);
				debug_can_read_ = true;
#endif
				tls_layer_.event_handler_->send_event<socket_event>(&tls_layer_, socket_event_flag::read, 0);
			}
#if DEBUG_SOCKETEVENTS
			assert(!debug_can_write_);
			debug_can_write_ = true;
#endif
		}

		return;
	}

	logger_.log(logmsg::error, fztranslate("Remote certificate not trusted."));
	failure(0, true);
}

static std::string bin2hex(unsigned char const* in, size_t size)
{
	std::string str;
	str.reserve(size * 3);
	for (size_t i = 0; i < size; ++i) {
		if (i) {
			str += ':';
		}
		str += int_to_hex_char<char>(in[i] >> 4);
		str += int_to_hex_char<char>(in[i] & 0xf);
	}

	return str;
}


bool tls_layer_impl::extract_cert(gnutls_x509_crt_t const& cert, x509_certificate& out, bool last)
{
	datetime expiration_time(gnutls_x509_crt_get_expiration_time(cert), datetime::seconds);
	datetime activation_time(gnutls_x509_crt_get_activation_time(cert), datetime::seconds);

	if (!activation_time || !expiration_time || expiration_time < activation_time) {
		logger_.log(logmsg::error, fztranslate("Could not extract validity period of certificate"));
		return false;
	}

	// Get the serial number of the certificate
	unsigned char buffer[40];
	size_t size = sizeof(buffer);
	int res = gnutls_x509_crt_get_serial(cert, buffer, &size);
	if (res != 0) {
		size = 0;
	}

	auto serial = bin2hex(buffer, size);

	unsigned int pk_bits;
	int pkAlgo = gnutls_x509_crt_get_pk_algorithm(cert, &pk_bits);
	std::string pk_algo_name;
	if (pkAlgo >= 0) {
		char const* pAlgo = gnutls_pk_algorithm_get_name((gnutls_pk_algorithm_t)pkAlgo);
		if (pAlgo) {
			pk_algo_name = pAlgo;
		}
	}

	int signAlgo = gnutls_x509_crt_get_signature_algorithm(cert);
	std::string signAlgoName;
	if (signAlgo >= 0) {
		char const* pAlgo = gnutls_sign_algorithm_get_name((gnutls_sign_algorithm_t)signAlgo);
		if (pAlgo) {
			signAlgoName = pAlgo;
		}
	}

	std::string subject, issuer;

	datum_holder raw_subject;
	if (!gnutls_x509_crt_get_dn3(cert, &raw_subject, 0)) {
		subject = raw_subject.to_string_view();
	}
	else {
		log_error(res, L"gnutls_x509_crt_get_dn3");
	}
	if (subject.empty()) {
		logger_.log(logmsg::error, fztranslate("Could not get distinguished name of certificate subject, gnutls_x509_get_dn failed"));
		return false;
	}

	std::vector<x509_certificate::subject_name> alt_subject_names = get_cert_subject_alt_names(cert);

	datum_holder raw_issuer;
	if (!gnutls_x509_crt_get_issuer_dn3(cert, &raw_issuer, 0)) {
		issuer = raw_issuer.to_string_view();
	}
	else {
		log_error(res, L"gnutls_x509_crt_get_issuer_dn3");
	}
	if (issuer.empty() ) {
		logger_.log(logmsg::error, fztranslate("Could not get distinguished name of certificate issuer, gnutls_x509_get_issuer_dn failed"));
		return false;
	}

	std::string fingerprint_sha256;
	std::string fingerprint_sha1;

	unsigned char digest[100];
	size = sizeof(digest) - 1;
	if (!gnutls_x509_crt_get_fingerprint(cert, GNUTLS_DIG_SHA256, digest, &size)) {
		digest[size] = 0;
		fingerprint_sha256 = bin2hex(digest, size);
	}
	size = sizeof(digest) - 1;
	if (!gnutls_x509_crt_get_fingerprint(cert, GNUTLS_DIG_SHA1, digest, &size)) {
		digest[size] = 0;
		fingerprint_sha1 = bin2hex(digest, size);
	}

	datum_holder der;
	if (gnutls_x509_crt_export2(cert, GNUTLS_X509_FMT_DER, &der) != GNUTLS_E_SUCCESS || !der.data || !der.size) {
		logger_.log(logmsg::error, L"gnutls_x509_crt_get_issuer_dn");
		return false;
	}
	std::vector<uint8_t> data(der.data, der.data + der.size);

	out = x509_certificate(
		std::move(data),
		activation_time, expiration_time,
		serial,
		pk_algo_name, pk_bits,
		signAlgoName,
		fingerprint_sha256,
		fingerprint_sha1,
		issuer,
		subject,
		std::move(alt_subject_names),
		last ? gnutls_x509_crt_check_issuer(cert, cert) : false);

	return true;
}


std::vector<x509_certificate::subject_name> tls_layer_impl::get_cert_subject_alt_names(gnutls_x509_crt_t cert)
{
	std::vector<x509_certificate::subject_name> ret;

	char san[4096];
	for (unsigned int i = 0; i < 10000; ++i) { // I assume this is a sane limit
		size_t san_size = sizeof(san) - 1;
		int const type_or_error = gnutls_x509_crt_get_subject_alt_name(cert, i, san, &san_size, nullptr);
		if (type_or_error == GNUTLS_E_SHORT_MEMORY_BUFFER) {
			continue;
		}
		else if (type_or_error < 0) {
			break;
		}

		if (type_or_error == GNUTLS_SAN_DNSNAME || type_or_error == GNUTLS_SAN_RFC822NAME) {
			std::string dns = san;
			if (!dns.empty()) {
				ret.emplace_back(x509_certificate::subject_name{std::move(dns), type_or_error == GNUTLS_SAN_DNSNAME});
			}
		}
		else if (type_or_error == GNUTLS_SAN_IPADDRESS) {
			std::string ip = socket::address_to_string(san, static_cast<int>(san_size));
			if (!ip.empty()) {
				ret.emplace_back(x509_certificate::subject_name{std::move(ip), false});
			}
		}
	}
	return ret;
}

bool tls_layer_impl::certificate_is_blacklisted(cert_list_holder const& certs)
{
	for (size_t i = 0; i < certs.certs_size; ++i) {
		if (certificate_is_blacklisted(certs.certs[i])) {
			return true;
		}
	}
	return false;
}

bool tls_layer_impl::certificate_is_blacklisted(gnutls_x509_crt_t const& cert)
{
	static std::set<std::string, std::less<>> const bad_authority_key_ids = {
		std::string("\xF4\x94\xBF\xDE\x50\xB6\xDB\x6B\x24\x3D\x9E\xF7\xBE\x3A\xAE\x36\xD7\xFB\x0E\x05", 20) // Nation-wide MITM in Kazakhstan
	};

	char buf[256];
	unsigned int critical{};
	size_t size = sizeof(buf);
	int res = gnutls_x509_crt_get_authority_key_id(cert, buf, &size, &critical);
	if (!res) {
		auto it = bad_authority_key_ids.find(std::string_view(buf, size));
		if (it != bad_authority_key_ids.cend()) {
			return true;
		}
	}

	return false;
}

int tls_layer_impl::get_algorithm_warnings() const
{
	int algorithmWarnings{};

	switch (gnutls_protocol_get_version(session_))
	{
		case GNUTLS_SSL3:
		case GNUTLS_VERSION_UNKNOWN:
			algorithmWarnings |= tls_session_info::tlsver;
			break;
		default:
			break;
	}

	switch (gnutls_cipher_get(session_)) {
		case GNUTLS_CIPHER_UNKNOWN:
		case GNUTLS_CIPHER_NULL:
		case GNUTLS_CIPHER_ARCFOUR_128:
		case GNUTLS_CIPHER_3DES_CBC:
		case GNUTLS_CIPHER_ARCFOUR_40:
		case GNUTLS_CIPHER_RC2_40_CBC:
		case GNUTLS_CIPHER_DES_CBC:
			algorithmWarnings |= tls_session_info::cipher;
			break;
		default:
			break;
	}

	switch (gnutls_mac_get(session_)) {
		case GNUTLS_MAC_UNKNOWN:
		case GNUTLS_MAC_NULL:
		case GNUTLS_MAC_MD5:
		case GNUTLS_MAC_MD2:
		case GNUTLS_MAC_UMAC_96:
			algorithmWarnings |= tls_session_info::mac;
			break;
		default:
			break;
	}

	switch (gnutls_kx_get(session_)) {
		case GNUTLS_KX_UNKNOWN:
		case GNUTLS_KX_ANON_DH:
		case GNUTLS_KX_RSA_EXPORT:
		case GNUTLS_KX_ANON_ECDH:
			algorithmWarnings |= tls_session_info::kex;
		default:
			break;
	}

	return algorithmWarnings;
}


bool tls_layer_impl::get_sorted_peer_certificates(gnutls_x509_crt_t *& certs, unsigned int & certs_size)
{
	certs = nullptr;
	certs_size = 0;

	// First get unsorted list of peer certificates in DER
	unsigned int cert_list_size;
	const gnutls_datum_t* cert_list = gnutls_certificate_get_peers(session_, &cert_list_size);
	if (!cert_list || !cert_list_size) {
		logger_.log(logmsg::error, fztranslate("gnutls_certificate_get_peers returned no certificates"));
		return false;
	}

	// Convert them all to PEM
	gnutls_datum_t *pem_cert_list = new gnutls_datum_t[cert_list_size];
	for (unsigned i = 0; i < cert_list_size; ++i) {
		if (gnutls_pem_base64_encode2("CERTIFICATE", cert_list + i, pem_cert_list + i) != 0) {
			for (unsigned int j = 0; j < i; ++j) {
				gnutls_free(pem_cert_list[j].data);
			}
			delete [] pem_cert_list;
			logger_.log(logmsg::error, fztranslate("gnutls_pem_base64_encode2 failed"));
			return false;
		}
	}

	// Concatenate them
	gnutls_datum_t concated_certs{};
	for (unsigned i = 0; i < cert_list_size; ++i) {
		concated_certs.size += pem_cert_list[i].size;
	}
	concated_certs.data = new unsigned char[concated_certs.size];
	concated_certs.size = 0;
	for (unsigned i = 0; i < cert_list_size; ++i) {
		memcpy(concated_certs.data + concated_certs.size, pem_cert_list[i].data, pem_cert_list[i].size);
		concated_certs.size += pem_cert_list[i].size;
	}

	for (unsigned i = 0; i < cert_list_size; ++i) {
		gnutls_free(pem_cert_list[i].data);
	}
	delete[] pem_cert_list;

	// And now import the certificates
	int res = gnutls_x509_crt_list_import2(&certs, &certs_size, &concated_certs, GNUTLS_X509_FMT_PEM, GNUTLS_X509_CRT_LIST_FAIL_IF_UNSORTED);
	if (res == GNUTLS_E_CERTIFICATE_LIST_UNSORTED) {
		logger_.log(logmsg::error, fztranslate("Server sent unsorted certificate chain in violation of the TLS specifications"));
		res = gnutls_x509_crt_list_import2(&certs, &certs_size, &concated_certs, GNUTLS_X509_FMT_PEM, GNUTLS_X509_CRT_LIST_SORT);
	}

	delete[] concated_certs.data;

	if (res != GNUTLS_E_SUCCESS) {
		certs = nullptr;
		certs_size = 0;
		logger_.log(logmsg::error, fztranslate("Could not sort peer certificates"));
		return false;
	}

	return true;
}

void tls_layer_impl::log_verification_error(int status)
{
	gnutls_datum_t buffer{};
	gnutls_certificate_verification_status_print(status, GNUTLS_CRT_X509, &buffer, 0);
	if (buffer.data) {
		logger_.log(logmsg::debug_warning, L"Gnutls Verification status: %s", buffer.data);
		gnutls_free(buffer.data);
	}

	if (status & GNUTLS_CERT_REVOKED) {
		logger_.log(logmsg::error, fztranslate("Beware! Certificate has been revoked"));

		// The remaining errors are no longer of interest
		return;
	}
	if (status & GNUTLS_CERT_SIGNATURE_FAILURE) {
		logger_.log(logmsg::error, fztranslate("Certificate signature verification failed"));
		status &= ~GNUTLS_CERT_SIGNATURE_FAILURE;
	}
	if (status & GNUTLS_CERT_INSECURE_ALGORITHM) {
		logger_.log(logmsg::error, fztranslate("A certificate in the chain was signed using an insecure algorithm"));
		status &= ~GNUTLS_CERT_INSECURE_ALGORITHM;
	}
	if (status & GNUTLS_CERT_SIGNER_NOT_CA) {
		logger_.log(logmsg::error, fztranslate("An issuer in the certificate chain is not a certificate authority"));
		status &= ~GNUTLS_CERT_SIGNER_NOT_CA;
	}
	if (status & GNUTLS_CERT_UNEXPECTED_OWNER) {
		logger_.log(logmsg::error, fztranslate("The server's hostname does not match the certificate's hostname"));
		status &= ~GNUTLS_CERT_UNEXPECTED_OWNER;
	}
#ifdef GNUTLS_CERT_MISSING_OCSP_STATUS
	if (status & GNUTLS_CERT_MISSING_OCSP_STATUS) {
		logger_.log(logmsg::error, fztranslate("The certificate requires the server to include an OCSP status in its response, but the OCSP status is missing."));
		status &= ~GNUTLS_CERT_MISSING_OCSP_STATUS;
	}
#endif
	if (status) {
		if (status == GNUTLS_CERT_INVALID) {
			logger_.log(logmsg::error, fztranslate("Received certificate chain could not be verified."));
		}
		else {
			logger_.log(logmsg::error, fztranslate("Received certificate chain could not be verified. Verification status is %d."), status);
		}
	}

}

int tls_layer_impl::verify_certificate()
{
	logger_.log(logmsg::debug_verbose, L"tls_layer_impl::verify_certificate()");

	if (state_ != socket_state::connecting) {
		logger_.log(logmsg::debug_warning, L"verify_certificate called at wrong time");
		return ENOTCONN;
	}

	if (gnutls_certificate_type_get(session_) != GNUTLS_CRT_X509) {
		logger_.log(logmsg::error, fztranslate("Unsupported certificate type"));
		failure(0, true);
		return EOPNOTSUPP;
	}

	cert_list_holder certs;
	if (!get_sorted_peer_certificates(certs.certs, certs.certs_size)) {
		failure(0, true);
		return EINVAL;
	}

	if (certificate_is_blacklisted(certs)) {
		logger_.log(logmsg::error, fztranslate("Man-in-the-Middle attack detected, aborting connection."));
		failure(0, true);
		return EINVAL;
	}

	if (!required_certificate_.empty()) {
		datum_holder cert_der{};
		int res = gnutls_x509_crt_export2(certs.certs[0], GNUTLS_X509_FMT_DER, &cert_der);
		if (res != GNUTLS_E_SUCCESS) {
			failure(res, true, L"gnutls_x509_crt_export2");
			return ECONNABORTED;
		}

		if (required_certificate_.size() != cert_der.size ||
			memcmp(required_certificate_.data(), cert_der.data, cert_der.size))
		{
			logger_.log(logmsg::error, fztranslate("Certificate of connection does not match expected certificate."));
			failure(0, true);
			return EINVAL;
		}

		set_verification_result(true);

		if (state_ != socket_state::connected && state_ != socket_state::shutting_down && state_ != socket_state::shut_down) {
			return ECONNABORTED;
		}
		return 0;
	}

	bool const uses_hostname = !hostname_.empty() && get_address_type(hostname_) == address_type::unknown;

	bool systemTrust = false;
	bool hostnameMismatch = false;

	// Our trust-model is user-guided TOFU on the host's certificate.
	//
	// First we verify it against the system trust store.
	//
	// If that fails, we validate the certificate chain sent by the server
	// allowing three impairments:
	// - Hostname mismatch
	// - Out of validity
	// - Signer not found
	//
	// In any case, actual trust decision is done later by the user.


	// First, check system trust
	if (uses_hostname && system_trust_store_) {

		auto lease = system_trust_store_->impl_->lease();
		auto cred = std::get<0>(lease);
		if (cred) {
			gnutls_credentials_set(session_, GNUTLS_CRD_CERTIFICATE, cred);
			unsigned int status = 0;
			int verifyResult = gnutls_certificate_verify_peers3(session_, to_utf8(hostname_).c_str(), &status);
			gnutls_credentials_set(session_, GNUTLS_CRD_CERTIFICATE, cert_credentials_);
			std::get<1>(lease).unlock();

			if (verifyResult < 0) {
				logger_.log(logmsg::debug_warning, L"gnutls_certificate_verify_peers2 returned %d with status %u", verifyResult, status);
				logger_.log(logmsg::error, fztranslate("Failed to verify peer certificate"));
				failure(0, true);
				return EINVAL;
			}

			if (!status) {
				systemTrust = true;
			}
		}
		else {
			std::get<1>(lease).unlock();
			logger_.log(logmsg::debug_warning, L"System trust store could not be loaded");
		}
	}

	if (!verification_handler_) {
		set_verification_result(systemTrust);
		return systemTrust ? 0 : ECONNABORTED;
	}
	else {
		if (!systemTrust) {
			// System trust store cannot verify this certificate. Allow three impairments:
			//
			// 1. For now, add the highest certificate from the chain to trust list. Otherwise
			// gnutls_certificate_verify_peers2 always stops with GNUTLS_CERT_SIGNER_NOT_FOUND
			// at the highest certificate in the chain.
			gnutls_x509_crt_t root{};
			clone_cert(certs.certs[certs.certs_size - 1], root);
			if (!root) {
				logger_.log(logmsg::error, fztranslate("Could not copy certificate"));
				failure(0, true);
				return ECONNABORTED;
			}

			gnutls_x509_trust_list_t tlist;
			gnutls_certificate_get_trust_list(cert_credentials_, &tlist);
			if (gnutls_x509_trust_list_add_cas(tlist, &root, 1, 0) != 1) {
				logger_.log(logmsg::error, fztranslate("Could not add certificate to temporary trust list"));
				failure(0, true);
				return ECONNABORTED;
			}

			// 2. Also disable time checks. We allow expired/not yet valid certificates, though only
			// after explicit user confirmation.
			gnutls_certificate_set_verify_flags(cert_credentials_, gnutls_certificate_get_verify_flags(cert_credentials_) | GNUTLS_VERIFY_DISABLE_TIME_CHECKS | GNUTLS_VERIFY_DISABLE_TRUSTED_TIME_CHECKS);

			unsigned int status = 0;
			int verifyResult = gnutls_certificate_verify_peers2(session_, &status);

			if (verifyResult < 0) {
				logger_.log(logmsg::debug_warning, L"gnutls_certificate_verify_peers2 returned %d with status %u", verifyResult, status);
				logger_.log(logmsg::error, fztranslate("Failed to verify peer certificate"));
				failure(0, true);
				return EINVAL;
			}

			if (status != 0) {
				log_verification_error(status);

				failure(0, true);
				return EINVAL;
			}

			// 3. Hostname mismatch
			if (uses_hostname) {
				if (!gnutls_x509_crt_check_hostname(certs.certs[0], to_utf8(hostname_).c_str())) {
					hostnameMismatch = true;
					logger_.log(logmsg::debug_warning, L"Hostname does not match certificate SANs");
				}
			}
		}

		logger_.log(logmsg::status, fztranslate("Verifying certificate..."));

		std::vector<x509_certificate> certificates;
		certificates.reserve(certs.certs_size);
		for (unsigned int i = 0; i < certs.certs_size; ++i) {
			x509_certificate cert;
			if (extract_cert(certs.certs[i], cert, i + 1 == certs.certs_size)) {
				certificates.push_back(cert);
			}
			else {
				failure(0, true);
				return ECONNABORTED;
			}
		}

		// Lengthen incomplete chains to the root using the trust store.
		if (!certificates.empty() && !certificates.back().self_signed() && system_trust_store_) {
			auto lease = system_trust_store_->impl_->lease();
			auto cred = std::get<0>(lease);
			if (cred) {
				gnutls_x509_crt_t cert = certs.certs[certs.certs_size - 1];
				while (!certificates.back().self_signed()) {
					gnutls_x509_crt_t issuer{};
					if (gnutls_certificate_get_issuer(cred, cert, &issuer, 0) || !issuer) {
						break;
					}

					// Why is this cert even in the trust store? Antivirus MITM?
					if (certificate_is_blacklisted(issuer)) {
						logger_.log(logmsg::error, fztranslate("Man-in-the-Middle attack detected, aborting connection."));
						failure(0, true);
						return EINVAL;
					}

					x509_certificate out;
					if (!extract_cert(issuer, out, true)) {
						failure(0, true);
						return ECONNABORTED;
					}
					certificates.push_back(out);
					cert = issuer;
				}
			}
		}

		int const algorithmWarnings = get_algorithm_warnings();

		int error;
		auto port = tls_layer_.peer_port(error);
		if (port == -1) {
			socket_error_ = error;
			failure(0, true);
			return ECONNABORTED;
		}

		tls_session_info session_info(
			to_utf8(to_wstring(hostname_)),
			port,
			get_protocol(),
			get_key_exchange(),
			get_cipher(),
			get_mac(),
			algorithmWarnings,
			std::move(certificates),
			systemTrust,
			hostnameMismatch
		);

		verification_handler_->send_event<certificate_verification_event>(&tls_layer_, std::move(session_info));

		return EAGAIN;
	}
}

std::string tls_layer_impl::get_protocol() const
{
	std::string ret;

	char const* s = gnutls_protocol_get_name(gnutls_protocol_get_version(session_));
	if (s && *s) {
		ret = s;
	}

	if (ret.empty()) {
		ret = to_utf8(fztranslate("unknown"));
	}

	return ret;
}

std::string tls_layer_impl::get_key_exchange() const
{
	std::string ret;

	char const* s{};
	gnutls_kx_algorithm_t alg = gnutls_kx_get(session_);
	bool const dh = (alg == GNUTLS_KX_DHE_RSA || alg == GNUTLS_KX_DHE_DSS);
	bool const ecdh = (alg == GNUTLS_KX_ECDHE_RSA || alg == GNUTLS_KX_ECDHE_ECDSA);
	if (dh || ecdh) {
		char const* const signature_name = gnutls_sign_get_name(static_cast<gnutls_sign_algorithm_t>(gnutls_sign_algorithm_get(session_)));
		ret = (ecdh ? "ECDHE" : "DHE");
#if GNUTLS_VERSION_NUMBER >= 0x030600
		s = gnutls_group_get_name(gnutls_group_get(session_));
		if (s && *s) {
			ret += "-";
			ret += s;
		}
#endif
		if (signature_name && *signature_name) {
			ret += "-";
			ret += signature_name;
		}
	}
	else {
		s = gnutls_kx_get_name(alg);
		if (s && *s) {
			ret = s;
		}
	}


	if (ret.empty()) {
		ret = to_utf8(fztranslate("unknown"));
	}

	return ret;
}

std::string tls_layer_impl::get_cipher() const
{
	std::string ret;

	char const* cipher = gnutls_cipher_get_name(gnutls_cipher_get(session_));
	if (cipher && *cipher) {
		ret = cipher;
	}

	if (ret.empty()) {
		ret = to_utf8(fztranslate("unknown"));
	}

	return ret;
}

std::string tls_layer_impl::get_mac() const
{
	std::string ret;

	char const* mac = gnutls_mac_get_name(gnutls_mac_get(session_));
	if (mac && *mac) {
		ret = mac;
	}

	if (ret.empty()) {
		ret = to_utf8(fztranslate("unknown"));
	}

	return ret;
}

std::string tls_layer_impl::list_tls_ciphers(std::string const& priority)
{
	auto list = sprintf("Ciphers for %s:\n", priority.empty() ? ciphers : priority);

	gnutls_priority_t pcache;
	char const* err = nullptr;
	int ret = gnutls_priority_init(&pcache, priority.empty() ? ciphers : priority.c_str(), &err);
	if (ret < 0) {
		list += sprintf("gnutls_priority_init failed with code %d: %s", ret, err ? err : "Unknown error");
		return list;
	}
	else {
		for (unsigned int i = 0; ; ++i) {
			unsigned int idx;
			ret = gnutls_priority_get_cipher_suite_index(pcache, i, &idx);
			if (ret == GNUTLS_E_REQUESTED_DATA_NOT_AVAILABLE) {
				break;
			}
			if (ret == GNUTLS_E_UNKNOWN_CIPHER_SUITE) {
				continue;
			}

			gnutls_protocol_t version;
			unsigned char id[2];
			char const* name = gnutls_cipher_suite_info(idx, id, nullptr, nullptr, nullptr, &version);

			if (name != nullptr) {
				list += sprintf(
					"%-50s    0x%02x, 0x%02x    %s\n",
					name,
					(unsigned char)id[0],
					(unsigned char)id[1],
					gnutls_protocol_get_name(version));
			}
		}
	}

	return list;
}

int tls_layer_impl::do_call_gnutls_record_recv(void* data, size_t len)
{
	ssize_t res = gnutls_record_recv(session_, data, len);
	while ((res == GNUTLS_E_AGAIN || res == GNUTLS_E_INTERRUPTED) && can_read_from_socket_ && !gnutls_record_get_direction(session_)) {
		// Spurious EAGAIN. Can happen if GnuTLS gets a partial
		// record and the socket got closed.
		// The unexpected close is being ignored in this case, unless
		// gnutls_record_recv is being called again.
		// Manually call gnutls_record_recv as in case of eof on the socket,
		// we are not getting another receive event.
		logger_.log(logmsg::debug_verbose, L"gnutls_record_recv returned spurious EAGAIN");
		res = gnutls_record_recv(session_, data, len);
	}

	if ((res == GNUTLS_E_AGAIN || res == GNUTLS_E_INTERRUPTED) && socket_error_) {
		res = GNUTLS_E_PULL_ERROR;
	}

	return static_cast<int>(res);
}

std::string tls_layer_impl::get_gnutls_version()
{
	char const* v = gnutls_check_version(nullptr);
	if (!v || !*v) {
		return "unknown";
	}

	return v;
}

void tls_layer_impl::set_hostname(native_string const& host)
{
	hostname_ = host;
	if (session_ && !hostname_.empty() && get_address_type(hostname_) == address_type::unknown) {
		auto const utf8 = to_utf8(hostname_);
		if (!utf8.empty()) {
			int res = gnutls_server_name_set(session_, GNUTLS_NAME_DNS, utf8.c_str(), utf8.size());
			if (res) {
				log_error(res, L"gnutls_server_name_set", logmsg::debug_warning);
			}
		}
	}
}

native_string tls_layer_impl::get_hostname() const
{
	if (!session_) {
		return {};
	}

	size_t len{};
	unsigned int type{};
	unsigned int i{};
	int ret;
	do {
		ret = gnutls_server_name_get(session_, nullptr, &len, &type, i++);
	}
	while (ret == GNUTLS_E_SHORT_MEMORY_BUFFER && type != GNUTLS_NAME_DNS);

	if (ret == GNUTLS_E_SHORT_MEMORY_BUFFER) {
		std::string name;
		name.resize(len - 1);
		ret = gnutls_server_name_get(session_, name.data(), &len, &type, --i);
		if (!ret) {
			return fz::to_native(name);
		}
	}

	return {};
}

int tls_layer_impl::connect(native_string const& host, unsigned int port, address_type family)
{
	if (hostname_.empty()) {
		set_hostname(host);
	}

	return tls_layer_.next_layer_.connect(host, port, family);
}

std::vector<uint8_t> tls_layer_impl::get_session_parameters() const
{
	std::vector<uint8_t> ret;

	if (is_client()) {
		datum_holder d;
		int res = gnutls_session_get_data2(session_, &d);
		if (res) {
			logger_.log(logmsg::debug_warning, L"gnutls_session_get_data2 failed: %d", res);
		}
		else {
			ret.assign(d.data, d.data + d.size);
		}
	}
	else {
		ret = ticket_key_;
	}

	return ret;
}

std::vector<uint8_t> tls_layer_impl::get_raw_certificate() const
{
	std::vector<uint8_t> ret;

	// Implicitly trust certificate of primary socket
	unsigned int cert_list_size;
	gnutls_datum_t const* const cert_list = gnutls_certificate_get_peers(session_, &cert_list_size);
	if (cert_list && cert_list_size) {
		ret.assign(cert_list[0].data, cert_list[0].data + cert_list[0].size);
	}

	return ret;
}

std::pair<std::string, std::string> tls_layer_impl::generate_selfsigned_certificate(native_string const& password, std::string const& distinguished_name, std::vector<std::string> const& hostnames)
{
	std::pair<std::string, std::string> ret;

	gnutls_x509_privkey_t priv;
	int res = gnutls_x509_privkey_init(&priv);
	if (res) {
		return ret;
	}

	auto fmt = GNUTLS_PK_ECDSA;
	unsigned int bits = gnutls_sec_param_to_pk_bits(fmt, GNUTLS_SEC_PARAM_HIGH);
	if (fmt == GNUTLS_PK_RSA && bits < 2048) {
		bits = 2048;
	}

	res = gnutls_x509_privkey_generate(priv, fmt, bits, 0);
	if (res) {
		gnutls_x509_privkey_deinit(priv);
		return ret;
	}

	datum_holder kh;

	if (password.empty()) {
		res = gnutls_x509_privkey_export2(priv, GNUTLS_X509_FMT_PEM, &kh);
	}
	else {
		res = gnutls_x509_privkey_export2_pkcs8(priv, GNUTLS_X509_FMT_PEM, to_utf8(password).c_str(), 0, &kh);
	}
	if (res) {
		gnutls_x509_privkey_deinit(priv);
		return ret;
	}

	gnutls_x509_crt_t crt;
	res = gnutls_x509_crt_init(&crt);
	if (res) {
		gnutls_x509_privkey_deinit(priv);
		return ret;
	}

	res = gnutls_x509_crt_set_version(crt, 3);
	if (res) {
		gnutls_x509_privkey_deinit(priv);
		gnutls_x509_crt_deinit(crt);
		return ret;
	}

	res = gnutls_x509_crt_set_key(crt, priv);
	if (res) {
		gnutls_x509_privkey_deinit(priv);
		gnutls_x509_crt_deinit(crt);
		return ret;
	}

	char const* out{};
	res = gnutls_x509_crt_set_dn(crt, distinguished_name.c_str(), &out);
	if (res) {
		gnutls_x509_privkey_deinit(priv);
		gnutls_x509_crt_deinit(crt);
		return ret;
	}

	for (auto const& hostname : hostnames) {
		res = gnutls_x509_crt_set_subject_alt_name(crt, GNUTLS_SAN_DNSNAME, hostname.c_str(), hostname.size(), GNUTLS_FSAN_APPEND);
		if (res) {
			gnutls_x509_privkey_deinit(priv);
			gnutls_x509_crt_deinit(crt);
			return ret;
		}
	}

	res = gnutls_x509_crt_set_serial(crt, random_bytes(20).data(), 20);
	if (res) {
		gnutls_x509_privkey_deinit(priv);
		gnutls_x509_crt_deinit(crt);
		return ret;
	}

	auto const now = datetime::now();

	res = gnutls_x509_crt_set_activation_time(crt, (now - duration::from_minutes(5)).get_time_t());
	if (res) {
		gnutls_x509_privkey_deinit(priv);
		gnutls_x509_crt_deinit(crt);
		return ret;
	}
	res = gnutls_x509_crt_set_expiration_time(crt, (now + duration::from_days(366)).get_time_t());
	if (res) {
		gnutls_x509_privkey_deinit(priv);
		gnutls_x509_crt_deinit(crt);
		return ret;
	}

	res = gnutls_x509_crt_set_key_usage(crt, GNUTLS_KEY_DIGITAL_SIGNATURE | GNUTLS_KEY_KEY_ENCIPHERMENT);
	if (res) {
		gnutls_x509_privkey_deinit(priv);
		gnutls_x509_crt_deinit(crt);
		return ret;
	}

	res = gnutls_x509_crt_set_basic_constraints(crt, 0, -1);
	if (res) {
		gnutls_x509_privkey_deinit(priv);
		gnutls_x509_crt_deinit(crt);
		return ret;
	}

	res = gnutls_x509_crt_sign2(crt, crt, priv, GNUTLS_DIG_SHA256, 0);
	if (res) {
		gnutls_x509_privkey_deinit(priv);
		gnutls_x509_crt_deinit(crt);
		return ret;
	}

	datum_holder ch;
	res = gnutls_x509_crt_export2(crt, GNUTLS_X509_FMT_PEM, &ch);
	if (res) {
		gnutls_x509_privkey_deinit(priv);
		gnutls_x509_crt_deinit(crt);
		return ret;
	}

	gnutls_x509_privkey_deinit(priv);
	gnutls_x509_crt_deinit(crt);
	ret.first = kh.to_string();
	ret.second = ch.to_string();

	return ret;
}

int tls_layer_impl::shutdown_read()
{
	if (!can_read_from_socket_) {
		return EAGAIN;
	}

	char c{};
	int error{};
	int res = tls_layer_.next_layer_.read(&c, 1, error);
	if (!res) {
		return tls_layer_.next_layer_.shutdown_read();
	}
	else if (res > 0) {
		// Have to fail the connection as we have now discarded data.
		return ECONNABORTED;
	}

	return error;
}

void tls_layer_impl::set_event_handler(event_handler* pEvtHandler, fz::socket_event_flag retrigger_block)
{
	write_blocked_by_send_buffer_ = false;

	fz::socket_event_flag const pending = change_socket_event_handler(tls_layer_.event_handler_, pEvtHandler, &tls_layer_, retrigger_block);
	tls_layer_.event_handler_ = pEvtHandler;

	if (pEvtHandler) {
		if ((state_ == socket_state::connected || state_ == socket_state::shutting_down) && !(pending & (socket_event_flag::write | socket_event_flag::connection)) && !(retrigger_block & socket_event_flag::write)) {
			pEvtHandler->send_event<socket_event>(&tls_layer_, socket_event_flag::write, 0);
		}
		if ((state_ == socket_state::connected || state_ == socket_state::shutting_down || state_ == socket_state::shut_down) && !(pending & socket_event_flag::read) && !(retrigger_block & socket_event_flag::read)) {
			pEvtHandler->send_event<socket_event>(&tls_layer_, socket_event_flag::read, 0);
		}

#if DEBUG_SOCKETEVENTS
		if (state_ == socket_state::connected || state_ == socket_state::shutting_down || state_ == socket_state::shut_down) {
			debug_can_read_ = true;
		}
		if (state_ == socket_state::connected || state_ == socket_state::shutting_down) {
			debug_can_write_ = true;
		}
#endif
	}

}

bool tls_layer_impl::do_set_alpn()
{
	if (alpn_.empty()) {
		return true;
	}

	gnutls_datum_t * data = new gnutls_datum_t[alpn_.size()];
	for (size_t i = 0; i < alpn_.size(); ++i) {
		data[i].data = reinterpret_cast<unsigned char *>(const_cast<char*>(alpn_[i].c_str()));
		data[i].size = alpn_[i].size();
	}
	int res = gnutls_alpn_set_protocols(session_, data, alpn_.size(), GNUTLS_ALPN_MANDATORY);

	if (res) {
		log_error(res, L"gnutls_alpn_set_protocols");
	}
	return res == 0;
}

std::string tls_layer_impl::get_alpn() const
{
	if (session_) {
		gnutls_datum_t protocol;
		if (!gnutls_alpn_get_selected_protocol(session_, &protocol)) {
			return to_string(protocol);
		}
	}
	return {};
}

}
