/*
 * Galago Object API
 *
 * Copyright (C) 2004-2006 Christian Hammond
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 */
#include <libgalago/galago-object.h>
#include <libgalago/galago-context.h>
#include <libgalago/galago-context-priv.h>
#include <libgalago/galago-core.h>
#include <libgalago/galago-enum-types.h>
#include <libgalago/galago-key-value.h>
#include <stdio.h>
#include <string.h>

struct _GalagoObjectPrivate
{
	GalagoContext *context;
	gchar *dbus_path;
	GalagoOrigin origin;
	GHashTable *attrs_table;
	GList *attrs_list;
	gboolean in_context : 1;
	gboolean watched : 1;
};

enum
{
	PROP_0,
	PROP_CONTEXT,
	PROP_ORIGIN,
	PROP_SUPPORTS_ATTRS
};

enum
{
	DESTROY,
	LAST_SIGNAL
};

#define GALAGO_OBJECT_SUPPORTS_ATTRS(object) \
	(GALAGO_OBJECT_GET_CLASS(GALAGO_OBJECT(object))->supports_attrs)

static void galago_object_dispose(GObject *gobject);
static void galago_object_finalize(GObject *gobject);
static void galago_object_real_dbus_push_full(GalagoObject *object);
static void galago_object_real_destroy(GalagoObject *object);
static void galago_object_set_property(GObject *gobject, guint prop_id,
									   const GValue *value, GParamSpec *pspec);
static void galago_object_get_property(GObject *gobject, guint prop_id,
									   GValue *value, GParamSpec *pspec);
static void galago_object_real_set_attribute(GalagoObject *object,
											 const char *name, GValue *value);
static gboolean galago_object_real_remove_attribute(GalagoObject *object,
													const char *name);
static const GValue *galago_object_real_get_attribute(
	const GalagoObject *object, const char *name);
static GList *galago_object_real_get_attributes(const GalagoObject *object);
static void _galago_dbus_object_set_attribute(GalagoObject *object,
											  const char *name,
											  GValue *value);
static GValue *_galago_dbus_object_get_attribute(const GalagoObject *object,
												 const char *name);
static GList *_galago_dbus_object_get_attributes(const GalagoObject *object);

static GObjectClass *parent_class = NULL;
static guint signals[LAST_SIGNAL] = {0};

G_DEFINE_TYPE(GalagoObject, galago_object, G_TYPE_OBJECT);

static void
galago_object_class_init(GalagoObjectClass *klass)
{
	GObjectClass *gobject_class = G_OBJECT_CLASS(klass);

	parent_class = g_type_class_peek_parent(klass);

	klass->supports_attrs = FALSE;

	klass->dbus_message_append = NULL;
	klass->dbus_message_get    = NULL;
	klass->dbus_push_full      = galago_object_real_dbus_push_full;
	klass->dbus_get_signature  = NULL;
	klass->destroy             = galago_object_real_destroy;
	klass->set_attribute       = galago_object_real_set_attribute;
	klass->remove_attribute    = galago_object_real_remove_attribute;
	klass->get_attribute       = galago_object_real_get_attribute;
	klass->get_attributes      = galago_object_real_get_attributes;

	gobject_class->dispose      = galago_object_dispose;
	gobject_class->finalize     = galago_object_finalize;
	gobject_class->set_property = galago_object_set_property;
	gobject_class->get_property = galago_object_get_property;

	/**
	 * GalagoObject::destroy:
	 * @object: The object which received the signal.
	 *
	 * Emitted when the object is undergoing a dispose. Signals that anything
	 * holding a reference to this object should release the reference.
	 * This may or may not end with the object being finalized, depending
	 * on whether there are any references left after this is emitted.
	 */
	signals[DESTROY] =
		g_signal_new("destroy",
					 G_TYPE_FROM_CLASS(klass),
					 G_SIGNAL_RUN_CLEANUP | G_SIGNAL_NO_RECURSE |
					 G_SIGNAL_NO_HOOKS,
					 G_STRUCT_OFFSET(GalagoObjectClass, destroy),
					 NULL, NULL,
					 g_cclosure_marshal_VOID__VOID,
					 G_TYPE_NONE, 0);

	g_object_class_install_property(gobject_class, PROP_CONTEXT,
		g_param_spec_pointer("context", "Context",
							 "The GalagoContext this object belongs to",
							 G_PARAM_READABLE |
							 G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));

	g_object_class_install_property(gobject_class, PROP_ORIGIN,
		g_param_spec_enum("origin", "Origin",
						  "The object's origin",
						  GALAGO_TYPE_ORIGIN,
						  GALAGO_LOCAL,
						  G_PARAM_READWRITE | G_PARAM_CONSTRUCT_ONLY |
						  G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));

	g_object_class_install_property(gobject_class, PROP_SUPPORTS_ATTRS,
		g_param_spec_boolean("supports-attrs",
							 "Support Remote Attributes",
							 "Indicates if this object supports remote "
							 "attributes",
							 FALSE, G_PARAM_READABLE |
							 G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));
}

static void
galago_object_init(GalagoObject *object)
{
	object->priv = g_new0(GalagoObjectPrivate, 1);
	object->priv->context = galago_context_get();
}

static void
galago_object_dispose(GObject *gobject)
{
	GalagoObject *object = GALAGO_OBJECT(gobject);

	if (!GALAGO_OBJECT_HAS_FLAG(object, GALAGO_OBJECT_IN_DESTRUCTION))
	{
		GALAGO_OBJECT_SET_FLAGS(object, GALAGO_OBJECT_IN_DESTRUCTION);

		g_signal_emit(object, signals[DESTROY], 0);

		if (object->priv->in_context)
		{
			galago_context_push(object->priv->context);
			galago_context_remove_object(object);
			galago_context_pop();
			object->priv->in_context = FALSE;
		}

		GALAGO_OBJECT_UNSET_FLAGS(object, GALAGO_OBJECT_IN_DESTRUCTION);
	}

	if (G_OBJECT_CLASS(parent_class)->dispose != NULL)
		G_OBJECT_CLASS(parent_class)->dispose(gobject);
}

static void
galago_object_finalize(GObject *gobject)
{
	GalagoObject *object = GALAGO_OBJECT(gobject);

	if (object->priv != NULL)
	{
		if (object->priv->dbus_path != NULL)
		{
			g_free(object->priv->dbus_path);
			object->priv->dbus_path = NULL;
		}

		if (object->priv->attrs_table != NULL)
			g_hash_table_destroy(object->priv->attrs_table);

		g_free(object->priv);
		object->priv = NULL;
	}

	if (G_OBJECT_CLASS(parent_class)->finalize != NULL)
		G_OBJECT_CLASS(parent_class)->finalize(gobject);
}

static void
_push_key(const char *name, GValue *value, GalagoObject *object)
{
	_galago_dbus_object_set_attribute(object, name, value);
}

static void
galago_object_real_dbus_push_full(GalagoObject *object)
{
	if (object->priv->attrs_table != NULL)
	{
		g_hash_table_foreach(object->priv->attrs_table,
							 (GHFunc)_push_key, object);
	}
}

static void
galago_object_real_destroy(GalagoObject *object)
{
	g_signal_handlers_destroy(object);
}

static void
galago_object_set_property(GObject *gobject, guint prop_id,
						   const GValue *value, GParamSpec *pspec)
{
	GalagoObject *object = GALAGO_OBJECT(gobject);

	switch (prop_id)
	{
		case PROP_ORIGIN:
			object->priv->origin = g_value_get_enum(value);
			g_object_notify(gobject, "origin");
			break;

		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID(gobject, prop_id, pspec);
			break;
	}
}

static void
galago_object_get_property(GObject *gobject, guint prop_id, GValue *value,
						   GParamSpec *pspec)
{
	GalagoObject *object = GALAGO_OBJECT(gobject);

	switch (prop_id)
	{
		case PROP_CONTEXT:
			g_value_set_pointer(value, galago_object_get_context(object));
			break;

		case PROP_ORIGIN:
			g_value_set_enum(value, galago_object_get_origin(object));
			break;

		case PROP_SUPPORTS_ATTRS:
			g_value_set_boolean(value,
				GALAGO_OBJECT_GET_CLASS(object)->supports_attrs);
			break;

		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID(gobject, prop_id, pspec);
			break;
	}
}


/**
 * galago_object_destroy
 * @object: The object to destroy.
 *
 * Emits the "destroy" signal for an object, and attempts to dispose of it.
 * The memory for the object won't actually be deleted until the reference
 * count drops to 0.
 */
void
galago_object_destroy(GalagoObject *object)
{
	g_return_if_fail(object != NULL);
	g_return_if_fail(GALAGO_IS_OBJECT(object));

	if (!GALAGO_OBJECT_HAS_FLAG(object, GALAGO_OBJECT_IN_DESTRUCTION))
	{
		g_object_run_dispose(G_OBJECT(object));
		g_object_unref(G_OBJECT(object));
	}
}

/**
 * galago_object_type_get_dbus_signature
 * @type: The object GType.
 *
 * Returns the D-BUS signature of the object type.
 *
 * Returns: The signature string, or NULL.
 */
const gchar *
galago_object_type_get_dbus_signature(GType type)
{
	GalagoObjectClass *klass;

	g_return_val_if_fail(g_type_is_a(type, GALAGO_TYPE_OBJECT), NULL);

	klass = g_type_class_ref(type);

	if (klass->dbus_get_signature == NULL)
	{
		g_type_class_unref(klass);
		return NULL;
	}

	if (klass->dbus_signature == NULL)
	{
		char *sig = klass->dbus_get_signature();
		klass->dbus_signature =
			g_strconcat(DBUS_STRUCT_BEGIN_CHAR_AS_STRING,
						sig,
						DBUS_STRUCT_END_CHAR_AS_STRING,
						NULL);
		g_free(sig);
	}

	g_type_class_unref(klass);

	return klass->dbus_signature;
}

/**
 * galago_object_set_dbus_path
 * @object:   The object.
 * @obj_path: The object path.
 *
 * Sets the D-BUS object path of an object.
 */
void
galago_object_set_dbus_path(GalagoObject *object, const gchar *obj_path)
{
	g_return_if_fail(object != NULL && GALAGO_IS_OBJECT(object));

	galago_context_push(object->priv->context);

	if (object->priv->dbus_path != NULL)
	{
		galago_context_remove_object(object);
		object->priv->in_context = FALSE;

		g_free(object->priv->dbus_path);
	}

	object->priv->dbus_path = (obj_path != NULL ? g_strdup(obj_path) : NULL);

	if (object->priv->dbus_path != NULL && !object->priv->in_context)
	{
		galago_context_add_object(object);
		object->priv->in_context = TRUE;
	}

	galago_context_pop();
}

/**
 * galago_object_get_dbus_path
 * @object: The object.
 *
 * Returns the D-BUS object path of an object.
 *
 * Returns: The object path.
 */
const gchar *
galago_object_get_dbus_path(const GalagoObject *object)
{
	g_return_val_if_fail(object != NULL && GALAGO_IS_OBJECT(object), NULL);

	return object->priv->dbus_path;
}

/**
 * galago_object_set_watch
 * @object: The object.
 * @watch:  %TRUE if this object should be watched, or %FALSE.
 *
 * Sets whether or not this object is watched for events.
 */
void
galago_object_set_watch(GalagoObject *object, gboolean watch)
{
	g_return_if_fail(object != NULL && GALAGO_IS_OBJECT(object));

	if (object->priv->watched == watch)
		return;

	object->priv->watched = watch;
}

/**
 * galago_object_is_watched
 * @object: The object.
 *
 * Returns whether or not an object is watched for events.
 *
 * Returns: %TRUE if this object is being watched, or %FALSE.
 */
gboolean
galago_object_is_watched(const GalagoObject *object)
{
	g_return_val_if_fail(object != NULL && GALAGO_IS_OBJECT(object), FALSE);

	return object->priv->watched;
}

/**
 * galago_object_get_origin
 * @object: The object.
 *
 * Returns the object's origin.
 *
 * Returns: The object's origin
 */
GalagoOrigin
galago_object_get_origin(const GalagoObject *object)
{
	g_return_val_if_fail(object != NULL && GALAGO_IS_OBJECT(object),
						 GALAGO_LOCAL);

	return object->priv->origin;
}

/**
 * galago_object_get_context
 * @object: The object.
 *
 * Returns the object's context.
 *
 * Returns: The object's context.
 */
GalagoContext *
galago_object_get_context(const GalagoObject *object)
{
	g_return_val_if_fail(object != NULL && GALAGO_IS_OBJECT(object), NULL);

	return object->priv->context;
}

/**
 * galago_object_set_attr_string
 * @object: The object.
 * @name:   The name of the attribute to set.
 * @value:  The value of the attribute.
 *
 * Sets a string attribute on an object.
 */
void
galago_object_set_attr_string(GalagoObject *object, const char *name,
							  const char *str_value)
{
	GValue *value;

	g_return_if_fail(object    != NULL);
	g_return_if_fail(name      != NULL && *name      != '\0');
	g_return_if_fail(str_value != NULL && *str_value != '\0');
	g_return_if_fail(GALAGO_IS_OBJECT(object));

	value = g_new0(GValue, 1);
	g_value_init(value, G_TYPE_STRING);
	g_value_set_string(value, str_value);
	galago_object_set_attribute(object, name, value);
}

/**
 * galago_object_set_attr_bool
 * @object: The object.
 * @name:   The name of the attribute to set.
 * @value:  The value of the attribute.
 *
 * Sets a boolean attribute on an object.
 */
void
galago_object_set_attr_bool(GalagoObject *object, const char *name,
							gboolean bool_value)
{
	GValue *value;

	g_return_if_fail(object != NULL);
	g_return_if_fail(name   != NULL && *name  != '\0');
	g_return_if_fail(GALAGO_IS_OBJECT(object));

	value = g_new0(GValue, 1);
	g_value_init(value, G_TYPE_BOOLEAN);
	g_value_set_boolean(value, bool_value);
	galago_object_set_attribute(object, name, value);
}

/**
 * galago_object_set_attr_int
 * @object: The object.
 * @name:   The name of the attribute to set.
 * @value:  The value of the attribute.
 *
 * Sets an integer attribute on an object.
 */
void
galago_object_set_attr_int(GalagoObject *object, const char *name,
						   gint32 int_value)
{
	GValue *value;

	g_return_if_fail(object != NULL);
	g_return_if_fail(name   != NULL && *name  != '\0');
	g_return_if_fail(GALAGO_IS_OBJECT(object));

	value = g_new0(GValue, 1);
	g_value_init(value, G_TYPE_INT);
	g_value_set_int(value, int_value);
	galago_object_set_attribute(object, name, value);
}

/**
 * galago_object_set_attr_double
 * @object: The object.
 * @name:   The name of the attribute to set.
 * @value:  The value of the attribute.
 *
 * Sets a double attribute on an object.
 */
void
galago_object_set_attr_double(GalagoObject *object, const char *name,
							  gdouble double_value)
{
	GValue *value;

	g_return_if_fail(object != NULL);
	g_return_if_fail(name   != NULL && *name  != '\0');
	g_return_if_fail(GALAGO_IS_OBJECT(object));

	value = g_new0(GValue, 1);
	g_value_init(value, G_TYPE_DOUBLE);
	g_value_set_int(value, double_value);
	galago_object_set_attribute(object, name, value);
}

static void
reset_attrs_list(GalagoObject *object)
{
	if (object->priv->attrs_list != NULL)
	{
		g_list_foreach(object->priv->attrs_list,
					   (GFunc)galago_key_value_destroy, NULL);
		g_list_free(object->priv->attrs_list);
		object->priv->attrs_list = NULL;
	}
}

static void
destroy_value(GValue *value)
{
	g_value_unset(value);
	g_free(value);
}

static void
galago_object_real_set_attribute(GalagoObject *object, const char *name,
								 GValue *value)
{
	if (object->priv->attrs_table == NULL)
	{
		object->priv->attrs_table =
			g_hash_table_new_full(g_str_hash, g_str_equal, g_free,
								  (GFreeFunc)destroy_value);
	}

	reset_attrs_list(object);

	g_hash_table_replace(object->priv->attrs_table,
						 g_ascii_strdown(name, -1), value);

	if (GALAGO_OBJECT_IS_LOCAL(object))
		_galago_dbus_object_set_attribute(object, name, value);
}

/**
 * galago_object_set_attribute
 * @object: The object.
 * @name:   The name of the attribute to set.
 * @value:  The value of the attribute.
 *
 * Sets an attribute on an object.
 *
 * This is limited to string, boolean, and integer value types.
 */
void
galago_object_set_attribute(GalagoObject *object, const char *name,
							GValue *value)
{
	g_return_if_fail(object != NULL);
	g_return_if_fail(GALAGO_IS_OBJECT(object));
	g_return_if_fail(GALAGO_OBJECT_SUPPORTS_ATTRS(object));
	g_return_if_fail(name   != NULL && *name  != '\0');
	g_return_if_fail(value  != NULL);
	g_return_if_fail(G_VALUE_HOLDS(value, G_TYPE_STRING) ||
					 G_VALUE_HOLDS(value, G_TYPE_BOOLEAN) ||
					 G_VALUE_HOLDS(value, G_TYPE_INT) ||
					 G_VALUE_HOLDS(value, G_TYPE_DOUBLE));

	g_return_if_fail(GALAGO_OBJECT_GET_CLASS(object)->get_attribute != NULL);

	GALAGO_OBJECT_GET_CLASS(object)->set_attribute(object, name, value);
}

static gboolean
galago_object_real_remove_attribute(GalagoObject *object, const char *name)
{
	char *temp;
	GList *l;

	if (object->priv->attrs_table == NULL)
		return TRUE;

	reset_attrs_list(object);

	temp = g_ascii_strdown(name, -1);
	g_hash_table_remove(object->priv->attrs_table, temp);

	for (l = object->priv->attrs_list; l != NULL; l = l->next)
	{
		GalagoKeyValue *key_value = (GalagoKeyValue *)l->data;

		if (!strcmp(galago_key_value_get_key(key_value), temp))
		{
			object->priv->attrs_list =
				g_list_remove_link(object->priv->attrs_list, l);
			g_list_free_1(l);
			break;
		}
	}

	g_free(temp);

	if (GALAGO_OBJECT_IS_LOCAL(object) && galago_is_connected() &&
		galago_is_feed())
	{
		galago_dbus_send_message(GALAGO_OBJECT(object), "RemoveAttribute",
			galago_value_new(GALAGO_VALUE_TYPE_STRING, &name, NULL),
			NULL);
	}

	return TRUE;
}

/**
 * galago_object_remove_attribute
 * @object: The object
 * @name:   The name of the attribute to remove
 *
 * Removes an attribute on an object.
 *
 * Returns: %TRUE if the attribute was removed, or %FALSE.
 */
gboolean
galago_object_remove_attribute(GalagoObject *object, const char *name)
{
	g_return_val_if_fail(object != NULL,                       FALSE);
	g_return_val_if_fail(GALAGO_IS_OBJECT(object),             FALSE);
	g_return_val_if_fail(GALAGO_OBJECT_SUPPORTS_ATTRS(object), FALSE);
	g_return_val_if_fail(name   != NULL && *name != '\0',      FALSE);

	g_return_val_if_fail(
		GALAGO_OBJECT_GET_CLASS(object)->remove_attribute != NULL, FALSE);

	return GALAGO_OBJECT_GET_CLASS(object)->remove_attribute(object, name);
}

/**
 * galago_object_get_attr_string
 * @object: The object.
 * @name:   The name of the attribute.
 *
 * Returns the value of a string attribute on an object.
 *
 * Returns: The attribute value, or NULL.
 */
const char *
galago_object_get_attr_string(const GalagoObject *object, const char *name)
{
	const GValue *value;

	g_return_val_if_fail(object != NULL,                       NULL);
	g_return_val_if_fail(GALAGO_IS_OBJECT(object),             NULL);
	g_return_val_if_fail(GALAGO_OBJECT_SUPPORTS_ATTRS(object), NULL);
	g_return_val_if_fail(name   != NULL && *name != '\0',      NULL);

	value = galago_object_get_attribute(object, name);

	if (value == NULL)
		return NULL;

	g_return_val_if_fail(G_VALUE_HOLDS(value, G_TYPE_STRING), NULL);

	return g_value_get_string(value);
}

/**
 * galago_object_get_attr_bool
 * @object: The object.
 * @name:   The name of the attribute.
 *
 * Returns the value of a boolean attribute on an object.
 *
 * Returns: The attribute value.
 */
gboolean
galago_object_get_attr_bool(const GalagoObject *object, const char *name)
{
	const GValue *value;

	g_return_val_if_fail(object != NULL,                       FALSE);
	g_return_val_if_fail(GALAGO_IS_OBJECT(object),             FALSE);
	g_return_val_if_fail(GALAGO_OBJECT_SUPPORTS_ATTRS(object), FALSE);
	g_return_val_if_fail(name   != NULL && *name != '\0',      FALSE);

	value = galago_object_get_attribute(object, name);

	if (value == NULL)
		return FALSE;

	g_return_val_if_fail(G_VALUE_HOLDS(value, G_TYPE_BOOLEAN), FALSE);

	return g_value_get_boolean(value);
}

/**
 * galago_object_get_attr_int
 * @object: The object.
 * @name:   The name of the attribute.
 *
 * Returns the value of an integer attribute on an object.
 *
 * Returns: The attribute value.
 */
gint32
galago_object_get_attr_int(const GalagoObject *object, const char *name)
{
	const GValue *value;

	g_return_val_if_fail(object != NULL,                       0);
	g_return_val_if_fail(GALAGO_IS_OBJECT(object),             0);
	g_return_val_if_fail(GALAGO_OBJECT_SUPPORTS_ATTRS(object), 0);
	g_return_val_if_fail(name   != NULL && *name != '\0',      0);

	value = galago_object_get_attribute(object, name);

	if (value == NULL)
		return 0;

	g_return_val_if_fail(G_VALUE_HOLDS(value, G_TYPE_INT), 0);

	return g_value_get_int(value);
}

/**
 * galago_object_get_attr_double
 * @object: The object.
 * @name:   The name of the attribute.
 *
 * Returns the value of a double attribute on an object.
 *
 * Returns: The attribute value.
 */
gdouble
galago_object_get_attr_double(const GalagoObject *object, const char *name)
{
	const GValue *value;

	g_return_val_if_fail(object != NULL,                       0);
	g_return_val_if_fail(GALAGO_IS_OBJECT(object),             0);
	g_return_val_if_fail(GALAGO_OBJECT_SUPPORTS_ATTRS(object), 0);
	g_return_val_if_fail(name   != NULL && *name != '\0',      0);

	value = galago_object_get_attribute(object, name);

	if (value == NULL)
		return 0.0;

	g_return_val_if_fail(G_VALUE_HOLDS(value, G_TYPE_DOUBLE), 0);

	return g_value_get_double(value);
}

static const GValue *
galago_object_real_get_attribute(const GalagoObject *object, const char *name)
{
	GValue *value = NULL;
	char *temp;

	temp = g_ascii_strdown(name, -1);

	if (object->priv->attrs_table != NULL)
		value = g_hash_table_lookup(object->priv->attrs_table, temp);

	if (value == NULL && GALAGO_OBJECT_IS_REMOTE(object))
		value = _galago_dbus_object_get_attribute(object, temp);

	g_free(temp);

	return value;
}

/**
 * galago_object_get_attribute
 * @object: The object.
 * @name:   The name of the attribute.
 *
 * Returns the value of an attribute on an object.
 *
 * Returns: The attribute value, or NULL.
 */
const GValue *
galago_object_get_attribute(const GalagoObject *object, const char *name)
{
	g_return_val_if_fail(object != NULL,                       NULL);
	g_return_val_if_fail(GALAGO_IS_OBJECT(object),             NULL);
	g_return_val_if_fail(GALAGO_OBJECT_SUPPORTS_ATTRS(object), NULL);
	g_return_val_if_fail(name   != NULL && *name != '\0',      NULL);

	g_return_val_if_fail(GALAGO_OBJECT_GET_CLASS(object)->get_attribute != NULL,
						 NULL);

	return GALAGO_OBJECT_GET_CLASS(object)->get_attribute(object, name);
}

/**
 * galago_object_get_has_attribute
 * @object: The object.
 * @name:   The name of the attribute.
 *
 * Returns whether or not an object has a specific attribute set.
 *
 * Returns: %TRUE if the attribute is set, or %FALSE.
 */
gboolean
galago_object_get_has_attribute(const GalagoObject *object, const char *name)
{
	g_return_val_if_fail(object != NULL,                       FALSE);
	g_return_val_if_fail(GALAGO_IS_OBJECT(object),             FALSE);
	g_return_val_if_fail(GALAGO_OBJECT_SUPPORTS_ATTRS(object), FALSE);
	g_return_val_if_fail(name   != NULL && *name != '\0',      FALSE);

	return galago_object_get_attribute(object, name) != NULL;
}

static void
get_attributes_foreach(const char *key, GValue *value, GalagoObject *object)
{
	object->priv->attrs_list = g_list_append(object->priv->attrs_list,
											 galago_key_value_new(key, value));
}

static GList *
galago_object_real_get_attributes(const GalagoObject *object)
{
	if (GALAGO_OBJECT_IS_REMOTE(object) && !galago_is_daemon())
	{
		reset_attrs_list((GalagoObject *)object);
		object->priv->attrs_list = _galago_dbus_object_get_attributes(object);
	}
	else if (object->priv->attrs_list == NULL &&
			 object->priv->attrs_table != NULL)
	{
		g_hash_table_foreach(object->priv->attrs_table,
							 (GHFunc)get_attributes_foreach,
							 (gpointer)object);
	}

	return object->priv->attrs_list;
}

/**
 * galago_object_get_attributes
 * @object: The object.
 *
 * Returns the list of attributes in an object, represented by GalagoKeyValue
 * structs.
 *
 * Returns: The attributes in the object. Each one is a GalagoKeyValue.
 */
GList *
galago_object_get_attributes(const GalagoObject *object)
{
	g_return_val_if_fail(object != NULL,                       NULL);
	g_return_val_if_fail(GALAGO_IS_OBJECT(object),             NULL);
	g_return_val_if_fail(GALAGO_OBJECT_SUPPORTS_ATTRS(object), NULL);

	g_return_val_if_fail(
		GALAGO_OBJECT_GET_CLASS(object)->get_attributes != NULL, NULL);

	return GALAGO_OBJECT_GET_CLASS(object)->get_attributes(object);
}


/**************************************************************************
 * D-BUS Functions
 **************************************************************************/
static void
_galago_dbus_object_set_attribute(GalagoObject *object, const char *name,
								  GValue *value)
{
	DBusMessage *message;
	DBusMessageIter iter, value_iter;
	gboolean success = TRUE;

	if (!galago_is_connected() || !galago_is_feed())
		return;

	message = galago_dbus_message_new_method_call(GALAGO_OBJECT(object),
												  "SetAttribute", FALSE,
												  &iter);
	dbus_message_iter_append_basic(&iter, DBUS_TYPE_STRING, &name);

	if (G_VALUE_HOLDS(value, G_TYPE_STRING))
	{
		const char *str = g_value_get_string(value);
		dbus_message_iter_open_container(&iter, DBUS_TYPE_VARIANT,
										 DBUS_TYPE_STRING_AS_STRING,
										 &value_iter);
		dbus_message_iter_append_basic(&value_iter, DBUS_TYPE_STRING, &str);
		dbus_message_iter_close_container(&iter, &value_iter);
	}
	else if (G_VALUE_HOLDS(value, G_TYPE_BOOLEAN))
	{
		gboolean b = g_value_get_boolean(value);
		dbus_message_iter_open_container(&iter, DBUS_TYPE_VARIANT,
										 DBUS_TYPE_BOOLEAN_AS_STRING,
										 &value_iter);
		dbus_message_iter_append_basic(&value_iter, DBUS_TYPE_BOOLEAN, &b);
		dbus_message_iter_close_container(&iter, &value_iter);
	}
	else if (G_VALUE_HOLDS(value, G_TYPE_INT))
	{
		gint32 i = g_value_get_int(value);
		dbus_message_iter_open_container(&iter, DBUS_TYPE_VARIANT,
										 DBUS_TYPE_INT32_AS_STRING,
										 &value_iter);
		dbus_message_iter_append_basic(&value_iter, DBUS_TYPE_INT32, &i);
		dbus_message_iter_close_container(&iter, &value_iter);
	}
	else if (G_VALUE_HOLDS(value, G_TYPE_DOUBLE))
	{
		gdouble i = g_value_get_double(value);
		dbus_message_iter_open_container(&iter, DBUS_TYPE_VARIANT,
										 DBUS_TYPE_DOUBLE_AS_STRING,
										 &value_iter);
		dbus_message_iter_append_basic(&value_iter, DBUS_TYPE_DOUBLE, &i);
		dbus_message_iter_close_container(&iter, &value_iter);
	}
	else
		g_assert_not_reached();

	if (success)
		dbus_connection_send(galago_get_dbus_conn(), message, NULL);

	dbus_message_unref(message);
}

static GValue *
_galago_dbus_object_get_attr_value(DBusMessageIter *iter)
{
	DBusMessageIter value_iter;
	GValue *value;

	dbus_message_iter_recurse(iter, &value_iter);

	value = g_new0(GValue, 1);

	int type = dbus_message_iter_get_arg_type(&value_iter);
	switch (type)
	{
		case DBUS_TYPE_STRING:
		{
			const char *data;
			dbus_message_iter_get_basic(&value_iter, &data);
			g_value_init(value, G_TYPE_STRING);
			g_value_set_string(value, data);
			break;
		}

		case DBUS_TYPE_BOOLEAN:
		{
			gboolean data;
			dbus_message_iter_get_basic(&value_iter, &data);
			g_value_init(value, G_TYPE_BOOLEAN);
			g_value_set_boolean(value, data);
			break;
		}

		case DBUS_TYPE_UINT32:
		{
			int data;
			dbus_message_iter_get_basic(&value_iter, &data);
			g_value_init(value, G_TYPE_INT);
			g_value_set_int(value, data);
			break;
		}

		case DBUS_TYPE_DOUBLE:
		{
			int data;
			dbus_message_iter_get_basic(&value_iter, &data);
			g_value_init(value, G_TYPE_DOUBLE);
			g_value_set_int(value, data);
			break;
		}

		default:
			g_value_unset(value);
			value = NULL;
			break;
	}

	return value;
}

static GValue *
_galago_dbus_object_get_attribute(const GalagoObject *object,
								  const char *name)
{
	DBusConnection *dbus_conn;
	DBusMessage *message, *reply;
	DBusMessageIter iter;
	DBusError error;
	GValue *value = NULL;

	if (!galago_is_connected())
		return NULL;

	dbus_conn = galago_get_dbus_conn();

	message = galago_dbus_message_new_method_call(GALAGO_OBJECT(object),
												  "GetAttribute", TRUE, &iter);
	g_return_val_if_fail(message != NULL, NULL);

	dbus_message_iter_append_basic(&iter, DBUS_TYPE_STRING, &name);

	dbus_error_init(&error);
	reply = dbus_connection_send_with_reply_and_block(dbus_conn, message,
													  -1, &error);
	dbus_message_unref(message);

	if (dbus_error_is_set(&error))
	{
		if (!dbus_error_has_name(&error, GALAGO_DBUS_ERROR_INVALID_ATTRIBUTE))
		{
			g_warning("Error sending Object.GetAttribute(%s, %s): %s",
					  galago_object_get_dbus_path(object), name, error.message);
		}

		goto exit;
	}

	dbus_message_iter_init(reply, &iter);
	value = _galago_dbus_object_get_attr_value(&iter);

exit:
	dbus_error_free(&error);

	if (reply != NULL)
		dbus_message_unref(reply);

	return value;
}

static GList *
_galago_dbus_object_get_attributes(const GalagoObject *object)
{
	DBusConnection *dbus_conn;
	DBusMessage *message, *reply;
	DBusMessageIter iter, array_iter, struct_iter;
	DBusError error;
	GList *attrs = NULL;

	if (!galago_is_connected())
		return NULL;

	dbus_conn = galago_get_dbus_conn();

	message = galago_dbus_message_new_method_call(GALAGO_OBJECT(object),
												  "GetAttributes", TRUE, NULL);
	g_return_val_if_fail(message != NULL, NULL);

	dbus_error_init(&error);
	reply = dbus_connection_send_with_reply_and_block(dbus_conn, message,
													  -1, &error);
	dbus_message_unref(message);

	if (dbus_error_is_set(&error))
	{
		g_warning("Error sending Object.GetAttributes(%s): %s",
				  galago_object_get_dbus_path(object), error.message);

		goto exit;
	}

	dbus_message_iter_init(reply, &iter);
	dbus_message_iter_recurse(&iter, &array_iter);

	while (dbus_message_iter_get_arg_type(&array_iter) != DBUS_TYPE_INVALID)
	{
		const char *attr_id;
		GValue *value;

		dbus_message_iter_recurse(&array_iter, &struct_iter);
		dbus_message_iter_get_basic(&struct_iter, &attr_id);
		dbus_message_iter_next(&struct_iter);

		value = _galago_dbus_object_get_attr_value(&struct_iter);
		dbus_message_iter_next(&array_iter);

		attrs = g_list_append(attrs, galago_key_value_new(attr_id, value));
	}

exit:
	dbus_error_free(&error);

	if (reply != NULL)
		dbus_message_unref(reply);

	return attrs;
}
