/* -*- Mode: C++; c-default-style: "k&r"; indent-tabs-mode: nil; tab-width: 2; c-basic-offset: 2 -*- */

/* libmwaw
* Version: MPL 2.0 / LGPLv2+
*
* The contents of this file are subject to the Mozilla Public License Version
* 2.0 (the "License"); you may not use this file except in compliance with
* the License or as specified alternatively below. You may obtain a copy of
* the License at http://www.mozilla.org/MPL/
*
* Software distributed under the License is distributed on an "AS IS" basis,
* WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
* for the specific language governing rights and limitations under the
* License.
*
* Major Contributor(s):
* Copyright (C) 2002 William Lachance (wrlach@gmail.com)
* Copyright (C) 2002,2004 Marc Maurer (uwog@uwog.net)
* Copyright (C) 2004-2006 Fridrich Strba (fridrich.strba@bluewin.ch)
* Copyright (C) 2006, 2007 Andrew Ziem
* Copyright (C) 2011, 2012 Alonso Laurent (alonso@loria.fr)
*
*
* All Rights Reserved.
*
* For minor contributions see the git repository.
*
* Alternatively, the contents of this file may be used under the terms of
* the GNU Lesser General Public License Version 2 or later (the "LGPLv2+"),
* in which case the provisions of the LGPLv2+ are applicable
* instead of those above.
*/

#include <algorithm>
#include <iomanip>
#include <iostream>
#include <set>
#include <sstream>
#include <utility>

#include <librevenge/librevenge.h>

#include "MWAWDebug.hxx"
#include "MWAWEntry.hxx"
#include "MWAWOLEParser.hxx"

#include "PowerPoint3OLE.hxx"

/** Internal: the structures of a PowerPoint3OLE */
namespace PowerPoint3OLEInternal
{
////////////////////////////////////////
//! Internal: the state of a PowerPoint3OLE
struct State {
  //! constructor
  explicit State(MWAWInputStreamPtr input) : m_input(input), m_oleParser(), m_unparsedNameSet()
  {
  }
  /** the input */
  MWAWInputStreamPtr m_input;
  /** the ole parser */
  shared_ptr<MWAWOLEParser> m_oleParser;
  /** the list of unparsed zone */
  std::set<std::string> m_unparsedNameSet;
};

}

////////////////////////////////////////////////////////////
// constructor/destructor, ...
////////////////////////////////////////////////////////////
PowerPoint3OLE::PowerPoint3OLE(MWAWInputStreamPtr input, MWAWFontConverterPtr fontConverter, int fId) : m_state(new PowerPoint3OLEInternal::State(input))
{
  if (input && input->isStructured() && input->getSubStreamByName("PP40"))
    m_state->m_oleParser.reset(new MWAWOLEParser("PP40", fontConverter, fId));
}

PowerPoint3OLE::~PowerPoint3OLE()
{
}

void PowerPoint3OLE::updateMetaData(librevenge::RVNGPropertyList &metaData) const
{
  if (m_state->m_oleParser)
    m_state->m_oleParser->updateMetaData(metaData);
}

////////////////////////////////////////////////////////////
// the parser
////////////////////////////////////////////////////////////
bool PowerPoint3OLE::parse()
{
  MWAWInputStreamPtr input=m_state->m_input;
  if (!input || !m_state->m_oleParser || !m_state->m_oleParser->parse(input)) return false;
  std::vector<std::string> unparsed = m_state->m_oleParser->getNotParse();

  size_t numUnparsed = unparsed.size();
  for (size_t i = 0; i < numUnparsed; i++) {
    std::string const &name = unparsed[i];
    // separated the directory and the name
    //    MatOST/MatadorObject1/Ole10Native
    //      -> dir="MatOST/MatadorObject1", base="Ole10Native"
    std::string::size_type pos = name.find_last_of('/');
    std::string dir, base;
    if (pos == std::string::npos) base = name;
    else if (pos == 0) base = name.substr(1);
    else {
      dir = name.substr(0,pos);
      base = name.substr(pos+1);
    }
    if (dir.empty() && base=="PP40") continue;

    MWAWInputStreamPtr ole = input->getSubStreamByName(name.c_str());
    if (!ole.get()) {
      MWAW_DEBUG_MSG(("PowerPoint3OLE::createZones: error: can not find OLE part: \"%s\"\n", name.c_str()));
      continue;
    }
    ole->setReadInverted(true);
    bool done=false;
    switch (base[0]) {
    case 'C':
      if (base=="Current User")
        done=parseCurrentUser(ole, name);
      else if (base=="Current ID")
        done=parseCurrentId(ole, name);
      break;
    default:
      break;
    }
    if (done) continue;
    m_state->m_unparsedNameSet.insert(name);
  }
  return true;
}

void PowerPoint3OLE::checkForUnparsedStream()
{
  for (std::set<std::string>::const_iterator it=m_state->m_unparsedNameSet.begin(); it!=m_state->m_unparsedNameSet.end(); ++it) {
    std::string const &name = *it;
    std::string::size_type pos = name.find_last_of('/');
    std::string dir, base;
    if (pos == std::string::npos) base = name;
    else if (pos == 0) base = name.substr(1);
    else {
      dir = name.substr(0,pos);
      base = name.substr(pos+1);
    }

    MWAWInputStreamPtr ole = m_state->m_input->getSubStreamByName(name.c_str());
    if (!ole.get()) {
      MWAW_DEBUG_MSG(("PowerPoint3OLE::checkForUnparsedStream: error: can not find OLE part: \"%s\"\n", name.c_str()));
      continue;
    }
    libmwaw::DebugFile asciiFile(ole);
    asciiFile.open(name);
    libmwaw::DebugStream f;
    f << "Entries(" << base << "):";
    asciiFile.addPos(0);
    asciiFile.addNote(f.str().c_str());
  }
}

////////////////////////////////////////////////////////////
// try to read the different stream
////////////////////////////////////////////////////////////
bool PowerPoint3OLE::parseCurrentId(MWAWInputStreamPtr input, std::string const &name)
{
  if (!input||input->size()!=4) {
    MWAW_DEBUG_MSG(("PowerPoint3OLE::parseCurrentId: can not find the input\n"));
    return false;
  }
  libmwaw::DebugFile ascFile(input);
  ascFile.open(name);
  input->seek(0, librevenge::RVNG_SEEK_SET);
  libmwaw::DebugStream f;
  f << "Entries(CurrentId):";
  int val=int(input->readLong(4));
  if (val) f << "id=" << val << ",";
  ascFile.addPos(0);
  ascFile.addNote(f.str().c_str());
  return true;
}

bool PowerPoint3OLE::parseCurrentUser(MWAWInputStreamPtr input, std::string const &name)
{
  if (!input||input->size()==0) {
    MWAW_DEBUG_MSG(("PowerPoint3OLE::parseCurrentUser: can not find the input\n"));
    return false;
  }
  libmwaw::DebugFile ascFile(input);
  ascFile.open(name);
  input->seek(0, librevenge::RVNG_SEEK_SET);
  long endPos=input->size();
  libmwaw::DebugStream f;
  f << "Entries(CurrentUser):";
  int sSz=int(input->readULong(1));
  if (sSz<0 || sSz+1>input->size()) {
    MWAW_DEBUG_MSG(("PowerPoint3OLE::parseCurrentUser: the stream size seems bad\n"));
    f << "###sSz,";
    ascFile.addPos(0);
    ascFile.addNote(f.str().c_str());
    return true;
  }
  std::string user;
  for (int c=0; c<sSz; ++c) {
    char ch=char(input->readULong(1));
    if (ch)
      user+=ch;
    else if (c+1!=sSz)
      f << "###";
  }
  f << user;
  ascFile.addPos(0);
  ascFile.addNote(f.str().c_str());
  if (input->tell()!=endPos) {
    ascFile.addPos(input->tell());
    ascFile.addNote("CurrentUser:##extra");
  }
  return true;
}


////////////////////////////////////////////////////////////
// try to send data
////////////////////////////////////////////////////////////


// vim: set filetype=cpp tabstop=2 shiftwidth=2 cindent autoindent smartindent noexpandtab:
