/*
 * This file is part of the Nice GLib ICE library.
 *
 * (C) 2007 Nokia Corporation. All rights reserved.
 *  Contact: Rémi Denis-Courmont
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is the Nice GLib ICE library.
 *
 * The Initial Developers of the Original Code are Collabora Ltd and Nokia
 * Corporation. All Rights Reserved.
 *
 * Contributors:
 *   Rémi Denis-Courmont, Nokia
 *
 * Alternatively, the contents of this file may be used under the terms of the
 * the GNU Lesser General Public License Version 2.1 (the "LGPL"), in which
 * case the provisions of LGPL are applicable instead of those above. If you
 * wish to allow use of your version of this file only under the terms of the
 * LGPL and not to allow others to use your version of this file under the
 * MPL, indicate your decision by deleting the provisions above and replace
 * them with the notice and other provisions required by the LGPL. If you do
 * not delete the provisions above, a recipient may use your version of this
 * file under either the MPL or the LGPL.
 */

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif

#ifndef _WIN32

#include <stdio.h>
#include <stdint.h>
#include <string.h>
#include <stdlib.h>
#include <signal.h>

#include <sys/types.h>


#include <sys/socket.h>
#include <netdb.h>
#include <netinet/in.h>

#include <unistd.h>
#include <errno.h>
#include <limits.h>

#ifndef SOL_IP
# define SOL_IP IPPROTO_IP
# define SOL_IPV6 IPPROTO_IPV6
#endif

#ifndef IPV6_RECVPKTINFO
# define IPV6_RECVPKTINFO IPV6_PKTINFO
#endif

/** Default port for STUN binding discovery */
#define IPPORT_STUN  3478

#include "stun/stunagent.h"
#include "stund.h"

static const uint16_t known_attributes[] =  {
  0
};

/**
 * Creates a listening socket
 */
int listen_socket (int fam, int type, int proto, unsigned int port)
{
  int yes = 1;
  int fd = socket (fam, type, proto);
  union {
    struct sockaddr addr;
    struct sockaddr_in in;
    struct sockaddr_in6 in6;
    struct sockaddr_storage storage;
  } addr;
  if (fd == -1)
  {
    perror ("Error opening IP port");
    return -1;
  }
  if (fd < 3)
    goto error;

  memset (&addr, 0, sizeof (addr));
  addr.storage.ss_family = fam;
#ifdef HAVE_SA_LEN
  addr.storage.ss_len = sizeof (addr);
#endif

  switch (fam)
  {
    case AF_INET:
      addr.in.sin_port = htons (port);
      break;

    case AF_INET6:
#ifdef IPV6_V6ONLY
      setsockopt (fd, SOL_IPV6, IPV6_V6ONLY, &yes, sizeof (yes));
#endif
      addr.in6.sin6_port = htons (port);
      break;
  }

  if (bind (fd, (struct sockaddr *)&addr, sizeof (addr)))
  {
    perror ("Error opening IP port");
    goto error;
  }

  if ((type == SOCK_DGRAM) || (type == SOCK_RAW))
  {
    switch (fam)
    {
      case AF_INET:
#ifdef IP_RECVERR
        setsockopt (fd, SOL_IP, IP_RECVERR, &yes, sizeof (yes));
#endif
        break;

      case AF_INET6:
#ifdef IPV6_RECVERR
        setsockopt (fd, SOL_IPV6, IPV6_RECVERR, &yes, sizeof (yes));
#endif
        break;
    }
  }
  else
  {
    if (listen (fd, INT_MAX))
    {
      perror ("Error opening IP port");
      goto error;
    }
  }

  return fd;

error:
  close (fd);
  return -1;
}


/** Dequeue error from a socket if applicable */
static int recv_err (int fd)
{
#ifdef MSG_ERRQUEUE
  struct msghdr hdr;
  memset (&hdr, 0, sizeof (hdr));
  return recvmsg (fd, &hdr, MSG_ERRQUEUE) >= 0;
#else
  return 0;
#endif
}


/** Receives a message or dequeues an error from a socket */
ssize_t recv_safe (int fd, struct msghdr *msg)
{
  ssize_t len = recvmsg (fd, msg, 0);
  if (len == -1)
    recv_err (fd);
  else
  if (msg->msg_flags & MSG_TRUNC)
  {
    errno = EMSGSIZE;
    return -1;
  }

  return len;
}


/** Sends a message through a socket */
ssize_t send_safe (int fd, const struct msghdr *msg)
{
  ssize_t len;

  do
    len = sendmsg (fd, msg, 0);
  while ((len == -1) && (recv_err (fd) == 0));

  return len;
}


static int dgram_process (int sock, StunAgent *oldagent, StunAgent *newagent)
{
  struct sockaddr_storage addr;
  uint8_t buf[STUN_MAX_MESSAGE_SIZE];
  char ctlbuf[CMSG_SPACE (sizeof (struct in6_pktinfo))];
  struct iovec iov = { buf, sizeof (buf) };
  StunMessage request;
  StunMessage response;
  StunValidationStatus validation;
  StunAgent *agent = NULL;

  struct msghdr mh =
  {
    .msg_name = (struct sockaddr *)&addr,
    .msg_namelen = sizeof (addr),
    .msg_iov = &iov,
    .msg_iovlen = 1,
    .msg_control = ctlbuf,
    .msg_controllen = sizeof (ctlbuf)
  };

  size_t len = recv_safe (sock, &mh);
  if (len == (size_t)-1)
    return -1;

  validation = stun_agent_validate (newagent, &request, buf, len, NULL, 0);

  if (validation == STUN_VALIDATION_SUCCESS) {
    agent = newagent;
  }
  else {
    validation = stun_agent_validate (oldagent, &request, buf, len, NULL, 0);
    agent = oldagent;
  }

  /* Unknown attributes */
  if (validation == STUN_VALIDATION_UNKNOWN_REQUEST_ATTRIBUTE)
  {
    stun_agent_build_unknown_attributes_error (agent, &response, buf,
        sizeof (buf), &request);
    goto send_buf;
  }

  /* Mal-formatted packets */
  if (validation != STUN_VALIDATION_SUCCESS ||
      stun_message_get_class (&request) != STUN_REQUEST) {
    return -1;
  }

  switch (stun_message_get_method (&request))
  {
    case STUN_BINDING:
      stun_agent_init_response (agent, &response, buf, sizeof (buf), &request);
      if (stun_message_has_cookie (&request))
        stun_message_append_xor_addr (&response,
                              STUN_ATTRIBUTE_XOR_MAPPED_ADDRESS,
                              mh.msg_name, mh.msg_namelen);
      else
         stun_message_append_addr (&response, STUN_ATTRIBUTE_MAPPED_ADDRESS,
                          mh.msg_name, mh.msg_namelen);
      break;

    default:
      stun_agent_init_error (agent, &response, buf, sizeof (buf),
          &request, STUN_ERROR_BAD_REQUEST);
  }

  iov.iov_len = stun_agent_finish_message (agent, &response, NULL, 0);
send_buf:

  len = send_safe (sock, &mh);
  return (len < iov.iov_len) ? -1 : 0;
}


static int run (int family, int protocol, unsigned port)
{
  StunAgent oldagent;
  StunAgent newagent;
  int sock = listen_socket (family, SOCK_DGRAM, protocol, port);
  if (sock == -1)
    return -1;

  stun_agent_init (&oldagent, known_attributes,
      STUN_COMPATIBILITY_RFC3489, 0);
  stun_agent_init (&newagent, known_attributes,
      STUN_COMPATIBILITY_RFC5389, STUN_AGENT_USAGE_USE_FINGERPRINT);

  for (;;)
    dgram_process (sock, &oldagent, &newagent);
}


/* Pretty useless dummy signal handler...
 * But calling exit() is needed for gcov to work properly. */
static void exit_handler (int signum)
{
  (void)signum;
  exit (0);
}


int main (int argc, char *argv[])
{
  int family = AF_INET;
  unsigned port = IPPORT_STUN;

  for (;;)
  {
    int c = getopt (argc, argv, "46");
    if (c == EOF)
      break;

    switch (c)
    {
      case '4':
        family = AF_INET;
        break;

      case '6':
        family = AF_INET6;
        break;
    }
  }

  if (optind < argc)
    port = atoi (argv[optind++]);

  signal (SIGINT, exit_handler);
  signal (SIGTERM, exit_handler);
  return run (family, IPPROTO_UDP, port) ? EXIT_FAILURE : EXIT_SUCCESS;
}

#else
int main () {
  return 0;
}
#endif
