// Divx for encore50

#include <quicktime/colormodels.h>
#include <funcprotos.h>
#include <quicktime/quicktime.h>
#include <workarounds.h>
#include "encore50/encore.h"
#include "decore50/decore.h"

#include <pthread.h>
#include <stdlib.h>
#include <string.h>


typedef struct
{
	unsigned char *work_buffer;
	char *temp_frame;
	long buffer_size;
	int decode_initialized;
	int encode_initialized;
	int bitrate;
	long rc_period; // the intended rate control averaging period
	long rc_reaction_period; // the reation period for rate control
	long rc_reaction_ratio; // the ratio for down/up rate control
	long max_key_interval; // the maximum interval between key frames
	int max_quantizer; // the upper limit of the quantizer
	int min_quantizer; // the lower limit of the quantizer
	int quantizer;    // For vbr
	int quality; // the forward search range for motion estimation
	int fix_bitrate;
// Last frame decoded
	long last_frame;  
	int encode_handle;

        DEC_SET dec_set;

	DEC_PARAM dec_param;
	ENC_PARAM enc_param;

	int decode_handle;
// Must count pframes in VBR
	int p_count;
} quicktime_divx_codec_t;

static pthread_mutex_t encode_mutex;
static pthread_mutex_t decode_mutex;
static int mutex_initialized = 0;
static int decode_handle = 1;
static int encode_handle = 0;

static int delete_codec(quicktime_video_map_t *vtrack)
{
	quicktime_divx_codec_t *codec;

	codec = ((quicktime_codec_t*)vtrack->codec)->priv;
	if(codec->encode_initialized)
	{
		pthread_mutex_lock(&encode_mutex);
		encore(codec->encode_handle,
			ENC_OPT_RELEASE,
			0,
			0);
		pthread_mutex_unlock(&encode_mutex);
	}
	if(codec->decode_initialized)
	{
		pthread_mutex_lock(&decode_mutex);





/*
 * 		decore(codec->handle,
 * 			DEC_OPT_DESTROY,
 * 			0,
 * 			0);
 */



		decore(codec->decode_handle,
			DEC_OPT_RELEASE,
			0,
			0);



		pthread_mutex_unlock(&decode_mutex);
	}
	if(codec->temp_frame) free(codec->temp_frame);
	if(codec->work_buffer) free(codec->work_buffer);
	free(codec);
	return 0;
}

static int reads_colormodel(quicktime_t *file, 
		int colormodel, 
		int track)
{
	return (colormodel == BC_YUV420P);
}

static int writes_colormodel(quicktime_t *file, 
		int colormodel, 
		int track)
{
	return (colormodel == BC_RGB888 ||
		colormodel == BC_RGBA8888 ||
		colormodel == BC_RGB161616 ||
		colormodel == BC_RGBA16161616 ||
		colormodel == BC_YUV888 ||
		colormodel == BC_YUVA8888 ||
		colormodel == BC_YUV161616 ||
		colormodel == BC_YUVA16161616 ||
		colormodel == BC_YUV420P ||
		colormodel == BC_YUV422 ||
		colormodel == BC_COMPRESSED);
}








static void init_mutex()
{
	if(!mutex_initialized)
	{
		pthread_mutexattr_t attr;
		mutex_initialized = 1;
		pthread_mutexattr_init(&attr);
		pthread_mutex_init(&decode_mutex, &attr);
		pthread_mutex_init(&encode_mutex, &attr);
	}
}




// Determine of the compressed frame is a keyframe for direct copy
int quicktime_divx_is_key(unsigned char *data, long size)
{
	int result = 0;
	int i;

	for(i = 0; i < size - 5; i++)
	{
		if( data[i]     == 0x00 && 
			data[i + 1] == 0x00 &&
			data[i + 2] == 0x01 &&
			data[i + 3] == 0xb6)
		{
			if((data[i + 4] & 0xc0) == 0x0) 
				return 1;
			else
				return 0;
		}
	}
	
	return result;
}


// Test for VOL header in frame
int quicktime_divx_has_vol(unsigned char *data)
{
	if( data[0] == 0x00 &&
		data[1] == 0x00 &&
		data[2] == 0x01 &&
		data[3] == 0x00 &&
		data[4] == 0x00 &&
		data[5] == 0x00 &&
		data[6] == 0x01 &&
		data[7] == 0x20)
		return 1;
	else
		return 0;
}




static void putbits(unsigned char **data, 
	int *bit_pos, 
	uint64_t *bit_store, 
	int *total, 
	int count, 
	uint64_t value)
{
	value &= 0xffffffffffffffff >> (64 - count);

	while(64 - *bit_pos < count)
	{
		*(*data)++ = (*bit_store) >> 56;
		(*bit_store) <<= 8;
		(*bit_pos) -= 8;
	}

	(*bit_store) |= value << (64 - count - *bit_pos);
	(*bit_pos) += count;
	(*total) += count;
}


static void flushbits(unsigned char **data, 
	int *bit_pos, 
	uint64_t *bit_store)
{
//printf("flushbits %llx\n", (*bit_store));
	while((*bit_pos) > 0)
	{
		*(*data)++ = (*bit_store) >> 56;
		(*bit_store) <<= 8;
		(*bit_pos) -= 8;
	}
}




#define VO_START_CODE 		0x8      
#define VO_START_CODE_LENGTH	27
#define VOL_START_CODE 0x12             /* 25-MAR-97 JDL : according to WD2 */
#define VOL_START_CODE_LENGTH 28



int quicktime_divx_write_vol(unsigned char *data_start,
	int vol_width, 
	int vol_height, 
	int time_increment_resolution, 
	float frame_rate)
{
	int written = 0;
	int bits, fixed_vop_time_increment;
	unsigned char *data = data_start;
	int bit_pos;
	uint64_t bit_store;
	int i, j;

	bit_store = 0;
	bit_pos = 0;
	vol_width = (int)((float)vol_width / 16 + 0.5) * 16;
	vol_height = (int)((float)vol_height / 16 + 0.5) * 16;


	putbits(&data, 
		&bit_pos, 
		&bit_store, 
		&written,
		VO_START_CODE_LENGTH, VO_START_CODE);
	putbits(&data, 
		&bit_pos, 
		&bit_store, 
		&written,
		5, 0);				/* vo_id = 0								*/

	putbits(&data, 
		&bit_pos, 
		&bit_store, 
		&written,
		VOL_START_CODE_LENGTH, VOL_START_CODE);




	putbits(&data, 
		&bit_pos, 
		&bit_store, 
		&written,
		4, 0);				/* vol_id = 0								*/

	putbits(&data, 
		&bit_pos, 
		&bit_store, 
		&written,
		1, 0);				/* random_accessible_vol = 0				*/
	putbits(&data, 
		&bit_pos, 
		&bit_store, 
		&written,
		8, 1);				/* video_object_type_indication = 1 video	*/
	putbits(&data, 
		&bit_pos, 
		&bit_store, 
		&written,
		1, 1);				/* is_object_layer_identifier = 1			*/
	putbits(&data, 
		&bit_pos, 
		&bit_store, 
		&written,
		4, 2);				/* visual_object_layer_ver_id = 2			*/
	putbits(&data, 
		&bit_pos, 
		&bit_store, 
		&written,
		3, 1);				/* visual_object_layer_priority = 1			*/
	putbits(&data, 
		&bit_pos, 
		&bit_store, 
		&written,
		4, 1);				/* aspect_ratio_info = 1					*/







	putbits(&data, 
		&bit_pos, 
		&bit_store, 
		&written,
		1, 0);				/* vol_control_parameter = 0				*/
	putbits(&data, 
		&bit_pos, 
		&bit_store, 
		&written,
		2, 0);				/* vol_shape = 0 rectangular				*/
	putbits(&data, 
		&bit_pos, 
		&bit_store, 
		&written,
		1, 1);				/* marker									*/







	putbits(&data, 
		&bit_pos, 
		&bit_store, 
		&written,
		16, time_increment_resolution);
	putbits(&data, 
		&bit_pos, 
		&bit_store, 
		&written,
		1, 1);				/* marker									*/
	putbits(&data, 
		&bit_pos, 
		&bit_store, 
		&written,
		1, 1);				/* fixed_vop_rate = 1						*/


	bits = 1;
	while((1 << bits) < time_increment_resolution) bits++;

// Log calculation fails for some reason
//	bits = (int)ceil(log((double)time_increment_resolution) / log(2.0));
//    if (bits < 1) bits=1;

	fixed_vop_time_increment = 
		(int)(time_increment_resolution / frame_rate + 0.1);

	putbits(&data, 
		&bit_pos, 
		&bit_store, 
		&written,
		bits, fixed_vop_time_increment);

	putbits(&data, 
		&bit_pos, 
		&bit_store, 
		&written,
		1, 1);				/* marker									*/

	putbits(&data, 
		&bit_pos, 
		&bit_store, 
		&written,
		13, vol_width);
	putbits(&data, 
		&bit_pos, 
		&bit_store, 
		&written,
		1, 1);				/* marker									*/
	putbits(&data, 
		&bit_pos, 
		&bit_store, 
		&written,
		13, vol_height);
	putbits(&data, 
		&bit_pos, 
		&bit_store, 
		&written,
		1, 1);				/* marker									*/

	putbits(&data, 
		&bit_pos, 
		&bit_store, 
		&written,
		1, 0);				/* interlaced = 0							*/
	putbits(&data, 
		&bit_pos, 
		&bit_store, 
		&written,
		1, 1);				/* OBMC_disabled = 1						*/
	putbits(&data, 
		&bit_pos, 
		&bit_store, 
		&written,
		2, 0);				/* vol_sprite_usage = 0						*/
	putbits(&data, 
		&bit_pos, 
		&bit_store, 
		&written,
		1, 0);				/* not_8_bit = 0							*/

	putbits(&data, 
		&bit_pos, 
		&bit_store, 
		&written,
		1, 0);				/* vol_quant_type = 0						*/
	putbits(&data, 
		&bit_pos, 
		&bit_store, 
		&written,
		1, 0);				/* vol_quarter_pixel = 0					*/
	putbits(&data, 
		&bit_pos, 
		&bit_store, 
		&written,
		1, 1);				/* complexity_estimation_disabled = 1		*/
	putbits(&data, 
		&bit_pos, 
		&bit_store, 
		&written,
		1, 1);				/* resync_marker_disabled = 1				*/
	putbits(&data, 
		&bit_pos, 
		&bit_store, 
		&written,
		1, 0);				/* data_partitioning_enabled = 0			*/
	putbits(&data, 
		&bit_pos, 
		&bit_store, 
		&written,
		1, 0);				/* scalability = 0							*/

	flushbits(&data, 
		&bit_pos,
		&bit_store);



/*
 * for(i = 0; i < data - data_start; i++)
 * 	for(j = 0x80; j >= 1; j /= 2)
 * 		printf("%d", (data_start[i] & j) ? 1 : 0);
 * printf("\n");
 */



	return data - data_start;
}





#define READ_RAW(framenum) \
{ \
	quicktime_set_video_position(file, framenum, track); \
	bytes = quicktime_frame_size(file, framenum, track); \
	if(!codec->work_buffer || codec->buffer_size < bytes) \
	{ \
		if(codec->work_buffer) free(codec->work_buffer); \
		codec->buffer_size = bytes; \
		codec->work_buffer = calloc(1, codec->buffer_size + 100); \
	} \
	result = !quicktime_read_data(file, codec->work_buffer, bytes); \
}





static int decode(quicktime_t *file, unsigned char **row_pointers, int track)
{
	int i;
	int64_t bytes;
	int result = 0;
	quicktime_video_map_t *vtrack = &(file->vtracks[track]);
	quicktime_trak_t *trak = vtrack->track;
	quicktime_divx_codec_t *codec = ((quicktime_codec_t*)vtrack->codec)->priv;
	int width = trak->tkhd.track_width;
	int height = trak->tkhd.track_height;
	int width_i = (int)((float)width / 16 + 0.5) * 16;
	int height_i = (int)((float)height / 16 + 0.5) * 16;
	DEC_FRAME dec_frame;
	int use_temp = 0;
	int input_cmodel;
	char *bmp_pointers[3];
	long temp_position;




//printf("decode 1 %d\n", file->color_model);

	init_mutex();
	pthread_mutex_lock(&decode_mutex);












	if(!codec->decode_initialized)
	{
		DEC_MEM_REQS dec_mem_reqs;

// decore requires handle to be > 1
		codec->decode_handle = decode_handle++;
		codec->last_frame = -1;
		codec->dec_param.x_dim = width_i;
		codec->dec_param.y_dim = height_i;
		codec->dec_param.output_format = DEC_420;
		codec->dec_param.time_incr = 0;

		decore(codec->decode_handle, DEC_OPT_MEMORY_REQS, &codec->dec_param, &dec_mem_reqs);
		codec->dec_param.buffers.mp4_edged_ref_buffers = calloc(1, dec_mem_reqs.mp4_edged_ref_buffers_size);
		codec->dec_param.buffers.mp4_edged_for_buffers = calloc(1, dec_mem_reqs.mp4_edged_for_buffers_size);
		codec->dec_param.buffers.mp4_display_buffers = calloc(1, dec_mem_reqs.mp4_display_buffers_size);
		codec->dec_param.buffers.mp4_state = calloc(1, dec_mem_reqs.mp4_state_size);
		codec->dec_param.buffers.mp4_tables = calloc(1, dec_mem_reqs.mp4_tables_size);
		codec->dec_param.buffers.mp4_stream = calloc(1, dec_mem_reqs.mp4_stream_size);
                
                decore(codec->decode_handle, DEC_OPT_INIT, &codec->dec_param, NULL);

                decore(codec->decode_handle, DEC_OPT_SETPP, &codec->dec_set, NULL);

		codec->temp_frame = malloc(width_i * height_i * 3 / 2);


/*
 * 		codec->dec_param.width = width_i;
 * 		codec->dec_param.height = height_i;
 * 		decore(0, DEC_OPT_CREATE, &codec->dec_param, NULL);
 */




// Must decode frame with VOL header first but only the first frame in the
// sequence has a VOL header.
		temp_position = vtrack->current_position;
		READ_RAW(0);
		vtrack->current_position = temp_position;
		dec_frame.bitstream = codec->work_buffer;
		dec_frame.length = bytes;
		dec_frame.stride = width_i;
		dec_frame.render_flag = 0;
		dec_frame.bmp[0] = codec->temp_frame;
		dec_frame.bmp[1] = codec->temp_frame + width_i * height_i;
		dec_frame.bmp[2] = codec->temp_frame + width_i * height_i * 5 / 4;
		decore(codec->decode_handle, 0, &dec_frame, NULL);



		codec->decode_initialized = 1;
	}

//printf("decode 1\n");




	input_cmodel = BC_YUV420P;
	if(file->color_model == input_cmodel &&
		file->out_w == width_i &&
		file->out_h == height_i &&
		file->in_x == 0 &&
		file->in_y == 0 &&
		file->in_w == width_i &&
		file->in_h == height_i)
	{
//		dec_frame.dst = row_pointers[0];
		dec_frame.bmp[0] = row_pointers[0];
		dec_frame.bmp[1] = row_pointers[1];
		dec_frame.bmp[2] = row_pointers[2];
		use_temp = 0;
	}
	else
	{
//		dec_frame.dst = codec->temp_frame;
		dec_frame.bmp[0] = codec->temp_frame;
		dec_frame.bmp[1] = codec->temp_frame + width_i * height_i;
		dec_frame.bmp[2] = codec->temp_frame + width_i * height_i * 5 / 4;
		use_temp = 1;
	}








	dec_frame.stride = width_i;

//	dec_frame.render = 1;
//	dec_frame.colorspace = DEC_420;












//printf("decode 1 %d %d\n", codec->last_frame, vtrack->current_position);

	if(quicktime_has_keyframes(file, track) && 
		vtrack->current_position != codec->last_frame + 1)
	{
		int frame1, frame2 = vtrack->current_position;

		frame1 = quicktime_get_keyframe_before(file, vtrack->current_position, track);

		if(frame1 < codec->last_frame &&
			frame2 > codec->last_frame) frame1 = codec->last_frame + 1;


		while(frame1 < frame2)
		{
			quicktime_set_video_position(file, frame1, track);
			bytes = quicktime_frame_size(file, frame1, track);


			if(!codec->work_buffer || codec->buffer_size < bytes)
			{
				if(codec->work_buffer) free(codec->work_buffer);
				codec->buffer_size = bytes;
				codec->work_buffer = calloc(1, codec->buffer_size + 100);
			}

			quicktime_read_data(file, codec->work_buffer, bytes);

//printf("decode 2 %d %d\n", frame1, frame2);
			dec_frame.bitstream = codec->work_buffer;
			dec_frame.length = bytes;
			dec_frame.render_flag = 0;
			decore(codec->decode_handle, 0, &dec_frame, NULL);
			frame1++;
		}
		
		
		vtrack->current_position = frame2;
	}














	codec->last_frame = vtrack->current_position;
//printf("decode 1\n");


	READ_RAW(vtrack->current_position);



//printf("decode 1\n");

//printf("decode 1\n");


	dec_frame.bitstream = codec->work_buffer;
	dec_frame.length = bytes;
	dec_frame.render_flag = 1;

//printf("decode 1\n");
//	decore(codec->dec_param.handle, DEC_OPT_DECODE, &dec_frame, NULL);
	decore(codec->decode_handle, 0, &dec_frame, NULL);
//printf("decode 2\n");


	pthread_mutex_unlock(&decode_mutex);
//printf("decode 1 %d %d\n", use_temp, file->color_model);

	if(use_temp)
	{
		unsigned char **input_rows = malloc(sizeof(unsigned char*) * height_i);
		for(i = 0; i < height_i; i++)
			input_rows[i] = codec->temp_frame + width_i * 3;
		
		
		cmodel_transfer(row_pointers, /* Leave NULL if non existent */
			input_rows,
			row_pointers[0], /* Leave NULL if non existent */
			row_pointers[1],
			row_pointers[2],
			codec->temp_frame, /* Leave NULL if non existent */
			codec->temp_frame + width_i * height_i,
			codec->temp_frame + width_i * height_i + width_i * height_i / 4,
			file->in_x,        /* Dimensions to capture from input frame */
			file->in_y, 
			file->in_w, 
			file->in_h,
			0,       /* Dimensions to project on output frame */
			0, 
			file->out_w, 
			file->out_h,
			input_cmodel, 
			file->color_model,
			0,         /* When transfering BC_RGBA8888 to non-alpha this is the background color in 0xRRGGBB hex */
			width_i,       /* For planar use the luma rowspan */
			width);
		
		free(input_rows);
	}




//printf("decode 2\n");

	return result;
}



static int encode(quicktime_t *file, unsigned char **row_pointers, int track)
{
//printf("encode 1\n");
	int64_t offset = quicktime_position(file);
	quicktime_video_map_t *vtrack = &(file->vtracks[track]);
	quicktime_divx_codec_t *codec = ((quicktime_codec_t*)vtrack->codec)->priv;
	quicktime_trak_t *trak = vtrack->track;
	int width = trak->tkhd.track_width;
	int height = trak->tkhd.track_height;
	int width_i = (int)((float)width / 16 + 0.5) * 16;
	int height_i = (int)((float)height / 16 + 0.5) * 16;
	int result = 0;
	int i;
	ENC_FRAME encore_input;
	ENC_RESULT encore_result;
//printf("encode 1 %d %d\n", width_i, height_i);

	init_mutex();
	pthread_mutex_lock(&encode_mutex);

	if(!codec->encode_initialized)
	{
		codec->encode_initialized = 1;
		codec->encode_handle = encode_handle++;
		codec->enc_param.x_dim = width_i;
		codec->enc_param.y_dim = height_i;
		codec->enc_param.framerate = quicktime_frame_rate(file, track);
		codec->enc_param.bitrate = codec->bitrate;
		codec->enc_param.rc_period = codec->rc_period;
		codec->enc_param.rc_reaction_period = codec->rc_reaction_period;
		codec->enc_param.rc_reaction_ratio = codec->rc_reaction_ratio;
		codec->enc_param.max_quantizer = codec->max_quantizer;
		codec->enc_param.min_quantizer = codec->min_quantizer;
		codec->enc_param.max_key_interval = codec->max_key_interval;

		codec->enc_param.search_range = codec->quality * 3;
		if(codec->enc_param.search_range > 15) codec->enc_param.search_range = 15;

		encore(codec->encode_handle, ENC_OPT_INIT, &codec->enc_param, NULL);
	}

//printf("encode 1\n");

// Assume planes are contiguous
	if(file->color_model == BC_YUV420P &&
		width == width_i &&
		height == height_i)
	{
//printf("encode 1.1\n");
		encore_input.image = row_pointers[0];
	}
// Convert to YUV420P
	else
	{
		if(!codec->temp_frame)
		{
			codec->temp_frame = malloc(width_i * height_i * 3 / 2);
		}
//printf("encode 2 %d %d %d %d %d %d\n", file->color_model, width, height, width_i, height_i);
		
//printf("encode 1.5\n");
		cmodel_transfer(0, /* Leave NULL if non existent */
			row_pointers,
			codec->temp_frame, /* Leave NULL if non existent */
			codec->temp_frame + width_i * height_i,
			codec->temp_frame + width_i * height_i + width_i * height_i / 4,
			row_pointers[0], /* Leave NULL if non existent */
			row_pointers[1],
			row_pointers[2],
			0,        /* Dimensions to capture from input frame */
			0, 
			width, 
			height,
			0,       /* Dimensions to project on output frame */
			0, 
			width, 
			height,
			file->color_model, 
			BC_YUV420P,
			0,         /* When transfering BC_RGBA8888 to non-alpha this is the background color in 0xRRGGBB hex */
			width,       /* For planar use the luma rowspan */
			width_i);
		
//printf("encode 2\n");
	
		encore_input.image = codec->temp_frame;
	}
//printf("encode 3\n");

	if(!codec->work_buffer)
	{
		codec->buffer_size = width * height;
//		codec->buffer_size = 0x1000000;
		codec->work_buffer = malloc(codec->buffer_size);
	}

//printf("encode 4\n");

	memset(codec->work_buffer, 0, codec->buffer_size);
	encore_input.bitstream = codec->work_buffer;
	encore_input.length = 0;
	encore_input.quant = !codec->fix_bitrate ? codec->quantizer : 0;

	if(codec->p_count == 0)
	{
		codec->p_count++;
	}
	else
	{
		codec->p_count++;
		if(codec->p_count >= codec->max_key_interval)
			codec->p_count = 0;
	}

//printf("encode 5 %d\n", encore_input.quant);
	encore(codec->encode_handle,	
		0,	
		&encore_input,
		&encore_result);
	pthread_mutex_unlock(&encode_mutex);
//printf("encode 6\n");

	result = !quicktime_write_data(file, codec->work_buffer, encore_input.length);
	quicktime_update_tables(file,
						file->vtracks[track].track,
						offset,
						file->vtracks[track].current_chunk,
						file->vtracks[track].current_position,
						1,
						encore_input.length);

	file->vtracks[track].current_chunk++;

	if(encore_result.isKeyFrame)
		quicktime_insert_keyframe(file, file->vtracks[track].current_position, track);

//printf("encode 7\n");

//printf("encode 8\n");

	return result;
}

static int set_parameter(quicktime_t *file, 
		int track, 
		char *key, 
		void *value)
{
	quicktime_video_map_t *vtrack = &(file->vtracks[track]);
	quicktime_divx_codec_t *codec = ((quicktime_codec_t*)vtrack->codec)->priv;

	if(!strcasecmp(key, "divx_bitrate"))
		codec->bitrate = *(int*)value;
	else
	if(!strcasecmp(key, "divx_rc_period"))
		codec->rc_period = *(int*)value;
	else
	if(!strcasecmp(key, "divx_rc_reaction_ratio"))
		codec->rc_reaction_ratio = *(int*)value;
	else
	if(!strcasecmp(key, "divx_rc_reaction_period"))
		codec->rc_reaction_period = *(int*)value;
	else
	if(!strcasecmp(key, "divx_max_key_interval"))
		codec->max_key_interval = *(int*)value;
	else
	if(!strcasecmp(key, "divx_max_quantizer"))
		codec->max_quantizer = *(int*)value;
	else
	if(!strcasecmp(key, "divx_min_quantizer"))
		codec->min_quantizer = *(int*)value;
	else
	if(!strcasecmp(key, "divx_quantizer"))
		codec->quantizer = *(int*)value;
	else
	if(!strcasecmp(key, "divx_quality"))
		codec->quality = *(int*)value;
	else
	if(!strcasecmp(key, "divx_fix_bitrate"))
		codec->fix_bitrate = *(int*)value;
	else
        if(!strcasecmp(key, "divx_postprocess"))
		codec->dec_set.postproc_level = *(int*)value;
	return 0;
}


void quicktime_init_codec_divx(quicktime_video_map_t *vtrack)
{
	int i;
	quicktime_divx_codec_t *codec;

/* Init public items */
	((quicktime_codec_t*)vtrack->codec)->priv = calloc(1, sizeof(quicktime_divx_codec_t));
	((quicktime_codec_t*)vtrack->codec)->delete_vcodec = delete_codec;
	((quicktime_codec_t*)vtrack->codec)->decode_video = decode;
	((quicktime_codec_t*)vtrack->codec)->encode_video = encode;
	((quicktime_codec_t*)vtrack->codec)->reads_colormodel = reads_colormodel;
	((quicktime_codec_t*)vtrack->codec)->writes_colormodel = writes_colormodel;
	((quicktime_codec_t*)vtrack->codec)->set_parameter = set_parameter;
	
	codec = ((quicktime_codec_t*)vtrack->codec)->priv;
	
	codec->bitrate = 1000000;
	codec->rc_period = 50;
	codec->rc_reaction_ratio = 45;
	codec->rc_reaction_period = 10;
	codec->max_key_interval = 45;
	codec->max_quantizer = 31;
	codec->min_quantizer = 1;
	codec->quantizer = 10;
	codec->quality = 5;
	codec->fix_bitrate = 1;
}

