// Divx for encore50

#include "config.h"
#include <quicktime/colormodels.h>
#include <funcprotos.h>
#include <quicktime/quicktime.h>
#include <workarounds.h>
#include "encore50/encore.h"
// #include "decore50/decore.h"

#include <pthread.h>
#include <stdlib.h>
#include <string.h>


typedef struct
{
	unsigned char *work_buffer;
	char *temp_frame;
	long buffer_size;
	int encode_initialized;
	int bitrate;
	long rc_period; // the intended rate control averaging period
	long rc_reaction_period; // the reation period for rate control
	long rc_reaction_ratio; // the ratio for down/up rate control
	long max_key_interval; // the maximum interval between key frames
	int max_quantizer; // the upper limit of the quantizer
	int min_quantizer; // the lower limit of the quantizer
	int quantizer;    // For vbr
	int quality; // the forward search range for motion estimation
	int fix_bitrate;
// Last frame decoded
	long last_frame;  
	int encode_handle;


	ENC_PARAM enc_param;

//	int decode_handle;
// Must count pframes in VBR
	int p_count;
} quicktime_divx_codec_t;

static pthread_mutex_t encode_mutex;
static int mutex_initialized = 0;
static int encode_handle = 0;

static int delete_codec(quicktime_video_map_t *vtrack)
{
	quicktime_divx_codec_t *codec;

	codec = ((quicktime_codec_t*)vtrack->codec)->priv;
	if(codec->encode_initialized)
	{
		pthread_mutex_lock(&encode_mutex);
		encore(codec->encode_handle,
			ENC_OPT_RELEASE,
			0,
			0);
		pthread_mutex_unlock(&encode_mutex);
	}
	if(codec->temp_frame) free(codec->temp_frame);
	if(codec->work_buffer) free(codec->work_buffer);
	free(codec);
	return 0;
}
#if 0
static int reads_colormodel(quicktime_t *file, 
		int colormodel, 
		int track)
{
	return (colormodel == BC_YUV420P);
}
#endif
static int writes_colormodel(quicktime_t *file, 
		int colormodel, 
		int track)
{
	return (colormodel == BC_RGB888 ||
		colormodel == BC_RGBA8888 ||
		colormodel == BC_RGB161616 ||
		colormodel == BC_RGBA16161616 ||
		colormodel == BC_YUV888 ||
		colormodel == BC_YUVA8888 ||
		colormodel == BC_YUV161616 ||
		colormodel == BC_YUVA16161616 ||
		colormodel == BC_YUV420P ||
		colormodel == BC_YUV422 ||
		colormodel == BC_COMPRESSED);
}








static void init_mutex()
{
	if(!mutex_initialized)
	{
		pthread_mutexattr_t attr;
		mutex_initialized = 1;
		pthread_mutexattr_init(&attr);
		pthread_mutex_init(&encode_mutex, &attr);
	}
}




// Determine of the compressed frame is a keyframe for direct copy
int quicktime_divx_is_key(unsigned char *data, long size)
{
	int result = 0;
	int i;

	for(i = 0; i < size - 5; i++)
	{
		if( data[i]     == 0x00 && 
			data[i + 1] == 0x00 &&
			data[i + 2] == 0x01 &&
			data[i + 3] == 0xb6)
		{
			if((data[i + 4] & 0xc0) == 0x0) 
				return 1;
			else
				return 0;
		}
	}
	
	return result;
}


// Test for VOL header in frame
int quicktime_divx_has_vol(unsigned char *data)
{
	if( data[0] == 0x00 &&
		data[1] == 0x00 &&
		data[2] == 0x01 &&
		data[3] == 0x00 &&
		data[4] == 0x00 &&
		data[5] == 0x00 &&
		data[6] == 0x01 &&
		data[7] == 0x20)
		return 1;
	else
		return 0;
}




static void putbits(unsigned char **data, 
	int *bit_pos, 
	uint64_t *bit_store, 
	int *total, 
	int count, 
	uint64_t value)
{
	value &= 0xffffffffffffffffLL >> (64 - count);

	while(64 - *bit_pos < count)
	{
		*(*data)++ = (*bit_store) >> 56;
		(*bit_store) <<= 8;
		(*bit_pos) -= 8;
	}

	(*bit_store) |= value << (64 - count - *bit_pos);
	(*bit_pos) += count;
	(*total) += count;
}


static void flushbits(unsigned char **data, 
	int *bit_pos, 
	uint64_t *bit_store)
{
//printf("flushbits %llx\n", (*bit_store));
	while((*bit_pos) > 0)
	{
		*(*data)++ = (*bit_store) >> 56;
		(*bit_store) <<= 8;
		(*bit_pos) -= 8;
	}
}




#define VO_START_CODE 		0x8      
#define VO_START_CODE_LENGTH	27
#define VOL_START_CODE 0x12             /* 25-MAR-97 JDL : according to WD2 */
#define VOL_START_CODE_LENGTH 28



int quicktime_divx_write_vol(unsigned char *data_start,
	int vol_width, 
	int vol_height, 
	int time_increment_resolution, 
	double frame_rate)
{
	int written = 0;
	int bits, fixed_vop_time_increment;
	unsigned char *data = data_start;
	int bit_pos;
	uint64_t bit_store;

	bit_store = 0;
	bit_pos = 0;
	vol_width = (int)((float)vol_width / 16 + 0.5) * 16;
	vol_height = (int)((float)vol_height / 16 + 0.5) * 16;


	putbits(&data, 
		&bit_pos, 
		&bit_store, 
		&written,
		VO_START_CODE_LENGTH, VO_START_CODE);
	putbits(&data, 
		&bit_pos, 
		&bit_store, 
		&written,
		5, 0);				/* vo_id = 0								*/

	putbits(&data, 
		&bit_pos, 
		&bit_store, 
		&written,
		VOL_START_CODE_LENGTH, VOL_START_CODE);




	putbits(&data, 
		&bit_pos, 
		&bit_store, 
		&written,
		4, 0);				/* vol_id = 0								*/

	putbits(&data, 
		&bit_pos, 
		&bit_store, 
		&written,
		1, 0);				/* random_accessible_vol = 0				*/
	putbits(&data, 
		&bit_pos, 
		&bit_store, 
		&written,
		8, 1);				/* video_object_type_indication = 1 video	*/
	putbits(&data, 
		&bit_pos, 
		&bit_store, 
		&written,
		1, 1);				/* is_object_layer_identifier = 1			*/
	putbits(&data, 
		&bit_pos, 
		&bit_store, 
		&written,
		4, 2);				/* visual_object_layer_ver_id = 2			*/
	putbits(&data, 
		&bit_pos, 
		&bit_store, 
		&written,
		3, 1);				/* visual_object_layer_priority = 1			*/
	putbits(&data, 
		&bit_pos, 
		&bit_store, 
		&written,
		4, 1);				/* aspect_ratio_info = 1					*/







	putbits(&data, 
		&bit_pos, 
		&bit_store, 
		&written,
		1, 0);				/* vol_control_parameter = 0				*/
	putbits(&data, 
		&bit_pos, 
		&bit_store, 
		&written,
		2, 0);				/* vol_shape = 0 rectangular				*/
	putbits(&data, 
		&bit_pos, 
		&bit_store, 
		&written,
		1, 1);				/* marker									*/







	putbits(&data, 
		&bit_pos, 
		&bit_store, 
		&written,
		16, time_increment_resolution);
	putbits(&data, 
		&bit_pos, 
		&bit_store, 
		&written,
		1, 1);				/* marker									*/
	putbits(&data, 
		&bit_pos, 
		&bit_store, 
		&written,
		1, 1);				/* fixed_vop_rate = 1						*/


	bits = 1;
	while((1 << bits) < time_increment_resolution) bits++;

// Log calculation fails for some reason
//	bits = (int)ceil(log((double)time_increment_resolution) / log(2.0));
//    if (bits < 1) bits=1;

	fixed_vop_time_increment = 
		(int)(time_increment_resolution / frame_rate + 0.1);

	putbits(&data, 
		&bit_pos, 
		&bit_store, 
		&written,
		bits, fixed_vop_time_increment);

	putbits(&data, 
		&bit_pos, 
		&bit_store, 
		&written,
		1, 1);				/* marker									*/

	putbits(&data, 
		&bit_pos, 
		&bit_store, 
		&written,
		13, vol_width);
	putbits(&data, 
		&bit_pos, 
		&bit_store, 
		&written,
		1, 1);				/* marker									*/
	putbits(&data, 
		&bit_pos, 
		&bit_store, 
		&written,
		13, vol_height);
	putbits(&data, 
		&bit_pos, 
		&bit_store, 
		&written,
		1, 1);				/* marker									*/

	putbits(&data, 
		&bit_pos, 
		&bit_store, 
		&written,
		1, 0);				/* interlaced = 0							*/
	putbits(&data, 
		&bit_pos, 
		&bit_store, 
		&written,
		1, 1);				/* OBMC_disabled = 1						*/
	putbits(&data, 
		&bit_pos, 
		&bit_store, 
		&written,
		2, 0);				/* vol_sprite_usage = 0						*/
	putbits(&data, 
		&bit_pos, 
		&bit_store, 
		&written,
		1, 0);				/* not_8_bit = 0							*/

	putbits(&data, 
		&bit_pos, 
		&bit_store, 
		&written,
		1, 0);				/* vol_quant_type = 0						*/
	putbits(&data, 
		&bit_pos, 
		&bit_store, 
		&written,
		1, 0);				/* vol_quarter_pixel = 0					*/
	putbits(&data, 
		&bit_pos, 
		&bit_store, 
		&written,
		1, 1);				/* complexity_estimation_disabled = 1		*/
	putbits(&data, 
		&bit_pos, 
		&bit_store, 
		&written,
		1, 1);				/* resync_marker_disabled = 1				*/
	putbits(&data, 
		&bit_pos, 
		&bit_store, 
		&written,
		1, 0);				/* data_partitioning_enabled = 0			*/
	putbits(&data, 
		&bit_pos, 
		&bit_store, 
		&written,
		1, 0);				/* scalability = 0							*/

	flushbits(&data, 
		&bit_pos,
		&bit_store);



/*
 * for(i = 0; i < data - data_start; i++)
 * 	for(j = 0x80; j >= 1; j /= 2)
 * 		printf("%d", (data_start[i] & j) ? 1 : 0);
 * printf("\n");
 */



	return data - data_start;
}





#define READ_RAW(framenum) \
{ \
	quicktime_set_video_position(file, framenum, track); \
	bytes = quicktime_frame_size(file, framenum, track); \
	if(!codec->work_buffer || codec->buffer_size < bytes) \
	{ \
		if(codec->work_buffer) free(codec->work_buffer); \
		codec->buffer_size = bytes; \
		codec->work_buffer = calloc(1, codec->buffer_size + 100); \
	} \
	result = !quicktime_read_data(file, codec->work_buffer, bytes); \
}


static int encode(quicktime_t *file, unsigned char **row_pointers, int track)
{
//printf("encode 1\n");
	quicktime_video_map_t *vtrack = &(file->vtracks[track]);
	quicktime_divx_codec_t *codec = ((quicktime_codec_t*)vtrack->codec)->priv;
	quicktime_trak_t *trak = vtrack->track;
	int width = trak->tkhd.track_width;
	int height = trak->tkhd.track_height;
	int width_i = (int)((float)width / 16 + 0.5) * 16;
	int height_i = (int)((float)height / 16 + 0.5) * 16;
	int result = 0;
        quicktime_atom_t chunk_atom;
	ENC_FRAME encore_input;
	ENC_RESULT encore_result;
//printf("encode 1 %d %d\n", width_i, height_i);

	init_mutex();
	pthread_mutex_lock(&encode_mutex);

	if(!codec->encode_initialized)
	{
		codec->encode_initialized = 1;
		codec->encode_handle = encode_handle++;
		codec->enc_param.x_dim = width_i;
		codec->enc_param.y_dim = height_i;
		codec->enc_param.framerate = quicktime_frame_rate(file, track);
		codec->enc_param.bitrate = codec->bitrate;
		codec->enc_param.rc_period = codec->rc_period;
		codec->enc_param.rc_reaction_period = codec->rc_reaction_period;
		codec->enc_param.rc_reaction_ratio = codec->rc_reaction_ratio;
		codec->enc_param.max_quantizer = codec->max_quantizer;
		codec->enc_param.min_quantizer = codec->min_quantizer;
		codec->enc_param.max_key_interval = codec->max_key_interval;

		codec->enc_param.search_range = codec->quality * 3;
		if(codec->enc_param.search_range > 15) codec->enc_param.search_range = 15;

		encore(codec->encode_handle, ENC_OPT_INIT, &codec->enc_param, NULL);
	}

//printf("encode 1\n");

// Assume planes are contiguous
	if(file->vtracks[track].color_model == BC_YUV420P &&
		width == width_i &&
		height == height_i)
	{
//printf("encode 1.1\n");
		encore_input.image = row_pointers[0];
	}
// Convert to YUV420P
	else
	{
		if(!codec->temp_frame)
		{
			codec->temp_frame = malloc(width_i * height_i * 3 / 2);
		}
//printf("encode 2 %d %d %d %d %d %d\n", file->color_model, width, height, width_i, height_i);
		
//printf("encode 1.5\n");
		cmodel_transfer(0, /* Leave NULL if non existent */
			row_pointers,
			codec->temp_frame, /* Leave NULL if non existent */
			codec->temp_frame + width_i * height_i,
			codec->temp_frame + width_i * height_i + width_i * height_i / 4,
			row_pointers[0], /* Leave NULL if non existent */
			row_pointers[1],
			row_pointers[2],
			0,        /* Dimensions to capture from input frame */
			0, 
			width, 
			height,
			0,       /* Dimensions to project on output frame */
			0, 
			width, 
			height,
			file->vtracks[track].color_model, 
			BC_YUV420P,
			0,         /* When transfering BC_RGBA8888 to non-alpha this is the background color in 0xRRGGBB hex */
			width,       /* For planar use the luma rowspan */
			width_i);
		
//printf("encode 2\n");
	
		encore_input.image = codec->temp_frame;
	}
//printf("encode 3\n");

	if(!codec->work_buffer)
	{
		codec->buffer_size = width * height;
//		codec->buffer_size = 0x1000000;
		codec->work_buffer = malloc(codec->buffer_size);
	}

//printf("encode 4\n");

	memset(codec->work_buffer, 0, codec->buffer_size);
	encore_input.bitstream = codec->work_buffer;
	encore_input.length = 0;
	encore_input.quant = !codec->fix_bitrate ? codec->quantizer : 0;

	if(codec->p_count == 0)
	{
		codec->p_count++;
	}
	else
	{
		codec->p_count++;
		if(codec->p_count >= codec->max_key_interval)
			codec->p_count = 0;
	}

//printf("encode 5 %d\n", encore_input.quant);
	encore(codec->encode_handle,	
		0,	
		&encore_input,
		&encore_result);
	pthread_mutex_unlock(&encode_mutex);
//printf("encode 6\n");
        quicktime_write_chunk_header(file, trak, &chunk_atom);
	result = !quicktime_write_data(file, codec->work_buffer, encore_input.length);
        quicktime_write_chunk_footer(file, trak, vtrack->current_chunk, &chunk_atom, 1);
	file->vtracks[track].current_chunk++;

	if(encore_result.isKeyFrame)
		quicktime_insert_keyframe(file, file->vtracks[track].current_position, track);

//printf("encode 7\n");

//printf("encode 8\n");

	return result;
}

static int set_parameter(quicktime_t *file, 
		int track, 
		char *key, 
		void *value)
{
	quicktime_video_map_t *vtrack = &(file->vtracks[track]);
	quicktime_divx_codec_t *codec = ((quicktime_codec_t*)vtrack->codec)->priv;

	if(!strcasecmp(key, "divx_bitrate"))
		codec->bitrate = *(int*)value;
	else
	if(!strcasecmp(key, "divx_rc_period"))
		codec->rc_period = *(int*)value;
	else
	if(!strcasecmp(key, "divx_rc_reaction_ratio"))
		codec->rc_reaction_ratio = *(int*)value;
	else
	if(!strcasecmp(key, "divx_rc_reaction_period"))
		codec->rc_reaction_period = *(int*)value;
	else
	if(!strcasecmp(key, "divx_max_key_interval"))
		codec->max_key_interval = *(int*)value;
	else
	if(!strcasecmp(key, "divx_max_quantizer"))
		codec->max_quantizer = *(int*)value;
	else
	if(!strcasecmp(key, "divx_min_quantizer"))
		codec->min_quantizer = *(int*)value;
	else
	if(!strcasecmp(key, "divx_quantizer"))
		codec->quantizer = *(int*)value;
	else
	if(!strcasecmp(key, "divx_quality"))
		codec->quality = *(int*)value;
	else
	if(!strcasecmp(key, "divx_fix_bitrate"))
		codec->fix_bitrate = *(int*)value;
	return 0;
}


void quicktime_init_codec_divx(quicktime_video_map_t *vtrack)
{
	quicktime_divx_codec_t *codec;

/* Init public items */
	((quicktime_codec_t*)vtrack->codec)->priv = calloc(1, sizeof(quicktime_divx_codec_t));
	((quicktime_codec_t*)vtrack->codec)->delete_vcodec = delete_codec;
	((quicktime_codec_t*)vtrack->codec)->encode_video = encode;
	((quicktime_codec_t*)vtrack->codec)->writes_colormodel = writes_colormodel;
	((quicktime_codec_t*)vtrack->codec)->set_parameter = set_parameter;
	
	codec = ((quicktime_codec_t*)vtrack->codec)->priv;
	
	codec->bitrate = 1000000;
	codec->rc_period = 50;
	codec->rc_reaction_ratio = 45;
	codec->rc_reaction_period = 10;
	codec->max_key_interval = 45;
	codec->max_quantizer = 31;
	codec->min_quantizer = 1;
	codec->quantizer = 10;
	codec->quality = 5;
	codec->fix_bitrate = 1;
}

