/****************************************************************************
**
** This file is part of the LibreCAD project, a 2D CAD program
**
** Copyright (C) 2010 R. van Twisk (librecad@rvt.dds.nl)
** Copyright (C) 2001-2003 RibbonSoft. All rights reserved.
**
**
** This file may be distributed and/or modified under the terms of the
** GNU General Public License version 2 as published by the Free Software
** Foundation and appearing in the file gpl-2.0.txt included in the
** packaging of this file.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
**
** This copyright notice MUST APPEAR in all copies of the script!
**
**********************************************************************/
#include "qg_printpreviewoptions.h"
#include "rs_settings.h"

#include "rs_actionprintpreview.h"

/*
 *  Constructs a QG_PrintPreviewOptions as a child of 'parent', with the
 *  name 'name' and widget flags set to 'f'.
 */
QG_PrintPreviewOptions::QG_PrintPreviewOptions(QWidget* parent, Qt::WindowFlags fl)
    : QWidget(parent, fl)
{
    setupUi(this);
    defaultScales=0;
    init();
}

/*
 *  Destroys the object and frees any allocated resources
 */
QG_PrintPreviewOptions::~QG_PrintPreviewOptions()
{
    destroy();
    // no need to delete child widgets, Qt does it all for us
}

/*
 *  Sets the strings of the subwidgets using the current
 *  language.
 */
void QG_PrintPreviewOptions::languageChange()
{
    retranslateUi(this);
}

void QG_PrintPreviewOptions::init() {
    updateDisabled = false;
    imperialScales
            << "1\" = 1\""
            << "1\" = 2\""
            << "1\" = 4\""
            << "1\" = 8\""
            << "1\" = 16\""
            << "1\" = 32\""
            << "1\" = 64\""
            << "1\" = 128\""
            << "1\" = 256\"";

    metricScales
            << "1:1" << "1:2" << "1:5" << "1:10"
            << "1:20" << "1:25" << "1:50" << "1:75" << "1:100"
            << "1:125" << "1:150" << "1:175" << "1:200"
            << "1:250" << "1:500" << "1:750" << "1:1000"
            << "1:2500" << "1:5000" << "1:7500" << "1:10000"
            << "1:25000" << "1:50000" << "1:75000" << "1:100000"
            << "2:1" << "5:1" << "10:1"
            << "20:1" << "25:1" << "50:1" << "75:1" << "100:1"
            << "125:1" << "150:1" << "175:1" << "200:1"
            << "250:1" << "500:1" << "750:1" << "1000:1"
            << "2500:1" << "5000:1" << "7500:1" << "10000:1"
            << "25000:1" << "50000:1" << "75000:1" << "100000:1";
    RS_SETTINGS->beginGroup("/PrintPreview");
    updateDisabled= RS_SETTINGS->readNumEntry("/PrintScaleFixed", 0)!=0;
    blackWhiteDisabled= RS_SETTINGS->readNumEntry("/BlackWhiteSet", 0)!=0;
    RS_SETTINGS->endGroup();
    action=NULL;
}

void QG_PrintPreviewOptions::destroy() {
    RS_SETTINGS->beginGroup("/PrintPreview");
    RS_SETTINGS->writeEntry("/PrintScaleFixed", QString(updateDisabled?"1":"0"));
    RS_SETTINGS->writeEntry("/BlackWhiteSet", QString(blackWhiteDisabled?"1":"0"));
    if(updateDisabled){
        RS_SETTINGS->writeEntry("/PrintScaleValue",cbScale->currentText());
    }
    RS_SETTINGS->endGroup();
    action=NULL;
}

/** print scale fixed to saved value **/
void QG_PrintPreviewOptions::setScaleFixed(bool fixed)
{
    if(action != NULL) action->setPaperScaleFixed(fixed);
    updateDisabled=fixed;
    cbScale->setDisabled(fixed);
    bFit->setVisible(!fixed);
    if(cFixed->isChecked() != fixed) {
        cFixed->setChecked(fixed);
    }
    if(updateDisabled){
        RS_SETTINGS->beginGroup("/PrintPreview");
        RS_SETTINGS->writeEntry("/PrintScaleFixed", QString(updateDisabled?"1":"0"));
        RS_SETTINGS->writeEntry("/PrintScaleValue",cbScale->currentText());
        RS_SETTINGS->endGroup();
    }
}

void QG_PrintPreviewOptions::setAction(RS_ActionInterface* a, bool update) {
    if (a!=NULL && a->rtti()==RS2::ActionPrintPreview) {
        action = static_cast<RS_ActionPrintPreview*>(a);
        /** fixed scale **/
        if(update){
//                        std::cout<<__FILE__<<" : "<<__FUNCTION__<<" : line "<<__LINE__<<std::endl;
//                        std::cout<<"update="<<update<<" : updateDisabled="<<updateDisabled <<std::endl;
//                        std::cout<<"update="<<update<<" : action->getPaperScaleFixed()="<<action->getPaperScaleFixed() <<std::endl;
            if(updateDisabled||action->getPaperScaleFixed()){
                if(action->getPaperScaleFixed()==false){
                    RS_SETTINGS->beginGroup("/PrintPreview");
                    QString&& s=RS_SETTINGS->readEntry("/PrintScaleValue", "1:1");
                    RS_SETTINGS->endGroup();
                    updateDisabled=false;
                    scale(s);
                }
                updateDisabled=true;
                setScaleFixed(true);
            }else{
                double&& currScale = action->getScale();
                if(  currScale > RS_TOLERANCE)
                    scale (currScale);
                else
                    fit();
                updateScaleBox();
                setScaleFixed(false);
            }
        }else{
            bool btmp=updateDisabled;
            updateDisabled = true;
            cbScale->setDuplicatesEnabled(false);
            RS2::Unit u = action->getUnit();
            if (u==RS2::Inch) {
                cbScale->insertItems(0,imperialScales);
            } else {
                cbScale->insertItems(0,metricScales);
            }
            defaultScales=cbScale->count();
            updateScaleBox();

            updateDisabled = btmp;
            setScaleFixed(updateDisabled);
        }
        setBlackWhite(blackWhiteDisabled);

    } else {
        RS_DEBUG->print(RS_Debug::D_ERROR,
                        "QG_PrintPreviewOptions::setAction: wrong action type");
        action = NULL;
    }
}

void QG_PrintPreviewOptions::updateData() {
    if (action!=NULL) {
        /*
        action->setAngle(RS_Math::deg2rad(RS_Math::eval(leAngle->text())));
        action->setFactor(RS_Math::eval(leFactor->text()));
        */
    }
}

void QG_PrintPreviewOptions::center() {
    if (action!=NULL) {
        action->center();
    }
}

void QG_PrintPreviewOptions::setBlackWhite(bool on) {
    if (action!=NULL) {
        if(bBlackWhite->isChecked() != on) {
            bBlackWhite->setChecked(on);
        }
        blackWhiteDisabled = on;
        action->setBlackWhite(on);
    }
}

void QG_PrintPreviewOptions::fit() {
    if(updateDisabled) return;
    if (action!=NULL) {
        action->fit();
        updateScaleBox();
    }
}


void QG_PrintPreviewOptions::scale(const double& factor) {
    double&& f=fabs(factor); // do we need negative factor at all?
    if(action->setScale(f)){
        //        std::cout<<"QG_PrintPreviewOptions::scale(const QString& s): line: "<<__LINE__<<" s="<<factor<<std::endl;
        updateScaleBox(f);
    }
}

void QG_PrintPreviewOptions::scale(const QString& s0) {
    QString s;
    if (updateDisabled) {
        s=cbScale->currentText();
    }else{
        s=s0;
    }
    //    std::cout<<"QG_PrintPreviewOptions::scale(const QString& s): s="<<qPrintable(s)<<std::endl;
    double factor(1.);

    if (s.contains(':')) {
        bool ok1 = false;
        bool ok2 = false;
        int i = s.indexOf(':');
        double n = RS_Math::eval(s.left(i),&ok1);
        double d = RS_Math::eval(s.mid(i+1),&ok2);
        if (ok1 && ok2 && d>1.0e-6 ){
            factor=n/d;
        }
    } else if (s.contains('=')) {
        bool ok = false;
        int i = s.indexOf('=');
        double d = RS_Math::eval(s.mid(i+2, s.length()-i-3),&ok);
        if (ok && d>1.0e-6) {
            factor=1.0/d;
        }
    } else {
        bool ok = false;
        double f = RS_Math::eval(s, &ok);
        if (ok) {
            factor=f;
        }
    }
    factor=fabs(factor); // do we need negative factor at all?
//    if(factor<1.0e-6 || factor>1.0e6) {
//        if(factor>1.0e6){
//            action->printWarning(tr("Paper scale factor larger than 1.0e6"));
//        }else{
//            action->printWarning(tr("Paper scale factor smaller than 1.0e-6"));
//        }
//        return;
//    }
    if(action->setScale(factor)){
        //        std::cout<<"QG_PrintPreviewOptions::scale(const QString& s): line: "<<__LINE__<<" s="<<factor<<std::endl;
        updateScaleBox(factor);
    }
}

//update the scalebox to
void QG_PrintPreviewOptions::updateScaleBox(){
    updateScaleBox(action->getScale());
}

void QG_PrintPreviewOptions::updateScaleBox(const double& f){
    //    std::cout<<"void QG_PrintPreviewOptions::updateScaleBox() f="<<f<<std::endl;
    int i;
    for(i=0;i<cbScale->count();i++){
        QString s=cbScale->itemText(i);
        int i0 = s.indexOf(':');
        bool ok1,ok2;
        double n = s.left(i0).toDouble(&ok1);
        double d = s.mid(i0+1).toDouble(&ok2);
        if(! (ok1 && ok2)|| fabs(d)<RS_TOLERANCE) continue;

        if(fabs(f-n/d)<RS_TOLERANCE) break;
    }
    if(i<cbScale->count()){
        cbScale->setCurrentIndex(i);
        //        std::cout<<"QG_PrintPreviewOptions::updateScaleBox(): old: "<<qPrintable(cbScale->currentText())<<std::endl;
        return;
    }
    QString s("");
    if(f>1.){
        s=QString("%1:1").arg(f);
    }else{
        if(fabs(f)>RS_TOLERANCE) s=QString("1:%1").arg(1./f);
    }
    if(cbScale->count()>defaultScales){
        i=defaultScales;
        cbScale->setItemText(defaultScales,s);
    }else{
        cbScale->addItem(s);
        i=cbScale->count()-1;
    }
    cbScale->setCurrentIndex(i);
    //    std::cout<<"QG_PrintPreviewOptions::updateScaleBox(): new: "<<qPrintable(cbScale->currentText())<<std::endl;
}

//void QG_PrintPreviewOptions::updateScaleBox(const QString& s) {
//    if(cbScale->count()>defaultScales) std::cout<<"cbScale->last()="<<qPrintable(cbScale->itemText(defaultScales))<<std::endl;
//    std::cout<<"void QG_PrintPreviewOptions::updateScaleBox(QString) s="<<qPrintable(s)<<std::endl;
//    int index=cbScale->findText(s);
//    std::cout<<"QG_PrintPreviewOptions::updateScaleBox(): cbScale->findText(s)="<<index<<std::endl;
//    //add the current sccale, bug#343794
//    if(index>=defaultScales){
//        index=defaultScales;
//        cbScale->setItemText(defaultScales,s);
//    }else{
//        if(index<0){
//            cbScale->addItem(s);
//            index=cbScale->count() -1;
//        }
//    }

//    cbScale->setCurrentIndex(index);
//}
