/*
 * qemu_monitor_json.h: interaction with QEMU monitor console
 *
 * Copyright (C) 2006-2009, 2011-2013 Red Hat, Inc.
 * Copyright (C) 2006 Daniel P. Berrange
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library.  If not, see
 * <http://www.gnu.org/licenses/>.
 *
 * Author: Daniel P. Berrange <berrange@redhat.com>
 */


#ifndef QEMU_MONITOR_JSON_H
# define QEMU_MONITOR_JSON_H

# include "internal.h"

# include "qemu_monitor.h"
# include "virbitmap.h"

int qemuMonitorJSONIOProcess(qemuMonitorPtr mon,
                             const char *data,
                             size_t len,
                             qemuMonitorMessagePtr msg);

int qemuMonitorJSONHumanCommandWithFd(qemuMonitorPtr mon,
                                      const char *cmd,
                                      int scm_fd,
                                      char **reply);

int qemuMonitorJSONSetCapabilities(qemuMonitorPtr mon);

int qemuMonitorJSONStartCPUs(qemuMonitorPtr mon,
                             virConnectPtr conn);
int qemuMonitorJSONStopCPUs(qemuMonitorPtr mon);
int qemuMonitorJSONGetStatus(qemuMonitorPtr mon,
                             bool *running,
                             virDomainPausedReason *reason);

int qemuMonitorJSONSystemPowerdown(qemuMonitorPtr mon);
int qemuMonitorJSONSystemReset(qemuMonitorPtr mon);

int qemuMonitorJSONGetCPUInfo(qemuMonitorPtr mon,
                              int **pids);
int qemuMonitorJSONGetVirtType(qemuMonitorPtr mon,
                               int *virtType);
int qemuMonitorJSONGetBalloonInfo(qemuMonitorPtr mon,
                                  unsigned long long *currmem);
int qemuMonitorJSONGetMemoryStats(qemuMonitorPtr mon,
                                  virDomainMemoryStatPtr stats,
                                  unsigned int nr_stats);
int qemuMonitorJSONGetBlockInfo(qemuMonitorPtr mon,
                                virHashTablePtr table);
int qemuMonitorJSONGetBlockStatsInfo(qemuMonitorPtr mon,
                                     const char *dev_name,
                                     long long *rd_req,
                                     long long *rd_bytes,
                                     long long *rd_total_times,
                                     long long *wr_req,
                                     long long *wr_bytes,
                                     long long *wr_total_times,
                                     long long *flush_req,
                                     long long *flush_total_times,
                                     long long *errs);
int qemuMonitorJSONGetBlockStatsParamsNumber(qemuMonitorPtr mon,
                                             int *nparams);
int qemuMonitorJSONGetBlockExtent(qemuMonitorPtr mon,
                                  const char *dev_name,
                                  unsigned long long *extent);
int qemuMonitorJSONBlockResize(qemuMonitorPtr mon,
                               const char *devce,
                               unsigned long long size);

int qemuMonitorJSONSetVNCPassword(qemuMonitorPtr mon,
                                  const char *password);
int qemuMonitorJSONSetPassword(qemuMonitorPtr mon,
                               const char *protocol,
                               const char *password,
                               const char *action_if_connected);
int qemuMonitorJSONExpirePassword(qemuMonitorPtr mon,
                                  const char *protocol,
                                  const char *expire_time);
int qemuMonitorJSONSetBalloon(qemuMonitorPtr mon,
                              unsigned long newmem);
int qemuMonitorJSONSetCPU(qemuMonitorPtr mon, int cpu, int online);

int qemuMonitorJSONEjectMedia(qemuMonitorPtr mon,
                              const char *dev_name,
                              bool force);
int qemuMonitorJSONChangeMedia(qemuMonitorPtr mon,
                               const char *dev_name,
                               const char *newmedia,
                               const char *format);


int qemuMonitorJSONSaveVirtualMemory(qemuMonitorPtr mon,
                                     unsigned long long offset,
                                     size_t length,
                                     const char *path);
int qemuMonitorJSONSavePhysicalMemory(qemuMonitorPtr mon,
                                      unsigned long long offset,
                                      size_t length,
                                      const char *path);

int qemuMonitorJSONSetMigrationSpeed(qemuMonitorPtr mon,
                                     unsigned long bandwidth);

int qemuMonitorJSONSetMigrationDowntime(qemuMonitorPtr mon,
                                        unsigned long long downtime);

int qemuMonitorJSONGetMigrationCacheSize(qemuMonitorPtr mon,
                                         unsigned long long *cacheSize);
int qemuMonitorJSONSetMigrationCacheSize(qemuMonitorPtr mon,
                                         unsigned long long cacheSize);

int qemuMonitorJSONGetMigrationStatus(qemuMonitorPtr mon,
                                      qemuMonitorMigrationStatusPtr status);

int qemuMonitorJSONGetMigrationCapability(qemuMonitorPtr mon,
                                          qemuMonitorMigrationCaps capability);
int qemuMonitorJSONSetMigrationCapability(qemuMonitorPtr mon,
                                          qemuMonitorMigrationCaps capability);

int qemuMonitorJSONMigrate(qemuMonitorPtr mon,
                           unsigned int flags,
                           const char *uri);
int qemuMonitorJSONGetSpiceMigrationStatus(qemuMonitorPtr mon,
                                           bool *spice_migrated);


int qemuMonitorJSONMigrateCancel(qemuMonitorPtr mon);

int qemuMonitorJSONDump(qemuMonitorPtr mon,
                        const char *protocol);

int qemuMonitorJSONGraphicsRelocate(qemuMonitorPtr mon,
                                    int type,
                                    const char *hostname,
                                    int port,
                                    int tlsPort,
                                    const char *tlsSubject);

int qemuMonitorJSONAddUSBDisk(qemuMonitorPtr mon,
                              const char *path);

int qemuMonitorJSONAddUSBDeviceExact(qemuMonitorPtr mon,
                                     int bus,
                                     int dev);
int qemuMonitorJSONAddUSBDeviceMatch(qemuMonitorPtr mon,
                                     int vendor,
                                     int product);


int qemuMonitorJSONAddPCIHostDevice(qemuMonitorPtr mon,
                                    virDevicePCIAddress *hostAddr,
                                    virDevicePCIAddress *guestAddr);

int qemuMonitorJSONAddPCIDisk(qemuMonitorPtr mon,
                              const char *path,
                              const char *bus,
                              virDevicePCIAddress *guestAddr);

int qemuMonitorJSONAddPCINetwork(qemuMonitorPtr mon,
                                 const char *nicstr,
                                 virDevicePCIAddress *guestAddr);

int qemuMonitorJSONRemovePCIDevice(qemuMonitorPtr mon,
                                   virDevicePCIAddress *guestAddr);

int qemuMonitorJSONSendFileHandle(qemuMonitorPtr mon,
                                  const char *fdname,
                                  int fd);
int qemuMonitorJSONAddFd(qemuMonitorPtr mon, int fdset, int fd,
                         const char *name);

int qemuMonitorJSONCloseFileHandle(qemuMonitorPtr mon,
                                   const char *fdname);
int qemuMonitorJSONRemoveFd(qemuMonitorPtr mon, int fdset, int fd);

int qemuMonitorJSONAddNetdev(qemuMonitorPtr mon,
                             const char *netdevstr);

int qemuMonitorJSONRemoveNetdev(qemuMonitorPtr mon,
                                const char *alias);

int qemuMonitorJSONGetPtyPaths(qemuMonitorPtr mon,
                               virHashTablePtr paths);

int qemuMonitorJSONAttachPCIDiskController(qemuMonitorPtr mon,
                                           const char *bus,
                                           virDevicePCIAddress *guestAddr);

int qemuMonitorJSONGetAllPCIAddresses(qemuMonitorPtr mon,
                                      qemuMonitorPCIAddress **addrs);

int qemuMonitorJSONAddDevice(qemuMonitorPtr mon,
                             const char *devicestr);

int qemuMonitorJSONDelDevice(qemuMonitorPtr mon,
                             const char *devalias);

int qemuMonitorJSONAddDrive(qemuMonitorPtr mon,
                            const char *drivestr);

int qemuMonitorJSONDriveDel(qemuMonitorPtr mon,
                            const char *drivestr);

int qemuMonitorJSONSetDrivePassphrase(qemuMonitorPtr mon,
                                      const char *alias,
                                      const char *passphrase);

int qemuMonitorJSONCreateSnapshot(qemuMonitorPtr mon, const char *name);
int qemuMonitorJSONLoadSnapshot(qemuMonitorPtr mon, const char *name);
int qemuMonitorJSONDeleteSnapshot(qemuMonitorPtr mon, const char *name);

int qemuMonitorJSONDiskSnapshot(qemuMonitorPtr mon,
                                virJSONValuePtr actions,
                                const char *device,
                                const char *file,
                                const char *format,
                                bool reuse)
    ATTRIBUTE_NONNULL(1) ATTRIBUTE_NONNULL(3)
    ATTRIBUTE_NONNULL(4) ATTRIBUTE_NONNULL(5);
int qemuMonitorJSONTransaction(qemuMonitorPtr mon, virJSONValuePtr actions)
    ATTRIBUTE_NONNULL(1) ATTRIBUTE_NONNULL(2);
int qemuMonitorJSONDriveMirror(qemuMonitorPtr mon,
                               const char *device,
                               const char *file,
                               const char *format,
                               unsigned long long speed,
                               unsigned int flags)
    ATTRIBUTE_NONNULL(1) ATTRIBUTE_NONNULL(2) ATTRIBUTE_NONNULL(3);
int qemuMonitorJSONDrivePivot(qemuMonitorPtr mon,
                              const char *device,
                              const char *file,
                              const char *format)
    ATTRIBUTE_NONNULL(1) ATTRIBUTE_NONNULL(2) ATTRIBUTE_NONNULL(3);

int qemuMonitorJSONBlockCommit(qemuMonitorPtr mon,
                               const char *device,
                               const char *top,
                               const char *base,
                               unsigned long long bandwidth)
    ATTRIBUTE_NONNULL(1) ATTRIBUTE_NONNULL(2) ATTRIBUTE_NONNULL(3);

int qemuMonitorJSONArbitraryCommand(qemuMonitorPtr mon,
                                    const char *cmd_str,
                                    char **reply_str,
                                    bool hmp);

int qemuMonitorJSONInjectNMI(qemuMonitorPtr mon);

int qemuMonitorJSONSendKey(qemuMonitorPtr mon,
                           unsigned int holdtime,
                           unsigned int *keycodes,
                           unsigned int nkeycodes);

int qemuMonitorJSONScreendump(qemuMonitorPtr mon,
                              const char *file);

int qemuMonitorJSONBlockJob(qemuMonitorPtr mon,
                            const char *device,
                            const char *base,
                            unsigned long long speed,
                            virDomainBlockJobInfoPtr info,
                            qemuMonitorBlockJobCmd mode,
                            bool modern)
    ATTRIBUTE_NONNULL(1) ATTRIBUTE_NONNULL(2);

int qemuMonitorJSONSetLink(qemuMonitorPtr mon,
                           const char *name,
                           enum virDomainNetInterfaceLinkState state);

int qemuMonitorJSONOpenGraphics(qemuMonitorPtr mon,
                                const char *protocol,
                                const char *fdname,
                                bool skipauth);

int qemuMonitorJSONSetBlockIoThrottle(qemuMonitorPtr mon,
                                      const char *device,
                                      virDomainBlockIoTuneInfoPtr info);

int qemuMonitorJSONGetBlockIoThrottle(qemuMonitorPtr mon,
                                      const char *device,
                                      virDomainBlockIoTuneInfoPtr reply);

int qemuMonitorJSONSystemWakeup(qemuMonitorPtr mon);

int qemuMonitorJSONGetVersion(qemuMonitorPtr mon,
                              int *major,
                              int *minor,
                              int *micro,
                              char **package)
    ATTRIBUTE_NONNULL(2) ATTRIBUTE_NONNULL(3) ATTRIBUTE_NONNULL(4);

int qemuMonitorJSONGetMachines(qemuMonitorPtr mon,
                               qemuMonitorMachineInfoPtr **machines)
    ATTRIBUTE_NONNULL(2);

int qemuMonitorJSONGetCPUDefinitions(qemuMonitorPtr mon,
                                     char ***cpus)
    ATTRIBUTE_NONNULL(2);

int qemuMonitorJSONGetCommands(qemuMonitorPtr mon,
                               char ***commands)
    ATTRIBUTE_NONNULL(2);
int qemuMonitorJSONGetEvents(qemuMonitorPtr mon,
                             char ***events)
    ATTRIBUTE_NONNULL(2);
int qemuMonitorJSONGetCommandLineOptionParameters(qemuMonitorPtr mon,
                                                  const char *option,
                                                  char ***params)
    ATTRIBUTE_NONNULL(2) ATTRIBUTE_NONNULL(3);

int qemuMonitorJSONGetKVMState(qemuMonitorPtr mon,
                               bool *enabled,
                               bool *present)
    ATTRIBUTE_NONNULL(2) ATTRIBUTE_NONNULL(3);

int qemuMonitorJSONGetObjectTypes(qemuMonitorPtr mon,
                                  char ***types)
    ATTRIBUTE_NONNULL(2);
int qemuMonitorJSONGetObjectProps(qemuMonitorPtr mon,
                                  const char *type,
                                  char ***props)
    ATTRIBUTE_NONNULL(2) ATTRIBUTE_NONNULL(3);
char *qemuMonitorJSONGetTargetArch(qemuMonitorPtr mon);

int qemuMonitorJSONNBDServerStart(qemuMonitorPtr mon,
                                  const char *host,
                                  unsigned int port);
int qemuMonitorJSONNBDServerAdd(qemuMonitorPtr mon,
                                const char *deviceID,
                                bool writable);
int qemuMonitorJSONNBDServerStop(qemuMonitorPtr mon);
int qemuMonitorJSONGetTPMModels(qemuMonitorPtr mon,
                                char ***tpmmodels)
    ATTRIBUTE_NONNULL(2);

int qemuMonitorJSONGetTPMTypes(qemuMonitorPtr mon,
                               char ***tpmtypes)
    ATTRIBUTE_NONNULL(2);

#endif /* QEMU_MONITOR_JSON_H */
