/*
 * qemu_conf.h: QEMU configuration management
 *
 * Copyright (C) 2006-2007, 2009-2013 Red Hat, Inc.
 * Copyright (C) 2006 Daniel P. Berrange
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library.  If not, see
 * <http://www.gnu.org/licenses/>.
 */

#pragma once

#include <unistd.h>

#include "virebtables.h"
#include "internal.h"
#include "capabilities.h"
#include "network_conf.h"
#include "domain_conf.h"
#include "checkpoint_conf.h"
#include "snapshot_conf.h"
#include "domain_event.h"
#include "virthread.h"
#include "security/security_manager.h"
#include "virpci.h"
#include "virusb.h"
#include "virscsi.h"
#include "cpu_conf.h"
#include "driver.h"
#include "virportallocator.h"
#include "vircommand.h"
#include "virthreadpool.h"
#include "locking/lock_manager.h"
#include "qemu_capabilities.h"
#include "virclosecallbacks.h"
#include "virhostdev.h"
#include "virfile.h"
#include "virfilecache.h"
#include "virfirmware.h"

#define QEMU_DRIVER_NAME "QEMU"

typedef struct _virQEMUDriver virQEMUDriver;
typedef virQEMUDriver *virQEMUDriverPtr;

typedef struct _virQEMUDriverConfig virQEMUDriverConfig;
typedef virQEMUDriverConfig *virQEMUDriverConfigPtr;

/* Main driver config. The data in these object
 * instances is immutable, so can be accessed
 * without locking. Threads must, however, hold
 * a valid reference on the object to prevent it
 * being released while they use it.
 *
 * eg
 *  g_autoptr(virQEMUDriverConfig) cfg = virQEMUDriverGetConfig(driver);
 *
 *  ...do stuff with 'cfg'..
 */
struct _virQEMUDriverConfig {
    virObject parent;

    char *uri;

    uid_t user;
    gid_t group;
    bool dynamicOwnership;

    virBitmapPtr namespaces;
    bool rememberOwner;

    int cgroupControllers;
    char **cgroupDeviceACL;

    /* These five directories are ones libvirtd uses (so must be root:root
     * to avoid security risk from QEMU processes */
    char *configBaseDir;
    char *configDir;
    char *autostartDir;
    char *logDir;
    char *swtpmLogDir;
    char *stateDir;
    char *swtpmStateDir;
    char *slirpStateDir;
    char *dbusStateDir;
    /* These two directories are ones QEMU processes use (so must match
     * the QEMU user/group */
    char *libDir;
    char *cacheDir;
    char *saveDir;
    char *snapshotDir;
    char *checkpointDir;
    char *channelTargetDir;
    char *nvramDir;
    char *swtpmStorageDir;

    char *defaultTLSx509certdir;
    bool defaultTLSx509certdirPresent;
    bool defaultTLSx509verify;
    bool defaultTLSx509verifyPresent;
    char *defaultTLSx509secretUUID;

    bool vncAutoUnixSocket;
    bool vncTLS;
    bool vncTLSx509verify;
    bool vncTLSx509verifyPresent;
    bool vncSASL;
    char *vncTLSx509certdir;
    char *vncTLSx509secretUUID;
    char *vncListen;
    char *vncPassword;
    char *vncSASLdir;

    bool spiceTLS;
    char *spiceTLSx509certdir;
    bool spiceSASL;
    char *spiceSASLdir;
    char *spiceListen;
    char *spicePassword;
    bool spiceAutoUnixSocket;

    bool chardevTLS;
    char *chardevTLSx509certdir;
    bool chardevTLSx509verify;
    bool chardevTLSx509verifyPresent;
    char *chardevTLSx509secretUUID;

    char *migrateTLSx509certdir;
    bool migrateTLSx509verify;
    bool migrateTLSx509verifyPresent;
    char *migrateTLSx509secretUUID;
    bool migrateTLSForce;

    char *backupTLSx509certdir;
    bool backupTLSx509verify;
    bool backupTLSx509verifyPresent;
    char *backupTLSx509secretUUID;

    bool vxhsTLS;
    char *vxhsTLSx509certdir;
    char *vxhsTLSx509secretUUID;

    bool nbdTLS;
    char *nbdTLSx509certdir;
    char *nbdTLSx509secretUUID;

    unsigned int remotePortMin;
    unsigned int remotePortMax;

    unsigned int webSocketPortMin;
    unsigned int webSocketPortMax;

    virHugeTLBFSPtr hugetlbfs;
    size_t nhugetlbfs;

    char *bridgeHelperName;
    char *prHelperName;
    char *slirpHelperName;
    char *dbusDaemonName;

    bool macFilter;

    bool relaxedACS;
    bool vncAllowHostAudio;
    bool nogfxAllowHostAudio;
    bool setProcessName;

    unsigned int maxProcesses;
    unsigned int maxFiles;
    unsigned int maxThreadsPerProc;
    unsigned long long maxCore;
    bool dumpGuestCore;

    unsigned int maxQueuedJobs;

    char **securityDriverNames;
    bool securityDefaultConfined;
    bool securityRequireConfined;

    char *saveImageFormat;
    char *dumpImageFormat;
    char *snapshotImageFormat;

    char *autoDumpPath;
    bool autoDumpBypassCache;
    bool autoStartBypassCache;

    char *lockManagerName;

    int keepAliveInterval;
    unsigned int keepAliveCount;

    int seccompSandbox;

    char *migrateHost;
    /* The default for -incoming */
    char *migrationAddress;
    unsigned int migrationPortMin;
    unsigned int migrationPortMax;

    bool logTimestamp;
    bool stdioLogD;

    virFirmwarePtr *firmwares;
    size_t nfirmwares;
    unsigned int glusterDebugLevel;
    bool virtiofsdDebug;

    char *memoryBackingDir;

    uid_t swtpm_user;
    gid_t swtpm_group;

    char **capabilityfilters;
};

G_DEFINE_AUTOPTR_CLEANUP_FUNC(virQEMUDriverConfig, virObjectUnref);


/* Main driver state */
struct _virQEMUDriver {
    virMutex lock;

    /* Require lock to get reference on 'config',
     * then lockless thereafter */
    virQEMUDriverConfigPtr config;

    /* pid file FD, ensures two copies of the driver can't use the same root */
    int lockFD;

    /* Immutable pointer, self-locking APIs */
    virThreadPoolPtr workerPool;

    /* Atomic increment only */
    int lastvmid;

    /* Atomic inc/dec only */
    unsigned int nactive;

    /* Immutable values */
    bool privileged;
    char *embeddedRoot;

    /* Immutable pointers. Caller must provide locking */
    virStateInhibitCallback inhibitCallback;
    void *inhibitOpaque;

    /* Immutable pointer, self-locking APIs */
    virDomainObjListPtr domains;

    /* Immutable pointer */
    char *qemuImgBinary;

    /* Immutable pointer, lockless APIs. Pointless abstraction */
    ebtablesContext *ebtables;

    /* Require lock to get a reference on the object,
     * lockless access thereafter
     */
    virCapsPtr caps;

    /* Lazy initialized on first use, immutable thereafter.
     * Require lock to get the pointer & do optional initialization
     */
    virCapsHostNUMAPtr hostnuma;

    /* Lazy initialized on first use, immutable thereafter.
     * Require lock to get the pointer & do optional initialization
     */
    virCPUDefPtr hostcpu;

    /* Immutable value */
    virArch hostarch;

    /* Immutable pointer, Immutable object */
    virDomainXMLOptionPtr xmlopt;

    /* Immutable pointer, self-locking APIs */
    virFileCachePtr qemuCapsCache;

    /* Immutable pointer, self-locking APIs */
    virObjectEventStatePtr domainEventState;

    /* Immutable pointer. self-locking APIs */
    virSecurityManagerPtr securityManager;

    virHostdevManagerPtr hostdevMgr;

    /* Immutable pointer. Unsafe APIs. XXX */
    GHashTable *sharedDevices;

    /* Immutable pointer, immutable object */
    virPortAllocatorRangePtr remotePorts;

    /* Immutable pointer, immutable object */
    virPortAllocatorRangePtr webSocketPorts;

    /* Immutable pointer, immutable object */
    virPortAllocatorRangePtr migrationPorts;

    /* Immutable pointer, lockless APIs */
    virSysinfoDefPtr hostsysinfo;

    /* Immutable pointer. lockless access */
    virLockManagerPluginPtr lockManager;

    /* Immutable pointer, self-clocking APIs */
    virCloseCallbacksPtr closeCallbacks;

    /* Immutable pointer, self-locking APIs */
    virHashAtomicPtr migrationErrors;
};

virQEMUDriverConfigPtr virQEMUDriverConfigNew(bool privileged,
                                              const char *root);

int virQEMUDriverConfigLoadFile(virQEMUDriverConfigPtr cfg,
                                const char *filename,
                                bool privileged);

int
virQEMUDriverConfigValidate(virQEMUDriverConfigPtr cfg);

int
virQEMUDriverConfigSetDefaults(virQEMUDriverConfigPtr cfg);

virQEMUDriverConfigPtr virQEMUDriverGetConfig(virQEMUDriverPtr driver);

virCapsHostNUMAPtr virQEMUDriverGetHostNUMACaps(virQEMUDriverPtr driver);
virCPUDefPtr virQEMUDriverGetHostCPU(virQEMUDriverPtr driver);
virCapsPtr virQEMUDriverCreateCapabilities(virQEMUDriverPtr driver);
virCapsPtr virQEMUDriverGetCapabilities(virQEMUDriverPtr driver,
                                        bool refresh);

virDomainCapsPtr
virQEMUDriverGetDomainCapabilities(virQEMUDriverPtr driver,
                                   virQEMUCapsPtr qemuCaps,
                                   const char *machine,
                                   virArch arch,
                                   virDomainVirtType virttype);

typedef struct _qemuSharedDeviceEntry qemuSharedDeviceEntry;
typedef qemuSharedDeviceEntry *qemuSharedDeviceEntryPtr;

bool qemuSharedDeviceEntryDomainExists(qemuSharedDeviceEntryPtr entry,
                                       const char *name,
                                       int *idx)
    ATTRIBUTE_NONNULL(1) ATTRIBUTE_NONNULL(2);

char *qemuGetSharedDeviceKey(const char *disk_path)
    ATTRIBUTE_NONNULL(1);

void qemuSharedDeviceEntryFree(void *payload);

int qemuAddSharedDisk(virQEMUDriverPtr driver,
                      virDomainDiskDefPtr disk,
                      const char *name)
    ATTRIBUTE_NONNULL(1) ATTRIBUTE_NONNULL(2) ATTRIBUTE_NONNULL(3);

int qemuAddSharedDevice(virQEMUDriverPtr driver,
                        virDomainDeviceDefPtr dev,
                        const char *name)
    ATTRIBUTE_NONNULL(1) ATTRIBUTE_NONNULL(2) ATTRIBUTE_NONNULL(3);

int qemuRemoveSharedDevice(virQEMUDriverPtr driver,
                           virDomainDeviceDefPtr dev,
                           const char *name)
    ATTRIBUTE_NONNULL(1) ATTRIBUTE_NONNULL(2) ATTRIBUTE_NONNULL(3);

int qemuRemoveSharedDisk(virQEMUDriverPtr driver,
                         virDomainDiskDefPtr disk,
                         const char *name)
    ATTRIBUTE_NONNULL(1) ATTRIBUTE_NONNULL(2) ATTRIBUTE_NONNULL(3);

int qemuSetUnprivSGIO(virDomainDeviceDefPtr dev);

int qemuDriverAllocateID(virQEMUDriverPtr driver);
virDomainXMLOptionPtr virQEMUDriverCreateXMLConf(virQEMUDriverPtr driver,
                                                 const char *defsecmodel);

int qemuTranslateSnapshotDiskSourcePool(virDomainSnapshotDiskDefPtr def);

char * qemuGetBaseHugepagePath(virQEMUDriverPtr driver,
                               virHugeTLBFSPtr hugepage);
char * qemuGetDomainHugepagePath(virQEMUDriverPtr driver,
                                 const virDomainDef *def,
                                 virHugeTLBFSPtr hugepage);

int qemuGetDomainHupageMemPath(virQEMUDriverPtr driver,
                               const virDomainDef *def,
                               unsigned long long pagesize,
                               char **memPath);

int qemuGetMemoryBackingDomainPath(virQEMUDriverPtr driver,
                                   const virDomainDef *def,
                                   char **path);
int qemuGetMemoryBackingPath(virQEMUDriverPtr driver,
                             const virDomainDef *def,
                             const char *alias,
                             char **memPath);
