/*
 *  linux-identd - an identification daemon for Linux
 * 
 *  Copyright (c) 2001-2003 by Per Liden <per@fukt.bth.se>
 * 
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, 
 *  USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <errno.h>
#include <string.h>
#include <signal.h>
#include <syslog.h>
#include <pwd.h>
#include <netdb.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/wait.h>
#include <sys/types.h>
#include <netinet/in.h>
#include <arpa/inet.h>

#define PORT                    113
#define BACKLOG                 8
#define TIMEOUT                 60
#define USER                    "nobody"

#define PROC_FILE               "/proc/net/tcp"
#define PROC_FILE_MAXLINE       256

#define TCP_INVALID_PORT        0
#define TCP_MIN_PORT            1
#define TCP_MAX_PORT            65535

#define LOOKUP_OK               0
#define LOOKUP_NOT_FOUND        1
#define LOOKUP_ERROR            2

#define MAX_RESPONSE            200
#define MAX_REQUEST             100

#ifdef DEBUG
#define debug(args...)          fprintf(stderr, args)
#else
#define debug(args...)
#endif

/* glibc 2.1.x doesn't declare in_addr_t */
#if defined(__GLIBC_MINOR__) && (__GLIBC_MINOR__ < 2)
typedef uint32_t in_addr_t;
#endif

static int log_connections = 0;

static void session_terminate(int s)
{
        while(wait(0) > 0);
}

static void session_timeout(int sig)
{
        exit(EXIT_SUCCESS);
}

static int read_request(int fd, char* buffer, int maxlen)
{
        int bytesread = 0;

        while (bytesread < maxlen - 1) {
                char c;
                if (read(fd, &c, 1) != 1)
                        return 0;
   
                if (c == '\n')
                        break;
      
                buffer[bytesread++] = c;
        }
   
        if (bytesread > 0 && buffer[bytesread - 1] == '\r')
                bytesread--;

        buffer[bytesread] = '\0';

        return 1;
}

static int write_response(int fd, char* buffer, int len)
{
        int byteswritten = 0;

        while (byteswritten < len) {
                int retval = write(fd, buffer + byteswritten, len - byteswritten);
                if (retval <= 0)
                        return 0;

                byteswritten += retval;
        }
   
        return 1;
}

static int lookup_uid(in_addr_t l_addr, unsigned int l_port,
                      in_addr_t r_addr, unsigned int r_port, uid_t* uid)
{
        FILE* file;

        if ((file = fopen(PROC_FILE, "r")) == 0) {
                syslog(LOG_ERR, "Failed to open %s: %s", PROC_FILE, strerror(errno));
                return LOOKUP_ERROR;
        }

        debug("Looking for: %.8X:%.4X %.8X:%.4X\n", l_addr, l_port, r_addr, r_port);

        while (!feof(file)) {      
                char buffer[PROC_FILE_MAXLINE];
                unsigned int la, ra, lp, rp;
                la = ra = lp = rp = 0;

                fgets(buffer, sizeof(buffer) - 1, file);
                if (sscanf(buffer, "%*u: %X:%X %X:%X %*X %*X:%*X %*X:%*X %*X %u", &la, &lp, &ra, &rp, uid) != 5)
                        continue;
      
                if (la == l_addr && ra == r_addr && lp == l_port && rp == r_port) {
                        debug("Found, uid = %d\n", *uid);
                        fclose(file);
                        return LOOKUP_OK;
                }
        }

        debug("Not found\n");
        fclose(file);
        return LOOKUP_NOT_FOUND;
}

static void compose_response(const int in, const char* request, char* response, unsigned int* response_len)
{
        struct sockaddr_in l_addr, r_addr;
        socklen_t l_addr_len, r_addr_len;
        unsigned int l_port, r_port;
        struct passwd* pwd;
        uid_t uid;

        l_addr_len = sizeof(l_addr);
        r_addr_len = sizeof(r_addr);
   
        if (getpeername(in, (struct sockaddr*)&r_addr, &r_addr_len) ||
            getsockname(in, (struct sockaddr*)&l_addr, &l_addr_len)) {
                *response_len = snprintf(response, *response_len, "%.20s : ERROR : UNKNOWN-ERROR\r\n", request);
        } else {
                l_port = r_port = TCP_INVALID_PORT;
                sscanf(request, " %u , %u ", &l_port, &r_port);
   
                if (l_port < TCP_MIN_PORT || l_port > TCP_MAX_PORT || r_port < TCP_MIN_PORT || r_port > TCP_MAX_PORT) {
                        *response_len = snprintf(response, *response_len, "%.20s : ERROR : INVALID-PORT\r\n", request);
                } else {
                        int result = lookup_uid(l_addr.sin_addr.s_addr, l_port, r_addr.sin_addr.s_addr, r_port, &uid);
         
                        if (result == LOOKUP_NOT_FOUND || (result == LOOKUP_OK && (pwd = getpwuid(uid)) == 0)) {
                                *response_len = snprintf(response, *response_len, "%.20s : ERROR : NO-USER\r\n", request);
                        } else if (result == LOOKUP_OK) {
                                *response_len = snprintf(response, *response_len, "%.20s : USERID : UNIX : %.20s\r\n", request, pwd->pw_name);
                        } else {
                                *response_len = snprintf(response, *response_len, "%.20s : ERROR : UNKNOWN-ERROR\r\n", request);
                        }
                }
        }
}

static void session_handler(int in, int out)
{
        unsigned int response_len;
        char response[MAX_RESPONSE];
        char request[MAX_REQUEST];
   
        signal(SIGALRM, session_timeout);
        alarm(TIMEOUT);
   
        for (;;) {
                if (!read_request(in, request, MAX_REQUEST))
                        break;

                debug("Request: %s\n", request);

                response_len = sizeof(response);
                compose_response(in, request, response, &response_len);

                debug("Response: %s\n", response);

                if(!write_response(out, response, response_len))
                        break;

                alarm(TIMEOUT);
        }
}

static void daemon_mode()
{
        int server_socket, session_socket;
        struct sockaddr_in l_addr, r_addr;
        struct passwd* pwd;
        int yes = 1;
        pid_t pid;

#ifndef DEBUG
        if (daemon(0, 0) == -1) {
                syslog(LOG_ERR, "Failed to become daemon: %s", strerror(errno));
                exit(EXIT_FAILURE);
        }
#endif

        if ((server_socket = socket(AF_INET, SOCK_STREAM, 0)) == -1) {
                syslog(LOG_ERR, "Failed to create socket: %s", strerror(errno));
                exit(EXIT_FAILURE);
        }

        if (setsockopt(server_socket, SOL_SOCKET, SO_REUSEADDR, &yes, sizeof(int)) == -1) {
                syslog(LOG_ERR, "Failed to set socket options: %s", strerror(errno));
                exit(EXIT_FAILURE);
        }
         
        memset(&l_addr, 0, sizeof(l_addr));
        l_addr.sin_family = AF_INET;
        l_addr.sin_port = htons(PORT);
        l_addr.sin_addr.s_addr = INADDR_ANY;

        if (bind(server_socket, (struct sockaddr*)&l_addr, sizeof(l_addr)) == -1) {
                syslog(LOG_ERR, "Failed to bind to port %d: %s", PORT, strerror(errno));
                exit(EXIT_FAILURE);
        }

        if (listen(server_socket, BACKLOG) == -1) {
                syslog(LOG_ERR, "Failed to listen to port %d: %s", PORT, strerror(errno));
                exit(EXIT_FAILURE);
        }

        if ((pwd = getpwnam(USER)) == 0) {
                syslog(LOG_ERR, "Failed to get user id of user %s, session terminated", USER);
                exit(EXIT_FAILURE);
        }
      
        if (setuid(pwd->pw_uid) == -1) {
                syslog(LOG_ERR, "Failed to set user id to %d (%s): %s", pwd->pw_uid, USER, strerror(errno));
                exit(EXIT_FAILURE);
        }

        signal(SIGCHLD, session_terminate);

        for(;;) {
                socklen_t r_addr_len = sizeof(r_addr);
                if ((session_socket = accept(server_socket, (struct sockaddr*)&r_addr, &r_addr_len)) == -1) {
                        if (errno != EINTR)
                                syslog(LOG_ERR, "Failed to accept connection on port %d: %s", PORT, strerror(errno));
                        continue;
                }

                debug("Got connection from %s\n", inet_ntoa(r_addr.sin_addr));

                if (log_connections) {
                        char hbuf[NI_MAXHOST];
                        if (!getnameinfo((struct sockaddr*)&r_addr, r_addr_len, hbuf, sizeof(hbuf) - 1, 0, 0, NI_NAMEREQD|NI_DGRAM))
                                syslog(LOG_INFO, "Accepted connection from %s (%s)", inet_ntoa(r_addr.sin_addr), hbuf);
                        else
                                syslog(LOG_INFO, "Accepted connection from %s", inet_ntoa(r_addr.sin_addr));
                }
      
                pid = fork();
                if (pid == 0) {
                        close(server_socket);
                        session_handler(session_socket, session_socket);
                        close(session_socket);
                        exit(EXIT_SUCCESS);
                } else if (pid == -1) {
                        syslog(LOG_ERR, "Failed to create new process: %s", strerror(errno));
                }

                close(session_socket);
        }
}

static void inetd_mode()
{
        session_handler(STDIN_FILENO, STDOUT_FILENO);
}

static void print_help()
{
        printf("linux-identd version %s, Copyright (c) 2001-2003 by Per Liden <per@fukt.bth.se>\n", VERSION);
        printf("options:\n");
        printf("   -d    run in daemon mode\n");
        printf("   -l    log connections\n");
        printf("   -v    print version and exit\n");
        printf("   -h    print this help and exit\n");
#if 0
        /* Future features? */
        printf("   -o         do not reveal host type\n");
        printf("   -t <sec>   connection timeout (default is 60)\n");
        printf("   -u <user>  run as <user> (default is nobody)\n");
#endif
}

static void print_version()
{
        printf("linux-identd %s\n", VERSION);
}

int main(int argc, char** argv)
{
        int i, become_daemon = 0;

        if (argc > 1) {
                for (i = 1; i < argc; i++) {
                        if (!strcmp(argv[i], "-d")) {
                                become_daemon = 1;
                        } else if (!strcmp(argv[i], "-l")) {
                                log_connections = 1;
                        } else if (!strcmp(argv[i], "-v")) {
                                print_version();
                                exit(EXIT_SUCCESS);
                        } else if (!strcmp(argv[i], "-h")) {
                                print_help();
                                exit(EXIT_SUCCESS);
                        } else {
                                fprintf(stderr, "invalid option '%s'\n\n", argv[i]);
                                print_help();
                                exit(EXIT_FAILURE);
                        }
                }
        }

        openlog(0, LOG_PID, LOG_DAEMON);

        if (become_daemon)
                daemon_mode();
        else
                inetd_mode();

        return EXIT_SUCCESS;
} 
