//===----------------------------------------------------------------------===//
//
// Part of the LLVM Project, under the Apache License v2.0 with LLVM Exceptions.
// See https://llvm.org/LICENSE.txt for license information.
// SPDX-License-Identifier: Apache-2.0 WITH LLVM-exception
//
//===----------------------------------------------------------------------===//

// <string>

// template <class T>
//    basic_string& replace(size_type pos1, size_type n1, const T& t,
//                          size_type pos2, size_type n=npos); // constexpr since C++20
//
//  Mostly we're testing string_view here

#include <string>
#include <stdexcept>
#include <algorithm>
#include <cassert>

#include "test_macros.h"
#include "min_allocator.h"

template <class S, class SV>
TEST_CONSTEXPR_CXX20 void
test(S s, typename S::size_type pos1, typename S::size_type n1,
     SV sv, typename S::size_type pos2, typename S::size_type n2,
     S expected)
{
    typedef typename S::size_type SizeT;
    static_assert((!std::is_same<S, SV>::value), "");

    // String and string_view may not always share the same size type,
    // but both types should have the same size (ex. int vs long)
    static_assert(sizeof(SizeT) == sizeof(typename SV::size_type), "");

    const SizeT old_size = s.size();
    S s0 = s;
    if (pos1 <= old_size && pos2 <= sv.size())
    {
        s.replace(pos1, n1, sv, pos2, n2);
        LIBCPP_ASSERT(s.__invariants());
        assert(s == expected);
        SizeT xlen = std::min<SizeT>(n1, old_size - pos1);
        SizeT rlen = std::min<SizeT>(n2, sv.size() - pos2);
        assert(s.size() == old_size - xlen + rlen);
    }
#ifndef TEST_HAS_NO_EXCEPTIONS
    else if (!TEST_IS_CONSTANT_EVALUATED)
    {
        try
        {
            s.replace(pos1, n1, sv, pos2, n2);
            assert(false);
        }
        catch (std::out_of_range&)
        {
            assert(pos1 > old_size || pos2 > sv.size());
            assert(s == s0);
        }
    }
#endif
}

template <class S, class SV>
TEST_CONSTEXPR_CXX20 void
test_npos(S s, typename S::size_type pos1, typename S::size_type n1,
          SV sv, typename S::size_type pos2,
          S expected)
{
    typedef typename S::size_type SizeT;
    static_assert((!std::is_same<S, SV>::value), "");
    const SizeT old_size = s.size();
    S s0 = s;
    if (pos1 <= old_size && pos2 <= sv.size())
    {
        s.replace(pos1, n1, sv, pos2);
        LIBCPP_ASSERT(s.__invariants());
        assert(s == expected);
        SizeT xlen = std::min<SizeT>(n1, old_size - pos1);
        SizeT rlen = std::min<SizeT>(S::npos, sv.size() - pos2);
        assert(s.size() == old_size - xlen + rlen);
    }
#ifndef TEST_HAS_NO_EXCEPTIONS
    else if (!TEST_IS_CONSTANT_EVALUATED)
    {
        try
        {
            s.replace(pos1, n1, sv, pos2);
            assert(false);
        }
        catch (std::out_of_range&)
        {
            assert(pos1 > old_size || pos2 > sv.size());
            assert(s == s0);
        }
    }
#endif
}


template <class S, class SV>
TEST_CONSTEXPR_CXX20 bool test0()
{
    test(S(""), 0, 0, SV(""), 0, 0, S(""));
    test(S(""), 0, 0, SV(""), 0, 1, S(""));
    test(S(""), 0, 0, SV(""), 1, 0, S("can't happen"));
    test(S(""), 0, 0, SV("12345"), 0, 0, S(""));
    test(S(""), 0, 0, SV("12345"), 0, 1, S("1"));
    test(S(""), 0, 0, SV("12345"), 0, 2, S("12"));
    test(S(""), 0, 0, SV("12345"), 0, 4, S("1234"));
    test(S(""), 0, 0, SV("12345"), 0, 5, S("12345"));
    test(S(""), 0, 0, SV("12345"), 0, 6, S("12345"));
    test(S(""), 0, 0, SV("12345"), 1, 0, S(""));
    test(S(""), 0, 0, SV("12345"), 1, 1, S("2"));
    test(S(""), 0, 0, SV("12345"), 1, 2, S("23"));
    test(S(""), 0, 0, SV("12345"), 1, 3, S("234"));
    test(S(""), 0, 0, SV("12345"), 1, 4, S("2345"));
    test(S(""), 0, 0, SV("12345"), 1, 5, S("2345"));
    test(S(""), 0, 0, SV("12345"), 2, 0, S(""));
    test(S(""), 0, 0, SV("12345"), 2, 1, S("3"));
    test(S(""), 0, 0, SV("12345"), 2, 2, S("34"));
    test(S(""), 0, 0, SV("12345"), 2, 3, S("345"));
    test(S(""), 0, 0, SV("12345"), 2, 4, S("345"));
    test(S(""), 0, 0, SV("12345"), 4, 0, S(""));
    test(S(""), 0, 0, SV("12345"), 4, 1, S("5"));
    test(S(""), 0, 0, SV("12345"), 4, 2, S("5"));
    test(S(""), 0, 0, SV("12345"), 5, 0, S(""));
    test(S(""), 0, 0, SV("12345"), 5, 1, S(""));
    test(S(""), 0, 0, SV("12345"), 6, 0, S("can't happen"));
    test(S(""), 0, 0, SV("1234567890"), 0, 0, S(""));
    test(S(""), 0, 0, SV("1234567890"), 0, 1, S("1"));
    test(S(""), 0, 0, SV("1234567890"), 0, 5, S("12345"));
    test(S(""), 0, 0, SV("1234567890"), 0, 9, S("123456789"));
    test(S(""), 0, 0, SV("1234567890"), 0, 10, S("1234567890"));
    test(S(""), 0, 0, SV("1234567890"), 0, 11, S("1234567890"));
    test(S(""), 0, 0, SV("1234567890"), 1, 0, S(""));
    test(S(""), 0, 0, SV("1234567890"), 1, 1, S("2"));
    test(S(""), 0, 0, SV("1234567890"), 1, 4, S("2345"));
    test(S(""), 0, 0, SV("1234567890"), 1, 8, S("23456789"));
    test(S(""), 0, 0, SV("1234567890"), 1, 9, S("234567890"));
    test(S(""), 0, 0, SV("1234567890"), 1, 10, S("234567890"));
    test(S(""), 0, 0, SV("1234567890"), 5, 0, S(""));
    test(S(""), 0, 0, SV("1234567890"), 5, 1, S("6"));
    test(S(""), 0, 0, SV("1234567890"), 5, 2, S("67"));
    test(S(""), 0, 0, SV("1234567890"), 5, 4, S("6789"));
    test(S(""), 0, 0, SV("1234567890"), 5, 5, S("67890"));
    test(S(""), 0, 0, SV("1234567890"), 5, 6, S("67890"));
    test(S(""), 0, 0, SV("1234567890"), 9, 0, S(""));
    test(S(""), 0, 0, SV("1234567890"), 9, 1, S("0"));
    test(S(""), 0, 0, SV("1234567890"), 9, 2, S("0"));
    test(S(""), 0, 0, SV("1234567890"), 10, 0, S(""));
    test(S(""), 0, 0, SV("1234567890"), 10, 1, S(""));
    test(S(""), 0, 0, SV("1234567890"), 11, 0, S("can't happen"));
    test(S(""), 0, 0, SV("12345678901234567890"), 0, 0, S(""));
    test(S(""), 0, 0, SV("12345678901234567890"), 0, 1, S("1"));
    test(S(""), 0, 0, SV("12345678901234567890"), 0, 10, S("1234567890"));
    test(S(""), 0, 0, SV("12345678901234567890"), 0, 19, S("1234567890123456789"));
    test(S(""), 0, 0, SV("12345678901234567890"), 0, 20, S("12345678901234567890"));
    test(S(""), 0, 0, SV("12345678901234567890"), 0, 21, S("12345678901234567890"));
    test(S(""), 0, 0, SV("12345678901234567890"), 1, 0, S(""));
    test(S(""), 0, 0, SV("12345678901234567890"), 1, 1, S("2"));
    test(S(""), 0, 0, SV("12345678901234567890"), 1, 9, S("234567890"));
    test(S(""), 0, 0, SV("12345678901234567890"), 1, 18, S("234567890123456789"));
    test(S(""), 0, 0, SV("12345678901234567890"), 1, 19, S("2345678901234567890"));
    test(S(""), 0, 0, SV("12345678901234567890"), 1, 20, S("2345678901234567890"));
    test(S(""), 0, 0, SV("12345678901234567890"), 10, 0, S(""));
    test(S(""), 0, 0, SV("12345678901234567890"), 10, 1, S("1"));
    test(S(""), 0, 0, SV("12345678901234567890"), 10, 5, S("12345"));
    test(S(""), 0, 0, SV("12345678901234567890"), 10, 9, S("123456789"));
    test(S(""), 0, 0, SV("12345678901234567890"), 10, 10, S("1234567890"));
    test(S(""), 0, 0, SV("12345678901234567890"), 10, 11, S("1234567890"));
    test(S(""), 0, 0, SV("12345678901234567890"), 19, 0, S(""));
    test(S(""), 0, 0, SV("12345678901234567890"), 19, 1, S("0"));
    test(S(""), 0, 0, SV("12345678901234567890"), 19, 2, S("0"));
    test(S(""), 0, 0, SV("12345678901234567890"), 20, 0, S(""));
    test(S(""), 0, 0, SV("12345678901234567890"), 20, 1, S(""));
    test(S(""), 0, 0, SV("12345678901234567890"), 21, 0, S("can't happen"));
    test(S(""), 0, 1, SV(""), 0, 0, S(""));
    test(S(""), 0, 1, SV(""), 0, 1, S(""));
    test(S(""), 0, 1, SV(""), 1, 0, S("can't happen"));
    test(S(""), 0, 1, SV("12345"), 0, 0, S(""));
    test(S(""), 0, 1, SV("12345"), 0, 1, S("1"));
    test(S(""), 0, 1, SV("12345"), 0, 2, S("12"));
    test(S(""), 0, 1, SV("12345"), 0, 4, S("1234"));
    test(S(""), 0, 1, SV("12345"), 0, 5, S("12345"));
    test(S(""), 0, 1, SV("12345"), 0, 6, S("12345"));
    test(S(""), 0, 1, SV("12345"), 1, 0, S(""));
    test(S(""), 0, 1, SV("12345"), 1, 1, S("2"));
    test(S(""), 0, 1, SV("12345"), 1, 2, S("23"));
    test(S(""), 0, 1, SV("12345"), 1, 3, S("234"));
    test(S(""), 0, 1, SV("12345"), 1, 4, S("2345"));
    test(S(""), 0, 1, SV("12345"), 1, 5, S("2345"));
    test(S(""), 0, 1, SV("12345"), 2, 0, S(""));
    test(S(""), 0, 1, SV("12345"), 2, 1, S("3"));
    test(S(""), 0, 1, SV("12345"), 2, 2, S("34"));
    test(S(""), 0, 1, SV("12345"), 2, 3, S("345"));
    test(S(""), 0, 1, SV("12345"), 2, 4, S("345"));
    test(S(""), 0, 1, SV("12345"), 4, 0, S(""));
    test(S(""), 0, 1, SV("12345"), 4, 1, S("5"));
    test(S(""), 0, 1, SV("12345"), 4, 2, S("5"));
    test(S(""), 0, 1, SV("12345"), 5, 0, S(""));
    test(S(""), 0, 1, SV("12345"), 5, 1, S(""));
    test(S(""), 0, 1, SV("12345"), 6, 0, S("can't happen"));

    return true;
}

template <class S, class SV>
TEST_CONSTEXPR_CXX20 bool test1()
{
    test(S(""), 0, 1, SV("1234567890"), 0, 0, S(""));
    test(S(""), 0, 1, SV("1234567890"), 0, 1, S("1"));
    test(S(""), 0, 1, SV("1234567890"), 0, 5, S("12345"));
    test(S(""), 0, 1, SV("1234567890"), 0, 9, S("123456789"));
    test(S(""), 0, 1, SV("1234567890"), 0, 10, S("1234567890"));
    test(S(""), 0, 1, SV("1234567890"), 0, 11, S("1234567890"));
    test(S(""), 0, 1, SV("1234567890"), 1, 0, S(""));
    test(S(""), 0, 1, SV("1234567890"), 1, 1, S("2"));
    test(S(""), 0, 1, SV("1234567890"), 1, 4, S("2345"));
    test(S(""), 0, 1, SV("1234567890"), 1, 8, S("23456789"));
    test(S(""), 0, 1, SV("1234567890"), 1, 9, S("234567890"));
    test(S(""), 0, 1, SV("1234567890"), 1, 10, S("234567890"));
    test(S(""), 0, 1, SV("1234567890"), 5, 0, S(""));
    test(S(""), 0, 1, SV("1234567890"), 5, 1, S("6"));
    test(S(""), 0, 1, SV("1234567890"), 5, 2, S("67"));
    test(S(""), 0, 1, SV("1234567890"), 5, 4, S("6789"));
    test(S(""), 0, 1, SV("1234567890"), 5, 5, S("67890"));
    test(S(""), 0, 1, SV("1234567890"), 5, 6, S("67890"));
    test(S(""), 0, 1, SV("1234567890"), 9, 0, S(""));
    test(S(""), 0, 1, SV("1234567890"), 9, 1, S("0"));
    test(S(""), 0, 1, SV("1234567890"), 9, 2, S("0"));
    test(S(""), 0, 1, SV("1234567890"), 10, 0, S(""));
    test(S(""), 0, 1, SV("1234567890"), 10, 1, S(""));
    test(S(""), 0, 1, SV("1234567890"), 11, 0, S("can't happen"));
    test(S(""), 0, 1, SV("12345678901234567890"), 0, 0, S(""));
    test(S(""), 0, 1, SV("12345678901234567890"), 0, 1, S("1"));
    test(S(""), 0, 1, SV("12345678901234567890"), 0, 10, S("1234567890"));
    test(S(""), 0, 1, SV("12345678901234567890"), 0, 19, S("1234567890123456789"));
    test(S(""), 0, 1, SV("12345678901234567890"), 0, 20, S("12345678901234567890"));
    test(S(""), 0, 1, SV("12345678901234567890"), 0, 21, S("12345678901234567890"));
    test(S(""), 0, 1, SV("12345678901234567890"), 1, 0, S(""));
    test(S(""), 0, 1, SV("12345678901234567890"), 1, 1, S("2"));
    test(S(""), 0, 1, SV("12345678901234567890"), 1, 9, S("234567890"));
    test(S(""), 0, 1, SV("12345678901234567890"), 1, 18, S("234567890123456789"));
    test(S(""), 0, 1, SV("12345678901234567890"), 1, 19, S("2345678901234567890"));
    test(S(""), 0, 1, SV("12345678901234567890"), 1, 20, S("2345678901234567890"));
    test(S(""), 0, 1, SV("12345678901234567890"), 10, 0, S(""));
    test(S(""), 0, 1, SV("12345678901234567890"), 10, 1, S("1"));
    test(S(""), 0, 1, SV("12345678901234567890"), 10, 5, S("12345"));
    test(S(""), 0, 1, SV("12345678901234567890"), 10, 9, S("123456789"));
    test(S(""), 0, 1, SV("12345678901234567890"), 10, 10, S("1234567890"));
    test(S(""), 0, 1, SV("12345678901234567890"), 10, 11, S("1234567890"));
    test(S(""), 0, 1, SV("12345678901234567890"), 19, 0, S(""));
    test(S(""), 0, 1, SV("12345678901234567890"), 19, 1, S("0"));
    test(S(""), 0, 1, SV("12345678901234567890"), 19, 2, S("0"));
    test(S(""), 0, 1, SV("12345678901234567890"), 20, 0, S(""));
    test(S(""), 0, 1, SV("12345678901234567890"), 20, 1, S(""));
    test(S(""), 0, 1, SV("12345678901234567890"), 21, 0, S("can't happen"));
    test(S(""), 1, 0, SV(""), 0, 0, S("can't happen"));
    test(S(""), 1, 0, SV(""), 0, 1, S("can't happen"));
    test(S(""), 1, 0, SV(""), 1, 0, S("can't happen"));
    test(S(""), 1, 0, SV("12345"), 0, 0, S("can't happen"));
    test(S(""), 1, 0, SV("12345"), 0, 1, S("can't happen"));
    test(S(""), 1, 0, SV("12345"), 0, 2, S("can't happen"));
    test(S(""), 1, 0, SV("12345"), 0, 4, S("can't happen"));
    test(S(""), 1, 0, SV("12345"), 0, 5, S("can't happen"));
    test(S(""), 1, 0, SV("12345"), 0, 6, S("can't happen"));
    test(S(""), 1, 0, SV("12345"), 1, 0, S("can't happen"));
    test(S(""), 1, 0, SV("12345"), 1, 1, S("can't happen"));
    test(S(""), 1, 0, SV("12345"), 1, 2, S("can't happen"));
    test(S(""), 1, 0, SV("12345"), 1, 3, S("can't happen"));
    test(S(""), 1, 0, SV("12345"), 1, 4, S("can't happen"));
    test(S(""), 1, 0, SV("12345"), 1, 5, S("can't happen"));
    test(S(""), 1, 0, SV("12345"), 2, 0, S("can't happen"));
    test(S(""), 1, 0, SV("12345"), 2, 1, S("can't happen"));
    test(S(""), 1, 0, SV("12345"), 2, 2, S("can't happen"));
    test(S(""), 1, 0, SV("12345"), 2, 3, S("can't happen"));
    test(S(""), 1, 0, SV("12345"), 2, 4, S("can't happen"));
    test(S(""), 1, 0, SV("12345"), 4, 0, S("can't happen"));
    test(S(""), 1, 0, SV("12345"), 4, 1, S("can't happen"));
    test(S(""), 1, 0, SV("12345"), 4, 2, S("can't happen"));
    test(S(""), 1, 0, SV("12345"), 5, 0, S("can't happen"));
    test(S(""), 1, 0, SV("12345"), 5, 1, S("can't happen"));
    test(S(""), 1, 0, SV("12345"), 6, 0, S("can't happen"));
    test(S(""), 1, 0, SV("1234567890"), 0, 0, S("can't happen"));
    test(S(""), 1, 0, SV("1234567890"), 0, 1, S("can't happen"));
    test(S(""), 1, 0, SV("1234567890"), 0, 5, S("can't happen"));
    test(S(""), 1, 0, SV("1234567890"), 0, 9, S("can't happen"));
    test(S(""), 1, 0, SV("1234567890"), 0, 10, S("can't happen"));
    test(S(""), 1, 0, SV("1234567890"), 0, 11, S("can't happen"));
    test(S(""), 1, 0, SV("1234567890"), 1, 0, S("can't happen"));
    test(S(""), 1, 0, SV("1234567890"), 1, 1, S("can't happen"));
    test(S(""), 1, 0, SV("1234567890"), 1, 4, S("can't happen"));
    test(S(""), 1, 0, SV("1234567890"), 1, 8, S("can't happen"));
    test(S(""), 1, 0, SV("1234567890"), 1, 9, S("can't happen"));
    test(S(""), 1, 0, SV("1234567890"), 1, 10, S("can't happen"));
    test(S(""), 1, 0, SV("1234567890"), 5, 0, S("can't happen"));
    test(S(""), 1, 0, SV("1234567890"), 5, 1, S("can't happen"));
    test(S(""), 1, 0, SV("1234567890"), 5, 2, S("can't happen"));
    test(S(""), 1, 0, SV("1234567890"), 5, 4, S("can't happen"));
    test(S(""), 1, 0, SV("1234567890"), 5, 5, S("can't happen"));
    test(S(""), 1, 0, SV("1234567890"), 5, 6, S("can't happen"));
    test(S(""), 1, 0, SV("1234567890"), 9, 0, S("can't happen"));
    test(S(""), 1, 0, SV("1234567890"), 9, 1, S("can't happen"));
    test(S(""), 1, 0, SV("1234567890"), 9, 2, S("can't happen"));
    test(S(""), 1, 0, SV("1234567890"), 10, 0, S("can't happen"));
    test(S(""), 1, 0, SV("1234567890"), 10, 1, S("can't happen"));
    test(S(""), 1, 0, SV("1234567890"), 11, 0, S("can't happen"));
    test(S(""), 1, 0, SV("12345678901234567890"), 0, 0, S("can't happen"));
    test(S(""), 1, 0, SV("12345678901234567890"), 0, 1, S("can't happen"));

    return true;
}

template <class S, class SV>
TEST_CONSTEXPR_CXX20 bool test2()
{
    test(S(""), 1, 0, SV("12345678901234567890"), 0, 10, S("can't happen"));
    test(S(""), 1, 0, SV("12345678901234567890"), 0, 19, S("can't happen"));
    test(S(""), 1, 0, SV("12345678901234567890"), 0, 20, S("can't happen"));
    test(S(""), 1, 0, SV("12345678901234567890"), 0, 21, S("can't happen"));
    test(S(""), 1, 0, SV("12345678901234567890"), 1, 0, S("can't happen"));
    test(S(""), 1, 0, SV("12345678901234567890"), 1, 1, S("can't happen"));
    test(S(""), 1, 0, SV("12345678901234567890"), 1, 9, S("can't happen"));
    test(S(""), 1, 0, SV("12345678901234567890"), 1, 18, S("can't happen"));
    test(S(""), 1, 0, SV("12345678901234567890"), 1, 19, S("can't happen"));
    test(S(""), 1, 0, SV("12345678901234567890"), 1, 20, S("can't happen"));
    test(S(""), 1, 0, SV("12345678901234567890"), 10, 0, S("can't happen"));
    test(S(""), 1, 0, SV("12345678901234567890"), 10, 1, S("can't happen"));
    test(S(""), 1, 0, SV("12345678901234567890"), 10, 5, S("can't happen"));
    test(S(""), 1, 0, SV("12345678901234567890"), 10, 9, S("can't happen"));
    test(S(""), 1, 0, SV("12345678901234567890"), 10, 10, S("can't happen"));
    test(S(""), 1, 0, SV("12345678901234567890"), 10, 11, S("can't happen"));
    test(S(""), 1, 0, SV("12345678901234567890"), 19, 0, S("can't happen"));
    test(S(""), 1, 0, SV("12345678901234567890"), 19, 1, S("can't happen"));
    test(S(""), 1, 0, SV("12345678901234567890"), 19, 2, S("can't happen"));
    test(S(""), 1, 0, SV("12345678901234567890"), 20, 0, S("can't happen"));
    test(S(""), 1, 0, SV("12345678901234567890"), 20, 1, S("can't happen"));
    test(S(""), 1, 0, SV("12345678901234567890"), 21, 0, S("can't happen"));
    test(S("abcde"), 0, 0, SV(""), 0, 0, S("abcde"));
    test(S("abcde"), 0, 0, SV(""), 0, 1, S("abcde"));
    test(S("abcde"), 0, 0, SV(""), 1, 0, S("can't happen"));
    test(S("abcde"), 0, 0, SV("12345"), 0, 0, S("abcde"));
    test(S("abcde"), 0, 0, SV("12345"), 0, 1, S("1abcde"));
    test(S("abcde"), 0, 0, SV("12345"), 0, 2, S("12abcde"));
    test(S("abcde"), 0, 0, SV("12345"), 0, 4, S("1234abcde"));
    test(S("abcde"), 0, 0, SV("12345"), 0, 5, S("12345abcde"));
    test(S("abcde"), 0, 0, SV("12345"), 0, 6, S("12345abcde"));
    test(S("abcde"), 0, 0, SV("12345"), 1, 0, S("abcde"));
    test(S("abcde"), 0, 0, SV("12345"), 1, 1, S("2abcde"));
    test(S("abcde"), 0, 0, SV("12345"), 1, 2, S("23abcde"));
    test(S("abcde"), 0, 0, SV("12345"), 1, 3, S("234abcde"));
    test(S("abcde"), 0, 0, SV("12345"), 1, 4, S("2345abcde"));
    test(S("abcde"), 0, 0, SV("12345"), 1, 5, S("2345abcde"));
    test(S("abcde"), 0, 0, SV("12345"), 2, 0, S("abcde"));
    test(S("abcde"), 0, 0, SV("12345"), 2, 1, S("3abcde"));
    test(S("abcde"), 0, 0, SV("12345"), 2, 2, S("34abcde"));
    test(S("abcde"), 0, 0, SV("12345"), 2, 3, S("345abcde"));
    test(S("abcde"), 0, 0, SV("12345"), 2, 4, S("345abcde"));
    test(S("abcde"), 0, 0, SV("12345"), 4, 0, S("abcde"));
    test(S("abcde"), 0, 0, SV("12345"), 4, 1, S("5abcde"));
    test(S("abcde"), 0, 0, SV("12345"), 4, 2, S("5abcde"));
    test(S("abcde"), 0, 0, SV("12345"), 5, 0, S("abcde"));
    test(S("abcde"), 0, 0, SV("12345"), 5, 1, S("abcde"));
    test(S("abcde"), 0, 0, SV("12345"), 6, 0, S("can't happen"));
    test(S("abcde"), 0, 0, SV("1234567890"), 0, 0, S("abcde"));
    test(S("abcde"), 0, 0, SV("1234567890"), 0, 1, S("1abcde"));
    test(S("abcde"), 0, 0, SV("1234567890"), 0, 5, S("12345abcde"));
    test(S("abcde"), 0, 0, SV("1234567890"), 0, 9, S("123456789abcde"));
    test(S("abcde"), 0, 0, SV("1234567890"), 0, 10, S("1234567890abcde"));
    test(S("abcde"), 0, 0, SV("1234567890"), 0, 11, S("1234567890abcde"));
    test(S("abcde"), 0, 0, SV("1234567890"), 1, 0, S("abcde"));
    test(S("abcde"), 0, 0, SV("1234567890"), 1, 1, S("2abcde"));
    test(S("abcde"), 0, 0, SV("1234567890"), 1, 4, S("2345abcde"));
    test(S("abcde"), 0, 0, SV("1234567890"), 1, 8, S("23456789abcde"));
    test(S("abcde"), 0, 0, SV("1234567890"), 1, 9, S("234567890abcde"));
    test(S("abcde"), 0, 0, SV("1234567890"), 1, 10, S("234567890abcde"));
    test(S("abcde"), 0, 0, SV("1234567890"), 5, 0, S("abcde"));
    test(S("abcde"), 0, 0, SV("1234567890"), 5, 1, S("6abcde"));
    test(S("abcde"), 0, 0, SV("1234567890"), 5, 2, S("67abcde"));
    test(S("abcde"), 0, 0, SV("1234567890"), 5, 4, S("6789abcde"));
    test(S("abcde"), 0, 0, SV("1234567890"), 5, 5, S("67890abcde"));
    test(S("abcde"), 0, 0, SV("1234567890"), 5, 6, S("67890abcde"));
    test(S("abcde"), 0, 0, SV("1234567890"), 9, 0, S("abcde"));
    test(S("abcde"), 0, 0, SV("1234567890"), 9, 1, S("0abcde"));
    test(S("abcde"), 0, 0, SV("1234567890"), 9, 2, S("0abcde"));
    test(S("abcde"), 0, 0, SV("1234567890"), 10, 0, S("abcde"));
    test(S("abcde"), 0, 0, SV("1234567890"), 10, 1, S("abcde"));
    test(S("abcde"), 0, 0, SV("1234567890"), 11, 0, S("can't happen"));
    test(S("abcde"), 0, 0, SV("12345678901234567890"), 0, 0, S("abcde"));
    test(S("abcde"), 0, 0, SV("12345678901234567890"), 0, 1, S("1abcde"));
    test(S("abcde"), 0, 0, SV("12345678901234567890"), 0, 10, S("1234567890abcde"));
    test(S("abcde"), 0, 0, SV("12345678901234567890"), 0, 19, S("1234567890123456789abcde"));
    test(S("abcde"), 0, 0, SV("12345678901234567890"), 0, 20, S("12345678901234567890abcde"));
    test(S("abcde"), 0, 0, SV("12345678901234567890"), 0, 21, S("12345678901234567890abcde"));
    test(S("abcde"), 0, 0, SV("12345678901234567890"), 1, 0, S("abcde"));
    test(S("abcde"), 0, 0, SV("12345678901234567890"), 1, 1, S("2abcde"));
    test(S("abcde"), 0, 0, SV("12345678901234567890"), 1, 9, S("234567890abcde"));
    test(S("abcde"), 0, 0, SV("12345678901234567890"), 1, 18, S("234567890123456789abcde"));
    test(S("abcde"), 0, 0, SV("12345678901234567890"), 1, 19, S("2345678901234567890abcde"));
    test(S("abcde"), 0, 0, SV("12345678901234567890"), 1, 20, S("2345678901234567890abcde"));
    test(S("abcde"), 0, 0, SV("12345678901234567890"), 10, 0, S("abcde"));
    test(S("abcde"), 0, 0, SV("12345678901234567890"), 10, 1, S("1abcde"));
    test(S("abcde"), 0, 0, SV("12345678901234567890"), 10, 5, S("12345abcde"));
    test(S("abcde"), 0, 0, SV("12345678901234567890"), 10, 9, S("123456789abcde"));
    test(S("abcde"), 0, 0, SV("12345678901234567890"), 10, 10, S("1234567890abcde"));
    test(S("abcde"), 0, 0, SV("12345678901234567890"), 10, 11, S("1234567890abcde"));
    test(S("abcde"), 0, 0, SV("12345678901234567890"), 19, 0, S("abcde"));
    test(S("abcde"), 0, 0, SV("12345678901234567890"), 19, 1, S("0abcde"));
    test(S("abcde"), 0, 0, SV("12345678901234567890"), 19, 2, S("0abcde"));
    test(S("abcde"), 0, 0, SV("12345678901234567890"), 20, 0, S("abcde"));
    test(S("abcde"), 0, 0, SV("12345678901234567890"), 20, 1, S("abcde"));
    test(S("abcde"), 0, 0, SV("12345678901234567890"), 21, 0, S("can't happen"));
    test(S("abcde"), 0, 1, SV(""), 0, 0, S("bcde"));
    test(S("abcde"), 0, 1, SV(""), 0, 1, S("bcde"));
    test(S("abcde"), 0, 1, SV(""), 1, 0, S("can't happen"));
    test(S("abcde"), 0, 1, SV("12345"), 0, 0, S("bcde"));

    return true;
}

template <class S, class SV>
TEST_CONSTEXPR_CXX20 bool test3()
{
    test(S("abcde"), 0, 1, SV("12345"), 0, 1, S("1bcde"));
    test(S("abcde"), 0, 1, SV("12345"), 0, 2, S("12bcde"));
    test(S("abcde"), 0, 1, SV("12345"), 0, 4, S("1234bcde"));
    test(S("abcde"), 0, 1, SV("12345"), 0, 5, S("12345bcde"));
    test(S("abcde"), 0, 1, SV("12345"), 0, 6, S("12345bcde"));
    test(S("abcde"), 0, 1, SV("12345"), 1, 0, S("bcde"));
    test(S("abcde"), 0, 1, SV("12345"), 1, 1, S("2bcde"));
    test(S("abcde"), 0, 1, SV("12345"), 1, 2, S("23bcde"));
    test(S("abcde"), 0, 1, SV("12345"), 1, 3, S("234bcde"));
    test(S("abcde"), 0, 1, SV("12345"), 1, 4, S("2345bcde"));
    test(S("abcde"), 0, 1, SV("12345"), 1, 5, S("2345bcde"));
    test(S("abcde"), 0, 1, SV("12345"), 2, 0, S("bcde"));
    test(S("abcde"), 0, 1, SV("12345"), 2, 1, S("3bcde"));
    test(S("abcde"), 0, 1, SV("12345"), 2, 2, S("34bcde"));
    test(S("abcde"), 0, 1, SV("12345"), 2, 3, S("345bcde"));
    test(S("abcde"), 0, 1, SV("12345"), 2, 4, S("345bcde"));
    test(S("abcde"), 0, 1, SV("12345"), 4, 0, S("bcde"));
    test(S("abcde"), 0, 1, SV("12345"), 4, 1, S("5bcde"));
    test(S("abcde"), 0, 1, SV("12345"), 4, 2, S("5bcde"));
    test(S("abcde"), 0, 1, SV("12345"), 5, 0, S("bcde"));
    test(S("abcde"), 0, 1, SV("12345"), 5, 1, S("bcde"));
    test(S("abcde"), 0, 1, SV("12345"), 6, 0, S("can't happen"));
    test(S("abcde"), 0, 1, SV("1234567890"), 0, 0, S("bcde"));
    test(S("abcde"), 0, 1, SV("1234567890"), 0, 1, S("1bcde"));
    test(S("abcde"), 0, 1, SV("1234567890"), 0, 5, S("12345bcde"));
    test(S("abcde"), 0, 1, SV("1234567890"), 0, 9, S("123456789bcde"));
    test(S("abcde"), 0, 1, SV("1234567890"), 0, 10, S("1234567890bcde"));
    test(S("abcde"), 0, 1, SV("1234567890"), 0, 11, S("1234567890bcde"));
    test(S("abcde"), 0, 1, SV("1234567890"), 1, 0, S("bcde"));
    test(S("abcde"), 0, 1, SV("1234567890"), 1, 1, S("2bcde"));
    test(S("abcde"), 0, 1, SV("1234567890"), 1, 4, S("2345bcde"));
    test(S("abcde"), 0, 1, SV("1234567890"), 1, 8, S("23456789bcde"));
    test(S("abcde"), 0, 1, SV("1234567890"), 1, 9, S("234567890bcde"));
    test(S("abcde"), 0, 1, SV("1234567890"), 1, 10, S("234567890bcde"));
    test(S("abcde"), 0, 1, SV("1234567890"), 5, 0, S("bcde"));
    test(S("abcde"), 0, 1, SV("1234567890"), 5, 1, S("6bcde"));
    test(S("abcde"), 0, 1, SV("1234567890"), 5, 2, S("67bcde"));
    test(S("abcde"), 0, 1, SV("1234567890"), 5, 4, S("6789bcde"));
    test(S("abcde"), 0, 1, SV("1234567890"), 5, 5, S("67890bcde"));
    test(S("abcde"), 0, 1, SV("1234567890"), 5, 6, S("67890bcde"));
    test(S("abcde"), 0, 1, SV("1234567890"), 9, 0, S("bcde"));
    test(S("abcde"), 0, 1, SV("1234567890"), 9, 1, S("0bcde"));
    test(S("abcde"), 0, 1, SV("1234567890"), 9, 2, S("0bcde"));
    test(S("abcde"), 0, 1, SV("1234567890"), 10, 0, S("bcde"));
    test(S("abcde"), 0, 1, SV("1234567890"), 10, 1, S("bcde"));
    test(S("abcde"), 0, 1, SV("1234567890"), 11, 0, S("can't happen"));
    test(S("abcde"), 0, 1, SV("12345678901234567890"), 0, 0, S("bcde"));
    test(S("abcde"), 0, 1, SV("12345678901234567890"), 0, 1, S("1bcde"));
    test(S("abcde"), 0, 1, SV("12345678901234567890"), 0, 10, S("1234567890bcde"));
    test(S("abcde"), 0, 1, SV("12345678901234567890"), 0, 19, S("1234567890123456789bcde"));
    test(S("abcde"), 0, 1, SV("12345678901234567890"), 0, 20, S("12345678901234567890bcde"));
    test(S("abcde"), 0, 1, SV("12345678901234567890"), 0, 21, S("12345678901234567890bcde"));
    test(S("abcde"), 0, 1, SV("12345678901234567890"), 1, 0, S("bcde"));
    test(S("abcde"), 0, 1, SV("12345678901234567890"), 1, 1, S("2bcde"));
    test(S("abcde"), 0, 1, SV("12345678901234567890"), 1, 9, S("234567890bcde"));
    test(S("abcde"), 0, 1, SV("12345678901234567890"), 1, 18, S("234567890123456789bcde"));
    test(S("abcde"), 0, 1, SV("12345678901234567890"), 1, 19, S("2345678901234567890bcde"));
    test(S("abcde"), 0, 1, SV("12345678901234567890"), 1, 20, S("2345678901234567890bcde"));
    test(S("abcde"), 0, 1, SV("12345678901234567890"), 10, 0, S("bcde"));
    test(S("abcde"), 0, 1, SV("12345678901234567890"), 10, 1, S("1bcde"));
    test(S("abcde"), 0, 1, SV("12345678901234567890"), 10, 5, S("12345bcde"));
    test(S("abcde"), 0, 1, SV("12345678901234567890"), 10, 9, S("123456789bcde"));
    test(S("abcde"), 0, 1, SV("12345678901234567890"), 10, 10, S("1234567890bcde"));
    test(S("abcde"), 0, 1, SV("12345678901234567890"), 10, 11, S("1234567890bcde"));
    test(S("abcde"), 0, 1, SV("12345678901234567890"), 19, 0, S("bcde"));
    test(S("abcde"), 0, 1, SV("12345678901234567890"), 19, 1, S("0bcde"));
    test(S("abcde"), 0, 1, SV("12345678901234567890"), 19, 2, S("0bcde"));
    test(S("abcde"), 0, 1, SV("12345678901234567890"), 20, 0, S("bcde"));
    test(S("abcde"), 0, 1, SV("12345678901234567890"), 20, 1, S("bcde"));
    test(S("abcde"), 0, 1, SV("12345678901234567890"), 21, 0, S("can't happen"));
    test(S("abcde"), 0, 2, SV(""), 0, 0, S("cde"));
    test(S("abcde"), 0, 2, SV(""), 0, 1, S("cde"));
    test(S("abcde"), 0, 2, SV(""), 1, 0, S("can't happen"));
    test(S("abcde"), 0, 2, SV("12345"), 0, 0, S("cde"));
    test(S("abcde"), 0, 2, SV("12345"), 0, 1, S("1cde"));
    test(S("abcde"), 0, 2, SV("12345"), 0, 2, S("12cde"));
    test(S("abcde"), 0, 2, SV("12345"), 0, 4, S("1234cde"));
    test(S("abcde"), 0, 2, SV("12345"), 0, 5, S("12345cde"));
    test(S("abcde"), 0, 2, SV("12345"), 0, 6, S("12345cde"));
    test(S("abcde"), 0, 2, SV("12345"), 1, 0, S("cde"));
    test(S("abcde"), 0, 2, SV("12345"), 1, 1, S("2cde"));
    test(S("abcde"), 0, 2, SV("12345"), 1, 2, S("23cde"));
    test(S("abcde"), 0, 2, SV("12345"), 1, 3, S("234cde"));
    test(S("abcde"), 0, 2, SV("12345"), 1, 4, S("2345cde"));
    test(S("abcde"), 0, 2, SV("12345"), 1, 5, S("2345cde"));
    test(S("abcde"), 0, 2, SV("12345"), 2, 0, S("cde"));
    test(S("abcde"), 0, 2, SV("12345"), 2, 1, S("3cde"));
    test(S("abcde"), 0, 2, SV("12345"), 2, 2, S("34cde"));
    test(S("abcde"), 0, 2, SV("12345"), 2, 3, S("345cde"));
    test(S("abcde"), 0, 2, SV("12345"), 2, 4, S("345cde"));
    test(S("abcde"), 0, 2, SV("12345"), 4, 0, S("cde"));
    test(S("abcde"), 0, 2, SV("12345"), 4, 1, S("5cde"));
    test(S("abcde"), 0, 2, SV("12345"), 4, 2, S("5cde"));
    test(S("abcde"), 0, 2, SV("12345"), 5, 0, S("cde"));
    test(S("abcde"), 0, 2, SV("12345"), 5, 1, S("cde"));
    test(S("abcde"), 0, 2, SV("12345"), 6, 0, S("can't happen"));
    test(S("abcde"), 0, 2, SV("1234567890"), 0, 0, S("cde"));
    test(S("abcde"), 0, 2, SV("1234567890"), 0, 1, S("1cde"));
    test(S("abcde"), 0, 2, SV("1234567890"), 0, 5, S("12345cde"));
    test(S("abcde"), 0, 2, SV("1234567890"), 0, 9, S("123456789cde"));

    return true;
}

template <class S, class SV>
TEST_CONSTEXPR_CXX20 bool test4()
{
    test(S("abcde"), 0, 2, SV("1234567890"), 0, 10, S("1234567890cde"));
    test(S("abcde"), 0, 2, SV("1234567890"), 0, 11, S("1234567890cde"));
    test(S("abcde"), 0, 2, SV("1234567890"), 1, 0, S("cde"));
    test(S("abcde"), 0, 2, SV("1234567890"), 1, 1, S("2cde"));
    test(S("abcde"), 0, 2, SV("1234567890"), 1, 4, S("2345cde"));
    test(S("abcde"), 0, 2, SV("1234567890"), 1, 8, S("23456789cde"));
    test(S("abcde"), 0, 2, SV("1234567890"), 1, 9, S("234567890cde"));
    test(S("abcde"), 0, 2, SV("1234567890"), 1, 10, S("234567890cde"));
    test(S("abcde"), 0, 2, SV("1234567890"), 5, 0, S("cde"));
    test(S("abcde"), 0, 2, SV("1234567890"), 5, 1, S("6cde"));
    test(S("abcde"), 0, 2, SV("1234567890"), 5, 2, S("67cde"));
    test(S("abcde"), 0, 2, SV("1234567890"), 5, 4, S("6789cde"));
    test(S("abcde"), 0, 2, SV("1234567890"), 5, 5, S("67890cde"));
    test(S("abcde"), 0, 2, SV("1234567890"), 5, 6, S("67890cde"));
    test(S("abcde"), 0, 2, SV("1234567890"), 9, 0, S("cde"));
    test(S("abcde"), 0, 2, SV("1234567890"), 9, 1, S("0cde"));
    test(S("abcde"), 0, 2, SV("1234567890"), 9, 2, S("0cde"));
    test(S("abcde"), 0, 2, SV("1234567890"), 10, 0, S("cde"));
    test(S("abcde"), 0, 2, SV("1234567890"), 10, 1, S("cde"));
    test(S("abcde"), 0, 2, SV("1234567890"), 11, 0, S("can't happen"));
    test(S("abcde"), 0, 2, SV("12345678901234567890"), 0, 0, S("cde"));
    test(S("abcde"), 0, 2, SV("12345678901234567890"), 0, 1, S("1cde"));
    test(S("abcde"), 0, 2, SV("12345678901234567890"), 0, 10, S("1234567890cde"));
    test(S("abcde"), 0, 2, SV("12345678901234567890"), 0, 19, S("1234567890123456789cde"));
    test(S("abcde"), 0, 2, SV("12345678901234567890"), 0, 20, S("12345678901234567890cde"));
    test(S("abcde"), 0, 2, SV("12345678901234567890"), 0, 21, S("12345678901234567890cde"));
    test(S("abcde"), 0, 2, SV("12345678901234567890"), 1, 0, S("cde"));
    test(S("abcde"), 0, 2, SV("12345678901234567890"), 1, 1, S("2cde"));
    test(S("abcde"), 0, 2, SV("12345678901234567890"), 1, 9, S("234567890cde"));
    test(S("abcde"), 0, 2, SV("12345678901234567890"), 1, 18, S("234567890123456789cde"));
    test(S("abcde"), 0, 2, SV("12345678901234567890"), 1, 19, S("2345678901234567890cde"));
    test(S("abcde"), 0, 2, SV("12345678901234567890"), 1, 20, S("2345678901234567890cde"));
    test(S("abcde"), 0, 2, SV("12345678901234567890"), 10, 0, S("cde"));
    test(S("abcde"), 0, 2, SV("12345678901234567890"), 10, 1, S("1cde"));
    test(S("abcde"), 0, 2, SV("12345678901234567890"), 10, 5, S("12345cde"));
    test(S("abcde"), 0, 2, SV("12345678901234567890"), 10, 9, S("123456789cde"));
    test(S("abcde"), 0, 2, SV("12345678901234567890"), 10, 10, S("1234567890cde"));
    test(S("abcde"), 0, 2, SV("12345678901234567890"), 10, 11, S("1234567890cde"));
    test(S("abcde"), 0, 2, SV("12345678901234567890"), 19, 0, S("cde"));
    test(S("abcde"), 0, 2, SV("12345678901234567890"), 19, 1, S("0cde"));
    test(S("abcde"), 0, 2, SV("12345678901234567890"), 19, 2, S("0cde"));
    test(S("abcde"), 0, 2, SV("12345678901234567890"), 20, 0, S("cde"));
    test(S("abcde"), 0, 2, SV("12345678901234567890"), 20, 1, S("cde"));
    test(S("abcde"), 0, 2, SV("12345678901234567890"), 21, 0, S("can't happen"));
    test(S("abcde"), 0, 4, SV(""), 0, 0, S("e"));
    test(S("abcde"), 0, 4, SV(""), 0, 1, S("e"));
    test(S("abcde"), 0, 4, SV(""), 1, 0, S("can't happen"));
    test(S("abcde"), 0, 4, SV("12345"), 0, 0, S("e"));
    test(S("abcde"), 0, 4, SV("12345"), 0, 1, S("1e"));
    test(S("abcde"), 0, 4, SV("12345"), 0, 2, S("12e"));
    test(S("abcde"), 0, 4, SV("12345"), 0, 4, S("1234e"));
    test(S("abcde"), 0, 4, SV("12345"), 0, 5, S("12345e"));
    test(S("abcde"), 0, 4, SV("12345"), 0, 6, S("12345e"));
    test(S("abcde"), 0, 4, SV("12345"), 1, 0, S("e"));
    test(S("abcde"), 0, 4, SV("12345"), 1, 1, S("2e"));
    test(S("abcde"), 0, 4, SV("12345"), 1, 2, S("23e"));
    test(S("abcde"), 0, 4, SV("12345"), 1, 3, S("234e"));
    test(S("abcde"), 0, 4, SV("12345"), 1, 4, S("2345e"));
    test(S("abcde"), 0, 4, SV("12345"), 1, 5, S("2345e"));
    test(S("abcde"), 0, 4, SV("12345"), 2, 0, S("e"));
    test(S("abcde"), 0, 4, SV("12345"), 2, 1, S("3e"));
    test(S("abcde"), 0, 4, SV("12345"), 2, 2, S("34e"));
    test(S("abcde"), 0, 4, SV("12345"), 2, 3, S("345e"));
    test(S("abcde"), 0, 4, SV("12345"), 2, 4, S("345e"));
    test(S("abcde"), 0, 4, SV("12345"), 4, 0, S("e"));
    test(S("abcde"), 0, 4, SV("12345"), 4, 1, S("5e"));
    test(S("abcde"), 0, 4, SV("12345"), 4, 2, S("5e"));
    test(S("abcde"), 0, 4, SV("12345"), 5, 0, S("e"));
    test(S("abcde"), 0, 4, SV("12345"), 5, 1, S("e"));
    test(S("abcde"), 0, 4, SV("12345"), 6, 0, S("can't happen"));
    test(S("abcde"), 0, 4, SV("1234567890"), 0, 0, S("e"));
    test(S("abcde"), 0, 4, SV("1234567890"), 0, 1, S("1e"));
    test(S("abcde"), 0, 4, SV("1234567890"), 0, 5, S("12345e"));
    test(S("abcde"), 0, 4, SV("1234567890"), 0, 9, S("123456789e"));
    test(S("abcde"), 0, 4, SV("1234567890"), 0, 10, S("1234567890e"));
    test(S("abcde"), 0, 4, SV("1234567890"), 0, 11, S("1234567890e"));
    test(S("abcde"), 0, 4, SV("1234567890"), 1, 0, S("e"));
    test(S("abcde"), 0, 4, SV("1234567890"), 1, 1, S("2e"));
    test(S("abcde"), 0, 4, SV("1234567890"), 1, 4, S("2345e"));
    test(S("abcde"), 0, 4, SV("1234567890"), 1, 8, S("23456789e"));
    test(S("abcde"), 0, 4, SV("1234567890"), 1, 9, S("234567890e"));
    test(S("abcde"), 0, 4, SV("1234567890"), 1, 10, S("234567890e"));
    test(S("abcde"), 0, 4, SV("1234567890"), 5, 0, S("e"));
    test(S("abcde"), 0, 4, SV("1234567890"), 5, 1, S("6e"));
    test(S("abcde"), 0, 4, SV("1234567890"), 5, 2, S("67e"));
    test(S("abcde"), 0, 4, SV("1234567890"), 5, 4, S("6789e"));
    test(S("abcde"), 0, 4, SV("1234567890"), 5, 5, S("67890e"));
    test(S("abcde"), 0, 4, SV("1234567890"), 5, 6, S("67890e"));
    test(S("abcde"), 0, 4, SV("1234567890"), 9, 0, S("e"));
    test(S("abcde"), 0, 4, SV("1234567890"), 9, 1, S("0e"));
    test(S("abcde"), 0, 4, SV("1234567890"), 9, 2, S("0e"));
    test(S("abcde"), 0, 4, SV("1234567890"), 10, 0, S("e"));
    test(S("abcde"), 0, 4, SV("1234567890"), 10, 1, S("e"));
    test(S("abcde"), 0, 4, SV("1234567890"), 11, 0, S("can't happen"));
    test(S("abcde"), 0, 4, SV("12345678901234567890"), 0, 0, S("e"));
    test(S("abcde"), 0, 4, SV("12345678901234567890"), 0, 1, S("1e"));
    test(S("abcde"), 0, 4, SV("12345678901234567890"), 0, 10, S("1234567890e"));
    test(S("abcde"), 0, 4, SV("12345678901234567890"), 0, 19, S("1234567890123456789e"));
    test(S("abcde"), 0, 4, SV("12345678901234567890"), 0, 20, S("12345678901234567890e"));
    test(S("abcde"), 0, 4, SV("12345678901234567890"), 0, 21, S("12345678901234567890e"));

    return true;
}

template <class S, class SV>
TEST_CONSTEXPR_CXX20 bool test5()
{
    test(S("abcde"), 0, 4, SV("12345678901234567890"), 1, 0, S("e"));
    test(S("abcde"), 0, 4, SV("12345678901234567890"), 1, 1, S("2e"));
    test(S("abcde"), 0, 4, SV("12345678901234567890"), 1, 9, S("234567890e"));
    test(S("abcde"), 0, 4, SV("12345678901234567890"), 1, 18, S("234567890123456789e"));
    test(S("abcde"), 0, 4, SV("12345678901234567890"), 1, 19, S("2345678901234567890e"));
    test(S("abcde"), 0, 4, SV("12345678901234567890"), 1, 20, S("2345678901234567890e"));
    test(S("abcde"), 0, 4, SV("12345678901234567890"), 10, 0, S("e"));
    test(S("abcde"), 0, 4, SV("12345678901234567890"), 10, 1, S("1e"));
    test(S("abcde"), 0, 4, SV("12345678901234567890"), 10, 5, S("12345e"));
    test(S("abcde"), 0, 4, SV("12345678901234567890"), 10, 9, S("123456789e"));
    test(S("abcde"), 0, 4, SV("12345678901234567890"), 10, 10, S("1234567890e"));
    test(S("abcde"), 0, 4, SV("12345678901234567890"), 10, 11, S("1234567890e"));
    test(S("abcde"), 0, 4, SV("12345678901234567890"), 19, 0, S("e"));
    test(S("abcde"), 0, 4, SV("12345678901234567890"), 19, 1, S("0e"));
    test(S("abcde"), 0, 4, SV("12345678901234567890"), 19, 2, S("0e"));
    test(S("abcde"), 0, 4, SV("12345678901234567890"), 20, 0, S("e"));
    test(S("abcde"), 0, 4, SV("12345678901234567890"), 20, 1, S("e"));
    test(S("abcde"), 0, 4, SV("12345678901234567890"), 21, 0, S("can't happen"));
    test(S("abcde"), 0, 5, SV(""), 0, 0, S(""));
    test(S("abcde"), 0, 5, SV(""), 0, 1, S(""));
    test(S("abcde"), 0, 5, SV(""), 1, 0, S("can't happen"));
    test(S("abcde"), 0, 5, SV("12345"), 0, 0, S(""));
    test(S("abcde"), 0, 5, SV("12345"), 0, 1, S("1"));
    test(S("abcde"), 0, 5, SV("12345"), 0, 2, S("12"));
    test(S("abcde"), 0, 5, SV("12345"), 0, 4, S("1234"));
    test(S("abcde"), 0, 5, SV("12345"), 0, 5, S("12345"));
    test(S("abcde"), 0, 5, SV("12345"), 0, 6, S("12345"));
    test(S("abcde"), 0, 5, SV("12345"), 1, 0, S(""));
    test(S("abcde"), 0, 5, SV("12345"), 1, 1, S("2"));
    test(S("abcde"), 0, 5, SV("12345"), 1, 2, S("23"));
    test(S("abcde"), 0, 5, SV("12345"), 1, 3, S("234"));
    test(S("abcde"), 0, 5, SV("12345"), 1, 4, S("2345"));
    test(S("abcde"), 0, 5, SV("12345"), 1, 5, S("2345"));
    test(S("abcde"), 0, 5, SV("12345"), 2, 0, S(""));
    test(S("abcde"), 0, 5, SV("12345"), 2, 1, S("3"));
    test(S("abcde"), 0, 5, SV("12345"), 2, 2, S("34"));
    test(S("abcde"), 0, 5, SV("12345"), 2, 3, S("345"));
    test(S("abcde"), 0, 5, SV("12345"), 2, 4, S("345"));
    test(S("abcde"), 0, 5, SV("12345"), 4, 0, S(""));
    test(S("abcde"), 0, 5, SV("12345"), 4, 1, S("5"));
    test(S("abcde"), 0, 5, SV("12345"), 4, 2, S("5"));
    test(S("abcde"), 0, 5, SV("12345"), 5, 0, S(""));
    test(S("abcde"), 0, 5, SV("12345"), 5, 1, S(""));
    test(S("abcde"), 0, 5, SV("12345"), 6, 0, S("can't happen"));
    test(S("abcde"), 0, 5, SV("1234567890"), 0, 0, S(""));
    test(S("abcde"), 0, 5, SV("1234567890"), 0, 1, S("1"));
    test(S("abcde"), 0, 5, SV("1234567890"), 0, 5, S("12345"));
    test(S("abcde"), 0, 5, SV("1234567890"), 0, 9, S("123456789"));
    test(S("abcde"), 0, 5, SV("1234567890"), 0, 10, S("1234567890"));
    test(S("abcde"), 0, 5, SV("1234567890"), 0, 11, S("1234567890"));
    test(S("abcde"), 0, 5, SV("1234567890"), 1, 0, S(""));
    test(S("abcde"), 0, 5, SV("1234567890"), 1, 1, S("2"));
    test(S("abcde"), 0, 5, SV("1234567890"), 1, 4, S("2345"));
    test(S("abcde"), 0, 5, SV("1234567890"), 1, 8, S("23456789"));
    test(S("abcde"), 0, 5, SV("1234567890"), 1, 9, S("234567890"));
    test(S("abcde"), 0, 5, SV("1234567890"), 1, 10, S("234567890"));
    test(S("abcde"), 0, 5, SV("1234567890"), 5, 0, S(""));
    test(S("abcde"), 0, 5, SV("1234567890"), 5, 1, S("6"));
    test(S("abcde"), 0, 5, SV("1234567890"), 5, 2, S("67"));
    test(S("abcde"), 0, 5, SV("1234567890"), 5, 4, S("6789"));
    test(S("abcde"), 0, 5, SV("1234567890"), 5, 5, S("67890"));
    test(S("abcde"), 0, 5, SV("1234567890"), 5, 6, S("67890"));
    test(S("abcde"), 0, 5, SV("1234567890"), 9, 0, S(""));
    test(S("abcde"), 0, 5, SV("1234567890"), 9, 1, S("0"));
    test(S("abcde"), 0, 5, SV("1234567890"), 9, 2, S("0"));
    test(S("abcde"), 0, 5, SV("1234567890"), 10, 0, S(""));
    test(S("abcde"), 0, 5, SV("1234567890"), 10, 1, S(""));
    test(S("abcde"), 0, 5, SV("1234567890"), 11, 0, S("can't happen"));
    test(S("abcde"), 0, 5, SV("12345678901234567890"), 0, 0, S(""));
    test(S("abcde"), 0, 5, SV("12345678901234567890"), 0, 1, S("1"));
    test(S("abcde"), 0, 5, SV("12345678901234567890"), 0, 10, S("1234567890"));
    test(S("abcde"), 0, 5, SV("12345678901234567890"), 0, 19, S("1234567890123456789"));
    test(S("abcde"), 0, 5, SV("12345678901234567890"), 0, 20, S("12345678901234567890"));
    test(S("abcde"), 0, 5, SV("12345678901234567890"), 0, 21, S("12345678901234567890"));
    test(S("abcde"), 0, 5, SV("12345678901234567890"), 1, 0, S(""));
    test(S("abcde"), 0, 5, SV("12345678901234567890"), 1, 1, S("2"));
    test(S("abcde"), 0, 5, SV("12345678901234567890"), 1, 9, S("234567890"));
    test(S("abcde"), 0, 5, SV("12345678901234567890"), 1, 18, S("234567890123456789"));
    test(S("abcde"), 0, 5, SV("12345678901234567890"), 1, 19, S("2345678901234567890"));
    test(S("abcde"), 0, 5, SV("12345678901234567890"), 1, 20, S("2345678901234567890"));
    test(S("abcde"), 0, 5, SV("12345678901234567890"), 10, 0, S(""));
    test(S("abcde"), 0, 5, SV("12345678901234567890"), 10, 1, S("1"));
    test(S("abcde"), 0, 5, SV("12345678901234567890"), 10, 5, S("12345"));
    test(S("abcde"), 0, 5, SV("12345678901234567890"), 10, 9, S("123456789"));
    test(S("abcde"), 0, 5, SV("12345678901234567890"), 10, 10, S("1234567890"));
    test(S("abcde"), 0, 5, SV("12345678901234567890"), 10, 11, S("1234567890"));
    test(S("abcde"), 0, 5, SV("12345678901234567890"), 19, 0, S(""));
    test(S("abcde"), 0, 5, SV("12345678901234567890"), 19, 1, S("0"));
    test(S("abcde"), 0, 5, SV("12345678901234567890"), 19, 2, S("0"));
    test(S("abcde"), 0, 5, SV("12345678901234567890"), 20, 0, S(""));
    test(S("abcde"), 0, 5, SV("12345678901234567890"), 20, 1, S(""));
    test(S("abcde"), 0, 5, SV("12345678901234567890"), 21, 0, S("can't happen"));
    test(S("abcde"), 0, 6, SV(""), 0, 0, S(""));
    test(S("abcde"), 0, 6, SV(""), 0, 1, S(""));
    test(S("abcde"), 0, 6, SV(""), 1, 0, S("can't happen"));
    test(S("abcde"), 0, 6, SV("12345"), 0, 0, S(""));
    test(S("abcde"), 0, 6, SV("12345"), 0, 1, S("1"));
    test(S("abcde"), 0, 6, SV("12345"), 0, 2, S("12"));
    test(S("abcde"), 0, 6, SV("12345"), 0, 4, S("1234"));
    test(S("abcde"), 0, 6, SV("12345"), 0, 5, S("12345"));

    return true;
}

template <class S, class SV>
TEST_CONSTEXPR_CXX20 bool test6()
{
    test(S("abcde"), 0, 6, SV("12345"), 0, 6, S("12345"));
    test(S("abcde"), 0, 6, SV("12345"), 1, 0, S(""));
    test(S("abcde"), 0, 6, SV("12345"), 1, 1, S("2"));
    test(S("abcde"), 0, 6, SV("12345"), 1, 2, S("23"));
    test(S("abcde"), 0, 6, SV("12345"), 1, 3, S("234"));
    test(S("abcde"), 0, 6, SV("12345"), 1, 4, S("2345"));
    test(S("abcde"), 0, 6, SV("12345"), 1, 5, S("2345"));
    test(S("abcde"), 0, 6, SV("12345"), 2, 0, S(""));
    test(S("abcde"), 0, 6, SV("12345"), 2, 1, S("3"));
    test(S("abcde"), 0, 6, SV("12345"), 2, 2, S("34"));
    test(S("abcde"), 0, 6, SV("12345"), 2, 3, S("345"));
    test(S("abcde"), 0, 6, SV("12345"), 2, 4, S("345"));
    test(S("abcde"), 0, 6, SV("12345"), 4, 0, S(""));
    test(S("abcde"), 0, 6, SV("12345"), 4, 1, S("5"));
    test(S("abcde"), 0, 6, SV("12345"), 4, 2, S("5"));
    test(S("abcde"), 0, 6, SV("12345"), 5, 0, S(""));
    test(S("abcde"), 0, 6, SV("12345"), 5, 1, S(""));
    test(S("abcde"), 0, 6, SV("12345"), 6, 0, S("can't happen"));
    test(S("abcde"), 0, 6, SV("1234567890"), 0, 0, S(""));
    test(S("abcde"), 0, 6, SV("1234567890"), 0, 1, S("1"));
    test(S("abcde"), 0, 6, SV("1234567890"), 0, 5, S("12345"));
    test(S("abcde"), 0, 6, SV("1234567890"), 0, 9, S("123456789"));
    test(S("abcde"), 0, 6, SV("1234567890"), 0, 10, S("1234567890"));
    test(S("abcde"), 0, 6, SV("1234567890"), 0, 11, S("1234567890"));
    test(S("abcde"), 0, 6, SV("1234567890"), 1, 0, S(""));
    test(S("abcde"), 0, 6, SV("1234567890"), 1, 1, S("2"));
    test(S("abcde"), 0, 6, SV("1234567890"), 1, 4, S("2345"));
    test(S("abcde"), 0, 6, SV("1234567890"), 1, 8, S("23456789"));
    test(S("abcde"), 0, 6, SV("1234567890"), 1, 9, S("234567890"));
    test(S("abcde"), 0, 6, SV("1234567890"), 1, 10, S("234567890"));
    test(S("abcde"), 0, 6, SV("1234567890"), 5, 0, S(""));
    test(S("abcde"), 0, 6, SV("1234567890"), 5, 1, S("6"));
    test(S("abcde"), 0, 6, SV("1234567890"), 5, 2, S("67"));
    test(S("abcde"), 0, 6, SV("1234567890"), 5, 4, S("6789"));
    test(S("abcde"), 0, 6, SV("1234567890"), 5, 5, S("67890"));
    test(S("abcde"), 0, 6, SV("1234567890"), 5, 6, S("67890"));
    test(S("abcde"), 0, 6, SV("1234567890"), 9, 0, S(""));
    test(S("abcde"), 0, 6, SV("1234567890"), 9, 1, S("0"));
    test(S("abcde"), 0, 6, SV("1234567890"), 9, 2, S("0"));
    test(S("abcde"), 0, 6, SV("1234567890"), 10, 0, S(""));
    test(S("abcde"), 0, 6, SV("1234567890"), 10, 1, S(""));
    test(S("abcde"), 0, 6, SV("1234567890"), 11, 0, S("can't happen"));
    test(S("abcde"), 0, 6, SV("12345678901234567890"), 0, 0, S(""));
    test(S("abcde"), 0, 6, SV("12345678901234567890"), 0, 1, S("1"));
    test(S("abcde"), 0, 6, SV("12345678901234567890"), 0, 10, S("1234567890"));
    test(S("abcde"), 0, 6, SV("12345678901234567890"), 0, 19, S("1234567890123456789"));
    test(S("abcde"), 0, 6, SV("12345678901234567890"), 0, 20, S("12345678901234567890"));
    test(S("abcde"), 0, 6, SV("12345678901234567890"), 0, 21, S("12345678901234567890"));
    test(S("abcde"), 0, 6, SV("12345678901234567890"), 1, 0, S(""));
    test(S("abcde"), 0, 6, SV("12345678901234567890"), 1, 1, S("2"));
    test(S("abcde"), 0, 6, SV("12345678901234567890"), 1, 9, S("234567890"));
    test(S("abcde"), 0, 6, SV("12345678901234567890"), 1, 18, S("234567890123456789"));
    test(S("abcde"), 0, 6, SV("12345678901234567890"), 1, 19, S("2345678901234567890"));
    test(S("abcde"), 0, 6, SV("12345678901234567890"), 1, 20, S("2345678901234567890"));
    test(S("abcde"), 0, 6, SV("12345678901234567890"), 10, 0, S(""));
    test(S("abcde"), 0, 6, SV("12345678901234567890"), 10, 1, S("1"));
    test(S("abcde"), 0, 6, SV("12345678901234567890"), 10, 5, S("12345"));
    test(S("abcde"), 0, 6, SV("12345678901234567890"), 10, 9, S("123456789"));
    test(S("abcde"), 0, 6, SV("12345678901234567890"), 10, 10, S("1234567890"));
    test(S("abcde"), 0, 6, SV("12345678901234567890"), 10, 11, S("1234567890"));
    test(S("abcde"), 0, 6, SV("12345678901234567890"), 19, 0, S(""));
    test(S("abcde"), 0, 6, SV("12345678901234567890"), 19, 1, S("0"));
    test(S("abcde"), 0, 6, SV("12345678901234567890"), 19, 2, S("0"));
    test(S("abcde"), 0, 6, SV("12345678901234567890"), 20, 0, S(""));
    test(S("abcde"), 0, 6, SV("12345678901234567890"), 20, 1, S(""));
    test(S("abcde"), 0, 6, SV("12345678901234567890"), 21, 0, S("can't happen"));
    test(S("abcde"), 1, 0, SV(""), 0, 0, S("abcde"));
    test(S("abcde"), 1, 0, SV(""), 0, 1, S("abcde"));
    test(S("abcde"), 1, 0, SV(""), 1, 0, S("can't happen"));
    test(S("abcde"), 1, 0, SV("12345"), 0, 0, S("abcde"));
    test(S("abcde"), 1, 0, SV("12345"), 0, 1, S("a1bcde"));
    test(S("abcde"), 1, 0, SV("12345"), 0, 2, S("a12bcde"));
    test(S("abcde"), 1, 0, SV("12345"), 0, 4, S("a1234bcde"));
    test(S("abcde"), 1, 0, SV("12345"), 0, 5, S("a12345bcde"));
    test(S("abcde"), 1, 0, SV("12345"), 0, 6, S("a12345bcde"));
    test(S("abcde"), 1, 0, SV("12345"), 1, 0, S("abcde"));
    test(S("abcde"), 1, 0, SV("12345"), 1, 1, S("a2bcde"));
    test(S("abcde"), 1, 0, SV("12345"), 1, 2, S("a23bcde"));
    test(S("abcde"), 1, 0, SV("12345"), 1, 3, S("a234bcde"));
    test(S("abcde"), 1, 0, SV("12345"), 1, 4, S("a2345bcde"));
    test(S("abcde"), 1, 0, SV("12345"), 1, 5, S("a2345bcde"));
    test(S("abcde"), 1, 0, SV("12345"), 2, 0, S("abcde"));
    test(S("abcde"), 1, 0, SV("12345"), 2, 1, S("a3bcde"));
    test(S("abcde"), 1, 0, SV("12345"), 2, 2, S("a34bcde"));
    test(S("abcde"), 1, 0, SV("12345"), 2, 3, S("a345bcde"));
    test(S("abcde"), 1, 0, SV("12345"), 2, 4, S("a345bcde"));
    test(S("abcde"), 1, 0, SV("12345"), 4, 0, S("abcde"));
    test(S("abcde"), 1, 0, SV("12345"), 4, 1, S("a5bcde"));
    test(S("abcde"), 1, 0, SV("12345"), 4, 2, S("a5bcde"));
    test(S("abcde"), 1, 0, SV("12345"), 5, 0, S("abcde"));
    test(S("abcde"), 1, 0, SV("12345"), 5, 1, S("abcde"));
    test(S("abcde"), 1, 0, SV("12345"), 6, 0, S("can't happen"));
    test(S("abcde"), 1, 0, SV("1234567890"), 0, 0, S("abcde"));
    test(S("abcde"), 1, 0, SV("1234567890"), 0, 1, S("a1bcde"));
    test(S("abcde"), 1, 0, SV("1234567890"), 0, 5, S("a12345bcde"));
    test(S("abcde"), 1, 0, SV("1234567890"), 0, 9, S("a123456789bcde"));
    test(S("abcde"), 1, 0, SV("1234567890"), 0, 10, S("a1234567890bcde"));
    test(S("abcde"), 1, 0, SV("1234567890"), 0, 11, S("a1234567890bcde"));
    test(S("abcde"), 1, 0, SV("1234567890"), 1, 0, S("abcde"));
    test(S("abcde"), 1, 0, SV("1234567890"), 1, 1, S("a2bcde"));

    return true;
}

template <class S, class SV>
TEST_CONSTEXPR_CXX20 bool test7()
{
    test(S("abcde"), 1, 0, SV("1234567890"), 1, 4, S("a2345bcde"));
    test(S("abcde"), 1, 0, SV("1234567890"), 1, 8, S("a23456789bcde"));
    test(S("abcde"), 1, 0, SV("1234567890"), 1, 9, S("a234567890bcde"));
    test(S("abcde"), 1, 0, SV("1234567890"), 1, 10, S("a234567890bcde"));
    test(S("abcde"), 1, 0, SV("1234567890"), 5, 0, S("abcde"));
    test(S("abcde"), 1, 0, SV("1234567890"), 5, 1, S("a6bcde"));
    test(S("abcde"), 1, 0, SV("1234567890"), 5, 2, S("a67bcde"));
    test(S("abcde"), 1, 0, SV("1234567890"), 5, 4, S("a6789bcde"));
    test(S("abcde"), 1, 0, SV("1234567890"), 5, 5, S("a67890bcde"));
    test(S("abcde"), 1, 0, SV("1234567890"), 5, 6, S("a67890bcde"));
    test(S("abcde"), 1, 0, SV("1234567890"), 9, 0, S("abcde"));
    test(S("abcde"), 1, 0, SV("1234567890"), 9, 1, S("a0bcde"));
    test(S("abcde"), 1, 0, SV("1234567890"), 9, 2, S("a0bcde"));
    test(S("abcde"), 1, 0, SV("1234567890"), 10, 0, S("abcde"));
    test(S("abcde"), 1, 0, SV("1234567890"), 10, 1, S("abcde"));
    test(S("abcde"), 1, 0, SV("1234567890"), 11, 0, S("can't happen"));
    test(S("abcde"), 1, 0, SV("12345678901234567890"), 0, 0, S("abcde"));
    test(S("abcde"), 1, 0, SV("12345678901234567890"), 0, 1, S("a1bcde"));
    test(S("abcde"), 1, 0, SV("12345678901234567890"), 0, 10, S("a1234567890bcde"));
    test(S("abcde"), 1, 0, SV("12345678901234567890"), 0, 19, S("a1234567890123456789bcde"));
    test(S("abcde"), 1, 0, SV("12345678901234567890"), 0, 20, S("a12345678901234567890bcde"));
    test(S("abcde"), 1, 0, SV("12345678901234567890"), 0, 21, S("a12345678901234567890bcde"));
    test(S("abcde"), 1, 0, SV("12345678901234567890"), 1, 0, S("abcde"));
    test(S("abcde"), 1, 0, SV("12345678901234567890"), 1, 1, S("a2bcde"));
    test(S("abcde"), 1, 0, SV("12345678901234567890"), 1, 9, S("a234567890bcde"));
    test(S("abcde"), 1, 0, SV("12345678901234567890"), 1, 18, S("a234567890123456789bcde"));
    test(S("abcde"), 1, 0, SV("12345678901234567890"), 1, 19, S("a2345678901234567890bcde"));
    test(S("abcde"), 1, 0, SV("12345678901234567890"), 1, 20, S("a2345678901234567890bcde"));
    test(S("abcde"), 1, 0, SV("12345678901234567890"), 10, 0, S("abcde"));
    test(S("abcde"), 1, 0, SV("12345678901234567890"), 10, 1, S("a1bcde"));
    test(S("abcde"), 1, 0, SV("12345678901234567890"), 10, 5, S("a12345bcde"));
    test(S("abcde"), 1, 0, SV("12345678901234567890"), 10, 9, S("a123456789bcde"));
    test(S("abcde"), 1, 0, SV("12345678901234567890"), 10, 10, S("a1234567890bcde"));
    test(S("abcde"), 1, 0, SV("12345678901234567890"), 10, 11, S("a1234567890bcde"));
    test(S("abcde"), 1, 0, SV("12345678901234567890"), 19, 0, S("abcde"));
    test(S("abcde"), 1, 0, SV("12345678901234567890"), 19, 1, S("a0bcde"));
    test(S("abcde"), 1, 0, SV("12345678901234567890"), 19, 2, S("a0bcde"));
    test(S("abcde"), 1, 0, SV("12345678901234567890"), 20, 0, S("abcde"));
    test(S("abcde"), 1, 0, SV("12345678901234567890"), 20, 1, S("abcde"));
    test(S("abcde"), 1, 0, SV("12345678901234567890"), 21, 0, S("can't happen"));
    test(S("abcde"), 1, 1, SV(""), 0, 0, S("acde"));
    test(S("abcde"), 1, 1, SV(""), 0, 1, S("acde"));
    test(S("abcde"), 1, 1, SV(""), 1, 0, S("can't happen"));
    test(S("abcde"), 1, 1, SV("12345"), 0, 0, S("acde"));
    test(S("abcde"), 1, 1, SV("12345"), 0, 1, S("a1cde"));
    test(S("abcde"), 1, 1, SV("12345"), 0, 2, S("a12cde"));
    test(S("abcde"), 1, 1, SV("12345"), 0, 4, S("a1234cde"));
    test(S("abcde"), 1, 1, SV("12345"), 0, 5, S("a12345cde"));
    test(S("abcde"), 1, 1, SV("12345"), 0, 6, S("a12345cde"));
    test(S("abcde"), 1, 1, SV("12345"), 1, 0, S("acde"));
    test(S("abcde"), 1, 1, SV("12345"), 1, 1, S("a2cde"));
    test(S("abcde"), 1, 1, SV("12345"), 1, 2, S("a23cde"));
    test(S("abcde"), 1, 1, SV("12345"), 1, 3, S("a234cde"));
    test(S("abcde"), 1, 1, SV("12345"), 1, 4, S("a2345cde"));
    test(S("abcde"), 1, 1, SV("12345"), 1, 5, S("a2345cde"));
    test(S("abcde"), 1, 1, SV("12345"), 2, 0, S("acde"));
    test(S("abcde"), 1, 1, SV("12345"), 2, 1, S("a3cde"));
    test(S("abcde"), 1, 1, SV("12345"), 2, 2, S("a34cde"));
    test(S("abcde"), 1, 1, SV("12345"), 2, 3, S("a345cde"));
    test(S("abcde"), 1, 1, SV("12345"), 2, 4, S("a345cde"));
    test(S("abcde"), 1, 1, SV("12345"), 4, 0, S("acde"));
    test(S("abcde"), 1, 1, SV("12345"), 4, 1, S("a5cde"));
    test(S("abcde"), 1, 1, SV("12345"), 4, 2, S("a5cde"));
    test(S("abcde"), 1, 1, SV("12345"), 5, 0, S("acde"));
    test(S("abcde"), 1, 1, SV("12345"), 5, 1, S("acde"));
    test(S("abcde"), 1, 1, SV("12345"), 6, 0, S("can't happen"));
    test(S("abcde"), 1, 1, SV("1234567890"), 0, 0, S("acde"));
    test(S("abcde"), 1, 1, SV("1234567890"), 0, 1, S("a1cde"));
    test(S("abcde"), 1, 1, SV("1234567890"), 0, 5, S("a12345cde"));
    test(S("abcde"), 1, 1, SV("1234567890"), 0, 9, S("a123456789cde"));
    test(S("abcde"), 1, 1, SV("1234567890"), 0, 10, S("a1234567890cde"));
    test(S("abcde"), 1, 1, SV("1234567890"), 0, 11, S("a1234567890cde"));
    test(S("abcde"), 1, 1, SV("1234567890"), 1, 0, S("acde"));
    test(S("abcde"), 1, 1, SV("1234567890"), 1, 1, S("a2cde"));
    test(S("abcde"), 1, 1, SV("1234567890"), 1, 4, S("a2345cde"));
    test(S("abcde"), 1, 1, SV("1234567890"), 1, 8, S("a23456789cde"));
    test(S("abcde"), 1, 1, SV("1234567890"), 1, 9, S("a234567890cde"));
    test(S("abcde"), 1, 1, SV("1234567890"), 1, 10, S("a234567890cde"));
    test(S("abcde"), 1, 1, SV("1234567890"), 5, 0, S("acde"));
    test(S("abcde"), 1, 1, SV("1234567890"), 5, 1, S("a6cde"));
    test(S("abcde"), 1, 1, SV("1234567890"), 5, 2, S("a67cde"));
    test(S("abcde"), 1, 1, SV("1234567890"), 5, 4, S("a6789cde"));
    test(S("abcde"), 1, 1, SV("1234567890"), 5, 5, S("a67890cde"));
    test(S("abcde"), 1, 1, SV("1234567890"), 5, 6, S("a67890cde"));
    test(S("abcde"), 1, 1, SV("1234567890"), 9, 0, S("acde"));
    test(S("abcde"), 1, 1, SV("1234567890"), 9, 1, S("a0cde"));
    test(S("abcde"), 1, 1, SV("1234567890"), 9, 2, S("a0cde"));
    test(S("abcde"), 1, 1, SV("1234567890"), 10, 0, S("acde"));
    test(S("abcde"), 1, 1, SV("1234567890"), 10, 1, S("acde"));
    test(S("abcde"), 1, 1, SV("1234567890"), 11, 0, S("can't happen"));
    test(S("abcde"), 1, 1, SV("12345678901234567890"), 0, 0, S("acde"));
    test(S("abcde"), 1, 1, SV("12345678901234567890"), 0, 1, S("a1cde"));
    test(S("abcde"), 1, 1, SV("12345678901234567890"), 0, 10, S("a1234567890cde"));
    test(S("abcde"), 1, 1, SV("12345678901234567890"), 0, 19, S("a1234567890123456789cde"));
    test(S("abcde"), 1, 1, SV("12345678901234567890"), 0, 20, S("a12345678901234567890cde"));
    test(S("abcde"), 1, 1, SV("12345678901234567890"), 0, 21, S("a12345678901234567890cde"));
    test(S("abcde"), 1, 1, SV("12345678901234567890"), 1, 0, S("acde"));
    test(S("abcde"), 1, 1, SV("12345678901234567890"), 1, 1, S("a2cde"));
    test(S("abcde"), 1, 1, SV("12345678901234567890"), 1, 9, S("a234567890cde"));
    test(S("abcde"), 1, 1, SV("12345678901234567890"), 1, 18, S("a234567890123456789cde"));

    return true;
}

template <class S, class SV>
TEST_CONSTEXPR_CXX20 bool test8()
{
    test(S("abcde"), 1, 1, SV("12345678901234567890"), 1, 19, S("a2345678901234567890cde"));
    test(S("abcde"), 1, 1, SV("12345678901234567890"), 1, 20, S("a2345678901234567890cde"));
    test(S("abcde"), 1, 1, SV("12345678901234567890"), 10, 0, S("acde"));
    test(S("abcde"), 1, 1, SV("12345678901234567890"), 10, 1, S("a1cde"));
    test(S("abcde"), 1, 1, SV("12345678901234567890"), 10, 5, S("a12345cde"));
    test(S("abcde"), 1, 1, SV("12345678901234567890"), 10, 9, S("a123456789cde"));
    test(S("abcde"), 1, 1, SV("12345678901234567890"), 10, 10, S("a1234567890cde"));
    test(S("abcde"), 1, 1, SV("12345678901234567890"), 10, 11, S("a1234567890cde"));
    test(S("abcde"), 1, 1, SV("12345678901234567890"), 19, 0, S("acde"));
    test(S("abcde"), 1, 1, SV("12345678901234567890"), 19, 1, S("a0cde"));
    test(S("abcde"), 1, 1, SV("12345678901234567890"), 19, 2, S("a0cde"));
    test(S("abcde"), 1, 1, SV("12345678901234567890"), 20, 0, S("acde"));
    test(S("abcde"), 1, 1, SV("12345678901234567890"), 20, 1, S("acde"));
    test(S("abcde"), 1, 1, SV("12345678901234567890"), 21, 0, S("can't happen"));
    test(S("abcde"), 1, 2, SV(""), 0, 0, S("ade"));
    test(S("abcde"), 1, 2, SV(""), 0, 1, S("ade"));
    test(S("abcde"), 1, 2, SV(""), 1, 0, S("can't happen"));
    test(S("abcde"), 1, 2, SV("12345"), 0, 0, S("ade"));
    test(S("abcde"), 1, 2, SV("12345"), 0, 1, S("a1de"));
    test(S("abcde"), 1, 2, SV("12345"), 0, 2, S("a12de"));
    test(S("abcde"), 1, 2, SV("12345"), 0, 4, S("a1234de"));
    test(S("abcde"), 1, 2, SV("12345"), 0, 5, S("a12345de"));
    test(S("abcde"), 1, 2, SV("12345"), 0, 6, S("a12345de"));
    test(S("abcde"), 1, 2, SV("12345"), 1, 0, S("ade"));
    test(S("abcde"), 1, 2, SV("12345"), 1, 1, S("a2de"));
    test(S("abcde"), 1, 2, SV("12345"), 1, 2, S("a23de"));
    test(S("abcde"), 1, 2, SV("12345"), 1, 3, S("a234de"));
    test(S("abcde"), 1, 2, SV("12345"), 1, 4, S("a2345de"));
    test(S("abcde"), 1, 2, SV("12345"), 1, 5, S("a2345de"));
    test(S("abcde"), 1, 2, SV("12345"), 2, 0, S("ade"));
    test(S("abcde"), 1, 2, SV("12345"), 2, 1, S("a3de"));
    test(S("abcde"), 1, 2, SV("12345"), 2, 2, S("a34de"));
    test(S("abcde"), 1, 2, SV("12345"), 2, 3, S("a345de"));
    test(S("abcde"), 1, 2, SV("12345"), 2, 4, S("a345de"));
    test(S("abcde"), 1, 2, SV("12345"), 4, 0, S("ade"));
    test(S("abcde"), 1, 2, SV("12345"), 4, 1, S("a5de"));
    test(S("abcde"), 1, 2, SV("12345"), 4, 2, S("a5de"));
    test(S("abcde"), 1, 2, SV("12345"), 5, 0, S("ade"));
    test(S("abcde"), 1, 2, SV("12345"), 5, 1, S("ade"));
    test(S("abcde"), 1, 2, SV("12345"), 6, 0, S("can't happen"));
    test(S("abcde"), 1, 2, SV("1234567890"), 0, 0, S("ade"));
    test(S("abcde"), 1, 2, SV("1234567890"), 0, 1, S("a1de"));
    test(S("abcde"), 1, 2, SV("1234567890"), 0, 5, S("a12345de"));
    test(S("abcde"), 1, 2, SV("1234567890"), 0, 9, S("a123456789de"));
    test(S("abcde"), 1, 2, SV("1234567890"), 0, 10, S("a1234567890de"));
    test(S("abcde"), 1, 2, SV("1234567890"), 0, 11, S("a1234567890de"));
    test(S("abcde"), 1, 2, SV("1234567890"), 1, 0, S("ade"));
    test(S("abcde"), 1, 2, SV("1234567890"), 1, 1, S("a2de"));
    test(S("abcde"), 1, 2, SV("1234567890"), 1, 4, S("a2345de"));
    test(S("abcde"), 1, 2, SV("1234567890"), 1, 8, S("a23456789de"));
    test(S("abcde"), 1, 2, SV("1234567890"), 1, 9, S("a234567890de"));
    test(S("abcde"), 1, 2, SV("1234567890"), 1, 10, S("a234567890de"));
    test(S("abcde"), 1, 2, SV("1234567890"), 5, 0, S("ade"));
    test(S("abcde"), 1, 2, SV("1234567890"), 5, 1, S("a6de"));
    test(S("abcde"), 1, 2, SV("1234567890"), 5, 2, S("a67de"));
    test(S("abcde"), 1, 2, SV("1234567890"), 5, 4, S("a6789de"));
    test(S("abcde"), 1, 2, SV("1234567890"), 5, 5, S("a67890de"));
    test(S("abcde"), 1, 2, SV("1234567890"), 5, 6, S("a67890de"));
    test(S("abcde"), 1, 2, SV("1234567890"), 9, 0, S("ade"));
    test(S("abcde"), 1, 2, SV("1234567890"), 9, 1, S("a0de"));
    test(S("abcde"), 1, 2, SV("1234567890"), 9, 2, S("a0de"));
    test(S("abcde"), 1, 2, SV("1234567890"), 10, 0, S("ade"));
    test(S("abcde"), 1, 2, SV("1234567890"), 10, 1, S("ade"));
    test(S("abcde"), 1, 2, SV("1234567890"), 11, 0, S("can't happen"));
    test(S("abcde"), 1, 2, SV("12345678901234567890"), 0, 0, S("ade"));
    test(S("abcde"), 1, 2, SV("12345678901234567890"), 0, 1, S("a1de"));
    test(S("abcde"), 1, 2, SV("12345678901234567890"), 0, 10, S("a1234567890de"));
    test(S("abcde"), 1, 2, SV("12345678901234567890"), 0, 19, S("a1234567890123456789de"));
    test(S("abcde"), 1, 2, SV("12345678901234567890"), 0, 20, S("a12345678901234567890de"));
    test(S("abcde"), 1, 2, SV("12345678901234567890"), 0, 21, S("a12345678901234567890de"));
    test(S("abcde"), 1, 2, SV("12345678901234567890"), 1, 0, S("ade"));
    test(S("abcde"), 1, 2, SV("12345678901234567890"), 1, 1, S("a2de"));
    test(S("abcde"), 1, 2, SV("12345678901234567890"), 1, 9, S("a234567890de"));
    test(S("abcde"), 1, 2, SV("12345678901234567890"), 1, 18, S("a234567890123456789de"));
    test(S("abcde"), 1, 2, SV("12345678901234567890"), 1, 19, S("a2345678901234567890de"));
    test(S("abcde"), 1, 2, SV("12345678901234567890"), 1, 20, S("a2345678901234567890de"));
    test(S("abcde"), 1, 2, SV("12345678901234567890"), 10, 0, S("ade"));
    test(S("abcde"), 1, 2, SV("12345678901234567890"), 10, 1, S("a1de"));
    test(S("abcde"), 1, 2, SV("12345678901234567890"), 10, 5, S("a12345de"));
    test(S("abcde"), 1, 2, SV("12345678901234567890"), 10, 9, S("a123456789de"));
    test(S("abcde"), 1, 2, SV("12345678901234567890"), 10, 10, S("a1234567890de"));
    test(S("abcde"), 1, 2, SV("12345678901234567890"), 10, 11, S("a1234567890de"));
    test(S("abcde"), 1, 2, SV("12345678901234567890"), 19, 0, S("ade"));
    test(S("abcde"), 1, 2, SV("12345678901234567890"), 19, 1, S("a0de"));
    test(S("abcde"), 1, 2, SV("12345678901234567890"), 19, 2, S("a0de"));
    test(S("abcde"), 1, 2, SV("12345678901234567890"), 20, 0, S("ade"));
    test(S("abcde"), 1, 2, SV("12345678901234567890"), 20, 1, S("ade"));
    test(S("abcde"), 1, 2, SV("12345678901234567890"), 21, 0, S("can't happen"));
    test(S("abcde"), 1, 3, SV(""), 0, 0, S("ae"));
    test(S("abcde"), 1, 3, SV(""), 0, 1, S("ae"));
    test(S("abcde"), 1, 3, SV(""), 1, 0, S("can't happen"));
    test(S("abcde"), 1, 3, SV("12345"), 0, 0, S("ae"));
    test(S("abcde"), 1, 3, SV("12345"), 0, 1, S("a1e"));
    test(S("abcde"), 1, 3, SV("12345"), 0, 2, S("a12e"));
    test(S("abcde"), 1, 3, SV("12345"), 0, 4, S("a1234e"));
    test(S("abcde"), 1, 3, SV("12345"), 0, 5, S("a12345e"));
    test(S("abcde"), 1, 3, SV("12345"), 0, 6, S("a12345e"));
    test(S("abcde"), 1, 3, SV("12345"), 1, 0, S("ae"));
    test(S("abcde"), 1, 3, SV("12345"), 1, 1, S("a2e"));
    test(S("abcde"), 1, 3, SV("12345"), 1, 2, S("a23e"));

    return true;
}

template <class S, class SV>
TEST_CONSTEXPR_CXX20 bool test9()
{
    test(S("abcde"), 1, 3, SV("12345"), 1, 3, S("a234e"));
    test(S("abcde"), 1, 3, SV("12345"), 1, 4, S("a2345e"));
    test(S("abcde"), 1, 3, SV("12345"), 1, 5, S("a2345e"));
    test(S("abcde"), 1, 3, SV("12345"), 2, 0, S("ae"));
    test(S("abcde"), 1, 3, SV("12345"), 2, 1, S("a3e"));
    test(S("abcde"), 1, 3, SV("12345"), 2, 2, S("a34e"));
    test(S("abcde"), 1, 3, SV("12345"), 2, 3, S("a345e"));
    test(S("abcde"), 1, 3, SV("12345"), 2, 4, S("a345e"));
    test(S("abcde"), 1, 3, SV("12345"), 4, 0, S("ae"));
    test(S("abcde"), 1, 3, SV("12345"), 4, 1, S("a5e"));
    test(S("abcde"), 1, 3, SV("12345"), 4, 2, S("a5e"));
    test(S("abcde"), 1, 3, SV("12345"), 5, 0, S("ae"));
    test(S("abcde"), 1, 3, SV("12345"), 5, 1, S("ae"));
    test(S("abcde"), 1, 3, SV("12345"), 6, 0, S("can't happen"));
    test(S("abcde"), 1, 3, SV("1234567890"), 0, 0, S("ae"));
    test(S("abcde"), 1, 3, SV("1234567890"), 0, 1, S("a1e"));
    test(S("abcde"), 1, 3, SV("1234567890"), 0, 5, S("a12345e"));
    test(S("abcde"), 1, 3, SV("1234567890"), 0, 9, S("a123456789e"));
    test(S("abcde"), 1, 3, SV("1234567890"), 0, 10, S("a1234567890e"));
    test(S("abcde"), 1, 3, SV("1234567890"), 0, 11, S("a1234567890e"));
    test(S("abcde"), 1, 3, SV("1234567890"), 1, 0, S("ae"));
    test(S("abcde"), 1, 3, SV("1234567890"), 1, 1, S("a2e"));
    test(S("abcde"), 1, 3, SV("1234567890"), 1, 4, S("a2345e"));
    test(S("abcde"), 1, 3, SV("1234567890"), 1, 8, S("a23456789e"));
    test(S("abcde"), 1, 3, SV("1234567890"), 1, 9, S("a234567890e"));
    test(S("abcde"), 1, 3, SV("1234567890"), 1, 10, S("a234567890e"));
    test(S("abcde"), 1, 3, SV("1234567890"), 5, 0, S("ae"));
    test(S("abcde"), 1, 3, SV("1234567890"), 5, 1, S("a6e"));
    test(S("abcde"), 1, 3, SV("1234567890"), 5, 2, S("a67e"));
    test(S("abcde"), 1, 3, SV("1234567890"), 5, 4, S("a6789e"));
    test(S("abcde"), 1, 3, SV("1234567890"), 5, 5, S("a67890e"));
    test(S("abcde"), 1, 3, SV("1234567890"), 5, 6, S("a67890e"));
    test(S("abcde"), 1, 3, SV("1234567890"), 9, 0, S("ae"));
    test(S("abcde"), 1, 3, SV("1234567890"), 9, 1, S("a0e"));
    test(S("abcde"), 1, 3, SV("1234567890"), 9, 2, S("a0e"));
    test(S("abcde"), 1, 3, SV("1234567890"), 10, 0, S("ae"));
    test(S("abcde"), 1, 3, SV("1234567890"), 10, 1, S("ae"));
    test(S("abcde"), 1, 3, SV("1234567890"), 11, 0, S("can't happen"));
    test(S("abcde"), 1, 3, SV("12345678901234567890"), 0, 0, S("ae"));
    test(S("abcde"), 1, 3, SV("12345678901234567890"), 0, 1, S("a1e"));
    test(S("abcde"), 1, 3, SV("12345678901234567890"), 0, 10, S("a1234567890e"));
    test(S("abcde"), 1, 3, SV("12345678901234567890"), 0, 19, S("a1234567890123456789e"));
    test(S("abcde"), 1, 3, SV("12345678901234567890"), 0, 20, S("a12345678901234567890e"));
    test(S("abcde"), 1, 3, SV("12345678901234567890"), 0, 21, S("a12345678901234567890e"));
    test(S("abcde"), 1, 3, SV("12345678901234567890"), 1, 0, S("ae"));
    test(S("abcde"), 1, 3, SV("12345678901234567890"), 1, 1, S("a2e"));
    test(S("abcde"), 1, 3, SV("12345678901234567890"), 1, 9, S("a234567890e"));
    test(S("abcde"), 1, 3, SV("12345678901234567890"), 1, 18, S("a234567890123456789e"));
    test(S("abcde"), 1, 3, SV("12345678901234567890"), 1, 19, S("a2345678901234567890e"));
    test(S("abcde"), 1, 3, SV("12345678901234567890"), 1, 20, S("a2345678901234567890e"));
    test(S("abcde"), 1, 3, SV("12345678901234567890"), 10, 0, S("ae"));
    test(S("abcde"), 1, 3, SV("12345678901234567890"), 10, 1, S("a1e"));
    test(S("abcde"), 1, 3, SV("12345678901234567890"), 10, 5, S("a12345e"));
    test(S("abcde"), 1, 3, SV("12345678901234567890"), 10, 9, S("a123456789e"));
    test(S("abcde"), 1, 3, SV("12345678901234567890"), 10, 10, S("a1234567890e"));
    test(S("abcde"), 1, 3, SV("12345678901234567890"), 10, 11, S("a1234567890e"));
    test(S("abcde"), 1, 3, SV("12345678901234567890"), 19, 0, S("ae"));
    test(S("abcde"), 1, 3, SV("12345678901234567890"), 19, 1, S("a0e"));
    test(S("abcde"), 1, 3, SV("12345678901234567890"), 19, 2, S("a0e"));
    test(S("abcde"), 1, 3, SV("12345678901234567890"), 20, 0, S("ae"));
    test(S("abcde"), 1, 3, SV("12345678901234567890"), 20, 1, S("ae"));
    test(S("abcde"), 1, 3, SV("12345678901234567890"), 21, 0, S("can't happen"));
    test(S("abcde"), 1, 4, SV(""), 0, 0, S("a"));
    test(S("abcde"), 1, 4, SV(""), 0, 1, S("a"));
    test(S("abcde"), 1, 4, SV(""), 1, 0, S("can't happen"));
    test(S("abcde"), 1, 4, SV("12345"), 0, 0, S("a"));
    test(S("abcde"), 1, 4, SV("12345"), 0, 1, S("a1"));
    test(S("abcde"), 1, 4, SV("12345"), 0, 2, S("a12"));
    test(S("abcde"), 1, 4, SV("12345"), 0, 4, S("a1234"));
    test(S("abcde"), 1, 4, SV("12345"), 0, 5, S("a12345"));
    test(S("abcde"), 1, 4, SV("12345"), 0, 6, S("a12345"));
    test(S("abcde"), 1, 4, SV("12345"), 1, 0, S("a"));
    test(S("abcde"), 1, 4, SV("12345"), 1, 1, S("a2"));
    test(S("abcde"), 1, 4, SV("12345"), 1, 2, S("a23"));
    test(S("abcde"), 1, 4, SV("12345"), 1, 3, S("a234"));
    test(S("abcde"), 1, 4, SV("12345"), 1, 4, S("a2345"));
    test(S("abcde"), 1, 4, SV("12345"), 1, 5, S("a2345"));
    test(S("abcde"), 1, 4, SV("12345"), 2, 0, S("a"));
    test(S("abcde"), 1, 4, SV("12345"), 2, 1, S("a3"));
    test(S("abcde"), 1, 4, SV("12345"), 2, 2, S("a34"));
    test(S("abcde"), 1, 4, SV("12345"), 2, 3, S("a345"));
    test(S("abcde"), 1, 4, SV("12345"), 2, 4, S("a345"));
    test(S("abcde"), 1, 4, SV("12345"), 4, 0, S("a"));
    test(S("abcde"), 1, 4, SV("12345"), 4, 1, S("a5"));
    test(S("abcde"), 1, 4, SV("12345"), 4, 2, S("a5"));
    test(S("abcde"), 1, 4, SV("12345"), 5, 0, S("a"));
    test(S("abcde"), 1, 4, SV("12345"), 5, 1, S("a"));
    test(S("abcde"), 1, 4, SV("12345"), 6, 0, S("can't happen"));
    test(S("abcde"), 1, 4, SV("1234567890"), 0, 0, S("a"));
    test(S("abcde"), 1, 4, SV("1234567890"), 0, 1, S("a1"));
    test(S("abcde"), 1, 4, SV("1234567890"), 0, 5, S("a12345"));
    test(S("abcde"), 1, 4, SV("1234567890"), 0, 9, S("a123456789"));
    test(S("abcde"), 1, 4, SV("1234567890"), 0, 10, S("a1234567890"));
    test(S("abcde"), 1, 4, SV("1234567890"), 0, 11, S("a1234567890"));
    test(S("abcde"), 1, 4, SV("1234567890"), 1, 0, S("a"));
    test(S("abcde"), 1, 4, SV("1234567890"), 1, 1, S("a2"));
    test(S("abcde"), 1, 4, SV("1234567890"), 1, 4, S("a2345"));
    test(S("abcde"), 1, 4, SV("1234567890"), 1, 8, S("a23456789"));
    test(S("abcde"), 1, 4, SV("1234567890"), 1, 9, S("a234567890"));
    test(S("abcde"), 1, 4, SV("1234567890"), 1, 10, S("a234567890"));

    return true;
}

template <class S, class SV>
TEST_CONSTEXPR_CXX20 bool test10()
{
    test(S("abcde"), 1, 4, SV("1234567890"), 5, 0, S("a"));
    test(S("abcde"), 1, 4, SV("1234567890"), 5, 1, S("a6"));
    test(S("abcde"), 1, 4, SV("1234567890"), 5, 2, S("a67"));
    test(S("abcde"), 1, 4, SV("1234567890"), 5, 4, S("a6789"));
    test(S("abcde"), 1, 4, SV("1234567890"), 5, 5, S("a67890"));
    test(S("abcde"), 1, 4, SV("1234567890"), 5, 6, S("a67890"));
    test(S("abcde"), 1, 4, SV("1234567890"), 9, 0, S("a"));
    test(S("abcde"), 1, 4, SV("1234567890"), 9, 1, S("a0"));
    test(S("abcde"), 1, 4, SV("1234567890"), 9, 2, S("a0"));
    test(S("abcde"), 1, 4, SV("1234567890"), 10, 0, S("a"));
    test(S("abcde"), 1, 4, SV("1234567890"), 10, 1, S("a"));
    test(S("abcde"), 1, 4, SV("1234567890"), 11, 0, S("can't happen"));
    test(S("abcde"), 1, 4, SV("12345678901234567890"), 0, 0, S("a"));
    test(S("abcde"), 1, 4, SV("12345678901234567890"), 0, 1, S("a1"));
    test(S("abcde"), 1, 4, SV("12345678901234567890"), 0, 10, S("a1234567890"));
    test(S("abcde"), 1, 4, SV("12345678901234567890"), 0, 19, S("a1234567890123456789"));
    test(S("abcde"), 1, 4, SV("12345678901234567890"), 0, 20, S("a12345678901234567890"));
    test(S("abcde"), 1, 4, SV("12345678901234567890"), 0, 21, S("a12345678901234567890"));
    test(S("abcde"), 1, 4, SV("12345678901234567890"), 1, 0, S("a"));
    test(S("abcde"), 1, 4, SV("12345678901234567890"), 1, 1, S("a2"));
    test(S("abcde"), 1, 4, SV("12345678901234567890"), 1, 9, S("a234567890"));
    test(S("abcde"), 1, 4, SV("12345678901234567890"), 1, 18, S("a234567890123456789"));
    test(S("abcde"), 1, 4, SV("12345678901234567890"), 1, 19, S("a2345678901234567890"));
    test(S("abcde"), 1, 4, SV("12345678901234567890"), 1, 20, S("a2345678901234567890"));
    test(S("abcde"), 1, 4, SV("12345678901234567890"), 10, 0, S("a"));
    test(S("abcde"), 1, 4, SV("12345678901234567890"), 10, 1, S("a1"));
    test(S("abcde"), 1, 4, SV("12345678901234567890"), 10, 5, S("a12345"));
    test(S("abcde"), 1, 4, SV("12345678901234567890"), 10, 9, S("a123456789"));
    test(S("abcde"), 1, 4, SV("12345678901234567890"), 10, 10, S("a1234567890"));
    test(S("abcde"), 1, 4, SV("12345678901234567890"), 10, 11, S("a1234567890"));
    test(S("abcde"), 1, 4, SV("12345678901234567890"), 19, 0, S("a"));
    test(S("abcde"), 1, 4, SV("12345678901234567890"), 19, 1, S("a0"));
    test(S("abcde"), 1, 4, SV("12345678901234567890"), 19, 2, S("a0"));
    test(S("abcde"), 1, 4, SV("12345678901234567890"), 20, 0, S("a"));
    test(S("abcde"), 1, 4, SV("12345678901234567890"), 20, 1, S("a"));
    test(S("abcde"), 1, 4, SV("12345678901234567890"), 21, 0, S("can't happen"));
    test(S("abcde"), 1, 5, SV(""), 0, 0, S("a"));
    test(S("abcde"), 1, 5, SV(""), 0, 1, S("a"));
    test(S("abcde"), 1, 5, SV(""), 1, 0, S("can't happen"));
    test(S("abcde"), 1, 5, SV("12345"), 0, 0, S("a"));
    test(S("abcde"), 1, 5, SV("12345"), 0, 1, S("a1"));
    test(S("abcde"), 1, 5, SV("12345"), 0, 2, S("a12"));
    test(S("abcde"), 1, 5, SV("12345"), 0, 4, S("a1234"));
    test(S("abcde"), 1, 5, SV("12345"), 0, 5, S("a12345"));
    test(S("abcde"), 1, 5, SV("12345"), 0, 6, S("a12345"));
    test(S("abcde"), 1, 5, SV("12345"), 1, 0, S("a"));
    test(S("abcde"), 1, 5, SV("12345"), 1, 1, S("a2"));
    test(S("abcde"), 1, 5, SV("12345"), 1, 2, S("a23"));
    test(S("abcde"), 1, 5, SV("12345"), 1, 3, S("a234"));
    test(S("abcde"), 1, 5, SV("12345"), 1, 4, S("a2345"));
    test(S("abcde"), 1, 5, SV("12345"), 1, 5, S("a2345"));
    test(S("abcde"), 1, 5, SV("12345"), 2, 0, S("a"));
    test(S("abcde"), 1, 5, SV("12345"), 2, 1, S("a3"));
    test(S("abcde"), 1, 5, SV("12345"), 2, 2, S("a34"));
    test(S("abcde"), 1, 5, SV("12345"), 2, 3, S("a345"));
    test(S("abcde"), 1, 5, SV("12345"), 2, 4, S("a345"));
    test(S("abcde"), 1, 5, SV("12345"), 4, 0, S("a"));
    test(S("abcde"), 1, 5, SV("12345"), 4, 1, S("a5"));
    test(S("abcde"), 1, 5, SV("12345"), 4, 2, S("a5"));
    test(S("abcde"), 1, 5, SV("12345"), 5, 0, S("a"));
    test(S("abcde"), 1, 5, SV("12345"), 5, 1, S("a"));
    test(S("abcde"), 1, 5, SV("12345"), 6, 0, S("can't happen"));
    test(S("abcde"), 1, 5, SV("1234567890"), 0, 0, S("a"));
    test(S("abcde"), 1, 5, SV("1234567890"), 0, 1, S("a1"));
    test(S("abcde"), 1, 5, SV("1234567890"), 0, 5, S("a12345"));
    test(S("abcde"), 1, 5, SV("1234567890"), 0, 9, S("a123456789"));
    test(S("abcde"), 1, 5, SV("1234567890"), 0, 10, S("a1234567890"));
    test(S("abcde"), 1, 5, SV("1234567890"), 0, 11, S("a1234567890"));
    test(S("abcde"), 1, 5, SV("1234567890"), 1, 0, S("a"));
    test(S("abcde"), 1, 5, SV("1234567890"), 1, 1, S("a2"));
    test(S("abcde"), 1, 5, SV("1234567890"), 1, 4, S("a2345"));
    test(S("abcde"), 1, 5, SV("1234567890"), 1, 8, S("a23456789"));
    test(S("abcde"), 1, 5, SV("1234567890"), 1, 9, S("a234567890"));
    test(S("abcde"), 1, 5, SV("1234567890"), 1, 10, S("a234567890"));
    test(S("abcde"), 1, 5, SV("1234567890"), 5, 0, S("a"));
    test(S("abcde"), 1, 5, SV("1234567890"), 5, 1, S("a6"));
    test(S("abcde"), 1, 5, SV("1234567890"), 5, 2, S("a67"));
    test(S("abcde"), 1, 5, SV("1234567890"), 5, 4, S("a6789"));
    test(S("abcde"), 1, 5, SV("1234567890"), 5, 5, S("a67890"));
    test(S("abcde"), 1, 5, SV("1234567890"), 5, 6, S("a67890"));
    test(S("abcde"), 1, 5, SV("1234567890"), 9, 0, S("a"));
    test(S("abcde"), 1, 5, SV("1234567890"), 9, 1, S("a0"));
    test(S("abcde"), 1, 5, SV("1234567890"), 9, 2, S("a0"));
    test(S("abcde"), 1, 5, SV("1234567890"), 10, 0, S("a"));
    test(S("abcde"), 1, 5, SV("1234567890"), 10, 1, S("a"));
    test(S("abcde"), 1, 5, SV("1234567890"), 11, 0, S("can't happen"));
    test(S("abcde"), 1, 5, SV("12345678901234567890"), 0, 0, S("a"));
    test(S("abcde"), 1, 5, SV("12345678901234567890"), 0, 1, S("a1"));
    test(S("abcde"), 1, 5, SV("12345678901234567890"), 0, 10, S("a1234567890"));
    test(S("abcde"), 1, 5, SV("12345678901234567890"), 0, 19, S("a1234567890123456789"));
    test(S("abcde"), 1, 5, SV("12345678901234567890"), 0, 20, S("a12345678901234567890"));
    test(S("abcde"), 1, 5, SV("12345678901234567890"), 0, 21, S("a12345678901234567890"));
    test(S("abcde"), 1, 5, SV("12345678901234567890"), 1, 0, S("a"));
    test(S("abcde"), 1, 5, SV("12345678901234567890"), 1, 1, S("a2"));
    test(S("abcde"), 1, 5, SV("12345678901234567890"), 1, 9, S("a234567890"));
    test(S("abcde"), 1, 5, SV("12345678901234567890"), 1, 18, S("a234567890123456789"));
    test(S("abcde"), 1, 5, SV("12345678901234567890"), 1, 19, S("a2345678901234567890"));
    test(S("abcde"), 1, 5, SV("12345678901234567890"), 1, 20, S("a2345678901234567890"));
    test(S("abcde"), 1, 5, SV("12345678901234567890"), 10, 0, S("a"));
    test(S("abcde"), 1, 5, SV("12345678901234567890"), 10, 1, S("a1"));

    return true;
}

template <class S, class SV>
TEST_CONSTEXPR_CXX20 bool test11()
{
    test(S("abcde"), 1, 5, SV("12345678901234567890"), 10, 5, S("a12345"));
    test(S("abcde"), 1, 5, SV("12345678901234567890"), 10, 9, S("a123456789"));
    test(S("abcde"), 1, 5, SV("12345678901234567890"), 10, 10, S("a1234567890"));
    test(S("abcde"), 1, 5, SV("12345678901234567890"), 10, 11, S("a1234567890"));
    test(S("abcde"), 1, 5, SV("12345678901234567890"), 19, 0, S("a"));
    test(S("abcde"), 1, 5, SV("12345678901234567890"), 19, 1, S("a0"));
    test(S("abcde"), 1, 5, SV("12345678901234567890"), 19, 2, S("a0"));
    test(S("abcde"), 1, 5, SV("12345678901234567890"), 20, 0, S("a"));
    test(S("abcde"), 1, 5, SV("12345678901234567890"), 20, 1, S("a"));
    test(S("abcde"), 1, 5, SV("12345678901234567890"), 21, 0, S("can't happen"));
    test(S("abcde"), 2, 0, SV(""), 0, 0, S("abcde"));
    test(S("abcde"), 2, 0, SV(""), 0, 1, S("abcde"));
    test(S("abcde"), 2, 0, SV(""), 1, 0, S("can't happen"));
    test(S("abcde"), 2, 0, SV("12345"), 0, 0, S("abcde"));
    test(S("abcde"), 2, 0, SV("12345"), 0, 1, S("ab1cde"));
    test(S("abcde"), 2, 0, SV("12345"), 0, 2, S("ab12cde"));
    test(S("abcde"), 2, 0, SV("12345"), 0, 4, S("ab1234cde"));
    test(S("abcde"), 2, 0, SV("12345"), 0, 5, S("ab12345cde"));
    test(S("abcde"), 2, 0, SV("12345"), 0, 6, S("ab12345cde"));
    test(S("abcde"), 2, 0, SV("12345"), 1, 0, S("abcde"));
    test(S("abcde"), 2, 0, SV("12345"), 1, 1, S("ab2cde"));
    test(S("abcde"), 2, 0, SV("12345"), 1, 2, S("ab23cde"));
    test(S("abcde"), 2, 0, SV("12345"), 1, 3, S("ab234cde"));
    test(S("abcde"), 2, 0, SV("12345"), 1, 4, S("ab2345cde"));
    test(S("abcde"), 2, 0, SV("12345"), 1, 5, S("ab2345cde"));
    test(S("abcde"), 2, 0, SV("12345"), 2, 0, S("abcde"));
    test(S("abcde"), 2, 0, SV("12345"), 2, 1, S("ab3cde"));
    test(S("abcde"), 2, 0, SV("12345"), 2, 2, S("ab34cde"));
    test(S("abcde"), 2, 0, SV("12345"), 2, 3, S("ab345cde"));
    test(S("abcde"), 2, 0, SV("12345"), 2, 4, S("ab345cde"));
    test(S("abcde"), 2, 0, SV("12345"), 4, 0, S("abcde"));
    test(S("abcde"), 2, 0, SV("12345"), 4, 1, S("ab5cde"));
    test(S("abcde"), 2, 0, SV("12345"), 4, 2, S("ab5cde"));
    test(S("abcde"), 2, 0, SV("12345"), 5, 0, S("abcde"));
    test(S("abcde"), 2, 0, SV("12345"), 5, 1, S("abcde"));
    test(S("abcde"), 2, 0, SV("12345"), 6, 0, S("can't happen"));
    test(S("abcde"), 2, 0, SV("1234567890"), 0, 0, S("abcde"));
    test(S("abcde"), 2, 0, SV("1234567890"), 0, 1, S("ab1cde"));
    test(S("abcde"), 2, 0, SV("1234567890"), 0, 5, S("ab12345cde"));
    test(S("abcde"), 2, 0, SV("1234567890"), 0, 9, S("ab123456789cde"));
    test(S("abcde"), 2, 0, SV("1234567890"), 0, 10, S("ab1234567890cde"));
    test(S("abcde"), 2, 0, SV("1234567890"), 0, 11, S("ab1234567890cde"));
    test(S("abcde"), 2, 0, SV("1234567890"), 1, 0, S("abcde"));
    test(S("abcde"), 2, 0, SV("1234567890"), 1, 1, S("ab2cde"));
    test(S("abcde"), 2, 0, SV("1234567890"), 1, 4, S("ab2345cde"));
    test(S("abcde"), 2, 0, SV("1234567890"), 1, 8, S("ab23456789cde"));
    test(S("abcde"), 2, 0, SV("1234567890"), 1, 9, S("ab234567890cde"));
    test(S("abcde"), 2, 0, SV("1234567890"), 1, 10, S("ab234567890cde"));
    test(S("abcde"), 2, 0, SV("1234567890"), 5, 0, S("abcde"));
    test(S("abcde"), 2, 0, SV("1234567890"), 5, 1, S("ab6cde"));
    test(S("abcde"), 2, 0, SV("1234567890"), 5, 2, S("ab67cde"));
    test(S("abcde"), 2, 0, SV("1234567890"), 5, 4, S("ab6789cde"));
    test(S("abcde"), 2, 0, SV("1234567890"), 5, 5, S("ab67890cde"));
    test(S("abcde"), 2, 0, SV("1234567890"), 5, 6, S("ab67890cde"));
    test(S("abcde"), 2, 0, SV("1234567890"), 9, 0, S("abcde"));
    test(S("abcde"), 2, 0, SV("1234567890"), 9, 1, S("ab0cde"));
    test(S("abcde"), 2, 0, SV("1234567890"), 9, 2, S("ab0cde"));
    test(S("abcde"), 2, 0, SV("1234567890"), 10, 0, S("abcde"));
    test(S("abcde"), 2, 0, SV("1234567890"), 10, 1, S("abcde"));
    test(S("abcde"), 2, 0, SV("1234567890"), 11, 0, S("can't happen"));
    test(S("abcde"), 2, 0, SV("12345678901234567890"), 0, 0, S("abcde"));
    test(S("abcde"), 2, 0, SV("12345678901234567890"), 0, 1, S("ab1cde"));
    test(S("abcde"), 2, 0, SV("12345678901234567890"), 0, 10, S("ab1234567890cde"));
    test(S("abcde"), 2, 0, SV("12345678901234567890"), 0, 19, S("ab1234567890123456789cde"));
    test(S("abcde"), 2, 0, SV("12345678901234567890"), 0, 20, S("ab12345678901234567890cde"));
    test(S("abcde"), 2, 0, SV("12345678901234567890"), 0, 21, S("ab12345678901234567890cde"));
    test(S("abcde"), 2, 0, SV("12345678901234567890"), 1, 0, S("abcde"));
    test(S("abcde"), 2, 0, SV("12345678901234567890"), 1, 1, S("ab2cde"));
    test(S("abcde"), 2, 0, SV("12345678901234567890"), 1, 9, S("ab234567890cde"));
    test(S("abcde"), 2, 0, SV("12345678901234567890"), 1, 18, S("ab234567890123456789cde"));
    test(S("abcde"), 2, 0, SV("12345678901234567890"), 1, 19, S("ab2345678901234567890cde"));
    test(S("abcde"), 2, 0, SV("12345678901234567890"), 1, 20, S("ab2345678901234567890cde"));
    test(S("abcde"), 2, 0, SV("12345678901234567890"), 10, 0, S("abcde"));
    test(S("abcde"), 2, 0, SV("12345678901234567890"), 10, 1, S("ab1cde"));
    test(S("abcde"), 2, 0, SV("12345678901234567890"), 10, 5, S("ab12345cde"));
    test(S("abcde"), 2, 0, SV("12345678901234567890"), 10, 9, S("ab123456789cde"));
    test(S("abcde"), 2, 0, SV("12345678901234567890"), 10, 10, S("ab1234567890cde"));
    test(S("abcde"), 2, 0, SV("12345678901234567890"), 10, 11, S("ab1234567890cde"));
    test(S("abcde"), 2, 0, SV("12345678901234567890"), 19, 0, S("abcde"));
    test(S("abcde"), 2, 0, SV("12345678901234567890"), 19, 1, S("ab0cde"));
    test(S("abcde"), 2, 0, SV("12345678901234567890"), 19, 2, S("ab0cde"));
    test(S("abcde"), 2, 0, SV("12345678901234567890"), 20, 0, S("abcde"));
    test(S("abcde"), 2, 0, SV("12345678901234567890"), 20, 1, S("abcde"));
    test(S("abcde"), 2, 0, SV("12345678901234567890"), 21, 0, S("can't happen"));
    test(S("abcde"), 2, 1, SV(""), 0, 0, S("abde"));
    test(S("abcde"), 2, 1, SV(""), 0, 1, S("abde"));
    test(S("abcde"), 2, 1, SV(""), 1, 0, S("can't happen"));
    test(S("abcde"), 2, 1, SV("12345"), 0, 0, S("abde"));
    test(S("abcde"), 2, 1, SV("12345"), 0, 1, S("ab1de"));
    test(S("abcde"), 2, 1, SV("12345"), 0, 2, S("ab12de"));
    test(S("abcde"), 2, 1, SV("12345"), 0, 4, S("ab1234de"));
    test(S("abcde"), 2, 1, SV("12345"), 0, 5, S("ab12345de"));
    test(S("abcde"), 2, 1, SV("12345"), 0, 6, S("ab12345de"));
    test(S("abcde"), 2, 1, SV("12345"), 1, 0, S("abde"));
    test(S("abcde"), 2, 1, SV("12345"), 1, 1, S("ab2de"));
    test(S("abcde"), 2, 1, SV("12345"), 1, 2, S("ab23de"));
    test(S("abcde"), 2, 1, SV("12345"), 1, 3, S("ab234de"));
    test(S("abcde"), 2, 1, SV("12345"), 1, 4, S("ab2345de"));
    test(S("abcde"), 2, 1, SV("12345"), 1, 5, S("ab2345de"));
    test(S("abcde"), 2, 1, SV("12345"), 2, 0, S("abde"));

    return true;
}

template <class S, class SV>
TEST_CONSTEXPR_CXX20 bool test12()
{
    test(S("abcde"), 2, 1, SV("12345"), 2, 1, S("ab3de"));
    test(S("abcde"), 2, 1, SV("12345"), 2, 2, S("ab34de"));
    test(S("abcde"), 2, 1, SV("12345"), 2, 3, S("ab345de"));
    test(S("abcde"), 2, 1, SV("12345"), 2, 4, S("ab345de"));
    test(S("abcde"), 2, 1, SV("12345"), 4, 0, S("abde"));
    test(S("abcde"), 2, 1, SV("12345"), 4, 1, S("ab5de"));
    test(S("abcde"), 2, 1, SV("12345"), 4, 2, S("ab5de"));
    test(S("abcde"), 2, 1, SV("12345"), 5, 0, S("abde"));
    test(S("abcde"), 2, 1, SV("12345"), 5, 1, S("abde"));
    test(S("abcde"), 2, 1, SV("12345"), 6, 0, S("can't happen"));
    test(S("abcde"), 2, 1, SV("1234567890"), 0, 0, S("abde"));
    test(S("abcde"), 2, 1, SV("1234567890"), 0, 1, S("ab1de"));
    test(S("abcde"), 2, 1, SV("1234567890"), 0, 5, S("ab12345de"));
    test(S("abcde"), 2, 1, SV("1234567890"), 0, 9, S("ab123456789de"));
    test(S("abcde"), 2, 1, SV("1234567890"), 0, 10, S("ab1234567890de"));
    test(S("abcde"), 2, 1, SV("1234567890"), 0, 11, S("ab1234567890de"));
    test(S("abcde"), 2, 1, SV("1234567890"), 1, 0, S("abde"));
    test(S("abcde"), 2, 1, SV("1234567890"), 1, 1, S("ab2de"));
    test(S("abcde"), 2, 1, SV("1234567890"), 1, 4, S("ab2345de"));
    test(S("abcde"), 2, 1, SV("1234567890"), 1, 8, S("ab23456789de"));
    test(S("abcde"), 2, 1, SV("1234567890"), 1, 9, S("ab234567890de"));
    test(S("abcde"), 2, 1, SV("1234567890"), 1, 10, S("ab234567890de"));
    test(S("abcde"), 2, 1, SV("1234567890"), 5, 0, S("abde"));
    test(S("abcde"), 2, 1, SV("1234567890"), 5, 1, S("ab6de"));
    test(S("abcde"), 2, 1, SV("1234567890"), 5, 2, S("ab67de"));
    test(S("abcde"), 2, 1, SV("1234567890"), 5, 4, S("ab6789de"));
    test(S("abcde"), 2, 1, SV("1234567890"), 5, 5, S("ab67890de"));
    test(S("abcde"), 2, 1, SV("1234567890"), 5, 6, S("ab67890de"));
    test(S("abcde"), 2, 1, SV("1234567890"), 9, 0, S("abde"));
    test(S("abcde"), 2, 1, SV("1234567890"), 9, 1, S("ab0de"));
    test(S("abcde"), 2, 1, SV("1234567890"), 9, 2, S("ab0de"));
    test(S("abcde"), 2, 1, SV("1234567890"), 10, 0, S("abde"));
    test(S("abcde"), 2, 1, SV("1234567890"), 10, 1, S("abde"));
    test(S("abcde"), 2, 1, SV("1234567890"), 11, 0, S("can't happen"));
    test(S("abcde"), 2, 1, SV("12345678901234567890"), 0, 0, S("abde"));
    test(S("abcde"), 2, 1, SV("12345678901234567890"), 0, 1, S("ab1de"));
    test(S("abcde"), 2, 1, SV("12345678901234567890"), 0, 10, S("ab1234567890de"));
    test(S("abcde"), 2, 1, SV("12345678901234567890"), 0, 19, S("ab1234567890123456789de"));
    test(S("abcde"), 2, 1, SV("12345678901234567890"), 0, 20, S("ab12345678901234567890de"));
    test(S("abcde"), 2, 1, SV("12345678901234567890"), 0, 21, S("ab12345678901234567890de"));
    test(S("abcde"), 2, 1, SV("12345678901234567890"), 1, 0, S("abde"));
    test(S("abcde"), 2, 1, SV("12345678901234567890"), 1, 1, S("ab2de"));
    test(S("abcde"), 2, 1, SV("12345678901234567890"), 1, 9, S("ab234567890de"));
    test(S("abcde"), 2, 1, SV("12345678901234567890"), 1, 18, S("ab234567890123456789de"));
    test(S("abcde"), 2, 1, SV("12345678901234567890"), 1, 19, S("ab2345678901234567890de"));
    test(S("abcde"), 2, 1, SV("12345678901234567890"), 1, 20, S("ab2345678901234567890de"));
    test(S("abcde"), 2, 1, SV("12345678901234567890"), 10, 0, S("abde"));
    test(S("abcde"), 2, 1, SV("12345678901234567890"), 10, 1, S("ab1de"));
    test(S("abcde"), 2, 1, SV("12345678901234567890"), 10, 5, S("ab12345de"));
    test(S("abcde"), 2, 1, SV("12345678901234567890"), 10, 9, S("ab123456789de"));
    test(S("abcde"), 2, 1, SV("12345678901234567890"), 10, 10, S("ab1234567890de"));
    test(S("abcde"), 2, 1, SV("12345678901234567890"), 10, 11, S("ab1234567890de"));
    test(S("abcde"), 2, 1, SV("12345678901234567890"), 19, 0, S("abde"));
    test(S("abcde"), 2, 1, SV("12345678901234567890"), 19, 1, S("ab0de"));
    test(S("abcde"), 2, 1, SV("12345678901234567890"), 19, 2, S("ab0de"));
    test(S("abcde"), 2, 1, SV("12345678901234567890"), 20, 0, S("abde"));
    test(S("abcde"), 2, 1, SV("12345678901234567890"), 20, 1, S("abde"));
    test(S("abcde"), 2, 1, SV("12345678901234567890"), 21, 0, S("can't happen"));
    test(S("abcde"), 2, 2, SV(""), 0, 0, S("abe"));
    test(S("abcde"), 2, 2, SV(""), 0, 1, S("abe"));
    test(S("abcde"), 2, 2, SV(""), 1, 0, S("can't happen"));
    test(S("abcde"), 2, 2, SV("12345"), 0, 0, S("abe"));
    test(S("abcde"), 2, 2, SV("12345"), 0, 1, S("ab1e"));
    test(S("abcde"), 2, 2, SV("12345"), 0, 2, S("ab12e"));
    test(S("abcde"), 2, 2, SV("12345"), 0, 4, S("ab1234e"));
    test(S("abcde"), 2, 2, SV("12345"), 0, 5, S("ab12345e"));
    test(S("abcde"), 2, 2, SV("12345"), 0, 6, S("ab12345e"));
    test(S("abcde"), 2, 2, SV("12345"), 1, 0, S("abe"));
    test(S("abcde"), 2, 2, SV("12345"), 1, 1, S("ab2e"));
    test(S("abcde"), 2, 2, SV("12345"), 1, 2, S("ab23e"));
    test(S("abcde"), 2, 2, SV("12345"), 1, 3, S("ab234e"));
    test(S("abcde"), 2, 2, SV("12345"), 1, 4, S("ab2345e"));
    test(S("abcde"), 2, 2, SV("12345"), 1, 5, S("ab2345e"));
    test(S("abcde"), 2, 2, SV("12345"), 2, 0, S("abe"));
    test(S("abcde"), 2, 2, SV("12345"), 2, 1, S("ab3e"));
    test(S("abcde"), 2, 2, SV("12345"), 2, 2, S("ab34e"));
    test(S("abcde"), 2, 2, SV("12345"), 2, 3, S("ab345e"));
    test(S("abcde"), 2, 2, SV("12345"), 2, 4, S("ab345e"));
    test(S("abcde"), 2, 2, SV("12345"), 4, 0, S("abe"));
    test(S("abcde"), 2, 2, SV("12345"), 4, 1, S("ab5e"));
    test(S("abcde"), 2, 2, SV("12345"), 4, 2, S("ab5e"));
    test(S("abcde"), 2, 2, SV("12345"), 5, 0, S("abe"));
    test(S("abcde"), 2, 2, SV("12345"), 5, 1, S("abe"));
    test(S("abcde"), 2, 2, SV("12345"), 6, 0, S("can't happen"));
    test(S("abcde"), 2, 2, SV("1234567890"), 0, 0, S("abe"));
    test(S("abcde"), 2, 2, SV("1234567890"), 0, 1, S("ab1e"));
    test(S("abcde"), 2, 2, SV("1234567890"), 0, 5, S("ab12345e"));
    test(S("abcde"), 2, 2, SV("1234567890"), 0, 9, S("ab123456789e"));
    test(S("abcde"), 2, 2, SV("1234567890"), 0, 10, S("ab1234567890e"));
    test(S("abcde"), 2, 2, SV("1234567890"), 0, 11, S("ab1234567890e"));
    test(S("abcde"), 2, 2, SV("1234567890"), 1, 0, S("abe"));
    test(S("abcde"), 2, 2, SV("1234567890"), 1, 1, S("ab2e"));
    test(S("abcde"), 2, 2, SV("1234567890"), 1, 4, S("ab2345e"));
    test(S("abcde"), 2, 2, SV("1234567890"), 1, 8, S("ab23456789e"));
    test(S("abcde"), 2, 2, SV("1234567890"), 1, 9, S("ab234567890e"));
    test(S("abcde"), 2, 2, SV("1234567890"), 1, 10, S("ab234567890e"));
    test(S("abcde"), 2, 2, SV("1234567890"), 5, 0, S("abe"));
    test(S("abcde"), 2, 2, SV("1234567890"), 5, 1, S("ab6e"));
    test(S("abcde"), 2, 2, SV("1234567890"), 5, 2, S("ab67e"));
    test(S("abcde"), 2, 2, SV("1234567890"), 5, 4, S("ab6789e"));

    return true;
}

template <class S, class SV>
TEST_CONSTEXPR_CXX20 bool test13()
{
    test(S("abcde"), 2, 2, SV("1234567890"), 5, 5, S("ab67890e"));
    test(S("abcde"), 2, 2, SV("1234567890"), 5, 6, S("ab67890e"));
    test(S("abcde"), 2, 2, SV("1234567890"), 9, 0, S("abe"));
    test(S("abcde"), 2, 2, SV("1234567890"), 9, 1, S("ab0e"));
    test(S("abcde"), 2, 2, SV("1234567890"), 9, 2, S("ab0e"));
    test(S("abcde"), 2, 2, SV("1234567890"), 10, 0, S("abe"));
    test(S("abcde"), 2, 2, SV("1234567890"), 10, 1, S("abe"));
    test(S("abcde"), 2, 2, SV("1234567890"), 11, 0, S("can't happen"));
    test(S("abcde"), 2, 2, SV("12345678901234567890"), 0, 0, S("abe"));
    test(S("abcde"), 2, 2, SV("12345678901234567890"), 0, 1, S("ab1e"));
    test(S("abcde"), 2, 2, SV("12345678901234567890"), 0, 10, S("ab1234567890e"));
    test(S("abcde"), 2, 2, SV("12345678901234567890"), 0, 19, S("ab1234567890123456789e"));
    test(S("abcde"), 2, 2, SV("12345678901234567890"), 0, 20, S("ab12345678901234567890e"));
    test(S("abcde"), 2, 2, SV("12345678901234567890"), 0, 21, S("ab12345678901234567890e"));
    test(S("abcde"), 2, 2, SV("12345678901234567890"), 1, 0, S("abe"));
    test(S("abcde"), 2, 2, SV("12345678901234567890"), 1, 1, S("ab2e"));
    test(S("abcde"), 2, 2, SV("12345678901234567890"), 1, 9, S("ab234567890e"));
    test(S("abcde"), 2, 2, SV("12345678901234567890"), 1, 18, S("ab234567890123456789e"));
    test(S("abcde"), 2, 2, SV("12345678901234567890"), 1, 19, S("ab2345678901234567890e"));
    test(S("abcde"), 2, 2, SV("12345678901234567890"), 1, 20, S("ab2345678901234567890e"));
    test(S("abcde"), 2, 2, SV("12345678901234567890"), 10, 0, S("abe"));
    test(S("abcde"), 2, 2, SV("12345678901234567890"), 10, 1, S("ab1e"));
    test(S("abcde"), 2, 2, SV("12345678901234567890"), 10, 5, S("ab12345e"));
    test(S("abcde"), 2, 2, SV("12345678901234567890"), 10, 9, S("ab123456789e"));
    test(S("abcde"), 2, 2, SV("12345678901234567890"), 10, 10, S("ab1234567890e"));
    test(S("abcde"), 2, 2, SV("12345678901234567890"), 10, 11, S("ab1234567890e"));
    test(S("abcde"), 2, 2, SV("12345678901234567890"), 19, 0, S("abe"));
    test(S("abcde"), 2, 2, SV("12345678901234567890"), 19, 1, S("ab0e"));
    test(S("abcde"), 2, 2, SV("12345678901234567890"), 19, 2, S("ab0e"));
    test(S("abcde"), 2, 2, SV("12345678901234567890"), 20, 0, S("abe"));
    test(S("abcde"), 2, 2, SV("12345678901234567890"), 20, 1, S("abe"));
    test(S("abcde"), 2, 2, SV("12345678901234567890"), 21, 0, S("can't happen"));
    test(S("abcde"), 2, 3, SV(""), 0, 0, S("ab"));
    test(S("abcde"), 2, 3, SV(""), 0, 1, S("ab"));
    test(S("abcde"), 2, 3, SV(""), 1, 0, S("can't happen"));
    test(S("abcde"), 2, 3, SV("12345"), 0, 0, S("ab"));
    test(S("abcde"), 2, 3, SV("12345"), 0, 1, S("ab1"));
    test(S("abcde"), 2, 3, SV("12345"), 0, 2, S("ab12"));
    test(S("abcde"), 2, 3, SV("12345"), 0, 4, S("ab1234"));
    test(S("abcde"), 2, 3, SV("12345"), 0, 5, S("ab12345"));
    test(S("abcde"), 2, 3, SV("12345"), 0, 6, S("ab12345"));
    test(S("abcde"), 2, 3, SV("12345"), 1, 0, S("ab"));
    test(S("abcde"), 2, 3, SV("12345"), 1, 1, S("ab2"));
    test(S("abcde"), 2, 3, SV("12345"), 1, 2, S("ab23"));
    test(S("abcde"), 2, 3, SV("12345"), 1, 3, S("ab234"));
    test(S("abcde"), 2, 3, SV("12345"), 1, 4, S("ab2345"));
    test(S("abcde"), 2, 3, SV("12345"), 1, 5, S("ab2345"));
    test(S("abcde"), 2, 3, SV("12345"), 2, 0, S("ab"));
    test(S("abcde"), 2, 3, SV("12345"), 2, 1, S("ab3"));
    test(S("abcde"), 2, 3, SV("12345"), 2, 2, S("ab34"));
    test(S("abcde"), 2, 3, SV("12345"), 2, 3, S("ab345"));
    test(S("abcde"), 2, 3, SV("12345"), 2, 4, S("ab345"));
    test(S("abcde"), 2, 3, SV("12345"), 4, 0, S("ab"));
    test(S("abcde"), 2, 3, SV("12345"), 4, 1, S("ab5"));
    test(S("abcde"), 2, 3, SV("12345"), 4, 2, S("ab5"));
    test(S("abcde"), 2, 3, SV("12345"), 5, 0, S("ab"));
    test(S("abcde"), 2, 3, SV("12345"), 5, 1, S("ab"));
    test(S("abcde"), 2, 3, SV("12345"), 6, 0, S("can't happen"));
    test(S("abcde"), 2, 3, SV("1234567890"), 0, 0, S("ab"));
    test(S("abcde"), 2, 3, SV("1234567890"), 0, 1, S("ab1"));
    test(S("abcde"), 2, 3, SV("1234567890"), 0, 5, S("ab12345"));
    test(S("abcde"), 2, 3, SV("1234567890"), 0, 9, S("ab123456789"));
    test(S("abcde"), 2, 3, SV("1234567890"), 0, 10, S("ab1234567890"));
    test(S("abcde"), 2, 3, SV("1234567890"), 0, 11, S("ab1234567890"));
    test(S("abcde"), 2, 3, SV("1234567890"), 1, 0, S("ab"));
    test(S("abcde"), 2, 3, SV("1234567890"), 1, 1, S("ab2"));
    test(S("abcde"), 2, 3, SV("1234567890"), 1, 4, S("ab2345"));
    test(S("abcde"), 2, 3, SV("1234567890"), 1, 8, S("ab23456789"));
    test(S("abcde"), 2, 3, SV("1234567890"), 1, 9, S("ab234567890"));
    test(S("abcde"), 2, 3, SV("1234567890"), 1, 10, S("ab234567890"));
    test(S("abcde"), 2, 3, SV("1234567890"), 5, 0, S("ab"));
    test(S("abcde"), 2, 3, SV("1234567890"), 5, 1, S("ab6"));
    test(S("abcde"), 2, 3, SV("1234567890"), 5, 2, S("ab67"));
    test(S("abcde"), 2, 3, SV("1234567890"), 5, 4, S("ab6789"));
    test(S("abcde"), 2, 3, SV("1234567890"), 5, 5, S("ab67890"));
    test(S("abcde"), 2, 3, SV("1234567890"), 5, 6, S("ab67890"));
    test(S("abcde"), 2, 3, SV("1234567890"), 9, 0, S("ab"));
    test(S("abcde"), 2, 3, SV("1234567890"), 9, 1, S("ab0"));
    test(S("abcde"), 2, 3, SV("1234567890"), 9, 2, S("ab0"));
    test(S("abcde"), 2, 3, SV("1234567890"), 10, 0, S("ab"));
    test(S("abcde"), 2, 3, SV("1234567890"), 10, 1, S("ab"));
    test(S("abcde"), 2, 3, SV("1234567890"), 11, 0, S("can't happen"));
    test(S("abcde"), 2, 3, SV("12345678901234567890"), 0, 0, S("ab"));
    test(S("abcde"), 2, 3, SV("12345678901234567890"), 0, 1, S("ab1"));
    test(S("abcde"), 2, 3, SV("12345678901234567890"), 0, 10, S("ab1234567890"));
    test(S("abcde"), 2, 3, SV("12345678901234567890"), 0, 19, S("ab1234567890123456789"));
    test(S("abcde"), 2, 3, SV("12345678901234567890"), 0, 20, S("ab12345678901234567890"));
    test(S("abcde"), 2, 3, SV("12345678901234567890"), 0, 21, S("ab12345678901234567890"));
    test(S("abcde"), 2, 3, SV("12345678901234567890"), 1, 0, S("ab"));
    test(S("abcde"), 2, 3, SV("12345678901234567890"), 1, 1, S("ab2"));
    test(S("abcde"), 2, 3, SV("12345678901234567890"), 1, 9, S("ab234567890"));
    test(S("abcde"), 2, 3, SV("12345678901234567890"), 1, 18, S("ab234567890123456789"));
    test(S("abcde"), 2, 3, SV("12345678901234567890"), 1, 19, S("ab2345678901234567890"));
    test(S("abcde"), 2, 3, SV("12345678901234567890"), 1, 20, S("ab2345678901234567890"));
    test(S("abcde"), 2, 3, SV("12345678901234567890"), 10, 0, S("ab"));
    test(S("abcde"), 2, 3, SV("12345678901234567890"), 10, 1, S("ab1"));
    test(S("abcde"), 2, 3, SV("12345678901234567890"), 10, 5, S("ab12345"));
    test(S("abcde"), 2, 3, SV("12345678901234567890"), 10, 9, S("ab123456789"));
    test(S("abcde"), 2, 3, SV("12345678901234567890"), 10, 10, S("ab1234567890"));
    test(S("abcde"), 2, 3, SV("12345678901234567890"), 10, 11, S("ab1234567890"));

    return true;
}

template <class S, class SV>
TEST_CONSTEXPR_CXX20 bool test14()
{
    test(S("abcde"), 2, 3, SV("12345678901234567890"), 19, 0, S("ab"));
    test(S("abcde"), 2, 3, SV("12345678901234567890"), 19, 1, S("ab0"));
    test(S("abcde"), 2, 3, SV("12345678901234567890"), 19, 2, S("ab0"));
    test(S("abcde"), 2, 3, SV("12345678901234567890"), 20, 0, S("ab"));
    test(S("abcde"), 2, 3, SV("12345678901234567890"), 20, 1, S("ab"));
    test(S("abcde"), 2, 3, SV("12345678901234567890"), 21, 0, S("can't happen"));
    test(S("abcde"), 2, 4, SV(""), 0, 0, S("ab"));
    test(S("abcde"), 2, 4, SV(""), 0, 1, S("ab"));
    test(S("abcde"), 2, 4, SV(""), 1, 0, S("can't happen"));
    test(S("abcde"), 2, 4, SV("12345"), 0, 0, S("ab"));
    test(S("abcde"), 2, 4, SV("12345"), 0, 1, S("ab1"));
    test(S("abcde"), 2, 4, SV("12345"), 0, 2, S("ab12"));
    test(S("abcde"), 2, 4, SV("12345"), 0, 4, S("ab1234"));
    test(S("abcde"), 2, 4, SV("12345"), 0, 5, S("ab12345"));
    test(S("abcde"), 2, 4, SV("12345"), 0, 6, S("ab12345"));
    test(S("abcde"), 2, 4, SV("12345"), 1, 0, S("ab"));
    test(S("abcde"), 2, 4, SV("12345"), 1, 1, S("ab2"));
    test(S("abcde"), 2, 4, SV("12345"), 1, 2, S("ab23"));
    test(S("abcde"), 2, 4, SV("12345"), 1, 3, S("ab234"));
    test(S("abcde"), 2, 4, SV("12345"), 1, 4, S("ab2345"));
    test(S("abcde"), 2, 4, SV("12345"), 1, 5, S("ab2345"));
    test(S("abcde"), 2, 4, SV("12345"), 2, 0, S("ab"));
    test(S("abcde"), 2, 4, SV("12345"), 2, 1, S("ab3"));
    test(S("abcde"), 2, 4, SV("12345"), 2, 2, S("ab34"));
    test(S("abcde"), 2, 4, SV("12345"), 2, 3, S("ab345"));
    test(S("abcde"), 2, 4, SV("12345"), 2, 4, S("ab345"));
    test(S("abcde"), 2, 4, SV("12345"), 4, 0, S("ab"));
    test(S("abcde"), 2, 4, SV("12345"), 4, 1, S("ab5"));
    test(S("abcde"), 2, 4, SV("12345"), 4, 2, S("ab5"));
    test(S("abcde"), 2, 4, SV("12345"), 5, 0, S("ab"));
    test(S("abcde"), 2, 4, SV("12345"), 5, 1, S("ab"));
    test(S("abcde"), 2, 4, SV("12345"), 6, 0, S("can't happen"));
    test(S("abcde"), 2, 4, SV("1234567890"), 0, 0, S("ab"));
    test(S("abcde"), 2, 4, SV("1234567890"), 0, 1, S("ab1"));
    test(S("abcde"), 2, 4, SV("1234567890"), 0, 5, S("ab12345"));
    test(S("abcde"), 2, 4, SV("1234567890"), 0, 9, S("ab123456789"));
    test(S("abcde"), 2, 4, SV("1234567890"), 0, 10, S("ab1234567890"));
    test(S("abcde"), 2, 4, SV("1234567890"), 0, 11, S("ab1234567890"));
    test(S("abcde"), 2, 4, SV("1234567890"), 1, 0, S("ab"));
    test(S("abcde"), 2, 4, SV("1234567890"), 1, 1, S("ab2"));
    test(S("abcde"), 2, 4, SV("1234567890"), 1, 4, S("ab2345"));
    test(S("abcde"), 2, 4, SV("1234567890"), 1, 8, S("ab23456789"));
    test(S("abcde"), 2, 4, SV("1234567890"), 1, 9, S("ab234567890"));
    test(S("abcde"), 2, 4, SV("1234567890"), 1, 10, S("ab234567890"));
    test(S("abcde"), 2, 4, SV("1234567890"), 5, 0, S("ab"));
    test(S("abcde"), 2, 4, SV("1234567890"), 5, 1, S("ab6"));
    test(S("abcde"), 2, 4, SV("1234567890"), 5, 2, S("ab67"));
    test(S("abcde"), 2, 4, SV("1234567890"), 5, 4, S("ab6789"));
    test(S("abcde"), 2, 4, SV("1234567890"), 5, 5, S("ab67890"));
    test(S("abcde"), 2, 4, SV("1234567890"), 5, 6, S("ab67890"));
    test(S("abcde"), 2, 4, SV("1234567890"), 9, 0, S("ab"));
    test(S("abcde"), 2, 4, SV("1234567890"), 9, 1, S("ab0"));
    test(S("abcde"), 2, 4, SV("1234567890"), 9, 2, S("ab0"));
    test(S("abcde"), 2, 4, SV("1234567890"), 10, 0, S("ab"));
    test(S("abcde"), 2, 4, SV("1234567890"), 10, 1, S("ab"));
    test(S("abcde"), 2, 4, SV("1234567890"), 11, 0, S("can't happen"));
    test(S("abcde"), 2, 4, SV("12345678901234567890"), 0, 0, S("ab"));
    test(S("abcde"), 2, 4, SV("12345678901234567890"), 0, 1, S("ab1"));
    test(S("abcde"), 2, 4, SV("12345678901234567890"), 0, 10, S("ab1234567890"));
    test(S("abcde"), 2, 4, SV("12345678901234567890"), 0, 19, S("ab1234567890123456789"));
    test(S("abcde"), 2, 4, SV("12345678901234567890"), 0, 20, S("ab12345678901234567890"));
    test(S("abcde"), 2, 4, SV("12345678901234567890"), 0, 21, S("ab12345678901234567890"));
    test(S("abcde"), 2, 4, SV("12345678901234567890"), 1, 0, S("ab"));
    test(S("abcde"), 2, 4, SV("12345678901234567890"), 1, 1, S("ab2"));
    test(S("abcde"), 2, 4, SV("12345678901234567890"), 1, 9, S("ab234567890"));
    test(S("abcde"), 2, 4, SV("12345678901234567890"), 1, 18, S("ab234567890123456789"));
    test(S("abcde"), 2, 4, SV("12345678901234567890"), 1, 19, S("ab2345678901234567890"));
    test(S("abcde"), 2, 4, SV("12345678901234567890"), 1, 20, S("ab2345678901234567890"));
    test(S("abcde"), 2, 4, SV("12345678901234567890"), 10, 0, S("ab"));
    test(S("abcde"), 2, 4, SV("12345678901234567890"), 10, 1, S("ab1"));
    test(S("abcde"), 2, 4, SV("12345678901234567890"), 10, 5, S("ab12345"));
    test(S("abcde"), 2, 4, SV("12345678901234567890"), 10, 9, S("ab123456789"));
    test(S("abcde"), 2, 4, SV("12345678901234567890"), 10, 10, S("ab1234567890"));
    test(S("abcde"), 2, 4, SV("12345678901234567890"), 10, 11, S("ab1234567890"));
    test(S("abcde"), 2, 4, SV("12345678901234567890"), 19, 0, S("ab"));
    test(S("abcde"), 2, 4, SV("12345678901234567890"), 19, 1, S("ab0"));
    test(S("abcde"), 2, 4, SV("12345678901234567890"), 19, 2, S("ab0"));
    test(S("abcde"), 2, 4, SV("12345678901234567890"), 20, 0, S("ab"));
    test(S("abcde"), 2, 4, SV("12345678901234567890"), 20, 1, S("ab"));
    test(S("abcde"), 2, 4, SV("12345678901234567890"), 21, 0, S("can't happen"));
    test(S("abcde"), 4, 0, SV(""), 0, 0, S("abcde"));
    test(S("abcde"), 4, 0, SV(""), 0, 1, S("abcde"));
    test(S("abcde"), 4, 0, SV(""), 1, 0, S("can't happen"));
    test(S("abcde"), 4, 0, SV("12345"), 0, 0, S("abcde"));
    test(S("abcde"), 4, 0, SV("12345"), 0, 1, S("abcd1e"));
    test(S("abcde"), 4, 0, SV("12345"), 0, 2, S("abcd12e"));
    test(S("abcde"), 4, 0, SV("12345"), 0, 4, S("abcd1234e"));
    test(S("abcde"), 4, 0, SV("12345"), 0, 5, S("abcd12345e"));
    test(S("abcde"), 4, 0, SV("12345"), 0, 6, S("abcd12345e"));
    test(S("abcde"), 4, 0, SV("12345"), 1, 0, S("abcde"));
    test(S("abcde"), 4, 0, SV("12345"), 1, 1, S("abcd2e"));
    test(S("abcde"), 4, 0, SV("12345"), 1, 2, S("abcd23e"));
    test(S("abcde"), 4, 0, SV("12345"), 1, 3, S("abcd234e"));
    test(S("abcde"), 4, 0, SV("12345"), 1, 4, S("abcd2345e"));
    test(S("abcde"), 4, 0, SV("12345"), 1, 5, S("abcd2345e"));
    test(S("abcde"), 4, 0, SV("12345"), 2, 0, S("abcde"));
    test(S("abcde"), 4, 0, SV("12345"), 2, 1, S("abcd3e"));
    test(S("abcde"), 4, 0, SV("12345"), 2, 2, S("abcd34e"));
    test(S("abcde"), 4, 0, SV("12345"), 2, 3, S("abcd345e"));
    test(S("abcde"), 4, 0, SV("12345"), 2, 4, S("abcd345e"));

    return true;
}

template <class S, class SV>
TEST_CONSTEXPR_CXX20 bool test15()
{
    test(S("abcde"), 4, 0, SV("12345"), 4, 0, S("abcde"));
    test(S("abcde"), 4, 0, SV("12345"), 4, 1, S("abcd5e"));
    test(S("abcde"), 4, 0, SV("12345"), 4, 2, S("abcd5e"));
    test(S("abcde"), 4, 0, SV("12345"), 5, 0, S("abcde"));
    test(S("abcde"), 4, 0, SV("12345"), 5, 1, S("abcde"));
    test(S("abcde"), 4, 0, SV("12345"), 6, 0, S("can't happen"));
    test(S("abcde"), 4, 0, SV("1234567890"), 0, 0, S("abcde"));
    test(S("abcde"), 4, 0, SV("1234567890"), 0, 1, S("abcd1e"));
    test(S("abcde"), 4, 0, SV("1234567890"), 0, 5, S("abcd12345e"));
    test(S("abcde"), 4, 0, SV("1234567890"), 0, 9, S("abcd123456789e"));
    test(S("abcde"), 4, 0, SV("1234567890"), 0, 10, S("abcd1234567890e"));
    test(S("abcde"), 4, 0, SV("1234567890"), 0, 11, S("abcd1234567890e"));
    test(S("abcde"), 4, 0, SV("1234567890"), 1, 0, S("abcde"));
    test(S("abcde"), 4, 0, SV("1234567890"), 1, 1, S("abcd2e"));
    test(S("abcde"), 4, 0, SV("1234567890"), 1, 4, S("abcd2345e"));
    test(S("abcde"), 4, 0, SV("1234567890"), 1, 8, S("abcd23456789e"));
    test(S("abcde"), 4, 0, SV("1234567890"), 1, 9, S("abcd234567890e"));
    test(S("abcde"), 4, 0, SV("1234567890"), 1, 10, S("abcd234567890e"));
    test(S("abcde"), 4, 0, SV("1234567890"), 5, 0, S("abcde"));
    test(S("abcde"), 4, 0, SV("1234567890"), 5, 1, S("abcd6e"));
    test(S("abcde"), 4, 0, SV("1234567890"), 5, 2, S("abcd67e"));
    test(S("abcde"), 4, 0, SV("1234567890"), 5, 4, S("abcd6789e"));
    test(S("abcde"), 4, 0, SV("1234567890"), 5, 5, S("abcd67890e"));
    test(S("abcde"), 4, 0, SV("1234567890"), 5, 6, S("abcd67890e"));
    test(S("abcde"), 4, 0, SV("1234567890"), 9, 0, S("abcde"));
    test(S("abcde"), 4, 0, SV("1234567890"), 9, 1, S("abcd0e"));
    test(S("abcde"), 4, 0, SV("1234567890"), 9, 2, S("abcd0e"));
    test(S("abcde"), 4, 0, SV("1234567890"), 10, 0, S("abcde"));
    test(S("abcde"), 4, 0, SV("1234567890"), 10, 1, S("abcde"));
    test(S("abcde"), 4, 0, SV("1234567890"), 11, 0, S("can't happen"));
    test(S("abcde"), 4, 0, SV("12345678901234567890"), 0, 0, S("abcde"));
    test(S("abcde"), 4, 0, SV("12345678901234567890"), 0, 1, S("abcd1e"));
    test(S("abcde"), 4, 0, SV("12345678901234567890"), 0, 10, S("abcd1234567890e"));
    test(S("abcde"), 4, 0, SV("12345678901234567890"), 0, 19, S("abcd1234567890123456789e"));
    test(S("abcde"), 4, 0, SV("12345678901234567890"), 0, 20, S("abcd12345678901234567890e"));
    test(S("abcde"), 4, 0, SV("12345678901234567890"), 0, 21, S("abcd12345678901234567890e"));
    test(S("abcde"), 4, 0, SV("12345678901234567890"), 1, 0, S("abcde"));
    test(S("abcde"), 4, 0, SV("12345678901234567890"), 1, 1, S("abcd2e"));
    test(S("abcde"), 4, 0, SV("12345678901234567890"), 1, 9, S("abcd234567890e"));
    test(S("abcde"), 4, 0, SV("12345678901234567890"), 1, 18, S("abcd234567890123456789e"));
    test(S("abcde"), 4, 0, SV("12345678901234567890"), 1, 19, S("abcd2345678901234567890e"));
    test(S("abcde"), 4, 0, SV("12345678901234567890"), 1, 20, S("abcd2345678901234567890e"));
    test(S("abcde"), 4, 0, SV("12345678901234567890"), 10, 0, S("abcde"));
    test(S("abcde"), 4, 0, SV("12345678901234567890"), 10, 1, S("abcd1e"));
    test(S("abcde"), 4, 0, SV("12345678901234567890"), 10, 5, S("abcd12345e"));
    test(S("abcde"), 4, 0, SV("12345678901234567890"), 10, 9, S("abcd123456789e"));
    test(S("abcde"), 4, 0, SV("12345678901234567890"), 10, 10, S("abcd1234567890e"));
    test(S("abcde"), 4, 0, SV("12345678901234567890"), 10, 11, S("abcd1234567890e"));
    test(S("abcde"), 4, 0, SV("12345678901234567890"), 19, 0, S("abcde"));
    test(S("abcde"), 4, 0, SV("12345678901234567890"), 19, 1, S("abcd0e"));
    test(S("abcde"), 4, 0, SV("12345678901234567890"), 19, 2, S("abcd0e"));
    test(S("abcde"), 4, 0, SV("12345678901234567890"), 20, 0, S("abcde"));
    test(S("abcde"), 4, 0, SV("12345678901234567890"), 20, 1, S("abcde"));
    test(S("abcde"), 4, 0, SV("12345678901234567890"), 21, 0, S("can't happen"));
    test(S("abcde"), 4, 1, SV(""), 0, 0, S("abcd"));
    test(S("abcde"), 4, 1, SV(""), 0, 1, S("abcd"));
    test(S("abcde"), 4, 1, SV(""), 1, 0, S("can't happen"));
    test(S("abcde"), 4, 1, SV("12345"), 0, 0, S("abcd"));
    test(S("abcde"), 4, 1, SV("12345"), 0, 1, S("abcd1"));
    test(S("abcde"), 4, 1, SV("12345"), 0, 2, S("abcd12"));
    test(S("abcde"), 4, 1, SV("12345"), 0, 4, S("abcd1234"));
    test(S("abcde"), 4, 1, SV("12345"), 0, 5, S("abcd12345"));
    test(S("abcde"), 4, 1, SV("12345"), 0, 6, S("abcd12345"));
    test(S("abcde"), 4, 1, SV("12345"), 1, 0, S("abcd"));
    test(S("abcde"), 4, 1, SV("12345"), 1, 1, S("abcd2"));
    test(S("abcde"), 4, 1, SV("12345"), 1, 2, S("abcd23"));
    test(S("abcde"), 4, 1, SV("12345"), 1, 3, S("abcd234"));
    test(S("abcde"), 4, 1, SV("12345"), 1, 4, S("abcd2345"));
    test(S("abcde"), 4, 1, SV("12345"), 1, 5, S("abcd2345"));
    test(S("abcde"), 4, 1, SV("12345"), 2, 0, S("abcd"));
    test(S("abcde"), 4, 1, SV("12345"), 2, 1, S("abcd3"));
    test(S("abcde"), 4, 1, SV("12345"), 2, 2, S("abcd34"));
    test(S("abcde"), 4, 1, SV("12345"), 2, 3, S("abcd345"));
    test(S("abcde"), 4, 1, SV("12345"), 2, 4, S("abcd345"));
    test(S("abcde"), 4, 1, SV("12345"), 4, 0, S("abcd"));
    test(S("abcde"), 4, 1, SV("12345"), 4, 1, S("abcd5"));
    test(S("abcde"), 4, 1, SV("12345"), 4, 2, S("abcd5"));
    test(S("abcde"), 4, 1, SV("12345"), 5, 0, S("abcd"));
    test(S("abcde"), 4, 1, SV("12345"), 5, 1, S("abcd"));
    test(S("abcde"), 4, 1, SV("12345"), 6, 0, S("can't happen"));
    test(S("abcde"), 4, 1, SV("1234567890"), 0, 0, S("abcd"));
    test(S("abcde"), 4, 1, SV("1234567890"), 0, 1, S("abcd1"));
    test(S("abcde"), 4, 1, SV("1234567890"), 0, 5, S("abcd12345"));
    test(S("abcde"), 4, 1, SV("1234567890"), 0, 9, S("abcd123456789"));
    test(S("abcde"), 4, 1, SV("1234567890"), 0, 10, S("abcd1234567890"));
    test(S("abcde"), 4, 1, SV("1234567890"), 0, 11, S("abcd1234567890"));
    test(S("abcde"), 4, 1, SV("1234567890"), 1, 0, S("abcd"));
    test(S("abcde"), 4, 1, SV("1234567890"), 1, 1, S("abcd2"));
    test(S("abcde"), 4, 1, SV("1234567890"), 1, 4, S("abcd2345"));
    test(S("abcde"), 4, 1, SV("1234567890"), 1, 8, S("abcd23456789"));
    test(S("abcde"), 4, 1, SV("1234567890"), 1, 9, S("abcd234567890"));
    test(S("abcde"), 4, 1, SV("1234567890"), 1, 10, S("abcd234567890"));
    test(S("abcde"), 4, 1, SV("1234567890"), 5, 0, S("abcd"));
    test(S("abcde"), 4, 1, SV("1234567890"), 5, 1, S("abcd6"));
    test(S("abcde"), 4, 1, SV("1234567890"), 5, 2, S("abcd67"));
    test(S("abcde"), 4, 1, SV("1234567890"), 5, 4, S("abcd6789"));
    test(S("abcde"), 4, 1, SV("1234567890"), 5, 5, S("abcd67890"));
    test(S("abcde"), 4, 1, SV("1234567890"), 5, 6, S("abcd67890"));
    test(S("abcde"), 4, 1, SV("1234567890"), 9, 0, S("abcd"));
    test(S("abcde"), 4, 1, SV("1234567890"), 9, 1, S("abcd0"));

    return true;
}

template <class S, class SV>
TEST_CONSTEXPR_CXX20 bool test16()
{
    test(S("abcde"), 4, 1, SV("1234567890"), 9, 2, S("abcd0"));
    test(S("abcde"), 4, 1, SV("1234567890"), 10, 0, S("abcd"));
    test(S("abcde"), 4, 1, SV("1234567890"), 10, 1, S("abcd"));
    test(S("abcde"), 4, 1, SV("1234567890"), 11, 0, S("can't happen"));
    test(S("abcde"), 4, 1, SV("12345678901234567890"), 0, 0, S("abcd"));
    test(S("abcde"), 4, 1, SV("12345678901234567890"), 0, 1, S("abcd1"));
    test(S("abcde"), 4, 1, SV("12345678901234567890"), 0, 10, S("abcd1234567890"));
    test(S("abcde"), 4, 1, SV("12345678901234567890"), 0, 19, S("abcd1234567890123456789"));
    test(S("abcde"), 4, 1, SV("12345678901234567890"), 0, 20, S("abcd12345678901234567890"));
    test(S("abcde"), 4, 1, SV("12345678901234567890"), 0, 21, S("abcd12345678901234567890"));
    test(S("abcde"), 4, 1, SV("12345678901234567890"), 1, 0, S("abcd"));
    test(S("abcde"), 4, 1, SV("12345678901234567890"), 1, 1, S("abcd2"));
    test(S("abcde"), 4, 1, SV("12345678901234567890"), 1, 9, S("abcd234567890"));
    test(S("abcde"), 4, 1, SV("12345678901234567890"), 1, 18, S("abcd234567890123456789"));
    test(S("abcde"), 4, 1, SV("12345678901234567890"), 1, 19, S("abcd2345678901234567890"));
    test(S("abcde"), 4, 1, SV("12345678901234567890"), 1, 20, S("abcd2345678901234567890"));
    test(S("abcde"), 4, 1, SV("12345678901234567890"), 10, 0, S("abcd"));
    test(S("abcde"), 4, 1, SV("12345678901234567890"), 10, 1, S("abcd1"));
    test(S("abcde"), 4, 1, SV("12345678901234567890"), 10, 5, S("abcd12345"));
    test(S("abcde"), 4, 1, SV("12345678901234567890"), 10, 9, S("abcd123456789"));
    test(S("abcde"), 4, 1, SV("12345678901234567890"), 10, 10, S("abcd1234567890"));
    test(S("abcde"), 4, 1, SV("12345678901234567890"), 10, 11, S("abcd1234567890"));
    test(S("abcde"), 4, 1, SV("12345678901234567890"), 19, 0, S("abcd"));
    test(S("abcde"), 4, 1, SV("12345678901234567890"), 19, 1, S("abcd0"));
    test(S("abcde"), 4, 1, SV("12345678901234567890"), 19, 2, S("abcd0"));
    test(S("abcde"), 4, 1, SV("12345678901234567890"), 20, 0, S("abcd"));
    test(S("abcde"), 4, 1, SV("12345678901234567890"), 20, 1, S("abcd"));
    test(S("abcde"), 4, 1, SV("12345678901234567890"), 21, 0, S("can't happen"));
    test(S("abcde"), 4, 2, SV(""), 0, 0, S("abcd"));
    test(S("abcde"), 4, 2, SV(""), 0, 1, S("abcd"));
    test(S("abcde"), 4, 2, SV(""), 1, 0, S("can't happen"));
    test(S("abcde"), 4, 2, SV("12345"), 0, 0, S("abcd"));
    test(S("abcde"), 4, 2, SV("12345"), 0, 1, S("abcd1"));
    test(S("abcde"), 4, 2, SV("12345"), 0, 2, S("abcd12"));
    test(S("abcde"), 4, 2, SV("12345"), 0, 4, S("abcd1234"));
    test(S("abcde"), 4, 2, SV("12345"), 0, 5, S("abcd12345"));
    test(S("abcde"), 4, 2, SV("12345"), 0, 6, S("abcd12345"));
    test(S("abcde"), 4, 2, SV("12345"), 1, 0, S("abcd"));
    test(S("abcde"), 4, 2, SV("12345"), 1, 1, S("abcd2"));
    test(S("abcde"), 4, 2, SV("12345"), 1, 2, S("abcd23"));
    test(S("abcde"), 4, 2, SV("12345"), 1, 3, S("abcd234"));
    test(S("abcde"), 4, 2, SV("12345"), 1, 4, S("abcd2345"));
    test(S("abcde"), 4, 2, SV("12345"), 1, 5, S("abcd2345"));
    test(S("abcde"), 4, 2, SV("12345"), 2, 0, S("abcd"));
    test(S("abcde"), 4, 2, SV("12345"), 2, 1, S("abcd3"));
    test(S("abcde"), 4, 2, SV("12345"), 2, 2, S("abcd34"));
    test(S("abcde"), 4, 2, SV("12345"), 2, 3, S("abcd345"));
    test(S("abcde"), 4, 2, SV("12345"), 2, 4, S("abcd345"));
    test(S("abcde"), 4, 2, SV("12345"), 4, 0, S("abcd"));
    test(S("abcde"), 4, 2, SV("12345"), 4, 1, S("abcd5"));
    test(S("abcde"), 4, 2, SV("12345"), 4, 2, S("abcd5"));
    test(S("abcde"), 4, 2, SV("12345"), 5, 0, S("abcd"));
    test(S("abcde"), 4, 2, SV("12345"), 5, 1, S("abcd"));
    test(S("abcde"), 4, 2, SV("12345"), 6, 0, S("can't happen"));
    test(S("abcde"), 4, 2, SV("1234567890"), 0, 0, S("abcd"));
    test(S("abcde"), 4, 2, SV("1234567890"), 0, 1, S("abcd1"));
    test(S("abcde"), 4, 2, SV("1234567890"), 0, 5, S("abcd12345"));
    test(S("abcde"), 4, 2, SV("1234567890"), 0, 9, S("abcd123456789"));
    test(S("abcde"), 4, 2, SV("1234567890"), 0, 10, S("abcd1234567890"));
    test(S("abcde"), 4, 2, SV("1234567890"), 0, 11, S("abcd1234567890"));
    test(S("abcde"), 4, 2, SV("1234567890"), 1, 0, S("abcd"));
    test(S("abcde"), 4, 2, SV("1234567890"), 1, 1, S("abcd2"));
    test(S("abcde"), 4, 2, SV("1234567890"), 1, 4, S("abcd2345"));
    test(S("abcde"), 4, 2, SV("1234567890"), 1, 8, S("abcd23456789"));
    test(S("abcde"), 4, 2, SV("1234567890"), 1, 9, S("abcd234567890"));
    test(S("abcde"), 4, 2, SV("1234567890"), 1, 10, S("abcd234567890"));
    test(S("abcde"), 4, 2, SV("1234567890"), 5, 0, S("abcd"));
    test(S("abcde"), 4, 2, SV("1234567890"), 5, 1, S("abcd6"));
    test(S("abcde"), 4, 2, SV("1234567890"), 5, 2, S("abcd67"));
    test(S("abcde"), 4, 2, SV("1234567890"), 5, 4, S("abcd6789"));
    test(S("abcde"), 4, 2, SV("1234567890"), 5, 5, S("abcd67890"));
    test(S("abcde"), 4, 2, SV("1234567890"), 5, 6, S("abcd67890"));
    test(S("abcde"), 4, 2, SV("1234567890"), 9, 0, S("abcd"));
    test(S("abcde"), 4, 2, SV("1234567890"), 9, 1, S("abcd0"));
    test(S("abcde"), 4, 2, SV("1234567890"), 9, 2, S("abcd0"));
    test(S("abcde"), 4, 2, SV("1234567890"), 10, 0, S("abcd"));
    test(S("abcde"), 4, 2, SV("1234567890"), 10, 1, S("abcd"));
    test(S("abcde"), 4, 2, SV("1234567890"), 11, 0, S("can't happen"));
    test(S("abcde"), 4, 2, SV("12345678901234567890"), 0, 0, S("abcd"));
    test(S("abcde"), 4, 2, SV("12345678901234567890"), 0, 1, S("abcd1"));
    test(S("abcde"), 4, 2, SV("12345678901234567890"), 0, 10, S("abcd1234567890"));
    test(S("abcde"), 4, 2, SV("12345678901234567890"), 0, 19, S("abcd1234567890123456789"));
    test(S("abcde"), 4, 2, SV("12345678901234567890"), 0, 20, S("abcd12345678901234567890"));
    test(S("abcde"), 4, 2, SV("12345678901234567890"), 0, 21, S("abcd12345678901234567890"));
    test(S("abcde"), 4, 2, SV("12345678901234567890"), 1, 0, S("abcd"));
    test(S("abcde"), 4, 2, SV("12345678901234567890"), 1, 1, S("abcd2"));
    test(S("abcde"), 4, 2, SV("12345678901234567890"), 1, 9, S("abcd234567890"));
    test(S("abcde"), 4, 2, SV("12345678901234567890"), 1, 18, S("abcd234567890123456789"));
    test(S("abcde"), 4, 2, SV("12345678901234567890"), 1, 19, S("abcd2345678901234567890"));
    test(S("abcde"), 4, 2, SV("12345678901234567890"), 1, 20, S("abcd2345678901234567890"));
    test(S("abcde"), 4, 2, SV("12345678901234567890"), 10, 0, S("abcd"));
    test(S("abcde"), 4, 2, SV("12345678901234567890"), 10, 1, S("abcd1"));
    test(S("abcde"), 4, 2, SV("12345678901234567890"), 10, 5, S("abcd12345"));
    test(S("abcde"), 4, 2, SV("12345678901234567890"), 10, 9, S("abcd123456789"));
    test(S("abcde"), 4, 2, SV("12345678901234567890"), 10, 10, S("abcd1234567890"));
    test(S("abcde"), 4, 2, SV("12345678901234567890"), 10, 11, S("abcd1234567890"));
    test(S("abcde"), 4, 2, SV("12345678901234567890"), 19, 0, S("abcd"));
    test(S("abcde"), 4, 2, SV("12345678901234567890"), 19, 1, S("abcd0"));
    test(S("abcde"), 4, 2, SV("12345678901234567890"), 19, 2, S("abcd0"));
    test(S("abcde"), 4, 2, SV("12345678901234567890"), 20, 0, S("abcd"));

    return true;
}

template <class S, class SV>
TEST_CONSTEXPR_CXX20 bool test17()
{
    test(S("abcde"), 4, 2, SV("12345678901234567890"), 20, 1, S("abcd"));
    test(S("abcde"), 4, 2, SV("12345678901234567890"), 21, 0, S("can't happen"));
    test(S("abcde"), 5, 0, SV(""), 0, 0, S("abcde"));
    test(S("abcde"), 5, 0, SV(""), 0, 1, S("abcde"));
    test(S("abcde"), 5, 0, SV(""), 1, 0, S("can't happen"));
    test(S("abcde"), 5, 0, SV("12345"), 0, 0, S("abcde"));
    test(S("abcde"), 5, 0, SV("12345"), 0, 1, S("abcde1"));
    test(S("abcde"), 5, 0, SV("12345"), 0, 2, S("abcde12"));
    test(S("abcde"), 5, 0, SV("12345"), 0, 4, S("abcde1234"));
    test(S("abcde"), 5, 0, SV("12345"), 0, 5, S("abcde12345"));
    test(S("abcde"), 5, 0, SV("12345"), 0, 6, S("abcde12345"));
    test(S("abcde"), 5, 0, SV("12345"), 1, 0, S("abcde"));
    test(S("abcde"), 5, 0, SV("12345"), 1, 1, S("abcde2"));
    test(S("abcde"), 5, 0, SV("12345"), 1, 2, S("abcde23"));
    test(S("abcde"), 5, 0, SV("12345"), 1, 3, S("abcde234"));
    test(S("abcde"), 5, 0, SV("12345"), 1, 4, S("abcde2345"));
    test(S("abcde"), 5, 0, SV("12345"), 1, 5, S("abcde2345"));
    test(S("abcde"), 5, 0, SV("12345"), 2, 0, S("abcde"));
    test(S("abcde"), 5, 0, SV("12345"), 2, 1, S("abcde3"));
    test(S("abcde"), 5, 0, SV("12345"), 2, 2, S("abcde34"));
    test(S("abcde"), 5, 0, SV("12345"), 2, 3, S("abcde345"));
    test(S("abcde"), 5, 0, SV("12345"), 2, 4, S("abcde345"));
    test(S("abcde"), 5, 0, SV("12345"), 4, 0, S("abcde"));
    test(S("abcde"), 5, 0, SV("12345"), 4, 1, S("abcde5"));
    test(S("abcde"), 5, 0, SV("12345"), 4, 2, S("abcde5"));
    test(S("abcde"), 5, 0, SV("12345"), 5, 0, S("abcde"));
    test(S("abcde"), 5, 0, SV("12345"), 5, 1, S("abcde"));
    test(S("abcde"), 5, 0, SV("12345"), 6, 0, S("can't happen"));
    test(S("abcde"), 5, 0, SV("1234567890"), 0, 0, S("abcde"));
    test(S("abcde"), 5, 0, SV("1234567890"), 0, 1, S("abcde1"));
    test(S("abcde"), 5, 0, SV("1234567890"), 0, 5, S("abcde12345"));
    test(S("abcde"), 5, 0, SV("1234567890"), 0, 9, S("abcde123456789"));
    test(S("abcde"), 5, 0, SV("1234567890"), 0, 10, S("abcde1234567890"));
    test(S("abcde"), 5, 0, SV("1234567890"), 0, 11, S("abcde1234567890"));
    test(S("abcde"), 5, 0, SV("1234567890"), 1, 0, S("abcde"));
    test(S("abcde"), 5, 0, SV("1234567890"), 1, 1, S("abcde2"));
    test(S("abcde"), 5, 0, SV("1234567890"), 1, 4, S("abcde2345"));
    test(S("abcde"), 5, 0, SV("1234567890"), 1, 8, S("abcde23456789"));
    test(S("abcde"), 5, 0, SV("1234567890"), 1, 9, S("abcde234567890"));
    test(S("abcde"), 5, 0, SV("1234567890"), 1, 10, S("abcde234567890"));
    test(S("abcde"), 5, 0, SV("1234567890"), 5, 0, S("abcde"));
    test(S("abcde"), 5, 0, SV("1234567890"), 5, 1, S("abcde6"));
    test(S("abcde"), 5, 0, SV("1234567890"), 5, 2, S("abcde67"));
    test(S("abcde"), 5, 0, SV("1234567890"), 5, 4, S("abcde6789"));
    test(S("abcde"), 5, 0, SV("1234567890"), 5, 5, S("abcde67890"));
    test(S("abcde"), 5, 0, SV("1234567890"), 5, 6, S("abcde67890"));
    test(S("abcde"), 5, 0, SV("1234567890"), 9, 0, S("abcde"));
    test(S("abcde"), 5, 0, SV("1234567890"), 9, 1, S("abcde0"));
    test(S("abcde"), 5, 0, SV("1234567890"), 9, 2, S("abcde0"));
    test(S("abcde"), 5, 0, SV("1234567890"), 10, 0, S("abcde"));
    test(S("abcde"), 5, 0, SV("1234567890"), 10, 1, S("abcde"));
    test(S("abcde"), 5, 0, SV("1234567890"), 11, 0, S("can't happen"));
    test(S("abcde"), 5, 0, SV("12345678901234567890"), 0, 0, S("abcde"));
    test(S("abcde"), 5, 0, SV("12345678901234567890"), 0, 1, S("abcde1"));
    test(S("abcde"), 5, 0, SV("12345678901234567890"), 0, 10, S("abcde1234567890"));
    test(S("abcde"), 5, 0, SV("12345678901234567890"), 0, 19, S("abcde1234567890123456789"));
    test(S("abcde"), 5, 0, SV("12345678901234567890"), 0, 20, S("abcde12345678901234567890"));
    test(S("abcde"), 5, 0, SV("12345678901234567890"), 0, 21, S("abcde12345678901234567890"));
    test(S("abcde"), 5, 0, SV("12345678901234567890"), 1, 0, S("abcde"));
    test(S("abcde"), 5, 0, SV("12345678901234567890"), 1, 1, S("abcde2"));
    test(S("abcde"), 5, 0, SV("12345678901234567890"), 1, 9, S("abcde234567890"));
    test(S("abcde"), 5, 0, SV("12345678901234567890"), 1, 18, S("abcde234567890123456789"));
    test(S("abcde"), 5, 0, SV("12345678901234567890"), 1, 19, S("abcde2345678901234567890"));
    test(S("abcde"), 5, 0, SV("12345678901234567890"), 1, 20, S("abcde2345678901234567890"));
    test(S("abcde"), 5, 0, SV("12345678901234567890"), 10, 0, S("abcde"));
    test(S("abcde"), 5, 0, SV("12345678901234567890"), 10, 1, S("abcde1"));
    test(S("abcde"), 5, 0, SV("12345678901234567890"), 10, 5, S("abcde12345"));
    test(S("abcde"), 5, 0, SV("12345678901234567890"), 10, 9, S("abcde123456789"));
    test(S("abcde"), 5, 0, SV("12345678901234567890"), 10, 10, S("abcde1234567890"));
    test(S("abcde"), 5, 0, SV("12345678901234567890"), 10, 11, S("abcde1234567890"));
    test(S("abcde"), 5, 0, SV("12345678901234567890"), 19, 0, S("abcde"));
    test(S("abcde"), 5, 0, SV("12345678901234567890"), 19, 1, S("abcde0"));
    test(S("abcde"), 5, 0, SV("12345678901234567890"), 19, 2, S("abcde0"));
    test(S("abcde"), 5, 0, SV("12345678901234567890"), 20, 0, S("abcde"));
    test(S("abcde"), 5, 0, SV("12345678901234567890"), 20, 1, S("abcde"));
    test(S("abcde"), 5, 0, SV("12345678901234567890"), 21, 0, S("can't happen"));
    test(S("abcde"), 5, 1, SV(""), 0, 0, S("abcde"));
    test(S("abcde"), 5, 1, SV(""), 0, 1, S("abcde"));
    test(S("abcde"), 5, 1, SV(""), 1, 0, S("can't happen"));
    test(S("abcde"), 5, 1, SV("12345"), 0, 0, S("abcde"));
    test(S("abcde"), 5, 1, SV("12345"), 0, 1, S("abcde1"));
    test(S("abcde"), 5, 1, SV("12345"), 0, 2, S("abcde12"));
    test(S("abcde"), 5, 1, SV("12345"), 0, 4, S("abcde1234"));
    test(S("abcde"), 5, 1, SV("12345"), 0, 5, S("abcde12345"));
    test(S("abcde"), 5, 1, SV("12345"), 0, 6, S("abcde12345"));
    test(S("abcde"), 5, 1, SV("12345"), 1, 0, S("abcde"));
    test(S("abcde"), 5, 1, SV("12345"), 1, 1, S("abcde2"));
    test(S("abcde"), 5, 1, SV("12345"), 1, 2, S("abcde23"));
    test(S("abcde"), 5, 1, SV("12345"), 1, 3, S("abcde234"));
    test(S("abcde"), 5, 1, SV("12345"), 1, 4, S("abcde2345"));
    test(S("abcde"), 5, 1, SV("12345"), 1, 5, S("abcde2345"));
    test(S("abcde"), 5, 1, SV("12345"), 2, 0, S("abcde"));
    test(S("abcde"), 5, 1, SV("12345"), 2, 1, S("abcde3"));
    test(S("abcde"), 5, 1, SV("12345"), 2, 2, S("abcde34"));
    test(S("abcde"), 5, 1, SV("12345"), 2, 3, S("abcde345"));
    test(S("abcde"), 5, 1, SV("12345"), 2, 4, S("abcde345"));
    test(S("abcde"), 5, 1, SV("12345"), 4, 0, S("abcde"));
    test(S("abcde"), 5, 1, SV("12345"), 4, 1, S("abcde5"));
    test(S("abcde"), 5, 1, SV("12345"), 4, 2, S("abcde5"));
    test(S("abcde"), 5, 1, SV("12345"), 5, 0, S("abcde"));

    return true;
}

template <class S, class SV>
TEST_CONSTEXPR_CXX20 bool test18()
{
    test(S("abcde"), 5, 1, SV("12345"), 5, 1, S("abcde"));
    test(S("abcde"), 5, 1, SV("12345"), 6, 0, S("can't happen"));
    test(S("abcde"), 5, 1, SV("1234567890"), 0, 0, S("abcde"));
    test(S("abcde"), 5, 1, SV("1234567890"), 0, 1, S("abcde1"));
    test(S("abcde"), 5, 1, SV("1234567890"), 0, 5, S("abcde12345"));
    test(S("abcde"), 5, 1, SV("1234567890"), 0, 9, S("abcde123456789"));
    test(S("abcde"), 5, 1, SV("1234567890"), 0, 10, S("abcde1234567890"));
    test(S("abcde"), 5, 1, SV("1234567890"), 0, 11, S("abcde1234567890"));
    test(S("abcde"), 5, 1, SV("1234567890"), 1, 0, S("abcde"));
    test(S("abcde"), 5, 1, SV("1234567890"), 1, 1, S("abcde2"));
    test(S("abcde"), 5, 1, SV("1234567890"), 1, 4, S("abcde2345"));
    test(S("abcde"), 5, 1, SV("1234567890"), 1, 8, S("abcde23456789"));
    test(S("abcde"), 5, 1, SV("1234567890"), 1, 9, S("abcde234567890"));
    test(S("abcde"), 5, 1, SV("1234567890"), 1, 10, S("abcde234567890"));
    test(S("abcde"), 5, 1, SV("1234567890"), 5, 0, S("abcde"));
    test(S("abcde"), 5, 1, SV("1234567890"), 5, 1, S("abcde6"));
    test(S("abcde"), 5, 1, SV("1234567890"), 5, 2, S("abcde67"));
    test(S("abcde"), 5, 1, SV("1234567890"), 5, 4, S("abcde6789"));
    test(S("abcde"), 5, 1, SV("1234567890"), 5, 5, S("abcde67890"));
    test(S("abcde"), 5, 1, SV("1234567890"), 5, 6, S("abcde67890"));
    test(S("abcde"), 5, 1, SV("1234567890"), 9, 0, S("abcde"));
    test(S("abcde"), 5, 1, SV("1234567890"), 9, 1, S("abcde0"));
    test(S("abcde"), 5, 1, SV("1234567890"), 9, 2, S("abcde0"));
    test(S("abcde"), 5, 1, SV("1234567890"), 10, 0, S("abcde"));
    test(S("abcde"), 5, 1, SV("1234567890"), 10, 1, S("abcde"));
    test(S("abcde"), 5, 1, SV("1234567890"), 11, 0, S("can't happen"));
    test(S("abcde"), 5, 1, SV("12345678901234567890"), 0, 0, S("abcde"));
    test(S("abcde"), 5, 1, SV("12345678901234567890"), 0, 1, S("abcde1"));
    test(S("abcde"), 5, 1, SV("12345678901234567890"), 0, 10, S("abcde1234567890"));
    test(S("abcde"), 5, 1, SV("12345678901234567890"), 0, 19, S("abcde1234567890123456789"));
    test(S("abcde"), 5, 1, SV("12345678901234567890"), 0, 20, S("abcde12345678901234567890"));
    test(S("abcde"), 5, 1, SV("12345678901234567890"), 0, 21, S("abcde12345678901234567890"));
    test(S("abcde"), 5, 1, SV("12345678901234567890"), 1, 0, S("abcde"));
    test(S("abcde"), 5, 1, SV("12345678901234567890"), 1, 1, S("abcde2"));
    test(S("abcde"), 5, 1, SV("12345678901234567890"), 1, 9, S("abcde234567890"));
    test(S("abcde"), 5, 1, SV("12345678901234567890"), 1, 18, S("abcde234567890123456789"));
    test(S("abcde"), 5, 1, SV("12345678901234567890"), 1, 19, S("abcde2345678901234567890"));
    test(S("abcde"), 5, 1, SV("12345678901234567890"), 1, 20, S("abcde2345678901234567890"));
    test(S("abcde"), 5, 1, SV("12345678901234567890"), 10, 0, S("abcde"));
    test(S("abcde"), 5, 1, SV("12345678901234567890"), 10, 1, S("abcde1"));
    test(S("abcde"), 5, 1, SV("12345678901234567890"), 10, 5, S("abcde12345"));
    test(S("abcde"), 5, 1, SV("12345678901234567890"), 10, 9, S("abcde123456789"));
    test(S("abcde"), 5, 1, SV("12345678901234567890"), 10, 10, S("abcde1234567890"));
    test(S("abcde"), 5, 1, SV("12345678901234567890"), 10, 11, S("abcde1234567890"));
    test(S("abcde"), 5, 1, SV("12345678901234567890"), 19, 0, S("abcde"));
    test(S("abcde"), 5, 1, SV("12345678901234567890"), 19, 1, S("abcde0"));
    test(S("abcde"), 5, 1, SV("12345678901234567890"), 19, 2, S("abcde0"));
    test(S("abcde"), 5, 1, SV("12345678901234567890"), 20, 0, S("abcde"));
    test(S("abcde"), 5, 1, SV("12345678901234567890"), 20, 1, S("abcde"));
    test(S("abcde"), 5, 1, SV("12345678901234567890"), 21, 0, S("can't happen"));
    test(S("abcde"), 6, 0, SV(""), 0, 0, S("can't happen"));
    test(S("abcde"), 6, 0, SV(""), 0, 1, S("can't happen"));
    test(S("abcde"), 6, 0, SV(""), 1, 0, S("can't happen"));
    test(S("abcde"), 6, 0, SV("12345"), 0, 0, S("can't happen"));
    test(S("abcde"), 6, 0, SV("12345"), 0, 1, S("can't happen"));
    test(S("abcde"), 6, 0, SV("12345"), 0, 2, S("can't happen"));
    test(S("abcde"), 6, 0, SV("12345"), 0, 4, S("can't happen"));
    test(S("abcde"), 6, 0, SV("12345"), 0, 5, S("can't happen"));
    test(S("abcde"), 6, 0, SV("12345"), 0, 6, S("can't happen"));
    test(S("abcde"), 6, 0, SV("12345"), 1, 0, S("can't happen"));
    test(S("abcde"), 6, 0, SV("12345"), 1, 1, S("can't happen"));
    test(S("abcde"), 6, 0, SV("12345"), 1, 2, S("can't happen"));
    test(S("abcde"), 6, 0, SV("12345"), 1, 3, S("can't happen"));
    test(S("abcde"), 6, 0, SV("12345"), 1, 4, S("can't happen"));
    test(S("abcde"), 6, 0, SV("12345"), 1, 5, S("can't happen"));
    test(S("abcde"), 6, 0, SV("12345"), 2, 0, S("can't happen"));
    test(S("abcde"), 6, 0, SV("12345"), 2, 1, S("can't happen"));
    test(S("abcde"), 6, 0, SV("12345"), 2, 2, S("can't happen"));
    test(S("abcde"), 6, 0, SV("12345"), 2, 3, S("can't happen"));
    test(S("abcde"), 6, 0, SV("12345"), 2, 4, S("can't happen"));
    test(S("abcde"), 6, 0, SV("12345"), 4, 0, S("can't happen"));
    test(S("abcde"), 6, 0, SV("12345"), 4, 1, S("can't happen"));
    test(S("abcde"), 6, 0, SV("12345"), 4, 2, S("can't happen"));
    test(S("abcde"), 6, 0, SV("12345"), 5, 0, S("can't happen"));
    test(S("abcde"), 6, 0, SV("12345"), 5, 1, S("can't happen"));
    test(S("abcde"), 6, 0, SV("12345"), 6, 0, S("can't happen"));
    test(S("abcde"), 6, 0, SV("1234567890"), 0, 0, S("can't happen"));
    test(S("abcde"), 6, 0, SV("1234567890"), 0, 1, S("can't happen"));
    test(S("abcde"), 6, 0, SV("1234567890"), 0, 5, S("can't happen"));
    test(S("abcde"), 6, 0, SV("1234567890"), 0, 9, S("can't happen"));
    test(S("abcde"), 6, 0, SV("1234567890"), 0, 10, S("can't happen"));
    test(S("abcde"), 6, 0, SV("1234567890"), 0, 11, S("can't happen"));
    test(S("abcde"), 6, 0, SV("1234567890"), 1, 0, S("can't happen"));
    test(S("abcde"), 6, 0, SV("1234567890"), 1, 1, S("can't happen"));
    test(S("abcde"), 6, 0, SV("1234567890"), 1, 4, S("can't happen"));
    test(S("abcde"), 6, 0, SV("1234567890"), 1, 8, S("can't happen"));
    test(S("abcde"), 6, 0, SV("1234567890"), 1, 9, S("can't happen"));
    test(S("abcde"), 6, 0, SV("1234567890"), 1, 10, S("can't happen"));
    test(S("abcde"), 6, 0, SV("1234567890"), 5, 0, S("can't happen"));
    test(S("abcde"), 6, 0, SV("1234567890"), 5, 1, S("can't happen"));
    test(S("abcde"), 6, 0, SV("1234567890"), 5, 2, S("can't happen"));
    test(S("abcde"), 6, 0, SV("1234567890"), 5, 4, S("can't happen"));
    test(S("abcde"), 6, 0, SV("1234567890"), 5, 5, S("can't happen"));
    test(S("abcde"), 6, 0, SV("1234567890"), 5, 6, S("can't happen"));
    test(S("abcde"), 6, 0, SV("1234567890"), 9, 0, S("can't happen"));
    test(S("abcde"), 6, 0, SV("1234567890"), 9, 1, S("can't happen"));
    test(S("abcde"), 6, 0, SV("1234567890"), 9, 2, S("can't happen"));
    test(S("abcde"), 6, 0, SV("1234567890"), 10, 0, S("can't happen"));
    test(S("abcde"), 6, 0, SV("1234567890"), 10, 1, S("can't happen"));
    test(S("abcde"), 6, 0, SV("1234567890"), 11, 0, S("can't happen"));

    return true;
}

template <class S, class SV>
TEST_CONSTEXPR_CXX20 bool test19()
{
    test(S("abcde"), 6, 0, SV("12345678901234567890"), 0, 0, S("can't happen"));
    test(S("abcde"), 6, 0, SV("12345678901234567890"), 0, 1, S("can't happen"));
    test(S("abcde"), 6, 0, SV("12345678901234567890"), 0, 10, S("can't happen"));
    test(S("abcde"), 6, 0, SV("12345678901234567890"), 0, 19, S("can't happen"));
    test(S("abcde"), 6, 0, SV("12345678901234567890"), 0, 20, S("can't happen"));
    test(S("abcde"), 6, 0, SV("12345678901234567890"), 0, 21, S("can't happen"));
    test(S("abcde"), 6, 0, SV("12345678901234567890"), 1, 0, S("can't happen"));
    test(S("abcde"), 6, 0, SV("12345678901234567890"), 1, 1, S("can't happen"));
    test(S("abcde"), 6, 0, SV("12345678901234567890"), 1, 9, S("can't happen"));
    test(S("abcde"), 6, 0, SV("12345678901234567890"), 1, 18, S("can't happen"));
    test(S("abcde"), 6, 0, SV("12345678901234567890"), 1, 19, S("can't happen"));
    test(S("abcde"), 6, 0, SV("12345678901234567890"), 1, 20, S("can't happen"));
    test(S("abcde"), 6, 0, SV("12345678901234567890"), 10, 0, S("can't happen"));
    test(S("abcde"), 6, 0, SV("12345678901234567890"), 10, 1, S("can't happen"));
    test(S("abcde"), 6, 0, SV("12345678901234567890"), 10, 5, S("can't happen"));
    test(S("abcde"), 6, 0, SV("12345678901234567890"), 10, 9, S("can't happen"));
    test(S("abcde"), 6, 0, SV("12345678901234567890"), 10, 10, S("can't happen"));
    test(S("abcde"), 6, 0, SV("12345678901234567890"), 10, 11, S("can't happen"));
    test(S("abcde"), 6, 0, SV("12345678901234567890"), 19, 0, S("can't happen"));
    test(S("abcde"), 6, 0, SV("12345678901234567890"), 19, 1, S("can't happen"));
    test(S("abcde"), 6, 0, SV("12345678901234567890"), 19, 2, S("can't happen"));
    test(S("abcde"), 6, 0, SV("12345678901234567890"), 20, 0, S("can't happen"));
    test(S("abcde"), 6, 0, SV("12345678901234567890"), 20, 1, S("can't happen"));
    test(S("abcde"), 6, 0, SV("12345678901234567890"), 21, 0, S("can't happen"));
    test(S("abcdefghij"), 0, 0, SV(""), 0, 0, S("abcdefghij"));
    test(S("abcdefghij"), 0, 0, SV(""), 0, 1, S("abcdefghij"));
    test(S("abcdefghij"), 0, 0, SV(""), 1, 0, S("can't happen"));
    test(S("abcdefghij"), 0, 0, SV("12345"), 0, 0, S("abcdefghij"));
    test(S("abcdefghij"), 0, 0, SV("12345"), 0, 1, S("1abcdefghij"));
    test(S("abcdefghij"), 0, 0, SV("12345"), 0, 2, S("12abcdefghij"));
    test(S("abcdefghij"), 0, 0, SV("12345"), 0, 4, S("1234abcdefghij"));
    test(S("abcdefghij"), 0, 0, SV("12345"), 0, 5, S("12345abcdefghij"));
    test(S("abcdefghij"), 0, 0, SV("12345"), 0, 6, S("12345abcdefghij"));
    test(S("abcdefghij"), 0, 0, SV("12345"), 1, 0, S("abcdefghij"));
    test(S("abcdefghij"), 0, 0, SV("12345"), 1, 1, S("2abcdefghij"));
    test(S("abcdefghij"), 0, 0, SV("12345"), 1, 2, S("23abcdefghij"));
    test(S("abcdefghij"), 0, 0, SV("12345"), 1, 3, S("234abcdefghij"));
    test(S("abcdefghij"), 0, 0, SV("12345"), 1, 4, S("2345abcdefghij"));
    test(S("abcdefghij"), 0, 0, SV("12345"), 1, 5, S("2345abcdefghij"));
    test(S("abcdefghij"), 0, 0, SV("12345"), 2, 0, S("abcdefghij"));
    test(S("abcdefghij"), 0, 0, SV("12345"), 2, 1, S("3abcdefghij"));
    test(S("abcdefghij"), 0, 0, SV("12345"), 2, 2, S("34abcdefghij"));
    test(S("abcdefghij"), 0, 0, SV("12345"), 2, 3, S("345abcdefghij"));
    test(S("abcdefghij"), 0, 0, SV("12345"), 2, 4, S("345abcdefghij"));
    test(S("abcdefghij"), 0, 0, SV("12345"), 4, 0, S("abcdefghij"));
    test(S("abcdefghij"), 0, 0, SV("12345"), 4, 1, S("5abcdefghij"));
    test(S("abcdefghij"), 0, 0, SV("12345"), 4, 2, S("5abcdefghij"));
    test(S("abcdefghij"), 0, 0, SV("12345"), 5, 0, S("abcdefghij"));
    test(S("abcdefghij"), 0, 0, SV("12345"), 5, 1, S("abcdefghij"));
    test(S("abcdefghij"), 0, 0, SV("12345"), 6, 0, S("can't happen"));
    test(S("abcdefghij"), 0, 0, SV("1234567890"), 0, 0, S("abcdefghij"));
    test(S("abcdefghij"), 0, 0, SV("1234567890"), 0, 1, S("1abcdefghij"));
    test(S("abcdefghij"), 0, 0, SV("1234567890"), 0, 5, S("12345abcdefghij"));
    test(S("abcdefghij"), 0, 0, SV("1234567890"), 0, 9, S("123456789abcdefghij"));
    test(S("abcdefghij"), 0, 0, SV("1234567890"), 0, 10, S("1234567890abcdefghij"));
    test(S("abcdefghij"), 0, 0, SV("1234567890"), 0, 11, S("1234567890abcdefghij"));
    test(S("abcdefghij"), 0, 0, SV("1234567890"), 1, 0, S("abcdefghij"));
    test(S("abcdefghij"), 0, 0, SV("1234567890"), 1, 1, S("2abcdefghij"));
    test(S("abcdefghij"), 0, 0, SV("1234567890"), 1, 4, S("2345abcdefghij"));
    test(S("abcdefghij"), 0, 0, SV("1234567890"), 1, 8, S("23456789abcdefghij"));
    test(S("abcdefghij"), 0, 0, SV("1234567890"), 1, 9, S("234567890abcdefghij"));
    test(S("abcdefghij"), 0, 0, SV("1234567890"), 1, 10, S("234567890abcdefghij"));
    test(S("abcdefghij"), 0, 0, SV("1234567890"), 5, 0, S("abcdefghij"));
    test(S("abcdefghij"), 0, 0, SV("1234567890"), 5, 1, S("6abcdefghij"));
    test(S("abcdefghij"), 0, 0, SV("1234567890"), 5, 2, S("67abcdefghij"));
    test(S("abcdefghij"), 0, 0, SV("1234567890"), 5, 4, S("6789abcdefghij"));
    test(S("abcdefghij"), 0, 0, SV("1234567890"), 5, 5, S("67890abcdefghij"));
    test(S("abcdefghij"), 0, 0, SV("1234567890"), 5, 6, S("67890abcdefghij"));
    test(S("abcdefghij"), 0, 0, SV("1234567890"), 9, 0, S("abcdefghij"));
    test(S("abcdefghij"), 0, 0, SV("1234567890"), 9, 1, S("0abcdefghij"));
    test(S("abcdefghij"), 0, 0, SV("1234567890"), 9, 2, S("0abcdefghij"));
    test(S("abcdefghij"), 0, 0, SV("1234567890"), 10, 0, S("abcdefghij"));
    test(S("abcdefghij"), 0, 0, SV("1234567890"), 10, 1, S("abcdefghij"));
    test(S("abcdefghij"), 0, 0, SV("1234567890"), 11, 0, S("can't happen"));
    test(S("abcdefghij"), 0, 0, SV("12345678901234567890"), 0, 0, S("abcdefghij"));
    test(S("abcdefghij"), 0, 0, SV("12345678901234567890"), 0, 1, S("1abcdefghij"));
    test(S("abcdefghij"), 0, 0, SV("12345678901234567890"), 0, 10, S("1234567890abcdefghij"));
    test(S("abcdefghij"), 0, 0, SV("12345678901234567890"), 0, 19, S("1234567890123456789abcdefghij"));
    test(S("abcdefghij"), 0, 0, SV("12345678901234567890"), 0, 20, S("12345678901234567890abcdefghij"));
    test(S("abcdefghij"), 0, 0, SV("12345678901234567890"), 0, 21, S("12345678901234567890abcdefghij"));
    test(S("abcdefghij"), 0, 0, SV("12345678901234567890"), 1, 0, S("abcdefghij"));
    test(S("abcdefghij"), 0, 0, SV("12345678901234567890"), 1, 1, S("2abcdefghij"));
    test(S("abcdefghij"), 0, 0, SV("12345678901234567890"), 1, 9, S("234567890abcdefghij"));
    test(S("abcdefghij"), 0, 0, SV("12345678901234567890"), 1, 18, S("234567890123456789abcdefghij"));
    test(S("abcdefghij"), 0, 0, SV("12345678901234567890"), 1, 19, S("2345678901234567890abcdefghij"));
    test(S("abcdefghij"), 0, 0, SV("12345678901234567890"), 1, 20, S("2345678901234567890abcdefghij"));
    test(S("abcdefghij"), 0, 0, SV("12345678901234567890"), 10, 0, S("abcdefghij"));
    test(S("abcdefghij"), 0, 0, SV("12345678901234567890"), 10, 1, S("1abcdefghij"));
    test(S("abcdefghij"), 0, 0, SV("12345678901234567890"), 10, 5, S("12345abcdefghij"));
    test(S("abcdefghij"), 0, 0, SV("12345678901234567890"), 10, 9, S("123456789abcdefghij"));
    test(S("abcdefghij"), 0, 0, SV("12345678901234567890"), 10, 10, S("1234567890abcdefghij"));
    test(S("abcdefghij"), 0, 0, SV("12345678901234567890"), 10, 11, S("1234567890abcdefghij"));
    test(S("abcdefghij"), 0, 0, SV("12345678901234567890"), 19, 0, S("abcdefghij"));
    test(S("abcdefghij"), 0, 0, SV("12345678901234567890"), 19, 1, S("0abcdefghij"));
    test(S("abcdefghij"), 0, 0, SV("12345678901234567890"), 19, 2, S("0abcdefghij"));
    test(S("abcdefghij"), 0, 0, SV("12345678901234567890"), 20, 0, S("abcdefghij"));
    test(S("abcdefghij"), 0, 0, SV("12345678901234567890"), 20, 1, S("abcdefghij"));
    test(S("abcdefghij"), 0, 0, SV("12345678901234567890"), 21, 0, S("can't happen"));
    test(S("abcdefghij"), 0, 1, SV(""), 0, 0, S("bcdefghij"));
    test(S("abcdefghij"), 0, 1, SV(""), 0, 1, S("bcdefghij"));

    return true;
}

template <class S, class SV>
TEST_CONSTEXPR_CXX20 bool test20()
{
    test(S("abcdefghij"), 0, 1, SV(""), 1, 0, S("can't happen"));
    test(S("abcdefghij"), 0, 1, SV("12345"), 0, 0, S("bcdefghij"));
    test(S("abcdefghij"), 0, 1, SV("12345"), 0, 1, S("1bcdefghij"));
    test(S("abcdefghij"), 0, 1, SV("12345"), 0, 2, S("12bcdefghij"));
    test(S("abcdefghij"), 0, 1, SV("12345"), 0, 4, S("1234bcdefghij"));
    test(S("abcdefghij"), 0, 1, SV("12345"), 0, 5, S("12345bcdefghij"));
    test(S("abcdefghij"), 0, 1, SV("12345"), 0, 6, S("12345bcdefghij"));
    test(S("abcdefghij"), 0, 1, SV("12345"), 1, 0, S("bcdefghij"));
    test(S("abcdefghij"), 0, 1, SV("12345"), 1, 1, S("2bcdefghij"));
    test(S("abcdefghij"), 0, 1, SV("12345"), 1, 2, S("23bcdefghij"));
    test(S("abcdefghij"), 0, 1, SV("12345"), 1, 3, S("234bcdefghij"));
    test(S("abcdefghij"), 0, 1, SV("12345"), 1, 4, S("2345bcdefghij"));
    test(S("abcdefghij"), 0, 1, SV("12345"), 1, 5, S("2345bcdefghij"));
    test(S("abcdefghij"), 0, 1, SV("12345"), 2, 0, S("bcdefghij"));
    test(S("abcdefghij"), 0, 1, SV("12345"), 2, 1, S("3bcdefghij"));
    test(S("abcdefghij"), 0, 1, SV("12345"), 2, 2, S("34bcdefghij"));
    test(S("abcdefghij"), 0, 1, SV("12345"), 2, 3, S("345bcdefghij"));
    test(S("abcdefghij"), 0, 1, SV("12345"), 2, 4, S("345bcdefghij"));
    test(S("abcdefghij"), 0, 1, SV("12345"), 4, 0, S("bcdefghij"));
    test(S("abcdefghij"), 0, 1, SV("12345"), 4, 1, S("5bcdefghij"));
    test(S("abcdefghij"), 0, 1, SV("12345"), 4, 2, S("5bcdefghij"));
    test(S("abcdefghij"), 0, 1, SV("12345"), 5, 0, S("bcdefghij"));
    test(S("abcdefghij"), 0, 1, SV("12345"), 5, 1, S("bcdefghij"));
    test(S("abcdefghij"), 0, 1, SV("12345"), 6, 0, S("can't happen"));
    test(S("abcdefghij"), 0, 1, SV("1234567890"), 0, 0, S("bcdefghij"));
    test(S("abcdefghij"), 0, 1, SV("1234567890"), 0, 1, S("1bcdefghij"));
    test(S("abcdefghij"), 0, 1, SV("1234567890"), 0, 5, S("12345bcdefghij"));
    test(S("abcdefghij"), 0, 1, SV("1234567890"), 0, 9, S("123456789bcdefghij"));
    test(S("abcdefghij"), 0, 1, SV("1234567890"), 0, 10, S("1234567890bcdefghij"));
    test(S("abcdefghij"), 0, 1, SV("1234567890"), 0, 11, S("1234567890bcdefghij"));
    test(S("abcdefghij"), 0, 1, SV("1234567890"), 1, 0, S("bcdefghij"));
    test(S("abcdefghij"), 0, 1, SV("1234567890"), 1, 1, S("2bcdefghij"));
    test(S("abcdefghij"), 0, 1, SV("1234567890"), 1, 4, S("2345bcdefghij"));
    test(S("abcdefghij"), 0, 1, SV("1234567890"), 1, 8, S("23456789bcdefghij"));
    test(S("abcdefghij"), 0, 1, SV("1234567890"), 1, 9, S("234567890bcdefghij"));
    test(S("abcdefghij"), 0, 1, SV("1234567890"), 1, 10, S("234567890bcdefghij"));
    test(S("abcdefghij"), 0, 1, SV("1234567890"), 5, 0, S("bcdefghij"));
    test(S("abcdefghij"), 0, 1, SV("1234567890"), 5, 1, S("6bcdefghij"));
    test(S("abcdefghij"), 0, 1, SV("1234567890"), 5, 2, S("67bcdefghij"));
    test(S("abcdefghij"), 0, 1, SV("1234567890"), 5, 4, S("6789bcdefghij"));
    test(S("abcdefghij"), 0, 1, SV("1234567890"), 5, 5, S("67890bcdefghij"));
    test(S("abcdefghij"), 0, 1, SV("1234567890"), 5, 6, S("67890bcdefghij"));
    test(S("abcdefghij"), 0, 1, SV("1234567890"), 9, 0, S("bcdefghij"));
    test(S("abcdefghij"), 0, 1, SV("1234567890"), 9, 1, S("0bcdefghij"));
    test(S("abcdefghij"), 0, 1, SV("1234567890"), 9, 2, S("0bcdefghij"));
    test(S("abcdefghij"), 0, 1, SV("1234567890"), 10, 0, S("bcdefghij"));
    test(S("abcdefghij"), 0, 1, SV("1234567890"), 10, 1, S("bcdefghij"));
    test(S("abcdefghij"), 0, 1, SV("1234567890"), 11, 0, S("can't happen"));
    test(S("abcdefghij"), 0, 1, SV("12345678901234567890"), 0, 0, S("bcdefghij"));
    test(S("abcdefghij"), 0, 1, SV("12345678901234567890"), 0, 1, S("1bcdefghij"));
    test(S("abcdefghij"), 0, 1, SV("12345678901234567890"), 0, 10, S("1234567890bcdefghij"));
    test(S("abcdefghij"), 0, 1, SV("12345678901234567890"), 0, 19, S("1234567890123456789bcdefghij"));
    test(S("abcdefghij"), 0, 1, SV("12345678901234567890"), 0, 20, S("12345678901234567890bcdefghij"));
    test(S("abcdefghij"), 0, 1, SV("12345678901234567890"), 0, 21, S("12345678901234567890bcdefghij"));
    test(S("abcdefghij"), 0, 1, SV("12345678901234567890"), 1, 0, S("bcdefghij"));
    test(S("abcdefghij"), 0, 1, SV("12345678901234567890"), 1, 1, S("2bcdefghij"));
    test(S("abcdefghij"), 0, 1, SV("12345678901234567890"), 1, 9, S("234567890bcdefghij"));
    test(S("abcdefghij"), 0, 1, SV("12345678901234567890"), 1, 18, S("234567890123456789bcdefghij"));
    test(S("abcdefghij"), 0, 1, SV("12345678901234567890"), 1, 19, S("2345678901234567890bcdefghij"));
    test(S("abcdefghij"), 0, 1, SV("12345678901234567890"), 1, 20, S("2345678901234567890bcdefghij"));
    test(S("abcdefghij"), 0, 1, SV("12345678901234567890"), 10, 0, S("bcdefghij"));
    test(S("abcdefghij"), 0, 1, SV("12345678901234567890"), 10, 1, S("1bcdefghij"));
    test(S("abcdefghij"), 0, 1, SV("12345678901234567890"), 10, 5, S("12345bcdefghij"));
    test(S("abcdefghij"), 0, 1, SV("12345678901234567890"), 10, 9, S("123456789bcdefghij"));
    test(S("abcdefghij"), 0, 1, SV("12345678901234567890"), 10, 10, S("1234567890bcdefghij"));
    test(S("abcdefghij"), 0, 1, SV("12345678901234567890"), 10, 11, S("1234567890bcdefghij"));
    test(S("abcdefghij"), 0, 1, SV("12345678901234567890"), 19, 0, S("bcdefghij"));
    test(S("abcdefghij"), 0, 1, SV("12345678901234567890"), 19, 1, S("0bcdefghij"));
    test(S("abcdefghij"), 0, 1, SV("12345678901234567890"), 19, 2, S("0bcdefghij"));
    test(S("abcdefghij"), 0, 1, SV("12345678901234567890"), 20, 0, S("bcdefghij"));
    test(S("abcdefghij"), 0, 1, SV("12345678901234567890"), 20, 1, S("bcdefghij"));
    test(S("abcdefghij"), 0, 1, SV("12345678901234567890"), 21, 0, S("can't happen"));
    test(S("abcdefghij"), 0, 5, SV(""), 0, 0, S("fghij"));
    test(S("abcdefghij"), 0, 5, SV(""), 0, 1, S("fghij"));
    test(S("abcdefghij"), 0, 5, SV(""), 1, 0, S("can't happen"));
    test(S("abcdefghij"), 0, 5, SV("12345"), 0, 0, S("fghij"));
    test(S("abcdefghij"), 0, 5, SV("12345"), 0, 1, S("1fghij"));
    test(S("abcdefghij"), 0, 5, SV("12345"), 0, 2, S("12fghij"));
    test(S("abcdefghij"), 0, 5, SV("12345"), 0, 4, S("1234fghij"));
    test(S("abcdefghij"), 0, 5, SV("12345"), 0, 5, S("12345fghij"));
    test(S("abcdefghij"), 0, 5, SV("12345"), 0, 6, S("12345fghij"));
    test(S("abcdefghij"), 0, 5, SV("12345"), 1, 0, S("fghij"));
    test(S("abcdefghij"), 0, 5, SV("12345"), 1, 1, S("2fghij"));
    test(S("abcdefghij"), 0, 5, SV("12345"), 1, 2, S("23fghij"));
    test(S("abcdefghij"), 0, 5, SV("12345"), 1, 3, S("234fghij"));
    test(S("abcdefghij"), 0, 5, SV("12345"), 1, 4, S("2345fghij"));
    test(S("abcdefghij"), 0, 5, SV("12345"), 1, 5, S("2345fghij"));
    test(S("abcdefghij"), 0, 5, SV("12345"), 2, 0, S("fghij"));
    test(S("abcdefghij"), 0, 5, SV("12345"), 2, 1, S("3fghij"));
    test(S("abcdefghij"), 0, 5, SV("12345"), 2, 2, S("34fghij"));
    test(S("abcdefghij"), 0, 5, SV("12345"), 2, 3, S("345fghij"));
    test(S("abcdefghij"), 0, 5, SV("12345"), 2, 4, S("345fghij"));
    test(S("abcdefghij"), 0, 5, SV("12345"), 4, 0, S("fghij"));
    test(S("abcdefghij"), 0, 5, SV("12345"), 4, 1, S("5fghij"));
    test(S("abcdefghij"), 0, 5, SV("12345"), 4, 2, S("5fghij"));
    test(S("abcdefghij"), 0, 5, SV("12345"), 5, 0, S("fghij"));
    test(S("abcdefghij"), 0, 5, SV("12345"), 5, 1, S("fghij"));
    test(S("abcdefghij"), 0, 5, SV("12345"), 6, 0, S("can't happen"));
    test(S("abcdefghij"), 0, 5, SV("1234567890"), 0, 0, S("fghij"));
    test(S("abcdefghij"), 0, 5, SV("1234567890"), 0, 1, S("1fghij"));

    return true;
}

template <class S, class SV>
TEST_CONSTEXPR_CXX20 bool test21()
{
    test(S("abcdefghij"), 0, 5, SV("1234567890"), 0, 5, S("12345fghij"));
    test(S("abcdefghij"), 0, 5, SV("1234567890"), 0, 9, S("123456789fghij"));
    test(S("abcdefghij"), 0, 5, SV("1234567890"), 0, 10, S("1234567890fghij"));
    test(S("abcdefghij"), 0, 5, SV("1234567890"), 0, 11, S("1234567890fghij"));
    test(S("abcdefghij"), 0, 5, SV("1234567890"), 1, 0, S("fghij"));
    test(S("abcdefghij"), 0, 5, SV("1234567890"), 1, 1, S("2fghij"));
    test(S("abcdefghij"), 0, 5, SV("1234567890"), 1, 4, S("2345fghij"));
    test(S("abcdefghij"), 0, 5, SV("1234567890"), 1, 8, S("23456789fghij"));
    test(S("abcdefghij"), 0, 5, SV("1234567890"), 1, 9, S("234567890fghij"));
    test(S("abcdefghij"), 0, 5, SV("1234567890"), 1, 10, S("234567890fghij"));
    test(S("abcdefghij"), 0, 5, SV("1234567890"), 5, 0, S("fghij"));
    test(S("abcdefghij"), 0, 5, SV("1234567890"), 5, 1, S("6fghij"));
    test(S("abcdefghij"), 0, 5, SV("1234567890"), 5, 2, S("67fghij"));
    test(S("abcdefghij"), 0, 5, SV("1234567890"), 5, 4, S("6789fghij"));
    test(S("abcdefghij"), 0, 5, SV("1234567890"), 5, 5, S("67890fghij"));
    test(S("abcdefghij"), 0, 5, SV("1234567890"), 5, 6, S("67890fghij"));
    test(S("abcdefghij"), 0, 5, SV("1234567890"), 9, 0, S("fghij"));
    test(S("abcdefghij"), 0, 5, SV("1234567890"), 9, 1, S("0fghij"));
    test(S("abcdefghij"), 0, 5, SV("1234567890"), 9, 2, S("0fghij"));
    test(S("abcdefghij"), 0, 5, SV("1234567890"), 10, 0, S("fghij"));
    test(S("abcdefghij"), 0, 5, SV("1234567890"), 10, 1, S("fghij"));
    test(S("abcdefghij"), 0, 5, SV("1234567890"), 11, 0, S("can't happen"));
    test(S("abcdefghij"), 0, 5, SV("12345678901234567890"), 0, 0, S("fghij"));
    test(S("abcdefghij"), 0, 5, SV("12345678901234567890"), 0, 1, S("1fghij"));
    test(S("abcdefghij"), 0, 5, SV("12345678901234567890"), 0, 10, S("1234567890fghij"));
    test(S("abcdefghij"), 0, 5, SV("12345678901234567890"), 0, 19, S("1234567890123456789fghij"));
    test(S("abcdefghij"), 0, 5, SV("12345678901234567890"), 0, 20, S("12345678901234567890fghij"));
    test(S("abcdefghij"), 0, 5, SV("12345678901234567890"), 0, 21, S("12345678901234567890fghij"));
    test(S("abcdefghij"), 0, 5, SV("12345678901234567890"), 1, 0, S("fghij"));
    test(S("abcdefghij"), 0, 5, SV("12345678901234567890"), 1, 1, S("2fghij"));
    test(S("abcdefghij"), 0, 5, SV("12345678901234567890"), 1, 9, S("234567890fghij"));
    test(S("abcdefghij"), 0, 5, SV("12345678901234567890"), 1, 18, S("234567890123456789fghij"));
    test(S("abcdefghij"), 0, 5, SV("12345678901234567890"), 1, 19, S("2345678901234567890fghij"));
    test(S("abcdefghij"), 0, 5, SV("12345678901234567890"), 1, 20, S("2345678901234567890fghij"));
    test(S("abcdefghij"), 0, 5, SV("12345678901234567890"), 10, 0, S("fghij"));
    test(S("abcdefghij"), 0, 5, SV("12345678901234567890"), 10, 1, S("1fghij"));
    test(S("abcdefghij"), 0, 5, SV("12345678901234567890"), 10, 5, S("12345fghij"));
    test(S("abcdefghij"), 0, 5, SV("12345678901234567890"), 10, 9, S("123456789fghij"));
    test(S("abcdefghij"), 0, 5, SV("12345678901234567890"), 10, 10, S("1234567890fghij"));
    test(S("abcdefghij"), 0, 5, SV("12345678901234567890"), 10, 11, S("1234567890fghij"));
    test(S("abcdefghij"), 0, 5, SV("12345678901234567890"), 19, 0, S("fghij"));
    test(S("abcdefghij"), 0, 5, SV("12345678901234567890"), 19, 1, S("0fghij"));
    test(S("abcdefghij"), 0, 5, SV("12345678901234567890"), 19, 2, S("0fghij"));
    test(S("abcdefghij"), 0, 5, SV("12345678901234567890"), 20, 0, S("fghij"));
    test(S("abcdefghij"), 0, 5, SV("12345678901234567890"), 20, 1, S("fghij"));
    test(S("abcdefghij"), 0, 5, SV("12345678901234567890"), 21, 0, S("can't happen"));
    test(S("abcdefghij"), 0, 9, SV(""), 0, 0, S("j"));
    test(S("abcdefghij"), 0, 9, SV(""), 0, 1, S("j"));
    test(S("abcdefghij"), 0, 9, SV(""), 1, 0, S("can't happen"));
    test(S("abcdefghij"), 0, 9, SV("12345"), 0, 0, S("j"));
    test(S("abcdefghij"), 0, 9, SV("12345"), 0, 1, S("1j"));
    test(S("abcdefghij"), 0, 9, SV("12345"), 0, 2, S("12j"));
    test(S("abcdefghij"), 0, 9, SV("12345"), 0, 4, S("1234j"));
    test(S("abcdefghij"), 0, 9, SV("12345"), 0, 5, S("12345j"));
    test(S("abcdefghij"), 0, 9, SV("12345"), 0, 6, S("12345j"));
    test(S("abcdefghij"), 0, 9, SV("12345"), 1, 0, S("j"));
    test(S("abcdefghij"), 0, 9, SV("12345"), 1, 1, S("2j"));
    test(S("abcdefghij"), 0, 9, SV("12345"), 1, 2, S("23j"));
    test(S("abcdefghij"), 0, 9, SV("12345"), 1, 3, S("234j"));
    test(S("abcdefghij"), 0, 9, SV("12345"), 1, 4, S("2345j"));
    test(S("abcdefghij"), 0, 9, SV("12345"), 1, 5, S("2345j"));
    test(S("abcdefghij"), 0, 9, SV("12345"), 2, 0, S("j"));
    test(S("abcdefghij"), 0, 9, SV("12345"), 2, 1, S("3j"));
    test(S("abcdefghij"), 0, 9, SV("12345"), 2, 2, S("34j"));
    test(S("abcdefghij"), 0, 9, SV("12345"), 2, 3, S("345j"));
    test(S("abcdefghij"), 0, 9, SV("12345"), 2, 4, S("345j"));
    test(S("abcdefghij"), 0, 9, SV("12345"), 4, 0, S("j"));
    test(S("abcdefghij"), 0, 9, SV("12345"), 4, 1, S("5j"));
    test(S("abcdefghij"), 0, 9, SV("12345"), 4, 2, S("5j"));
    test(S("abcdefghij"), 0, 9, SV("12345"), 5, 0, S("j"));
    test(S("abcdefghij"), 0, 9, SV("12345"), 5, 1, S("j"));
    test(S("abcdefghij"), 0, 9, SV("12345"), 6, 0, S("can't happen"));
    test(S("abcdefghij"), 0, 9, SV("1234567890"), 0, 0, S("j"));
    test(S("abcdefghij"), 0, 9, SV("1234567890"), 0, 1, S("1j"));
    test(S("abcdefghij"), 0, 9, SV("1234567890"), 0, 5, S("12345j"));
    test(S("abcdefghij"), 0, 9, SV("1234567890"), 0, 9, S("123456789j"));
    test(S("abcdefghij"), 0, 9, SV("1234567890"), 0, 10, S("1234567890j"));
    test(S("abcdefghij"), 0, 9, SV("1234567890"), 0, 11, S("1234567890j"));
    test(S("abcdefghij"), 0, 9, SV("1234567890"), 1, 0, S("j"));
    test(S("abcdefghij"), 0, 9, SV("1234567890"), 1, 1, S("2j"));
    test(S("abcdefghij"), 0, 9, SV("1234567890"), 1, 4, S("2345j"));
    test(S("abcdefghij"), 0, 9, SV("1234567890"), 1, 8, S("23456789j"));
    test(S("abcdefghij"), 0, 9, SV("1234567890"), 1, 9, S("234567890j"));
    test(S("abcdefghij"), 0, 9, SV("1234567890"), 1, 10, S("234567890j"));
    test(S("abcdefghij"), 0, 9, SV("1234567890"), 5, 0, S("j"));
    test(S("abcdefghij"), 0, 9, SV("1234567890"), 5, 1, S("6j"));
    test(S("abcdefghij"), 0, 9, SV("1234567890"), 5, 2, S("67j"));
    test(S("abcdefghij"), 0, 9, SV("1234567890"), 5, 4, S("6789j"));
    test(S("abcdefghij"), 0, 9, SV("1234567890"), 5, 5, S("67890j"));
    test(S("abcdefghij"), 0, 9, SV("1234567890"), 5, 6, S("67890j"));
    test(S("abcdefghij"), 0, 9, SV("1234567890"), 9, 0, S("j"));
    test(S("abcdefghij"), 0, 9, SV("1234567890"), 9, 1, S("0j"));
    test(S("abcdefghij"), 0, 9, SV("1234567890"), 9, 2, S("0j"));
    test(S("abcdefghij"), 0, 9, SV("1234567890"), 10, 0, S("j"));
    test(S("abcdefghij"), 0, 9, SV("1234567890"), 10, 1, S("j"));
    test(S("abcdefghij"), 0, 9, SV("1234567890"), 11, 0, S("can't happen"));
    test(S("abcdefghij"), 0, 9, SV("12345678901234567890"), 0, 0, S("j"));
    test(S("abcdefghij"), 0, 9, SV("12345678901234567890"), 0, 1, S("1j"));
    test(S("abcdefghij"), 0, 9, SV("12345678901234567890"), 0, 10, S("1234567890j"));
    test(S("abcdefghij"), 0, 9, SV("12345678901234567890"), 0, 19, S("1234567890123456789j"));

    return true;
}

template <class S, class SV>
TEST_CONSTEXPR_CXX20 bool test22()
{
    test(S("abcdefghij"), 0, 9, SV("12345678901234567890"), 0, 20, S("12345678901234567890j"));
    test(S("abcdefghij"), 0, 9, SV("12345678901234567890"), 0, 21, S("12345678901234567890j"));
    test(S("abcdefghij"), 0, 9, SV("12345678901234567890"), 1, 0, S("j"));
    test(S("abcdefghij"), 0, 9, SV("12345678901234567890"), 1, 1, S("2j"));
    test(S("abcdefghij"), 0, 9, SV("12345678901234567890"), 1, 9, S("234567890j"));
    test(S("abcdefghij"), 0, 9, SV("12345678901234567890"), 1, 18, S("234567890123456789j"));
    test(S("abcdefghij"), 0, 9, SV("12345678901234567890"), 1, 19, S("2345678901234567890j"));
    test(S("abcdefghij"), 0, 9, SV("12345678901234567890"), 1, 20, S("2345678901234567890j"));
    test(S("abcdefghij"), 0, 9, SV("12345678901234567890"), 10, 0, S("j"));
    test(S("abcdefghij"), 0, 9, SV("12345678901234567890"), 10, 1, S("1j"));
    test(S("abcdefghij"), 0, 9, SV("12345678901234567890"), 10, 5, S("12345j"));
    test(S("abcdefghij"), 0, 9, SV("12345678901234567890"), 10, 9, S("123456789j"));
    test(S("abcdefghij"), 0, 9, SV("12345678901234567890"), 10, 10, S("1234567890j"));
    test(S("abcdefghij"), 0, 9, SV("12345678901234567890"), 10, 11, S("1234567890j"));
    test(S("abcdefghij"), 0, 9, SV("12345678901234567890"), 19, 0, S("j"));
    test(S("abcdefghij"), 0, 9, SV("12345678901234567890"), 19, 1, S("0j"));
    test(S("abcdefghij"), 0, 9, SV("12345678901234567890"), 19, 2, S("0j"));
    test(S("abcdefghij"), 0, 9, SV("12345678901234567890"), 20, 0, S("j"));
    test(S("abcdefghij"), 0, 9, SV("12345678901234567890"), 20, 1, S("j"));
    test(S("abcdefghij"), 0, 9, SV("12345678901234567890"), 21, 0, S("can't happen"));
    test(S("abcdefghij"), 0, 10, SV(""), 0, 0, S(""));
    test(S("abcdefghij"), 0, 10, SV(""), 0, 1, S(""));
    test(S("abcdefghij"), 0, 10, SV(""), 1, 0, S("can't happen"));
    test(S("abcdefghij"), 0, 10, SV("12345"), 0, 0, S(""));
    test(S("abcdefghij"), 0, 10, SV("12345"), 0, 1, S("1"));
    test(S("abcdefghij"), 0, 10, SV("12345"), 0, 2, S("12"));
    test(S("abcdefghij"), 0, 10, SV("12345"), 0, 4, S("1234"));
    test(S("abcdefghij"), 0, 10, SV("12345"), 0, 5, S("12345"));
    test(S("abcdefghij"), 0, 10, SV("12345"), 0, 6, S("12345"));
    test(S("abcdefghij"), 0, 10, SV("12345"), 1, 0, S(""));
    test(S("abcdefghij"), 0, 10, SV("12345"), 1, 1, S("2"));
    test(S("abcdefghij"), 0, 10, SV("12345"), 1, 2, S("23"));
    test(S("abcdefghij"), 0, 10, SV("12345"), 1, 3, S("234"));
    test(S("abcdefghij"), 0, 10, SV("12345"), 1, 4, S("2345"));
    test(S("abcdefghij"), 0, 10, SV("12345"), 1, 5, S("2345"));
    test(S("abcdefghij"), 0, 10, SV("12345"), 2, 0, S(""));
    test(S("abcdefghij"), 0, 10, SV("12345"), 2, 1, S("3"));
    test(S("abcdefghij"), 0, 10, SV("12345"), 2, 2, S("34"));
    test(S("abcdefghij"), 0, 10, SV("12345"), 2, 3, S("345"));
    test(S("abcdefghij"), 0, 10, SV("12345"), 2, 4, S("345"));
    test(S("abcdefghij"), 0, 10, SV("12345"), 4, 0, S(""));
    test(S("abcdefghij"), 0, 10, SV("12345"), 4, 1, S("5"));
    test(S("abcdefghij"), 0, 10, SV("12345"), 4, 2, S("5"));
    test(S("abcdefghij"), 0, 10, SV("12345"), 5, 0, S(""));
    test(S("abcdefghij"), 0, 10, SV("12345"), 5, 1, S(""));
    test(S("abcdefghij"), 0, 10, SV("12345"), 6, 0, S("can't happen"));
    test(S("abcdefghij"), 0, 10, SV("1234567890"), 0, 0, S(""));
    test(S("abcdefghij"), 0, 10, SV("1234567890"), 0, 1, S("1"));
    test(S("abcdefghij"), 0, 10, SV("1234567890"), 0, 5, S("12345"));
    test(S("abcdefghij"), 0, 10, SV("1234567890"), 0, 9, S("123456789"));
    test(S("abcdefghij"), 0, 10, SV("1234567890"), 0, 10, S("1234567890"));
    test(S("abcdefghij"), 0, 10, SV("1234567890"), 0, 11, S("1234567890"));
    test(S("abcdefghij"), 0, 10, SV("1234567890"), 1, 0, S(""));
    test(S("abcdefghij"), 0, 10, SV("1234567890"), 1, 1, S("2"));
    test(S("abcdefghij"), 0, 10, SV("1234567890"), 1, 4, S("2345"));
    test(S("abcdefghij"), 0, 10, SV("1234567890"), 1, 8, S("23456789"));
    test(S("abcdefghij"), 0, 10, SV("1234567890"), 1, 9, S("234567890"));
    test(S("abcdefghij"), 0, 10, SV("1234567890"), 1, 10, S("234567890"));
    test(S("abcdefghij"), 0, 10, SV("1234567890"), 5, 0, S(""));
    test(S("abcdefghij"), 0, 10, SV("1234567890"), 5, 1, S("6"));
    test(S("abcdefghij"), 0, 10, SV("1234567890"), 5, 2, S("67"));
    test(S("abcdefghij"), 0, 10, SV("1234567890"), 5, 4, S("6789"));
    test(S("abcdefghij"), 0, 10, SV("1234567890"), 5, 5, S("67890"));
    test(S("abcdefghij"), 0, 10, SV("1234567890"), 5, 6, S("67890"));
    test(S("abcdefghij"), 0, 10, SV("1234567890"), 9, 0, S(""));
    test(S("abcdefghij"), 0, 10, SV("1234567890"), 9, 1, S("0"));
    test(S("abcdefghij"), 0, 10, SV("1234567890"), 9, 2, S("0"));
    test(S("abcdefghij"), 0, 10, SV("1234567890"), 10, 0, S(""));
    test(S("abcdefghij"), 0, 10, SV("1234567890"), 10, 1, S(""));
    test(S("abcdefghij"), 0, 10, SV("1234567890"), 11, 0, S("can't happen"));
    test(S("abcdefghij"), 0, 10, SV("12345678901234567890"), 0, 0, S(""));
    test(S("abcdefghij"), 0, 10, SV("12345678901234567890"), 0, 1, S("1"));
    test(S("abcdefghij"), 0, 10, SV("12345678901234567890"), 0, 10, S("1234567890"));
    test(S("abcdefghij"), 0, 10, SV("12345678901234567890"), 0, 19, S("1234567890123456789"));
    test(S("abcdefghij"), 0, 10, SV("12345678901234567890"), 0, 20, S("12345678901234567890"));
    test(S("abcdefghij"), 0, 10, SV("12345678901234567890"), 0, 21, S("12345678901234567890"));
    test(S("abcdefghij"), 0, 10, SV("12345678901234567890"), 1, 0, S(""));
    test(S("abcdefghij"), 0, 10, SV("12345678901234567890"), 1, 1, S("2"));
    test(S("abcdefghij"), 0, 10, SV("12345678901234567890"), 1, 9, S("234567890"));
    test(S("abcdefghij"), 0, 10, SV("12345678901234567890"), 1, 18, S("234567890123456789"));
    test(S("abcdefghij"), 0, 10, SV("12345678901234567890"), 1, 19, S("2345678901234567890"));
    test(S("abcdefghij"), 0, 10, SV("12345678901234567890"), 1, 20, S("2345678901234567890"));
    test(S("abcdefghij"), 0, 10, SV("12345678901234567890"), 10, 0, S(""));
    test(S("abcdefghij"), 0, 10, SV("12345678901234567890"), 10, 1, S("1"));
    test(S("abcdefghij"), 0, 10, SV("12345678901234567890"), 10, 5, S("12345"));
    test(S("abcdefghij"), 0, 10, SV("12345678901234567890"), 10, 9, S("123456789"));
    test(S("abcdefghij"), 0, 10, SV("12345678901234567890"), 10, 10, S("1234567890"));
    test(S("abcdefghij"), 0, 10, SV("12345678901234567890"), 10, 11, S("1234567890"));
    test(S("abcdefghij"), 0, 10, SV("12345678901234567890"), 19, 0, S(""));
    test(S("abcdefghij"), 0, 10, SV("12345678901234567890"), 19, 1, S("0"));
    test(S("abcdefghij"), 0, 10, SV("12345678901234567890"), 19, 2, S("0"));
    test(S("abcdefghij"), 0, 10, SV("12345678901234567890"), 20, 0, S(""));
    test(S("abcdefghij"), 0, 10, SV("12345678901234567890"), 20, 1, S(""));
    test(S("abcdefghij"), 0, 10, SV("12345678901234567890"), 21, 0, S("can't happen"));
    test(S("abcdefghij"), 0, 11, SV(""), 0, 0, S(""));
    test(S("abcdefghij"), 0, 11, SV(""), 0, 1, S(""));
    test(S("abcdefghij"), 0, 11, SV(""), 1, 0, S("can't happen"));
    test(S("abcdefghij"), 0, 11, SV("12345"), 0, 0, S(""));
    test(S("abcdefghij"), 0, 11, SV("12345"), 0, 1, S("1"));
    test(S("abcdefghij"), 0, 11, SV("12345"), 0, 2, S("12"));

    return true;
}

template <class S, class SV>
TEST_CONSTEXPR_CXX20 bool test23()
{
    test(S("abcdefghij"), 0, 11, SV("12345"), 0, 4, S("1234"));
    test(S("abcdefghij"), 0, 11, SV("12345"), 0, 5, S("12345"));
    test(S("abcdefghij"), 0, 11, SV("12345"), 0, 6, S("12345"));
    test(S("abcdefghij"), 0, 11, SV("12345"), 1, 0, S(""));
    test(S("abcdefghij"), 0, 11, SV("12345"), 1, 1, S("2"));
    test(S("abcdefghij"), 0, 11, SV("12345"), 1, 2, S("23"));
    test(S("abcdefghij"), 0, 11, SV("12345"), 1, 3, S("234"));
    test(S("abcdefghij"), 0, 11, SV("12345"), 1, 4, S("2345"));
    test(S("abcdefghij"), 0, 11, SV("12345"), 1, 5, S("2345"));
    test(S("abcdefghij"), 0, 11, SV("12345"), 2, 0, S(""));
    test(S("abcdefghij"), 0, 11, SV("12345"), 2, 1, S("3"));
    test(S("abcdefghij"), 0, 11, SV("12345"), 2, 2, S("34"));
    test(S("abcdefghij"), 0, 11, SV("12345"), 2, 3, S("345"));
    test(S("abcdefghij"), 0, 11, SV("12345"), 2, 4, S("345"));
    test(S("abcdefghij"), 0, 11, SV("12345"), 4, 0, S(""));
    test(S("abcdefghij"), 0, 11, SV("12345"), 4, 1, S("5"));
    test(S("abcdefghij"), 0, 11, SV("12345"), 4, 2, S("5"));
    test(S("abcdefghij"), 0, 11, SV("12345"), 5, 0, S(""));
    test(S("abcdefghij"), 0, 11, SV("12345"), 5, 1, S(""));
    test(S("abcdefghij"), 0, 11, SV("12345"), 6, 0, S("can't happen"));
    test(S("abcdefghij"), 0, 11, SV("1234567890"), 0, 0, S(""));
    test(S("abcdefghij"), 0, 11, SV("1234567890"), 0, 1, S("1"));
    test(S("abcdefghij"), 0, 11, SV("1234567890"), 0, 5, S("12345"));
    test(S("abcdefghij"), 0, 11, SV("1234567890"), 0, 9, S("123456789"));
    test(S("abcdefghij"), 0, 11, SV("1234567890"), 0, 10, S("1234567890"));
    test(S("abcdefghij"), 0, 11, SV("1234567890"), 0, 11, S("1234567890"));
    test(S("abcdefghij"), 0, 11, SV("1234567890"), 1, 0, S(""));
    test(S("abcdefghij"), 0, 11, SV("1234567890"), 1, 1, S("2"));
    test(S("abcdefghij"), 0, 11, SV("1234567890"), 1, 4, S("2345"));
    test(S("abcdefghij"), 0, 11, SV("1234567890"), 1, 8, S("23456789"));
    test(S("abcdefghij"), 0, 11, SV("1234567890"), 1, 9, S("234567890"));
    test(S("abcdefghij"), 0, 11, SV("1234567890"), 1, 10, S("234567890"));
    test(S("abcdefghij"), 0, 11, SV("1234567890"), 5, 0, S(""));
    test(S("abcdefghij"), 0, 11, SV("1234567890"), 5, 1, S("6"));
    test(S("abcdefghij"), 0, 11, SV("1234567890"), 5, 2, S("67"));
    test(S("abcdefghij"), 0, 11, SV("1234567890"), 5, 4, S("6789"));
    test(S("abcdefghij"), 0, 11, SV("1234567890"), 5, 5, S("67890"));
    test(S("abcdefghij"), 0, 11, SV("1234567890"), 5, 6, S("67890"));
    test(S("abcdefghij"), 0, 11, SV("1234567890"), 9, 0, S(""));
    test(S("abcdefghij"), 0, 11, SV("1234567890"), 9, 1, S("0"));
    test(S("abcdefghij"), 0, 11, SV("1234567890"), 9, 2, S("0"));
    test(S("abcdefghij"), 0, 11, SV("1234567890"), 10, 0, S(""));
    test(S("abcdefghij"), 0, 11, SV("1234567890"), 10, 1, S(""));
    test(S("abcdefghij"), 0, 11, SV("1234567890"), 11, 0, S("can't happen"));
    test(S("abcdefghij"), 0, 11, SV("12345678901234567890"), 0, 0, S(""));
    test(S("abcdefghij"), 0, 11, SV("12345678901234567890"), 0, 1, S("1"));
    test(S("abcdefghij"), 0, 11, SV("12345678901234567890"), 0, 10, S("1234567890"));
    test(S("abcdefghij"), 0, 11, SV("12345678901234567890"), 0, 19, S("1234567890123456789"));
    test(S("abcdefghij"), 0, 11, SV("12345678901234567890"), 0, 20, S("12345678901234567890"));
    test(S("abcdefghij"), 0, 11, SV("12345678901234567890"), 0, 21, S("12345678901234567890"));
    test(S("abcdefghij"), 0, 11, SV("12345678901234567890"), 1, 0, S(""));
    test(S("abcdefghij"), 0, 11, SV("12345678901234567890"), 1, 1, S("2"));
    test(S("abcdefghij"), 0, 11, SV("12345678901234567890"), 1, 9, S("234567890"));
    test(S("abcdefghij"), 0, 11, SV("12345678901234567890"), 1, 18, S("234567890123456789"));
    test(S("abcdefghij"), 0, 11, SV("12345678901234567890"), 1, 19, S("2345678901234567890"));
    test(S("abcdefghij"), 0, 11, SV("12345678901234567890"), 1, 20, S("2345678901234567890"));
    test(S("abcdefghij"), 0, 11, SV("12345678901234567890"), 10, 0, S(""));
    test(S("abcdefghij"), 0, 11, SV("12345678901234567890"), 10, 1, S("1"));
    test(S("abcdefghij"), 0, 11, SV("12345678901234567890"), 10, 5, S("12345"));
    test(S("abcdefghij"), 0, 11, SV("12345678901234567890"), 10, 9, S("123456789"));
    test(S("abcdefghij"), 0, 11, SV("12345678901234567890"), 10, 10, S("1234567890"));
    test(S("abcdefghij"), 0, 11, SV("12345678901234567890"), 10, 11, S("1234567890"));
    test(S("abcdefghij"), 0, 11, SV("12345678901234567890"), 19, 0, S(""));
    test(S("abcdefghij"), 0, 11, SV("12345678901234567890"), 19, 1, S("0"));
    test(S("abcdefghij"), 0, 11, SV("12345678901234567890"), 19, 2, S("0"));
    test(S("abcdefghij"), 0, 11, SV("12345678901234567890"), 20, 0, S(""));
    test(S("abcdefghij"), 0, 11, SV("12345678901234567890"), 20, 1, S(""));
    test(S("abcdefghij"), 0, 11, SV("12345678901234567890"), 21, 0, S("can't happen"));
    test(S("abcdefghij"), 1, 0, SV(""), 0, 0, S("abcdefghij"));
    test(S("abcdefghij"), 1, 0, SV(""), 0, 1, S("abcdefghij"));
    test(S("abcdefghij"), 1, 0, SV(""), 1, 0, S("can't happen"));
    test(S("abcdefghij"), 1, 0, SV("12345"), 0, 0, S("abcdefghij"));
    test(S("abcdefghij"), 1, 0, SV("12345"), 0, 1, S("a1bcdefghij"));
    test(S("abcdefghij"), 1, 0, SV("12345"), 0, 2, S("a12bcdefghij"));
    test(S("abcdefghij"), 1, 0, SV("12345"), 0, 4, S("a1234bcdefghij"));
    test(S("abcdefghij"), 1, 0, SV("12345"), 0, 5, S("a12345bcdefghij"));
    test(S("abcdefghij"), 1, 0, SV("12345"), 0, 6, S("a12345bcdefghij"));
    test(S("abcdefghij"), 1, 0, SV("12345"), 1, 0, S("abcdefghij"));
    test(S("abcdefghij"), 1, 0, SV("12345"), 1, 1, S("a2bcdefghij"));
    test(S("abcdefghij"), 1, 0, SV("12345"), 1, 2, S("a23bcdefghij"));
    test(S("abcdefghij"), 1, 0, SV("12345"), 1, 3, S("a234bcdefghij"));
    test(S("abcdefghij"), 1, 0, SV("12345"), 1, 4, S("a2345bcdefghij"));
    test(S("abcdefghij"), 1, 0, SV("12345"), 1, 5, S("a2345bcdefghij"));
    test(S("abcdefghij"), 1, 0, SV("12345"), 2, 0, S("abcdefghij"));
    test(S("abcdefghij"), 1, 0, SV("12345"), 2, 1, S("a3bcdefghij"));
    test(S("abcdefghij"), 1, 0, SV("12345"), 2, 2, S("a34bcdefghij"));
    test(S("abcdefghij"), 1, 0, SV("12345"), 2, 3, S("a345bcdefghij"));
    test(S("abcdefghij"), 1, 0, SV("12345"), 2, 4, S("a345bcdefghij"));
    test(S("abcdefghij"), 1, 0, SV("12345"), 4, 0, S("abcdefghij"));
    test(S("abcdefghij"), 1, 0, SV("12345"), 4, 1, S("a5bcdefghij"));
    test(S("abcdefghij"), 1, 0, SV("12345"), 4, 2, S("a5bcdefghij"));
    test(S("abcdefghij"), 1, 0, SV("12345"), 5, 0, S("abcdefghij"));
    test(S("abcdefghij"), 1, 0, SV("12345"), 5, 1, S("abcdefghij"));
    test(S("abcdefghij"), 1, 0, SV("12345"), 6, 0, S("can't happen"));
    test(S("abcdefghij"), 1, 0, SV("1234567890"), 0, 0, S("abcdefghij"));
    test(S("abcdefghij"), 1, 0, SV("1234567890"), 0, 1, S("a1bcdefghij"));
    test(S("abcdefghij"), 1, 0, SV("1234567890"), 0, 5, S("a12345bcdefghij"));
    test(S("abcdefghij"), 1, 0, SV("1234567890"), 0, 9, S("a123456789bcdefghij"));
    test(S("abcdefghij"), 1, 0, SV("1234567890"), 0, 10, S("a1234567890bcdefghij"));
    test(S("abcdefghij"), 1, 0, SV("1234567890"), 0, 11, S("a1234567890bcdefghij"));

    return true;
}

template <class S, class SV>
TEST_CONSTEXPR_CXX20 bool test24()
{
    test(S("abcdefghij"), 1, 0, SV("1234567890"), 1, 0, S("abcdefghij"));
    test(S("abcdefghij"), 1, 0, SV("1234567890"), 1, 1, S("a2bcdefghij"));
    test(S("abcdefghij"), 1, 0, SV("1234567890"), 1, 4, S("a2345bcdefghij"));
    test(S("abcdefghij"), 1, 0, SV("1234567890"), 1, 8, S("a23456789bcdefghij"));
    test(S("abcdefghij"), 1, 0, SV("1234567890"), 1, 9, S("a234567890bcdefghij"));
    test(S("abcdefghij"), 1, 0, SV("1234567890"), 1, 10, S("a234567890bcdefghij"));
    test(S("abcdefghij"), 1, 0, SV("1234567890"), 5, 0, S("abcdefghij"));
    test(S("abcdefghij"), 1, 0, SV("1234567890"), 5, 1, S("a6bcdefghij"));
    test(S("abcdefghij"), 1, 0, SV("1234567890"), 5, 2, S("a67bcdefghij"));
    test(S("abcdefghij"), 1, 0, SV("1234567890"), 5, 4, S("a6789bcdefghij"));
    test(S("abcdefghij"), 1, 0, SV("1234567890"), 5, 5, S("a67890bcdefghij"));
    test(S("abcdefghij"), 1, 0, SV("1234567890"), 5, 6, S("a67890bcdefghij"));
    test(S("abcdefghij"), 1, 0, SV("1234567890"), 9, 0, S("abcdefghij"));
    test(S("abcdefghij"), 1, 0, SV("1234567890"), 9, 1, S("a0bcdefghij"));
    test(S("abcdefghij"), 1, 0, SV("1234567890"), 9, 2, S("a0bcdefghij"));
    test(S("abcdefghij"), 1, 0, SV("1234567890"), 10, 0, S("abcdefghij"));
    test(S("abcdefghij"), 1, 0, SV("1234567890"), 10, 1, S("abcdefghij"));
    test(S("abcdefghij"), 1, 0, SV("1234567890"), 11, 0, S("can't happen"));
    test(S("abcdefghij"), 1, 0, SV("12345678901234567890"), 0, 0, S("abcdefghij"));
    test(S("abcdefghij"), 1, 0, SV("12345678901234567890"), 0, 1, S("a1bcdefghij"));
    test(S("abcdefghij"), 1, 0, SV("12345678901234567890"), 0, 10, S("a1234567890bcdefghij"));
    test(S("abcdefghij"), 1, 0, SV("12345678901234567890"), 0, 19, S("a1234567890123456789bcdefghij"));
    test(S("abcdefghij"), 1, 0, SV("12345678901234567890"), 0, 20, S("a12345678901234567890bcdefghij"));
    test(S("abcdefghij"), 1, 0, SV("12345678901234567890"), 0, 21, S("a12345678901234567890bcdefghij"));
    test(S("abcdefghij"), 1, 0, SV("12345678901234567890"), 1, 0, S("abcdefghij"));
    test(S("abcdefghij"), 1, 0, SV("12345678901234567890"), 1, 1, S("a2bcdefghij"));
    test(S("abcdefghij"), 1, 0, SV("12345678901234567890"), 1, 9, S("a234567890bcdefghij"));
    test(S("abcdefghij"), 1, 0, SV("12345678901234567890"), 1, 18, S("a234567890123456789bcdefghij"));
    test(S("abcdefghij"), 1, 0, SV("12345678901234567890"), 1, 19, S("a2345678901234567890bcdefghij"));
    test(S("abcdefghij"), 1, 0, SV("12345678901234567890"), 1, 20, S("a2345678901234567890bcdefghij"));
    test(S("abcdefghij"), 1, 0, SV("12345678901234567890"), 10, 0, S("abcdefghij"));
    test(S("abcdefghij"), 1, 0, SV("12345678901234567890"), 10, 1, S("a1bcdefghij"));
    test(S("abcdefghij"), 1, 0, SV("12345678901234567890"), 10, 5, S("a12345bcdefghij"));
    test(S("abcdefghij"), 1, 0, SV("12345678901234567890"), 10, 9, S("a123456789bcdefghij"));
    test(S("abcdefghij"), 1, 0, SV("12345678901234567890"), 10, 10, S("a1234567890bcdefghij"));
    test(S("abcdefghij"), 1, 0, SV("12345678901234567890"), 10, 11, S("a1234567890bcdefghij"));
    test(S("abcdefghij"), 1, 0, SV("12345678901234567890"), 19, 0, S("abcdefghij"));
    test(S("abcdefghij"), 1, 0, SV("12345678901234567890"), 19, 1, S("a0bcdefghij"));
    test(S("abcdefghij"), 1, 0, SV("12345678901234567890"), 19, 2, S("a0bcdefghij"));
    test(S("abcdefghij"), 1, 0, SV("12345678901234567890"), 20, 0, S("abcdefghij"));
    test(S("abcdefghij"), 1, 0, SV("12345678901234567890"), 20, 1, S("abcdefghij"));
    test(S("abcdefghij"), 1, 0, SV("12345678901234567890"), 21, 0, S("can't happen"));
    test(S("abcdefghij"), 1, 1, SV(""), 0, 0, S("acdefghij"));
    test(S("abcdefghij"), 1, 1, SV(""), 0, 1, S("acdefghij"));
    test(S("abcdefghij"), 1, 1, SV(""), 1, 0, S("can't happen"));
    test(S("abcdefghij"), 1, 1, SV("12345"), 0, 0, S("acdefghij"));
    test(S("abcdefghij"), 1, 1, SV("12345"), 0, 1, S("a1cdefghij"));
    test(S("abcdefghij"), 1, 1, SV("12345"), 0, 2, S("a12cdefghij"));
    test(S("abcdefghij"), 1, 1, SV("12345"), 0, 4, S("a1234cdefghij"));
    test(S("abcdefghij"), 1, 1, SV("12345"), 0, 5, S("a12345cdefghij"));
    test(S("abcdefghij"), 1, 1, SV("12345"), 0, 6, S("a12345cdefghij"));
    test(S("abcdefghij"), 1, 1, SV("12345"), 1, 0, S("acdefghij"));
    test(S("abcdefghij"), 1, 1, SV("12345"), 1, 1, S("a2cdefghij"));
    test(S("abcdefghij"), 1, 1, SV("12345"), 1, 2, S("a23cdefghij"));
    test(S("abcdefghij"), 1, 1, SV("12345"), 1, 3, S("a234cdefghij"));
    test(S("abcdefghij"), 1, 1, SV("12345"), 1, 4, S("a2345cdefghij"));
    test(S("abcdefghij"), 1, 1, SV("12345"), 1, 5, S("a2345cdefghij"));
    test(S("abcdefghij"), 1, 1, SV("12345"), 2, 0, S("acdefghij"));
    test(S("abcdefghij"), 1, 1, SV("12345"), 2, 1, S("a3cdefghij"));
    test(S("abcdefghij"), 1, 1, SV("12345"), 2, 2, S("a34cdefghij"));
    test(S("abcdefghij"), 1, 1, SV("12345"), 2, 3, S("a345cdefghij"));
    test(S("abcdefghij"), 1, 1, SV("12345"), 2, 4, S("a345cdefghij"));
    test(S("abcdefghij"), 1, 1, SV("12345"), 4, 0, S("acdefghij"));
    test(S("abcdefghij"), 1, 1, SV("12345"), 4, 1, S("a5cdefghij"));
    test(S("abcdefghij"), 1, 1, SV("12345"), 4, 2, S("a5cdefghij"));
    test(S("abcdefghij"), 1, 1, SV("12345"), 5, 0, S("acdefghij"));
    test(S("abcdefghij"), 1, 1, SV("12345"), 5, 1, S("acdefghij"));
    test(S("abcdefghij"), 1, 1, SV("12345"), 6, 0, S("can't happen"));
    test(S("abcdefghij"), 1, 1, SV("1234567890"), 0, 0, S("acdefghij"));
    test(S("abcdefghij"), 1, 1, SV("1234567890"), 0, 1, S("a1cdefghij"));
    test(S("abcdefghij"), 1, 1, SV("1234567890"), 0, 5, S("a12345cdefghij"));
    test(S("abcdefghij"), 1, 1, SV("1234567890"), 0, 9, S("a123456789cdefghij"));
    test(S("abcdefghij"), 1, 1, SV("1234567890"), 0, 10, S("a1234567890cdefghij"));
    test(S("abcdefghij"), 1, 1, SV("1234567890"), 0, 11, S("a1234567890cdefghij"));
    test(S("abcdefghij"), 1, 1, SV("1234567890"), 1, 0, S("acdefghij"));
    test(S("abcdefghij"), 1, 1, SV("1234567890"), 1, 1, S("a2cdefghij"));
    test(S("abcdefghij"), 1, 1, SV("1234567890"), 1, 4, S("a2345cdefghij"));
    test(S("abcdefghij"), 1, 1, SV("1234567890"), 1, 8, S("a23456789cdefghij"));
    test(S("abcdefghij"), 1, 1, SV("1234567890"), 1, 9, S("a234567890cdefghij"));
    test(S("abcdefghij"), 1, 1, SV("1234567890"), 1, 10, S("a234567890cdefghij"));
    test(S("abcdefghij"), 1, 1, SV("1234567890"), 5, 0, S("acdefghij"));
    test(S("abcdefghij"), 1, 1, SV("1234567890"), 5, 1, S("a6cdefghij"));
    test(S("abcdefghij"), 1, 1, SV("1234567890"), 5, 2, S("a67cdefghij"));
    test(S("abcdefghij"), 1, 1, SV("1234567890"), 5, 4, S("a6789cdefghij"));
    test(S("abcdefghij"), 1, 1, SV("1234567890"), 5, 5, S("a67890cdefghij"));
    test(S("abcdefghij"), 1, 1, SV("1234567890"), 5, 6, S("a67890cdefghij"));
    test(S("abcdefghij"), 1, 1, SV("1234567890"), 9, 0, S("acdefghij"));
    test(S("abcdefghij"), 1, 1, SV("1234567890"), 9, 1, S("a0cdefghij"));
    test(S("abcdefghij"), 1, 1, SV("1234567890"), 9, 2, S("a0cdefghij"));
    test(S("abcdefghij"), 1, 1, SV("1234567890"), 10, 0, S("acdefghij"));
    test(S("abcdefghij"), 1, 1, SV("1234567890"), 10, 1, S("acdefghij"));
    test(S("abcdefghij"), 1, 1, SV("1234567890"), 11, 0, S("can't happen"));
    test(S("abcdefghij"), 1, 1, SV("12345678901234567890"), 0, 0, S("acdefghij"));
    test(S("abcdefghij"), 1, 1, SV("12345678901234567890"), 0, 1, S("a1cdefghij"));
    test(S("abcdefghij"), 1, 1, SV("12345678901234567890"), 0, 10, S("a1234567890cdefghij"));
    test(S("abcdefghij"), 1, 1, SV("12345678901234567890"), 0, 19, S("a1234567890123456789cdefghij"));
    test(S("abcdefghij"), 1, 1, SV("12345678901234567890"), 0, 20, S("a12345678901234567890cdefghij"));
    test(S("abcdefghij"), 1, 1, SV("12345678901234567890"), 0, 21, S("a12345678901234567890cdefghij"));
    test(S("abcdefghij"), 1, 1, SV("12345678901234567890"), 1, 0, S("acdefghij"));
    test(S("abcdefghij"), 1, 1, SV("12345678901234567890"), 1, 1, S("a2cdefghij"));

    return true;
}

template <class S, class SV>
TEST_CONSTEXPR_CXX20 bool test25()
{
    test(S("abcdefghij"), 1, 1, SV("12345678901234567890"), 1, 9, S("a234567890cdefghij"));
    test(S("abcdefghij"), 1, 1, SV("12345678901234567890"), 1, 18, S("a234567890123456789cdefghij"));
    test(S("abcdefghij"), 1, 1, SV("12345678901234567890"), 1, 19, S("a2345678901234567890cdefghij"));
    test(S("abcdefghij"), 1, 1, SV("12345678901234567890"), 1, 20, S("a2345678901234567890cdefghij"));
    test(S("abcdefghij"), 1, 1, SV("12345678901234567890"), 10, 0, S("acdefghij"));
    test(S("abcdefghij"), 1, 1, SV("12345678901234567890"), 10, 1, S("a1cdefghij"));
    test(S("abcdefghij"), 1, 1, SV("12345678901234567890"), 10, 5, S("a12345cdefghij"));
    test(S("abcdefghij"), 1, 1, SV("12345678901234567890"), 10, 9, S("a123456789cdefghij"));
    test(S("abcdefghij"), 1, 1, SV("12345678901234567890"), 10, 10, S("a1234567890cdefghij"));
    test(S("abcdefghij"), 1, 1, SV("12345678901234567890"), 10, 11, S("a1234567890cdefghij"));
    test(S("abcdefghij"), 1, 1, SV("12345678901234567890"), 19, 0, S("acdefghij"));
    test(S("abcdefghij"), 1, 1, SV("12345678901234567890"), 19, 1, S("a0cdefghij"));
    test(S("abcdefghij"), 1, 1, SV("12345678901234567890"), 19, 2, S("a0cdefghij"));
    test(S("abcdefghij"), 1, 1, SV("12345678901234567890"), 20, 0, S("acdefghij"));
    test(S("abcdefghij"), 1, 1, SV("12345678901234567890"), 20, 1, S("acdefghij"));
    test(S("abcdefghij"), 1, 1, SV("12345678901234567890"), 21, 0, S("can't happen"));
    test(S("abcdefghij"), 1, 4, SV(""), 0, 0, S("afghij"));
    test(S("abcdefghij"), 1, 4, SV(""), 0, 1, S("afghij"));
    test(S("abcdefghij"), 1, 4, SV(""), 1, 0, S("can't happen"));
    test(S("abcdefghij"), 1, 4, SV("12345"), 0, 0, S("afghij"));
    test(S("abcdefghij"), 1, 4, SV("12345"), 0, 1, S("a1fghij"));
    test(S("abcdefghij"), 1, 4, SV("12345"), 0, 2, S("a12fghij"));
    test(S("abcdefghij"), 1, 4, SV("12345"), 0, 4, S("a1234fghij"));
    test(S("abcdefghij"), 1, 4, SV("12345"), 0, 5, S("a12345fghij"));
    test(S("abcdefghij"), 1, 4, SV("12345"), 0, 6, S("a12345fghij"));
    test(S("abcdefghij"), 1, 4, SV("12345"), 1, 0, S("afghij"));
    test(S("abcdefghij"), 1, 4, SV("12345"), 1, 1, S("a2fghij"));
    test(S("abcdefghij"), 1, 4, SV("12345"), 1, 2, S("a23fghij"));
    test(S("abcdefghij"), 1, 4, SV("12345"), 1, 3, S("a234fghij"));
    test(S("abcdefghij"), 1, 4, SV("12345"), 1, 4, S("a2345fghij"));
    test(S("abcdefghij"), 1, 4, SV("12345"), 1, 5, S("a2345fghij"));
    test(S("abcdefghij"), 1, 4, SV("12345"), 2, 0, S("afghij"));
    test(S("abcdefghij"), 1, 4, SV("12345"), 2, 1, S("a3fghij"));
    test(S("abcdefghij"), 1, 4, SV("12345"), 2, 2, S("a34fghij"));
    test(S("abcdefghij"), 1, 4, SV("12345"), 2, 3, S("a345fghij"));
    test(S("abcdefghij"), 1, 4, SV("12345"), 2, 4, S("a345fghij"));
    test(S("abcdefghij"), 1, 4, SV("12345"), 4, 0, S("afghij"));
    test(S("abcdefghij"), 1, 4, SV("12345"), 4, 1, S("a5fghij"));
    test(S("abcdefghij"), 1, 4, SV("12345"), 4, 2, S("a5fghij"));
    test(S("abcdefghij"), 1, 4, SV("12345"), 5, 0, S("afghij"));
    test(S("abcdefghij"), 1, 4, SV("12345"), 5, 1, S("afghij"));
    test(S("abcdefghij"), 1, 4, SV("12345"), 6, 0, S("can't happen"));
    test(S("abcdefghij"), 1, 4, SV("1234567890"), 0, 0, S("afghij"));
    test(S("abcdefghij"), 1, 4, SV("1234567890"), 0, 1, S("a1fghij"));
    test(S("abcdefghij"), 1, 4, SV("1234567890"), 0, 5, S("a12345fghij"));
    test(S("abcdefghij"), 1, 4, SV("1234567890"), 0, 9, S("a123456789fghij"));
    test(S("abcdefghij"), 1, 4, SV("1234567890"), 0, 10, S("a1234567890fghij"));
    test(S("abcdefghij"), 1, 4, SV("1234567890"), 0, 11, S("a1234567890fghij"));
    test(S("abcdefghij"), 1, 4, SV("1234567890"), 1, 0, S("afghij"));
    test(S("abcdefghij"), 1, 4, SV("1234567890"), 1, 1, S("a2fghij"));
    test(S("abcdefghij"), 1, 4, SV("1234567890"), 1, 4, S("a2345fghij"));
    test(S("abcdefghij"), 1, 4, SV("1234567890"), 1, 8, S("a23456789fghij"));
    test(S("abcdefghij"), 1, 4, SV("1234567890"), 1, 9, S("a234567890fghij"));
    test(S("abcdefghij"), 1, 4, SV("1234567890"), 1, 10, S("a234567890fghij"));
    test(S("abcdefghij"), 1, 4, SV("1234567890"), 5, 0, S("afghij"));
    test(S("abcdefghij"), 1, 4, SV("1234567890"), 5, 1, S("a6fghij"));
    test(S("abcdefghij"), 1, 4, SV("1234567890"), 5, 2, S("a67fghij"));
    test(S("abcdefghij"), 1, 4, SV("1234567890"), 5, 4, S("a6789fghij"));
    test(S("abcdefghij"), 1, 4, SV("1234567890"), 5, 5, S("a67890fghij"));
    test(S("abcdefghij"), 1, 4, SV("1234567890"), 5, 6, S("a67890fghij"));
    test(S("abcdefghij"), 1, 4, SV("1234567890"), 9, 0, S("afghij"));
    test(S("abcdefghij"), 1, 4, SV("1234567890"), 9, 1, S("a0fghij"));
    test(S("abcdefghij"), 1, 4, SV("1234567890"), 9, 2, S("a0fghij"));
    test(S("abcdefghij"), 1, 4, SV("1234567890"), 10, 0, S("afghij"));
    test(S("abcdefghij"), 1, 4, SV("1234567890"), 10, 1, S("afghij"));
    test(S("abcdefghij"), 1, 4, SV("1234567890"), 11, 0, S("can't happen"));
    test(S("abcdefghij"), 1, 4, SV("12345678901234567890"), 0, 0, S("afghij"));
    test(S("abcdefghij"), 1, 4, SV("12345678901234567890"), 0, 1, S("a1fghij"));
    test(S("abcdefghij"), 1, 4, SV("12345678901234567890"), 0, 10, S("a1234567890fghij"));
    test(S("abcdefghij"), 1, 4, SV("12345678901234567890"), 0, 19, S("a1234567890123456789fghij"));
    test(S("abcdefghij"), 1, 4, SV("12345678901234567890"), 0, 20, S("a12345678901234567890fghij"));
    test(S("abcdefghij"), 1, 4, SV("12345678901234567890"), 0, 21, S("a12345678901234567890fghij"));
    test(S("abcdefghij"), 1, 4, SV("12345678901234567890"), 1, 0, S("afghij"));
    test(S("abcdefghij"), 1, 4, SV("12345678901234567890"), 1, 1, S("a2fghij"));
    test(S("abcdefghij"), 1, 4, SV("12345678901234567890"), 1, 9, S("a234567890fghij"));
    test(S("abcdefghij"), 1, 4, SV("12345678901234567890"), 1, 18, S("a234567890123456789fghij"));
    test(S("abcdefghij"), 1, 4, SV("12345678901234567890"), 1, 19, S("a2345678901234567890fghij"));
    test(S("abcdefghij"), 1, 4, SV("12345678901234567890"), 1, 20, S("a2345678901234567890fghij"));
    test(S("abcdefghij"), 1, 4, SV("12345678901234567890"), 10, 0, S("afghij"));
    test(S("abcdefghij"), 1, 4, SV("12345678901234567890"), 10, 1, S("a1fghij"));
    test(S("abcdefghij"), 1, 4, SV("12345678901234567890"), 10, 5, S("a12345fghij"));
    test(S("abcdefghij"), 1, 4, SV("12345678901234567890"), 10, 9, S("a123456789fghij"));
    test(S("abcdefghij"), 1, 4, SV("12345678901234567890"), 10, 10, S("a1234567890fghij"));
    test(S("abcdefghij"), 1, 4, SV("12345678901234567890"), 10, 11, S("a1234567890fghij"));
    test(S("abcdefghij"), 1, 4, SV("12345678901234567890"), 19, 0, S("afghij"));
    test(S("abcdefghij"), 1, 4, SV("12345678901234567890"), 19, 1, S("a0fghij"));
    test(S("abcdefghij"), 1, 4, SV("12345678901234567890"), 19, 2, S("a0fghij"));
    test(S("abcdefghij"), 1, 4, SV("12345678901234567890"), 20, 0, S("afghij"));
    test(S("abcdefghij"), 1, 4, SV("12345678901234567890"), 20, 1, S("afghij"));
    test(S("abcdefghij"), 1, 4, SV("12345678901234567890"), 21, 0, S("can't happen"));
    test(S("abcdefghij"), 1, 8, SV(""), 0, 0, S("aj"));
    test(S("abcdefghij"), 1, 8, SV(""), 0, 1, S("aj"));
    test(S("abcdefghij"), 1, 8, SV(""), 1, 0, S("can't happen"));
    test(S("abcdefghij"), 1, 8, SV("12345"), 0, 0, S("aj"));
    test(S("abcdefghij"), 1, 8, SV("12345"), 0, 1, S("a1j"));
    test(S("abcdefghij"), 1, 8, SV("12345"), 0, 2, S("a12j"));
    test(S("abcdefghij"), 1, 8, SV("12345"), 0, 4, S("a1234j"));
    test(S("abcdefghij"), 1, 8, SV("12345"), 0, 5, S("a12345j"));
    test(S("abcdefghij"), 1, 8, SV("12345"), 0, 6, S("a12345j"));
    test(S("abcdefghij"), 1, 8, SV("12345"), 1, 0, S("aj"));

    return true;
}

template <class S, class SV>
TEST_CONSTEXPR_CXX20 bool test26()
{
    test(S("abcdefghij"), 1, 8, SV("12345"), 1, 1, S("a2j"));
    test(S("abcdefghij"), 1, 8, SV("12345"), 1, 2, S("a23j"));
    test(S("abcdefghij"), 1, 8, SV("12345"), 1, 3, S("a234j"));
    test(S("abcdefghij"), 1, 8, SV("12345"), 1, 4, S("a2345j"));
    test(S("abcdefghij"), 1, 8, SV("12345"), 1, 5, S("a2345j"));
    test(S("abcdefghij"), 1, 8, SV("12345"), 2, 0, S("aj"));
    test(S("abcdefghij"), 1, 8, SV("12345"), 2, 1, S("a3j"));
    test(S("abcdefghij"), 1, 8, SV("12345"), 2, 2, S("a34j"));
    test(S("abcdefghij"), 1, 8, SV("12345"), 2, 3, S("a345j"));
    test(S("abcdefghij"), 1, 8, SV("12345"), 2, 4, S("a345j"));
    test(S("abcdefghij"), 1, 8, SV("12345"), 4, 0, S("aj"));
    test(S("abcdefghij"), 1, 8, SV("12345"), 4, 1, S("a5j"));
    test(S("abcdefghij"), 1, 8, SV("12345"), 4, 2, S("a5j"));
    test(S("abcdefghij"), 1, 8, SV("12345"), 5, 0, S("aj"));
    test(S("abcdefghij"), 1, 8, SV("12345"), 5, 1, S("aj"));
    test(S("abcdefghij"), 1, 8, SV("12345"), 6, 0, S("can't happen"));
    test(S("abcdefghij"), 1, 8, SV("1234567890"), 0, 0, S("aj"));
    test(S("abcdefghij"), 1, 8, SV("1234567890"), 0, 1, S("a1j"));
    test(S("abcdefghij"), 1, 8, SV("1234567890"), 0, 5, S("a12345j"));
    test(S("abcdefghij"), 1, 8, SV("1234567890"), 0, 9, S("a123456789j"));
    test(S("abcdefghij"), 1, 8, SV("1234567890"), 0, 10, S("a1234567890j"));
    test(S("abcdefghij"), 1, 8, SV("1234567890"), 0, 11, S("a1234567890j"));
    test(S("abcdefghij"), 1, 8, SV("1234567890"), 1, 0, S("aj"));
    test(S("abcdefghij"), 1, 8, SV("1234567890"), 1, 1, S("a2j"));
    test(S("abcdefghij"), 1, 8, SV("1234567890"), 1, 4, S("a2345j"));
    test(S("abcdefghij"), 1, 8, SV("1234567890"), 1, 8, S("a23456789j"));
    test(S("abcdefghij"), 1, 8, SV("1234567890"), 1, 9, S("a234567890j"));
    test(S("abcdefghij"), 1, 8, SV("1234567890"), 1, 10, S("a234567890j"));
    test(S("abcdefghij"), 1, 8, SV("1234567890"), 5, 0, S("aj"));
    test(S("abcdefghij"), 1, 8, SV("1234567890"), 5, 1, S("a6j"));
    test(S("abcdefghij"), 1, 8, SV("1234567890"), 5, 2, S("a67j"));
    test(S("abcdefghij"), 1, 8, SV("1234567890"), 5, 4, S("a6789j"));
    test(S("abcdefghij"), 1, 8, SV("1234567890"), 5, 5, S("a67890j"));
    test(S("abcdefghij"), 1, 8, SV("1234567890"), 5, 6, S("a67890j"));
    test(S("abcdefghij"), 1, 8, SV("1234567890"), 9, 0, S("aj"));
    test(S("abcdefghij"), 1, 8, SV("1234567890"), 9, 1, S("a0j"));
    test(S("abcdefghij"), 1, 8, SV("1234567890"), 9, 2, S("a0j"));
    test(S("abcdefghij"), 1, 8, SV("1234567890"), 10, 0, S("aj"));
    test(S("abcdefghij"), 1, 8, SV("1234567890"), 10, 1, S("aj"));
    test(S("abcdefghij"), 1, 8, SV("1234567890"), 11, 0, S("can't happen"));
    test(S("abcdefghij"), 1, 8, SV("12345678901234567890"), 0, 0, S("aj"));
    test(S("abcdefghij"), 1, 8, SV("12345678901234567890"), 0, 1, S("a1j"));
    test(S("abcdefghij"), 1, 8, SV("12345678901234567890"), 0, 10, S("a1234567890j"));
    test(S("abcdefghij"), 1, 8, SV("12345678901234567890"), 0, 19, S("a1234567890123456789j"));
    test(S("abcdefghij"), 1, 8, SV("12345678901234567890"), 0, 20, S("a12345678901234567890j"));
    test(S("abcdefghij"), 1, 8, SV("12345678901234567890"), 0, 21, S("a12345678901234567890j"));
    test(S("abcdefghij"), 1, 8, SV("12345678901234567890"), 1, 0, S("aj"));
    test(S("abcdefghij"), 1, 8, SV("12345678901234567890"), 1, 1, S("a2j"));
    test(S("abcdefghij"), 1, 8, SV("12345678901234567890"), 1, 9, S("a234567890j"));
    test(S("abcdefghij"), 1, 8, SV("12345678901234567890"), 1, 18, S("a234567890123456789j"));
    test(S("abcdefghij"), 1, 8, SV("12345678901234567890"), 1, 19, S("a2345678901234567890j"));
    test(S("abcdefghij"), 1, 8, SV("12345678901234567890"), 1, 20, S("a2345678901234567890j"));
    test(S("abcdefghij"), 1, 8, SV("12345678901234567890"), 10, 0, S("aj"));
    test(S("abcdefghij"), 1, 8, SV("12345678901234567890"), 10, 1, S("a1j"));
    test(S("abcdefghij"), 1, 8, SV("12345678901234567890"), 10, 5, S("a12345j"));
    test(S("abcdefghij"), 1, 8, SV("12345678901234567890"), 10, 9, S("a123456789j"));
    test(S("abcdefghij"), 1, 8, SV("12345678901234567890"), 10, 10, S("a1234567890j"));
    test(S("abcdefghij"), 1, 8, SV("12345678901234567890"), 10, 11, S("a1234567890j"));
    test(S("abcdefghij"), 1, 8, SV("12345678901234567890"), 19, 0, S("aj"));
    test(S("abcdefghij"), 1, 8, SV("12345678901234567890"), 19, 1, S("a0j"));
    test(S("abcdefghij"), 1, 8, SV("12345678901234567890"), 19, 2, S("a0j"));
    test(S("abcdefghij"), 1, 8, SV("12345678901234567890"), 20, 0, S("aj"));
    test(S("abcdefghij"), 1, 8, SV("12345678901234567890"), 20, 1, S("aj"));
    test(S("abcdefghij"), 1, 8, SV("12345678901234567890"), 21, 0, S("can't happen"));
    test(S("abcdefghij"), 1, 9, SV(""), 0, 0, S("a"));
    test(S("abcdefghij"), 1, 9, SV(""), 0, 1, S("a"));
    test(S("abcdefghij"), 1, 9, SV(""), 1, 0, S("can't happen"));
    test(S("abcdefghij"), 1, 9, SV("12345"), 0, 0, S("a"));
    test(S("abcdefghij"), 1, 9, SV("12345"), 0, 1, S("a1"));
    test(S("abcdefghij"), 1, 9, SV("12345"), 0, 2, S("a12"));
    test(S("abcdefghij"), 1, 9, SV("12345"), 0, 4, S("a1234"));
    test(S("abcdefghij"), 1, 9, SV("12345"), 0, 5, S("a12345"));
    test(S("abcdefghij"), 1, 9, SV("12345"), 0, 6, S("a12345"));
    test(S("abcdefghij"), 1, 9, SV("12345"), 1, 0, S("a"));
    test(S("abcdefghij"), 1, 9, SV("12345"), 1, 1, S("a2"));
    test(S("abcdefghij"), 1, 9, SV("12345"), 1, 2, S("a23"));
    test(S("abcdefghij"), 1, 9, SV("12345"), 1, 3, S("a234"));
    test(S("abcdefghij"), 1, 9, SV("12345"), 1, 4, S("a2345"));
    test(S("abcdefghij"), 1, 9, SV("12345"), 1, 5, S("a2345"));
    test(S("abcdefghij"), 1, 9, SV("12345"), 2, 0, S("a"));
    test(S("abcdefghij"), 1, 9, SV("12345"), 2, 1, S("a3"));
    test(S("abcdefghij"), 1, 9, SV("12345"), 2, 2, S("a34"));
    test(S("abcdefghij"), 1, 9, SV("12345"), 2, 3, S("a345"));
    test(S("abcdefghij"), 1, 9, SV("12345"), 2, 4, S("a345"));
    test(S("abcdefghij"), 1, 9, SV("12345"), 4, 0, S("a"));
    test(S("abcdefghij"), 1, 9, SV("12345"), 4, 1, S("a5"));
    test(S("abcdefghij"), 1, 9, SV("12345"), 4, 2, S("a5"));
    test(S("abcdefghij"), 1, 9, SV("12345"), 5, 0, S("a"));
    test(S("abcdefghij"), 1, 9, SV("12345"), 5, 1, S("a"));
    test(S("abcdefghij"), 1, 9, SV("12345"), 6, 0, S("can't happen"));
    test(S("abcdefghij"), 1, 9, SV("1234567890"), 0, 0, S("a"));
    test(S("abcdefghij"), 1, 9, SV("1234567890"), 0, 1, S("a1"));
    test(S("abcdefghij"), 1, 9, SV("1234567890"), 0, 5, S("a12345"));
    test(S("abcdefghij"), 1, 9, SV("1234567890"), 0, 9, S("a123456789"));
    test(S("abcdefghij"), 1, 9, SV("1234567890"), 0, 10, S("a1234567890"));
    test(S("abcdefghij"), 1, 9, SV("1234567890"), 0, 11, S("a1234567890"));
    test(S("abcdefghij"), 1, 9, SV("1234567890"), 1, 0, S("a"));
    test(S("abcdefghij"), 1, 9, SV("1234567890"), 1, 1, S("a2"));
    test(S("abcdefghij"), 1, 9, SV("1234567890"), 1, 4, S("a2345"));
    test(S("abcdefghij"), 1, 9, SV("1234567890"), 1, 8, S("a23456789"));

    return true;
}

template <class S, class SV>
TEST_CONSTEXPR_CXX20 bool test27()
{
    test(S("abcdefghij"), 1, 9, SV("1234567890"), 1, 9, S("a234567890"));
    test(S("abcdefghij"), 1, 9, SV("1234567890"), 1, 10, S("a234567890"));
    test(S("abcdefghij"), 1, 9, SV("1234567890"), 5, 0, S("a"));
    test(S("abcdefghij"), 1, 9, SV("1234567890"), 5, 1, S("a6"));
    test(S("abcdefghij"), 1, 9, SV("1234567890"), 5, 2, S("a67"));
    test(S("abcdefghij"), 1, 9, SV("1234567890"), 5, 4, S("a6789"));
    test(S("abcdefghij"), 1, 9, SV("1234567890"), 5, 5, S("a67890"));
    test(S("abcdefghij"), 1, 9, SV("1234567890"), 5, 6, S("a67890"));
    test(S("abcdefghij"), 1, 9, SV("1234567890"), 9, 0, S("a"));
    test(S("abcdefghij"), 1, 9, SV("1234567890"), 9, 1, S("a0"));
    test(S("abcdefghij"), 1, 9, SV("1234567890"), 9, 2, S("a0"));
    test(S("abcdefghij"), 1, 9, SV("1234567890"), 10, 0, S("a"));
    test(S("abcdefghij"), 1, 9, SV("1234567890"), 10, 1, S("a"));
    test(S("abcdefghij"), 1, 9, SV("1234567890"), 11, 0, S("can't happen"));
    test(S("abcdefghij"), 1, 9, SV("12345678901234567890"), 0, 0, S("a"));
    test(S("abcdefghij"), 1, 9, SV("12345678901234567890"), 0, 1, S("a1"));
    test(S("abcdefghij"), 1, 9, SV("12345678901234567890"), 0, 10, S("a1234567890"));
    test(S("abcdefghij"), 1, 9, SV("12345678901234567890"), 0, 19, S("a1234567890123456789"));
    test(S("abcdefghij"), 1, 9, SV("12345678901234567890"), 0, 20, S("a12345678901234567890"));
    test(S("abcdefghij"), 1, 9, SV("12345678901234567890"), 0, 21, S("a12345678901234567890"));
    test(S("abcdefghij"), 1, 9, SV("12345678901234567890"), 1, 0, S("a"));
    test(S("abcdefghij"), 1, 9, SV("12345678901234567890"), 1, 1, S("a2"));
    test(S("abcdefghij"), 1, 9, SV("12345678901234567890"), 1, 9, S("a234567890"));
    test(S("abcdefghij"), 1, 9, SV("12345678901234567890"), 1, 18, S("a234567890123456789"));
    test(S("abcdefghij"), 1, 9, SV("12345678901234567890"), 1, 19, S("a2345678901234567890"));
    test(S("abcdefghij"), 1, 9, SV("12345678901234567890"), 1, 20, S("a2345678901234567890"));
    test(S("abcdefghij"), 1, 9, SV("12345678901234567890"), 10, 0, S("a"));
    test(S("abcdefghij"), 1, 9, SV("12345678901234567890"), 10, 1, S("a1"));
    test(S("abcdefghij"), 1, 9, SV("12345678901234567890"), 10, 5, S("a12345"));
    test(S("abcdefghij"), 1, 9, SV("12345678901234567890"), 10, 9, S("a123456789"));
    test(S("abcdefghij"), 1, 9, SV("12345678901234567890"), 10, 10, S("a1234567890"));
    test(S("abcdefghij"), 1, 9, SV("12345678901234567890"), 10, 11, S("a1234567890"));
    test(S("abcdefghij"), 1, 9, SV("12345678901234567890"), 19, 0, S("a"));
    test(S("abcdefghij"), 1, 9, SV("12345678901234567890"), 19, 1, S("a0"));
    test(S("abcdefghij"), 1, 9, SV("12345678901234567890"), 19, 2, S("a0"));
    test(S("abcdefghij"), 1, 9, SV("12345678901234567890"), 20, 0, S("a"));
    test(S("abcdefghij"), 1, 9, SV("12345678901234567890"), 20, 1, S("a"));
    test(S("abcdefghij"), 1, 9, SV("12345678901234567890"), 21, 0, S("can't happen"));
    test(S("abcdefghij"), 1, 10, SV(""), 0, 0, S("a"));
    test(S("abcdefghij"), 1, 10, SV(""), 0, 1, S("a"));
    test(S("abcdefghij"), 1, 10, SV(""), 1, 0, S("can't happen"));
    test(S("abcdefghij"), 1, 10, SV("12345"), 0, 0, S("a"));
    test(S("abcdefghij"), 1, 10, SV("12345"), 0, 1, S("a1"));
    test(S("abcdefghij"), 1, 10, SV("12345"), 0, 2, S("a12"));
    test(S("abcdefghij"), 1, 10, SV("12345"), 0, 4, S("a1234"));
    test(S("abcdefghij"), 1, 10, SV("12345"), 0, 5, S("a12345"));
    test(S("abcdefghij"), 1, 10, SV("12345"), 0, 6, S("a12345"));
    test(S("abcdefghij"), 1, 10, SV("12345"), 1, 0, S("a"));
    test(S("abcdefghij"), 1, 10, SV("12345"), 1, 1, S("a2"));
    test(S("abcdefghij"), 1, 10, SV("12345"), 1, 2, S("a23"));
    test(S("abcdefghij"), 1, 10, SV("12345"), 1, 3, S("a234"));
    test(S("abcdefghij"), 1, 10, SV("12345"), 1, 4, S("a2345"));
    test(S("abcdefghij"), 1, 10, SV("12345"), 1, 5, S("a2345"));
    test(S("abcdefghij"), 1, 10, SV("12345"), 2, 0, S("a"));
    test(S("abcdefghij"), 1, 10, SV("12345"), 2, 1, S("a3"));
    test(S("abcdefghij"), 1, 10, SV("12345"), 2, 2, S("a34"));
    test(S("abcdefghij"), 1, 10, SV("12345"), 2, 3, S("a345"));
    test(S("abcdefghij"), 1, 10, SV("12345"), 2, 4, S("a345"));
    test(S("abcdefghij"), 1, 10, SV("12345"), 4, 0, S("a"));
    test(S("abcdefghij"), 1, 10, SV("12345"), 4, 1, S("a5"));
    test(S("abcdefghij"), 1, 10, SV("12345"), 4, 2, S("a5"));
    test(S("abcdefghij"), 1, 10, SV("12345"), 5, 0, S("a"));
    test(S("abcdefghij"), 1, 10, SV("12345"), 5, 1, S("a"));
    test(S("abcdefghij"), 1, 10, SV("12345"), 6, 0, S("can't happen"));
    test(S("abcdefghij"), 1, 10, SV("1234567890"), 0, 0, S("a"));
    test(S("abcdefghij"), 1, 10, SV("1234567890"), 0, 1, S("a1"));
    test(S("abcdefghij"), 1, 10, SV("1234567890"), 0, 5, S("a12345"));
    test(S("abcdefghij"), 1, 10, SV("1234567890"), 0, 9, S("a123456789"));
    test(S("abcdefghij"), 1, 10, SV("1234567890"), 0, 10, S("a1234567890"));
    test(S("abcdefghij"), 1, 10, SV("1234567890"), 0, 11, S("a1234567890"));
    test(S("abcdefghij"), 1, 10, SV("1234567890"), 1, 0, S("a"));
    test(S("abcdefghij"), 1, 10, SV("1234567890"), 1, 1, S("a2"));
    test(S("abcdefghij"), 1, 10, SV("1234567890"), 1, 4, S("a2345"));
    test(S("abcdefghij"), 1, 10, SV("1234567890"), 1, 8, S("a23456789"));
    test(S("abcdefghij"), 1, 10, SV("1234567890"), 1, 9, S("a234567890"));
    test(S("abcdefghij"), 1, 10, SV("1234567890"), 1, 10, S("a234567890"));
    test(S("abcdefghij"), 1, 10, SV("1234567890"), 5, 0, S("a"));
    test(S("abcdefghij"), 1, 10, SV("1234567890"), 5, 1, S("a6"));
    test(S("abcdefghij"), 1, 10, SV("1234567890"), 5, 2, S("a67"));
    test(S("abcdefghij"), 1, 10, SV("1234567890"), 5, 4, S("a6789"));
    test(S("abcdefghij"), 1, 10, SV("1234567890"), 5, 5, S("a67890"));
    test(S("abcdefghij"), 1, 10, SV("1234567890"), 5, 6, S("a67890"));
    test(S("abcdefghij"), 1, 10, SV("1234567890"), 9, 0, S("a"));
    test(S("abcdefghij"), 1, 10, SV("1234567890"), 9, 1, S("a0"));
    test(S("abcdefghij"), 1, 10, SV("1234567890"), 9, 2, S("a0"));
    test(S("abcdefghij"), 1, 10, SV("1234567890"), 10, 0, S("a"));
    test(S("abcdefghij"), 1, 10, SV("1234567890"), 10, 1, S("a"));
    test(S("abcdefghij"), 1, 10, SV("1234567890"), 11, 0, S("can't happen"));
    test(S("abcdefghij"), 1, 10, SV("12345678901234567890"), 0, 0, S("a"));
    test(S("abcdefghij"), 1, 10, SV("12345678901234567890"), 0, 1, S("a1"));
    test(S("abcdefghij"), 1, 10, SV("12345678901234567890"), 0, 10, S("a1234567890"));
    test(S("abcdefghij"), 1, 10, SV("12345678901234567890"), 0, 19, S("a1234567890123456789"));
    test(S("abcdefghij"), 1, 10, SV("12345678901234567890"), 0, 20, S("a12345678901234567890"));
    test(S("abcdefghij"), 1, 10, SV("12345678901234567890"), 0, 21, S("a12345678901234567890"));
    test(S("abcdefghij"), 1, 10, SV("12345678901234567890"), 1, 0, S("a"));
    test(S("abcdefghij"), 1, 10, SV("12345678901234567890"), 1, 1, S("a2"));
    test(S("abcdefghij"), 1, 10, SV("12345678901234567890"), 1, 9, S("a234567890"));
    test(S("abcdefghij"), 1, 10, SV("12345678901234567890"), 1, 18, S("a234567890123456789"));
    test(S("abcdefghij"), 1, 10, SV("12345678901234567890"), 1, 19, S("a2345678901234567890"));
    test(S("abcdefghij"), 1, 10, SV("12345678901234567890"), 1, 20, S("a2345678901234567890"));

    return true;
}

template <class S, class SV>
TEST_CONSTEXPR_CXX20 bool test28()
{
    test(S("abcdefghij"), 1, 10, SV("12345678901234567890"), 10, 0, S("a"));
    test(S("abcdefghij"), 1, 10, SV("12345678901234567890"), 10, 1, S("a1"));
    test(S("abcdefghij"), 1, 10, SV("12345678901234567890"), 10, 5, S("a12345"));
    test(S("abcdefghij"), 1, 10, SV("12345678901234567890"), 10, 9, S("a123456789"));
    test(S("abcdefghij"), 1, 10, SV("12345678901234567890"), 10, 10, S("a1234567890"));
    test(S("abcdefghij"), 1, 10, SV("12345678901234567890"), 10, 11, S("a1234567890"));
    test(S("abcdefghij"), 1, 10, SV("12345678901234567890"), 19, 0, S("a"));
    test(S("abcdefghij"), 1, 10, SV("12345678901234567890"), 19, 1, S("a0"));
    test(S("abcdefghij"), 1, 10, SV("12345678901234567890"), 19, 2, S("a0"));
    test(S("abcdefghij"), 1, 10, SV("12345678901234567890"), 20, 0, S("a"));
    test(S("abcdefghij"), 1, 10, SV("12345678901234567890"), 20, 1, S("a"));
    test(S("abcdefghij"), 1, 10, SV("12345678901234567890"), 21, 0, S("can't happen"));
    test(S("abcdefghij"), 5, 0, SV(""), 0, 0, S("abcdefghij"));
    test(S("abcdefghij"), 5, 0, SV(""), 0, 1, S("abcdefghij"));
    test(S("abcdefghij"), 5, 0, SV(""), 1, 0, S("can't happen"));
    test(S("abcdefghij"), 5, 0, SV("12345"), 0, 0, S("abcdefghij"));
    test(S("abcdefghij"), 5, 0, SV("12345"), 0, 1, S("abcde1fghij"));
    test(S("abcdefghij"), 5, 0, SV("12345"), 0, 2, S("abcde12fghij"));
    test(S("abcdefghij"), 5, 0, SV("12345"), 0, 4, S("abcde1234fghij"));
    test(S("abcdefghij"), 5, 0, SV("12345"), 0, 5, S("abcde12345fghij"));
    test(S("abcdefghij"), 5, 0, SV("12345"), 0, 6, S("abcde12345fghij"));
    test(S("abcdefghij"), 5, 0, SV("12345"), 1, 0, S("abcdefghij"));
    test(S("abcdefghij"), 5, 0, SV("12345"), 1, 1, S("abcde2fghij"));
    test(S("abcdefghij"), 5, 0, SV("12345"), 1, 2, S("abcde23fghij"));
    test(S("abcdefghij"), 5, 0, SV("12345"), 1, 3, S("abcde234fghij"));
    test(S("abcdefghij"), 5, 0, SV("12345"), 1, 4, S("abcde2345fghij"));
    test(S("abcdefghij"), 5, 0, SV("12345"), 1, 5, S("abcde2345fghij"));
    test(S("abcdefghij"), 5, 0, SV("12345"), 2, 0, S("abcdefghij"));
    test(S("abcdefghij"), 5, 0, SV("12345"), 2, 1, S("abcde3fghij"));
    test(S("abcdefghij"), 5, 0, SV("12345"), 2, 2, S("abcde34fghij"));
    test(S("abcdefghij"), 5, 0, SV("12345"), 2, 3, S("abcde345fghij"));
    test(S("abcdefghij"), 5, 0, SV("12345"), 2, 4, S("abcde345fghij"));
    test(S("abcdefghij"), 5, 0, SV("12345"), 4, 0, S("abcdefghij"));
    test(S("abcdefghij"), 5, 0, SV("12345"), 4, 1, S("abcde5fghij"));
    test(S("abcdefghij"), 5, 0, SV("12345"), 4, 2, S("abcde5fghij"));
    test(S("abcdefghij"), 5, 0, SV("12345"), 5, 0, S("abcdefghij"));
    test(S("abcdefghij"), 5, 0, SV("12345"), 5, 1, S("abcdefghij"));
    test(S("abcdefghij"), 5, 0, SV("12345"), 6, 0, S("can't happen"));
    test(S("abcdefghij"), 5, 0, SV("1234567890"), 0, 0, S("abcdefghij"));
    test(S("abcdefghij"), 5, 0, SV("1234567890"), 0, 1, S("abcde1fghij"));
    test(S("abcdefghij"), 5, 0, SV("1234567890"), 0, 5, S("abcde12345fghij"));
    test(S("abcdefghij"), 5, 0, SV("1234567890"), 0, 9, S("abcde123456789fghij"));
    test(S("abcdefghij"), 5, 0, SV("1234567890"), 0, 10, S("abcde1234567890fghij"));
    test(S("abcdefghij"), 5, 0, SV("1234567890"), 0, 11, S("abcde1234567890fghij"));
    test(S("abcdefghij"), 5, 0, SV("1234567890"), 1, 0, S("abcdefghij"));
    test(S("abcdefghij"), 5, 0, SV("1234567890"), 1, 1, S("abcde2fghij"));
    test(S("abcdefghij"), 5, 0, SV("1234567890"), 1, 4, S("abcde2345fghij"));
    test(S("abcdefghij"), 5, 0, SV("1234567890"), 1, 8, S("abcde23456789fghij"));
    test(S("abcdefghij"), 5, 0, SV("1234567890"), 1, 9, S("abcde234567890fghij"));
    test(S("abcdefghij"), 5, 0, SV("1234567890"), 1, 10, S("abcde234567890fghij"));
    test(S("abcdefghij"), 5, 0, SV("1234567890"), 5, 0, S("abcdefghij"));
    test(S("abcdefghij"), 5, 0, SV("1234567890"), 5, 1, S("abcde6fghij"));
    test(S("abcdefghij"), 5, 0, SV("1234567890"), 5, 2, S("abcde67fghij"));
    test(S("abcdefghij"), 5, 0, SV("1234567890"), 5, 4, S("abcde6789fghij"));
    test(S("abcdefghij"), 5, 0, SV("1234567890"), 5, 5, S("abcde67890fghij"));
    test(S("abcdefghij"), 5, 0, SV("1234567890"), 5, 6, S("abcde67890fghij"));
    test(S("abcdefghij"), 5, 0, SV("1234567890"), 9, 0, S("abcdefghij"));
    test(S("abcdefghij"), 5, 0, SV("1234567890"), 9, 1, S("abcde0fghij"));
    test(S("abcdefghij"), 5, 0, SV("1234567890"), 9, 2, S("abcde0fghij"));
    test(S("abcdefghij"), 5, 0, SV("1234567890"), 10, 0, S("abcdefghij"));
    test(S("abcdefghij"), 5, 0, SV("1234567890"), 10, 1, S("abcdefghij"));
    test(S("abcdefghij"), 5, 0, SV("1234567890"), 11, 0, S("can't happen"));
    test(S("abcdefghij"), 5, 0, SV("12345678901234567890"), 0, 0, S("abcdefghij"));
    test(S("abcdefghij"), 5, 0, SV("12345678901234567890"), 0, 1, S("abcde1fghij"));
    test(S("abcdefghij"), 5, 0, SV("12345678901234567890"), 0, 10, S("abcde1234567890fghij"));
    test(S("abcdefghij"), 5, 0, SV("12345678901234567890"), 0, 19, S("abcde1234567890123456789fghij"));
    test(S("abcdefghij"), 5, 0, SV("12345678901234567890"), 0, 20, S("abcde12345678901234567890fghij"));
    test(S("abcdefghij"), 5, 0, SV("12345678901234567890"), 0, 21, S("abcde12345678901234567890fghij"));
    test(S("abcdefghij"), 5, 0, SV("12345678901234567890"), 1, 0, S("abcdefghij"));
    test(S("abcdefghij"), 5, 0, SV("12345678901234567890"), 1, 1, S("abcde2fghij"));
    test(S("abcdefghij"), 5, 0, SV("12345678901234567890"), 1, 9, S("abcde234567890fghij"));
    test(S("abcdefghij"), 5, 0, SV("12345678901234567890"), 1, 18, S("abcde234567890123456789fghij"));
    test(S("abcdefghij"), 5, 0, SV("12345678901234567890"), 1, 19, S("abcde2345678901234567890fghij"));
    test(S("abcdefghij"), 5, 0, SV("12345678901234567890"), 1, 20, S("abcde2345678901234567890fghij"));
    test(S("abcdefghij"), 5, 0, SV("12345678901234567890"), 10, 0, S("abcdefghij"));
    test(S("abcdefghij"), 5, 0, SV("12345678901234567890"), 10, 1, S("abcde1fghij"));
    test(S("abcdefghij"), 5, 0, SV("12345678901234567890"), 10, 5, S("abcde12345fghij"));
    test(S("abcdefghij"), 5, 0, SV("12345678901234567890"), 10, 9, S("abcde123456789fghij"));
    test(S("abcdefghij"), 5, 0, SV("12345678901234567890"), 10, 10, S("abcde1234567890fghij"));
    test(S("abcdefghij"), 5, 0, SV("12345678901234567890"), 10, 11, S("abcde1234567890fghij"));
    test(S("abcdefghij"), 5, 0, SV("12345678901234567890"), 19, 0, S("abcdefghij"));
    test(S("abcdefghij"), 5, 0, SV("12345678901234567890"), 19, 1, S("abcde0fghij"));
    test(S("abcdefghij"), 5, 0, SV("12345678901234567890"), 19, 2, S("abcde0fghij"));
    test(S("abcdefghij"), 5, 0, SV("12345678901234567890"), 20, 0, S("abcdefghij"));
    test(S("abcdefghij"), 5, 0, SV("12345678901234567890"), 20, 1, S("abcdefghij"));
    test(S("abcdefghij"), 5, 0, SV("12345678901234567890"), 21, 0, S("can't happen"));
    test(S("abcdefghij"), 5, 1, SV(""), 0, 0, S("abcdeghij"));
    test(S("abcdefghij"), 5, 1, SV(""), 0, 1, S("abcdeghij"));
    test(S("abcdefghij"), 5, 1, SV(""), 1, 0, S("can't happen"));
    test(S("abcdefghij"), 5, 1, SV("12345"), 0, 0, S("abcdeghij"));
    test(S("abcdefghij"), 5, 1, SV("12345"), 0, 1, S("abcde1ghij"));
    test(S("abcdefghij"), 5, 1, SV("12345"), 0, 2, S("abcde12ghij"));
    test(S("abcdefghij"), 5, 1, SV("12345"), 0, 4, S("abcde1234ghij"));
    test(S("abcdefghij"), 5, 1, SV("12345"), 0, 5, S("abcde12345ghij"));
    test(S("abcdefghij"), 5, 1, SV("12345"), 0, 6, S("abcde12345ghij"));
    test(S("abcdefghij"), 5, 1, SV("12345"), 1, 0, S("abcdeghij"));
    test(S("abcdefghij"), 5, 1, SV("12345"), 1, 1, S("abcde2ghij"));
    test(S("abcdefghij"), 5, 1, SV("12345"), 1, 2, S("abcde23ghij"));
    test(S("abcdefghij"), 5, 1, SV("12345"), 1, 3, S("abcde234ghij"));
    test(S("abcdefghij"), 5, 1, SV("12345"), 1, 4, S("abcde2345ghij"));

    return true;
}

template <class S, class SV>
TEST_CONSTEXPR_CXX20 bool test29()
{
    test(S("abcdefghij"), 5, 1, SV("12345"), 1, 5, S("abcde2345ghij"));
    test(S("abcdefghij"), 5, 1, SV("12345"), 2, 0, S("abcdeghij"));
    test(S("abcdefghij"), 5, 1, SV("12345"), 2, 1, S("abcde3ghij"));
    test(S("abcdefghij"), 5, 1, SV("12345"), 2, 2, S("abcde34ghij"));
    test(S("abcdefghij"), 5, 1, SV("12345"), 2, 3, S("abcde345ghij"));
    test(S("abcdefghij"), 5, 1, SV("12345"), 2, 4, S("abcde345ghij"));
    test(S("abcdefghij"), 5, 1, SV("12345"), 4, 0, S("abcdeghij"));
    test(S("abcdefghij"), 5, 1, SV("12345"), 4, 1, S("abcde5ghij"));
    test(S("abcdefghij"), 5, 1, SV("12345"), 4, 2, S("abcde5ghij"));
    test(S("abcdefghij"), 5, 1, SV("12345"), 5, 0, S("abcdeghij"));
    test(S("abcdefghij"), 5, 1, SV("12345"), 5, 1, S("abcdeghij"));
    test(S("abcdefghij"), 5, 1, SV("12345"), 6, 0, S("can't happen"));
    test(S("abcdefghij"), 5, 1, SV("1234567890"), 0, 0, S("abcdeghij"));
    test(S("abcdefghij"), 5, 1, SV("1234567890"), 0, 1, S("abcde1ghij"));
    test(S("abcdefghij"), 5, 1, SV("1234567890"), 0, 5, S("abcde12345ghij"));
    test(S("abcdefghij"), 5, 1, SV("1234567890"), 0, 9, S("abcde123456789ghij"));
    test(S("abcdefghij"), 5, 1, SV("1234567890"), 0, 10, S("abcde1234567890ghij"));
    test(S("abcdefghij"), 5, 1, SV("1234567890"), 0, 11, S("abcde1234567890ghij"));
    test(S("abcdefghij"), 5, 1, SV("1234567890"), 1, 0, S("abcdeghij"));
    test(S("abcdefghij"), 5, 1, SV("1234567890"), 1, 1, S("abcde2ghij"));
    test(S("abcdefghij"), 5, 1, SV("1234567890"), 1, 4, S("abcde2345ghij"));
    test(S("abcdefghij"), 5, 1, SV("1234567890"), 1, 8, S("abcde23456789ghij"));
    test(S("abcdefghij"), 5, 1, SV("1234567890"), 1, 9, S("abcde234567890ghij"));
    test(S("abcdefghij"), 5, 1, SV("1234567890"), 1, 10, S("abcde234567890ghij"));
    test(S("abcdefghij"), 5, 1, SV("1234567890"), 5, 0, S("abcdeghij"));
    test(S("abcdefghij"), 5, 1, SV("1234567890"), 5, 1, S("abcde6ghij"));
    test(S("abcdefghij"), 5, 1, SV("1234567890"), 5, 2, S("abcde67ghij"));
    test(S("abcdefghij"), 5, 1, SV("1234567890"), 5, 4, S("abcde6789ghij"));
    test(S("abcdefghij"), 5, 1, SV("1234567890"), 5, 5, S("abcde67890ghij"));
    test(S("abcdefghij"), 5, 1, SV("1234567890"), 5, 6, S("abcde67890ghij"));
    test(S("abcdefghij"), 5, 1, SV("1234567890"), 9, 0, S("abcdeghij"));
    test(S("abcdefghij"), 5, 1, SV("1234567890"), 9, 1, S("abcde0ghij"));
    test(S("abcdefghij"), 5, 1, SV("1234567890"), 9, 2, S("abcde0ghij"));
    test(S("abcdefghij"), 5, 1, SV("1234567890"), 10, 0, S("abcdeghij"));
    test(S("abcdefghij"), 5, 1, SV("1234567890"), 10, 1, S("abcdeghij"));
    test(S("abcdefghij"), 5, 1, SV("1234567890"), 11, 0, S("can't happen"));
    test(S("abcdefghij"), 5, 1, SV("12345678901234567890"), 0, 0, S("abcdeghij"));
    test(S("abcdefghij"), 5, 1, SV("12345678901234567890"), 0, 1, S("abcde1ghij"));
    test(S("abcdefghij"), 5, 1, SV("12345678901234567890"), 0, 10, S("abcde1234567890ghij"));
    test(S("abcdefghij"), 5, 1, SV("12345678901234567890"), 0, 19, S("abcde1234567890123456789ghij"));
    test(S("abcdefghij"), 5, 1, SV("12345678901234567890"), 0, 20, S("abcde12345678901234567890ghij"));
    test(S("abcdefghij"), 5, 1, SV("12345678901234567890"), 0, 21, S("abcde12345678901234567890ghij"));
    test(S("abcdefghij"), 5, 1, SV("12345678901234567890"), 1, 0, S("abcdeghij"));
    test(S("abcdefghij"), 5, 1, SV("12345678901234567890"), 1, 1, S("abcde2ghij"));
    test(S("abcdefghij"), 5, 1, SV("12345678901234567890"), 1, 9, S("abcde234567890ghij"));
    test(S("abcdefghij"), 5, 1, SV("12345678901234567890"), 1, 18, S("abcde234567890123456789ghij"));
    test(S("abcdefghij"), 5, 1, SV("12345678901234567890"), 1, 19, S("abcde2345678901234567890ghij"));
    test(S("abcdefghij"), 5, 1, SV("12345678901234567890"), 1, 20, S("abcde2345678901234567890ghij"));
    test(S("abcdefghij"), 5, 1, SV("12345678901234567890"), 10, 0, S("abcdeghij"));
    test(S("abcdefghij"), 5, 1, SV("12345678901234567890"), 10, 1, S("abcde1ghij"));
    test(S("abcdefghij"), 5, 1, SV("12345678901234567890"), 10, 5, S("abcde12345ghij"));
    test(S("abcdefghij"), 5, 1, SV("12345678901234567890"), 10, 9, S("abcde123456789ghij"));
    test(S("abcdefghij"), 5, 1, SV("12345678901234567890"), 10, 10, S("abcde1234567890ghij"));
    test(S("abcdefghij"), 5, 1, SV("12345678901234567890"), 10, 11, S("abcde1234567890ghij"));
    test(S("abcdefghij"), 5, 1, SV("12345678901234567890"), 19, 0, S("abcdeghij"));
    test(S("abcdefghij"), 5, 1, SV("12345678901234567890"), 19, 1, S("abcde0ghij"));
    test(S("abcdefghij"), 5, 1, SV("12345678901234567890"), 19, 2, S("abcde0ghij"));
    test(S("abcdefghij"), 5, 1, SV("12345678901234567890"), 20, 0, S("abcdeghij"));
    test(S("abcdefghij"), 5, 1, SV("12345678901234567890"), 20, 1, S("abcdeghij"));
    test(S("abcdefghij"), 5, 1, SV("12345678901234567890"), 21, 0, S("can't happen"));
    test(S("abcdefghij"), 5, 2, SV(""), 0, 0, S("abcdehij"));
    test(S("abcdefghij"), 5, 2, SV(""), 0, 1, S("abcdehij"));
    test(S("abcdefghij"), 5, 2, SV(""), 1, 0, S("can't happen"));
    test(S("abcdefghij"), 5, 2, SV("12345"), 0, 0, S("abcdehij"));
    test(S("abcdefghij"), 5, 2, SV("12345"), 0, 1, S("abcde1hij"));
    test(S("abcdefghij"), 5, 2, SV("12345"), 0, 2, S("abcde12hij"));
    test(S("abcdefghij"), 5, 2, SV("12345"), 0, 4, S("abcde1234hij"));
    test(S("abcdefghij"), 5, 2, SV("12345"), 0, 5, S("abcde12345hij"));
    test(S("abcdefghij"), 5, 2, SV("12345"), 0, 6, S("abcde12345hij"));
    test(S("abcdefghij"), 5, 2, SV("12345"), 1, 0, S("abcdehij"));
    test(S("abcdefghij"), 5, 2, SV("12345"), 1, 1, S("abcde2hij"));
    test(S("abcdefghij"), 5, 2, SV("12345"), 1, 2, S("abcde23hij"));
    test(S("abcdefghij"), 5, 2, SV("12345"), 1, 3, S("abcde234hij"));
    test(S("abcdefghij"), 5, 2, SV("12345"), 1, 4, S("abcde2345hij"));
    test(S("abcdefghij"), 5, 2, SV("12345"), 1, 5, S("abcde2345hij"));
    test(S("abcdefghij"), 5, 2, SV("12345"), 2, 0, S("abcdehij"));
    test(S("abcdefghij"), 5, 2, SV("12345"), 2, 1, S("abcde3hij"));
    test(S("abcdefghij"), 5, 2, SV("12345"), 2, 2, S("abcde34hij"));
    test(S("abcdefghij"), 5, 2, SV("12345"), 2, 3, S("abcde345hij"));
    test(S("abcdefghij"), 5, 2, SV("12345"), 2, 4, S("abcde345hij"));
    test(S("abcdefghij"), 5, 2, SV("12345"), 4, 0, S("abcdehij"));
    test(S("abcdefghij"), 5, 2, SV("12345"), 4, 1, S("abcde5hij"));
    test(S("abcdefghij"), 5, 2, SV("12345"), 4, 2, S("abcde5hij"));
    test(S("abcdefghij"), 5, 2, SV("12345"), 5, 0, S("abcdehij"));
    test(S("abcdefghij"), 5, 2, SV("12345"), 5, 1, S("abcdehij"));
    test(S("abcdefghij"), 5, 2, SV("12345"), 6, 0, S("can't happen"));
    test(S("abcdefghij"), 5, 2, SV("1234567890"), 0, 0, S("abcdehij"));
    test(S("abcdefghij"), 5, 2, SV("1234567890"), 0, 1, S("abcde1hij"));
    test(S("abcdefghij"), 5, 2, SV("1234567890"), 0, 5, S("abcde12345hij"));
    test(S("abcdefghij"), 5, 2, SV("1234567890"), 0, 9, S("abcde123456789hij"));
    test(S("abcdefghij"), 5, 2, SV("1234567890"), 0, 10, S("abcde1234567890hij"));
    test(S("abcdefghij"), 5, 2, SV("1234567890"), 0, 11, S("abcde1234567890hij"));
    test(S("abcdefghij"), 5, 2, SV("1234567890"), 1, 0, S("abcdehij"));
    test(S("abcdefghij"), 5, 2, SV("1234567890"), 1, 1, S("abcde2hij"));
    test(S("abcdefghij"), 5, 2, SV("1234567890"), 1, 4, S("abcde2345hij"));
    test(S("abcdefghij"), 5, 2, SV("1234567890"), 1, 8, S("abcde23456789hij"));
    test(S("abcdefghij"), 5, 2, SV("1234567890"), 1, 9, S("abcde234567890hij"));
    test(S("abcdefghij"), 5, 2, SV("1234567890"), 1, 10, S("abcde234567890hij"));
    test(S("abcdefghij"), 5, 2, SV("1234567890"), 5, 0, S("abcdehij"));
    test(S("abcdefghij"), 5, 2, SV("1234567890"), 5, 1, S("abcde6hij"));

    return true;
}

template <class S, class SV>
TEST_CONSTEXPR_CXX20 bool test30()
{
    test(S("abcdefghij"), 5, 2, SV("1234567890"), 5, 2, S("abcde67hij"));
    test(S("abcdefghij"), 5, 2, SV("1234567890"), 5, 4, S("abcde6789hij"));
    test(S("abcdefghij"), 5, 2, SV("1234567890"), 5, 5, S("abcde67890hij"));
    test(S("abcdefghij"), 5, 2, SV("1234567890"), 5, 6, S("abcde67890hij"));
    test(S("abcdefghij"), 5, 2, SV("1234567890"), 9, 0, S("abcdehij"));
    test(S("abcdefghij"), 5, 2, SV("1234567890"), 9, 1, S("abcde0hij"));
    test(S("abcdefghij"), 5, 2, SV("1234567890"), 9, 2, S("abcde0hij"));
    test(S("abcdefghij"), 5, 2, SV("1234567890"), 10, 0, S("abcdehij"));
    test(S("abcdefghij"), 5, 2, SV("1234567890"), 10, 1, S("abcdehij"));
    test(S("abcdefghij"), 5, 2, SV("1234567890"), 11, 0, S("can't happen"));
    test(S("abcdefghij"), 5, 2, SV("12345678901234567890"), 0, 0, S("abcdehij"));
    test(S("abcdefghij"), 5, 2, SV("12345678901234567890"), 0, 1, S("abcde1hij"));
    test(S("abcdefghij"), 5, 2, SV("12345678901234567890"), 0, 10, S("abcde1234567890hij"));
    test(S("abcdefghij"), 5, 2, SV("12345678901234567890"), 0, 19, S("abcde1234567890123456789hij"));
    test(S("abcdefghij"), 5, 2, SV("12345678901234567890"), 0, 20, S("abcde12345678901234567890hij"));
    test(S("abcdefghij"), 5, 2, SV("12345678901234567890"), 0, 21, S("abcde12345678901234567890hij"));
    test(S("abcdefghij"), 5, 2, SV("12345678901234567890"), 1, 0, S("abcdehij"));
    test(S("abcdefghij"), 5, 2, SV("12345678901234567890"), 1, 1, S("abcde2hij"));
    test(S("abcdefghij"), 5, 2, SV("12345678901234567890"), 1, 9, S("abcde234567890hij"));
    test(S("abcdefghij"), 5, 2, SV("12345678901234567890"), 1, 18, S("abcde234567890123456789hij"));
    test(S("abcdefghij"), 5, 2, SV("12345678901234567890"), 1, 19, S("abcde2345678901234567890hij"));
    test(S("abcdefghij"), 5, 2, SV("12345678901234567890"), 1, 20, S("abcde2345678901234567890hij"));
    test(S("abcdefghij"), 5, 2, SV("12345678901234567890"), 10, 0, S("abcdehij"));
    test(S("abcdefghij"), 5, 2, SV("12345678901234567890"), 10, 1, S("abcde1hij"));
    test(S("abcdefghij"), 5, 2, SV("12345678901234567890"), 10, 5, S("abcde12345hij"));
    test(S("abcdefghij"), 5, 2, SV("12345678901234567890"), 10, 9, S("abcde123456789hij"));
    test(S("abcdefghij"), 5, 2, SV("12345678901234567890"), 10, 10, S("abcde1234567890hij"));
    test(S("abcdefghij"), 5, 2, SV("12345678901234567890"), 10, 11, S("abcde1234567890hij"));
    test(S("abcdefghij"), 5, 2, SV("12345678901234567890"), 19, 0, S("abcdehij"));
    test(S("abcdefghij"), 5, 2, SV("12345678901234567890"), 19, 1, S("abcde0hij"));
    test(S("abcdefghij"), 5, 2, SV("12345678901234567890"), 19, 2, S("abcde0hij"));
    test(S("abcdefghij"), 5, 2, SV("12345678901234567890"), 20, 0, S("abcdehij"));
    test(S("abcdefghij"), 5, 2, SV("12345678901234567890"), 20, 1, S("abcdehij"));
    test(S("abcdefghij"), 5, 2, SV("12345678901234567890"), 21, 0, S("can't happen"));
    test(S("abcdefghij"), 5, 4, SV(""), 0, 0, S("abcdej"));
    test(S("abcdefghij"), 5, 4, SV(""), 0, 1, S("abcdej"));
    test(S("abcdefghij"), 5, 4, SV(""), 1, 0, S("can't happen"));
    test(S("abcdefghij"), 5, 4, SV("12345"), 0, 0, S("abcdej"));
    test(S("abcdefghij"), 5, 4, SV("12345"), 0, 1, S("abcde1j"));
    test(S("abcdefghij"), 5, 4, SV("12345"), 0, 2, S("abcde12j"));
    test(S("abcdefghij"), 5, 4, SV("12345"), 0, 4, S("abcde1234j"));
    test(S("abcdefghij"), 5, 4, SV("12345"), 0, 5, S("abcde12345j"));
    test(S("abcdefghij"), 5, 4, SV("12345"), 0, 6, S("abcde12345j"));
    test(S("abcdefghij"), 5, 4, SV("12345"), 1, 0, S("abcdej"));
    test(S("abcdefghij"), 5, 4, SV("12345"), 1, 1, S("abcde2j"));
    test(S("abcdefghij"), 5, 4, SV("12345"), 1, 2, S("abcde23j"));
    test(S("abcdefghij"), 5, 4, SV("12345"), 1, 3, S("abcde234j"));
    test(S("abcdefghij"), 5, 4, SV("12345"), 1, 4, S("abcde2345j"));
    test(S("abcdefghij"), 5, 4, SV("12345"), 1, 5, S("abcde2345j"));
    test(S("abcdefghij"), 5, 4, SV("12345"), 2, 0, S("abcdej"));
    test(S("abcdefghij"), 5, 4, SV("12345"), 2, 1, S("abcde3j"));
    test(S("abcdefghij"), 5, 4, SV("12345"), 2, 2, S("abcde34j"));
    test(S("abcdefghij"), 5, 4, SV("12345"), 2, 3, S("abcde345j"));
    test(S("abcdefghij"), 5, 4, SV("12345"), 2, 4, S("abcde345j"));
    test(S("abcdefghij"), 5, 4, SV("12345"), 4, 0, S("abcdej"));
    test(S("abcdefghij"), 5, 4, SV("12345"), 4, 1, S("abcde5j"));
    test(S("abcdefghij"), 5, 4, SV("12345"), 4, 2, S("abcde5j"));
    test(S("abcdefghij"), 5, 4, SV("12345"), 5, 0, S("abcdej"));
    test(S("abcdefghij"), 5, 4, SV("12345"), 5, 1, S("abcdej"));
    test(S("abcdefghij"), 5, 4, SV("12345"), 6, 0, S("can't happen"));
    test(S("abcdefghij"), 5, 4, SV("1234567890"), 0, 0, S("abcdej"));
    test(S("abcdefghij"), 5, 4, SV("1234567890"), 0, 1, S("abcde1j"));
    test(S("abcdefghij"), 5, 4, SV("1234567890"), 0, 5, S("abcde12345j"));
    test(S("abcdefghij"), 5, 4, SV("1234567890"), 0, 9, S("abcde123456789j"));
    test(S("abcdefghij"), 5, 4, SV("1234567890"), 0, 10, S("abcde1234567890j"));
    test(S("abcdefghij"), 5, 4, SV("1234567890"), 0, 11, S("abcde1234567890j"));
    test(S("abcdefghij"), 5, 4, SV("1234567890"), 1, 0, S("abcdej"));
    test(S("abcdefghij"), 5, 4, SV("1234567890"), 1, 1, S("abcde2j"));
    test(S("abcdefghij"), 5, 4, SV("1234567890"), 1, 4, S("abcde2345j"));
    test(S("abcdefghij"), 5, 4, SV("1234567890"), 1, 8, S("abcde23456789j"));
    test(S("abcdefghij"), 5, 4, SV("1234567890"), 1, 9, S("abcde234567890j"));
    test(S("abcdefghij"), 5, 4, SV("1234567890"), 1, 10, S("abcde234567890j"));
    test(S("abcdefghij"), 5, 4, SV("1234567890"), 5, 0, S("abcdej"));
    test(S("abcdefghij"), 5, 4, SV("1234567890"), 5, 1, S("abcde6j"));
    test(S("abcdefghij"), 5, 4, SV("1234567890"), 5, 2, S("abcde67j"));
    test(S("abcdefghij"), 5, 4, SV("1234567890"), 5, 4, S("abcde6789j"));
    test(S("abcdefghij"), 5, 4, SV("1234567890"), 5, 5, S("abcde67890j"));
    test(S("abcdefghij"), 5, 4, SV("1234567890"), 5, 6, S("abcde67890j"));
    test(S("abcdefghij"), 5, 4, SV("1234567890"), 9, 0, S("abcdej"));
    test(S("abcdefghij"), 5, 4, SV("1234567890"), 9, 1, S("abcde0j"));
    test(S("abcdefghij"), 5, 4, SV("1234567890"), 9, 2, S("abcde0j"));
    test(S("abcdefghij"), 5, 4, SV("1234567890"), 10, 0, S("abcdej"));
    test(S("abcdefghij"), 5, 4, SV("1234567890"), 10, 1, S("abcdej"));
    test(S("abcdefghij"), 5, 4, SV("1234567890"), 11, 0, S("can't happen"));
    test(S("abcdefghij"), 5, 4, SV("12345678901234567890"), 0, 0, S("abcdej"));
    test(S("abcdefghij"), 5, 4, SV("12345678901234567890"), 0, 1, S("abcde1j"));
    test(S("abcdefghij"), 5, 4, SV("12345678901234567890"), 0, 10, S("abcde1234567890j"));
    test(S("abcdefghij"), 5, 4, SV("12345678901234567890"), 0, 19, S("abcde1234567890123456789j"));
    test(S("abcdefghij"), 5, 4, SV("12345678901234567890"), 0, 20, S("abcde12345678901234567890j"));
    test(S("abcdefghij"), 5, 4, SV("12345678901234567890"), 0, 21, S("abcde12345678901234567890j"));
    test(S("abcdefghij"), 5, 4, SV("12345678901234567890"), 1, 0, S("abcdej"));
    test(S("abcdefghij"), 5, 4, SV("12345678901234567890"), 1, 1, S("abcde2j"));
    test(S("abcdefghij"), 5, 4, SV("12345678901234567890"), 1, 9, S("abcde234567890j"));
    test(S("abcdefghij"), 5, 4, SV("12345678901234567890"), 1, 18, S("abcde234567890123456789j"));
    test(S("abcdefghij"), 5, 4, SV("12345678901234567890"), 1, 19, S("abcde2345678901234567890j"));
    test(S("abcdefghij"), 5, 4, SV("12345678901234567890"), 1, 20, S("abcde2345678901234567890j"));
    test(S("abcdefghij"), 5, 4, SV("12345678901234567890"), 10, 0, S("abcdej"));
    test(S("abcdefghij"), 5, 4, SV("12345678901234567890"), 10, 1, S("abcde1j"));
    test(S("abcdefghij"), 5, 4, SV("12345678901234567890"), 10, 5, S("abcde12345j"));
    test(S("abcdefghij"), 5, 4, SV("12345678901234567890"), 10, 9, S("abcde123456789j"));

    return true;
}

template <class S, class SV>
TEST_CONSTEXPR_CXX20 bool test31()
{
    test(S("abcdefghij"), 5, 4, SV("12345678901234567890"), 10, 10, S("abcde1234567890j"));
    test(S("abcdefghij"), 5, 4, SV("12345678901234567890"), 10, 11, S("abcde1234567890j"));
    test(S("abcdefghij"), 5, 4, SV("12345678901234567890"), 19, 0, S("abcdej"));
    test(S("abcdefghij"), 5, 4, SV("12345678901234567890"), 19, 1, S("abcde0j"));
    test(S("abcdefghij"), 5, 4, SV("12345678901234567890"), 19, 2, S("abcde0j"));
    test(S("abcdefghij"), 5, 4, SV("12345678901234567890"), 20, 0, S("abcdej"));
    test(S("abcdefghij"), 5, 4, SV("12345678901234567890"), 20, 1, S("abcdej"));
    test(S("abcdefghij"), 5, 4, SV("12345678901234567890"), 21, 0, S("can't happen"));
    test(S("abcdefghij"), 5, 5, SV(""), 0, 0, S("abcde"));
    test(S("abcdefghij"), 5, 5, SV(""), 0, 1, S("abcde"));
    test(S("abcdefghij"), 5, 5, SV(""), 1, 0, S("can't happen"));
    test(S("abcdefghij"), 5, 5, SV("12345"), 0, 0, S("abcde"));
    test(S("abcdefghij"), 5, 5, SV("12345"), 0, 1, S("abcde1"));
    test(S("abcdefghij"), 5, 5, SV("12345"), 0, 2, S("abcde12"));
    test(S("abcdefghij"), 5, 5, SV("12345"), 0, 4, S("abcde1234"));
    test(S("abcdefghij"), 5, 5, SV("12345"), 0, 5, S("abcde12345"));
    test(S("abcdefghij"), 5, 5, SV("12345"), 0, 6, S("abcde12345"));
    test(S("abcdefghij"), 5, 5, SV("12345"), 1, 0, S("abcde"));
    test(S("abcdefghij"), 5, 5, SV("12345"), 1, 1, S("abcde2"));
    test(S("abcdefghij"), 5, 5, SV("12345"), 1, 2, S("abcde23"));
    test(S("abcdefghij"), 5, 5, SV("12345"), 1, 3, S("abcde234"));
    test(S("abcdefghij"), 5, 5, SV("12345"), 1, 4, S("abcde2345"));
    test(S("abcdefghij"), 5, 5, SV("12345"), 1, 5, S("abcde2345"));
    test(S("abcdefghij"), 5, 5, SV("12345"), 2, 0, S("abcde"));
    test(S("abcdefghij"), 5, 5, SV("12345"), 2, 1, S("abcde3"));
    test(S("abcdefghij"), 5, 5, SV("12345"), 2, 2, S("abcde34"));
    test(S("abcdefghij"), 5, 5, SV("12345"), 2, 3, S("abcde345"));
    test(S("abcdefghij"), 5, 5, SV("12345"), 2, 4, S("abcde345"));
    test(S("abcdefghij"), 5, 5, SV("12345"), 4, 0, S("abcde"));
    test(S("abcdefghij"), 5, 5, SV("12345"), 4, 1, S("abcde5"));
    test(S("abcdefghij"), 5, 5, SV("12345"), 4, 2, S("abcde5"));
    test(S("abcdefghij"), 5, 5, SV("12345"), 5, 0, S("abcde"));
    test(S("abcdefghij"), 5, 5, SV("12345"), 5, 1, S("abcde"));
    test(S("abcdefghij"), 5, 5, SV("12345"), 6, 0, S("can't happen"));
    test(S("abcdefghij"), 5, 5, SV("1234567890"), 0, 0, S("abcde"));
    test(S("abcdefghij"), 5, 5, SV("1234567890"), 0, 1, S("abcde1"));
    test(S("abcdefghij"), 5, 5, SV("1234567890"), 0, 5, S("abcde12345"));
    test(S("abcdefghij"), 5, 5, SV("1234567890"), 0, 9, S("abcde123456789"));
    test(S("abcdefghij"), 5, 5, SV("1234567890"), 0, 10, S("abcde1234567890"));
    test(S("abcdefghij"), 5, 5, SV("1234567890"), 0, 11, S("abcde1234567890"));
    test(S("abcdefghij"), 5, 5, SV("1234567890"), 1, 0, S("abcde"));
    test(S("abcdefghij"), 5, 5, SV("1234567890"), 1, 1, S("abcde2"));
    test(S("abcdefghij"), 5, 5, SV("1234567890"), 1, 4, S("abcde2345"));
    test(S("abcdefghij"), 5, 5, SV("1234567890"), 1, 8, S("abcde23456789"));
    test(S("abcdefghij"), 5, 5, SV("1234567890"), 1, 9, S("abcde234567890"));
    test(S("abcdefghij"), 5, 5, SV("1234567890"), 1, 10, S("abcde234567890"));
    test(S("abcdefghij"), 5, 5, SV("1234567890"), 5, 0, S("abcde"));
    test(S("abcdefghij"), 5, 5, SV("1234567890"), 5, 1, S("abcde6"));
    test(S("abcdefghij"), 5, 5, SV("1234567890"), 5, 2, S("abcde67"));
    test(S("abcdefghij"), 5, 5, SV("1234567890"), 5, 4, S("abcde6789"));
    test(S("abcdefghij"), 5, 5, SV("1234567890"), 5, 5, S("abcde67890"));
    test(S("abcdefghij"), 5, 5, SV("1234567890"), 5, 6, S("abcde67890"));
    test(S("abcdefghij"), 5, 5, SV("1234567890"), 9, 0, S("abcde"));
    test(S("abcdefghij"), 5, 5, SV("1234567890"), 9, 1, S("abcde0"));
    test(S("abcdefghij"), 5, 5, SV("1234567890"), 9, 2, S("abcde0"));
    test(S("abcdefghij"), 5, 5, SV("1234567890"), 10, 0, S("abcde"));
    test(S("abcdefghij"), 5, 5, SV("1234567890"), 10, 1, S("abcde"));
    test(S("abcdefghij"), 5, 5, SV("1234567890"), 11, 0, S("can't happen"));
    test(S("abcdefghij"), 5, 5, SV("12345678901234567890"), 0, 0, S("abcde"));
    test(S("abcdefghij"), 5, 5, SV("12345678901234567890"), 0, 1, S("abcde1"));
    test(S("abcdefghij"), 5, 5, SV("12345678901234567890"), 0, 10, S("abcde1234567890"));
    test(S("abcdefghij"), 5, 5, SV("12345678901234567890"), 0, 19, S("abcde1234567890123456789"));
    test(S("abcdefghij"), 5, 5, SV("12345678901234567890"), 0, 20, S("abcde12345678901234567890"));
    test(S("abcdefghij"), 5, 5, SV("12345678901234567890"), 0, 21, S("abcde12345678901234567890"));
    test(S("abcdefghij"), 5, 5, SV("12345678901234567890"), 1, 0, S("abcde"));
    test(S("abcdefghij"), 5, 5, SV("12345678901234567890"), 1, 1, S("abcde2"));
    test(S("abcdefghij"), 5, 5, SV("12345678901234567890"), 1, 9, S("abcde234567890"));
    test(S("abcdefghij"), 5, 5, SV("12345678901234567890"), 1, 18, S("abcde234567890123456789"));
    test(S("abcdefghij"), 5, 5, SV("12345678901234567890"), 1, 19, S("abcde2345678901234567890"));
    test(S("abcdefghij"), 5, 5, SV("12345678901234567890"), 1, 20, S("abcde2345678901234567890"));
    test(S("abcdefghij"), 5, 5, SV("12345678901234567890"), 10, 0, S("abcde"));
    test(S("abcdefghij"), 5, 5, SV("12345678901234567890"), 10, 1, S("abcde1"));
    test(S("abcdefghij"), 5, 5, SV("12345678901234567890"), 10, 5, S("abcde12345"));
    test(S("abcdefghij"), 5, 5, SV("12345678901234567890"), 10, 9, S("abcde123456789"));
    test(S("abcdefghij"), 5, 5, SV("12345678901234567890"), 10, 10, S("abcde1234567890"));
    test(S("abcdefghij"), 5, 5, SV("12345678901234567890"), 10, 11, S("abcde1234567890"));
    test(S("abcdefghij"), 5, 5, SV("12345678901234567890"), 19, 0, S("abcde"));
    test(S("abcdefghij"), 5, 5, SV("12345678901234567890"), 19, 1, S("abcde0"));
    test(S("abcdefghij"), 5, 5, SV("12345678901234567890"), 19, 2, S("abcde0"));
    test(S("abcdefghij"), 5, 5, SV("12345678901234567890"), 20, 0, S("abcde"));
    test(S("abcdefghij"), 5, 5, SV("12345678901234567890"), 20, 1, S("abcde"));
    test(S("abcdefghij"), 5, 5, SV("12345678901234567890"), 21, 0, S("can't happen"));
    test(S("abcdefghij"), 5, 6, SV(""), 0, 0, S("abcde"));
    test(S("abcdefghij"), 5, 6, SV(""), 0, 1, S("abcde"));
    test(S("abcdefghij"), 5, 6, SV(""), 1, 0, S("can't happen"));
    test(S("abcdefghij"), 5, 6, SV("12345"), 0, 0, S("abcde"));
    test(S("abcdefghij"), 5, 6, SV("12345"), 0, 1, S("abcde1"));
    test(S("abcdefghij"), 5, 6, SV("12345"), 0, 2, S("abcde12"));
    test(S("abcdefghij"), 5, 6, SV("12345"), 0, 4, S("abcde1234"));
    test(S("abcdefghij"), 5, 6, SV("12345"), 0, 5, S("abcde12345"));
    test(S("abcdefghij"), 5, 6, SV("12345"), 0, 6, S("abcde12345"));
    test(S("abcdefghij"), 5, 6, SV("12345"), 1, 0, S("abcde"));
    test(S("abcdefghij"), 5, 6, SV("12345"), 1, 1, S("abcde2"));
    test(S("abcdefghij"), 5, 6, SV("12345"), 1, 2, S("abcde23"));
    test(S("abcdefghij"), 5, 6, SV("12345"), 1, 3, S("abcde234"));
    test(S("abcdefghij"), 5, 6, SV("12345"), 1, 4, S("abcde2345"));
    test(S("abcdefghij"), 5, 6, SV("12345"), 1, 5, S("abcde2345"));
    test(S("abcdefghij"), 5, 6, SV("12345"), 2, 0, S("abcde"));
    test(S("abcdefghij"), 5, 6, SV("12345"), 2, 1, S("abcde3"));
    test(S("abcdefghij"), 5, 6, SV("12345"), 2, 2, S("abcde34"));

    return true;
}

template <class S, class SV>
TEST_CONSTEXPR_CXX20 bool test32()
{
    test(S("abcdefghij"), 5, 6, SV("12345"), 2, 3, S("abcde345"));
    test(S("abcdefghij"), 5, 6, SV("12345"), 2, 4, S("abcde345"));
    test(S("abcdefghij"), 5, 6, SV("12345"), 4, 0, S("abcde"));
    test(S("abcdefghij"), 5, 6, SV("12345"), 4, 1, S("abcde5"));
    test(S("abcdefghij"), 5, 6, SV("12345"), 4, 2, S("abcde5"));
    test(S("abcdefghij"), 5, 6, SV("12345"), 5, 0, S("abcde"));
    test(S("abcdefghij"), 5, 6, SV("12345"), 5, 1, S("abcde"));
    test(S("abcdefghij"), 5, 6, SV("12345"), 6, 0, S("can't happen"));
    test(S("abcdefghij"), 5, 6, SV("1234567890"), 0, 0, S("abcde"));
    test(S("abcdefghij"), 5, 6, SV("1234567890"), 0, 1, S("abcde1"));
    test(S("abcdefghij"), 5, 6, SV("1234567890"), 0, 5, S("abcde12345"));
    test(S("abcdefghij"), 5, 6, SV("1234567890"), 0, 9, S("abcde123456789"));
    test(S("abcdefghij"), 5, 6, SV("1234567890"), 0, 10, S("abcde1234567890"));
    test(S("abcdefghij"), 5, 6, SV("1234567890"), 0, 11, S("abcde1234567890"));
    test(S("abcdefghij"), 5, 6, SV("1234567890"), 1, 0, S("abcde"));
    test(S("abcdefghij"), 5, 6, SV("1234567890"), 1, 1, S("abcde2"));
    test(S("abcdefghij"), 5, 6, SV("1234567890"), 1, 4, S("abcde2345"));
    test(S("abcdefghij"), 5, 6, SV("1234567890"), 1, 8, S("abcde23456789"));
    test(S("abcdefghij"), 5, 6, SV("1234567890"), 1, 9, S("abcde234567890"));
    test(S("abcdefghij"), 5, 6, SV("1234567890"), 1, 10, S("abcde234567890"));
    test(S("abcdefghij"), 5, 6, SV("1234567890"), 5, 0, S("abcde"));
    test(S("abcdefghij"), 5, 6, SV("1234567890"), 5, 1, S("abcde6"));
    test(S("abcdefghij"), 5, 6, SV("1234567890"), 5, 2, S("abcde67"));
    test(S("abcdefghij"), 5, 6, SV("1234567890"), 5, 4, S("abcde6789"));
    test(S("abcdefghij"), 5, 6, SV("1234567890"), 5, 5, S("abcde67890"));
    test(S("abcdefghij"), 5, 6, SV("1234567890"), 5, 6, S("abcde67890"));
    test(S("abcdefghij"), 5, 6, SV("1234567890"), 9, 0, S("abcde"));
    test(S("abcdefghij"), 5, 6, SV("1234567890"), 9, 1, S("abcde0"));
    test(S("abcdefghij"), 5, 6, SV("1234567890"), 9, 2, S("abcde0"));
    test(S("abcdefghij"), 5, 6, SV("1234567890"), 10, 0, S("abcde"));
    test(S("abcdefghij"), 5, 6, SV("1234567890"), 10, 1, S("abcde"));
    test(S("abcdefghij"), 5, 6, SV("1234567890"), 11, 0, S("can't happen"));
    test(S("abcdefghij"), 5, 6, SV("12345678901234567890"), 0, 0, S("abcde"));
    test(S("abcdefghij"), 5, 6, SV("12345678901234567890"), 0, 1, S("abcde1"));
    test(S("abcdefghij"), 5, 6, SV("12345678901234567890"), 0, 10, S("abcde1234567890"));
    test(S("abcdefghij"), 5, 6, SV("12345678901234567890"), 0, 19, S("abcde1234567890123456789"));
    test(S("abcdefghij"), 5, 6, SV("12345678901234567890"), 0, 20, S("abcde12345678901234567890"));
    test(S("abcdefghij"), 5, 6, SV("12345678901234567890"), 0, 21, S("abcde12345678901234567890"));
    test(S("abcdefghij"), 5, 6, SV("12345678901234567890"), 1, 0, S("abcde"));
    test(S("abcdefghij"), 5, 6, SV("12345678901234567890"), 1, 1, S("abcde2"));
    test(S("abcdefghij"), 5, 6, SV("12345678901234567890"), 1, 9, S("abcde234567890"));
    test(S("abcdefghij"), 5, 6, SV("12345678901234567890"), 1, 18, S("abcde234567890123456789"));
    test(S("abcdefghij"), 5, 6, SV("12345678901234567890"), 1, 19, S("abcde2345678901234567890"));
    test(S("abcdefghij"), 5, 6, SV("12345678901234567890"), 1, 20, S("abcde2345678901234567890"));
    test(S("abcdefghij"), 5, 6, SV("12345678901234567890"), 10, 0, S("abcde"));
    test(S("abcdefghij"), 5, 6, SV("12345678901234567890"), 10, 1, S("abcde1"));
    test(S("abcdefghij"), 5, 6, SV("12345678901234567890"), 10, 5, S("abcde12345"));
    test(S("abcdefghij"), 5, 6, SV("12345678901234567890"), 10, 9, S("abcde123456789"));
    test(S("abcdefghij"), 5, 6, SV("12345678901234567890"), 10, 10, S("abcde1234567890"));
    test(S("abcdefghij"), 5, 6, SV("12345678901234567890"), 10, 11, S("abcde1234567890"));
    test(S("abcdefghij"), 5, 6, SV("12345678901234567890"), 19, 0, S("abcde"));
    test(S("abcdefghij"), 5, 6, SV("12345678901234567890"), 19, 1, S("abcde0"));
    test(S("abcdefghij"), 5, 6, SV("12345678901234567890"), 19, 2, S("abcde0"));
    test(S("abcdefghij"), 5, 6, SV("12345678901234567890"), 20, 0, S("abcde"));
    test(S("abcdefghij"), 5, 6, SV("12345678901234567890"), 20, 1, S("abcde"));
    test(S("abcdefghij"), 5, 6, SV("12345678901234567890"), 21, 0, S("can't happen"));
    test(S("abcdefghij"), 9, 0, SV(""), 0, 0, S("abcdefghij"));
    test(S("abcdefghij"), 9, 0, SV(""), 0, 1, S("abcdefghij"));
    test(S("abcdefghij"), 9, 0, SV(""), 1, 0, S("can't happen"));
    test(S("abcdefghij"), 9, 0, SV("12345"), 0, 0, S("abcdefghij"));
    test(S("abcdefghij"), 9, 0, SV("12345"), 0, 1, S("abcdefghi1j"));
    test(S("abcdefghij"), 9, 0, SV("12345"), 0, 2, S("abcdefghi12j"));
    test(S("abcdefghij"), 9, 0, SV("12345"), 0, 4, S("abcdefghi1234j"));
    test(S("abcdefghij"), 9, 0, SV("12345"), 0, 5, S("abcdefghi12345j"));
    test(S("abcdefghij"), 9, 0, SV("12345"), 0, 6, S("abcdefghi12345j"));
    test(S("abcdefghij"), 9, 0, SV("12345"), 1, 0, S("abcdefghij"));
    test(S("abcdefghij"), 9, 0, SV("12345"), 1, 1, S("abcdefghi2j"));
    test(S("abcdefghij"), 9, 0, SV("12345"), 1, 2, S("abcdefghi23j"));
    test(S("abcdefghij"), 9, 0, SV("12345"), 1, 3, S("abcdefghi234j"));
    test(S("abcdefghij"), 9, 0, SV("12345"), 1, 4, S("abcdefghi2345j"));
    test(S("abcdefghij"), 9, 0, SV("12345"), 1, 5, S("abcdefghi2345j"));
    test(S("abcdefghij"), 9, 0, SV("12345"), 2, 0, S("abcdefghij"));
    test(S("abcdefghij"), 9, 0, SV("12345"), 2, 1, S("abcdefghi3j"));
    test(S("abcdefghij"), 9, 0, SV("12345"), 2, 2, S("abcdefghi34j"));
    test(S("abcdefghij"), 9, 0, SV("12345"), 2, 3, S("abcdefghi345j"));
    test(S("abcdefghij"), 9, 0, SV("12345"), 2, 4, S("abcdefghi345j"));
    test(S("abcdefghij"), 9, 0, SV("12345"), 4, 0, S("abcdefghij"));
    test(S("abcdefghij"), 9, 0, SV("12345"), 4, 1, S("abcdefghi5j"));
    test(S("abcdefghij"), 9, 0, SV("12345"), 4, 2, S("abcdefghi5j"));
    test(S("abcdefghij"), 9, 0, SV("12345"), 5, 0, S("abcdefghij"));
    test(S("abcdefghij"), 9, 0, SV("12345"), 5, 1, S("abcdefghij"));
    test(S("abcdefghij"), 9, 0, SV("12345"), 6, 0, S("can't happen"));
    test(S("abcdefghij"), 9, 0, SV("1234567890"), 0, 0, S("abcdefghij"));
    test(S("abcdefghij"), 9, 0, SV("1234567890"), 0, 1, S("abcdefghi1j"));
    test(S("abcdefghij"), 9, 0, SV("1234567890"), 0, 5, S("abcdefghi12345j"));
    test(S("abcdefghij"), 9, 0, SV("1234567890"), 0, 9, S("abcdefghi123456789j"));
    test(S("abcdefghij"), 9, 0, SV("1234567890"), 0, 10, S("abcdefghi1234567890j"));
    test(S("abcdefghij"), 9, 0, SV("1234567890"), 0, 11, S("abcdefghi1234567890j"));
    test(S("abcdefghij"), 9, 0, SV("1234567890"), 1, 0, S("abcdefghij"));
    test(S("abcdefghij"), 9, 0, SV("1234567890"), 1, 1, S("abcdefghi2j"));
    test(S("abcdefghij"), 9, 0, SV("1234567890"), 1, 4, S("abcdefghi2345j"));
    test(S("abcdefghij"), 9, 0, SV("1234567890"), 1, 8, S("abcdefghi23456789j"));
    test(S("abcdefghij"), 9, 0, SV("1234567890"), 1, 9, S("abcdefghi234567890j"));
    test(S("abcdefghij"), 9, 0, SV("1234567890"), 1, 10, S("abcdefghi234567890j"));
    test(S("abcdefghij"), 9, 0, SV("1234567890"), 5, 0, S("abcdefghij"));
    test(S("abcdefghij"), 9, 0, SV("1234567890"), 5, 1, S("abcdefghi6j"));
    test(S("abcdefghij"), 9, 0, SV("1234567890"), 5, 2, S("abcdefghi67j"));
    test(S("abcdefghij"), 9, 0, SV("1234567890"), 5, 4, S("abcdefghi6789j"));
    test(S("abcdefghij"), 9, 0, SV("1234567890"), 5, 5, S("abcdefghi67890j"));
    test(S("abcdefghij"), 9, 0, SV("1234567890"), 5, 6, S("abcdefghi67890j"));

    return true;
}

template <class S, class SV>
TEST_CONSTEXPR_CXX20 bool test33()
{
    test(S("abcdefghij"), 9, 0, SV("1234567890"), 9, 0, S("abcdefghij"));
    test(S("abcdefghij"), 9, 0, SV("1234567890"), 9, 1, S("abcdefghi0j"));
    test(S("abcdefghij"), 9, 0, SV("1234567890"), 9, 2, S("abcdefghi0j"));
    test(S("abcdefghij"), 9, 0, SV("1234567890"), 10, 0, S("abcdefghij"));
    test(S("abcdefghij"), 9, 0, SV("1234567890"), 10, 1, S("abcdefghij"));
    test(S("abcdefghij"), 9, 0, SV("1234567890"), 11, 0, S("can't happen"));
    test(S("abcdefghij"), 9, 0, SV("12345678901234567890"), 0, 0, S("abcdefghij"));
    test(S("abcdefghij"), 9, 0, SV("12345678901234567890"), 0, 1, S("abcdefghi1j"));
    test(S("abcdefghij"), 9, 0, SV("12345678901234567890"), 0, 10, S("abcdefghi1234567890j"));
    test(S("abcdefghij"), 9, 0, SV("12345678901234567890"), 0, 19, S("abcdefghi1234567890123456789j"));
    test(S("abcdefghij"), 9, 0, SV("12345678901234567890"), 0, 20, S("abcdefghi12345678901234567890j"));
    test(S("abcdefghij"), 9, 0, SV("12345678901234567890"), 0, 21, S("abcdefghi12345678901234567890j"));
    test(S("abcdefghij"), 9, 0, SV("12345678901234567890"), 1, 0, S("abcdefghij"));
    test(S("abcdefghij"), 9, 0, SV("12345678901234567890"), 1, 1, S("abcdefghi2j"));
    test(S("abcdefghij"), 9, 0, SV("12345678901234567890"), 1, 9, S("abcdefghi234567890j"));
    test(S("abcdefghij"), 9, 0, SV("12345678901234567890"), 1, 18, S("abcdefghi234567890123456789j"));
    test(S("abcdefghij"), 9, 0, SV("12345678901234567890"), 1, 19, S("abcdefghi2345678901234567890j"));
    test(S("abcdefghij"), 9, 0, SV("12345678901234567890"), 1, 20, S("abcdefghi2345678901234567890j"));
    test(S("abcdefghij"), 9, 0, SV("12345678901234567890"), 10, 0, S("abcdefghij"));
    test(S("abcdefghij"), 9, 0, SV("12345678901234567890"), 10, 1, S("abcdefghi1j"));
    test(S("abcdefghij"), 9, 0, SV("12345678901234567890"), 10, 5, S("abcdefghi12345j"));
    test(S("abcdefghij"), 9, 0, SV("12345678901234567890"), 10, 9, S("abcdefghi123456789j"));
    test(S("abcdefghij"), 9, 0, SV("12345678901234567890"), 10, 10, S("abcdefghi1234567890j"));
    test(S("abcdefghij"), 9, 0, SV("12345678901234567890"), 10, 11, S("abcdefghi1234567890j"));
    test(S("abcdefghij"), 9, 0, SV("12345678901234567890"), 19, 0, S("abcdefghij"));
    test(S("abcdefghij"), 9, 0, SV("12345678901234567890"), 19, 1, S("abcdefghi0j"));
    test(S("abcdefghij"), 9, 0, SV("12345678901234567890"), 19, 2, S("abcdefghi0j"));
    test(S("abcdefghij"), 9, 0, SV("12345678901234567890"), 20, 0, S("abcdefghij"));
    test(S("abcdefghij"), 9, 0, SV("12345678901234567890"), 20, 1, S("abcdefghij"));
    test(S("abcdefghij"), 9, 0, SV("12345678901234567890"), 21, 0, S("can't happen"));
    test(S("abcdefghij"), 9, 1, SV(""), 0, 0, S("abcdefghi"));
    test(S("abcdefghij"), 9, 1, SV(""), 0, 1, S("abcdefghi"));
    test(S("abcdefghij"), 9, 1, SV(""), 1, 0, S("can't happen"));
    test(S("abcdefghij"), 9, 1, SV("12345"), 0, 0, S("abcdefghi"));
    test(S("abcdefghij"), 9, 1, SV("12345"), 0, 1, S("abcdefghi1"));
    test(S("abcdefghij"), 9, 1, SV("12345"), 0, 2, S("abcdefghi12"));
    test(S("abcdefghij"), 9, 1, SV("12345"), 0, 4, S("abcdefghi1234"));
    test(S("abcdefghij"), 9, 1, SV("12345"), 0, 5, S("abcdefghi12345"));
    test(S("abcdefghij"), 9, 1, SV("12345"), 0, 6, S("abcdefghi12345"));
    test(S("abcdefghij"), 9, 1, SV("12345"), 1, 0, S("abcdefghi"));
    test(S("abcdefghij"), 9, 1, SV("12345"), 1, 1, S("abcdefghi2"));
    test(S("abcdefghij"), 9, 1, SV("12345"), 1, 2, S("abcdefghi23"));
    test(S("abcdefghij"), 9, 1, SV("12345"), 1, 3, S("abcdefghi234"));
    test(S("abcdefghij"), 9, 1, SV("12345"), 1, 4, S("abcdefghi2345"));
    test(S("abcdefghij"), 9, 1, SV("12345"), 1, 5, S("abcdefghi2345"));
    test(S("abcdefghij"), 9, 1, SV("12345"), 2, 0, S("abcdefghi"));
    test(S("abcdefghij"), 9, 1, SV("12345"), 2, 1, S("abcdefghi3"));
    test(S("abcdefghij"), 9, 1, SV("12345"), 2, 2, S("abcdefghi34"));
    test(S("abcdefghij"), 9, 1, SV("12345"), 2, 3, S("abcdefghi345"));
    test(S("abcdefghij"), 9, 1, SV("12345"), 2, 4, S("abcdefghi345"));
    test(S("abcdefghij"), 9, 1, SV("12345"), 4, 0, S("abcdefghi"));
    test(S("abcdefghij"), 9, 1, SV("12345"), 4, 1, S("abcdefghi5"));
    test(S("abcdefghij"), 9, 1, SV("12345"), 4, 2, S("abcdefghi5"));
    test(S("abcdefghij"), 9, 1, SV("12345"), 5, 0, S("abcdefghi"));
    test(S("abcdefghij"), 9, 1, SV("12345"), 5, 1, S("abcdefghi"));
    test(S("abcdefghij"), 9, 1, SV("12345"), 6, 0, S("can't happen"));
    test(S("abcdefghij"), 9, 1, SV("1234567890"), 0, 0, S("abcdefghi"));
    test(S("abcdefghij"), 9, 1, SV("1234567890"), 0, 1, S("abcdefghi1"));
    test(S("abcdefghij"), 9, 1, SV("1234567890"), 0, 5, S("abcdefghi12345"));
    test(S("abcdefghij"), 9, 1, SV("1234567890"), 0, 9, S("abcdefghi123456789"));
    test(S("abcdefghij"), 9, 1, SV("1234567890"), 0, 10, S("abcdefghi1234567890"));
    test(S("abcdefghij"), 9, 1, SV("1234567890"), 0, 11, S("abcdefghi1234567890"));
    test(S("abcdefghij"), 9, 1, SV("1234567890"), 1, 0, S("abcdefghi"));
    test(S("abcdefghij"), 9, 1, SV("1234567890"), 1, 1, S("abcdefghi2"));
    test(S("abcdefghij"), 9, 1, SV("1234567890"), 1, 4, S("abcdefghi2345"));
    test(S("abcdefghij"), 9, 1, SV("1234567890"), 1, 8, S("abcdefghi23456789"));
    test(S("abcdefghij"), 9, 1, SV("1234567890"), 1, 9, S("abcdefghi234567890"));
    test(S("abcdefghij"), 9, 1, SV("1234567890"), 1, 10, S("abcdefghi234567890"));
    test(S("abcdefghij"), 9, 1, SV("1234567890"), 5, 0, S("abcdefghi"));
    test(S("abcdefghij"), 9, 1, SV("1234567890"), 5, 1, S("abcdefghi6"));
    test(S("abcdefghij"), 9, 1, SV("1234567890"), 5, 2, S("abcdefghi67"));
    test(S("abcdefghij"), 9, 1, SV("1234567890"), 5, 4, S("abcdefghi6789"));
    test(S("abcdefghij"), 9, 1, SV("1234567890"), 5, 5, S("abcdefghi67890"));
    test(S("abcdefghij"), 9, 1, SV("1234567890"), 5, 6, S("abcdefghi67890"));
    test(S("abcdefghij"), 9, 1, SV("1234567890"), 9, 0, S("abcdefghi"));
    test(S("abcdefghij"), 9, 1, SV("1234567890"), 9, 1, S("abcdefghi0"));
    test(S("abcdefghij"), 9, 1, SV("1234567890"), 9, 2, S("abcdefghi0"));
    test(S("abcdefghij"), 9, 1, SV("1234567890"), 10, 0, S("abcdefghi"));
    test(S("abcdefghij"), 9, 1, SV("1234567890"), 10, 1, S("abcdefghi"));
    test(S("abcdefghij"), 9, 1, SV("1234567890"), 11, 0, S("can't happen"));
    test(S("abcdefghij"), 9, 1, SV("12345678901234567890"), 0, 0, S("abcdefghi"));
    test(S("abcdefghij"), 9, 1, SV("12345678901234567890"), 0, 1, S("abcdefghi1"));
    test(S("abcdefghij"), 9, 1, SV("12345678901234567890"), 0, 10, S("abcdefghi1234567890"));
    test(S("abcdefghij"), 9, 1, SV("12345678901234567890"), 0, 19, S("abcdefghi1234567890123456789"));
    test(S("abcdefghij"), 9, 1, SV("12345678901234567890"), 0, 20, S("abcdefghi12345678901234567890"));
    test(S("abcdefghij"), 9, 1, SV("12345678901234567890"), 0, 21, S("abcdefghi12345678901234567890"));
    test(S("abcdefghij"), 9, 1, SV("12345678901234567890"), 1, 0, S("abcdefghi"));
    test(S("abcdefghij"), 9, 1, SV("12345678901234567890"), 1, 1, S("abcdefghi2"));
    test(S("abcdefghij"), 9, 1, SV("12345678901234567890"), 1, 9, S("abcdefghi234567890"));
    test(S("abcdefghij"), 9, 1, SV("12345678901234567890"), 1, 18, S("abcdefghi234567890123456789"));
    test(S("abcdefghij"), 9, 1, SV("12345678901234567890"), 1, 19, S("abcdefghi2345678901234567890"));
    test(S("abcdefghij"), 9, 1, SV("12345678901234567890"), 1, 20, S("abcdefghi2345678901234567890"));
    test(S("abcdefghij"), 9, 1, SV("12345678901234567890"), 10, 0, S("abcdefghi"));
    test(S("abcdefghij"), 9, 1, SV("12345678901234567890"), 10, 1, S("abcdefghi1"));
    test(S("abcdefghij"), 9, 1, SV("12345678901234567890"), 10, 5, S("abcdefghi12345"));
    test(S("abcdefghij"), 9, 1, SV("12345678901234567890"), 10, 9, S("abcdefghi123456789"));
    test(S("abcdefghij"), 9, 1, SV("12345678901234567890"), 10, 10, S("abcdefghi1234567890"));
    test(S("abcdefghij"), 9, 1, SV("12345678901234567890"), 10, 11, S("abcdefghi1234567890"));
    test(S("abcdefghij"), 9, 1, SV("12345678901234567890"), 19, 0, S("abcdefghi"));
    test(S("abcdefghij"), 9, 1, SV("12345678901234567890"), 19, 1, S("abcdefghi0"));

    return true;
}

template <class S, class SV>
TEST_CONSTEXPR_CXX20 bool test34()
{
    test(S("abcdefghij"), 9, 1, SV("12345678901234567890"), 19, 2, S("abcdefghi0"));
    test(S("abcdefghij"), 9, 1, SV("12345678901234567890"), 20, 0, S("abcdefghi"));
    test(S("abcdefghij"), 9, 1, SV("12345678901234567890"), 20, 1, S("abcdefghi"));
    test(S("abcdefghij"), 9, 1, SV("12345678901234567890"), 21, 0, S("can't happen"));
    test(S("abcdefghij"), 9, 2, SV(""), 0, 0, S("abcdefghi"));
    test(S("abcdefghij"), 9, 2, SV(""), 0, 1, S("abcdefghi"));
    test(S("abcdefghij"), 9, 2, SV(""), 1, 0, S("can't happen"));
    test(S("abcdefghij"), 9, 2, SV("12345"), 0, 0, S("abcdefghi"));
    test(S("abcdefghij"), 9, 2, SV("12345"), 0, 1, S("abcdefghi1"));
    test(S("abcdefghij"), 9, 2, SV("12345"), 0, 2, S("abcdefghi12"));
    test(S("abcdefghij"), 9, 2, SV("12345"), 0, 4, S("abcdefghi1234"));
    test(S("abcdefghij"), 9, 2, SV("12345"), 0, 5, S("abcdefghi12345"));
    test(S("abcdefghij"), 9, 2, SV("12345"), 0, 6, S("abcdefghi12345"));
    test(S("abcdefghij"), 9, 2, SV("12345"), 1, 0, S("abcdefghi"));
    test(S("abcdefghij"), 9, 2, SV("12345"), 1, 1, S("abcdefghi2"));
    test(S("abcdefghij"), 9, 2, SV("12345"), 1, 2, S("abcdefghi23"));
    test(S("abcdefghij"), 9, 2, SV("12345"), 1, 3, S("abcdefghi234"));
    test(S("abcdefghij"), 9, 2, SV("12345"), 1, 4, S("abcdefghi2345"));
    test(S("abcdefghij"), 9, 2, SV("12345"), 1, 5, S("abcdefghi2345"));
    test(S("abcdefghij"), 9, 2, SV("12345"), 2, 0, S("abcdefghi"));
    test(S("abcdefghij"), 9, 2, SV("12345"), 2, 1, S("abcdefghi3"));
    test(S("abcdefghij"), 9, 2, SV("12345"), 2, 2, S("abcdefghi34"));
    test(S("abcdefghij"), 9, 2, SV("12345"), 2, 3, S("abcdefghi345"));
    test(S("abcdefghij"), 9, 2, SV("12345"), 2, 4, S("abcdefghi345"));
    test(S("abcdefghij"), 9, 2, SV("12345"), 4, 0, S("abcdefghi"));
    test(S("abcdefghij"), 9, 2, SV("12345"), 4, 1, S("abcdefghi5"));
    test(S("abcdefghij"), 9, 2, SV("12345"), 4, 2, S("abcdefghi5"));
    test(S("abcdefghij"), 9, 2, SV("12345"), 5, 0, S("abcdefghi"));
    test(S("abcdefghij"), 9, 2, SV("12345"), 5, 1, S("abcdefghi"));
    test(S("abcdefghij"), 9, 2, SV("12345"), 6, 0, S("can't happen"));
    test(S("abcdefghij"), 9, 2, SV("1234567890"), 0, 0, S("abcdefghi"));
    test(S("abcdefghij"), 9, 2, SV("1234567890"), 0, 1, S("abcdefghi1"));
    test(S("abcdefghij"), 9, 2, SV("1234567890"), 0, 5, S("abcdefghi12345"));
    test(S("abcdefghij"), 9, 2, SV("1234567890"), 0, 9, S("abcdefghi123456789"));
    test(S("abcdefghij"), 9, 2, SV("1234567890"), 0, 10, S("abcdefghi1234567890"));
    test(S("abcdefghij"), 9, 2, SV("1234567890"), 0, 11, S("abcdefghi1234567890"));
    test(S("abcdefghij"), 9, 2, SV("1234567890"), 1, 0, S("abcdefghi"));
    test(S("abcdefghij"), 9, 2, SV("1234567890"), 1, 1, S("abcdefghi2"));
    test(S("abcdefghij"), 9, 2, SV("1234567890"), 1, 4, S("abcdefghi2345"));
    test(S("abcdefghij"), 9, 2, SV("1234567890"), 1, 8, S("abcdefghi23456789"));
    test(S("abcdefghij"), 9, 2, SV("1234567890"), 1, 9, S("abcdefghi234567890"));
    test(S("abcdefghij"), 9, 2, SV("1234567890"), 1, 10, S("abcdefghi234567890"));
    test(S("abcdefghij"), 9, 2, SV("1234567890"), 5, 0, S("abcdefghi"));
    test(S("abcdefghij"), 9, 2, SV("1234567890"), 5, 1, S("abcdefghi6"));
    test(S("abcdefghij"), 9, 2, SV("1234567890"), 5, 2, S("abcdefghi67"));
    test(S("abcdefghij"), 9, 2, SV("1234567890"), 5, 4, S("abcdefghi6789"));
    test(S("abcdefghij"), 9, 2, SV("1234567890"), 5, 5, S("abcdefghi67890"));
    test(S("abcdefghij"), 9, 2, SV("1234567890"), 5, 6, S("abcdefghi67890"));
    test(S("abcdefghij"), 9, 2, SV("1234567890"), 9, 0, S("abcdefghi"));
    test(S("abcdefghij"), 9, 2, SV("1234567890"), 9, 1, S("abcdefghi0"));
    test(S("abcdefghij"), 9, 2, SV("1234567890"), 9, 2, S("abcdefghi0"));
    test(S("abcdefghij"), 9, 2, SV("1234567890"), 10, 0, S("abcdefghi"));
    test(S("abcdefghij"), 9, 2, SV("1234567890"), 10, 1, S("abcdefghi"));
    test(S("abcdefghij"), 9, 2, SV("1234567890"), 11, 0, S("can't happen"));
    test(S("abcdefghij"), 9, 2, SV("12345678901234567890"), 0, 0, S("abcdefghi"));
    test(S("abcdefghij"), 9, 2, SV("12345678901234567890"), 0, 1, S("abcdefghi1"));
    test(S("abcdefghij"), 9, 2, SV("12345678901234567890"), 0, 10, S("abcdefghi1234567890"));
    test(S("abcdefghij"), 9, 2, SV("12345678901234567890"), 0, 19, S("abcdefghi1234567890123456789"));
    test(S("abcdefghij"), 9, 2, SV("12345678901234567890"), 0, 20, S("abcdefghi12345678901234567890"));
    test(S("abcdefghij"), 9, 2, SV("12345678901234567890"), 0, 21, S("abcdefghi12345678901234567890"));
    test(S("abcdefghij"), 9, 2, SV("12345678901234567890"), 1, 0, S("abcdefghi"));
    test(S("abcdefghij"), 9, 2, SV("12345678901234567890"), 1, 1, S("abcdefghi2"));
    test(S("abcdefghij"), 9, 2, SV("12345678901234567890"), 1, 9, S("abcdefghi234567890"));
    test(S("abcdefghij"), 9, 2, SV("12345678901234567890"), 1, 18, S("abcdefghi234567890123456789"));
    test(S("abcdefghij"), 9, 2, SV("12345678901234567890"), 1, 19, S("abcdefghi2345678901234567890"));
    test(S("abcdefghij"), 9, 2, SV("12345678901234567890"), 1, 20, S("abcdefghi2345678901234567890"));
    test(S("abcdefghij"), 9, 2, SV("12345678901234567890"), 10, 0, S("abcdefghi"));
    test(S("abcdefghij"), 9, 2, SV("12345678901234567890"), 10, 1, S("abcdefghi1"));
    test(S("abcdefghij"), 9, 2, SV("12345678901234567890"), 10, 5, S("abcdefghi12345"));
    test(S("abcdefghij"), 9, 2, SV("12345678901234567890"), 10, 9, S("abcdefghi123456789"));
    test(S("abcdefghij"), 9, 2, SV("12345678901234567890"), 10, 10, S("abcdefghi1234567890"));
    test(S("abcdefghij"), 9, 2, SV("12345678901234567890"), 10, 11, S("abcdefghi1234567890"));
    test(S("abcdefghij"), 9, 2, SV("12345678901234567890"), 19, 0, S("abcdefghi"));
    test(S("abcdefghij"), 9, 2, SV("12345678901234567890"), 19, 1, S("abcdefghi0"));
    test(S("abcdefghij"), 9, 2, SV("12345678901234567890"), 19, 2, S("abcdefghi0"));
    test(S("abcdefghij"), 9, 2, SV("12345678901234567890"), 20, 0, S("abcdefghi"));
    test(S("abcdefghij"), 9, 2, SV("12345678901234567890"), 20, 1, S("abcdefghi"));
    test(S("abcdefghij"), 9, 2, SV("12345678901234567890"), 21, 0, S("can't happen"));
    test(S("abcdefghij"), 10, 0, SV(""), 0, 0, S("abcdefghij"));
    test(S("abcdefghij"), 10, 0, SV(""), 0, 1, S("abcdefghij"));
    test(S("abcdefghij"), 10, 0, SV(""), 1, 0, S("can't happen"));
    test(S("abcdefghij"), 10, 0, SV("12345"), 0, 0, S("abcdefghij"));
    test(S("abcdefghij"), 10, 0, SV("12345"), 0, 1, S("abcdefghij1"));
    test(S("abcdefghij"), 10, 0, SV("12345"), 0, 2, S("abcdefghij12"));
    test(S("abcdefghij"), 10, 0, SV("12345"), 0, 4, S("abcdefghij1234"));
    test(S("abcdefghij"), 10, 0, SV("12345"), 0, 5, S("abcdefghij12345"));
    test(S("abcdefghij"), 10, 0, SV("12345"), 0, 6, S("abcdefghij12345"));
    test(S("abcdefghij"), 10, 0, SV("12345"), 1, 0, S("abcdefghij"));
    test(S("abcdefghij"), 10, 0, SV("12345"), 1, 1, S("abcdefghij2"));
    test(S("abcdefghij"), 10, 0, SV("12345"), 1, 2, S("abcdefghij23"));
    test(S("abcdefghij"), 10, 0, SV("12345"), 1, 3, S("abcdefghij234"));
    test(S("abcdefghij"), 10, 0, SV("12345"), 1, 4, S("abcdefghij2345"));
    test(S("abcdefghij"), 10, 0, SV("12345"), 1, 5, S("abcdefghij2345"));
    test(S("abcdefghij"), 10, 0, SV("12345"), 2, 0, S("abcdefghij"));
    test(S("abcdefghij"), 10, 0, SV("12345"), 2, 1, S("abcdefghij3"));
    test(S("abcdefghij"), 10, 0, SV("12345"), 2, 2, S("abcdefghij34"));
    test(S("abcdefghij"), 10, 0, SV("12345"), 2, 3, S("abcdefghij345"));
    test(S("abcdefghij"), 10, 0, SV("12345"), 2, 4, S("abcdefghij345"));
    test(S("abcdefghij"), 10, 0, SV("12345"), 4, 0, S("abcdefghij"));
    test(S("abcdefghij"), 10, 0, SV("12345"), 4, 1, S("abcdefghij5"));

    return true;
}

template <class S, class SV>
TEST_CONSTEXPR_CXX20 bool test35()
{
    test(S("abcdefghij"), 10, 0, SV("12345"), 4, 2, S("abcdefghij5"));
    test(S("abcdefghij"), 10, 0, SV("12345"), 5, 0, S("abcdefghij"));
    test(S("abcdefghij"), 10, 0, SV("12345"), 5, 1, S("abcdefghij"));
    test(S("abcdefghij"), 10, 0, SV("12345"), 6, 0, S("can't happen"));
    test(S("abcdefghij"), 10, 0, SV("1234567890"), 0, 0, S("abcdefghij"));
    test(S("abcdefghij"), 10, 0, SV("1234567890"), 0, 1, S("abcdefghij1"));
    test(S("abcdefghij"), 10, 0, SV("1234567890"), 0, 5, S("abcdefghij12345"));
    test(S("abcdefghij"), 10, 0, SV("1234567890"), 0, 9, S("abcdefghij123456789"));
    test(S("abcdefghij"), 10, 0, SV("1234567890"), 0, 10, S("abcdefghij1234567890"));
    test(S("abcdefghij"), 10, 0, SV("1234567890"), 0, 11, S("abcdefghij1234567890"));
    test(S("abcdefghij"), 10, 0, SV("1234567890"), 1, 0, S("abcdefghij"));
    test(S("abcdefghij"), 10, 0, SV("1234567890"), 1, 1, S("abcdefghij2"));
    test(S("abcdefghij"), 10, 0, SV("1234567890"), 1, 4, S("abcdefghij2345"));
    test(S("abcdefghij"), 10, 0, SV("1234567890"), 1, 8, S("abcdefghij23456789"));
    test(S("abcdefghij"), 10, 0, SV("1234567890"), 1, 9, S("abcdefghij234567890"));
    test(S("abcdefghij"), 10, 0, SV("1234567890"), 1, 10, S("abcdefghij234567890"));
    test(S("abcdefghij"), 10, 0, SV("1234567890"), 5, 0, S("abcdefghij"));
    test(S("abcdefghij"), 10, 0, SV("1234567890"), 5, 1, S("abcdefghij6"));
    test(S("abcdefghij"), 10, 0, SV("1234567890"), 5, 2, S("abcdefghij67"));
    test(S("abcdefghij"), 10, 0, SV("1234567890"), 5, 4, S("abcdefghij6789"));
    test(S("abcdefghij"), 10, 0, SV("1234567890"), 5, 5, S("abcdefghij67890"));
    test(S("abcdefghij"), 10, 0, SV("1234567890"), 5, 6, S("abcdefghij67890"));
    test(S("abcdefghij"), 10, 0, SV("1234567890"), 9, 0, S("abcdefghij"));
    test(S("abcdefghij"), 10, 0, SV("1234567890"), 9, 1, S("abcdefghij0"));
    test(S("abcdefghij"), 10, 0, SV("1234567890"), 9, 2, S("abcdefghij0"));
    test(S("abcdefghij"), 10, 0, SV("1234567890"), 10, 0, S("abcdefghij"));
    test(S("abcdefghij"), 10, 0, SV("1234567890"), 10, 1, S("abcdefghij"));
    test(S("abcdefghij"), 10, 0, SV("1234567890"), 11, 0, S("can't happen"));
    test(S("abcdefghij"), 10, 0, SV("12345678901234567890"), 0, 0, S("abcdefghij"));
    test(S("abcdefghij"), 10, 0, SV("12345678901234567890"), 0, 1, S("abcdefghij1"));
    test(S("abcdefghij"), 10, 0, SV("12345678901234567890"), 0, 10, S("abcdefghij1234567890"));
    test(S("abcdefghij"), 10, 0, SV("12345678901234567890"), 0, 19, S("abcdefghij1234567890123456789"));
    test(S("abcdefghij"), 10, 0, SV("12345678901234567890"), 0, 20, S("abcdefghij12345678901234567890"));
    test(S("abcdefghij"), 10, 0, SV("12345678901234567890"), 0, 21, S("abcdefghij12345678901234567890"));
    test(S("abcdefghij"), 10, 0, SV("12345678901234567890"), 1, 0, S("abcdefghij"));
    test(S("abcdefghij"), 10, 0, SV("12345678901234567890"), 1, 1, S("abcdefghij2"));
    test(S("abcdefghij"), 10, 0, SV("12345678901234567890"), 1, 9, S("abcdefghij234567890"));
    test(S("abcdefghij"), 10, 0, SV("12345678901234567890"), 1, 18, S("abcdefghij234567890123456789"));
    test(S("abcdefghij"), 10, 0, SV("12345678901234567890"), 1, 19, S("abcdefghij2345678901234567890"));
    test(S("abcdefghij"), 10, 0, SV("12345678901234567890"), 1, 20, S("abcdefghij2345678901234567890"));
    test(S("abcdefghij"), 10, 0, SV("12345678901234567890"), 10, 0, S("abcdefghij"));
    test(S("abcdefghij"), 10, 0, SV("12345678901234567890"), 10, 1, S("abcdefghij1"));
    test(S("abcdefghij"), 10, 0, SV("12345678901234567890"), 10, 5, S("abcdefghij12345"));
    test(S("abcdefghij"), 10, 0, SV("12345678901234567890"), 10, 9, S("abcdefghij123456789"));
    test(S("abcdefghij"), 10, 0, SV("12345678901234567890"), 10, 10, S("abcdefghij1234567890"));
    test(S("abcdefghij"), 10, 0, SV("12345678901234567890"), 10, 11, S("abcdefghij1234567890"));
    test(S("abcdefghij"), 10, 0, SV("12345678901234567890"), 19, 0, S("abcdefghij"));
    test(S("abcdefghij"), 10, 0, SV("12345678901234567890"), 19, 1, S("abcdefghij0"));
    test(S("abcdefghij"), 10, 0, SV("12345678901234567890"), 19, 2, S("abcdefghij0"));
    test(S("abcdefghij"), 10, 0, SV("12345678901234567890"), 20, 0, S("abcdefghij"));
    test(S("abcdefghij"), 10, 0, SV("12345678901234567890"), 20, 1, S("abcdefghij"));
    test(S("abcdefghij"), 10, 0, SV("12345678901234567890"), 21, 0, S("can't happen"));
    test(S("abcdefghij"), 10, 1, SV(""), 0, 0, S("abcdefghij"));
    test(S("abcdefghij"), 10, 1, SV(""), 0, 1, S("abcdefghij"));
    test(S("abcdefghij"), 10, 1, SV(""), 1, 0, S("can't happen"));
    test(S("abcdefghij"), 10, 1, SV("12345"), 0, 0, S("abcdefghij"));
    test(S("abcdefghij"), 10, 1, SV("12345"), 0, 1, S("abcdefghij1"));
    test(S("abcdefghij"), 10, 1, SV("12345"), 0, 2, S("abcdefghij12"));
    test(S("abcdefghij"), 10, 1, SV("12345"), 0, 4, S("abcdefghij1234"));
    test(S("abcdefghij"), 10, 1, SV("12345"), 0, 5, S("abcdefghij12345"));
    test(S("abcdefghij"), 10, 1, SV("12345"), 0, 6, S("abcdefghij12345"));
    test(S("abcdefghij"), 10, 1, SV("12345"), 1, 0, S("abcdefghij"));
    test(S("abcdefghij"), 10, 1, SV("12345"), 1, 1, S("abcdefghij2"));
    test(S("abcdefghij"), 10, 1, SV("12345"), 1, 2, S("abcdefghij23"));
    test(S("abcdefghij"), 10, 1, SV("12345"), 1, 3, S("abcdefghij234"));
    test(S("abcdefghij"), 10, 1, SV("12345"), 1, 4, S("abcdefghij2345"));
    test(S("abcdefghij"), 10, 1, SV("12345"), 1, 5, S("abcdefghij2345"));
    test(S("abcdefghij"), 10, 1, SV("12345"), 2, 0, S("abcdefghij"));
    test(S("abcdefghij"), 10, 1, SV("12345"), 2, 1, S("abcdefghij3"));
    test(S("abcdefghij"), 10, 1, SV("12345"), 2, 2, S("abcdefghij34"));
    test(S("abcdefghij"), 10, 1, SV("12345"), 2, 3, S("abcdefghij345"));
    test(S("abcdefghij"), 10, 1, SV("12345"), 2, 4, S("abcdefghij345"));
    test(S("abcdefghij"), 10, 1, SV("12345"), 4, 0, S("abcdefghij"));
    test(S("abcdefghij"), 10, 1, SV("12345"), 4, 1, S("abcdefghij5"));
    test(S("abcdefghij"), 10, 1, SV("12345"), 4, 2, S("abcdefghij5"));
    test(S("abcdefghij"), 10, 1, SV("12345"), 5, 0, S("abcdefghij"));
    test(S("abcdefghij"), 10, 1, SV("12345"), 5, 1, S("abcdefghij"));
    test(S("abcdefghij"), 10, 1, SV("12345"), 6, 0, S("can't happen"));
    test(S("abcdefghij"), 10, 1, SV("1234567890"), 0, 0, S("abcdefghij"));
    test(S("abcdefghij"), 10, 1, SV("1234567890"), 0, 1, S("abcdefghij1"));
    test(S("abcdefghij"), 10, 1, SV("1234567890"), 0, 5, S("abcdefghij12345"));
    test(S("abcdefghij"), 10, 1, SV("1234567890"), 0, 9, S("abcdefghij123456789"));
    test(S("abcdefghij"), 10, 1, SV("1234567890"), 0, 10, S("abcdefghij1234567890"));
    test(S("abcdefghij"), 10, 1, SV("1234567890"), 0, 11, S("abcdefghij1234567890"));
    test(S("abcdefghij"), 10, 1, SV("1234567890"), 1, 0, S("abcdefghij"));
    test(S("abcdefghij"), 10, 1, SV("1234567890"), 1, 1, S("abcdefghij2"));
    test(S("abcdefghij"), 10, 1, SV("1234567890"), 1, 4, S("abcdefghij2345"));
    test(S("abcdefghij"), 10, 1, SV("1234567890"), 1, 8, S("abcdefghij23456789"));
    test(S("abcdefghij"), 10, 1, SV("1234567890"), 1, 9, S("abcdefghij234567890"));
    test(S("abcdefghij"), 10, 1, SV("1234567890"), 1, 10, S("abcdefghij234567890"));
    test(S("abcdefghij"), 10, 1, SV("1234567890"), 5, 0, S("abcdefghij"));
    test(S("abcdefghij"), 10, 1, SV("1234567890"), 5, 1, S("abcdefghij6"));
    test(S("abcdefghij"), 10, 1, SV("1234567890"), 5, 2, S("abcdefghij67"));
    test(S("abcdefghij"), 10, 1, SV("1234567890"), 5, 4, S("abcdefghij6789"));
    test(S("abcdefghij"), 10, 1, SV("1234567890"), 5, 5, S("abcdefghij67890"));
    test(S("abcdefghij"), 10, 1, SV("1234567890"), 5, 6, S("abcdefghij67890"));
    test(S("abcdefghij"), 10, 1, SV("1234567890"), 9, 0, S("abcdefghij"));
    test(S("abcdefghij"), 10, 1, SV("1234567890"), 9, 1, S("abcdefghij0"));
    test(S("abcdefghij"), 10, 1, SV("1234567890"), 9, 2, S("abcdefghij0"));
    test(S("abcdefghij"), 10, 1, SV("1234567890"), 10, 0, S("abcdefghij"));

    return true;
}

template <class S, class SV>
TEST_CONSTEXPR_CXX20 bool test36()
{
    test(S("abcdefghij"), 10, 1, SV("1234567890"), 10, 1, S("abcdefghij"));
    test(S("abcdefghij"), 10, 1, SV("1234567890"), 11, 0, S("can't happen"));
    test(S("abcdefghij"), 10, 1, SV("12345678901234567890"), 0, 0, S("abcdefghij"));
    test(S("abcdefghij"), 10, 1, SV("12345678901234567890"), 0, 1, S("abcdefghij1"));
    test(S("abcdefghij"), 10, 1, SV("12345678901234567890"), 0, 10, S("abcdefghij1234567890"));
    test(S("abcdefghij"), 10, 1, SV("12345678901234567890"), 0, 19, S("abcdefghij1234567890123456789"));
    test(S("abcdefghij"), 10, 1, SV("12345678901234567890"), 0, 20, S("abcdefghij12345678901234567890"));
    test(S("abcdefghij"), 10, 1, SV("12345678901234567890"), 0, 21, S("abcdefghij12345678901234567890"));
    test(S("abcdefghij"), 10, 1, SV("12345678901234567890"), 1, 0, S("abcdefghij"));
    test(S("abcdefghij"), 10, 1, SV("12345678901234567890"), 1, 1, S("abcdefghij2"));
    test(S("abcdefghij"), 10, 1, SV("12345678901234567890"), 1, 9, S("abcdefghij234567890"));
    test(S("abcdefghij"), 10, 1, SV("12345678901234567890"), 1, 18, S("abcdefghij234567890123456789"));
    test(S("abcdefghij"), 10, 1, SV("12345678901234567890"), 1, 19, S("abcdefghij2345678901234567890"));
    test(S("abcdefghij"), 10, 1, SV("12345678901234567890"), 1, 20, S("abcdefghij2345678901234567890"));
    test(S("abcdefghij"), 10, 1, SV("12345678901234567890"), 10, 0, S("abcdefghij"));
    test(S("abcdefghij"), 10, 1, SV("12345678901234567890"), 10, 1, S("abcdefghij1"));
    test(S("abcdefghij"), 10, 1, SV("12345678901234567890"), 10, 5, S("abcdefghij12345"));
    test(S("abcdefghij"), 10, 1, SV("12345678901234567890"), 10, 9, S("abcdefghij123456789"));
    test(S("abcdefghij"), 10, 1, SV("12345678901234567890"), 10, 10, S("abcdefghij1234567890"));
    test(S("abcdefghij"), 10, 1, SV("12345678901234567890"), 10, 11, S("abcdefghij1234567890"));
    test(S("abcdefghij"), 10, 1, SV("12345678901234567890"), 19, 0, S("abcdefghij"));
    test(S("abcdefghij"), 10, 1, SV("12345678901234567890"), 19, 1, S("abcdefghij0"));
    test(S("abcdefghij"), 10, 1, SV("12345678901234567890"), 19, 2, S("abcdefghij0"));
    test(S("abcdefghij"), 10, 1, SV("12345678901234567890"), 20, 0, S("abcdefghij"));
    test(S("abcdefghij"), 10, 1, SV("12345678901234567890"), 20, 1, S("abcdefghij"));
    test(S("abcdefghij"), 10, 1, SV("12345678901234567890"), 21, 0, S("can't happen"));
    test(S("abcdefghij"), 11, 0, SV(""), 0, 0, S("can't happen"));
    test(S("abcdefghij"), 11, 0, SV(""), 0, 1, S("can't happen"));
    test(S("abcdefghij"), 11, 0, SV(""), 1, 0, S("can't happen"));
    test(S("abcdefghij"), 11, 0, SV("12345"), 0, 0, S("can't happen"));
    test(S("abcdefghij"), 11, 0, SV("12345"), 0, 1, S("can't happen"));
    test(S("abcdefghij"), 11, 0, SV("12345"), 0, 2, S("can't happen"));
    test(S("abcdefghij"), 11, 0, SV("12345"), 0, 4, S("can't happen"));
    test(S("abcdefghij"), 11, 0, SV("12345"), 0, 5, S("can't happen"));
    test(S("abcdefghij"), 11, 0, SV("12345"), 0, 6, S("can't happen"));
    test(S("abcdefghij"), 11, 0, SV("12345"), 1, 0, S("can't happen"));
    test(S("abcdefghij"), 11, 0, SV("12345"), 1, 1, S("can't happen"));
    test(S("abcdefghij"), 11, 0, SV("12345"), 1, 2, S("can't happen"));
    test(S("abcdefghij"), 11, 0, SV("12345"), 1, 3, S("can't happen"));
    test(S("abcdefghij"), 11, 0, SV("12345"), 1, 4, S("can't happen"));
    test(S("abcdefghij"), 11, 0, SV("12345"), 1, 5, S("can't happen"));
    test(S("abcdefghij"), 11, 0, SV("12345"), 2, 0, S("can't happen"));
    test(S("abcdefghij"), 11, 0, SV("12345"), 2, 1, S("can't happen"));
    test(S("abcdefghij"), 11, 0, SV("12345"), 2, 2, S("can't happen"));
    test(S("abcdefghij"), 11, 0, SV("12345"), 2, 3, S("can't happen"));
    test(S("abcdefghij"), 11, 0, SV("12345"), 2, 4, S("can't happen"));
    test(S("abcdefghij"), 11, 0, SV("12345"), 4, 0, S("can't happen"));
    test(S("abcdefghij"), 11, 0, SV("12345"), 4, 1, S("can't happen"));
    test(S("abcdefghij"), 11, 0, SV("12345"), 4, 2, S("can't happen"));
    test(S("abcdefghij"), 11, 0, SV("12345"), 5, 0, S("can't happen"));
    test(S("abcdefghij"), 11, 0, SV("12345"), 5, 1, S("can't happen"));
    test(S("abcdefghij"), 11, 0, SV("12345"), 6, 0, S("can't happen"));
    test(S("abcdefghij"), 11, 0, SV("1234567890"), 0, 0, S("can't happen"));
    test(S("abcdefghij"), 11, 0, SV("1234567890"), 0, 1, S("can't happen"));
    test(S("abcdefghij"), 11, 0, SV("1234567890"), 0, 5, S("can't happen"));
    test(S("abcdefghij"), 11, 0, SV("1234567890"), 0, 9, S("can't happen"));
    test(S("abcdefghij"), 11, 0, SV("1234567890"), 0, 10, S("can't happen"));
    test(S("abcdefghij"), 11, 0, SV("1234567890"), 0, 11, S("can't happen"));
    test(S("abcdefghij"), 11, 0, SV("1234567890"), 1, 0, S("can't happen"));
    test(S("abcdefghij"), 11, 0, SV("1234567890"), 1, 1, S("can't happen"));
    test(S("abcdefghij"), 11, 0, SV("1234567890"), 1, 4, S("can't happen"));
    test(S("abcdefghij"), 11, 0, SV("1234567890"), 1, 8, S("can't happen"));
    test(S("abcdefghij"), 11, 0, SV("1234567890"), 1, 9, S("can't happen"));
    test(S("abcdefghij"), 11, 0, SV("1234567890"), 1, 10, S("can't happen"));
    test(S("abcdefghij"), 11, 0, SV("1234567890"), 5, 0, S("can't happen"));
    test(S("abcdefghij"), 11, 0, SV("1234567890"), 5, 1, S("can't happen"));
    test(S("abcdefghij"), 11, 0, SV("1234567890"), 5, 2, S("can't happen"));
    test(S("abcdefghij"), 11, 0, SV("1234567890"), 5, 4, S("can't happen"));
    test(S("abcdefghij"), 11, 0, SV("1234567890"), 5, 5, S("can't happen"));
    test(S("abcdefghij"), 11, 0, SV("1234567890"), 5, 6, S("can't happen"));
    test(S("abcdefghij"), 11, 0, SV("1234567890"), 9, 0, S("can't happen"));
    test(S("abcdefghij"), 11, 0, SV("1234567890"), 9, 1, S("can't happen"));
    test(S("abcdefghij"), 11, 0, SV("1234567890"), 9, 2, S("can't happen"));
    test(S("abcdefghij"), 11, 0, SV("1234567890"), 10, 0, S("can't happen"));
    test(S("abcdefghij"), 11, 0, SV("1234567890"), 10, 1, S("can't happen"));
    test(S("abcdefghij"), 11, 0, SV("1234567890"), 11, 0, S("can't happen"));
    test(S("abcdefghij"), 11, 0, SV("12345678901234567890"), 0, 0, S("can't happen"));
    test(S("abcdefghij"), 11, 0, SV("12345678901234567890"), 0, 1, S("can't happen"));
    test(S("abcdefghij"), 11, 0, SV("12345678901234567890"), 0, 10, S("can't happen"));
    test(S("abcdefghij"), 11, 0, SV("12345678901234567890"), 0, 19, S("can't happen"));
    test(S("abcdefghij"), 11, 0, SV("12345678901234567890"), 0, 20, S("can't happen"));
    test(S("abcdefghij"), 11, 0, SV("12345678901234567890"), 0, 21, S("can't happen"));
    test(S("abcdefghij"), 11, 0, SV("12345678901234567890"), 1, 0, S("can't happen"));
    test(S("abcdefghij"), 11, 0, SV("12345678901234567890"), 1, 1, S("can't happen"));
    test(S("abcdefghij"), 11, 0, SV("12345678901234567890"), 1, 9, S("can't happen"));
    test(S("abcdefghij"), 11, 0, SV("12345678901234567890"), 1, 18, S("can't happen"));
    test(S("abcdefghij"), 11, 0, SV("12345678901234567890"), 1, 19, S("can't happen"));
    test(S("abcdefghij"), 11, 0, SV("12345678901234567890"), 1, 20, S("can't happen"));
    test(S("abcdefghij"), 11, 0, SV("12345678901234567890"), 10, 0, S("can't happen"));
    test(S("abcdefghij"), 11, 0, SV("12345678901234567890"), 10, 1, S("can't happen"));
    test(S("abcdefghij"), 11, 0, SV("12345678901234567890"), 10, 5, S("can't happen"));
    test(S("abcdefghij"), 11, 0, SV("12345678901234567890"), 10, 9, S("can't happen"));
    test(S("abcdefghij"), 11, 0, SV("12345678901234567890"), 10, 10, S("can't happen"));
    test(S("abcdefghij"), 11, 0, SV("12345678901234567890"), 10, 11, S("can't happen"));
    test(S("abcdefghij"), 11, 0, SV("12345678901234567890"), 19, 0, S("can't happen"));
    test(S("abcdefghij"), 11, 0, SV("12345678901234567890"), 19, 1, S("can't happen"));
    test(S("abcdefghij"), 11, 0, SV("12345678901234567890"), 19, 2, S("can't happen"));
    test(S("abcdefghij"), 11, 0, SV("12345678901234567890"), 20, 0, S("can't happen"));
    test(S("abcdefghij"), 11, 0, SV("12345678901234567890"), 20, 1, S("can't happen"));
    test(S("abcdefghij"), 11, 0, SV("12345678901234567890"), 21, 0, S("can't happen"));

    return true;
}

template <class S, class SV>
TEST_CONSTEXPR_CXX20 bool test37()
{
    test(S("abcdefghijklmnopqrst"), 0, 0, SV(""), 0, 0, S("abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 0, SV(""), 0, 1, S("abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 0, SV(""), 1, 0, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 0, 0, SV("12345"), 0, 0, S("abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 0, SV("12345"), 0, 1, S("1abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 0, SV("12345"), 0, 2, S("12abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 0, SV("12345"), 0, 4, S("1234abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 0, SV("12345"), 0, 5, S("12345abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 0, SV("12345"), 0, 6, S("12345abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 0, SV("12345"), 1, 0, S("abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 0, SV("12345"), 1, 1, S("2abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 0, SV("12345"), 1, 2, S("23abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 0, SV("12345"), 1, 3, S("234abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 0, SV("12345"), 1, 4, S("2345abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 0, SV("12345"), 1, 5, S("2345abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 0, SV("12345"), 2, 0, S("abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 0, SV("12345"), 2, 1, S("3abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 0, SV("12345"), 2, 2, S("34abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 0, SV("12345"), 2, 3, S("345abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 0, SV("12345"), 2, 4, S("345abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 0, SV("12345"), 4, 0, S("abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 0, SV("12345"), 4, 1, S("5abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 0, SV("12345"), 4, 2, S("5abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 0, SV("12345"), 5, 0, S("abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 0, SV("12345"), 5, 1, S("abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 0, SV("12345"), 6, 0, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 0, 0, SV("1234567890"), 0, 0, S("abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 0, SV("1234567890"), 0, 1, S("1abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 0, SV("1234567890"), 0, 5, S("12345abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 0, SV("1234567890"), 0, 9, S("123456789abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 0, SV("1234567890"), 0, 10, S("1234567890abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 0, SV("1234567890"), 0, 11, S("1234567890abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 0, SV("1234567890"), 1, 0, S("abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 0, SV("1234567890"), 1, 1, S("2abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 0, SV("1234567890"), 1, 4, S("2345abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 0, SV("1234567890"), 1, 8, S("23456789abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 0, SV("1234567890"), 1, 9, S("234567890abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 0, SV("1234567890"), 1, 10, S("234567890abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 0, SV("1234567890"), 5, 0, S("abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 0, SV("1234567890"), 5, 1, S("6abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 0, SV("1234567890"), 5, 2, S("67abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 0, SV("1234567890"), 5, 4, S("6789abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 0, SV("1234567890"), 5, 5, S("67890abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 0, SV("1234567890"), 5, 6, S("67890abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 0, SV("1234567890"), 9, 0, S("abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 0, SV("1234567890"), 9, 1, S("0abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 0, SV("1234567890"), 9, 2, S("0abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 0, SV("1234567890"), 10, 0, S("abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 0, SV("1234567890"), 10, 1, S("abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 0, SV("1234567890"), 11, 0, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 0, 0, SV("12345678901234567890"), 0, 0, S("abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 0, SV("12345678901234567890"), 0, 1, S("1abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 0, SV("12345678901234567890"), 0, 10, S("1234567890abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 0, SV("12345678901234567890"), 0, 19, S("1234567890123456789abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 0, SV("12345678901234567890"), 0, 20, S("12345678901234567890abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 0, SV("12345678901234567890"), 0, 21, S("12345678901234567890abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 0, SV("12345678901234567890"), 1, 0, S("abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 0, SV("12345678901234567890"), 1, 1, S("2abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 0, SV("12345678901234567890"), 1, 9, S("234567890abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 0, SV("12345678901234567890"), 1, 18, S("234567890123456789abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 0, SV("12345678901234567890"), 1, 19, S("2345678901234567890abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 0, SV("12345678901234567890"), 1, 20, S("2345678901234567890abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 0, SV("12345678901234567890"), 10, 0, S("abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 0, SV("12345678901234567890"), 10, 1, S("1abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 0, SV("12345678901234567890"), 10, 5, S("12345abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 0, SV("12345678901234567890"), 10, 9, S("123456789abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 0, SV("12345678901234567890"), 10, 10, S("1234567890abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 0, SV("12345678901234567890"), 10, 11, S("1234567890abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 0, SV("12345678901234567890"), 19, 0, S("abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 0, SV("12345678901234567890"), 19, 1, S("0abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 0, SV("12345678901234567890"), 19, 2, S("0abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 0, SV("12345678901234567890"), 20, 0, S("abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 0, SV("12345678901234567890"), 20, 1, S("abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 0, SV("12345678901234567890"), 21, 0, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 0, 1, SV(""), 0, 0, S("bcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 1, SV(""), 0, 1, S("bcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 1, SV(""), 1, 0, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 0, 1, SV("12345"), 0, 0, S("bcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 1, SV("12345"), 0, 1, S("1bcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 1, SV("12345"), 0, 2, S("12bcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 1, SV("12345"), 0, 4, S("1234bcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 1, SV("12345"), 0, 5, S("12345bcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 1, SV("12345"), 0, 6, S("12345bcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 1, SV("12345"), 1, 0, S("bcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 1, SV("12345"), 1, 1, S("2bcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 1, SV("12345"), 1, 2, S("23bcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 1, SV("12345"), 1, 3, S("234bcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 1, SV("12345"), 1, 4, S("2345bcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 1, SV("12345"), 1, 5, S("2345bcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 1, SV("12345"), 2, 0, S("bcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 1, SV("12345"), 2, 1, S("3bcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 1, SV("12345"), 2, 2, S("34bcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 1, SV("12345"), 2, 3, S("345bcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 1, SV("12345"), 2, 4, S("345bcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 1, SV("12345"), 4, 0, S("bcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 1, SV("12345"), 4, 1, S("5bcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 1, SV("12345"), 4, 2, S("5bcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 1, SV("12345"), 5, 0, S("bcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 1, SV("12345"), 5, 1, S("bcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 1, SV("12345"), 6, 0, S("can't happen"));

    return true;
}

template <class S, class SV>
TEST_CONSTEXPR_CXX20 bool test38()
{
    test(S("abcdefghijklmnopqrst"), 0, 1, SV("1234567890"), 0, 0, S("bcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 1, SV("1234567890"), 0, 1, S("1bcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 1, SV("1234567890"), 0, 5, S("12345bcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 1, SV("1234567890"), 0, 9, S("123456789bcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 1, SV("1234567890"), 0, 10, S("1234567890bcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 1, SV("1234567890"), 0, 11, S("1234567890bcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 1, SV("1234567890"), 1, 0, S("bcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 1, SV("1234567890"), 1, 1, S("2bcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 1, SV("1234567890"), 1, 4, S("2345bcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 1, SV("1234567890"), 1, 8, S("23456789bcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 1, SV("1234567890"), 1, 9, S("234567890bcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 1, SV("1234567890"), 1, 10, S("234567890bcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 1, SV("1234567890"), 5, 0, S("bcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 1, SV("1234567890"), 5, 1, S("6bcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 1, SV("1234567890"), 5, 2, S("67bcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 1, SV("1234567890"), 5, 4, S("6789bcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 1, SV("1234567890"), 5, 5, S("67890bcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 1, SV("1234567890"), 5, 6, S("67890bcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 1, SV("1234567890"), 9, 0, S("bcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 1, SV("1234567890"), 9, 1, S("0bcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 1, SV("1234567890"), 9, 2, S("0bcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 1, SV("1234567890"), 10, 0, S("bcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 1, SV("1234567890"), 10, 1, S("bcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 1, SV("1234567890"), 11, 0, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 0, 1, SV("12345678901234567890"), 0, 0, S("bcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 1, SV("12345678901234567890"), 0, 1, S("1bcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 1, SV("12345678901234567890"), 0, 10, S("1234567890bcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 1, SV("12345678901234567890"), 0, 19, S("1234567890123456789bcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 1, SV("12345678901234567890"), 0, 20, S("12345678901234567890bcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 1, SV("12345678901234567890"), 0, 21, S("12345678901234567890bcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 1, SV("12345678901234567890"), 1, 0, S("bcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 1, SV("12345678901234567890"), 1, 1, S("2bcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 1, SV("12345678901234567890"), 1, 9, S("234567890bcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 1, SV("12345678901234567890"), 1, 18, S("234567890123456789bcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 1, SV("12345678901234567890"), 1, 19, S("2345678901234567890bcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 1, SV("12345678901234567890"), 1, 20, S("2345678901234567890bcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 1, SV("12345678901234567890"), 10, 0, S("bcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 1, SV("12345678901234567890"), 10, 1, S("1bcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 1, SV("12345678901234567890"), 10, 5, S("12345bcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 1, SV("12345678901234567890"), 10, 9, S("123456789bcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 1, SV("12345678901234567890"), 10, 10, S("1234567890bcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 1, SV("12345678901234567890"), 10, 11, S("1234567890bcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 1, SV("12345678901234567890"), 19, 0, S("bcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 1, SV("12345678901234567890"), 19, 1, S("0bcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 1, SV("12345678901234567890"), 19, 2, S("0bcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 1, SV("12345678901234567890"), 20, 0, S("bcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 1, SV("12345678901234567890"), 20, 1, S("bcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 1, SV("12345678901234567890"), 21, 0, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 0, 10, SV(""), 0, 0, S("klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 10, SV(""), 0, 1, S("klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 10, SV(""), 1, 0, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 0, 10, SV("12345"), 0, 0, S("klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 10, SV("12345"), 0, 1, S("1klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 10, SV("12345"), 0, 2, S("12klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 10, SV("12345"), 0, 4, S("1234klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 10, SV("12345"), 0, 5, S("12345klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 10, SV("12345"), 0, 6, S("12345klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 10, SV("12345"), 1, 0, S("klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 10, SV("12345"), 1, 1, S("2klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 10, SV("12345"), 1, 2, S("23klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 10, SV("12345"), 1, 3, S("234klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 10, SV("12345"), 1, 4, S("2345klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 10, SV("12345"), 1, 5, S("2345klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 10, SV("12345"), 2, 0, S("klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 10, SV("12345"), 2, 1, S("3klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 10, SV("12345"), 2, 2, S("34klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 10, SV("12345"), 2, 3, S("345klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 10, SV("12345"), 2, 4, S("345klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 10, SV("12345"), 4, 0, S("klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 10, SV("12345"), 4, 1, S("5klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 10, SV("12345"), 4, 2, S("5klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 10, SV("12345"), 5, 0, S("klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 10, SV("12345"), 5, 1, S("klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 10, SV("12345"), 6, 0, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 0, 10, SV("1234567890"), 0, 0, S("klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 10, SV("1234567890"), 0, 1, S("1klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 10, SV("1234567890"), 0, 5, S("12345klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 10, SV("1234567890"), 0, 9, S("123456789klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 10, SV("1234567890"), 0, 10, S("1234567890klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 10, SV("1234567890"), 0, 11, S("1234567890klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 10, SV("1234567890"), 1, 0, S("klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 10, SV("1234567890"), 1, 1, S("2klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 10, SV("1234567890"), 1, 4, S("2345klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 10, SV("1234567890"), 1, 8, S("23456789klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 10, SV("1234567890"), 1, 9, S("234567890klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 10, SV("1234567890"), 1, 10, S("234567890klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 10, SV("1234567890"), 5, 0, S("klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 10, SV("1234567890"), 5, 1, S("6klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 10, SV("1234567890"), 5, 2, S("67klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 10, SV("1234567890"), 5, 4, S("6789klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 10, SV("1234567890"), 5, 5, S("67890klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 10, SV("1234567890"), 5, 6, S("67890klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 10, SV("1234567890"), 9, 0, S("klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 10, SV("1234567890"), 9, 1, S("0klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 10, SV("1234567890"), 9, 2, S("0klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 10, SV("1234567890"), 10, 0, S("klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 10, SV("1234567890"), 10, 1, S("klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 10, SV("1234567890"), 11, 0, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 0, 10, SV("12345678901234567890"), 0, 0, S("klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 10, SV("12345678901234567890"), 0, 1, S("1klmnopqrst"));

    return true;
}

template <class S, class SV>
TEST_CONSTEXPR_CXX20 bool test39()
{
    test(S("abcdefghijklmnopqrst"), 0, 10, SV("12345678901234567890"), 0, 10, S("1234567890klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 10, SV("12345678901234567890"), 0, 19, S("1234567890123456789klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 10, SV("12345678901234567890"), 0, 20, S("12345678901234567890klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 10, SV("12345678901234567890"), 0, 21, S("12345678901234567890klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 10, SV("12345678901234567890"), 1, 0, S("klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 10, SV("12345678901234567890"), 1, 1, S("2klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 10, SV("12345678901234567890"), 1, 9, S("234567890klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 10, SV("12345678901234567890"), 1, 18, S("234567890123456789klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 10, SV("12345678901234567890"), 1, 19, S("2345678901234567890klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 10, SV("12345678901234567890"), 1, 20, S("2345678901234567890klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 10, SV("12345678901234567890"), 10, 0, S("klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 10, SV("12345678901234567890"), 10, 1, S("1klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 10, SV("12345678901234567890"), 10, 5, S("12345klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 10, SV("12345678901234567890"), 10, 9, S("123456789klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 10, SV("12345678901234567890"), 10, 10, S("1234567890klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 10, SV("12345678901234567890"), 10, 11, S("1234567890klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 10, SV("12345678901234567890"), 19, 0, S("klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 10, SV("12345678901234567890"), 19, 1, S("0klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 10, SV("12345678901234567890"), 19, 2, S("0klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 10, SV("12345678901234567890"), 20, 0, S("klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 10, SV("12345678901234567890"), 20, 1, S("klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 0, 10, SV("12345678901234567890"), 21, 0, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 0, 19, SV(""), 0, 0, S("t"));
    test(S("abcdefghijklmnopqrst"), 0, 19, SV(""), 0, 1, S("t"));
    test(S("abcdefghijklmnopqrst"), 0, 19, SV(""), 1, 0, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 0, 19, SV("12345"), 0, 0, S("t"));
    test(S("abcdefghijklmnopqrst"), 0, 19, SV("12345"), 0, 1, S("1t"));
    test(S("abcdefghijklmnopqrst"), 0, 19, SV("12345"), 0, 2, S("12t"));
    test(S("abcdefghijklmnopqrst"), 0, 19, SV("12345"), 0, 4, S("1234t"));
    test(S("abcdefghijklmnopqrst"), 0, 19, SV("12345"), 0, 5, S("12345t"));
    test(S("abcdefghijklmnopqrst"), 0, 19, SV("12345"), 0, 6, S("12345t"));
    test(S("abcdefghijklmnopqrst"), 0, 19, SV("12345"), 1, 0, S("t"));
    test(S("abcdefghijklmnopqrst"), 0, 19, SV("12345"), 1, 1, S("2t"));
    test(S("abcdefghijklmnopqrst"), 0, 19, SV("12345"), 1, 2, S("23t"));
    test(S("abcdefghijklmnopqrst"), 0, 19, SV("12345"), 1, 3, S("234t"));
    test(S("abcdefghijklmnopqrst"), 0, 19, SV("12345"), 1, 4, S("2345t"));
    test(S("abcdefghijklmnopqrst"), 0, 19, SV("12345"), 1, 5, S("2345t"));
    test(S("abcdefghijklmnopqrst"), 0, 19, SV("12345"), 2, 0, S("t"));
    test(S("abcdefghijklmnopqrst"), 0, 19, SV("12345"), 2, 1, S("3t"));
    test(S("abcdefghijklmnopqrst"), 0, 19, SV("12345"), 2, 2, S("34t"));
    test(S("abcdefghijklmnopqrst"), 0, 19, SV("12345"), 2, 3, S("345t"));
    test(S("abcdefghijklmnopqrst"), 0, 19, SV("12345"), 2, 4, S("345t"));
    test(S("abcdefghijklmnopqrst"), 0, 19, SV("12345"), 4, 0, S("t"));
    test(S("abcdefghijklmnopqrst"), 0, 19, SV("12345"), 4, 1, S("5t"));
    test(S("abcdefghijklmnopqrst"), 0, 19, SV("12345"), 4, 2, S("5t"));
    test(S("abcdefghijklmnopqrst"), 0, 19, SV("12345"), 5, 0, S("t"));
    test(S("abcdefghijklmnopqrst"), 0, 19, SV("12345"), 5, 1, S("t"));
    test(S("abcdefghijklmnopqrst"), 0, 19, SV("12345"), 6, 0, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 0, 19, SV("1234567890"), 0, 0, S("t"));
    test(S("abcdefghijklmnopqrst"), 0, 19, SV("1234567890"), 0, 1, S("1t"));
    test(S("abcdefghijklmnopqrst"), 0, 19, SV("1234567890"), 0, 5, S("12345t"));
    test(S("abcdefghijklmnopqrst"), 0, 19, SV("1234567890"), 0, 9, S("123456789t"));
    test(S("abcdefghijklmnopqrst"), 0, 19, SV("1234567890"), 0, 10, S("1234567890t"));
    test(S("abcdefghijklmnopqrst"), 0, 19, SV("1234567890"), 0, 11, S("1234567890t"));
    test(S("abcdefghijklmnopqrst"), 0, 19, SV("1234567890"), 1, 0, S("t"));
    test(S("abcdefghijklmnopqrst"), 0, 19, SV("1234567890"), 1, 1, S("2t"));
    test(S("abcdefghijklmnopqrst"), 0, 19, SV("1234567890"), 1, 4, S("2345t"));
    test(S("abcdefghijklmnopqrst"), 0, 19, SV("1234567890"), 1, 8, S("23456789t"));
    test(S("abcdefghijklmnopqrst"), 0, 19, SV("1234567890"), 1, 9, S("234567890t"));
    test(S("abcdefghijklmnopqrst"), 0, 19, SV("1234567890"), 1, 10, S("234567890t"));
    test(S("abcdefghijklmnopqrst"), 0, 19, SV("1234567890"), 5, 0, S("t"));
    test(S("abcdefghijklmnopqrst"), 0, 19, SV("1234567890"), 5, 1, S("6t"));
    test(S("abcdefghijklmnopqrst"), 0, 19, SV("1234567890"), 5, 2, S("67t"));
    test(S("abcdefghijklmnopqrst"), 0, 19, SV("1234567890"), 5, 4, S("6789t"));
    test(S("abcdefghijklmnopqrst"), 0, 19, SV("1234567890"), 5, 5, S("67890t"));
    test(S("abcdefghijklmnopqrst"), 0, 19, SV("1234567890"), 5, 6, S("67890t"));
    test(S("abcdefghijklmnopqrst"), 0, 19, SV("1234567890"), 9, 0, S("t"));
    test(S("abcdefghijklmnopqrst"), 0, 19, SV("1234567890"), 9, 1, S("0t"));
    test(S("abcdefghijklmnopqrst"), 0, 19, SV("1234567890"), 9, 2, S("0t"));
    test(S("abcdefghijklmnopqrst"), 0, 19, SV("1234567890"), 10, 0, S("t"));
    test(S("abcdefghijklmnopqrst"), 0, 19, SV("1234567890"), 10, 1, S("t"));
    test(S("abcdefghijklmnopqrst"), 0, 19, SV("1234567890"), 11, 0, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 0, 19, SV("12345678901234567890"), 0, 0, S("t"));
    test(S("abcdefghijklmnopqrst"), 0, 19, SV("12345678901234567890"), 0, 1, S("1t"));
    test(S("abcdefghijklmnopqrst"), 0, 19, SV("12345678901234567890"), 0, 10, S("1234567890t"));
    test(S("abcdefghijklmnopqrst"), 0, 19, SV("12345678901234567890"), 0, 19, S("1234567890123456789t"));
    test(S("abcdefghijklmnopqrst"), 0, 19, SV("12345678901234567890"), 0, 20, S("12345678901234567890t"));
    test(S("abcdefghijklmnopqrst"), 0, 19, SV("12345678901234567890"), 0, 21, S("12345678901234567890t"));
    test(S("abcdefghijklmnopqrst"), 0, 19, SV("12345678901234567890"), 1, 0, S("t"));
    test(S("abcdefghijklmnopqrst"), 0, 19, SV("12345678901234567890"), 1, 1, S("2t"));
    test(S("abcdefghijklmnopqrst"), 0, 19, SV("12345678901234567890"), 1, 9, S("234567890t"));
    test(S("abcdefghijklmnopqrst"), 0, 19, SV("12345678901234567890"), 1, 18, S("234567890123456789t"));
    test(S("abcdefghijklmnopqrst"), 0, 19, SV("12345678901234567890"), 1, 19, S("2345678901234567890t"));
    test(S("abcdefghijklmnopqrst"), 0, 19, SV("12345678901234567890"), 1, 20, S("2345678901234567890t"));
    test(S("abcdefghijklmnopqrst"), 0, 19, SV("12345678901234567890"), 10, 0, S("t"));
    test(S("abcdefghijklmnopqrst"), 0, 19, SV("12345678901234567890"), 10, 1, S("1t"));
    test(S("abcdefghijklmnopqrst"), 0, 19, SV("12345678901234567890"), 10, 5, S("12345t"));
    test(S("abcdefghijklmnopqrst"), 0, 19, SV("12345678901234567890"), 10, 9, S("123456789t"));
    test(S("abcdefghijklmnopqrst"), 0, 19, SV("12345678901234567890"), 10, 10, S("1234567890t"));
    test(S("abcdefghijklmnopqrst"), 0, 19, SV("12345678901234567890"), 10, 11, S("1234567890t"));
    test(S("abcdefghijklmnopqrst"), 0, 19, SV("12345678901234567890"), 19, 0, S("t"));
    test(S("abcdefghijklmnopqrst"), 0, 19, SV("12345678901234567890"), 19, 1, S("0t"));
    test(S("abcdefghijklmnopqrst"), 0, 19, SV("12345678901234567890"), 19, 2, S("0t"));
    test(S("abcdefghijklmnopqrst"), 0, 19, SV("12345678901234567890"), 20, 0, S("t"));
    test(S("abcdefghijklmnopqrst"), 0, 19, SV("12345678901234567890"), 20, 1, S("t"));
    test(S("abcdefghijklmnopqrst"), 0, 19, SV("12345678901234567890"), 21, 0, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 0, 20, SV(""), 0, 0, S(""));
    test(S("abcdefghijklmnopqrst"), 0, 20, SV(""), 0, 1, S(""));
    test(S("abcdefghijklmnopqrst"), 0, 20, SV(""), 1, 0, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 0, 20, SV("12345"), 0, 0, S(""));

    return true;
}

template <class S, class SV>
TEST_CONSTEXPR_CXX20 bool test40()
{
    test(S("abcdefghijklmnopqrst"), 0, 20, SV("12345"), 0, 1, S("1"));
    test(S("abcdefghijklmnopqrst"), 0, 20, SV("12345"), 0, 2, S("12"));
    test(S("abcdefghijklmnopqrst"), 0, 20, SV("12345"), 0, 4, S("1234"));
    test(S("abcdefghijklmnopqrst"), 0, 20, SV("12345"), 0, 5, S("12345"));
    test(S("abcdefghijklmnopqrst"), 0, 20, SV("12345"), 0, 6, S("12345"));
    test(S("abcdefghijklmnopqrst"), 0, 20, SV("12345"), 1, 0, S(""));
    test(S("abcdefghijklmnopqrst"), 0, 20, SV("12345"), 1, 1, S("2"));
    test(S("abcdefghijklmnopqrst"), 0, 20, SV("12345"), 1, 2, S("23"));
    test(S("abcdefghijklmnopqrst"), 0, 20, SV("12345"), 1, 3, S("234"));
    test(S("abcdefghijklmnopqrst"), 0, 20, SV("12345"), 1, 4, S("2345"));
    test(S("abcdefghijklmnopqrst"), 0, 20, SV("12345"), 1, 5, S("2345"));
    test(S("abcdefghijklmnopqrst"), 0, 20, SV("12345"), 2, 0, S(""));
    test(S("abcdefghijklmnopqrst"), 0, 20, SV("12345"), 2, 1, S("3"));
    test(S("abcdefghijklmnopqrst"), 0, 20, SV("12345"), 2, 2, S("34"));
    test(S("abcdefghijklmnopqrst"), 0, 20, SV("12345"), 2, 3, S("345"));
    test(S("abcdefghijklmnopqrst"), 0, 20, SV("12345"), 2, 4, S("345"));
    test(S("abcdefghijklmnopqrst"), 0, 20, SV("12345"), 4, 0, S(""));
    test(S("abcdefghijklmnopqrst"), 0, 20, SV("12345"), 4, 1, S("5"));
    test(S("abcdefghijklmnopqrst"), 0, 20, SV("12345"), 4, 2, S("5"));
    test(S("abcdefghijklmnopqrst"), 0, 20, SV("12345"), 5, 0, S(""));
    test(S("abcdefghijklmnopqrst"), 0, 20, SV("12345"), 5, 1, S(""));
    test(S("abcdefghijklmnopqrst"), 0, 20, SV("12345"), 6, 0, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 0, 20, SV("1234567890"), 0, 0, S(""));
    test(S("abcdefghijklmnopqrst"), 0, 20, SV("1234567890"), 0, 1, S("1"));
    test(S("abcdefghijklmnopqrst"), 0, 20, SV("1234567890"), 0, 5, S("12345"));
    test(S("abcdefghijklmnopqrst"), 0, 20, SV("1234567890"), 0, 9, S("123456789"));
    test(S("abcdefghijklmnopqrst"), 0, 20, SV("1234567890"), 0, 10, S("1234567890"));
    test(S("abcdefghijklmnopqrst"), 0, 20, SV("1234567890"), 0, 11, S("1234567890"));
    test(S("abcdefghijklmnopqrst"), 0, 20, SV("1234567890"), 1, 0, S(""));
    test(S("abcdefghijklmnopqrst"), 0, 20, SV("1234567890"), 1, 1, S("2"));
    test(S("abcdefghijklmnopqrst"), 0, 20, SV("1234567890"), 1, 4, S("2345"));
    test(S("abcdefghijklmnopqrst"), 0, 20, SV("1234567890"), 1, 8, S("23456789"));
    test(S("abcdefghijklmnopqrst"), 0, 20, SV("1234567890"), 1, 9, S("234567890"));
    test(S("abcdefghijklmnopqrst"), 0, 20, SV("1234567890"), 1, 10, S("234567890"));
    test(S("abcdefghijklmnopqrst"), 0, 20, SV("1234567890"), 5, 0, S(""));
    test(S("abcdefghijklmnopqrst"), 0, 20, SV("1234567890"), 5, 1, S("6"));
    test(S("abcdefghijklmnopqrst"), 0, 20, SV("1234567890"), 5, 2, S("67"));
    test(S("abcdefghijklmnopqrst"), 0, 20, SV("1234567890"), 5, 4, S("6789"));
    test(S("abcdefghijklmnopqrst"), 0, 20, SV("1234567890"), 5, 5, S("67890"));
    test(S("abcdefghijklmnopqrst"), 0, 20, SV("1234567890"), 5, 6, S("67890"));
    test(S("abcdefghijklmnopqrst"), 0, 20, SV("1234567890"), 9, 0, S(""));
    test(S("abcdefghijklmnopqrst"), 0, 20, SV("1234567890"), 9, 1, S("0"));
    test(S("abcdefghijklmnopqrst"), 0, 20, SV("1234567890"), 9, 2, S("0"));
    test(S("abcdefghijklmnopqrst"), 0, 20, SV("1234567890"), 10, 0, S(""));
    test(S("abcdefghijklmnopqrst"), 0, 20, SV("1234567890"), 10, 1, S(""));
    test(S("abcdefghijklmnopqrst"), 0, 20, SV("1234567890"), 11, 0, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 0, 20, SV("12345678901234567890"), 0, 0, S(""));
    test(S("abcdefghijklmnopqrst"), 0, 20, SV("12345678901234567890"), 0, 1, S("1"));
    test(S("abcdefghijklmnopqrst"), 0, 20, SV("12345678901234567890"), 0, 10, S("1234567890"));
    test(S("abcdefghijklmnopqrst"), 0, 20, SV("12345678901234567890"), 0, 19, S("1234567890123456789"));
    test(S("abcdefghijklmnopqrst"), 0, 20, SV("12345678901234567890"), 0, 20, S("12345678901234567890"));
    test(S("abcdefghijklmnopqrst"), 0, 20, SV("12345678901234567890"), 0, 21, S("12345678901234567890"));
    test(S("abcdefghijklmnopqrst"), 0, 20, SV("12345678901234567890"), 1, 0, S(""));
    test(S("abcdefghijklmnopqrst"), 0, 20, SV("12345678901234567890"), 1, 1, S("2"));
    test(S("abcdefghijklmnopqrst"), 0, 20, SV("12345678901234567890"), 1, 9, S("234567890"));
    test(S("abcdefghijklmnopqrst"), 0, 20, SV("12345678901234567890"), 1, 18, S("234567890123456789"));
    test(S("abcdefghijklmnopqrst"), 0, 20, SV("12345678901234567890"), 1, 19, S("2345678901234567890"));
    test(S("abcdefghijklmnopqrst"), 0, 20, SV("12345678901234567890"), 1, 20, S("2345678901234567890"));
    test(S("abcdefghijklmnopqrst"), 0, 20, SV("12345678901234567890"), 10, 0, S(""));
    test(S("abcdefghijklmnopqrst"), 0, 20, SV("12345678901234567890"), 10, 1, S("1"));
    test(S("abcdefghijklmnopqrst"), 0, 20, SV("12345678901234567890"), 10, 5, S("12345"));
    test(S("abcdefghijklmnopqrst"), 0, 20, SV("12345678901234567890"), 10, 9, S("123456789"));
    test(S("abcdefghijklmnopqrst"), 0, 20, SV("12345678901234567890"), 10, 10, S("1234567890"));
    test(S("abcdefghijklmnopqrst"), 0, 20, SV("12345678901234567890"), 10, 11, S("1234567890"));
    test(S("abcdefghijklmnopqrst"), 0, 20, SV("12345678901234567890"), 19, 0, S(""));
    test(S("abcdefghijklmnopqrst"), 0, 20, SV("12345678901234567890"), 19, 1, S("0"));
    test(S("abcdefghijklmnopqrst"), 0, 20, SV("12345678901234567890"), 19, 2, S("0"));
    test(S("abcdefghijklmnopqrst"), 0, 20, SV("12345678901234567890"), 20, 0, S(""));
    test(S("abcdefghijklmnopqrst"), 0, 20, SV("12345678901234567890"), 20, 1, S(""));
    test(S("abcdefghijklmnopqrst"), 0, 20, SV("12345678901234567890"), 21, 0, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 0, 21, SV(""), 0, 0, S(""));
    test(S("abcdefghijklmnopqrst"), 0, 21, SV(""), 0, 1, S(""));
    test(S("abcdefghijklmnopqrst"), 0, 21, SV(""), 1, 0, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 0, 21, SV("12345"), 0, 0, S(""));
    test(S("abcdefghijklmnopqrst"), 0, 21, SV("12345"), 0, 1, S("1"));
    test(S("abcdefghijklmnopqrst"), 0, 21, SV("12345"), 0, 2, S("12"));
    test(S("abcdefghijklmnopqrst"), 0, 21, SV("12345"), 0, 4, S("1234"));
    test(S("abcdefghijklmnopqrst"), 0, 21, SV("12345"), 0, 5, S("12345"));
    test(S("abcdefghijklmnopqrst"), 0, 21, SV("12345"), 0, 6, S("12345"));
    test(S("abcdefghijklmnopqrst"), 0, 21, SV("12345"), 1, 0, S(""));
    test(S("abcdefghijklmnopqrst"), 0, 21, SV("12345"), 1, 1, S("2"));
    test(S("abcdefghijklmnopqrst"), 0, 21, SV("12345"), 1, 2, S("23"));
    test(S("abcdefghijklmnopqrst"), 0, 21, SV("12345"), 1, 3, S("234"));
    test(S("abcdefghijklmnopqrst"), 0, 21, SV("12345"), 1, 4, S("2345"));
    test(S("abcdefghijklmnopqrst"), 0, 21, SV("12345"), 1, 5, S("2345"));
    test(S("abcdefghijklmnopqrst"), 0, 21, SV("12345"), 2, 0, S(""));
    test(S("abcdefghijklmnopqrst"), 0, 21, SV("12345"), 2, 1, S("3"));
    test(S("abcdefghijklmnopqrst"), 0, 21, SV("12345"), 2, 2, S("34"));
    test(S("abcdefghijklmnopqrst"), 0, 21, SV("12345"), 2, 3, S("345"));
    test(S("abcdefghijklmnopqrst"), 0, 21, SV("12345"), 2, 4, S("345"));
    test(S("abcdefghijklmnopqrst"), 0, 21, SV("12345"), 4, 0, S(""));
    test(S("abcdefghijklmnopqrst"), 0, 21, SV("12345"), 4, 1, S("5"));
    test(S("abcdefghijklmnopqrst"), 0, 21, SV("12345"), 4, 2, S("5"));
    test(S("abcdefghijklmnopqrst"), 0, 21, SV("12345"), 5, 0, S(""));
    test(S("abcdefghijklmnopqrst"), 0, 21, SV("12345"), 5, 1, S(""));
    test(S("abcdefghijklmnopqrst"), 0, 21, SV("12345"), 6, 0, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 0, 21, SV("1234567890"), 0, 0, S(""));
    test(S("abcdefghijklmnopqrst"), 0, 21, SV("1234567890"), 0, 1, S("1"));
    test(S("abcdefghijklmnopqrst"), 0, 21, SV("1234567890"), 0, 5, S("12345"));
    test(S("abcdefghijklmnopqrst"), 0, 21, SV("1234567890"), 0, 9, S("123456789"));

    return true;
}

template <class S, class SV>
TEST_CONSTEXPR_CXX20 bool test41()
{
    test(S("abcdefghijklmnopqrst"), 0, 21, SV("1234567890"), 0, 10, S("1234567890"));
    test(S("abcdefghijklmnopqrst"), 0, 21, SV("1234567890"), 0, 11, S("1234567890"));
    test(S("abcdefghijklmnopqrst"), 0, 21, SV("1234567890"), 1, 0, S(""));
    test(S("abcdefghijklmnopqrst"), 0, 21, SV("1234567890"), 1, 1, S("2"));
    test(S("abcdefghijklmnopqrst"), 0, 21, SV("1234567890"), 1, 4, S("2345"));
    test(S("abcdefghijklmnopqrst"), 0, 21, SV("1234567890"), 1, 8, S("23456789"));
    test(S("abcdefghijklmnopqrst"), 0, 21, SV("1234567890"), 1, 9, S("234567890"));
    test(S("abcdefghijklmnopqrst"), 0, 21, SV("1234567890"), 1, 10, S("234567890"));
    test(S("abcdefghijklmnopqrst"), 0, 21, SV("1234567890"), 5, 0, S(""));
    test(S("abcdefghijklmnopqrst"), 0, 21, SV("1234567890"), 5, 1, S("6"));
    test(S("abcdefghijklmnopqrst"), 0, 21, SV("1234567890"), 5, 2, S("67"));
    test(S("abcdefghijklmnopqrst"), 0, 21, SV("1234567890"), 5, 4, S("6789"));
    test(S("abcdefghijklmnopqrst"), 0, 21, SV("1234567890"), 5, 5, S("67890"));
    test(S("abcdefghijklmnopqrst"), 0, 21, SV("1234567890"), 5, 6, S("67890"));
    test(S("abcdefghijklmnopqrst"), 0, 21, SV("1234567890"), 9, 0, S(""));
    test(S("abcdefghijklmnopqrst"), 0, 21, SV("1234567890"), 9, 1, S("0"));
    test(S("abcdefghijklmnopqrst"), 0, 21, SV("1234567890"), 9, 2, S("0"));
    test(S("abcdefghijklmnopqrst"), 0, 21, SV("1234567890"), 10, 0, S(""));
    test(S("abcdefghijklmnopqrst"), 0, 21, SV("1234567890"), 10, 1, S(""));
    test(S("abcdefghijklmnopqrst"), 0, 21, SV("1234567890"), 11, 0, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 0, 21, SV("12345678901234567890"), 0, 0, S(""));
    test(S("abcdefghijklmnopqrst"), 0, 21, SV("12345678901234567890"), 0, 1, S("1"));
    test(S("abcdefghijklmnopqrst"), 0, 21, SV("12345678901234567890"), 0, 10, S("1234567890"));
    test(S("abcdefghijklmnopqrst"), 0, 21, SV("12345678901234567890"), 0, 19, S("1234567890123456789"));
    test(S("abcdefghijklmnopqrst"), 0, 21, SV("12345678901234567890"), 0, 20, S("12345678901234567890"));
    test(S("abcdefghijklmnopqrst"), 0, 21, SV("12345678901234567890"), 0, 21, S("12345678901234567890"));
    test(S("abcdefghijklmnopqrst"), 0, 21, SV("12345678901234567890"), 1, 0, S(""));
    test(S("abcdefghijklmnopqrst"), 0, 21, SV("12345678901234567890"), 1, 1, S("2"));
    test(S("abcdefghijklmnopqrst"), 0, 21, SV("12345678901234567890"), 1, 9, S("234567890"));
    test(S("abcdefghijklmnopqrst"), 0, 21, SV("12345678901234567890"), 1, 18, S("234567890123456789"));
    test(S("abcdefghijklmnopqrst"), 0, 21, SV("12345678901234567890"), 1, 19, S("2345678901234567890"));
    test(S("abcdefghijklmnopqrst"), 0, 21, SV("12345678901234567890"), 1, 20, S("2345678901234567890"));
    test(S("abcdefghijklmnopqrst"), 0, 21, SV("12345678901234567890"), 10, 0, S(""));
    test(S("abcdefghijklmnopqrst"), 0, 21, SV("12345678901234567890"), 10, 1, S("1"));
    test(S("abcdefghijklmnopqrst"), 0, 21, SV("12345678901234567890"), 10, 5, S("12345"));
    test(S("abcdefghijklmnopqrst"), 0, 21, SV("12345678901234567890"), 10, 9, S("123456789"));
    test(S("abcdefghijklmnopqrst"), 0, 21, SV("12345678901234567890"), 10, 10, S("1234567890"));
    test(S("abcdefghijklmnopqrst"), 0, 21, SV("12345678901234567890"), 10, 11, S("1234567890"));
    test(S("abcdefghijklmnopqrst"), 0, 21, SV("12345678901234567890"), 19, 0, S(""));
    test(S("abcdefghijklmnopqrst"), 0, 21, SV("12345678901234567890"), 19, 1, S("0"));
    test(S("abcdefghijklmnopqrst"), 0, 21, SV("12345678901234567890"), 19, 2, S("0"));
    test(S("abcdefghijklmnopqrst"), 0, 21, SV("12345678901234567890"), 20, 0, S(""));
    test(S("abcdefghijklmnopqrst"), 0, 21, SV("12345678901234567890"), 20, 1, S(""));
    test(S("abcdefghijklmnopqrst"), 0, 21, SV("12345678901234567890"), 21, 0, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 1, 0, SV(""), 0, 0, S("abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 0, SV(""), 0, 1, S("abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 0, SV(""), 1, 0, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 1, 0, SV("12345"), 0, 0, S("abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 0, SV("12345"), 0, 1, S("a1bcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 0, SV("12345"), 0, 2, S("a12bcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 0, SV("12345"), 0, 4, S("a1234bcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 0, SV("12345"), 0, 5, S("a12345bcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 0, SV("12345"), 0, 6, S("a12345bcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 0, SV("12345"), 1, 0, S("abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 0, SV("12345"), 1, 1, S("a2bcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 0, SV("12345"), 1, 2, S("a23bcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 0, SV("12345"), 1, 3, S("a234bcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 0, SV("12345"), 1, 4, S("a2345bcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 0, SV("12345"), 1, 5, S("a2345bcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 0, SV("12345"), 2, 0, S("abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 0, SV("12345"), 2, 1, S("a3bcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 0, SV("12345"), 2, 2, S("a34bcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 0, SV("12345"), 2, 3, S("a345bcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 0, SV("12345"), 2, 4, S("a345bcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 0, SV("12345"), 4, 0, S("abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 0, SV("12345"), 4, 1, S("a5bcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 0, SV("12345"), 4, 2, S("a5bcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 0, SV("12345"), 5, 0, S("abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 0, SV("12345"), 5, 1, S("abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 0, SV("12345"), 6, 0, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 1, 0, SV("1234567890"), 0, 0, S("abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 0, SV("1234567890"), 0, 1, S("a1bcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 0, SV("1234567890"), 0, 5, S("a12345bcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 0, SV("1234567890"), 0, 9, S("a123456789bcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 0, SV("1234567890"), 0, 10, S("a1234567890bcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 0, SV("1234567890"), 0, 11, S("a1234567890bcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 0, SV("1234567890"), 1, 0, S("abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 0, SV("1234567890"), 1, 1, S("a2bcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 0, SV("1234567890"), 1, 4, S("a2345bcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 0, SV("1234567890"), 1, 8, S("a23456789bcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 0, SV("1234567890"), 1, 9, S("a234567890bcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 0, SV("1234567890"), 1, 10, S("a234567890bcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 0, SV("1234567890"), 5, 0, S("abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 0, SV("1234567890"), 5, 1, S("a6bcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 0, SV("1234567890"), 5, 2, S("a67bcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 0, SV("1234567890"), 5, 4, S("a6789bcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 0, SV("1234567890"), 5, 5, S("a67890bcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 0, SV("1234567890"), 5, 6, S("a67890bcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 0, SV("1234567890"), 9, 0, S("abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 0, SV("1234567890"), 9, 1, S("a0bcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 0, SV("1234567890"), 9, 2, S("a0bcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 0, SV("1234567890"), 10, 0, S("abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 0, SV("1234567890"), 10, 1, S("abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 0, SV("1234567890"), 11, 0, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 1, 0, SV("12345678901234567890"), 0, 0, S("abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 0, SV("12345678901234567890"), 0, 1, S("a1bcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 0, SV("12345678901234567890"), 0, 10, S("a1234567890bcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 0, SV("12345678901234567890"), 0, 19, S("a1234567890123456789bcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 0, SV("12345678901234567890"), 0, 20, S("a12345678901234567890bcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 0, SV("12345678901234567890"), 0, 21, S("a12345678901234567890bcdefghijklmnopqrst"));

    return true;
}

template <class S, class SV>
TEST_CONSTEXPR_CXX20 bool test42()
{
    test(S("abcdefghijklmnopqrst"), 1, 0, SV("12345678901234567890"), 1, 0, S("abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 0, SV("12345678901234567890"), 1, 1, S("a2bcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 0, SV("12345678901234567890"), 1, 9, S("a234567890bcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 0, SV("12345678901234567890"), 1, 18, S("a234567890123456789bcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 0, SV("12345678901234567890"), 1, 19, S("a2345678901234567890bcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 0, SV("12345678901234567890"), 1, 20, S("a2345678901234567890bcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 0, SV("12345678901234567890"), 10, 0, S("abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 0, SV("12345678901234567890"), 10, 1, S("a1bcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 0, SV("12345678901234567890"), 10, 5, S("a12345bcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 0, SV("12345678901234567890"), 10, 9, S("a123456789bcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 0, SV("12345678901234567890"), 10, 10, S("a1234567890bcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 0, SV("12345678901234567890"), 10, 11, S("a1234567890bcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 0, SV("12345678901234567890"), 19, 0, S("abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 0, SV("12345678901234567890"), 19, 1, S("a0bcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 0, SV("12345678901234567890"), 19, 2, S("a0bcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 0, SV("12345678901234567890"), 20, 0, S("abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 0, SV("12345678901234567890"), 20, 1, S("abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 0, SV("12345678901234567890"), 21, 0, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 1, 1, SV(""), 0, 0, S("acdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 1, SV(""), 0, 1, S("acdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 1, SV(""), 1, 0, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 1, 1, SV("12345"), 0, 0, S("acdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 1, SV("12345"), 0, 1, S("a1cdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 1, SV("12345"), 0, 2, S("a12cdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 1, SV("12345"), 0, 4, S("a1234cdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 1, SV("12345"), 0, 5, S("a12345cdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 1, SV("12345"), 0, 6, S("a12345cdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 1, SV("12345"), 1, 0, S("acdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 1, SV("12345"), 1, 1, S("a2cdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 1, SV("12345"), 1, 2, S("a23cdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 1, SV("12345"), 1, 3, S("a234cdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 1, SV("12345"), 1, 4, S("a2345cdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 1, SV("12345"), 1, 5, S("a2345cdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 1, SV("12345"), 2, 0, S("acdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 1, SV("12345"), 2, 1, S("a3cdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 1, SV("12345"), 2, 2, S("a34cdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 1, SV("12345"), 2, 3, S("a345cdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 1, SV("12345"), 2, 4, S("a345cdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 1, SV("12345"), 4, 0, S("acdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 1, SV("12345"), 4, 1, S("a5cdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 1, SV("12345"), 4, 2, S("a5cdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 1, SV("12345"), 5, 0, S("acdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 1, SV("12345"), 5, 1, S("acdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 1, SV("12345"), 6, 0, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 1, 1, SV("1234567890"), 0, 0, S("acdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 1, SV("1234567890"), 0, 1, S("a1cdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 1, SV("1234567890"), 0, 5, S("a12345cdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 1, SV("1234567890"), 0, 9, S("a123456789cdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 1, SV("1234567890"), 0, 10, S("a1234567890cdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 1, SV("1234567890"), 0, 11, S("a1234567890cdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 1, SV("1234567890"), 1, 0, S("acdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 1, SV("1234567890"), 1, 1, S("a2cdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 1, SV("1234567890"), 1, 4, S("a2345cdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 1, SV("1234567890"), 1, 8, S("a23456789cdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 1, SV("1234567890"), 1, 9, S("a234567890cdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 1, SV("1234567890"), 1, 10, S("a234567890cdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 1, SV("1234567890"), 5, 0, S("acdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 1, SV("1234567890"), 5, 1, S("a6cdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 1, SV("1234567890"), 5, 2, S("a67cdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 1, SV("1234567890"), 5, 4, S("a6789cdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 1, SV("1234567890"), 5, 5, S("a67890cdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 1, SV("1234567890"), 5, 6, S("a67890cdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 1, SV("1234567890"), 9, 0, S("acdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 1, SV("1234567890"), 9, 1, S("a0cdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 1, SV("1234567890"), 9, 2, S("a0cdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 1, SV("1234567890"), 10, 0, S("acdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 1, SV("1234567890"), 10, 1, S("acdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 1, SV("1234567890"), 11, 0, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 1, 1, SV("12345678901234567890"), 0, 0, S("acdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 1, SV("12345678901234567890"), 0, 1, S("a1cdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 1, SV("12345678901234567890"), 0, 10, S("a1234567890cdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 1, SV("12345678901234567890"), 0, 19, S("a1234567890123456789cdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 1, SV("12345678901234567890"), 0, 20, S("a12345678901234567890cdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 1, SV("12345678901234567890"), 0, 21, S("a12345678901234567890cdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 1, SV("12345678901234567890"), 1, 0, S("acdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 1, SV("12345678901234567890"), 1, 1, S("a2cdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 1, SV("12345678901234567890"), 1, 9, S("a234567890cdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 1, SV("12345678901234567890"), 1, 18, S("a234567890123456789cdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 1, SV("12345678901234567890"), 1, 19, S("a2345678901234567890cdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 1, SV("12345678901234567890"), 1, 20, S("a2345678901234567890cdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 1, SV("12345678901234567890"), 10, 0, S("acdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 1, SV("12345678901234567890"), 10, 1, S("a1cdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 1, SV("12345678901234567890"), 10, 5, S("a12345cdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 1, SV("12345678901234567890"), 10, 9, S("a123456789cdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 1, SV("12345678901234567890"), 10, 10, S("a1234567890cdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 1, SV("12345678901234567890"), 10, 11, S("a1234567890cdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 1, SV("12345678901234567890"), 19, 0, S("acdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 1, SV("12345678901234567890"), 19, 1, S("a0cdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 1, SV("12345678901234567890"), 19, 2, S("a0cdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 1, SV("12345678901234567890"), 20, 0, S("acdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 1, SV("12345678901234567890"), 20, 1, S("acdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 1, SV("12345678901234567890"), 21, 0, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 1, 9, SV(""), 0, 0, S("aklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 9, SV(""), 0, 1, S("aklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 9, SV(""), 1, 0, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 1, 9, SV("12345"), 0, 0, S("aklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 9, SV("12345"), 0, 1, S("a1klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 9, SV("12345"), 0, 2, S("a12klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 9, SV("12345"), 0, 4, S("a1234klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 9, SV("12345"), 0, 5, S("a12345klmnopqrst"));

    return true;
}

template <class S, class SV>
TEST_CONSTEXPR_CXX20 bool test43()
{
    test(S("abcdefghijklmnopqrst"), 1, 9, SV("12345"), 0, 6, S("a12345klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 9, SV("12345"), 1, 0, S("aklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 9, SV("12345"), 1, 1, S("a2klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 9, SV("12345"), 1, 2, S("a23klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 9, SV("12345"), 1, 3, S("a234klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 9, SV("12345"), 1, 4, S("a2345klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 9, SV("12345"), 1, 5, S("a2345klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 9, SV("12345"), 2, 0, S("aklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 9, SV("12345"), 2, 1, S("a3klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 9, SV("12345"), 2, 2, S("a34klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 9, SV("12345"), 2, 3, S("a345klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 9, SV("12345"), 2, 4, S("a345klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 9, SV("12345"), 4, 0, S("aklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 9, SV("12345"), 4, 1, S("a5klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 9, SV("12345"), 4, 2, S("a5klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 9, SV("12345"), 5, 0, S("aklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 9, SV("12345"), 5, 1, S("aklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 9, SV("12345"), 6, 0, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 1, 9, SV("1234567890"), 0, 0, S("aklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 9, SV("1234567890"), 0, 1, S("a1klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 9, SV("1234567890"), 0, 5, S("a12345klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 9, SV("1234567890"), 0, 9, S("a123456789klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 9, SV("1234567890"), 0, 10, S("a1234567890klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 9, SV("1234567890"), 0, 11, S("a1234567890klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 9, SV("1234567890"), 1, 0, S("aklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 9, SV("1234567890"), 1, 1, S("a2klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 9, SV("1234567890"), 1, 4, S("a2345klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 9, SV("1234567890"), 1, 8, S("a23456789klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 9, SV("1234567890"), 1, 9, S("a234567890klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 9, SV("1234567890"), 1, 10, S("a234567890klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 9, SV("1234567890"), 5, 0, S("aklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 9, SV("1234567890"), 5, 1, S("a6klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 9, SV("1234567890"), 5, 2, S("a67klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 9, SV("1234567890"), 5, 4, S("a6789klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 9, SV("1234567890"), 5, 5, S("a67890klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 9, SV("1234567890"), 5, 6, S("a67890klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 9, SV("1234567890"), 9, 0, S("aklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 9, SV("1234567890"), 9, 1, S("a0klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 9, SV("1234567890"), 9, 2, S("a0klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 9, SV("1234567890"), 10, 0, S("aklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 9, SV("1234567890"), 10, 1, S("aklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 9, SV("1234567890"), 11, 0, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 1, 9, SV("12345678901234567890"), 0, 0, S("aklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 9, SV("12345678901234567890"), 0, 1, S("a1klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 9, SV("12345678901234567890"), 0, 10, S("a1234567890klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 9, SV("12345678901234567890"), 0, 19, S("a1234567890123456789klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 9, SV("12345678901234567890"), 0, 20, S("a12345678901234567890klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 9, SV("12345678901234567890"), 0, 21, S("a12345678901234567890klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 9, SV("12345678901234567890"), 1, 0, S("aklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 9, SV("12345678901234567890"), 1, 1, S("a2klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 9, SV("12345678901234567890"), 1, 9, S("a234567890klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 9, SV("12345678901234567890"), 1, 18, S("a234567890123456789klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 9, SV("12345678901234567890"), 1, 19, S("a2345678901234567890klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 9, SV("12345678901234567890"), 1, 20, S("a2345678901234567890klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 9, SV("12345678901234567890"), 10, 0, S("aklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 9, SV("12345678901234567890"), 10, 1, S("a1klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 9, SV("12345678901234567890"), 10, 5, S("a12345klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 9, SV("12345678901234567890"), 10, 9, S("a123456789klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 9, SV("12345678901234567890"), 10, 10, S("a1234567890klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 9, SV("12345678901234567890"), 10, 11, S("a1234567890klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 9, SV("12345678901234567890"), 19, 0, S("aklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 9, SV("12345678901234567890"), 19, 1, S("a0klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 9, SV("12345678901234567890"), 19, 2, S("a0klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 9, SV("12345678901234567890"), 20, 0, S("aklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 9, SV("12345678901234567890"), 20, 1, S("aklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 1, 9, SV("12345678901234567890"), 21, 0, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 1, 18, SV(""), 0, 0, S("at"));
    test(S("abcdefghijklmnopqrst"), 1, 18, SV(""), 0, 1, S("at"));
    test(S("abcdefghijklmnopqrst"), 1, 18, SV(""), 1, 0, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 1, 18, SV("12345"), 0, 0, S("at"));
    test(S("abcdefghijklmnopqrst"), 1, 18, SV("12345"), 0, 1, S("a1t"));
    test(S("abcdefghijklmnopqrst"), 1, 18, SV("12345"), 0, 2, S("a12t"));
    test(S("abcdefghijklmnopqrst"), 1, 18, SV("12345"), 0, 4, S("a1234t"));
    test(S("abcdefghijklmnopqrst"), 1, 18, SV("12345"), 0, 5, S("a12345t"));
    test(S("abcdefghijklmnopqrst"), 1, 18, SV("12345"), 0, 6, S("a12345t"));
    test(S("abcdefghijklmnopqrst"), 1, 18, SV("12345"), 1, 0, S("at"));
    test(S("abcdefghijklmnopqrst"), 1, 18, SV("12345"), 1, 1, S("a2t"));
    test(S("abcdefghijklmnopqrst"), 1, 18, SV("12345"), 1, 2, S("a23t"));
    test(S("abcdefghijklmnopqrst"), 1, 18, SV("12345"), 1, 3, S("a234t"));
    test(S("abcdefghijklmnopqrst"), 1, 18, SV("12345"), 1, 4, S("a2345t"));
    test(S("abcdefghijklmnopqrst"), 1, 18, SV("12345"), 1, 5, S("a2345t"));
    test(S("abcdefghijklmnopqrst"), 1, 18, SV("12345"), 2, 0, S("at"));
    test(S("abcdefghijklmnopqrst"), 1, 18, SV("12345"), 2, 1, S("a3t"));
    test(S("abcdefghijklmnopqrst"), 1, 18, SV("12345"), 2, 2, S("a34t"));
    test(S("abcdefghijklmnopqrst"), 1, 18, SV("12345"), 2, 3, S("a345t"));
    test(S("abcdefghijklmnopqrst"), 1, 18, SV("12345"), 2, 4, S("a345t"));
    test(S("abcdefghijklmnopqrst"), 1, 18, SV("12345"), 4, 0, S("at"));
    test(S("abcdefghijklmnopqrst"), 1, 18, SV("12345"), 4, 1, S("a5t"));
    test(S("abcdefghijklmnopqrst"), 1, 18, SV("12345"), 4, 2, S("a5t"));
    test(S("abcdefghijklmnopqrst"), 1, 18, SV("12345"), 5, 0, S("at"));
    test(S("abcdefghijklmnopqrst"), 1, 18, SV("12345"), 5, 1, S("at"));
    test(S("abcdefghijklmnopqrst"), 1, 18, SV("12345"), 6, 0, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 1, 18, SV("1234567890"), 0, 0, S("at"));
    test(S("abcdefghijklmnopqrst"), 1, 18, SV("1234567890"), 0, 1, S("a1t"));
    test(S("abcdefghijklmnopqrst"), 1, 18, SV("1234567890"), 0, 5, S("a12345t"));
    test(S("abcdefghijklmnopqrst"), 1, 18, SV("1234567890"), 0, 9, S("a123456789t"));
    test(S("abcdefghijklmnopqrst"), 1, 18, SV("1234567890"), 0, 10, S("a1234567890t"));
    test(S("abcdefghijklmnopqrst"), 1, 18, SV("1234567890"), 0, 11, S("a1234567890t"));
    test(S("abcdefghijklmnopqrst"), 1, 18, SV("1234567890"), 1, 0, S("at"));
    test(S("abcdefghijklmnopqrst"), 1, 18, SV("1234567890"), 1, 1, S("a2t"));

    return true;
}

template <class S, class SV>
TEST_CONSTEXPR_CXX20 bool test44()
{
    test(S("abcdefghijklmnopqrst"), 1, 18, SV("1234567890"), 1, 4, S("a2345t"));
    test(S("abcdefghijklmnopqrst"), 1, 18, SV("1234567890"), 1, 8, S("a23456789t"));
    test(S("abcdefghijklmnopqrst"), 1, 18, SV("1234567890"), 1, 9, S("a234567890t"));
    test(S("abcdefghijklmnopqrst"), 1, 18, SV("1234567890"), 1, 10, S("a234567890t"));
    test(S("abcdefghijklmnopqrst"), 1, 18, SV("1234567890"), 5, 0, S("at"));
    test(S("abcdefghijklmnopqrst"), 1, 18, SV("1234567890"), 5, 1, S("a6t"));
    test(S("abcdefghijklmnopqrst"), 1, 18, SV("1234567890"), 5, 2, S("a67t"));
    test(S("abcdefghijklmnopqrst"), 1, 18, SV("1234567890"), 5, 4, S("a6789t"));
    test(S("abcdefghijklmnopqrst"), 1, 18, SV("1234567890"), 5, 5, S("a67890t"));
    test(S("abcdefghijklmnopqrst"), 1, 18, SV("1234567890"), 5, 6, S("a67890t"));
    test(S("abcdefghijklmnopqrst"), 1, 18, SV("1234567890"), 9, 0, S("at"));
    test(S("abcdefghijklmnopqrst"), 1, 18, SV("1234567890"), 9, 1, S("a0t"));
    test(S("abcdefghijklmnopqrst"), 1, 18, SV("1234567890"), 9, 2, S("a0t"));
    test(S("abcdefghijklmnopqrst"), 1, 18, SV("1234567890"), 10, 0, S("at"));
    test(S("abcdefghijklmnopqrst"), 1, 18, SV("1234567890"), 10, 1, S("at"));
    test(S("abcdefghijklmnopqrst"), 1, 18, SV("1234567890"), 11, 0, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 1, 18, SV("12345678901234567890"), 0, 0, S("at"));
    test(S("abcdefghijklmnopqrst"), 1, 18, SV("12345678901234567890"), 0, 1, S("a1t"));
    test(S("abcdefghijklmnopqrst"), 1, 18, SV("12345678901234567890"), 0, 10, S("a1234567890t"));
    test(S("abcdefghijklmnopqrst"), 1, 18, SV("12345678901234567890"), 0, 19, S("a1234567890123456789t"));
    test(S("abcdefghijklmnopqrst"), 1, 18, SV("12345678901234567890"), 0, 20, S("a12345678901234567890t"));
    test(S("abcdefghijklmnopqrst"), 1, 18, SV("12345678901234567890"), 0, 21, S("a12345678901234567890t"));
    test(S("abcdefghijklmnopqrst"), 1, 18, SV("12345678901234567890"), 1, 0, S("at"));
    test(S("abcdefghijklmnopqrst"), 1, 18, SV("12345678901234567890"), 1, 1, S("a2t"));
    test(S("abcdefghijklmnopqrst"), 1, 18, SV("12345678901234567890"), 1, 9, S("a234567890t"));
    test(S("abcdefghijklmnopqrst"), 1, 18, SV("12345678901234567890"), 1, 18, S("a234567890123456789t"));
    test(S("abcdefghijklmnopqrst"), 1, 18, SV("12345678901234567890"), 1, 19, S("a2345678901234567890t"));
    test(S("abcdefghijklmnopqrst"), 1, 18, SV("12345678901234567890"), 1, 20, S("a2345678901234567890t"));
    test(S("abcdefghijklmnopqrst"), 1, 18, SV("12345678901234567890"), 10, 0, S("at"));
    test(S("abcdefghijklmnopqrst"), 1, 18, SV("12345678901234567890"), 10, 1, S("a1t"));
    test(S("abcdefghijklmnopqrst"), 1, 18, SV("12345678901234567890"), 10, 5, S("a12345t"));
    test(S("abcdefghijklmnopqrst"), 1, 18, SV("12345678901234567890"), 10, 9, S("a123456789t"));
    test(S("abcdefghijklmnopqrst"), 1, 18, SV("12345678901234567890"), 10, 10, S("a1234567890t"));
    test(S("abcdefghijklmnopqrst"), 1, 18, SV("12345678901234567890"), 10, 11, S("a1234567890t"));
    test(S("abcdefghijklmnopqrst"), 1, 18, SV("12345678901234567890"), 19, 0, S("at"));
    test(S("abcdefghijklmnopqrst"), 1, 18, SV("12345678901234567890"), 19, 1, S("a0t"));
    test(S("abcdefghijklmnopqrst"), 1, 18, SV("12345678901234567890"), 19, 2, S("a0t"));
    test(S("abcdefghijklmnopqrst"), 1, 18, SV("12345678901234567890"), 20, 0, S("at"));
    test(S("abcdefghijklmnopqrst"), 1, 18, SV("12345678901234567890"), 20, 1, S("at"));
    test(S("abcdefghijklmnopqrst"), 1, 18, SV("12345678901234567890"), 21, 0, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 1, 19, SV(""), 0, 0, S("a"));
    test(S("abcdefghijklmnopqrst"), 1, 19, SV(""), 0, 1, S("a"));
    test(S("abcdefghijklmnopqrst"), 1, 19, SV(""), 1, 0, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 1, 19, SV("12345"), 0, 0, S("a"));
    test(S("abcdefghijklmnopqrst"), 1, 19, SV("12345"), 0, 1, S("a1"));
    test(S("abcdefghijklmnopqrst"), 1, 19, SV("12345"), 0, 2, S("a12"));
    test(S("abcdefghijklmnopqrst"), 1, 19, SV("12345"), 0, 4, S("a1234"));
    test(S("abcdefghijklmnopqrst"), 1, 19, SV("12345"), 0, 5, S("a12345"));
    test(S("abcdefghijklmnopqrst"), 1, 19, SV("12345"), 0, 6, S("a12345"));
    test(S("abcdefghijklmnopqrst"), 1, 19, SV("12345"), 1, 0, S("a"));
    test(S("abcdefghijklmnopqrst"), 1, 19, SV("12345"), 1, 1, S("a2"));
    test(S("abcdefghijklmnopqrst"), 1, 19, SV("12345"), 1, 2, S("a23"));
    test(S("abcdefghijklmnopqrst"), 1, 19, SV("12345"), 1, 3, S("a234"));
    test(S("abcdefghijklmnopqrst"), 1, 19, SV("12345"), 1, 4, S("a2345"));
    test(S("abcdefghijklmnopqrst"), 1, 19, SV("12345"), 1, 5, S("a2345"));
    test(S("abcdefghijklmnopqrst"), 1, 19, SV("12345"), 2, 0, S("a"));
    test(S("abcdefghijklmnopqrst"), 1, 19, SV("12345"), 2, 1, S("a3"));
    test(S("abcdefghijklmnopqrst"), 1, 19, SV("12345"), 2, 2, S("a34"));
    test(S("abcdefghijklmnopqrst"), 1, 19, SV("12345"), 2, 3, S("a345"));
    test(S("abcdefghijklmnopqrst"), 1, 19, SV("12345"), 2, 4, S("a345"));
    test(S("abcdefghijklmnopqrst"), 1, 19, SV("12345"), 4, 0, S("a"));
    test(S("abcdefghijklmnopqrst"), 1, 19, SV("12345"), 4, 1, S("a5"));
    test(S("abcdefghijklmnopqrst"), 1, 19, SV("12345"), 4, 2, S("a5"));
    test(S("abcdefghijklmnopqrst"), 1, 19, SV("12345"), 5, 0, S("a"));
    test(S("abcdefghijklmnopqrst"), 1, 19, SV("12345"), 5, 1, S("a"));
    test(S("abcdefghijklmnopqrst"), 1, 19, SV("12345"), 6, 0, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 1, 19, SV("1234567890"), 0, 0, S("a"));
    test(S("abcdefghijklmnopqrst"), 1, 19, SV("1234567890"), 0, 1, S("a1"));
    test(S("abcdefghijklmnopqrst"), 1, 19, SV("1234567890"), 0, 5, S("a12345"));
    test(S("abcdefghijklmnopqrst"), 1, 19, SV("1234567890"), 0, 9, S("a123456789"));
    test(S("abcdefghijklmnopqrst"), 1, 19, SV("1234567890"), 0, 10, S("a1234567890"));
    test(S("abcdefghijklmnopqrst"), 1, 19, SV("1234567890"), 0, 11, S("a1234567890"));
    test(S("abcdefghijklmnopqrst"), 1, 19, SV("1234567890"), 1, 0, S("a"));
    test(S("abcdefghijklmnopqrst"), 1, 19, SV("1234567890"), 1, 1, S("a2"));
    test(S("abcdefghijklmnopqrst"), 1, 19, SV("1234567890"), 1, 4, S("a2345"));
    test(S("abcdefghijklmnopqrst"), 1, 19, SV("1234567890"), 1, 8, S("a23456789"));
    test(S("abcdefghijklmnopqrst"), 1, 19, SV("1234567890"), 1, 9, S("a234567890"));
    test(S("abcdefghijklmnopqrst"), 1, 19, SV("1234567890"), 1, 10, S("a234567890"));
    test(S("abcdefghijklmnopqrst"), 1, 19, SV("1234567890"), 5, 0, S("a"));
    test(S("abcdefghijklmnopqrst"), 1, 19, SV("1234567890"), 5, 1, S("a6"));
    test(S("abcdefghijklmnopqrst"), 1, 19, SV("1234567890"), 5, 2, S("a67"));
    test(S("abcdefghijklmnopqrst"), 1, 19, SV("1234567890"), 5, 4, S("a6789"));
    test(S("abcdefghijklmnopqrst"), 1, 19, SV("1234567890"), 5, 5, S("a67890"));
    test(S("abcdefghijklmnopqrst"), 1, 19, SV("1234567890"), 5, 6, S("a67890"));
    test(S("abcdefghijklmnopqrst"), 1, 19, SV("1234567890"), 9, 0, S("a"));
    test(S("abcdefghijklmnopqrst"), 1, 19, SV("1234567890"), 9, 1, S("a0"));
    test(S("abcdefghijklmnopqrst"), 1, 19, SV("1234567890"), 9, 2, S("a0"));
    test(S("abcdefghijklmnopqrst"), 1, 19, SV("1234567890"), 10, 0, S("a"));
    test(S("abcdefghijklmnopqrst"), 1, 19, SV("1234567890"), 10, 1, S("a"));
    test(S("abcdefghijklmnopqrst"), 1, 19, SV("1234567890"), 11, 0, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 1, 19, SV("12345678901234567890"), 0, 0, S("a"));
    test(S("abcdefghijklmnopqrst"), 1, 19, SV("12345678901234567890"), 0, 1, S("a1"));
    test(S("abcdefghijklmnopqrst"), 1, 19, SV("12345678901234567890"), 0, 10, S("a1234567890"));
    test(S("abcdefghijklmnopqrst"), 1, 19, SV("12345678901234567890"), 0, 19, S("a1234567890123456789"));
    test(S("abcdefghijklmnopqrst"), 1, 19, SV("12345678901234567890"), 0, 20, S("a12345678901234567890"));
    test(S("abcdefghijklmnopqrst"), 1, 19, SV("12345678901234567890"), 0, 21, S("a12345678901234567890"));
    test(S("abcdefghijklmnopqrst"), 1, 19, SV("12345678901234567890"), 1, 0, S("a"));
    test(S("abcdefghijklmnopqrst"), 1, 19, SV("12345678901234567890"), 1, 1, S("a2"));
    test(S("abcdefghijklmnopqrst"), 1, 19, SV("12345678901234567890"), 1, 9, S("a234567890"));
    test(S("abcdefghijklmnopqrst"), 1, 19, SV("12345678901234567890"), 1, 18, S("a234567890123456789"));

    return true;
}

template <class S, class SV>
TEST_CONSTEXPR_CXX20 bool test45()
{
    test(S("abcdefghijklmnopqrst"), 1, 19, SV("12345678901234567890"), 1, 19, S("a2345678901234567890"));
    test(S("abcdefghijklmnopqrst"), 1, 19, SV("12345678901234567890"), 1, 20, S("a2345678901234567890"));
    test(S("abcdefghijklmnopqrst"), 1, 19, SV("12345678901234567890"), 10, 0, S("a"));
    test(S("abcdefghijklmnopqrst"), 1, 19, SV("12345678901234567890"), 10, 1, S("a1"));
    test(S("abcdefghijklmnopqrst"), 1, 19, SV("12345678901234567890"), 10, 5, S("a12345"));
    test(S("abcdefghijklmnopqrst"), 1, 19, SV("12345678901234567890"), 10, 9, S("a123456789"));
    test(S("abcdefghijklmnopqrst"), 1, 19, SV("12345678901234567890"), 10, 10, S("a1234567890"));
    test(S("abcdefghijklmnopqrst"), 1, 19, SV("12345678901234567890"), 10, 11, S("a1234567890"));
    test(S("abcdefghijklmnopqrst"), 1, 19, SV("12345678901234567890"), 19, 0, S("a"));
    test(S("abcdefghijklmnopqrst"), 1, 19, SV("12345678901234567890"), 19, 1, S("a0"));
    test(S("abcdefghijklmnopqrst"), 1, 19, SV("12345678901234567890"), 19, 2, S("a0"));
    test(S("abcdefghijklmnopqrst"), 1, 19, SV("12345678901234567890"), 20, 0, S("a"));
    test(S("abcdefghijklmnopqrst"), 1, 19, SV("12345678901234567890"), 20, 1, S("a"));
    test(S("abcdefghijklmnopqrst"), 1, 19, SV("12345678901234567890"), 21, 0, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 1, 20, SV(""), 0, 0, S("a"));
    test(S("abcdefghijklmnopqrst"), 1, 20, SV(""), 0, 1, S("a"));
    test(S("abcdefghijklmnopqrst"), 1, 20, SV(""), 1, 0, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 1, 20, SV("12345"), 0, 0, S("a"));
    test(S("abcdefghijklmnopqrst"), 1, 20, SV("12345"), 0, 1, S("a1"));
    test(S("abcdefghijklmnopqrst"), 1, 20, SV("12345"), 0, 2, S("a12"));
    test(S("abcdefghijklmnopqrst"), 1, 20, SV("12345"), 0, 4, S("a1234"));
    test(S("abcdefghijklmnopqrst"), 1, 20, SV("12345"), 0, 5, S("a12345"));
    test(S("abcdefghijklmnopqrst"), 1, 20, SV("12345"), 0, 6, S("a12345"));
    test(S("abcdefghijklmnopqrst"), 1, 20, SV("12345"), 1, 0, S("a"));
    test(S("abcdefghijklmnopqrst"), 1, 20, SV("12345"), 1, 1, S("a2"));
    test(S("abcdefghijklmnopqrst"), 1, 20, SV("12345"), 1, 2, S("a23"));
    test(S("abcdefghijklmnopqrst"), 1, 20, SV("12345"), 1, 3, S("a234"));
    test(S("abcdefghijklmnopqrst"), 1, 20, SV("12345"), 1, 4, S("a2345"));
    test(S("abcdefghijklmnopqrst"), 1, 20, SV("12345"), 1, 5, S("a2345"));
    test(S("abcdefghijklmnopqrst"), 1, 20, SV("12345"), 2, 0, S("a"));
    test(S("abcdefghijklmnopqrst"), 1, 20, SV("12345"), 2, 1, S("a3"));
    test(S("abcdefghijklmnopqrst"), 1, 20, SV("12345"), 2, 2, S("a34"));
    test(S("abcdefghijklmnopqrst"), 1, 20, SV("12345"), 2, 3, S("a345"));
    test(S("abcdefghijklmnopqrst"), 1, 20, SV("12345"), 2, 4, S("a345"));
    test(S("abcdefghijklmnopqrst"), 1, 20, SV("12345"), 4, 0, S("a"));
    test(S("abcdefghijklmnopqrst"), 1, 20, SV("12345"), 4, 1, S("a5"));
    test(S("abcdefghijklmnopqrst"), 1, 20, SV("12345"), 4, 2, S("a5"));
    test(S("abcdefghijklmnopqrst"), 1, 20, SV("12345"), 5, 0, S("a"));
    test(S("abcdefghijklmnopqrst"), 1, 20, SV("12345"), 5, 1, S("a"));
    test(S("abcdefghijklmnopqrst"), 1, 20, SV("12345"), 6, 0, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 1, 20, SV("1234567890"), 0, 0, S("a"));
    test(S("abcdefghijklmnopqrst"), 1, 20, SV("1234567890"), 0, 1, S("a1"));
    test(S("abcdefghijklmnopqrst"), 1, 20, SV("1234567890"), 0, 5, S("a12345"));
    test(S("abcdefghijklmnopqrst"), 1, 20, SV("1234567890"), 0, 9, S("a123456789"));
    test(S("abcdefghijklmnopqrst"), 1, 20, SV("1234567890"), 0, 10, S("a1234567890"));
    test(S("abcdefghijklmnopqrst"), 1, 20, SV("1234567890"), 0, 11, S("a1234567890"));
    test(S("abcdefghijklmnopqrst"), 1, 20, SV("1234567890"), 1, 0, S("a"));
    test(S("abcdefghijklmnopqrst"), 1, 20, SV("1234567890"), 1, 1, S("a2"));
    test(S("abcdefghijklmnopqrst"), 1, 20, SV("1234567890"), 1, 4, S("a2345"));
    test(S("abcdefghijklmnopqrst"), 1, 20, SV("1234567890"), 1, 8, S("a23456789"));
    test(S("abcdefghijklmnopqrst"), 1, 20, SV("1234567890"), 1, 9, S("a234567890"));
    test(S("abcdefghijklmnopqrst"), 1, 20, SV("1234567890"), 1, 10, S("a234567890"));
    test(S("abcdefghijklmnopqrst"), 1, 20, SV("1234567890"), 5, 0, S("a"));
    test(S("abcdefghijklmnopqrst"), 1, 20, SV("1234567890"), 5, 1, S("a6"));
    test(S("abcdefghijklmnopqrst"), 1, 20, SV("1234567890"), 5, 2, S("a67"));
    test(S("abcdefghijklmnopqrst"), 1, 20, SV("1234567890"), 5, 4, S("a6789"));
    test(S("abcdefghijklmnopqrst"), 1, 20, SV("1234567890"), 5, 5, S("a67890"));
    test(S("abcdefghijklmnopqrst"), 1, 20, SV("1234567890"), 5, 6, S("a67890"));
    test(S("abcdefghijklmnopqrst"), 1, 20, SV("1234567890"), 9, 0, S("a"));
    test(S("abcdefghijklmnopqrst"), 1, 20, SV("1234567890"), 9, 1, S("a0"));
    test(S("abcdefghijklmnopqrst"), 1, 20, SV("1234567890"), 9, 2, S("a0"));
    test(S("abcdefghijklmnopqrst"), 1, 20, SV("1234567890"), 10, 0, S("a"));
    test(S("abcdefghijklmnopqrst"), 1, 20, SV("1234567890"), 10, 1, S("a"));
    test(S("abcdefghijklmnopqrst"), 1, 20, SV("1234567890"), 11, 0, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 1, 20, SV("12345678901234567890"), 0, 0, S("a"));
    test(S("abcdefghijklmnopqrst"), 1, 20, SV("12345678901234567890"), 0, 1, S("a1"));
    test(S("abcdefghijklmnopqrst"), 1, 20, SV("12345678901234567890"), 0, 10, S("a1234567890"));
    test(S("abcdefghijklmnopqrst"), 1, 20, SV("12345678901234567890"), 0, 19, S("a1234567890123456789"));
    test(S("abcdefghijklmnopqrst"), 1, 20, SV("12345678901234567890"), 0, 20, S("a12345678901234567890"));
    test(S("abcdefghijklmnopqrst"), 1, 20, SV("12345678901234567890"), 0, 21, S("a12345678901234567890"));
    test(S("abcdefghijklmnopqrst"), 1, 20, SV("12345678901234567890"), 1, 0, S("a"));
    test(S("abcdefghijklmnopqrst"), 1, 20, SV("12345678901234567890"), 1, 1, S("a2"));
    test(S("abcdefghijklmnopqrst"), 1, 20, SV("12345678901234567890"), 1, 9, S("a234567890"));
    test(S("abcdefghijklmnopqrst"), 1, 20, SV("12345678901234567890"), 1, 18, S("a234567890123456789"));
    test(S("abcdefghijklmnopqrst"), 1, 20, SV("12345678901234567890"), 1, 19, S("a2345678901234567890"));
    test(S("abcdefghijklmnopqrst"), 1, 20, SV("12345678901234567890"), 1, 20, S("a2345678901234567890"));
    test(S("abcdefghijklmnopqrst"), 1, 20, SV("12345678901234567890"), 10, 0, S("a"));
    test(S("abcdefghijklmnopqrst"), 1, 20, SV("12345678901234567890"), 10, 1, S("a1"));
    test(S("abcdefghijklmnopqrst"), 1, 20, SV("12345678901234567890"), 10, 5, S("a12345"));
    test(S("abcdefghijklmnopqrst"), 1, 20, SV("12345678901234567890"), 10, 9, S("a123456789"));
    test(S("abcdefghijklmnopqrst"), 1, 20, SV("12345678901234567890"), 10, 10, S("a1234567890"));
    test(S("abcdefghijklmnopqrst"), 1, 20, SV("12345678901234567890"), 10, 11, S("a1234567890"));
    test(S("abcdefghijklmnopqrst"), 1, 20, SV("12345678901234567890"), 19, 0, S("a"));
    test(S("abcdefghijklmnopqrst"), 1, 20, SV("12345678901234567890"), 19, 1, S("a0"));
    test(S("abcdefghijklmnopqrst"), 1, 20, SV("12345678901234567890"), 19, 2, S("a0"));
    test(S("abcdefghijklmnopqrst"), 1, 20, SV("12345678901234567890"), 20, 0, S("a"));
    test(S("abcdefghijklmnopqrst"), 1, 20, SV("12345678901234567890"), 20, 1, S("a"));
    test(S("abcdefghijklmnopqrst"), 1, 20, SV("12345678901234567890"), 21, 0, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 10, 0, SV(""), 0, 0, S("abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 0, SV(""), 0, 1, S("abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 0, SV(""), 1, 0, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 10, 0, SV("12345"), 0, 0, S("abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 0, SV("12345"), 0, 1, S("abcdefghij1klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 0, SV("12345"), 0, 2, S("abcdefghij12klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 0, SV("12345"), 0, 4, S("abcdefghij1234klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 0, SV("12345"), 0, 5, S("abcdefghij12345klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 0, SV("12345"), 0, 6, S("abcdefghij12345klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 0, SV("12345"), 1, 0, S("abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 0, SV("12345"), 1, 1, S("abcdefghij2klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 0, SV("12345"), 1, 2, S("abcdefghij23klmnopqrst"));

    return true;
}

template <class S, class SV>
TEST_CONSTEXPR_CXX20 bool test46()
{
    test(S("abcdefghijklmnopqrst"), 10, 0, SV("12345"), 1, 3, S("abcdefghij234klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 0, SV("12345"), 1, 4, S("abcdefghij2345klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 0, SV("12345"), 1, 5, S("abcdefghij2345klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 0, SV("12345"), 2, 0, S("abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 0, SV("12345"), 2, 1, S("abcdefghij3klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 0, SV("12345"), 2, 2, S("abcdefghij34klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 0, SV("12345"), 2, 3, S("abcdefghij345klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 0, SV("12345"), 2, 4, S("abcdefghij345klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 0, SV("12345"), 4, 0, S("abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 0, SV("12345"), 4, 1, S("abcdefghij5klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 0, SV("12345"), 4, 2, S("abcdefghij5klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 0, SV("12345"), 5, 0, S("abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 0, SV("12345"), 5, 1, S("abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 0, SV("12345"), 6, 0, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 10, 0, SV("1234567890"), 0, 0, S("abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 0, SV("1234567890"), 0, 1, S("abcdefghij1klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 0, SV("1234567890"), 0, 5, S("abcdefghij12345klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 0, SV("1234567890"), 0, 9, S("abcdefghij123456789klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 0, SV("1234567890"), 0, 10, S("abcdefghij1234567890klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 0, SV("1234567890"), 0, 11, S("abcdefghij1234567890klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 0, SV("1234567890"), 1, 0, S("abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 0, SV("1234567890"), 1, 1, S("abcdefghij2klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 0, SV("1234567890"), 1, 4, S("abcdefghij2345klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 0, SV("1234567890"), 1, 8, S("abcdefghij23456789klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 0, SV("1234567890"), 1, 9, S("abcdefghij234567890klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 0, SV("1234567890"), 1, 10, S("abcdefghij234567890klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 0, SV("1234567890"), 5, 0, S("abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 0, SV("1234567890"), 5, 1, S("abcdefghij6klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 0, SV("1234567890"), 5, 2, S("abcdefghij67klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 0, SV("1234567890"), 5, 4, S("abcdefghij6789klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 0, SV("1234567890"), 5, 5, S("abcdefghij67890klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 0, SV("1234567890"), 5, 6, S("abcdefghij67890klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 0, SV("1234567890"), 9, 0, S("abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 0, SV("1234567890"), 9, 1, S("abcdefghij0klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 0, SV("1234567890"), 9, 2, S("abcdefghij0klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 0, SV("1234567890"), 10, 0, S("abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 0, SV("1234567890"), 10, 1, S("abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 0, SV("1234567890"), 11, 0, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 10, 0, SV("12345678901234567890"), 0, 0, S("abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 0, SV("12345678901234567890"), 0, 1, S("abcdefghij1klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 0, SV("12345678901234567890"), 0, 10, S("abcdefghij1234567890klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 0, SV("12345678901234567890"), 0, 19, S("abcdefghij1234567890123456789klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 0, SV("12345678901234567890"), 0, 20, S("abcdefghij12345678901234567890klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 0, SV("12345678901234567890"), 0, 21, S("abcdefghij12345678901234567890klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 0, SV("12345678901234567890"), 1, 0, S("abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 0, SV("12345678901234567890"), 1, 1, S("abcdefghij2klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 0, SV("12345678901234567890"), 1, 9, S("abcdefghij234567890klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 0, SV("12345678901234567890"), 1, 18, S("abcdefghij234567890123456789klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 0, SV("12345678901234567890"), 1, 19, S("abcdefghij2345678901234567890klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 0, SV("12345678901234567890"), 1, 20, S("abcdefghij2345678901234567890klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 0, SV("12345678901234567890"), 10, 0, S("abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 0, SV("12345678901234567890"), 10, 1, S("abcdefghij1klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 0, SV("12345678901234567890"), 10, 5, S("abcdefghij12345klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 0, SV("12345678901234567890"), 10, 9, S("abcdefghij123456789klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 0, SV("12345678901234567890"), 10, 10, S("abcdefghij1234567890klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 0, SV("12345678901234567890"), 10, 11, S("abcdefghij1234567890klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 0, SV("12345678901234567890"), 19, 0, S("abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 0, SV("12345678901234567890"), 19, 1, S("abcdefghij0klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 0, SV("12345678901234567890"), 19, 2, S("abcdefghij0klmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 0, SV("12345678901234567890"), 20, 0, S("abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 0, SV("12345678901234567890"), 20, 1, S("abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 0, SV("12345678901234567890"), 21, 0, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 10, 1, SV(""), 0, 0, S("abcdefghijlmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 1, SV(""), 0, 1, S("abcdefghijlmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 1, SV(""), 1, 0, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 10, 1, SV("12345"), 0, 0, S("abcdefghijlmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 1, SV("12345"), 0, 1, S("abcdefghij1lmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 1, SV("12345"), 0, 2, S("abcdefghij12lmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 1, SV("12345"), 0, 4, S("abcdefghij1234lmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 1, SV("12345"), 0, 5, S("abcdefghij12345lmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 1, SV("12345"), 0, 6, S("abcdefghij12345lmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 1, SV("12345"), 1, 0, S("abcdefghijlmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 1, SV("12345"), 1, 1, S("abcdefghij2lmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 1, SV("12345"), 1, 2, S("abcdefghij23lmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 1, SV("12345"), 1, 3, S("abcdefghij234lmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 1, SV("12345"), 1, 4, S("abcdefghij2345lmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 1, SV("12345"), 1, 5, S("abcdefghij2345lmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 1, SV("12345"), 2, 0, S("abcdefghijlmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 1, SV("12345"), 2, 1, S("abcdefghij3lmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 1, SV("12345"), 2, 2, S("abcdefghij34lmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 1, SV("12345"), 2, 3, S("abcdefghij345lmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 1, SV("12345"), 2, 4, S("abcdefghij345lmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 1, SV("12345"), 4, 0, S("abcdefghijlmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 1, SV("12345"), 4, 1, S("abcdefghij5lmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 1, SV("12345"), 4, 2, S("abcdefghij5lmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 1, SV("12345"), 5, 0, S("abcdefghijlmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 1, SV("12345"), 5, 1, S("abcdefghijlmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 1, SV("12345"), 6, 0, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 10, 1, SV("1234567890"), 0, 0, S("abcdefghijlmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 1, SV("1234567890"), 0, 1, S("abcdefghij1lmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 1, SV("1234567890"), 0, 5, S("abcdefghij12345lmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 1, SV("1234567890"), 0, 9, S("abcdefghij123456789lmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 1, SV("1234567890"), 0, 10, S("abcdefghij1234567890lmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 1, SV("1234567890"), 0, 11, S("abcdefghij1234567890lmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 1, SV("1234567890"), 1, 0, S("abcdefghijlmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 1, SV("1234567890"), 1, 1, S("abcdefghij2lmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 1, SV("1234567890"), 1, 4, S("abcdefghij2345lmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 1, SV("1234567890"), 1, 8, S("abcdefghij23456789lmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 1, SV("1234567890"), 1, 9, S("abcdefghij234567890lmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 1, SV("1234567890"), 1, 10, S("abcdefghij234567890lmnopqrst"));

    return true;
}

template <class S, class SV>
TEST_CONSTEXPR_CXX20 bool test47()
{
    test(S("abcdefghijklmnopqrst"), 10, 1, SV("1234567890"), 5, 0, S("abcdefghijlmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 1, SV("1234567890"), 5, 1, S("abcdefghij6lmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 1, SV("1234567890"), 5, 2, S("abcdefghij67lmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 1, SV("1234567890"), 5, 4, S("abcdefghij6789lmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 1, SV("1234567890"), 5, 5, S("abcdefghij67890lmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 1, SV("1234567890"), 5, 6, S("abcdefghij67890lmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 1, SV("1234567890"), 9, 0, S("abcdefghijlmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 1, SV("1234567890"), 9, 1, S("abcdefghij0lmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 1, SV("1234567890"), 9, 2, S("abcdefghij0lmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 1, SV("1234567890"), 10, 0, S("abcdefghijlmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 1, SV("1234567890"), 10, 1, S("abcdefghijlmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 1, SV("1234567890"), 11, 0, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 10, 1, SV("12345678901234567890"), 0, 0, S("abcdefghijlmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 1, SV("12345678901234567890"), 0, 1, S("abcdefghij1lmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 1, SV("12345678901234567890"), 0, 10, S("abcdefghij1234567890lmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 1, SV("12345678901234567890"), 0, 19, S("abcdefghij1234567890123456789lmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 1, SV("12345678901234567890"), 0, 20, S("abcdefghij12345678901234567890lmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 1, SV("12345678901234567890"), 0, 21, S("abcdefghij12345678901234567890lmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 1, SV("12345678901234567890"), 1, 0, S("abcdefghijlmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 1, SV("12345678901234567890"), 1, 1, S("abcdefghij2lmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 1, SV("12345678901234567890"), 1, 9, S("abcdefghij234567890lmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 1, SV("12345678901234567890"), 1, 18, S("abcdefghij234567890123456789lmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 1, SV("12345678901234567890"), 1, 19, S("abcdefghij2345678901234567890lmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 1, SV("12345678901234567890"), 1, 20, S("abcdefghij2345678901234567890lmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 1, SV("12345678901234567890"), 10, 0, S("abcdefghijlmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 1, SV("12345678901234567890"), 10, 1, S("abcdefghij1lmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 1, SV("12345678901234567890"), 10, 5, S("abcdefghij12345lmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 1, SV("12345678901234567890"), 10, 9, S("abcdefghij123456789lmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 1, SV("12345678901234567890"), 10, 10, S("abcdefghij1234567890lmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 1, SV("12345678901234567890"), 10, 11, S("abcdefghij1234567890lmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 1, SV("12345678901234567890"), 19, 0, S("abcdefghijlmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 1, SV("12345678901234567890"), 19, 1, S("abcdefghij0lmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 1, SV("12345678901234567890"), 19, 2, S("abcdefghij0lmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 1, SV("12345678901234567890"), 20, 0, S("abcdefghijlmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 1, SV("12345678901234567890"), 20, 1, S("abcdefghijlmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 1, SV("12345678901234567890"), 21, 0, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 10, 5, SV(""), 0, 0, S("abcdefghijpqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 5, SV(""), 0, 1, S("abcdefghijpqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 5, SV(""), 1, 0, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 10, 5, SV("12345"), 0, 0, S("abcdefghijpqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 5, SV("12345"), 0, 1, S("abcdefghij1pqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 5, SV("12345"), 0, 2, S("abcdefghij12pqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 5, SV("12345"), 0, 4, S("abcdefghij1234pqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 5, SV("12345"), 0, 5, S("abcdefghij12345pqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 5, SV("12345"), 0, 6, S("abcdefghij12345pqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 5, SV("12345"), 1, 0, S("abcdefghijpqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 5, SV("12345"), 1, 1, S("abcdefghij2pqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 5, SV("12345"), 1, 2, S("abcdefghij23pqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 5, SV("12345"), 1, 3, S("abcdefghij234pqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 5, SV("12345"), 1, 4, S("abcdefghij2345pqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 5, SV("12345"), 1, 5, S("abcdefghij2345pqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 5, SV("12345"), 2, 0, S("abcdefghijpqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 5, SV("12345"), 2, 1, S("abcdefghij3pqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 5, SV("12345"), 2, 2, S("abcdefghij34pqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 5, SV("12345"), 2, 3, S("abcdefghij345pqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 5, SV("12345"), 2, 4, S("abcdefghij345pqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 5, SV("12345"), 4, 0, S("abcdefghijpqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 5, SV("12345"), 4, 1, S("abcdefghij5pqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 5, SV("12345"), 4, 2, S("abcdefghij5pqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 5, SV("12345"), 5, 0, S("abcdefghijpqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 5, SV("12345"), 5, 1, S("abcdefghijpqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 5, SV("12345"), 6, 0, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 10, 5, SV("1234567890"), 0, 0, S("abcdefghijpqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 5, SV("1234567890"), 0, 1, S("abcdefghij1pqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 5, SV("1234567890"), 0, 5, S("abcdefghij12345pqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 5, SV("1234567890"), 0, 9, S("abcdefghij123456789pqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 5, SV("1234567890"), 0, 10, S("abcdefghij1234567890pqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 5, SV("1234567890"), 0, 11, S("abcdefghij1234567890pqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 5, SV("1234567890"), 1, 0, S("abcdefghijpqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 5, SV("1234567890"), 1, 1, S("abcdefghij2pqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 5, SV("1234567890"), 1, 4, S("abcdefghij2345pqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 5, SV("1234567890"), 1, 8, S("abcdefghij23456789pqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 5, SV("1234567890"), 1, 9, S("abcdefghij234567890pqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 5, SV("1234567890"), 1, 10, S("abcdefghij234567890pqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 5, SV("1234567890"), 5, 0, S("abcdefghijpqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 5, SV("1234567890"), 5, 1, S("abcdefghij6pqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 5, SV("1234567890"), 5, 2, S("abcdefghij67pqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 5, SV("1234567890"), 5, 4, S("abcdefghij6789pqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 5, SV("1234567890"), 5, 5, S("abcdefghij67890pqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 5, SV("1234567890"), 5, 6, S("abcdefghij67890pqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 5, SV("1234567890"), 9, 0, S("abcdefghijpqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 5, SV("1234567890"), 9, 1, S("abcdefghij0pqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 5, SV("1234567890"), 9, 2, S("abcdefghij0pqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 5, SV("1234567890"), 10, 0, S("abcdefghijpqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 5, SV("1234567890"), 10, 1, S("abcdefghijpqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 5, SV("1234567890"), 11, 0, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 10, 5, SV("12345678901234567890"), 0, 0, S("abcdefghijpqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 5, SV("12345678901234567890"), 0, 1, S("abcdefghij1pqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 5, SV("12345678901234567890"), 0, 10, S("abcdefghij1234567890pqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 5, SV("12345678901234567890"), 0, 19, S("abcdefghij1234567890123456789pqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 5, SV("12345678901234567890"), 0, 20, S("abcdefghij12345678901234567890pqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 5, SV("12345678901234567890"), 0, 21, S("abcdefghij12345678901234567890pqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 5, SV("12345678901234567890"), 1, 0, S("abcdefghijpqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 5, SV("12345678901234567890"), 1, 1, S("abcdefghij2pqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 5, SV("12345678901234567890"), 1, 9, S("abcdefghij234567890pqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 5, SV("12345678901234567890"), 1, 18, S("abcdefghij234567890123456789pqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 5, SV("12345678901234567890"), 1, 19, S("abcdefghij2345678901234567890pqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 5, SV("12345678901234567890"), 1, 20, S("abcdefghij2345678901234567890pqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 5, SV("12345678901234567890"), 10, 0, S("abcdefghijpqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 5, SV("12345678901234567890"), 10, 1, S("abcdefghij1pqrst"));

    return true;
}

template <class S, class SV>
TEST_CONSTEXPR_CXX20 bool test48()
{
    test(S("abcdefghijklmnopqrst"), 10, 5, SV("12345678901234567890"), 10, 5, S("abcdefghij12345pqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 5, SV("12345678901234567890"), 10, 9, S("abcdefghij123456789pqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 5, SV("12345678901234567890"), 10, 10, S("abcdefghij1234567890pqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 5, SV("12345678901234567890"), 10, 11, S("abcdefghij1234567890pqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 5, SV("12345678901234567890"), 19, 0, S("abcdefghijpqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 5, SV("12345678901234567890"), 19, 1, S("abcdefghij0pqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 5, SV("12345678901234567890"), 19, 2, S("abcdefghij0pqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 5, SV("12345678901234567890"), 20, 0, S("abcdefghijpqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 5, SV("12345678901234567890"), 20, 1, S("abcdefghijpqrst"));
    test(S("abcdefghijklmnopqrst"), 10, 5, SV("12345678901234567890"), 21, 0, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 10, 9, SV(""), 0, 0, S("abcdefghijt"));
    test(S("abcdefghijklmnopqrst"), 10, 9, SV(""), 0, 1, S("abcdefghijt"));
    test(S("abcdefghijklmnopqrst"), 10, 9, SV(""), 1, 0, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 10, 9, SV("12345"), 0, 0, S("abcdefghijt"));
    test(S("abcdefghijklmnopqrst"), 10, 9, SV("12345"), 0, 1, S("abcdefghij1t"));
    test(S("abcdefghijklmnopqrst"), 10, 9, SV("12345"), 0, 2, S("abcdefghij12t"));
    test(S("abcdefghijklmnopqrst"), 10, 9, SV("12345"), 0, 4, S("abcdefghij1234t"));
    test(S("abcdefghijklmnopqrst"), 10, 9, SV("12345"), 0, 5, S("abcdefghij12345t"));
    test(S("abcdefghijklmnopqrst"), 10, 9, SV("12345"), 0, 6, S("abcdefghij12345t"));
    test(S("abcdefghijklmnopqrst"), 10, 9, SV("12345"), 1, 0, S("abcdefghijt"));
    test(S("abcdefghijklmnopqrst"), 10, 9, SV("12345"), 1, 1, S("abcdefghij2t"));
    test(S("abcdefghijklmnopqrst"), 10, 9, SV("12345"), 1, 2, S("abcdefghij23t"));
    test(S("abcdefghijklmnopqrst"), 10, 9, SV("12345"), 1, 3, S("abcdefghij234t"));
    test(S("abcdefghijklmnopqrst"), 10, 9, SV("12345"), 1, 4, S("abcdefghij2345t"));
    test(S("abcdefghijklmnopqrst"), 10, 9, SV("12345"), 1, 5, S("abcdefghij2345t"));
    test(S("abcdefghijklmnopqrst"), 10, 9, SV("12345"), 2, 0, S("abcdefghijt"));
    test(S("abcdefghijklmnopqrst"), 10, 9, SV("12345"), 2, 1, S("abcdefghij3t"));
    test(S("abcdefghijklmnopqrst"), 10, 9, SV("12345"), 2, 2, S("abcdefghij34t"));
    test(S("abcdefghijklmnopqrst"), 10, 9, SV("12345"), 2, 3, S("abcdefghij345t"));
    test(S("abcdefghijklmnopqrst"), 10, 9, SV("12345"), 2, 4, S("abcdefghij345t"));
    test(S("abcdefghijklmnopqrst"), 10, 9, SV("12345"), 4, 0, S("abcdefghijt"));
    test(S("abcdefghijklmnopqrst"), 10, 9, SV("12345"), 4, 1, S("abcdefghij5t"));
    test(S("abcdefghijklmnopqrst"), 10, 9, SV("12345"), 4, 2, S("abcdefghij5t"));
    test(S("abcdefghijklmnopqrst"), 10, 9, SV("12345"), 5, 0, S("abcdefghijt"));
    test(S("abcdefghijklmnopqrst"), 10, 9, SV("12345"), 5, 1, S("abcdefghijt"));
    test(S("abcdefghijklmnopqrst"), 10, 9, SV("12345"), 6, 0, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 10, 9, SV("1234567890"), 0, 0, S("abcdefghijt"));
    test(S("abcdefghijklmnopqrst"), 10, 9, SV("1234567890"), 0, 1, S("abcdefghij1t"));
    test(S("abcdefghijklmnopqrst"), 10, 9, SV("1234567890"), 0, 5, S("abcdefghij12345t"));
    test(S("abcdefghijklmnopqrst"), 10, 9, SV("1234567890"), 0, 9, S("abcdefghij123456789t"));
    test(S("abcdefghijklmnopqrst"), 10, 9, SV("1234567890"), 0, 10, S("abcdefghij1234567890t"));
    test(S("abcdefghijklmnopqrst"), 10, 9, SV("1234567890"), 0, 11, S("abcdefghij1234567890t"));
    test(S("abcdefghijklmnopqrst"), 10, 9, SV("1234567890"), 1, 0, S("abcdefghijt"));
    test(S("abcdefghijklmnopqrst"), 10, 9, SV("1234567890"), 1, 1, S("abcdefghij2t"));
    test(S("abcdefghijklmnopqrst"), 10, 9, SV("1234567890"), 1, 4, S("abcdefghij2345t"));
    test(S("abcdefghijklmnopqrst"), 10, 9, SV("1234567890"), 1, 8, S("abcdefghij23456789t"));
    test(S("abcdefghijklmnopqrst"), 10, 9, SV("1234567890"), 1, 9, S("abcdefghij234567890t"));
    test(S("abcdefghijklmnopqrst"), 10, 9, SV("1234567890"), 1, 10, S("abcdefghij234567890t"));
    test(S("abcdefghijklmnopqrst"), 10, 9, SV("1234567890"), 5, 0, S("abcdefghijt"));
    test(S("abcdefghijklmnopqrst"), 10, 9, SV("1234567890"), 5, 1, S("abcdefghij6t"));
    test(S("abcdefghijklmnopqrst"), 10, 9, SV("1234567890"), 5, 2, S("abcdefghij67t"));
    test(S("abcdefghijklmnopqrst"), 10, 9, SV("1234567890"), 5, 4, S("abcdefghij6789t"));
    test(S("abcdefghijklmnopqrst"), 10, 9, SV("1234567890"), 5, 5, S("abcdefghij67890t"));
    test(S("abcdefghijklmnopqrst"), 10, 9, SV("1234567890"), 5, 6, S("abcdefghij67890t"));
    test(S("abcdefghijklmnopqrst"), 10, 9, SV("1234567890"), 9, 0, S("abcdefghijt"));
    test(S("abcdefghijklmnopqrst"), 10, 9, SV("1234567890"), 9, 1, S("abcdefghij0t"));
    test(S("abcdefghijklmnopqrst"), 10, 9, SV("1234567890"), 9, 2, S("abcdefghij0t"));
    test(S("abcdefghijklmnopqrst"), 10, 9, SV("1234567890"), 10, 0, S("abcdefghijt"));
    test(S("abcdefghijklmnopqrst"), 10, 9, SV("1234567890"), 10, 1, S("abcdefghijt"));
    test(S("abcdefghijklmnopqrst"), 10, 9, SV("1234567890"), 11, 0, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 10, 9, SV("12345678901234567890"), 0, 0, S("abcdefghijt"));
    test(S("abcdefghijklmnopqrst"), 10, 9, SV("12345678901234567890"), 0, 1, S("abcdefghij1t"));
    test(S("abcdefghijklmnopqrst"), 10, 9, SV("12345678901234567890"), 0, 10, S("abcdefghij1234567890t"));
    test(S("abcdefghijklmnopqrst"), 10, 9, SV("12345678901234567890"), 0, 19, S("abcdefghij1234567890123456789t"));
    test(S("abcdefghijklmnopqrst"), 10, 9, SV("12345678901234567890"), 0, 20, S("abcdefghij12345678901234567890t"));
    test(S("abcdefghijklmnopqrst"), 10, 9, SV("12345678901234567890"), 0, 21, S("abcdefghij12345678901234567890t"));
    test(S("abcdefghijklmnopqrst"), 10, 9, SV("12345678901234567890"), 1, 0, S("abcdefghijt"));
    test(S("abcdefghijklmnopqrst"), 10, 9, SV("12345678901234567890"), 1, 1, S("abcdefghij2t"));
    test(S("abcdefghijklmnopqrst"), 10, 9, SV("12345678901234567890"), 1, 9, S("abcdefghij234567890t"));
    test(S("abcdefghijklmnopqrst"), 10, 9, SV("12345678901234567890"), 1, 18, S("abcdefghij234567890123456789t"));
    test(S("abcdefghijklmnopqrst"), 10, 9, SV("12345678901234567890"), 1, 19, S("abcdefghij2345678901234567890t"));
    test(S("abcdefghijklmnopqrst"), 10, 9, SV("12345678901234567890"), 1, 20, S("abcdefghij2345678901234567890t"));
    test(S("abcdefghijklmnopqrst"), 10, 9, SV("12345678901234567890"), 10, 0, S("abcdefghijt"));
    test(S("abcdefghijklmnopqrst"), 10, 9, SV("12345678901234567890"), 10, 1, S("abcdefghij1t"));
    test(S("abcdefghijklmnopqrst"), 10, 9, SV("12345678901234567890"), 10, 5, S("abcdefghij12345t"));
    test(S("abcdefghijklmnopqrst"), 10, 9, SV("12345678901234567890"), 10, 9, S("abcdefghij123456789t"));
    test(S("abcdefghijklmnopqrst"), 10, 9, SV("12345678901234567890"), 10, 10, S("abcdefghij1234567890t"));
    test(S("abcdefghijklmnopqrst"), 10, 9, SV("12345678901234567890"), 10, 11, S("abcdefghij1234567890t"));
    test(S("abcdefghijklmnopqrst"), 10, 9, SV("12345678901234567890"), 19, 0, S("abcdefghijt"));
    test(S("abcdefghijklmnopqrst"), 10, 9, SV("12345678901234567890"), 19, 1, S("abcdefghij0t"));
    test(S("abcdefghijklmnopqrst"), 10, 9, SV("12345678901234567890"), 19, 2, S("abcdefghij0t"));
    test(S("abcdefghijklmnopqrst"), 10, 9, SV("12345678901234567890"), 20, 0, S("abcdefghijt"));
    test(S("abcdefghijklmnopqrst"), 10, 9, SV("12345678901234567890"), 20, 1, S("abcdefghijt"));
    test(S("abcdefghijklmnopqrst"), 10, 9, SV("12345678901234567890"), 21, 0, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 10, 10, SV(""), 0, 0, S("abcdefghij"));
    test(S("abcdefghijklmnopqrst"), 10, 10, SV(""), 0, 1, S("abcdefghij"));
    test(S("abcdefghijklmnopqrst"), 10, 10, SV(""), 1, 0, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 10, 10, SV("12345"), 0, 0, S("abcdefghij"));
    test(S("abcdefghijklmnopqrst"), 10, 10, SV("12345"), 0, 1, S("abcdefghij1"));
    test(S("abcdefghijklmnopqrst"), 10, 10, SV("12345"), 0, 2, S("abcdefghij12"));
    test(S("abcdefghijklmnopqrst"), 10, 10, SV("12345"), 0, 4, S("abcdefghij1234"));
    test(S("abcdefghijklmnopqrst"), 10, 10, SV("12345"), 0, 5, S("abcdefghij12345"));
    test(S("abcdefghijklmnopqrst"), 10, 10, SV("12345"), 0, 6, S("abcdefghij12345"));
    test(S("abcdefghijklmnopqrst"), 10, 10, SV("12345"), 1, 0, S("abcdefghij"));
    test(S("abcdefghijklmnopqrst"), 10, 10, SV("12345"), 1, 1, S("abcdefghij2"));
    test(S("abcdefghijklmnopqrst"), 10, 10, SV("12345"), 1, 2, S("abcdefghij23"));
    test(S("abcdefghijklmnopqrst"), 10, 10, SV("12345"), 1, 3, S("abcdefghij234"));
    test(S("abcdefghijklmnopqrst"), 10, 10, SV("12345"), 1, 4, S("abcdefghij2345"));
    test(S("abcdefghijklmnopqrst"), 10, 10, SV("12345"), 1, 5, S("abcdefghij2345"));
    test(S("abcdefghijklmnopqrst"), 10, 10, SV("12345"), 2, 0, S("abcdefghij"));

    return true;
}

template <class S, class SV>
TEST_CONSTEXPR_CXX20 bool test49()
{
    test(S("abcdefghijklmnopqrst"), 10, 10, SV("12345"), 2, 1, S("abcdefghij3"));
    test(S("abcdefghijklmnopqrst"), 10, 10, SV("12345"), 2, 2, S("abcdefghij34"));
    test(S("abcdefghijklmnopqrst"), 10, 10, SV("12345"), 2, 3, S("abcdefghij345"));
    test(S("abcdefghijklmnopqrst"), 10, 10, SV("12345"), 2, 4, S("abcdefghij345"));
    test(S("abcdefghijklmnopqrst"), 10, 10, SV("12345"), 4, 0, S("abcdefghij"));
    test(S("abcdefghijklmnopqrst"), 10, 10, SV("12345"), 4, 1, S("abcdefghij5"));
    test(S("abcdefghijklmnopqrst"), 10, 10, SV("12345"), 4, 2, S("abcdefghij5"));
    test(S("abcdefghijklmnopqrst"), 10, 10, SV("12345"), 5, 0, S("abcdefghij"));
    test(S("abcdefghijklmnopqrst"), 10, 10, SV("12345"), 5, 1, S("abcdefghij"));
    test(S("abcdefghijklmnopqrst"), 10, 10, SV("12345"), 6, 0, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 10, 10, SV("1234567890"), 0, 0, S("abcdefghij"));
    test(S("abcdefghijklmnopqrst"), 10, 10, SV("1234567890"), 0, 1, S("abcdefghij1"));
    test(S("abcdefghijklmnopqrst"), 10, 10, SV("1234567890"), 0, 5, S("abcdefghij12345"));
    test(S("abcdefghijklmnopqrst"), 10, 10, SV("1234567890"), 0, 9, S("abcdefghij123456789"));
    test(S("abcdefghijklmnopqrst"), 10, 10, SV("1234567890"), 0, 10, S("abcdefghij1234567890"));
    test(S("abcdefghijklmnopqrst"), 10, 10, SV("1234567890"), 0, 11, S("abcdefghij1234567890"));
    test(S("abcdefghijklmnopqrst"), 10, 10, SV("1234567890"), 1, 0, S("abcdefghij"));
    test(S("abcdefghijklmnopqrst"), 10, 10, SV("1234567890"), 1, 1, S("abcdefghij2"));
    test(S("abcdefghijklmnopqrst"), 10, 10, SV("1234567890"), 1, 4, S("abcdefghij2345"));
    test(S("abcdefghijklmnopqrst"), 10, 10, SV("1234567890"), 1, 8, S("abcdefghij23456789"));
    test(S("abcdefghijklmnopqrst"), 10, 10, SV("1234567890"), 1, 9, S("abcdefghij234567890"));
    test(S("abcdefghijklmnopqrst"), 10, 10, SV("1234567890"), 1, 10, S("abcdefghij234567890"));
    test(S("abcdefghijklmnopqrst"), 10, 10, SV("1234567890"), 5, 0, S("abcdefghij"));
    test(S("abcdefghijklmnopqrst"), 10, 10, SV("1234567890"), 5, 1, S("abcdefghij6"));
    test(S("abcdefghijklmnopqrst"), 10, 10, SV("1234567890"), 5, 2, S("abcdefghij67"));
    test(S("abcdefghijklmnopqrst"), 10, 10, SV("1234567890"), 5, 4, S("abcdefghij6789"));
    test(S("abcdefghijklmnopqrst"), 10, 10, SV("1234567890"), 5, 5, S("abcdefghij67890"));
    test(S("abcdefghijklmnopqrst"), 10, 10, SV("1234567890"), 5, 6, S("abcdefghij67890"));
    test(S("abcdefghijklmnopqrst"), 10, 10, SV("1234567890"), 9, 0, S("abcdefghij"));
    test(S("abcdefghijklmnopqrst"), 10, 10, SV("1234567890"), 9, 1, S("abcdefghij0"));
    test(S("abcdefghijklmnopqrst"), 10, 10, SV("1234567890"), 9, 2, S("abcdefghij0"));
    test(S("abcdefghijklmnopqrst"), 10, 10, SV("1234567890"), 10, 0, S("abcdefghij"));
    test(S("abcdefghijklmnopqrst"), 10, 10, SV("1234567890"), 10, 1, S("abcdefghij"));
    test(S("abcdefghijklmnopqrst"), 10, 10, SV("1234567890"), 11, 0, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 10, 10, SV("12345678901234567890"), 0, 0, S("abcdefghij"));
    test(S("abcdefghijklmnopqrst"), 10, 10, SV("12345678901234567890"), 0, 1, S("abcdefghij1"));
    test(S("abcdefghijklmnopqrst"), 10, 10, SV("12345678901234567890"), 0, 10, S("abcdefghij1234567890"));
    test(S("abcdefghijklmnopqrst"), 10, 10, SV("12345678901234567890"), 0, 19, S("abcdefghij1234567890123456789"));
    test(S("abcdefghijklmnopqrst"), 10, 10, SV("12345678901234567890"), 0, 20, S("abcdefghij12345678901234567890"));
    test(S("abcdefghijklmnopqrst"), 10, 10, SV("12345678901234567890"), 0, 21, S("abcdefghij12345678901234567890"));
    test(S("abcdefghijklmnopqrst"), 10, 10, SV("12345678901234567890"), 1, 0, S("abcdefghij"));
    test(S("abcdefghijklmnopqrst"), 10, 10, SV("12345678901234567890"), 1, 1, S("abcdefghij2"));
    test(S("abcdefghijklmnopqrst"), 10, 10, SV("12345678901234567890"), 1, 9, S("abcdefghij234567890"));
    test(S("abcdefghijklmnopqrst"), 10, 10, SV("12345678901234567890"), 1, 18, S("abcdefghij234567890123456789"));
    test(S("abcdefghijklmnopqrst"), 10, 10, SV("12345678901234567890"), 1, 19, S("abcdefghij2345678901234567890"));
    test(S("abcdefghijklmnopqrst"), 10, 10, SV("12345678901234567890"), 1, 20, S("abcdefghij2345678901234567890"));
    test(S("abcdefghijklmnopqrst"), 10, 10, SV("12345678901234567890"), 10, 0, S("abcdefghij"));
    test(S("abcdefghijklmnopqrst"), 10, 10, SV("12345678901234567890"), 10, 1, S("abcdefghij1"));
    test(S("abcdefghijklmnopqrst"), 10, 10, SV("12345678901234567890"), 10, 5, S("abcdefghij12345"));
    test(S("abcdefghijklmnopqrst"), 10, 10, SV("12345678901234567890"), 10, 9, S("abcdefghij123456789"));
    test(S("abcdefghijklmnopqrst"), 10, 10, SV("12345678901234567890"), 10, 10, S("abcdefghij1234567890"));
    test(S("abcdefghijklmnopqrst"), 10, 10, SV("12345678901234567890"), 10, 11, S("abcdefghij1234567890"));
    test(S("abcdefghijklmnopqrst"), 10, 10, SV("12345678901234567890"), 19, 0, S("abcdefghij"));
    test(S("abcdefghijklmnopqrst"), 10, 10, SV("12345678901234567890"), 19, 1, S("abcdefghij0"));
    test(S("abcdefghijklmnopqrst"), 10, 10, SV("12345678901234567890"), 19, 2, S("abcdefghij0"));
    test(S("abcdefghijklmnopqrst"), 10, 10, SV("12345678901234567890"), 20, 0, S("abcdefghij"));
    test(S("abcdefghijklmnopqrst"), 10, 10, SV("12345678901234567890"), 20, 1, S("abcdefghij"));
    test(S("abcdefghijklmnopqrst"), 10, 10, SV("12345678901234567890"), 21, 0, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 10, 11, SV(""), 0, 0, S("abcdefghij"));
    test(S("abcdefghijklmnopqrst"), 10, 11, SV(""), 0, 1, S("abcdefghij"));
    test(S("abcdefghijklmnopqrst"), 10, 11, SV(""), 1, 0, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 10, 11, SV("12345"), 0, 0, S("abcdefghij"));
    test(S("abcdefghijklmnopqrst"), 10, 11, SV("12345"), 0, 1, S("abcdefghij1"));
    test(S("abcdefghijklmnopqrst"), 10, 11, SV("12345"), 0, 2, S("abcdefghij12"));
    test(S("abcdefghijklmnopqrst"), 10, 11, SV("12345"), 0, 4, S("abcdefghij1234"));
    test(S("abcdefghijklmnopqrst"), 10, 11, SV("12345"), 0, 5, S("abcdefghij12345"));
    test(S("abcdefghijklmnopqrst"), 10, 11, SV("12345"), 0, 6, S("abcdefghij12345"));
    test(S("abcdefghijklmnopqrst"), 10, 11, SV("12345"), 1, 0, S("abcdefghij"));
    test(S("abcdefghijklmnopqrst"), 10, 11, SV("12345"), 1, 1, S("abcdefghij2"));
    test(S("abcdefghijklmnopqrst"), 10, 11, SV("12345"), 1, 2, S("abcdefghij23"));
    test(S("abcdefghijklmnopqrst"), 10, 11, SV("12345"), 1, 3, S("abcdefghij234"));
    test(S("abcdefghijklmnopqrst"), 10, 11, SV("12345"), 1, 4, S("abcdefghij2345"));
    test(S("abcdefghijklmnopqrst"), 10, 11, SV("12345"), 1, 5, S("abcdefghij2345"));
    test(S("abcdefghijklmnopqrst"), 10, 11, SV("12345"), 2, 0, S("abcdefghij"));
    test(S("abcdefghijklmnopqrst"), 10, 11, SV("12345"), 2, 1, S("abcdefghij3"));
    test(S("abcdefghijklmnopqrst"), 10, 11, SV("12345"), 2, 2, S("abcdefghij34"));
    test(S("abcdefghijklmnopqrst"), 10, 11, SV("12345"), 2, 3, S("abcdefghij345"));
    test(S("abcdefghijklmnopqrst"), 10, 11, SV("12345"), 2, 4, S("abcdefghij345"));
    test(S("abcdefghijklmnopqrst"), 10, 11, SV("12345"), 4, 0, S("abcdefghij"));
    test(S("abcdefghijklmnopqrst"), 10, 11, SV("12345"), 4, 1, S("abcdefghij5"));
    test(S("abcdefghijklmnopqrst"), 10, 11, SV("12345"), 4, 2, S("abcdefghij5"));
    test(S("abcdefghijklmnopqrst"), 10, 11, SV("12345"), 5, 0, S("abcdefghij"));
    test(S("abcdefghijklmnopqrst"), 10, 11, SV("12345"), 5, 1, S("abcdefghij"));
    test(S("abcdefghijklmnopqrst"), 10, 11, SV("12345"), 6, 0, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 10, 11, SV("1234567890"), 0, 0, S("abcdefghij"));
    test(S("abcdefghijklmnopqrst"), 10, 11, SV("1234567890"), 0, 1, S("abcdefghij1"));
    test(S("abcdefghijklmnopqrst"), 10, 11, SV("1234567890"), 0, 5, S("abcdefghij12345"));
    test(S("abcdefghijklmnopqrst"), 10, 11, SV("1234567890"), 0, 9, S("abcdefghij123456789"));
    test(S("abcdefghijklmnopqrst"), 10, 11, SV("1234567890"), 0, 10, S("abcdefghij1234567890"));
    test(S("abcdefghijklmnopqrst"), 10, 11, SV("1234567890"), 0, 11, S("abcdefghij1234567890"));
    test(S("abcdefghijklmnopqrst"), 10, 11, SV("1234567890"), 1, 0, S("abcdefghij"));
    test(S("abcdefghijklmnopqrst"), 10, 11, SV("1234567890"), 1, 1, S("abcdefghij2"));
    test(S("abcdefghijklmnopqrst"), 10, 11, SV("1234567890"), 1, 4, S("abcdefghij2345"));
    test(S("abcdefghijklmnopqrst"), 10, 11, SV("1234567890"), 1, 8, S("abcdefghij23456789"));
    test(S("abcdefghijklmnopqrst"), 10, 11, SV("1234567890"), 1, 9, S("abcdefghij234567890"));
    test(S("abcdefghijklmnopqrst"), 10, 11, SV("1234567890"), 1, 10, S("abcdefghij234567890"));
    test(S("abcdefghijklmnopqrst"), 10, 11, SV("1234567890"), 5, 0, S("abcdefghij"));
    test(S("abcdefghijklmnopqrst"), 10, 11, SV("1234567890"), 5, 1, S("abcdefghij6"));
    test(S("abcdefghijklmnopqrst"), 10, 11, SV("1234567890"), 5, 2, S("abcdefghij67"));
    test(S("abcdefghijklmnopqrst"), 10, 11, SV("1234567890"), 5, 4, S("abcdefghij6789"));

    return true;
}

template <class S, class SV>
TEST_CONSTEXPR_CXX20 bool test50()
{
    test(S("abcdefghijklmnopqrst"), 10, 11, SV("1234567890"), 5, 5, S("abcdefghij67890"));
    test(S("abcdefghijklmnopqrst"), 10, 11, SV("1234567890"), 5, 6, S("abcdefghij67890"));
    test(S("abcdefghijklmnopqrst"), 10, 11, SV("1234567890"), 9, 0, S("abcdefghij"));
    test(S("abcdefghijklmnopqrst"), 10, 11, SV("1234567890"), 9, 1, S("abcdefghij0"));
    test(S("abcdefghijklmnopqrst"), 10, 11, SV("1234567890"), 9, 2, S("abcdefghij0"));
    test(S("abcdefghijklmnopqrst"), 10, 11, SV("1234567890"), 10, 0, S("abcdefghij"));
    test(S("abcdefghijklmnopqrst"), 10, 11, SV("1234567890"), 10, 1, S("abcdefghij"));
    test(S("abcdefghijklmnopqrst"), 10, 11, SV("1234567890"), 11, 0, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 10, 11, SV("12345678901234567890"), 0, 0, S("abcdefghij"));
    test(S("abcdefghijklmnopqrst"), 10, 11, SV("12345678901234567890"), 0, 1, S("abcdefghij1"));
    test(S("abcdefghijklmnopqrst"), 10, 11, SV("12345678901234567890"), 0, 10, S("abcdefghij1234567890"));
    test(S("abcdefghijklmnopqrst"), 10, 11, SV("12345678901234567890"), 0, 19, S("abcdefghij1234567890123456789"));
    test(S("abcdefghijklmnopqrst"), 10, 11, SV("12345678901234567890"), 0, 20, S("abcdefghij12345678901234567890"));
    test(S("abcdefghijklmnopqrst"), 10, 11, SV("12345678901234567890"), 0, 21, S("abcdefghij12345678901234567890"));
    test(S("abcdefghijklmnopqrst"), 10, 11, SV("12345678901234567890"), 1, 0, S("abcdefghij"));
    test(S("abcdefghijklmnopqrst"), 10, 11, SV("12345678901234567890"), 1, 1, S("abcdefghij2"));
    test(S("abcdefghijklmnopqrst"), 10, 11, SV("12345678901234567890"), 1, 9, S("abcdefghij234567890"));
    test(S("abcdefghijklmnopqrst"), 10, 11, SV("12345678901234567890"), 1, 18, S("abcdefghij234567890123456789"));
    test(S("abcdefghijklmnopqrst"), 10, 11, SV("12345678901234567890"), 1, 19, S("abcdefghij2345678901234567890"));
    test(S("abcdefghijklmnopqrst"), 10, 11, SV("12345678901234567890"), 1, 20, S("abcdefghij2345678901234567890"));
    test(S("abcdefghijklmnopqrst"), 10, 11, SV("12345678901234567890"), 10, 0, S("abcdefghij"));
    test(S("abcdefghijklmnopqrst"), 10, 11, SV("12345678901234567890"), 10, 1, S("abcdefghij1"));
    test(S("abcdefghijklmnopqrst"), 10, 11, SV("12345678901234567890"), 10, 5, S("abcdefghij12345"));
    test(S("abcdefghijklmnopqrst"), 10, 11, SV("12345678901234567890"), 10, 9, S("abcdefghij123456789"));
    test(S("abcdefghijklmnopqrst"), 10, 11, SV("12345678901234567890"), 10, 10, S("abcdefghij1234567890"));
    test(S("abcdefghijklmnopqrst"), 10, 11, SV("12345678901234567890"), 10, 11, S("abcdefghij1234567890"));
    test(S("abcdefghijklmnopqrst"), 10, 11, SV("12345678901234567890"), 19, 0, S("abcdefghij"));
    test(S("abcdefghijklmnopqrst"), 10, 11, SV("12345678901234567890"), 19, 1, S("abcdefghij0"));
    test(S("abcdefghijklmnopqrst"), 10, 11, SV("12345678901234567890"), 19, 2, S("abcdefghij0"));
    test(S("abcdefghijklmnopqrst"), 10, 11, SV("12345678901234567890"), 20, 0, S("abcdefghij"));
    test(S("abcdefghijklmnopqrst"), 10, 11, SV("12345678901234567890"), 20, 1, S("abcdefghij"));
    test(S("abcdefghijklmnopqrst"), 10, 11, SV("12345678901234567890"), 21, 0, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 19, 0, SV(""), 0, 0, S("abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 19, 0, SV(""), 0, 1, S("abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 19, 0, SV(""), 1, 0, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 19, 0, SV("12345"), 0, 0, S("abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 19, 0, SV("12345"), 0, 1, S("abcdefghijklmnopqrs1t"));
    test(S("abcdefghijklmnopqrst"), 19, 0, SV("12345"), 0, 2, S("abcdefghijklmnopqrs12t"));
    test(S("abcdefghijklmnopqrst"), 19, 0, SV("12345"), 0, 4, S("abcdefghijklmnopqrs1234t"));
    test(S("abcdefghijklmnopqrst"), 19, 0, SV("12345"), 0, 5, S("abcdefghijklmnopqrs12345t"));
    test(S("abcdefghijklmnopqrst"), 19, 0, SV("12345"), 0, 6, S("abcdefghijklmnopqrs12345t"));
    test(S("abcdefghijklmnopqrst"), 19, 0, SV("12345"), 1, 0, S("abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 19, 0, SV("12345"), 1, 1, S("abcdefghijklmnopqrs2t"));
    test(S("abcdefghijklmnopqrst"), 19, 0, SV("12345"), 1, 2, S("abcdefghijklmnopqrs23t"));
    test(S("abcdefghijklmnopqrst"), 19, 0, SV("12345"), 1, 3, S("abcdefghijklmnopqrs234t"));
    test(S("abcdefghijklmnopqrst"), 19, 0, SV("12345"), 1, 4, S("abcdefghijklmnopqrs2345t"));
    test(S("abcdefghijklmnopqrst"), 19, 0, SV("12345"), 1, 5, S("abcdefghijklmnopqrs2345t"));
    test(S("abcdefghijklmnopqrst"), 19, 0, SV("12345"), 2, 0, S("abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 19, 0, SV("12345"), 2, 1, S("abcdefghijklmnopqrs3t"));
    test(S("abcdefghijklmnopqrst"), 19, 0, SV("12345"), 2, 2, S("abcdefghijklmnopqrs34t"));
    test(S("abcdefghijklmnopqrst"), 19, 0, SV("12345"), 2, 3, S("abcdefghijklmnopqrs345t"));
    test(S("abcdefghijklmnopqrst"), 19, 0, SV("12345"), 2, 4, S("abcdefghijklmnopqrs345t"));
    test(S("abcdefghijklmnopqrst"), 19, 0, SV("12345"), 4, 0, S("abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 19, 0, SV("12345"), 4, 1, S("abcdefghijklmnopqrs5t"));
    test(S("abcdefghijklmnopqrst"), 19, 0, SV("12345"), 4, 2, S("abcdefghijklmnopqrs5t"));
    test(S("abcdefghijklmnopqrst"), 19, 0, SV("12345"), 5, 0, S("abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 19, 0, SV("12345"), 5, 1, S("abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 19, 0, SV("12345"), 6, 0, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 19, 0, SV("1234567890"), 0, 0, S("abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 19, 0, SV("1234567890"), 0, 1, S("abcdefghijklmnopqrs1t"));
    test(S("abcdefghijklmnopqrst"), 19, 0, SV("1234567890"), 0, 5, S("abcdefghijklmnopqrs12345t"));
    test(S("abcdefghijklmnopqrst"), 19, 0, SV("1234567890"), 0, 9, S("abcdefghijklmnopqrs123456789t"));
    test(S("abcdefghijklmnopqrst"), 19, 0, SV("1234567890"), 0, 10, S("abcdefghijklmnopqrs1234567890t"));
    test(S("abcdefghijklmnopqrst"), 19, 0, SV("1234567890"), 0, 11, S("abcdefghijklmnopqrs1234567890t"));
    test(S("abcdefghijklmnopqrst"), 19, 0, SV("1234567890"), 1, 0, S("abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 19, 0, SV("1234567890"), 1, 1, S("abcdefghijklmnopqrs2t"));
    test(S("abcdefghijklmnopqrst"), 19, 0, SV("1234567890"), 1, 4, S("abcdefghijklmnopqrs2345t"));
    test(S("abcdefghijklmnopqrst"), 19, 0, SV("1234567890"), 1, 8, S("abcdefghijklmnopqrs23456789t"));
    test(S("abcdefghijklmnopqrst"), 19, 0, SV("1234567890"), 1, 9, S("abcdefghijklmnopqrs234567890t"));
    test(S("abcdefghijklmnopqrst"), 19, 0, SV("1234567890"), 1, 10, S("abcdefghijklmnopqrs234567890t"));
    test(S("abcdefghijklmnopqrst"), 19, 0, SV("1234567890"), 5, 0, S("abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 19, 0, SV("1234567890"), 5, 1, S("abcdefghijklmnopqrs6t"));
    test(S("abcdefghijklmnopqrst"), 19, 0, SV("1234567890"), 5, 2, S("abcdefghijklmnopqrs67t"));
    test(S("abcdefghijklmnopqrst"), 19, 0, SV("1234567890"), 5, 4, S("abcdefghijklmnopqrs6789t"));
    test(S("abcdefghijklmnopqrst"), 19, 0, SV("1234567890"), 5, 5, S("abcdefghijklmnopqrs67890t"));
    test(S("abcdefghijklmnopqrst"), 19, 0, SV("1234567890"), 5, 6, S("abcdefghijklmnopqrs67890t"));
    test(S("abcdefghijklmnopqrst"), 19, 0, SV("1234567890"), 9, 0, S("abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 19, 0, SV("1234567890"), 9, 1, S("abcdefghijklmnopqrs0t"));
    test(S("abcdefghijklmnopqrst"), 19, 0, SV("1234567890"), 9, 2, S("abcdefghijklmnopqrs0t"));
    test(S("abcdefghijklmnopqrst"), 19, 0, SV("1234567890"), 10, 0, S("abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 19, 0, SV("1234567890"), 10, 1, S("abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 19, 0, SV("1234567890"), 11, 0, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 19, 0, SV("12345678901234567890"), 0, 0, S("abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 19, 0, SV("12345678901234567890"), 0, 1, S("abcdefghijklmnopqrs1t"));
    test(S("abcdefghijklmnopqrst"), 19, 0, SV("12345678901234567890"), 0, 10, S("abcdefghijklmnopqrs1234567890t"));
    test(S("abcdefghijklmnopqrst"), 19, 0, SV("12345678901234567890"), 0, 19, S("abcdefghijklmnopqrs1234567890123456789t"));
    test(S("abcdefghijklmnopqrst"), 19, 0, SV("12345678901234567890"), 0, 20, S("abcdefghijklmnopqrs12345678901234567890t"));
    test(S("abcdefghijklmnopqrst"), 19, 0, SV("12345678901234567890"), 0, 21, S("abcdefghijklmnopqrs12345678901234567890t"));
    test(S("abcdefghijklmnopqrst"), 19, 0, SV("12345678901234567890"), 1, 0, S("abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 19, 0, SV("12345678901234567890"), 1, 1, S("abcdefghijklmnopqrs2t"));
    test(S("abcdefghijklmnopqrst"), 19, 0, SV("12345678901234567890"), 1, 9, S("abcdefghijklmnopqrs234567890t"));
    test(S("abcdefghijklmnopqrst"), 19, 0, SV("12345678901234567890"), 1, 18, S("abcdefghijklmnopqrs234567890123456789t"));
    test(S("abcdefghijklmnopqrst"), 19, 0, SV("12345678901234567890"), 1, 19, S("abcdefghijklmnopqrs2345678901234567890t"));
    test(S("abcdefghijklmnopqrst"), 19, 0, SV("12345678901234567890"), 1, 20, S("abcdefghijklmnopqrs2345678901234567890t"));
    test(S("abcdefghijklmnopqrst"), 19, 0, SV("12345678901234567890"), 10, 0, S("abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 19, 0, SV("12345678901234567890"), 10, 1, S("abcdefghijklmnopqrs1t"));
    test(S("abcdefghijklmnopqrst"), 19, 0, SV("12345678901234567890"), 10, 5, S("abcdefghijklmnopqrs12345t"));
    test(S("abcdefghijklmnopqrst"), 19, 0, SV("12345678901234567890"), 10, 9, S("abcdefghijklmnopqrs123456789t"));
    test(S("abcdefghijklmnopqrst"), 19, 0, SV("12345678901234567890"), 10, 10, S("abcdefghijklmnopqrs1234567890t"));
    test(S("abcdefghijklmnopqrst"), 19, 0, SV("12345678901234567890"), 10, 11, S("abcdefghijklmnopqrs1234567890t"));

    return true;
}

template <class S, class SV>
TEST_CONSTEXPR_CXX20 bool test51()
{
    test(S("abcdefghijklmnopqrst"), 19, 0, SV("12345678901234567890"), 19, 0, S("abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 19, 0, SV("12345678901234567890"), 19, 1, S("abcdefghijklmnopqrs0t"));
    test(S("abcdefghijklmnopqrst"), 19, 0, SV("12345678901234567890"), 19, 2, S("abcdefghijklmnopqrs0t"));
    test(S("abcdefghijklmnopqrst"), 19, 0, SV("12345678901234567890"), 20, 0, S("abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 19, 0, SV("12345678901234567890"), 20, 1, S("abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 19, 0, SV("12345678901234567890"), 21, 0, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 19, 1, SV(""), 0, 0, S("abcdefghijklmnopqrs"));
    test(S("abcdefghijklmnopqrst"), 19, 1, SV(""), 0, 1, S("abcdefghijklmnopqrs"));
    test(S("abcdefghijklmnopqrst"), 19, 1, SV(""), 1, 0, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 19, 1, SV("12345"), 0, 0, S("abcdefghijklmnopqrs"));
    test(S("abcdefghijklmnopqrst"), 19, 1, SV("12345"), 0, 1, S("abcdefghijklmnopqrs1"));
    test(S("abcdefghijklmnopqrst"), 19, 1, SV("12345"), 0, 2, S("abcdefghijklmnopqrs12"));
    test(S("abcdefghijklmnopqrst"), 19, 1, SV("12345"), 0, 4, S("abcdefghijklmnopqrs1234"));
    test(S("abcdefghijklmnopqrst"), 19, 1, SV("12345"), 0, 5, S("abcdefghijklmnopqrs12345"));
    test(S("abcdefghijklmnopqrst"), 19, 1, SV("12345"), 0, 6, S("abcdefghijklmnopqrs12345"));
    test(S("abcdefghijklmnopqrst"), 19, 1, SV("12345"), 1, 0, S("abcdefghijklmnopqrs"));
    test(S("abcdefghijklmnopqrst"), 19, 1, SV("12345"), 1, 1, S("abcdefghijklmnopqrs2"));
    test(S("abcdefghijklmnopqrst"), 19, 1, SV("12345"), 1, 2, S("abcdefghijklmnopqrs23"));
    test(S("abcdefghijklmnopqrst"), 19, 1, SV("12345"), 1, 3, S("abcdefghijklmnopqrs234"));
    test(S("abcdefghijklmnopqrst"), 19, 1, SV("12345"), 1, 4, S("abcdefghijklmnopqrs2345"));
    test(S("abcdefghijklmnopqrst"), 19, 1, SV("12345"), 1, 5, S("abcdefghijklmnopqrs2345"));
    test(S("abcdefghijklmnopqrst"), 19, 1, SV("12345"), 2, 0, S("abcdefghijklmnopqrs"));
    test(S("abcdefghijklmnopqrst"), 19, 1, SV("12345"), 2, 1, S("abcdefghijklmnopqrs3"));
    test(S("abcdefghijklmnopqrst"), 19, 1, SV("12345"), 2, 2, S("abcdefghijklmnopqrs34"));
    test(S("abcdefghijklmnopqrst"), 19, 1, SV("12345"), 2, 3, S("abcdefghijklmnopqrs345"));
    test(S("abcdefghijklmnopqrst"), 19, 1, SV("12345"), 2, 4, S("abcdefghijklmnopqrs345"));
    test(S("abcdefghijklmnopqrst"), 19, 1, SV("12345"), 4, 0, S("abcdefghijklmnopqrs"));
    test(S("abcdefghijklmnopqrst"), 19, 1, SV("12345"), 4, 1, S("abcdefghijklmnopqrs5"));
    test(S("abcdefghijklmnopqrst"), 19, 1, SV("12345"), 4, 2, S("abcdefghijklmnopqrs5"));
    test(S("abcdefghijklmnopqrst"), 19, 1, SV("12345"), 5, 0, S("abcdefghijklmnopqrs"));
    test(S("abcdefghijklmnopqrst"), 19, 1, SV("12345"), 5, 1, S("abcdefghijklmnopqrs"));
    test(S("abcdefghijklmnopqrst"), 19, 1, SV("12345"), 6, 0, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 19, 1, SV("1234567890"), 0, 0, S("abcdefghijklmnopqrs"));
    test(S("abcdefghijklmnopqrst"), 19, 1, SV("1234567890"), 0, 1, S("abcdefghijklmnopqrs1"));
    test(S("abcdefghijklmnopqrst"), 19, 1, SV("1234567890"), 0, 5, S("abcdefghijklmnopqrs12345"));
    test(S("abcdefghijklmnopqrst"), 19, 1, SV("1234567890"), 0, 9, S("abcdefghijklmnopqrs123456789"));
    test(S("abcdefghijklmnopqrst"), 19, 1, SV("1234567890"), 0, 10, S("abcdefghijklmnopqrs1234567890"));
    test(S("abcdefghijklmnopqrst"), 19, 1, SV("1234567890"), 0, 11, S("abcdefghijklmnopqrs1234567890"));
    test(S("abcdefghijklmnopqrst"), 19, 1, SV("1234567890"), 1, 0, S("abcdefghijklmnopqrs"));
    test(S("abcdefghijklmnopqrst"), 19, 1, SV("1234567890"), 1, 1, S("abcdefghijklmnopqrs2"));
    test(S("abcdefghijklmnopqrst"), 19, 1, SV("1234567890"), 1, 4, S("abcdefghijklmnopqrs2345"));
    test(S("abcdefghijklmnopqrst"), 19, 1, SV("1234567890"), 1, 8, S("abcdefghijklmnopqrs23456789"));
    test(S("abcdefghijklmnopqrst"), 19, 1, SV("1234567890"), 1, 9, S("abcdefghijklmnopqrs234567890"));
    test(S("abcdefghijklmnopqrst"), 19, 1, SV("1234567890"), 1, 10, S("abcdefghijklmnopqrs234567890"));
    test(S("abcdefghijklmnopqrst"), 19, 1, SV("1234567890"), 5, 0, S("abcdefghijklmnopqrs"));
    test(S("abcdefghijklmnopqrst"), 19, 1, SV("1234567890"), 5, 1, S("abcdefghijklmnopqrs6"));
    test(S("abcdefghijklmnopqrst"), 19, 1, SV("1234567890"), 5, 2, S("abcdefghijklmnopqrs67"));
    test(S("abcdefghijklmnopqrst"), 19, 1, SV("1234567890"), 5, 4, S("abcdefghijklmnopqrs6789"));
    test(S("abcdefghijklmnopqrst"), 19, 1, SV("1234567890"), 5, 5, S("abcdefghijklmnopqrs67890"));
    test(S("abcdefghijklmnopqrst"), 19, 1, SV("1234567890"), 5, 6, S("abcdefghijklmnopqrs67890"));
    test(S("abcdefghijklmnopqrst"), 19, 1, SV("1234567890"), 9, 0, S("abcdefghijklmnopqrs"));
    test(S("abcdefghijklmnopqrst"), 19, 1, SV("1234567890"), 9, 1, S("abcdefghijklmnopqrs0"));
    test(S("abcdefghijklmnopqrst"), 19, 1, SV("1234567890"), 9, 2, S("abcdefghijklmnopqrs0"));
    test(S("abcdefghijklmnopqrst"), 19, 1, SV("1234567890"), 10, 0, S("abcdefghijklmnopqrs"));
    test(S("abcdefghijklmnopqrst"), 19, 1, SV("1234567890"), 10, 1, S("abcdefghijklmnopqrs"));
    test(S("abcdefghijklmnopqrst"), 19, 1, SV("1234567890"), 11, 0, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 19, 1, SV("12345678901234567890"), 0, 0, S("abcdefghijklmnopqrs"));
    test(S("abcdefghijklmnopqrst"), 19, 1, SV("12345678901234567890"), 0, 1, S("abcdefghijklmnopqrs1"));
    test(S("abcdefghijklmnopqrst"), 19, 1, SV("12345678901234567890"), 0, 10, S("abcdefghijklmnopqrs1234567890"));
    test(S("abcdefghijklmnopqrst"), 19, 1, SV("12345678901234567890"), 0, 19, S("abcdefghijklmnopqrs1234567890123456789"));
    test(S("abcdefghijklmnopqrst"), 19, 1, SV("12345678901234567890"), 0, 20, S("abcdefghijklmnopqrs12345678901234567890"));
    test(S("abcdefghijklmnopqrst"), 19, 1, SV("12345678901234567890"), 0, 21, S("abcdefghijklmnopqrs12345678901234567890"));
    test(S("abcdefghijklmnopqrst"), 19, 1, SV("12345678901234567890"), 1, 0, S("abcdefghijklmnopqrs"));
    test(S("abcdefghijklmnopqrst"), 19, 1, SV("12345678901234567890"), 1, 1, S("abcdefghijklmnopqrs2"));
    test(S("abcdefghijklmnopqrst"), 19, 1, SV("12345678901234567890"), 1, 9, S("abcdefghijklmnopqrs234567890"));
    test(S("abcdefghijklmnopqrst"), 19, 1, SV("12345678901234567890"), 1, 18, S("abcdefghijklmnopqrs234567890123456789"));
    test(S("abcdefghijklmnopqrst"), 19, 1, SV("12345678901234567890"), 1, 19, S("abcdefghijklmnopqrs2345678901234567890"));
    test(S("abcdefghijklmnopqrst"), 19, 1, SV("12345678901234567890"), 1, 20, S("abcdefghijklmnopqrs2345678901234567890"));
    test(S("abcdefghijklmnopqrst"), 19, 1, SV("12345678901234567890"), 10, 0, S("abcdefghijklmnopqrs"));
    test(S("abcdefghijklmnopqrst"), 19, 1, SV("12345678901234567890"), 10, 1, S("abcdefghijklmnopqrs1"));
    test(S("abcdefghijklmnopqrst"), 19, 1, SV("12345678901234567890"), 10, 5, S("abcdefghijklmnopqrs12345"));
    test(S("abcdefghijklmnopqrst"), 19, 1, SV("12345678901234567890"), 10, 9, S("abcdefghijklmnopqrs123456789"));
    test(S("abcdefghijklmnopqrst"), 19, 1, SV("12345678901234567890"), 10, 10, S("abcdefghijklmnopqrs1234567890"));
    test(S("abcdefghijklmnopqrst"), 19, 1, SV("12345678901234567890"), 10, 11, S("abcdefghijklmnopqrs1234567890"));
    test(S("abcdefghijklmnopqrst"), 19, 1, SV("12345678901234567890"), 19, 0, S("abcdefghijklmnopqrs"));
    test(S("abcdefghijklmnopqrst"), 19, 1, SV("12345678901234567890"), 19, 1, S("abcdefghijklmnopqrs0"));
    test(S("abcdefghijklmnopqrst"), 19, 1, SV("12345678901234567890"), 19, 2, S("abcdefghijklmnopqrs0"));
    test(S("abcdefghijklmnopqrst"), 19, 1, SV("12345678901234567890"), 20, 0, S("abcdefghijklmnopqrs"));
    test(S("abcdefghijklmnopqrst"), 19, 1, SV("12345678901234567890"), 20, 1, S("abcdefghijklmnopqrs"));
    test(S("abcdefghijklmnopqrst"), 19, 1, SV("12345678901234567890"), 21, 0, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 19, 2, SV(""), 0, 0, S("abcdefghijklmnopqrs"));
    test(S("abcdefghijklmnopqrst"), 19, 2, SV(""), 0, 1, S("abcdefghijklmnopqrs"));
    test(S("abcdefghijklmnopqrst"), 19, 2, SV(""), 1, 0, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 19, 2, SV("12345"), 0, 0, S("abcdefghijklmnopqrs"));
    test(S("abcdefghijklmnopqrst"), 19, 2, SV("12345"), 0, 1, S("abcdefghijklmnopqrs1"));
    test(S("abcdefghijklmnopqrst"), 19, 2, SV("12345"), 0, 2, S("abcdefghijklmnopqrs12"));
    test(S("abcdefghijklmnopqrst"), 19, 2, SV("12345"), 0, 4, S("abcdefghijklmnopqrs1234"));
    test(S("abcdefghijklmnopqrst"), 19, 2, SV("12345"), 0, 5, S("abcdefghijklmnopqrs12345"));
    test(S("abcdefghijklmnopqrst"), 19, 2, SV("12345"), 0, 6, S("abcdefghijklmnopqrs12345"));
    test(S("abcdefghijklmnopqrst"), 19, 2, SV("12345"), 1, 0, S("abcdefghijklmnopqrs"));
    test(S("abcdefghijklmnopqrst"), 19, 2, SV("12345"), 1, 1, S("abcdefghijklmnopqrs2"));
    test(S("abcdefghijklmnopqrst"), 19, 2, SV("12345"), 1, 2, S("abcdefghijklmnopqrs23"));
    test(S("abcdefghijklmnopqrst"), 19, 2, SV("12345"), 1, 3, S("abcdefghijklmnopqrs234"));
    test(S("abcdefghijklmnopqrst"), 19, 2, SV("12345"), 1, 4, S("abcdefghijklmnopqrs2345"));
    test(S("abcdefghijklmnopqrst"), 19, 2, SV("12345"), 1, 5, S("abcdefghijklmnopqrs2345"));
    test(S("abcdefghijklmnopqrst"), 19, 2, SV("12345"), 2, 0, S("abcdefghijklmnopqrs"));
    test(S("abcdefghijklmnopqrst"), 19, 2, SV("12345"), 2, 1, S("abcdefghijklmnopqrs3"));
    test(S("abcdefghijklmnopqrst"), 19, 2, SV("12345"), 2, 2, S("abcdefghijklmnopqrs34"));
    test(S("abcdefghijklmnopqrst"), 19, 2, SV("12345"), 2, 3, S("abcdefghijklmnopqrs345"));
    test(S("abcdefghijklmnopqrst"), 19, 2, SV("12345"), 2, 4, S("abcdefghijklmnopqrs345"));

    return true;
}

template <class S, class SV>
TEST_CONSTEXPR_CXX20 bool test52()
{
    test(S("abcdefghijklmnopqrst"), 19, 2, SV("12345"), 4, 0, S("abcdefghijklmnopqrs"));
    test(S("abcdefghijklmnopqrst"), 19, 2, SV("12345"), 4, 1, S("abcdefghijklmnopqrs5"));
    test(S("abcdefghijklmnopqrst"), 19, 2, SV("12345"), 4, 2, S("abcdefghijklmnopqrs5"));
    test(S("abcdefghijklmnopqrst"), 19, 2, SV("12345"), 5, 0, S("abcdefghijklmnopqrs"));
    test(S("abcdefghijklmnopqrst"), 19, 2, SV("12345"), 5, 1, S("abcdefghijklmnopqrs"));
    test(S("abcdefghijklmnopqrst"), 19, 2, SV("12345"), 6, 0, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 19, 2, SV("1234567890"), 0, 0, S("abcdefghijklmnopqrs"));
    test(S("abcdefghijklmnopqrst"), 19, 2, SV("1234567890"), 0, 1, S("abcdefghijklmnopqrs1"));
    test(S("abcdefghijklmnopqrst"), 19, 2, SV("1234567890"), 0, 5, S("abcdefghijklmnopqrs12345"));
    test(S("abcdefghijklmnopqrst"), 19, 2, SV("1234567890"), 0, 9, S("abcdefghijklmnopqrs123456789"));
    test(S("abcdefghijklmnopqrst"), 19, 2, SV("1234567890"), 0, 10, S("abcdefghijklmnopqrs1234567890"));
    test(S("abcdefghijklmnopqrst"), 19, 2, SV("1234567890"), 0, 11, S("abcdefghijklmnopqrs1234567890"));
    test(S("abcdefghijklmnopqrst"), 19, 2, SV("1234567890"), 1, 0, S("abcdefghijklmnopqrs"));
    test(S("abcdefghijklmnopqrst"), 19, 2, SV("1234567890"), 1, 1, S("abcdefghijklmnopqrs2"));
    test(S("abcdefghijklmnopqrst"), 19, 2, SV("1234567890"), 1, 4, S("abcdefghijklmnopqrs2345"));
    test(S("abcdefghijklmnopqrst"), 19, 2, SV("1234567890"), 1, 8, S("abcdefghijklmnopqrs23456789"));
    test(S("abcdefghijklmnopqrst"), 19, 2, SV("1234567890"), 1, 9, S("abcdefghijklmnopqrs234567890"));
    test(S("abcdefghijklmnopqrst"), 19, 2, SV("1234567890"), 1, 10, S("abcdefghijklmnopqrs234567890"));
    test(S("abcdefghijklmnopqrst"), 19, 2, SV("1234567890"), 5, 0, S("abcdefghijklmnopqrs"));
    test(S("abcdefghijklmnopqrst"), 19, 2, SV("1234567890"), 5, 1, S("abcdefghijklmnopqrs6"));
    test(S("abcdefghijklmnopqrst"), 19, 2, SV("1234567890"), 5, 2, S("abcdefghijklmnopqrs67"));
    test(S("abcdefghijklmnopqrst"), 19, 2, SV("1234567890"), 5, 4, S("abcdefghijklmnopqrs6789"));
    test(S("abcdefghijklmnopqrst"), 19, 2, SV("1234567890"), 5, 5, S("abcdefghijklmnopqrs67890"));
    test(S("abcdefghijklmnopqrst"), 19, 2, SV("1234567890"), 5, 6, S("abcdefghijklmnopqrs67890"));
    test(S("abcdefghijklmnopqrst"), 19, 2, SV("1234567890"), 9, 0, S("abcdefghijklmnopqrs"));
    test(S("abcdefghijklmnopqrst"), 19, 2, SV("1234567890"), 9, 1, S("abcdefghijklmnopqrs0"));
    test(S("abcdefghijklmnopqrst"), 19, 2, SV("1234567890"), 9, 2, S("abcdefghijklmnopqrs0"));
    test(S("abcdefghijklmnopqrst"), 19, 2, SV("1234567890"), 10, 0, S("abcdefghijklmnopqrs"));
    test(S("abcdefghijklmnopqrst"), 19, 2, SV("1234567890"), 10, 1, S("abcdefghijklmnopqrs"));
    test(S("abcdefghijklmnopqrst"), 19, 2, SV("1234567890"), 11, 0, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 19, 2, SV("12345678901234567890"), 0, 0, S("abcdefghijklmnopqrs"));
    test(S("abcdefghijklmnopqrst"), 19, 2, SV("12345678901234567890"), 0, 1, S("abcdefghijklmnopqrs1"));
    test(S("abcdefghijklmnopqrst"), 19, 2, SV("12345678901234567890"), 0, 10, S("abcdefghijklmnopqrs1234567890"));
    test(S("abcdefghijklmnopqrst"), 19, 2, SV("12345678901234567890"), 0, 19, S("abcdefghijklmnopqrs1234567890123456789"));
    test(S("abcdefghijklmnopqrst"), 19, 2, SV("12345678901234567890"), 0, 20, S("abcdefghijklmnopqrs12345678901234567890"));
    test(S("abcdefghijklmnopqrst"), 19, 2, SV("12345678901234567890"), 0, 21, S("abcdefghijklmnopqrs12345678901234567890"));
    test(S("abcdefghijklmnopqrst"), 19, 2, SV("12345678901234567890"), 1, 0, S("abcdefghijklmnopqrs"));
    test(S("abcdefghijklmnopqrst"), 19, 2, SV("12345678901234567890"), 1, 1, S("abcdefghijklmnopqrs2"));
    test(S("abcdefghijklmnopqrst"), 19, 2, SV("12345678901234567890"), 1, 9, S("abcdefghijklmnopqrs234567890"));
    test(S("abcdefghijklmnopqrst"), 19, 2, SV("12345678901234567890"), 1, 18, S("abcdefghijklmnopqrs234567890123456789"));
    test(S("abcdefghijklmnopqrst"), 19, 2, SV("12345678901234567890"), 1, 19, S("abcdefghijklmnopqrs2345678901234567890"));
    test(S("abcdefghijklmnopqrst"), 19, 2, SV("12345678901234567890"), 1, 20, S("abcdefghijklmnopqrs2345678901234567890"));
    test(S("abcdefghijklmnopqrst"), 19, 2, SV("12345678901234567890"), 10, 0, S("abcdefghijklmnopqrs"));
    test(S("abcdefghijklmnopqrst"), 19, 2, SV("12345678901234567890"), 10, 1, S("abcdefghijklmnopqrs1"));
    test(S("abcdefghijklmnopqrst"), 19, 2, SV("12345678901234567890"), 10, 5, S("abcdefghijklmnopqrs12345"));
    test(S("abcdefghijklmnopqrst"), 19, 2, SV("12345678901234567890"), 10, 9, S("abcdefghijklmnopqrs123456789"));
    test(S("abcdefghijklmnopqrst"), 19, 2, SV("12345678901234567890"), 10, 10, S("abcdefghijklmnopqrs1234567890"));
    test(S("abcdefghijklmnopqrst"), 19, 2, SV("12345678901234567890"), 10, 11, S("abcdefghijklmnopqrs1234567890"));
    test(S("abcdefghijklmnopqrst"), 19, 2, SV("12345678901234567890"), 19, 0, S("abcdefghijklmnopqrs"));
    test(S("abcdefghijklmnopqrst"), 19, 2, SV("12345678901234567890"), 19, 1, S("abcdefghijklmnopqrs0"));
    test(S("abcdefghijklmnopqrst"), 19, 2, SV("12345678901234567890"), 19, 2, S("abcdefghijklmnopqrs0"));
    test(S("abcdefghijklmnopqrst"), 19, 2, SV("12345678901234567890"), 20, 0, S("abcdefghijklmnopqrs"));
    test(S("abcdefghijklmnopqrst"), 19, 2, SV("12345678901234567890"), 20, 1, S("abcdefghijklmnopqrs"));
    test(S("abcdefghijklmnopqrst"), 19, 2, SV("12345678901234567890"), 21, 0, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 20, 0, SV(""), 0, 0, S("abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 20, 0, SV(""), 0, 1, S("abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 20, 0, SV(""), 1, 0, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 20, 0, SV("12345"), 0, 0, S("abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 20, 0, SV("12345"), 0, 1, S("abcdefghijklmnopqrst1"));
    test(S("abcdefghijklmnopqrst"), 20, 0, SV("12345"), 0, 2, S("abcdefghijklmnopqrst12"));
    test(S("abcdefghijklmnopqrst"), 20, 0, SV("12345"), 0, 4, S("abcdefghijklmnopqrst1234"));
    test(S("abcdefghijklmnopqrst"), 20, 0, SV("12345"), 0, 5, S("abcdefghijklmnopqrst12345"));
    test(S("abcdefghijklmnopqrst"), 20, 0, SV("12345"), 0, 6, S("abcdefghijklmnopqrst12345"));
    test(S("abcdefghijklmnopqrst"), 20, 0, SV("12345"), 1, 0, S("abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 20, 0, SV("12345"), 1, 1, S("abcdefghijklmnopqrst2"));
    test(S("abcdefghijklmnopqrst"), 20, 0, SV("12345"), 1, 2, S("abcdefghijklmnopqrst23"));
    test(S("abcdefghijklmnopqrst"), 20, 0, SV("12345"), 1, 3, S("abcdefghijklmnopqrst234"));
    test(S("abcdefghijklmnopqrst"), 20, 0, SV("12345"), 1, 4, S("abcdefghijklmnopqrst2345"));
    test(S("abcdefghijklmnopqrst"), 20, 0, SV("12345"), 1, 5, S("abcdefghijklmnopqrst2345"));
    test(S("abcdefghijklmnopqrst"), 20, 0, SV("12345"), 2, 0, S("abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 20, 0, SV("12345"), 2, 1, S("abcdefghijklmnopqrst3"));
    test(S("abcdefghijklmnopqrst"), 20, 0, SV("12345"), 2, 2, S("abcdefghijklmnopqrst34"));
    test(S("abcdefghijklmnopqrst"), 20, 0, SV("12345"), 2, 3, S("abcdefghijklmnopqrst345"));
    test(S("abcdefghijklmnopqrst"), 20, 0, SV("12345"), 2, 4, S("abcdefghijklmnopqrst345"));
    test(S("abcdefghijklmnopqrst"), 20, 0, SV("12345"), 4, 0, S("abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 20, 0, SV("12345"), 4, 1, S("abcdefghijklmnopqrst5"));
    test(S("abcdefghijklmnopqrst"), 20, 0, SV("12345"), 4, 2, S("abcdefghijklmnopqrst5"));
    test(S("abcdefghijklmnopqrst"), 20, 0, SV("12345"), 5, 0, S("abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 20, 0, SV("12345"), 5, 1, S("abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 20, 0, SV("12345"), 6, 0, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 20, 0, SV("1234567890"), 0, 0, S("abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 20, 0, SV("1234567890"), 0, 1, S("abcdefghijklmnopqrst1"));
    test(S("abcdefghijklmnopqrst"), 20, 0, SV("1234567890"), 0, 5, S("abcdefghijklmnopqrst12345"));
    test(S("abcdefghijklmnopqrst"), 20, 0, SV("1234567890"), 0, 9, S("abcdefghijklmnopqrst123456789"));
    test(S("abcdefghijklmnopqrst"), 20, 0, SV("1234567890"), 0, 10, S("abcdefghijklmnopqrst1234567890"));
    test(S("abcdefghijklmnopqrst"), 20, 0, SV("1234567890"), 0, 11, S("abcdefghijklmnopqrst1234567890"));
    test(S("abcdefghijklmnopqrst"), 20, 0, SV("1234567890"), 1, 0, S("abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 20, 0, SV("1234567890"), 1, 1, S("abcdefghijklmnopqrst2"));
    test(S("abcdefghijklmnopqrst"), 20, 0, SV("1234567890"), 1, 4, S("abcdefghijklmnopqrst2345"));
    test(S("abcdefghijklmnopqrst"), 20, 0, SV("1234567890"), 1, 8, S("abcdefghijklmnopqrst23456789"));
    test(S("abcdefghijklmnopqrst"), 20, 0, SV("1234567890"), 1, 9, S("abcdefghijklmnopqrst234567890"));
    test(S("abcdefghijklmnopqrst"), 20, 0, SV("1234567890"), 1, 10, S("abcdefghijklmnopqrst234567890"));
    test(S("abcdefghijklmnopqrst"), 20, 0, SV("1234567890"), 5, 0, S("abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 20, 0, SV("1234567890"), 5, 1, S("abcdefghijklmnopqrst6"));
    test(S("abcdefghijklmnopqrst"), 20, 0, SV("1234567890"), 5, 2, S("abcdefghijklmnopqrst67"));
    test(S("abcdefghijklmnopqrst"), 20, 0, SV("1234567890"), 5, 4, S("abcdefghijklmnopqrst6789"));
    test(S("abcdefghijklmnopqrst"), 20, 0, SV("1234567890"), 5, 5, S("abcdefghijklmnopqrst67890"));
    test(S("abcdefghijklmnopqrst"), 20, 0, SV("1234567890"), 5, 6, S("abcdefghijklmnopqrst67890"));
    test(S("abcdefghijklmnopqrst"), 20, 0, SV("1234567890"), 9, 0, S("abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 20, 0, SV("1234567890"), 9, 1, S("abcdefghijklmnopqrst0"));

    return true;
}

template <class S, class SV>
TEST_CONSTEXPR_CXX20 bool test53()
{
    test(S("abcdefghijklmnopqrst"), 20, 0, SV("1234567890"), 9, 2, S("abcdefghijklmnopqrst0"));
    test(S("abcdefghijklmnopqrst"), 20, 0, SV("1234567890"), 10, 0, S("abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 20, 0, SV("1234567890"), 10, 1, S("abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 20, 0, SV("1234567890"), 11, 0, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 20, 0, SV("12345678901234567890"), 0, 0, S("abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 20, 0, SV("12345678901234567890"), 0, 1, S("abcdefghijklmnopqrst1"));
    test(S("abcdefghijklmnopqrst"), 20, 0, SV("12345678901234567890"), 0, 10, S("abcdefghijklmnopqrst1234567890"));
    test(S("abcdefghijklmnopqrst"), 20, 0, SV("12345678901234567890"), 0, 19, S("abcdefghijklmnopqrst1234567890123456789"));
    test(S("abcdefghijklmnopqrst"), 20, 0, SV("12345678901234567890"), 0, 20, S("abcdefghijklmnopqrst12345678901234567890"));
    test(S("abcdefghijklmnopqrst"), 20, 0, SV("12345678901234567890"), 0, 21, S("abcdefghijklmnopqrst12345678901234567890"));
    test(S("abcdefghijklmnopqrst"), 20, 0, SV("12345678901234567890"), 1, 0, S("abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 20, 0, SV("12345678901234567890"), 1, 1, S("abcdefghijklmnopqrst2"));
    test(S("abcdefghijklmnopqrst"), 20, 0, SV("12345678901234567890"), 1, 9, S("abcdefghijklmnopqrst234567890"));
    test(S("abcdefghijklmnopqrst"), 20, 0, SV("12345678901234567890"), 1, 18, S("abcdefghijklmnopqrst234567890123456789"));
    test(S("abcdefghijklmnopqrst"), 20, 0, SV("12345678901234567890"), 1, 19, S("abcdefghijklmnopqrst2345678901234567890"));
    test(S("abcdefghijklmnopqrst"), 20, 0, SV("12345678901234567890"), 1, 20, S("abcdefghijklmnopqrst2345678901234567890"));
    test(S("abcdefghijklmnopqrst"), 20, 0, SV("12345678901234567890"), 10, 0, S("abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 20, 0, SV("12345678901234567890"), 10, 1, S("abcdefghijklmnopqrst1"));
    test(S("abcdefghijklmnopqrst"), 20, 0, SV("12345678901234567890"), 10, 5, S("abcdefghijklmnopqrst12345"));
    test(S("abcdefghijklmnopqrst"), 20, 0, SV("12345678901234567890"), 10, 9, S("abcdefghijklmnopqrst123456789"));
    test(S("abcdefghijklmnopqrst"), 20, 0, SV("12345678901234567890"), 10, 10, S("abcdefghijklmnopqrst1234567890"));
    test(S("abcdefghijklmnopqrst"), 20, 0, SV("12345678901234567890"), 10, 11, S("abcdefghijklmnopqrst1234567890"));
    test(S("abcdefghijklmnopqrst"), 20, 0, SV("12345678901234567890"), 19, 0, S("abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 20, 0, SV("12345678901234567890"), 19, 1, S("abcdefghijklmnopqrst0"));
    test(S("abcdefghijklmnopqrst"), 20, 0, SV("12345678901234567890"), 19, 2, S("abcdefghijklmnopqrst0"));
    test(S("abcdefghijklmnopqrst"), 20, 0, SV("12345678901234567890"), 20, 0, S("abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 20, 0, SV("12345678901234567890"), 20, 1, S("abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 20, 0, SV("12345678901234567890"), 21, 0, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 20, 1, SV(""), 0, 0, S("abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 20, 1, SV(""), 0, 1, S("abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 20, 1, SV(""), 1, 0, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 20, 1, SV("12345"), 0, 0, S("abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 20, 1, SV("12345"), 0, 1, S("abcdefghijklmnopqrst1"));
    test(S("abcdefghijklmnopqrst"), 20, 1, SV("12345"), 0, 2, S("abcdefghijklmnopqrst12"));
    test(S("abcdefghijklmnopqrst"), 20, 1, SV("12345"), 0, 4, S("abcdefghijklmnopqrst1234"));
    test(S("abcdefghijklmnopqrst"), 20, 1, SV("12345"), 0, 5, S("abcdefghijklmnopqrst12345"));
    test(S("abcdefghijklmnopqrst"), 20, 1, SV("12345"), 0, 6, S("abcdefghijklmnopqrst12345"));
    test(S("abcdefghijklmnopqrst"), 20, 1, SV("12345"), 1, 0, S("abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 20, 1, SV("12345"), 1, 1, S("abcdefghijklmnopqrst2"));
    test(S("abcdefghijklmnopqrst"), 20, 1, SV("12345"), 1, 2, S("abcdefghijklmnopqrst23"));
    test(S("abcdefghijklmnopqrst"), 20, 1, SV("12345"), 1, 3, S("abcdefghijklmnopqrst234"));
    test(S("abcdefghijklmnopqrst"), 20, 1, SV("12345"), 1, 4, S("abcdefghijklmnopqrst2345"));
    test(S("abcdefghijklmnopqrst"), 20, 1, SV("12345"), 1, 5, S("abcdefghijklmnopqrst2345"));
    test(S("abcdefghijklmnopqrst"), 20, 1, SV("12345"), 2, 0, S("abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 20, 1, SV("12345"), 2, 1, S("abcdefghijklmnopqrst3"));
    test(S("abcdefghijklmnopqrst"), 20, 1, SV("12345"), 2, 2, S("abcdefghijklmnopqrst34"));
    test(S("abcdefghijklmnopqrst"), 20, 1, SV("12345"), 2, 3, S("abcdefghijklmnopqrst345"));
    test(S("abcdefghijklmnopqrst"), 20, 1, SV("12345"), 2, 4, S("abcdefghijklmnopqrst345"));
    test(S("abcdefghijklmnopqrst"), 20, 1, SV("12345"), 4, 0, S("abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 20, 1, SV("12345"), 4, 1, S("abcdefghijklmnopqrst5"));
    test(S("abcdefghijklmnopqrst"), 20, 1, SV("12345"), 4, 2, S("abcdefghijklmnopqrst5"));
    test(S("abcdefghijklmnopqrst"), 20, 1, SV("12345"), 5, 0, S("abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 20, 1, SV("12345"), 5, 1, S("abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 20, 1, SV("12345"), 6, 0, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 20, 1, SV("1234567890"), 0, 0, S("abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 20, 1, SV("1234567890"), 0, 1, S("abcdefghijklmnopqrst1"));
    test(S("abcdefghijklmnopqrst"), 20, 1, SV("1234567890"), 0, 5, S("abcdefghijklmnopqrst12345"));
    test(S("abcdefghijklmnopqrst"), 20, 1, SV("1234567890"), 0, 9, S("abcdefghijklmnopqrst123456789"));
    test(S("abcdefghijklmnopqrst"), 20, 1, SV("1234567890"), 0, 10, S("abcdefghijklmnopqrst1234567890"));
    test(S("abcdefghijklmnopqrst"), 20, 1, SV("1234567890"), 0, 11, S("abcdefghijklmnopqrst1234567890"));
    test(S("abcdefghijklmnopqrst"), 20, 1, SV("1234567890"), 1, 0, S("abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 20, 1, SV("1234567890"), 1, 1, S("abcdefghijklmnopqrst2"));
    test(S("abcdefghijklmnopqrst"), 20, 1, SV("1234567890"), 1, 4, S("abcdefghijklmnopqrst2345"));
    test(S("abcdefghijklmnopqrst"), 20, 1, SV("1234567890"), 1, 8, S("abcdefghijklmnopqrst23456789"));
    test(S("abcdefghijklmnopqrst"), 20, 1, SV("1234567890"), 1, 9, S("abcdefghijklmnopqrst234567890"));
    test(S("abcdefghijklmnopqrst"), 20, 1, SV("1234567890"), 1, 10, S("abcdefghijklmnopqrst234567890"));
    test(S("abcdefghijklmnopqrst"), 20, 1, SV("1234567890"), 5, 0, S("abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 20, 1, SV("1234567890"), 5, 1, S("abcdefghijklmnopqrst6"));
    test(S("abcdefghijklmnopqrst"), 20, 1, SV("1234567890"), 5, 2, S("abcdefghijklmnopqrst67"));
    test(S("abcdefghijklmnopqrst"), 20, 1, SV("1234567890"), 5, 4, S("abcdefghijklmnopqrst6789"));
    test(S("abcdefghijklmnopqrst"), 20, 1, SV("1234567890"), 5, 5, S("abcdefghijklmnopqrst67890"));
    test(S("abcdefghijklmnopqrst"), 20, 1, SV("1234567890"), 5, 6, S("abcdefghijklmnopqrst67890"));
    test(S("abcdefghijklmnopqrst"), 20, 1, SV("1234567890"), 9, 0, S("abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 20, 1, SV("1234567890"), 9, 1, S("abcdefghijklmnopqrst0"));
    test(S("abcdefghijklmnopqrst"), 20, 1, SV("1234567890"), 9, 2, S("abcdefghijklmnopqrst0"));
    test(S("abcdefghijklmnopqrst"), 20, 1, SV("1234567890"), 10, 0, S("abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 20, 1, SV("1234567890"), 10, 1, S("abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 20, 1, SV("1234567890"), 11, 0, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 20, 1, SV("12345678901234567890"), 0, 0, S("abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 20, 1, SV("12345678901234567890"), 0, 1, S("abcdefghijklmnopqrst1"));
    test(S("abcdefghijklmnopqrst"), 20, 1, SV("12345678901234567890"), 0, 10, S("abcdefghijklmnopqrst1234567890"));
    test(S("abcdefghijklmnopqrst"), 20, 1, SV("12345678901234567890"), 0, 19, S("abcdefghijklmnopqrst1234567890123456789"));
    test(S("abcdefghijklmnopqrst"), 20, 1, SV("12345678901234567890"), 0, 20, S("abcdefghijklmnopqrst12345678901234567890"));
    test(S("abcdefghijklmnopqrst"), 20, 1, SV("12345678901234567890"), 0, 21, S("abcdefghijklmnopqrst12345678901234567890"));
    test(S("abcdefghijklmnopqrst"), 20, 1, SV("12345678901234567890"), 1, 0, S("abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 20, 1, SV("12345678901234567890"), 1, 1, S("abcdefghijklmnopqrst2"));
    test(S("abcdefghijklmnopqrst"), 20, 1, SV("12345678901234567890"), 1, 9, S("abcdefghijklmnopqrst234567890"));
    test(S("abcdefghijklmnopqrst"), 20, 1, SV("12345678901234567890"), 1, 18, S("abcdefghijklmnopqrst234567890123456789"));
    test(S("abcdefghijklmnopqrst"), 20, 1, SV("12345678901234567890"), 1, 19, S("abcdefghijklmnopqrst2345678901234567890"));
    test(S("abcdefghijklmnopqrst"), 20, 1, SV("12345678901234567890"), 1, 20, S("abcdefghijklmnopqrst2345678901234567890"));
    test(S("abcdefghijklmnopqrst"), 20, 1, SV("12345678901234567890"), 10, 0, S("abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 20, 1, SV("12345678901234567890"), 10, 1, S("abcdefghijklmnopqrst1"));
    test(S("abcdefghijklmnopqrst"), 20, 1, SV("12345678901234567890"), 10, 5, S("abcdefghijklmnopqrst12345"));
    test(S("abcdefghijklmnopqrst"), 20, 1, SV("12345678901234567890"), 10, 9, S("abcdefghijklmnopqrst123456789"));
    test(S("abcdefghijklmnopqrst"), 20, 1, SV("12345678901234567890"), 10, 10, S("abcdefghijklmnopqrst1234567890"));
    test(S("abcdefghijklmnopqrst"), 20, 1, SV("12345678901234567890"), 10, 11, S("abcdefghijklmnopqrst1234567890"));
    test(S("abcdefghijklmnopqrst"), 20, 1, SV("12345678901234567890"), 19, 0, S("abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 20, 1, SV("12345678901234567890"), 19, 1, S("abcdefghijklmnopqrst0"));
    test(S("abcdefghijklmnopqrst"), 20, 1, SV("12345678901234567890"), 19, 2, S("abcdefghijklmnopqrst0"));
    test(S("abcdefghijklmnopqrst"), 20, 1, SV("12345678901234567890"), 20, 0, S("abcdefghijklmnopqrst"));

    return true;
}

template <class S, class SV>
TEST_CONSTEXPR_CXX20 bool test54()
{
    test(S("abcdefghijklmnopqrst"), 20, 1, SV("12345678901234567890"), 20, 1, S("abcdefghijklmnopqrst"));
    test(S("abcdefghijklmnopqrst"), 20, 1, SV("12345678901234567890"), 21, 0, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 21, 0, SV(""), 0, 0, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 21, 0, SV(""), 0, 1, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 21, 0, SV(""), 1, 0, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 21, 0, SV("12345"), 0, 0, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 21, 0, SV("12345"), 0, 1, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 21, 0, SV("12345"), 0, 2, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 21, 0, SV("12345"), 0, 4, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 21, 0, SV("12345"), 0, 5, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 21, 0, SV("12345"), 0, 6, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 21, 0, SV("12345"), 1, 0, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 21, 0, SV("12345"), 1, 1, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 21, 0, SV("12345"), 1, 2, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 21, 0, SV("12345"), 1, 3, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 21, 0, SV("12345"), 1, 4, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 21, 0, SV("12345"), 1, 5, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 21, 0, SV("12345"), 2, 0, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 21, 0, SV("12345"), 2, 1, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 21, 0, SV("12345"), 2, 2, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 21, 0, SV("12345"), 2, 3, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 21, 0, SV("12345"), 2, 4, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 21, 0, SV("12345"), 4, 0, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 21, 0, SV("12345"), 4, 1, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 21, 0, SV("12345"), 4, 2, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 21, 0, SV("12345"), 5, 0, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 21, 0, SV("12345"), 5, 1, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 21, 0, SV("12345"), 6, 0, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 21, 0, SV("1234567890"), 0, 0, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 21, 0, SV("1234567890"), 0, 1, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 21, 0, SV("1234567890"), 0, 5, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 21, 0, SV("1234567890"), 0, 9, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 21, 0, SV("1234567890"), 0, 10, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 21, 0, SV("1234567890"), 0, 11, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 21, 0, SV("1234567890"), 1, 0, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 21, 0, SV("1234567890"), 1, 1, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 21, 0, SV("1234567890"), 1, 4, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 21, 0, SV("1234567890"), 1, 8, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 21, 0, SV("1234567890"), 1, 9, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 21, 0, SV("1234567890"), 1, 10, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 21, 0, SV("1234567890"), 5, 0, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 21, 0, SV("1234567890"), 5, 1, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 21, 0, SV("1234567890"), 5, 2, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 21, 0, SV("1234567890"), 5, 4, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 21, 0, SV("1234567890"), 5, 5, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 21, 0, SV("1234567890"), 5, 6, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 21, 0, SV("1234567890"), 9, 0, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 21, 0, SV("1234567890"), 9, 1, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 21, 0, SV("1234567890"), 9, 2, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 21, 0, SV("1234567890"), 10, 0, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 21, 0, SV("1234567890"), 10, 1, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 21, 0, SV("1234567890"), 11, 0, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 21, 0, SV("12345678901234567890"), 0, 0, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 21, 0, SV("12345678901234567890"), 0, 1, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 21, 0, SV("12345678901234567890"), 0, 10, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 21, 0, SV("12345678901234567890"), 0, 19, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 21, 0, SV("12345678901234567890"), 0, 20, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 21, 0, SV("12345678901234567890"), 0, 21, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 21, 0, SV("12345678901234567890"), 1, 0, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 21, 0, SV("12345678901234567890"), 1, 1, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 21, 0, SV("12345678901234567890"), 1, 9, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 21, 0, SV("12345678901234567890"), 1, 18, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 21, 0, SV("12345678901234567890"), 1, 19, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 21, 0, SV("12345678901234567890"), 1, 20, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 21, 0, SV("12345678901234567890"), 10, 0, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 21, 0, SV("12345678901234567890"), 10, 1, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 21, 0, SV("12345678901234567890"), 10, 5, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 21, 0, SV("12345678901234567890"), 10, 9, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 21, 0, SV("12345678901234567890"), 10, 10, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 21, 0, SV("12345678901234567890"), 10, 11, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 21, 0, SV("12345678901234567890"), 19, 0, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 21, 0, SV("12345678901234567890"), 19, 1, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 21, 0, SV("12345678901234567890"), 19, 2, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 21, 0, SV("12345678901234567890"), 20, 0, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 21, 0, SV("12345678901234567890"), 20, 1, S("can't happen"));
    test(S("abcdefghijklmnopqrst"), 21, 0, SV("12345678901234567890"), 21, 0, S("can't happen"));

    return true;
}

template <class S, class SV>
TEST_CONSTEXPR_CXX20 bool test55()
{
    test_npos(S("abcdefghij"), 9, 1, SV("12345678901234567890"), 10, S("abcdefghi1234567890"));
    test_npos(S("abcdefghij"), 9, 1, SV("12345678901234567890"), 19, S("abcdefghi0"));
    test_npos(S("abcdefghij"), 9, 1, SV("12345678901234567890"), 20, S("abcdefghi"));
    test_npos(S("abcdefghij"), 9, 1, SV("12345678901234567890"), 20, S("abcdefghi"));
    test_npos(S("abcdefghij"), 9, 1, SV("12345678901234567890"), 21, S("can't happen"));
    test_npos(S("abcdefghij"), 9, 2, SV(""), 0, S("abcdefghi"));
    test_npos(S("abcdefghij"), 9, 2, SV(""), 1, S("can't happen"));
    test_npos(S("abcdefghij"), 9, 2, SV("12345"), 0, S("abcdefghi12345"));
    test_npos(S("abcdefghij"), 9, 2, SV("12345"), 1, S("abcdefghi2345"));
    test_npos(S("abcdefghij"), 9, 2, SV("12345"), 2, S("abcdefghi345"));
    test_npos(S("abcdefghij"), 9, 2, SV("12345"), 4, S("abcdefghi5"));
    test_npos(S("abcdefghij"), 9, 2, SV("12345"), 5, S("abcdefghi"));
    test_npos(S("abcdefghij"), 9, 2, SV("12345"), 6, S("can't happen"));

    return true;
}

TEST_CONSTEXPR_CXX20 bool test56() {
    typedef std::string S;
    typedef std::string_view SV;
    S s0 = "ABCD";
    S s;
    SV sv = "EFGH";
    char arr[] = "IJKL";

    s = s0;
    s.replace(0, 4, "CDEF", 0);    // calls replace(pos1, n1, const char *, len)
    assert(s == "");

    s = s0;
    s.replace(0, 4, "QRST", 0, std::string::npos); // calls replace(pos1, n1, string("QRST"), pos, npos)
    assert(s == "QRST");

    s = s0;
    s.replace(0, 4, sv, 0);  // calls replace(pos1, n1, T, pos, npos)
    assert(s == sv);

    s = s0;
    s.replace(0, 4, sv, 0, std::string::npos);   // calls replace(pos1, n1, T, pos, npos)
    assert(s == sv);

    s = s0;
    s.replace(0, 4, arr, 0);     // calls replace(pos1, n1, const char *, len)
    assert(s == "");

    s = s0;
    s.replace(0, 4, arr, 0, std::string::npos);    // calls replace(pos1, n1, string("IJKL"), pos, npos)
    assert(s == "IJKL");

    return true;
}

template <class S, class SV>
TEST_CONSTEXPR_CXX20 bool test() {
  test0<S, SV>();
  test1<S, SV>();
  test2<S, SV>();
  test3<S, SV>();
  test4<S, SV>();
  test5<S, SV>();
  test6<S, SV>();
  test7<S, SV>();
  test8<S, SV>();
  test9<S, SV>();
  test10<S, SV>();
  test11<S, SV>();
  test12<S, SV>();
  test13<S, SV>();
  test14<S, SV>();
  test15<S, SV>();
  test16<S, SV>();
  test17<S, SV>();
  test18<S, SV>();
  test19<S, SV>();
  test20<S, SV>();
  test21<S, SV>();
  test22<S, SV>();
  test23<S, SV>();
  test24<S, SV>();
  test25<S, SV>();
  test26<S, SV>();
  test27<S, SV>();
  test28<S, SV>();
  test29<S, SV>();
  test30<S, SV>();
  test31<S, SV>();
  test32<S, SV>();
  test33<S, SV>();
  test34<S, SV>();
  test35<S, SV>();
  test36<S, SV>();
  test37<S, SV>();
  test38<S, SV>();
  test39<S, SV>();
  test40<S, SV>();
  test41<S, SV>();
  test42<S, SV>();
  test43<S, SV>();
  test44<S, SV>();
  test45<S, SV>();
  test46<S, SV>();
  test47<S, SV>();
  test48<S, SV>();
  test49<S, SV>();
  test50<S, SV>();
  test51<S, SV>();
  test52<S, SV>();
  test53<S, SV>();
  test54<S, SV>();
  test55<S, SV>();

#if TEST_STD_VER > 17
  static_assert(test0<S, SV>());
  static_assert(test1<S, SV>());
  static_assert(test2<S, SV>());
  static_assert(test3<S, SV>());
  static_assert(test4<S, SV>());
  static_assert(test5<S, SV>());
  static_assert(test6<S, SV>());
  static_assert(test7<S, SV>());
  static_assert(test8<S, SV>());
  static_assert(test9<S, SV>());
  static_assert(test10<S, SV>());
  static_assert(test11<S, SV>());
  static_assert(test12<S, SV>());
  static_assert(test13<S, SV>());
  static_assert(test14<S, SV>());
  static_assert(test15<S, SV>());
  static_assert(test16<S, SV>());
  static_assert(test17<S, SV>());
  static_assert(test18<S, SV>());
  static_assert(test19<S, SV>());
  static_assert(test20<S, SV>());
  static_assert(test21<S, SV>());
  static_assert(test22<S, SV>());
  static_assert(test23<S, SV>());
  static_assert(test24<S, SV>());
  static_assert(test25<S, SV>());
  static_assert(test26<S, SV>());
  static_assert(test27<S, SV>());
  static_assert(test28<S, SV>());
  static_assert(test29<S, SV>());
  static_assert(test30<S, SV>());
  static_assert(test31<S, SV>());
  static_assert(test32<S, SV>());
  static_assert(test33<S, SV>());
  static_assert(test34<S, SV>());
  static_assert(test35<S, SV>());
  static_assert(test36<S, SV>());
  static_assert(test37<S, SV>());
  static_assert(test38<S, SV>());
  static_assert(test39<S, SV>());
  static_assert(test40<S, SV>());
  static_assert(test41<S, SV>());
  static_assert(test42<S, SV>());
  static_assert(test43<S, SV>());
  static_assert(test44<S, SV>());
  static_assert(test45<S, SV>());
  static_assert(test46<S, SV>());
  static_assert(test47<S, SV>());
  static_assert(test48<S, SV>());
  static_assert(test49<S, SV>());
  static_assert(test50<S, SV>());
  static_assert(test51<S, SV>());
  static_assert(test52<S, SV>());
  static_assert(test53<S, SV>());
  static_assert(test54<S, SV>());
  static_assert(test55<S, SV>());
#endif

  return true;
}

int main(int, char**)
{
  test<std::string, std::string_view>();
#if TEST_STD_VER >= 11
  test<std::basic_string<char, std::char_traits<char>, min_allocator<char>>,
       std::basic_string_view<char, std::char_traits<char>>>();
#endif

  return 0;
}
