/*
    chips.c - Part of libsensors, a Linux library for reading sensor data.
    Copyright (c) 1998, 1999  Frodo Looijaard <frodol@dds.nl> 

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "chips.h"
#include "data.h"
#include "sensors.h"
#include "kernel/include/sensors.h"

/* For each chip, a list of features;
   For each feature, the number, its name, the logical mapping (for example,
   fan1_div belongs to fan1), the compute mapping, an access field, the 
   SYSCTL id of the /proc field, the value location of the /proc field,
   and the magnitude.
   The access field  indicates whether resources are readable and writable.
   Sometimes, a resource is neither; in this case, it is just a catch-all
   category, for mappings. 
   The value location is the byte-offset where this value can be found in
   the sysctl buffer.
   The magnitude is the 10-power through which the read value must be
   divided. */


#define VALUE(x) (x-1) * sizeof(long)

static sensors_chip_feature lm78_features[] =
  { 
    { SENSORS_LM78_IN0, "in0", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING, 
                        SENSORS_MODE_R, LM78_SYSCTL_IN0, VALUE(3), 2 },
    { SENSORS_LM78_IN1, "in1", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, LM78_SYSCTL_IN1, VALUE(3), 2 },
    { SENSORS_LM78_IN2, "in2", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, LM78_SYSCTL_IN2, VALUE(3), 2 },
    { SENSORS_LM78_IN3, "in3", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, LM78_SYSCTL_IN3, VALUE(3), 2 },
    { SENSORS_LM78_IN4, "in4", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, LM78_SYSCTL_IN4, VALUE(3), 2 },
    { SENSORS_LM78_IN5, "in5", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, LM78_SYSCTL_IN5, VALUE(3), 2 },
    { SENSORS_LM78_IN6, "in6", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, LM78_SYSCTL_IN6, VALUE(3), 2 },
    { SENSORS_LM78_IN0_MIN, "in0_min", SENSORS_LM78_IN0, SENSORS_LM78_IN0,
                            SENSORS_MODE_RW, LM78_SYSCTL_IN0, VALUE(1), 2 },
    { SENSORS_LM78_IN1_MIN, "in1_min", SENSORS_LM78_IN1, SENSORS_LM78_IN1,
                            SENSORS_MODE_RW, LM78_SYSCTL_IN1, VALUE(1), 2 },
    { SENSORS_LM78_IN2_MIN, "in2_min", SENSORS_LM78_IN2, SENSORS_LM78_IN2,
                            SENSORS_MODE_RW, LM78_SYSCTL_IN2, VALUE(1), 2 },
    { SENSORS_LM78_IN3_MIN, "in3_min", SENSORS_LM78_IN3, SENSORS_LM78_IN3,
                            SENSORS_MODE_RW, LM78_SYSCTL_IN3, VALUE(1), 2 },
    { SENSORS_LM78_IN4_MIN, "in4_min", SENSORS_LM78_IN4, SENSORS_LM78_IN4,
                            SENSORS_MODE_RW, LM78_SYSCTL_IN4, VALUE(1), 2 },
/* swap 5 and 6 min/max because compute lines will be inverting */
    { SENSORS_LM78_IN5_MIN, "in5_min", SENSORS_LM78_IN5, SENSORS_LM78_IN5,
                            SENSORS_MODE_RW, LM78_SYSCTL_IN5, VALUE(2), 2 },
    { SENSORS_LM78_IN6_MIN, "in6_min", SENSORS_LM78_IN6, SENSORS_LM78_IN6,
                            SENSORS_MODE_RW, LM78_SYSCTL_IN6, VALUE(2), 2 },
    { SENSORS_LM78_IN0_MAX, "in0_max", SENSORS_LM78_IN0, SENSORS_LM78_IN0,
                            SENSORS_MODE_RW, LM78_SYSCTL_IN0, VALUE(2), 2 },
    { SENSORS_LM78_IN1_MAX, "in1_max", SENSORS_LM78_IN1, SENSORS_LM78_IN1,
                            SENSORS_MODE_RW, LM78_SYSCTL_IN1, VALUE(2), 2 },
    { SENSORS_LM78_IN2_MAX, "in2_max", SENSORS_LM78_IN2, SENSORS_LM78_IN2,
                            SENSORS_MODE_RW, LM78_SYSCTL_IN2, VALUE(2), 2 },
    { SENSORS_LM78_IN3_MAX, "in3_max", SENSORS_LM78_IN3, SENSORS_LM78_IN3,
                            SENSORS_MODE_RW, LM78_SYSCTL_IN3, VALUE(2), 2 },
    { SENSORS_LM78_IN4_MAX, "in4_max", SENSORS_LM78_IN4, SENSORS_LM78_IN4,
                            SENSORS_MODE_RW, LM78_SYSCTL_IN4, VALUE(2), 2 },
    { SENSORS_LM78_IN5_MAX, "in5_max", SENSORS_LM78_IN5, SENSORS_LM78_IN5,
                            SENSORS_MODE_RW, LM78_SYSCTL_IN5, VALUE(1), 2 },
    { SENSORS_LM78_IN6_MAX, "in6_max", SENSORS_LM78_IN6, SENSORS_LM78_IN6,
                            SENSORS_MODE_RW, LM78_SYSCTL_IN6, VALUE(1), 2 },
    { SENSORS_LM78_FAN1, "fan1", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, LM78_SYSCTL_FAN1, VALUE(2), 0 },
    { SENSORS_LM78_FAN2, "fan2", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, LM78_SYSCTL_FAN2, VALUE(2), 0 },
    { SENSORS_LM78_FAN3, "fan3", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, LM78_SYSCTL_FAN3, VALUE(2), 0 },
    { SENSORS_LM78_FAN1_MIN, "fan1_min", SENSORS_LM78_FAN1, SENSORS_LM78_FAN1,
                             SENSORS_MODE_RW, LM78_SYSCTL_FAN1, VALUE(1), 0 },
    { SENSORS_LM78_FAN2_MIN, "fan2_min", SENSORS_LM78_FAN2, SENSORS_LM78_FAN2,
                             SENSORS_MODE_RW, LM78_SYSCTL_FAN2, VALUE(1), 0 },
    { SENSORS_LM78_FAN3_MIN, "fan3_min", SENSORS_LM78_FAN3, SENSORS_LM78_FAN3,
                             SENSORS_MODE_RW, LM78_SYSCTL_FAN3, VALUE(1), 0 },
    { SENSORS_LM78_TEMP, "temp", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, LM78_SYSCTL_TEMP, VALUE(3), 1 },
    { SENSORS_LM78_TEMP_HYST, "temp_hyst", SENSORS_LM78_TEMP,
                              SENSORS_LM78_TEMP, SENSORS_MODE_RW, 
                              LM78_SYSCTL_TEMP, VALUE(2), 1 },
    { SENSORS_LM78_TEMP_OVER, "temp_over", SENSORS_LM78_TEMP,
                              SENSORS_LM78_TEMP, SENSORS_MODE_RW, 
                              LM78_SYSCTL_TEMP, VALUE(1), 1 },
    { SENSORS_LM78_VID, "vid", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, LM78_SYSCTL_VID, VALUE(1), 2 },
    { SENSORS_LM78_FAN1_DIV, "fan1_div", SENSORS_LM78_FAN1, SENSORS_NO_MAPPING,
                             SENSORS_MODE_RW, LM78_SYSCTL_FAN_DIV, VALUE(1), 
                             0 },
    { SENSORS_LM78_FAN2_DIV, "fan2_div", SENSORS_LM78_FAN2, SENSORS_NO_MAPPING,
                             SENSORS_MODE_RW, LM78_SYSCTL_FAN_DIV, VALUE(2), 
                             0 },
    { SENSORS_LM78_FAN3_DIV, "fan3_div", SENSORS_LM78_FAN3, SENSORS_NO_MAPPING,
                              SENSORS_MODE_R, LM78_SYSCTL_FAN_DIV, VALUE(3), 
                              0 },
    { SENSORS_LM78_ALARMS, "alarms", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING, 
                           SENSORS_MODE_R, LM78_SYSCTL_ALARMS, VALUE(1), 0 },
    { 0 }
  };

static sensors_chip_feature lm78j_features[] =
  { 
    { SENSORS_LM78J_IN0, "in0", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING, 
                         SENSORS_MODE_R, LM78_SYSCTL_IN0, VALUE(3), 2 },
    { SENSORS_LM78J_IN1, "in1", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, LM78_SYSCTL_IN1, VALUE(3), 2 },
    { SENSORS_LM78J_IN2, "in2", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, LM78_SYSCTL_IN2, VALUE(3), 2 },
    { SENSORS_LM78J_IN3, "in3", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, LM78_SYSCTL_IN3, VALUE(3), 2 },
    { SENSORS_LM78J_IN4, "in4", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, LM78_SYSCTL_IN4, VALUE(3), 2 },
    { SENSORS_LM78J_IN5, "in5", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, LM78_SYSCTL_IN5, VALUE(3), 2 },
    { SENSORS_LM78J_IN6, "in6", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, LM78_SYSCTL_IN6, VALUE(3), 2 },
    { SENSORS_LM78J_IN0_MIN, "in0_min", SENSORS_LM78J_IN0, SENSORS_LM78J_IN0,
                             SENSORS_MODE_RW, LM78_SYSCTL_IN0, VALUE(1), 2 },
    { SENSORS_LM78J_IN1_MIN, "in1_min", SENSORS_LM78J_IN1, SENSORS_LM78J_IN1,
                             SENSORS_MODE_RW, LM78_SYSCTL_IN1, VALUE(1), 2 },
    { SENSORS_LM78J_IN2_MIN, "in2_min", SENSORS_LM78J_IN2, SENSORS_LM78J_IN2,
                             SENSORS_MODE_RW, LM78_SYSCTL_IN2, VALUE(1), 2 },
    { SENSORS_LM78J_IN3_MIN, "in3_min", SENSORS_LM78J_IN3, SENSORS_LM78J_IN3,
                             SENSORS_MODE_RW, LM78_SYSCTL_IN3, VALUE(1), 2 },
    { SENSORS_LM78J_IN4_MIN, "in4_min", SENSORS_LM78J_IN4, SENSORS_LM78J_IN4,
                             SENSORS_MODE_RW, LM78_SYSCTL_IN4, VALUE(1), 2 },
/* swap 5 and 6 min/max because compute lines will be inverting */
    { SENSORS_LM78J_IN5_MIN, "in5_min", SENSORS_LM78J_IN5, SENSORS_LM78J_IN5,
                             SENSORS_MODE_RW, LM78_SYSCTL_IN5, VALUE(2), 2 },
    { SENSORS_LM78J_IN6_MIN, "in6_min", SENSORS_LM78J_IN6, SENSORS_LM78J_IN6,
                             SENSORS_MODE_RW, LM78_SYSCTL_IN6, VALUE(2), 2 },
    { SENSORS_LM78J_IN0_MAX, "in0_max", SENSORS_LM78J_IN0, SENSORS_LM78J_IN0,
                             SENSORS_MODE_RW, LM78_SYSCTL_IN0, VALUE(2), 2 },
    { SENSORS_LM78J_IN1_MAX, "in1_max", SENSORS_LM78J_IN1, SENSORS_LM78J_IN1,
                             SENSORS_MODE_RW, LM78_SYSCTL_IN1, VALUE(2), 2 },
    { SENSORS_LM78J_IN2_MAX, "in2_max", SENSORS_LM78J_IN2, SENSORS_LM78J_IN2,
                             SENSORS_MODE_RW, LM78_SYSCTL_IN2, VALUE(2), 2 },
    { SENSORS_LM78J_IN3_MAX, "in3_max", SENSORS_LM78J_IN3, SENSORS_LM78J_IN3,
                             SENSORS_MODE_RW, LM78_SYSCTL_IN3, VALUE(2), 2 },
    { SENSORS_LM78J_IN4_MAX, "in4_max", SENSORS_LM78J_IN4, SENSORS_LM78J_IN4,
                             SENSORS_MODE_RW, LM78_SYSCTL_IN4, VALUE(2), 2 },
    { SENSORS_LM78J_IN5_MAX, "in5_max", SENSORS_LM78J_IN5, SENSORS_LM78J_IN5,
                             SENSORS_MODE_RW, LM78_SYSCTL_IN5, VALUE(1), 2 },
    { SENSORS_LM78J_IN6_MAX, "in6_max", SENSORS_LM78J_IN6, SENSORS_LM78J_IN6,
                             SENSORS_MODE_RW, LM78_SYSCTL_IN6, VALUE(1), 2 },
    { SENSORS_LM78J_FAN1, "fan1", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                          SENSORS_MODE_R, LM78_SYSCTL_FAN1, VALUE(2), 0 },
    { SENSORS_LM78J_FAN2, "fan2", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                          SENSORS_MODE_R, LM78_SYSCTL_FAN2, VALUE(2), 0 },
    { SENSORS_LM78J_FAN3, "fan3", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                          SENSORS_MODE_R, LM78_SYSCTL_FAN3, VALUE(2), 0 },
    { SENSORS_LM78J_FAN1_MIN, "fan1_min", SENSORS_LM78J_FAN1,
                              SENSORS_LM78J_FAN1, SENSORS_MODE_RW, 
                              LM78_SYSCTL_FAN1, VALUE(1), 0 },
    { SENSORS_LM78J_FAN2_MIN, "fan2_min", SENSORS_LM78J_FAN2,
                              SENSORS_LM78J_FAN2, SENSORS_MODE_RW, 
                              LM78_SYSCTL_FAN2, VALUE(1), 0 },
    { SENSORS_LM78J_FAN3_MIN, "fan3_min", SENSORS_LM78J_FAN3,
                              SENSORS_LM78J_FAN3, SENSORS_MODE_RW, 
                              LM78_SYSCTL_FAN3, VALUE(1), 0 },
    { SENSORS_LM78J_TEMP, "temp", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                          SENSORS_MODE_R, LM78_SYSCTL_TEMP, VALUE(3), 1 },
    { SENSORS_LM78J_TEMP_HYST, "temp_hyst", SENSORS_LM78J_TEMP,
                               SENSORS_LM78J_TEMP, SENSORS_MODE_RW, 
                               LM78_SYSCTL_TEMP, VALUE(2), 1 },
    { SENSORS_LM78J_TEMP_OVER, "temp_over", SENSORS_LM78J_TEMP,
                               SENSORS_LM78J_TEMP, SENSORS_MODE_RW, 
                               LM78_SYSCTL_TEMP, VALUE(1), 1 },
    { SENSORS_LM78J_VID, "vid", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, LM78_SYSCTL_VID, VALUE(1), 2 },
    { SENSORS_LM78J_FAN1_DIV, "fan1_div", SENSORS_LM78J_FAN1,
                              SENSORS_NO_MAPPING, SENSORS_MODE_RW, 
                              LM78_SYSCTL_FAN_DIV, VALUE(1), 0 },
    { SENSORS_LM78J_FAN2_DIV, "fan2_div", SENSORS_LM78J_FAN2,
                              SENSORS_NO_MAPPING, SENSORS_MODE_RW,
                              LM78_SYSCTL_FAN_DIV, VALUE(2), 0 },
    { SENSORS_LM78J_FAN3_DIV, "fan3_div", SENSORS_LM78J_FAN3,
                              SENSORS_NO_MAPPING, SENSORS_MODE_R,
                              LM78_SYSCTL_FAN_DIV, VALUE(3), 0 },
    { SENSORS_LM78J_ALARMS, "alarms", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING, 
                            SENSORS_MODE_R, LM78_SYSCTL_ALARMS, VALUE(1), 0 },
    { 0 }
  };

static sensors_chip_feature lm79_features[] =
  { 
    { SENSORS_LM79_IN0, "in0", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING, 
                        SENSORS_MODE_R, LM78_SYSCTL_IN0, VALUE(3), 2 },
    { SENSORS_LM79_IN1, "in1", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, LM78_SYSCTL_IN1, VALUE(3), 2 },
    { SENSORS_LM79_IN2, "in2", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, LM78_SYSCTL_IN2, VALUE(3), 2 },
    { SENSORS_LM79_IN3, "in3", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, LM78_SYSCTL_IN3, VALUE(3), 2 },
    { SENSORS_LM79_IN4, "in4", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, LM78_SYSCTL_IN4, VALUE(3), 2 },
    { SENSORS_LM79_IN5, "in5", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, LM78_SYSCTL_IN5, VALUE(3), 2 },
    { SENSORS_LM79_IN6, "in6", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, LM78_SYSCTL_IN6, VALUE(3), 2 },
    { SENSORS_LM79_IN0_MIN, "in0_min", SENSORS_LM79_IN0, SENSORS_LM79_IN0,
                            SENSORS_MODE_RW, LM78_SYSCTL_IN0, VALUE(1), 2 },
    { SENSORS_LM79_IN1_MIN, "in1_min", SENSORS_LM79_IN1, SENSORS_LM79_IN1,
                            SENSORS_MODE_RW, LM78_SYSCTL_IN1, VALUE(1), 2 },
    { SENSORS_LM79_IN2_MIN, "in2_min", SENSORS_LM79_IN2, SENSORS_LM79_IN2,
                            SENSORS_MODE_RW, LM78_SYSCTL_IN2, VALUE(1), 2 },
    { SENSORS_LM79_IN3_MIN, "in3_min", SENSORS_LM79_IN3, SENSORS_LM79_IN3,
                            SENSORS_MODE_RW, LM78_SYSCTL_IN3, VALUE(1), 2 },
    { SENSORS_LM79_IN4_MIN, "in4_min", SENSORS_LM79_IN4, SENSORS_LM79_IN4,
                            SENSORS_MODE_RW, LM78_SYSCTL_IN4, VALUE(1), 2 },
/* swap 5 and 6 min/max because compute lines will be inverting */
    { SENSORS_LM79_IN5_MIN, "in5_min", SENSORS_LM79_IN5, SENSORS_LM79_IN5,
                            SENSORS_MODE_RW, LM78_SYSCTL_IN5, VALUE(2), 2 },
    { SENSORS_LM79_IN6_MIN, "in6_min", SENSORS_LM79_IN6, SENSORS_LM79_IN6,
                            SENSORS_MODE_RW, LM78_SYSCTL_IN6, VALUE(2), 2 },
    { SENSORS_LM79_IN0_MAX, "in0_max", SENSORS_LM79_IN0, SENSORS_LM79_IN0,
                            SENSORS_MODE_RW, LM78_SYSCTL_IN0, VALUE(2), 2 },
    { SENSORS_LM79_IN1_MAX, "in1_max", SENSORS_LM79_IN1, SENSORS_LM79_IN1,
                            SENSORS_MODE_RW, LM78_SYSCTL_IN1, VALUE(2), 2 },
    { SENSORS_LM79_IN2_MAX, "in2_max", SENSORS_LM79_IN2, SENSORS_LM79_IN2,
                            SENSORS_MODE_RW, LM78_SYSCTL_IN2, VALUE(2), 2 },
    { SENSORS_LM79_IN3_MAX, "in3_max", SENSORS_LM79_IN3, SENSORS_LM79_IN3,
                            SENSORS_MODE_RW, LM78_SYSCTL_IN3, VALUE(2), 2 },
    { SENSORS_LM79_IN4_MAX, "in4_max", SENSORS_LM79_IN4, SENSORS_LM79_IN4,
                            SENSORS_MODE_RW, LM78_SYSCTL_IN4, VALUE(2), 2 },
    { SENSORS_LM79_IN5_MAX, "in5_max", SENSORS_LM79_IN5, SENSORS_LM79_IN5,
                            SENSORS_MODE_RW, LM78_SYSCTL_IN5, VALUE(1), 2 },
    { SENSORS_LM79_IN6_MAX, "in6_max", SENSORS_LM79_IN6, SENSORS_LM79_IN6,
                            SENSORS_MODE_RW, LM78_SYSCTL_IN6, VALUE(1), 2 },
    { SENSORS_LM79_FAN1, "fan1", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, LM78_SYSCTL_FAN1, VALUE(2), 0 },
    { SENSORS_LM79_FAN2, "fan2", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, LM78_SYSCTL_FAN2, VALUE(2), 0 },
    { SENSORS_LM79_FAN3, "fan3", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, LM78_SYSCTL_FAN3, VALUE(2), 0 },
    { SENSORS_LM79_FAN1_MIN, "fan1_min", SENSORS_LM79_FAN1, SENSORS_LM79_FAN1,
                             SENSORS_MODE_RW, LM78_SYSCTL_FAN1, VALUE(1), 0 },
    { SENSORS_LM79_FAN2_MIN, "fan2_min", SENSORS_LM79_FAN2, SENSORS_LM79_FAN2,
                             SENSORS_MODE_RW, LM78_SYSCTL_FAN2, VALUE(1), 0 },
    { SENSORS_LM79_FAN3_MIN, "fan3_min", SENSORS_LM79_FAN3, SENSORS_LM79_FAN3,
                             SENSORS_MODE_RW, LM78_SYSCTL_FAN3, VALUE(1), 0 },
    { SENSORS_LM79_TEMP, "temp", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, LM78_SYSCTL_TEMP, VALUE(3), 1 },
    { SENSORS_LM79_TEMP_HYST, "temp_hyst", SENSORS_LM79_TEMP,
                              SENSORS_LM79_TEMP, SENSORS_MODE_RW, 
                              LM78_SYSCTL_TEMP, VALUE(2), 1 },
    { SENSORS_LM79_TEMP_OVER, "temp_over", SENSORS_LM79_TEMP,
                              SENSORS_LM79_TEMP, SENSORS_MODE_RW, 
                              LM78_SYSCTL_TEMP, VALUE(1), 1 },
    { SENSORS_LM79_VID, "vid", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, LM78_SYSCTL_VID, VALUE(1), 2 },
    { SENSORS_LM79_FAN1_DIV, "fan1_div", SENSORS_LM79_FAN1, SENSORS_NO_MAPPING,
                             SENSORS_MODE_RW, LM78_SYSCTL_FAN_DIV, VALUE(1), 
                             0 },
    { SENSORS_LM79_FAN2_DIV, "fan2_div", SENSORS_LM79_FAN2, SENSORS_NO_MAPPING,
                             SENSORS_MODE_RW, LM78_SYSCTL_FAN_DIV, VALUE(2), 
                             0 },
    { SENSORS_LM79_FAN3_DIV, "fan3_div", SENSORS_LM79_FAN3, SENSORS_NO_MAPPING,
                              SENSORS_MODE_R, LM78_SYSCTL_FAN_DIV, VALUE(3), 
                              0 },
    { SENSORS_LM79_ALARMS, "alarms", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING, 
                           SENSORS_MODE_R, LM78_SYSCTL_ALARMS, VALUE(1), 0 },
    { 0 }
  };


static sensors_chip_feature lm75_features[] =
  {
    { SENSORS_LM75_TEMP, "temp", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, LM75_SYSCTL_TEMP, VALUE(3), 1 },
    { SENSORS_LM75_TEMP_HYST, "temp_hyst", SENSORS_LM75_TEMP,
                              SENSORS_LM75_TEMP, SENSORS_MODE_RW, 
                              LM75_SYSCTL_TEMP, VALUE(2), 1 },
    { SENSORS_LM75_TEMP_OVER, "temp_over", SENSORS_LM75_TEMP,
                              SENSORS_LM75_TEMP, SENSORS_MODE_RW, 
                              LM75_SYSCTL_TEMP, VALUE(1), 1 },
    { 0 }
  };

static sensors_chip_feature adm1021_features[] =
  {
    { SENSORS_ADM1021_TEMP, "temp", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                              SENSORS_MODE_R, ADM1021_SYSCTL_TEMP, VALUE(3), 
                              0 },
    { SENSORS_ADM1021_TEMP_HYST, "temp_low", SENSORS_ADM1021_TEMP,
                              SENSORS_ADM1021_TEMP, SENSORS_MODE_RW, 
                              ADM1021_SYSCTL_TEMP, VALUE(2), 0 },
    { SENSORS_ADM1021_TEMP_OVER, "temp_over", SENSORS_ADM1021_TEMP,
                              SENSORS_ADM1021_TEMP, SENSORS_MODE_RW,
                              ADM1021_SYSCTL_TEMP, VALUE(1), 0 },
    { SENSORS_ADM1021_REMOTE_TEMP, "remote_temp", SENSORS_NO_MAPPING, 
                              SENSORS_NO_MAPPING, SENSORS_MODE_R, 
                              ADM1021_SYSCTL_REMOTE_TEMP, VALUE(3), 0 },
    { SENSORS_ADM1021_REMOTE_TEMP_HYST, "remote_temp_low", 
                              SENSORS_ADM1021_REMOTE_TEMP,
                              SENSORS_ADM1021_REMOTE_TEMP, SENSORS_MODE_RW, 
                              ADM1021_SYSCTL_REMOTE_TEMP, VALUE(2), 0 },
    { SENSORS_ADM1021_REMOTE_TEMP_OVER, "remote_temp_over", 
                              SENSORS_ADM1021_REMOTE_TEMP,
                              SENSORS_ADM1021_REMOTE_TEMP, SENSORS_MODE_RW,
                              ADM1021_SYSCTL_REMOTE_TEMP, VALUE(1), 0 },
    { SENSORS_ADM1021_ALARMS, "alarms", SENSORS_NO_MAPPING,
                              SENSORS_NO_MAPPING, SENSORS_MODE_R,
                              ADM1021_SYSCTL_ALARMS, VALUE(1), 0 },
    { SENSORS_ADM1021_DIE_CODE, "die_code", SENSORS_NO_MAPPING,
                              SENSORS_NO_MAPPING, SENSORS_MODE_R,
                              ADM1021_SYSCTL_DIE_CODE, VALUE(1), 0 },
    { 0 }
  };

static sensors_chip_feature adm1023_features[] =
  {
    { SENSORS_ADM1021_TEMP, "temp", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                              SENSORS_MODE_R, ADM1021_SYSCTL_TEMP, VALUE(3), 
                              0 },
    { SENSORS_ADM1021_TEMP_HYST, "temp_low", SENSORS_ADM1021_TEMP,
                              SENSORS_ADM1021_TEMP, SENSORS_MODE_RW, 
                              ADM1021_SYSCTL_TEMP, VALUE(2), 0 },
    { SENSORS_ADM1021_TEMP_OVER, "temp_over", SENSORS_ADM1021_TEMP,
                              SENSORS_ADM1021_TEMP, SENSORS_MODE_RW,
                              ADM1021_SYSCTL_TEMP, VALUE(1), 0 },
    { SENSORS_ADM1021_REMOTE_TEMP, "remote_temp", SENSORS_NO_MAPPING, 
                              SENSORS_NO_MAPPING, SENSORS_MODE_R, 
                              ADM1021_SYSCTL_REMOTE_TEMP, VALUE(4), 3 },
    { SENSORS_ADM1021_REMOTE_TEMP_HYST, "remote_temp_low", 
                              SENSORS_ADM1021_REMOTE_TEMP,
                              SENSORS_ADM1021_REMOTE_TEMP, SENSORS_MODE_RW, 
                              ADM1021_SYSCTL_REMOTE_TEMP, VALUE(2), 3 },
    { SENSORS_ADM1021_REMOTE_TEMP_OVER, "remote_temp_over", 
                              SENSORS_ADM1021_REMOTE_TEMP,
                              SENSORS_ADM1021_REMOTE_TEMP, SENSORS_MODE_RW,
                              ADM1021_SYSCTL_REMOTE_TEMP, VALUE(1), 3 },
    { SENSORS_ADM1021_ALARMS, "alarms", SENSORS_NO_MAPPING,
                              SENSORS_NO_MAPPING, SENSORS_MODE_R,
                              ADM1021_SYSCTL_ALARMS, VALUE(1), 0 },
    { 0 }
  };

static sensors_chip_feature max1617_features[] =
  {
    { SENSORS_MAX1617_TEMP, "temp", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                              SENSORS_MODE_R, ADM1021_SYSCTL_TEMP, VALUE(3), 
                              0 },
    { SENSORS_MAX1617_TEMP_HYST, "temp_low", SENSORS_MAX1617_TEMP,
                              SENSORS_MAX1617_TEMP, SENSORS_MODE_RW, 
                              ADM1021_SYSCTL_TEMP, VALUE(2), 0 },
    { SENSORS_MAX1617_TEMP_OVER, "temp_over", SENSORS_MAX1617_TEMP,
                              SENSORS_MAX1617_TEMP, SENSORS_MODE_RW,
                              ADM1021_SYSCTL_TEMP, VALUE(1), 0 },
    { SENSORS_MAX1617_REMOTE_TEMP, "remote_temp", SENSORS_NO_MAPPING, 
                              SENSORS_NO_MAPPING, SENSORS_MODE_R, 
                              ADM1021_SYSCTL_REMOTE_TEMP, VALUE(3), 0 },
    { SENSORS_MAX1617_REMOTE_TEMP_HYST, "remote_temp_low", 
                              SENSORS_MAX1617_REMOTE_TEMP,
                              SENSORS_MAX1617_REMOTE_TEMP, SENSORS_MODE_RW, 
                              ADM1021_SYSCTL_REMOTE_TEMP, VALUE(2), 0 },
    { SENSORS_MAX1617_REMOTE_TEMP_OVER, "remote_temp_over", 
                              SENSORS_MAX1617_REMOTE_TEMP,
                              SENSORS_MAX1617_REMOTE_TEMP, SENSORS_MODE_RW,
                              ADM1021_SYSCTL_REMOTE_TEMP, VALUE(1), 0 },
    { SENSORS_MAX1617_ALARMS, "alarms", SENSORS_NO_MAPPING,
                              SENSORS_NO_MAPPING, SENSORS_MODE_R,
                              ADM1021_SYSCTL_ALARMS, VALUE(1), 0 },
    { 0 }
  };

static sensors_chip_feature max1617a_features[] =
  {
    { SENSORS_MAX1617A_TEMP, "temp", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                              SENSORS_MODE_R, ADM1021_SYSCTL_TEMP, VALUE(3), 
                              0 },
    { SENSORS_MAX1617A_TEMP_HYST, "temp_low", SENSORS_MAX1617A_TEMP,
                              SENSORS_MAX1617A_TEMP, SENSORS_MODE_RW, 
                              ADM1021_SYSCTL_TEMP, VALUE(2), 0 },
    { SENSORS_MAX1617A_TEMP_OVER, "temp_over", SENSORS_MAX1617A_TEMP,
                              SENSORS_MAX1617A_TEMP, SENSORS_MODE_RW,
                              ADM1021_SYSCTL_TEMP, VALUE(1), 0 },
    { SENSORS_MAX1617A_REMOTE_TEMP, "remote_temp", SENSORS_NO_MAPPING, 
                              SENSORS_NO_MAPPING, SENSORS_MODE_R, 
                              ADM1021_SYSCTL_REMOTE_TEMP, VALUE(3), 0 },
    { SENSORS_MAX1617A_REMOTE_TEMP_HYST, "remote_temp_low", 
                              SENSORS_MAX1617A_REMOTE_TEMP,
                              SENSORS_MAX1617A_REMOTE_TEMP, SENSORS_MODE_RW, 
                              ADM1021_SYSCTL_REMOTE_TEMP, VALUE(2), 0 },
    { SENSORS_MAX1617A_REMOTE_TEMP_OVER, "remote_temp_over", 
                              SENSORS_MAX1617A_REMOTE_TEMP,
                              SENSORS_MAX1617A_REMOTE_TEMP, SENSORS_MODE_RW,
                              ADM1021_SYSCTL_REMOTE_TEMP, VALUE(1), 0 },
    { SENSORS_MAX1617A_ALARMS, "alarms", SENSORS_NO_MAPPING,
                              SENSORS_NO_MAPPING, SENSORS_MODE_R,
                              ADM1021_SYSCTL_ALARMS, VALUE(1), 0 },
    { 0 }
  };

static sensors_chip_feature gl518_features[] =
  {
    { SENSORS_GL518_VDD, "vdd", SENSORS_NO_MAPPING,SENSORS_NO_MAPPING,
                            SENSORS_MODE_R, GL518_SYSCTL_VDD, VALUE(3), 2 },
    { SENSORS_GL518_VIN1, "vin1", SENSORS_NO_MAPPING,SENSORS_NO_MAPPING,
                             SENSORS_MODE_R, GL518_SYSCTL_VIN1, VALUE(3), 
                             2 },
    { SENSORS_GL518_VIN2, "vin2", SENSORS_NO_MAPPING,SENSORS_NO_MAPPING,
                             SENSORS_MODE_R, GL518_SYSCTL_VIN2, VALUE(3), 
                             2 },
    { SENSORS_GL518_VIN3, "vin3", SENSORS_NO_MAPPING,SENSORS_NO_MAPPING,
                             SENSORS_MODE_R, GL518_SYSCTL_VIN3, VALUE(3), 2 },
    { SENSORS_GL518_VDD_MIN, "vdd_min", SENSORS_GL518_VDD,
                                SENSORS_GL518_VDD, SENSORS_MODE_RW,
                                GL518_SYSCTL_VDD, VALUE(1), 2 },
    { SENSORS_GL518_VIN1_MIN, "vin1_min", SENSORS_GL518_VIN1, 
                                 SENSORS_GL518_VIN1, SENSORS_MODE_RW,
                                 GL518_SYSCTL_VIN1, VALUE(1), 2 },
    { SENSORS_GL518_VIN2_MIN, "vin2_min", SENSORS_GL518_VIN2, 
                                 SENSORS_GL518_VIN2, SENSORS_MODE_RW,
                                 GL518_SYSCTL_VIN2, VALUE(1), 2 },
    { SENSORS_GL518_VIN3_MIN, "vin3_min", SENSORS_GL518_VIN3, 
                                 SENSORS_GL518_VIN3, SENSORS_MODE_RW,
                                 GL518_SYSCTL_VIN3, VALUE(1), 2 },
    { SENSORS_GL518_VDD_MAX, "vdd_max", SENSORS_GL518_VDD,
                                SENSORS_GL518_VDD, SENSORS_MODE_RW,
                                GL518_SYSCTL_VDD, VALUE(2), 2 },
    { SENSORS_GL518_VIN1_MAX, "vin1_max", SENSORS_GL518_VIN1, 
                                 SENSORS_GL518_VIN1, SENSORS_MODE_RW,
                                 GL518_SYSCTL_VIN1, VALUE(2), 2 },
    { SENSORS_GL518_VIN2_MAX, "vin2_max", SENSORS_GL518_VIN2, 
                                 SENSORS_GL518_VIN2, SENSORS_MODE_RW,
                                 GL518_SYSCTL_VIN2, VALUE(2), 2 },
    { SENSORS_GL518_VIN3_MAX, "vin3_max", SENSORS_GL518_VIN3, 
                                 SENSORS_GL518_VIN3, SENSORS_MODE_RW,
                                 GL518_SYSCTL_VIN3, VALUE(2), 2 },
    { SENSORS_GL518_FAN1, "fan1", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                             SENSORS_MODE_R, GL518_SYSCTL_FAN1, VALUE(2), 0 },
    { SENSORS_GL518_FAN2, "fan2", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                             SENSORS_MODE_R, GL518_SYSCTL_FAN2, VALUE(2), 0 },
    { SENSORS_GL518_FAN1_MIN, "fan1_min", SENSORS_GL518_FAN1, 
                                 SENSORS_GL518_FAN1, SENSORS_MODE_RW,
                                 GL518_SYSCTL_FAN1, VALUE(1), 0 },
    { SENSORS_GL518_FAN2_MIN, "fan2_min", SENSORS_GL518_FAN2, 
                                 SENSORS_GL518_FAN2, SENSORS_MODE_RW,
                                 GL518_SYSCTL_FAN2, VALUE(1), 0 },
    { SENSORS_GL518_TEMP, "temp", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                             SENSORS_MODE_R, GL518_SYSCTL_TEMP, VALUE(3), 1 },
    { SENSORS_GL518_TEMP_HYST, "temp_hyst", SENSORS_GL518_TEMP, 
                                  SENSORS_GL518_TEMP, SENSORS_MODE_RW,
                                  GL518_SYSCTL_TEMP, VALUE(2), 1 },
    { SENSORS_GL518_TEMP_OVER, "temp_over", SENSORS_GL518_TEMP, 
                                  SENSORS_GL518_TEMP, SENSORS_MODE_RW,
                                  GL518_SYSCTL_TEMP, VALUE(1), 1 },
    { SENSORS_GL518_FAN1_DIV, "fan1_div", SENSORS_GL518_FAN1, 
                                 SENSORS_NO_MAPPING, SENSORS_MODE_RW,
                                 GL518_SYSCTL_FAN_DIV, VALUE(1), 0 },
    { SENSORS_GL518_FAN2_DIV, "fan2_div", SENSORS_GL518_FAN2, 
                                 SENSORS_NO_MAPPING, SENSORS_MODE_RW,
                                 GL518_SYSCTL_FAN_DIV, VALUE(2), 0 },
    { SENSORS_GL518_ALARMS, "alarms", SENSORS_NO_MAPPING,
                               SENSORS_NO_MAPPING, SENSORS_MODE_R,
                               GL518_SYSCTL_ALARMS, VALUE(1), 0 },
    { SENSORS_GL518_BEEP_ENABLE, "beep_enable", SENSORS_GL518_ALARMS,
                               SENSORS_NO_MAPPING, SENSORS_MODE_RW,
                               GL518_SYSCTL_BEEP, VALUE(1), 0 },
    { SENSORS_GL518_BEEPS, "beeps", SENSORS_GL518_ALARMS,
                               SENSORS_GL518_ALARMS, SENSORS_MODE_RW,
                               GL518_SYSCTL_BEEP, VALUE(2), 0 },
    { SENSORS_GL518_FAN1OFF, "fan1_off", SENSORS_GL518_FAN1,
                               SENSORS_NO_MAPPING, SENSORS_MODE_RW,
                               GL518_SYSCTL_FAN1OFF, VALUE(1), 0 },
    { SENSORS_GL518_FAN1PIN, "fan1_off_pin", SENSORS_GL518_FAN1OFF,
                               SENSORS_NO_MAPPING, SENSORS_MODE_RW,
                               GL518_SYSCTL_FAN1OFF, VALUE(2), 0 },
    { SENSORS_GL518_ITERATE, "iterate", SENSORS_NO_MAPPING,
                               SENSORS_NO_MAPPING, SENSORS_MODE_RW,
                               GL518_SYSCTL_ITERATE, VALUE(1), 0 },
    { 0 }
  };

static sensors_chip_feature gl520_features[] =
  {
    { SENSORS_GL520_VDD, "vdd", SENSORS_NO_MAPPING,SENSORS_NO_MAPPING,
                            SENSORS_MODE_R, GL520_SYSCTL_VDD, VALUE(3), 2 },
    { SENSORS_GL520_VIN1, "vin1", SENSORS_NO_MAPPING,SENSORS_NO_MAPPING,
                             SENSORS_MODE_R, GL520_SYSCTL_VIN1, VALUE(3), 
                             2 },
    { SENSORS_GL520_VIN2, "vin2", SENSORS_NO_MAPPING,SENSORS_NO_MAPPING,
                             SENSORS_MODE_R, GL520_SYSCTL_VIN2, VALUE(3), 
                             2 },
    { SENSORS_GL520_VIN3, "vin3", SENSORS_NO_MAPPING,SENSORS_NO_MAPPING,
                             SENSORS_MODE_R, GL520_SYSCTL_VIN3, VALUE(3), 2 },
    { SENSORS_GL520_VIN4, "vin4", SENSORS_NO_MAPPING,SENSORS_NO_MAPPING,
                             SENSORS_MODE_R, GL520_SYSCTL_VIN4, VALUE(3), 2 },
    { SENSORS_GL520_VDD_MIN, "vdd_min", SENSORS_GL520_VDD,
                                SENSORS_GL520_VDD, SENSORS_MODE_RW,
                                GL520_SYSCTL_VDD, VALUE(1), 2 },
    { SENSORS_GL520_VIN1_MIN, "vin1_min", SENSORS_GL520_VIN1, 
                                 SENSORS_GL520_VIN1, SENSORS_MODE_RW,
                                 GL520_SYSCTL_VIN1, VALUE(1), 2 },
    { SENSORS_GL520_VIN2_MIN, "vin2_min", SENSORS_GL520_VIN2, 
                                 SENSORS_GL520_VIN2, SENSORS_MODE_RW,
                                 GL520_SYSCTL_VIN2, VALUE(1), 2 },
    { SENSORS_GL520_VIN3_MIN, "vin3_min", SENSORS_GL520_VIN3, 
                                 SENSORS_GL520_VIN3, SENSORS_MODE_RW,
                                 GL520_SYSCTL_VIN3, VALUE(1), 2 },
    { SENSORS_GL520_VIN4_MIN, "vin4_min", SENSORS_GL520_VIN4, 
                                 SENSORS_GL520_VIN4, SENSORS_MODE_RW,
                                 GL520_SYSCTL_VIN4, VALUE(1), 2 },
    { SENSORS_GL520_VDD_MAX, "vdd_max", SENSORS_GL520_VDD,
                                SENSORS_GL520_VDD, SENSORS_MODE_RW,
                                GL520_SYSCTL_VDD, VALUE(2), 2 },
    { SENSORS_GL520_VIN1_MAX, "vin1_max", SENSORS_GL520_VIN1, 
                                 SENSORS_GL520_VIN1, SENSORS_MODE_RW,
                                 GL520_SYSCTL_VIN1, VALUE(2), 2 },
    { SENSORS_GL520_VIN2_MAX, "vin2_max", SENSORS_GL520_VIN2, 
                                 SENSORS_GL520_VIN2, SENSORS_MODE_RW,
                                 GL520_SYSCTL_VIN2, VALUE(2), 2 },
    { SENSORS_GL520_VIN3_MAX, "vin3_max", SENSORS_GL520_VIN3, 
                                 SENSORS_GL520_VIN3, SENSORS_MODE_RW,
                                 GL520_SYSCTL_VIN3, VALUE(2), 2 },
    { SENSORS_GL520_VIN4_MAX, "vin4_max", SENSORS_GL520_VIN4, 
                                 SENSORS_GL520_VIN4, SENSORS_MODE_RW,
                                 GL520_SYSCTL_VIN4, VALUE(2), 2 },
    { SENSORS_GL520_FAN1, "fan1", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                             SENSORS_MODE_R, GL520_SYSCTL_FAN1, VALUE(2), 0 },
    { SENSORS_GL520_FAN2, "fan2", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                             SENSORS_MODE_R, GL520_SYSCTL_FAN2, VALUE(2), 0 },
    { SENSORS_GL520_FAN1_MIN, "fan1_min", SENSORS_GL520_FAN1, 
                                 SENSORS_GL520_FAN1, SENSORS_MODE_RW,
                                 GL520_SYSCTL_FAN1, VALUE(1), 0 },
    { SENSORS_GL520_FAN2_MIN, "fan2_min", SENSORS_GL520_FAN2, 
                                 SENSORS_GL520_FAN2, SENSORS_MODE_RW,
                                 GL520_SYSCTL_FAN2, VALUE(1), 0 },
    { SENSORS_GL520_TEMP1, "temp1", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                             SENSORS_MODE_R, GL520_SYSCTL_TEMP1, VALUE(3), 1 },
    { SENSORS_GL520_TEMP1_HYST, "temp1_hyst", SENSORS_GL520_TEMP1, 
                                  SENSORS_GL520_TEMP1, SENSORS_MODE_RW,
                                  GL520_SYSCTL_TEMP1, VALUE(2), 1 },
    { SENSORS_GL520_TEMP1_OVER, "temp1_over", SENSORS_GL520_TEMP1, 
                                  SENSORS_GL520_TEMP1, SENSORS_MODE_RW,
                                  GL520_SYSCTL_TEMP1, VALUE(1), 1 },
    { SENSORS_GL520_TEMP2, "temp2", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                             SENSORS_MODE_R, GL520_SYSCTL_TEMP2, VALUE(3), 1 },
    { SENSORS_GL520_TEMP2_HYST, "temp2_hyst", SENSORS_GL520_TEMP2, 
                                  SENSORS_GL520_TEMP2, SENSORS_MODE_RW,
                                  GL520_SYSCTL_TEMP2, VALUE(2), 1 },
    { SENSORS_GL520_TEMP2_OVER, "temp2_over", SENSORS_GL520_TEMP2, 
                                  SENSORS_GL520_TEMP2, SENSORS_MODE_RW,
                                  GL520_SYSCTL_TEMP2, VALUE(1), 1 },
    { SENSORS_GL520_VID, "vid", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                              SENSORS_MODE_R, GL520_SYSCTL_VID, VALUE(1), 2 },
    { SENSORS_GL520_FAN1_DIV, "fan1_div", SENSORS_GL520_FAN1, 
                                 SENSORS_NO_MAPPING, SENSORS_MODE_RW,
                                 GL520_SYSCTL_FAN_DIV, VALUE(1), 0 },
    { SENSORS_GL520_FAN2_DIV, "fan2_div", SENSORS_GL520_FAN2, 
                                 SENSORS_NO_MAPPING, SENSORS_MODE_RW,
                                 GL520_SYSCTL_FAN_DIV, VALUE(2), 0 },
    { SENSORS_GL520_ALARMS, "alarms", SENSORS_NO_MAPPING,
                               SENSORS_NO_MAPPING, SENSORS_MODE_R,
                               GL520_SYSCTL_ALARMS, VALUE(1), 0 },
    { SENSORS_GL520_BEEP_ENABLE, "beep_enable", SENSORS_GL520_ALARMS,
                               SENSORS_NO_MAPPING, SENSORS_MODE_RW,
                               GL520_SYSCTL_BEEP, VALUE(1), 0 },
    { SENSORS_GL520_BEEPS, "beeps", SENSORS_GL520_ALARMS,
                               SENSORS_GL520_ALARMS, SENSORS_MODE_RW,
                               GL520_SYSCTL_BEEP, VALUE(2), 0 },
    { SENSORS_GL520_FAN1OFF, "fan1_off", SENSORS_GL520_FAN1,
                               SENSORS_NO_MAPPING, SENSORS_MODE_RW,
                               GL520_SYSCTL_FAN1OFF, VALUE(1), 0 },
    { SENSORS_GL520_TWOTEMPS, "two_temps", SENSORS_NO_MAPPING,
                               SENSORS_NO_MAPPING, SENSORS_MODE_RW,
                               GL520_SYSCTL_CONFIG, VALUE(1), 0 },
    { 0 }
  };

static sensors_chip_feature lm80_features[] =
  { 
    { SENSORS_LM80_IN0, "in0", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING, 
                        SENSORS_MODE_R, LM80_SYSCTL_IN0, VALUE(3), 2 },
    { SENSORS_LM80_IN1, "in1", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, LM80_SYSCTL_IN1, VALUE(3), 2 },
    { SENSORS_LM80_IN2, "in2", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, LM80_SYSCTL_IN2, VALUE(3), 2 },
    { SENSORS_LM80_IN3, "in3", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, LM80_SYSCTL_IN3, VALUE(3), 2 },
    { SENSORS_LM80_IN4, "in4", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, LM80_SYSCTL_IN4, VALUE(3), 2 },
    { SENSORS_LM80_IN5, "in5", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, LM80_SYSCTL_IN5, VALUE(3), 2 },
    { SENSORS_LM80_IN6, "in6", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, LM80_SYSCTL_IN6, VALUE(3), 2 },
    { SENSORS_LM80_IN0_MIN, "in0_min", SENSORS_LM80_IN0, SENSORS_LM80_IN0,
                            SENSORS_MODE_RW, LM80_SYSCTL_IN0, VALUE(1), 2 },
    { SENSORS_LM80_IN1_MIN, "in1_min", SENSORS_LM80_IN1, SENSORS_LM80_IN1,
                            SENSORS_MODE_RW, LM80_SYSCTL_IN1, VALUE(1), 2 },
    { SENSORS_LM80_IN2_MIN, "in2_min", SENSORS_LM80_IN2, SENSORS_LM80_IN2,
                            SENSORS_MODE_RW, LM80_SYSCTL_IN2, VALUE(1), 2 },
    { SENSORS_LM80_IN3_MIN, "in3_min", SENSORS_LM80_IN3, SENSORS_LM80_IN3,
                            SENSORS_MODE_RW, LM80_SYSCTL_IN3, VALUE(1), 2 },
    { SENSORS_LM80_IN4_MIN, "in4_min", SENSORS_LM80_IN4, SENSORS_LM80_IN4,
                            SENSORS_MODE_RW, LM80_SYSCTL_IN4, VALUE(1), 2 },
    { SENSORS_LM80_IN5_MIN, "in5_min", SENSORS_LM80_IN5, SENSORS_LM80_IN5,
                            SENSORS_MODE_RW, LM80_SYSCTL_IN5, VALUE(1), 2 },
    { SENSORS_LM80_IN6_MIN, "in6_min", SENSORS_LM80_IN6, SENSORS_LM80_IN6,
                            SENSORS_MODE_RW, LM80_SYSCTL_IN6, VALUE(1), 2 },
    { SENSORS_LM80_IN0_MAX, "in0_max", SENSORS_LM80_IN0, SENSORS_LM80_IN0,
                            SENSORS_MODE_RW, LM80_SYSCTL_IN0, VALUE(2), 2 },
    { SENSORS_LM80_IN1_MAX, "in1_max", SENSORS_LM80_IN1, SENSORS_LM80_IN1,
                            SENSORS_MODE_RW, LM80_SYSCTL_IN1, VALUE(2), 2 },
    { SENSORS_LM80_IN2_MAX, "in2_max", SENSORS_LM80_IN2, SENSORS_LM80_IN2,
                            SENSORS_MODE_RW, LM80_SYSCTL_IN2, VALUE(2), 2 },
    { SENSORS_LM80_IN3_MAX, "in3_max", SENSORS_LM80_IN3, SENSORS_LM80_IN3,
                            SENSORS_MODE_RW, LM80_SYSCTL_IN3, VALUE(2), 2 },
    { SENSORS_LM80_IN4_MAX, "in4_max", SENSORS_LM80_IN4, SENSORS_LM80_IN4,
                            SENSORS_MODE_RW, LM80_SYSCTL_IN4, VALUE(2), 2 },
    { SENSORS_LM80_IN5_MAX, "in5_max", SENSORS_LM80_IN5, SENSORS_LM80_IN5,
                            SENSORS_MODE_RW, LM80_SYSCTL_IN5, VALUE(2), 2 },
    { SENSORS_LM80_IN6_MAX, "in6_max", SENSORS_LM80_IN6, SENSORS_LM80_IN6,
                            SENSORS_MODE_RW, LM80_SYSCTL_IN6, VALUE(2), 2 },
    { SENSORS_LM80_FAN1, "fan1", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, LM80_SYSCTL_FAN1, VALUE(2), 0 },
    { SENSORS_LM80_FAN2, "fan2", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, LM80_SYSCTL_FAN2, VALUE(2), 0 },
    { SENSORS_LM80_FAN1_MIN, "fan1_min", SENSORS_LM80_FAN1, SENSORS_NO_MAPPING,
                             SENSORS_MODE_RW, LM80_SYSCTL_FAN1, VALUE(1), 0 },
    { SENSORS_LM80_FAN2_MIN, "fan2_min", SENSORS_LM80_FAN2, SENSORS_NO_MAPPING,
                             SENSORS_MODE_RW, LM80_SYSCTL_FAN2, VALUE(1), 0 },
    { SENSORS_LM80_TEMP, "temp", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, LM80_SYSCTL_TEMP, VALUE(5), 2 },
    { SENSORS_LM80_TEMP_HOT_HYST, "temp_hot_hyst", SENSORS_LM80_TEMP,
                              SENSORS_LM80_TEMP, SENSORS_MODE_RW, 
                              LM80_SYSCTL_TEMP, VALUE(2), 2 },
    { SENSORS_LM80_TEMP_HOT_MAX, "temp_hot_max", SENSORS_LM80_TEMP,
                              SENSORS_LM80_TEMP, SENSORS_MODE_RW, 
                              LM80_SYSCTL_TEMP, VALUE(1), 2 },
    { SENSORS_LM80_TEMP_OS_HYST, "temp_os_hyst", SENSORS_LM80_TEMP,
                              SENSORS_LM80_TEMP, SENSORS_MODE_RW, 
                              LM80_SYSCTL_TEMP, VALUE(4), 2 },
    { SENSORS_LM80_TEMP_OS_MAX, "temp_os_max", SENSORS_LM80_TEMP,
                              SENSORS_LM80_TEMP, SENSORS_MODE_RW, 
                              LM80_SYSCTL_TEMP, VALUE(3), 2 },
    { SENSORS_LM80_FAN1_DIV, "fan1_div", SENSORS_LM80_FAN1, SENSORS_NO_MAPPING,
                             SENSORS_MODE_RW, LM80_SYSCTL_FAN_DIV, VALUE(1), 
                             0 },
    { SENSORS_LM80_FAN2_DIV, "fan2_div", SENSORS_LM80_FAN2, SENSORS_NO_MAPPING,
                             SENSORS_MODE_RW, LM80_SYSCTL_FAN_DIV, VALUE(2), 
                             0 },
    { SENSORS_LM80_ALARMS, "alarms", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING, 
                           SENSORS_MODE_R, LM80_SYSCTL_ALARMS, VALUE(1), 0 },
    { 0 }
  };


static sensors_chip_feature lm83_features[] =
  {
    { SENSORS_LM83_LOCAL_TEMP, "temp1", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, LM83_SYSCTL_LOCAL_TEMP, VALUE(2), 0 },
    { SENSORS_LM83_LOCAL_HIGH, "temp1_high", SENSORS_LM83_LOCAL_TEMP,
                              SENSORS_LM83_LOCAL_TEMP, SENSORS_MODE_RW, 
                              LM83_SYSCTL_LOCAL_TEMP, VALUE(1), 0 },
    { SENSORS_LM83_REMOTE1_TEMP, "temp2", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, LM83_SYSCTL_REMOTE1_TEMP, VALUE(2), 0 },
    { SENSORS_LM83_REMOTE1_HIGH, "temp2_high", SENSORS_LM83_REMOTE1_TEMP,
                              SENSORS_LM83_REMOTE1_TEMP, SENSORS_MODE_RW, 
                              LM83_SYSCTL_REMOTE1_TEMP, VALUE(1), 0 },
    { SENSORS_LM83_REMOTE2_TEMP, "temp3", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, LM83_SYSCTL_REMOTE2_TEMP, VALUE(2), 0 },
    { SENSORS_LM83_REMOTE2_HIGH, "temp3_high", SENSORS_LM83_REMOTE2_TEMP,
                              SENSORS_LM83_REMOTE2_TEMP, SENSORS_MODE_RW, 
                              LM83_SYSCTL_REMOTE2_TEMP, VALUE(1), 0 },
    { SENSORS_LM83_REMOTE3_TEMP, "temp4", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, LM83_SYSCTL_REMOTE3_TEMP, VALUE(2), 0 },
    { SENSORS_LM83_REMOTE3_HIGH, "temp4_high", SENSORS_LM83_REMOTE3_TEMP,
                              SENSORS_LM83_REMOTE3_TEMP, SENSORS_MODE_RW, 
                              LM83_SYSCTL_REMOTE3_TEMP, VALUE(1), 0 },
    { SENSORS_LM83_TCRIT, "tcrit", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_RW, LM83_SYSCTL_TCRIT, VALUE(1), 0 },
    { SENSORS_LM83_ALARMS, "alarms", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, LM83_SYSCTL_ALARMS, VALUE(1), 0 },
    { 0 }
  };


static sensors_chip_feature lm90_features[] =
  {
    { SENSORS_LM90_LOCAL_TEMP, "temp1",
      SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
      SENSORS_MODE_R, LM90_SYSCTL_LOCAL_TEMP, VALUE(3), 0 },
    { SENSORS_LM90_LOCAL_HIGH, "temp1_high",
      SENSORS_LM90_LOCAL_TEMP, SENSORS_LM90_LOCAL_TEMP,
      SENSORS_MODE_RW, LM90_SYSCTL_LOCAL_TEMP, VALUE(1), 0 },
    { SENSORS_LM90_LOCAL_LOW, "temp1_low",
      SENSORS_LM90_LOCAL_TEMP, SENSORS_LM90_LOCAL_TEMP,
      SENSORS_MODE_RW, LM90_SYSCTL_LOCAL_TEMP, VALUE(2), 0 },
    { SENSORS_LM90_REMOTE_TEMP, "temp2",
      SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
      SENSORS_MODE_R, LM90_SYSCTL_REMOTE_TEMP, VALUE(3), 1 },
    { SENSORS_LM90_REMOTE_HIGH, "temp2_high",
      SENSORS_LM90_REMOTE_TEMP, SENSORS_LM90_REMOTE_TEMP,
      SENSORS_MODE_RW, LM90_SYSCTL_REMOTE_TEMP, VALUE(1), 1 },
    { SENSORS_LM90_REMOTE_LOW, "temp2_low",
      SENSORS_LM90_REMOTE_TEMP, SENSORS_LM90_REMOTE_TEMP,
      SENSORS_MODE_RW, LM90_SYSCTL_REMOTE_TEMP, VALUE(2), 1 },
    { SENSORS_LM90_LOCAL_TCRIT, "tcrit1",
      SENSORS_LM90_LOCAL_TEMP, SENSORS_LM90_LOCAL_TEMP,
      SENSORS_MODE_RW, LM90_SYSCTL_LOCAL_TCRIT, VALUE(1), 0 },
    { SENSORS_LM90_REMOTE_TCRIT, "tcrit2",
      SENSORS_LM90_REMOTE_TEMP, SENSORS_LM90_REMOTE_TEMP,
      SENSORS_MODE_RW, LM90_SYSCTL_REMOTE_TCRIT, VALUE(1), 0 },
    { SENSORS_LM90_TCRIT_HYST, "hyst",
      SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
      SENSORS_MODE_RW, LM90_SYSCTL_HYST, VALUE(1), 0 },
    { SENSORS_LM90_ALARMS, "alarms",
      SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
      SENSORS_MODE_R, LM90_SYSCTL_ALARMS, VALUE(1), 0 },
    { 0 }
  };


static sensors_chip_feature w83781d_features[] =
  { 
    { SENSORS_W83781D_IN0, "in0", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING, 
                        SENSORS_MODE_R, W83781D_SYSCTL_IN0, VALUE(3), 2 },
    { SENSORS_W83781D_IN1, "in1", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, W83781D_SYSCTL_IN1, VALUE(3), 2 },
    { SENSORS_W83781D_IN2, "in2", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, W83781D_SYSCTL_IN2, VALUE(3), 2 },
    { SENSORS_W83781D_IN3, "in3", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, W83781D_SYSCTL_IN3, VALUE(3), 2 },
    { SENSORS_W83781D_IN4, "in4", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, W83781D_SYSCTL_IN4, VALUE(3), 2 },
    { SENSORS_W83781D_IN5, "in5", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, W83781D_SYSCTL_IN5, VALUE(3), 2 },
    { SENSORS_W83781D_IN6, "in6", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, W83781D_SYSCTL_IN6, VALUE(3), 2 },
    { SENSORS_W83781D_IN0_MIN, "in0_min", SENSORS_W83781D_IN0, 
                        SENSORS_W83781D_IN0, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_IN0, VALUE(1), 2 },
    { SENSORS_W83781D_IN1_MIN, "in1_min", SENSORS_W83781D_IN1, 
                        SENSORS_W83781D_IN1, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_IN1, VALUE(1), 2 },
    { SENSORS_W83781D_IN2_MIN, "in2_min", SENSORS_W83781D_IN2, 
                        SENSORS_W83781D_IN2, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_IN2, VALUE(1), 2 },
    { SENSORS_W83781D_IN3_MIN, "in3_min", SENSORS_W83781D_IN3, 
                        SENSORS_W83781D_IN3, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_IN3, VALUE(1), 2 },
    { SENSORS_W83781D_IN4_MIN, "in4_min", SENSORS_W83781D_IN4, 
                        SENSORS_W83781D_IN4, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_IN4, VALUE(1), 2 },
/* swap 5 and 6 min/max because compute lines will be inverting */
    { SENSORS_W83781D_IN5_MIN, "in5_min", SENSORS_W83781D_IN5, 
                        SENSORS_W83781D_IN5, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_IN5, VALUE(2), 2 },
    { SENSORS_W83781D_IN6_MIN, "in6_min", SENSORS_W83781D_IN6, 
                        SENSORS_W83781D_IN6, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_IN6, VALUE(2), 2 },
    { SENSORS_W83781D_IN0_MAX, "in0_max", SENSORS_W83781D_IN0, 
                        SENSORS_W83781D_IN0, SENSORS_MODE_RW,
                        W83781D_SYSCTL_IN0, VALUE(2), 2 },
    { SENSORS_W83781D_IN1_MAX, "in1_max", SENSORS_W83781D_IN1, 
                        SENSORS_W83781D_IN1, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_IN1, VALUE(2), 2 },
    { SENSORS_W83781D_IN2_MAX, "in2_max", SENSORS_W83781D_IN2, 
                        SENSORS_W83781D_IN2, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_IN2, VALUE(2), 2 },
    { SENSORS_W83781D_IN3_MAX, "in3_max", SENSORS_W83781D_IN3, 
                        SENSORS_W83781D_IN3, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_IN3, VALUE(2), 2 },
    { SENSORS_W83781D_IN4_MAX, "in4_max", SENSORS_W83781D_IN4, 
                        SENSORS_W83781D_IN4, SENSORS_MODE_RW,
                        W83781D_SYSCTL_IN4, VALUE(2), 2 },
    { SENSORS_W83781D_IN5_MAX, "in5_max", SENSORS_W83781D_IN5, 
                        SENSORS_W83781D_IN5, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_IN5, VALUE(1), 2 },
    { SENSORS_W83781D_IN6_MAX, "in6_max", SENSORS_W83781D_IN6, 
                        SENSORS_W83781D_IN6, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_IN6, VALUE(1), 2 },
    { SENSORS_W83781D_FAN1, "fan1", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, W83781D_SYSCTL_FAN1, VALUE(2), 0 },
    { SENSORS_W83781D_FAN2, "fan2", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, W83781D_SYSCTL_FAN2, VALUE(2), 0 },
    { SENSORS_W83781D_FAN3, "fan3", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, W83781D_SYSCTL_FAN3, VALUE(2), 0 },
    { SENSORS_W83781D_FAN1_MIN, "fan1_min", SENSORS_W83781D_FAN1,
                        SENSORS_W83781D_FAN1, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_FAN1, VALUE(1), 0 },
    { SENSORS_W83781D_FAN2_MIN, "fan2_min", SENSORS_W83781D_FAN2, 
                        SENSORS_W83781D_FAN2, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_FAN2, VALUE(1), 0 },
    { SENSORS_W83781D_FAN3_MIN, "fan3_min", SENSORS_W83781D_FAN3, 
                        SENSORS_W83781D_FAN3, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_FAN3, VALUE(1), 0 },
    { SENSORS_W83781D_TEMP1, "temp1", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, W83781D_SYSCTL_TEMP1, VALUE(3), 1 },
    { SENSORS_W83781D_TEMP1_HYST, "temp1_hyst", SENSORS_W83781D_TEMP1,
                         SENSORS_W83781D_TEMP1, SENSORS_MODE_RW, 
                         W83781D_SYSCTL_TEMP1, VALUE(2), 1 },
    { SENSORS_W83781D_TEMP1_OVER, "temp1_over", SENSORS_W83781D_TEMP1,
                         SENSORS_W83781D_TEMP1, SENSORS_MODE_RW, 
                         W83781D_SYSCTL_TEMP1, VALUE(1), 1 },
    { SENSORS_W83781D_TEMP2, "temp2", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, W83781D_SYSCTL_TEMP2, VALUE(3), 1 },
    { SENSORS_W83781D_TEMP2_HYST, "temp2_hyst", SENSORS_W83781D_TEMP2,
                         SENSORS_W83781D_TEMP2, SENSORS_MODE_RW, 
                         W83781D_SYSCTL_TEMP2, VALUE(2), 1 },
    { SENSORS_W83781D_TEMP2_OVER, "temp2_over", SENSORS_W83781D_TEMP2,
                         SENSORS_W83781D_TEMP2, SENSORS_MODE_RW, 
                         W83781D_SYSCTL_TEMP2, VALUE(1), 1 },
    { SENSORS_W83781D_TEMP3, "temp3", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, W83781D_SYSCTL_TEMP3, VALUE(3), 1 },
    { SENSORS_W83781D_TEMP3_HYST, "temp3_hyst", SENSORS_W83781D_TEMP3,
                         SENSORS_W83781D_TEMP3, SENSORS_MODE_RW, 
                         W83781D_SYSCTL_TEMP3, VALUE(2), 1 },
    { SENSORS_W83781D_TEMP3_OVER, "temp3_over", SENSORS_W83781D_TEMP3,
                         SENSORS_W83781D_TEMP3, SENSORS_MODE_RW, 
                         W83781D_SYSCTL_TEMP3, VALUE(1), 1 },
    { SENSORS_W83781D_VID, "vid", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, W83781D_SYSCTL_VID, VALUE(1), 3 },
    { SENSORS_W83781D_VRM, "vrm", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_RW, W83781D_SYSCTL_VRM, VALUE(1), 1 },
    { SENSORS_W83781D_FAN1_DIV, "fan1_div", SENSORS_W83781D_FAN1, 
                         SENSORS_NO_MAPPING, SENSORS_MODE_RW, 
                         W83781D_SYSCTL_FAN_DIV, VALUE(1), 0 },
    { SENSORS_W83781D_FAN2_DIV, "fan2_div", SENSORS_W83781D_FAN2, 
                         SENSORS_NO_MAPPING, SENSORS_MODE_RW, 
                         W83781D_SYSCTL_FAN_DIV, VALUE(2), 0 },
    { SENSORS_W83781D_FAN3_DIV, "fan3_div", SENSORS_W83781D_FAN3, 
                         SENSORS_NO_MAPPING, SENSORS_MODE_RW, 
                         W83781D_SYSCTL_FAN_DIV, VALUE(3), 0 },
    { SENSORS_W83781D_ALARMS, "alarms", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING, 
                         SENSORS_MODE_R, W83781D_SYSCTL_ALARMS, VALUE(1), 0 },
    { SENSORS_W83781D_BEEP_ENABLE, "beep_enable", SENSORS_W83781D_ALARMS,
                         SENSORS_NO_MAPPING, SENSORS_MODE_RW,
                         W83781D_SYSCTL_BEEP, VALUE(1), 0 },
    { SENSORS_W83781D_BEEPS, "beeps", SENSORS_W83781D_ALARMS,
                         SENSORS_W83781D_ALARMS, SENSORS_MODE_RW,
                         W83781D_SYSCTL_BEEP, VALUE(2), 0 },
    { 0 }
  };
  
static sensors_chip_feature as99127f_features[] =
  { 
    { SENSORS_W83782D_IN0, "in0", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING, 
                        SENSORS_MODE_R, W83781D_SYSCTL_IN0, VALUE(3), 2 },
    { SENSORS_W83782D_IN1, "in1", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, W83781D_SYSCTL_IN1, VALUE(3), 2 },
    { SENSORS_W83782D_IN2, "in2", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, W83781D_SYSCTL_IN2, VALUE(3), 2 },
    { SENSORS_W83782D_IN3, "in3", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, W83781D_SYSCTL_IN3, VALUE(3), 2 },
    { SENSORS_W83782D_IN4, "in4", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, W83781D_SYSCTL_IN4, VALUE(3), 2 },
    { SENSORS_W83782D_IN5, "in5", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, W83781D_SYSCTL_IN5, VALUE(3), 2 },
    { SENSORS_W83782D_IN6, "in6", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, W83781D_SYSCTL_IN6, VALUE(3), 2 },
    { SENSORS_W83782D_IN7, "in7", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, W83781D_SYSCTL_IN7, VALUE(3), 2 },
    { SENSORS_W83782D_IN8, "in8", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, W83781D_SYSCTL_IN8, VALUE(3), 2 },
    { SENSORS_W83782D_IN0_MIN, "in0_min", SENSORS_W83782D_IN0, 
                        SENSORS_W83782D_IN0, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_IN0, VALUE(1), 2 },
    { SENSORS_W83782D_IN1_MIN, "in1_min", SENSORS_W83782D_IN1, 
                        SENSORS_W83782D_IN1, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_IN1, VALUE(1), 2 },
    { SENSORS_W83782D_IN2_MIN, "in2_min", SENSORS_W83782D_IN2, 
                        SENSORS_W83782D_IN2, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_IN2, VALUE(1), 2 },
    { SENSORS_W83782D_IN3_MIN, "in3_min", SENSORS_W83782D_IN3, 
                        SENSORS_W83782D_IN3, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_IN3, VALUE(1), 2 },
    { SENSORS_W83782D_IN4_MIN, "in4_min", SENSORS_W83782D_IN4, 
                        SENSORS_W83782D_IN4, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_IN4, VALUE(1), 2 },
/* swap 5 and 6 min/max because compute lines will be inverting */
    { SENSORS_W83782D_IN5_MIN, "in5_min", SENSORS_W83782D_IN5, 
                        SENSORS_W83782D_IN5, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_IN5, VALUE(2), 2 },
    { SENSORS_W83782D_IN6_MIN, "in6_min", SENSORS_W83782D_IN6, 
                        SENSORS_W83782D_IN6, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_IN6, VALUE(2), 2 },
    { SENSORS_W83782D_IN7_MIN, "in7_min", SENSORS_W83782D_IN7, 
                        SENSORS_W83782D_IN7, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_IN7, VALUE(1), 2 },
    { SENSORS_W83782D_IN8_MIN, "in8_min", SENSORS_W83782D_IN8, 
                        SENSORS_W83782D_IN8, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_IN8, VALUE(1), 2 },
    { SENSORS_W83782D_IN0_MAX, "in0_max", SENSORS_W83782D_IN0, 
                        SENSORS_W83782D_IN0, SENSORS_MODE_RW,
                        W83781D_SYSCTL_IN0, VALUE(2), 2 },
    { SENSORS_W83782D_IN1_MAX, "in1_max", SENSORS_W83782D_IN1, 
                        SENSORS_W83782D_IN1, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_IN1, VALUE(2), 2 },
    { SENSORS_W83782D_IN2_MAX, "in2_max", SENSORS_W83782D_IN2, 
                        SENSORS_W83782D_IN2, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_IN2, VALUE(2), 2 },
    { SENSORS_W83782D_IN3_MAX, "in3_max", SENSORS_W83782D_IN3, 
                        SENSORS_W83782D_IN3, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_IN3, VALUE(2), 2 },
    { SENSORS_W83782D_IN4_MAX, "in4_max", SENSORS_W83782D_IN4, 
                        SENSORS_W83782D_IN4, SENSORS_MODE_RW,
                        W83781D_SYSCTL_IN4, VALUE(2), 2 },
    { SENSORS_W83782D_IN5_MAX, "in5_max", SENSORS_W83782D_IN5, 
                        SENSORS_W83782D_IN5, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_IN5, VALUE(1), 2 },
    { SENSORS_W83782D_IN6_MAX, "in6_max", SENSORS_W83782D_IN6, 
                        SENSORS_W83782D_IN6, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_IN6, VALUE(1), 2 },
    { SENSORS_W83782D_IN7_MAX, "in7_max", SENSORS_W83782D_IN7, 
                        SENSORS_W83782D_IN7, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_IN7, VALUE(2), 2 },
    { SENSORS_W83782D_IN8_MAX, "in8_max", SENSORS_W83782D_IN8, 
                        SENSORS_W83782D_IN8, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_IN8, VALUE(2), 2 },
    { SENSORS_W83782D_FAN1, "fan1", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, W83781D_SYSCTL_FAN1, VALUE(2), 0 },
    { SENSORS_W83782D_FAN2, "fan2", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, W83781D_SYSCTL_FAN2, VALUE(2), 0 },
    { SENSORS_W83781D_FAN3, "fan3", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, W83781D_SYSCTL_FAN3, VALUE(2), 0 },
    { SENSORS_W83782D_FAN1_MIN, "fan1_min", SENSORS_W83782D_FAN1,
                        SENSORS_W83782D_FAN1, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_FAN1, VALUE(1), 0 },
    { SENSORS_W83782D_FAN2_MIN, "fan2_min", SENSORS_W83782D_FAN2, 
                        SENSORS_W83782D_FAN2, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_FAN2, VALUE(1), 0 },
    { SENSORS_W83781D_FAN3_MIN, "fan3_min", SENSORS_W83781D_FAN3, 
                        SENSORS_W83782D_FAN3, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_FAN3, VALUE(1), 0 },
    { SENSORS_W83782D_TEMP1, "temp1", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, W83781D_SYSCTL_TEMP1, VALUE(3), 1 },
    { SENSORS_W83782D_TEMP1_HYST, "temp1_hyst", SENSORS_W83782D_TEMP1,
                         SENSORS_W83782D_TEMP1, SENSORS_MODE_RW, 
                         W83781D_SYSCTL_TEMP1, VALUE(2), 1 },
    { SENSORS_W83782D_TEMP1_OVER, "temp1_over", SENSORS_W83782D_TEMP1,
                         SENSORS_W83782D_TEMP1, SENSORS_MODE_RW, 
                         W83781D_SYSCTL_TEMP1, VALUE(1), 1 },
    { SENSORS_W83782D_TEMP2, "temp2", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, W83781D_SYSCTL_TEMP2, VALUE(3), 1 },
    { SENSORS_W83782D_TEMP2_HYST, "temp2_hyst", SENSORS_W83782D_TEMP2,
                         SENSORS_W83782D_TEMP2, SENSORS_MODE_RW, 
                         W83781D_SYSCTL_TEMP2, VALUE(2), 1 },
    { SENSORS_W83782D_TEMP2_OVER, "temp2_over", SENSORS_W83782D_TEMP2,
                         SENSORS_W83782D_TEMP2, SENSORS_MODE_RW, 
                         W83781D_SYSCTL_TEMP2, VALUE(1), 1 },
    { SENSORS_W83782D_TEMP3, "temp3", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, W83781D_SYSCTL_TEMP3, VALUE(3), 1 },
    { SENSORS_W83782D_TEMP3_HYST, "temp3_hyst", SENSORS_W83782D_TEMP3,
                         SENSORS_W83782D_TEMP3, SENSORS_MODE_RW, 
                         W83781D_SYSCTL_TEMP3, VALUE(2), 1 },
    { SENSORS_W83782D_TEMP3_OVER, "temp3_over", SENSORS_W83782D_TEMP3,
                         SENSORS_W83782D_TEMP3, SENSORS_MODE_RW, 
                         W83781D_SYSCTL_TEMP3, VALUE(1), 1 },
    { SENSORS_W83782D_VID, "vid", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, W83781D_SYSCTL_VID, VALUE(1), 3 },
    { SENSORS_W83782D_VRM, "vrm", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_RW, W83781D_SYSCTL_VRM, VALUE(1), 1 },
    { SENSORS_W83782D_FAN1_DIV, "fan1_div", SENSORS_W83782D_FAN1, 
                         SENSORS_NO_MAPPING, SENSORS_MODE_RW, 
                         W83781D_SYSCTL_FAN_DIV, VALUE(1), 0 },
    { SENSORS_W83782D_FAN2_DIV, "fan2_div", SENSORS_W83782D_FAN2, 
                         SENSORS_NO_MAPPING, SENSORS_MODE_RW, 
                         W83781D_SYSCTL_FAN_DIV, VALUE(2), 0 },
    { SENSORS_W83782D_FAN3_DIV, "fan3_div", SENSORS_W83781D_FAN3, 
                         SENSORS_NO_MAPPING, SENSORS_MODE_RW, 
                         W83781D_SYSCTL_FAN_DIV, VALUE(3), 0 },
    { SENSORS_W83782D_ALARMS, "alarms", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING, 
                         SENSORS_MODE_R, W83781D_SYSCTL_ALARMS, VALUE(1), 0 },
    { SENSORS_W83782D_BEEP_ENABLE, "beep_enable", SENSORS_W83782D_ALARMS,
                         SENSORS_NO_MAPPING, SENSORS_MODE_RW,
                         W83781D_SYSCTL_BEEP, VALUE(1), 0 },
    { SENSORS_W83782D_BEEPS, "beeps", SENSORS_W83782D_ALARMS,
                         SENSORS_W83782D_ALARMS, SENSORS_MODE_RW,
                         W83781D_SYSCTL_BEEP, VALUE(2), 0 },
    { SENSORS_W83782D_SENS1, "sensor1", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_RW, W83781D_SYSCTL_SENS1, VALUE(1), 0 },
    { SENSORS_W83782D_SENS2, "sensor2", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_RW, W83781D_SYSCTL_SENS2, VALUE(1), 0 },
    { SENSORS_W83782D_SENS3, "sensor3", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_RW, W83781D_SYSCTL_SENS3, VALUE(1), 0 },
    { 0 }
  };
  
static sensors_chip_feature w83782d_features[] =
  { 
    { SENSORS_W83782D_IN0, "in0", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING, 
                        SENSORS_MODE_R, W83781D_SYSCTL_IN0, VALUE(3), 2 },
    { SENSORS_W83782D_IN1, "in1", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, W83781D_SYSCTL_IN1, VALUE(3), 2 },
    { SENSORS_W83782D_IN2, "in2", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, W83781D_SYSCTL_IN2, VALUE(3), 2 },
    { SENSORS_W83782D_IN3, "in3", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, W83781D_SYSCTL_IN3, VALUE(3), 2 },
    { SENSORS_W83782D_IN4, "in4", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, W83781D_SYSCTL_IN4, VALUE(3), 2 },
    { SENSORS_W83782D_IN5, "in5", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, W83781D_SYSCTL_IN5, VALUE(3), 2 },
    { SENSORS_W83782D_IN6, "in6", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, W83781D_SYSCTL_IN6, VALUE(3), 2 },
    { SENSORS_W83782D_IN7, "in7", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, W83781D_SYSCTL_IN7, VALUE(3), 2 },
    { SENSORS_W83782D_IN8, "in8", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, W83781D_SYSCTL_IN8, VALUE(3), 2 },
    { SENSORS_W83782D_IN0_MIN, "in0_min", SENSORS_W83782D_IN0, 
                        SENSORS_W83782D_IN0, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_IN0, VALUE(1), 2 },
    { SENSORS_W83782D_IN1_MIN, "in1_min", SENSORS_W83782D_IN1, 
                        SENSORS_W83782D_IN1, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_IN1, VALUE(1), 2 },
    { SENSORS_W83782D_IN2_MIN, "in2_min", SENSORS_W83782D_IN2, 
                        SENSORS_W83782D_IN2, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_IN2, VALUE(1), 2 },
    { SENSORS_W83782D_IN3_MIN, "in3_min", SENSORS_W83782D_IN3, 
                        SENSORS_W83782D_IN3, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_IN3, VALUE(1), 2 },
    { SENSORS_W83782D_IN4_MIN, "in4_min", SENSORS_W83782D_IN4, 
                        SENSORS_W83782D_IN4, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_IN4, VALUE(1), 2 },
    { SENSORS_W83782D_IN5_MIN, "in5_min", SENSORS_W83782D_IN5, 
                        SENSORS_W83782D_IN5, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_IN5, VALUE(1), 2 },
    { SENSORS_W83782D_IN6_MIN, "in6_min", SENSORS_W83782D_IN6, 
                        SENSORS_W83782D_IN6, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_IN6, VALUE(1), 2 },
    { SENSORS_W83782D_IN7_MIN, "in7_min", SENSORS_W83782D_IN7, 
                        SENSORS_W83782D_IN7, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_IN7, VALUE(1), 2 },
    { SENSORS_W83782D_IN8_MIN, "in8_min", SENSORS_W83782D_IN8, 
                        SENSORS_W83782D_IN8, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_IN8, VALUE(1), 2 },
    { SENSORS_W83782D_IN0_MAX, "in0_max", SENSORS_W83782D_IN0, 
                        SENSORS_W83782D_IN0, SENSORS_MODE_RW,
                        W83781D_SYSCTL_IN0, VALUE(2), 2 },
    { SENSORS_W83782D_IN1_MAX, "in1_max", SENSORS_W83782D_IN1, 
                        SENSORS_W83782D_IN1, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_IN1, VALUE(2), 2 },
    { SENSORS_W83782D_IN2_MAX, "in2_max", SENSORS_W83782D_IN2, 
                        SENSORS_W83782D_IN2, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_IN2, VALUE(2), 2 },
    { SENSORS_W83782D_IN3_MAX, "in3_max", SENSORS_W83782D_IN3, 
                        SENSORS_W83782D_IN3, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_IN3, VALUE(2), 2 },
    { SENSORS_W83782D_IN4_MAX, "in4_max", SENSORS_W83782D_IN4, 
                        SENSORS_W83782D_IN4, SENSORS_MODE_RW,
                        W83781D_SYSCTL_IN4, VALUE(2), 2 },
    { SENSORS_W83782D_IN5_MAX, "in5_max", SENSORS_W83782D_IN5, 
                        SENSORS_W83782D_IN5, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_IN5, VALUE(2), 2 },
    { SENSORS_W83782D_IN6_MAX, "in6_max", SENSORS_W83782D_IN6, 
                        SENSORS_W83782D_IN6, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_IN6, VALUE(2), 2 },
    { SENSORS_W83782D_IN7_MAX, "in7_max", SENSORS_W83782D_IN7, 
                        SENSORS_W83782D_IN7, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_IN7, VALUE(2), 2 },
    { SENSORS_W83782D_IN8_MAX, "in8_max", SENSORS_W83782D_IN8, 
                        SENSORS_W83782D_IN8, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_IN8, VALUE(2), 2 },
    { SENSORS_W83782D_FAN1, "fan1", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, W83781D_SYSCTL_FAN1, VALUE(2), 0 },
    { SENSORS_W83782D_FAN2, "fan2", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, W83781D_SYSCTL_FAN2, VALUE(2), 0 },
    { SENSORS_W83781D_FAN3, "fan3", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, W83781D_SYSCTL_FAN3, VALUE(2), 0 },
    { SENSORS_W83782D_FAN1_MIN, "fan1_min", SENSORS_W83782D_FAN1,
                        SENSORS_W83782D_FAN1, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_FAN1, VALUE(1), 0 },
    { SENSORS_W83782D_FAN2_MIN, "fan2_min", SENSORS_W83782D_FAN2, 
                        SENSORS_W83782D_FAN2, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_FAN2, VALUE(1), 0 },
    { SENSORS_W83781D_FAN3_MIN, "fan3_min", SENSORS_W83781D_FAN3, 
                        SENSORS_W83782D_FAN3, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_FAN3, VALUE(1), 0 },
    { SENSORS_W83782D_TEMP1, "temp1", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, W83781D_SYSCTL_TEMP1, VALUE(3), 1 },
    { SENSORS_W83782D_TEMP1_HYST, "temp1_hyst", SENSORS_W83782D_TEMP1,
                         SENSORS_W83782D_TEMP1, SENSORS_MODE_RW, 
                         W83781D_SYSCTL_TEMP1, VALUE(2), 1 },
    { SENSORS_W83782D_TEMP1_OVER, "temp1_over", SENSORS_W83782D_TEMP1,
                         SENSORS_W83782D_TEMP1, SENSORS_MODE_RW, 
                         W83781D_SYSCTL_TEMP1, VALUE(1), 1 },
    { SENSORS_W83782D_TEMP2, "temp2", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, W83781D_SYSCTL_TEMP2, VALUE(3), 1 },
    { SENSORS_W83782D_TEMP2_HYST, "temp2_hyst", SENSORS_W83782D_TEMP2,
                         SENSORS_W83782D_TEMP2, SENSORS_MODE_RW, 
                         W83781D_SYSCTL_TEMP2, VALUE(2), 1 },
    { SENSORS_W83782D_TEMP2_OVER, "temp2_over", SENSORS_W83782D_TEMP2,
                         SENSORS_W83782D_TEMP2, SENSORS_MODE_RW, 
                         W83781D_SYSCTL_TEMP2, VALUE(1), 1 },
    { SENSORS_W83782D_TEMP3, "temp3", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, W83781D_SYSCTL_TEMP3, VALUE(3), 1 },
    { SENSORS_W83782D_TEMP3_HYST, "temp3_hyst", SENSORS_W83782D_TEMP3,
                         SENSORS_W83782D_TEMP3, SENSORS_MODE_RW, 
                         W83781D_SYSCTL_TEMP3, VALUE(2), 1 },
    { SENSORS_W83782D_TEMP3_OVER, "temp3_over", SENSORS_W83782D_TEMP3,
                         SENSORS_W83782D_TEMP3, SENSORS_MODE_RW, 
                         W83781D_SYSCTL_TEMP3, VALUE(1), 1 },
    { SENSORS_W83782D_VID, "vid", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, W83781D_SYSCTL_VID, VALUE(1), 3 },
    { SENSORS_W83782D_VRM, "vrm", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_RW, W83781D_SYSCTL_VRM, VALUE(1), 1 },
    { SENSORS_W83782D_FAN1_DIV, "fan1_div", SENSORS_W83782D_FAN1, 
                         SENSORS_NO_MAPPING, SENSORS_MODE_RW, 
                         W83781D_SYSCTL_FAN_DIV, VALUE(1), 0 },
    { SENSORS_W83782D_FAN2_DIV, "fan2_div", SENSORS_W83782D_FAN2, 
                         SENSORS_NO_MAPPING, SENSORS_MODE_RW, 
                         W83781D_SYSCTL_FAN_DIV, VALUE(2), 0 },
    { SENSORS_W83782D_FAN3_DIV, "fan3_div", SENSORS_W83781D_FAN3, 
                         SENSORS_NO_MAPPING, SENSORS_MODE_RW, 
                         W83781D_SYSCTL_FAN_DIV, VALUE(3), 0 },
    { SENSORS_W83782D_ALARMS, "alarms", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING, 
                         SENSORS_MODE_R, W83781D_SYSCTL_ALARMS, VALUE(1), 0 },
    { SENSORS_W83782D_BEEP_ENABLE, "beep_enable", SENSORS_W83782D_ALARMS,
                         SENSORS_NO_MAPPING, SENSORS_MODE_RW,
                         W83781D_SYSCTL_BEEP, VALUE(1), 0 },
    { SENSORS_W83782D_BEEPS, "beeps", SENSORS_W83782D_ALARMS,
                         SENSORS_W83782D_ALARMS, SENSORS_MODE_RW,
                         W83781D_SYSCTL_BEEP, VALUE(2), 0 },
    { SENSORS_W83782D_SENS1, "sensor1", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_RW, W83781D_SYSCTL_SENS1, VALUE(1), 0 },
    { SENSORS_W83782D_SENS2, "sensor2", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_RW, W83781D_SYSCTL_SENS2, VALUE(1), 0 },
    { SENSORS_W83782D_SENS3, "sensor3", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_RW, W83781D_SYSCTL_SENS3, VALUE(1), 0 },
    { 0 }
  };
  
static sensors_chip_feature w83783s_features[] =
  { 
    { SENSORS_W83783S_IN0, "in0", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING, 
                        SENSORS_MODE_R, W83781D_SYSCTL_IN0, VALUE(3), 2 },
    { SENSORS_W83783S_IN1, "in1", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, W83781D_SYSCTL_IN1, VALUE(3), 2 },
    { SENSORS_W83783S_IN2, "in2", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, W83781D_SYSCTL_IN2, VALUE(3), 2 },
    { SENSORS_W83783S_IN3, "in3", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, W83781D_SYSCTL_IN3, VALUE(3), 2 },
    { SENSORS_W83783S_IN4, "in4", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, W83781D_SYSCTL_IN4, VALUE(3), 2 },
    { SENSORS_W83783S_IN5, "in5", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, W83781D_SYSCTL_IN5, VALUE(3), 2 },
    { SENSORS_W83783S_IN6, "in6", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, W83781D_SYSCTL_IN6, VALUE(3), 2 },
    { SENSORS_W83783S_IN0_MIN, "in0_min", SENSORS_W83783S_IN0, 
                        SENSORS_W83783S_IN0, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_IN0, VALUE(1), 2 },
    { SENSORS_W83783S_IN1_MIN, "in1_min", SENSORS_W83783S_IN1, 
                        SENSORS_W83783S_IN1, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_IN1, VALUE(1), 2 },
    { SENSORS_W83783S_IN2_MIN, "in2_min", SENSORS_W83783S_IN2, 
                        SENSORS_W83783S_IN2, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_IN2, VALUE(1), 2 },
    { SENSORS_W83783S_IN3_MIN, "in3_min", SENSORS_W83783S_IN3, 
                        SENSORS_W83783S_IN3, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_IN3, VALUE(1), 2 },
    { SENSORS_W83783S_IN4_MIN, "in4_min", SENSORS_W83783S_IN4, 
                        SENSORS_W83783S_IN4, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_IN4, VALUE(1), 2 },
    { SENSORS_W83783S_IN5_MIN, "in5_min", SENSORS_W83783S_IN5, 
                        SENSORS_W83783S_IN5, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_IN5, VALUE(1), 2 },
    { SENSORS_W83783S_IN6_MIN, "in6_min", SENSORS_W83783S_IN6, 
                        SENSORS_W83783S_IN6, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_IN6, VALUE(1), 2 },
    { SENSORS_W83783S_IN0_MAX, "in0_max", SENSORS_W83783S_IN0, 
                        SENSORS_W83783S_IN0, SENSORS_MODE_RW,
                        W83781D_SYSCTL_IN0, VALUE(2), 2 },
    { SENSORS_W83783S_IN1_MAX, "in1_max", SENSORS_W83783S_IN1, 
                        SENSORS_W83783S_IN1, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_IN1, VALUE(2), 2 },
    { SENSORS_W83783S_IN2_MAX, "in2_max", SENSORS_W83783S_IN2, 
                        SENSORS_W83783S_IN2, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_IN2, VALUE(2), 2 },
    { SENSORS_W83783S_IN3_MAX, "in3_max", SENSORS_W83783S_IN3, 
                        SENSORS_W83783S_IN3, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_IN3, VALUE(2), 2 },
    { SENSORS_W83783S_IN4_MAX, "in4_max", SENSORS_W83783S_IN4, 
                        SENSORS_W83783S_IN4, SENSORS_MODE_RW,
                        W83781D_SYSCTL_IN4, VALUE(2), 2 },
    { SENSORS_W83783S_IN5_MAX, "in5_max", SENSORS_W83783S_IN5, 
                        SENSORS_W83783S_IN5, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_IN5, VALUE(2), 2 },
    { SENSORS_W83783S_IN6_MAX, "in6_max", SENSORS_W83783S_IN6, 
                        SENSORS_W83783S_IN6, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_IN6, VALUE(2), 2 },
    { SENSORS_W83783S_FAN1, "fan1", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, W83781D_SYSCTL_FAN1, VALUE(2), 0 },
    { SENSORS_W83783S_FAN2, "fan2", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, W83781D_SYSCTL_FAN2, VALUE(2), 0 },
    { SENSORS_W83783S_FAN3, "fan3", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, W83781D_SYSCTL_FAN3, VALUE(2), 0 },
    { SENSORS_W83783S_FAN1_MIN, "fan1_min", SENSORS_W83783S_FAN1,
                        SENSORS_W83783S_FAN1, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_FAN1, VALUE(1), 0 },
    { SENSORS_W83783S_FAN2_MIN, "fan2_min", SENSORS_W83783S_FAN2, 
                        SENSORS_W83783S_FAN2, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_FAN2, VALUE(1), 0 },
    { SENSORS_W83783S_FAN3_MIN, "fan3_min", SENSORS_W83783S_FAN3, 
                        SENSORS_W83783S_FAN3, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_FAN3, VALUE(1), 0 },
    { SENSORS_W83783S_TEMP1, "temp1", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, W83781D_SYSCTL_TEMP1, VALUE(3), 1 },
    { SENSORS_W83783S_TEMP1_HYST, "temp1_hyst", SENSORS_W83783S_TEMP1,
                         SENSORS_W83783S_TEMP1, SENSORS_MODE_RW, 
                         W83781D_SYSCTL_TEMP1, VALUE(2), 1 },
    { SENSORS_W83783S_TEMP1_OVER, "temp1_over", SENSORS_W83783S_TEMP1,
                         SENSORS_W83783S_TEMP1, SENSORS_MODE_RW, 
                         W83781D_SYSCTL_TEMP1, VALUE(1), 1 },
    { SENSORS_W83783S_TEMP2, "temp2", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, W83781D_SYSCTL_TEMP2, VALUE(3), 1 },
    { SENSORS_W83783S_TEMP2_HYST, "temp2_hyst", SENSORS_W83783S_TEMP2,
                         SENSORS_W83783S_TEMP2, SENSORS_MODE_RW, 
                         W83781D_SYSCTL_TEMP2, VALUE(2), 1 },
    { SENSORS_W83783S_TEMP2_OVER, "temp2_over", SENSORS_W83783S_TEMP2,
                         SENSORS_W83783S_TEMP2, SENSORS_MODE_RW, 
                         W83781D_SYSCTL_TEMP2, VALUE(1), 1 },
    { SENSORS_W83783S_VID, "vid", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, W83781D_SYSCTL_VID, VALUE(1), 3 },
    { SENSORS_W83783S_VRM, "vrm", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_RW, W83781D_SYSCTL_VRM, VALUE(1), 1 },
    { SENSORS_W83783S_FAN1_DIV, "fan1_div", SENSORS_W83783S_FAN1, 
                         SENSORS_NO_MAPPING, SENSORS_MODE_RW, 
                         W83781D_SYSCTL_FAN_DIV, VALUE(1), 0 },
    { SENSORS_W83783S_FAN2_DIV, "fan2_div", SENSORS_W83783S_FAN2, 
                         SENSORS_NO_MAPPING, SENSORS_MODE_RW, 
                         W83781D_SYSCTL_FAN_DIV, VALUE(2), 0 },
    { SENSORS_W83783S_FAN3_DIV, "fan3_div", SENSORS_W83783S_FAN3, 
                         SENSORS_NO_MAPPING, SENSORS_MODE_RW, 
                         W83781D_SYSCTL_FAN_DIV, VALUE(3), 0 },
    { SENSORS_W83783S_ALARMS, "alarms", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING, 
                         SENSORS_MODE_R, W83781D_SYSCTL_ALARMS, VALUE(1), 0 },
    { SENSORS_W83783S_BEEP_ENABLE, "beep_enable", SENSORS_W83783S_ALARMS,
                         SENSORS_NO_MAPPING, SENSORS_MODE_RW,
                         W83781D_SYSCTL_BEEP, VALUE(1), 0 },
    { SENSORS_W83783S_BEEPS, "beeps", SENSORS_W83783S_ALARMS,
                         SENSORS_W83783S_ALARMS, SENSORS_MODE_RW,
                         W83781D_SYSCTL_BEEP, VALUE(2), 0 },
    { SENSORS_W83783S_SENS1, "sensor1", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_RW, W83781D_SYSCTL_SENS1, VALUE(1), 0 },
    { SENSORS_W83783S_SENS2, "sensor2", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_RW, W83781D_SYSCTL_SENS2, VALUE(1), 0 },
    { 0 }
  };
  
static sensors_chip_feature w83791d_features[] =
  { 
    { SENSORS_W83791D_IN0, "in0", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING, 
                        SENSORS_MODE_R, W83781D_SYSCTL_IN0, VALUE(3), 2 },
    { SENSORS_W83791D_IN1, "in1", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, W83781D_SYSCTL_IN1, VALUE(3), 2 },
    { SENSORS_W83791D_IN2, "in2", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, W83781D_SYSCTL_IN2, VALUE(3), 2 },
    { SENSORS_W83791D_IN3, "in3", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, W83781D_SYSCTL_IN3, VALUE(3), 2 },
    { SENSORS_W83791D_IN4, "in4", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, W83781D_SYSCTL_IN4, VALUE(3), 2 },
    { SENSORS_W83791D_IN5, "in5", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, W83781D_SYSCTL_IN5, VALUE(3), 2 },
    { SENSORS_W83791D_IN6, "in6", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, W83781D_SYSCTL_IN6, VALUE(3), 2 },
    { SENSORS_W83791D_IN7, "in7", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, W83781D_SYSCTL_IN7, VALUE(3), 2 },
    { SENSORS_W83791D_IN8, "in8", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, W83781D_SYSCTL_IN8, VALUE(3), 2 },
    { SENSORS_W83791D_IN9, "in9", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, W83781D_SYSCTL_IN9, VALUE(3), 2 },
    { SENSORS_W83791D_IN0_MIN, "in0_min", SENSORS_W83791D_IN0, 
                        SENSORS_W83791D_IN0, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_IN0, VALUE(1), 2 },
    { SENSORS_W83791D_IN1_MIN, "in1_min", SENSORS_W83791D_IN1, 
                        SENSORS_W83791D_IN1, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_IN1, VALUE(1), 2 },
    { SENSORS_W83791D_IN2_MIN, "in2_min", SENSORS_W83791D_IN2, 
                        SENSORS_W83791D_IN2, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_IN2, VALUE(1), 2 },
    { SENSORS_W83791D_IN3_MIN, "in3_min", SENSORS_W83791D_IN3, 
                        SENSORS_W83791D_IN3, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_IN3, VALUE(1), 2 },
    { SENSORS_W83791D_IN4_MIN, "in4_min", SENSORS_W83791D_IN4, 
                        SENSORS_W83791D_IN4, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_IN4, VALUE(1), 2 },
    { SENSORS_W83791D_IN5_MIN, "in5_min", SENSORS_W83791D_IN5, 
                        SENSORS_W83791D_IN5, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_IN5, VALUE(1), 2 },
    { SENSORS_W83791D_IN6_MIN, "in6_min", SENSORS_W83791D_IN6, 
                        SENSORS_W83791D_IN6, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_IN6, VALUE(1), 2 },
    { SENSORS_W83791D_IN7_MIN, "in7_min", SENSORS_W83791D_IN7, 
                        SENSORS_W83791D_IN7, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_IN7, VALUE(1), 2 },
    { SENSORS_W83791D_IN8_MIN, "in8_min", SENSORS_W83791D_IN8, 
                        SENSORS_W83791D_IN8, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_IN8, VALUE(1), 2 },
    { SENSORS_W83791D_IN9_MIN, "in9_min", SENSORS_W83791D_IN9, 
                        SENSORS_W83791D_IN9, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_IN9, VALUE(1), 2 },
    { SENSORS_W83791D_IN0_MAX, "in0_max", SENSORS_W83791D_IN0, 
                        SENSORS_W83791D_IN0, SENSORS_MODE_RW,
                        W83781D_SYSCTL_IN0, VALUE(2), 2 },
    { SENSORS_W83791D_IN1_MAX, "in1_max", SENSORS_W83791D_IN1, 
                        SENSORS_W83791D_IN1, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_IN1, VALUE(2), 2 },
    { SENSORS_W83791D_IN2_MAX, "in2_max", SENSORS_W83791D_IN2, 
                        SENSORS_W83791D_IN2, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_IN2, VALUE(2), 2 },
    { SENSORS_W83791D_IN3_MAX, "in3_max", SENSORS_W83791D_IN3, 
                        SENSORS_W83791D_IN3, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_IN3, VALUE(2), 2 },
    { SENSORS_W83791D_IN4_MAX, "in4_max", SENSORS_W83791D_IN4, 
                        SENSORS_W83791D_IN4, SENSORS_MODE_RW,
                        W83781D_SYSCTL_IN4, VALUE(2), 2 },
    { SENSORS_W83791D_IN5_MAX, "in5_max", SENSORS_W83791D_IN5, 
                        SENSORS_W83791D_IN5, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_IN5, VALUE(2), 2 },
    { SENSORS_W83791D_IN6_MAX, "in6_max", SENSORS_W83791D_IN6, 
                        SENSORS_W83791D_IN6, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_IN6, VALUE(2), 2 },
    { SENSORS_W83791D_IN7_MAX, "in7_max", SENSORS_W83791D_IN7, 
                        SENSORS_W83791D_IN7, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_IN7, VALUE(2), 2 },
    { SENSORS_W83791D_IN8_MAX, "in8_max", SENSORS_W83791D_IN8, 
                        SENSORS_W83791D_IN8, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_IN8, VALUE(2), 2 },
    { SENSORS_W83791D_IN9_MAX, "in9_max", SENSORS_W83791D_IN9, 
                        SENSORS_W83791D_IN9, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_IN9, VALUE(2), 2 },
    { SENSORS_W83791D_FAN1, "fan1", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, W83781D_SYSCTL_FAN1, VALUE(2), 0 },
    { SENSORS_W83791D_FAN2, "fan2", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, W83781D_SYSCTL_FAN2, VALUE(2), 0 },
    { SENSORS_W83791D_FAN3, "fan3", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, W83781D_SYSCTL_FAN3, VALUE(2), 0 },
    { SENSORS_W83791D_FAN4, "fan4", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, W83781D_SYSCTL_FAN4, VALUE(2), 0 },
    { SENSORS_W83791D_FAN5, "fan5", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, W83781D_SYSCTL_FAN5, VALUE(2), 0 },
    { SENSORS_W83791D_FAN1_MIN, "fan1_min", SENSORS_W83791D_FAN1,
                        SENSORS_W83791D_FAN1, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_FAN1, VALUE(1), 0 },
    { SENSORS_W83791D_FAN2_MIN, "fan2_min", SENSORS_W83791D_FAN2, 
                        SENSORS_W83791D_FAN2, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_FAN2, VALUE(1), 0 },
    { SENSORS_W83791D_FAN3_MIN, "fan3_min", SENSORS_W83791D_FAN3, 
                        SENSORS_W83791D_FAN3, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_FAN3, VALUE(1), 0 },
    { SENSORS_W83791D_FAN4_MIN, "fan4_min", SENSORS_W83791D_FAN4, 
                        SENSORS_W83791D_FAN4, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_FAN4, VALUE(1), 0 },
    { SENSORS_W83791D_FAN5_MIN, "fan5_min", SENSORS_W83791D_FAN5, 
                        SENSORS_W83791D_FAN5, SENSORS_MODE_RW, 
                        W83781D_SYSCTL_FAN5, VALUE(1), 0 },
    { SENSORS_W83791D_TEMP1, "temp1", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, W83781D_SYSCTL_TEMP1, VALUE(3), 1 },
    { SENSORS_W83791D_TEMP1_HYST, "temp1_hyst", SENSORS_W83791D_TEMP1,
                         SENSORS_W83791D_TEMP1, SENSORS_MODE_RW, 
                         W83781D_SYSCTL_TEMP1, VALUE(2), 1 },
    { SENSORS_W83791D_TEMP1_OVER, "temp1_over", SENSORS_W83791D_TEMP1,
                         SENSORS_W83791D_TEMP1, SENSORS_MODE_RW, 
                         W83781D_SYSCTL_TEMP1, VALUE(1), 1 },
    { SENSORS_W83791D_TEMP2, "temp2", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, W83781D_SYSCTL_TEMP2, VALUE(3), 1 },
    { SENSORS_W83791D_TEMP2_HYST, "temp2_hyst", SENSORS_W83791D_TEMP2,
                         SENSORS_W83791D_TEMP2, SENSORS_MODE_RW, 
                         W83781D_SYSCTL_TEMP2, VALUE(2), 1 },
    { SENSORS_W83791D_TEMP2_OVER, "temp2_over", SENSORS_W83791D_TEMP2,
                         SENSORS_W83791D_TEMP2, SENSORS_MODE_RW, 
                         W83781D_SYSCTL_TEMP2, VALUE(1), 1 },
    { SENSORS_W83791D_TEMP3, "temp3", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, W83781D_SYSCTL_TEMP3, VALUE(3), 1 },
    { SENSORS_W83791D_TEMP3_HYST, "temp3_hyst", SENSORS_W83791D_TEMP3,
                         SENSORS_W83791D_TEMP3, SENSORS_MODE_RW, 
                         W83781D_SYSCTL_TEMP3, VALUE(2), 1 },
    { SENSORS_W83791D_TEMP3_OVER, "temp3_over", SENSORS_W83791D_TEMP3,
                         SENSORS_W83791D_TEMP3, SENSORS_MODE_RW, 
                         W83781D_SYSCTL_TEMP3, VALUE(1), 1 },
    { SENSORS_W83791D_VID, "vid", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, W83781D_SYSCTL_VID, VALUE(1), 3 },
    { SENSORS_W83791D_VRM, "vrm", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_RW, W83781D_SYSCTL_VRM, VALUE(1), 1 },
    { SENSORS_W83791D_FAN1_DIV, "fan1_div", SENSORS_W83791D_FAN1, 
                         SENSORS_NO_MAPPING, SENSORS_MODE_RW, 
                         W83781D_SYSCTL_FAN_DIV, VALUE(1), 0 },
    { SENSORS_W83791D_FAN2_DIV, "fan2_div", SENSORS_W83791D_FAN2, 
                         SENSORS_NO_MAPPING, SENSORS_MODE_RW, 
                         W83781D_SYSCTL_FAN_DIV, VALUE(2), 0 },
    { SENSORS_W83791D_FAN3_DIV, "fan3_div", SENSORS_W83781D_FAN3, 
                         SENSORS_NO_MAPPING, SENSORS_MODE_RW, 
                         W83781D_SYSCTL_FAN_DIV, VALUE(3), 0 },
    { SENSORS_W83791D_FAN4_DIV, "fan4_div", SENSORS_W83791D_FAN4, 
                         SENSORS_NO_MAPPING, SENSORS_MODE_RW, 
                         W83781D_SYSCTL_FAN_DIV, VALUE(4), 0 },
    { SENSORS_W83791D_FAN5_DIV, "fan5_div", SENSORS_W83791D_FAN5, 
                         SENSORS_NO_MAPPING, SENSORS_MODE_RW, 
                         W83781D_SYSCTL_FAN_DIV, VALUE(5), 0 },
    { SENSORS_W83791D_ALARMS, "alarms", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING, 
                         SENSORS_MODE_R, W83781D_SYSCTL_ALARMS, VALUE(1), 0 },
    { SENSORS_W83791D_BEEP_ENABLE, "beep_enable", SENSORS_W83791D_ALARMS,
                         SENSORS_NO_MAPPING, SENSORS_MODE_RW,
                         W83781D_SYSCTL_BEEP, VALUE(1), 0 },
    { SENSORS_W83791D_BEEPS, "beeps", SENSORS_W83791D_ALARMS,
                         SENSORS_W83791D_ALARMS, SENSORS_MODE_RW,
                         W83781D_SYSCTL_BEEP, VALUE(2), 0 },
    { 0 }
  };

static sensors_chip_feature adm9240_features[] =
  { 
    { SENSORS_ADM9240_IN0, "2.5V", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING, 
                        SENSORS_MODE_R, ADM9240_SYSCTL_IN0, VALUE(3), 2 },
    { SENSORS_ADM9240_IN1, "Vccp1", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, ADM9240_SYSCTL_IN1, VALUE(3), 2 },
    { SENSORS_ADM9240_IN2, "3.3V", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, ADM9240_SYSCTL_IN2, VALUE(3), 2 },
    { SENSORS_ADM9240_IN3, "5V", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, ADM9240_SYSCTL_IN3, VALUE(3), 2 },
    { SENSORS_ADM9240_IN4, "12V", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, ADM9240_SYSCTL_IN4, VALUE(3), 2 },
    { SENSORS_ADM9240_IN5, "Vccp2", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, ADM9240_SYSCTL_IN5, VALUE(3), 2 },
    { SENSORS_ADM9240_IN0_MIN, "2.5V_min", SENSORS_ADM9240_IN0, 
                        SENSORS_ADM9240_IN0, SENSORS_MODE_RW, 
                        ADM9240_SYSCTL_IN0, VALUE(1), 2 },
    { SENSORS_ADM9240_IN1_MIN, "Vccp1_min", SENSORS_ADM9240_IN1, 
                        SENSORS_ADM9240_IN1, SENSORS_MODE_RW, 
                        ADM9240_SYSCTL_IN1, VALUE(1), 2 },
    { SENSORS_ADM9240_IN2_MIN, "3.3V_min", SENSORS_ADM9240_IN2, 
                        SENSORS_ADM9240_IN2, SENSORS_MODE_RW, 
                        ADM9240_SYSCTL_IN2, VALUE(1), 2 },
    { SENSORS_ADM9240_IN3_MIN, "5V_min", SENSORS_ADM9240_IN3, 
                        SENSORS_ADM9240_IN3, SENSORS_MODE_RW, 
                        ADM9240_SYSCTL_IN3, VALUE(1), 2 },
    { SENSORS_ADM9240_IN4_MIN, "12V_min", SENSORS_ADM9240_IN4, 
                        SENSORS_ADM9240_IN4, SENSORS_MODE_RW, 
                        ADM9240_SYSCTL_IN4, VALUE(1), 2 },
    { SENSORS_ADM9240_IN5_MIN, "Vccp2_min", SENSORS_ADM9240_IN5, 
                        SENSORS_ADM9240_IN5, SENSORS_MODE_RW, 
                        ADM9240_SYSCTL_IN5, VALUE(1), 2 },
    { SENSORS_ADM9240_IN0_MAX, "2.5V_max", SENSORS_ADM9240_IN0, 
                        SENSORS_ADM9240_IN0, SENSORS_MODE_RW,
                        ADM9240_SYSCTL_IN0, VALUE(2), 2 },
    { SENSORS_ADM9240_IN1_MAX, "Vccp1_max", SENSORS_ADM9240_IN1, 
                        SENSORS_ADM9240_IN1, SENSORS_MODE_RW, 
                        ADM9240_SYSCTL_IN1, VALUE(2), 2 },
    { SENSORS_ADM9240_IN2_MAX, "3.3V_max", SENSORS_ADM9240_IN2, 
                        SENSORS_ADM9240_IN2, SENSORS_MODE_RW, 
                        ADM9240_SYSCTL_IN2, VALUE(2), 2 },
    { SENSORS_ADM9240_IN3_MAX, "5V_max", SENSORS_ADM9240_IN3, 
                        SENSORS_ADM9240_IN3, SENSORS_MODE_RW, 
                        ADM9240_SYSCTL_IN3, VALUE(2), 2 },
    { SENSORS_ADM9240_IN4_MAX, "12V_max", SENSORS_ADM9240_IN4, 
                        SENSORS_ADM9240_IN4, SENSORS_MODE_RW,
                        ADM9240_SYSCTL_IN4, VALUE(2), 2 },
    { SENSORS_ADM9240_IN5_MAX, "Vccp2_max", SENSORS_ADM9240_IN5, 
                        SENSORS_ADM9240_IN5, SENSORS_MODE_RW, 
                        ADM9240_SYSCTL_IN5, VALUE(2), 2 },
    { SENSORS_ADM9240_FAN1, "fan1", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, ADM9240_SYSCTL_FAN1, VALUE(2), 0 },
    { SENSORS_ADM9240_FAN2, "fan2", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, ADM9240_SYSCTL_FAN2, VALUE(2), 0 },
    { SENSORS_ADM9240_FAN1_MIN, "fan1_min", SENSORS_ADM9240_FAN1,
                        SENSORS_ADM9240_FAN1, SENSORS_MODE_RW, 
                        ADM9240_SYSCTL_FAN1, VALUE(1), 0 },
    { SENSORS_ADM9240_FAN2_MIN, "fan2_min", SENSORS_ADM9240_FAN2, 
                        SENSORS_ADM9240_FAN2, SENSORS_MODE_RW, 
                        ADM9240_SYSCTL_FAN2, VALUE(1), 0 },
    { SENSORS_ADM9240_TEMP, "temp", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, ADM9240_SYSCTL_TEMP, VALUE(3), 1 },
    { SENSORS_ADM9240_TEMP_HYST, "temp_hyst", SENSORS_ADM9240_TEMP,
                         SENSORS_ADM9240_TEMP, SENSORS_MODE_RW, 
                         ADM9240_SYSCTL_TEMP, VALUE(2), 1 },
    { SENSORS_ADM9240_TEMP_OVER, "temp_over", SENSORS_ADM9240_TEMP,
                         SENSORS_ADM9240_TEMP, SENSORS_MODE_RW, 
                         ADM9240_SYSCTL_TEMP, VALUE(1), 1 },

    { SENSORS_ADM9240_VID, "vid", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, ADM9240_SYSCTL_VID, VALUE(1), 2 },
    { SENSORS_ADM9240_FAN1_DIV, "fan1_div", SENSORS_ADM9240_FAN1, 
                         SENSORS_NO_MAPPING, SENSORS_MODE_RW, 
                         ADM9240_SYSCTL_FAN_DIV, VALUE(1), 0 },
    { SENSORS_ADM9240_FAN2_DIV, "fan2_div", SENSORS_ADM9240_FAN2, 
                         SENSORS_NO_MAPPING, SENSORS_MODE_RW, 
                         ADM9240_SYSCTL_FAN_DIV, VALUE(2), 0 },
    { SENSORS_ADM9240_ALARMS, "alarms", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING, 
                         SENSORS_MODE_R, ADM9240_SYSCTL_ALARMS, VALUE(1), 0 },
    { SENSORS_ADM9240_ANALOG_OUT, "analog_out", SENSORS_NO_MAPPING,
                         SENSORS_NO_MAPPING, SENSORS_MODE_RW,
                         ADM9240_SYSCTL_ANALOG_OUT, VALUE(1), 0 },
    { 0 }
  };

static sensors_chip_feature adm1024_features[] =
  { 
    { SENSORS_ADM1024_IN0, "2.5V", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING, 
                        SENSORS_MODE_R, ADM1024_SYSCTL_IN0, VALUE(3), 2 },
    { SENSORS_ADM1024_IN1, "Vccp1", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, ADM1024_SYSCTL_IN1, VALUE(3), 2 },
    { SENSORS_ADM1024_IN2, "3.3V", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, ADM1024_SYSCTL_IN2, VALUE(3), 2 },
    { SENSORS_ADM1024_IN3, "5V", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, ADM1024_SYSCTL_IN3, VALUE(3), 2 },
    { SENSORS_ADM1024_IN4, "12V", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, ADM1024_SYSCTL_IN4, VALUE(3), 2 },
    { SENSORS_ADM1024_IN5, "Vccp2", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, ADM1024_SYSCTL_IN5, VALUE(3), 2 },
    { SENSORS_ADM1024_IN0_MIN, "2.5V_min", SENSORS_ADM1024_IN0, 
                        SENSORS_ADM1024_IN0, SENSORS_MODE_RW, 
                        ADM1024_SYSCTL_IN0, VALUE(1), 2 },
    { SENSORS_ADM1024_IN1_MIN, "Vccp1_min", SENSORS_ADM1024_IN1, 
                        SENSORS_ADM1024_IN1, SENSORS_MODE_RW, 
                        ADM1024_SYSCTL_IN1, VALUE(1), 2 },
    { SENSORS_ADM1024_IN2_MIN, "3.3V_min", SENSORS_ADM1024_IN2, 
                        SENSORS_ADM1024_IN2, SENSORS_MODE_RW, 
                        ADM1024_SYSCTL_IN2, VALUE(1), 2 },
    { SENSORS_ADM1024_IN3_MIN, "5V_min", SENSORS_ADM1024_IN3, 
                        SENSORS_ADM1024_IN3, SENSORS_MODE_RW, 
                        ADM1024_SYSCTL_IN3, VALUE(1), 2 },
    { SENSORS_ADM1024_IN4_MIN, "12V_min", SENSORS_ADM1024_IN4, 
                        SENSORS_ADM1024_IN4, SENSORS_MODE_RW, 
                        ADM1024_SYSCTL_IN4, VALUE(1), 2 },
    { SENSORS_ADM1024_IN5_MIN, "Vccp2_min", SENSORS_ADM1024_IN5, 
                        SENSORS_ADM1024_IN5, SENSORS_MODE_RW, 
                        ADM1024_SYSCTL_IN5, VALUE(1), 2 },
    { SENSORS_ADM1024_IN0_MAX, "2.5V_max", SENSORS_ADM1024_IN0, 
                        SENSORS_ADM1024_IN0, SENSORS_MODE_RW,
                        ADM1024_SYSCTL_IN0, VALUE(2), 2 },
    { SENSORS_ADM1024_IN1_MAX, "Vccp1_max", SENSORS_ADM1024_IN1, 
                        SENSORS_ADM1024_IN1, SENSORS_MODE_RW, 
                        ADM1024_SYSCTL_IN1, VALUE(2), 2 },
    { SENSORS_ADM1024_IN2_MAX, "3.3V_max", SENSORS_ADM1024_IN2, 
                        SENSORS_ADM1024_IN2, SENSORS_MODE_RW, 
                        ADM1024_SYSCTL_IN2, VALUE(2), 2 },
    { SENSORS_ADM1024_IN3_MAX, "5V_max", SENSORS_ADM1024_IN3, 
                        SENSORS_ADM1024_IN3, SENSORS_MODE_RW, 
                        ADM1024_SYSCTL_IN3, VALUE(2), 2 },
    { SENSORS_ADM1024_IN4_MAX, "12V_max", SENSORS_ADM1024_IN4, 
                        SENSORS_ADM1024_IN4, SENSORS_MODE_RW,
                        ADM1024_SYSCTL_IN4, VALUE(2), 2 },
    { SENSORS_ADM1024_IN5_MAX, "Vccp2_max", SENSORS_ADM1024_IN5, 
                        SENSORS_ADM1024_IN5, SENSORS_MODE_RW, 
                        ADM1024_SYSCTL_IN5, VALUE(2), 2 },
    { SENSORS_ADM1024_FAN1, "fan1", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, ADM1024_SYSCTL_FAN1, VALUE(2), 0 },
    { SENSORS_ADM1024_FAN2, "fan2", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, ADM1024_SYSCTL_FAN2, VALUE(2), 0 },
    { SENSORS_ADM1024_FAN1_MIN, "fan1_min", SENSORS_ADM1024_FAN1,
                        SENSORS_ADM1024_FAN1, SENSORS_MODE_RW, 
                        ADM1024_SYSCTL_FAN1, VALUE(1), 0 },
    { SENSORS_ADM1024_FAN2_MIN, "fan2_min", SENSORS_ADM1024_FAN2, 
                        SENSORS_ADM1024_FAN2, SENSORS_MODE_RW, 
                        ADM1024_SYSCTL_FAN2, VALUE(1), 0 },
    { SENSORS_ADM1024_TEMP, "temp", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, ADM1024_SYSCTL_TEMP, VALUE(3), 1 },
    { SENSORS_ADM1024_TEMP1, "temp1", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, ADM1024_SYSCTL_TEMP1, VALUE(3), 1 },
    { SENSORS_ADM1024_TEMP2, "temp2", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, ADM1024_SYSCTL_TEMP2, VALUE(3), 1 },
    { SENSORS_ADM1024_TEMP_HYST, "temp_hyst", SENSORS_ADM1024_TEMP,
                         SENSORS_ADM1024_TEMP, SENSORS_MODE_RW, 
                         ADM1024_SYSCTL_TEMP, VALUE(2), 1 },
    { SENSORS_ADM1024_TEMP_OVER, "temp_over", SENSORS_ADM1024_TEMP,
                         SENSORS_ADM1024_TEMP, SENSORS_MODE_RW, 
                         ADM1024_SYSCTL_TEMP, VALUE(1), 1 },
    { SENSORS_ADM1024_TEMP1_HYST, "temp1_hyst", SENSORS_ADM1024_TEMP1,
                         SENSORS_ADM1024_TEMP1, SENSORS_MODE_RW, 
                         ADM1024_SYSCTL_TEMP1, VALUE(2), 1 },
    { SENSORS_ADM1024_TEMP1_OVER, "temp1_over", SENSORS_ADM1024_TEMP1,
                         SENSORS_ADM1024_TEMP1, SENSORS_MODE_RW, 
                         ADM1024_SYSCTL_TEMP1, VALUE(1), 1 },
    { SENSORS_ADM1024_TEMP2_HYST, "temp2_hyst", SENSORS_ADM1024_TEMP2,
                         SENSORS_ADM1024_TEMP2, SENSORS_MODE_RW, 
                         ADM1024_SYSCTL_TEMP2, VALUE(2), 1 },
    { SENSORS_ADM1024_TEMP2_OVER, "temp2_over", SENSORS_ADM1024_TEMP2,
                         SENSORS_ADM1024_TEMP2, SENSORS_MODE_RW, 
                         ADM1024_SYSCTL_TEMP2, VALUE(1), 1 },

    { SENSORS_ADM1024_VID, "vid", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, ADM1024_SYSCTL_VID, VALUE(1), 2 },
    { SENSORS_ADM1024_FAN1_DIV, "fan1_div", SENSORS_ADM1024_FAN1, 
                         SENSORS_NO_MAPPING, SENSORS_MODE_RW, 
                         ADM1024_SYSCTL_FAN_DIV, VALUE(1), 0 },
    { SENSORS_ADM1024_FAN2_DIV, "fan2_div", SENSORS_ADM1024_FAN2, 
                         SENSORS_NO_MAPPING, SENSORS_MODE_RW, 
                         ADM1024_SYSCTL_FAN_DIV, VALUE(2), 0 },
    { SENSORS_ADM1024_ALARMS, "alarms", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING, 
                         SENSORS_MODE_R, ADM1024_SYSCTL_ALARMS, VALUE(1), 0 },
    { SENSORS_ADM1024_ANALOG_OUT, "analog_out", SENSORS_NO_MAPPING,
                         SENSORS_NO_MAPPING, SENSORS_MODE_RW,
                         ADM1024_SYSCTL_ANALOG_OUT, VALUE(1), 0 },
    { 0 }
  };

static sensors_chip_feature ds1780_features[] =
  { 
    { SENSORS_DS1780_IN0, "2.5V", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING, 
                        SENSORS_MODE_R, ADM9240_SYSCTL_IN0, VALUE(3), 2 },
    { SENSORS_DS1780_IN1, "Vccp1", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, ADM9240_SYSCTL_IN1, VALUE(3), 2 },
    { SENSORS_DS1780_IN2, "3.3V", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, ADM9240_SYSCTL_IN2, VALUE(3), 2 },
    { SENSORS_DS1780_IN3, "5V", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, ADM9240_SYSCTL_IN3, VALUE(3), 2 },
    { SENSORS_DS1780_IN4, "12V", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, ADM9240_SYSCTL_IN4, VALUE(3), 2 },
    { SENSORS_DS1780_IN5, "Vccp2", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, ADM9240_SYSCTL_IN5, VALUE(3), 2 },
    { SENSORS_DS1780_IN0_MIN, "2.5V_min", SENSORS_DS1780_IN0, 
                        SENSORS_DS1780_IN0, SENSORS_MODE_RW, 
                        ADM9240_SYSCTL_IN0, VALUE(1), 2 },
    { SENSORS_DS1780_IN1_MIN, "Vccp1_min", SENSORS_DS1780_IN1, 
                        SENSORS_DS1780_IN1, SENSORS_MODE_RW, 
                        ADM9240_SYSCTL_IN1, VALUE(1), 2 },
    { SENSORS_DS1780_IN2_MIN, "3.3V_min", SENSORS_DS1780_IN2, 
                        SENSORS_DS1780_IN2, SENSORS_MODE_RW, 
                        ADM9240_SYSCTL_IN2, VALUE(1), 2 },
    { SENSORS_DS1780_IN3_MIN, "5V_min", SENSORS_DS1780_IN3, 
                        SENSORS_DS1780_IN3, SENSORS_MODE_RW, 
                        ADM9240_SYSCTL_IN3, VALUE(1), 2 },
    { SENSORS_DS1780_IN4_MIN, "12V_min", SENSORS_DS1780_IN4, 
                        SENSORS_DS1780_IN4, SENSORS_MODE_RW, 
                        ADM9240_SYSCTL_IN4, VALUE(1), 2 },
    { SENSORS_DS1780_IN5_MIN, "Vccp2_min", SENSORS_DS1780_IN5, 
                        SENSORS_DS1780_IN5, SENSORS_MODE_RW, 
                        ADM9240_SYSCTL_IN5, VALUE(1), 2 },
    { SENSORS_DS1780_IN0_MAX, "2.5V_max", SENSORS_DS1780_IN0, 
                        SENSORS_DS1780_IN0, SENSORS_MODE_RW,
                        ADM9240_SYSCTL_IN0, VALUE(2), 2 },
    { SENSORS_DS1780_IN1_MAX, "Vccp1_max", SENSORS_DS1780_IN1, 
                        SENSORS_DS1780_IN1, SENSORS_MODE_RW, 
                        ADM9240_SYSCTL_IN1, VALUE(2), 2 },
    { SENSORS_DS1780_IN2_MAX, "3.3V_max", SENSORS_DS1780_IN2, 
                        SENSORS_DS1780_IN2, SENSORS_MODE_RW, 
                        ADM9240_SYSCTL_IN2, VALUE(2), 2 },
    { SENSORS_DS1780_IN3_MAX, "5V_max", SENSORS_DS1780_IN3, 
                        SENSORS_DS1780_IN3, SENSORS_MODE_RW, 
                        ADM9240_SYSCTL_IN3, VALUE(2), 2 },
    { SENSORS_DS1780_IN4_MAX, "12V_max", SENSORS_DS1780_IN4, 
                        SENSORS_DS1780_IN4, SENSORS_MODE_RW,
                        ADM9240_SYSCTL_IN4, VALUE(2), 2 },
    { SENSORS_DS1780_IN5_MAX, "Vccp2_max", SENSORS_DS1780_IN5, 
                        SENSORS_DS1780_IN5, SENSORS_MODE_RW, 
                        ADM9240_SYSCTL_IN5, VALUE(2), 2 },
    { SENSORS_DS1780_FAN1, "fan1", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, ADM9240_SYSCTL_FAN1, VALUE(2), 0 },
    { SENSORS_DS1780_FAN2, "fan2", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, ADM9240_SYSCTL_FAN2, VALUE(2), 0 },
    { SENSORS_DS1780_FAN1_MIN, "fan1_min", SENSORS_DS1780_FAN1,
                        SENSORS_DS1780_FAN1, SENSORS_MODE_RW, 
                        ADM9240_SYSCTL_FAN1, VALUE(1), 0 },
    { SENSORS_DS1780_FAN2_MIN, "fan2_min", SENSORS_DS1780_FAN2, 
                        SENSORS_DS1780_FAN2, SENSORS_MODE_RW, 
                        ADM9240_SYSCTL_FAN2, VALUE(1), 0 },
    { SENSORS_DS1780_TEMP, "temp", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, ADM9240_SYSCTL_TEMP, VALUE(3), 1 },
    { SENSORS_DS1780_TEMP_HYST, "temp_hyst", SENSORS_DS1780_TEMP,
                         SENSORS_DS1780_TEMP, SENSORS_MODE_RW, 
                         ADM9240_SYSCTL_TEMP, VALUE(2), 1 },
    { SENSORS_DS1780_TEMP_OVER, "temp_over", SENSORS_DS1780_TEMP,
                         SENSORS_DS1780_TEMP, SENSORS_MODE_RW, 
                         ADM9240_SYSCTL_TEMP, VALUE(1), 1 },

    { SENSORS_DS1780_VID, "vid", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, ADM9240_SYSCTL_VID, VALUE(1), 2 },
    { SENSORS_DS1780_FAN1_DIV, "fan1_div", SENSORS_DS1780_FAN1, 
                         SENSORS_NO_MAPPING, SENSORS_MODE_RW, 
                         ADM9240_SYSCTL_FAN_DIV, VALUE(1), 0 },
    { SENSORS_DS1780_FAN2_DIV, "fan2_div", SENSORS_DS1780_FAN2, 
                         SENSORS_NO_MAPPING, SENSORS_MODE_RW, 
                         ADM9240_SYSCTL_FAN_DIV, VALUE(2), 0 },
    { SENSORS_DS1780_ALARMS, "alarms", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING, 
                         SENSORS_MODE_R, ADM9240_SYSCTL_ALARMS, VALUE(1), 0 },
    { SENSORS_DS1780_ANALOG_OUT, "analog_out", SENSORS_NO_MAPPING,
                         SENSORS_NO_MAPPING, SENSORS_MODE_RW,
                         ADM9240_SYSCTL_ANALOG_OUT, VALUE(1), 0 },
    { 0 }
  };

static sensors_chip_feature sis5595_features[] =
  { 
    { SENSORS_SIS5595_IN0, "in0", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING, 
                        SENSORS_MODE_R, LM78_SYSCTL_IN0, VALUE(3), 2 },
    { SENSORS_SIS5595_IN1, "in1", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, LM78_SYSCTL_IN1, VALUE(3), 2 },
    { SENSORS_SIS5595_IN2, "in2", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, LM78_SYSCTL_IN2, VALUE(3), 2 },
    { SENSORS_SIS5595_IN3, "in3", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, LM78_SYSCTL_IN3, VALUE(3), 2 },
    { SENSORS_SIS5595_IN4, "in4", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, LM78_SYSCTL_IN4, VALUE(3), 2 },
    { SENSORS_SIS5595_IN0_MIN, "in0_min", SENSORS_SIS5595_IN0, SENSORS_SIS5595_IN0,
                            SENSORS_MODE_RW, LM78_SYSCTL_IN0, VALUE(1), 2 },
    { SENSORS_SIS5595_IN1_MIN, "in1_min", SENSORS_SIS5595_IN1, SENSORS_SIS5595_IN1,
                            SENSORS_MODE_RW, LM78_SYSCTL_IN1, VALUE(1), 2 },
    { SENSORS_SIS5595_IN2_MIN, "in2_min", SENSORS_SIS5595_IN2, SENSORS_SIS5595_IN2,
                            SENSORS_MODE_RW, LM78_SYSCTL_IN2, VALUE(1), 2 },
    { SENSORS_SIS5595_IN3_MIN, "in3_min", SENSORS_SIS5595_IN3, SENSORS_SIS5595_IN3,
                            SENSORS_MODE_RW, LM78_SYSCTL_IN3, VALUE(1), 2 },
    { SENSORS_SIS5595_IN4_MIN, "in4_min", SENSORS_SIS5595_IN4, SENSORS_SIS5595_IN4,
                            SENSORS_MODE_RW, LM78_SYSCTL_IN4, VALUE(1), 2 },
    { SENSORS_SIS5595_IN0_MAX, "in0_max", SENSORS_SIS5595_IN0, SENSORS_SIS5595_IN0,
                            SENSORS_MODE_RW, LM78_SYSCTL_IN0, VALUE(2), 2 },
    { SENSORS_SIS5595_IN1_MAX, "in1_max", SENSORS_SIS5595_IN1, SENSORS_SIS5595_IN1,
                            SENSORS_MODE_RW, LM78_SYSCTL_IN1, VALUE(2), 2 },
    { SENSORS_SIS5595_IN2_MAX, "in2_max", SENSORS_SIS5595_IN2, SENSORS_SIS5595_IN2,
                            SENSORS_MODE_RW, LM78_SYSCTL_IN2, VALUE(2), 2 },
    { SENSORS_SIS5595_IN3_MAX, "in3_max", SENSORS_SIS5595_IN3, SENSORS_SIS5595_IN3,
                            SENSORS_MODE_RW, LM78_SYSCTL_IN3, VALUE(2), 2 },
    { SENSORS_SIS5595_IN4_MAX, "in4_max", SENSORS_SIS5595_IN4, SENSORS_SIS5595_IN4,
                            SENSORS_MODE_RW, LM78_SYSCTL_IN4, VALUE(2), 2 },
    { SENSORS_SIS5595_FAN1, "fan1", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, LM78_SYSCTL_FAN1, VALUE(2), 0 },
    { SENSORS_SIS5595_FAN2, "fan2", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, LM78_SYSCTL_FAN2, VALUE(2), 0 },
    { SENSORS_SIS5595_FAN1_MIN, "fan1_min", SENSORS_SIS5595_FAN1, SENSORS_SIS5595_FAN1,
                             SENSORS_MODE_RW, LM78_SYSCTL_FAN1, VALUE(1), 0 },
    { SENSORS_SIS5595_FAN2_MIN, "fan2_min", SENSORS_SIS5595_FAN2, SENSORS_SIS5595_FAN2,
                             SENSORS_MODE_RW, LM78_SYSCTL_FAN2, VALUE(1), 0 },
    { SENSORS_SIS5595_TEMP, "temp", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, LM78_SYSCTL_TEMP, VALUE(3), 1 },
    { SENSORS_SIS5595_TEMP_HYST, "temp_hyst", SENSORS_SIS5595_TEMP,
                              SENSORS_SIS5595_TEMP, SENSORS_MODE_RW, 
                              LM78_SYSCTL_TEMP, VALUE(2), 1 },
    { SENSORS_SIS5595_TEMP_OVER, "temp_over", SENSORS_SIS5595_TEMP,
                              SENSORS_SIS5595_TEMP, SENSORS_MODE_RW, 
                              LM78_SYSCTL_TEMP, VALUE(1), 1 },
    { SENSORS_SIS5595_FAN1_DIV, "fan1_div", SENSORS_SIS5595_FAN1, SENSORS_NO_MAPPING,
                             SENSORS_MODE_RW, LM78_SYSCTL_FAN_DIV, VALUE(1), 
                             0 },
    { SENSORS_SIS5595_FAN2_DIV, "fan2_div", SENSORS_SIS5595_FAN2, SENSORS_NO_MAPPING,
                             SENSORS_MODE_RW, LM78_SYSCTL_FAN_DIV, VALUE(2), 
                             0 },
    { SENSORS_SIS5595_ALARMS, "alarms", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING, 
                           SENSORS_MODE_R, LM78_SYSCTL_ALARMS, VALUE(1), 0 },
    { 0 }
  };

static sensors_chip_feature maxi_cg_features[] =
  { 
    { SENSORS_MAXI_CG_FAN1, "fan1", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, MAXI_SYSCTL_FAN1, VALUE(3), 0 },
    { SENSORS_MAXI_CG_FAN2, "fan2", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, MAXI_SYSCTL_FAN2, VALUE(3), 0 },
    { SENSORS_MAXI_CG_FAN3, "fan3", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, MAXI_SYSCTL_FAN3, VALUE(3), 0 },
    { SENSORS_MAXI_CG_FAN1_MIN, "fan1_min", SENSORS_MAXI_CG_FAN1, SENSORS_MAXI_CG_FAN1,
                             SENSORS_MODE_RW, MAXI_SYSCTL_FAN1, VALUE(1), 0 },
    { SENSORS_MAXI_CG_FAN2_MIN, "fan2_min", SENSORS_MAXI_CG_FAN2, SENSORS_MAXI_CG_FAN2,
                             SENSORS_MODE_RW, MAXI_SYSCTL_FAN2, VALUE(1), 0 },
    { SENSORS_MAXI_CG_FAN3_MIN, "fan3_min", SENSORS_MAXI_CG_FAN3, SENSORS_MAXI_CG_FAN3,
                             SENSORS_MODE_RW, MAXI_SYSCTL_FAN3, VALUE(1), 0 },
    { SENSORS_MAXI_CG_FAN1_DIV, "fan1_div", SENSORS_MAXI_CG_FAN1, SENSORS_NO_MAPPING,
                             SENSORS_MODE_R, MAXI_SYSCTL_FAN1, VALUE(2), 0 },
    { SENSORS_MAXI_CG_FAN2_DIV, "fan2_div", SENSORS_MAXI_CG_FAN2, SENSORS_NO_MAPPING,
                             SENSORS_MODE_R, MAXI_SYSCTL_FAN2, VALUE(2), 0 },
    { SENSORS_MAXI_CG_FAN3_DIV, "fan3_div", SENSORS_MAXI_CG_FAN3, SENSORS_NO_MAPPING,
                             SENSORS_MODE_R, MAXI_SYSCTL_FAN3, VALUE(2), 0 },
    { SENSORS_MAXI_CG_TEMP1, "temp1", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                          SENSORS_MODE_R, MAXI_SYSCTL_TEMP1, VALUE(3), 1 },
    { SENSORS_MAXI_CG_TEMP2, "temp2", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                          SENSORS_MODE_R, MAXI_SYSCTL_TEMP2, VALUE(3), 1 },
    { SENSORS_MAXI_CG_TEMP3, "temp3", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                          SENSORS_MODE_R, MAXI_SYSCTL_TEMP3, VALUE(3), 1 },
    { SENSORS_MAXI_CG_TEMP4, "temp4", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                          SENSORS_MODE_R, MAXI_SYSCTL_TEMP4, VALUE(3), 1 },
    { SENSORS_MAXI_CG_TEMP5, "temp5", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                          SENSORS_MODE_R, MAXI_SYSCTL_TEMP5, VALUE(3), 1 },
    { SENSORS_MAXI_CG_TEMP1_MAX, "temp1_max", SENSORS_MAXI_CG_TEMP1,
                              SENSORS_MAXI_CG_TEMP1, SENSORS_MODE_R, 
                              MAXI_SYSCTL_TEMP1, VALUE(1), 1 },
    { SENSORS_MAXI_CG_TEMP2_MAX, "temp2_max", SENSORS_MAXI_CG_TEMP2,
                              SENSORS_MAXI_CG_TEMP2, SENSORS_MODE_R, 
                              MAXI_SYSCTL_TEMP2, VALUE(1), 1 },
    { SENSORS_MAXI_CG_TEMP3_MAX, "temp3_max", SENSORS_MAXI_CG_TEMP3,
                              SENSORS_MAXI_CG_TEMP3, SENSORS_MODE_R, 
                              MAXI_SYSCTL_TEMP3, VALUE(1), 1 },
    { SENSORS_MAXI_CG_TEMP4_MAX, "temp4_max", SENSORS_MAXI_CG_TEMP4,
                              SENSORS_MAXI_CG_TEMP4, SENSORS_MODE_R, 
                              MAXI_SYSCTL_TEMP4, VALUE(1), 1 },
    { SENSORS_MAXI_CG_TEMP5_MAX, "temp5_max", SENSORS_MAXI_CG_TEMP5,
                              SENSORS_MAXI_CG_TEMP5, SENSORS_MODE_R, 
                              MAXI_SYSCTL_TEMP5, VALUE(1), 1 },
    { SENSORS_MAXI_CG_TEMP1_HYST, "temp1_hyst", SENSORS_MAXI_CG_TEMP1,
                              SENSORS_MAXI_CG_TEMP1, SENSORS_MODE_R, 
                              MAXI_SYSCTL_TEMP1, VALUE(2), 1 },
    { SENSORS_MAXI_CG_TEMP2_HYST, "temp2_hyst", SENSORS_MAXI_CG_TEMP2,
                              SENSORS_MAXI_CG_TEMP2, SENSORS_MODE_R, 
                              MAXI_SYSCTL_TEMP2, VALUE(2), 1 },
    { SENSORS_MAXI_CG_TEMP3_HYST, "temp3_hyst", SENSORS_MAXI_CG_TEMP3,
                              SENSORS_MAXI_CG_TEMP3, SENSORS_MODE_R, 
                              MAXI_SYSCTL_TEMP3, VALUE(2), 1 },
    { SENSORS_MAXI_CG_TEMP4_HYST, "temp4_hyst", SENSORS_MAXI_CG_TEMP4,
                              SENSORS_MAXI_CG_TEMP4, SENSORS_MODE_R, 
                              MAXI_SYSCTL_TEMP4, VALUE(2), 1 },
    { SENSORS_MAXI_CG_TEMP5_HYST, "temp5_hyst", SENSORS_MAXI_CG_TEMP5,
                              SENSORS_MAXI_CG_TEMP5, SENSORS_MODE_R, 
                              MAXI_SYSCTL_TEMP5, VALUE(2), 1 },
    { SENSORS_MAXI_CG_PLL, "pll", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, MAXI_SYSCTL_PLL, VALUE(3), 2 },
    { SENSORS_MAXI_CG_PLL_MIN, "pll_min", SENSORS_MAXI_CG_PLL, SENSORS_MAXI_CG_PLL,
                            SENSORS_MODE_RW, MAXI_SYSCTL_PLL, VALUE(1), 2 },
    { SENSORS_MAXI_CG_PLL_MAX, "pll_max", SENSORS_MAXI_CG_PLL, SENSORS_MAXI_CG_PLL,
                            SENSORS_MODE_RW, MAXI_SYSCTL_PLL, VALUE(2), 2 },
    { SENSORS_MAXI_CG_VID1, "vid1", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, MAXI_SYSCTL_VID1, VALUE(3), 4 },
    { SENSORS_MAXI_CG_VID2, "vid2", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, MAXI_SYSCTL_VID2, VALUE(3), 4 },
    { SENSORS_MAXI_CG_VID3, "vid3", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, MAXI_SYSCTL_VID3, VALUE(3), 4 },
    { SENSORS_MAXI_CG_VID4, "vid4", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, MAXI_SYSCTL_VID4, VALUE(3), 4 },
    { SENSORS_MAXI_CG_VID1_MIN, "vid1_min", SENSORS_MAXI_CG_VID1, SENSORS_MAXI_CG_VID1,
                             SENSORS_MODE_RW, MAXI_SYSCTL_VID1, VALUE(1), 4 },
    { SENSORS_MAXI_CG_VID2_MIN, "vid2_min", SENSORS_MAXI_CG_VID2, SENSORS_MAXI_CG_VID2,
                             SENSORS_MODE_RW, MAXI_SYSCTL_VID2, VALUE(1), 4 },
    { SENSORS_MAXI_CG_VID3_MIN, "vid3_min", SENSORS_MAXI_CG_VID3, SENSORS_MAXI_CG_VID3,
                             SENSORS_MODE_RW, MAXI_SYSCTL_VID3, VALUE(1), 4 },
    { SENSORS_MAXI_CG_VID4_MIN, "vid4_min", SENSORS_MAXI_CG_VID4, SENSORS_MAXI_CG_VID4,
                             SENSORS_MODE_RW, MAXI_SYSCTL_VID4, VALUE(1), 4 },
    { SENSORS_MAXI_CG_VID1_MAX, "vid1_max", SENSORS_MAXI_CG_VID1, SENSORS_MAXI_CG_VID1,
                             SENSORS_MODE_RW, MAXI_SYSCTL_VID1, VALUE(2), 4 },
    { SENSORS_MAXI_CG_VID2_MAX, "vid2_max", SENSORS_MAXI_CG_VID2, SENSORS_MAXI_CG_VID2,
                             SENSORS_MODE_RW, MAXI_SYSCTL_VID2, VALUE(2), 4 },
    { SENSORS_MAXI_CG_VID3_MAX, "vid3_max", SENSORS_MAXI_CG_VID3, SENSORS_MAXI_CG_VID3,
                             SENSORS_MODE_RW, MAXI_SYSCTL_VID3, VALUE(2), 4 },
    { SENSORS_MAXI_CG_VID4_MAX, "vid4_max", SENSORS_MAXI_CG_VID4, SENSORS_MAXI_CG_VID4,
                             SENSORS_MODE_RW, MAXI_SYSCTL_VID4, VALUE(2), 4 },
    { SENSORS_MAXI_CG_ALARMS, "alarms", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING, 
                           SENSORS_MODE_R, MAXI_SYSCTL_ALARMS, VALUE(1), 0 },
    { 0 }
  };

static sensors_chip_feature maxi_co_features[] =
  { 
    { SENSORS_MAXI_CO_FAN1, "fan1", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, MAXI_SYSCTL_FAN1, VALUE(3), 0 },
    { SENSORS_MAXI_CO_FAN2, "fan2", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, MAXI_SYSCTL_FAN2, VALUE(3), 0 },
    { SENSORS_MAXI_CO_FAN3, "fan3", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, MAXI_SYSCTL_FAN3, VALUE(3), 0 },
    { SENSORS_MAXI_CO_FAN1_MIN, "fan1_min", SENSORS_MAXI_CO_FAN1, SENSORS_MAXI_CO_FAN1,
                             SENSORS_MODE_RW, MAXI_SYSCTL_FAN1, VALUE(1), 0 },
    { SENSORS_MAXI_CO_FAN2_MIN, "fan2_min", SENSORS_MAXI_CO_FAN2, SENSORS_MAXI_CO_FAN2,
                             SENSORS_MODE_RW, MAXI_SYSCTL_FAN2, VALUE(1), 0 },
    { SENSORS_MAXI_CO_FAN3_MIN, "fan3_min", SENSORS_MAXI_CO_FAN3, SENSORS_MAXI_CO_FAN3,
                             SENSORS_MODE_RW, MAXI_SYSCTL_FAN3, VALUE(1), 0 },
    { SENSORS_MAXI_CO_FAN1_DIV, "fan1_div", SENSORS_MAXI_CO_FAN1, SENSORS_NO_MAPPING,
                             SENSORS_MODE_R, MAXI_SYSCTL_FAN1, VALUE(2), 0 },
    { SENSORS_MAXI_CO_FAN2_DIV, "fan2_div", SENSORS_MAXI_CO_FAN2, SENSORS_NO_MAPPING,
                             SENSORS_MODE_R, MAXI_SYSCTL_FAN2, VALUE(2), 0 },
    { SENSORS_MAXI_CO_FAN3_DIV, "fan3_div", SENSORS_MAXI_CO_FAN3, SENSORS_NO_MAPPING,
                             SENSORS_MODE_R, MAXI_SYSCTL_FAN3, VALUE(2), 0 },
    { SENSORS_MAXI_CO_TEMP1, "temp1", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                          SENSORS_MODE_R, MAXI_SYSCTL_TEMP1, VALUE(3), 1 },
    { SENSORS_MAXI_CO_TEMP2, "temp2", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                          SENSORS_MODE_R, MAXI_SYSCTL_TEMP2, VALUE(3), 1 },
    { SENSORS_MAXI_CO_TEMP3, "temp3", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                          SENSORS_MODE_R, MAXI_SYSCTL_TEMP3, VALUE(3), 1 },
    { SENSORS_MAXI_CO_TEMP4, "temp4", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                          SENSORS_MODE_R, MAXI_SYSCTL_TEMP4, VALUE(3), 1 },
    { SENSORS_MAXI_CO_TEMP5, "temp5", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                          SENSORS_MODE_R, MAXI_SYSCTL_TEMP5, VALUE(3), 1 },
    { SENSORS_MAXI_CO_TEMP1_MAX, "temp1_max", SENSORS_MAXI_CO_TEMP1,
                              SENSORS_MAXI_CO_TEMP1, SENSORS_MODE_R, 
                              MAXI_SYSCTL_TEMP1, VALUE(1), 1 },
    { SENSORS_MAXI_CO_TEMP2_MAX, "temp2_max", SENSORS_MAXI_CO_TEMP2,
                              SENSORS_MAXI_CO_TEMP2, SENSORS_MODE_R, 
                              MAXI_SYSCTL_TEMP2, VALUE(1), 1 },
    { SENSORS_MAXI_CO_TEMP3_MAX, "temp3_max", SENSORS_MAXI_CO_TEMP3,
                              SENSORS_MAXI_CO_TEMP3, SENSORS_MODE_R, 
                              MAXI_SYSCTL_TEMP3, VALUE(1), 1 },
    { SENSORS_MAXI_CO_TEMP4_MAX, "temp4_max", SENSORS_MAXI_CO_TEMP4,
                              SENSORS_MAXI_CO_TEMP4, SENSORS_MODE_R, 
                              MAXI_SYSCTL_TEMP4, VALUE(1), 1 },
    { SENSORS_MAXI_CO_TEMP5_MAX, "temp5_max", SENSORS_MAXI_CO_TEMP5,
                              SENSORS_MAXI_CO_TEMP5, SENSORS_MODE_R, 
                              MAXI_SYSCTL_TEMP5, VALUE(1), 1 },
    { SENSORS_MAXI_CO_TEMP1_HYST, "temp1_hyst", SENSORS_MAXI_CO_TEMP1,
                              SENSORS_MAXI_CO_TEMP1, SENSORS_MODE_R, 
                              MAXI_SYSCTL_TEMP1, VALUE(2), 1 },
    { SENSORS_MAXI_CO_TEMP2_HYST, "temp2_hyst", SENSORS_MAXI_CO_TEMP2,
                              SENSORS_MAXI_CO_TEMP2, SENSORS_MODE_R, 
                              MAXI_SYSCTL_TEMP2, VALUE(2), 1 },
    { SENSORS_MAXI_CO_TEMP3_HYST, "temp3_hyst", SENSORS_MAXI_CO_TEMP3,
                              SENSORS_MAXI_CO_TEMP3, SENSORS_MODE_R, 
                              MAXI_SYSCTL_TEMP3, VALUE(2), 1 },
    { SENSORS_MAXI_CO_TEMP4_HYST, "temp4_hyst", SENSORS_MAXI_CO_TEMP4,
                              SENSORS_MAXI_CO_TEMP4, SENSORS_MODE_R, 
                              MAXI_SYSCTL_TEMP4, VALUE(2), 1 },
    { SENSORS_MAXI_CO_TEMP5_HYST, "temp5_hyst", SENSORS_MAXI_CO_TEMP5,
                              SENSORS_MAXI_CO_TEMP5, SENSORS_MODE_R, 
                              MAXI_SYSCTL_TEMP5, VALUE(2), 1 },
    { SENSORS_MAXI_CO_PLL, "pll", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, MAXI_SYSCTL_PLL, VALUE(3), 2 },
    { SENSORS_MAXI_CO_PLL_MIN, "pll_min", SENSORS_MAXI_CO_PLL, SENSORS_MAXI_CO_PLL,
                            SENSORS_MODE_RW, MAXI_SYSCTL_PLL, VALUE(1), 2 },
    { SENSORS_MAXI_CO_PLL_MAX, "pll_max", SENSORS_MAXI_CO_PLL, SENSORS_MAXI_CO_PLL,
                            SENSORS_MODE_RW, MAXI_SYSCTL_PLL, VALUE(2), 2 },
    { SENSORS_MAXI_CO_VID1, "vid1", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, MAXI_SYSCTL_VID1, VALUE(3), 4 },
    { SENSORS_MAXI_CO_VID2, "vid2", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, MAXI_SYSCTL_VID2, VALUE(3), 4 },
    { SENSORS_MAXI_CO_VID3, "vid3", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, MAXI_SYSCTL_VID3, VALUE(3), 4 },
    { SENSORS_MAXI_CO_VID4, "vid4", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, MAXI_SYSCTL_VID4, VALUE(3), 4 },
    { SENSORS_MAXI_CO_VID1_MIN, "vid1_min", SENSORS_MAXI_CO_VID1, SENSORS_MAXI_CO_VID1,
                             SENSORS_MODE_RW, MAXI_SYSCTL_VID1, VALUE(1), 4 },
    { SENSORS_MAXI_CO_VID2_MIN, "vid2_min", SENSORS_MAXI_CO_VID2, SENSORS_MAXI_CO_VID2,
                             SENSORS_MODE_RW, MAXI_SYSCTL_VID2, VALUE(1), 4 },
    { SENSORS_MAXI_CO_VID3_MIN, "vid3_min", SENSORS_MAXI_CO_VID3, SENSORS_MAXI_CO_VID3,
                             SENSORS_MODE_RW, MAXI_SYSCTL_VID3, VALUE(1), 4 },
    { SENSORS_MAXI_CO_VID4_MIN, "vid4_min", SENSORS_MAXI_CO_VID4, SENSORS_MAXI_CO_VID4,
                             SENSORS_MODE_RW, MAXI_SYSCTL_VID4, VALUE(1), 4 },
    { SENSORS_MAXI_CO_VID1_MAX, "vid1_max", SENSORS_MAXI_CO_VID1, SENSORS_MAXI_CO_VID1,
                             SENSORS_MODE_RW, MAXI_SYSCTL_VID1, VALUE(2), 4 },
    { SENSORS_MAXI_CO_VID2_MAX, "vid2_max", SENSORS_MAXI_CO_VID2, SENSORS_MAXI_CO_VID2,
                             SENSORS_MODE_RW, MAXI_SYSCTL_VID2, VALUE(2), 4 },
    { SENSORS_MAXI_CG_VID3_MAX, "vid3_max", SENSORS_MAXI_CO_VID3, SENSORS_MAXI_CO_VID3,
                             SENSORS_MODE_RW, MAXI_SYSCTL_VID3, VALUE(2), 4 },
    { SENSORS_MAXI_CO_VID4_MAX, "vid4_max", SENSORS_MAXI_CO_VID4, SENSORS_MAXI_CG_VID4,
                             SENSORS_MODE_RW, MAXI_SYSCTL_VID4, VALUE(2), 4 },
    { SENSORS_MAXI_CO_ALARMS, "alarms", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING, 
                           SENSORS_MODE_R, MAXI_SYSCTL_ALARMS, VALUE(1), 0 },
    { 0 }
  };

static sensors_chip_feature maxi_as_features[] =
  { 
    { SENSORS_MAXI_AS_FAN1, "fan1", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, MAXI_SYSCTL_FAN1, VALUE(3), 0 },
    { SENSORS_MAXI_AS_FAN2, "fan2", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, MAXI_SYSCTL_FAN2, VALUE(3), 0 },
    { SENSORS_MAXI_AS_FAN3, "fan3", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, MAXI_SYSCTL_FAN3, VALUE(3), 0 },
    { SENSORS_MAXI_AS_FAN1_MIN, "fan1_min", SENSORS_MAXI_AS_FAN1, SENSORS_MAXI_AS_FAN1,
                             SENSORS_MODE_RW, MAXI_SYSCTL_FAN1, VALUE(1), 0 },
    { SENSORS_MAXI_AS_FAN2_MIN, "fan2_min", SENSORS_MAXI_AS_FAN2, SENSORS_MAXI_AS_FAN2,
                             SENSORS_MODE_RW, MAXI_SYSCTL_FAN2, VALUE(1), 0 },
    { SENSORS_MAXI_AS_FAN3_MIN, "fan3_min", SENSORS_MAXI_AS_FAN3, SENSORS_MAXI_AS_FAN3,
                             SENSORS_MODE_RW, MAXI_SYSCTL_FAN3, VALUE(1), 0 },
    { SENSORS_MAXI_AS_FAN1_DIV, "fan1_div", SENSORS_MAXI_AS_FAN1, SENSORS_NO_MAPPING,
                             SENSORS_MODE_R, MAXI_SYSCTL_FAN1, VALUE(2), 0 },
    { SENSORS_MAXI_AS_FAN2_DIV, "fan2_div", SENSORS_MAXI_AS_FAN2, SENSORS_NO_MAPPING,
                             SENSORS_MODE_R, MAXI_SYSCTL_FAN2, VALUE(2), 0 },
    { SENSORS_MAXI_AS_FAN3_DIV, "fan3_div", SENSORS_MAXI_AS_FAN3, SENSORS_NO_MAPPING,
                             SENSORS_MODE_R, MAXI_SYSCTL_FAN3, VALUE(2), 0 },
    { SENSORS_MAXI_AS_TEMP1, "temp1", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                          SENSORS_MODE_R, MAXI_SYSCTL_TEMP1, VALUE(3), 1 },
    { SENSORS_MAXI_AS_TEMP2, "temp2", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                          SENSORS_MODE_R, MAXI_SYSCTL_TEMP2, VALUE(3), 1 },
    { SENSORS_MAXI_AS_TEMP3, "temp3", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                          SENSORS_MODE_R, MAXI_SYSCTL_TEMP3, VALUE(3), 1 },
    { SENSORS_MAXI_AS_TEMP4, "temp4", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                          SENSORS_MODE_R, MAXI_SYSCTL_TEMP4, VALUE(3), 1 },
    { SENSORS_MAXI_AS_TEMP5, "temp5", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                          SENSORS_MODE_R, MAXI_SYSCTL_TEMP5, VALUE(3), 1 },
    { SENSORS_MAXI_AS_TEMP1_MAX, "temp1_max", SENSORS_MAXI_AS_TEMP1,
                              SENSORS_MAXI_AS_TEMP1, SENSORS_MODE_R, 
                              MAXI_SYSCTL_TEMP1, VALUE(1), 1 },
    { SENSORS_MAXI_AS_TEMP2_MAX, "temp2_max", SENSORS_MAXI_AS_TEMP2,
                              SENSORS_MAXI_AS_TEMP2, SENSORS_MODE_R, 
                              MAXI_SYSCTL_TEMP2, VALUE(1), 1 },
    { SENSORS_MAXI_AS_TEMP3_MAX, "temp3_max", SENSORS_MAXI_AS_TEMP3,
                              SENSORS_MAXI_AS_TEMP3, SENSORS_MODE_R, 
                              MAXI_SYSCTL_TEMP3, VALUE(1), 1 },
    { SENSORS_MAXI_AS_TEMP4_MAX, "temp4_max", SENSORS_MAXI_AS_TEMP4,
                              SENSORS_MAXI_AS_TEMP4, SENSORS_MODE_R, 
                              MAXI_SYSCTL_TEMP4, VALUE(1), 1 },
    { SENSORS_MAXI_AS_TEMP5_MAX, "temp5_max", SENSORS_MAXI_AS_TEMP5,
                              SENSORS_MAXI_AS_TEMP5, SENSORS_MODE_R, 
                              MAXI_SYSCTL_TEMP5, VALUE(1), 1 },
    { SENSORS_MAXI_AS_TEMP1_HYST, "temp1_hyst", SENSORS_MAXI_AS_TEMP1,
                              SENSORS_MAXI_AS_TEMP1, SENSORS_MODE_R, 
                              MAXI_SYSCTL_TEMP1, VALUE(2), 1 },
    { SENSORS_MAXI_AS_TEMP2_HYST, "temp2_hyst", SENSORS_MAXI_AS_TEMP2,
                              SENSORS_MAXI_AS_TEMP2, SENSORS_MODE_R, 
                              MAXI_SYSCTL_TEMP2, VALUE(2), 1 },
    { SENSORS_MAXI_AS_TEMP3_HYST, "temp3_hyst", SENSORS_MAXI_AS_TEMP3,
                              SENSORS_MAXI_AS_TEMP3, SENSORS_MODE_R, 
                              MAXI_SYSCTL_TEMP3, VALUE(2), 1 },
    { SENSORS_MAXI_AS_TEMP4_HYST, "temp4_hyst", SENSORS_MAXI_AS_TEMP4,
                              SENSORS_MAXI_AS_TEMP4, SENSORS_MODE_R, 
                              MAXI_SYSCTL_TEMP4, VALUE(2), 1 },
    { SENSORS_MAXI_AS_TEMP5_HYST, "temp5_hyst", SENSORS_MAXI_AS_TEMP5,
                              SENSORS_MAXI_AS_TEMP5, SENSORS_MODE_R, 
                              MAXI_SYSCTL_TEMP5, VALUE(2), 1 },
    { SENSORS_MAXI_AS_PLL, "pll", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, MAXI_SYSCTL_PLL, VALUE(3), 2 },
    { SENSORS_MAXI_AS_PLL_MIN, "pll_min", SENSORS_MAXI_AS_PLL, SENSORS_MAXI_AS_PLL,
                            SENSORS_MODE_RW, MAXI_SYSCTL_PLL, VALUE(1), 2 },
    { SENSORS_MAXI_AS_PLL_MAX, "pll_max", SENSORS_MAXI_AS_PLL, SENSORS_MAXI_AS_PLL,
                            SENSORS_MODE_RW, MAXI_SYSCTL_PLL, VALUE(2), 2 },
    { SENSORS_MAXI_AS_VID1, "vid1", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, MAXI_SYSCTL_VID1, VALUE(3), 4 },
    { SENSORS_MAXI_AS_VID2, "vid2", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, MAXI_SYSCTL_VID2, VALUE(3), 4 },
    { SENSORS_MAXI_AS_VID3, "vid3", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, MAXI_SYSCTL_VID3, VALUE(3), 4 },
    { SENSORS_MAXI_AS_VID4, "vid4", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, MAXI_SYSCTL_VID4, VALUE(3), 4 },
    { SENSORS_MAXI_AS_VID1_MIN, "vid1_min", SENSORS_MAXI_AS_VID1, SENSORS_MAXI_AS_VID1,
                             SENSORS_MODE_RW, MAXI_SYSCTL_VID1, VALUE(1), 4 },
    { SENSORS_MAXI_AS_VID2_MIN, "vid2_min", SENSORS_MAXI_AS_VID2, SENSORS_MAXI_AS_VID2,
                             SENSORS_MODE_RW, MAXI_SYSCTL_VID2, VALUE(1), 4 },
    { SENSORS_MAXI_AS_VID3_MIN, "vid3_min", SENSORS_MAXI_AS_VID3, SENSORS_MAXI_AS_VID3,
                             SENSORS_MODE_RW, MAXI_SYSCTL_VID3, VALUE(1), 4 },
    { SENSORS_MAXI_AS_VID4_MIN, "vid4_min", SENSORS_MAXI_AS_VID4, SENSORS_MAXI_AS_VID4,
                             SENSORS_MODE_RW, MAXI_SYSCTL_VID4, VALUE(1), 4 },
    { SENSORS_MAXI_AS_VID1_MAX, "vid1_max", SENSORS_MAXI_AS_VID1, SENSORS_MAXI_AS_VID1,
                             SENSORS_MODE_RW, MAXI_SYSCTL_VID1, VALUE(2), 4 },
    { SENSORS_MAXI_AS_VID2_MAX, "vid2_max", SENSORS_MAXI_AS_VID2, SENSORS_MAXI_AS_VID2,
                             SENSORS_MODE_RW, MAXI_SYSCTL_VID2, VALUE(2), 4 },
    { SENSORS_MAXI_AS_VID3_MAX, "vid3_max", SENSORS_MAXI_AS_VID3, SENSORS_MAXI_AS_VID3,
                             SENSORS_MODE_RW, MAXI_SYSCTL_VID3, VALUE(2), 4 },
    { SENSORS_MAXI_AS_VID4_MAX, "vid4_max", SENSORS_MAXI_AS_VID4, SENSORS_MAXI_CG_VID4,
                             SENSORS_MODE_RW, MAXI_SYSCTL_VID4, VALUE(2), 4 },
    { SENSORS_MAXI_AS_ALARMS, "alarms", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING, 
                           SENSORS_MODE_R, MAXI_SYSCTL_ALARMS, VALUE(1), 0 },
    { 0 }
  };

static sensors_chip_feature thmc50_features[] =
  {
    { SENSORS_THMC50_TEMP, "temp", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                              SENSORS_MODE_R, THMC50_SYSCTL_TEMP, VALUE(3), 
                              0 },
    { SENSORS_THMC50_TEMP_HYST, "temp_hyst", SENSORS_THMC50_TEMP,
                              SENSORS_THMC50_TEMP, SENSORS_MODE_RW, 
                              THMC50_SYSCTL_TEMP, VALUE(2), 0 },
    { SENSORS_THMC50_TEMP_OVER, "temp_over", SENSORS_THMC50_TEMP,
                              SENSORS_THMC50_TEMP, SENSORS_MODE_RW,
                              THMC50_SYSCTL_TEMP, VALUE(1), 0 },
    { SENSORS_THMC50_REMOTE_TEMP, "remote_temp", SENSORS_NO_MAPPING, 
                              SENSORS_NO_MAPPING, SENSORS_MODE_R, 
                              THMC50_SYSCTL_REMOTE_TEMP, VALUE(3), 0 },
    { SENSORS_THMC50_REMOTE_TEMP_HYST, "remote_temp_hyst", 
                              SENSORS_THMC50_REMOTE_TEMP,
                              SENSORS_THMC50_REMOTE_TEMP, SENSORS_MODE_RW, 
                              THMC50_SYSCTL_REMOTE_TEMP, VALUE(2), 0 },
    { SENSORS_THMC50_REMOTE_TEMP_OVER, "remote_temp_over", 
                              SENSORS_THMC50_REMOTE_TEMP,
                              SENSORS_THMC50_REMOTE_TEMP, SENSORS_MODE_RW,
                              THMC50_SYSCTL_REMOTE_TEMP, VALUE(1), 0 },
    { SENSORS_THMC50_INTER_MASK, "analog output", SENSORS_NO_MAPPING,
                              SENSORS_NO_MAPPING, SENSORS_MODE_RW,
                              THMC50_SYSCTL_ANALOG_OUT, VALUE(1), 0 },
    { SENSORS_THMC50_INTER, "interrupts", SENSORS_NO_MAPPING,
                              SENSORS_NO_MAPPING, SENSORS_MODE_R,
                              THMC50_SYSCTL_INTER, VALUE(1), 0 },
    { SENSORS_THMC50_INTER_MASK, "interrupt mask", SENSORS_NO_MAPPING,
                              SENSORS_NO_MAPPING, SENSORS_MODE_RW,
                              THMC50_SYSCTL_INTER_MASK, VALUE(1), 0 },
    { SENSORS_THMC50_DIE_CODE, "die_code", SENSORS_NO_MAPPING,
                              SENSORS_NO_MAPPING, SENSORS_MODE_R,
                              THMC50_SYSCTL_DIE_CODE, VALUE(1), 0 },
    { 0 }
  };

static sensors_chip_feature adm1025_features[] =
  { 
    { SENSORS_ADM1025_IN0, "in0", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING, 
                        SENSORS_MODE_R, ADM1025_SYSCTL_IN0, VALUE(3), 2 },
    { SENSORS_ADM1025_IN1, "in1", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, ADM1025_SYSCTL_IN1, VALUE(3), 2 },
    { SENSORS_ADM1025_IN2, "in2", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, ADM1025_SYSCTL_IN2, VALUE(3), 2 },
    { SENSORS_ADM1025_IN3, "in3", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, ADM1025_SYSCTL_IN3, VALUE(3), 2 },
    { SENSORS_ADM1025_IN4, "in4", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, ADM1025_SYSCTL_IN4, VALUE(3), 2 },
    { SENSORS_ADM1025_IN5, "in5", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, ADM1025_SYSCTL_IN5, VALUE(3), 2 },
    { SENSORS_ADM1025_IN0_MIN, "in0_min", SENSORS_ADM1025_IN0, 
                        SENSORS_ADM1025_IN0, SENSORS_MODE_RW, 
                        ADM1025_SYSCTL_IN0, VALUE(1), 2 },
    { SENSORS_ADM1025_IN1_MIN, "in1_min", SENSORS_ADM1025_IN1, 
                        SENSORS_ADM1025_IN1, SENSORS_MODE_RW, 
                        ADM1025_SYSCTL_IN1, VALUE(1), 2 },
    { SENSORS_ADM1025_IN2_MIN, "in2_min", SENSORS_ADM1025_IN2, 
                        SENSORS_ADM1025_IN2, SENSORS_MODE_RW, 
                        ADM1025_SYSCTL_IN2, VALUE(1), 2 },
    { SENSORS_ADM1025_IN3_MIN, "in3_min", SENSORS_ADM1025_IN3, 
                        SENSORS_ADM1025_IN3, SENSORS_MODE_RW, 
                        ADM1025_SYSCTL_IN3, VALUE(1), 2 },
    { SENSORS_ADM1025_IN4_MIN, "in4_min", SENSORS_ADM1025_IN4, 
                        SENSORS_ADM1025_IN4, SENSORS_MODE_RW, 
                        ADM1025_SYSCTL_IN4, VALUE(1), 2 },
    { SENSORS_ADM1025_IN5_MIN, "in5_min", SENSORS_ADM1025_IN5, 
                        SENSORS_ADM1025_IN5, SENSORS_MODE_RW, 
                        ADM1025_SYSCTL_IN5, VALUE(1), 2 },
    { SENSORS_ADM1025_IN0_MAX, "in0_max", SENSORS_ADM1025_IN0, 
                        SENSORS_ADM1025_IN0, SENSORS_MODE_RW,
                        ADM1025_SYSCTL_IN0, VALUE(2), 2 },
    { SENSORS_ADM1025_IN1_MAX, "in1_max", SENSORS_ADM1025_IN1, 
                        SENSORS_ADM1025_IN1, SENSORS_MODE_RW, 
                        ADM1025_SYSCTL_IN1, VALUE(2), 2 },
    { SENSORS_ADM1025_IN2_MAX, "in2_max", SENSORS_ADM1025_IN2, 
                        SENSORS_ADM1025_IN2, SENSORS_MODE_RW, 
                        ADM1025_SYSCTL_IN2, VALUE(2), 2 },
    { SENSORS_ADM1025_IN3_MAX, "in3_max", SENSORS_ADM1025_IN3, 
                        SENSORS_ADM1025_IN3, SENSORS_MODE_RW, 
                        ADM1025_SYSCTL_IN3, VALUE(2), 2 },
    { SENSORS_ADM1025_IN4_MAX, "in4_max", SENSORS_ADM1025_IN4, 
                        SENSORS_ADM1025_IN4, SENSORS_MODE_RW,
                        ADM1025_SYSCTL_IN4, VALUE(2), 2 },
    { SENSORS_ADM1025_IN5_MAX, "in5_max", SENSORS_ADM1025_IN5, 
                        SENSORS_ADM1025_IN5, SENSORS_MODE_RW, 
                        ADM1025_SYSCTL_IN5, VALUE(2), 2 },
    { SENSORS_ADM1025_TEMP1, "temp1", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, ADM1025_SYSCTL_RTEMP, VALUE(3), 1 },
    { SENSORS_ADM1025_TEMP1_LOW, "temp1_low", SENSORS_ADM1025_TEMP1,
                         SENSORS_ADM1025_TEMP1, SENSORS_MODE_RW, 
                         ADM1025_SYSCTL_RTEMP, VALUE(2), 1 },
    { SENSORS_ADM1025_TEMP1_HIGH, "temp1_high", SENSORS_ADM1025_TEMP1,
                         SENSORS_ADM1025_TEMP1, SENSORS_MODE_RW, 
                         ADM1025_SYSCTL_RTEMP, VALUE(1), 1 },
    { SENSORS_ADM1025_TEMP2, "temp2", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, ADM1025_SYSCTL_TEMP, VALUE(3), 1 },
    { SENSORS_ADM1025_TEMP2_LOW, "temp2_low", SENSORS_ADM1025_TEMP2,
                         SENSORS_ADM1025_TEMP2, SENSORS_MODE_RW, 
                         ADM1025_SYSCTL_TEMP, VALUE(2), 1 },
    { SENSORS_ADM1025_TEMP2_HIGH, "temp2_high", SENSORS_ADM1025_TEMP2,
                         SENSORS_ADM1025_TEMP2, SENSORS_MODE_RW, 
                         ADM1025_SYSCTL_TEMP, VALUE(1), 1 },
    { SENSORS_ADM1025_VID, "vid", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, ADM1025_SYSCTL_VID, VALUE(1), 3 },
    { SENSORS_ADM1025_VRM, "vrm", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_RW, ADM1025_SYSCTL_VRM, VALUE(1), 1 },
    { SENSORS_ADM1025_ALARMS, "alarms", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING, 
                         SENSORS_MODE_R, ADM1025_SYSCTL_ALARMS, VALUE(1), 0 },
    { 0 }
  };


static sensors_chip_feature adm1026_features[] = {
    { SENSORS_ADM1026_ALARMS, "alarms",
		SENSORS_NO_MAPPING, SENSORS_NO_MAPPING, SENSORS_MODE_R,
		ADM1026_SYSCTL_ALARMS, VALUE(1), 0 },
    { SENSORS_ADM1026_ALARM_MASK, "alarm_mask",
		SENSORS_ADM1026_ALARMS, SENSORS_NO_MAPPING, SENSORS_MODE_R,
		ADM1026_SYSCTL_ALARM_MASK, VALUE(1), 0 },
    { SENSORS_ADM1026_GPIO, "gpio",
		SENSORS_NO_MAPPING, SENSORS_NO_MAPPING, SENSORS_MODE_R,
		ADM1026_SYSCTL_GPIO, VALUE(1), 0 },
    { SENSORS_ADM1026_GPIO_MASK, "gpio_mask",
		SENSORS_ADM1026_GPIO, SENSORS_NO_MAPPING, SENSORS_MODE_R,
		ADM1026_SYSCTL_GPIO_MASK, VALUE(1), 0 },
    { SENSORS_ADM1026_VID, "vid",
		SENSORS_NO_MAPPING, SENSORS_NO_MAPPING, SENSORS_MODE_RW,
		ADM1026_SYSCTL_VID, VALUE(1), 3 },
    { SENSORS_ADM1026_VRM, "vrm",
		SENSORS_NO_MAPPING, SENSORS_NO_MAPPING, SENSORS_MODE_RW,
		ADM1026_SYSCTL_VRM, VALUE(1), 1 },
    { SENSORS_ADM1026_PWM, "pwm",
		SENSORS_NO_MAPPING, SENSORS_NO_MAPPING, SENSORS_MODE_RW,
		ADM1026_SYSCTL_PWM, VALUE(1), 0 },
    { SENSORS_ADM1026_AFC_PWM, "afc_pwm",
		SENSORS_NO_MAPPING, SENSORS_NO_MAPPING, SENSORS_MODE_RW,
		ADM1026_SYSCTL_PWM, VALUE(1), 0 },
    { SENSORS_ADM1026_DAC, "analog_out",
		SENSORS_NO_MAPPING, SENSORS_NO_MAPPING, SENSORS_MODE_RW,
		ADM1026_SYSCTL_ANALOG_OUT, VALUE(1), 0 },
    { SENSORS_ADM1026_AFC_DAC, "afc_analog_out",
		SENSORS_ADM1026_DAC, SENSORS_NO_MAPPING, SENSORS_MODE_RW,
		ADM1026_SYSCTL_AFC, VALUE(2), 0 },

    { SENSORS_ADM1026_IN0, "in0",
		SENSORS_NO_MAPPING, SENSORS_NO_MAPPING, SENSORS_MODE_R,
		ADM1026_SYSCTL_IN0, VALUE(3), 3 },
    { SENSORS_ADM1026_IN0_MIN, "in0_min",
		SENSORS_ADM1026_IN0, SENSORS_ADM1026_IN0, SENSORS_MODE_RW,
		ADM1026_SYSCTL_IN0, VALUE(1), 3 },
    { SENSORS_ADM1026_IN0_MAX, "in0_max",
		SENSORS_ADM1026_IN0, SENSORS_ADM1026_IN0, SENSORS_MODE_RW,
		ADM1026_SYSCTL_IN0, VALUE(2), 3 },
    { SENSORS_ADM1026_IN1, "in1",
		SENSORS_NO_MAPPING, SENSORS_NO_MAPPING, SENSORS_MODE_R,
		ADM1026_SYSCTL_IN1, VALUE(3), 3 },
    { SENSORS_ADM1026_IN1_MIN, "in1_min",
		SENSORS_ADM1026_IN1, SENSORS_ADM1026_IN1, SENSORS_MODE_RW,
		ADM1026_SYSCTL_IN1, VALUE(1), 3 },
    { SENSORS_ADM1026_IN1_MAX, "in1_max",
		SENSORS_ADM1026_IN1, SENSORS_ADM1026_IN1, SENSORS_MODE_RW,
		ADM1026_SYSCTL_IN1, VALUE(2), 3 },
    { SENSORS_ADM1026_IN2, "in2",
		SENSORS_NO_MAPPING, SENSORS_NO_MAPPING, SENSORS_MODE_R,
		ADM1026_SYSCTL_IN2, VALUE(3), 3 },
    { SENSORS_ADM1026_IN2_MIN, "in2_min",
		SENSORS_ADM1026_IN2, SENSORS_ADM1026_IN2, SENSORS_MODE_RW,
		ADM1026_SYSCTL_IN2, VALUE(1), 3 },
    { SENSORS_ADM1026_IN2_MAX, "in2_max",
		SENSORS_ADM1026_IN2, SENSORS_ADM1026_IN2, SENSORS_MODE_RW,
		ADM1026_SYSCTL_IN2, VALUE(2), 3 },
    { SENSORS_ADM1026_IN3, "in3",
		SENSORS_NO_MAPPING, SENSORS_NO_MAPPING, SENSORS_MODE_R,
		ADM1026_SYSCTL_IN3, VALUE(3), 3 },
    { SENSORS_ADM1026_IN3_MIN, "in3_min",
		SENSORS_ADM1026_IN3, SENSORS_ADM1026_IN3, SENSORS_MODE_RW,
		ADM1026_SYSCTL_IN3, VALUE(1), 3 },
    { SENSORS_ADM1026_IN3_MAX, "in3_max",
		SENSORS_ADM1026_IN3, SENSORS_ADM1026_IN3, SENSORS_MODE_RW,
		ADM1026_SYSCTL_IN3, VALUE(2), 3 },
    { SENSORS_ADM1026_IN4, "in4",
		SENSORS_NO_MAPPING, SENSORS_NO_MAPPING, SENSORS_MODE_R,
		ADM1026_SYSCTL_IN4, VALUE(3), 3 },
    { SENSORS_ADM1026_IN4_MIN, "in4_min",
		SENSORS_ADM1026_IN4, SENSORS_ADM1026_IN4, SENSORS_MODE_RW,
		ADM1026_SYSCTL_IN4, VALUE(1), 3 },
    { SENSORS_ADM1026_IN4_MAX, "in4_max",
		SENSORS_ADM1026_IN4, SENSORS_ADM1026_IN4, SENSORS_MODE_RW,
		ADM1026_SYSCTL_IN4, VALUE(2), 3 },
    { SENSORS_ADM1026_IN5, "in5",
		SENSORS_NO_MAPPING, SENSORS_NO_MAPPING, SENSORS_MODE_R,
		ADM1026_SYSCTL_IN5, VALUE(3), 3 },
    { SENSORS_ADM1026_IN5_MIN, "in5_min",
		SENSORS_ADM1026_IN5, SENSORS_ADM1026_IN5, SENSORS_MODE_RW,
		ADM1026_SYSCTL_IN5, VALUE(1), 3 },
    { SENSORS_ADM1026_IN5_MAX, "in5_max",
		SENSORS_ADM1026_IN5, SENSORS_ADM1026_IN5, SENSORS_MODE_RW,
		ADM1026_SYSCTL_IN5, VALUE(2), 3 },
    { SENSORS_ADM1026_IN6, "in6",
		SENSORS_NO_MAPPING, SENSORS_NO_MAPPING, SENSORS_MODE_R,
		ADM1026_SYSCTL_IN6, VALUE(3), 3 },
    { SENSORS_ADM1026_IN6_MIN, "in6_min",
		SENSORS_ADM1026_IN6, SENSORS_ADM1026_IN6, SENSORS_MODE_RW,
		ADM1026_SYSCTL_IN6, VALUE(1), 3 },
    { SENSORS_ADM1026_IN6_MAX, "in6_max",
		SENSORS_ADM1026_IN6, SENSORS_ADM1026_IN6, SENSORS_MODE_RW,
		ADM1026_SYSCTL_IN6, VALUE(2), 3 },
    { SENSORS_ADM1026_IN7, "in7",
		SENSORS_NO_MAPPING, SENSORS_NO_MAPPING, SENSORS_MODE_R,
		ADM1026_SYSCTL_IN7, VALUE(3), 3 },
    { SENSORS_ADM1026_IN7_MIN, "in7_min",
		SENSORS_ADM1026_IN7, SENSORS_ADM1026_IN7, SENSORS_MODE_RW,
		ADM1026_SYSCTL_IN7, VALUE(1), 3 },
    { SENSORS_ADM1026_IN7_MAX, "in7_max",
		SENSORS_ADM1026_IN7, SENSORS_ADM1026_IN7, SENSORS_MODE_RW,
		ADM1026_SYSCTL_IN7, VALUE(2), 3 },
    { SENSORS_ADM1026_IN8, "in8",
		SENSORS_NO_MAPPING, SENSORS_NO_MAPPING, SENSORS_MODE_R,
		ADM1026_SYSCTL_IN8, VALUE(3), 3 },
    { SENSORS_ADM1026_IN8_MIN, "in8_min",
		SENSORS_ADM1026_IN8, SENSORS_ADM1026_IN8, SENSORS_MODE_RW,
		ADM1026_SYSCTL_IN8, VALUE(1), 3 },
    { SENSORS_ADM1026_IN8_MAX, "in8_max",
		SENSORS_ADM1026_IN8, SENSORS_ADM1026_IN8, SENSORS_MODE_RW,
		ADM1026_SYSCTL_IN8, VALUE(2), 3 },
    { SENSORS_ADM1026_IN9, "in9",
		SENSORS_NO_MAPPING, SENSORS_NO_MAPPING, SENSORS_MODE_R,
		ADM1026_SYSCTL_IN9, VALUE(3), 3 },
    { SENSORS_ADM1026_IN9_MIN, "in9_min",
		SENSORS_ADM1026_IN9, SENSORS_ADM1026_IN9, SENSORS_MODE_RW,
		ADM1026_SYSCTL_IN9, VALUE(1), 3 },
    { SENSORS_ADM1026_IN9_MAX, "in9_max",
		SENSORS_ADM1026_IN9, SENSORS_ADM1026_IN9, SENSORS_MODE_RW,
		ADM1026_SYSCTL_IN9, VALUE(2), 3 },
    { SENSORS_ADM1026_IN10, "in10",
		SENSORS_NO_MAPPING, SENSORS_NO_MAPPING, SENSORS_MODE_R,
		ADM1026_SYSCTL_IN10, VALUE(3), 3 },
    { SENSORS_ADM1026_IN10_MIN, "in10_min",
		SENSORS_ADM1026_IN10, SENSORS_ADM1026_IN10, SENSORS_MODE_RW,
		ADM1026_SYSCTL_IN10, VALUE(1), 3 },
    { SENSORS_ADM1026_IN10_MAX, "in10_max",
		SENSORS_ADM1026_IN10, SENSORS_ADM1026_IN10, SENSORS_MODE_RW,
		ADM1026_SYSCTL_IN10, VALUE(2), 3 },
    { SENSORS_ADM1026_IN11, "in11",
		SENSORS_NO_MAPPING, SENSORS_NO_MAPPING, SENSORS_MODE_R,
		ADM1026_SYSCTL_IN11, VALUE(3), 3 },
    { SENSORS_ADM1026_IN11_MIN, "in11_min",
		SENSORS_ADM1026_IN11, SENSORS_ADM1026_IN11, SENSORS_MODE_RW,
		ADM1026_SYSCTL_IN11, VALUE(1), 3 },
    { SENSORS_ADM1026_IN11_MAX, "in11_max",
		SENSORS_ADM1026_IN11, SENSORS_ADM1026_IN11, SENSORS_MODE_RW,
		ADM1026_SYSCTL_IN11, VALUE(2), 3 },
    { SENSORS_ADM1026_IN12, "in12",
		SENSORS_NO_MAPPING, SENSORS_NO_MAPPING, SENSORS_MODE_R,
		ADM1026_SYSCTL_IN12, VALUE(3), 3 },
    { SENSORS_ADM1026_IN12_MIN, "in12_min",
		SENSORS_ADM1026_IN12, SENSORS_ADM1026_IN12, SENSORS_MODE_RW,
		ADM1026_SYSCTL_IN12, VALUE(1), 3 },
    { SENSORS_ADM1026_IN12_MAX, "in12_max",
		SENSORS_ADM1026_IN12, SENSORS_ADM1026_IN12, SENSORS_MODE_RW,
		ADM1026_SYSCTL_IN12, VALUE(2), 3 },
    { SENSORS_ADM1026_IN13, "in13",
		SENSORS_NO_MAPPING, SENSORS_NO_MAPPING, SENSORS_MODE_R,
		ADM1026_SYSCTL_IN13, VALUE(3), 3 },
    { SENSORS_ADM1026_IN13_MIN, "in13_min",
		SENSORS_ADM1026_IN13, SENSORS_ADM1026_IN13, SENSORS_MODE_RW,
		ADM1026_SYSCTL_IN13, VALUE(1), 3 },
    { SENSORS_ADM1026_IN13_MAX, "in13_max",
		SENSORS_ADM1026_IN13, SENSORS_ADM1026_IN13, SENSORS_MODE_RW,
		ADM1026_SYSCTL_IN13, VALUE(2), 3 },
    { SENSORS_ADM1026_IN14, "in14",
		SENSORS_NO_MAPPING, SENSORS_NO_MAPPING, SENSORS_MODE_R,
		ADM1026_SYSCTL_IN14, VALUE(3), 3 },
    { SENSORS_ADM1026_IN14_MIN, "in14_min",
		SENSORS_ADM1026_IN14, SENSORS_ADM1026_IN14, SENSORS_MODE_RW,
		ADM1026_SYSCTL_IN14, VALUE(1), 3 },
    { SENSORS_ADM1026_IN14_MAX, "in14_max",
		SENSORS_ADM1026_IN14, SENSORS_ADM1026_IN14, SENSORS_MODE_RW,
		ADM1026_SYSCTL_IN14, VALUE(2), 3 },
    { SENSORS_ADM1026_IN15, "in15",
		SENSORS_NO_MAPPING, SENSORS_NO_MAPPING, SENSORS_MODE_R,
		ADM1026_SYSCTL_IN15, VALUE(3), 3 },
    { SENSORS_ADM1026_IN15_MIN, "in15_min",
		SENSORS_ADM1026_IN15, SENSORS_ADM1026_IN15, SENSORS_MODE_RW,
		ADM1026_SYSCTL_IN15, VALUE(1), 3 },
    { SENSORS_ADM1026_IN15_MAX, "in15_max",
		SENSORS_ADM1026_IN15, SENSORS_ADM1026_IN15, SENSORS_MODE_RW,
		ADM1026_SYSCTL_IN15, VALUE(2), 3 },
    { SENSORS_ADM1026_IN16, "in16",
		SENSORS_NO_MAPPING, SENSORS_NO_MAPPING, SENSORS_MODE_R,
		ADM1026_SYSCTL_IN16, VALUE(3), 3 },
    { SENSORS_ADM1026_IN16_MIN, "in16_min",
		SENSORS_ADM1026_IN16, SENSORS_ADM1026_IN16, SENSORS_MODE_RW,
		ADM1026_SYSCTL_IN16, VALUE(1), 3 },
    { SENSORS_ADM1026_IN16_MAX, "in16_max",
		SENSORS_ADM1026_IN16, SENSORS_ADM1026_IN16, SENSORS_MODE_RW,
		ADM1026_SYSCTL_IN16, VALUE(2), 3 },
    { SENSORS_ADM1026_FAN0, "fan0",
		SENSORS_NO_MAPPING, SENSORS_NO_MAPPING, SENSORS_MODE_R,
		ADM1026_SYSCTL_FAN0, VALUE(2), 0 },
    { SENSORS_ADM1026_FAN0_MIN, "fan0_min",
		SENSORS_ADM1026_FAN0, SENSORS_ADM1026_FAN0, SENSORS_MODE_RW,
		ADM1026_SYSCTL_FAN0, VALUE(1), 0 },
    { SENSORS_ADM1026_FAN0_DIV, "fan0_div",
		SENSORS_ADM1026_FAN0, SENSORS_NO_MAPPING, SENSORS_MODE_RW,
		ADM1026_SYSCTL_FAN_DIV, VALUE(1), 0 },
    { SENSORS_ADM1026_FAN1, "fan1",
		SENSORS_NO_MAPPING, SENSORS_NO_MAPPING, SENSORS_MODE_R,
		ADM1026_SYSCTL_FAN1, VALUE(2), 0 },
    { SENSORS_ADM1026_FAN1_MIN, "fan1_min",
		SENSORS_ADM1026_FAN1, SENSORS_ADM1026_FAN1, SENSORS_MODE_RW,
		ADM1026_SYSCTL_FAN1, VALUE(1), 0 },
    { SENSORS_ADM1026_FAN1_DIV, "fan1_div",
		SENSORS_ADM1026_FAN1, SENSORS_NO_MAPPING, SENSORS_MODE_RW,
		ADM1026_SYSCTL_FAN_DIV, VALUE(2), 0 },
    { SENSORS_ADM1026_FAN2, "fan2",
		SENSORS_NO_MAPPING, SENSORS_NO_MAPPING, SENSORS_MODE_R,
		ADM1026_SYSCTL_FAN2, VALUE(2), 0 },
    { SENSORS_ADM1026_FAN2_MIN, "fan2_min",
		SENSORS_ADM1026_FAN2, SENSORS_ADM1026_FAN2, SENSORS_MODE_RW,
		ADM1026_SYSCTL_FAN2, VALUE(1), 0 },
    { SENSORS_ADM1026_FAN2_DIV, "fan2_div",
		SENSORS_ADM1026_FAN2, SENSORS_NO_MAPPING, SENSORS_MODE_RW,
		ADM1026_SYSCTL_FAN_DIV, VALUE(3), 0 },
    { SENSORS_ADM1026_FAN3, "fan3",
		SENSORS_NO_MAPPING, SENSORS_NO_MAPPING, SENSORS_MODE_R,
		ADM1026_SYSCTL_FAN3, VALUE(2), 0 },
    { SENSORS_ADM1026_FAN3_MIN, "fan3_min",
		SENSORS_ADM1026_FAN3, SENSORS_ADM1026_FAN3, SENSORS_MODE_RW,
		ADM1026_SYSCTL_FAN3, VALUE(1), 0 },
    { SENSORS_ADM1026_FAN3_DIV, "fan3_div",
		SENSORS_ADM1026_FAN3, SENSORS_NO_MAPPING, SENSORS_MODE_RW,
		ADM1026_SYSCTL_FAN_DIV, VALUE(4), 0 },
    { SENSORS_ADM1026_FAN4, "fan4",
		SENSORS_NO_MAPPING, SENSORS_NO_MAPPING, SENSORS_MODE_R,
		ADM1026_SYSCTL_FAN4, VALUE(2), 0 },
    { SENSORS_ADM1026_FAN4_MIN, "fan4_min",
		SENSORS_ADM1026_FAN4, SENSORS_ADM1026_FAN4, SENSORS_MODE_RW,
		ADM1026_SYSCTL_FAN4, VALUE(1), 0 },
    { SENSORS_ADM1026_FAN4_DIV, "fan4_div",
		SENSORS_ADM1026_FAN4, SENSORS_NO_MAPPING, SENSORS_MODE_RW,
		ADM1026_SYSCTL_FAN_DIV, VALUE(5), 0 },
    { SENSORS_ADM1026_FAN5, "fan5",
		SENSORS_NO_MAPPING, SENSORS_NO_MAPPING, SENSORS_MODE_R,
		ADM1026_SYSCTL_FAN5, VALUE(2), 0 },
    { SENSORS_ADM1026_FAN5_MIN, "fan5_min",
		SENSORS_ADM1026_FAN5, SENSORS_ADM1026_FAN5, SENSORS_MODE_RW,
		ADM1026_SYSCTL_FAN5, VALUE(1), 0 },
    { SENSORS_ADM1026_FAN5_DIV, "fan5_div",
		SENSORS_ADM1026_FAN5, SENSORS_NO_MAPPING, SENSORS_MODE_RW,
		ADM1026_SYSCTL_FAN_DIV, VALUE(6), 0 },
    { SENSORS_ADM1026_FAN6, "fan6",
		SENSORS_NO_MAPPING, SENSORS_NO_MAPPING, SENSORS_MODE_R,
		ADM1026_SYSCTL_FAN6, VALUE(2), 0 },
    { SENSORS_ADM1026_FAN6_MIN, "fan6_min",
		SENSORS_ADM1026_FAN6, SENSORS_ADM1026_FAN6, SENSORS_MODE_RW,
		ADM1026_SYSCTL_FAN6, VALUE(1), 0 },
    { SENSORS_ADM1026_FAN6_DIV, "fan6_div",
		SENSORS_ADM1026_FAN6, SENSORS_NO_MAPPING, SENSORS_MODE_RW,
		ADM1026_SYSCTL_FAN_DIV, VALUE(7), 0 },
    { SENSORS_ADM1026_FAN7, "fan7",
		SENSORS_NO_MAPPING, SENSORS_NO_MAPPING, SENSORS_MODE_R,
		ADM1026_SYSCTL_FAN7, VALUE(2), 0 },
    { SENSORS_ADM1026_FAN7_MIN, "fan7_min",
		SENSORS_ADM1026_FAN7, SENSORS_ADM1026_FAN7, SENSORS_MODE_RW,
		ADM1026_SYSCTL_FAN7, VALUE(1), 0 },
    { SENSORS_ADM1026_FAN7_DIV, "fan7_div",
		SENSORS_ADM1026_FAN7, SENSORS_NO_MAPPING, SENSORS_MODE_RW,
		ADM1026_SYSCTL_FAN_DIV, VALUE(8), 0 },
    { SENSORS_ADM1026_TEMP1, "temp1",
		SENSORS_NO_MAPPING, SENSORS_NO_MAPPING, SENSORS_MODE_R,
		ADM1026_SYSCTL_TEMP1, VALUE(3), 0 },
    { SENSORS_ADM1026_TEMP1_MIN, "temp1_min",
		SENSORS_ADM1026_TEMP1, SENSORS_ADM1026_TEMP1, SENSORS_MODE_RW,
		ADM1026_SYSCTL_TEMP1, VALUE(1), 0 },
    { SENSORS_ADM1026_TEMP1_MAX, "temp1_max",
		SENSORS_ADM1026_TEMP1, SENSORS_ADM1026_TEMP1, SENSORS_MODE_RW,
		ADM1026_SYSCTL_TEMP1, VALUE(2), 0 },
    { SENSORS_ADM1026_TEMP1_OFFSET, "temp1_offset",
		SENSORS_ADM1026_TEMP1, SENSORS_NO_MAPPING, SENSORS_MODE_RW,
		ADM1026_SYSCTL_TEMP_OFFSET1, VALUE(1), 0 },
    { SENSORS_ADM1026_TEMP1_TMIN, "temp1_tmin",
		SENSORS_ADM1026_TEMP1, SENSORS_ADM1026_TEMP1, SENSORS_MODE_RW,
		ADM1026_SYSCTL_TEMP_TMIN1, VALUE(1), 0 },
    { SENSORS_ADM1026_TEMP1_THERM, "temp1_therm",
		SENSORS_ADM1026_TEMP1, SENSORS_ADM1026_TEMP1, SENSORS_MODE_RW,
		ADM1026_SYSCTL_TEMP_THERM1, VALUE(1), 0 },
    { SENSORS_ADM1026_TEMP2, "temp2",
		SENSORS_NO_MAPPING, SENSORS_NO_MAPPING, SENSORS_MODE_R,
		ADM1026_SYSCTL_TEMP2, VALUE(3), 0 },
    { SENSORS_ADM1026_TEMP2_MIN, "temp2_min",
		SENSORS_ADM1026_TEMP2, SENSORS_ADM1026_TEMP2, SENSORS_MODE_RW,
		ADM1026_SYSCTL_TEMP2, VALUE(1), 0 },
    { SENSORS_ADM1026_TEMP2_MAX, "temp2_max",
		SENSORS_ADM1026_TEMP2, SENSORS_ADM1026_TEMP2, SENSORS_MODE_RW,
		ADM1026_SYSCTL_TEMP2, VALUE(2), 0 },
    { SENSORS_ADM1026_TEMP2_OFFSET, "temp2_offset",
		SENSORS_ADM1026_TEMP2, SENSORS_NO_MAPPING, SENSORS_MODE_RW,
		ADM1026_SYSCTL_TEMP_OFFSET2, VALUE(1), 0 },
    { SENSORS_ADM1026_TEMP2_TMIN, "temp2_tmin",
		SENSORS_ADM1026_TEMP2, SENSORS_ADM1026_TEMP2, SENSORS_MODE_RW,
		ADM1026_SYSCTL_TEMP_TMIN2, VALUE(1), 0 },
    { SENSORS_ADM1026_TEMP2_THERM, "temp2_therm",
		SENSORS_ADM1026_TEMP2, SENSORS_ADM1026_TEMP2, SENSORS_MODE_RW,
		ADM1026_SYSCTL_TEMP_THERM2, VALUE(1), 0 },
    { SENSORS_ADM1026_TEMP3, "temp3",
		SENSORS_NO_MAPPING, SENSORS_NO_MAPPING, SENSORS_MODE_R,
		ADM1026_SYSCTL_TEMP3, VALUE(3), 0 },
    { SENSORS_ADM1026_TEMP3_MIN, "temp3_min",
		SENSORS_ADM1026_TEMP3, SENSORS_ADM1026_TEMP3, SENSORS_MODE_RW,
		ADM1026_SYSCTL_TEMP3, VALUE(1), 0 },
    { SENSORS_ADM1026_TEMP3_MAX, "temp3_max",
		SENSORS_ADM1026_TEMP3, SENSORS_ADM1026_TEMP3, SENSORS_MODE_RW,
		ADM1026_SYSCTL_TEMP3, VALUE(2), 0 },
    { SENSORS_ADM1026_TEMP3_OFFSET, "temp3_offset",
		SENSORS_ADM1026_TEMP3, SENSORS_NO_MAPPING, SENSORS_MODE_RW,
		ADM1026_SYSCTL_TEMP_OFFSET3, VALUE(1), 0 },
    { SENSORS_ADM1026_TEMP3_TMIN, "temp3_tmin",
		SENSORS_ADM1026_TEMP3, SENSORS_ADM1026_TEMP3, SENSORS_MODE_RW,
		ADM1026_SYSCTL_TEMP_TMIN3, VALUE(1), 0 },
    { SENSORS_ADM1026_TEMP3_THERM, "temp3_therm",
		SENSORS_ADM1026_TEMP3, SENSORS_ADM1026_TEMP3, SENSORS_MODE_RW,
		ADM1026_SYSCTL_TEMP_THERM3, VALUE(1), 0 },
    { 0 }
  };


static sensors_chip_feature via686a_features[] =
  { 
    { SENSORS_VIA686A_IN0, "2.0V", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING, 
                        SENSORS_MODE_R, VIA686A_SYSCTL_IN0, VALUE(3), 2 },
    { SENSORS_VIA686A_IN1, "2.5V", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, VIA686A_SYSCTL_IN1, VALUE(3), 2 },
    { SENSORS_VIA686A_IN2, "3.3V", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, VIA686A_SYSCTL_IN2, VALUE(3), 2 },
    { SENSORS_VIA686A_IN3, "5.0V", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, VIA686A_SYSCTL_IN3, VALUE(3), 2 },
    { SENSORS_VIA686A_IN4, "12V", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, VIA686A_SYSCTL_IN4, VALUE(3), 2 },
    { SENSORS_VIA686A_IN0_MIN, "in0_min", SENSORS_VIA686A_IN0,
                            SENSORS_VIA686A_IN0,
                            SENSORS_MODE_RW, VIA686A_SYSCTL_IN0, VALUE(1), 2 },
    { SENSORS_VIA686A_IN1_MIN, "in1_min", SENSORS_VIA686A_IN1,
                            SENSORS_VIA686A_IN1,
                            SENSORS_MODE_RW, VIA686A_SYSCTL_IN1, VALUE(1), 2 },
    { SENSORS_VIA686A_IN2_MIN, "in2_min", SENSORS_VIA686A_IN2,
                            SENSORS_VIA686A_IN2,
                            SENSORS_MODE_RW, VIA686A_SYSCTL_IN2, VALUE(1), 2 },
    { SENSORS_VIA686A_IN3_MIN, "in3_min", SENSORS_VIA686A_IN3,
                            SENSORS_VIA686A_IN3,
                            SENSORS_MODE_RW, VIA686A_SYSCTL_IN3, VALUE(1), 2 },
    { SENSORS_VIA686A_IN4_MIN, "in4_min", SENSORS_VIA686A_IN4,
                            SENSORS_VIA686A_IN4,
                            SENSORS_MODE_RW, VIA686A_SYSCTL_IN4, VALUE(1), 2 },
    { SENSORS_VIA686A_IN0_MAX, "in0_max", SENSORS_VIA686A_IN0,
                            SENSORS_VIA686A_IN0,
                            SENSORS_MODE_RW, VIA686A_SYSCTL_IN0, VALUE(2), 2 },
    { SENSORS_VIA686A_IN1_MAX, "in1_max", SENSORS_VIA686A_IN1,
                            SENSORS_VIA686A_IN1,
                            SENSORS_MODE_RW, VIA686A_SYSCTL_IN1, VALUE(2), 2 },
    { SENSORS_VIA686A_IN2_MAX, "in2_max", SENSORS_VIA686A_IN2,
                            SENSORS_VIA686A_IN2,
                            SENSORS_MODE_RW, VIA686A_SYSCTL_IN2, VALUE(2), 2 },
    { SENSORS_VIA686A_IN3_MAX, "in3_max", SENSORS_VIA686A_IN3,
                            SENSORS_VIA686A_IN3,
                            SENSORS_MODE_RW, VIA686A_SYSCTL_IN3, VALUE(2), 2 },
    { SENSORS_VIA686A_IN4_MAX, "in4_max", SENSORS_VIA686A_IN4,
                            SENSORS_VIA686A_IN4,
                            SENSORS_MODE_RW, VIA686A_SYSCTL_IN4, VALUE(2), 2 },
    { SENSORS_VIA686A_FAN1, "fan1", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, VIA686A_SYSCTL_FAN1, VALUE(2), 0 },
    { SENSORS_VIA686A_FAN2, "fan2", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, VIA686A_SYSCTL_FAN2, VALUE(2), 0 },
    { SENSORS_VIA686A_FAN1_MIN, "fan1_min", SENSORS_VIA686A_FAN1,
                             SENSORS_VIA686A_FAN1,
                             SENSORS_MODE_RW, VIA686A_SYSCTL_FAN1, VALUE(1), 0 },
    { SENSORS_VIA686A_FAN2_MIN, "fan2_min", SENSORS_VIA686A_FAN2,
                             SENSORS_VIA686A_FAN2,
                             SENSORS_MODE_RW, VIA686A_SYSCTL_FAN2, VALUE(1), 0 },
    { SENSORS_VIA686A_TEMP, "temp1", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, VIA686A_SYSCTL_TEMP, VALUE(3), 1 },
    { SENSORS_VIA686A_TEMP_HYST, "temp1_hyst", SENSORS_VIA686A_TEMP,
                              SENSORS_VIA686A_TEMP, SENSORS_MODE_RW, 
                              VIA686A_SYSCTL_TEMP, VALUE(2), 1 },
    { SENSORS_VIA686A_TEMP_OVER, "temp1_over", SENSORS_VIA686A_TEMP,
                              SENSORS_VIA686A_TEMP, SENSORS_MODE_RW, 
                              VIA686A_SYSCTL_TEMP, VALUE(1), 1 },
    { SENSORS_VIA686A_TEMP2, "temp2", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, VIA686A_SYSCTL_TEMP2, VALUE(3), 1 },
    { SENSORS_VIA686A_TEMP2_HYST, "temp2_hyst", SENSORS_VIA686A_TEMP2,
                              SENSORS_VIA686A_TEMP2, SENSORS_MODE_RW, 
                              VIA686A_SYSCTL_TEMP2, VALUE(2), 1 },
    { SENSORS_VIA686A_TEMP2_OVER, "temp2_over", SENSORS_VIA686A_TEMP2,
                              SENSORS_VIA686A_TEMP2, SENSORS_MODE_RW, 
                              VIA686A_SYSCTL_TEMP2, VALUE(1), 1 },
    { SENSORS_VIA686A_TEMP3, "temp3", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, VIA686A_SYSCTL_TEMP3, VALUE(3), 1 },
    { SENSORS_VIA686A_TEMP3_HYST, "temp3_hyst", SENSORS_VIA686A_TEMP3,
                              SENSORS_VIA686A_TEMP3, SENSORS_MODE_RW, 
                              VIA686A_SYSCTL_TEMP3, VALUE(2), 1 },
    { SENSORS_VIA686A_TEMP3_OVER, "temp3_over", SENSORS_VIA686A_TEMP3,
                              SENSORS_VIA686A_TEMP3, SENSORS_MODE_RW, 
                              VIA686A_SYSCTL_TEMP3, VALUE(1), 1 },
    { SENSORS_VIA686A_FAN1_DIV, "fan1_div", SENSORS_VIA686A_FAN1,
                             SENSORS_NO_MAPPING,
                             SENSORS_MODE_RW, VIA686A_SYSCTL_FAN_DIV, VALUE(1), 
                             0 },
    { SENSORS_VIA686A_FAN2_DIV, "fan2_div", SENSORS_VIA686A_FAN2,
                             SENSORS_NO_MAPPING,
                             SENSORS_MODE_RW, VIA686A_SYSCTL_FAN_DIV, VALUE(2), 
                             0 },
    { SENSORS_VIA686A_ALARMS, "alarms", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING, 
                           SENSORS_MODE_R, VIA686A_SYSCTL_ALARMS, VALUE(1), 0 },
    { 0 }
  };

static sensors_chip_feature ddcmon_features[] =
  {
    { SENSORS_DDCMON_MAN_ID, "Manufacturer ID",
                         SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, DDCMON_SYSCTL_ID, VALUE(1), 0 },
    { SENSORS_DDCMON_PROD_ID, "Model Number",
                         SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, DDCMON_SYSCTL_ID, VALUE(2), 0 },
    { SENSORS_DDCMON_VERSIZE, "Monitor Size (cm)",
                         SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, DDCMON_SYSCTL_SIZE, VALUE(1), 0 },
    { SENSORS_DDCMON_HORSIZE, "hsize",
                         SENSORS_DDCMON_VERSIZE, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, DDCMON_SYSCTL_SIZE, VALUE(2), 0 },
    { SENSORS_DDCMON_VERSYNCMIN, "Vertical Sync (Hz)",
                         SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, DDCMON_SYSCTL_SYNC, VALUE(1), 0 },
    { SENSORS_DDCMON_VERSYNCMAX, "v_sync_max",
                         SENSORS_DDCMON_VERSYNCMIN, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, DDCMON_SYSCTL_SYNC, VALUE(2), 0 },
    { SENSORS_DDCMON_HORSYNCMIN, "Horizontal Sync (KHz)",
                         SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, DDCMON_SYSCTL_SYNC, VALUE(3), 0 },
    { SENSORS_DDCMON_HORSYNCMAX, "h_sync_max",
                         SENSORS_DDCMON_HORSYNCMIN, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, DDCMON_SYSCTL_SYNC, VALUE(4), 0 },
    { SENSORS_DDCMON_TIMINGS, "Established Timings",
                         SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, DDCMON_SYSCTL_TIMINGS, VALUE(1), 0 },
    { SENSORS_DDCMON_SERIAL, "Serial Number",
                         SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, DDCMON_SYSCTL_SERIAL, VALUE(1), 0 },
    { SENSORS_DDCMON_YEAR, "Manufacture Time",
                         SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, DDCMON_SYSCTL_TIME, VALUE(1), 0 },
    { SENSORS_DDCMON_WEEK, "week",
                         SENSORS_DDCMON_YEAR, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, DDCMON_SYSCTL_TIME, VALUE(2), 0 },
    { SENSORS_DDCMON_EDID_VER, "EDID Version",
                         SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, DDCMON_SYSCTL_EDID, VALUE(1), 0 },
    { SENSORS_DDCMON_EDID_REV, "revision",
                         SENSORS_DDCMON_EDID_VER, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, DDCMON_SYSCTL_EDID, VALUE(2), 0 },
    { SENSORS_DDCMON_GAMMA, "Gamma Factor",
                         SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, DDCMON_SYSCTL_GAMMA, VALUE(1), 2 },
    { SENSORS_DDCMON_DPMS, "DPMS Modes",
                         SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, DDCMON_SYSCTL_DPMS, VALUE(1), 0 },
    { SENSORS_DDCMON_TIMING1_HOR, "Standard Timing 1",
                         SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, DDCMON_SYSCTL_TIMING1, VALUE(1), 0 },
    { SENSORS_DDCMON_TIMING1_VER, "vertical",
                         SENSORS_DDCMON_TIMING1_HOR, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, DDCMON_SYSCTL_TIMING1, VALUE(2), 0 },
    { SENSORS_DDCMON_TIMING1_REF, "refresh",
                         SENSORS_DDCMON_TIMING1_HOR, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, DDCMON_SYSCTL_TIMING1, VALUE(3), 0 },
    { SENSORS_DDCMON_TIMING2_HOR, "Standard Timing 2",
                         SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, DDCMON_SYSCTL_TIMING2, VALUE(1), 0 },
    { SENSORS_DDCMON_TIMING2_VER, "vertical",
                         SENSORS_DDCMON_TIMING2_HOR, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, DDCMON_SYSCTL_TIMING2, VALUE(2), 0 },
    { SENSORS_DDCMON_TIMING2_REF, "refresh",
                         SENSORS_DDCMON_TIMING2_HOR, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, DDCMON_SYSCTL_TIMING2, VALUE(3), 0 },
    { SENSORS_DDCMON_TIMING3_HOR, "Standard Timing 3",
                         SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, DDCMON_SYSCTL_TIMING3, VALUE(1), 0 },
    { SENSORS_DDCMON_TIMING3_VER, "vertical",
                         SENSORS_DDCMON_TIMING3_HOR, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, DDCMON_SYSCTL_TIMING3, VALUE(2), 0 },
    { SENSORS_DDCMON_TIMING3_REF, "refresh",
                         SENSORS_DDCMON_TIMING3_HOR, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, DDCMON_SYSCTL_TIMING3, VALUE(3), 0 },
    { SENSORS_DDCMON_TIMING4_HOR, "Standard Timing 4",
                         SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, DDCMON_SYSCTL_TIMING4, VALUE(1), 0 },
    { SENSORS_DDCMON_TIMING4_VER, "vertical",
                         SENSORS_DDCMON_TIMING4_HOR, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, DDCMON_SYSCTL_TIMING4, VALUE(2), 0 },
    { SENSORS_DDCMON_TIMING4_REF, "refresh",
                         SENSORS_DDCMON_TIMING4_HOR, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, DDCMON_SYSCTL_TIMING4, VALUE(3), 0 },
    { SENSORS_DDCMON_TIMING5_HOR, "Standard Timing 5",
                         SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, DDCMON_SYSCTL_TIMING5, VALUE(1), 0 },
    { SENSORS_DDCMON_TIMING5_VER, "vertical",
                         SENSORS_DDCMON_TIMING5_HOR, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, DDCMON_SYSCTL_TIMING5, VALUE(2), 0 },
    { SENSORS_DDCMON_TIMING5_REF, "refresh",
                         SENSORS_DDCMON_TIMING5_HOR, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, DDCMON_SYSCTL_TIMING5, VALUE(3), 0 },
    { SENSORS_DDCMON_TIMING6_HOR, "Standard Timing 6",
                         SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, DDCMON_SYSCTL_TIMING6, VALUE(1), 0 },
    { SENSORS_DDCMON_TIMING6_VER, "vertical",
                         SENSORS_DDCMON_TIMING6_HOR, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, DDCMON_SYSCTL_TIMING6, VALUE(2), 0 },
    { SENSORS_DDCMON_TIMING6_REF, "refresh",
                         SENSORS_DDCMON_TIMING6_HOR, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, DDCMON_SYSCTL_TIMING6, VALUE(3), 0 },
    { SENSORS_DDCMON_TIMING7_HOR, "Standard Timing 7",
                         SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, DDCMON_SYSCTL_TIMING7, VALUE(1), 0 },
    { SENSORS_DDCMON_TIMING7_VER, "vertical",
                         SENSORS_DDCMON_TIMING7_HOR, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, DDCMON_SYSCTL_TIMING7, VALUE(2), 0 },
    { SENSORS_DDCMON_TIMING7_REF, "refresh",
                         SENSORS_DDCMON_TIMING7_HOR, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, DDCMON_SYSCTL_TIMING7, VALUE(3), 0 },
    { SENSORS_DDCMON_TIMING8_HOR, "Standard Timing 8",
                         SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, DDCMON_SYSCTL_TIMING8, VALUE(1), 0 },
    { SENSORS_DDCMON_TIMING8_VER, "vertical",
                         SENSORS_DDCMON_TIMING8_HOR, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, DDCMON_SYSCTL_TIMING8, VALUE(2), 0 },
    { SENSORS_DDCMON_TIMING8_REF, "refresh",
                         SENSORS_DDCMON_TIMING8_HOR, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, DDCMON_SYSCTL_TIMING8, VALUE(3), 0 },
    { SENSORS_DDCMON_MAXCLOCK, "Max Pixel Clock (MHz)",
                         SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, DDCMON_SYSCTL_MAXCLOCK, VALUE(1), 0 },
    { 0 }
  };

static sensors_chip_feature eeprom_features[] =
  {
    { SENSORS_EEPROM_TYPE, "Memory type",
                         SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, EEPROM_SYSCTL1, VALUE(3), 0 },
/* cheat and use the row-address-bits label for the total size in sensors */
    { SENSORS_EEPROM_ROWADDR, "Memory size (MB)",
                         SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, EEPROM_SYSCTL1, VALUE(4), 0 },
    { SENSORS_EEPROM_COLADDR, "col",
                         SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, EEPROM_SYSCTL1, VALUE(5), 0 },
    { SENSORS_EEPROM_NUMROWS, "num rows",
                         SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, EEPROM_SYSCTL1, VALUE(6), 0 },
    { SENSORS_EEPROM_BANKS, "banks",
                         SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, EEPROM_SYSCTL2, VALUE(2), 0 },
/* for Sony Vaio's eeproms */
    { SENSORS_EEPROM_VAIO_NAME, "Machine name",
                         SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, EEPROM_SYSCTL9, VALUE(1), 0 },
    { SENSORS_EEPROM_VAIO_NAME+1, "name2",
                         SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, EEPROM_SYSCTL9, VALUE(2), 0 },
    { SENSORS_EEPROM_VAIO_NAME+2, "name3",
                         SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, EEPROM_SYSCTL9, VALUE(3), 0 },
    { SENSORS_EEPROM_VAIO_NAME+3, "name4",
                         SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, EEPROM_SYSCTL9, VALUE(4), 0 },
    { SENSORS_EEPROM_VAIO_NAME+4, "name5",
                         SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, EEPROM_SYSCTL9, VALUE(5), 0 },
    { SENSORS_EEPROM_VAIO_NAME+5, "name6",
                         SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, EEPROM_SYSCTL9, VALUE(6), 0 },
    { SENSORS_EEPROM_VAIO_NAME+6, "name7",
                         SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, EEPROM_SYSCTL9, VALUE(7), 0 },
    { SENSORS_EEPROM_VAIO_NAME+7, "name8",
                         SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, EEPROM_SYSCTL9, VALUE(8), 0 },
    { SENSORS_EEPROM_VAIO_NAME+8, "name9",
                         SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, EEPROM_SYSCTL9, VALUE(9), 0 },
    { SENSORS_EEPROM_VAIO_NAME+9, "name10",
                         SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, EEPROM_SYSCTL9, VALUE(10), 0 },
    { SENSORS_EEPROM_VAIO_NAME+10, "name11",
                         SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, EEPROM_SYSCTL9, VALUE(11), 0 },
    { SENSORS_EEPROM_VAIO_NAME+11, "name12",
                         SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, EEPROM_SYSCTL9, VALUE(12), 0 },
    { SENSORS_EEPROM_VAIO_NAME+12, "name13",
                         SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, EEPROM_SYSCTL9, VALUE(13), 0 },
    { SENSORS_EEPROM_VAIO_NAME+13, "name14",
                         SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, EEPROM_SYSCTL9, VALUE(14), 0 },
    { SENSORS_EEPROM_VAIO_NAME+14, "name15",
                         SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, EEPROM_SYSCTL9, VALUE(15), 0 },
    { SENSORS_EEPROM_VAIO_NAME+15, "name16",
                         SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, EEPROM_SYSCTL9, VALUE(16), 0 },
    { SENSORS_EEPROM_VAIO_NAME+16, "name17",
                         SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, EEPROM_SYSCTL10, VALUE(1), 0 },
    { SENSORS_EEPROM_VAIO_NAME+17, "name18",
                         SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, EEPROM_SYSCTL10, VALUE(2), 0 },
    { SENSORS_EEPROM_VAIO_NAME+18, "name19",
                         SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, EEPROM_SYSCTL10, VALUE(3), 0 },
    { SENSORS_EEPROM_VAIO_NAME+19, "name20",
                         SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, EEPROM_SYSCTL10, VALUE(4), 0 },
    { SENSORS_EEPROM_VAIO_NAME+20, "name21",
                         SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, EEPROM_SYSCTL10, VALUE(5), 0 },
    { SENSORS_EEPROM_VAIO_NAME+21, "name22",
                         SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, EEPROM_SYSCTL10, VALUE(6), 0 },
    { SENSORS_EEPROM_VAIO_NAME+22, "name23",
                         SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, EEPROM_SYSCTL10, VALUE(7), 0 },
    { SENSORS_EEPROM_VAIO_NAME+23, "name24",
                         SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, EEPROM_SYSCTL10, VALUE(8), 0 },
    { SENSORS_EEPROM_VAIO_NAME+24, "name25",
                         SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, EEPROM_SYSCTL10, VALUE(9), 0 },
    { SENSORS_EEPROM_VAIO_NAME+25, "name26",
                         SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, EEPROM_SYSCTL10, VALUE(10), 0 },
    { SENSORS_EEPROM_VAIO_NAME+26, "name27",
                         SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, EEPROM_SYSCTL10, VALUE(11), 0 },
    { SENSORS_EEPROM_VAIO_NAME+27, "name28",
                         SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, EEPROM_SYSCTL10, VALUE(12), 0 },
    { SENSORS_EEPROM_VAIO_NAME+28, "name29",
                         SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, EEPROM_SYSCTL10, VALUE(13), 0 },
    { SENSORS_EEPROM_VAIO_NAME+29, "name30",
                         SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, EEPROM_SYSCTL10, VALUE(14), 0 },
    { SENSORS_EEPROM_VAIO_NAME+30, "name31",
                         SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, EEPROM_SYSCTL10, VALUE(15), 0 },
    { SENSORS_EEPROM_VAIO_NAME+31, "name32",
                         SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, EEPROM_SYSCTL10, VALUE(16), 0 },
    { SENSORS_EEPROM_VAIO_SERIAL, "Serial number",
                         SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, EEPROM_SYSCTL13, VALUE(1), 0 },
    { SENSORS_EEPROM_VAIO_SERIAL+1, "serial2",
                         SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, EEPROM_SYSCTL13, VALUE(2), 0 },
    { SENSORS_EEPROM_VAIO_SERIAL+2, "serial3",
                         SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, EEPROM_SYSCTL13, VALUE(3), 0 },
    { SENSORS_EEPROM_VAIO_SERIAL+3, "serial4",
                         SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, EEPROM_SYSCTL13, VALUE(4), 0 },
    { SENSORS_EEPROM_VAIO_SERIAL+4, "serial5",
                         SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, EEPROM_SYSCTL13, VALUE(5), 0 },
    { SENSORS_EEPROM_VAIO_SERIAL+5, "serial6",
                         SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, EEPROM_SYSCTL13, VALUE(6), 0 },
    { SENSORS_EEPROM_VAIO_SERIAL+6, "serial7",
                         SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, EEPROM_SYSCTL13, VALUE(7), 0 },
    { SENSORS_EEPROM_VAIO_SERIAL+7, "serial8",
                         SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, EEPROM_SYSCTL13, VALUE(8), 0 },
    { SENSORS_EEPROM_VAIO_SERIAL+8, "serial9",
                         SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, EEPROM_SYSCTL13, VALUE(9), 0 },
    { SENSORS_EEPROM_VAIO_SERIAL+9, "serial10",
                         SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, EEPROM_SYSCTL13, VALUE(10), 0 },
    { SENSORS_EEPROM_VAIO_SERIAL+10, "serial11",
                         SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, EEPROM_SYSCTL13, VALUE(11), 0 },
    { SENSORS_EEPROM_VAIO_SERIAL+11, "serial12",
                         SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, EEPROM_SYSCTL13, VALUE(12), 0 },
    { SENSORS_EEPROM_VAIO_SERIAL+12, "serial13",
                         SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, EEPROM_SYSCTL13, VALUE(13), 0 },
    { SENSORS_EEPROM_VAIO_SERIAL+13, "serial14",
                         SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, EEPROM_SYSCTL13, VALUE(14), 0 },
    { SENSORS_EEPROM_VAIO_SERIAL+14, "serial15",
                         SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, EEPROM_SYSCTL13, VALUE(15), 0 },
    { SENSORS_EEPROM_VAIO_SERIAL+15, "serial16",
                         SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, EEPROM_SYSCTL13, VALUE(16), 0 },
    { SENSORS_EEPROM_VAIO_SERIAL+16, "serial17",
                         SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, EEPROM_SYSCTL14, VALUE(1), 0 },
    { SENSORS_EEPROM_VAIO_SERIAL+17, "serial18",
                         SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, EEPROM_SYSCTL14, VALUE(2), 0 },
    { SENSORS_EEPROM_VAIO_SERIAL+18, "serial19",
                         SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, EEPROM_SYSCTL14, VALUE(3), 0 },
    { SENSORS_EEPROM_VAIO_SERIAL+19, "serial20",
                         SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, EEPROM_SYSCTL14, VALUE(4), 0 },
    { SENSORS_EEPROM_VAIO_SERIAL+20, "serial21",
                         SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, EEPROM_SYSCTL14, VALUE(5), 0 },
    { SENSORS_EEPROM_VAIO_SERIAL+21, "serial22",
                         SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, EEPROM_SYSCTL14, VALUE(6), 0 },
    { SENSORS_EEPROM_VAIO_SERIAL+22, "serial23",
                         SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, EEPROM_SYSCTL14, VALUE(7), 0 },
    { SENSORS_EEPROM_VAIO_SERIAL+23, "serial24",
                         SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, EEPROM_SYSCTL14, VALUE(8), 0 },
    { SENSORS_EEPROM_VAIO_SERIAL+24, "serial25",
                         SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, EEPROM_SYSCTL14, VALUE(9), 0 },
    { SENSORS_EEPROM_VAIO_SERIAL+25, "serial26",
                         SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, EEPROM_SYSCTL14, VALUE(10), 0 },
    { SENSORS_EEPROM_VAIO_SERIAL+26, "serial27",
                         SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, EEPROM_SYSCTL14, VALUE(11), 0 },
    { SENSORS_EEPROM_VAIO_SERIAL+27, "serial28",
                         SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, EEPROM_SYSCTL14, VALUE(12), 0 },
    { SENSORS_EEPROM_VAIO_SERIAL+28, "serial29",
                         SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, EEPROM_SYSCTL14, VALUE(13), 0 },
    { SENSORS_EEPROM_VAIO_SERIAL+29, "serial30",
                         SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, EEPROM_SYSCTL14, VALUE(14), 0 },
    { SENSORS_EEPROM_VAIO_SERIAL+30, "serial31",
                         SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, EEPROM_SYSCTL14, VALUE(15), 0 },
    { SENSORS_EEPROM_VAIO_SERIAL+31, "serial32",
                         SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, EEPROM_SYSCTL14, VALUE(16), 0 },
/* for monitor's edid */
    { SENSORS_EEPROM_EDID_HEADER, "EDID header",
                         SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, EEPROM_SYSCTL1, VALUE(1), 0 },
    { 0 }
  };

/* NOTE: Many shared values with adm1027_features !!! */
static sensors_chip_feature lm85_features[] =
  { 
    { SENSORS_LM85_IN0, "in0", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING, 
                        SENSORS_MODE_R, LM85_SYSCTL_IN0, VALUE(3), 3 },
    { SENSORS_LM85_IN1, "in1", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, LM85_SYSCTL_IN1, VALUE(3), 3 },
    { SENSORS_LM85_IN2, "in2", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, LM85_SYSCTL_IN2, VALUE(3), 3 },
    { SENSORS_LM85_IN3, "in3", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, LM85_SYSCTL_IN3, VALUE(3), 3 },
    { SENSORS_LM85_IN4, "in4", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, LM85_SYSCTL_IN4, VALUE(3), 3 },
    { SENSORS_LM85_IN5, "in5", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, EMC6D100_SYSCTL_IN5, VALUE(3), 3 },
    { SENSORS_LM85_IN6, "in6", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, EMC6D100_SYSCTL_IN6, VALUE(3), 3 },
    { SENSORS_LM85_IN7, "in7", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, EMC6D100_SYSCTL_IN7, VALUE(3), 3 },
    { SENSORS_LM85_IN0_MIN, "in0_min", SENSORS_LM85_IN0, SENSORS_LM85_IN0,
                            SENSORS_MODE_RW, LM85_SYSCTL_IN0, VALUE(1), 3 },
    { SENSORS_LM85_IN1_MIN, "in1_min", SENSORS_LM85_IN1, SENSORS_LM85_IN1,
                            SENSORS_MODE_RW, LM85_SYSCTL_IN1, VALUE(1), 3 },
    { SENSORS_LM85_IN2_MIN, "in2_min", SENSORS_LM85_IN2, SENSORS_LM85_IN2,
                            SENSORS_MODE_RW, LM85_SYSCTL_IN2, VALUE(1), 3 },
    { SENSORS_LM85_IN3_MIN, "in3_min", SENSORS_LM85_IN3, SENSORS_LM85_IN3,
                            SENSORS_MODE_RW, LM85_SYSCTL_IN3, VALUE(1), 3 },
    { SENSORS_LM85_IN4_MIN, "in4_min", SENSORS_LM85_IN4, SENSORS_LM85_IN4,
                            SENSORS_MODE_RW, LM85_SYSCTL_IN4, VALUE(1), 3 },
    { SENSORS_LM85_IN5_MIN, "in5_min", SENSORS_LM85_IN5, SENSORS_LM85_IN5,
                            SENSORS_MODE_RW, EMC6D100_SYSCTL_IN5, VALUE(1), 3 },
    { SENSORS_LM85_IN6_MIN, "in6_min", SENSORS_LM85_IN6, SENSORS_LM85_IN6,
                            SENSORS_MODE_RW, EMC6D100_SYSCTL_IN6, VALUE(1), 3 },
    { SENSORS_LM85_IN7_MIN, "in7_min", SENSORS_LM85_IN7, SENSORS_LM85_IN7,
                            SENSORS_MODE_RW, EMC6D100_SYSCTL_IN7, VALUE(1), 3 },
    { SENSORS_LM85_IN0_MAX, "in0_max", SENSORS_LM85_IN0, SENSORS_LM85_IN0,
                            SENSORS_MODE_RW, LM85_SYSCTL_IN0, VALUE(2), 3 },
    { SENSORS_LM85_IN1_MAX, "in1_max", SENSORS_LM85_IN1, SENSORS_LM85_IN1,
                            SENSORS_MODE_RW, LM85_SYSCTL_IN1, VALUE(2), 3 },
    { SENSORS_LM85_IN2_MAX, "in2_max", SENSORS_LM85_IN2, SENSORS_LM85_IN2,
                            SENSORS_MODE_RW, LM85_SYSCTL_IN2, VALUE(2), 3 },
    { SENSORS_LM85_IN3_MAX, "in3_max", SENSORS_LM85_IN3, SENSORS_LM85_IN3,
                            SENSORS_MODE_RW, LM85_SYSCTL_IN3, VALUE(2), 3 },
    { SENSORS_LM85_IN4_MAX, "in4_max", SENSORS_LM85_IN4, SENSORS_LM85_IN4,
                            SENSORS_MODE_RW, LM85_SYSCTL_IN4, VALUE(2), 3 },
    { SENSORS_LM85_IN5_MAX, "in5_max", SENSORS_LM85_IN5, SENSORS_LM85_IN5,
                            SENSORS_MODE_RW, EMC6D100_SYSCTL_IN5, VALUE(2), 3 },
    { SENSORS_LM85_IN6_MAX, "in6_max", SENSORS_LM85_IN6, SENSORS_LM85_IN6,
                            SENSORS_MODE_RW, EMC6D100_SYSCTL_IN6, VALUE(2), 3 },
    { SENSORS_LM85_IN7_MAX, "in7_max", SENSORS_LM85_IN7, SENSORS_LM85_IN7,
                            SENSORS_MODE_RW, EMC6D100_SYSCTL_IN7, VALUE(2), 3 },
    { SENSORS_LM85_FAN1, "fan1", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, LM85_SYSCTL_FAN1, VALUE(2), 0 },
    { SENSORS_LM85_FAN2, "fan2", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, LM85_SYSCTL_FAN2, VALUE(2), 0 },
    { SENSORS_LM85_FAN3, "fan3", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, LM85_SYSCTL_FAN3, VALUE(2), 0 },
    { SENSORS_LM85_FAN4, "fan4", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, LM85_SYSCTL_FAN4, VALUE(2), 0 },
    { SENSORS_LM85_FAN1_MIN, "fan1_min", SENSORS_LM85_FAN1, SENSORS_LM85_FAN1,
                             SENSORS_MODE_RW, LM85_SYSCTL_FAN1, VALUE(1), 0 },
    { SENSORS_LM85_FAN2_MIN, "fan2_min", SENSORS_LM85_FAN2, SENSORS_LM85_FAN2,
                             SENSORS_MODE_RW, LM85_SYSCTL_FAN2, VALUE(1), 0 },
    { SENSORS_LM85_FAN3_MIN, "fan3_min", SENSORS_LM85_FAN3, SENSORS_LM85_FAN3,
                             SENSORS_MODE_RW, LM85_SYSCTL_FAN3, VALUE(1), 0 },
    { SENSORS_LM85_FAN4_MIN, "fan4_min", SENSORS_LM85_FAN4, SENSORS_LM85_FAN4,
                             SENSORS_MODE_RW, LM85_SYSCTL_FAN4, VALUE(1), 0 },
    { SENSORS_LM85_TEMP1, "temp1", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, LM85_SYSCTL_TEMP1, VALUE(3), 2 },
    { SENSORS_LM85_TEMP2, "temp2", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, LM85_SYSCTL_TEMP2, VALUE(3), 2 },
    { SENSORS_LM85_TEMP3, "temp3", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, LM85_SYSCTL_TEMP3, VALUE(3), 2 },
    { SENSORS_LM85_TEMP1_MIN, "temp1_min", SENSORS_LM85_TEMP1,
                              SENSORS_LM85_TEMP1, SENSORS_MODE_RW, 
                              LM85_SYSCTL_TEMP1, VALUE(1), 2 },
    { SENSORS_LM85_TEMP2_MIN, "temp2_min", SENSORS_LM85_TEMP2,
                              SENSORS_LM85_TEMP2, SENSORS_MODE_RW, 
                              LM85_SYSCTL_TEMP2, VALUE(1), 2 },
    { SENSORS_LM85_TEMP3_MIN, "temp3_min", SENSORS_LM85_TEMP3,
                              SENSORS_LM85_TEMP3, SENSORS_MODE_RW, 
                              LM85_SYSCTL_TEMP3, VALUE(1), 2 },
    { SENSORS_LM85_TEMP1_MAX, "temp1_max", SENSORS_LM85_TEMP1,
                              SENSORS_LM85_TEMP1, SENSORS_MODE_RW, 
                              LM85_SYSCTL_TEMP1, VALUE(2), 2 },
    { SENSORS_LM85_TEMP2_MAX, "temp2_max", SENSORS_LM85_TEMP2,
                              SENSORS_LM85_TEMP2, SENSORS_MODE_RW, 
                              LM85_SYSCTL_TEMP2, VALUE(2), 2 },
    { SENSORS_LM85_TEMP3_MAX, "temp3_max", SENSORS_LM85_TEMP3,
                              SENSORS_LM85_TEMP3, SENSORS_MODE_RW, 
                              LM85_SYSCTL_TEMP3, VALUE(2), 2 },
    { SENSORS_LM85_VID, "vid", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, LM85_SYSCTL_VID, VALUE(1), 3 },
    { SENSORS_LM85_VRM, "vrm", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_RW, LM85_SYSCTL_VRM, VALUE(1), 1 },
    { SENSORS_LM85_ALARMS, "alarms", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING, 
                           SENSORS_MODE_R, LM85_SYSCTL_ALARMS, VALUE(1), 0 },
    { SENSORS_LM85_PWM1, "pwm1", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_RW, LM85_SYSCTL_PWM1, VALUE(1), 0 },
    { SENSORS_LM85_PWM2, "pwm2", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_RW, LM85_SYSCTL_PWM2, VALUE(1), 0 },
    { SENSORS_LM85_PWM3, "pwm3", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_RW, LM85_SYSCTL_PWM3, VALUE(1), 0 },
    { SENSORS_LM85_PWM1_SPINUP, "pwm1_spinup", SENSORS_LM85_PWM1, 
                         SENSORS_NO_MAPPING, SENSORS_MODE_RW, 
                         LM85_SYSCTL_PWM_CFG1, VALUE(1), 1 },
    { SENSORS_LM85_PWM2_SPINUP, "pwm2_spinup", SENSORS_LM85_PWM2, 
                         SENSORS_NO_MAPPING, SENSORS_MODE_RW, 
                         LM85_SYSCTL_PWM_CFG2, VALUE(1), 1 },
    { SENSORS_LM85_PWM3_SPINUP, "pwm3_spinup", SENSORS_LM85_PWM3, 
                         SENSORS_NO_MAPPING, SENSORS_MODE_RW, 
                         LM85_SYSCTL_PWM_CFG3, VALUE(1), 1 },
    { SENSORS_LM85_PWM1_MIN, "pwm1_min", SENSORS_LM85_PWM1, 
                         SENSORS_NO_MAPPING, SENSORS_MODE_RW, 
                         LM85_SYSCTL_PWM_CFG1, VALUE(2), 1 },
    { SENSORS_LM85_PWM2_MIN, "pwm2_min", SENSORS_LM85_PWM2, 
                         SENSORS_NO_MAPPING, SENSORS_MODE_RW, 
                         LM85_SYSCTL_PWM_CFG2, VALUE(2), 1 },
    { SENSORS_LM85_PWM3_MIN, "pwm3_min", SENSORS_LM85_PWM3, 
                         SENSORS_NO_MAPPING, SENSORS_MODE_RW, 
                         LM85_SYSCTL_PWM_CFG3, VALUE(2), 1 },
    { SENSORS_LM85_PWM1_FREQ, "pwm1_freq", SENSORS_LM85_PWM1, 
                         SENSORS_NO_MAPPING, SENSORS_MODE_RW, 
                         LM85_SYSCTL_PWM_CFG1, VALUE(3), 1 },
    { SENSORS_LM85_PWM2_FREQ, "pwm2_freq", SENSORS_LM85_PWM2, 
                         SENSORS_NO_MAPPING, SENSORS_MODE_RW, 
                         LM85_SYSCTL_PWM_CFG2, VALUE(3), 1 },
    { SENSORS_LM85_PWM3_FREQ, "pwm3_freq", SENSORS_LM85_PWM3, 
                         SENSORS_NO_MAPPING, SENSORS_MODE_RW, 
                         LM85_SYSCTL_PWM_CFG3, VALUE(3), 1 },
    { SENSORS_LM85_PWM1_MIN_CTL, "pwm1_min_ctl", SENSORS_LM85_PWM1, 
                         SENSORS_NO_MAPPING, SENSORS_MODE_RW, 
                         LM85_SYSCTL_PWM_CFG1, VALUE(4), 1 },
    { SENSORS_LM85_PWM2_MIN_CTL, "pwm2_min_ctl", SENSORS_LM85_PWM2, 
                         SENSORS_NO_MAPPING, SENSORS_MODE_RW, 
                         LM85_SYSCTL_PWM_CFG2, VALUE(4), 1 },
    { SENSORS_LM85_PWM3_MIN_CTL, "pwm3_min_ctl", SENSORS_LM85_PWM3, 
                         SENSORS_NO_MAPPING, SENSORS_MODE_RW, 
                         LM85_SYSCTL_PWM_CFG3, VALUE(4), 1 },
    { SENSORS_LM85_PWM1_INVERT, "pwm1_invert", SENSORS_LM85_PWM1, 
                         SENSORS_NO_MAPPING, SENSORS_MODE_RW, 
                         LM85_SYSCTL_PWM_CFG1, VALUE(5), 0 },
    { SENSORS_LM85_PWM2_INVERT, "pwm2_invert", SENSORS_LM85_PWM2, 
                         SENSORS_NO_MAPPING, SENSORS_MODE_RW, 
                         LM85_SYSCTL_PWM_CFG2, VALUE(5), 0 },
    { SENSORS_LM85_PWM3_INVERT, "pwm3_invert", SENSORS_LM85_PWM3, 
                         SENSORS_NO_MAPPING, SENSORS_MODE_RW, 
                         LM85_SYSCTL_PWM_CFG3, VALUE(5), 0 },
    { SENSORS_LM85_PWM1_ZONE, "pwm1_zone", SENSORS_LM85_PWM1, 
                         SENSORS_NO_MAPPING, SENSORS_MODE_RW, 
                         LM85_SYSCTL_PWM_ZONE1, VALUE(1), 0 },
    { SENSORS_LM85_PWM2_ZONE, "pwm2_zone", SENSORS_LM85_PWM2, 
                         SENSORS_NO_MAPPING, SENSORS_MODE_RW, 
                         LM85_SYSCTL_PWM_ZONE2, VALUE(1), 0 },
    { SENSORS_LM85_PWM3_ZONE, "pwm3_zone", SENSORS_LM85_PWM3, 
                         SENSORS_NO_MAPPING, SENSORS_MODE_RW, 
                         LM85_SYSCTL_PWM_ZONE3, VALUE(1), 0 },
    { SENSORS_LM85_ZONE1_LIMIT, "zone1_limit", SENSORS_LM85_TEMP1, 
                         SENSORS_NO_MAPPING, SENSORS_MODE_RW, 
                         LM85_SYSCTL_ZONE1, VALUE(1), 1 },
    { SENSORS_LM85_ZONE2_LIMIT, "zone2_limit", SENSORS_LM85_TEMP2, 
                         SENSORS_NO_MAPPING, SENSORS_MODE_RW, 
                         LM85_SYSCTL_ZONE2, VALUE(1), 1 },
    { SENSORS_LM85_ZONE3_LIMIT, "zone3_limit", SENSORS_LM85_TEMP3, 
                         SENSORS_NO_MAPPING, SENSORS_MODE_RW, 
                         LM85_SYSCTL_ZONE3, VALUE(1), 1 },
    { SENSORS_LM85_ZONE1_HYST, "zone1_hyst", SENSORS_LM85_TEMP1, 
                         SENSORS_NO_MAPPING, SENSORS_MODE_RW, 
                         LM85_SYSCTL_ZONE1, VALUE(2), 1 },
    { SENSORS_LM85_ZONE2_HYST, "zone2_hyst", SENSORS_LM85_TEMP2, 
                         SENSORS_NO_MAPPING, SENSORS_MODE_RW, 
                         LM85_SYSCTL_ZONE2, VALUE(2), 1 },
    { SENSORS_LM85_ZONE3_HYST, "zone3_hyst", SENSORS_LM85_TEMP3, 
                         SENSORS_NO_MAPPING, SENSORS_MODE_RW, 
                         LM85_SYSCTL_ZONE3, VALUE(2), 1 },
    { SENSORS_LM85_ZONE1_RANGE, "zone1_range", SENSORS_LM85_TEMP1, 
                         SENSORS_NO_MAPPING, SENSORS_MODE_RW, 
                         LM85_SYSCTL_ZONE1, VALUE(3), 1 },
    { SENSORS_LM85_ZONE2_RANGE, "zone2_range", SENSORS_LM85_TEMP2, 
                         SENSORS_NO_MAPPING, SENSORS_MODE_RW, 
                         LM85_SYSCTL_ZONE2, VALUE(3), 1 },
    { SENSORS_LM85_ZONE3_RANGE, "zone3_range", SENSORS_LM85_TEMP3, 
                         SENSORS_NO_MAPPING, SENSORS_MODE_RW, 
                         LM85_SYSCTL_ZONE3, VALUE(3), 1 },
    { SENSORS_LM85_ZONE1_CRITICAL, "zone1_critical", SENSORS_LM85_TEMP1, 
                         SENSORS_NO_MAPPING, SENSORS_MODE_RW, 
                         LM85_SYSCTL_ZONE1, VALUE(4), 1 },
    { SENSORS_LM85_ZONE2_CRITICAL, "zone2_critical", SENSORS_LM85_TEMP2, 
                         SENSORS_NO_MAPPING, SENSORS_MODE_RW, 
                         LM85_SYSCTL_ZONE2, VALUE(4), 1 },
    { SENSORS_LM85_ZONE3_CRITICAL, "zone3_critical", SENSORS_LM85_TEMP3, 
                         SENSORS_NO_MAPPING, SENSORS_MODE_RW, 
                         LM85_SYSCTL_ZONE3, VALUE(4), 1 },

    { SENSORS_LM85_PWM1_SPINUP_CTL, "pwm1_spinup_ctl", SENSORS_LM85_PWM1, 
                         SENSORS_NO_MAPPING, SENSORS_MODE_RW, 
                         LM85_SYSCTL_SPINUP_CTL, VALUE(1), 0 },
    { SENSORS_LM85_PWM2_SPINUP_CTL, "pwm2_spinup_ctl", SENSORS_LM85_PWM2, 
                         SENSORS_NO_MAPPING, SENSORS_MODE_RW, 
                         LM85_SYSCTL_SPINUP_CTL, VALUE(2), 0 },
    { SENSORS_LM85_PWM3_SPINUP_CTL, "pwm3_spinup_ctl", SENSORS_LM85_PWM3, 
                         SENSORS_NO_MAPPING, SENSORS_MODE_RW, 
                         LM85_SYSCTL_SPINUP_CTL, VALUE(3), 0 },
    { SENSORS_LM85_FAN1_TACH_MODE, "fan1_tach_mode", SENSORS_LM85_FAN1, 
                         SENSORS_NO_MAPPING, SENSORS_MODE_RW, 
                         LM85_SYSCTL_TACH_MODE, VALUE(1), 0 },
    { SENSORS_LM85_FAN2_TACH_MODE, "fan2_tach_mode", SENSORS_LM85_FAN2, 
                         SENSORS_NO_MAPPING, SENSORS_MODE_RW, 
                         LM85_SYSCTL_TACH_MODE, VALUE(2), 0 },
    { SENSORS_LM85_FAN3_TACH_MODE, "fan3_tach_mode", SENSORS_LM85_FAN3, 
                         SENSORS_NO_MAPPING, SENSORS_MODE_RW, 
                         LM85_SYSCTL_TACH_MODE, VALUE(3), 0 },
    { SENSORS_LM85_ZONE1_SMOOTH, "zone1_smooth", SENSORS_LM85_TEMP1, 
                         SENSORS_NO_MAPPING, SENSORS_MODE_RW, 
                         LM85_SYSCTL_SMOOTH1, VALUE(1), 1 },
    { SENSORS_LM85_ZONE2_SMOOTH, "zone2_smooth", SENSORS_LM85_TEMP2, 
                         SENSORS_NO_MAPPING, SENSORS_MODE_RW, 
                         LM85_SYSCTL_SMOOTH2, VALUE(1), 1 },
    { SENSORS_LM85_ZONE3_SMOOTH, "zone3_smooth", SENSORS_LM85_TEMP3, 
                         SENSORS_NO_MAPPING, SENSORS_MODE_RW, 
                         LM85_SYSCTL_SMOOTH3, VALUE(1), 1 },
    { 0 }
  };

/* NOTE: Many shared values with lm85_features !!! */
static sensors_chip_feature adm1027_features[] =
  { 
    { SENSORS_LM85_IN0, "in0", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING, 
                        SENSORS_MODE_R, LM85_SYSCTL_IN0, VALUE(3), 3 },
    { SENSORS_LM85_IN1, "in1", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, LM85_SYSCTL_IN1, VALUE(3), 3 },
    { SENSORS_LM85_IN2, "in2", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, LM85_SYSCTL_IN2, VALUE(3), 3 },
    { SENSORS_LM85_IN3, "in3", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, LM85_SYSCTL_IN3, VALUE(3), 3 },
    { SENSORS_LM85_IN4, "in4", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, LM85_SYSCTL_IN4, VALUE(3), 3 },
    { SENSORS_LM85_IN0_MIN, "in0_min", SENSORS_LM85_IN0, SENSORS_LM85_IN0,
                            SENSORS_MODE_RW, LM85_SYSCTL_IN0, VALUE(1), 3 },
    { SENSORS_LM85_IN1_MIN, "in1_min", SENSORS_LM85_IN1, SENSORS_LM85_IN1,
                            SENSORS_MODE_RW, LM85_SYSCTL_IN1, VALUE(1), 3 },
    { SENSORS_LM85_IN2_MIN, "in2_min", SENSORS_LM85_IN2, SENSORS_LM85_IN2,
                            SENSORS_MODE_RW, LM85_SYSCTL_IN2, VALUE(1), 3 },
    { SENSORS_LM85_IN3_MIN, "in3_min", SENSORS_LM85_IN3, SENSORS_LM85_IN3,
                            SENSORS_MODE_RW, LM85_SYSCTL_IN3, VALUE(1), 3 },
    { SENSORS_LM85_IN4_MIN, "in4_min", SENSORS_LM85_IN4, SENSORS_LM85_IN4,
                            SENSORS_MODE_RW, LM85_SYSCTL_IN4, VALUE(1), 3 },
    { SENSORS_LM85_IN0_MAX, "in0_max", SENSORS_LM85_IN0, SENSORS_LM85_IN0,
                            SENSORS_MODE_RW, LM85_SYSCTL_IN0, VALUE(2), 3 },
    { SENSORS_LM85_IN1_MAX, "in1_max", SENSORS_LM85_IN1, SENSORS_LM85_IN1,
                            SENSORS_MODE_RW, LM85_SYSCTL_IN1, VALUE(2), 3 },
    { SENSORS_LM85_IN2_MAX, "in2_max", SENSORS_LM85_IN2, SENSORS_LM85_IN2,
                            SENSORS_MODE_RW, LM85_SYSCTL_IN2, VALUE(2), 3 },
    { SENSORS_LM85_IN3_MAX, "in3_max", SENSORS_LM85_IN3, SENSORS_LM85_IN3,
                            SENSORS_MODE_RW, LM85_SYSCTL_IN3, VALUE(2), 3 },
    { SENSORS_LM85_IN4_MAX, "in4_max", SENSORS_LM85_IN4, SENSORS_LM85_IN4,
                            SENSORS_MODE_RW, LM85_SYSCTL_IN4, VALUE(2), 3 },
    { SENSORS_LM85_FAN1, "fan1", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, LM85_SYSCTL_FAN1, VALUE(2), 0 },
    { SENSORS_LM85_FAN2, "fan2", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, LM85_SYSCTL_FAN2, VALUE(2), 0 },
    { SENSORS_LM85_FAN3, "fan3", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, LM85_SYSCTL_FAN3, VALUE(2), 0 },
    { SENSORS_LM85_FAN4, "fan4", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, LM85_SYSCTL_FAN4, VALUE(2), 0 },
    { SENSORS_LM85_FAN1_MIN, "fan1_min", SENSORS_LM85_FAN1, SENSORS_LM85_FAN1,
                             SENSORS_MODE_RW, LM85_SYSCTL_FAN1, VALUE(1), 0 },
    { SENSORS_LM85_FAN2_MIN, "fan2_min", SENSORS_LM85_FAN2, SENSORS_LM85_FAN2,
                             SENSORS_MODE_RW, LM85_SYSCTL_FAN2, VALUE(1), 0 },
    { SENSORS_LM85_FAN3_MIN, "fan3_min", SENSORS_LM85_FAN3, SENSORS_LM85_FAN3,
                             SENSORS_MODE_RW, LM85_SYSCTL_FAN3, VALUE(1), 0 },
    { SENSORS_LM85_FAN4_MIN, "fan4_min", SENSORS_LM85_FAN4, SENSORS_LM85_FAN4,
                             SENSORS_MODE_RW, LM85_SYSCTL_FAN4, VALUE(1), 0 },
    { SENSORS_LM85_TEMP1, "temp1", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, LM85_SYSCTL_TEMP1, VALUE(3), 2 },
    { SENSORS_LM85_TEMP2, "temp2", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, LM85_SYSCTL_TEMP2, VALUE(3), 2 },
    { SENSORS_LM85_TEMP3, "temp3", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, LM85_SYSCTL_TEMP3, VALUE(3), 2 },
    { SENSORS_LM85_TEMP1_MIN, "temp1_min", SENSORS_LM85_TEMP1,
                              SENSORS_LM85_TEMP1, SENSORS_MODE_RW, 
                              LM85_SYSCTL_TEMP1, VALUE(1), 2 },
    { SENSORS_LM85_TEMP2_MIN, "temp2_min", SENSORS_LM85_TEMP2,
                              SENSORS_LM85_TEMP2, SENSORS_MODE_RW, 
                              LM85_SYSCTL_TEMP2, VALUE(1), 2 },
    { SENSORS_LM85_TEMP3_MIN, "temp3_min", SENSORS_LM85_TEMP3,
                              SENSORS_LM85_TEMP3, SENSORS_MODE_RW, 
                              LM85_SYSCTL_TEMP3, VALUE(1), 2 },
    { SENSORS_LM85_TEMP1_MAX, "temp1_max", SENSORS_LM85_TEMP1,
                              SENSORS_LM85_TEMP1, SENSORS_MODE_RW, 
                              LM85_SYSCTL_TEMP1, VALUE(2), 2 },
    { SENSORS_LM85_TEMP2_MAX, "temp2_max", SENSORS_LM85_TEMP2,
                              SENSORS_LM85_TEMP2, SENSORS_MODE_RW, 
                              LM85_SYSCTL_TEMP2, VALUE(2), 2 },
    { SENSORS_LM85_TEMP3_MAX, "temp3_max", SENSORS_LM85_TEMP3,
                              SENSORS_LM85_TEMP3, SENSORS_MODE_RW, 
                              LM85_SYSCTL_TEMP3, VALUE(2), 2 },
    { SENSORS_LM85_VID, "vid", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, LM85_SYSCTL_VID, VALUE(1), 3 },
    { SENSORS_LM85_VRM, "vrm", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_RW, LM85_SYSCTL_VRM, VALUE(1), 1 },
    { SENSORS_LM85_ALARMS, "alarms", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING, 
                           SENSORS_MODE_R, LM85_SYSCTL_ALARMS, VALUE(1), 0 },
    { SENSORS_LM85_PWM1, "pwm1", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_RW, LM85_SYSCTL_PWM1, VALUE(1), 0 },
    { SENSORS_LM85_PWM2, "pwm2", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_RW, LM85_SYSCTL_PWM2, VALUE(1), 0 },
    { SENSORS_LM85_PWM3, "pwm3", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_RW, LM85_SYSCTL_PWM3, VALUE(1), 0 },
    { SENSORS_LM85_PWM1_SPINUP, "pwm1_spinup", SENSORS_LM85_PWM1, 
                         SENSORS_NO_MAPPING, SENSORS_MODE_RW, 
                         LM85_SYSCTL_PWM_CFG1, VALUE(1), 1 },
    { SENSORS_LM85_PWM2_SPINUP, "pwm2_spinup", SENSORS_LM85_PWM2, 
                         SENSORS_NO_MAPPING, SENSORS_MODE_RW, 
                         LM85_SYSCTL_PWM_CFG2, VALUE(1), 1 },
    { SENSORS_LM85_PWM3_SPINUP, "pwm3_spinup", SENSORS_LM85_PWM3, 
                         SENSORS_NO_MAPPING, SENSORS_MODE_RW, 
                         LM85_SYSCTL_PWM_CFG3, VALUE(1), 1 },
    { SENSORS_LM85_PWM1_MIN, "pwm1_min", SENSORS_LM85_PWM1, 
                         SENSORS_NO_MAPPING, SENSORS_MODE_RW, 
                         LM85_SYSCTL_PWM_CFG1, VALUE(2), 1 },
    { SENSORS_LM85_PWM2_MIN, "pwm2_min", SENSORS_LM85_PWM2, 
                         SENSORS_NO_MAPPING, SENSORS_MODE_RW, 
                         LM85_SYSCTL_PWM_CFG2, VALUE(2), 1 },
    { SENSORS_LM85_PWM3_MIN, "pwm3_min", SENSORS_LM85_PWM3, 
                         SENSORS_NO_MAPPING, SENSORS_MODE_RW, 
                         LM85_SYSCTL_PWM_CFG3, VALUE(2), 1 },
    { SENSORS_LM85_PWM1_FREQ, "pwm1_freq", SENSORS_LM85_PWM1, 
                         SENSORS_NO_MAPPING, SENSORS_MODE_RW, 
                         LM85_SYSCTL_PWM_CFG1, VALUE(3), 1 },
    { SENSORS_LM85_PWM2_FREQ, "pwm2_freq", SENSORS_LM85_PWM2, 
                         SENSORS_NO_MAPPING, SENSORS_MODE_RW, 
                         LM85_SYSCTL_PWM_CFG2, VALUE(3), 1 },
    { SENSORS_LM85_PWM3_FREQ, "pwm3_freq", SENSORS_LM85_PWM3, 
                         SENSORS_NO_MAPPING, SENSORS_MODE_RW, 
                         LM85_SYSCTL_PWM_CFG3, VALUE(3), 1 },
    { SENSORS_LM85_PWM1_MIN_CTL, "pwm1_min_ctl", SENSORS_LM85_PWM1, 
                         SENSORS_NO_MAPPING, SENSORS_MODE_RW, 
                         LM85_SYSCTL_PWM_CFG1, VALUE(4), 1 },
    { SENSORS_LM85_PWM2_MIN_CTL, "pwm2_min_ctl", SENSORS_LM85_PWM2, 
                         SENSORS_NO_MAPPING, SENSORS_MODE_RW, 
                         LM85_SYSCTL_PWM_CFG2, VALUE(4), 1 },
    { SENSORS_LM85_PWM3_MIN_CTL, "pwm3_min_ctl", SENSORS_LM85_PWM3, 
                         SENSORS_NO_MAPPING, SENSORS_MODE_RW, 
                         LM85_SYSCTL_PWM_CFG3, VALUE(4), 1 },
    { SENSORS_LM85_PWM1_INVERT, "pwm1_invert", SENSORS_LM85_PWM1, 
                         SENSORS_NO_MAPPING, SENSORS_MODE_RW, 
                         LM85_SYSCTL_PWM_CFG1, VALUE(5), 0 },
    { SENSORS_LM85_PWM2_INVERT, "pwm2_invert", SENSORS_LM85_PWM2, 
                         SENSORS_NO_MAPPING, SENSORS_MODE_RW, 
                         LM85_SYSCTL_PWM_CFG2, VALUE(5), 0 },
    { SENSORS_LM85_PWM3_INVERT, "pwm3_invert", SENSORS_LM85_PWM3, 
                         SENSORS_NO_MAPPING, SENSORS_MODE_RW, 
                         LM85_SYSCTL_PWM_CFG3, VALUE(5), 0 },
    { SENSORS_LM85_PWM1_ZONE, "pwm1_zone", SENSORS_LM85_PWM1, 
                         SENSORS_NO_MAPPING, SENSORS_MODE_RW, 
                         LM85_SYSCTL_PWM_ZONE1, VALUE(1), 0 },
    { SENSORS_LM85_PWM2_ZONE, "pwm2_zone", SENSORS_LM85_PWM2, 
                         SENSORS_NO_MAPPING, SENSORS_MODE_RW, 
                         LM85_SYSCTL_PWM_ZONE2, VALUE(1), 0 },
    { SENSORS_LM85_PWM3_ZONE, "pwm3_zone", SENSORS_LM85_PWM3, 
                         SENSORS_NO_MAPPING, SENSORS_MODE_RW, 
                         LM85_SYSCTL_PWM_ZONE3, VALUE(1), 0 },
    { SENSORS_LM85_ZONE1_LIMIT, "zone1_limit", SENSORS_LM85_TEMP1, 
                         SENSORS_NO_MAPPING, SENSORS_MODE_RW, 
                         LM85_SYSCTL_ZONE1, VALUE(1), 1 },
    { SENSORS_LM85_ZONE2_LIMIT, "zone2_limit", SENSORS_LM85_TEMP2, 
                         SENSORS_NO_MAPPING, SENSORS_MODE_RW, 
                         LM85_SYSCTL_ZONE2, VALUE(1), 1 },
    { SENSORS_LM85_ZONE3_LIMIT, "zone3_limit", SENSORS_LM85_TEMP3, 
                         SENSORS_NO_MAPPING, SENSORS_MODE_RW, 
                         LM85_SYSCTL_ZONE3, VALUE(1), 1 },
    { SENSORS_LM85_ZONE1_HYST, "zone1_hyst", SENSORS_LM85_TEMP1, 
                         SENSORS_NO_MAPPING, SENSORS_MODE_RW, 
                         LM85_SYSCTL_ZONE1, VALUE(2), 1 },
    { SENSORS_LM85_ZONE2_HYST, "zone2_hyst", SENSORS_LM85_TEMP2, 
                         SENSORS_NO_MAPPING, SENSORS_MODE_RW, 
                         LM85_SYSCTL_ZONE2, VALUE(2), 1 },
    { SENSORS_LM85_ZONE3_HYST, "zone3_hyst", SENSORS_LM85_TEMP3, 
                         SENSORS_NO_MAPPING, SENSORS_MODE_RW, 
                         LM85_SYSCTL_ZONE3, VALUE(2), 1 },
    { SENSORS_LM85_ZONE1_RANGE, "zone1_range", SENSORS_LM85_TEMP1, 
                         SENSORS_NO_MAPPING, SENSORS_MODE_RW, 
                         LM85_SYSCTL_ZONE1, VALUE(3), 1 },
    { SENSORS_LM85_ZONE2_RANGE, "zone2_range", SENSORS_LM85_TEMP2, 
                         SENSORS_NO_MAPPING, SENSORS_MODE_RW, 
                         LM85_SYSCTL_ZONE2, VALUE(3), 1 },
    { SENSORS_LM85_ZONE3_RANGE, "zone3_range", SENSORS_LM85_TEMP3, 
                         SENSORS_NO_MAPPING, SENSORS_MODE_RW, 
                         LM85_SYSCTL_ZONE3, VALUE(3), 1 },
    { SENSORS_LM85_ZONE1_CRITICAL, "zone1_critical", SENSORS_LM85_TEMP1, 
                         SENSORS_NO_MAPPING, SENSORS_MODE_RW, 
                         LM85_SYSCTL_ZONE1, VALUE(4), 1 },
    { SENSORS_LM85_ZONE2_CRITICAL, "zone2_critical", SENSORS_LM85_TEMP2, 
                         SENSORS_NO_MAPPING, SENSORS_MODE_RW, 
                         LM85_SYSCTL_ZONE2, VALUE(4), 1 },
    { SENSORS_LM85_ZONE3_CRITICAL, "zone3_critical", SENSORS_LM85_TEMP3, 
                         SENSORS_NO_MAPPING, SENSORS_MODE_RW, 
                         LM85_SYSCTL_ZONE3, VALUE(4), 1 },

    { SENSORS_ADM1027_FAN1_TACH_MODE, "fan1_tach_mode", SENSORS_LM85_FAN1, 
                         SENSORS_NO_MAPPING, SENSORS_MODE_RW, 
                         ADM1027_SYSCTL_TACH_MODE, VALUE(1), 0 },
    { SENSORS_ADM1027_FAN2_TACH_MODE, "fan2_tach_mode", SENSORS_LM85_FAN2, 
                         SENSORS_NO_MAPPING, SENSORS_MODE_RW, 
                         ADM1027_SYSCTL_TACH_MODE, VALUE(2), 0 },
    { SENSORS_ADM1027_FAN3_TACH_MODE, "fan3_tach_mode", SENSORS_LM85_FAN3, 
                         SENSORS_NO_MAPPING, SENSORS_MODE_RW, 
                         ADM1027_SYSCTL_TACH_MODE, VALUE(3), 0 },
    { SENSORS_ADM1027_FAN4_TACH_MODE, "fan4_tach_mode", SENSORS_LM85_FAN4, 
                         SENSORS_NO_MAPPING, SENSORS_MODE_RW, 
                         ADM1027_SYSCTL_TACH_MODE, VALUE(4), 0 },
    { SENSORS_ADM1027_PWM1_SMOOTH, "pwm1_smooth", SENSORS_LM85_PWM1, 
                         SENSORS_NO_MAPPING, SENSORS_MODE_RW, 
                         LM85_SYSCTL_SMOOTH1, VALUE(1), 1 },
    { SENSORS_ADM1027_PWM2_SMOOTH, "pwm2_smooth", SENSORS_LM85_PWM2, 
                         SENSORS_NO_MAPPING, SENSORS_MODE_RW, 
                         LM85_SYSCTL_SMOOTH2, VALUE(1), 1 },
    { SENSORS_ADM1027_PWM3_SMOOTH, "pwm3_smooth", SENSORS_LM85_PWM3, 
                         SENSORS_NO_MAPPING, SENSORS_MODE_RW, 
                         LM85_SYSCTL_SMOOTH3, VALUE(1), 1 },
    { SENSORS_ADM1027_TEMP1_OFFSET, "temp1_offset", SENSORS_LM85_TEMP1,
                         SENSORS_LM85_TEMP1, SENSORS_MODE_RW, 
                         ADM1027_SYSCTL_TEMP_OFFSET1, VALUE(1), 2 },
    { SENSORS_ADM1027_TEMP2_OFFSET, "temp2_offset", SENSORS_LM85_TEMP2,
                         SENSORS_LM85_TEMP2, SENSORS_MODE_RW, 
                         ADM1027_SYSCTL_TEMP_OFFSET2, VALUE(1), 2 },
    { SENSORS_ADM1027_TEMP3_OFFSET, "temp3_offset", SENSORS_LM85_TEMP3,
                         SENSORS_LM85_TEMP3, SENSORS_MODE_RW, 
                         ADM1027_SYSCTL_TEMP_OFFSET3, VALUE(1), 2 },
    { SENSORS_ADM1027_FAN1_PPR, "fan1_ppr", SENSORS_LM85_FAN1, 
                         SENSORS_NO_MAPPING, SENSORS_MODE_RW, 
                         ADM1027_SYSCTL_FAN_PPR, VALUE(1), 0 },
    { SENSORS_ADM1027_FAN2_PPR, "fan2_ppr", SENSORS_LM85_FAN2, 
                         SENSORS_NO_MAPPING, SENSORS_MODE_RW, 
                         ADM1027_SYSCTL_FAN_PPR, VALUE(2), 0 },
    { SENSORS_ADM1027_FAN3_PPR, "fan3_ppr", SENSORS_LM85_FAN3, 
                         SENSORS_NO_MAPPING, SENSORS_MODE_RW, 
                         ADM1027_SYSCTL_FAN_PPR, VALUE(3), 0 },
    { SENSORS_ADM1027_FAN4_PPR, "fan4_ppr", SENSORS_LM85_FAN4, 
                         SENSORS_NO_MAPPING, SENSORS_MODE_RW, 
                         ADM1027_SYSCTL_FAN_PPR, VALUE(4), 0 },
    { SENSORS_ADM1027_ALARM_MASK, "alarm_mask", SENSORS_NO_MAPPING, 
                         SENSORS_NO_MAPPING, SENSORS_MODE_RW, 
                         ADM1027_SYSCTL_ALARM_MASK, VALUE(1), 0 },
    { 0 }
  };

static sensors_chip_feature lm87_features[] =
  { 
    { SENSORS_LM87_IN0, "2.5V", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, LM87_SYSCTL_IN0, VALUE(3), 2 },
    { SENSORS_LM87_IN1, "Vccp1", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, LM87_SYSCTL_IN1, VALUE(3), 2 },
    { SENSORS_LM87_IN2, "3.3V", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, LM87_SYSCTL_IN2, VALUE(3), 2 },
    { SENSORS_LM87_IN3, "5V", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, LM87_SYSCTL_IN3, VALUE(3), 2 },
    { SENSORS_LM87_IN4, "12V", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, LM87_SYSCTL_IN4, VALUE(3), 2 },
    { SENSORS_LM87_IN5, "Vccp2", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, LM87_SYSCTL_IN5, VALUE(3), 2 },
    { SENSORS_LM87_AIN1, "AIN1", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, LM87_SYSCTL_AIN1, VALUE(3), 0 },
    { SENSORS_LM87_AIN2, "AIN2", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, LM87_SYSCTL_AIN2, VALUE(3), 0 },
    { SENSORS_LM87_IN0_MIN, "AmbTemp_min", SENSORS_LM87_IN0, 
                        SENSORS_LM87_IN0, SENSORS_MODE_RW, 
                        LM87_SYSCTL_IN0, VALUE(1), 2 },
    { SENSORS_LM87_IN1_MIN, "Vccp1_min", SENSORS_LM87_IN1, 
                        SENSORS_LM87_IN1, SENSORS_MODE_RW, 
                        LM87_SYSCTL_IN1, VALUE(1), 2 },
    { SENSORS_LM87_IN2_MIN, "3.3V_min", SENSORS_LM87_IN2, 
                        SENSORS_LM87_IN2, SENSORS_MODE_RW, 
                        LM87_SYSCTL_IN2, VALUE(1), 2 },
    { SENSORS_LM87_IN3_MIN, "5V_min", SENSORS_LM87_IN3, 
                        SENSORS_LM87_IN3, SENSORS_MODE_RW, 
                        LM87_SYSCTL_IN3, VALUE(1), 2 },
    { SENSORS_LM87_IN4_MIN, "12V_min", SENSORS_LM87_IN4, 
                        SENSORS_LM87_IN4, SENSORS_MODE_RW, 
                        LM87_SYSCTL_IN4, VALUE(1), 2 },
    { SENSORS_LM87_IN5_MIN, "Vccp2_min", SENSORS_LM87_IN5, 
                        SENSORS_LM87_IN5, SENSORS_MODE_RW, 
                        LM87_SYSCTL_IN5, VALUE(1), 2 },
    { SENSORS_LM87_AIN1_MIN, "AIN1_min", SENSORS_LM87_AIN1, 
                        SENSORS_LM87_AIN1, SENSORS_MODE_RW, 
                        LM87_SYSCTL_AIN1, VALUE(1), 0 },
    { SENSORS_LM87_AIN2_MIN, "AIN2_min", SENSORS_LM87_AIN2, 
                        SENSORS_LM87_AIN2, SENSORS_MODE_RW, 
                        LM87_SYSCTL_AIN2, VALUE(1), 0 },
    { SENSORS_LM87_IN0_MAX, "AmbTemp_max", SENSORS_LM87_IN0, 
                        SENSORS_LM87_IN0, SENSORS_MODE_RW,
                        LM87_SYSCTL_IN0, VALUE(2), 2 },
    { SENSORS_LM87_IN1_MAX, "Vccp1_max", SENSORS_LM87_IN1, 
                        SENSORS_LM87_IN1, SENSORS_MODE_RW, 
                        LM87_SYSCTL_IN1, VALUE(2), 2 },
    { SENSORS_LM87_IN2_MAX, "3.3V_max", SENSORS_LM87_IN2, 
                        SENSORS_LM87_IN2, SENSORS_MODE_RW, 
                        LM87_SYSCTL_IN2, VALUE(2), 2 },
    { SENSORS_LM87_IN3_MAX, "5V_max", SENSORS_LM87_IN3, 
                        SENSORS_LM87_IN3, SENSORS_MODE_RW, 
                        LM87_SYSCTL_IN3, VALUE(2), 2 },
    { SENSORS_LM87_IN4_MAX, "12V_max", SENSORS_LM87_IN4, 
                        SENSORS_LM87_IN4, SENSORS_MODE_RW,
                        LM87_SYSCTL_IN4, VALUE(2), 2 },
    { SENSORS_LM87_IN5_MAX, "Vccp2_max", SENSORS_LM87_IN5, 
                        SENSORS_LM87_IN5, SENSORS_MODE_RW, 
                        LM87_SYSCTL_IN5, VALUE(2), 2 },
    { SENSORS_LM87_AIN1_MAX, "AIN1_max", SENSORS_LM87_AIN1, 
                        SENSORS_LM87_AIN1, SENSORS_MODE_RW, 
                        LM87_SYSCTL_AIN1, VALUE(2), 0 },
    { SENSORS_LM87_AIN2_MAX, "AIN2_max", SENSORS_LM87_AIN2, 
                        SENSORS_LM87_AIN2, SENSORS_MODE_RW, 
                        LM87_SYSCTL_AIN2, VALUE(2), 0 },
    { SENSORS_LM87_FAN1, "fan1", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, LM87_SYSCTL_FAN1, VALUE(2), 0 },
    { SENSORS_LM87_FAN2, "fan2", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, LM87_SYSCTL_FAN2, VALUE(2), 0 },
    { SENSORS_LM87_FAN1_MIN, "fan1_min", SENSORS_LM87_FAN1,
                        SENSORS_LM87_FAN1, SENSORS_MODE_RW, 
                        LM87_SYSCTL_FAN1, VALUE(1), 0 },
    { SENSORS_LM87_FAN2_MIN, "fan2_min", SENSORS_LM87_FAN2, 
                        SENSORS_LM87_FAN2, SENSORS_MODE_RW, 
                        LM87_SYSCTL_FAN2, VALUE(1), 0 },
    { SENSORS_LM87_TEMP3, "CPU2_Temp", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, LM87_SYSCTL_TEMP3, VALUE(3), 1 },
    { SENSORS_LM87_TEMP2, "CPU_Temp", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING, 
                        SENSORS_MODE_R, LM87_SYSCTL_TEMP2, VALUE(3), 1 },
    { SENSORS_LM87_TEMP1, "temp1", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, LM87_SYSCTL_TEMP1, VALUE(3), 1 },
    { SENSORS_LM87_TEMP1_HYST, "temp1_min", SENSORS_LM87_TEMP1,
                         SENSORS_LM87_TEMP1, SENSORS_MODE_RW, 
                         LM87_SYSCTL_TEMP1, VALUE(2), 1 },
    { SENSORS_LM87_TEMP1_OVER, "temp1_max", SENSORS_LM87_TEMP1,
                         SENSORS_LM87_TEMP1, SENSORS_MODE_RW, 
                         LM87_SYSCTL_TEMP1, VALUE(1), 1 },
    { SENSORS_LM87_TEMP2_HYST, "temp2_min", SENSORS_LM87_TEMP2,
                         SENSORS_LM87_TEMP2, SENSORS_MODE_RW, 
                         LM87_SYSCTL_TEMP2, VALUE(2), 1 },
    { SENSORS_LM87_TEMP2_OVER, "temp2_max", SENSORS_LM87_TEMP2,
                         SENSORS_LM87_TEMP2, SENSORS_MODE_RW, 
                         LM87_SYSCTL_TEMP2, VALUE(1), 1 },
    { SENSORS_LM87_TEMP3_HYST, "temp3_min", SENSORS_LM87_TEMP3,
                         SENSORS_LM87_TEMP3, SENSORS_MODE_RW, 
                         LM87_SYSCTL_TEMP3, VALUE(2), 1 },
    { SENSORS_LM87_TEMP3_OVER, "temp3_max", SENSORS_LM87_TEMP3,
                         SENSORS_LM87_TEMP3, SENSORS_MODE_RW, 
                         LM87_SYSCTL_TEMP3, VALUE(1), 1 },
    { SENSORS_LM87_VID, "vid", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, LM87_SYSCTL_VID, VALUE(1), 3 },
    { SENSORS_LM87_VRM, "vrm", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_RW, LM87_SYSCTL_VRM, VALUE(1), 1 },
/* Ho hum, this will be wrong if fan1 is disabled, but fan2 isn't.. fix?? */
    { SENSORS_LM87_FAN1_DIV, "fan1_div", SENSORS_LM87_FAN1, 
                         SENSORS_NO_MAPPING, SENSORS_MODE_RW, 
                         LM87_SYSCTL_FAN_DIV, VALUE(1), 0 },
    { SENSORS_LM87_FAN2_DIV, "fan2_div", SENSORS_LM87_FAN2, 
                         SENSORS_NO_MAPPING, SENSORS_MODE_RW, 
                         LM87_SYSCTL_FAN_DIV, VALUE(2), 0 },
    { SENSORS_LM87_ALARMS, "alarms", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING, 
                         SENSORS_MODE_R, LM87_SYSCTL_ALARMS, VALUE(1), 0 },
    { SENSORS_LM87_ANALOG_OUT, "analog_out", SENSORS_NO_MAPPING,
                         SENSORS_NO_MAPPING, SENSORS_MODE_RW,
                         LM87_SYSCTL_ANALOG_OUT, VALUE(1), 0 },
    { 0 }
  };

static sensors_chip_feature mtp008_features[] =
  { 
    { SENSORS_MTP008_IN0, "in0", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, MTP008_SYSCTL_IN0, VALUE(3), 2 },
    { SENSORS_MTP008_IN1, "in1", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, MTP008_SYSCTL_IN1, VALUE(3), 2 },
    { SENSORS_MTP008_IN2, "in2", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, MTP008_SYSCTL_IN2, VALUE(3), 2 },
    { SENSORS_MTP008_IN3, "in3", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, MTP008_SYSCTL_IN3, VALUE(3), 2 },
    { SENSORS_MTP008_IN4, "in4", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, MTP008_SYSCTL_IN4, VALUE(3), 2 },
    { SENSORS_MTP008_IN5, "in5", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, MTP008_SYSCTL_IN5, VALUE(3), 2 },
    { SENSORS_MTP008_IN6, "in6", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, MTP008_SYSCTL_IN6, VALUE(3), 2 },
    { SENSORS_MTP008_IN0_MIN, "in0_min", SENSORS_MTP008_IN0, 
                        SENSORS_MTP008_IN0, SENSORS_MODE_RW, 
                        MTP008_SYSCTL_IN0, VALUE(1), 2 },
    { SENSORS_MTP008_IN1_MIN, "in1_min", SENSORS_MTP008_IN1, 
                        SENSORS_MTP008_IN1, SENSORS_MODE_RW, 
                        MTP008_SYSCTL_IN1, VALUE(1), 2 },
    { SENSORS_MTP008_IN2_MIN, "in2_min", SENSORS_MTP008_IN2, 
                        SENSORS_MTP008_IN2, SENSORS_MODE_RW, 
                        MTP008_SYSCTL_IN2, VALUE(1), 2 },
    { SENSORS_MTP008_IN3_MIN, "in3_min", SENSORS_MTP008_IN3, 
                        SENSORS_MTP008_IN3, SENSORS_MODE_RW, 
                        MTP008_SYSCTL_IN3, VALUE(1), 2 },
    { SENSORS_MTP008_IN4_MIN, "in4_min", SENSORS_MTP008_IN4, 
                        SENSORS_MTP008_IN4, SENSORS_MODE_RW, 
                        MTP008_SYSCTL_IN4, VALUE(1), 2 },
    { SENSORS_MTP008_IN5_MIN, "in5_min", SENSORS_MTP008_IN5, 
                        SENSORS_MTP008_IN5, SENSORS_MODE_RW, 
                        MTP008_SYSCTL_IN5, VALUE(1), 2 },
    { SENSORS_MTP008_IN6_MIN, "in6_min", SENSORS_MTP008_IN6, 
                        SENSORS_MTP008_IN6, SENSORS_MODE_RW, 
                        MTP008_SYSCTL_IN6, VALUE(1), 2 },
    { SENSORS_MTP008_IN0_MAX, "in0_max", SENSORS_MTP008_IN0, 
                        SENSORS_MTP008_IN0, SENSORS_MODE_RW,
                        MTP008_SYSCTL_IN0, VALUE(2), 2 },
    { SENSORS_MTP008_IN1_MAX, "in1_max", SENSORS_MTP008_IN1, 
                        SENSORS_MTP008_IN1, SENSORS_MODE_RW, 
                        MTP008_SYSCTL_IN1, VALUE(2), 2 },
    { SENSORS_MTP008_IN2_MAX, "in2_max", SENSORS_MTP008_IN2, 
                        SENSORS_MTP008_IN2, SENSORS_MODE_RW, 
                        MTP008_SYSCTL_IN2, VALUE(2), 2 },
    { SENSORS_MTP008_IN3_MAX, "in3_max", SENSORS_MTP008_IN3, 
                        SENSORS_MTP008_IN3, SENSORS_MODE_RW, 
                        MTP008_SYSCTL_IN3, VALUE(2), 2 },
    { SENSORS_MTP008_IN4_MAX, "in4_max", SENSORS_MTP008_IN4, 
                        SENSORS_MTP008_IN4, SENSORS_MODE_RW,
                        MTP008_SYSCTL_IN4, VALUE(2), 2 },
    { SENSORS_MTP008_IN5_MAX, "in5_max", SENSORS_MTP008_IN5, 
                        SENSORS_MTP008_IN5, SENSORS_MODE_RW, 
                        MTP008_SYSCTL_IN5, VALUE(2), 2 },
    { SENSORS_MTP008_IN6_MAX, "in6_max", SENSORS_MTP008_IN6, 
                        SENSORS_MTP008_IN6, SENSORS_MODE_RW, 
                        MTP008_SYSCTL_IN6, VALUE(2), 2 },
    { SENSORS_MTP008_FAN1, "fan1", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, MTP008_SYSCTL_FAN1, VALUE(2), 0 },
    { SENSORS_MTP008_FAN2, "fan2", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, MTP008_SYSCTL_FAN2, VALUE(2), 0 },
    { SENSORS_MTP008_FAN3, "fan3", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, MTP008_SYSCTL_FAN3, VALUE(2), 0 },
    { SENSORS_MTP008_FAN1_MIN, "fan1_min", SENSORS_MTP008_FAN1,
                        SENSORS_MTP008_FAN1, SENSORS_MODE_RW, 
                        MTP008_SYSCTL_FAN1, VALUE(1), 0 },
    { SENSORS_MTP008_FAN2_MIN, "fan2_min", SENSORS_MTP008_FAN2, 
                        SENSORS_MTP008_FAN2, SENSORS_MODE_RW, 
                        MTP008_SYSCTL_FAN2, VALUE(1), 0 },
    { SENSORS_MTP008_FAN3_MIN, "fan3_min", SENSORS_MTP008_FAN3, 
                        SENSORS_MTP008_FAN3, SENSORS_MODE_RW, 
                        MTP008_SYSCTL_FAN3, VALUE(1), 0 },
    { SENSORS_MTP008_TEMP1, "temp1", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, MTP008_SYSCTL_TEMP1, VALUE(3), 1 },
    { SENSORS_MTP008_TEMP2, "temp2", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING, 
                        SENSORS_MODE_R, MTP008_SYSCTL_TEMP2, VALUE(3), 1 },
    { SENSORS_MTP008_TEMP3, "temp3", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, MTP008_SYSCTL_TEMP3, VALUE(3), 1 },
    { SENSORS_MTP008_TEMP1_OVER, "temp1_over", SENSORS_MTP008_TEMP1,
			SENSORS_MTP008_TEMP1, SENSORS_MODE_RW,
			MTP008_SYSCTL_TEMP1, VALUE(1), 1 },
    { SENSORS_MTP008_TEMP2_OVER, "temp2_over", SENSORS_MTP008_TEMP2,
			SENSORS_MTP008_TEMP2, SENSORS_MODE_RW,
			MTP008_SYSCTL_TEMP2, VALUE(1), 1 },
    { SENSORS_MTP008_TEMP3_OVER, "temp3_over", SENSORS_MTP008_TEMP3,
			SENSORS_MTP008_TEMP3, SENSORS_MODE_RW,
			MTP008_SYSCTL_TEMP3, VALUE(1), 1 },
    { SENSORS_MTP008_TEMP1_HYST, "temp1_hyst", SENSORS_MTP008_TEMP1,
			SENSORS_MTP008_TEMP1, SENSORS_MODE_RW,
			MTP008_SYSCTL_TEMP1, VALUE(2), 1 },
    { SENSORS_MTP008_TEMP2_HYST, "temp2_hyst", SENSORS_MTP008_TEMP2,
			SENSORS_MTP008_TEMP2, SENSORS_MODE_RW,
			MTP008_SYSCTL_TEMP2, VALUE(2), 1 },
    { SENSORS_MTP008_TEMP3_HYST, "temp3_hyst", SENSORS_MTP008_TEMP3,
			SENSORS_MTP008_TEMP3, SENSORS_MODE_RW,
			MTP008_SYSCTL_TEMP3, VALUE(2), 1 },
    { SENSORS_MTP008_VID, "vid", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, MTP008_SYSCTL_VID, VALUE(1), 2 },
    { SENSORS_MTP008_FAN1_DIV, "fan1_div", SENSORS_NO_MAPPING, 
                         SENSORS_NO_MAPPING, SENSORS_MODE_RW, 
                         MTP008_SYSCTL_FAN_DIV, VALUE(1), 0 },
    { SENSORS_MTP008_FAN2_DIV, "fan2_div", SENSORS_NO_MAPPING, 
                         SENSORS_NO_MAPPING, SENSORS_MODE_RW, 
                         MTP008_SYSCTL_FAN_DIV, VALUE(2), 0 },
    { SENSORS_MTP008_FAN3_DIV, "fan3_div", SENSORS_NO_MAPPING, 
                         SENSORS_NO_MAPPING, SENSORS_MODE_RW, 
                         MTP008_SYSCTL_FAN_DIV, VALUE(3), 0 },
    { SENSORS_MTP008_ALARMS, "alarms", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING, 
                         SENSORS_MODE_R, MTP008_SYSCTL_ALARMS, VALUE(1), 0 },
    { SENSORS_MTP008_BEEP, "beeps", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING, 
                         SENSORS_MODE_R, MTP008_SYSCTL_BEEP, VALUE(1), 0 },
    { 0 }
  };

static sensors_chip_feature ds1621_features[] =
  {
    { SENSORS_DS1621_TEMP, "temp", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, DS1621_SYSCTL_TEMP, VALUE(3), 1 },
    { SENSORS_DS1621_TEMP_HYST, "temp_hyst", SENSORS_DS1621_TEMP,
                              SENSORS_DS1621_TEMP, SENSORS_MODE_RW, 
                              DS1621_SYSCTL_TEMP, VALUE(2), 1 },
    { SENSORS_DS1621_TEMP_OVER, "temp_over", SENSORS_DS1621_TEMP,
                              SENSORS_DS1621_TEMP, SENSORS_MODE_RW, 
                              DS1621_SYSCTL_TEMP, VALUE(1), 1 },
    { SENSORS_DS1621_ALARMS, "alarms", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                           SENSORS_MODE_R, DS1621_SYSCTL_ALARMS, VALUE(1), 0 },
    { 0 }
  };

static sensors_chip_feature it87_features[] =
  {
    { SENSORS_IT87_IN0, "in0", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, IT87_SYSCTL_IN0, VALUE(3), 2 },
    { SENSORS_IT87_IN1, "in1", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, IT87_SYSCTL_IN1, VALUE(3), 2 },
    { SENSORS_IT87_IN2, "in2", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, IT87_SYSCTL_IN2, VALUE(3), 2 },
    { SENSORS_IT87_IN3, "in3", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, IT87_SYSCTL_IN3, VALUE(3), 2 },
    { SENSORS_IT87_IN4, "in4", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, IT87_SYSCTL_IN4, VALUE(3), 2 },
    { SENSORS_IT87_IN5, "in5", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, IT87_SYSCTL_IN5, VALUE(3), 2 },
    { SENSORS_IT87_IN6, "in6", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, IT87_SYSCTL_IN6, VALUE(3), 2 },
    { SENSORS_IT87_IN7, "in7", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, IT87_SYSCTL_IN7, VALUE(3), 2 },
    { SENSORS_IT87_IN8, "in8", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, IT87_SYSCTL_IN8, VALUE(3), 2 },
    { SENSORS_IT87_IN0_MIN, "in0_min", SENSORS_IT87_IN0, SENSORS_IT87_IN0,
                            SENSORS_MODE_RW, IT87_SYSCTL_IN0, VALUE(1), 2 },
    { SENSORS_IT87_IN1_MIN, "in1_min", SENSORS_IT87_IN1, SENSORS_IT87_IN1,
                            SENSORS_MODE_RW, IT87_SYSCTL_IN1, VALUE(1), 2 },
    { SENSORS_IT87_IN2_MIN, "in2_min", SENSORS_IT87_IN2, SENSORS_IT87_IN2,
                            SENSORS_MODE_RW, IT87_SYSCTL_IN2, VALUE(1), 2 },
    { SENSORS_IT87_IN3_MIN, "in3_min", SENSORS_IT87_IN3, SENSORS_IT87_IN3,
                            SENSORS_MODE_RW, IT87_SYSCTL_IN3, VALUE(1), 2 },
    { SENSORS_IT87_IN4_MIN, "in4_min", SENSORS_IT87_IN4, SENSORS_IT87_IN4,
                            SENSORS_MODE_RW, IT87_SYSCTL_IN4, VALUE(1), 2 },
    { SENSORS_IT87_IN5_MIN, "in5_min", SENSORS_IT87_IN5, SENSORS_IT87_IN5,
                            SENSORS_MODE_RW, IT87_SYSCTL_IN5, VALUE(1), 2 },
    { SENSORS_IT87_IN6_MIN, "in6_min", SENSORS_IT87_IN6, SENSORS_IT87_IN6,
                            SENSORS_MODE_RW, IT87_SYSCTL_IN6, VALUE(1), 2 },
    { SENSORS_IT87_IN7_MIN, "in7_min", SENSORS_IT87_IN7, SENSORS_IT87_IN7,
                            SENSORS_MODE_RW, IT87_SYSCTL_IN7, VALUE(1), 2 },
    { SENSORS_IT87_IN0_MAX, "in0_max", SENSORS_IT87_IN0, SENSORS_IT87_IN0,
                            SENSORS_MODE_RW, IT87_SYSCTL_IN0, VALUE(2), 2 },
    { SENSORS_IT87_IN1_MAX, "in1_max", SENSORS_IT87_IN1, SENSORS_IT87_IN1,
                            SENSORS_MODE_RW, IT87_SYSCTL_IN1, VALUE(2), 2 },
    { SENSORS_IT87_IN2_MAX, "in2_max", SENSORS_IT87_IN2, SENSORS_IT87_IN2,
                            SENSORS_MODE_RW, IT87_SYSCTL_IN2, VALUE(2), 2 },
    { SENSORS_IT87_IN3_MAX, "in3_max", SENSORS_IT87_IN3, SENSORS_IT87_IN3,
                            SENSORS_MODE_RW, IT87_SYSCTL_IN3, VALUE(2), 2 },
    { SENSORS_IT87_IN4_MAX, "in4_max", SENSORS_IT87_IN4, SENSORS_IT87_IN4,
                            SENSORS_MODE_RW, IT87_SYSCTL_IN4, VALUE(2), 2 },
    { SENSORS_IT87_IN5_MAX, "in5_max", SENSORS_IT87_IN5, SENSORS_IT87_IN5,
                            SENSORS_MODE_RW, IT87_SYSCTL_IN5, VALUE(2), 2 },
    { SENSORS_IT87_IN6_MAX, "in6_max", SENSORS_IT87_IN6, SENSORS_IT87_IN6,
                            SENSORS_MODE_RW, IT87_SYSCTL_IN6, VALUE(2), 2 },
    { SENSORS_IT87_IN7_MAX, "in7_max", SENSORS_IT87_IN7, SENSORS_IT87_IN7,
                            SENSORS_MODE_RW, IT87_SYSCTL_IN7, VALUE(2), 2 },
    { SENSORS_IT87_FAN1, "fan1", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, IT87_SYSCTL_FAN1, VALUE(2), 0 },
    { SENSORS_IT87_FAN2, "fan2", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, IT87_SYSCTL_FAN2, VALUE(2), 0 },
    { SENSORS_IT87_FAN3, "fan3", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, IT87_SYSCTL_FAN3, VALUE(2), 0 },
    { SENSORS_IT87_FAN1_MIN, "fan1_min", SENSORS_IT87_FAN1, SENSORS_IT87_FAN1,
                             SENSORS_MODE_RW, IT87_SYSCTL_FAN1, VALUE(1), 0 },
    { SENSORS_IT87_FAN2_MIN, "fan2_min", SENSORS_IT87_FAN2, SENSORS_IT87_FAN2,
                             SENSORS_MODE_RW, IT87_SYSCTL_FAN2, VALUE(1), 0 },
    { SENSORS_IT87_FAN3_MIN, "fan3_min", SENSORS_IT87_FAN3, SENSORS_IT87_FAN3,
                             SENSORS_MODE_RW, IT87_SYSCTL_FAN3, VALUE(1), 0 },
    { SENSORS_IT87_TEMP1, "temp1", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, IT87_SYSCTL_TEMP1, VALUE(3), 1 },
    { SENSORS_IT87_TEMP2, "temp2", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, IT87_SYSCTL_TEMP2, VALUE(3), 1 },
    { SENSORS_IT87_TEMP3, "temp3", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, IT87_SYSCTL_TEMP3, VALUE(3), 1 },
    { SENSORS_IT87_TEMP1_LOW, "temp1_hyst", SENSORS_IT87_TEMP1,
                              SENSORS_IT87_TEMP1, SENSORS_MODE_RW,
                              IT87_SYSCTL_TEMP1, VALUE(2), 1 },
    { SENSORS_IT87_TEMP2_LOW, "temp2_hyst", SENSORS_IT87_TEMP2,
                              SENSORS_IT87_TEMP2, SENSORS_MODE_RW,
                              IT87_SYSCTL_TEMP2, VALUE(2), 1 },
    { SENSORS_IT87_TEMP3_LOW, "temp3_hyst", SENSORS_IT87_TEMP3,
                              SENSORS_IT87_TEMP3, SENSORS_MODE_RW,
                              IT87_SYSCTL_TEMP3, VALUE(2), 1 },
    { SENSORS_IT87_TEMP1_HIGH, "temp1_over", SENSORS_IT87_TEMP1,
                              SENSORS_IT87_TEMP1, SENSORS_MODE_RW,
                              IT87_SYSCTL_TEMP1, VALUE(1), 1 },
    { SENSORS_IT87_TEMP2_HIGH, "temp2_over", SENSORS_IT87_TEMP2,
                              SENSORS_IT87_TEMP2, SENSORS_MODE_RW,
                              IT87_SYSCTL_TEMP2, VALUE(1), 1 },
    { SENSORS_IT87_TEMP3_HIGH, "temp3_over", SENSORS_IT87_TEMP3,
                              SENSORS_IT87_TEMP3, SENSORS_MODE_RW,
                              IT87_SYSCTL_TEMP3, VALUE(1), 1 },
    { SENSORS_IT87_VID, "vid", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, IT87_SYSCTL_VID, VALUE(1), 2 },
    { SENSORS_IT87_FAN1_DIV, "fan1_div", SENSORS_IT87_FAN1, SENSORS_NO_MAPPING,
                             SENSORS_MODE_RW, IT87_SYSCTL_FAN_DIV, VALUE(1),
                             0 },
    { SENSORS_IT87_FAN2_DIV, "fan2_div", SENSORS_IT87_FAN2, SENSORS_NO_MAPPING,
                             SENSORS_MODE_RW, IT87_SYSCTL_FAN_DIV, VALUE(2),
                             0 },
    { SENSORS_IT87_FAN3_DIV, "fan3_div", SENSORS_IT87_FAN3, SENSORS_NO_MAPPING,
                              SENSORS_MODE_RW, IT87_SYSCTL_FAN_DIV, VALUE(3),
                              0 },
    { SENSORS_IT87_ALARMS, "alarms", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                           SENSORS_MODE_R, IT87_SYSCTL_ALARMS, VALUE(1), 0 },
    { 0 }
  };

static sensors_chip_feature fscpos_features[] =
  {
    { SENSORS_FSCPOS_REV, "rev", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, FSCPOS_SYSCTL_REV, VALUE(1), 0 },
    { SENSORS_FSCPOS_EVENT, "event", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, FSCPOS_SYSCTL_EVENT, VALUE(1), 0 },
    { SENSORS_FSCPOS_CONTROL, "control", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_RW, FSCPOS_SYSCTL_CONTROL, VALUE(1), 0 },
    { SENSORS_FSCPOS_VOLTAGE1, "volt12", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, FSCPOS_SYSCTL_VOLT0, VALUE(1), 2 },
    { SENSORS_FSCPOS_VOLTAGE2, "volt5", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, FSCPOS_SYSCTL_VOLT1, VALUE(1), 2 },
    { SENSORS_FSCPOS_VOLTAGE3, "voltbatt", SENSORS_NO_MAPPING,
			SENSORS_NO_MAPPING, SENSORS_MODE_R, 
			FSCPOS_SYSCTL_VOLT2, VALUE(1), 2 },
    { SENSORS_FSCPOS_TEMP1, "temp1", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, FSCPOS_SYSCTL_TEMP0, VALUE(2), 0 },
    { SENSORS_FSCPOS_TEMP2, "temp2", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, FSCPOS_SYSCTL_TEMP1, VALUE(2), 0 },
    { SENSORS_FSCPOS_TEMP3, "temp3", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, FSCPOS_SYSCTL_TEMP2, VALUE(2), 0 },
    { SENSORS_FSCPOS_TEMP1_STATE, "temp1_state", SENSORS_FSCPOS_TEMP1,
			SENSORS_FSCPOS_TEMP1, SENSORS_MODE_RW, 
			FSCPOS_SYSCTL_TEMP0, VALUE(1), 0 },
    { SENSORS_FSCPOS_TEMP2_STATE, "temp2_state", SENSORS_FSCPOS_TEMP2, 
			SENSORS_FSCPOS_TEMP2, SENSORS_MODE_RW, 
			FSCPOS_SYSCTL_TEMP1, VALUE(1), 0 },
    { SENSORS_FSCPOS_TEMP3_STATE, "temp3_state", SENSORS_FSCPOS_TEMP3, 
			SENSORS_FSCPOS_TEMP3, SENSORS_MODE_RW, 
			FSCPOS_SYSCTL_TEMP2, VALUE(1), 0 },
    { SENSORS_FSCPOS_FAN1, "fan1", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, FSCPOS_SYSCTL_FAN0, VALUE(4), 0 },
    { SENSORS_FSCPOS_FAN2, "fan2", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, FSCPOS_SYSCTL_FAN1, VALUE(4), 0 },
    { SENSORS_FSCPOS_FAN3, "fan3", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, FSCPOS_SYSCTL_FAN2, VALUE(4), 0 },
    { SENSORS_FSCPOS_FAN1_MIN, "fan1_min", SENSORS_FSCPOS_FAN1, 
			SENSORS_FSCPOS_FAN1, SENSORS_MODE_RW, 
			FSCPOS_SYSCTL_FAN0, VALUE(2), 0 },
    { SENSORS_FSCPOS_FAN2_MIN, "fan2_min", SENSORS_FSCPOS_FAN2, 
			SENSORS_FSCPOS_FAN2, SENSORS_MODE_RW, 
			FSCPOS_SYSCTL_FAN1, VALUE(2), 0 },
    { SENSORS_FSCPOS_FAN1_STATE, "fan1_state", SENSORS_FSCPOS_FAN1, 
			SENSORS_FSCPOS_FAN1, SENSORS_MODE_RW, 
			FSCPOS_SYSCTL_FAN0, VALUE(1), 0 },
    { SENSORS_FSCPOS_FAN2_STATE, "fan2_state", SENSORS_FSCPOS_FAN2, 
			SENSORS_FSCPOS_FAN2, SENSORS_MODE_RW, 
			FSCPOS_SYSCTL_FAN1, VALUE(1), 0 },
    { SENSORS_FSCPOS_FAN3_STATE, "fan3_state", SENSORS_FSCPOS_FAN3, 
			SENSORS_FSCPOS_FAN3, SENSORS_MODE_RW, 
			FSCPOS_SYSCTL_FAN2, VALUE(1), 0 },
    { SENSORS_FSCPOS_FAN1_RIPPLE, "fan1_ripple", SENSORS_FSCPOS_FAN1, 
			SENSORS_FSCPOS_FAN1, SENSORS_MODE_RW, 
			FSCPOS_SYSCTL_FAN0, VALUE(3), 0 },
    { SENSORS_FSCPOS_FAN2_RIPPLE, "fan2_ripple", SENSORS_FSCPOS_FAN2, 
			SENSORS_FSCPOS_FAN2, SENSORS_MODE_RW, 
			FSCPOS_SYSCTL_FAN1, VALUE(3), 0 },
    { SENSORS_FSCPOS_FAN3_RIPPLE, "fan3_ripple", SENSORS_FSCPOS_FAN3, 
			SENSORS_FSCPOS_FAN3, SENSORS_MODE_RW, 
			FSCPOS_SYSCTL_FAN2, VALUE(3), 0 },
    { SENSORS_FSCPOS_WDOG_PRESET, "wdog_preset", SENSORS_NO_MAPPING, 
			SENSORS_NO_MAPPING, SENSORS_MODE_RW, 
			FSCPOS_SYSCTL_WDOG, VALUE(1), 0 },
    { SENSORS_FSCPOS_WDOG_STATE, "wdog_state", SENSORS_FSCPOS_WDOG_PRESET, 
			SENSORS_FSCPOS_WDOG_PRESET, SENSORS_MODE_RW, 
			FSCPOS_SYSCTL_WDOG, VALUE(2), 0 },
    { SENSORS_FSCPOS_WDOG_CONTROL, "wdog_control", SENSORS_FSCPOS_WDOG_PRESET, 
			SENSORS_FSCPOS_WDOG_PRESET, SENSORS_MODE_RW, 
			FSCPOS_SYSCTL_WDOG, VALUE(3), 0 },
    { 0 }
  };

static sensors_chip_feature fscscy_features[] =
  {
    { SENSORS_FSCSCY_REV, "rev", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, FSCSCY_SYSCTL_REV, VALUE(1), 0 },
    { SENSORS_FSCSCY_EVENT, "event", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, FSCSCY_SYSCTL_EVENT, VALUE(1), 0 },
    { SENSORS_FSCSCY_CONTROL, "control", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_RW, FSCSCY_SYSCTL_CONTROL, VALUE(1), 0 },
    { SENSORS_FSCSCY_VOLTAGE1, "volt12", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, FSCSCY_SYSCTL_VOLT0, VALUE(1), 2 },
    { SENSORS_FSCSCY_VOLTAGE2, "volt5", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, FSCSCY_SYSCTL_VOLT1, VALUE(1), 2 },
    { SENSORS_FSCSCY_VOLTAGE3, "voltbatt", SENSORS_NO_MAPPING,
			SENSORS_NO_MAPPING, SENSORS_MODE_R, 
			FSCSCY_SYSCTL_VOLT2, VALUE(1), 2 },
    { SENSORS_FSCSCY_TEMP1, "temp1", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, FSCSCY_SYSCTL_TEMP0, VALUE(2), 0 },
    { SENSORS_FSCSCY_TEMP2, "temp2", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, FSCSCY_SYSCTL_TEMP1, VALUE(2), 0 },
    { SENSORS_FSCSCY_TEMP3, "temp3", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, FSCSCY_SYSCTL_TEMP2, VALUE(2), 0 },
    { SENSORS_FSCSCY_TEMP4, "temp4", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, FSCSCY_SYSCTL_TEMP3, VALUE(2), 0 },
    { SENSORS_FSCSCY_TEMP1_STATE, "temp1_state", SENSORS_FSCSCY_TEMP1,
			SENSORS_FSCSCY_TEMP1, SENSORS_MODE_RW, 
			FSCSCY_SYSCTL_TEMP0, VALUE(1), 0 },
    { SENSORS_FSCSCY_TEMP2_STATE, "temp2_state", SENSORS_FSCSCY_TEMP2, 
			SENSORS_FSCSCY_TEMP2, SENSORS_MODE_RW, 
			FSCSCY_SYSCTL_TEMP1, VALUE(1), 0 },
    { SENSORS_FSCSCY_TEMP3_STATE, "temp3_state", SENSORS_FSCSCY_TEMP3, 
			SENSORS_FSCSCY_TEMP3, SENSORS_MODE_RW, 
			FSCSCY_SYSCTL_TEMP2, VALUE(1), 0 },
    { SENSORS_FSCSCY_TEMP4_STATE, "temp4_state", SENSORS_FSCSCY_TEMP4, 
			SENSORS_FSCSCY_TEMP4, SENSORS_MODE_RW, 
			FSCSCY_SYSCTL_TEMP3, VALUE(1), 0 },
    { SENSORS_FSCSCY_TEMP1_LIM, "temp1_lim", SENSORS_FSCSCY_TEMP1,
			SENSORS_FSCSCY_TEMP1, SENSORS_MODE_R, 
			FSCSCY_SYSCTL_TEMP0, VALUE(3), 0 },
    { SENSORS_FSCSCY_TEMP2_LIM, "temp2_lim", SENSORS_FSCSCY_TEMP2,
			SENSORS_FSCSCY_TEMP2, SENSORS_MODE_R, 
			FSCSCY_SYSCTL_TEMP1, VALUE(3), 0 },
    { SENSORS_FSCSCY_TEMP3_LIM, "temp3_lim", SENSORS_FSCSCY_TEMP3,
			SENSORS_FSCSCY_TEMP3, SENSORS_MODE_R, 
			FSCSCY_SYSCTL_TEMP2, VALUE(3), 0 },
    { SENSORS_FSCSCY_TEMP4_LIM, "temp4_lim", SENSORS_FSCSCY_TEMP4,
			SENSORS_FSCSCY_TEMP4, SENSORS_MODE_R, 
			FSCSCY_SYSCTL_TEMP3, VALUE(3), 0 },
    { SENSORS_FSCSCY_TEMP1_MIN, "temp1_min", SENSORS_FSCSCY_TEMP1,
			SENSORS_FSCSCY_TEMP1, SENSORS_MODE_R, 
			FSCSCY_SYSCTL_TEMP0, VALUE(4), 0 },
    { SENSORS_FSCSCY_TEMP2_MIN, "temp2_min", SENSORS_FSCSCY_TEMP2,
			SENSORS_FSCSCY_TEMP2, SENSORS_MODE_R, 
			FSCSCY_SYSCTL_TEMP1, VALUE(4), 0 },
    { SENSORS_FSCSCY_TEMP3_MIN, "temp3_min", SENSORS_FSCSCY_TEMP3,
			SENSORS_FSCSCY_TEMP3, SENSORS_MODE_R, 
			FSCSCY_SYSCTL_TEMP2, VALUE(4), 0 },
    { SENSORS_FSCSCY_TEMP4_MIN, "temp4_min", SENSORS_FSCSCY_TEMP4,
			SENSORS_FSCSCY_TEMP4, SENSORS_MODE_R, 
			FSCSCY_SYSCTL_TEMP3, VALUE(4), 0 },
    { SENSORS_FSCSCY_TEMP1_MAX, "temp1_max", SENSORS_FSCSCY_TEMP1,
			SENSORS_FSCSCY_TEMP1, SENSORS_MODE_R, 
			FSCSCY_SYSCTL_TEMP0, VALUE(5), 0 },
    { SENSORS_FSCSCY_TEMP2_MAX, "temp2_max", SENSORS_FSCSCY_TEMP2,
			SENSORS_FSCSCY_TEMP2, SENSORS_MODE_R, 
			FSCSCY_SYSCTL_TEMP1, VALUE(5), 0 },
    { SENSORS_FSCSCY_TEMP3_MAX, "temp3_max", SENSORS_FSCSCY_TEMP3,
			SENSORS_FSCSCY_TEMP3, SENSORS_MODE_R, 
			FSCSCY_SYSCTL_TEMP2, VALUE(5), 0 },
    { SENSORS_FSCSCY_TEMP4_MAX, "temp4_max", SENSORS_FSCSCY_TEMP4,
			SENSORS_FSCSCY_TEMP4, SENSORS_MODE_R, 
			FSCSCY_SYSCTL_TEMP3, VALUE(5), 0 },
    { SENSORS_FSCSCY_FAN1, "fan1", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, FSCSCY_SYSCTL_FAN0, VALUE(4), 0 },
    { SENSORS_FSCSCY_FAN2, "fan2", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, FSCSCY_SYSCTL_FAN1, VALUE(4), 0 },
    { SENSORS_FSCSCY_FAN3, "fan3", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, FSCSCY_SYSCTL_FAN2, VALUE(4), 0 },
    { SENSORS_FSCSCY_FAN4, "fan4", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, FSCSCY_SYSCTL_FAN3, VALUE(4), 0 },
    { SENSORS_FSCSCY_FAN5, "fan5", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, FSCSCY_SYSCTL_FAN4, VALUE(4), 0 },
    { SENSORS_FSCSCY_FAN6, "fan6", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, FSCSCY_SYSCTL_FAN5, VALUE(4), 0 },
    { SENSORS_FSCSCY_FAN1_MIN, "fan1_min", SENSORS_FSCSCY_FAN1, 
			SENSORS_FSCSCY_FAN1, SENSORS_MODE_RW, 
			FSCSCY_SYSCTL_FAN0, VALUE(2), 0 },
    { SENSORS_FSCSCY_FAN2_MIN, "fan2_min", SENSORS_FSCSCY_FAN2, 
			SENSORS_FSCSCY_FAN2, SENSORS_MODE_RW, 
			FSCSCY_SYSCTL_FAN1, VALUE(2), 0 },
    { SENSORS_FSCSCY_FAN3_MIN, "fan3_min", SENSORS_FSCSCY_FAN3, 
			SENSORS_FSCSCY_FAN3, SENSORS_MODE_RW, 
			FSCSCY_SYSCTL_FAN2, VALUE(2), 0 },
    { SENSORS_FSCSCY_FAN4_MIN, "fan4_min", SENSORS_FSCSCY_FAN4, 
			SENSORS_FSCSCY_FAN4, SENSORS_MODE_RW, 
			FSCSCY_SYSCTL_FAN3, VALUE(2), 0 },
    { SENSORS_FSCSCY_FAN5_MIN, "fan5_min", SENSORS_FSCSCY_FAN5, 
			SENSORS_FSCSCY_FAN5, SENSORS_MODE_RW, 
			FSCSCY_SYSCTL_FAN4, VALUE(2), 0 },
    { SENSORS_FSCSCY_FAN6_MIN, "fan6_min", SENSORS_FSCSCY_FAN6, 
			SENSORS_FSCSCY_FAN6, SENSORS_MODE_RW, 
			FSCSCY_SYSCTL_FAN5, VALUE(2), 0 },
    { SENSORS_FSCSCY_FAN1_STATE, "fan1_state", SENSORS_FSCSCY_FAN1, 
			SENSORS_FSCSCY_FAN1, SENSORS_MODE_RW, 
			FSCSCY_SYSCTL_FAN0, VALUE(1), 0 },
    { SENSORS_FSCSCY_FAN2_STATE, "fan2_state", SENSORS_FSCSCY_FAN2, 
			SENSORS_FSCSCY_FAN2, SENSORS_MODE_RW, 
			FSCSCY_SYSCTL_FAN1, VALUE(1), 0 },
    { SENSORS_FSCSCY_FAN3_STATE, "fan3_state", SENSORS_FSCSCY_FAN3, 
			SENSORS_FSCSCY_FAN3, SENSORS_MODE_RW, 
			FSCSCY_SYSCTL_FAN2, VALUE(1), 0 },
    { SENSORS_FSCSCY_FAN4_STATE, "fan4_state", SENSORS_FSCSCY_FAN4, 
			SENSORS_FSCSCY_FAN4, SENSORS_MODE_RW, 
			FSCSCY_SYSCTL_FAN3, VALUE(1), 0 },
    { SENSORS_FSCSCY_FAN5_STATE, "fan5_state", SENSORS_FSCSCY_FAN5, 
			SENSORS_FSCSCY_FAN5, SENSORS_MODE_RW, 
			FSCSCY_SYSCTL_FAN4, VALUE(1), 0 },
    { SENSORS_FSCSCY_FAN6_STATE, "fan6_state", SENSORS_FSCSCY_FAN6, 
			SENSORS_FSCSCY_FAN6, SENSORS_MODE_RW, 
			FSCSCY_SYSCTL_FAN5, VALUE(1), 0 },
    { SENSORS_FSCSCY_FAN1_RIPPLE, "fan1_ripple", SENSORS_FSCSCY_FAN1, 
			SENSORS_FSCSCY_FAN1, SENSORS_MODE_RW, 
			FSCSCY_SYSCTL_FAN0, VALUE(3), 0 },
    { SENSORS_FSCSCY_FAN2_RIPPLE, "fan2_ripple", SENSORS_FSCSCY_FAN2, 
			SENSORS_FSCSCY_FAN2, SENSORS_MODE_RW, 
			FSCSCY_SYSCTL_FAN1, VALUE(3), 0 },
    { SENSORS_FSCSCY_FAN3_RIPPLE, "fan3_ripple", SENSORS_FSCSCY_FAN3, 
			SENSORS_FSCSCY_FAN3, SENSORS_MODE_RW, 
			FSCSCY_SYSCTL_FAN2, VALUE(3), 0 },
    { SENSORS_FSCSCY_FAN4_RIPPLE, "fan4_ripple", SENSORS_FSCSCY_FAN4, 
			SENSORS_FSCSCY_FAN4, SENSORS_MODE_RW, 
			FSCSCY_SYSCTL_FAN3, VALUE(3), 0 },
    { SENSORS_FSCSCY_FAN5_RIPPLE, "fan5_ripple", SENSORS_FSCSCY_FAN5, 
			SENSORS_FSCSCY_FAN5, SENSORS_MODE_RW, 
			FSCSCY_SYSCTL_FAN4, VALUE(3), 0 },
    { SENSORS_FSCSCY_FAN6_RIPPLE, "fan6_ripple", SENSORS_FSCSCY_FAN6, 
			SENSORS_FSCSCY_FAN6, SENSORS_MODE_RW, 
			FSCSCY_SYSCTL_FAN5, VALUE(3), 0 },
    { SENSORS_FSCSCY_WDOG_PRESET, "wdog_preset", SENSORS_NO_MAPPING, 
			SENSORS_NO_MAPPING, SENSORS_MODE_RW, 
			FSCSCY_SYSCTL_WDOG, VALUE(1), 0 },
    { SENSORS_FSCSCY_WDOG_STATE, "wdog_state", SENSORS_FSCSCY_WDOG_PRESET, 
			SENSORS_FSCSCY_WDOG_PRESET, SENSORS_MODE_RW, 
			FSCSCY_SYSCTL_WDOG, VALUE(2), 0 },
    { SENSORS_FSCSCY_WDOG_CONTROL, "wdog_control", SENSORS_FSCSCY_WDOG_PRESET, 
			SENSORS_FSCSCY_WDOG_PRESET, SENSORS_MODE_RW, 
			FSCSCY_SYSCTL_WDOG, VALUE(3), 0 },
    { 0 }
  };

  static sensors_chip_feature pcf8591_features[] =
  {
    { SENSORS_PCF8591_AIN_CONF, "ain_conf", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                              SENSORS_MODE_RW, PCF8591_SYSCTL_AIN_CONF, VALUE(1), 0 },
    { SENSORS_PCF8591_CH0, "ch0", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, PCF8591_SYSCTL_CH0, VALUE(1), 0 },
    { SENSORS_PCF8591_CH1, "ch1", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, PCF8591_SYSCTL_CH1, VALUE(1), 0 },
    { SENSORS_PCF8591_CH2, "ch2", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, PCF8591_SYSCTL_CH2, VALUE(1), 0 },
    { SENSORS_PCF8591_CH3, "ch3", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, PCF8591_SYSCTL_CH3, VALUE(1), 0 },
    { SENSORS_PCF8591_AOUT_ENABLE, "aout_enable", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                                 SENSORS_MODE_RW, PCF8591_SYSCTL_AOUT_ENABLE, VALUE(1), 0 },
    { SENSORS_PCF8591_AOUT, "aout", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                          SENSORS_MODE_RW, PCF8591_SYSCTL_AOUT, VALUE(1), 0 },
    { 0 }
  };


static sensors_chip_feature vt1211_features[] =
  { 
    { SENSORS_VT1211_IN0, "in0", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING, 
                        SENSORS_MODE_R, VT1211_SYSCTL_IN0, VALUE(3), 2 },
    { SENSORS_VT1211_IN1, "in1", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, VT1211_SYSCTL_IN1, VALUE(3), 2 },
    { SENSORS_VT1211_IN2, "in2", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, VT1211_SYSCTL_IN2, VALUE(3), 2 },
    { SENSORS_VT1211_IN3, "in3", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, VT1211_SYSCTL_IN3, VALUE(3), 2 },
    { SENSORS_VT1211_IN4, "in4", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, VT1211_SYSCTL_IN4, VALUE(3), 2 },
    { SENSORS_VT1211_IN5, "in5", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, VT1211_SYSCTL_IN5, VALUE(3), 2 },
    { SENSORS_VT1211_IN6, "in6", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, VT1211_SYSCTL_IN6, VALUE(3), 2 },
    { SENSORS_VT1211_IN0_MIN, "in0_min", SENSORS_VT1211_IN0,
                            SENSORS_VT1211_IN0,
                            SENSORS_MODE_RW, VT1211_SYSCTL_IN0, VALUE(1), 2 },
    { SENSORS_VT1211_IN1_MIN, "in1_min", SENSORS_VT1211_IN1,
                            SENSORS_VT1211_IN1,
                            SENSORS_MODE_RW, VT1211_SYSCTL_IN1, VALUE(1), 2 },
    { SENSORS_VT1211_IN2_MIN, "in2_min", SENSORS_VT1211_IN2,
                            SENSORS_VT1211_IN2,
                            SENSORS_MODE_RW, VT1211_SYSCTL_IN2, VALUE(1), 2 },
    { SENSORS_VT1211_IN3_MIN, "in3_min", SENSORS_VT1211_IN3,
                            SENSORS_VT1211_IN3,
                            SENSORS_MODE_RW, VT1211_SYSCTL_IN3, VALUE(1), 2 },
    { SENSORS_VT1211_IN4_MIN, "in4_min", SENSORS_VT1211_IN4,
                            SENSORS_VT1211_IN4,
                            SENSORS_MODE_RW, VT1211_SYSCTL_IN4, VALUE(1), 2 },
    { SENSORS_VT1211_IN5_MIN, "in5_min", SENSORS_VT1211_IN5,
                            SENSORS_VT1211_IN5,
                            SENSORS_MODE_RW, VT1211_SYSCTL_IN5, VALUE(1), 2 },
    { SENSORS_VT1211_IN6_MIN, "in6_min", SENSORS_VT1211_IN6,
                            SENSORS_VT1211_IN6,
                            SENSORS_MODE_RW, VT1211_SYSCTL_IN6, VALUE(1), 2 },
    { SENSORS_VT1211_IN0_MAX, "in0_max", SENSORS_VT1211_IN0,
                            SENSORS_VT1211_IN0,
                            SENSORS_MODE_RW, VT1211_SYSCTL_IN0, VALUE(2), 2 },
    { SENSORS_VT1211_IN1_MAX, "in1_max", SENSORS_VT1211_IN1,
                            SENSORS_VT1211_IN1,
                            SENSORS_MODE_RW, VT1211_SYSCTL_IN1, VALUE(2), 2 },
    { SENSORS_VT1211_IN2_MAX, "in2_max", SENSORS_VT1211_IN2,
                            SENSORS_VT1211_IN2,
                            SENSORS_MODE_RW, VT1211_SYSCTL_IN2, VALUE(2), 2 },
    { SENSORS_VT1211_IN3_MAX, "in3_max", SENSORS_VT1211_IN3,
                            SENSORS_VT1211_IN3,
                            SENSORS_MODE_RW, VT1211_SYSCTL_IN3, VALUE(2), 2 },
    { SENSORS_VT1211_IN4_MAX, "in4_max", SENSORS_VT1211_IN4,
                            SENSORS_VT1211_IN4,
                            SENSORS_MODE_RW, VT1211_SYSCTL_IN4, VALUE(2), 2 },
    { SENSORS_VT1211_IN5_MAX, "in5_max", SENSORS_VT1211_IN5,
                            SENSORS_VT1211_IN5,
                            SENSORS_MODE_RW, VT1211_SYSCTL_IN5, VALUE(2), 2 },
    { SENSORS_VT1211_IN6_MAX, "in6_max", SENSORS_VT1211_IN6,
                            SENSORS_VT1211_IN6,
                            SENSORS_MODE_RW, VT1211_SYSCTL_IN6, VALUE(2), 2 },
    { SENSORS_VT1211_FAN1, "fan1", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, VT1211_SYSCTL_FAN1, VALUE(2), 0 },
    { SENSORS_VT1211_FAN2, "fan2", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, VT1211_SYSCTL_FAN2, VALUE(2), 0 },
    { SENSORS_VT1211_FAN1_MIN, "fan1_min", SENSORS_VT1211_FAN1,
                             SENSORS_VT1211_FAN1,
                             SENSORS_MODE_RW, VT1211_SYSCTL_FAN1, VALUE(1), 0 },
    { SENSORS_VT1211_FAN2_MIN, "fan2_min", SENSORS_VT1211_FAN2,
                             SENSORS_VT1211_FAN2,
                             SENSORS_MODE_RW, VT1211_SYSCTL_FAN2, VALUE(1), 0 },
    { SENSORS_VT1211_TEMP, "temp1", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, VT1211_SYSCTL_TEMP, VALUE(3), 1 },
    { SENSORS_VT1211_TEMP_HYST, "temp1_hyst", SENSORS_VT1211_TEMP,
                              SENSORS_VT1211_TEMP, SENSORS_MODE_RW, 
                              VT1211_SYSCTL_TEMP, VALUE(2), 1 },
    { SENSORS_VT1211_TEMP_OVER, "temp1_over", SENSORS_VT1211_TEMP,
                              SENSORS_VT1211_TEMP, SENSORS_MODE_RW, 
                              VT1211_SYSCTL_TEMP, VALUE(1), 1 },
    { SENSORS_VT1211_TEMP2, "temp2", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, VT1211_SYSCTL_TEMP2, VALUE(3), 1 },
    { SENSORS_VT1211_TEMP2_HYST, "temp2_hyst", SENSORS_VT1211_TEMP2,
                              SENSORS_VT1211_TEMP2, SENSORS_MODE_RW, 
                              VT1211_SYSCTL_TEMP2, VALUE(2), 1 },
    { SENSORS_VT1211_TEMP2_OVER, "temp2_over", SENSORS_VT1211_TEMP2,
                              SENSORS_VT1211_TEMP2, SENSORS_MODE_RW, 
                              VT1211_SYSCTL_TEMP2, VALUE(1), 1 },
    { SENSORS_VT1211_TEMP3, "temp3", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, VT1211_SYSCTL_TEMP3, VALUE(3), 1 },
    { SENSORS_VT1211_TEMP3_HYST, "temp3_hyst", SENSORS_VT1211_TEMP3,
                              SENSORS_VT1211_TEMP3, SENSORS_MODE_RW, 
                              VT1211_SYSCTL_TEMP3, VALUE(2), 1 },
    { SENSORS_VT1211_TEMP3_OVER, "temp3_over", SENSORS_VT1211_TEMP3,
                              SENSORS_VT1211_TEMP3, SENSORS_MODE_RW, 
                              VT1211_SYSCTL_TEMP3, VALUE(1), 1 },
    { SENSORS_VT1211_TEMP4, "temp4", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, VT1211_SYSCTL_TEMP4, VALUE(3), 1 },
    { SENSORS_VT1211_TEMP4_HYST, "temp4_hyst", SENSORS_VT1211_TEMP4,
                              SENSORS_VT1211_TEMP4, SENSORS_MODE_RW, 
                              VT1211_SYSCTL_TEMP4, VALUE(2), 1 },
    { SENSORS_VT1211_TEMP4_OVER, "temp4_over", SENSORS_VT1211_TEMP4,
                              SENSORS_VT1211_TEMP4, SENSORS_MODE_RW, 
                              VT1211_SYSCTL_TEMP4, VALUE(1), 1 },
    { SENSORS_VT1211_TEMP5, "temp5", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, VT1211_SYSCTL_TEMP5, VALUE(3), 1 },
    { SENSORS_VT1211_TEMP5_HYST, "temp5_hyst", SENSORS_VT1211_TEMP5,
                              SENSORS_VT1211_TEMP5, SENSORS_MODE_RW, 
                              VT1211_SYSCTL_TEMP5, VALUE(2), 1 },
    { SENSORS_VT1211_TEMP5_OVER, "temp5_over", SENSORS_VT1211_TEMP5,
                              SENSORS_VT1211_TEMP5, SENSORS_MODE_RW, 
                              VT1211_SYSCTL_TEMP5, VALUE(1), 1 },
    { SENSORS_VT1211_TEMP6, "temp6", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, VT1211_SYSCTL_TEMP6, VALUE(3), 1 },
    { SENSORS_VT1211_TEMP6_HYST, "temp6_hyst", SENSORS_VT1211_TEMP6,
                              SENSORS_VT1211_TEMP6, SENSORS_MODE_RW, 
                              VT1211_SYSCTL_TEMP6, VALUE(2), 1 },
    { SENSORS_VT1211_TEMP6_OVER, "temp6_over", SENSORS_VT1211_TEMP6,
                              SENSORS_VT1211_TEMP6, SENSORS_MODE_RW, 
                              VT1211_SYSCTL_TEMP6, VALUE(1), 1 },
    { SENSORS_VT1211_TEMP7, "temp7", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, VT1211_SYSCTL_TEMP7, VALUE(3), 1 },
    { SENSORS_VT1211_TEMP7_HYST, "temp7_hyst", SENSORS_VT1211_TEMP7,
                              SENSORS_VT1211_TEMP7, SENSORS_MODE_RW, 
                              VT1211_SYSCTL_TEMP7, VALUE(2), 1 },
    { SENSORS_VT1211_TEMP7_OVER, "temp7_over", SENSORS_VT1211_TEMP7,
                              SENSORS_VT1211_TEMP7, SENSORS_MODE_RW, 
                              VT1211_SYSCTL_TEMP7, VALUE(1), 1 },
    { SENSORS_VT1211_FAN1_DIV, "fan1_div", SENSORS_VT1211_FAN1,
                             SENSORS_NO_MAPPING,
                             SENSORS_MODE_RW, VT1211_SYSCTL_FAN_DIV, VALUE(1), 
                             0 },
    { SENSORS_VT1211_FAN2_DIV, "fan2_div", SENSORS_VT1211_FAN2,
                             SENSORS_NO_MAPPING,
                             SENSORS_MODE_RW, VT1211_SYSCTL_FAN_DIV, VALUE(2), 
                             0 },
    { SENSORS_VT1211_ALARMS, "alarms", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING, 
                           SENSORS_MODE_R, VT1211_SYSCTL_ALARMS, VALUE(1), 0 },
    { SENSORS_VT1211_VID, "vid", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING, 
                           SENSORS_MODE_R, VT1211_SYSCTL_VID, VALUE(1), 3 },
    { SENSORS_VT1211_VRM, "vrm", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING, 
                           SENSORS_MODE_RW, VT1211_SYSCTL_VRM, VALUE(1), 1 },
    { SENSORS_VT1211_UCH, "config", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING, 
                           SENSORS_MODE_R, VT1211_SYSCTL_UCH, VALUE(1), 0 },
    { 0 }
  };

static sensors_chip_feature smsc47m1_features[] =
  { 
    { SENSORS_SMSC47M1_FAN1, "fan1", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, SMSC47M1_SYSCTL_FAN1, VALUE(2), 0 },
    { SENSORS_SMSC47M1_FAN2, "fan2", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, SMSC47M1_SYSCTL_FAN2, VALUE(2), 0 },
    { SENSORS_SMSC47M1_FAN1_MIN, "fan1_min", SENSORS_SMSC47M1_FAN1,
                             SENSORS_SMSC47M1_FAN1,
                             SENSORS_MODE_RW, SMSC47M1_SYSCTL_FAN1, VALUE(1), 0 },
    { SENSORS_SMSC47M1_FAN2_MIN, "fan2_min", SENSORS_SMSC47M1_FAN2,
                             SENSORS_SMSC47M1_FAN2,
                             SENSORS_MODE_RW, SMSC47M1_SYSCTL_FAN2, VALUE(1), 0 },
    { SENSORS_SMSC47M1_FAN1_DIV, "fan1_div", SENSORS_SMSC47M1_FAN1,
                             SENSORS_NO_MAPPING,
                             SENSORS_MODE_RW, SMSC47M1_SYSCTL_FAN_DIV, VALUE(1), 
                             0 },
    { SENSORS_SMSC47M1_FAN2_DIV, "fan2_div", SENSORS_SMSC47M1_FAN2,
                             SENSORS_NO_MAPPING,
                             SENSORS_MODE_RW, SMSC47M1_SYSCTL_FAN_DIV, VALUE(2), 
                             0 },
    { SENSORS_SMSC47M1_ALARMS, "alarms", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING, 
                           SENSORS_MODE_R, SMSC47M1_SYSCTL_ALARMS, VALUE(1), 0 },
    { 0 }
  };

static sensors_chip_feature lm92_features[] =
  {
    { SENSORS_LM92_TEMP, "temp", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                       SENSORS_MODE_R, LM92_SYSCTL_TEMP, VALUE(1), 4 },
    { SENSORS_LM92_TEMP_HIGH, "temp_high", SENSORS_LM92_TEMP, SENSORS_LM92_TEMP,
                       SENSORS_MODE_RW, LM92_SYSCTL_TEMP, VALUE(2), 4 },
    { SENSORS_LM92_TEMP_LOW, "temp_low", SENSORS_LM92_TEMP, SENSORS_LM92_TEMP,
                       SENSORS_MODE_RW, LM92_SYSCTL_TEMP, VALUE(3), 4 },
    { SENSORS_LM92_TEMP_CRIT, "temp_crit", SENSORS_LM92_TEMP, SENSORS_LM92_TEMP,
                       SENSORS_MODE_RW, LM92_SYSCTL_TEMP, VALUE(4), 4 },
    { SENSORS_LM92_TEMP_HYST, "temp_hyst", SENSORS_LM92_TEMP, SENSORS_LM92_TEMP,
                       SENSORS_MODE_RW, LM92_SYSCTL_TEMP, VALUE(5), 4 },
    { SENSORS_LM92_ALARMS, "alarms", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                       SENSORS_MODE_R, LM92_SYSCTL_ALARMS, VALUE(1), 0 },
    { 0 }
  };

static sensors_chip_feature vt8231_features[] =
  { 
    { SENSORS_VT8231_IN0, "in0", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING, 
                        SENSORS_MODE_R, VT8231_SYSCTL_IN0, VALUE(3), 2 },
    { SENSORS_VT8231_IN1, "in1", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, VT8231_SYSCTL_IN1, VALUE(3), 2 },
    { SENSORS_VT8231_IN2, "in2", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, VT8231_SYSCTL_IN2, VALUE(3), 2 },
    { SENSORS_VT8231_IN3, "in3", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, VT8231_SYSCTL_IN3, VALUE(3), 2 },
    { SENSORS_VT8231_IN4, "in4", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, VT8231_SYSCTL_IN4, VALUE(3), 2 },
    { SENSORS_VT8231_IN5, "in5", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, VT8231_SYSCTL_IN5, VALUE(3), 2 },
    { SENSORS_VT8231_IN6, "in6", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                        SENSORS_MODE_R, VT8231_SYSCTL_IN6, VALUE(3), 2 },
    { SENSORS_VT8231_IN0_MIN, "in0_min", SENSORS_VT8231_IN0,
                            SENSORS_VT8231_IN0,
                            SENSORS_MODE_RW, VT8231_SYSCTL_IN0, VALUE(1), 2 },
    { SENSORS_VT8231_IN1_MIN, "in1_min", SENSORS_VT8231_IN1,
                            SENSORS_VT8231_IN1,
                            SENSORS_MODE_RW, VT8231_SYSCTL_IN1, VALUE(1), 2 },
    { SENSORS_VT8231_IN2_MIN, "in2_min", SENSORS_VT8231_IN2,
                            SENSORS_VT8231_IN2,
                            SENSORS_MODE_RW, VT8231_SYSCTL_IN2, VALUE(1), 2 },
    { SENSORS_VT8231_IN3_MIN, "in3_min", SENSORS_VT8231_IN3,
                            SENSORS_VT8231_IN3,
                            SENSORS_MODE_RW, VT8231_SYSCTL_IN3, VALUE(1), 2 },
    { SENSORS_VT8231_IN4_MIN, "in4_min", SENSORS_VT8231_IN4,
                            SENSORS_VT8231_IN4,
                            SENSORS_MODE_RW, VT8231_SYSCTL_IN4, VALUE(1), 2 },
    { SENSORS_VT8231_IN5_MIN, "in5_min", SENSORS_VT8231_IN5,
                            SENSORS_VT8231_IN5,
                            SENSORS_MODE_RW, VT8231_SYSCTL_IN5, VALUE(1), 2 },
    { SENSORS_VT8231_IN6_MIN, "in6_min", SENSORS_VT8231_IN6,
                            SENSORS_VT8231_IN6,
                            SENSORS_MODE_RW, VT8231_SYSCTL_IN6, VALUE(1), 2 },
    { SENSORS_VT8231_IN0_MAX, "in0_max", SENSORS_VT8231_IN0,
                            SENSORS_VT8231_IN0,
                            SENSORS_MODE_RW, VT8231_SYSCTL_IN0, VALUE(2), 2 },
    { SENSORS_VT8231_IN1_MAX, "in1_max", SENSORS_VT8231_IN1,
                            SENSORS_VT8231_IN1,
                            SENSORS_MODE_RW, VT8231_SYSCTL_IN1, VALUE(2), 2 },
    { SENSORS_VT8231_IN2_MAX, "in2_max", SENSORS_VT8231_IN2,
                            SENSORS_VT8231_IN2,
                            SENSORS_MODE_RW, VT8231_SYSCTL_IN2, VALUE(2), 2 },
    { SENSORS_VT8231_IN3_MAX, "in3_max", SENSORS_VT8231_IN3,
                            SENSORS_VT8231_IN3,
                            SENSORS_MODE_RW, VT8231_SYSCTL_IN3, VALUE(2), 2 },
    { SENSORS_VT8231_IN4_MAX, "in4_max", SENSORS_VT8231_IN4,
                            SENSORS_VT8231_IN4,
                            SENSORS_MODE_RW, VT8231_SYSCTL_IN4, VALUE(2), 2 },
    { SENSORS_VT8231_IN5_MAX, "in5_max", SENSORS_VT8231_IN5,
                            SENSORS_VT8231_IN5,
                            SENSORS_MODE_RW, VT8231_SYSCTL_IN5, VALUE(2), 2 },
    { SENSORS_VT8231_IN6_MAX, "in6_max", SENSORS_VT8231_IN6,
                            SENSORS_VT8231_IN6,
                            SENSORS_MODE_RW, VT8231_SYSCTL_IN6, VALUE(2), 2 },
    { SENSORS_VT8231_FAN1, "fan1", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, VT8231_SYSCTL_FAN1, VALUE(2), 0 },
    { SENSORS_VT8231_FAN2, "fan2", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, VT8231_SYSCTL_FAN2, VALUE(2), 0 },
    { SENSORS_VT8231_FAN1_MIN, "fan1_min", SENSORS_VT8231_FAN1,
                             SENSORS_VT8231_FAN1,
                             SENSORS_MODE_RW, VT8231_SYSCTL_FAN1, VALUE(1), 0 },
    { SENSORS_VT8231_FAN2_MIN, "fan2_min", SENSORS_VT8231_FAN2,
                             SENSORS_VT8231_FAN2,
                             SENSORS_MODE_RW, VT8231_SYSCTL_FAN2, VALUE(1), 0 },
    { SENSORS_VT8231_TEMP, "temp1", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, VT8231_SYSCTL_TEMP, VALUE(3), 1 },
    { SENSORS_VT8231_TEMP_HYST, "temp1_hyst", SENSORS_VT8231_TEMP,
                              SENSORS_VT8231_TEMP, SENSORS_MODE_RW, 
                              VT8231_SYSCTL_TEMP, VALUE(2), 1 },
    { SENSORS_VT8231_TEMP_OVER, "temp1_over", SENSORS_VT8231_TEMP,
                              SENSORS_VT8231_TEMP, SENSORS_MODE_RW, 
                              VT8231_SYSCTL_TEMP, VALUE(1), 1 },
    { SENSORS_VT8231_TEMP2, "temp2", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, VT8231_SYSCTL_TEMP2, VALUE(3), 1 },
    { SENSORS_VT8231_TEMP2_HYST, "temp2_hyst", SENSORS_VT8231_TEMP2,
                              SENSORS_VT8231_TEMP2, SENSORS_MODE_RW, 
                              VT8231_SYSCTL_TEMP2, VALUE(2), 1 },
    { SENSORS_VT8231_TEMP2_OVER, "temp2_over", SENSORS_VT8231_TEMP2,
                              SENSORS_VT8231_TEMP2, SENSORS_MODE_RW, 
                              VT8231_SYSCTL_TEMP2, VALUE(1), 1 },
    { SENSORS_VT8231_TEMP3, "temp3", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, VT8231_SYSCTL_TEMP3, VALUE(3), 1 },
    { SENSORS_VT8231_TEMP3_HYST, "temp3_hyst", SENSORS_VT8231_TEMP3,
                              SENSORS_VT8231_TEMP3, SENSORS_MODE_RW, 
                              VT8231_SYSCTL_TEMP3, VALUE(2), 1 },
    { SENSORS_VT8231_TEMP3_OVER, "temp3_over", SENSORS_VT8231_TEMP3,
                              SENSORS_VT8231_TEMP3, SENSORS_MODE_RW, 
                              VT8231_SYSCTL_TEMP3, VALUE(1), 1 },
    { SENSORS_VT8231_TEMP4, "temp4", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, VT8231_SYSCTL_TEMP4, VALUE(3), 1 },
    { SENSORS_VT8231_TEMP4_HYST, "temp4_hyst", SENSORS_VT8231_TEMP4,
                              SENSORS_VT8231_TEMP4, SENSORS_MODE_RW, 
                              VT8231_SYSCTL_TEMP4, VALUE(2), 1 },
    { SENSORS_VT8231_TEMP4_OVER, "temp4_over", SENSORS_VT8231_TEMP4,
                              SENSORS_VT8231_TEMP4, SENSORS_MODE_RW, 
                              VT8231_SYSCTL_TEMP4, VALUE(1), 1 },
    { SENSORS_VT8231_TEMP5, "temp5", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, VT8231_SYSCTL_TEMP5, VALUE(3), 1 },
    { SENSORS_VT8231_TEMP5_HYST, "temp5_hyst", SENSORS_VT8231_TEMP5,
                              SENSORS_VT8231_TEMP5, SENSORS_MODE_RW, 
                              VT8231_SYSCTL_TEMP5, VALUE(2), 1 },
    { SENSORS_VT8231_TEMP5_OVER, "temp5_over", SENSORS_VT8231_TEMP5,
                              SENSORS_VT8231_TEMP5, SENSORS_MODE_RW, 
                              VT8231_SYSCTL_TEMP5, VALUE(1), 1 },
    { SENSORS_VT8231_TEMP6, "temp6", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, VT8231_SYSCTL_TEMP6, VALUE(3), 1 },
    { SENSORS_VT8231_TEMP6_HYST, "temp6_hyst", SENSORS_VT8231_TEMP6,
                              SENSORS_VT8231_TEMP6, SENSORS_MODE_RW, 
                              VT8231_SYSCTL_TEMP6, VALUE(2), 1 },
    { SENSORS_VT8231_TEMP6_OVER, "temp6_over", SENSORS_VT8231_TEMP6,
                              SENSORS_VT8231_TEMP6, SENSORS_MODE_RW, 
                              VT8231_SYSCTL_TEMP6, VALUE(1), 1 },
    { SENSORS_VT8231_TEMP7, "temp7", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, VT8231_SYSCTL_TEMP7, VALUE(3), 1 },
    { SENSORS_VT8231_TEMP7_HYST, "temp7_hyst", SENSORS_VT8231_TEMP7,
                              SENSORS_VT8231_TEMP7, SENSORS_MODE_RW, 
                              VT8231_SYSCTL_TEMP7, VALUE(2), 1 },
    { SENSORS_VT8231_TEMP7_OVER, "temp7_over", SENSORS_VT8231_TEMP7,
                              SENSORS_VT8231_TEMP7, SENSORS_MODE_RW, 
                              VT8231_SYSCTL_TEMP7, VALUE(1), 1 },
    { SENSORS_VT8231_FAN1_DIV, "fan1_div", SENSORS_VT8231_FAN1,
                             SENSORS_NO_MAPPING,
                             SENSORS_MODE_RW, VT8231_SYSCTL_FAN_DIV, VALUE(1), 
                             0 },
    { SENSORS_VT8231_FAN2_DIV, "fan2_div", SENSORS_VT8231_FAN2,
                             SENSORS_NO_MAPPING,
                             SENSORS_MODE_RW, VT8231_SYSCTL_FAN_DIV, VALUE(2), 
                             0 },
    { SENSORS_VT8231_ALARMS, "alarms", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING, 
                           SENSORS_MODE_R, VT8231_SYSCTL_ALARMS, VALUE(1), 0 },
    { SENSORS_VT8231_VID, "vid", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING, 
                           SENSORS_MODE_R, VT8231_SYSCTL_VID, VALUE(1), 3 },
    { SENSORS_VT8231_VRM, "vrm", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING, 
                           SENSORS_MODE_RW, VT8231_SYSCTL_VRM, VALUE(1), 1 },
    { SENSORS_VT8231_UCH, "config", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING, 
                           SENSORS_MODE_R, VT8231_SYSCTL_UCH, VALUE(1), 0 },
    { 0 }
  };

static sensors_chip_feature bmc_features[] =
  { 
    { SENSORS_BMC_IN1, "in1", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING, 
                        SENSORS_MODE_R, BMC_SYSCTL_IN1, VALUE(3), 2 },
    { SENSORS_BMC_IN1_MIN, "in1_min", SENSORS_BMC_IN1,
                            SENSORS_BMC_IN1,
                            SENSORS_MODE_RW, BMC_SYSCTL_IN1, VALUE(1), 2 },
    { SENSORS_BMC_IN1_MAX, "in1_max", SENSORS_BMC_IN1,
                            SENSORS_BMC_IN1,
                            SENSORS_MODE_RW, BMC_SYSCTL_IN1, VALUE(2), 2 },
    { SENSORS_BMC_IN1+1, "in2", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING, 
                        SENSORS_MODE_R, BMC_SYSCTL_IN1+1, VALUE(3), 2 },
    { SENSORS_BMC_IN1_MIN+1, "in2_min", SENSORS_BMC_IN1+1,
                            SENSORS_BMC_IN1+1,
                            SENSORS_MODE_RW, BMC_SYSCTL_IN1+1, VALUE(1), 2 },
    { SENSORS_BMC_IN1_MAX+1, "in2_max", SENSORS_BMC_IN1+1,
                            SENSORS_BMC_IN1+1,
                            SENSORS_MODE_RW, BMC_SYSCTL_IN1+1, VALUE(2), 2 },
    { SENSORS_BMC_IN1+2, "in3", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING, 
                        SENSORS_MODE_R, BMC_SYSCTL_IN1+2, VALUE(3), 2 },
    { SENSORS_BMC_IN1_MIN+2, "in3_min", SENSORS_BMC_IN1+2,
                            SENSORS_BMC_IN1+2,
                            SENSORS_MODE_RW, BMC_SYSCTL_IN1+2, VALUE(1), 2 },
    { SENSORS_BMC_IN1_MAX+2, "in3_max", SENSORS_BMC_IN1+2,
                            SENSORS_BMC_IN1+2,
                            SENSORS_MODE_RW, BMC_SYSCTL_IN1+2, VALUE(2), 2 },
    { SENSORS_BMC_IN1+3, "in4", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING, 
                        SENSORS_MODE_R, BMC_SYSCTL_IN1+3, VALUE(3), 2 },
    { SENSORS_BMC_IN1_MIN+3, "in4_min", SENSORS_BMC_IN1+3,
                            SENSORS_BMC_IN1+3,
                            SENSORS_MODE_RW, BMC_SYSCTL_IN1+3, VALUE(1), 2 },
    { SENSORS_BMC_IN1_MAX+3, "in4_max", SENSORS_BMC_IN1+3,
                            SENSORS_BMC_IN1+3,
                            SENSORS_MODE_RW, BMC_SYSCTL_IN1+3, VALUE(2), 2 },
    { SENSORS_BMC_IN1+4, "in5", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING, 
                        SENSORS_MODE_R, BMC_SYSCTL_IN1+4, VALUE(3), 2 },
    { SENSORS_BMC_IN1_MIN+4, "in5_min", SENSORS_BMC_IN1+4,
                            SENSORS_BMC_IN1+4,
                            SENSORS_MODE_RW, BMC_SYSCTL_IN1+4, VALUE(1), 2 },
    { SENSORS_BMC_IN1_MAX+4, "in5_max", SENSORS_BMC_IN1+4,
                            SENSORS_BMC_IN1+4,
                            SENSORS_MODE_RW, BMC_SYSCTL_IN1+4, VALUE(2), 2 },
    { SENSORS_BMC_IN1+5, "in6", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING, 
                        SENSORS_MODE_R, BMC_SYSCTL_IN1+5, VALUE(3), 2 },
    { SENSORS_BMC_IN1_MIN+5, "in6_min", SENSORS_BMC_IN1+5,
                            SENSORS_BMC_IN1+5,
                            SENSORS_MODE_RW, BMC_SYSCTL_IN1+5, VALUE(1), 2 },
    { SENSORS_BMC_IN1_MAX+5, "in6_max", SENSORS_BMC_IN1+5,
                            SENSORS_BMC_IN1+5,
                            SENSORS_MODE_RW, BMC_SYSCTL_IN1+5, VALUE(2), 2 },
    { SENSORS_BMC_IN1+6, "in7", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING, 
                        SENSORS_MODE_R, BMC_SYSCTL_IN1+6, VALUE(3), 2 },
    { SENSORS_BMC_IN1_MIN+6, "in7_min", SENSORS_BMC_IN1+6,
                            SENSORS_BMC_IN1+6,
                            SENSORS_MODE_RW, BMC_SYSCTL_IN1+6, VALUE(1), 2 },
    { SENSORS_BMC_IN1_MAX+6, "in7_max", SENSORS_BMC_IN1+6,
                            SENSORS_BMC_IN1+6,
                            SENSORS_MODE_RW, BMC_SYSCTL_IN1+6, VALUE(2), 2 },
    { SENSORS_BMC_FAN1, "fan1", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, BMC_SYSCTL_FAN1, VALUE(2), 0 },
    { SENSORS_BMC_FAN1+1, "fan2", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, BMC_SYSCTL_FAN1+1, VALUE(2), 0 },
    { SENSORS_BMC_FAN1+2, "fan3", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, BMC_SYSCTL_FAN1+2, VALUE(2), 0 },
    { SENSORS_BMC_FAN1+3, "fan4", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, BMC_SYSCTL_FAN1+3, VALUE(2), 0 },
    { SENSORS_BMC_FAN1+4, "fan5", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, BMC_SYSCTL_FAN1+4, VALUE(2), 0 },
    { SENSORS_BMC_FAN1+5, "fan6", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, BMC_SYSCTL_FAN1+5, VALUE(2), 0 },
    { SENSORS_BMC_FAN1+6, "fan7", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, BMC_SYSCTL_FAN1+6, VALUE(2), 0 },
    { SENSORS_BMC_FAN1_MIN, "fan1_min", SENSORS_BMC_FAN1,
                             SENSORS_BMC_FAN1,
                             SENSORS_MODE_RW, BMC_SYSCTL_FAN1, VALUE(1), 0 },
    { SENSORS_BMC_FAN1_MIN+1, "fan2_min", SENSORS_BMC_FAN1+1,
                             SENSORS_BMC_FAN1+1,
                             SENSORS_MODE_RW, BMC_SYSCTL_FAN1+1, VALUE(1), 0 },
    { SENSORS_BMC_FAN1_MIN+2, "fan3_min", SENSORS_BMC_FAN1+2,
                             SENSORS_BMC_FAN1+2,
                             SENSORS_MODE_RW, BMC_SYSCTL_FAN1+2, VALUE(1), 0 },
    { SENSORS_BMC_FAN1_MIN+3, "fan4_min", SENSORS_BMC_FAN1+3,
                             SENSORS_BMC_FAN1+3,
                             SENSORS_MODE_RW, BMC_SYSCTL_FAN1+3, VALUE(1), 0 },
    { SENSORS_BMC_FAN1_MIN+4, "fan5_min", SENSORS_BMC_FAN1+4,
                             SENSORS_BMC_FAN1+4,
                             SENSORS_MODE_RW, BMC_SYSCTL_FAN1+4, VALUE(1), 0 },
    { SENSORS_BMC_FAN1_MIN+5, "fan6_min", SENSORS_BMC_FAN1+5,
                             SENSORS_BMC_FAN1+5,
                             SENSORS_MODE_RW, BMC_SYSCTL_FAN1+5, VALUE(1), 0 },
    { SENSORS_BMC_FAN1_MIN+6, "fan7_min", SENSORS_BMC_FAN1+6,
                             SENSORS_BMC_FAN1+6,
                             SENSORS_MODE_RW, BMC_SYSCTL_FAN1+6, VALUE(1), 0 },
    { SENSORS_BMC_TEMP1, "temp1", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, BMC_SYSCTL_TEMP1, VALUE(3), 0 },
    { SENSORS_BMC_TEMP1_MIN, "temp1_min", SENSORS_BMC_TEMP1,
                              SENSORS_BMC_TEMP1, SENSORS_MODE_RW, 
                              BMC_SYSCTL_TEMP1, VALUE(2), 0 },
    { SENSORS_BMC_TEMP1_MAX, "temp1_max", SENSORS_BMC_TEMP1,
                              SENSORS_BMC_TEMP1, SENSORS_MODE_RW, 
                              BMC_SYSCTL_TEMP1, VALUE(1), 0 },
    { SENSORS_BMC_TEMP1+1, "temp2", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, BMC_SYSCTL_TEMP1+1, VALUE(3), 0 },
    { SENSORS_BMC_TEMP1_MIN+1, "temp2_min", SENSORS_BMC_TEMP1+1,
                              SENSORS_BMC_TEMP1+1, SENSORS_MODE_RW, 
                              BMC_SYSCTL_TEMP1+1, VALUE(2), 0 },
    { SENSORS_BMC_TEMP1_MAX+1, "temp2_max", SENSORS_BMC_TEMP1+1,
                              SENSORS_BMC_TEMP1+1, SENSORS_MODE_RW, 
                              BMC_SYSCTL_TEMP1+1, VALUE(1), 0 },
    { SENSORS_BMC_TEMP1+2, "temp3", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, BMC_SYSCTL_TEMP1+2, VALUE(3), 0 },
    { SENSORS_BMC_TEMP1_MIN+2, "temp3_min", SENSORS_BMC_TEMP1+2,
                              SENSORS_BMC_TEMP1+2, SENSORS_MODE_RW, 
                              BMC_SYSCTL_TEMP1+2, VALUE(2), 0 },
    { SENSORS_BMC_TEMP1_MAX+2, "temp3_max", SENSORS_BMC_TEMP1+2,
                              SENSORS_BMC_TEMP1+2, SENSORS_MODE_RW, 
                              BMC_SYSCTL_TEMP1+2, VALUE(1), 0 },
	/* scalings from here down */
    { SENSORS_BMC_TEMP1+3, "temp4", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, BMC_SYSCTL_TEMP1+3, VALUE(3), 1 },
    { SENSORS_BMC_TEMP1_MIN+3, "temp4_min", SENSORS_BMC_TEMP1+3,
                              SENSORS_BMC_TEMP1+3, SENSORS_MODE_RW, 
                              BMC_SYSCTL_TEMP1+3, VALUE(2), 1 },
    { SENSORS_BMC_TEMP1_MAX+3, "temp4_max", SENSORS_BMC_TEMP1+3,
                              SENSORS_BMC_TEMP1+3, SENSORS_MODE_RW, 
                              BMC_SYSCTL_TEMP1+3, VALUE(1), 1 },
    { SENSORS_BMC_TEMP1+4, "temp5", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, BMC_SYSCTL_TEMP1+4, VALUE(3), 1 },
    { SENSORS_BMC_TEMP1_MIN+4, "temp5_min", SENSORS_BMC_TEMP1+4,
                              SENSORS_BMC_TEMP1+4, SENSORS_MODE_RW, 
                              BMC_SYSCTL_TEMP1+4, VALUE(2), 1 },
    { SENSORS_BMC_TEMP1_MAX+4, "temp5_max", SENSORS_BMC_TEMP1+4,
                              SENSORS_BMC_TEMP1+4, SENSORS_MODE_RW, 
                              BMC_SYSCTL_TEMP1+4, VALUE(1), 1 },
    { SENSORS_BMC_TEMP1+5, "temp6", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, BMC_SYSCTL_TEMP1+5, VALUE(3), 1 },
    { SENSORS_BMC_TEMP1_MIN+5, "temp6_min", SENSORS_BMC_TEMP1+5,
                              SENSORS_BMC_TEMP1+5, SENSORS_MODE_RW, 
                              BMC_SYSCTL_TEMP1+5, VALUE(2), 1 },
    { SENSORS_BMC_TEMP1_MAX+5, "temp6_max", SENSORS_BMC_TEMP1+5,
                              SENSORS_BMC_TEMP1+5, SENSORS_MODE_RW, 
                              BMC_SYSCTL_TEMP1+5, VALUE(1), 1 },
    { SENSORS_BMC_TEMP1+6, "temp7", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING,
                         SENSORS_MODE_R, BMC_SYSCTL_TEMP1+6, VALUE(3), 1 },
    { SENSORS_BMC_TEMP1_MIN+6, "temp7_min", SENSORS_BMC_TEMP1+6,
                              SENSORS_BMC_TEMP1+6, SENSORS_MODE_RW, 
                              BMC_SYSCTL_TEMP1+6, VALUE(2), 1 },
    { SENSORS_BMC_TEMP1_MAX+6, "temp7_max", SENSORS_BMC_TEMP1+6,
                              SENSORS_BMC_TEMP1+6, SENSORS_MODE_RW, 
                              BMC_SYSCTL_TEMP1+6, VALUE(1), 1 },
    { SENSORS_BMC_ALARMS, "alarms", SENSORS_NO_MAPPING, SENSORS_NO_MAPPING, 
                           SENSORS_MODE_R, BMC_SYSCTL_ALARMS, VALUE(1), 0 },
    { 0 }
  };

static sensors_chip_feature xeontemp_features[] =
  {
    { SENSORS_XEONTEMP_REMOTE_TEMP, "temp", SENSORS_NO_MAPPING, 
                              SENSORS_NO_MAPPING, SENSORS_MODE_R, 
                              XEONTEMP_SYSCTL_REMOTE_TEMP, VALUE(3), 0 },
    { SENSORS_XEONTEMP_REMOTE_TEMP_HYST, "temp_low", 
                              SENSORS_XEONTEMP_REMOTE_TEMP,
                              SENSORS_XEONTEMP_REMOTE_TEMP, SENSORS_MODE_RW, 
                              XEONTEMP_SYSCTL_REMOTE_TEMP, VALUE(2), 0 },
    { SENSORS_XEONTEMP_REMOTE_TEMP_OVER, "temp_over", 
                              SENSORS_XEONTEMP_REMOTE_TEMP,
                              SENSORS_XEONTEMP_REMOTE_TEMP, SENSORS_MODE_RW,
                              XEONTEMP_SYSCTL_REMOTE_TEMP, VALUE(1), 0 },
    { SENSORS_XEONTEMP_ALARMS, "alarms", SENSORS_NO_MAPPING,
                              SENSORS_NO_MAPPING, SENSORS_MODE_R,
                              XEONTEMP_SYSCTL_ALARMS, VALUE(1), 0 },
    { 0 }
  };

static sensors_chip_feature max6650_features[] =
  {
    { SENSORS_MAX6650_FAN1_TACH, "fan1", SENSORS_NO_MAPPING,
                                 SENSORS_NO_MAPPING, SENSORS_MODE_R,
                                 MAX6650_SYSCTL_FAN1, VALUE(1), 0 },
    { SENSORS_MAX6650_FAN2_TACH, "fan2", SENSORS_NO_MAPPING,
                                 SENSORS_NO_MAPPING, SENSORS_MODE_R,
                                 MAX6650_SYSCTL_FAN2, VALUE(1), 0 },
    { SENSORS_MAX6650_FAN3_TACH, "fan3", SENSORS_NO_MAPPING,
                                 SENSORS_NO_MAPPING, SENSORS_MODE_R,
                                 MAX6650_SYSCTL_FAN3, VALUE(1), 0 },
    { SENSORS_MAX6650_FAN4_TACH, "fan4", SENSORS_NO_MAPPING,
                                 SENSORS_NO_MAPPING, SENSORS_MODE_R,
                                 MAX6650_SYSCTL_FAN4, VALUE(1), 0 },
    { SENSORS_MAX6650_SPEED, "speed", SENSORS_NO_MAPPING,
                                 SENSORS_NO_MAPPING, SENSORS_MODE_RW,
                                 MAX6650_SYSCTL_SPEED, VALUE(1), 0 },
  };

sensors_chip_features sensors_chip_features_list[] =
{
 { SENSORS_LM78_PREFIX, lm78_features },
 { SENSORS_LM78J_PREFIX, lm78j_features },
 { SENSORS_LM79_PREFIX, lm79_features },
 { SENSORS_LM75_PREFIX, lm75_features },
 { SENSORS_GL518_PREFIX, gl518_features },
 { SENSORS_GL520_PREFIX, gl520_features },
 { SENSORS_LM80_PREFIX, lm80_features },
 { SENSORS_W83781D_PREFIX, w83781d_features },
 { SENSORS_W83782D_PREFIX, w83782d_features },
 { SENSORS_W83783S_PREFIX, w83783s_features },
 { SENSORS_W83697HF_PREFIX, w83782d_features }, /* same as 782d */
		/* Cheat on 83627HF for now - no separate #defines */
 { SENSORS_W83627HF_PREFIX, w83782d_features },
 { SENSORS_W83791D_PREFIX, w83791d_features },
 { SENSORS_AS99127F_PREFIX, as99127f_features },
 { SENSORS_ADM9240_PREFIX, adm9240_features },
		/* Cheat on LM81 for now - no separate #defines */
 { SENSORS_LM81_PREFIX, adm9240_features },
 { SENSORS_DS1780_PREFIX, ds1780_features },
 { SENSORS_ADM1021_PREFIX, adm1021_features },
 { SENSORS_MAX1617_PREFIX, max1617_features },
 { SENSORS_MC1066_PREFIX, max1617_features },
 { SENSORS_MAX1617A_PREFIX, max1617a_features },
		/* Cheat on LM84 for now - no separate #defines */
 { SENSORS_LM84_PREFIX, adm1021_features },
		/* Cheat on GL523 for now - no separate #defines */
 { SENSORS_GL523_PREFIX, adm1021_features },
 { SENSORS_ADM1023_PREFIX, adm1023_features },
		/* Cheat on THMC10 for now - no separate #defines */
 { SENSORS_THMC10_PREFIX, adm1021_features },
 { SENSORS_SIS5595_PREFIX, sis5595_features },
 { SENSORS_MAXI_CG_PREFIX, maxi_cg_features },
 { SENSORS_MAXI_CO_PREFIX, maxi_co_features },
 { SENSORS_MAXI_AS_PREFIX, maxi_as_features },
		/* Cheat on NBA for now - no separate #defines */
 { SENSORS_MAXI_NBA_PREFIX, maxi_as_features },
 { SENSORS_THMC50_PREFIX, thmc50_features },
		/* Cheat on ADM1022 for now - no separate #defines */
 { SENSORS_ADM1022_PREFIX, thmc50_features },
 { SENSORS_ADM1025_PREFIX, adm1025_features },
 { SENSORS_ADM1026_PREFIX, adm1026_features },
 { SENSORS_VIA686A_PREFIX, via686a_features },
 { SENSORS_DDCMON_PREFIX, ddcmon_features },
 { SENSORS_EEPROM_PREFIX, eeprom_features },
 { SENSORS_LM87_PREFIX, lm87_features },
 { SENSORS_MTP008_PREFIX, mtp008_features },
 { SENSORS_DS1621_PREFIX, ds1621_features },
 { SENSORS_ADM1024_PREFIX, adm1024_features },
 { SENSORS_IT87_PREFIX, it87_features },
 { SENSORS_FSCPOS_PREFIX, fscpos_features },
 { SENSORS_FSCSCY_PREFIX, fscscy_features },
 { SENSORS_PCF8591_PREFIX, pcf8591_features }, 
 { SENSORS_VT1211_PREFIX, vt1211_features }, 
 { SENSORS_SMSC47M1_PREFIX, smsc47m1_features }, 
 { SENSORS_LM92_PREFIX, lm92_features }, 
 { SENSORS_VT8231_PREFIX, vt8231_features }, 
 { SENSORS_BMC_PREFIX, bmc_features }, 
 { SENSORS_LM85_PREFIX, lm85_features },
 { SENSORS_LM85B_PREFIX, lm85_features },
 { SENSORS_LM85C_PREFIX, lm85_features },
 { SENSORS_EMC6D100_PREFIX, lm85_features },
 { SENSORS_ADM1027_PREFIX, adm1027_features },
 { SENSORS_ADT7463_PREFIX, adm1027_features },
 { SENSORS_LM83_PREFIX, lm83_features },
 { SENSORS_LM90_PREFIX, lm90_features },
 { SENSORS_XEONTEMP_PREFIX, xeontemp_features },
 { SENSORS_MAX6650_PREFIX, max6650_features },
 { 0 }
};
